﻿--[[ RECIPE BOOK SKILLS :
Contains most of the functions used for skill display (i.e. /rb skill)

]]--

--============================== VARIABLES ==============================--
-- Search Frame Globals --
local RB_SearchResults = {};
RBSearch = {};
RBSKILLSEARCHFRAME_NUMITEMS = 13;
RBSKILLSEARCH_CBXSTATUS = {
	["RBUI_SearchFrame_CBX_SearchItems"]= {["check"] = true},
	["RBUI_SearchFrame_CBX_SearchMats"]= {["check"] = false},
	["RBUI_SearchFrame_CBX_SearchExact"]= {["check"] = false},
	["RBUI_SearchFrame_CBX_SearchBoosts"]= {["check"] = false},
	};

--============================== FUNCTIONS ==============================--
--[[ OnShow() ]]--
function RBSearch:OnShow()
	for name, data in pairs(RBSKILLSEARCH_CBXSTATUS) do
		getglobal(name):SetChecked(data["check"]);
	end
	RBSearch:OnUpdate();
end

--[[ OnUpdate() ]]--
function RBSearch:OnUpdate()
	RBUI_SearchFrame_ScrollFrame_ScrollBarUpdate();
end


--[[ OnHide() ]]--
function RBSearch:OnHide()
	RB_SearchResults = {};
end

--[[ DoSearch() --> Searches data ]]--
function RBSearch:DoSearch()
	-- Blank previous search results
	RB_SearchResults = {};
	RBUI_SearchFrame_ScrollFrame_ScrollBarUpdate();
	-- Search with parameters provided
	RBSearch:Search(RBUI_SearchFrame_EDIT_SearchFor:GetText(), RBUI_SearchFrame_CBX_SearchItems:GetChecked(), RBUI_SearchFrame_CBX_SearchMats:GetChecked());
	table.sort(RB_SearchResults, function(a,b) return a[3]<b[3] end);
	if #RB_SearchResults < 1 then table.insert(RB_SearchResults, {nil, nil, RECIPEBOOK_ERR_NOSEARCHRESULT, 0, {}, "", false, {}}) end;
	RBUI_SearchFrame_EDIT_SearchFor:HighlightText(0);
	RBSearch:OnUpdate();
end


--[[ SearchItems(what, items, materials) --> Searches items data for matches 
	Searching items yields all items which match what.
		resultIT= {id, rid, name, skill, {all mats}, {known}};
	Searching materials yields all items using a material matching what.
		resultMT = {id, rid, name, skill,  {all mats}, {known}, {matching mats}};
	Ultimately, these are combined into a meaningful whole.
		RB_SearchResults = { id, rid, name, skill, known, flavor text, {left-click tooltip}, {right-click tooltip}}
		left-click tooltip should generally behave by linking, etc.
		right-click tooltip should display a list of who knows the item and who has the skill.
]]--
function RBSearch:Search(what, matchItems, matchMats)
	RecipeBook:Debug("Search called.");
	local resultIT = {};
	local resultMT = {};

	local itemT, matsT = RBDB:GetAllItems(matchMats);
	local exact = RBUI_SearchFrame_CBX_SearchExact:GetChecked();
	if matchItems and itemT and next(itemT) ~= nil then
		resultIT = RBSearch:MatchText(what, itemT, exact, true);
		RBSearch:FormatData(resultIT, false);
	end
	if matchMats and matsT and next(matsT) ~= nil then
		resultMT = RBSearch:MatchText(what, matsT, exact, false);
		RBSearch:FormatData(resultMT, true);
	end
end

--[[ FormatData(match table, do we have mats data?) --> Inserts the results into the RecipeBook search results list. 
	RB_SearchResults = { id, rid, name, skill, known, flavor text, {left-click tooltip}, {right-click tooltip}}
	left-click tooltip should generally behave by linking, etc.  Use "false" to represent the default behavior.
	right-click tooltip should display a list of who knows the item and who has the skill.
]]--
function RBSearch:FormatData(infoT, hasMats)
	local id, rid, name, skill;
	local flavorText = "";
	local leftClick, rightClick = {}, {};
	local matsT, knownT, matchT;
	local valid = true;
	
	for i, data in pairs(infoT) do
		id, rid, name, skill, matsT, knownT, matchT = unpack(data);
		if not hasMats then matchT = nil end;
		
		-- Format left-click (false: SetItemRef)
		leftClick = false;
		
		-- Format right-click lines and filter for Boost checkbox - you must qualify to be included if SearchBoosts is checked.
		if RBUI_SearchFrame_CBX_SearchBoosts:GetChecked() then valid = false else valid = true end;
		local haveT = RBDB:GetAllCharacters(skill) -- Everyone who has the skill and is displayed
		rightClick = {RECIPEBOOK_TXT_KNOWNBY};
		local kc, hc = 0, 0;
		for fac, altT in pairs(knownT) do
			for who, diff in pairs(altT) do -- Now we have faction, name, difficulty
				-- If we are filtering for boosts only then this determines whether we add this item.
				if valid == false and diff > 1 then valid = true end; -- If this item has someone it boosts, it is valid.
				-- If this item will be added:
				if valid then 
					-- Only count people who are being displayed, remove them from the skills table, and increment counter
					if haveT[who] ~= nil then 
						table.insert(rightClick, (haveT[who][1]..who..RB_HEXCOLOR["end"].." ("..RBDB:Difficulty_DBToText(diff)..")") ); -- "Bob (optimal)"
						haveT[who] = nil;
						kc = kc + 1;
					end
				end
			end
		end
		
		-- If we are going to insert this item then also add the 'has' data to the right-click tooltip
		if valid then
			table.insert(rightClick, string.format(RECIPEBOOK_TXT_HASKILL, RBDB:Skill_DBToText(skill)));
			for who, data in pairs(haveT) do
				table.insert(rightClick, (data[1]..who..RB_HEXCOLOR["end"].." ("..data[2][RBDB:TradeskillIndex_TextToDB("rank")]..")"));
				hc = hc + 1;
			end
		end
		
		-- Format flavor text if we are displaying
		if valid then 
			if hasMats then 
				flavorText = RECIPEBOOK_TEMPLATE_KNOWNMATS;
				for i, mats in ipairs(matchT) do -- {item, item name}
					local name = mats[2];
					if flavorText == RECIPEBOOK_TEMPLATE_KNOWNMATS then -- First item.
						flavorText = flavorText .. name;
					elseif string.sub(flavorText, -3) == "..." then -- Already at max length
					elseif string.len(flavorText .. ", ".. name) < 60 then -- OK to add item
						flavorText = flavorText .. ", "..name;
					else -- This will make it too long
						flavorText = flavorText .. "...";
					end
				end
			else
				local text = "";
				if kc == 1 then text = RECIPEBOOK_TEMPLATE_KNOWNS else text = string.format(RECIPEBOOK_TEMPLATE_KNOWNM, kc) end;
				if hc == 1 then flavorText = text .. string.format(RECIPEBOOK_TEMPLATE_HAVES, RBDB:Skill_DBToText(skill));
				else flavorText = text .. string.format(RECIPEBOOK_TEMPLATE_HAVEM, hc, RBDB:Skill_DBToText(skill));
				end;
			end
			-- Insert your formatted data string into the Search Results.
			-- RecipeBook:Debug("Inserting data for : "..name..(hasMats and " (With Mats)" or ""));
			table.insert(RB_SearchResults, {id, rid, name, skill, flavorText, leftClick, rightClick});
		end
	end
end
		
--[[ MatchText(match string, potential matches, return table, match exact only?, indexes irrelevant?) --> Returns a table of items that match the info ]]--
function RBSearch:MatchText(what, matchT, matchExact, indexed)
	local resultT = {};
	local itemsT = {};
	local indexT = {};
	local id;
	-- matchT = { id = {"Name to match to", other data} - indexed false
	-- resultT = { {id, rid, "Name to match to", other data} } - indexed true
	for i, data in pairs(matchT) do
		-- Find the string we are matching on.
		if indexed then rname = data[3];
		else rname = data[1];
		end
		
		name = string.lower(rname);
		-- Determine whether text matches
		local match = false;
	    if matchExact then
	        if string.find(name, string.lower(what)) then match = true end;
		else
			for _, word in pairs(RBOutput:Words(what)) do
			    -- Requires all words to be present
			    if string.find(name, string.lower(word)) then match = true;
				else
					match = false;
					break;
				end;
			end
		end
		-- Format returned data
		-- indexed = {id, rid, name, skill, {all mats}, {known}};
		-- not indexed = {id, rid, name, skill,  {all mats} {known},  {matching mats},};
		
		-- If a match has been found:
		if match then
			-- We have all the indexed stuff; just send it back (does not require mats table).
			if indexed then table.insert(resultT, data);
			else
				-- Add this item for all recipes which use it
				for x, idT in pairs(data[2]) do
					local id, rid = unpack(idT);
					-- RecipeBook:Debug("Adding for item: "..i.." ("..rname..") - ID: "..id.." RID: "..rid);
					if itemsT[rid] == nil then itemsT[rid] = {} end; -- Create a table for the given recipe.
					itemsT[rid]["ID"] = id; -- Create a subtable for the item ID; we'll need it later
					table.insert(itemsT[rid], {i, rname}) -- Insert this materials item
				end
				-- Now we have our mats table.  Format our return data.
				local dbinfo = {};
				for rid, info in pairs(itemsT) do
					local id = info["ID"];
					info["ID"] = nil;
					dbinfo = info;
					local rinfo = RBDB:GetItemInfo(id, rid);
					if next(rinfo) ~= nil then 
						local minfo = RBDB:GetMaterialsInfo(id, rinfo["Skill"], rid);
						if minfo[1] ~= nil then minfo = minfo[1][2] end; -- Passing RID we know only one returned result.						
						-- RecipeBook:Debug("Adding result info for ".. rinfo["Name"].. " with ".. #info .." items matches. ");
						if indexT[rid] then 
							resultT[indexT[rid]][7] = info; -- Replace with updated information.
						else
							table.insert(resultT, {id, rid, rinfo["Name"], rinfo["Skill"], minfo, RBDB:GetKnownInfo(id, rinfo["Skill"], rid), info});
							indexT[rid] = #resultT;
						end
					end
				end
			end
		end
	end
	return resultT;
end

function RBSearch:ItemOnEnter()
	GameTooltip:SetOwner(RBUI_SearchFrame, "ANCHOR_BOTTOMRIGHT", -6, 448);
	local id, rid = unpack(RB_SearchResults[FauxScrollFrame_GetOffset(RBUI_SearchFrame_ResultsScrollFrame) + this:GetID()]);
	GameTooltip:SetHyperlink("enchant:"..rid);
end;

--[[ ItemOnClick(button) --> Displays search data on an item ]]--
function RBSearch:ItemOnClick(button)
	local id, rid, name, skill, flavorText, leftClick, rightClick = unpack(RB_SearchResults[FauxScrollFrame_GetOffset(RBUI_SearchFrame_ResultsScrollFrame) + this:GetID()]);
	if name == RECIPEBOOK_ERR_NOSEARCHRESULT then return end;
	if ( button == "LeftButton" ) then

		-- Alt+Lclick outputs materials data.
		if IsAltKeyDown() then
			local mlink = RBDB:Link_DBToText(id, rid, "gold");
			if ( ChatFrameEditBox:IsVisible() ) then
				ChatFrameEditBox:Insert(mlink);
			else
				local mats = RBDB:GetMaterialsInfo(id, skill, rid);
				if mats[1] ~= nil then mats = mats[1][2] end;
				local data = {};
			    for a, b in pairs(mats) do
			        tinsert(data, RBDB:Link_DBToText(a, 0, "mat").." x"..b);
				end
				RBOutput:Print(RECIPEBOOK_TXT_SUPPLIES..mlink..": "..table.concat(data, ", "), "output", nil, 0.25,0.5,1);
			end
		-- Otherwise, display the item tooltip.
		else
		    local link = RBDB:Link_DBToText(id, rid);
			if id == rid then
				id = "enchant:"..id;
			else id = RBOutput:MakeLink(id);
			end
			SetItemRef(id, link, button);
		end
	elseif ( button == "RightButton" ) then
		if RecipeBookUITooltip:IsVisible() then RecipeBookUITooltip:Hide();
		else
			RecipeBookUITooltip:ClearLines();
			RecipeBookUITooltip:SetOwner(this, "ANCHOR_CURSOR");
			RecipeBookUITooltip:AddLine(name, 1, 1, 0);
			for i, line in pairs(rightClick) do
				RecipeBookUITooltip:AddLine(line);
			end
			RecipeBookUITooltip:Show();
		end
	end
end

--[[ ScrollBarUpdate() --> Scrolls ]]--
function RBUI_SearchFrame_ScrollFrame_ScrollBarUpdate()
	local length = #RB_SearchResults;
	local line, index, button, text;

	local offset = FauxScrollFrame_GetOffset(RBUI_SearchFrame_ResultsScrollFrame);
	FauxScrollFrame_Update(RBUI_SearchFrame_ResultsScrollFrame,length+1,RBSKILLSEARCHFRAME_NUMITEMS,30);
	for line = 1,RBSKILLSEARCHFRAME_NUMITEMS,1 do
		index = offset + line;
		button = getglobal("RBUI_SearchFrame_ScrollFrameItem"..line);
		button:SetID(line);
		if index <= length then
			local _, _, name, _, flavorText = unpack(RB_SearchResults[index]);
			getglobal("RBUI_SearchFrame_ScrollFrameItem"..line.."_TXT-Item"):SetText(name);
			getglobal("RBUI_SearchFrame_ScrollFrameItem"..line.."_TXT-Flavor"):SetText(flavorText);
			button:Show();
			button:Enable();
			if name == RECIPEBOOK_ERR_NOSEARCHRESULT then button:Disable() end;
		else
		    button:Hide();
		end
	end
end
