--[[ RECIPE BOOK OPTIONS : 
Contains most of the functions used for dealing with options and options data 

]]--

--============================== DEFAULT OPTIONS TABLE ==============================--
local RECIPEBOOK_DEFAULT_OPTIONS = {
	["Status"] 	= true;
	["Output"]	= {
		["TT"] 	= true;
		["CF"]	= false;
		["Info"] = true;
	},
	["Display"]	= {
	    ["Me"]	= true; -- This player
	    ["ASF"]	= true; -- Alts same
	    ["AOF"]	= false;  -- Alts other
	    ["SSF"]	= true; -- Shared same
	    ["SOF"]	= false; -- Shared other
	    ["AK"]	= true; -- Alts know
	    ["AC"]	= true; -- Alts can learn
	    ["AW"]	= true; -- Alts will learn
 	    ["B"]	= true; -- Banked
	    ["SK"]	= true; -- Shared know
	    ["SC"]	= true; -- Shared can learn
	    ["SW"]	= true; -- Shared will learn
		["AM"] 	= true; -- Alts can make
		["SM"] 	= false; -- Shared can make
	},
	["TipC"]	= {
		["ASF"]			= "|cff00ff00";
		["AOF"]			= "|cff33ccff";
		["SSF"] 		= "|cffffff00";
		["SOF"]      	= "|cffff0000";
		["K"] 			= {1, 0, 0};
		["C"] 			= {0, 1, 0};
		["W"] 			= {0, 0.75, 0.75};
		["B"] 			= {1, 0.5, 0.1};
		["M"]			= {1, 1, 1};
	},
	["AucC"]	= {
	    ["Show"]      	= true;
		["AC"] 			= {0.1, 1, 0.1}; -- Alts can learn
		["MW"] 			= {1, 0.75, 0.1}; -- You will learn
		["AW"] 			= {0.1, 0.75, 1.0}; -- Alts will learn
		["OC"] 			= {1.0, 0.1, 0.1}; -- No alts can learn
		["AK"] 			= {0, 0, 1.0}; -- All alts know
		["B"] 			= {0.1, 0.1, 0.1}; -- Banked
	},
	["Bank"] = {
	    ["Show"] = true;
        ["Bank"] = true;
        ["Bags"] = true;
        ["Tell"] = true;
	},
	["Browse"] = {
		["Sort"] = "Alphabetical";
		["Unknown"] = false;
		["CleanData"] = true;
	},
	["TrackMe"] = true; -- track me!
	["TrackSkills"]	= {true, true, true, true, true, true, true, true, true, true, true, true, true}; -- default skill tracking
	["ShareGroups"] = {["_Guild"] = -1, ["_Friend"] = -1, ["_Default"] = -1}; -- Auto-accept status for friends, guildmates, and defaults
	["Updates"] = {
		["SendGuild"] = false; -- Send Guild update on login
		["Guild"] = true; -- Accept Guild channel broadcasts
		["Freq"] = 0; -- Send updates to your AutoUpdate list (valid: day = 1, week = 2, never = 0)
		["Accept"] = true;
	};
	["MM"] = {
		["Show"] = true,
		["Pos"] = 0
	}; -- Minimap button variables
};

--============================== VARIABLES ==============================--
RecipeBook_Defaults = {}; -- Saved globally
RecipeBook_Options = {}; -- Saved per character
RecipeBook_ShareOptions = {}; -- Names of persons with custom accept/decline status.  Saved globally, indexed per realm.
RBOptions = {};

-- Subframes for tabs
RBOPTIONSFRAME_SUBFRAMES = { "RBOptions_GeneralFrame", "RBOptions_SharingFrame", "RBOptions_AuctionFrame"};

--============================== FUNCTIONS ==============================--
function RBOptions:Initialize()
	if RecipeBook_ShareOptions[RecipeBook.Globals.Realm] == nil then RecipeBook_ShareOptions[RecipeBook.Globals.Realm] = {} end;
	-- reinitialize options which have been incorrectly set.
	for x,y in pairs(RecipeBook_Defaults) do
		if type(y) ~= "table" and type(RECIPEBOOK_DEFAULT_OPTIONS[x]) == "table" then RecipeBook_Defaults[x] = nil end;
	end
	for a,b in pairs(RecipeBook_Options) do
		if type(b) ~= "table" and type(RECIPEBOOK_DEFAULT_OPTIONS[a]) == "table" then RecipeBook_Options[a] = nil end;
	end
end

--============================== Option setting and access ==============================--
--[[ IsValid(o, so) --> Checks if an option exists
	Returns true | false ]]--
function RBOptions:IsValid(o,so)
	if RECIPEBOOK_DEFAULT_OPTIONS[o] ~= nil and (not so or RECIPEBOOK_DEFAULT_OPTIONS[o][so] ~= nil) then
		return true;
	else return false;
	end
end

--[[ GetOption(o, so) --> Returns the value of a given option ]]--
function RBOptions:GetOption(o, so)
	if not RBOptions:IsValid(o, so) then return nil end;
    if o and RecipeBook_Options[o] then
		if so then -- Suboption passed
			if RecipeBook_Options[o][so] ~= nil then
	    		return RecipeBook_Options[o][so];
			else
			    return RBOptions:DefaultOption(o, so);
			end
		else
		    return RecipeBook_Options[o];
		end
	else -- We know there's a default for the option.
		return RBOptions:DefaultOption(o, so)
	end
end

--[[ SetOption(o, [so,] value) --> Sets an option o /suboption so to value
	Returns: true | false if set ]]--
function RBOptions:SetOption(o, so, value)
	if value == nil then -- No suboption.
	    value = so;
	    so = nil;
	end
	if (not RBOptions:IsValid(o, so)) or (value == nil) then return false end; -- Option does not exist, or value is nil.
	local def = (value == RBOptions:DefaultOption(o, so) and true or false); -- Is this a reset to default option?
	if so then
	    if def then -- Reset to default.
			 -- RecipeBook:Debug("Resetting o: "..o);
			 -- RecipeBook:Debug("Resetting so: "..so);
	        if RecipeBook_Options[o] ~= nil and type(RecipeBook_Options[o] == "table") then
				if RecipeBook_Options[o][so] ~= nil then
					RecipeBook_Options[o][so] = nil;
				end
				if next(RecipeBook_Options[o]) == nil then RecipeBook_Options[o] = nil end;
			end
		else
			if RecipeBook_Options[o] == nil then RecipeBook_Options[o] = {} end;
			RecipeBook_Options[o][so] = value;
		end
	else-- Reset to default.
	    if def then RecipeBook_Options[o] = nil;
		else RecipeBook_Options[o] = value;
		end
	end
	if o == "MM" and so == "Show" then 
		if value then RBMM:ShowButton() else RBMM:HideButton() end;
	end;
	return true;
end

--[[ DefaultOption(o, [so]) --> Returns the default (accounting for customs) of an option ]]--
function RBOptions:DefaultOption(o, so)
	if not RBOptions:IsValid(o, so) then return nil end;
	
	if RecipeBook_Defaults[o] ~= nil then -- Custom default
	    if so then -- Suboption exists
	        if RecipeBook_Defaults[o][so] ~= nil then
				return RecipeBook_Defaults[o][so];
			else
			    return RECIPEBOOK_DEFAULT_OPTIONS[o][so];
			end
		else
		    return RecipeBook_Defaults[o];
		end
	elseif so then -- Standard default, suboption
	    return RECIPEBOOK_DEFAULT_OPTIONS[o][so];
	else -- Standard default, no suboption
	    return RECIPEBOOK_DEFAULT_OPTIONS[o];
	end
end

--[[ RevertOptions() --> Resets all options to defaults. ]]--
function RBOptions:RevertOptions()
	RecipeBook_Options = {};
end

--[[ SetDefaultOptions() --> Sets custom defaults. ]]--
function RBOptions:SetDefaultOptions(subset)
	if subset then
		local o, so, v;
		for _, data in pairs(subset) do
			o, so, v = unpack(data);
			if (v == nil) then v = so; so = nil; end; -- Only one option level.
			if RBOptions:IsValid(o, so) then
				if so then 
					if not RecipeBook_Defaults[o] then RecipeBook_Defaults[o] = {} end;
					RecipeBook_Defaults[o][so] = v;
				else
					RecipeBook_Defaults[o] = v;
				end
			end
		end
	else
		RecipeBook_Defaults = RecipeBook_Options;
		RBOptions:RevertOptions();
	end
end

--============================== Sharing Options ==============================--
--[[ GetShareStatus(who, [faction - defaults to current]) --> Returns whether a particular person has custom share options.
	RecipeBook_ShareOptions is indexed by "yy-Name", where yy is the faction.]]--
function RBOptions:GetShareStatus(who, fac)
	-- "_" denotes a group setting.
	if not (string.sub(who, 1, 1) == "_") then
		if not fac then fac = RecipeBook.Globals.Faction end;
	    who = RBDB:Faction_TextToDB(fac, true).."-"..who;
		return RecipeBook_ShareOptions[RecipeBook.Globals.Realm][who];
	else
	    return RBOptions:GetOption("ShareGroups", who);
	end
end

--[[ SetShareStatus(who, value) --> Sets the custom share option for a player in your faction:
	-1: prompt
	0: decline
	1: accept
	nil: return to default
	RecipeBook_ShareOptions is indexed by "yy-Name", where yy is the faction.]]--
function RBOptions:SetShareStatus(who, value)
	if not who then return end;
	if not (string.sub(who, 1, 1) == "_") then
		who = RBDB:Faction_TextToDB(RecipeBook.Globals.Faction, true).."-"..who;
		if value and type(value) ~= "number" then return end;
		RecipeBook_ShareOptions[RecipeBook.Globals.Realm][who] = value;
	else
	    RBOptions:SetOption("ShareGroups", who, value);
	end
end

--============================== Options Frame Functions ==============================--
--[[ ColorOptionText() --> Colorizing Auction Text samples ]]--
function RBOptions:ColorOptionText()
	RBOptions_FS_ColorAltsCanLearn:SetVertexColor(unpack(RBOptions:GetOption("AucC", "AC")));
	RBOptions_FS_ColorAltsWillLearn:SetVertexColor(unpack(RBOptions:GetOption("AucC", "AW")));
	RBOptions_FS_ColorYouWillLearn:SetVertexColor(unpack(RBOptions:GetOption("AucC", "MW")));
	RBOptions_FS_ColorNoAltsCanLearn:SetVertexColor(unpack(RBOptions:GetOption("AucC", "OC")));
	RBOptions_FS_ColorAllAltsKnow:SetVertexColor(unpack(RBOptions:GetOption("AucC", "AK")));
end