﻿--[[ RecipeBook Database functions
Functions in this file are for accessing the RecipeBook database in various and sundry ways. 
Static DB's: 
	RecipeBook_ItemDB (stores global data on tradeskill items)
	RecipeBook_CharacterDB (includes character tradeskills, banked items)
Volatile DB's (not saved to SavedVariables):
	RecipeBook_ItemNameDB (stores name to id conversions; note that names are kept as lowercase/no spaces/no special characters)
]]--

--======================= Globals =======================--
RBDB = {};
--======================================================================================--
--============================== RECIPEBOOK ITEM DATABASE ==============================--
--======================================================================================--
--[[ 
RecipeBook_ItemDB = {
	[Item ID] = {
		["Name"] = name -- The name of  created item
		["Enchant"] = name -- If this id also has an enchantment, the name of the enchantment (a la Runecloth Robe and Enchant Chest: Minor Mana)
		[Recipe ID] = { -- This is the Tradeskill: Item link with mats included a la enchanting.
			["Skill"] = skill, -- This is the skill which can craft this particular recipe
			["Name"] = name -- Sometimes a recipe produces a different name for an item (a la Mechanical Squirrel/Box and Festival Suit); for matching purposes this name is stored
			["Mats"] = {
				[Material ID] = number (number required),
				...
			},
			["Known"] = {
				[Server ID] = {
					[Data type] = { -- 10/20 = Alliance/Horde + 0/1 = Personal/Shared
						["Character"] = number (converted difficulty),
						...
					},
					...
				},
				... -- Other servers here
			},
		},
		... -- Other skills here.
	},
	... -- Other items here
};

]]--


-- --======================= Accessing Database Information =======================--
--[[ GetItemInfo(id, rid) --> Returns the subtable for the given recipe, pointer-safe.  Replaces "Name" with item name if not listed. ]]--
function RBDB:GetItemInfo(id, rid)
	if not id or not rid then 
		-- if not id then RecipeBook:Debug("No id provided to GetItemInfo.") end;
		-- if not rid then RecipeBook:Debug("No rID provided to GetItemInfo.") end;
		return {};
	end;
	local returnT = {};
	if RecipeBook_ItemDB[id] and RecipeBook_ItemDB[id][rid] then
		for k, v in pairs(RecipeBook_ItemDB[id][rid]) do
			if k == "Name" and not v then 
				if id == rid then
					v = RecipeBook_ItemDB[id]["Enchant"];
				else
					v = RecipeBook_ItemDB[id]["Name"];
				end
			end
			returnT[k]= v;
		end
	end
	return returnT;
end

--[[ GetMaterialsInfo(id, [skill], [rid]) --> materials list and current RecipeBook.Globals.Realm's known data for a given item ID/tradeskill.
	If only id is provided, returns ALL recipes that create this item, regardless of skill
	If id and skill are provided, or multiple skills are provided, returns ALL recipes within that skill which create this item.
	If id, skill, and recipe id are provided, returns the ONE set of data for that recipe (still needs to be referenced as returnvalue[1])
	Returns: indexed table containing: {rid, {matslist}} style data.
]]--
function RBDB:GetMaterialsInfo(id, skill, rid)
	local returnT = {};
	if not id then return returnT else id = tonumber(id) end;
	if ( (not skill) or (type(skill) == "table") ) then skill = 99 end; -- 99 will represent "all skills"; we will be returning a list of information.
	-- Does this item exist at all in the DB?
	if RecipeBook_ItemDB[id] == nil then
		RecipeBook:Debug("No such Item Known: "..id);
		return returnT;
	end
	-- RecipeBook:Debug("Getting info for item: "..id.. ", skill: "..skill);
	if type(skill) ~= "number" then skill = RB DB:Skill_TextToDB(skill) end;
	-- If you provide a specific recipe ID then we return data for that recipe alone.
	if rid and RecipeBook_ItemDB[id][rid] then
		if RecipeBook_ItemDB[id][rid]["Mats"] and next(RecipeBook_ItemDB[id][rid]["Mats"]) ~= nil then
			table.insert(returnT, {rid, RecipeBook_ItemDB[id][rid]["Mats"]});
		else
			RecipeBook:Debug("No Materials Data available for rid: "..rid);
			table.insert(returnT, {rid, {}});
		end
		return returnT;
	end
	-- No specific recipe id: return all item lists which create [item] within [skill] or all skills.
	for rid, data in pairs(RecipeBook_ItemDB[id]) do
		if type(rid) ~= "number" then
			-- Name/Enchant information; discard
		elseif ( (skill == 99) or (data["Skill"] == skill) ) then -- Recipe ID with a matching skill/returning all skills
			if data["Mats"] and next(data["Mats"]) ~= nil then
				table.insert(returnT, {rid, data["Mats"]});
			else
				RecipeBook:Debug("No Materials Data available for rid: "..rid);
				table.insert(returnT, {rid, {}});
			end
		end
	end
	
	return returnT;
end

--[[ GetKnownInfo(id, [skill], [rid]) --> Generates a table of who (by faction) can make the item in some fashion.
If provided a list of skills then returns data for all skills (accepts 99 to make this default).  If rid is provided, moves to a specific recipe only.
Returns a table, indexed by faction
fac = {
	alt = diff,
	alt = diff,
	},
fac2 = ...
]]--
function RBDB:GetKnownInfo(id, skill, rid)
	local returnT = {};
	-- I need an id.
	if not id then return returnT else id = tonumber(id) end;
	if skill == nil then skill = 99 end;
	-- Does this item exist at all in the DB?
	if RecipeBook_ItemDB[id] == nil then
		RecipeBook:Debug("No such Item Known: "..id);
		return returnT;
	end
	-- RecipeBook:Debug("Getting info for item: "..id.. ", skill: "..skill);
	if type(skill) == "string" then skill = RBDB:Skill_TextToDB(skill) end;
	-- If you provide a specific recipe ID then we return data for that recipe alone.
	if rid and RecipeBook_ItemDB[id][rid] then
		if RecipeBook_ItemDB[id][rid]["Known"] and RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm] ~= nil then
			returnT = RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm];
		else
			RecipeBook:Debug("GetKnownInfo: No Known Data on this server available for rid: "..rid);
		end
		return returnT;
	end
	-- No specific recipe id: return all item lists which create [item] within [skill] or all skills.
	for rid, data in pairs(RecipeBook_ItemDB[id]) do
		if type(rid) ~= "number" then
			-- Name/Enchant information; discard
		else
			local match = false;
			if type(skill) == "table" then 
				for x, s in ipairs(skill) do 
					if data["Skill"] == s then match = true; end;
				end
			elseif ( (skill == 99) or (data["Skill"] == skill) ) then match = true;
			end
			
			if match then -- Recipe ID with a matching skill/returning all skills
				if RecipeBook_ItemDB[id][rid]["Known"] and RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm] ~= nil then
					for fac, whos in pairs(RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm]) do
						if returnT[fac] == nil then returnT[fac] = {} end;
						for who, diff in pairs(whos) do
							returnT[fac][who] = diff;
						end
					end
				else
					RecipeBook:Debug("GetKnownInfo: No ID and no Known Data on this server available for rid: "..rid);
				end
			end
		end
	end
	return returnT;
end

--[[ Item_DBToText(id, rid) --> Returns the stored name for an item, or generates one if it does not exist. 
	Returns the item name and its alternate name.  If no alternate name, then returns the name twice.
]]--
function RBDB:Item_DBToText(id, rid)
	id, rid = tonumber(id), tonumber(rid);
	local iname, rname;
	-- Name: the item name.
	if id and RecipeBook_ItemDB[id] then
		if id == rid then iname = RecipeBook_ItemDB[id]["Enchant"];
		else iname = RecipeBook_ItemDB[id]["Name"];
		end;
	else
		if id == rid then iname = GetSpellInfo(id);
		else iname = GetItemInfo(id);
		end;
	end
	-- AltName: The alternate item name, generated from the recipe.
	if id and rid and RecipeBook_ItemDB[id] and RecipeBook_ItemDB[id][rid] then
		rname = RecipeBook_ItemDB[id][rid]["Name"];
		if not rname then rname = iname end;
	else
		rname = iname;
	end
	-- table.insert(RecipeBook_CharacterDB["Debug"], {id, rid, iname, rname});
	return iname, rname;
end

--[[ GetMaterialsID(id, skill) --> Returns a materials link for an item ]]--
function RBDB:GetMaterialsID(id, skill)
	id = tonumber(id);
	if type(skill) ~= "number" then skill = RBDB:Skill_TextToDB(skill) end;
	if RecipeBook_ItemDB[id] then
		for rid, data in RecipeBook_ItemDB[id] do
			if data["Skill"] == skill then return rid end;
		end
	else
		return 0;
	end
end

--[[ GetPlayerKnownItems(who, faction, skill, include all?) --> Returns a list of item id's the player knows in a particular tradeskill 
	If all is true, then returns a list of all items in the tradeskill.
	Returns {list} = matched id's. ]]--
function RBDB:GetPlayerSkillItems(who, fac, skill, all)
	if (skill == "all") or (skill == 99) then 
		local bigT = {};
		for n in 1, #RecipeBook.TSTable do
			table.insert(bigT, RBDB:GetPlayerSkillItems(who, nil, n, all)); -- Recurse through all tradeskills.
		end
		return bigT;
	end
	if type(skill) ~= "number" then skill = RBDB:Skill_TextToDB(skill) end;
	if not fac then fac = RBDB:GetPlayerFaction(who) end;
	local itemT = {};
	if not skill or skill == 0 or not fac then
	    -- RecipeBook:Debug("No skill/no player");
		return {};
	end; -- player doesn't exist on this server, or skill does not match
	
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill] then
	    local id, rid, data, info;
		for id, info in pairs(RecipeBook_ItemDB) do
			for rid, data in pairs(info) do 
				if data["Skill"] == skill then 
					local known = RBDB:GetKnownInfo(id, skill, rid);
					if known[fac] ~= nil and known[fac][who] ~= nil then
						itemT[rid] = {id, known[fac][who]}; -- rid = {id, difficulty};
					elseif all and itemT[rid] == nil then
						itemT[rid] = {id, -1}; -- rid = {id, unknown flag};
					end
				end
			end
		end
		return itemT;
	else
		return {}; -- Player doesn't know this skill.
	end
end

--[[ GetAllItems(boolean mats) --> Returns a list : names of all items in the DB.  If mats is true then returns names of all material items
	Returns two tables: 
	itemT { {id, rid, name,  skill, materials, known list } } 
	matsT { id = { name, { {id, rid}, {id, rid} } } }
]]--
function RBDB:GetAllItems(mats)
	local itemT = {};
	local matsT = {};
	
	for id, info in pairs(RecipeBook_ItemDB) do
		for rid, data in pairs(info) do
			if type(rid) == "number" then
				-- Generate Mats Table
				if mats then 
					for mid, _ in pairs(data["Mats"]) do
						if matsT[mid] and next(matsT[mid]) ~= nil then 
							-- matsT[id] = {name, { {id, rid}... } }
							table.insert(matsT[mid][2], {id, rid})
						else
							local mname = GetItemInfo(mid);
							if not mname then mname = "No Mat Name Match" end;
							matsT[mid] = {mname, {}};
							table.insert(matsT[mid][2], {id, rid})
						end
					end
				end
				-- Generate Item Table
				local skill = data["Skill"];
				local name = data["Name"];
				if not name then
					if id == rid then name = info["Enchant"];
					else name = info["Name"];
					end
				end
				local known = RBDB:GetKnownInfo(id, skill, rid);
				local minfo = RBDB:GetMaterialsInfo(id, skill, rid);
				if minfo[1] ~= nil then minfo = minfo[1][2] end; -- Pare down to mats table alone; we know the RID.
				table.insert(itemT, {id, rid, name, skill, minfo, known})
			end
		end
	end
	return itemT, matsT;
end

function RBDB:IsEnchant(skill)
	if skill == RBDB:Skill_TextToDB(RECIPEBOOK_TRADESKILLS["Enchanting"]) then return true;
	else return false;
	end
end

function RBDB:IsValidWho(who, fac)
	if not who or not fac then return false end;
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac] and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who] then return true;
	else return false;
	end;
end
--======================= Altering Database Information =======================--
--[[ AddDBItem(id, rid, name, altname, skill) --> Adds an item to the database, checking for duplication. ]]--
function RBDB:AddDBItem(id, rid, name, altname, skill)
	-- Assumes we are being given numerical id, rid, skill.
	if (not skill) or (not id) or (not rid) or (skill * id * rid == 0) then return false end; -- Bad data.
	if not name then return false end; -- We need a name.
	if not altname then altname = name end;
	
	RecipeBook:Debug("Adding DB Item: "..id);
	
	-- Totally new?  Let's prepare.
	if RecipeBook_ItemDB[id] == nil then RecipeBook_ItemDB[id] = {} end;
	-- Does the item exist in the database?
	if RecipeBook_ItemDB[id] ~= nil and RecipeBook_ItemDB[id][rid] ~= nil then
		if id == rid then 
			RecipeBook_ItemDB[id]["Enchant"] = name; -- Updated data for an enchantment
		else
			RecipeBook_ItemDB[id]["Name"] = name; -- Updated data for an item.
		end
		if name ~= altname then
			RecipeBook_ItemDB[id][rid]["Name"] = altname;
		else
			RecipeBook_ItemDB[id][rid]["Name"] = false;
		end
		return false; -- Data exists for that reccipe.
	-- Data exist but not for this recipe ID, or it is a new add.
	else
		if id == rid then 
			RecipeBook_ItemDB[id]["Enchant"] = name; -- Updated data for an enchantment
		else
			RecipeBook_ItemDB[id]["Name"] = name; -- Updated data for an item.
		end
		RecipeBook_ItemDB[id][rid] = {};
		RecipeBook_ItemDB[id][rid]["Skill"] = skill;
		RecipeBook_ItemDB[id][rid]["Mats"] = {}; -- Materials are added separately.
		if name ~= altname then -- Add alternate item name
			RecipeBook_ItemDB[id][rid]["Name"] = altname;
		else
			RecipeBook_ItemDB[id][rid]["Name"] = false;
		end
		RecipeBook_ItemDB[id][rid]["Known"] = {};
		return true;
	end
end

--[[ AddMaterialsData(id, rid, {materials list}) --> Checks that the item exists and then applies materials data to it ]]--
function RBDB:AddMaterialsData(id, rid, mlist)
	-- Assumes we are sending numerical id and rid
	if not id or not rid then return false end;
	-- Check for valid site
	if RecipeBook_ItemDB[id] == nil or RecipeBook_ItemDB[id][rid] == nil then
		return false;
	else
		local tmp = {};
		for k,v in pairs(mlist) do
			k = tonumber(k);
			v = tonumber(v);
			if (not k) or (not v) or (k * v == 0) then return false end; -- Something's got the wrong data info.  Scrap the whole list.
			tmp[k] = v;
		end
		if next(tmp) ~= nil then 
			RecipeBook_ItemDB[id][rid]["Mats"] = tmp;
			return true;
		else
			return false;
		end;
	end
end

--[[ SetMetaData(id, rid, index, metatable) --> Adds meta data to a recipe.  This function is designed to be called from an outside mod which uses string 'index' to identify its metadata.  
	.Meta is the field for outside mods to attach data to RecipeBook's stored items.  It is indexed .Meta.index = data.
	Please give your data a sensible index.]]--
function RBDB:SetMetaData(id, rid, index, data)
	if RecipeBook_ItemDB[id] == nil or RecipeBook_ItemDB[id][rid] == nil then return false end;
	if RecipeBook_ItemDB[id][rid]["Meta"] == nil then RecipeBook_ItemDB[id][rid]["Meta"] = {} end;
	if data == {} then data = nil end; -- Setting to an empty table makes metadata go away.
	RecipeBook_ItemDB[id][rid]["Meta"][index] = data;
	return true;
end

--[[ GetMetaData(id, rid, index,) --> Retrieves meta data for a recipe, pointer-safe.  This function is designed to be called from an outside mod which uses string 'index' to identify its metadata.  
	.Meta is the field for outside mods to attach data to RecipeBook's stored items.  It is indexed .Meta.index = data.
	It is permissible to attempt to access other mods' metadata (if they are storing something you want)]]--
function RBDB:GetMetaData(id, rid, index)
	local returnT = RBDB:GetItemInfo(id, rid);
	if returnT["Meta"] == nil or returnT["Meta"][index] == nil then return {};
	else
		return returnT["Meta"][index];
	end
end

--[[ AppendMetaData(id, rid, index, key, value) --> Appends meta data to a metatable.  
	If 'key' is a table, iterates through that table and calls AppendMetaData on each key, value pair.  This can be recursed.
	If .Meta.index == nil then creates the Meta field.
	.Meta is the field for outside mods to attach data to RecipeBook's stored items.  It is indexed .Meta.index = data.
	Please give your data a sensible index.]]--
function RBDB:AppendMetaData(id, rid, index, key, value)
	if RecipeBook_ItemDB[id] == nil or RecipeBook_ItemDB[id][rid] == nil then return false end;
	if RecipeBook_ItemDB[id][rid]["Meta"] == nil then RecipeBook_ItemDB[id][rid]["Meta"] = {} end;
	if RecipeBook_ItemDB[id][rid]["Meta"][index] == nil then RecipeBook_ItemDB[id][rid]["Meta"][index] = {} end;
	if type(key) == "table" then
		for k, v in pairs(key) do RBDB:AppendMetaData(id, rid, index, k, v) end;
	else
		RecipeBook_ItemDB[id][rid]["Meta"][index][key] = value;
	end
	return true;
end

--[[ DeleteMetaData(id, rid, index, key) --> Deletes meta data from a metatable, returning the value of the deleted key.  
	if 'key' does not exist then returns nil.
	.Meta is the field for outside mods to attach data to RecipeBook's stored items.  It is indexed .Meta.index = data.
	Please give your data a sensible index.]]--
function RBDB:DeleteMetaData(id, rid, index, key)
	if RecipeBook_ItemDB[id] == nil or RecipeBook_ItemDB[id][rid] == nil then return nil end;
	if RecipeBook_ItemDB[id][rid]["Meta"] == nil or RecipeBook_ItemDB[id][rid]["Meta"][index] == nil then return nil end;
	local v;
	if type(key) == "table" then 
		v = {};
		for i, k in ipairs(key) do table.insert(v, RBDB:DeleteMetaData(id, rid, index, k)) end;
	else
		v = RecipeBook_ItemDB[id][rid]["Meta"][index][key];	
		RecipeBook_ItemDB[id][rid]["Meta"][index][key] = nil;
	end
	return v;
end

--[[ AddKnownData(id, tradeskill, player, difficulty, faction code) --> Checks that the item exists and then adds the passed player to it. ]]--
function RBDB:AddKnownData(id, rid, who, diff, fac)
	-- Assuming we have numerical id, rid; do a check on diff.
	if type(diff) ~= "number" then diff = RBDB:Difficulty_TextToDB(diff) end;
	fac = tonumber(fac);
	if (diff == nil) or (not id) or (not fac) or (not rid) then RecipeBook:Debug("missing diff/id/fac/rid"); return false end;
	if not who then return false end;
	if RecipeBook_ItemDB[id] == nil or RecipeBook_ItemDB[id][rid] == nil then return false end;
	
	RecipeBook:Debug("Adding Known Data");
	
	-- If we already know this, just replace the item.
	if RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm] and RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac] and RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac][who] then
		RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac][who] = diff;
		return true;
	else
		if RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm] == nil then RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm] = {} end;
		if RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac] == nil then RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac] = {} end;
		RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac][who] = diff;
		return true;
	end
	return false;
end

--[[ DeleteKnownData(player, id, rid, faction code) --> Checks that the item exists and then deletes the passed player from it. ]]--
function RBDB:DeleteKnownData(who, id, rid, fac)	
	-- Assume numerical id, rid, fac
	if (not id) or (not fac) or (not rid) then return false end;
	if not who then return false end;

	 if RecipeBook_ItemDB[id] ~= nil and RecipeBook_ItemDB[id][rid] ~= nil then
		-- If who is listed for this recipe then delete who
	 	if RecipeBook_ItemDB[id][rid]["Known"] and RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm] and RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac] then
		 	RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac][who] = nil;
			-- If faction is now empty then delete faction.
			if next(RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac]) == nil then
			    RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm][fac] = nil;
				-- If Realm is now empty then delete realm.
				if next(RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm]) == nil then
			        RecipeBook_ItemDB[id][rid]["Known"][RecipeBook.Globals.Realm] = nil;
					-- If Known is now empty and we are keeping our data clean then delete the recipe.
			        if RBOptions:GetOption("Browse", "CleanData") and next(RecipeBook_ItemDB[id][rid]["Known"]) == nil then
			            RecipeBook_ItemDB[id][rid] = nil;
						-- If there are no more recipes known for this item, delete the item.
						local valid = false;
						for x, y in pairs(RecipeBook_ItemDB[id]) do
							if type(x) == "number" then valid = true end;
						end
						if not valid then 
				            RecipeBook_ItemDB[id] = nil;
						end
					end
				end
			end
		end
		return true;
	end
	return false;
end

function RBDB:CleanDB() 
	if not RBOptions:GetOption("Browse", "CleanData") then return end; -- CleanData flags for the DB to be scrubbed.
	for id, info in pairs(RecipeBook_ItemDB) do
		for rid, data in pairs(info) do
			if type(rid) == "number" and next(data["Known"]) == nil then
				RecipeBook_ItemDB[id][rid] = nil;
				local valid = false;
				for x, y in pairs(RecipeBook_ItemDB[id]) do
					if type(x) == "number" then valid = true end;
				end
				if not valid then 
					RecipeBook_ItemDB[id] = nil;
				end
			end
		end
	end
end
--===========================================================================================--
--============================== RECIPEBOOK CHARACTER DATABASE ==============================--
--===========================================================================================--

--[[
RecipeBook_CharacterDB = {
	[Server ID] = { -- Indexed in RecipeBook_Indices
		[Data Type] = { -- 10/20 = Alliance/Horde + 0/1 = Personal/Shared
			["Character"] = {
				[1] = nil / {rank, maxrank, spec, subspec, [timestamp]}
				...
				[12] = nil / {rank, maxrank, spec, subspec, [timestamp]}
				["Rep"] = {faction_id = 3, ...}
				["Track"] = {custom tracking options};
				["AutoUpdate"] = {{who, time}...};
			},
			... -- Other characters here
		},
		... -- Other data types on this server here
		["Unsafe Items"] = {
			[2045] = true;
			...  -- Other unsafe data for this server here.
		},
		["Pending Mail"] = {
		    [item link] = who to
		};
		["Last Processed"] = false; -- false/last ID processed.
		["Held Items"] = {
		    [Player Name] = {
				[Item ID] = {
				    "Skill" = tradeskill
					"Rank" = skill level required to use the recipe
					"Bank" = true if item is held in bank
				},
				...
			};
			... -- More banking characters here
		};
		["Bags Pending"] = {
			["who"] = "Name"; 
			[1] = [Item ID or hyperlink];
			... -- More items here
		}
	},
	... -- Other servers here
}
]]--

--======================= Accessing Database Information =======================--
--[[ GetPlayerTradeskillInfo(who, skill, [key]) --> returns the tradeskill data for a character.  "all" as skill will return all tradeskill data for that player
	For each skill returns information as it is stored in the tradeskill data table.
	For "All" returns skillT[1] = {skill 1}, etc.
]]--
function RBDB:GetPlayerTradeskillInfo(who, fac, skill, key)
	who = RBOutput:Capitalize(who);
	if not fac or not who or not skill then return {} end;
	-- all tradeskills
	if skill == "all" or skill == 99 then 
		if RBDB:IsValidWho(who, fac) then
			local skillT = {};
		    for data, _ in pairs(RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]) do
		        if type(data) == "number" then skillT[data] = RBDB:GetPlayerTradeskillInfo(who, fac, data) end;
			end
			return skillT;
		else
			return {};
		end
	end
	-- individual skills
	if type(skill) ~= "number" then skill = RBDB:Skill_TextToDB(skill) end;
	
	if RBDB:IsValidWho(who, fac) and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill] then
		local data = RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill];
		local dataT = {};
		for k, v in pairs(RECIPEBOOK_TRADESKILL_DEFAULTS) do
		    if key and key == k then
				if data[v[1]] ~= nil then return data[v[1]];
				else return v[3];
				end;
		    elseif not key then
			    if data[v[1]] ~= nil then dataT[v[1]] = data[v[1]];
			    else dataT[v[1]] = v[3];
			    end;
			end
		end;
		return dataT;
	else
		return {};
	end
end

--[[ GetPlayerSpecializations(who, fac, skill) --> accesses the specialization information for a given player.
	Returns: spec, subspec numbers (0 is none) ]]--
function RBDB:GetPlayerSpecializations(who, fac, skill)
	if not who or not fac or not skill then return 0,0 end;
	
	if RBDB:IsValidWho(who, fac) then
	    if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill] then
	        return RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill][RBDB:TradeskillIndex_TextToDB("spec")], RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill][RBDB:TradeskillIndex_TextToDB("subspec")];
		else
		    return 0,0;
		end
	else
		return 0,0;
	end
end

--[[ GetPlayerFaction(who) --> returns the faction number for a given player, 0 if not found. ]]--
function RBDB:GetPlayerFaction(who)
 	for i,d in pairs(RecipeBook_CharacterDB[RecipeBook.Globals.Realm]) do
 	    if type(i) == "number" then
			for k, _ in pairs(d) do
				if k == who then return i end;
			end
		end
	end
	return 0;
end

--[[ GetPlayerReputation(who,  fac, faction) ]]--
function RBDB:GetPlayerReputation(who, fac, rep)
	if type(rep) ~= "number" then rep = RBDB:Rep_TextToDB(rep) end;
	if RBDB:IsValidWho(who, fac) and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"] ~= nil and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"][rep] ~= nil then
		return RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"][rep];
	else return 0;
	end
end

--[[ GetSkillIsTracked(who, fac, skill) --> returns whether or not to track a particular skill ]]--
function RBDB:GetSkillIsTracked(who, fac, skill)
	if type(skill) ~= "number" then skill = RBDB:Skill_TextToDB(skill) end;
	if RBDB:IsValidWho(who, fac) and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Track"] ~= nil and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Track"][skill] ~= nil then
	    return RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Track"][skill];
	else
	    return RBOptions:GetOption("TrackSkills", skill);
	end
end

--[[ GetUpdatesTo(who) --> returns all alts currently AutoUpdating to the given who]]--
function RBDB:GetUpdatesTo(who, fac)
	local nameT = {};
	if not RBDB:IsValidWho(who, fac) then return {} end;
	for _, name in pairs(RBDB:GetAllInFaction(fac)) do
		if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][name]["AutoUpdate"] and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][name]["AutoUpdate"][who] ~= nil then
			table.insert(nameT, name);
		end
	end
	return nameT;
end

--[[ GetUpdatesFrom(alt) --> Returns the list of all whos currently receiving AutoUpdates from the given alt, and last updated. ]]--
function RBDB:GetUpdatesFrom(who, fac)
	if not RBDB:IsValidWho(who, fac) then return {} end;
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["AutoUpdate"] ~= nil then 
		return RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["AutoUpdate"];
	else
		return {};
	end
end

--[[ AltUpdatesTo(alt, who) --> returns true if the given alt is set to autoupdate to the given who ]]--
function RBDB:AltUpdatesTo(alt, fac, who)
	local nameT = RBDB:GetUpdatesFrom(alt, fac);
	if nameT[who] ~= nil then return true end;
	return false;
end

--[[ GetSkillIsShown(who, skill) --> returns whether or not to show a particular skill ]]--
function RBDB:GetSkillIsShown(who, fac, skill)
	-- Assumes numerical skill,
	if not RBDB:IsValidWho(who, fac) then return false end;
	local show = RBDB:GetPlayerTradeskillInfo(who, fac, skill, "show");
	local num = RBDB:GetPlayerTradeskillInfo(who, fac, skill, "numskills");
	
	if (math.fmod(fac, 10) == 0) and (num == 0) then return false; --Incomplete skill.
	elseif show ~= nil then return show;
	else return true; -- Default to displaying
	end
end

--[[ GetItemIsBanked(item) --> Returns the info for who the item is banked by, false if none ]]--
function RBDB:GetItemIsBanked(id)
	local rlist = {};
	for who, items in pairs(RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"]) do
	    if items[id] then
			local fac = RBDB:GetPlayerFaction(who);
			if fac and (math.fmod(fac, 10) > 0 and ( RBOptions:GetOption("Display", "SSF") or RBOptions:GetOption("Display", "S"))) then
				if (((RecipeBook.Globals.Faction == FACTION_HORDE and fac/10 > 1) or fac/10 < 2) and RBOptions:GetOption("Display", "SSF")) or RBOptions:GetOption("Display", "SOF") then -- OK to display
					table.insert(rlist, RBOutput:PlayerNameColor(fac)..who..RB_HEXCOLOR["end"]);
				end
			elseif fac and (((RecipeBook.Globals.Faction == FACTION_HORDE and fac/10 > 1) or fac/10 < 2) and RBOptions:GetOption("Display", "ASF")) or RBOptions:GetOption("Display", "AOF") then -- Same faction as player
				table.insert(rlist, RBOutput:PlayerNameColor(fac)..who..RB_HEXCOLOR["end"]);
			end
		end
	end
	
	if next(rlist) ~= nil then 
		return rlist;
	else 
		return false; 
	end
end

--[[ GetBankedForSkill(skill, rank) --> Returns info regarding any banked recipes which have a skill requirement of rank exactly x
	Returns: table of banked item ID's + who has it banked or false ]]--
function RBDB:GetBankedForSkill(skill, rank)
	if not skill or skill < 1 then return {} end;
	
	local bankT = {};
	for who, items in pairs(RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"]) do
		for item, data in pairs(items) do
			if not bankT[item] and data.Skill == skill and data.Rank == rank then
			    bankT[item] = RBDB:GetItemIsBanked(item);
			end
		end
	end
	
	if next(bankT) then return bankT else return false end;
end

--[[ GetAllCharacters(skill) --> Returns a colorized list of all displayable characters who have a skill and their current data
	Returns: {name = {color, {skill table}}, name = {color, {skill table}}]]--
function RBDB:GetAllCharacters(skill)
	if not skill or skill < 1 then return {} end;
	local nameT = {};
    local sfac, ofac = RBDB:Faction_TextToDB(RecipeBook.Globals.Faction, false), RBDB:Faction_TextToDB(RecipeBook.Globals.OFaction, false)
	local asf, ssf = RBOptions:GetOption("Display", "ASF"), RBOptions:GetOption("Display", "SSF");
	local aof, sof = RBOptions:GetOption("Display", "AOF"), RBOptions:GetOption("Display", "SOF");
	local shared = ssf or sof;
	local display = false;

	for fac,d in pairs(RecipeBook_CharacterDB[RecipeBook.Globals.Realm]) do
	    if type(fac) == 'number' then
    		display = false;
			if math.fmod(fac, 10) > 0 then
			    if shared then
					if fac == (ofac + 1) and sof then display = true; -- Shared other faction
					elseif fac == (sfac + 1) and ssf then display = true; -- Shared same faction
					end
				end
			elseif fac == ofac and aof then display = true; -- Alts other faction
			elseif fac == sfac and asf then display = true; -- Alts same faction
			end
			if display then
				local color = RBOutput:PlayerNameColor(fac);
				for p, _ in pairs(d) do
					if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][p][skill] then -- p has the skill
						nameT[p] = {color, RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][p][skill]};
						-- Reputation data goes later.
					end
				end
			end
		end
	end
	return nameT;
end

--[[ GetAllInFaction(faction) --> Returns a list of all the player names in a faction ]]--
function RBDB:GetAllInFaction(fac, share)
	if type(fac) ~= "number" then fac = RBDB:Faction_TextToDB(fac, share) end  -- Presuming you are passing a share value if such is needed. 
	local nameT = {};
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm] and RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac] then
		for who, _ in pairs(RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac]) do
		    tinsert(nameT, who);
		end
	end
	return nameT;
end

--======================= Altering Database Information =======================--
--[[ AddNewPlayer(who, faction, shared) --> Adds a brand new player to the database, with all default values ]]--
function RBDB:AddNewPlayer(who, fac, share)
	if type(fac) == "string" then fac = RBDB:Faction_TextToDB(fac,share) end;
	if not fac then return false end;
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm] == nil then
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm] = {["Unsafe Items"] = {}, ["Last Processed"] = false, ["Held Items"] = {}, ["Bags Pending"] = {}};
	elseif RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"] == nil then
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"] = {};
	end
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac] == nil then RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac] = {} end;
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who] ~= nil then
	    if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"] == nil then RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"] = {} end;
	    if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Track"] == nil then RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Track"] = {} end;
		if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["AutoUpdate"] == nil then RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["AutoUpdate"] = {} end;
--	    if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Show"] == nil then RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Show"] = {} end;
		RBTradeskill:ProfessionScan();
		return false;
	end -- Player already exists; update tradeskills.
	RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who] = {["Rep"] = {}, ["Track"] = {}, ["AutoUpdate"] = {} };
	RBTradeskill:ProfessionScan(); -- Update tradeskills :)
	RBTradeskill:ReputationScan(); -- Update reputations :)
	return true;
end

--[[ UpdateTrackedSkills(who) --> Updates which skills are being tracked based on what options are set ]]--
function RBDB:UpdateTrackedSkills(who, fac)
	if not RBDB:IsValidWho(who, fac) then return end;
	for skill, _ in pairs(RBOptions:GetOption("TrackSkills")) do
		RecipeBook:Debug("Running Check for skill: "..skill);
	    if not RBDB:GetSkillIsTracked(who, fac, skill) then RecipeBook:Debug("Deleting: "..skill); RBDB:DeleteTradeskill(who, fac, skill) end;
	end
	RBTradeskill:ProfessionScan();
	if RBUI_SkillFrame:IsVisible() then RBSkill:Refresh() end;
end

--[[ SetSkillIsTracked(who, skill, value) --> sets whether or not to track a particular skill ]]--
function RBDB:SetSkillIsTracked(who, fac, skill, value)
	if not RBDB:IsValidWho(who, fac) then return end;
	-- If same as default value then set to nil.
	if value == RBOptions:GetOption("TrackSkills", skill) then value = nil end;
	-- Otherwise set value.
	RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Track"][skill] = value;
end


--[[ SetAutoUpdate(who, when, alt) --> sets AutoUpdate data up.  Who is the person being updated to; when is the last update time (0 for initial), optional alt is if you are setting someone else's AutoUpdate.]]--
function RBDB:SetAutoUpdateData(who, when, alt)
	if who == RecipeBook.Globals.Player then return end; -- Don't AutoUpdate to yourself.
	if not alt then alt = RecipeBook.Globals.Player end;
	local fac = RBDB:GetPlayerFaction(alt);
	if not RBDB:IsValidWho(alt, fac) then return end;
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][alt]["AutoUpdate"] == nil then RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][alt]["AutoUpdate"] = {} end;
	RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][alt]["AutoUpdate"][who] = when;
end

--[[ SetSkillIsShown(who, skill, value) --> sets whether or not to show a particular skill (false = deprecate) ]]--
function RBDB:SetSkillIsShown(who, fac, skill, value)
	if not RBDB:IsValidWho(who, fac) then return end;
	RBDB:SetPlayerTradeskillInfo(who, fac, skill, "show", value);
end


--[[ SetPlayerTradeskillInfo(who, skill, key, value) --> Sets a tradeskill data item for a player ]]--
function RBDB:SetPlayerTradeskillInfo(who, fac, skill, key, value)
	if not RBDB:IsValidWho(who, fac) then return false end;
	if not skill or skill < 1 then return false end; -- Guaranteed that who exists now.
	
	-- Brand new skill info?  Set up a default table.
	if not RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill] then
	    RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill] = {};
		for i,v in pairs(RECIPEBOOK_TRADESKILL_DEFAULTS) do
			RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill][v[1]] = v[3];
		end
	end
	
	if RECIPEBOOK_TRADESKILL_DEFAULTS[key] ~= nil then
		-- rudimentary type checking at least, since that much I know should be consistent.
		if RECIPEBOOK_TRADESKILL_DEFAULTS[key][2] == type(value) then
			RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill][RECIPEBOOK_TRADESKILL_DEFAULTS[key][1]] = value;
		end
	end
	
	return true;	
end

--[[ SetPlayerReputation(who, faction, value) --> Sets the reputation data for a specific faction. ]]--
function RBDB:SetPlayerReputation(who, fac, reput, value)
	if not RBDB:IsValidWho(who, fac) then return end;
	if type(reput) ~= "number" then reput = RBDB:Rep_TextToDB(reput) end;
	value = tonumber(value);
	-- No recipe requires less than Neutral.
	if value < 4 then return end;
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"] ~= nil then
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"][reput] = value;
	else 
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who]["Rep"] = {[reput] = value};
	end
end

--[[ DeletePlayer(who) --> Removes all data for a particular player from the database ]]--
function RBDB:DeletePlayer(who, fac)
	RecipeBook:Debug("Running delete for "..who);
	if player == who then return false end;
	if not RBDB:IsValidWho(who, fac) then return false end;

	-- known items
	for id, data in pairs(RecipeBook_ItemDB) do
	    for rid, info in pairs(data) do
  			RBDB:DeleteKnownData(who, id, rid, fac);
		end
	end
	-- banked items
	RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who] = nil;
	-- tradeskill data
	RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who] = nil;

	return true;
end

--[[ DeleteTradeskill(who, faction, skill) --> Unlearns/removes a particular tradeskill from the database ]]--
function RBDB:DeleteTradeskill(who, fac, skill)
	if not RBDB:IsValidWho(who, fac) then return false end;
	if skill < 1 then return false end;
	-- Iterate through and remove the player's known information from each item; remove item if no known.
	for rid, data in pairs(RBDB:GetPlayerSkillItems(who, fac, skill, false)) do
	    RBDB:DeleteKnownData(who, data[1], rid, fac); 
	end
	RecipeBook_CharacterDB[RecipeBook.Globals.Realm][fac][who][skill] = nil; -- Blank character data.
	RBTradeskill:ForceSkillUpdate(skill);
	return true;
end

--[[ AddHeldItem(id, who, [item in bank]) --> Adds/alters a banked item; returns nil.  Item in bank overrides item not in bank. ]]--
function RBDB:AddHeldItem(id, who, bank)
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"] == nil then RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"] = {} end;
	if RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who] == nil then
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who] = {};
	end
	local valid, skill = RecipeBook:ParseItemLink(id);
	if valid and RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who][id] == nil then
	    RecipeBookTooltip:SetHyperlink(RBOutput:MakeLink(id));
		if type(skill) ~= "number" then skill = RBDB:Skill_TextToDB(skill) end;
		rank = RecipeBook:GetSkillInfo(RecipeBookTooltip, skill);
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who][id] = {Skill = skill, Rank = rank, Bank = bank};
		local _, what = GetItemInfo(id);
		if what and RBOptions:GetOption("Bank", "Tell") then
			RBOutput:Print(RECIPEBOOK_AUTOHELD..what, "info");
		end
	elseif bank and not RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who][id].Bank then
	    RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who][id].Bank = bank;
	end
end

--[[ RemoveHeldItem(id, who) --> Removes a banked item.]]--
function RBDB:RemoveHeldItem(id, who)
	if who and RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who] and RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who][id] then
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Held Items"][who][id] = nil;
		local _, what = GetItemInfo(id);
		if what and RBOptions:GetOption("Bank", "Tell") then
			RBOutput:Print(string.format(RECIPEBOOK_AUTOUNHELDSUCCEED, who, what), "info");
		end
	else
		local _, what = GetItemInfo(id);
		RBOutput:Print(string.format(RECIPEBOOK_AUTOUNHELDFAIL, who, what), "info");
	end
end

--===========================================================================================--
--============================== DATABASE CONVERSION FUNCTIONS ==============================--
--===========================================================================================--

--======================= Converting Information to/from Database Format =======================--
--[[ Difficulty_TextToDB(diff) --> converts string diff to numeric (1-4) value of the difficulty; returns 0 if no match. ]]--
function RBDB:Difficulty_TextToDB(diff)
	diff = string.lower(diff);
	for i,v in ipairs(RECIPEBOOK_DIFFICULTY) do 
		if string.lower(v) == diff then return i - 2 end; -- -1-0-1-2-3-4
	end
	return 0; -- no match; must be the default.
end

--[[ Difficulty_DBToText(diff) --> converts numeric index (0-4) diff to string value of the difficulty. ]]--
function RBDB:Difficulty_DBToText(diff)
	diff = tonumber(diff) + 2;
	return RECIPEBOOK_DIFFICULTY[diff] ~= nil and RECIPEBOOK_DIFFICULTY[diff] or RECIPEBOOK_DBERROR_NODIFFID;
end

--[[ Difficulty_DBToRGB(diff) --> returns RGB value of the difficulty. ]]--
function RBDB:Difficulty_DBToRGB(diff)
	if type(diff) ~= "number" then diff = RBDB:Difficulty_TextToDB(diff) end;
	diff = diff + 2; --0 is default color.
	if RB_DIFFCOLOR[diff] then
		return RB_DIFFCOLOR[diff]["r"], RB_DIFFCOLOR[diff]["g"], RB_DIFFCOLOR[diff]["b"];
	else -- default
		return RB_DIFFCOLOR[1]["r"], RB_DIFFCOLOR[1]["g"], RB_DIFFCOLOR[1]["b"];
	end
end

--[[ Skill_TextToDB(skill) --> converts localized tradeskill name to numeric tradeskill value ]]--
function RBDB:Skill_TextToDB(skill)
	if not skill then return 0 end;
	if type(skill) == "table" then 
		local skillT = {};
		for x, s in ipairs(skill) do
			table.insert(skillT, RBDB:Skill_TextToDB(s));
		end
		return skillT;
	else
		skill = string.lower(skill);
		for i,v in ipairs(RecipeBook.TSTable) do
			if string.lower(v) == skill or string.lower(string.gsub(v, "%s", "")) == skill then return i end;
		end
	end
	return 0;	
end

--[[ Skill_DBToText(skill) --> converts numeric tradeskill to localized tradeskill name]]--
function RBDB:Skill_DBToText(skill)
	if type(skill) == "table" then 
		local skillT = {};
		for x, s in ipairs(skill) do
			table.insert(skillT, RBDB:Skill_TextToDB(s));
		end
		return table.concat(skillT, "/");
	else
		skill = tonumber(skill);
		if skill == 99 then return RECIPEBOOK_TRADESKILLS["Enchanting"] end;
		if RecipeBook.TSTable[skill] ~= nil then 
			return RecipeBook.TSTable[skill];
		else
			return RECIPEBOOK_DBERROR_NOSKILLID;
		end
	end
end

--[[ Skill_DelocalizeText(skill) --> converts localized tradeskill name to English tradeskill name ]]--
function RBDB:Skill_DelocalizeText(skill)
	if not skill then return "No Tradeskill" end;
	skill = string.lower(skill);
	for i,v in pairs(RECIPEBOOK_TRADESKILLS) do
		if string.lower(v) == skill or string.lower(string.gsub(v, "%s", "")) == skill then return i end;
	end
	return "No Tradeskill";
end


--[[ Specialty_TextToDB(skill, spec) --> converts text info on specialties to the appropriate numeric values. 
	Returns: number spec, boolean subspec (if it's a subspecialty) ]]--
function RBDB:Specialty_TextToDB(skill, spec)
	if type(skill) == "number" then skill = RBDB:Skill_DBToText(skill) end;
	if spec == nil then return false, false end;
	if RECIPEBOOK_SPECIALS[skill] then 
		for i,v in ipairs(RECIPEBOOK_SPECIALS[skill]) do
			if v == spec then return i, false end; -- specialty only
		end
	end
	if RECIPEBOOK_SUBSPECIALS[skill] ~= nil then 
		for i,v in ipairs(RECIPEBOOK_SUBSPECIALS[skill]) do
			if v == subspec then return i, true end;
		end
	end
	return false, false;
end

--[[ Specialty_DBToText(skill, spec, boolean subspec) --> converts numeric info on specialties to the appropriate text strings. ]]--
function RBDB:Specialty_DBToText(skill, spec, subspec)
	if type(skill) == "number" then skill = RBDB:Skill_DBToText(skill) end;
	
	if spec and tonumber(spec) then
		spec = tonumber(spec);
		if not subspec and RECIPEBOOK_SPECIALS[skill] and RECIPEBOOK_SPECIALS[skill][spec] then
			spec = RECIPEBOOK_SPECIALS[skill][spec];
		elseif subspec and RECIPEBOOK_SUBSPECIALS[skill] and RECIPEBOOK_SUBSPECIALS[skill][spec] then
		    spec = RECIPEBOOK_SUBSPECIALS[skill][spec];
		else
		    spec = RECIPEBOOK_DBERROR_NOSPECID;
		end
	else
	    spec = RECIPEBOOK_DBERROR_NOSPECID;
	end
	return spec;
end


--[[ Rep_TextToDB(rep) --> Converts a faction ("Darnassus", etc) to a numerical ID.  Stored by account, so convert back before sending. ]]--
function RBDB:Rep_TextToDB(rep)
	if RecipeBook_Indices["Factions"] ~= nil and RecipeBook_Indices["Factions"][rep] ~= nil then
		return RecipeBook_Indices["Factions"][rep];
	else
	    if RecipeBook_Indices["Factions"] == nil then RecipeBook_Indices["Factions"] = {} end;
		if RecipeBook_Indices["Factions"]["Next Faction"] == nil then RecipeBook_Indices["Factions"]["Next Faction"] = 1 end;
		if RecipeBook_Indices["Factions"][rep] == nil then RecipeBook_Indices["Factions"][rep] = RecipeBook_Indices["Factions"]["Next Faction"];
		    RecipeBook_Indices["Factions"]["Next Faction"] = RecipeBook_Indices["Factions"]["Next Faction"] + 1;
		end
		return RecipeBook_Indices["Factions"][rep];
	end
end

--[[ Rep_DBToText(rep) --> Converts a faction id to the faction name ("Darnassus", etc)  ]]--
function RBDB:Rep_DBToText(rep)
	for faction, id in pairs(RecipeBook_Indices["Factions"]) do
	    if id == rep then return faction end;
	end
	return "No faction match.";
end

--[[ RepValue_TextToDB(value) --> Converts a faction standing ("Honored" etc) to a numerical value. ]]--
function RBDB:RepValue_TextToDB(value)
	for n = 1, 8 do
	    if getglobal("FACTION_STANDING_LABEL"..n) == value or getglobal("FACTION_STANDING_LABEL"..n.."_FEMALE") == value then return n end;
	end
	return 0;
end


--[[ Faction_TextToDB(faction, shared) --> Converts faction data ("alliance/horde") + shared info (true if info is shared/false if not) to single database number ]]--
function RBDB:Faction_TextToDB(fac, share)
	if fac ~= FACTION_ALLIANCE and fac ~= FACTION_HORDE then return false end;
	if share == nil then share = false end;
	if type(share) ~= "boolean" then
		share = (share == RECIPEBOOK_BROWSE_FS_SHARED and true or false)
	end
	return ((fac == FACTION_ALLIANCE and 10 or 20) + (share and 1 or 0));
end

--[[ Faction_DBToText(faction, shared) --> Converts database ID to text info on faction ("alliance/horde") + shared status (true if info is shared/false if not)]]--
function RBDB:Faction_DBToText(fac)
	if type(fac) ~= "number" then return "No faction", false end;
	return (math.floor(fac/10) == 1 and FACTION_ALLIANCE or FACTION_HORDE), (math.fmod(fac, 10) == 1 and true or false);
end

--[[ Item_TextToDB(name, [skill]) --> Returns the known item ID and recipe id for a given item name; 0 if not found ]]--
function RBDB:Item_TextToDB(name, skill)
	local id, rid = 0, 0;
	if not name then return id, rid end;
	if string.sub(name, 1, 1) == "\n" then name = string.sub(name, 2) end;
	name = string.lower(string.gsub(name, "[^%w]", ""));
	-- if RECIPEBOOK_ENCHANTMISMATCH[name] then name = RECIPEBOOK_ENCHANTMISMATCH[name] end; -- Fix enchant name errors
	local idt = RecipeBook_ItemNameDB[name];
	if idt then id, rid = unpack(idt) end;
	if id == 0 and skill and RBDB:IsEnchant(skill) then -- Partial matches may be a workaround for enchanting items, but very resource-intensive.
		name = "^"..name; -- Anchor at the beginning of the string.
		for what, i in pairs(RecipeBook_ItemNameDB) do
			if string.find(what, name) then id, rid = unpack(i) end; 
		end
	end
	return id, rid;
end

--[[ Recipe_TextToDB(item ID) --> When given a recipe item (i.e. "Recipe: Crocolisk Gumbo") attempts to determine the SINGLE id, rid pair which matches.  ]]--
function RBDB:Recipe_ItemToDB(iid, name)
	local skill;
	if not name then 
		name, _, _, _, _, _, skill = GetItemInfo(iid);
	end
	if not name then return 0,0 end;
	name = string.gsub(name, "^(%w+: )", "");
	skill = RBDB:Skill_TextToDB(skill);
	local id, rid = RBDB:Item_TextToDB(name, skill);
	if id == 0 then return 0,0;
	elseif type (rid) == "table" then
		for x, rx in ipairs(rid) do
			rname = GetSpellInfo(rx);
			if rname == name then return id, rx;
			elseif string.find(name, rname) or string.find(rname, name) then return id, rx;
			end
			return id, 0;
		end
	else
		return id, rid;
	end;
end

--[[ Link_DBToText(item id, recipe id, [special needs]) --> Returns the full hyperlink of an item, given its item ID ]]--
function RBDB:Link_DBToText(id, rid, special)
	local link = "[No link available (item id: "..id..", recipe id: "..rid..")]"
	if special then
		-- We are generating the "Tradeskill:Item" link for this item
		if special == "gold" then
			if id and RecipeBook_ItemDB[id] and RecipeBook_ItemDB[id][rid] then
				skill = RecipeBook_ItemDB[id][rid]["Skill"];
			else
				skill = "Skill Not Available";
			end
			local name = GetSpellInfo(rid);
			link = "|cffffd000|Henchant:"..rid.."|h["..RBDB:Skill_DBToText(skill)..": "..name.."]|h|r";
		-- We are generating an item link for a materials item.
		elseif special == "mat" then
			_, link = GetItemInfo(id);
		end
	else
		if id == rid then 
			local name = GetSpellInfo(rid);
			link = "|cffffd000|Henchant:"..rid.."|h["..name.."]|h|r";
			-- link = GetSpellLink(id);
		else _, link = GetItemInfo(id);
		end
	end
	
	return link;
end

--[[ Link_TextToDB(link) --> Returns the ID of an item, given its hyperlink]]--
function RBDB:Link_TextToDB(link)
	if not link then return false end;
	id = string.match(link,".*|Hitem:(%d+)[:%-?%d+]*|h.*");
	if not id then id = string.match(link,".*|Henchant:(%d+)[:%-?%d+]*|h.*") end;
	if not id then
		return false;
	else
		return tonumber(id);
	end
end


--[[ TradeskillIndex_TextToDB(index) --> Checking for the correct index of the requested tradeskill table item ]]--
function RBDB:TradeskillIndex_TextToDB(index)
	if RECIPEBOOK_TRADESKILL_DEFAULTS[index] then return RECIPEBOOK_TRADESKILL_DEFAULTS[index][1];
	else return 0; -- Not found.
	end
end

--[[ Reputation_TextToNumber(reputation) --> Returns a numeric code for your particular reputation in a faction ]]-- 
function RBDB:Reputation_TextToNumber(reput)
	for i in 1, 8 do
		if getglobal("FACTION_STANDING_LABEL"..i) == reput then return i end;
	end
	return 0;
end

--===========================================================================================--
--============================== RECIPEBOOK ITEM NAME DATABASE ==============================--
--===========================================================================================--
--[[ This is a volatile database - not stored - of name to id conversions; it's updated regularly ]]--
function RBDB:UpdateNameList()
	-- At least 15 seconds between update calls.
	local elapsed = time() - RecipeBook.Parms.LastDBUpdate;
	if elapsed < 15 then return end;
	
	-- Code to warn user why they have been disconnected; possibly may happen again.
	if next(RecipeBook_ItemNameDB) == nil and RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Last Processed"] == true then
	    RecipeBook:Debug("RecipeBook was disconned for bad link last load!");
		RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Last Processed"] = false;
		StaticPopup_Show("RECIPEBOOK_ITEM_DISCONNECTED");
		return;
	end

	RecipeBook_ItemNameDB = {};
	
	-- Inserts id and recipe id data into the reverse-mapping table.
	local function makeDB(id, rid, name)
		name = string.lower(string.gsub(name, "[^%w]", "")); -- converts to DB format.
		if RecipeBook_ItemNameDB[name] ~= nil then
			if type(RecipeBook_ItemNameDB[name][2]) == "table" then
				table.insert(RecipeBook_ItemNameDB[name][2], rid);
			else
				RecipeBook_ItemNameDB[name][2] = {RecipeBook_ItemNameDB[name][2]};
				table.insert(RecipeBook_ItemNameDB[name][2], rid);
			end
		else
			RecipeBook_ItemNameDB[name] = {id, rid};
		end
		if RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Unsafe Items"][id] then RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Unsafe Items"][id] = nil; RecipeBook:Debug("Item "..id.." is now safe.") end;
	end
	
	local cached, uncached = false, {};
	for id, info in pairs(RecipeBook_ItemDB) do 
		for rid, data in pairs(info) do
			if type(rid) == "number" then  -- Now we have an id and an item ID.
				-- Special item which has a special name.  This item is double-mapped because one or the other will be correct.
				if data.Name then makeDB(id, rid, data.Name) end;
				-- All items should have either a base name or an enchant name.
				if id == rid and RecipeBook_ItemDB[id]["Enchant"] ~= nil then makeDB(id, rid, RecipeBook_ItemDB[id]["Enchant"]) 
				elseif RecipeBook_ItemDB[id]["Name"] ~= nil then makeDB(id, rid, RecipeBook_ItemDB[id]["Name"]);
				else
					RecipeBook:Debug("Incorrectly cached item: "..id.."/"..rid);
				end
			end
		end
	end

	RecipeBook.Parms.Button = {RBData ={Name = "No Item", Modify = nil,Tooltip = {}, Chatframe = {}}};
	RecipeBook.Parms.LastDBUpdate = time();
	RecipeBook:ScheduleEvent(RBDB.FinishScan, 5); -- After five seconds, end processing loop.
	return true;
end


function RBDB:FinishScan()
    RecipeBook_CharacterDB[RecipeBook.Globals.Realm]["Last Processed"] = false;
end;



