﻿--[[ Synchronization functions ]]

local _G = getfenv(0)
local math_floor = _G.math.floor
local math_max = _G.math.max
local math_min = _G.math.min
local string_find = _G.string.find

function Recap_SetSyncDisplay()

	if not recap_temp.Loaded then
		return
	end

	if recap.Opt.IgnoreGUIDs.value then
		-- if ignoring GUIDs, disallow sync
		recap.Opt.EnableSync.value = false
		recap.Opt.SyncState.value = "Ignore"
	end

	if recap.Opt.SyncState.value == "Ignore" then
		RecapOpt_SyncState:SetText(RECAP_SYNC_STATE_IGNORE)
		RecapOpt_SyncStateDetail:SetText("")
		RecapOpt_SyncStateLastUpdate:SetText("")
		RecapStartSyncMergePetsOffButton:Disable()
		RecapStartSyncMergePetsOnButton:Disable()
		RecapManualSyncButton:Disable()
		RecapBecomeLeaderButton:Disable()
		RecapListMembersButton:Disable()
		RecapClearInactiveMembersButton:Disable()
		RecapSkipNextFightButton:Enable()
	elseif recap.Opt.SyncState.value == "Ready" then
		RecapOpt_SyncState:SetText(RECAP_SYNC_STATE_READY)
		RecapOpt_SyncStateDetail:SetText("")
		RecapOpt_SyncStateLastUpdate:SetText("")
		RecapStartSyncMergePetsOffButton:Enable()
		RecapStartSyncMergePetsOnButton:Enable()
		RecapManualSyncButton:Disable()
		RecapBecomeLeaderButton:Disable()
		RecapListMembersButton:Disable()
		RecapClearInactiveMembersButton:Disable()
		RecapSkipNextFightButton:Enable()
	elseif recap.Opt.SyncState.value == "Member" then
		RecapOpt_SyncState:SetText(RECAP_SYNC_STATE_MEMBER)
		Recap_SetSyncStateDetail()
		RecapStartSyncMergePetsOffButton:Enable()
		RecapStartSyncMergePetsOnButton:Enable()
		RecapManualSyncButton:Disable()
		RecapBecomeLeaderButton:Enable()
		RecapListMembersButton:Enable()
		RecapClearInactiveMembersButton:Enable()
		RecapSkipNextFightButton:Disable()
	elseif recap.Opt.SyncState.value == "Leader" then
		RecapOpt_SyncState:SetText(RECAP_SYNC_STATE_LEADER)
		Recap_SetSyncStateDetail()
		RecapStartSyncMergePetsOffButton:Enable()
		RecapStartSyncMergePetsOnButton:Enable()
		RecapManualSyncButton:Enable()
		RecapBecomeLeaderButton:Disable()
		RecapListMembersButton:Enable()
		RecapClearInactiveMembersButton:Enable()
		RecapSkipNextFightButton:Enable()
	else
		-- safety code
		recap.Opt.EnableSync.value = false
		recap.Opt.SyncState.value = "Ignore"
		-- clear all sync values (including the spinlock)
		recap.Opt.SyncTimestamp.value = false
		recap.Opt.SyncLeader.value = false
		recap.Opt.SyncLeaderName.value = false
		recap.Opt.SyncLeaderServer.value = false
		recap.Opt.SyncMergePets.value = false
		recap.Opt.SyncUpdateWhen.value = false
		recap_temp.SyncInviteTimestamp = false
		recap_temp.SyncInviteLeader = false
		recap_temp.SyncInviteLeaderName = false
		recap_temp.SyncInviteLeaderServer = false
		recap_temp.SyncInviteLeaderRecapVersion = false
		recap_temp.SyncInviteMergePets = false
		recap_temp.SyncInviteWhen = false
		recap_temp.SyncInviteNewOngoing = false
		recap_temp.AskingJoinSync = false
		recap_temp.SyncData = {}
		recap_temp.SyncMembers = {}
		recap_temp.OnCommReceived = false
		RecapOpt_SyncState:SetText(RECAP_SYNC_STATE_IGNORE)
		RecapOpt_SyncStateDetail:SetText("")
		RecapOpt_SyncStateLastUpdate:SetText("")
		RecapStartSyncMergePetsOffButton:Disable()
		RecapStartSyncMergePetsOnButton:Disable()
		RecapManualSyncButton:Disable()
		RecapBecomeLeaderButton:Disable()
		RecapListMembersButton:Disable()
		RecapClearInactiveMembersButton:Disable()
		RecapSkipNextFightButton:Enable()
	end
end

function Recap_SetSyncStateDetail()

	local text

	text = ""
	if recap.Opt.SyncTimestamp.value then
		text = text..recap_temp.Localize.Timestamp..": "..tostring(recap.Opt.SyncTimestamp.value)
	end
	if recap.Opt.SyncState.value and (recap.Opt.SyncState.value == "Member") and recap.Opt.SyncLeader.value then
		-- somehow (don't know how) SyncLeader is losing its value
		-- details for leader distinguished by not listing leader
		text = text.."   "..recap_temp.Localize.Leader..": "..Recap_NameOnlyFromCombatant(recap.Opt.SyncLeaderName.value).." ("..recap.Opt.SyncLeaderServer.value..")"
	end
	if recap.Opt.SyncMergePets.value and (recap.Opt.SyncMergePets.value == "on") then
		text = text.."  ("..recap_temp.Localize.MergePetsOn..")"
	else
		text = text.."  ("..recap_temp.Localize.MergePetsOff..")"
	end
	RecapOpt_SyncStateDetail:SetText(text)
	text = ""
	if recap.Opt.SyncUpdateWhen.value then
		text = recap_temp.Localize.LastUpdateWas..": "..Recap_FormatTime(GetTime()-(recap.Opt.SyncUpdateWhen.value/1000)).." "..recap_temp.Localize.Ago
	end
	RecapOpt_SyncStateLastUpdate:SetText(text)
end

function Recap_StartSync(mergePets)

	-- paranoia check
	if recap.Opt.SyncState.value == "Ignore" then
		-- bail if ignoring synchronization
		return
	end

	if (recap.Opt.State.value == "Stopped") then
		-- we've got Recap turned off at the moment (okay to start if Recap is in combat, invitation will be presented when people are out of combat)
		PlaySound("igQuestFailed")
		return
	end

	-- lock OnCommReceived event
	if recap_temp.OnCommReceived == true then
		PlaySound("igQuestFailed")
		return
	end
	recap_temp.OnCommReceived = true

	-- abandon existing synchronization (if any), start a new one as leader
	recap.Opt.SyncState.value = "Leader"
	recap.Opt.SyncTimestamp.value = math_floor(1000*GetTime()) -- integer thousandths
	recap.Opt.SyncLeader.value = recap_temp.PlayerGUID.."_"..recap_temp.Server
	recap.Opt.SyncLeaderName.value = recap_temp.PlayerGUID
	recap.Opt.SyncLeaderServer.value = recap_temp.Server
	recap.Opt.SyncMergePets.value = mergePets
	recap.Opt.SyncUpdateWhen.value = recap.Opt.SyncTimestamp.value
	recap_temp.AskingJoinSync = false
	recap_temp.SyncData = {}
	recap_temp.SyncMembers = {}
	Recap_UpdateSyncMember(recap_temp.PlayerGUID, recap_temp.Server, Recap_Version)
	Recap_SetSyncDisplay()

	-- reset All Fights
	RecapPanel_Hide(1)
	RecapRecent_Hide(1)
	Recap_ResetAllCombatants(true) -- force reset even of locked combatants
	Recap_MakeFriends()

	-- perhaps remove pet effects
	if recap.Opt.SyncMergePets.value == "off" then
		Recap_RemovePetEffects()
	end

	-- display
	Recap_SetState(false)
	Recap_SetView()

	-- issue invitation (version, timestamp, leader, mergePets, name, server, recapVersion)
	if not Recap:SendCommMessage("GROUP", recap_temp.Invite, Recap_Sync_Version, recap.Opt.SyncTimestamp.value, recap.Opt.SyncLeader.value, recap.Opt.SyncMergePets.value, recap_temp.PlayerGUID, recap_temp.Server, Recap_Version) then
		-- not sent
		DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendInvite)
	end

	-- unlock OnCommReceived event
	recap_temp.OnCommReceived = false
end

function Recap_ManualSync()

	-- paranoia check
	if not (recap.Opt.SyncState.value == "Leader") then
		-- bail if not Leader
		return
	end

	if (recap.Opt.State.value == "Stopped") or (recap.Opt.State.value == "Active") then
		-- we've got Recap turned off at the moment, or are in combat
		PlaySound("igQuestFailed")
		return
	end

	-- lock OnCommReceived event
	if recap_temp.OnCommReceived == true then
		PlaySound("igQuestFailed")
		return
	end
	recap_temp.OnCommReceived = true

	Recap_SendSummary()

	-- unlock OnCommReceived event
	recap_temp.OnCommReceived = false
end

function Recap_BecomeLeader()

	-- paranoia check
	if not (recap.Opt.SyncState.value == "Member") then
		-- bail if not Member
		return
	end

	if (recap.Opt.State.value == "Stopped") or (recap.Opt.State.value == "Active") then
		-- we've got Recap turned off at the moment, or are in combat
		PlaySound("igQuestFailed")
		return
	end

	-- lock OnCommReceived event
	if recap_temp.OnCommReceived == true then
		PlaySound("igQuestFailed")
		return
	end
	recap_temp.OnCommReceived = true

	-- take over an existing synchronization as leader
	recap.Opt.SyncState.value = "Leader"
	recap.Opt.SyncLeader.value = recap_temp.PlayerGUID.."_"..recap_temp.Server
	recap.Opt.SyncLeaderName.value = recap_temp.PlayerGUID
	recap.Opt.SyncLeaderServer.value = recap_temp.Server
	Recap_SetSyncDisplay()

	-- do a manual synchronization to let other players know about the change of leader
	Recap_SendSummary()

	-- unlock OnCommReceived event
	recap_temp.OnCommReceived = false
end

-- standard receipt of Recap synchronization messages
function Recap:OnCommReceived(prefix, sender, distribution, request, version, timestamp, leader, mergePets,
							  Name, Time, TimeIn, TimeHeal, DmgOut, DmgIn, RawHeal, OverHeal, MaxHit, Kills, Seen, Flags)

	-- in some contexts (Invite, Accept, Decline, Heartbeat) two parameters have different meanings
	local Server = Time
	local RecapVersion = TimeIn

	if recap.Opt.SyncState.value == "Ignore" then
		-- we're ignoring synchronization
		-- (TODO: future optimisation would be to dynamically Register and Unregister from the channel)
		return
	end

	if recap.Opt.State.value == "Stopped" then
		-- we've got Recap turned off at the moment
		return
	end

	-- between AceComm (r47060) and AceComm (r48940) there is a serious incompatibility, so that good values
	--   sent using the former are received via the latter as nils -- bad Ace !
	--   and later (r66043) as false with sync supposedly turned off -- damn you Ace !
	if (request == nil) or (version == nil) or (timestamp == nil) or (leader == nil) or (mergePets == nil) or
	   (request == false) or (version == false) or (timestamp == false) or (leader == false) then
		-- silently ignore messages with nil or false for a selection of parameters
		return
	end

	if not (version == Recap_Sync_Version) then
		-- incompatible version (at moment exact match required)
		if version and Recap_Sync_Version and tonumber(version) and tonumber(Recap_Sync_Version) and (tonumber(version) > tonumber(Recap_Sync_Version)) then
			-- running an outdated version
			DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.OutOfDateSyncVersion)
		end
		return
	end

	-- lock OnCommReceived event
	if recap_temp.OnCommReceived == true then return end
	recap_temp.OnCommReceived = true

	-- State machine based on SyncState and request

	-- Invite messages (Ready, Member, Leader)
	if request == recap_temp.Invite then
		if recap.Opt.State.value == "Active" then
			-- in combat, remember the request (overwrites any earlier memo)
			Recap_RememberInvite(timestamp, leader, mergePets, Name, Server, RecapVersion, "new")
		elseif recap.Opt.State.value == "Idle" then
			-- idle (not in combat), okay to pop the question (but only if not already asking it)
			if not recap_temp.AskingJoinSync then
				recap_temp.AskingJoinSync = true
				Recap_AskJoinSync(timestamp, leader, mergePets, Name, Server, RecapVersion, "new")
			end
		end
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end

	-- Ready (Summary, treated as an invitation)
	if recap.Opt.SyncState.value == "Ready" then
		-- not currently a member of a synchronization
		if request == recap_temp.Summary then
			-- treat a Summary record as an implicit invitation
			if recap.Opt.State.value == "Active" then
				-- in combat, remember the invitation for later (overwrites any earlier memo)
				Recap_RememberInvite(timestamp, leader, mergePets, sender, Recap_SyncExtractServer(leader), recap_temp.Localize.Unknown, "ongoing")
			else
				-- idle (not in combat), okay to pop the question (but only if not already asking it)
				if not recap_temp.AskingJoinSync then
					recap_temp.AskingJoinSync = true
					Recap_AskJoinSync(timestamp, leader, mergePets, sender, Recap_SyncExtractServer(leader), recap_temp.Localize.Unknown, "ongoing")
				end
			end
			-- note that until the player actually accepts the invite, the Summary data are discarded
		end
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end
	-- assert: Member or Leader only


	-- assert: further processing must match on timestamp
	if not (recap.Opt.SyncTimestamp.value == timestamp) then
		-- no match, bail early
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end

	-- Heartbeat message
	if request == recap_temp.Heartbeat then
		-- note this person as part of the sync
		Recap_UpdateSyncMember(Name, Server, RecapVersion)
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end

	-- Accept and Decline messages (only reported to leader)
	if request == recap_temp.Accept then
		if recap.Opt.SyncState.value == "Leader" then
			DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..Recap_NameOnlyFromCombatant(Name).." ("..Server..") "..recap_temp.Localize.AcceptedSync)
		end
		-- note this person as part of the sync
		Recap_UpdateSyncMember(Name, Server, RecapVersion)
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end
	if request == recap_temp.Decline then
		if recap.Opt.SyncState.value == "Leader" then
			DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..Recap_NameOnlyFromCombatant(Name).." ("..Server..") "..recap_temp.Localize.DeclinedSync)
		end
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end

	-- SkipNextFight message
	if request == recap_temp.SkipNextFight then
		-- skip next fight
		recap.Opt.SkipNextFight.value = true
		DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..":  "..recap_temp.Localize.SkipNextFightBegins)
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end
	-- assert: Reply or Summary messages only

	if not ((request == recap_temp.Reply) or (request == recap_temp.Summary)) then
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end

	-- assert: everyone needs to be on the same page for synchronization to work reliably
	-- assert: the All Fights panel is stable from the end of one fight until just before the accumulation at the end of the next fight
	-- assert: we're going to be extra cautious, and only process and respond to updates until the beginning of the next fight
	-- assert: short inter-fight gaps will probably mean incomplete synchronization

	-- if we're in combat, remember the data for later on (whether Summary or Reply) (overwrites any earlier memo)
	if recap.Opt.State.value == "Active" then
		Recap_RememberSyncData(timestamp, Name, Time, TimeIn, TimeHeal, DmgOut, DmgIn, RawHeal, OverHeal, MaxHit, Kills, Seen, Flags)
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end

	-- Reply messages
	if request == recap_temp.Reply then
		-- ignore any possible new leader information from a Reply in case it is stale-dated
		Recap_ProcessReply(Name, Time, TimeIn, TimeHeal, DmgOut, DmgIn, RawHeal, OverHeal, MaxHit, Kills, Seen, Flags)
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end
	-- assert: Summary messages only

	if not (request == recap_temp.Summary) then
		recap_temp.OnCommReceived = false -- unlock OnCommReceived event
		return
	end

	-- check for new leader
	if not (recap.Opt.SyncLeader.value == leader) then
		-- new leader (if we were the Leader, we silently become a Member)
		recap.Opt.SyncState.value = "Member"
		recap.Opt.SyncLeader.value = leader
		recap.Opt.SyncLeaderName.value = Recap_SyncExtractNameGUID(leader)
		recap.Opt.SyncLeaderServer.value = Recap_SyncExtractServer(leader)
		Recap_SetSyncDisplay()
	end

	-- and finally, process the Summary update
	Recap_ProcessSummary(Name, Time, TimeIn, TimeHeal, DmgOut, DmgIn, RawHeal, OverHeal, MaxHit, Kills, Seen, Flags)

	-- unlock OnCommReceived event
	recap_temp.OnCommReceived = false
end

function Recap_RememberInvite(timestamp, leader, mergePets, name, server, recapVersion, newOngoing)
	recap_temp.SyncInviteTimestamp = timestamp
	recap_temp.SyncInviteLeader = leader
	recap_temp.SyncInviteLeaderName = name
	recap_temp.SyncInviteLeaderServer = server
	recap_temp.SyncInviteLeaderRecapVersion = recapVersion
	recap_temp.SyncInviteMergePets = mergePets
	recap_temp.SyncInviteWhen = math_floor(1000*GetTime()) -- integer thousandths
	recap_temp.SyncInviteNewOngoing = newOngoing
end

function Recap_RememberSyncData(timestamp, Name, Time, TimeIn, TimeHeal, DmgOut, DmgIn, RawHeal, OverHeal, MaxHit, Kills, Seen, Flags)
	if not recap_temp.SyncData[Name] then
		recap_temp.SyncData[Name] = {}
	end
	local iName = recap_temp.SyncData[Name]
	iName.Timestamp = timestamp
	iName.Time = math_max((iName.Time or 0), (Time or 0))
	iName.TimeIn = math_max((iName.TimeIn or 0), (TimeIn or 0))
	iName.TimeHeal = math_max((iName.TimeHeal or 0), (TimeHeal or 0))
	iName.DmgOut = math_max((iName.DmgOut or 0), (DmgOut or 0))
	iName.DmgIn = math_max((iName.DmgIn or 0), (DmgIn or 0))
	iName.RawHeal = math_max((iName.RawHeal or 0), (RawHeal or 0))
	iName.OverHeal = math_max((iName.OverHeal or 0), (OverHeal or 0))
	iName.MaxHit = math_max((iName.MaxHit or 0), (MaxHit or 0))
	iName.Kills = math_max((iName.Kills or 0), (Kills or 0))
	if Seen and (Seen > 0) then
		if iName.Seen and (iName.Seen > 0) then
			iName.Seen = math_min(iName.Seen, Seen)
		else
			iName.Seen = Seen
		end
	end
	if Flags and (Flags > 0) then
		iName.Flags = Flags
	end
	iName.When = math_floor(1000*GetTime()) -- integer thousandths
end

function Recap_CheckJoinSync()
	-- see if there is an outstanding invitation to a synchronization that we saved during combat
	-- called immediately after EndFight

	if recap.Opt.SyncState.value == "Ignore" then
		-- we're ignoring synchronization
		return
	end

	if (recap.Opt.State.value == "Stopped") or (recap.Opt.State.value == "Active") then
		-- we've got Recap turned off at the moment, or are in combat
		return
	end

	if not (recap_temp.SyncInviteTimestamp and recap_temp.SyncInviteLeader and recap_temp.SyncInviteLeaderName and recap_temp.SyncInviteLeaderServer and recap_temp.SyncInviteLeaderRecapVersion and recap_temp.SyncInviteMergePets and recap_temp.SyncInviteWhen and recap_temp.SyncInviteNewOngoing) then
		-- paranoia check that the saved invite info is present and complete
		return
	end

	-- lock OnCommReceived event
	if recap_temp.OnCommReceived == true then return end
	recap_temp.OnCommReceived = true

	if (GetTime()-(recap_temp.SyncInviteWhen/1000)) <= 180 then
		-- within the time limit (in seconds)
		-- pop the question (we were out of combat when Recap_CheckJoinSync was called)
		local timestamp, leader, name, server, recapVersion, mergePets, newOngoing
		timestamp = recap_temp.SyncInviteTimestamp
		leader = recap_temp.SyncInviteLeader
		name = recap_temp.SyncInviteLeaderName
		server = recap_temp.SyncInviteLeaderServer
		recapVersion = recap_temp.SyncInviteLeaderRecapVersion
		mergePets = recap_temp.SyncInviteMergePets
		newOngoing = recap_temp.SyncInviteNewOngoing
		recap_temp.SyncInviteTimestamp = false
		recap_temp.SyncInviteLeader = false
		recap_temp.SyncInviteLeaderName = false
		recap_temp.SyncInviteLeaderServer = false
		recap_temp.SyncInviteLeaderRecapVersion = false
		recap_temp.SyncInviteMergePets = false
		recap_temp.SyncInviteWhen = false
		recap_temp.SyncInviteNewOngoing = false
		recap_temp.AskingJoinSync = true
		Recap_AskJoinSync(timestamp, leader, mergePets, name, server, recapVersion, newOngoing)
	end

	-- unlock OnCommReceived event
	recap_temp.OnCommReceived = false
end

function Recap_AskJoinSync(timestamp, leader, mergePets, name, server, recapVersion, newOngoing)

	local msg

	msg = recap_temp.Recap..": "..Recap_NameOnlyFromCombatant(name).." ("..server..") "..recap_temp.Localize.InvitesYouToSynchronize
	if mergePets == "on" then
		msg = msg.."  ("..recap_temp.Localize.MergePetsOn..")"
	else
		msg = msg.."  ("..recap_temp.Localize.MergePetsOff..")"
	end
	if newOngoing == "new" then
		msg = msg.."\n\n--  "..recap_temp.Localize.NewSynchronization.."  --"
	else
		msg = msg.."\n\n--  "..recap_temp.Localize.OngoingSynchronization.."  --"
	end
	StaticPopupDialogs["recap_temp.ASKJOINSYNC"] = {
		text = TEXT(msg),
		button1 = TEXT(ACCEPT),
		button2 = TEXT(CANCEL),
		OnAccept = function()
			-- join the synchronization
			Recap_JoinSync(timestamp, leader, mergePets, name, server, recapVersion)
			-- send accept message
			if not Recap:SendCommMessage("GROUP", recap_temp.Accept, Recap_Sync_Version, timestamp, leader, mergePets, recap_temp.PlayerGUID, recap_temp.Server, Recap_Version) then
				-- not sent
				DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendAccept)
			end
			collectgarbage("collect")
			-- free the spinlock
			recap_temp.AskingJoinSync = false
		end,
		OnCancel = function()
			-- send decline message
			if not Recap:SendCommMessage("GROUP", recap_temp.Decline, Recap_Sync_Version, timestamp, leader, mergePets, recap_temp.PlayerGUID, recap_temp.Server, Recap_Version) then
				-- not sent
				DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendDecline)
			end
			-- free the spinlock
			recap_temp.AskingJoinSync = false
		end,
		timeout = 0,
		showAlert = 1,
		whileDead = 1
	}
	StaticPopup_Show("recap_temp.ASKJOINSYNC") -- Note: NOT a blocking call

end

function Recap_JoinSync(timestamp, leader, mergePets, name, server, recapVersion)

	recap.Opt.SyncState.value = "Member"
	recap.Opt.SyncTimestamp.value = timestamp
	recap.Opt.SyncLeader.value = leader
	recap.Opt.SyncLeaderName.value = name
	recap.Opt.SyncLeaderServer.value = server
	recap.Opt.SyncMergePets.value = mergePets
	recap.Opt.SyncUpdateWhen.value = math_floor(1000*GetTime()) -- integer thousandths
	recap_temp.SyncData = {}
	recap_temp.SyncMembers = {}
	Recap_UpdateSyncMember(name, server, recapVersion)
	Recap_UpdateSyncMember(recap_temp.PlayerGUID, recap_temp.Server, Recap_Version)
	Recap_SetSyncDisplay()

	-- reset All Fights
	RecapPanel_Hide(1)
	RecapRecent_Hide(1)
	Recap_ResetAllCombatants(true) -- force reset even of locked combatants
	Recap_MakeFriends()

	-- perhaps remove pet effects
	if recap.Opt.SyncMergePets.value == "off" then
		Recap_RemovePetEffects()
	end

	-- display
	Recap_SetState(false)
	Recap_SetView()

end

function Recap_CheckSendSummary()
	-- see if we as leader need to broadcast a Summary
	-- called immediately after EndFight

	if not (recap.Opt.SyncState.value == "Leader") then
		-- bail if not Leader
		return
	end

	if (recap.Opt.State.value == "Stopped") or (recap.Opt.State.value == "Active") then
		-- we've got Recap turned off at the moment, or are in combat
		return
	end

	if recap.Opt.SkipNextFight.value == true then
		-- we've been skipping this fight, so don't send a summary
		return
	end

	if math_max(1+GetNumPartyMembers(),GetNumRaidMembers()) == 1 then
		-- we're not in a group, skip the summary (can still send one manually)
		return
	end

	-- lock OnCommReceived event
	if recap_temp.OnCommReceived == true then return end
	recap_temp.OnCommReceived = true

	-- do it
	Recap_SendSummary()

	-- unlock OnCommReceived event
	recap_temp.OnCommReceived = false
end

function Recap_SendSummary()

	local i, iCombatant

	if not recap.Combatant then
		recap.Combatant = {}
	end
	for i in pairs(recap.Combatant) do
		if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) then
			iCombatant = recap.Combatant[i]
			-- skip combatants with all combat data nil
			if iCombatant.TotalTime or iCombatant.TotalTimeIn or iCombatant.TotalTimeHeal or iCombatant.TotalDmgOut or iCombatant.TotalDmgIn or
			   iCombatant.TotalRawHeal or iCombatant.TotalOverHeal or iCombatant.TotalMaxHit or iCombatant.TotalKills then
				-- one message per combatant (version, timestamp, leader, mergePets, summary values)
				-- send times as integer thousandths for more compact transmission
				if not Recap:SendCommMessage("GROUP", recap_temp.Summary, Recap_Sync_Version, recap.Opt.SyncTimestamp.value, recap.Opt.SyncLeader.value, recap.Opt.SyncMergePets.value,
											 i,
											 math_floor(1000*(iCombatant.TotalTime or 0)), math_floor(1000*(iCombatant.TotalTimeIn or 0)), math_floor(1000*(iCombatant.TotalTimeHeal or 0)),
											 (iCombatant.TotalDmgOut or 0), (iCombatant.TotalDmgIn or 0), (iCombatant.TotalRawHeal or 0), (iCombatant.TotalOverHeal or 0),
											 (iCombatant.TotalMaxHit or 0), (iCombatant.TotalKills or 0), (iCombatant.Seen or 0), (iCombatant.Flags or 0)) then
					-- not sent
					DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendSummary)
					-- leave loop, no point in continuing after one failure
					break
				end
			end
		end
	end

	-- send fight start and end points and total duration as a special combatant called "Last Fight Start and End"
	if not Recap:SendCommMessage("GROUP", recap_temp.Summary, Recap_Sync_Version, recap.Opt.SyncTimestamp.value, recap.Opt.SyncLeader.value, recap.Opt.SyncMergePets.value,
								 "Last Fight Start and End",
								 math_floor(1000*(recap[recap_temp.p].LastFightStart or 0)), math_floor(1000*(recap[recap_temp.p].LastFightEnd or 0)),
								 (recap[recap_temp.p].TotalDuration or 0), false, false, false, false, false, false) then
		-- not sent
		DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendSummary)
	end

	-- turn on sync status light blink
	if recap_temp.SyncStatusLightBlink == 0 then
		recap_temp.SyncStatusLightBlink = 1
	end

	-- update our own heartbeat as leader
	Recap_UpdateSyncMember(recap_temp.PlayerGUID, recap_temp.Server, Recap_Version)

	-- remember the local time of the latest update activity
	recap.Opt.SyncUpdateWhen.value = math_floor(1000*GetTime()) -- integer thousandths
	Recap_SetSyncDisplay()

	-- start a timer to detect when synchronization activity is over
	recap_temp.EndSyncDelayTimer = 0

	-- turn on sync status light blink
	if recap_temp.SyncStatusLightBlink == 0 then
		recap_temp.SyncStatusLightBlink = 1
	end
end

function Recap_CheckProcessSyncData()
	-- see if we need to process any remembered updates that were saved during combat
	-- could include Summary or Reply data, and we treat both kinds of update data as if it were Summary data (so in this case we might Reply to a Reply)
	-- called immediately after EndFight

	local Name, i, iName

	if ((recap.Opt.SyncState.value == "Ignore") or (recap.Opt.SyncState.value == "Ready")) then
		-- we're not in synchronization
		return
	end

	if (recap.Opt.State.value == "Stopped") or (recap.Opt.State.value == "Active") then
		-- we've got Recap turned off at the moment, or are in combat
		return
	end

	-- lock OnCommReceived event
	if recap_temp.OnCommReceived == true then return end
	recap_temp.OnCommReceived = true

	-- scan for saved SyncData
	for Name in pairs(recap_temp.SyncData) do
		iName = recap_temp.SyncData[Name]
		if iName.Timestamp then
			-- data present
			if iName.Timestamp == recap.Opt.SyncTimestamp.value then
				-- data matches our timestamp (paranoia check)
				if (GetTime()-(iName.When/1000)) <= 180 then
					-- within the time limit (in seconds)
					Recap_ProcessSummary(Name, iName.Time, iName.TimeIn, iName.TimeHeal,
										iName.DmgOut, iName.DmgIn, iName.RawHeal,
										iName.OverHeal, iName.MaxHit, iName.Kills, iName.Seen, iName.Flags)
				end
			end
		end
		-- clear the data
		for i in pairs(iName) do
			iName[i] = nil
		end
	end

	-- unlock OnCommReceived event
	recap_temp.OnCommReceived = false
end

function Recap_ProcessSummary(NameGUID, Time, TimeIn, TimeHeal, DmgOut, DmgIn, RawHeal, OverHeal, MaxHit, Kills, Seen, Flags)

	local upTime, upTimeIn, upTimeHeal, upDmgOut, upDmgIn, upRawHeal, upOverHeal, upMaxHit, upKills, upSeen, upFlags, iCombatant
	upTime = false
	upTimeIn = false
	upTimeHeal = false
	upDmgOut = false
	upDmgIn = false
	upRawHeal = false
	upOverHeal = false
	upMaxHit = false
	upKills = false
	upSeen = false
	upFlags = false


	if NameGUID == "Last Fight Start and End" then
		-- special record for last fight start and end points and total duration
		-- note that since we can't absolutely guarantee that all copies of Recap are talking about the same last fight, we could get out of sync here

		local FightStart = Time -- renamed argument
		local FightEnd = TimeIn -- renamed argument
		local TotalDuration = TimeHeal -- renamed argument
		local upStart = false
		local upEnd = false
		local upDuration = false
		local delta

--DEFAULT_CHAT_FRAME:AddMessage("Summary inc  "..Recap_FormatTime((FightStart or 0)/1000).." "..Recap_FormatTime((FightEnd or 0)/1000).." "..string_format("%.3f", (TotalDuration or 0)/1000))
--DEFAULT_CHAT_FRAME:AddMessage("Summary mine "..Recap_FormatTime((recap[recap_temp.p].LastFightStart or 0)).." "..Recap_FormatTime((recap[recap_temp.p].LastFightEnd or 0)).." "..string_format("%.3f", (recap[recap_temp.p].TotalDuration or 0)))

		if FightStart and (FightStart > 0) and FightEnd and (FightEnd > 0) then

			-- check whether our interval is entirely earlier than the Summary interval, or entirely later, or intersecting with the Summary interval
			if (recap[recap_temp.p].LastFightStart == nil) or (recap[recap_temp.p].LastFightStart == 0) or
			   (recap[recap_temp.p].LastFightEnd == nil) or (recap[recap_temp.p].LastFightEnd == 0) or
			   (FightStart > math_floor(1000*recap[recap_temp.p].LastFightEnd)) then
				-- we are entirely earlier (or have no fight yet), very probably not the same 'last fight'; play 'catch up' by adopting the Summary start and end, and adjusting our TotalDuration
				-- that way we will pick up properly on any Reply messages that update the last fight start and end
				-- we preserve our own last fight duration since sync is for All Fights
				-- not ideal, but relatively safe for most fight situations
				-- we don't compare the leader's total duration, this will only be compared when we have a probably matching 'last fight'
				delta = FightEnd/1000 - FightStart/1000
				recap[recap_temp.p].LastFightStart = FightStart/1000
				recap[recap_temp.p].LastFightEnd = FightEnd/1000
				recap[recap_temp.p].TotalDuration = (recap[recap_temp.p].TotalDuration or 0) + delta
			elseif (FightEnd < math_floor(1000*recap[recap_temp.p].LastFightStart)) then
				-- we are entirely later, very probably not the same 'last fight', we're probably in our own little punch-up elsewhere; ignore entirely
				-- not ideal, but relatively safe for most fight situations
			else
				-- very probably a matching 'last fight'
				-- if the incoming start value is earlier, or if the incoming end value is later, it replaces our value
				-- if our values are wider, we'll send them out
				-- the times come in as integer thousandths for more compact transmission
				if (FightStart <= math_floor(1000*recap[recap_temp.p].LastFightStart)) then
					-- they started earlier
					delta = recap[recap_temp.p].LastFightStart - FightStart/1000
					recap[recap_temp.p].LastFightStart = recap[recap_temp.p].LastFightStart - delta
					recap[recap_temp.p].LastDuration = recap[recap_temp.p].LastDuration + delta
					recap[recap_temp.p].TotalDuration = (recap[recap_temp.p].TotalDuration or 0) + delta
				else
					-- we started earlier
--Recap_AuthorDebug("Reply my FightStart "..tostring(recap[recap_temp.p].LastFightStart).." beats the summary "..tostring(FightStart/1000))
					upStart = math_floor(1000*recap[recap_temp.p].LastFightStart)
				end
				if (FightEnd >= math_floor(1000*recap[recap_temp.p].LastFightEnd)) then
					-- they ended later
					delta = FightEnd/1000 - recap[recap_temp.p].LastFightEnd
					recap[recap_temp.p].LastFightEnd = recap[recap_temp.p].LastFightEnd + delta
					recap[recap_temp.p].LastDuration = recap[recap_temp.p].LastDuration + delta
					recap[recap_temp.p].TotalDuration = (recap[recap_temp.p].TotalDuration or 0) + delta
				else
					-- we ended later
--Recap_AuthorDebug("Reply my FightEnd "..tostring(recap[recap_temp.p].LastFightEnd).." beats the summary "..tostring(FightEnd/1000))
					upEnd = math_floor(1000*recap[recap_temp.p].LastFightEnd)
				end

				-- compare the leader's total duration (we may have already adjusted our own in the preceding code)
				if TotalDuration and (TotalDuration > 0) then
					if recap[recap_temp.p].TotalDuration and (math_floor(1000*recap[recap_temp.p].TotalDuration) > TotalDuration) then
						upDuration = math_floor(1000*recap[recap_temp.p].TotalDuration)
					else
						recap[recap_temp.p].TotalDuration = TotalDuration/1000
					end
				end
			end

			if upStart or upEnd or upDuration then
				-- send an update if appropriate
				-- values that do not need updating are sent as 'false'
				if not Recap:SendCommMessage("GROUP", recap_temp.Reply, Recap_Sync_Version, recap.Opt.SyncTimestamp.value, recap.Opt.SyncLeader.value, recap.Opt.SyncMergePets.value,
											 NameGUID, upStart, upEnd, upDuration, false, false, false, false, false, false) then
					-- not sent (could be a lot of these messages if something is wrong)
					DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendReply)
				end
			end
		end

	else

		-- ordinary combatant
		if not recap.Combatant[NameGUID] then
			-- new combatant
			Recap_CreateBlankCombatant(NameGUID, Seen, Flags)
		end

--Recap_AuthorDebug("Reply my Time "..tostring(recap.Combatant[NameGUID].TotalTime).." beats the summary "..tostring(Time/1000).." for "..tostring(NameGUID))

		-- if an incoming value is larger than our value, it replaces our value
		-- if our value is larger, we'll send that out
		-- the times come in as integer thousandths for more compact transmission
		iCombatant = recap.Combatant[NameGUID]
		if iCombatant.TotalTime and (math_floor(1000*iCombatant.TotalTime) > Time) then
			upTime = math_floor(1000*iCombatant.TotalTime)
		elseif (Time > 0) then
			iCombatant.TotalTime = Time/1000
		end
		if iCombatant.TotalTimeIn and (math_floor(1000*iCombatant.TotalTimeIn) > TimeIn) then
			upTimeIn = math_floor(1000*iCombatant.TotalTimeIn)
		elseif (TimeIn > 0) then
			iCombatant.TotalTimeIn = TimeIn/1000
		end
		if iCombatant.TotalTimeHeal and (math_floor(1000*iCombatant.TotalTimeHeal) > TimeHeal) then
			upTimeHeal = math_floor(1000*iCombatant.TotalTimeHeal)
		elseif (TimeHeal > 0) then
			iCombatant.TotalTimeHeal = TimeHeal/1000
		end
		if iCombatant.TotalDmgOut and (iCombatant.TotalDmgOut > DmgOut) then
			upDmgOut = iCombatant.TotalDmgOut
		elseif (DmgOut > 0) then
			iCombatant.TotalDmgOut = DmgOut
		end
		if iCombatant.TotalDmgIn and (iCombatant.TotalDmgIn > DmgIn) then
			upDmgIn = iCombatant.TotalDmgIn
		elseif (DmgIn > 0) then
			iCombatant.TotalDmgIn = DmgIn
		end
		if iCombatant.TotalRawHeal and (iCombatant.TotalRawHeal > RawHeal) then
			upRawHeal = iCombatant.TotalRawHeal
		elseif (RawHeal > 0) then
			iCombatant.TotalRawHeal = RawHeal
		end
		if iCombatant.TotalOverHeal and (iCombatant.TotalOverHeal > OverHeal) then
			upOverHeal = iCombatant.TotalOverHeal
		elseif (OverHeal > 0) then
			iCombatant.TotalOverHeal = OverHeal
		end
		if iCombatant.TotalMaxHit and (iCombatant.TotalMaxHit > MaxHit) then
			upMaxHit = iCombatant.TotalMaxHit
		elseif (MaxHit > 0) then
			iCombatant.TotalMaxHit = MaxHit
		end
		if iCombatant.TotalKills and (iCombatant.TotalKills > Kills) then
			upKills = iCombatant.TotalKills
		elseif (Kills > 0) then
			iCombatant.TotalKills = Kills
		end
		if iCombatant.Seen and (iCombatant.Seen > 0) and Seen and (iCombatant.Seen < Seen) then
			upSeen = iCombatant.Seen
		elseif Seen and (Seen > 0) then
			iCombatant.Seen = Seen
		end
		if iCombatant.Flags and (iCombatant.Flags > 0) and ((not Flags) or (Flags == 0)) then
			-- we have flags and sender doesn't
			upFlags = iCombatant.Flags
		elseif Flags and (Flags > 0) then
			-- we accept sender's flags, which may not always be the latest value (tough)
			iCombatant.Flags = Flags
		end

		-- silently adjust overhealing number if necessary (could leave us out of strict sync)
		if iCombatant.TotalOverHeal and (iCombatant.TotalOverHeal > 0) and iCombatant.TotalRawHeal and (iCombatant.TotalRawHeal > 0) then
			iCombatant.TotalOverHeal = math_min(iCombatant.TotalOverHeal, iCombatant.TotalRawHeal)
		end

		if upTime or upTimeIn or upTimeHeal or upDmgOut or upDmgIn or upRawHeal or upOverHeal or upMaxHit or upKills or upSeen or upFlags then
			-- one message per combatant (version, timestamp, leader, mergePets, summary values)
			-- values that do not need updating are sent as 'false'
			if not Recap:SendCommMessage("GROUP", recap_temp.Reply, Recap_Sync_Version, recap.Opt.SyncTimestamp.value, recap.Opt.SyncLeader.value, recap.Opt.SyncMergePets.value,
										 NameGUID, upTime, upTimeIn, upTimeHeal, upDmgOut, upDmgIn, upRawHeal, upOverHeal, upMaxHit, upKills, upSeen, upFlags) then
				-- not sent (could be a lot of these messages if something is wrong)
				DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendReply)
			end
		else
			-- we're not sending any update for this combatant
			-- if this is a summary record for the leader of the synchronization (i.e. only once per Summary set),
			--   then send a Heartbeat to show that we are still here
			local tempName
			_, _, tempName = string_find(recap.Opt.SyncLeader.value, "^(%u%l+)")
			if NameGUID == tempName then
				Recap_SendSyncHeartbeat()
			end
		end

		-- derivative calculated values
		if iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
			iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
		end
		if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
			iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
		end
		if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and (((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)) > 0) then
			iCombatant.TotalHPS = Recap_Div1(((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)), iCombatant.TotalTimeHeal)
		end
	end

	-- update our own heartbeat as member
	Recap_UpdateSyncMember(recap_temp.PlayerGUID, recap_temp.Server, Recap_Version)

	-- remember the local time of the latest update activity
	recap.Opt.SyncUpdateWhen.value = math_floor(1000*GetTime()) -- integer thousandths
	Recap_SetSyncDisplay()

	-- start a timer to detect when synchronization activity is over
	recap_temp.EndSyncDelayTimer = 0

	-- turn on sync status light blink
	if recap_temp.SyncStatusLightBlink == 0 then
		recap_temp.SyncStatusLightBlink = 1
	end
end

function Recap_ProcessReply(NameGUID, Time, TimeIn, TimeHeal, DmgOut, DmgIn, RawHeal, OverHeal, MaxHit, Kills, Seen, Flags)

	if NameGUID == "Last Fight Start and End" then
		-- special record for last fight start and end points and total duration
		-- note that since we can't guarantee that all copies of Recap are talking about the same last fight, we can get permanently out of sync here

		local FightStart = Time -- renamed argument
		local FightEnd = TimeIn -- renamed argument
		local TotalDuration = TimeHeal -- renamed argument
		local delta

--DEFAULT_CHAT_FRAME:AddMessage("Reply inc  "..Recap_FormatTime((FightStart or 0)/1000).." "..Recap_FormatTime((FightEnd or 0)/1000).." "..string_format("%.3f", (TotalDuration or 0)/1000))
--DEFAULT_CHAT_FRAME:AddMessage("Reply mine "..Recap_FormatTime((recap[recap_temp.p].LastFightStart or 0)).." "..Recap_FormatTime((recap[recap_temp.p].LastFightEnd or 0)).." "..string_format("%.3f", (recap[recap_temp.p].TotalDuration or 0)))

		-- check whether our interval is entirely earlier than the Reply interval, or entirely later, or intersecting with the Reply interval
		if (recap[recap_temp.p].LastFightStart == nil) or (recap[recap_temp.p].LastFightStart == 0) or
		   (recap[recap_temp.p].LastFightEnd == nil) or (recap[recap_temp.p].LastFightEnd == 0) or
		   (FightStart and (FightStart > math_floor(1000*recap[recap_temp.p].LastFightEnd))) then
			-- we are entirely earlier (or have no fight yet), very probably not the same 'last fight'; play 'catch up' by adopting the Reply start and end, and adjusting our TotalDuration
			-- that way we will pick up properly on any Reply messages that update the last fight start and end
			-- we preserve our own last fight duration since sync is for All Fights
			-- not ideal, but relatively safe for most fight situations
			if FightStart and (FightStart > 0) and FightEnd and (FightEnd > 0) then
				delta = FightEnd/1000 - FightStart/1000
				recap[recap_temp.p].LastFightStart = FightStart/1000
				recap[recap_temp.p].LastFightEnd = FightEnd/1000
				recap[recap_temp.p].TotalDuration = (recap[recap_temp.p].TotalDuration or 0) + delta
			end
		elseif FightEnd and (FightEnd < math_floor(1000*recap[recap_temp.p].LastFightStart)) then
			-- we are entirely later, very probably not the same 'last fight', we're probably in our own little punch-up elsewhere; ignore entirely
			-- not ideal, but relatively safe for most fight situations
		else
			-- very probably a matching 'last fight'
			-- if the incoming start value is earlier, or if the incoming end value is later, it replaces our value
			-- the times come in as integer thousandths for more compact transmission
			if FightStart and (FightStart <= math_floor(1000*recap[recap_temp.p].LastFightStart)) then
				delta = recap[recap_temp.p].LastFightStart - FightStart/1000
				recap[recap_temp.p].LastFightStart = recap[recap_temp.p].LastFightStart - delta
				recap[recap_temp.p].LastDuration = recap[recap_temp.p].LastDuration + delta
				recap[recap_temp.p].TotalDuration = (recap[recap_temp.p].TotalDuration or 0) + delta
			end
			if FightEnd and (FightEnd >= math_floor(1000*recap[recap_temp.p].LastFightEnd)) then
				delta = FightEnd/1000 - recap[recap_temp.p].LastFightEnd
				recap[recap_temp.p].LastFightEnd = recap[recap_temp.p].LastFightEnd + delta
				recap[recap_temp.p].LastDuration = recap[recap_temp.p].LastDuration + delta
				recap[recap_temp.p].TotalDuration = (recap[recap_temp.p].TotalDuration or 0) + delta
			end

			-- compare the leader's total duration (we may have already adjusted our own in the preceding code)
			if TotalDuration and (TotalDuration > 0) then
				if TotalDuration > math_floor(1000*(recap[recap_temp.p].TotalDuration or 0)) then
					recap[recap_temp.p].TotalDuration = TotalDuration/1000
				end
			end
		end

	else

		-- ordinary combatant
		if not recap.Combatant[NameGUID] then
			-- new combatant
			Recap_CreateBlankCombatant(NameGUID, Seen, Flags)
		end

		-- if an incoming value is false, it is ignored
		-- if it is larger than our value, it replaces our value
		-- if our value is larger, we don't reply (we would have done so in response to the initial Summary)
		-- the times come in as integer thousandths for more compact transmission
		local iCombatant = recap.Combatant[NameGUID]
		if Time and (Time > 0) then
			if Time > math_floor(1000*(iCombatant.TotalTime or 0)) then
				iCombatant.TotalTime = Time/1000
			end
		end
		if TimeIn and (TimeIn > 0) then
			if TimeIn > math_floor(1000*(iCombatant.TotalTimeIn or 0)) then
				iCombatant.TotalTimeIn = TimeIn/1000
			end
		end
		if TimeHeal and (TimeHeal > 0) then
			if TimeHeal > math_floor(1000*(iCombatant.TotalTimeHeal or 0)) then
				iCombatant.TotalTimeHeal = TimeHeal/1000
			end
		end
		if DmgOut and (DmgOut > 0) then
			if DmgOut > (iCombatant.TotalDmgOut or 0) then
				iCombatant.TotalDmgOut = DmgOut
			end
		end
		if DmgIn and (DmgIn > 0) then
			if DmgIn > (iCombatant.TotalDmgIn or 0) then
				iCombatant.TotalDmgIn = DmgIn
			end
		end
		if RawHeal and (RawHeal > 0) then
			if RawHeal > (iCombatant.TotalRawHeal or 0) then
				iCombatant.TotalRawHeal = RawHeal
			end
		end
		if OverHeal and (OverHeal > 0) then
			if OverHeal > (iCombatant.TotalOverHeal or 0) then
				iCombatant.TotalOverHeal = OverHeal
			end
		end
		if MaxHit and (MaxHit > 0) then
			if MaxHit > (iCombatant.TotalMaxHit or 0) then
				iCombatant.TotalMaxHit = MaxHit
			end
		end
		if Kills and (Kills > 0) then
			if Kills > (iCombatant.TotalKills or 0) then
				iCombatant.TotalKills = Kills
			end
		end
		if Seen and (Seen > 0) then
			if Seen < (iCombatant.Seen or 0) then
				iCombatant.Seen = Seen
			end
		end
		if Flags and (Flags > 0) then
			-- we accept sender's flags, which may not always be the latest value (tough)
			iCombatant.Flags = Flags
		end

		-- silently adjust overhealing number if necessary (could leave us out of strict sync)
		if iCombatant.TotalOverHeal and (iCombatant.TotalOverHeal > 0) and iCombatant.TotalRawHeal and (iCombatant.TotalRawHeal > 0) then
			iCombatant.TotalOverHeal = math_min(iCombatant.TotalOverHeal, iCombatant.TotalRawHeal)
		end

		-- derivative calculated values
		if iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
			iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
		end
		if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
			iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
		end
		if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and (((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)) > 0) then
			iCombatant.TotalHPS = Recap_Div1(((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)), iCombatant.TotalTimeHeal)
		end

	end

	-- remember the local time of the latest update activity
	recap.Opt.SyncUpdateWhen.value = math_floor(1000*GetTime()) -- integer thousandths
	Recap_SetSyncDisplay()

	-- start a timer to detect when synchronization activity is over
	recap_temp.EndSyncDelayTimer = 0

	-- turn on sync status light blink
	if recap_temp.SyncStatusLightBlink == 0 then
		recap_temp.SyncStatusLightBlink = 1
	end
end

function Recap_SendSyncHeartbeat()
	if not Recap:SendCommMessage("GROUP", recap_temp.Heartbeat, Recap_Sync_Version, recap.Opt.SyncTimestamp.value, recap.Opt.SyncLeader.value, recap.Opt.SyncMergePets.value,
								 recap_temp.PlayerGUID, recap_temp.Server, Recap_Version) then
		-- not sent
		DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendHeartbeat)
	end
end

function Recap_WarnNotAllowedDuringSync()
	StaticPopupDialogs["recap_temp.CONFIRMDATAMODE"] = {
		text = TEXT(recap_temp.Localize.NotAllowedDuringSync),
		button1 = TEXT(OKAY),
		OnAccept = function()
		end,
		timeout = 0,
		showAlert = 1,
		whileDead = 1
	}
	StaticPopup_Show("recap_temp.CONFIRMDATAMODE")
end

function Recap_UpdateSyncMember(name, server, recapVersion)
	if not recap_temp.SyncMembers[name..server] then
		-- new
		recap_temp.SyncMembers[name..server] = {}
	end
	local iMember = recap_temp.SyncMembers[name..server]
	iMember.Name = name
	iMember.Server = server
	iMember.RecapVersion = recapVersion
	iMember.When = math_floor(1000*GetTime())
	iMember.Ago = 0
end

function Recap_ListSyncMembers()

	local list_time, i, index, iMember, inactive

	list_time = GetTime()
	index = 0
	DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.SyncMembers)
	-- three passes
	-- first calculate time ago in seconds
	for i in pairs(recap_temp.SyncMembers) do
		iMember = recap_temp.SyncMembers[i]
		iMember.Ago = list_time-(iMember.When/1000)
	end
	-- second list members active within the past ten seconds
	for i in pairs(recap_temp.SyncMembers) do
		iMember = recap_temp.SyncMembers[i]
		if iMember.Ago <= 10 then
			index = index + 1
			DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..tostring(index)..": "..Recap_NameOnlyFromCombatant(iMember.Name).." ("..iMember.Server..") ["..tostring(iMember.RecapVersion).."] "..recap_temp.Localize.LastUpdateWas.." "..Recap_FormatTime(list_time-(iMember.When/1000)).." "..recap_temp.Localize.Ago)
		end
	end
	-- third list members not active within the past ten seconds
	inactive = false
	for i in pairs(recap_temp.SyncMembers) do
		iMember = recap_temp.SyncMembers[i]
		if iMember.Ago > 10 then
			index = index + 1
			if inactive == false then
				inactive = true
				DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..":  "..recap_temp.Localize.InactiveSyncMembers)
			end
			DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..tostring(index)..": "..Recap_NameOnlyFromCombatant(iMember.Name).." ("..iMember.Server..") ["..tostring(iMember.RecapVersion).."] "..recap_temp.Localize.LastUpdateWas.." "..Recap_FormatTime(list_time-(iMember.When/1000)).." "..recap_temp.Localize.Ago)
		end
	end
end

-- remove from the list members who have not been active within the past ten seconds
function Recap_SyncClearInactiveMembers()

	local list_time, i, j, iMember

	list_time = GetTime()
	-- calculate time ago in seconds
	for i in pairs(recap_temp.SyncMembers) do
		iMember = recap_temp.SyncMembers[i]
		iMember.Ago = list_time-(iMember.When/1000)
	end
	-- delete members not active within the past ten seconds
	j = 0
	for i in pairs(recap_temp.SyncMembers) do
		iMember = recap_temp.SyncMembers[i]
		if iMember.Ago > 10 then
			recap_temp.SyncMembers[i] = nil
			j = j + 1
		end
	end
	DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..":  "..tostring(j).." "..recap_temp.Localize.InactiveSyncMembers.." "..recap_temp.Localize.Removed)
end

-- skip the next fight (or the rest of the current one if already in combat)
function Recap_SkipNextFight()

	if (recap.Opt.State.value == "Stopped") then
		-- we've got Recap turned off at the moment
		PlaySound("igQuestFailed")
		return
	end

	-- paranoia check
	if (recap.Opt.SyncState.value == "Member") then
		-- bail if Member
		PlaySound("igQuestFailed")
		return
	end

	-- skip our own next fight (works for people not in sync)
	recap.Opt.SkipNextFight.value = true
	DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..":  "..recap_temp.Localize.SkipNextFightBegins)

	if (recap.Opt.SyncState.value == "Leader") then
		-- if Leader, tell others to skip next fight

		-- lock OnCommReceived event
		if recap_temp.OnCommReceived == true then
			PlaySound("igQuestFailed")
			return
		end
		recap_temp.OnCommReceived = true

		if not Recap:SendCommMessage("GROUP", recap_temp.SkipNextFight, Recap_Sync_Version, recap.Opt.SyncTimestamp.value, recap.Opt.SyncLeader.value, recap.Opt.SyncMergePets.value) then
			-- not sent
			DEFAULT_CHAT_FRAME:AddMessage(recap_temp.Recap..": "..recap_temp.Localize.FailSendSkipNextFight)
		end

		-- unlock OnCommReceived event
		recap_temp.OnCommReceived = false
	end
end

function Recap_SyncExtractNameGUID(nameServer)
	local name
	-- format is name_GUID_server
	_, _, name = string_find(nameServer, "^(.+)_.-$")
	return name
end

function Recap_SyncExtractServer(nameServer)
	local server
	-- format is name_GUID_server
	_, _, server = string_find(nameServer, "^.+_(.-)$")
	return server
end

RecapSync_lua_411 = true
