﻿--[[ Fight functions ]]

local _G = getfenv(0)
local bit_band = _G.bit.band
local bit_bor = _G.bit.bor
local math_floor = _G.math.floor
local math_max = _G.math.max
local math_min = _G.math.min
local string_find = _G.string.find
local string_format = _G.string.format
local string_gmatch = _G.string.gmatch
local string_gsub = _G.string.gsub
local string_lower = _G.string.lower
local string_sub = _G.string.sub
local string_upper = _G.string.upper

function Recap_CreateCombatantAndUpdateTimers(thisCombatant, timestamp, flags, timeType)

	local iLast

	-- guard code
	if not thisCombatant then
		return
	end

	-- if thisCombatant not a .Combatant, add an entry for them
	if not recap.Combatant[thisCombatant] then
		Recap_CreateBlankCombatant(thisCombatant, math_floor(1000*timestamp), flags)
	end

	-- if they haven't fought this session, add a .Last entry for them (only do it once, we get called repeatedly)
	if not recap_temp.Last[thisCombatant] then
		Recap_InitializeLastFight(thisCombatant)
	end

	-- If we get here for any reason we might as well start the overall fight timers
	--   (Recap 3.32 would only do so for a source doing damage, or for a spellcast start)
	--   (Is healing after a fight, thus starting a 'new' fight, any different from someone ganking a toad?)
	-- now three independent overall fight timers
	if recap_temp.InFriend[thisCombatant] then
		-- overall fight timers should be for group only -- this has been in error for years !
		if timeType == "Out" then
			if recap_temp.FightStart==0 then
				recap_temp.FightStart = timestamp
			end
			recap_temp.FightEnd = timestamp
		end
		if timeType == "In" then
			if recap_temp.FightStartIn==0 then
				recap_temp.FightStartIn = timestamp
			end
			recap_temp.FightEndIn = timestamp
		end
		if timeType == "Heal" then
			if recap_temp.FightStartHeal==0 then
				recap_temp.FightStartHeal = timestamp
			end
			recap_temp.FightEndHeal = timestamp
		end
	end

	-- three independent timers per combatant for damage dealt, damage received, and healing done
	--   (at the moment no timer for healing received)
	iLast = recap_temp.Last[thisCombatant]
	if timeType == "Out" then
		if iLast.StartOut==0 then
			iLast.StartOut = timestamp
		end
		iLast.EndOut = timestamp
	elseif timeType == "In" then
		if iLast.StartIn==0 then
			iLast.StartIn = timestamp
		end
		iLast.EndIn = timestamp
	elseif timeType == "Heal" then
		if iLast.StartHeal==0 then
			iLast.StartHeal = timestamp
		end
		iLast.EndHeal = timestamp
	end

	-- anyone who does damage, takes damage, heals, or receives heals will be flagged as having been in the fight
	recap.Combatant[thisCombatant].WasInCurrent = true

	-- remember any of the player's pets who are involved in the Last fight (to optimize live update calculations)
	if Recap_IsOwnedBy(thisCombatant, recap_temp.PlayerGUID) then
		recap_temp.LastPlayerPets[thisCombatant] = true
	end
end

function Recap_StartFight()

	local i, iFriend, overhealing

	if recap.Opt.AutoHide.value and not recap.Opt.Minimized.value then
		RecapFrame_Hide()
		RecapPanel_Hide(1)
		RecapRecent_Hide(1)
		RecapOptFrame:Hide()
	end

	-- check group at start of fight to get HP for more accurate calculation of overhealing, and to get group pet information.
	Recap_MakeFriends()

	-- only clear WasInCurrent once per fight (also used for other once-per-fight actions)
	if not recap_temp.ResetWasInCurrent then

		-- silent repair
		if not recap.Combatant then
			recap.Combatant = {}
		end
		for i in pairs(recap.Combatant) do
			if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) then
				recap.Combatant[i].WasInCurrent = false
			end
		end

		-- clear list of player pets who are active in the last fight
		recap_temp.LastPlayerPets = {}

		-- clear any existing synchronization update data (done outside the spinlock, I'm hoping that's safe)
		recap_temp.SyncData = {}

		-- pre-calculate some numbers that will be used for live update calculations
		recap_temp.TotalGroupDmgIn = 0
		recap_temp.TotalGroupDmgOut = 0
		recap_temp.TotalGroupHeal = 0
		for i in pairs(recap_temp.InFriend) do
			iFriend = recap.Combatant[i]
			if iFriend then
				recap_temp.TotalGroupDmgIn = recap_temp.TotalGroupDmgIn + (iFriend.TotalDmgIn or 0)
				recap_temp.TotalGroupDmgOut = recap_temp.TotalGroupDmgOut + (iFriend.TotalDmgOut or 0)
				overhealing = math_min((iFriend.TotalOverHeal or 0), (iFriend.TotalRawHeal or 0))
				recap_temp.TotalGroupHeal = recap_temp.TotalGroupHeal + (iFriend.TotalRawHeal or 0) - overhealing
			end
		end

		-- clear count of casts of Prayer of Mending (includes a total so we don't have to scan it repeatedly)
		-- TODO: not completely right, doesn't properly handle e.g. battlegrounds where some allies might not be in your group
		recap_temp.PoMCasts = {}
		recap_temp.PoMTotalCastsGroup = 0
		recap_temp.PoMTotalCastsNonGroup = 0

		-- clear count of casts of Lifebloom (includes a total so we don't have to scan it repeatedly)
		recap_temp.LBCasts = {}
		recap_temp.LBTotalCastsGroup = 0
		recap_temp.LBTotalCastsNonGroup = 0

		-- clear count of casts of Earth Shield (includes a total so we don't have to scan it repeatedly)
		recap_temp.ESCasts = {}
		recap_temp.ESTotalCastsGroup = 0
		recap_temp.ESTotalCastsNonGroup = 0

		-- clear count of casts of Judgement of Light (includes a total so we don't have to scan it repeatedly)
		recap_temp.JoLCasts = {}
		recap_temp.JoLTotalCastsGroup = 0
		recap_temp.JoLTotalCastsNonGroup = 0

		-- clear count of casts of Improved Leader of the Pack (includes a total so we don't have to scan it repeatedly)
		recap_temp.ILotPCasts = {}
		recap_temp.ILotPTotalCastsGroup = 0
		recap_temp.ILotPTotalCastsNonGroup = 0

		-- mark that we've now done this once-only stuff
		recap_temp.ResetWasInCurrent = true
	end

	-- start idle timers
	if not recap.Opt.LimitFights.value and recap.Opt.IdleFight.value then
		recap_temp.IdleTimer = 0
	end
	if not recap.Opt.LimitFights.value then
		recap_temp.HiddenIdleTimer = 0
	end

	-- make sure end of fight delay timer is off
	recap_temp.EndFightDelayTimer = -1

	Recap_SetState("Active")
end

function Recap_AddPetToOwner(pet, owner)

	local combatantPet = recap.Combatant[pet]
	local combatantOwner = recap.Combatant[owner]
	local lastPet = recap_temp.Last[pet]
	local lastOwner = recap_temp.Last[owner]

	if not combatantPet then
		return
	end
	if not combatantOwner then
		return
	end
	if not lastPet then
		return
	end
	if not lastOwner then
		return
	end

	-- now three independent times, for damage, incoming, and healing
	if (combatantPet.WasInCurrent and (lastPet.StartOut > 0)) and
	   (combatantOwner.WasInCurrent and (lastOwner.StartOut > 0)) then
		-- both pet and owner were in the fight and did damage
		lastOwner.StartOut = math_min(lastPet.StartOut,lastOwner.StartOut)
		lastOwner.EndOut = math_max(lastPet.EndOut,lastOwner.EndOut)
	elseif (combatantPet.WasInCurrent and (lastPet.StartOut > 0)) and
		   (not combatantOwner.WasInCurrent or (lastOwner.StartOut == 0)) then
		-- the pet was in the fight and did damage, the owner did no damage
		lastOwner.StartOut = lastPet.StartOut
		lastOwner.EndOut = lastPet.EndOut
	end
	if (combatantPet.WasInCurrent and (lastPet.StartIn > 0)) and
	   (combatantOwner.WasInCurrent and (lastOwner.StartIn > 0)) then
		-- both pet and owner were in the fight and took damage
		lastOwner.StartIn = math_min(lastPet.StartIn,lastOwner.StartIn)
		lastOwner.EndIn = math_max(lastPet.EndIn,lastOwner.EndIn)
	elseif (combatantPet.WasInCurrent and (lastPet.StartIn > 0)) and
		   (not combatantOwner.WasInCurrent or (lastOwner.StartIn == 0)) then
		-- the pet was in the fight and took damage, the owner took no damage
		lastOwner.StartIn = lastPet.StartIn
		lastOwner.EndIn = lastPet.EndIn
	end
	if (combatantPet.WasInCurrent and (lastPet.StartHeal > 0)) and
	   (combatantOwner.WasInCurrent and (lastOwner.StartHeal > 0)) then
		-- both pet and owner were in the fight and did healing
		lastOwner.StartHeal = math_min(lastPet.StartHeal,lastOwner.StartHeal)
		lastOwner.EndHeal = math_max(lastPet.EndHeal,lastOwner.EndHeal)
	elseif (combatantPet.WasInCurrent and (lastPet.StartHeal > 0)) and
		   (not combatantOwner.WasInCurrent or (lastOwner.StartHeal == 0)) then
		-- the pet was in the fight and did healing, the owner did no healing
		lastOwner.StartHeal = lastPet.StartHeal
		lastOwner.EndHeal = lastPet.EndHeal
	end

	lastOwner.DmgIn = lastOwner.DmgIn + lastPet.DmgIn
	lastOwner.DmgOut = lastOwner.DmgOut + lastPet.DmgOut
	lastOwner.MaxHit = math_max(lastOwner.MaxHit,lastPet.MaxHit)
	-- no longer copy pet deaths (misleadingly called Kills) up to owner
	-- lastOwner.Kills = lastOwner.Kills + lastPet.Kills
	lastOwner.RawHeal = lastOwner.RawHeal + lastPet.RawHeal
	lastOwner.OverHeal = math_min((lastOwner.OverHeal + lastPet.OverHeal),lastOwner.RawHeal)

	lastPet.StartOut = 0
	lastPet.EndOut = 0
	lastPet.StartIn = 0
	lastPet.EndIn = 0
	lastPet.StartHeal = 0
	lastPet.EndHeal = 0
	lastPet.DmgIn = 0
	lastPet.DmgOut = 0
	lastPet.MaxHit = 0
	lastPet.Kills = 0
	lastPet.RawHeal = 0
	lastPet.OverHeal = 0

	combatantPet.WasInCurrent = false
	combatantOwner.WasInCurrent = true -- if the owner wasn't in the fight, the owner is now

end

function Recap_AccumulateCombatant(i, iCombatant, iLast)

	local k, lastDmgIn, lastDmgOut, lastTimeOut, lastTimeIn, lastTimeHeal, lastMaxHit, lastKills, lastRawHeal, lastOverHeal

	lastDmgIn = iLast.DmgIn
	lastDmgOut = iLast.DmgOut
	-- these three if statements shouldn't be necessary, but I'm getting paranoid about the reported bad Time values
	if iLast.StartOut > 0 and (iLast.EndOut > iLast.StartOut) then
		lastTimeOut = iLast.EndOut - iLast.StartOut
	else
		lastTimeOut = 0
	end
	if iLast.StartIn > 0 and (iLast.EndIn > iLast.StartIn) then
		lastTimeIn = iLast.EndIn - iLast.StartIn
	else
		lastTimeIn = 0
	end
	if iLast.StartHeal > 0 and (iLast.EndHeal > iLast.StartHeal) then
		lastTimeHeal = iLast.EndHeal - iLast.StartHeal
	else
		lastTimeHeal = 0
	end
	lastMaxHit = iLast.MaxHit
	lastKills = iLast.Kills
	lastRawHeal = iLast.RawHeal
	lastOverHeal = math_min(iLast.OverHeal,lastRawHeal)

	-- If there is non-zero damage or healing, force the matching time to be
	--   (somewhat arbitrarily) at least as large as the global cooldown ("DeemedTime").
	--   This gives some duration to single-shot fights (which are accumulated), keeping
	--   the anomalous DPS down a bit, without displaying them or their DPS.
	if (lastDmgOut > 0) and (lastTimeOut < recap_temp.DeemedTime) then
		lastTimeOut = recap_temp.DeemedTime
	end
	if (lastDmgIn > 0) and (lastTimeIn < recap_temp.DeemedTime) then
		lastTimeIn = recap_temp.DeemedTime
	end
	if (lastRawHeal > 0) and (lastTimeHeal < recap_temp.DeemedTime) then
		lastTimeHeal = recap_temp.DeemedTime
	end

	if recap_temp.LastFightSignificant then
		-- put the Last Fight summary data for this combatant where the display code can see it
		k = recap_temp.ActiveLastFight
		if (lastDmgIn > 0) then
			iCombatant["LastDmgIn_"..k] = lastDmgIn
		end
		if (lastDmgOut > 0) then
			iCombatant["LastDmgOut_"..k] = lastDmgOut
		end
		if (lastTimeOut > 0) then
			iCombatant["LastTime_"..k] = lastTimeOut
		end
		if (lastTimeIn > 0) then
			iCombatant["LastTimeIn_"..k] = lastTimeIn
		end
		if (lastTimeHeal > 0) then
			iCombatant["LastTimeHeal_"..k] = lastTimeHeal
		end
		if (lastMaxHit > 0) then
			iCombatant["LastMaxHit_"..k] = lastMaxHit
		end
		if (lastKills > 0) then
			iCombatant["LastKills_"..k] = lastKills
		end
		if (lastRawHeal > 0) then
			iCombatant["LastRawHeal_"..k] = lastRawHeal
		end
		if (lastOverHeal > 0) and (lastRawHeal > 0) then
			iCombatant["LastOverHeal_"..k] = math_min(lastOverHeal, lastRawHeal)
		end
	end

	iLast.StartOut = 0
	iLast.EndOut = 0
	iLast.StartIn = 0
	iLast.EndIn = 0
	iLast.StartHeal = 0
	iLast.EndHeal = 0
	iLast.DmgIn = 0
	iLast.DmgOut = 0
	iLast.MaxHit = 0
	iLast.Kills = 0
	iLast.RawHeal = 0
	iLast.OverHeal = 0

	if (lastDmgIn > 0) then
		iCombatant.TotalDmgIn = (iCombatant.TotalDmgIn or 0) + lastDmgIn
	end
	if (lastDmgOut > 0) then
		iCombatant.TotalDmgOut = (iCombatant.TotalDmgOut or 0) + lastDmgOut
	end
	if (lastTimeOut > 0) then
		iCombatant.TotalTime = (iCombatant.TotalTime or 0) + lastTimeOut
	end
	if (lastTimeIn > 0) then
		iCombatant.TotalTimeIn = (iCombatant.TotalTimeIn or 0) + lastTimeIn
	end
	if (lastTimeHeal > 0) then
		iCombatant.TotalTimeHeal = (iCombatant.TotalTimeHeal or 0) + lastTimeHeal
	end
	if (lastMaxHit > 0) then
		iCombatant.TotalMaxHit = math_max((iCombatant.TotalMaxHit or 0), lastMaxHit)
	end
	if (lastKills > 0) then
		iCombatant.TotalKills = (iCombatant.TotalKills or 0) + lastKills
	end
	if (lastRawHeal > 0) then
		iCombatant.TotalRawHeal = (iCombatant.TotalRawHeal or 0) + lastRawHeal
	end
	if (lastOverHeal > 0) then
		iCombatant.TotalOverHeal = math_min(((iCombatant.TotalOverHeal or 0) + lastOverHeal), (iCombatant.TotalRawHeal or 0))
	end

	-- three flavours of "per second" calculation for this combatant
	if recap_temp.LastFightSignificant then
		if (lastTimeOut > recap_temp.MinTime) and (lastDmgOut > 0) then
			iCombatant["LastDPS_"..recap_temp.ActiveLastFight] = Recap_Div1(lastDmgOut, lastTimeOut)
		end
		if (lastTimeIn > recap_temp.MinTime) and (lastDmgIn > 0) then
			iCombatant["LastDPSIn_"..recap_temp.ActiveLastFight] = Recap_Div1(lastDmgIn, lastTimeIn)
		end
		if (lastTimeHeal > recap_temp.MinTime) and ((lastRawHeal - lastOverHeal) > 0) then
			iCombatant["LastHPS_"..recap_temp.ActiveLastFight] = Recap_Div1((lastRawHeal - lastOverHeal), lastTimeHeal)
		end
	end
	if iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
		iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
	end
	if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
		iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
	end
	if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and (((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)) > 0) then
		iCombatant.TotalHPS = Recap_Div1(((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)), iCombatant.TotalTimeHeal)
	end

	iCombatant.WasInCurrent = false
	if recap_temp.LastFightSignificant then
		iCombatant.WasInLast = true
	end
end


-- used to remove pet effects when Merge Pets has been set to off, after having been on
function Recap_RemovePetEffects()

	local i, j, effectNum

	-- silent repair
	if not recap.Combatant then
		recap.Combatant = {}
	end
	for i in pairs(recap.Combatant) do
		if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) then
			local iCombatant = recap.Combatant[i]
			if iCombatant.OutgoingDetail then
				for j in pairs(iCombatant.OutgoingDetail) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.OutgoingDetail[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.TargetDetail then
				for j in pairs(iCombatant.TargetDetail) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") then
						-- remove pet effect
						iCombatant.TargetDetail[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.IncomingDetail then
				for j in pairs(iCombatant.IncomingDetail) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.IncomingDetail[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.SourceDetail then
				for j in pairs(iCombatant.SourceDetail) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") then
						-- remove pet effect
						iCombatant.SourceDetail[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.OtherDetail then
				for j in pairs(iCombatant.OtherDetail) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.OtherDetail[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastOutgoingDetail_1 then
				for j in pairs(iCombatant.LastOutgoingDetail_1) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.LastOutgoingDetail_1[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastTargetDetail_1 then
				for j in pairs(iCombatant.LastTargetDetail_1) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") then
						-- remove pet effect
						iCombatant.LastTargetDetail_1[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastIncomingDetail_1 then
				for j in pairs(iCombatant.LastIncomingDetail_1) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.LastIncomingDetail_1[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastSourceDetail_1 then
				for j in pairs(iCombatant.LastSourceDetail_1) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") then
						-- remove pet effect
						iCombatant.LastSourceDetail_1[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastOtherDetail_1 then
				for j in pairs(iCombatant.LastOtherDetail_1) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.LastOtherDetail_1[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastOutgoingDetail_2 then
				for j in pairs(iCombatant.LastOutgoingDetail_2) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.LastOutgoingDetail_2[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastTargetDetail_2 then
				for j in pairs(iCombatant.LastTargetDetail_2) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") then
						-- remove pet effect
						iCombatant.LastTargetDetail_2[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastIncomingDetail_2 then
				for j in pairs(iCombatant.LastIncomingDetail_2) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.LastIncomingDetail_2[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastSourceDetail_2 then
				for j in pairs(iCombatant.LastSourceDetail_2) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") then
						-- remove pet effect
						iCombatant.LastSourceDetail_2[j] = nil -- keep as nil
					end
				end
			end
			if iCombatant.LastOtherDetail_2 then
				for j in pairs(iCombatant.LastOtherDetail_2) do
					effectNum = string_sub(j,1,1)
					if (effectNum == "2") or (effectNum == "4") or (effectNum == "6") or (effectNum == "8") then
						-- remove pet effect
						iCombatant.LastOtherDetail_2[j] = nil -- keep as nil
					end
				end
			end
		end
	end
	-- and personal details
	-- with 3.67 this next bit is now only for cleanup: no new effects of this type will be created
	for j in pairs(recap.Self[recap_temp.s]) do
		effectNum = string_sub(j,1,1)
		if (effectNum == "2") or (effectNum == "4") then
			-- remove pet effect
			recap.Self[recap_temp.s][j] = nil -- keep as nil
		end
	end
end

-- that the format of thisPet is "owner:owner:pet" or "owner:pet"
-- we are always adding from the pet last fight to the owner
function Recap_AddPetDetailsToOwner(thisPet, thisOwner, ownerAllLastOutgoingDetail, ownerAllLastIncomingDetail, ownerAllLastOtherDetail, ownerAllLastTargetDetail, ownerAllLastSourceDetail,
									petAllLastOutgoingDetail, petAllLastIncomingDetail, petAllLastOtherDetail, petAllLastTargetDetail, petAllLastSourceDetail)

	local i, j
	local petEffect
	local iType, iEffect, iTarget, iSource
	local petDetail, ownerDetail
	local combatantPet = recap.Combatant[thisPet]
	local slimPet = Recap_StripGUIDsFromCombatant(thisPet)

	if not recap.Opt.LightData.value then
		if combatantPet[petAllLastOutgoingDetail] then
			for i in pairs(combatantPet[petAllLastOutgoingDetail]) do
				iType = tonumber(string_sub(i,1,1))
				iEffect = string_sub(i,2)
				petEffect = tostring(iType+1)..slimPet..": "..iEffect
				petDetail = combatantPet[petAllLastOutgoingDetail][i]
				if not recap.Combatant[thisOwner] then
					Recap_CreateBlankCombatant(thisOwner, combatantPet.Seen, nil)
					if not recap_temp.Last[thisOwner] then
						Recap_InitializeLastFight(thisOwner)
					end
				end
				if not recap.Combatant[thisOwner][ownerAllLastOutgoingDetail] then
					recap.Combatant[thisOwner][ownerAllLastOutgoingDetail] = {}
				end
				if not recap.Combatant[thisOwner][ownerAllLastOutgoingDetail][petEffect] then
					recap.Combatant[thisOwner][ownerAllLastOutgoingDetail][petEffect] = {}
				end
				ownerDetail = recap.Combatant[thisOwner][ownerAllLastOutgoingDetail][petEffect]
				Recap_AmalgamatePetIncomingOutgoingDetails(ownerDetail, petDetail)
			end
		end

		if recap.Opt.MatrixData.value then
			if combatantPet[petAllLastTargetDetail] then
				for i in pairs(combatantPet[petAllLastTargetDetail]) do
					iType = tonumber(string_sub(i,1,1))
					iTarget = tostring(iType+1)..string_sub(i,2)
					petDetail = combatantPet[petAllLastTargetDetail][i]
					if not recap.Combatant[thisOwner] then
						Recap_CreateBlankCombatant(thisOwner, combatantPet.Seen, nil)
						if not recap_temp.Last[thisOwner] then
							Recap_InitializeLastFight(thisOwner)
						end
					end
					if petDetail and petDetail.Total and (petDetail.Total > 0) then
						if not recap.Combatant[thisOwner][ownerAllLastTargetDetail] then
							recap.Combatant[thisOwner][ownerAllLastTargetDetail] = {}
						end
						if not recap.Combatant[thisOwner][ownerAllLastTargetDetail][iTarget] then
							recap.Combatant[thisOwner][ownerAllLastTargetDetail][iTarget] = {}
						end
						ownerDetail = recap.Combatant[thisOwner][ownerAllLastTargetDetail][iTarget]
						-- with GUIDs off, a pet named 'Total' will be absorbed into the owner's details
						ownerDetail[slimPet] = (ownerDetail[slimPet] or 0) + petDetail.Total
					end
				end
			end
		end

		if combatantPet[petAllLastIncomingDetail] then
			for i in pairs(combatantPet[petAllLastIncomingDetail]) do
				iType = tonumber(string_sub(i,1,1))
				iEffect = string_sub(i,2)
				petEffect = tostring(iType+1)..slimPet..": "..iEffect
				petDetail = combatantPet[petAllLastIncomingDetail][i]
				if not recap.Combatant[thisOwner] then
					Recap_CreateBlankCombatant(thisOwner, combatantPet.Seen, nil)
					if not recap_temp.Last[thisOwner] then
						Recap_InitializeLastFight(thisOwner)
					end
				end
				if not recap.Combatant[thisOwner][ownerAllLastIncomingDetail] then
					recap.Combatant[thisOwner][ownerAllLastIncomingDetail] = {}
				end
				if not recap.Combatant[thisOwner][ownerAllLastIncomingDetail][petEffect] then
					recap.Combatant[thisOwner][ownerAllLastIncomingDetail][petEffect] = {}
				end
				ownerDetail = recap.Combatant[thisOwner][ownerAllLastIncomingDetail][petEffect]
				Recap_AmalgamatePetIncomingOutgoingDetails(ownerDetail, petDetail)
			end
		end

		if recap.Opt.MatrixData.value then
			if combatantPet[petAllLastSourceDetail] then
				for i in pairs(combatantPet[petAllLastSourceDetail]) do
					iType = tonumber(string_sub(i,1,1))
					iSource = tostring(iType+1)..string_sub(i,2)
					petDetail = combatantPet[petAllLastSourceDetail][i]
					if not recap.Combatant[thisOwner] then
						Recap_CreateBlankCombatant(thisOwner, combatantPet.Seen, nil)
						if not recap_temp.Last[thisOwner] then
							Recap_InitializeLastFight(thisOwner)
						end
					end
					if petDetail and petDetail.Total and (petDetail.Total > 0) then
						if not recap.Combatant[thisOwner][ownerAllLastSourceDetail] then
							recap.Combatant[thisOwner][ownerAllLastSourceDetail] = {}
						end
						if not recap.Combatant[thisOwner][ownerAllLastSourceDetail][iSource] then
							recap.Combatant[thisOwner][ownerAllLastSourceDetail][iSource] = {}
						end
						ownerDetail = recap.Combatant[thisOwner][ownerAllLastSourceDetail][iSource]
						-- with GUIDs off, a pet named 'Total' will be absorbed into the owner's details
						ownerDetail[slimPet] = (ownerDetail[slimPet] or 0) + petDetail.Total
					end
				end
			end
		end

		if recap.Opt.OtherData.value then
			if combatantPet[petAllLastOtherDetail] then
				for i in pairs(combatantPet[petAllLastOtherDetail]) do
					iType = tonumber(string_sub(i,1,1))
					iEffect = string_sub(i,2)
					petEffect = tostring(iType+1)..slimPet..": "..iEffect
					petDetail = combatantPet[petAllLastOtherDetail][i]
					if not recap.Combatant[thisOwner] then
						Recap_CreateBlankCombatant(thisOwner, combatantPet.Seen, nil)
						if not recap_temp.Last[thisOwner] then
							Recap_InitializeLastFight(thisOwner)
						end
					end
					if not recap.Combatant[thisOwner][ownerAllLastOtherDetail] then
						recap.Combatant[thisOwner][ownerAllLastOtherDetail] = {}
					end
					if not recap.Combatant[thisOwner][ownerAllLastOtherDetail][petEffect] then
						recap.Combatant[thisOwner][ownerAllLastOtherDetail][petEffect] = {}
					end
					ownerDetail = recap.Combatant[thisOwner][ownerAllLastOtherDetail][petEffect]
					Recap_AmalgamatePetOtherDetails(ownerDetail, petDetail)
				end
			end
		end
	end
end
function Recap_AmalgamatePetIncomingOutgoingDetails(ownerDetail, petDetail)
	if petDetail.Hits then
		ownerDetail.Hits = (ownerDetail.Hits or 0) + petDetail.Hits
	end
	if petDetail.HitsDmg then
		ownerDetail.HitsDmg = (ownerDetail.HitsDmg or 0) + petDetail.HitsDmg
	end
	if petDetail.HitsMax then
		ownerDetail.HitsMax = math_max((ownerDetail.HitsMax or 0), petDetail.HitsMax)
	end
	if petDetail.Crits then
		ownerDetail.Crits = (ownerDetail.Crits or 0) + petDetail.Crits
	end
	if petDetail.CritsEvents then
		ownerDetail.CritsEvents = (ownerDetail.CritsEvents or 0) + petDetail.CritsEvents
	end
	if petDetail.CritsDmg then
		ownerDetail.CritsDmg = (ownerDetail.CritsDmg or 0) + petDetail.CritsDmg
	end
	if petDetail.CritsMax then
		ownerDetail.CritsMax = math_max((ownerDetail.CritsMax or 0), petDetail.CritsMax)
	end
	if petDetail.Missed then
		ownerDetail.Missed = (ownerDetail.Missed or 0) + petDetail.Missed
	end
	if petDetail.Ticks then
		ownerDetail.Ticks = (ownerDetail.Ticks or 0) + petDetail.Ticks
	end
	if petDetail.TicksDmg then
		ownerDetail.TicksDmg = (ownerDetail.TicksDmg or 0) + petDetail.TicksDmg
	end
	if petDetail.TicksMax then
		ownerDetail.TicksMax = math_max((ownerDetail.TicksMax or 0), petDetail.TicksMax)
	end
	ownerDetail.Element = Recap_MergeElements(ownerDetail.Element, petDetail.Element)
	if ownerDetail.Element and string_find(ownerDetail.Element, "%+", 1, true) and (string_find(ownerDetail.Element, "%?", 1, true) or string_find(ownerDetail.Element, recap_temp.Localize.ElementOther, 1, true)) then
		-- strip any "?" or "Other" that isn't solo
		ownerDetail.Element = string_gsub(ownerDetail.Element, "%?%+", "")
		ownerDetail.Element = string_gsub(ownerDetail.Element, "%+%?", "")
		ownerDetail.Element = string_gsub(ownerDetail.Element, recap_temp.Localize.ElementOther.."%+", "")
		ownerDetail.Element = string_gsub(ownerDetail.Element, "%+"..recap_temp.Localize.ElementOther, "")
	end
	if petDetail.Dodged then
		ownerDetail.Dodged = (ownerDetail.Dodged or 0) + petDetail.Dodged
	end
	if petDetail.Parried then
		ownerDetail.Parried = (ownerDetail.Parried or 0) + petDetail.Parried
	end
	if petDetail.Blocked then
		ownerDetail.Blocked = (ownerDetail.Blocked or 0) + petDetail.Blocked
	end
	if petDetail.Absorbed then
		ownerDetail.Absorbed = (ownerDetail.Absorbed or 0) + petDetail.Absorbed
	end
	if petDetail.Deflected then
		ownerDetail.Deflected = (ownerDetail.Deflected or 0) + petDetail.Deflected
	end
	if petDetail.Evaded then
		ownerDetail.Evaded = (ownerDetail.Evaded or 0) + petDetail.Evaded
	end
	if petDetail.Resisted then
		ownerDetail.Resisted = (ownerDetail.Resisted or 0) + petDetail.Resisted
	end
	if petDetail.Reflected then
		ownerDetail.Reflected = (ownerDetail.Reflected or 0) + petDetail.Reflected
	end
	if petDetail.Immune then
		ownerDetail.Immune = (ownerDetail.Immune or 0) + petDetail.Immune
	end
	if petDetail.ICount then
		ownerDetail.ICount = (ownerDetail.ICount or 0) + petDetail.ICount
	end
	if petDetail.ITotal then
		ownerDetail.ITotal = (ownerDetail.ITotal or 0) + petDetail.ITotal
	end
	if petDetail.Glances then
		ownerDetail.Glances = (ownerDetail.Glances or 0) + petDetail.Glances
	end
	if petDetail.GlancesDmg then
		ownerDetail.GlancesDmg = (ownerDetail.GlancesDmg or 0) + petDetail.GlancesDmg
	end
	if petDetail.GlancesMax then
		ownerDetail.GlancesMax = math_max((ownerDetail.GlancesMax or 0), petDetail.GlancesMax)
	end
	if petDetail.HitsMin then
		ownerDetail.HitsMin = Recap_Min(ownerDetail.HitsMin, petDetail.HitsMin)
	end
	if petDetail.CritsMin then
		ownerDetail.CritsMin = Recap_Min(ownerDetail.CritsMin, petDetail.CritsMin)
	end
	if petDetail.TicksMin then
		ownerDetail.TicksMin = Recap_Min(ownerDetail.TicksMin, petDetail.TicksMin)
	end
	if petDetail.GlancesMin then
		ownerDetail.GlancesMin = Recap_Min(ownerDetail.GlancesMin, petDetail.GlancesMin)
	end
	if petDetail.Crushes then
		ownerDetail.Crushes = (ownerDetail.Crushes or 0) + petDetail.Crushes
	end
	if petDetail.CrushDmg then
		ownerDetail.Hits = (ownerDetail.CrushDmg or 0) + petDetail.CrushDmg
	end
	if petDetail.CrushMax then
		ownerDetail.CrushMax = math_max((ownerDetail.CrushMax or 0), petDetail.CrushMax)
	end
	if petDetail.CrushMin then
		ownerDetail.CrushMin = Recap_Min(ownerDetail.CrushMin, petDetail.CrushMin)
	end
	if petDetail.PAbsorbs then
		ownerDetail.PAbsorbs = (ownerDetail.PAbsorbs or 0) + petDetail.PAbsorbs
	end
	if petDetail.PAbsorbsDmg then
		ownerDetail.PAbsorbsDmg = (ownerDetail.PAbsorbsDmg or 0) + petDetail.PAbsorbsDmg
	end
	if petDetail.PBlocks then
		ownerDetail.PBlocks = (ownerDetail.PBlocks or 0) + petDetail.PBlocks
	end
	if petDetail.PBlocksDmg then
		ownerDetail.PBlocksDmg = (ownerDetail.PBlocksDmg or 0) + petDetail.PBlocksDmg
	end
	if petDetail.PResists25 then
		ownerDetail.PResists25 = (ownerDetail.PResists25 or 0) + petDetail.PResists25
	end
	if petDetail.PResists25Dmg then
		ownerDetail.PResists25Dmg = (ownerDetail.PResists25Dmg or 0) + petDetail.PResists25Dmg
	end
	if petDetail.PResists50 then
		ownerDetail.PResists50 = (ownerDetail.PResists50 or 0) + petDetail.PResists50
	end
	if petDetail.PResists50Dmg then
		ownerDetail.PResists50Dmg = (ownerDetail.PResists50Dmg or 0) + petDetail.PResists50Dmg
	end
	if petDetail.PResists75 then
		ownerDetail.PResists75 = (ownerDetail.PResists75 or 0) + petDetail.PResists75
	end
	if petDetail.PResists75Dmg then
		ownerDetail.PResists75Dmg = (ownerDetail.PResists75Dmg or 0) + petDetail.PResists75Dmg
	end
	if petDetail.EstResistedDmg then
		ownerDetail.EstResistedDmg = (ownerDetail.EstResistedDmg or 0) + petDetail.EstResistedDmg
	end
	if petDetail.EstResistableDmg then
		ownerDetail.EstResistableDmg = (ownerDetail.EstResistableDmg or 0) + petDetail.EstResistableDmg
	end
end
function Recap_AmalgamatePetOtherDetails(ownerDetail, petDetail)
	if petDetail.Hits then
		ownerDetail.Hits = (ownerDetail.Hits or 0) + petDetail.Hits
	end
	if petDetail.Total then
		ownerDetail.Total = (ownerDetail.Total or 0) + petDetail.Total
	end
	if petDetail.Max then
		ownerDetail.Max = math_max((ownerDetail.Max or 0), petDetail.Max)
	end
	if petDetail.Attribute and not ownerDetail.Attribute then
		-- we arbitrarily assume that there is only one possible Attribute for this effect
		ownerDetail.Attribute = petDetail.Attribute
	end
	if petDetail.Missed then
		ownerDetail.Missed = (ownerDetail.Missed or 0) + petDetail.Missed
	end
	if petDetail.Dispels then
		ownerDetail.Dispels = (ownerDetail.Dispels or 0) + petDetail.Dispels
	end
	if petDetail.Steals then
		ownerDetail.Steals = (ownerDetail.Steals or 0) + petDetail.Steals
	end
	if petDetail.ICount then
		ownerDetail.ICount = (ownerDetail.ICount or 0) + petDetail.ICount
	end
	if petDetail.ITotal then
		ownerDetail.ITotal = (ownerDetail.ITotal or 0) + petDetail.ITotal
	end
	if petDetail.DCount then
		ownerDetail.DCount = (ownerDetail.DCount or 0) + petDetail.DCount
	end
	if petDetail.DTotal then
		ownerDetail.DTotal = (ownerDetail.DTotal or 0) + petDetail.DTotal
	end
end

function Recap_EndFight(forceEnd)

	local i, j, k, iPlayer, iCombatant, iLast
	local iDurationOut, iDurationIn, iDurationHeal

	-- turn off idle and end of fight delay timers
	recap_temp.IdleTimer = -1
	recap_temp.HiddenIdleTimer = -1
	recap_temp.EndFightDelayTimer = -1
	recap_temp.EndSyncDelayTimer = -1

	if forceEnd or recap.Opt.State.value=="Active" then

		local debug_time = GetTime()

		Recap_MakeFriends()

		RecapUpdateFrame:Hide() -- disable OnUpdate
		recap.Opt.State.value = "Stopped" -- suspend logging to calculate

		-- silent repairs
		if not recap_temp.Last then
			recap_temp.Last = {}
		end
		if not recap.Combatant then
			recap.Combatant = {}
		end

		if recap.Opt.SkipNextFight.value == true then
			-- we are skipping this fight, so do a whole bunch of nothing
			-- we might have recorded some data before setting the flag, so clear some data that is lying around
			for i in pairs(recap_temp.Last) do
				iCombatant = recap.Combatant[i]
				iCombatant.WasInCurrent = false
				iLast = recap_temp.Last[i]
				iLast.StartOut = 0
				iLast.EndOut = 0
				iLast.StartIn = 0
				iLast.EndIn = 0
				iLast.StartHeal = 0
				iLast.EndHeal = 0
				iLast.DmgIn = 0
				iLast.DmgOut = 0
				iLast.MaxHit = 0
				iLast.Kills = 0
				iLast.RawHeal = 0
				iLast.OverHeal = 0
				-- Last Fight details are not cleared at this time
			end
			-- prevent switching last fight pointers
			recap_temp.LastFightSignificant = false
		else
			-- do the regular end of fight stuff

			-- if we were personally in combat, calculate the live dps and hps numbers one last time (regardless of whether we have plugins or are minimized)
			-- this means that the live numbers are always calculated at least once per fight
			-- note that this will not include any synchronization updates, which happen later
			if recap_temp.PlayerInCombat then
				Recap_UpdateMinimizedDPS()
				if ((IB_Recap_Update or TitanPanelRecap_Update) or (RecapFrame:IsShown() and recap.Opt.Minimized.value)) then
					-- display if we have plugins or are minimized
					Recap_DisplayMinimizedDPS() -- also sends to plugins
				end
				-- turn off the flag that shows whether we were personally in combat
				recap_temp.PlayerInCombat = false
			end

			-- merge pets if appropriate
			if Recap_PetsMerged() then
				-- add all pets to all ultimate owners
				-- data was tracked during a fight to the current ultimate owner, so amalgamate only to the first owner in any string of owners
				-- an entity doing something on its own behalf (when not controlled, and when Recap can work out that it wasn't controlled) will have data of its own
				-- there is a definite benefit to doing this at the end of the fight rather than during the fight, since during the fight ownership may not always be known properly
				for i in pairs(recap_temp.Last) do
					if recap.Combatant[i].WasInCurrent then
						j = Recap_ExtractOwner(i)
						if j then
							Recap_AddPetToOwner(i, j) -- assert: clears pet's WasInCurrent, sets owner's WasInCurrent
							-- the details are always added from the last fight into the owner all fights and into the owner last fight
							k = recap_temp.ActiveLastFight
							Recap_AddPetDetailsToOwner(i, j, "OutgoingDetail", "IncomingDetail", "OtherDetail", "TargetDetail", "SourceDetail",
															 "LastOutgoingDetail_"..k, "LastIncomingDetail_"..k, "LastOtherDetail_"..k, "LastTargetDetail_"..k, "LastSourceDetail_"..k)
							Recap_AddPetDetailsToOwner(i, j, "LastOutgoingDetail_"..k, "LastIncomingDetail_"..k, "LastOtherDetail_"..k, "LastTargetDetail_"..k, "LastSourceDetail_"..k,
															 "LastOutgoingDetail_"..k, "LastIncomingDetail_"..k, "LastOtherDetail_"..k, "LastTargetDetail_"..k, "LastSourceDetail_"..k)
						end
					end
				end
			end

			-- scan all combatants in Last to see if any dealt or took damage for more than the significant time
			--   If so, then we will update combatant Last Fight information
			--   If not, we won't update combatant Last Fight information, but we will still accumulate to the All Fights information
			recap_temp.LastFightSignificant = false
			for i in pairs(recap_temp.Last) do
				iCombatant = recap.Combatant[i]
				if iCombatant.WasInCurrent then
					iLast = recap_temp.Last[i]
					if (iLast.DmgOut > 0) and (iLast.StartOut > 0) and ((iLast.EndOut - iLast.StartOut) > recap_temp.LastFightSignificantTime) then
						-- damage dealt, for more than the significant time
						recap_temp.LastFightSignificant = true
						break
					end
					if (iLast.DmgIn > 0) and (iLast.StartIn > 0) and ((iLast.EndIn - iLast.StartIn) > recap_temp.LastFightSignificantTime) then
						-- damage taken, for more than the significant time
						recap_temp.LastFightSignificant = true
						break
					end
				end
			end

			-- if we are going to have fresh Last Fight data, clear all WasInLast values
			if recap_temp.LastFightSignificant then
				for i in pairs(recap.Combatant) do
					recap.Combatant[i].WasInLast = false
				end
			end

			-- accumulate all combatants who were in the Last Fight
			for i in pairs(recap_temp.Last) do
				iCombatant = recap.Combatant[i]
				iLast = recap_temp.Last[i]
				if iCombatant.WasInCurrent then
-- TODO: this could accumulate e.g. Totems who really should be getting added to their owners via a compound name
-- TODO: somehow we are missing getting the ownership properly -- but there is usually only a single outgoing damage hit, and zero details -- not sure how that is happening
					Recap_AccumulateCombatant(i, iCombatant, iLast) -- assert: clears WasInCurrent, sets WasInLast if LastFightSignificant

					-- special case correction for some dual damage / healing tickers
					Recap_CorrectDualDoTHoT(iCombatant, "OutgoingDetail", "IncomingDetail")
					Recap_CorrectDualDoTHoT(iCombatant, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
				end
				-- ensure that every combatant has their 'Last' record cleared, even if not in current fight
				iLast.StartOut = 0
				iLast.EndOut = 0
				iLast.StartIn = 0
				iLast.EndIn = 0
				iLast.StartHeal = 0
				iLast.EndHeal = 0
				iLast.DmgIn = 0
				iLast.DmgOut = 0
				iLast.MaxHit = 0
				iLast.Kills = 0
				iLast.RawHeal = 0
				iLast.OverHeal = 0

				-- Last Fight details are not cleared at this time
			end

			-- update the overall (All Fights) durations (for group only)
			iPlayer = recap[recap_temp.p]
			if recap_temp.FightStart>0 and (recap_temp.FightEnd > recap_temp.FightStart) then
				iDurationOut = recap_temp.FightEnd - recap_temp.FightStart
			else
				iDurationOut = 0
			end
			if recap_temp.FightStartIn>0 and (recap_temp.FightEndIn > recap_temp.FightStartIn) then
				iDurationIn = recap_temp.FightEndIn - recap_temp.FightStartIn
			else
				iDurationIn = 0
			end
			if recap_temp.FightStartHeal>0 and (recap_temp.FightEndHeal > recap_temp.FightStartHeal) then
				iDurationHeal = recap_temp.FightEndHeal - recap_temp.FightStartHeal
			else
				iDurationHeal = 0
			end
			if recap_temp.LastFightSignificant then
				iPlayer.LastDuration = iDurationOut
				iPlayer.LastDurationIn = iDurationIn
				iPlayer.LastDurationHeal = iDurationHeal
			end
			iPlayer.TotalDuration = iPlayer.TotalDuration + iDurationOut
			iPlayer.TotalDurationIn = iPlayer.TotalDurationIn + iDurationIn
			iPlayer.TotalDurationHeal = iPlayer.TotalDurationHeal + iDurationHeal
			-- save the FightStart and FightEnd values for use in synchronization
			iPlayer.LastFightStart = recap_temp.FightStart
			iPlayer.LastFightEnd = recap_temp.FightEnd
		end

		-- clear temporaries
		recap_temp.FightStart = 0
		recap_temp.FightEnd = 0
		recap_temp.FightStartIn = 0
		recap_temp.FightEndIn = 0
		recap_temp.FightStartHeal = 0
		recap_temp.FightEndHeal = 0

		-- clear effect interval timers
		Recap_ClearEffectIntervalTimers()

		-- if the Last Fight panel is being held, this supports multiple fights happening in the background
		recap_temp.ResetWasInCurrent = false

		-- if Recent Data Mode is active, add an 'end of fight' event (signalled by false values for sourceNameGUID and destNameGUID)
		--   (added even if we are in SkipNextFight)
		if recap.Opt.RecentData.value then
			-- this 'debug_time' is not in sync with Blizzard's combat log event timestamps
			Recap_AddRecentEvent(debug_time, false, false, false, false, false, false, false, false)
		end

		-- special case correction for some dual damage:healing tickers (in active set)
		Recap_CorrectDualDoTHoTForSelf()

		-- the following should happen after all processing of the current active data has been complete (modulo any later changes due to synchronization)
		if recap_temp.LastFightSignificant then
			for i in pairs(recap.Combatant) do
				-- swap the two Last Fight detail data structures, and clear the older one so it can collect the next set of fight data
				iCombatant = recap.Combatant[i]
				if recap_temp.ActiveLastFight == "1" then
					-- we have been storing data in 1, and displaying from 2
					-- clear 2
					iCombatant.LastTime_2 = nil
					iCombatant.LastTimeIn_2 = nil
					iCombatant.LastTimeHeal_2 = nil
					iCombatant.LastMaxHit_2 = nil
					iCombatant.LastDmgIn_2 = nil
					iCombatant.LastDmgOut_2 = nil
					iCombatant.LastHPS_2 = nil
					iCombatant.LastDPSIn_2 = nil
					iCombatant.LastDPS_2 = nil
					iCombatant.LastKills_2 = nil
					iCombatant.LastRawHeal_2 = nil
					iCombatant.LastOverHeal_2 = nil
					iCombatant.LastOutgoingDetail_2 = nil
					iCombatant.LastTargetDetail_2 = nil
					iCombatant.LastIncomingDetail_2 = nil
					iCombatant.LastSourceDetail_2 = nil
					iCombatant.LastOtherDetail_2 = nil
				else
					-- we have been storing data in 2, and displaying from 1
					-- clear 1
					iCombatant.LastTime_1 = nil
					iCombatant.LastTimeIn_1 = nil
					iCombatant.LastTimeHeal_1 = nil
					iCombatant.LastMaxHit_1 = nil
					iCombatant.LastDmgIn_1 = nil
					iCombatant.LastDmgOut_1 = nil
					iCombatant.LastHPS_1 = nil
					iCombatant.LastDPSIn_1 = nil
					iCombatant.LastDPS_1 = nil
					iCombatant.LastKills_1 = nil
					iCombatant.LastRawHeal_1 = nil
					iCombatant.LastOverHeal_1 = nil
					iCombatant.LastOutgoingDetail_1 = nil
					iCombatant.LastTargetDetail_1 = nil
					iCombatant.LastIncomingDetail_1 = nil
					iCombatant.LastSourceDetail_1 = nil
					iCombatant.LastOtherDetail_1 = nil
				end
			end
			if recap_temp.ActiveLastFight == "1" then
				-- swap to storing data in 2, and displaying from 1
				recap_temp.ActiveLastFight = "2"
				recap_temp.DisplayLastFight = "1"
			else
				-- swap to storing data in 1, and displaying from 2
				recap_temp.ActiveLastFight = "1"
				recap_temp.DisplayLastFight = "2"
			end
		end

		if recap.Opt.AutoPost.value then
			recap.Opt.SortBy.value = Recap_AutoPostGetStatID(recap.Opt.AutoPost.Stat)
			recap.Opt.SortDir.value = false
		end

		if (not recap.Opt.Minimized.value) or recap.Opt.AutoPost.value or recap.Opt.AutoLeader.value then
			-- don't construct a list if minimized, unless we are doing one of the auto things which needs the list
			Recap_ConstructList()
		end
		Recap_SetState("Idle")
		Recap_TitleBar()
		RecapHeader_Name:SetText((recap_temp.ListSize-1).." "..RECAP_COMBATANTS)
		Recap_SetTitleBackground()

		if recap_temp.LastFightSignificant then
			if not ((recap.Opt.SyncState.value == "Member") or (recap.Opt.SyncState.value == "Leader")) then
				-- if we're not in synchronization then we do these auto-posts immediately, otherwise we will wait for the EndSyncDelayTimer to expire
				if recap.Opt.AutoPost.value then
					Recap_PostFight()
				end
				if recap.Opt.AutoLeader.value then
					Recap_PostLeader()
				end
			end
		end

		if recap.debug then
			DEFAULT_CHAT_FRAME:AddMessage("(recap) Fight processed in "..string_format("%.3f",GetTime()-debug_time).." secs")
		end

		if recap.Opt.AutoHide.value and not recap.Opt.Minimized.value then
			RecapFrame_Show()
			if recap.Opt.AutoFade.value then
				recap_temp.FadeTimer = 0
			end
		end

		-- force a garbage collection, as this is probably the best time, just after the end of a fight
		if recap.Opt.CollectGarbage.value then
			local totalBefore, totalAfter, recapBefore, recapAfter
			if recap.debug then
				-- enable profiling (wasn't needed until 2.4)
				SetCVar("scriptProfile", 1)
				-- memory before gc
				totalBefore = Recap_Div2(collectgarbage("count"), 1000)
				-- update numbers for all addons
				UpdateAddOnMemoryUsage()
				-- Print memory usage for Recap specifically
				recapBefore = Recap_Div2(GetAddOnMemoryUsage("Recap"), 1000)
				DEFAULT_CHAT_FRAME:AddMessage("(recap) Total addon memory before GC "..string_format("%.2f MB",totalBefore).."  (Recap "..string_format("%.2f MB)",recapBefore))
			end
			local gc_time = GetTime()
			collectgarbage("collect")
			if recap.debug then
				local total, forRecap
				DEFAULT_CHAT_FRAME:AddMessage("(recap) Garbage collection time "..string_format("%.3f secs",GetTime()-gc_time))
				totalAfter = Recap_Div2(collectgarbage("count"), 1000)
				-- update memory amounts for all addons
				UpdateAddOnMemoryUsage()
				recapAfter = Recap_Div2(GetAddOnMemoryUsage("Recap"), 1000)
				DEFAULT_CHAT_FRAME:AddMessage("(recap) Total addon memory after GC "..string_format("%.2f MB",totalAfter).."  (Recap "..string_format("%.2f MB, %.1f%% of total)",recapAfter,Recap_Round1(100*recapAfter/totalAfter)))
				total = totalBefore-totalAfter
				forRecap = recapBefore-recapAfter
				DEFAULT_CHAT_FRAME:AddMessage("(recap) Garbage recycled "..string_format("%.2f MB",total).."  (Recap "..string_format("%.2f MB, %.1f%% of total)",forRecap,Recap_Round1(100*forRecap/total)))
				total = GetScriptCPUUsage()/1000
				-- update CPU times for all addons
				UpdateAddOnCPUUsage()
				forRecap = GetAddOnCPUUsage("Recap")/1000
				DEFAULT_CHAT_FRAME:AddMessage("(recap) Total addon CPU time "..string_format("%.3f secs",total).."  (Recap "..string_format("%.3f secs, %.1f%% of total)",forRecap,Recap_Round1(100*forRecap/total)))
				-- Hawksy: the following is specific to Hawksy's needs -- modify or delete to suit
				if (recap_temp.p == "Hawksy_Elune") or (recap_temp.p == "Galassa_Elune") or (recap_temp.p == "Metrognome_Elune") then
					local text = "(recap) Other addon CPU times as a proportion of total "
					forRecap = Recap_Div2(GetAddOnCPUUsage("Recount"), 1000)
					text = text.."Recount "..string_format("(%.1f%%) ",Recap_Round1(100*forRecap/total))
					forRecap = Recap_Div2(GetAddOnCPUUsage("Omen"), 1000)
					text = text.."Omen "..string_format("(%.1f%%) ",Recap_Round1(100*forRecap/total))
					forRecap = Recap_Div2(GetAddOnCPUUsage("CT_RaidAssist"), 1000)
					text = text.."CT_RaidAssist "..string_format("(%.1f%%) ",Recap_Round1(100*forRecap/total))
					forRecap = Recap_Div2(GetAddOnCPUUsage("Clique"), 1000)
					text = text.."Clique "..string_format("(%.1f%%) ",Recap_Round1(100*forRecap/total))
					forRecap = Recap_Div2(GetAddOnCPUUsage("ItemRack"), 1000)
					text = text.."ItemRack "..string_format("(%.1f%%) ",Recap_Round1(100*forRecap/total))
					forRecap = Recap_Div2((GetAddOnCPUUsage("DBM_API")+GetAddOnCPUUsage("DBM_GUI")+GetAddOnCPUUsage("DBM_Karazhan")+GetAddOnCPUUsage("DBM_Serpentshrine")+GetAddOnCPUUsage("DBM_ZulAman")+GetAddOnCPUUsage("DBM_TheEye")+GetAddOnCPUUsage("DBM_Outlands")), 1000)
					text = text.."DBM "..string_format("(%.1f%%) ",Recap_Round1(100*forRecap/total))
					DEFAULT_CHAT_FRAME:AddMessage(text)
					-- reset CPU profiling numbers at the end of every fight
					ResetCPUUsage()
				end
			end
		end
	end
end

function Recap_ClearAsIfEndOfFight()
	if not recap_temp.Last then
		recap_temp.Last = {}
	end
	for i in pairs(recap_temp.Last) do
		-- ensure that every 'Last' record is cleared
		local iLast = recap_temp.Last[i]
		iLast.StartOut = 0
		iLast.EndOut = 0
		iLast.StartIn = 0
		iLast.EndIn = 0
		iLast.StartHeal = 0
		iLast.EndHeal = 0
		iLast.DmgIn = 0
		iLast.DmgOut = 0
		iLast.MaxHit = 0
		iLast.Kills = 0
		iLast.RawHeal = 0
		iLast.OverHeal = 0

		-- Last Fight details are not cleared at this time
	end
	-- clear effect interval timers
	Recap_ClearEffectIntervalTimers()
	-- turn off idle and end of fight delay timers
	recap_temp.IdleTimer = -1
	recap_temp.HiddenIdleTimer = -1
	recap_temp.EndFightDelayTimer = -1
end

-- clear all of the effect interval timers, since intervals are only meaningful when measured within a single fight
-- the effect duration timers, however, are not reset, since the effects can last between fights
-- but they are for the Last details
function Recap_ClearEffectIntervalTimers()

	local i, j

	-- silent repair
	if not recap.Combatant then
		recap.Combatant = {}
	end
	for i in pairs(recap.Combatant) do
		if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) then
			local iCombatant = recap.Combatant[i]
			if iCombatant.OutgoingDetail then
				for j in pairs(iCombatant.OutgoingDetail) do
					if iCombatant.OutgoingDetail[j].PrevTime then
						iCombatant.OutgoingDetail[j].PrevTime = nil
					end
				end
			end
			if iCombatant.IncomingDetail then
				for j in pairs(iCombatant.IncomingDetail) do
					if iCombatant.IncomingDetail[j].PrevTime then
						iCombatant.IncomingDetail[j].PrevTime = nil
					end
				end
			end
			if iCombatant.OtherDetail then
				for j in pairs(iCombatant.OtherDetail) do
					if iCombatant.OtherDetail[j].PrevTime then
						iCombatant.OtherDetail[j].PrevTime = nil
					end
				end
			end
			if iCombatant.LastOutgoingDetail_1 then
				for j in pairs(iCombatant.LastOutgoingDetail_1) do
					if iCombatant.LastOutgoingDetail_1[j].PrevTime then
						iCombatant.LastOutgoingDetail_1[j].PrevTime = nil
						iCombatant.LastOutgoingDetail_1[j].PrevTimeDur = nil
					end
				end
			end
			if iCombatant.LastIncomingDetail_1 then
				for j in pairs(iCombatant.LastIncomingDetail_1) do
					if iCombatant.LastIncomingDetail_1[j].PrevTime then
						iCombatant.LastIncomingDetail_1[j].PrevTime = nil
						iCombatant.LastIncomingDetail_1[j].PrevTimeDur = nil
					end
				end
			end
			if iCombatant.LastOtherDetail_1 then
				for j in pairs(iCombatant.LastOtherDetail_1) do
					if iCombatant.LastOtherDetail_1[j].PrevTime then
						iCombatant.LastOtherDetail_1[j].PrevTime = nil
						iCombatant.LastOtherDetail_1[j].PrevTimeDur = nil
					end
				end
			end
			if iCombatant.LastOutgoingDetail_2 then
				for j in pairs(iCombatant.LastOutgoingDetail_2) do
					if iCombatant.LastOutgoingDetail_2[j].PrevTime then
						iCombatant.LastOutgoingDetail_2[j].PrevTime = nil
						iCombatant.LastOutgoingDetail_2[j].PrevTimeDur = nil
					end
				end
			end
			if iCombatant.LastIncomingDetail_2 then
				for j in pairs(iCombatant.LastIncomingDetail_2) do
					if iCombatant.LastIncomingDetail_2[j].PrevTime then
						iCombatant.LastIncomingDetail_2[j].PrevTime = nil
						iCombatant.LastIncomingDetail_2[j].PrevTimeDur = nil
					end
				end
			end
			if iCombatant.LastOtherDetail_2 then
				for j in pairs(iCombatant.LastOtherDetail_2) do
					if iCombatant.LastOtherDetail_2[j].PrevTime then
						iCombatant.LastOtherDetail_2[j].PrevTime = nil
						iCombatant.LastOtherDetail_2[j].PrevTimeDur = nil
					end
				end
			end
		end
	end
end


-- we don't track the Outgoing component of damage that is from one group member to another group member (otherwise would inflate DPS)
-- this includes not tracking, for example, Shatter damage in Gruul's Lair
-- this includes not tracking self-damage (e.g. Shadow Word: Death) and damage from owner to pet (e.g. Soul Link) (both of these whether the combatant is in the group or not)
function Recap_TrackAsOutgoing(sourceNameGUID, destNameGUID)
	-- check for self-damage
	if sourceNameGUID == destNameGUID then
		-- ignore if self-damage (whether in group or not)
		return false
	end
	-- check for group <=> group damage
	if recap_temp.InGroup[sourceNameGUID] and recap_temp.InGroup[destNameGUID] then
		-- ignore if both combatants are in the group (for most group pets this is faster than checking the pet the hard way)
		-- note that damage between combatants who are not in the group, but who could be allied with each other, is still counted, potentially inflating *their* DPS
		return false
	end
	-- check for owner <=> pet damage the hard way
	local sourceOwner = Recap_ExtractOwner(sourceNameGUID)
	local destOwner = Recap_ExtractOwner(destNameGUID)
	if (sourceOwner and (sourceOwner == Recap_ExtractCombatant(destNameGUID))) or (destOwner and (destOwner == Recap_ExtractCombatant(sourceNameGUID))) then
		-- ignore if source is controlled by dest, or dest is controlled by source (whether in group or not)
		return false
	end
	-- no contra-indications
	return true
end


function Recap_CreateEffect(event_time, sourceNameGUID, destNameGUID, effect, myType, total, overheal, did_crit, is_dot, element, missType, did_crush, did_glance, absorb, block, resist, doSelf, AllLastOutgoingDetail, AllLastIncomingDetail)

	local iEffect, sourceCombatant, destCombatant, typeEffect, myElement, elementEffect, selfIndex, trackAsOutgoing, damageEffect, deltaEstResisted, deltaEstResistable

	-- parameter is_dot is not currently used (will be 1 or nil)

	if not effect then
		return
	end

	-- table lookup shortcuts
	if sourceNameGUID then
		sourceCombatant = recap.Combatant[sourceNameGUID]
	else
		sourceCombatant = nil
	end
	if destNameGUID then
		destCombatant = recap.Combatant[destNameGUID]
	else
		destCombatant = nil
	end

	typeEffect = myType..effect

	-- some misses will get here with Other or "?" as the element
	myElement = element
	if missType and ((myElement == recap_temp.Localize.ElementOther) or (myElement == "?")) then
		-- if so attempt a correction by looking for a non-default element in the corresponding Outgoing effect
		if sourceCombatant and sourceCombatant[AllLastOutgoingDetail] and sourceCombatant[AllLastOutgoingDetail]["1"..effect] and sourceCombatant[AllLastOutgoingDetail]["1"..effect].Element then
			for i in string_gmatch(sourceCombatant[AllLastOutgoingDetail]["1"..effect].Element, "[^%+]+") do
				if (i ~= recap_temp.Localize.ElementOther) and (i ~= "?") then
					myElement = i
					break
				end
			end
		end
		-- if still Other or "?" then check among the corresponding Incoming effects
		if (myElement == recap_temp.Localize.ElementOther) or (myElement == "?") then
			if destCombatant and destCombatant[AllLastIncomingDetail] and destCombatant[AllLastIncomingDetail]["1"..effect] and destCombatant[AllLastIncomingDetail]["1"..effect].Element then
				for i in string_gmatch(destCombatant[AllLastIncomingDetail]["1"..effect].Element, "[^%+]+") do
					if (i ~= recap_temp.Localize.ElementOther) and (i ~= "?") then
						myElement = i
						break
					end
				end
			end
		end
		-- if still "?" then force to Other
		if myElement == "?" then
			myElement = recap_temp.Localize.ElementOther
		end
	end

	-- prepare for subtotals and totals
	if myType == 1 then
		-- keep element subtotals separate for damage and healing
		elementEffect = "5"..recap_temp.Localize.Subtotal..": "..myElement
		-- overall damage total
		damageEffect = "7"..RECAP_TOTAL..": "..RECAP_DAMAGE
	elseif myType == 3 then
		-- overall healing total
		elementEffect = "7"..RECAP_TOTAL..": "..recap_temp.Localize.ElementHealing
	end

	-- is the sourceNameGUID the player or one of the player's pets?
	selfIndex = nil -- keep as nil
	if sourceNameGUID then
		if sourceNameGUID == recap_temp.PlayerGUID then
			selfIndex = recap_temp.s
		elseif Recap_IsOwnedBy(sourceNameGUID, recap_temp.PlayerGUID) then
			selfIndex = recap_temp.s..":"..Recap_NameOnlyFromCombatant(sourceNameGUID)
		end
	end

	-- we don't track the Outgoing component of some damage (self-damage, damage between owner and pet, and damage from one group member to another group member) (otherwise would inflate DPS)
	-- we still track the Incoming component and (elsewhere) the Matrix component
	trackAsOutgoing = true
	if (myType == 1) and sourceNameGUID and destNameGUID then
		trackAsOutgoing = Recap_TrackAsOutgoing(sourceNameGUID, destNameGUID)
	end

	-- main processing block
	if (did_crit==1) then
		if sourceNameGUID and trackAsOutgoing then
			Recap_MakeOutgoingDetail(sourceCombatant, typeEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][typeEffect]
			Recap_AddCrits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, elementEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][elementEffect]
			Recap_AddCrits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeOutgoingDetail(sourceCombatant, damageEffect, AllLastOutgoingDetail)
				iEffect = sourceCombatant[AllLastOutgoingDetail][damageEffect]
				Recap_AddCrits(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end

			-- only add when adding to AllFights (otherwise counts twice)
			if selfIndex and doSelf then
				Recap_MakeSelfDetail(selfIndex, typeEffect)
				iEffect = recap.Self[selfIndex][typeEffect]
				iEffect.CritsEvents = (iEffect.CritsEvents or 0) + 1
				iEffect.Crits = (iEffect.Crits or 0) + 1
				iEffect.CritsDmg = (iEffect.CritsDmg or 0) + total
				iEffect.CritsMax = math_max(total,iEffect.CritsMax or 0)
				if recap.gmin then
					-- for the cognoscenti, factor back in any partials to get a gross minimum
					iEffect.CritsMin = Recap_Min(total+(absorb or 0)+(block or 0)+(resist or 0),iEffect.CritsMin)
				else
					-- normal net minimum
					iEffect.CritsMin = Recap_Min(total,iEffect.CritsMin)
				end
				Recap_AddSelfElement(iEffect, myElement)
			end
		end
		if destCombatant then
			Recap_MakeIncomingDetail(destCombatant, typeEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][typeEffect]
			Recap_AddCrits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeIncomingDetail(destCombatant, elementEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][elementEffect]
			Recap_AddCrits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeIncomingDetail(destCombatant, damageEffect, AllLastIncomingDetail)
				iEffect = destCombatant[AllLastIncomingDetail][damageEffect]
				Recap_AddCrits(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end
		end
	elseif (did_crush==1) then
		if sourceNameGUID and trackAsOutgoing then
			Recap_MakeOutgoingDetail(sourceCombatant, typeEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][typeEffect]
			Recap_AddCrushes(iEffect, total)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, elementEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][elementEffect]
			Recap_AddCrushes(iEffect, total)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			-- damage subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, damageEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][damageEffect]
			Recap_AddCrushes(iEffect, total)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)

			-- self and pets can never crush, so there is nothing done with Self
		end
		if destCombatant then
			Recap_MakeIncomingDetail(destCombatant, typeEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][typeEffect]
			Recap_AddCrushes(iEffect, total)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeIncomingDetail(destCombatant, elementEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][elementEffect]
			Recap_AddCrushes(iEffect, total)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeIncomingDetail(destCombatant, damageEffect, AllLastIncomingDetail)
				iEffect = destCombatant[AllLastIncomingDetail][damageEffect]
				Recap_AddCrushes(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end
		end
	elseif (did_glance==1) then
		if sourceNameGUID and trackAsOutgoing then
			Recap_MakeOutgoingDetail(sourceCombatant, typeEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][typeEffect]
			Recap_AddGlances(iEffect, total)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, elementEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][elementEffect]
			Recap_AddGlances(iEffect, total)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			-- damage subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, damageEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][damageEffect]
			Recap_AddGlances(iEffect, total)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)

			-- only add when adding to AllFights (otherwise counts twice)
			if selfIndex and doSelf then
				Recap_MakeSelfDetail(selfIndex, typeEffect)
				iEffect = recap.Self[selfIndex][typeEffect]
				iEffect.CritsEvents = (iEffect.CritsEvents or 0) + 1
				iEffect.Glances = (iEffect.Glances or 0) + 1
				iEffect.GlancesDmg = (iEffect.GlancesDmg or 0) + total
				iEffect.GlancesMax = math_max(total,iEffect.GlancesMax or 0)
				if recap.gmin then
					-- for the cognoscenti, factor back in any partials to get a gross minimum
					iEffect.GlancesMin = Recap_Min(total+(absorb or 0)+(block or 0)+(resist or 0),iEffect.GlancesMin)
				else
					-- normal net minimum
					iEffect.GlancesMin = Recap_Min(total,iEffect.GlancesMin)
				end
				Recap_AddSelfElement(iEffect, myElement)
			end
		end
		if destCombatant then
			Recap_MakeIncomingDetail(destCombatant, typeEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][typeEffect]
			Recap_AddGlances(iEffect, total)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeIncomingDetail(destCombatant, elementEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][elementEffect]
			Recap_AddGlances(iEffect, total)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeIncomingDetail(destCombatant, damageEffect, AllLastIncomingDetail)
				iEffect = destCombatant[AllLastIncomingDetail][damageEffect]
				Recap_AddGlances(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end
		end
	elseif (did_crit==false) and (did_crush==false) and (did_glance==false) then
		-- hit that could have critted (does not include misses, so that crit percentages are accurate)
		if sourceNameGUID and trackAsOutgoing then
			Recap_MakeOutgoingDetail(sourceCombatant, typeEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][typeEffect]
			Recap_AddHits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, elementEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][elementEffect]
			Recap_AddHits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeOutgoingDetail(sourceCombatant, damageEffect, AllLastOutgoingDetail)
				iEffect = sourceCombatant[AllLastOutgoingDetail][damageEffect]
				Recap_AddHits(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end

			-- only add when adding to AllFights (otherwise counts twice)
			if selfIndex and doSelf then
				Recap_MakeSelfDetail(selfIndex, typeEffect)
				iEffect = recap.Self[selfIndex][typeEffect]
				iEffect.CritsEvents = (iEffect.CritsEvents or 0) + 1
				iEffect.Hits = (iEffect.Hits or 0) + 1
				iEffect.HitsDmg = (iEffect.HitsDmg or 0) + total
				iEffect.HitsMax = math_max(total, iEffect.HitsMax or 0)
				if recap.gmin then
					-- for the cognoscenti, factor back in any partials to get a gross minimum
					iEffect.HitsMin = Recap_Min(total+(absorb or 0)+(block or 0)+(resist or 0),iEffect.HitsMin)
				else
					-- normal net minimum
					iEffect.HitsMin = Recap_Min(total,iEffect.HitsMin)
				end
				Recap_AddSelfElement(iEffect, myElement)
			end
		end
		if destCombatant then
			Recap_MakeIncomingDetail(destCombatant, typeEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][typeEffect]
			Recap_AddHits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeIncomingDetail(destCombatant, elementEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][elementEffect]
			Recap_AddHits(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeIncomingDetail(destCombatant, damageEffect, AllLastIncomingDetail)
				iEffect = destCombatant[AllLastIncomingDetail][damageEffect]
				Recap_AddHits(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end
		end
	elseif (total==0) then
		-- miss
		if sourceNameGUID and trackAsOutgoing then
			Recap_MakeOutgoingDetail(sourceCombatant, typeEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][typeEffect]
			Recap_AddMisses(iEffect, missType)
			Recap_AddElement(iEffect, myElement)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, elementEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][elementEffect]
			Recap_AddMisses(iEffect, missType)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			-- damage subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, damageEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][damageEffect]
			Recap_AddMisses(iEffect, missType)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)

			-- only add when adding to AllFights (otherwise counts twice)
			if selfIndex and doSelf then
				Recap_MakeSelfDetail(selfIndex, typeEffect)
				iEffect = recap.Self[selfIndex][typeEffect]
				if (missType == "MISS") then
					iEffect.Missed = (iEffect.Missed or 0) + 1
				elseif (missType == "ABSORB") then
					iEffect.Absorbed = (iEffect.Absorbed or 0) + 1
				elseif (missType == "DODGE") then
					iEffect.Dodged = (iEffect.Dodged or 0) + 1
				elseif (missType == "PARRY") then
					iEffect.Parried = (iEffect.Parried or 0) + 1
				elseif (missType == "BLOCK") then
					iEffect.Blocked = (iEffect.Blocked or 0) + 1
				elseif (missType == "DEFLECT") then
					iEffect.Deflected = (iEffect.Deflected or 0) + 1
				elseif (missType == "EVADE") then
					iEffect.Evaded = (iEffect.Evaded or 0) + 1
				elseif (missType == "RESIST") then
					iEffect.Resisted = (iEffect.Resisted or 0) + 1
				elseif (missType == "REFLECT") then
					iEffect.Reflected = (iEffect.Reflected or 0) + 1
				elseif (missType == "IMMUNE") then
					iEffect.Immune = (iEffect.Immune or 0) + 1
				end
				Recap_AddSelfElement(iEffect, myElement)
			end
		end
		if destCombatant then
			Recap_MakeIncomingDetail(destCombatant, typeEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][typeEffect]
			Recap_AddMisses(iEffect, missType)
			Recap_AddElement(iEffect, myElement)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeIncomingDetail(destCombatant, elementEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][elementEffect]
			Recap_AddMisses(iEffect, missType)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeIncomingDetail(destCombatant, damageEffect, AllLastIncomingDetail)
				iEffect = destCombatant[AllLastIncomingDetail][damageEffect]
				Recap_AddMisses(iEffect, missType)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end
		end
	else
		-- hit that couldn't have critted (must be a DoT or HoT)
		if sourceNameGUID and trackAsOutgoing then
			Recap_MakeOutgoingDetail(sourceCombatant, typeEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][typeEffect]
			Recap_AddTicks(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeOutgoingDetail(sourceCombatant, elementEffect, AllLastOutgoingDetail)
			iEffect = sourceCombatant[AllLastOutgoingDetail][elementEffect]
			Recap_AddTicks(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeOutgoingDetail(sourceCombatant, damageEffect, AllLastOutgoingDetail)
				iEffect = sourceCombatant[AllLastOutgoingDetail][damageEffect]
				Recap_AddTicks(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end

			-- only add when adding to AllFights (otherwise counts twice)
			if selfIndex and doSelf then
				Recap_MakeSelfDetail(selfIndex, typeEffect)
				iEffect = recap.Self[selfIndex][typeEffect]
				iEffect.Ticks = (iEffect.Ticks or 0) + 1
				iEffect.TicksDmg = (iEffect.TicksDmg or 0) + total
				iEffect.TicksMax = math_max(total,iEffect.TicksMax or 0)
				if recap.gmin then
					-- for the cognoscenti, factor back in any partials to get a gross minimum
					iEffect.TicksMin = Recap_Min(total+(absorb or 0)+(block or 0)+(resist or 0),iEffect.TicksMin)
				else
					-- normal net minimum
					iEffect.TicksMin = Recap_Min(total,iEffect.TicksMin)
				end
				Recap_AddSelfElement(iEffect, myElement)
			end
		end
		if destCombatant then
			Recap_MakeIncomingDetail(destCombatant, typeEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][typeEffect]
			Recap_AddTicks(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddElement(iEffect, myElement)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			deltaEstResisted, deltaEstResistable = Recap_CalcEstimatedResistance(iEffect)
			Recap_CalcInterval(iEffect, event_time)
			-- element subtotal
			Recap_MakeIncomingDetail(destCombatant, elementEffect, AllLastIncomingDetail)
			iEffect = destCombatant[AllLastIncomingDetail][elementEffect]
			Recap_AddTicks(iEffect, total)
			Recap_AddOverhealing(iEffect, overheal)
			Recap_AddPartials(iEffect, total, absorb, block, resist)
			Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
			Recap_CalcInterval(iEffect, event_time)
			if myType == 1 then
				-- damage subtotal
				Recap_MakeIncomingDetail(destCombatant, damageEffect, AllLastIncomingDetail)
				iEffect = destCombatant[AllLastIncomingDetail][damageEffect]
				Recap_AddTicks(iEffect, total)
				Recap_AddPartials(iEffect, total, absorb, block, resist)
				Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
				Recap_CalcInterval(iEffect, event_time)
			end
		end
	end
end

-- six auxiliary functions for Recap_CreateEffect to create spots in the tables as necessary
function Recap_MakeIncomingDetail(iCombatant, typeEffect, AllLastIncomingDetail)
	if not iCombatant[AllLastIncomingDetail] then
		iCombatant[AllLastIncomingDetail] = {}
	end
	if not iCombatant[AllLastIncomingDetail][typeEffect] then
		iCombatant[AllLastIncomingDetail][typeEffect] = {}
	end
end
function Recap_MakeOutgoingDetail(iCombatant, typeEffect, AllLastOutgoingDetail)
	if not iCombatant[AllLastOutgoingDetail] then
		iCombatant[AllLastOutgoingDetail] = {}
	end
	if not iCombatant[AllLastOutgoingDetail][typeEffect] then
		iCombatant[AllLastOutgoingDetail][typeEffect] = {}
	end
end
function Recap_MakeSelfDetail(selfIndex, typeEffect)
	if not recap.Self[selfIndex] then
		recap.Self[selfIndex] = {}
	end
	if not recap.Self[selfIndex][typeEffect] then
		recap.Self[selfIndex][typeEffect] = {}
	end
end
function Recap_AddCrits(iEffect, total)
	iEffect.CritsEvents = (iEffect.CritsEvents or 0) + 1
	iEffect.Crits = (iEffect.Crits or 0) + 1
	iEffect.CritsDmg = (iEffect.CritsDmg or 0) + total
	iEffect.CritsMax = math_max(total, iEffect.CritsMax or 0)
	iEffect.CritsMin = Recap_Min(total, iEffect.CritsMin)
end
function Recap_AddCrushes(iEffect, total)
	iEffect.CritsEvents = (iEffect.CritsEvents or 0) + 1
	iEffect.Crushes = (iEffect.Crushes or 0) + 1
	iEffect.CrushDmg = (iEffect.CrushDmg or 0) + total
	iEffect.CrushMax = math_max(total, iEffect.CrushMax or 0)
	iEffect.CrushMin = Recap_Min(total, iEffect.CrushMin)
end
function Recap_AddGlances(iEffect, total)
	iEffect.CritsEvents = (iEffect.CritsEvents or 0) + 1
	iEffect.Glances = (iEffect.Glances or 0) + 1
	iEffect.GlancesDmg = (iEffect.GlancesDmg or 0) + total
	iEffect.GlancesMax = math_max(total, iEffect.GlancesMax or 0)
	iEffect.GlancesMin = Recap_Min(total, iEffect.GlancesMin)
end
function Recap_AddHits(iEffect, total)
	iEffect.CritsEvents = (iEffect.CritsEvents or 0) + 1
	iEffect.Hits = (iEffect.Hits or 0) + 1
	iEffect.HitsDmg = (iEffect.HitsDmg or 0) + total
	iEffect.HitsMax = math_max(total, iEffect.HitsMax or 0)
	iEffect.HitsMin = Recap_Min(total, iEffect.HitsMin)
end
function Recap_AddTicks(iEffect, total)
	iEffect.Ticks = (iEffect.Ticks or 0) + 1
	iEffect.TicksDmg = (iEffect.TicksDmg or 0) + total
	iEffect.TicksMax = math_max(total, iEffect.TicksMax or 0)
	iEffect.TicksMin = Recap_Min(total, iEffect.TicksMin)
end
function Recap_AddOverhealing(iEffect, overhealing)
	if overhealing > 0 then
		-- reuse .Missed detail to store overhealing since heals never miss
		iEffect.Missed = (iEffect.Missed or 0) + overhealing
	end
end
function Recap_AddMisses(iEffect, missType)
	if (missType == "MISS") then
		iEffect.Missed = (iEffect.Missed or 0) + 1
	elseif (missType == "ABSORB") then
		iEffect.Absorbed = (iEffect.Absorbed or 0) + 1
	elseif (missType == "DODGE") then
		iEffect.Dodged = (iEffect.Dodged or 0) + 1
	elseif (missType == "PARRY") then
		iEffect.Parried = (iEffect.Parried or 0) + 1
	elseif (missType == "BLOCK") then
		iEffect.Blocked = (iEffect.Blocked or 0) + 1
	elseif (missType == "DEFLECT") then
		iEffect.Deflected = (iEffect.Deflected or 0) + 1
	elseif (missType == "EVADE") then
		iEffect.Evaded = (iEffect.Evaded or 0) + 1
	elseif (missType == "RESIST") then
		iEffect.Resisted = (iEffect.Resisted or 0) + 1
	elseif (missType == "REFLECT") then
		iEffect.Reflected = (iEffect.Reflected or 0) + 1
	elseif (missType == "IMMUNE") then
		iEffect.Immune = (iEffect.Immune or 0) + 1
	end
end
function Recap_AddElement(iEffect, element)
	if not iEffect.Element then
		-- no element yet, set it
		iEffect.Element = element
	else
		-- if there is a new element that is not 'Other', check whether the new and old elements are the same
		if element and (element ~= recap_temp.Localize.ElementOther) then
			if not string_find(iEffect.Element, element, 1, true) then
				-- new element not found in old element, append it
				-- for example, perhaps the combatant switched wands
				if (iEffect.Element == "?") or (iEffect.Element == recap_temp.Localize.ElementOther) then
					-- replace a "?" or "Other"
					iEffect.Element = element
				else
					-- append to anything else
					iEffect.Element = iEffect.Element.."+"..element
				end
			end
		end
	end
end
function Recap_AddPartials(iEffect, total, absorb, block, resist)

	local proportion

	if absorb and absorb>0 then
		iEffect.PAbsorbs = (iEffect.PAbsorbs or 0) + 1
		iEffect.PAbsorbsDmg = (iEffect.PAbsorbsDmg or 0) + absorb
	end
	if block and block>0 then
		iEffect.PBlocks = (iEffect.PBlocks or 0) + 1
		iEffect.PBlocksDmg = (iEffect.PBlocksDmg or 0) + block
	end
	if resist and resist>0 then
		-- partial resist percentages are quantised to 25%, 50%, and 75% (with some variability)
		proportion = resist / (resist + total)
		if proportion < 0.325 then
			iEffect.PResists25 = (iEffect.PResists25 or 0) + 1
			iEffect.PResists25Dmg = (iEffect.PResists25Dmg or 0) + resist
		elseif proportion < 0.625 then
			iEffect.PResists50 = (iEffect.PResists50 or 0) + 1
			iEffect.PResists50Dmg = (iEffect.PResists50Dmg or 0) + resist
		else
			iEffect.PResists75 = (iEffect.PResists75 or 0) + 1
			iEffect.PResists75Dmg = (iEffect.PResists75Dmg or 0) + resist
		end
	end
end
-- function returns two values (delta resisted damage and delta resistable damage) for accumulation into subtotals and totals
function Recap_CalcEstimatedResistance(iEffect) -- this is for an effect other than a subtotal or total

	local PartialResistedDmg, PartialPreventedDmg, LandedDmg, Landings, CompletelyResistedDmg
	local EstimatedResistedDmg, EstimatedResistableDmg, DeltaEstimatedResistedDmg, DeltaEstimatedResistableDmg

	PartialResistedDmg = (iEffect.PResists25Dmg or 0)+(iEffect.PResists50Dmg or 0)+(iEffect.PResists75Dmg or 0)
	PartialPreventedDmg = (iEffect.PAbsorbsDmg or 0)+(iEffect.PBlocksDmg or 0)+PartialResistedDmg
	LandedDmg = (iEffect.GlancesDmg or 0)+(iEffect.HitsDmg or 0)+(iEffect.CritsDmg or 0)+(iEffect.CrushDmg or 0)+(iEffect.TicksDmg or 0)+PartialPreventedDmg
	-- if this effect has an initial hit plus a DoT, we count only the initial hits and assume that all complete resists are of initial hits
	Landings = (iEffect.Glances or 0)+(iEffect.Hits or 0)+(iEffect.Crits or 0)+(iEffect.Crushes or 0)
	if (Landings == 0) then
		-- if this effect has ticks only, use that count instead
		Landings = (iEffect.Ticks or 0)
	end
	if (Landings == 0) then
		-- we have only complete resists, and so cannot give meaningful resisted and resistable values
		return 0, 0
	end
	-- this is an estimate only, we cannot know what the hits would have been if not completely resisted
	CompletelyResistedDmg = (iEffect.Resisted or 0) * Recap_Div0(LandedDmg, Landings)

	-- we can calculate resisted and resistable for the effect
	-- we cannot do that for a subtotal or total
	-- instead we pass back deltas (which could go negative) and accumulate them for the subtotal or total
	EstimatedResistedDmg = PartialResistedDmg + CompletelyResistedDmg
	EstimatedResistableDmg = LandedDmg + CompletelyResistedDmg
	DeltaEstimatedResistedDmg = EstimatedResistedDmg - (iEffect.EstResistedDmg or 0)
	DeltaEstimatedResistableDmg = EstimatedResistableDmg - (iEffect.EstResistableDmg or 0)
	-- estimated resisted is a visible column, estimated resistable is a hidden column
	-- only write if non-zero
	if EstimatedResistedDmg > 0 then
		iEffect.EstResistedDmg = EstimatedResistedDmg
	end
	if EstimatedResistableDmg > 0 then
		iEffect.EstResistableDmg = EstimatedResistableDmg
	end

	return DeltaEstimatedResistedDmg, DeltaEstimatedResistableDmg
end
-- for subtotals and totals
function Recap_AccumulateEstimatedResistance(iEffect, deltaEstResisted, deltaEstResistable)
	-- the deltas could go negative
	if not (deltaEstResisted == 0) then
		iEffect.EstResistedDmg = (iEffect.EstResistedDmg or 0) + deltaEstResisted
	end
	if not (deltaEstResistable == 0) then
		iEffect.EstResistableDmg = (iEffect.EstResistableDmg or 0) + deltaEstResistable
	end
end
function Recap_CalcInterval(iEffect, event_time)
	if iEffect.PrevTime and (iEffect.PrevTime > 0) then
		local delta_time = (1000 * event_time) - (iEffect.PrevTime or 0)
		if (delta_time > 0) and (delta_time < 130000) then
			-- only pay attention if the two events happened less than 130 seconds apart
			-- count the number of time deltas, and sum their values (will be used to produce an average)
			iEffect.ICount = (iEffect.ICount or 0) + 1
			iEffect.ITotal = (iEffect.ITotal or 0) + delta_time
		end
	end
	iEffect.PrevTime = (1000 * event_time)
end
function Recap_AddSelfElement(iEffect, element)
	if not iEffect.Element then
		-- no element yet, set it
		iEffect.Element = element
	else
		-- if there is a new element that is not 'Other', check whether the new and old elements are the same
		if element and (element ~= recap_temp.Localize.ElementOther) then
			if not string_find(iEffect.Element, element, 1, true) then
				-- new element not found in old element, append it
				-- for example, perhaps the combatant switched wands
				if (iEffect.Element == "?") or (iEffect.Element == recap_temp.Localize.ElementOther) then
					-- replace a "?" or "Other"
					iEffect.Element = element
				else
					-- append to anything else
					iEffect.Element = iEffect.Element.."+"..element
				end
			end
		end
	end
end


-- new function to store the matrices of who did what and to whom
function Recap_AccumulateMatrixDetails(sourceNameGUID, sourceFlags, destNameGUID, destFlags, myType, total, timestamp, AllLastTargetDetail, AllLastSourceDetail)

	local typeSource, typeDest, iWhom, sourceCombatant, destCombatant

	if (sourceNameGUID == nil) or (destNameGUID == nil) then
		return
	end

	-- create combatants if needed for the To / From Data Mode
	--   (even if they would not otherwise get created)
	if not recap.Combatant[sourceNameGUID] then
		Recap_CreateBlankCombatant(sourceNameGUID, math_floor(1000*timestamp), sourceFlags)
	end
	if not recap.Combatant[destNameGUID] then
		Recap_CreateBlankCombatant(destNameGUID, math_floor(1000*timestamp), destFlags)
	end

	-- table lookup shortcuts
	sourceCombatant = recap.Combatant[sourceNameGUID]
	destCombatant = recap.Combatant[destNameGUID]
	typeSource = tostring(myType)..sourceNameGUID
	typeDest = tostring(myType)..destNameGUID

	Recap_MakeTargetDetail(sourceCombatant, typeDest, AllLastTargetDetail)
	iWhom = sourceCombatant[AllLastTargetDetail][typeDest]
	iWhom.Total = (iWhom.Total or 0) + total
	Recap_MakeSourceDetail(destCombatant, typeSource, AllLastSourceDetail)
	iWhom = destCombatant[AllLastSourceDetail][typeSource]
	iWhom.Total = (iWhom.Total or 0) + total
end
function Recap_MakeTargetDetail(iCombatant, typeWhom, AllLastTargetDetail)
	if not iCombatant[AllLastTargetDetail] then
		iCombatant[AllLastTargetDetail] = {}
	end
	if not iCombatant[AllLastTargetDetail][typeWhom] then
		iCombatant[AllLastTargetDetail][typeWhom] = {}
	end
end
function Recap_MakeSourceDetail(iCombatant, typeWhom, AllLastSourceDetail)
	if not iCombatant[AllLastSourceDetail] then
		iCombatant[AllLastSourceDetail] = {}
	end
	if not iCombatant[AllLastSourceDetail][typeWhom] then
		iCombatant[AllLastSourceDetail][typeWhom] = {}
	end
end

function Recap_CreateMissEffect(sourceNameGUID, destNameGUID, effect, missText, AllLastOtherDetail)

	-- currently ignoring missText

	local typeEffect, iDetail

	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		-- check for the existence of a cast, buff, debuff, or unknown (in that order)
		typeEffect = nil
		iDetail = recap.Combatant[sourceNameGUID][AllLastOtherDetail]
		if iDetail then
			if iDetail["1"..effect] then
				typeEffect = "1"..effect
			elseif iDetail["3"..effect] then
				typeEffect = "3"..effect
			elseif iDetail["5"..effect] then
				typeEffect = "5"..effect
			elseif iDetail["7"..effect] then
				typeEffect = "7"..effect
			end
			if typeEffect then
				iDetail[typeEffect].Hits = (iDetail[typeEffect].Hits or 0) + 1
				iDetail[typeEffect].Missed = (iDetail[typeEffect].Missed or 0) + 1
			end
		end
	else
		-- misses not recorded before sourceNameGUID has been registered as a combatant
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		-- check for the existence of a cast, buff, debuff, or unknown (in that order)
		typeEffect = nil
		iDetail = recap.Combatant[destNameGUID][AllLastOtherDetail]
		if iDetail then
			if iDetail["1"..effect] then
				typeEffect = "1"..effect
			elseif iDetail["3"..effect] then
				typeEffect = "3"..effect
			elseif iDetail["5"..effect] then
				typeEffect = "5"..effect
			elseif iDetail["7"..effect] then
				typeEffect = "7"..effect
			end
			if typeEffect then
				iDetail[typeEffect].Hits = (iDetail[typeEffect].Hits or 0) + 1
				iDetail[typeEffect].Missed = (iDetail[typeEffect].Missed or 0) + 1
			end
		end
	else
		-- misses not recorded before destNameGUID has been registered as a combatant
	end
end

function Recap_CalcOtherInterval(iEffect, event_time)
	if iEffect.PrevTime and (iEffect.PrevTime > 0) then
		local delta_time = (1000 * event_time) - (iEffect.PrevTime or 0)
		if (delta_time > 0) and (delta_time < 130000) then
			-- only pay attention if the two events happened less than 130 seconds apart
			-- count the number of time deltas, and sum their values (will be used to produce an average)
			iEffect.ICount = (iEffect.ICount or 0) + 1
			iEffect.ITotal = (iEffect.ITotal or 0) + delta_time
		end
	end
	iEffect.PrevTime = (1000 * event_time)
	-- set the timer for measuring durations
	iEffect.PrevTimeDur = (1000 * event_time)
end
function Recap_CalcOtherDuration(iEffect, event_time)
	if iEffect.PrevTimeDur and (iEffect.PrevTimeDur > 0) then
		local delta_time = (1000 * event_time) - (iEffect.PrevTimeDur or 0)
		if (delta_time > 0) and (delta_time < 130000) then
			-- only pay attention if the apparent duration is less than 130 seconds
			-- count the number of time deltas, and sum their values (will be used to produce an average)
			iEffect.DCount = (iEffect.DCount or 0) + 1
			iEffect.DTotal = (iEffect.DTotal or 0) + delta_time
		end
	end
	-- clear the duration timer
	iEffect.PrevTimeDur = nil
end

function Recap_MakeOtherDetail(thisCombatant, typeEffect, AllLastOtherDetail)
	if not recap.Combatant[thisCombatant][AllLastOtherDetail] then
		recap.Combatant[thisCombatant][AllLastOtherDetail] = {}
	end
	if not recap.Combatant[thisCombatant][AllLastOtherDetail][typeEffect] then
		recap.Combatant[thisCombatant][AllLastOtherDetail][typeEffect] = {}
	end
end
function Recap_CreateBasicOtherDetail(thisCombatant, timestamp, myType, effect, AllLastOtherDetail)
	local typeEffect, iEffect
	typeEffect = myType..effect
	Recap_MakeOtherDetail(thisCombatant, typeEffect, AllLastOtherDetail)
	iEffect = recap.Combatant[thisCombatant][AllLastOtherDetail][typeEffect]
	iEffect.Hits = (iEffect.Hits or 0) + 1
	Recap_CalcOtherInterval(iEffect, timestamp)
end
function Recap_CreateFullOtherDetail(thisCombatant, timestamp, myType, effect, total, attribute, AllLastOtherDetail)
	local typeEffect, iEffect
	typeEffect = myType..effect
	Recap_MakeOtherDetail(thisCombatant, typeEffect, AllLastOtherDetail)
	iEffect = recap.Combatant[thisCombatant][AllLastOtherDetail][typeEffect]
	iEffect.Hits = (iEffect.Hits or 0) + 1
	iEffect.Total = (iEffect.Total or 0) + total
	iEffect.Max = math_max(total,iEffect.Max or 1)
	if attribute then
		iEffect.Attribute = attribute
	end
	Recap_CalcOtherInterval(iEffect, timestamp)
end

-- removes combatant i from last fight
function Recap_ResetLastFight(i)

	local iCombatant = recap.Combatant[i]
	local j

	j = (iCombatant.TotalTime or 0) - (iCombatant["LastTime_"..recap_temp.DisplayLastFight] or 0)
	if j>0 then
		iCombatant.TotalTime = j
	end
	j = (iCombatant.TotalTimeIn or 0) - (iCombatant["LastTimeIn_"..recap_temp.DisplayLastFight] or 0)
	if j>0 then
		iCombatant.TotalTimeIn = j
	end
	j = (iCombatant.TotalTimeHeal or 0) - (iCombatant["LastTimeHeal_"..recap_temp.DisplayLastFight] or 0)
	if j>0 then
		iCombatant.TotalTimeHeal = j
	end
	j = (iCombatant.TotalDmgIn or 0) - (iCombatant["LastDmgIn_"..recap_temp.DisplayLastFight] or 0)
	if j>0 then
		iCombatant.TotalDmgIn = j
	end
	j = (iCombatant.TotalDmgOut or 0) - (iCombatant["LastDmgOut_"..recap_temp.DisplayLastFight] or 0)
	if j>0 then
		iCombatant.TotalDmgOut = j
	end
	j = (iCombatant.TotalRawHeal or 0) - (iCombatant["LastRawHeal_"..recap_temp.DisplayLastFight] or 0)
	if j>0 then
		iCombatant.TotalRawHeal = j
	end
	j = math_min(((iCombatant.TotalOverHeal or 0) - (iCombatant["LastOverHeal_"..recap_temp.DisplayLastFight] or 0)), (iCombatant.TotalRawHeal or 0))
	if j>0 then
		iCombatant.TotalOverHeal = j
	end
	j = (iCombatant.TotalKills or 0) - (iCombatant["LastKills_"..recap_temp.DisplayLastFight] or 0)
	if j>0 then
		iCombatant.TotalKills = j
	end
	if iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
		iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
	end
	if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
		iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
	end
	if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and (((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)) > 0) then
		iCombatant.TotalHPS = Recap_Div1(((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)), iCombatant.TotalTimeHeal)
	end
	iCombatant.WasInLast = false

	-- clear the currently displayed Last Fight details
	if recap_temp.ActiveLastFight == "1" then
		-- we have been storing data in 1, and displaying from 2
		-- clear 2
		iCombatant.LastTime_2 = nil
		iCombatant.LastTimeIn_2 = nil
		iCombatant.LastTimeHeal_2 = nil
		iCombatant.LastMaxHit_2 = nil
		iCombatant.LastDmgIn_2 = nil
		iCombatant.LastDmgOut_2 = nil
		iCombatant.LastHPS_2 = nil
		iCombatant.LastDPSIn_2 = nil
		iCombatant.LastDPS_2 = nil
		iCombatant.LastKills_2 = nil
		iCombatant.LastRawHeal_2 = nil
		iCombatant.LastOverHeal_2 = nil
		iCombatant.LastOutgoingDetail_2 = nil
		iCombatant.LastTargetDetail_2 = nil
		iCombatant.LastIncomingDetail_2 = nil
		iCombatant.LastSourceDetail_2 = nil
		iCombatant.LastOtherDetail_2 = nil
	else
		-- we have been storing data in 2, and displaying from 1
		-- clear 1
		iCombatant.LastTime_1 = nil
		iCombatant.LastTimeIn_1 = nil
		iCombatant.LastTimeHeal_1 = nil
		iCombatant.LastMaxHit_1 = nil
		iCombatant.LastDmgIn_1 = nil
		iCombatant.LastDmgOut_1 = nil
		iCombatant.LastHPS_1 = nil
		iCombatant.LastDPSIn_1 = nil
		iCombatant.LastDPS_1 = nil
		iCombatant.LastKills_1 = nil
		iCombatant.LastRawHeal_1 = nil
		iCombatant.LastOverHeal_1 = nil
		iCombatant.LastOutgoingDetail_1 = nil
		iCombatant.LastTargetDetail_1 = nil
		iCombatant.LastIncomingDetail_1 = nil
		iCombatant.LastSourceDetail_1 = nil
		iCombatant.LastOtherDetail_1 = nil
	end

	-- we do not attempt to subtract the Last Fight details from the All Fights details -- tough
end


-- two functions to support Recent Data Mode
function Recap_CreateRecent()

	local i

	recap_temp.Recent = {}
	i = 1
	while i <= recap.Opt.RecentEventCount.value do
		recap_temp.Recent[i] = {}
		i = i + 1
	end
	recap_temp.Recent.First = 1
	recap_temp.Recent.Last = 1

end

function Recap_AddRecentEvent(event_time, sourceNameGUID, destNameGUID, effectType, effect, element, amount, crit, crush)

	local i, iRecent

	-- play with the pointers
	if recap_temp.Recent[recap_temp.Recent.Last] and recap_temp.Recent[recap_temp.Recent.Last].T then
		-- events already stored, move Last pointer
		if recap_temp.Recent.Last >= recap.Opt.RecentEventCount.value then
			recap_temp.Recent.Last = 1
		else
			recap_temp.Recent.Last = recap_temp.Recent.Last + 1
		end
		if recap_temp.Recent.First == recap_temp.Recent.Last then
			-- pointers have met, move the First pointer out of the way
			if recap_temp.Recent.First >= recap.Opt.RecentEventCount.value then
				recap_temp.Recent.First = 1
			else
				recap_temp.Recent.First = recap_temp.Recent.First + 1
			end
		end
	else
		-- no event stored yet, store at Last
	end
	i = recap_temp.Recent.Last
	iRecent = recap_temp.Recent[i]
	-- single character keys to keep memory cost down (T = time, S = source D = destination, E = effect, L = element, A = amount, C = crit, R = crush)
	iRecent.T = event_time
	if sourceNameGUID or destNameGUID then
		iRecent.S = sourceNameGUID
		iRecent.D = destNameGUID
		iRecent.E = tostring(effectType)..effect
		iRecent.L = element
		iRecent.A = amount
		iRecent.C = crit
		iRecent.R = crush
	else
		-- if sourceNameGUID and destNameGUID are false, this represents an 'end of fight' event
		iRecent.S = "**EOF**"
		iRecent.D = nil -- keep as nil
		iRecent.E = nil -- keep as nil
		iRecent.L = nil -- keep as nil
		iRecent.A = nil -- keep as nil
		iRecent.C = nil -- keep as nil
		iRecent.R = nil -- keep as nil
	end
end


-- main combat log event dispatcher for WoW patch 2.4
function Recap_ProcessCombatLogEvent(self, event, ...)

	local onIgnore

	-- bail if Recap is suspended
	if (recap.Opt.State.value == "Stopped") then
		return
	end

	-- dump everything as we get it
	-- doesn't tell us much that "/combatlog" wouldn't have told us, so I'm not sure how useful this is any longer
	if recap.dump then
		Recap_DumpMessage(Recap_EventText(...))
	end

	-- get the eight arguments common to all events (the ellipsis is a variable pointing to a list of arguments)
	-- timestamp appears to be a flavour of UTC
	local timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags = ...

	-- we seem to get events with standard arguments equal to nil, so they need to be ignored
	if (timestamp == nil) or (eventType == nil) then
		return
	end

	-- if this is going to be an event that does damage or healing (or equivalent), we need to check for group pet ownership before we do anything else, otherwise
	--   we can end up with orphan pets if they damage or heal before ownership is determined (e.g. a pet that appears upon dismount and is then sent to begin combat)
	if (recap.Opt.State.value == "Idle") and not recap.Opt.LimitFights.value then
		-- the fight hasn't started, and if we are not limiting the fight to when we are personally in combat
		if (eventType == "SWING_DAMAGE") or (eventType == "RANGE_DAMAGE") or (eventType == "SPELL_DAMAGE") or (eventType == "SPELL_PERIODIC_DAMAGE") or
		   (eventType == "SPELL_HEAL") or (eventType == "SPELL_PERIODIC_HEAL") or (eventType == "DAMAGE_SHIELD") or (eventType == "DAMAGE_SPLIT") or (eventType == "ENVIRONMENTAL_DAMAGE") or
		   (eventType == "SWING_MISSED") or (eventType == "RANGE_MISSED") or (eventType == "SPELL_MISSED") or (eventType == "SPELL_PERIODIC_MISSED") or (eventType == "DAMAGE_SHIELD_MISSED") then
			-- start the fight
			Recap_StartFight() -- calls MakeFriends
		end
	elseif (recap.Opt.State.value == "Active") then
		-- the fight is ongoing
		if (eventType == "SWING_DAMAGE") or (eventType == "RANGE_DAMAGE") or (eventType == "SPELL_DAMAGE") or (eventType == "SPELL_PERIODIC_DAMAGE") or
		   (eventType == "SPELL_HEAL") or (eventType == "SPELL_PERIODIC_HEAL") or (eventType == "DAMAGE_SHIELD") or (eventType == "DAMAGE_SPLIT") or (eventType == "ENVIRONMENTAL_DAMAGE") or
		   (eventType == "SWING_MISSED") or (eventType == "RANGE_MISSED") or (eventType == "SPELL_MISSED") or (eventType == "SPELL_PERIODIC_MISSED") or (eventType == "DAMAGE_SHIELD_MISSED") then
			-- if this is a damage or healing event, start or restart the idle timers
			if recap.Opt.IdleFight.value then
				recap_temp.IdleTimer = 0
			end
			recap_temp.HiddenIdleTimer = 0
			-- if the end of fight delay timer is running, restart the end of fight delay timer
			--   (the player left combat, perhaps temporarily, but the fight continues)
			if recap_temp.EndFightDelayTimer~=-1 then
				recap_temp.EndFightDelayTimer = 0
			end
		end
	end

	if recap.Opt.SkipNextFight.value == true then
		-- we are skipping this fight
		-- we've perhaps started the fight, and we've tickled the idle timers, but we won't track damage or healing or buffs or pet ownership changes (which could lead to anomalies later, tough)
		return
	end

-- TODO: we seem to occasionally have orphan Searing Totems for which ownership has not been determined -- is this an occasional absence of summon messages from Blizzard? -- or is it simply a range or visibility issue?
--       it might be only the initial cast that is not identified
-- this is NOT related to starting a fight, it is occurring mid-fight
-- but but but if a shaman can have more than one totem then we can't be finding this out using 'MakeFriends' which as far as I know is only asking for one pet, not multiples
-- perhaps the summon message is coming in *after* the first damage message ??????? -- that's all I can think of
-- if we get a summon for an existing combatant then *move* any existing data?


	-- form the full combatant name as "name_GUID"
	-- we note the reuse of GUIDs for non-player mobs at server restarts, but the odds on a "name_GUID" combination recurring are very low and would (I think) have no serious negative consequences
	local sourceNameGUID = nil
	if sourceName then
		sourceNameGUID = sourceName
		if not recap.Opt.IgnoreGUIDs.value then
			-- using GUIDs (the normal case)
			sourceNameGUID = sourceNameGUID.."_"..Recap_TrimGUID(sourceGUID)
		end
	end
	local destNameGUID = nil
	if destName then
		destNameGUID = destName
		if not recap.Opt.IgnoreGUIDs.value then
			-- using GUIDs (the normal case)
			destNameGUID = destNameGUID.."_"..Recap_TrimGUID(destGUID)
		end
	end

	-- various processing for the bare names (happens again later with fully-qualified names)
	onIgnore = Recap_CheckCombatant(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, ...)
	if onIgnore then
		return
	end

	-- mind control effects occur with SPELL_AURA_APPLIED but unfortunately we don't get both source and dest so that we could set up a pet immediately

	-- Ownership change code (Part 1)
	-- if both source and dest exist and one is recorded as controlling the other, and if the eventType is a hostile spell (heuristic only), then remove the ownership information
	if sourceNameGUID and recap.Combatant[sourceNameGUID] and destNameGUID and recap.Combatant[destNameGUID] and
	   ((recap.Combatant[sourceNameGUID].OwnedBy == destNameGUID) or (recap.Combatant[destNameGUID].OwnedBy == sourceNameGUID)) then
		if Recap_HostileEvent(eventType, sourceNameGUID, destNameGUID, ...) then
			-- remove OwnedBy information
			if (recap.Combatant[sourceNameGUID].OwnedBy == destNameGUID) then
				recap.Combatant[sourceNameGUID].OwnedBy = nil
			end
			if (recap.Combatant[destNameGUID].OwnedBy == sourceNameGUID) then
				recap.Combatant[destNameGUID].OwnedBy = nil
			end
		end
	end

	-- assert: at this point sourceNameGUID and destNameGUID do not have any 'owner' prepended


	-- Ownership change code (Part 2)
	-- if a combatant is controlled, replace any simple name with a fully qualified name "owner:pet"
	-- if we can tell conclusively that a combatant is no longer controlled, remove the ownership information
	local sourceOwnerNameGUID = sourceNameGUID
	local destOwnerNameGUID = destNameGUID
	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		-- existing combatant
		if recap.Combatant[sourceNameGUID].OwnedBy then
			-- there is recorded ownership
			local owner =  recap.Combatant[sourceNameGUID].OwnedBy
			local wasControlled = 0
			if recap.Combatant[sourceNameGUID].Flags then
				wasControlled = bit_band(recap.Combatant[sourceNameGUID].Flags, COMBATLOG_OBJECT_TYPE_PET)
			end
			local isControlled = bit_band(sourceFlags, COMBATLOG_OBJECT_TYPE_PET)
			if (wasControlled ~= 0) and (isControlled == 0) then
				-- the combatant was flagged as a controlled pet but is no longer, so remove ownership information
				-- note that we never do this for guardians and objects (which always remain pets)
				recap.Combatant[sourceNameGUID].OwnedBy = nil
			else
				-- probably still controlled, so slap the owner of record in front
				-- note that this could now be "owner:pet" or "owner:owner:pet" (and possibly stacked even higher though we haven't seen one yet, and our code doesn't currently handle triple-decker owners)
				sourceOwnerNameGUID = owner..":"..sourceNameGUID

				-- we don't yet have Hunter owning Snake Trap, and Snake Trap owning Viper, but we're ready ...
				-- Midnight summons Attumen who summons Attumen-on-horseback, three different GUIDs, leading to "owner:owner:pet"
				-- Illhoof summons Fiendish Portal which summons Fiendish Imps
				-- mind controlling someone who has a pet does not gain ownership of the pet, which becomes instead hostile to its now-mind-controlled former owner
				-- no examples so far of a mind controlled entity being able to summon a pet
			end
		else
			-- no recorded ownership, do nothing even if they are flagged as controlled
		end
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		-- existing combatant
		if recap.Combatant[destNameGUID].OwnedBy then
			-- there is recorded ownership
			local owner = recap.Combatant[destNameGUID].OwnedBy
			local wasControlled = 0
			if recap.Combatant[destNameGUID].Flags then
				wasControlled = bit_band(recap.Combatant[destNameGUID].Flags, COMBATLOG_OBJECT_TYPE_PET)
			end
			local isControlled = bit_band(destFlags, COMBATLOG_OBJECT_TYPE_PET)
			if (wasControlled ~= 0) and (isControlled == 0) then
				-- the combatant was flagged as a controlled pet but is no longer, so remove ownership information
				-- note that we never do this for guardians and objects (which always remain pets)
				recap.Combatant[destNameGUID].OwnedBy = nil
			else
				-- probably still controlled, so slap the owner of record in front
				-- note that this could now be "owner:pet" or "owner:owner:pet" (and possibly stacked even higher though we haven't seen one yet, and our code doesn't currently handle triple-decker owners)
				destOwnerNameGUID = owner..":"..destNameGUID
			end
		else
			-- no recorded ownership, do nothing even if they are flagged as controlled (would be an unknown owner)
		end
	end

	-- at this point we have both sourceNameGUID and sourceOwnerNameGUID (which may have no owner), and similarly for dest
	-- we use destOwnerNameGUID for everything that follows *except* for a summon or create event for which we must have an unqualified name

	if (sourceOwnerNameGUID ~= sourceNameGUID) or (destOwnerNameGUID ~= destNameGUID) then
		-- various processing for the fully-qualified names (already happened for the bare names)
		onIgnore = Recap_CheckCombatant(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, ...)
		if onIgnore then
			return
		end
	end


	local amount, school, resisted, blocked, absorbed, critical, glancing, crushing, missType, enviromentalType
	local spellID, spellName, spellSchool, powerType, extraAmount, extraSpellId, extraSpellName, extraSpellSchool, auraType
	local schoolName, powerName, extraSchoolName -- we translate the school number and the power type number to a localized name


	-- the Booleans (critical, glancing, crushing) come in as either 1 or nil (I changed Recap code to match this, so as to save unnecessary conversions)
	-- similarly missType arrives as all upper-case
	-- with patch 2.4.3 the schools (which are bit significant) can be multiple, e.g. Sunblade Protector spell Fel Lightning is now Shadow+Nature instead of Fire


	-- damaging events and misses

	if (eventType == "SWING_DAMAGE") then
		-- can be non-Physical damage (e.g. Melee that lands as Shadow damage)
		amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(9, ...)
		schoolName = Recap_DecodeSchoolName(school, ...)
		if amount and (amount > 0) then
			-- non-zero damage
			Recap_DamageEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, nil, RECAP_MELEE, 1, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)
		end
		return

	elseif (eventType == "SWING_MISSED") then
		missType = select(9, ...)
		if (missType == "MISS") or (missType == "ABSORB") or (missType == "DODGE") or (missType == "PARRY") or (missType == "BLOCK") or
		   (missType == "DEFLECT") or (missType == "EVADE") or (missType == "RESIST") or (missType == "REFLECT") or (missType == "IMMUNE") then
			-- expected, do nothing
		else
			-- unexpected miss type
			Recap_CheckUnexpectedEvent("unexpected miss type", ...)
			missType = "MISS"
		end
		schoolName = recap_temp.Localize.SchoolName[1] -- hardcoded to Physical
		Recap_MissEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, nil, RECAP_MELEE, 1, schoolName, missType)
		return

	elseif (eventType == "RANGE_DAMAGE") then
		-- for a wand the spellSchool is Physical but the school is e.g. Shadow, so don't check whether they differ
		spellID, spellName, spellSchool, amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(9, ...)
		schoolName = Recap_DecodeSchoolName(school, ...)
		if amount and (amount > 0) then
			-- non-zero damage
			Recap_DamageEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)
		end
		return

	elseif (eventType == "SPELL_DAMAGE") or (eventType == "SPELL_PERIODIC_DAMAGE") then
		-- implication is that PERIODIC damage could be critical, glancing, or crushing, but I've not seen that
		spellID, spellName, spellSchool, amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(9, ...)
		schoolName = Recap_DecodeSchoolName(school, ...)
		-- sometimes spellSchool does not equal school
		-- for example Warden Icoshock does a Shadowsurge with the spell being Shadow but with the damage caused being Physical
		-- we track the school according to the damage done, in this example Physical
		-- the fact that the spell is Shadow is not tracked
		-- presumably abilities that interrupt Shadow casting would block the spell
		if amount and (amount > 0) then
			-- non-zero damage
			Recap_DamageEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)
		end
		return

	elseif (eventType == "DAMAGE_SHIELD") then
		spellID, spellName, spellSchool, amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(9, ...)
		schoolName = Recap_DecodeSchoolName(school, ...)
		-- sometimes spellSchool does not equal school
		if amount and (amount > 0) then
			-- non-zero damage
			Recap_DamageEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)
		end
		return

	elseif (eventType == "DAMAGE_SPLIT") then
		-- not sure if SPLIT damage needs to be treated differently -- the two spell schools can certainly be different
		spellID, spellName, spellSchool, amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(9, ...)
		schoolName = Recap_DecodeSchoolName(school, ...)
		if amount and (amount > 0) then
			-- non-zero damage
			Recap_DamageEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)
		end
		return

	elseif (eventType == "RANGE_MISSED") or (eventType == "SPELL_MISSED") or (eventType == "SPELL_PERIODIC_MISSED") or (eventType == "DAMAGE_SHIELD_MISSED") then
		-- SPELL_PERIODIC_MISSED is, for example, a damage tick being absorbed
		spellID, spellName, spellSchool, missType = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		if (missType == "MISS") or (missType == "ABSORB") or (missType == "DODGE") or (missType == "PARRY") or (missType == "BLOCK") or
		   (missType == "DEFLECT") or (missType == "EVADE") or (missType == "RESIST") or (missType == "REFLECT") or (missType == "IMMUNE") then
			-- expected, do nothing
		else
			-- unexpected miss type
			Recap_CheckUnexpectedEvent("unexpected miss type", ...)
			missType = "MISS"
		end
		Recap_MissEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, missType)
		return

	elseif (eventType == "ENVIRONMENTAL_DAMAGE") then
		enviromentalType, amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(9, ...)
		schoolName = Recap_DecodeSchoolName(school, ...)
		if amount and (amount > 0) then
			-- non-zero damage
			Recap_EnvironmentDamageEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, enviromentalType, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)
		end
		return


	-- healing events

	elseif (eventType == "SPELL_HEAL") or (eventType == "SPELL_PERIODIC_HEAL") then
		spellID, spellName, spellSchool, amount, critical = select(9, ...)
		if amount and (amount > 0) then
			-- non-zero healing
			schoolName = Recap_DecodeSchoolName(spellSchool, ...)
			Recap_HealEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, critical)
		end
		return


	-- death events

	elseif (eventType == "UNIT_DIED") then
		-- no extra arguments
		-- note still absolutely no difference between priest death and priest second death after Spirit of Redemption
		Recap_DeathEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags)
		return
	elseif (eventType == "PARTY_KILL") then
		-- no extra arguments
		-- this event partially duplicates UNIT_DIED, so is used only for Recent Events
		Recap_KillEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags)
		return


	-- casting events that can affect standard data (and may add to Other data if that mode is enabled)

	elseif (eventType == "SPELL_CAST_SUCCESS") then
		-- this seems to show both source and destination, so we may not need to 'remember' from SPELL_CAST_START (which used to be the case)
		spellID, spellName, spellSchool = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		Recap_CastEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName)
		return

	elseif (eventType == "SPELL_INSTAKILL") then
		spellID, spellName, spellSchool = select(9, ...)
		-- for example, a Boom Bot self-destructing; a warlock sacrificing their demon; a paladin killing themselves during Divine Intervention
		-- will count twice, once as source (green) and once as destination (white)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		Recap_CastEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName)
		return


	-- summoning and creating can create pets (and may add to Other data if that mode is enabled)

	elseif (eventType == "SPELL_SUMMON") or (eventType == "SPELL_CREATE") then
		-- gives the GUID of a summoned creature or created object (e.g. totem or trap)
		-- we will associate these 'pets' with their owners (and if Blizzard supported it then we could do it transitively from hunter trap to snakes)
		-- this event does NOT happen for mind control
		spellID, spellName, spellSchool = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)

		-- NB: the dest uses the unqualified name (breaks with GUIDs off if we don't do this)
		Recap_SummonOrCreateEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName)
		return
	end

	-- here endeth the standard events


	-- here beginneth the Other events

	-- an Other event does not start a fight
	-- an Other event does not affect the idle timer
	-- an Other event does not affect the end of fight delay timer
	-- an Other event does not create a source or destination combatant, nor affect their combat timers

	-- some Other events apply partially even in Light Data Mode and with Other Data turned off, so process them first


	-- buffs and debuffs

	if (eventType == "SPELL_AURA_APPLIED") or (eventType == "SPELL_AURA_REFRESH") then
		-- source arguments are nil
		-- I don't currently distinguish these two event types (the latter being new with patch 2.4.3)
		spellID, spellName, spellSchool, auraType = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		Recap_BuffOrDebuffEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, auraType, 1)
		return

	elseif (eventType == "SPELL_AURA_APPLIED_DOSE") then
		-- source arguments are nil
		-- for example Sunder Armor, a stacking debuff
		spellID, spellName, spellSchool, auraType, amount = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		Recap_BuffOrDebuffEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, auraType, amount)
		return

	elseif (eventType == "SPELL_AURA_REMOVED") then
		-- source arguments are nil
		spellID, spellName, spellSchool, auraType = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		Recap_RemoveEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, auraType)
		return

	elseif (eventType == "SPELL_AURA_REMOVED_DOSE") then
		-- source arguments are nil
		-- spellID, spellName, spellSchool, auraType, amount = select(9, ...)
		-- I think this is where a stacked buff or debuff loses some but not all of its stacking
		-- ignore
		return

	elseif (eventType == "SPELL_DISPEL") then
		-- the first spell is the one causing the aura to be dispelled or otherwise cancelled, and the *extra* is the name of the aura that is being dispelled
		spellID, spellName, spellSchool, extraSpellId, extraSpellName, extraSpellSchool, auraType = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		extraSchoolName = Recap_DecodeSchoolName(extraSpellSchool, ...)
		Recap_DispelEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName, auraType)
		return

	elseif (eventType == "SPELL_AURA_BROKEN_SPELL") then
		-- okay, this one (new with patch 2.4.3) is backwards compared to SPELL_DISPEL, just to keep us addon authors on our toes
		-- the first spell is the name of the aura that is being dispelled, and the *extra* is the one causing the aura to be broken
		spellID, spellName, spellSchool, extraSpellId, extraSpellName, extraSpellSchool, auraType = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		extraSchoolName = Recap_DecodeSchoolName(extraSpellSchool, ...)
		Recap_DispelEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName, spellID, spellName, spellSchool, schoolName, auraType)
		return

	elseif (eventType == "SPELL_AURA_BROKEN") then
		-- new with patch 2.4.3
		-- I interpret this event as a physical attack (arbitrarily chosen to be 'Melee') dispelling an aura
		extraSpellId, extraSpellName, extraSpellSchool, auraType = select(9, ...)
		schoolName = recap_temp.Localize.SchoolName[1] -- hardcoded to Physical
		extraSchoolName = Recap_DecodeSchoolName(extraSpellSchool, ...)
		Recap_DispelEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, nil, RECAP_MELEE, 1, schoolName, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName, auraType)
		return

	elseif (eventType == "SPELL_STOLEN") then
		spellID, spellName, spellSchool, extraSpellId, extraSpellName, extraSpellSchool, auraType = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		extraSchoolName = Recap_DecodeSchoolName(extraSpellSchool, ...)
		Recap_StealEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName, auraType)
		return
	end


	-- the remaining other events are only processed if necessary

	if recap.Opt.LightData.value or not recap.Opt.OtherData.value then
		-- only track these Other events if not in Light Data Mode, and with the Other Data Mode enabled
		return
	end

	-- gains, drains, leeches, extra attacks

	if (eventType == "SPELL_ENERGIZE") or (eventType == "SPELL_PERIODIC_ENERGIZE") then
		spellID, spellName, spellSchool, amount, powerType = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		powerName = recap_temp.Localize.PowerName[powerType]
		if not powerName then
			Recap_CheckUnexpectedEvent("unexpected power type", ...)
			powerName = recap_temp.Localize.ElementOther
		end
		Recap_GainEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, powerType, powerName)
		return

	elseif (eventType == "SPELL_DRAIN") or (eventType == "SPELL_PERIODIC_DRAIN") then
		spellID, spellName, spellSchool, amount, powerType, extraAmount = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		powerName = recap_temp.Localize.PowerName[powerType]
		if not powerName then
			Recap_CheckUnexpectedEvent("unexpected power type", ...)
			powerName = recap_temp.Localize.ElementOther
		end
-- do we ever see an extraAmount, and if so what is it?
-- TODO
if recap_temp.snap then
if extraAmount ~= nil then
local evCom, evRest
evCom = "*****timestamp "..tostring(timestamp).."; eventType "..tostring(eventType).."; sourceGUID "..tostring(sourceGUID).."; sourceName "..tostring(sourceName).."; sourceFlags "..string_format("0x%X", sourceFlags).."; destGUID "..tostring(destGUID).."; destName "..tostring(destName).."; destFlags "..string_format("0x%X", destFlags)
evRest = "spellID "..tostring(spellID).."; spellName "..tostring(spellName).."; spellSchool "..tostring(spellSchool).."; schoolName "..tostring(schoolName).."; amount "..tostring(amount).."; powerType "..tostring(powerType).."; powerName "..tostring(powerName).."; extraAmount "..tostring(extraAmount)
 DEFAULT_CHAT_FRAME:AddMessage(evCom.."; "..evRest)
Screenshot()
end
end
		Recap_DrainEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, powerType, powerName, extraAmount)
		return

	elseif (eventType == "SPELL_LEECH") or (eventType == "SPELL_PERIODIC_LEECH") then
		-- could be Dark Pact moving mana from an imp to its warlock
		-- it seems that amount is the gain for the source and extraAmount is the loss for the dest
		spellID, spellName, spellSchool, amount, powerType, extraAmount = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		powerName = recap_temp.Localize.PowerName[powerType]
		if not powerName then
			Recap_CheckUnexpectedEvent("unexpected power type", ...)
			powerName = recap_temp.Localize.ElementOther
		end
		Recap_LeechEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, powerType, powerName, extraAmount)
		return

	elseif (eventType == "SPELL_EXTRA_ATTACKS") then
		spellID, spellName, spellSchool, amount = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		Recap_ExtraAttackEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount)
		return


	-- interrupting

	elseif (eventType == "SPELL_INTERRUPT") then
		-- spellName does the interrupting, extraSpellName is what gets interrupted
		spellID, spellName, spellSchool, extraSpellId, extraSpellName, extraSpellSchool = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
		extraSchoolName = Recap_DecodeSchoolName(extraSpellSchool, ...)
		Recap_InterruptEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName)
		return


	-- durability

	elseif (eventType == "SPELL_DURABILITY_DAMAGE") or (eventType == "SPELL_DURABILITY_DAMAGE_ALL") then
		-- we are not currently distinguishing between these two, and I've never actually captured one of these
		spellID, spellName, spellSchool = select(9, ...)
		schoolName = Recap_DecodeSchoolName(spellSchool, ...)
-- do we ever see a durability effect?
-- TODO
if recap_temp.snap then
local evCom, evRest
evCom = "*****timestamp "..tostring(timestamp).."; eventType "..tostring(eventType).."; sourceGUID "..tostring(sourceGUID).."; sourceName "..tostring(sourceName).."; sourceFlags "..string_format("0x%X", sourceFlags).."; destGUID "..tostring(destGUID).."; destName "..tostring(destName).."; destFlags "..string_format("0x%X", destFlags)
evRest = "spellID "..tostring(spellID).."; spellName "..tostring(spellName).."; spellSchool "..tostring(spellSchool).."; schoolName "..tostring(schoolName)
 DEFAULT_CHAT_FRAME:AddMessage(evCom.."; "..evRest)
Screenshot()
end
		Recap_DurabilityEvent(timestamp, eventType, sourceOwnerNameGUID, sourceFlags, destOwnerNameGUID, destFlags, spellID, spellName, spellSchool, schoolName)
		return


	-- other casting events

	elseif (eventType == "SPELL_CAST_START") then
		-- spellID, spellName, spellSchool = select(9, ...)
		-- ignore
		return
	elseif (eventType == "SPELL_CAST_FAILED") then
		-- spellID, spellName, spellSchool, missType = select(9, ...)
		-- ignore
		return
	elseif (eventType == "SPELL_DISPEL_FAILED") then
		-- spellID, spellName, spellSchool, extraSpellId, extraSpellName, extraSpellSchool = select(9, ...)
		-- ignore
		return


	-- ignored events

	elseif (eventType == "UNIT_DESTROYED") then
		-- for example, summoned elemental ending its allotted life
		-- no extra arguments
		-- ignore
		return
	elseif (eventType == "ENCHANT_APPLIED") then
		-- local itemId, itemName
		-- spellName, itemId, itemName = select(9, ...)
		-- ignore
		return
	elseif (eventType == "ENCHANT_REMOVED") then
		-- local itemId, itemName
		-- spellName, itemId, itemName = select(9, ...)
		-- ignore
		return


	-- unexpected events (it is only possible to get here with Other Data Mode turned on)

	else
		-- dump any others to find ones that we have missed
		Recap_CheckUnexpectedEvent("unexpected event", ...)
	end
end


function Recap_CheckCombatant(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, ...)

	local wasControlled, isControlled

	-- check for ignores on the name before we go any further
	if (recap.Opt.SyncState.value == "Member") or (recap.Opt.SyncState.value == "Leader") then
		-- in synchronization, don't ignore anyone (if on 'ignore' they will still be hidden on the Fights panel)
	else
		if (sourceNameGUID and recap.Combatant[sourceNameGUID] and recap.Combatant[sourceNameGUID].Ignore) then
			-- sourceNameGUID is an ignored combatant, pretend the event didn't happen at all
			return true
		end
		if (destNameGUID and recap.Combatant[destNameGUID] and recap.Combatant[destNameGUID].Ignore) then
			-- destNameGUID is an ignored combatant, pretend the event didn't happen at all
			return true
		end
	end

	-- if the combatant exists but has no First Seen time, add it
	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		if (recap.Combatant[sourceNameGUID].Seen == nil) or (recap.Combatant[sourceNameGUID].Seen == 0) then
			recap.Combatant[sourceNameGUID].Seen = math_floor(1000*timestamp)
		end
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		if (recap.Combatant[destNameGUID].Seen == nil) or (recap.Combatant[destNameGUID].Seen == 0) then
			recap.Combatant[destNameGUID].Seen = math_floor(1000*timestamp)
		end
	end

	-- if the combatant exists, update flags
	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		if sourceFlags and (sourceFlags > 0) then
			wasControlled = 0
			if recap.Combatant[sourceNameGUID].Flags then
				wasControlled = bit_band(recap.Combatant[sourceNameGUID].Flags, COMBATLOG_OBJECT_TYPE_PET)
			end
			isControlled = bit_band(sourceFlags, COMBATLOG_OBJECT_TYPE_PET)
			if (wasControlled == 0) and (isControlled ~= 0) then
				-- was not controlled as a pet, is now, so check friends immediately in case it is a new pet for us
				Recap_MakeFriends()
			end
			recap.Combatant[sourceNameGUID].Flags = sourceFlags
		end
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		if destFlags and (destFlags > 0) then
			wasControlled = 0
			if recap.Combatant[destNameGUID].Flags then
				wasControlled = bit_band(recap.Combatant[destNameGUID].Flags, COMBATLOG_OBJECT_TYPE_PET)
			end
			isControlled = bit_band(destFlags, COMBATLOG_OBJECT_TYPE_PET)
			if (wasControlled == 0) and (isControlled ~= 0) then
				-- was not controlled as a pet, is now, so check friends immediately in case it is a new pet for us
				Recap_MakeFriends()
			end
			recap.Combatant[destNameGUID].Flags = destFlags
		end
	end

	-- if the combatant exists and is Player-owned, add a Trash value of false
	-- Trash is nil if not known, true if we have counted more than one of them, and false if the combatant is Player-owned
	if sourceNameGUID and recap.Combatant[sourceNameGUID] and (bit_band(sourceFlags, COMBATLOG_OBJECT_CONTROL_PLAYER) ~= 0) then
		recap.Combatant[sourceNameGUID].Trash = false -- keep as false
	end
	if destNameGUID and recap.Combatant[destNameGUID] and (bit_band(destFlags, COMBATLOG_OBJECT_CONTROL_PLAYER) ~= 0) then
		recap.Combatant[destNameGUID].Trash = false -- keep as false
	end

	-- if the source combatant is marked as Friend, and is not in the group, and is doing something hostile to someone who is in the group, then remove the Friend mark (heuristic only)
	if sourceNameGUID and destNameGUID and recap.Combatant[sourceNameGUID] and recap.Combatant[destNameGUID] and recap.Combatant[sourceNameGUID].Friend and (not recap_temp.InGroup[sourceNameGUID]) and recap_temp.InGroup[destNameGUID] then
		if Recap_HostileEvent(eventType, sourceNameGUID, destNameGUID, ...) then
			recap.Combatant[sourceNameGUID].Friend = false
			recap_temp.InFriend[sourceNameGUID] = nil
		end
	end
	-- similarly if someone in the group is doing something hostile to a dest combatant marked as Friend but who is not in the group
	if sourceNameGUID and destNameGUID and recap.Combatant[sourceNameGUID] and recap.Combatant[destNameGUID] and recap.Combatant[destNameGUID].Friend and (not recap_temp.InGroup[destNameGUID]) and recap_temp.InGroup[sourceNameGUID] then
		if Recap_HostileEvent(eventType, sourceNameGUID, destNameGUID, ...) then
			recap.Combatant[destNameGUID].Friend = false
			recap_temp.InFriend[destNameGUID] = nil
		end
	end

	return false
end


-- Is this a hostile event? (heuristic only, we won't be able to spot all possible hostile events)
-- These are 'hostile' events in two specific contexts
--   The first is casting between an owner and its apparent pet, or from pet to apparent owner
--	   So if, for example, an owner successfully damages a pet with melee, or vice versa, we can I think safely conclude that the pet is not currently controlled by the (former) owner
--	   I'm trying to guess which events would only ever be permitted to happen by Blizzard between entities which are at that instant hostile to each other, and hence not currently in a proper owner-pet relationship
--	   Note that it is frequently the case that we don't learn the initial OwnedBy relationship and so an NPC Imp Minion (say) begins with an unknown owner
--   The second is casting between a member of the group and an apparent Friend, who may have earlier been a temporary pet (e.g. mind controlled) but who is now presumably hostile again
function Recap_HostileEvent(eventType, sourceNameGUID, destNameGUID, ...)

	local auraType, spellTypeName

	if (eventType == "SWING_DAMAGE") or (eventType == "SWING_MISSED") or (eventType == "RANGE_DAMAGE") or (eventType == "RANGE_MISSED") or
	   (eventType == "SPELL_DAMAGE") or (eventType == "SPELL_MISSED") or (eventType == "DAMAGE_SHIELD") or (eventType == "DAMAGE_SHIELD_MISSED") or
	   (eventType == "SPELL_INTERRUPT") then
		-- we assume that this is hostile although it might be involuntary (as in Shatter damage which is presumably SPELL_DAMAGE)
		return true
	else
		-- more complex situations (no guarantee that these are always hostile, because they might be involuntary)
		if (eventType == "SPELL_AURA_APPLIED") or (eventType == "SPELL_AURA_APPLIED_DOSE") then
			-- source debuffing dest
			auraType = select(12, ...)
			if (auraType == "DEBUFF") then
				-- debuff applied
				return true
			end
		elseif (eventType == "SPELL_DISPEL") or (eventType == "SPELL_AURA_STOLEN") then
			-- source dispelling or stealing dest's buff
			auraType = select(12, ...)
			if (auraType == "BUFF") then
				-- buff dispelled
				return true
			end
		elseif (eventType == "SPELL_CAST_SUCCESS") then
			-- while a DoT tick could legitimately do damage between friends, I think Blizzard would not permit a successful damaging spell *cast* between friends
			spellTypeName = "1"..select(10, ...)
			if (recap.Combatant[sourceNameGUID].OutgoingDetail and recap.Combatant[sourceNameGUID].OutgoingDetail[spellTypeName]) or
			   (recap.Combatant[sourceNameGUID].IncomingDetail and recap.Combatant[sourceNameGUID].IncomingDetail[spellTypeName]) or
			   (recap.Combatant[destNameGUID].OutgoingDetail and recap.Combatant[destNameGUID].OutgoingDetail[spellTypeName]) or
			   (recap.Combatant[destNameGUID].IncomingDetail and recap.Combatant[destNameGUID].IncomingDetail[spellTypeName]) then
				-- this is a cast of a spell which has done damage before (we check only the source and dest, not the details for all combatants, so we might miss a damaging cast)
				return true
			end
		end
		-- probably friendly: SPELL_HEAL, SPELL_PERIODIC_HEAL, SPELL_SUMMON, SPELL_CREATE, SPELL_ENERGIZE, SPELL_PERIODIC_ENERGIZE, ENCHANT_APPLIED, ENCHANT_REMOVED
		-- potentially ambiguous: PARTY_KILL, SPELL_PERIODIC_DAMAGE, SPELL_PERIODIC_MISSED, DAMAGE_SPLIT, SPELL_DRAIN, SPELL_PERIODIC_DRAIN, SPELL_LEECH, SPELL_PERIODIC_LEECH
		--   for example: the SPELL_PERIODIC_DAMAGE is potentially ambiguous if I DoT someone then mind control them while the DoT is still active -- the tick will do damage but is not hostile for this purpose
		-- potentially ambiguous: SPELL_CAST_START, SPELL_CAST_FAILED, SPELL_DISPEL_FAILED
		-- other: ENVIRONMENTAL_DAMAGE, UNIT_DIED, SPELL_INSTAKILL, SPELL_EXTRA_ATTACKS, SPELL_AURA_REMOVED, SPELL_AURA_REMOVED_DOSE, SPELL_DURABILITY_DAMAGE, SPELL_DURABILITY_DAMAGE_ALL, UNIT_DESTROYED
	end
	return false
end

function Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)

	-- does a combatant have "do not store" status due to the setting of options?
	if (not recap.Opt.StoreOnlyDisplayed.value) or (recap.Opt.SyncState.value == "Member") or (recap.Opt.SyncState.value == "Leader") then
		-- in synchronization, or the StoreOnlyDisplayed option is off, carry on as normal
	else
		-- the StoreOnlyDisplayed option is on
		if sourceNameGUID then
			-- note that 'friends' includes group members, former group members, hostiles controlled for a while, and combatants explicitly named as friends
			if recap.Opt.HideOthers.value and recap.Combatant[sourceNameGUID] and (not recap.Combatant[sourceNameGUID].Friend) then
				sourceNameGUID = nil
			elseif recap.Opt.HideGroup.value and (not Recap_CheckForSelf(sourceNameGUID)) and recap.Combatant[sourceNameGUID] and recap.Combatant[sourceNameGUID].Friend then
				sourceNameGUID = nil
			end
		end
		if destNameGUID then
			-- note that 'friends' includes group members, former group members, hostiles controlled for a while, and combatants explicitly named as friends
			if recap.Opt.HideOthers.value and recap.Combatant[destNameGUID] and (not recap.Combatant[destNameGUID].Friend) then
				destNameGUID = nil
			elseif recap.Opt.HideGroup.value and (not Recap_CheckForSelf(destNameGUID)) and recap.Combatant[destNameGUID] and recap.Combatant[destNameGUID].Friend then
				destNameGUID = nil
			end
		end
	end

	return sourceNameGUID, destNameGUID
end

function Recap_DecodeSchoolName(school, ...)
	-- needed beginning with patch 2.4.3, which introduces composite school types
	local i, result
	result = ""
	if recap_temp.Localize.SchoolName[school] then
		-- shortcut, we have a simple match
		result = recap_temp.Localize.SchoolName[school]
	else
		-- perhaps it is a composite
		for i in pairs(recap_temp.Localize.SchoolName) do
			if bit_band(school, i) ~= 0 then
				if result == "" then
					result = recap_temp.Localize.SchoolName[i]
				else
					result = result.."+"..recap_temp.Localize.SchoolName[i]
				end
			end
		end
		if result == "" then
			-- still no luck
			Recap_CheckUnexpectedEvent("unexpected school, spell school, or extra spell school ("..school..")", ...)
			result = recap_temp.Localize.ElementOther
		end
	end
	return result
end

function Recap_DamageEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)

	-- currently ignoring sourceFlags, destFlags
	local sourceCombatant, destCombatant, trackAsOutgoing

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- let's start the timers
	if sourceNameGUID then
		Recap_CreateCombatantAndUpdateTimers(sourceNameGUID, timestamp, sourceFlags, "Out")
	end
	if destNameGUID then
		Recap_CreateCombatantAndUpdateTimers(destNameGUID, timestamp, destFlags, "In")
	end

	-- table lookup shortcuts
	if sourceNameGUID then
		sourceCombatant = recap.Combatant[sourceNameGUID]
	else
		sourceCombatant = nil
	end
	if destNameGUID then
		destCombatant = recap.Combatant[destNameGUID]
	else
		destCombatant = nil
	end

	-- assert: caller has already checked that 'amount' is greater than zero

	trackAsOutgoing = true
	if sourceNameGUID and destNameGUID then
		-- we don't track the Outgoing component of some damage (self-damage, damage between owner and pet, and damage from one group member to another group member) (otherwise would inflate DPS)
		-- we still track the Incoming component and the Matrix component
		trackAsOutgoing = Recap_TrackAsOutgoing(sourceNameGUID, destNameGUID)
	end
	if sourceNameGUID and trackAsOutgoing then
		local iLast = recap_temp.Last[sourceNameGUID]
		iLast.DmgOut = iLast.DmgOut + amount
		if amount > iLast.MaxHit then
			iLast.MaxHit = amount
		end
	end

	if destNameGUID then
		iLast = recap_temp.Last[destNameGUID]
		iLast.DmgIn = iLast.DmgIn + amount
		if iLast.HP then
			iLast.HP = iLast.HP - amount
		end
	end

	if sourceNameGUID and destNameGUID then -- guard
		-- special case code for paladin Judgement of Light
		if (recap_temp.JoLRegistration[destNameGUID]) then
			-- there is probably a Judgement of Light registered on the target
			Recap_RegisterJoLHit(timestamp, sourceNameGUID, destNameGUID)
		end
	end

	if not recap.Opt.LightData.value then
		-- determine whether it could have critted
		if string_find(eventType, "PERIODIC", 1, true) then

			-- periodic event, assume not crit-capable, pass 'nil' in the crit slot (and in the crush and glance slots) -- keep crit and crush and glance as 'nil', not false
			Recap_CreateEffect(timestamp, sourceNameGUID, destNameGUID, spellName, 1, amount, 0, nil, 1, schoolName, false, nil, nil, absorbed, blocked, resisted, true, "OutgoingDetail", "IncomingDetail")
			Recap_CreateEffect(timestamp, sourceNameGUID, destNameGUID, spellName, 1, amount, 0, nil, 1, schoolName, false, nil, nil, absorbed, blocked, resisted, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)

		else
			-- assume all others are crit or crush or glance capable
			local critty, crushy, glancy
			critty = critical or false
			crushy = crushing or false
			glancy = glancing or false
			Recap_CreateEffect(timestamp, sourceNameGUID, destNameGUID, spellName, 1, amount, 0, critty, 1, schoolName, false, crushy, glancy, absorbed, blocked, resisted, true, "OutgoingDetail", "IncomingDetail")
			Recap_CreateEffect(timestamp, sourceNameGUID, destNameGUID, spellName, 1, amount, 0, critty, 1, schoolName, false, crushy, glancy, absorbed, blocked, resisted, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
		end

		if recap.Opt.MatrixData.value then
			Recap_AccumulateMatrixDetails((sourceNameGUID or recap_temp.Localize.Unknown), (sourceFlags or 0), (destNameGUID or recap_temp.Localize.Unknown), (destFlags or 0), 1, amount, timestamp, "TargetDetail", "SourceDetail")
			Recap_AccumulateMatrixDetails((sourceNameGUID or recap_temp.Localize.Unknown), (sourceFlags or 0), (destNameGUID or recap_temp.Localize.Unknown), (destFlags or 0), 1, amount, timestamp, "LastTargetDetail_"..recap_temp.ActiveLastFight, "LastSourceDetail_"..recap_temp.ActiveLastFight)
		end
	end

	if recap.Opt.RecentData.value then
		Recap_AddRecentEvent(timestamp, sourceNameGUID, destNameGUID, 1, spellName, schoolName, amount, critical, crushing)
	end
end

function Recap_RegisterJoLHit(timestamp, sourceNameGUID, destNameGUID)

	local regDest

	-- check whether this is a standard registered JoL on the target
	if recap_temp.JoLRegistration[destNameGUID] and recap_temp.JoLRegistration[destNameGUID].OriginalCaster and recap_temp.JoLRegistration[destNameGUID].Light then
		-- fully registered, create a record for the hitter (overwrites any existing record)
		-- this registers everyone who does damage (or misses), not just the melee who are the only ones who might get heals -- just a bit of extra storage, no harm
		if not recap_temp.JoLRegistrationHitter[sourceNameGUID] then
			recap_temp.JoLRegistrationHitter[sourceNameGUID] = {}
		end
		regDest = recap_temp.JoLRegistrationHitter[sourceNameGUID]
		regDest.OriginalCaster = recap_temp.JoLRegistration[destNameGUID].OriginalCaster
		regDest.Timestamp = timestamp
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." JoL new registration hitter for "..sourceNameGUID.." on behalf of source "..recap_temp.JoLRegistration[destNameGUID].OriginalCaster)
	else
		-- not fully registered
		if regDest then
			-- we have an existing record for this hitter, check the timestamp
			if (timestamp - regDest.Timestamp) > 20 then
				-- clear the existing record only if the timestamp is over 20 seconds old
				-- allows us to hit other things (e.g. with AoE) without losing registration for JoL
				recap_temp.JoLRegistrationHitter[sourceNameGUID] = nil
			end
		end
	end
	-- the JoLRegistrationHitter table could get a lot of thrashing in the presence of Judgement of Light
end

function Recap_MissEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, missType)

	local sourceCombatant, destCombatant, iLast

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- let's start the timers
	if sourceNameGUID then
		Recap_CreateCombatantAndUpdateTimers(sourceNameGUID, timestamp, sourceFlags, "Out")
	end
	if destNameGUID then
		Recap_CreateCombatantAndUpdateTimers(destNameGUID, timestamp, destFlags, "In")
	end

	-- table lookup shortcuts
	if sourceNameGUID then
		sourceCombatant = recap.Combatant[sourceNameGUID]
	else
		sourceCombatant = nil
	end
	if destNameGUID then
		destCombatant = recap.Combatant[destNameGUID]
	else
		destCombatant = nil
	end

	if sourceNameGUID and destNameGUID then -- guard
		-- special case code for paladin Judgement of Light
		if (recap_temp.JoLRegistration[destNameGUID]) then
			-- there is probably a Judgement of Light registered on the target
			Recap_RegisterJoLHit(timestamp, sourceNameGUID, destNameGUID)
		end
	end

	if not recap.Opt.LightData.value then
		if (sourceNameGUID == destNameGUID) then
			-- ignore misses related to self-damage (if any)
		else
			local typeEffect, total
			typeEffect = "1"..spellName
			total = 0
			if sourceCombatant and sourceCombatant.OutgoingDetail and sourceCombatant.OutgoingDetail[typeEffect] then
				local iEffect = sourceCombatant.OutgoingDetail[typeEffect]
				total = (iEffect.GlancesDmg or 0) + (iEffect.HitsDmg or 0) + (iEffect.CritsDmg or 0) + (iEffect.CrushDmg or 0) + (iEffect.TicksDmg or 0)
			end
			local theDetail = "LastOutgoingDetail_"..recap_temp.ActiveLastFight
			if sourceCombatant and sourceCombatant[theDetail] and sourceCombatant[theDetail][typeEffect] then
				local iEffect = sourceCombatant[theDetail][typeEffect]
				total = (iEffect.GlancesDmg or 0) + (iEffect.HitsDmg or 0) + (iEffect.CritsDmg or 0) + (iEffect.CrushDmg or 0) + (iEffect.TicksDmg or 0)
			end
			-- hmm, Arcane Orb (Void Reaver fight) does damage but has no sourceNameGUID, so total from sourceNameGUID is always zero
			-- check whether the target has taken damage from this spellName
			if destCombatant and destCombatant.IncomingDetail and destCombatant.IncomingDetail[typeEffect] then
				local iEffect = destCombatant.IncomingDetail[typeEffect]
				total = total + (iEffect.GlancesDmg or 0) + (iEffect.HitsDmg or 0) + (iEffect.CritsDmg or 0) + (iEffect.CrushDmg or 0) + (iEffect.TicksDmg or 0)
			end
			local theDetail = "LastIncomingDetail_"..recap_temp.ActiveLastFight
			if destCombatant and destCombatant[theDetail] and destCombatant[theDetail][typeEffect] then
				local iEffect = destCombatant[theDetail][typeEffect]
				total = total + (iEffect.GlancesDmg or 0) + (iEffect.HitsDmg or 0) + (iEffect.CritsDmg or 0) + (iEffect.CrushDmg or 0) + (iEffect.TicksDmg or 0)
			end
			if total > 0 then
				-- this spellName has done damage, so count the miss directly
				-- register the miss as a non-crit-capable event -- keep crit and crush and glance as 'nil', not false
				Recap_CreateEffect(timestamp, sourceNameGUID, destNameGUID, spellName, 1, 0, 0, nil, nil, schoolName, missType, nil, nil, nil, nil, nil, true, "OutgoingDetail", "IncomingDetail")
				Recap_CreateEffect(timestamp, sourceNameGUID, destNameGUID, spellName, 1, 0, 0, nil, nil, schoolName, missType, nil, nil, nil, nil, nil, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
				if recap.Opt.RecentData.value then
					local missText = (missType or "MISS")
					Recap_AddRecentEvent(timestamp, sourceNameGUID, destNameGUID, 1, spellName, schoolName, string_lower(missText), false, false)
				end
			else
				-- this spellName has not done damage, treat it as a miss for any matching cast, debuff, or buff
				local missText = (missType or "MISS")
				Recap_CreateMissEffect(sourceNameGUID, destNameGUID, spellName, missText, "OtherDetail")
				Recap_CreateMissEffect(sourceNameGUID, destNameGUID, spellName, missText, "LastOtherDetail_"..recap_temp.ActiveLastFight)
			end
		end
	end
end

function Recap_EnvironmentDamageEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, enviromentalType, amount, school, schoolName, resisted, blocked, absorbed, critical, glancing, crushing)

	local effect, element, iLast

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- table lookup shortcut
	destCombatant = recap.Combatant[destNameGUID]

	-- effect name is e.g. "Environment Fire"
	effect = RECAP_ENVIRONMENT.." "..string_upper(string_sub(enviromentalType,1,1))..string_lower(string_sub(enviromentalType,2))

	-- start the timers
	Recap_CreateCombatantAndUpdateTimers(destNameGUID, timestamp, destFlags, "In")

	if amount and (amount > 0) then
		iLast = recap_temp.Last[destNameGUID]
		iLast.DmgIn = iLast.DmgIn + amount
		if iLast.HP then
			iLast.HP = iLast.HP - amount
		end

		if not recap.Opt.LightData.value then
			-- environment effect
			Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, amount, 0, false, nil, schoolName, false, false, false, absorbed, blocked, resisted, true, "OutgoingDetail", "IncomingDetail")
			Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, amount, 0, false, nil, schoolName, false, false, false, absorbed, blocked, resisted, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
		end

		if recap.Opt.RecentData.value then
			Recap_AddRecentEvent(timestamp, false, destNameGUID, 1, effect, schoolName, amount, false, false)
		end
	else
		-- no actual damage, treat as a miss
		if not recap.Opt.LightData.value then
			if absorbed and (absorbed > 0) then
				-- entire amount absorbed
				-- register the miss as a non-crit-capable event -- keep crit and crush and glance as 'nil', not false
				-- ( this one happened, an ordinary fire that did 12 damage and was entirely absorbed )
				Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, 0, 0, nil, nil, schoolName, "ABSORB", nil, nil, nil, nil, nil, true, "OutgoingDetail", "IncomingDetail")
				Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, 0, 0, nil, nil, schoolName, "ABSORB", nil, nil, nil, nil, nil, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
			elseif blocked and (blocked > 0) then
				-- entire amount blocked
				-- ( this might never occur )
				Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, 0, 0, nil, nil, schoolName, "BLOCK", nil, nil, nil, nil, nil, true, "OutgoingDetail", "IncomingDetail")
				Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, 0, 0, nil, nil, schoolName, "BLOCK", nil, nil, nil, nil, nil, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
			elseif resisted and (resisted > 0) then
				-- entire amount resisted
				-- ( this might never occur )
				Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, 0, 0, nil, nil, schoolName, "RESIST", nil, nil, nil, nil, nil, true, "OutgoingDetail", "IncomingDetail")
				Recap_CreateEffect(timestamp, false, destNameGUID, effect, 1, 0, 0, nil, nil, schoolName, "RESIST", nil, nil, nil, nil, nil, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
			else
				-- no damage and not a miss
				-- ignore at this time
			end
		end
	end
end

function Recap_HealEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, critical)

	local rawheal, actualheal, overheal, sourceCombatant, destCombatant, iLast, localSourceNameGUID, newSourceNameGUID

	rawheal = tonumber(amount)

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- special case code might override this default source
	localSourceNameGUID = sourceNameGUID

	if destNameGUID then -- guard
		-- special case code for priest Prayer of Mending
		if (spellID == recap_temp.PoMHealID) then
			newSourceNameGUID = Recap_PoMHeal(timestamp, destNameGUID)
			if newSourceNameGUID then
				localSourceNameGUID = newSourceNameGUID
			end
		end

		-- special case code for druid Lifebloom
		if (spellID == recap_temp.LBHealID) then
			newSourceNameGUID = Recap_LBHeal(timestamp, destNameGUID)
			if newSourceNameGUID then
				localSourceNameGUID = newSourceNameGUID
			end
		end

		-- special case code for shaman Earth Shield
		if (spellID == recap_temp.ESHealID) then
			newSourceNameGUID = Recap_ESHeal(timestamp, destNameGUID)
			if newSourceNameGUID then
				localSourceNameGUID = newSourceNameGUID
			end
		end

		-- special case code for paladin Judgement of Light
		if (spellID == recap_temp.JoLHealID) then
			newSourceNameGUID = Recap_JoLHeal(timestamp, destNameGUID)
			if newSourceNameGUID then
				localSourceNameGUID = newSourceNameGUID
			end
		end

		-- special case code for druid Improved Leader of the Pack
		if (spellID == recap_temp.ILotPHealID) then
			newSourceNameGUID = Recap_ILotPHeal(timestamp, destNameGUID)
			if newSourceNameGUID then
				localSourceNameGUID = newSourceNameGUID
			end
		end

		if sourceNameGUID then -- guard
			-- special case code for druid Lifebloom tick (we might need to create a synthetic registration based on seeing the tick happen)
			if (spellID == recap_temp.LBTickID) then
				Recap_LBTick(timestamp, sourceNameGUID, destNameGUID)
			end
		end
	end

	-- Now keeping an independent timer for healing, and setting the 'WasInCurrent' flag for the healer
	-- Now also start the timers for the recipient of a heal
	if localSourceNameGUID then
		Recap_CreateCombatantAndUpdateTimers(localSourceNameGUID, timestamp, sourceFlags, "Heal")
	end
	if destNameGUID then
		Recap_CreateCombatantAndUpdateTimers(destNameGUID, timestamp, destFlags, "Heal")
	end

	-- table lookup shortcuts
	if localSourceNameGUID then
		sourceCombatant = recap.Combatant[localSourceNameGUID]
	else
		sourceCombatant = nil
	end
	if destNameGUID then
		destCombatant = recap.Combatant[destNameGUID]
	else
		destCombatant = nil
	end

	if rawheal and (rawheal > 0) then

		actualheal = rawheal
		overheal = 0
		if destNameGUID then
			iLast = recap_temp.Last[destNameGUID]
			if iLast and iLast.HP then
				-- reduce 'actualheal' if we know the hit points of the target
				actualheal = math_max(0,math_min(iLast.MaxHP-iLast.HP,rawheal))
				iLast.HP = iLast.HP + actualheal
				overheal = rawheal - actualheal
			end
		end

		-- two hidden columns, gross healing (RawHeal) and overhealing (OverHeal)
		-- a calculated column, Heal, includes actual healing
		if localSourceNameGUID then
			iLast = recap_temp.Last[localSourceNameGUID]
			iLast.RawHeal = iLast.RawHeal + rawheal
			iLast.OverHeal = math_min((iLast.OverHeal + overheal),iLast.RawHeal)
		end

		if not recap.Opt.LightData.value then
			-- the effect is credited with the full value of the heal
			-- determine whether it could have critted
			if string_find(eventType, "PERIODIC", 1, true) then
				-- periodic event, assume not crit-capable, pass 'nil' in the crit slot -- keep crit and crush and glance as 'nil', not false
				Recap_CreateEffect(timestamp, localSourceNameGUID, destNameGUID, spellName, 3, rawheal, overheal, nil, 1, schoolName, false, nil, nil, nil, nil, nil, true, "OutgoingDetail", "IncomingDetail")
				Recap_CreateEffect(timestamp, localSourceNameGUID, destNameGUID, spellName, 3, rawheal, overheal, nil, 1, schoolName, false, nil, nil, nil, nil, nil, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
			else
				-- assume all others are crit capable
				local critty
				critty = critical or false
				Recap_CreateEffect(timestamp, localSourceNameGUID, destNameGUID, spellName, 3, rawheal, overheal, critty, nil, schoolName, false, false, false, nil, nil, nil, true, "OutgoingDetail", "IncomingDetail")
				Recap_CreateEffect(timestamp, localSourceNameGUID, destNameGUID, spellName, 3, rawheal, overheal, critty, nil, schoolName, false, false, false, nil, nil, nil, false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight, "LastIncomingDetail_"..recap_temp.ActiveLastFight)
			end

			if recap.Opt.MatrixData.value and (actualheal > 0) then
				Recap_AccumulateMatrixDetails((localSourceNameGUID or recap_temp.Localize.Unknown), (sourceFlags or 0), (destNameGUID or recap_temp.Localize.Unknown), (destFlags or 0), 3, actualheal, timestamp, "TargetDetail", "SourceDetail")
				Recap_AccumulateMatrixDetails((localSourceNameGUID or recap_temp.Localize.Unknown), (sourceFlags or 0), (destNameGUID or recap_temp.Localize.Unknown), (destFlags or 0), 3, actualheal, timestamp, "LastTargetDetail_"..recap_temp.ActiveLastFight, "LastSourceDetail_"..recap_temp.ActiveLastFight)
			end
		end

		if recap.Opt.RecentData.value then
			Recap_AddRecentEvent(timestamp, localSourceNameGUID, destNameGUID, 3, spellName, schoolName, rawheal, critical, false)
		end
	end
end

function Recap_PoMHeal(timestamp, destNameGUID)
	-- Prayer of Mending heal going off
	-- datum: in fight against Pathaleon when he mind-controlled me it looks like a PoM went from me to him and back (or something like that)
	-- datum: the Supremus fight is sufficiently spread out that not all events are received due to range issues, so for example we see a Remove Aura event but do not see the matching PoM Heal event

	local newSourceNameGUID

	newSourceNameGUID = Recap_FindPoMSourceInRegistrationPriorToHeal(timestamp, destNameGUID)
	if newSourceNameGUID then
		-- source found in Registration table (newSourceNameGUID can equal sourceNameGUID if the priest's own PoM is doing the healing on the priest)
		-- if there had already been a migration (due to seeing the fading of the aura), then the registration will have been removed
		-- if there had not been a migration, then a migration will have been synthesized
		-- in either case the registration will have been removed
		return newSourceNameGUID
	else
		-- PoM not registered in the standard way, try something else
		-- there is evidence that there is not always an Add Aura event, so go check the Migration record (any new synthetic registration will have a timestamp that is perhaps too generous)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..destNameGUID.." has no registered aura -- about to look for migration record")
		-- since we will be using the youngest record found, we need to check only once, using the full length window
		Recap_MovePoMFromMigrationToRegistration(timestamp, destNameGUID, false, 29.3)
		-- try the heal again (we could have provided a return value from the Move call to tell us whether we'd find one, but checking Registration again is fast so don't bother)
		newSourceNameGUID = Recap_FindPoMSourceInRegistrationPriorToHeal(timestamp, destNameGUID)
		if newSourceNameGUID then
			-- source was found in the Registration table (having just been successfully moved from the Migration table)
			-- a further migration will have been synthesized, and the registration will have been removed
			return newSourceNameGUID
		else
			-- still no luck, apply an heuristic assignment randomly in proportion to the casting of PoM during the current fight (with separate counts for group and non-group)
			local i, randomCast
			newSourceNameGUID = nil -- to signal failure to find any casts
			if recap_temp.InGroup[destNameGUID] then
				if (recap_temp.PoMTotalCastsGroup > 0) then
					-- check only group
					randomCast = math.random(recap_temp.PoMTotalCastsGroup)
					for i in pairs(recap_temp.PoMCasts) do
						if recap_temp.InGroup[i] then
							-- only from group
							if randomCast <= recap_temp.PoMCasts[i] then
								-- this is the heuristic source to use
								newSourceNameGUID = i
								break
							else
								-- keep looking
								randomCast = randomCast - recap_temp.PoMCasts[i]
							end
						end
					end
				end
			else
				if (recap_temp.PoMTotalCastsNonGroup > 0) then
					-- check only non-group
					randomCast = math.random(recap_temp.PoMTotalCastsNonGroup)
					for i in pairs(recap_temp.PoMCasts) do
						if not recap_temp.InGroup[i] then
							-- only from non-group
							if randomCast <= recap_temp.PoMCasts[i] then
								-- this is the heuristic source to use
								newSourceNameGUID = i
								break
							else
								-- keep looking
								randomCast = randomCast - recap_temp.PoMCasts[i]
							end
						end
					end
				end
			end
			if newSourceNameGUID then
				-- found something
				-- this heuristic will somewhat favour the priest running Recap, since they are more likely to miss other priest casts due to range issues
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM heal for "..destNameGUID.." replaced by original source "..newSourceNameGUID.." using random heuristic")
				return newSourceNameGUID
			else
				-- we don't expect to see these messages often (cast could have happened out of range, so there might be no registered casts at all for this fight)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM heal for "..destNameGUID.." uncorrected: will be credited to "..destNameGUID)
				return nil
			end
		end
	end
	return nil
end

function Recap_LBHeal(timestamp, destNameGUID)
	-- Lifebloom heal going off

	local newSourceNameGUID

	newSourceNameGUID = Recap_FindLBSourceInRegistrationPriorToHeal(timestamp, destNameGUID)
	if newSourceNameGUID then
		-- source found in Registration table (newSourceNameGUID can equal sourceNameGUID if the druid's own LB is doing the healing on the druid)
		return newSourceNameGUID
	else
		-- LB not registered in the standard way, apply an heuristic assignment randomly in proportion to the casting of LB during the current fight (with separate counts for group and non-group)
		local i, randomCast
		newSourceNameGUID = nil -- to signal failure to find any casts
		if recap_temp.InGroup[destNameGUID] then
			if (recap_temp.LBTotalCastsGroup > 0) then
				-- check only group
				randomCast = math.random(recap_temp.LBTotalCastsGroup)
				for i in pairs(recap_temp.LBCasts) do
					if recap_temp.InGroup[i] then
						-- only from group
						if randomCast <= recap_temp.LBCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.LBCasts[i]
						end
					end
				end
			end
		else
			if (recap_temp.LBTotalCastsNonGroup > 0) then
				-- check only non-group
				randomCast = math.random(recap_temp.LBTotalCastsNonGroup)
				for i in pairs(recap_temp.LBCasts) do
					if not recap_temp.InGroup[i] then
						-- only from non-group
						if randomCast <= recap_temp.LBCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.LBCasts[i]
						end
					end
				end
			end
		end
		if newSourceNameGUID then
			-- found something
			-- this heuristic will somewhat favour the druid running Recap, since they are more likely to miss other druid casts due to range issues
			-- will happen less than for priest PoM due to being able to recreate credit from any preceding ticks that we see
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." LB heal for "..destNameGUID.." replaced by original source "..newSourceNameGUID.." using random heuristic")
			return newSourceNameGUID
		else
			-- we don't expect to see these messages often (cast could have happened out of range, so there might be no registered casts at all for this fight)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." LB heal for "..destNameGUID.." uncorrected: will be credited to "..destNameGUID)
			return nil
		end
	end
	return nil
end

function Recap_ESHeal(timestamp, destNameGUID)
	-- Earth Shield heal going off

	local newSourceNameGUID

	newSourceNameGUID = Recap_FindESSourceInRegistrationPriorToHeal(timestamp, destNameGUID)
	if newSourceNameGUID then
		-- source found in Registration table (newSourceNameGUID can equal sourceNameGUID if the shaman's own ES is doing the healing on the shaman)
		return newSourceNameGUID
	else
		-- ES not registered in the standard way, apply an heuristic assignment randomly in proportion to the casting of ES during the current fight (with separate counts for group and non-group)
		local i, randomCast
		newSourceNameGUID = nil -- to signal failure to find any casts
		if recap_temp.InGroup[destNameGUID] then
			if (recap_temp.ESTotalCastsGroup > 0) then
				-- check only group
				randomCast = math.random(recap_temp.ESTotalCastsGroup)
				for i in pairs(recap_temp.ESCasts) do
					if recap_temp.InGroup[i] then
						-- only from group
						if randomCast <= recap_temp.ESCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.ESCasts[i]
						end
					end
				end
			end
		else
			if (recap_temp.ESTotalCastsNonGroup > 0) then
				-- check only non-group
				randomCast = math.random(recap_temp.ESTotalCastsNonGroup)
				for i in pairs(recap_temp.ESCasts) do
					if not recap_temp.InGroup[i] then
						-- only from non-group
						if randomCast <= recap_temp.ESCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.ESCasts[i]
						end
					end
				end
			end
		end
		if newSourceNameGUID then
			-- found something
			-- this heuristic will somewhat favour the shaman running Recap, since they are more likely to miss other shaman casts due to range issues
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ES heal for "..destNameGUID.." replaced by original source "..newSourceNameGUID.." using random heuristic")
			return newSourceNameGUID
		else
			-- we don't expect to see these messages often (cast could have happened out of range, so there might be no registered casts at all for this fight)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ES heal for "..destNameGUID.." uncorrected: will be credited to "..destNameGUID)
			return nil
		end
	end
	return nil
end

function Recap_JoLHeal(timestamp, destNameGUID)
	-- Judgement of Light heal going off

	local newSourceNameGUID

	newSourceNameGUID = Recap_FindJoLSourceInRegistrationPriorToHeal(timestamp, destNameGUID)
	if newSourceNameGUID then
		-- source found in Registration table (newSourceNameGUID can equal sourceNameGUID if the paladin's own JoL is doing the healing on the paladin)
		return newSourceNameGUID
	else
		-- JoL not registered in the standard way, apply an heuristic assignment randomly in proportion to the casting of JoL during the current fight (with separate counts for group and non-group)
		local i, randomCast
		newSourceNameGUID = nil -- to signal failure to find any casts
		if recap_temp.InGroup[destNameGUID] then
			if (recap_temp.JoLTotalCastsGroup > 0) then
				-- check only group
				randomCast = math.random(recap_temp.JoLTotalCastsGroup)
				for i in pairs(recap_temp.JoLCasts) do
					if recap_temp.InGroup[i] then
						-- only from group
						if randomCast <= recap_temp.JoLCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.JoLCasts[i]
						end
					end
				end
			end
		else
			if (recap_temp.JoLTotalCastsNonGroup > 0) then
				-- check only non-group
				randomCast = math.random(recap_temp.JoLTotalCastsNonGroup)
				for i in pairs(recap_temp.JoLCasts) do
					if not recap_temp.InGroup[i] then
						-- only from non-group
						if randomCast <= recap_temp.JoLCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.JoLCasts[i]
						end
					end
				end
			end
		end
		if newSourceNameGUID then
			-- found something
			-- this heuristic will somewhat favour the shaman running Recap, since they are more likely to miss other shaman casts due to range issues
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." JoL heal for "..destNameGUID.." replaced by original source "..newSourceNameGUID.." using random heuristic")
			return newSourceNameGUID
		else
			-- we don't expect to see these messages often (cast could have happened out of range, so there might be no registered casts at all for this fight)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." JoL heal for "..destNameGUID.." uncorrected: will be credited to "..destNameGUID)
			return nil
		end
	end
	return nil
end

function Recap_ILotPHeal(timestamp, destNameGUID)
	-- Improved Leader of the Pack heal going off

	local newSourceNameGUID

	newSourceNameGUID = Recap_FindILotPSourceInRegistrationPriorToHeal(timestamp, destNameGUID)
	if newSourceNameGUID then
		-- source found in Registration table (newSourceNameGUID can equal sourceNameGUID if the druid's own ILotP is doing the healing on the druid)
		-- use this to create a cast belatedly for the druid for Improved Leader of the Pack
		if recap_temp.ILotPRegistrationFeral[newSourceNameGUID] and (recap_temp.ILotPRegistrationFeral[newSourceNameGUID].Improved == false) then
			-- registration for the feral form for the druid, but not yet confirmed as implying Improved Leader of the Pack
			recap_temp.ILotPRegistrationFeral[newSourceNameGUID].Improved = true
			-- count casts of ILotP (table was cleared during StartFight -- not ideal but close enough -- misses casts that preceded the start of the fight if any)
			if not recap_temp.ILotPCasts[newSourceNameGUID] then
				recap_temp.ILotPCasts[newSourceNameGUID] = 0
			end
			recap_temp.ILotPCasts[newSourceNameGUID] = recap_temp.ILotPCasts[newSourceNameGUID] + 1
			if recap_temp.InGroup[newSourceNameGUID] then
				recap_temp.ILotPTotalCastsGroup = recap_temp.ILotPTotalCastsGroup + 1
			else
				recap_temp.ILotPTotalCastsNonGroup = recap_temp.ILotPTotalCastsNonGroup + 1
			end
		end
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ILotP heal for "..destNameGUID.." replaced by original source "..newSourceNameGUID)
		return newSourceNameGUID
	else
		-- ILotP not registered in the standard way, apply an heuristic assignment randomly in proportion to the casting of ILotP during the current fight (with separate counts for group and non-group)
		local i, randomCast
		newSourceNameGUID = nil -- to signal failure to find any casts
		if recap_temp.InGroup[destNameGUID] then
			if (recap_temp.ILotPTotalCastsGroup > 0) then
				-- check only group
				randomCast = math.random(recap_temp.ILotPTotalCastsGroup)
				for i in pairs(recap_temp.ILotPCasts) do
					if recap_temp.InGroup[i] then
						-- only from group
						if randomCast <= recap_temp.ILotPCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.ILotPCasts[i]
						end
					end
				end
			end
		else
			if (recap_temp.ILotPTotalCastsNonGroup > 0) then
				-- check only non-group
				randomCast = math.random(recap_temp.ILotPTotalCastsNonGroup)
				for i in pairs(recap_temp.ILotPCasts) do
					if not recap_temp.InGroup[i] then
						-- only from non-group
						if randomCast <= recap_temp.ILotPCasts[i] then
							-- this is the heuristic source to use
							newSourceNameGUID = i
							break
						else
							-- keep looking
							randomCast = randomCast - recap_temp.ILotPCasts[i]
						end
					end
				end
			end
		end
		if newSourceNameGUID then
			-- found something
			-- this heuristic will somewhat favour the shaman running Recap, since they are more likely to miss other shaman casts due to range issues
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ILotP heal for "..destNameGUID.." replaced by original source "..newSourceNameGUID.." using random heuristic")
			return newSourceNameGUID
		else
			-- we don't expect to see these messages often (cast could have happened out of range, so there might be no registered casts at all for this fight)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ILotP heal for "..destNameGUID.." uncorrected: will be credited to "..destNameGUID)
			return nil
		end
	end
	return nil
end

function Recap_LBTick(timestamp, sourceNameGUID, destNameGUID)
	-- Lifebloom tick, so if necessary create a synthetic registration

	local regDest

	if not recap_temp.LBRegistration[destNameGUID] then
		recap_temp.LBRegistration[destNameGUID] = {}
	end
	regDest = recap_temp.LBRegistration[destNameGUID]
	if regDest[sourceNameGUID] then
		-- registered, check the timestamp
		if (timestamp - regDest[sourceNameGUID]) < 9 then
			-- assume that it is a current registration (the usual case for Lifebloom ticks), do nothing
		else
			-- assume that it was a stale registration, replace it with a synthetic registration
			regDest[sourceNameGUID] = timestamp
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." LB synthetic registration based on seeing a tick for "..destNameGUID.." on behalf of source "..sourceNameGUID)
			-- we don't add this synthetic registration to the counts, as it is already unusual
		end
	else
		-- not registered, create a synthetic registration
		regDest[sourceNameGUID] = timestamp
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." LB synthetic registration based on seeing a tick for "..destNameGUID.." on behalf of source "..sourceNameGUID)
		-- we don't add this synthetic registration to the counts, as it is already unusual
	end
end

function Recap_FindPoMSourceInRegistrationPriorToHeal(timestamp, destNameGUID)

	local regDest

	-- check whether this is a standard registered PoM
	if recap_temp.PoMRegistration[destNameGUID] and recap_temp.PoMRegistration[destNameGUID].OriginalCaster then
		-- has registration, check timestamp
		regDest = recap_temp.PoMRegistration[destNameGUID]
		if (timestamp - regDest.Timestamp) < 29.3 then
			-- current, replace the source with the original caster
			-- empirically, it looks as though some intervals over 29.1 and certainly those over 29.7, have expired -- there is a slight risk of deleting something that migrates with its last breath
			local newSourceNameGUID = regDest.OriginalCaster
if regDest.HasAura == true then
if regDest.Migrated == true then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." heal for "..destNameGUID.." (has aura) replaced by original source "..newSourceNameGUID.." using charge "..regDest.Charges.." (old registration deleted)")
else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." heal for "..destNameGUID.." (has aura) replaced by original source "..newSourceNameGUID.." using charge "..regDest.Charges)
end
else
if regDest.Migrated == true then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." heal for "..destNameGUID.." (no aura) replaced by original source "..newSourceNameGUID.." using charge "..regDest.Charges.." (old registration deleted)")
else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." heal for "..destNameGUID.." (no aura) replaced by original source "..newSourceNameGUID.." using charge "..regDest.Charges)
end
end
			regDest.Healed = true
			if (regDest.Migrated == true) then
				-- already copied to migration, we can safely delete this registration
				--   (the code supports migration before healing which is the usual case, and healing before migration (see below))
				recap_temp.PoMRegistration[destNameGUID] = nil
			else
				-- we have not already migrated this registration, so migrate it now (migration synthesized immediately following heal)
				Recap_CreatePoMMigrationEntryFromRegistration(timestamp, destNameGUID)
				-- at this point the registration will have been removed
				-- if the combatant has an aura (whether we noticed it or not) then there may be a subsequent loss-of-aura event which we will ignore because there will be no registration
				--   (we could get bad sequencing if the combatant gets a new registration prior to our seeing the loss-of-aura event)
			end
			-- assert: one way or another the registration is now gone
			return newSourceNameGUID
		else
			-- expired
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." registration for "..destNameGUID.." deleted (expired after "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." seconds)")
			recap_temp.PoMRegistration[destNameGUID] = nil
			return nil
		end
	else
		-- no registration
		return nil
	end
end

function Recap_FindLBSourceInRegistrationPriorToHeal(timestamp, destNameGUID)

	-- check whether this is a standard registered LB
	if recap_temp.LBRegistration[destNameGUID] then
		-- we might have registrations
		-- scan the registrations discarding stale ones, and remember the youngest of those remaining
		local i, regDest
		local somethingChanged = true
		local safetyCount = 0
		local youngestTimestamp = nil
		regDest = recap_temp.LBRegistration[destNameGUID]
		while somethingChanged do
			somethingChanged = false
			safetyCount = safetyCount + 1
			for i in pairs(regDest) do
				if (timestamp - regDest[i]) > 10 then
					-- expired for sure
					regDest[i] = nil
					somethingChanged = true
					break
				end
				-- no anomalies, remember the youngest Timestamp encountered
				if youngestTimestamp == nil then
					youngestTimestamp = regDest[i]
				else
					if regDest[i] > youngestTimestamp then
						youngestTimestamp = regDest[i]
					end
				end
			end
			if safetyCount >= 20 then
				-- hmm, 20 loops, time to bail (paranoia check)
				break
			end
		end

		-- LBRegistration is now clean, scan it again and find the youngest Timestamp and use that one (not perfect, there could be mis-attribution, tough luck bunkie)
		for i in pairs(regDest) do
			if regDest[i] == youngestTimestamp then
				-- got a source, remove the entry, return the source
				regDest[i] = nil
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." LB final bloom for "..destNameGUID.." found original source "..i)
				return i
			end
		end

		-- we could get here without having found anything in LBRegistration
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." LB final bloom failed to find source in LBRegistration for "..destNameGUID)
		return nil
	else
		-- no registrations at all
		return nil
	end
end

function Recap_FindESSourceInRegistrationPriorToHeal(timestamp, destNameGUID)

	local regDest

	-- check whether this is a standard registered ES
	if recap_temp.ESRegistration[destNameGUID] and recap_temp.ESRegistration[destNameGUID].OriginalCaster then
		-- has registration, check count
		regDest = recap_temp.ESRegistration[destNameGUID]
		if regDest.Charges > 0 then
			-- count okay, check timestamp
			if (timestamp - regDest.Timestamp) < 602 then
				-- still live, replace the source with the original caster
				local newSourceNameGUID = regDest.OriginalCaster
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ES heal for "..destNameGUID.." replaced by original source "..newSourceNameGUID.." using charge "..regDest.Charges)
				regDest.Charges = regDest.Charges - 1
				return newSourceNameGUID
			else
				-- expired
				recap_temp.ESRegistration[destNameGUID] = nil
				return nil
			end
		else
			-- charges used up
			recap_temp.ESRegistration[destNameGUID] = nil
			return nil
		end
	else
		-- no registration
		return nil
	end
end

function Recap_FindJoLSourceInRegistrationPriorToHeal(timestamp, destNameGUID)

	-- check whether this is a standard registered JoL
	if recap_temp.JoLRegistrationHitter[destNameGUID] then
		-- registered, this combatant has recently hit a mob that has a Judgement of Light on it
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." JoL heal for "..destNameGUID.." replaced by original source "..recap_temp.JoLRegistrationHitter[destNameGUID].OriginalCaster)
		return recap_temp.JoLRegistrationHitter[destNameGUID].OriginalCaster
	else
		return nil
	end
end

function Recap_FindILotPSourceInRegistrationPriorToHeal(timestamp, destNameGUID)

	-- check whether this is a standard registered ILotP
	if recap_temp.ILotPRegistration[destNameGUID] then
		-- registered, this combatant has acquired the Leader of the Pack aura
		-- note that the registration is removed only on death; or when the aura fades (or is dispelled or stolen -- if either of those is possible)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ILotP heal for "..destNameGUID.." replaced by original source "..recap_temp.ILotPRegistration[destNameGUID])
		return recap_temp.ILotPRegistration[destNameGUID]
	else
		return nil
	end
end

function Recap_DeathEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags)

	local j, spot, feignDeath, groupPetDeath

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- we often get two death events for a single death, one event with source and one without
	-- Some testing indicated that the event without the source was more reliable, so we're using that one, but we will miss some deaths
	if sourceNameGUID == nil then

		spot = nil
		feignDeath = false
		groupPetDeath = false

		-- check for FD
		if recap.Combatant[destNameGUID] and recap.Combatant[destNameGUID].Class and recap.Combatant[destNameGUID].Class=="Hunter" then
			if recap.Opt.IgnoreGUIDs.value then
				-- ignoring GUIDs
				if recap_temp.Player==destNameGUID then
					spot = "player"
				end
				if not spot then
					for j=1,4 do
						if (UnitName("party"..j))==destNameGUID then
							spot = "party"..j
							break
						end
					end
				end
				if not spot then
					for j=1,40 do
						if (UnitName("raid"..j))==destNameGUID then
							spot = "raid"..j
							break
						end
					end
				end
			else
				-- using GUIDs (the normal case)
				if recap_temp.Player.."_"..Recap_TrimGUID((UnitGUID("player")))==destNameGUID then
					spot = "player"
				end
				if not spot then
					for j=1,4 do
						if (UnitName("party"..j)).."_"..Recap_TrimGUID((UnitGUID("party"..j)))==destNameGUID then
							spot = "party"..j
							break
						end
					end
				end
				if not spot then
					for j=1,40 do
						if (UnitName("raid"..j)).."_"..Recap_TrimGUID((UnitGUID("raid"..j)))==destNameGUID then
							spot = "raid"..j
							break
						end
					end
				end
			end
			if spot then
				if UnitIsFeignDeath(spot) then
					-- ignore feign deaths
					feignDeath = true
				end
			end
		end

		-- is this a group pet death ?
		if (not spot) and recap_temp.InGroup[destNameGUID] and Recap_ExtractOwner(destNameGUID) then
			groupPetDeath = true

			-- code for various special case healing spells (death removes registration, might not always be correct if for example an aura loss event follows death)
			recap_temp.PoMRegistration[destNameGUID] = nil
			recap_temp.PoMMigration[destNameGUID] = nil
			recap_temp.LBRegistration[destNameGUID] = nil
			recap_temp.ESRegistration[destNameGUID] = nil
			recap_temp.JoLRegistration[destNameGUID] = nil
			recap_temp.JoLRegistrationHitter[destNameGUID] = nil
			recap_temp.ILotPRegistrationFeral[destNameGUID] = nil
			recap_temp.ILotPRegistration[destNameGUID] = nil
		end

		-- count real deaths for combatants other than group pets (non-group pets are still counted against their owner's death count)
		if (not feignDeath) and (not groupPetDeath) and recap_temp.Last[destNameGUID] then
			recap_temp.Last[destNameGUID].Kills = recap_temp.Last[destNameGUID].Kills + 1

			-- code for various special case healing spells (death removes registration, might not always be correct if for example an aura loss event follows death)
			recap_temp.PoMRegistration[destNameGUID] = nil
			recap_temp.PoMMigration[destNameGUID] = nil
			recap_temp.LBRegistration[destNameGUID] = nil
			recap_temp.ESRegistration[destNameGUID] = nil
			recap_temp.JoLRegistration[destNameGUID] = nil
			recap_temp.JoLRegistrationHitter[destNameGUID] = nil
			recap_temp.ILotPRegistrationFeral[destNameGUID] = nil
			recap_temp.ILotPRegistration[destNameGUID] = nil
		end

		-- we still include group pet deaths in recent events
		if (not feignDeath) and recap.Opt.RecentData.value then
			-- effectType 5 is death
			Recap_AddRecentEvent(timestamp, false, destNameGUID, 5, recap_temp.Localize.Died, false, false, false, false)
		end
	end
end

function Recap_KillEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags)

	local j, spot

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- because the underlying event partially duplicates the DeathEvent, we use this one only to credit kills in recent events
	-- there is an anomaly, in that this kill event may arrive before the damage event that accomplished it (although time stamps may be identical) (shrug)
	if sourceNameGUID and recap_temp.Last[sourceNameGUID] and destNameGUID and recap_temp.Last[destNameGUID] then
		if recap.Opt.RecentData.value then
			-- effectType 5 is death
			Recap_AddRecentEvent(timestamp, sourceNameGUID, destNameGUID, 5, recap_temp.Localize.Killed, false, false, false, false)
		end
	end
end

function Recap_CastEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName)

	local damageEffect, healEffect, didPoM, didLB, didES, didILotP

	-- Note that a Cast event does not trigger timers per se
	-- SPELL_CAST_SUCCESS may have a lot of extra noise, such as dismounting, equipping an Argent Dawn commission, etc.

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- However, if this is the Cast of a known DoT or HoT then it may trigger timers

	if sourceNameGUID and destNameGUID then -- guard
		-- special case code for priest Prayer of Mending -- the first such special case code in Recap
		didPoM = false
		if (spellID == recap_temp.PoMCastID) then
			-- someone casting Prayer of Mending
			Recap_RegisterPoMCast(timestamp, sourceNameGUID, destNameGUID)
			didPoM = true
		end

		-- special case code for druid Lifebloom
		didLB = false
		if (spellID == recap_temp.LBCastID) then
			-- someone casting Lifebloom
			Recap_RegisterLBCast(timestamp, sourceNameGUID, destNameGUID)
			didLB = true
		end

		-- special case code for shaman Earth Shield
		didES = false
		if recap_temp.ESCastID[spellID] then
			-- someone casting one of the ranks of Earth Shield
			Recap_RegisterESCast(timestamp, sourceNameGUID, destNameGUID)
			didES = true
		end

		-- special case code for paladin Judgement
		didJoL = false
		if (spellID == recap_temp.JoLCastID) then
			-- someone casting Judgement
			Recap_RegisterJoLCast(timestamp, sourceNameGUID, destNameGUID)
			didJoL = true
		end
	end
	if sourceNameGUID then -- guard
		-- special case code for druid Cat Form or Dire Bear Form
		didILotP = false
		if recap_temp.ILotPCastID[spellID] then
			-- someone casting a feral form that might later be associated with Improved Leader of the Pack
			Recap_RegisterILotPCast(timestamp, sourceNameGUID)
			didILotP = true
		end
	end

	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		-- we have data for this combatant
		if (didPoM == false) and (didLB == false) and (didES == false) and (didJoL == false) and (didILotP == false) then
			-- not the initial cast of any of the special cases
			damageEffect = "1"..spellName
			healEffect = "3"..spellName
			if recap.Combatant[sourceNameGUID].OutgoingDetail and (recap.Combatant[sourceNameGUID].OutgoingDetail[damageEffect] or recap.Combatant[sourceNameGUID].OutgoingDetail[healEffect]) then -- no need to check LastOutgoingDetail

				-- this is a cast with which this combatant has done damage or healing before

				-- we used to call Recap_StartFight here, but I now think that's wrong
				-- that could have started a fight with the initial cast of a Prayer of Mending, which is not what we want
				-- this means that fights that start with e.g. a fireball will now be slightly shorter (by the flight time of the fireball)

				-- if the fight is ongoing, start or restart the idle timers
				if recap.Opt.State.value=="Active" and recap.Opt.IdleFight.value then
					recap_temp.IdleTimer = 0
				end
				if recap.Opt.State.value=="Active" then
					recap_temp.HiddenIdleTimer = 0
				end

				-- if the fight is ongoing and if the end of fight delay timer is running, restart the end of fight delay timer
				--   (the player left combat, perhaps temporarily, but the fight continues)
				if recap.Opt.State.value=="Active" and recap_temp.EndFightDelayTimer~=-1 then
					recap_temp.EndFightDelayTimer = 0
				end

				-- perhaps add the combatant into the fight, and update timers
				-- this takes the time from the moment of successful cast, rather than from when it strikes the target
				if recap.Combatant[sourceNameGUID].OutgoingDetail[damageEffect] then		-- no need to check LastOutgoingDetail
					Recap_CreateCastForTicks(sourceNameGUID, timestamp, sourceFlags, damageEffect, "Out", true, "OutgoingDetail")
					Recap_CreateCastForTicks(sourceNameGUID, timestamp, sourceFlags, damageEffect, "Out", false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight)
				end

				-- for heals, similarly
				if recap.Combatant[sourceNameGUID].OutgoingDetail[healEffect] then		-- no need to check LastOutgoingDetail
					Recap_CreateCastForTicks(sourceNameGUID, timestamp, sourceFlags, healEffect, "Heal", true, "OutgoingDetail")
					Recap_CreateCastForTicks(sourceNameGUID, timestamp, sourceFlags, healEffect, "Heal", false, "LastOutgoingDetail_"..recap_temp.ActiveLastFight)
				end
			end
		end

		-- at risk of adding a tonne of duplicate events, and a tonne of non-combat events, record this spell as an Other event
		-- this is needed to (approximately) count the initial casts of Prayer of Mending, which would otherwise often be ignored since they do healing later
		if not recap.Opt.LightData.value and recap.Opt.OtherData.value then
			Recap_CreateBasicOtherDetail(sourceNameGUID, timestamp, "1", spellName, "OtherDetail")
			Recap_CreateBasicOtherDetail(sourceNameGUID, timestamp, "1", spellName, "LastOtherDetail_"..recap_temp.ActiveLastFight)
		end
	end

	if destNameGUID and recap.Combatant[destNameGUID] then
		if not recap.Opt.LightData.value and recap.Opt.OtherData.value then
			-- we don't know whether a 'cast' event with a destNameGUID is a buff or a debuff, so insert it as an unknown (will be coloured white)
			-- 'cast' events sometimes overlap with buffs and debuffs, so there will be some double counting
			Recap_CreateFullOtherDetail(destNameGUID, timestamp, "7", spellName, 1, nil, "OtherDetail")
			Recap_CreateFullOtherDetail(destNameGUID, timestamp, "7", spellName, 1, nil, "LastOtherDetail_"..recap_temp.ActiveLastFight)
		end
	end
end

function Recap_CreateCastForTicks(thisCombatant, timestamp, flags, effect, direction, doSelf, AllLastOutgoingDetail)
	if recap.Combatant[thisCombatant][AllLastOutgoingDetail] and recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect] then
		if (recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].GlancesDmg) or (recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].HitsDmg) or
		   (recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].CritsDmg) or (recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].CrushDmg) then
			-- this has a direct component, so it is already being counted; do nothing
		else
			if recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].TicksDmg then
				-- this is the casting of a DoT or Hot (with no direct component), so count it as a hit
				Recap_CreateCombatantAndUpdateTimers(thisCombatant, timestamp, flags, direction)
				recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].Hits = (recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].Hits or 0) + 1
				recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].CritsEvents = (recap.Combatant[thisCombatant][AllLastOutgoingDetail][effect].CritsEvents or 0) + 1
				-- is the thisCombatant the player or one of the player's pets?
				local selfIndex = nil -- keep as nil
				if thisCombatant then
					if thisCombatant == recap_temp.PlayerGUID then
						selfIndex = recap_temp.s
					elseif Recap_IsOwnedBy(thisCombatant, recap_temp.PlayerGUID) then
						selfIndex = recap_temp.s..":"..Recap_NameOnlyFromCombatant(thisCombatant)
					end
				end

				-- only add to Self when adding to AllFights (otherwise counts twice)
				if selfIndex and doSelf then
					Recap_MakeSelfDetail(selfIndex, effect)
					recap.Self[selfIndex][effect].Hits = (recap.Self[selfIndex][effect].Hits or 0) + 1
					recap.Self[selfIndex][effect].CritsEvents = (recap.Self[selfIndex][effect].CritsEvents or 0) + 1
				end
			end
		end
	end
end

function Recap_RegisterPoMCast(timestamp, sourceNameGUID, destNameGUID)

	local regDest

	if not recap_temp.PoMRegistration[destNameGUID] then
		recap_temp.PoMRegistration[destNameGUID] = {}
	end
	regDest = recap_temp.PoMRegistration[destNameGUID]
	-- fresh registration, overwrites whatever was there without checking (we assume Blizzard enforces the rules)
	regDest.OriginalCaster = sourceNameGUID
	regDest.OriginalTimestamp = timestamp -- both a unique ID for this PoM, and allows us to track total duration if we want to
	regDest.Timestamp = timestamp
	regDest.Charges = 5
	regDest.Healed = false
	regDest.Migrated = false
	regDest.HasAura = false
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." new registration for "..destNameGUID.." on behalf of source "..sourceNameGUID)
	-- count casts of PoM (table was cleared during StartFight -- not ideal but close enough -- misses casts that preceded the start of the fight)
	if not recap_temp.PoMCasts[sourceNameGUID] then
		recap_temp.PoMCasts[sourceNameGUID] = 0
	end
	recap_temp.PoMCasts[sourceNameGUID] = recap_temp.PoMCasts[sourceNameGUID] + 1
	if recap_temp.InGroup[sourceNameGUID] then
		recap_temp.PoMTotalCastsGroup = recap_temp.PoMTotalCastsGroup + 1
	else
		recap_temp.PoMTotalCastsNonGroup = recap_temp.PoMTotalCastsNonGroup + 1
	end
end

function Recap_RegisterLBCast(timestamp, sourceNameGUID, destNameGUID)

	local regDest

	if not recap_temp.LBRegistration[destNameGUID] then
		recap_temp.LBRegistration[destNameGUID] = {}
	end
	regDest = recap_temp.LBRegistration[destNameGUID]
	-- fresh registration, overwrites whatever was there without checking (this has the effect of refreshing any existing Lifebloom stack registration for this source)
	regDest[sourceNameGUID] = timestamp
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." LB fresh registration for "..destNameGUID.." on behalf of source "..sourceNameGUID)
	-- count casts of LB (table was cleared during StartFight -- not ideal but close enough -- misses casts that preceded the start of the fight)
	-- note that a druid who takes care to keep Lifebloom refreshed rather than allowing it to bloom may be credited with more than their share of otherwise unattributed blooms -- tough, inaccurate, but perhaps they deserve the credit for taking the care
	if not recap_temp.LBCasts[sourceNameGUID] then
		recap_temp.LBCasts[sourceNameGUID] = 0
	end
	recap_temp.LBCasts[sourceNameGUID] = recap_temp.LBCasts[sourceNameGUID] + 1
	if recap_temp.InGroup[sourceNameGUID] then
		recap_temp.LBTotalCastsGroup = recap_temp.LBTotalCastsGroup + 1
	else
		recap_temp.LBTotalCastsNonGroup = recap_temp.LBTotalCastsNonGroup + 1
	end
end

function Recap_RegisterESCast(timestamp, sourceNameGUID, destNameGUID)

	local regDest

	if not recap_temp.ESRegistration[destNameGUID] then
		recap_temp.ESRegistration[destNameGUID] = {}
	end
	regDest = recap_temp.ESRegistration[destNameGUID]
	-- fresh registration, overwrites whatever was there without checking (we assume Blizzard enforces the rules)
	regDest.OriginalCaster = sourceNameGUID
	regDest.Timestamp = timestamp
	regDest.Charges = 6
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ES new registration for "..destNameGUID.." on behalf of source "..sourceNameGUID)
	-- count casts of ES (table was cleared during StartFight -- not ideal but close enough -- misses casts that preceded the start of the fight)
	if not recap_temp.ESCasts[sourceNameGUID] then
		recap_temp.ESCasts[sourceNameGUID] = 0
	end
	recap_temp.ESCasts[sourceNameGUID] = recap_temp.ESCasts[sourceNameGUID] + 1
	if recap_temp.InGroup[sourceNameGUID] then
		recap_temp.ESTotalCastsGroup = recap_temp.ESTotalCastsGroup + 1
	else
		recap_temp.ESTotalCastsNonGroup = recap_temp.ESTotalCastsNonGroup + 1
	end
end

function Recap_RegisterJoLCast(timestamp, sourceNameGUID, destNameGUID)

	-- this registers the Judgement on the target
	-- we will need the target to acquire the Judgement of Light aura to complete the registration

	local regDest

	if not destNameGUID then
		-- hmmm, seems that there can be a Judgement without a target
		return
	end

	if not recap_temp.JoLRegistration[destNameGUID] then
		recap_temp.JoLRegistration[destNameGUID] = {}
	end
	regDest = recap_temp.JoLRegistration[destNameGUID]
	-- fresh registration, overwrites whatever was there without checking (we assume Blizzard enforces the rules)
	regDest.OriginalCaster = sourceNameGUID
	regDest.Timestamp = timestamp
	regDest.Light = false
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." JoL new registration (Judgement only) for "..destNameGUID.." on behalf of source "..sourceNameGUID)
	-- we don't count this partial registration as a cast -- that will only happen when we see the Judgement of Light aura on the target
end

function Recap_RegisterILotPCast(timestamp, sourceNameGUID)

	-- this registers the druid form change to Cat or Dire Bear
	-- we will need combatants to acquire the Leader of the Pack aura to get their registrations

	local regDest

	if not recap_temp.ILotPRegistrationFeral[sourceNameGUID] then
		recap_temp.ILotPRegistrationFeral[sourceNameGUID] = {}
	end
	regDest = recap_temp.ILotPRegistrationFeral[sourceNameGUID]
	-- fresh registration, overwrites whatever was there without checking (we assume Blizzard enforces the rules)
	regDest.Timestamp = timestamp
	regDest.Improved = false
	-- register an aura on the caster
	recap_temp.ILotPRegistration[sourceNameGUID] = sourceNameGUID
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ILotP new registration (Cat or Dire Bear Form only) for druid "..sourceNameGUID)
	-- we don't count this as a cast -- that will only happen if we see an Improved Leader of the Pack heal attributed to this druid
end

function Recap_SummonOrCreateEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName)

	-- Note that a Summon or Create event does not trigger timers, but could create new combatants

	-- since the destNameGUID does not yet exist, we don't filter on it, only on the combatant summoning or creating it
	sourceNameGUID = Recap_FilterDoNotStore(sourceNameGUID, nil)
	if (sourceNameGUID == nil) then
		-- ignore entirely
		return
	end

	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		if not recap.Opt.LightData.value and recap.Opt.OtherData.value then
			-- the summon or create is an Other event similar to a Cast, so track only if Other Data mode is enabled
			Recap_CreateBasicOtherDetail(sourceNameGUID, timestamp, "1", spellName, "OtherDetail")
			Recap_CreateBasicOtherDetail(sourceNameGUID, timestamp, "1", spellName, "LastOtherDetail_"..recap_temp.ActiveLastFight)
		end
	end

	if destNameGUID then
		-- this could be (for example) summoning a Water Elemental or creating a Magma Totem
		-- treat as the creation of a new pet (with same 'Friend' status as its owner)
		-- note that destFlags does not yet include the controlled pet flags, so we insert the controlled object flag
		-- the flags may later be overwritten with a set of flags indicating either controlled or uncontrolled (go figure) (this could produce incorrect death counts if they die while marked as uncontrolled)
		local petDestFlags = bit_bor(destFlags, COMBATLOG_OBJECT_TYPE_OBJECT)
		Recap_CreatePet(destNameGUID, petDestFlags, sourceNameGUID, sourceFlags, ((sourceNameGUID and recap.Combatant[sourceNameGUID] and recap.Combatant[sourceNameGUID].Friend) or false), math_floor(1000*timestamp))
		-- we ignore the fully qualified pet name that is returned by Recap_CreatePet
	end
end

function Recap_GainEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, powerType, powerName)

	-- Note that a Gain event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- it looks as though we can ignore schoolName (the spell might be a Nature spell but the gain is Mana)
	-- we are ignoring any distinction between PERIODIC and non-PERIODIC
	if destNameGUID and recap.Combatant[destNameGUID] then
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "5", spellName, amount, powerName, "OtherDetail")
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "5", spellName, amount, powerName, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
end

function Recap_DrainEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, powerType, powerName, extraAmount)

	-- Note that a Drain event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if destNameGUID and recap.Combatant[destNameGUID] then
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "3", spellName, amount, powerName, "OtherDetail")
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "3", spellName, amount, powerName, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
end

function Recap_LeechEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount, powerType, powerName, extraAmount)

	-- Note that a Leech event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		-- this is the implicit 'cast' from the sourceNameGUID
		Recap_CreateBasicOtherDetail(sourceNameGUID, timestamp, "1", spellName, "OtherDetail")
		Recap_CreateBasicOtherDetail(sourceNameGUID, timestamp, "1", spellName, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		-- this is the beneficiary of the leech
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "5", spellName, amount, powerName, "OtherDetail")
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "5", spellName, amount, powerName, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		-- this is the loser from the leech
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "3", spellName, extraAmount, powerName, "OtherDetail")
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "3", spellName, extraAmount, powerName, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
end

function Recap_ExtraAttackEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, amount)

	-- Note that an Extra Attack event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if destNameGUID and recap.Combatant[destNameGUID] then
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "5", spellName, amount, recap_temp.Localize.Attacks, "OtherDetail")
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "5", spellName, amount, recap_temp.Localize.Attacks, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
end

function Recap_BuffOrDebuffEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, auraType, amount)

	local wasControlled, isControlled

	-- Note that a Buff or Debuff event does not trigger timers
	-- this is where mind-control spells come (but unfortunately they don't include both source and dest)

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if destNameGUID then -- guard
		-- special case code for priest Prayer of Mending
		if (spellID == recap_temp.PoMAuraID) then
			-- allow 2 seconds for normal migration of aura -- the vast majority are 0.5 seconds or less
			-- at worst this toon will get a PoM heal with no registration and will have to check again in Migration table
			Recap_MovePoMFromMigrationToRegistration(timestamp, destNameGUID, true, 2)
		end

		-- special case code for paladin Judgement of Light
		if (spellID == recap_temp.JoLAuraID) then
			Recap_RegisterJoLAura(timestamp, destNameGUID)
		end

		-- special case code for druid Leader of the Pack
		if (spellID == recap_temp.ILotPAuraID) then
			Recap_RegisterILotPAura(timestamp, destNameGUID)
		end
	end

	if recap.Opt.LightData.value or not recap.Opt.OtherData.value then
		-- only track as Other events if not in Light Data Mode, and with the Other Data Mode enabled
		return
	end

	if destNameGUID and recap.Combatant[destNameGUID] then
		local myType
		if auraType == "BUFF" then
			myType = "5"
		else
			myType = "3"
		end
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, myType, spellName, amount, nil, "OtherDetail")
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, myType, spellName, amount, nil, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	else
		-- buffs and debuffs are not recorded before destNameGUID has been registered as a combatant
	end
end

function Recap_MovePoMFromMigrationToRegistration(timestamp, destNameGUID, gainsAura, migrationInterval)
	-- PoM aura applied (or should have been)

	-- case one, we have a freshly cast PoM already registered to us, do nothing, the aura is duplicated information
	if recap_temp.PoMRegistration[destNameGUID] and recap_temp.PoMRegistration[destNameGUID].OriginalCaster then
		-- has registration, check timestamp
		if (timestamp - recap_temp.PoMRegistration[destNameGUID].Timestamp) < 29.3 then
			-- current registration
			if gainsAura and (recap_temp.PoMRegistration[destNameGUID].Charges ~= 5) then
				recap_temp.PoMRegistration[destNameGUID].HasAura = true
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(recap_temp.PoMRegistration[destNameGUID].OriginalTimestamp).." registration for "..destNameGUID.." gains aura at charge "..recap_temp.PoMRegistration[destNameGUID].Charges)
			end
			return
		else
			-- expired, delete the registration and drop through to treat as case two
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(recap_temp.PoMRegistration[destNameGUID].OriginalTimestamp).." registration for "..destNameGUID.." cleared so we can check in migration (expired "..Recap_FormatTimePoM(timestamp - recap_temp.PoMRegistration[destNameGUID].Timestamp).." seconds)")
			recap_temp.PoMRegistration[destNameGUID] = nil
		end
	end

	-- case two, we are catching a migrated PoM, and we need to find out where it came from and copy that information
	-- we have nothing registered to us, so look in PoMMigration for something to migrate from
	-- first scan PoMMigration repeatedly until we find no more errors or stale records
	-- PoMMigration includes both group and non-group, so only use records that match (group to group, or non-group to non-group)
	local i, mig
	local somethingChanged = true
	local safetyCount = 0
	local youngestTimestamp = nil
	while somethingChanged do
		somethingChanged = false
		safetyCount = safetyCount + 1
		for i in pairs(recap_temp.PoMMigration) do
			mig = recap_temp.PoMMigration[i]
			if not mig.OriginalCaster then
				-- no registration (anomalous)
				recap_temp.PoMMigration[i] = nil
				somethingChanged = true
				break
			end
			if mig.Charges == 0 then
				-- no charges (possibly anomalous)
				recap_temp.PoMMigration[i] = nil
				somethingChanged = true
				break
			end
			if (timestamp - mig.Timestamp) > 29.3 then
				-- expired for sure
				recap_temp.PoMMigration[i] = nil
				somethingChanged = true
				break
			end
			-- no anomalies, remember the youngest Timestamp encountered
			if i == destNameGUID then
				-- skip if the migration would be to the same combatant
			else
				if recap_temp.InGroup[i] ~= recap_temp.InGroup[destNameGUID] then
					-- skip if not both group, or not both non-group
					-- TODO: this is not completely right, it fails in a battleground with allies who are not in your group
				else
					if (timestamp - mig.Timestamp) > migrationInterval then
						-- skip if the allowed migration interval would be too long
						-- we keep the migration record around for up to 29.3 seconds in case we need it for an otherwise unattributed heal (see Recap_HealEvent)
						-- for an ordinary immediate migration the interval is 2 seconds, and for a "see if anything might fit" unattributed heal it is the full 29.3 seconds
					else
						if youngestTimestamp == nil then
							youngestTimestamp = mig.Timestamp
						else
							if mig.Timestamp > youngestTimestamp then
								youngestTimestamp = mig.Timestamp
							end
						end
					end
				end
			end
		end
		if safetyCount >= 20 then
			-- hmm, 20 loops, time to bail (paranoia check)
			break
		end
	end

	-- TODO: loop used for diagnostic printout only
	for i in pairs(recap_temp.PoMMigration) do
		mig = recap_temp.PoMMigration[i]
		if i == destNameGUID then
			-- skip if the migration would be to the same combatant
		else
			if recap_temp.InGroup[i] ~= recap_temp.InGroup[destNameGUID] then
				-- skip if not both group, or not both non-group
			else
				if (timestamp - mig.Timestamp) > migrationInterval then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM candidate for migration from "..i.." to "..destNameGUID..": PoM "..Recap_FormatTimeRecent(mig.OriginalTimestamp).." for original "..mig.OriginalCaster.." charges "..mig.Charges.." after "..Recap_FormatTimePoM(timestamp - mig.Timestamp).." secs ('stale')")
				else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM candidate for migration from "..i.." to "..destNameGUID..": PoM "..Recap_FormatTimeRecent(mig.OriginalTimestamp).." for original "..mig.OriginalCaster.." charges "..mig.Charges.." after "..Recap_FormatTimePoM(timestamp - mig.Timestamp).." secs")
				end
			end
		end
	end

	-- PoMMigration is now clean, scan it again and find the youngest Timestamp and use that one (not perfect, there could be mis-attribution, tough luck bunkie)
	for i in pairs(recap_temp.PoMMigration) do
		mig = recap_temp.PoMMigration[i]
		if i == destNameGUID then
			-- ignore if the migration would be to the same combatant
		else
			if recap_temp.InGroup[i] ~= recap_temp.InGroup[destNameGUID] then
				-- ignore if not both group, or not both non-group
				-- TODO: this is not completely right, it fails in a battleground with allies who are not in your group
			else
				if (timestamp - mig.Timestamp) > migrationInterval then
					-- ignore if the allowed migration interval would be too long
					--   (if a registration and a loss-of-aura event get sent to us in the wrong order then we will have migrated the registration prematurely, and could miss an attribution using the short interval)
				else
					if mig.Timestamp == youngestTimestamp then
						-- copy from Migration to Registration
						recap_temp.PoMRegistration[destNameGUID] = {}
						local regDest = recap_temp.PoMRegistration[destNameGUID]
						regDest.OriginalCaster = mig.OriginalCaster
						regDest.OriginalTimestamp = mig.OriginalTimestamp
						regDest.Timestamp = timestamp -- fresh timestamp
						regDest.Charges = mig.Charges -- already reduced by one when this was copied into Migration
						regDest.Healed = false
						regDest.Migrated = false
						regDest.HasAura = gainsAura
						-- remove the Migration entry
						recap_temp.PoMMigration[i] = nil
if regDest.HasAura == true then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." migrated from "..i.." to "..destNameGUID.." (has aura) for original "..regDest.OriginalCaster.." and charges are now "..regDest.Charges)
else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." migrated from "..i.." to "..destNameGUID.." (no aura) for original "..regDest.OriginalCaster.." and charges are now "..regDest.Charges)
end
						return -- rather than break, so that we can detect during testing any failure to find a match
					end
				end
			end
		end
	end
	-- we could get here without having found anything in PoMMigration (tough, we have the aura, or should have the aura, but have nobody to credit it to)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM failed to find source in PoMMigration for aura for "..destNameGUID.." (using window of "..migrationInterval.." secs)")
end

function Recap_RegisterJoLAura(timestamp, destNameGUID)

	-- this registers the Judgement of Light aura on the target, which needs to have already had a Judgement cast on it

	local regDest

	if recap_temp.JoLRegistration[destNameGUID] and recap_temp.JoLRegistration[destNameGUID].OriginalCaster then
		-- registration could be stale, or could be from another paladin, but we will arbitrarily assume that everything is okay
		-- after all, the target is still alive so our previous information can't be entirely outdated
		-- because the Judgement of Light aura can be refreshed, and we're not currently tracking any such refreshment, we don't actually use the timestamp
		regDest = recap_temp.JoLRegistration[destNameGUID]
		regDest.Timestamp = timestamp -- refresh the timestamp
		regDest.Light = true -- indicate that we have seen the Judgement of Light aura
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." JoL new registration (Judgement of Light aura) for "..destNameGUID.." on behalf of source "..regDest.OriginalCaster)
		-- count casts of JoL (table was cleared during StartFight -- not ideal but close enough -- misses casts that preceded the start of the fight if any)
		if not recap_temp.JoLCasts[regDest.OriginalCaster] then
			recap_temp.JoLCasts[regDest.OriginalCaster] = 0
		end
		recap_temp.JoLCasts[regDest.OriginalCaster] = recap_temp.JoLCasts[regDest.OriginalCaster] + 1
		if recap_temp.InGroup[regDest.OriginalCaster] then
			recap_temp.JoLTotalCastsGroup = recap_temp.JoLTotalCastsGroup + 1
		else
			recap_temp.JoLTotalCastsNonGroup = recap_temp.JoLTotalCastsNonGroup + 1
		end
	else
		-- no registration, ignore the aura since we can't attribute it to an original caster
	end
end

function Recap_RegisterILotPAura(timestamp, destNameGUID)

	-- this registers the Leader of the Pack aura on a combatant, if it comes shortly after a druid has assumed a Cat or Dire Bear form
	-- there could be misassignment if two druids go into feral form at about the same time

	local i

	-- scan feral form registrations
	for i in pairs(recap_temp.ILotPRegistrationFeral) do
		if recap_temp.InGroup[i] == recap_temp.InGroup[destNameGUID] then
			-- use only if both group, or both non-group
			if (timestamp - recap_temp.ILotPRegistrationFeral[i].Timestamp) < 2 then
				-- only recent feral forms count, take the first one we find (could be not entirely accurate, tough) (overwrites any previous registration)
				recap_temp.ILotPRegistration[destNameGUID] = i
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." ILotP new registration (Leader of the Pack only) for "..destNameGUID.." on behalf of source "..i)
				return
			end
		end
	end
end

function Recap_RemoveEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, auraType)

	local typeEffect

	-- Note that a Buff or Debuff Removed event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if destNameGUID then -- guard
		-- special case code for priest Prayer of Mending
		if (spellID == recap_temp.PoMAuraID) then
			Recap_CreatePoMMigrationEntryFromRegistration(timestamp, destNameGUID)
		end

		-- special case code for paladin Judgement of Light
		if (spellID == recap_temp.JoLAuraID) then
			recap_temp.JoLRegistration[destNameGUID] = nil
		end

		-- special case code for druid Leader of the Pack
		if (spellID == recap_temp.ILotPAuraID) then
			recap_temp.ILotPRegistration[destNameGUID] = nil
		end
	end

	if recap.Opt.LightData.value or not recap.Opt.OtherData.value then
		-- only track as Other events if not in Light Data Mode, and with the Other Data Mode enabled
		return
	end

	-- we ignore de-stacking events
	if destNameGUID and recap.Combatant[destNameGUID] then
		-- look for the matching buff or debuff
		if auraType == "BUFF" then
			typeEffect = "5"..spellName
		else
			typeEffect = "3"..spellName
		end
		if recap.Combatant[destNameGUID].OtherDetail and recap.Combatant[destNameGUID].OtherDetail[typeEffect] then
			-- Removal of a buff or debuff implies that it has run out, so mark its end and calculate a duration
			Recap_CalcOtherDuration(recap.Combatant[destNameGUID].OtherDetail[typeEffect], timestamp)
		end
		local theDetail = "LastOtherDetail_"..recap_temp.ActiveLastFight
		if recap.Combatant[destNameGUID][theDetail] and recap.Combatant[destNameGUID][theDetail][typeEffect] then
			-- Removal of a buff or debuff implies that it has run out, so mark its end and calculate a duration
			Recap_CalcOtherDuration(recap.Combatant[destNameGUID][theDetail][typeEffect], timestamp)
		end
	end
end

function Recap_CreatePoMMigrationEntryFromRegistration(timestamp, destNameGUID)

	local regDest, migDest

	-- PoM aura removed, either prior to a heal or on expiry
	if recap_temp.PoMRegistration[destNameGUID] and recap_temp.PoMRegistration[destNameGUID].OriginalCaster then
		-- registered, check timestamp
		regDest = recap_temp.PoMRegistration[destNameGUID]
		if (timestamp - regDest.Timestamp) < 29.3 then -- empirically it looks as though an expired record may show as having lasted 29.1 seconds or longer, so assume (at a small risk) that anything over 29.3 has expired
			-- current, check count
			if regDest.Charges > 0 then
				-- count okay, two things might happen
				-- if count is 2 or more, we copy to Migration, because there's still life left in this PoM
				-- if count is 1 then this is the final heal for this PoM, so we don't actually copy it to Migration, but we do everything else
				if regDest.Charges > 1 then
					-- copy to migration
					if not recap_temp.PoMMigration[destNameGUID] then
						recap_temp.PoMMigration[destNameGUID] = {}
					end
					migDest = recap_temp.PoMMigration[destNameGUID]
					-- if there happens to already be a Migration record, we overwrite it without checking (we assume that Blizzard follows the rules)
					-- NOTE: the expiry of a PoM aura (no migration intended) [case 1] can be ambiguous with the removal of a PoM aura because it is about to migrate [case 2],
					--       and with the removal of a PoM aura because it would be able to migrate but has no valid migration target [case 3] (which could be because of range,
					--       or because all targets within range already have the aura).
					--     As far as I can tell we have no reliable way of distinguishing among these cases in advance.
					--     Case 3 is okay because we will only occasionally see another acquisition of a PoM aura within the short time allowed by the code.
					--     Cases 1 and 2 can be confused and lead to mis-attribution if case 1 precedes case 2 and if there is an acquisition of a PoM aura within
					--       the short time allowed by the code for both.
					migDest.OriginalCaster = regDest.OriginalCaster
					migDest.OriginalTimestamp = regDest.OriginalTimestamp
					migDest.Timestamp = timestamp -- update as of the loss of aura (there is normally only a short time before another target acquires the aura)
					migDest.Charges = regDest.Charges - 1 -- update by removing one charge
if regDest.HasAura == true then
if regDest.Healed == true then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." queued up for migration from "..destNameGUID.." (has aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs (old registration deleted)")
else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." queued up for migration from "..destNameGUID.." (has aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs")
end
else
if regDest.Healed == true then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." queued up for migration from "..destNameGUID.." (no aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs (old registration deleted)")
else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." queued up for migration from "..destNameGUID.." (no aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs")
end
end
				else
					-- one charge, this is the final one, so we don't actually migrate (but we pretend we did in case the heal comes later)
if regDest.HasAura == true then
if regDest.Healed == true then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." no actual migration (final charge) from "..destNameGUID.." (has aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs (old registration deleted)")
else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." no actual migration (final charge) from "..destNameGUID.." (has aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs")
end
else
if regDest.Healed == true then
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." no actual migration (final charge) from "..destNameGUID.." (no aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs (old registration deleted)")
else
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." no actual migration (final charge) from "..destNameGUID.." (no aura) after having lasted "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." secs")
end
end
				end
				regDest.Migrated = true
				if (regDest.Healed == true) then
					-- already healed, we can safely delete this registration
					--   (code supports migration before healing which is the usual case, and healing before migration)
					recap_temp.PoMRegistration[destNameGUID] = nil
				end
				-- note that if a registration and a loss-of-aura event get sent to us in the wrong order then we will migrate the registration prematurely, and could miss an attribution
			else
				-- no charges left (possibly anomalous)
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." no migration, ran out of charges for "..destNameGUID.." (deleted)")
				recap_temp.PoMRegistration[destNameGUID] = nil
			end
		else
			-- expired
--Recap_DumpMessage(Recap_FormatTimeRecent(timestamp).." PoM "..Recap_FormatTimeRecent(regDest.OriginalTimestamp).." registration for "..destNameGUID.." deleted (expired "..Recap_FormatTimePoM(timestamp - regDest.Timestamp).." seconds)")
			recap_temp.PoMRegistration[destNameGUID] = nil
		end
	else
		-- no registration entry, we may have already migrated as part of the heal (migration follows heal), ignore the loss of aura
	end
end

function Recap_DispelEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName, auraType)

	local typeEffect

	-- Note that a Dispel event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if destNameGUID then -- guard
		-- special case code for priest Prayer of Mending
		if (spellID == recap_temp.PoMAuraID) then
			-- PoM aura dispelled (in case that's possible)
			recap_temp.PoMRegistration[destNameGUID] = nil
		end

		-- special case code for paladin Judgement of Light
		if (spellID == recap_temp.JoLAuraID) then
			recap_temp.JoLRegistration[destNameGUID] = nil
		end

		-- special case code for druid Leader of the Pack
		if (spellID == recap_temp.ILotPAuraID) then
			recap_temp.ILotPRegistration[destNameGUID] = nil
		end
	end

	if recap.Opt.LightData.value or not recap.Opt.OtherData.value then
		-- only track as Other events if not in Light Data Mode, and with the Other Data Mode enabled
		return
	end

	-- the aura being dispelled is extraSpellName, and the dispelling or breaking is being done by spellName
	if sourceNameGUID and recap.Combatant[sourceNameGUID] and spellName then
		-- we treat this as an unknown event since we can't tell whether dispelling the aura is a good thing or a bad thing
		-- this can include things like a DoT breaking a shackle
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "7", spellName, 1, recap_temp.Localize.Dispels, "OtherDetail")
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "7", spellName, 1, recap_temp.Localize.Dispels, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		-- look for the matching buff or debuff
		if auraType == "BUFF" then
			typeEffect = "5"..extraSpellName
		else
			typeEffect = "3"..extraSpellName
		end
		if recap.Combatant[destNameGUID].OtherDetail and recap.Combatant[destNameGUID].OtherDetail[typeEffect] then
			recap.Combatant[destNameGUID].OtherDetail[typeEffect].Dispels = (recap.Combatant[destNameGUID].OtherDetail[typeEffect].Dispels or 0) + 1
		end
		local theDetail = "LastOtherDetail_"..recap_temp.ActiveLastFight
		if recap.Combatant[destNameGUID][theDetail] and recap.Combatant[destNameGUID][theDetail][typeEffect] then
			recap.Combatant[destNameGUID][theDetail][typeEffect].Dispels = (recap.Combatant[destNameGUID][theDetail][typeEffect].Dispels or 0) + 1
		end
	end

	-- TODO: this is semi-experimental -- we might eventually do this full dispel info (including dispelling and the breaking of crowd control) in some other way
	if recap.Opt.RecentData.value and sourceNameGUID and recap.Combatant[sourceNameGUID] then
		-- effectType 6 is dispel (the ones without a source appear to be uninteresting)
		Recap_AddRecentEvent(timestamp, sourceNameGUID, destNameGUID, 6, spellName, extraSpellName, nil, nil, nil)
	end
end

-- a new event type that wasn't handled before patch 2.4
function Recap_StealEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName, auraType)

	local typeEffect, iDetail

	-- Note that a Steal event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if destNameGUID then -- guard
		-- special case code for priest Prayer of Mending
		if (extraSpellId == recap_temp.PoMAuraID) then
			recap_temp.PoMRegistration[destNameGUID] = nil
		end

		-- special case code for paladin Judgement of Light
		if (extraSpellId == recap_temp.JoLAuraID) then
			recap_temp.JoLRegistration[destNameGUID] = nil
		end

		-- special case code for druid Leader of the Pack
		if (spellID == recap_temp.ILotPAuraID) then
			recap_temp.ILotPRegistration[destNameGUID] = nil
		end
	end

	if recap.Opt.LightData.value or not recap.Opt.OtherData.value then
		-- only track as Other events if not in Light Data Mode, and with the Other Data Mode enabled
		return
	end

	-- the aura being stolen is extraSpellName, and the stealing is being done by spellName
	if sourceNameGUID and recap.Combatant[sourceNameGUID] and extraSpellName then
		-- for consistency with the dispelling (intentional or otherwise), we report this intentional stealing as an unknown
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "7", spellName, 1, RECAP_STEALS, "OtherDetail")
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "7", spellName, 1, RECAP_STEALS, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		-- look for the matching buff or debuff
		if auraType == "BUFF" then
			typeEffect = "5"..extraSpellName
		else
			typeEffect = "3"..extraSpellName
		end
		if recap.Combatant[destNameGUID].OtherDetail and recap.Combatant[destNameGUID].OtherDetail[typeEffect] then
			recap.Combatant[destNameGUID].OtherDetail[typeEffect].Steals = (recap.Combatant[destNameGUID].OtherDetail[typeEffect].Steals or 0) + 1
		end
		local theDetail = "LastOtherDetail_"..recap_temp.ActiveLastFight
		if recap.Combatant[destNameGUID][theDetail] and recap.Combatant[destNameGUID][theDetail][typeEffect] then
			recap.Combatant[destNameGUID][theDetail][typeEffect].Steals = (recap.Combatant[destNameGUID][theDetail][typeEffect].Steals or 0) + 1
		end
	end
end

function Recap_InterruptEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName, extraSpellId, extraSpellName, extraSpellSchool, extraSchoolName)

	-- Note that an Interrupt event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	-- spellName does the interrupting, extraSpellName is what gets interrupted
	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		-- we treat this as an unknown event since we can't tell whether interrupting is a good thing or a bad thing
		-- besides, it is consistent with 'Dispels' (as 'Interrupts')
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "7", spellName, 1, recap_temp.Localize.Interrupts, "OtherDetail")
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "7", spellName, 1, recap_temp.Localize.Interrupts, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		-- treat an incoming interrupt in a similar way (as 'Interrupted')
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "7", spellName, 1, recap_temp.Localize.Interrupted, "OtherDetail")
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "7", spellName, 1, recap_temp.Localize.Interrupted, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
end

function Recap_DurabilityEvent(timestamp, eventType, sourceNameGUID, sourceFlags, destNameGUID, destFlags, spellID, spellName, spellSchool, schoolName)

	-- Note that a Buff or Debuff event does not trigger timers

	sourceNameGUID, destNameGUID = Recap_FilterDoNotStore(sourceNameGUID, destNameGUID)
	if (sourceNameGUID == nil) and (destNameGUID == nil) then
		-- ignore entirely
		return
	end

	if sourceNameGUID and recap.Combatant[sourceNameGUID] then
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "1", spellName, 1, recap_temp.Localize.Durability, "OtherDetail")
		Recap_CreateFullOtherDetail(sourceNameGUID, timestamp, "1", spellName, 1, recap_temp.Localize.Durability, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
	if destNameGUID and recap.Combatant[destNameGUID] then
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "3", spellName, 1, recap_temp.Localize.Durability, "OtherDetail")
		Recap_CreateFullOtherDetail(destNameGUID, timestamp, "3", spellName, 1, recap_temp.Localize.Durability, "LastOtherDetail_"..recap_temp.ActiveLastFight)
	end
end

function Recap_Register_CombatEvents()

	if not recap_temp.CombatEventsRegistered then
		-- events for WoW patch 2.4
		RecapCombatEvents:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED");

		-- two other events
		RecapCombatEvents:RegisterEvent("UPDATE_MOUSEOVER_UNIT")
		RecapCombatEvents:RegisterEvent("PLAYER_TARGET_CHANGED")

		-- register for receiving messages on sync channel
		--   in a sense this doesn't really belong here (oh well)
		Recap:RegisterComm(Recap.commPrefix, "GROUP", "OnCommReceived")

		recap_temp.CombatEventsRegistered = true
	end
end

function Recap_Unregister_CombatEvents()
	if recap_temp.CombatEventsRegistered then
		RecapCombatEvents:UnregisterAllEvents()
		Recap:UnregisterComm(Recap.commPrefix, "GROUP")
		recap_temp.CombatEventsRegistered = false
	end
end


RecapCombat_lua_411 = true
