﻿--[[ Auxiliary functions, assorted ]]

local _G = getfenv(0)
local math_floor = _G.math.floor
local math_max = _G.math.max
local math_min = _G.math.min
local string_find = _G.string.find
local string_format = _G.string.format
local string_gmatch = _G.string.gmatch
local string_gsub = _G.string.gsub
local string_len = _G.string.len
local string_sub = _G.string.sub
local string_upper = _G.string.upper

--[[ local tables ]]

-- keys for (det)ails
-- 2008-February-12: E again available with removal of 'vulnerable'
local detkey =	{ ["HitsDmg"]="d", ["Hits"]="n", ["HitsMax"]="x", ["CritsDmg"]="D", ["Crits"]="N",
				  ["CritsMax"]="X", ["CritsEvents"]="e", ["Missed"]="m", ["TicksDmg"]="t", ["TicksMax"]="T", ["Ticks"]="i",
				  ["Dodged"]="G", ["Parried"]="P", ["Blocked"]="B", ["Absorbed"]="A", ["Deflected"]="F",
				  ["Evaded"]="V", ["Resisted"]="R", ["Reflected"]="S", ["Immune"]="I",
				  ["Crushes"]="r", ["CrushDmg"]="s", ["CrushMax"]="u",
				  ["Total"]="v", ["Max"]="w", ["Dispels"]="y", ["PrevTime"]="z", ["ICount"]="a", ["ITotal"]="b", ["DCount"]="c",
				  ["DTotal"]="f", ["PrevTimeDur"]="g", ["GlancesDmg"]="h", ["Glances"]="j", ["GlancesMax"]="k",
				  ["PAbsorbs"]="l", ["PAbsorbsDmg"]="o", ["PBlocks"]="p", ["PBlocksDmg"]="q",
				  ["PResists25"]="H", ["PResists25Dmg"]="J", ["PResists50"]="K",
				  ["PResists50Dmg"]="L", ["PResists75"]="M", ["PResists75Dmg"]="O",
				  ["GlancesMin"]="Q", ["HitsMin"]="U", ["CritsMin"]="W", ["CrushMin"]="Y", ["TicksMin"]="Z",
				  ["EstResistedDmg"]="aa", ["EstResistableDmg"]="bb", ["Steals"]="C" }

-- keys for (inc)oming data sets
-- added NonMeleeTicks -- for Incoming, will eventually be obsolete
local inckey = { ["MeleeDamage"]="a", ["MeleeMax"]="b", ["MeleeHits"]="c", ["MeleeCrits"]="d", ["MeleeMissed"]="e",
				 ["MeleeDodged"]="f", ["MeleeParried"]="g", ["MeleeBlocked"]="h", ["NonMeleeDamage"]="i",
				 ["NonMeleeMax"]="j", ["NonMeleeHits"]="k", ["NonMeleeCrits"]="l", ["NonMeleeMissed"]="m",
				 ["NonMeleeTicks"]="n" }


--[[ Auxiliary functions, assorted ]]


function Recap_Round0(a)
	return math_floor(a + 0.5)
end
function Recap_Round2(a)
	return math_floor(a * 100 + 0.5) / 100
end
function Recap_Div0(a, b)
	return math_floor(a/b + 0.5)
end
function Recap_Round1(a)
	return math_floor(a * 10 + 0.5) / 10
end
function Recap_Div1(a, b)
	return math_floor(a/b * 10 + 0.5) / 10
end
function Recap_Div2(a, b)
	return math_floor(a/b * 100 + 0.5) / 100
end
function Recap_Min(a, b)
	if a and b then
		return math_min(a, b)
	elseif a then
		return a
	else
		return b
	end
end

function Recap_PetsMerged()
	if (recap.Opt.SyncState.value == "Member") or (recap.Opt.SyncState.value == "Leader") then
		if recap.Opt.SyncMergePets.value == "on" then
			-- synchronization active and merge pets is on
			return true
		end
	else
		if recap.Opt.MergePets.value then
			-- synchronization not active and merge pets is on
			return true
		end
	end
	return false
end

function Recap_PetsMergedText()
	if Recap_PetsMerged() then
		return recap_temp.Localize.MergePetsOn
	else
		return recap_temp.Localize.MergePetsOff
	end
	return false
end

-- do our own error display since WoW 2.1 will have lua error display off by default
function Recap_TrapError(message, frame)
	debuginfo();
	if string_find(message, "Recap", 1, true) then
		-- this is an error with the string "Recap" in it, pretend it is ours
		if (not ScriptErrors:IsVisible()) then
			-- but only if there is not already an error message visible
			ScriptErrors_Message:SetFormattedText("Recap [%.2f]: %s", Recap_Version, message)
			ScriptErrors:Show()
			if recap_temp.snap then
				Screenshot()
			end
		end
	else
		-- probably not a Recap error, pass it to the original error handler
		if recap_Original_ErrorHandler then
			return recap_Original_ErrorHandler(message, frame)
		end
	end
end

function Recap_AuthorDebug(msg)
	if (recap_temp.p == "Hawksy_Elune") or (recap_temp.p == "Galassa_Elune") or (recap_temp.p == "Metrognome_Elune") then
		DEFAULT_CHAT_FRAME:AddMessage(msg)
	end
end

-- dump combat events to WOWChatLog.txt
function Recap_DumpMessage(printable)
	-- include a line count so that we can detect if any lines get dropped
	recap.dumplines = recap.dumplines + 1
	-- limit message to 244 leaving plenty of room for the line number
	ChatThrottleLib:SendChatMessage("BULK", "Recap", tostring(recap.dumplines)..", "..string_sub(printable,1,244), "CHANNEL", nil, recap_temp.LogChatnum)
end

function Recap_EventText(...)
	local timestamp, eventtype, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14, x15 = ...
	local printable = tostring(timestamp)..", "..tostring(eventtype)..", "..tostring(sourceGUID)..", "..tostring(sourceName)..", "..string_format("0x%X", sourceFlags)..", "..tostring(destGUID)..", "..tostring(destName)..", "..string_format("0x%X", destFlags)
	-- only dump extras up to the last non-nil value
	-- this does not distinguish between trailing values which are nil and trailing values which are absent
	local extras = ""
	if x15 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)..", "..tostring(x9)..", "..tostring(x10)..", "..tostring(x11)..", "..tostring(x12)..", "..tostring(x13)..", "..tostring(x14)..", "..tostring(x15)
	elseif x14 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)..", "..tostring(x9)..", "..tostring(x10)..", "..tostring(x11)..", "..tostring(x12)..", "..tostring(x13)..", "..tostring(x14)
	elseif x13 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)..", "..tostring(x9)..", "..tostring(x10)..", "..tostring(x11)..", "..tostring(x12)..", "..tostring(x13)
	elseif x12 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)..", "..tostring(x9)..", "..tostring(x10)..", "..tostring(x11)..", "..tostring(x12)
	elseif x11 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)..", "..tostring(x9)..", "..tostring(x10)..", "..tostring(x11)
	elseif x10 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)..", "..tostring(x9)..", "..tostring(x10)
	elseif x9 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)..", "..tostring(x9)
	elseif x8 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)..", "..tostring(x8)
	elseif x7 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)..", "..tostring(x7)
	elseif x6 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)..", "..tostring(x6)
	elseif x5 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)..", "..tostring(x5)
	elseif x4 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)..", "..tostring(x4)
	elseif x3 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)..", "..tostring(x3)
	elseif x2 ~= nil then
		extras = tostring(x1)..", "..tostring(x2)
	elseif x1 ~= nil then
		extras = tostring(x1)
	end
	if extras ~= "" then
		printable = printable..", "..extras
	end
	return printable
end


function Recap_CheckUnexpectedEvent(reason, ...)
	if recap_temp.UnexpectedErrorMessageCount < 10 then
		-- only emit the first few error messages after a log or reload
		local text = string_format("Recap [%.2f]: %s: %s", Recap_Version, reason, Recap_EventText(...))
		if recap.dump then
			Recap_DumpMessage(text)
		end
		DEFAULT_CHAT_FRAME:AddMessage(text)
		if recap_temp.snap then
			Screenshot()
		end
		recap_temp.UnexpectedErrorMessageCount = recap_temp.UnexpectedErrorMessageCount + 1
	end
end


-- is this the player (or pet)?
function Recap_CheckForSelf(combatant)

	local okayToShow = false

	if combatant == recap_temp.PlayerGUID then
		-- player
		okayToShow = true
	else
		if Recap_IsOwnedBy(combatant, recap_temp.PlayerGUID) then
			-- is or was controlled by player
			okayToShow = true
		end
	end

	return okayToShow
end


--[[ Misc functions ]]

-- creates a new .Combatant[thisCombatant]
function Recap_CreateBlankCombatant(thisCombatant, seen, flags)

-- a number of fields have been commented out to save space for combatants who don't use them -- they will be created as needed by other code
-- TotalTime, TotalTimeIn, TotalTimeHeal, TotalDmgIn, TotalDmgOut, TotalDPS, TotalDPSIn, TotalHPS, TotalMaxHit, TotalKills, TotalRawHeal, TotalOverHeal
-- Seen, Trash, Locked, Ignore, OwnedBy, Class, Faction, Level
-- Controlled (flags whether Blizzard flags said this combatant was controlled as a pet while also OwnedBy)
-- OutgoingDetail, TargetDetail, IncomingDetail, SourceDetail, OtherDetail

-- we have two sets of Last Fight numbers, and we swap between them at the end of any 'significant' fight (avoids extra copying)
-- the 'hard coded' 1 and 2 are less pure but match the hierarchical structure of OutgoingDetail etc. for easier parallel coding elsewhere
-- LastTime_1, LastTimeIn_1, LastTimeHeal_1, LastMaxHit_1, LastDmgIn_1, LastDmgOut_1, LastHPS_1, LastDPSIn_1, LastDPS_1, LastKills_1, LastRawHeal_1, LastOverHeal_1
-- LastOutgoingDetail_1, LastTargetDetail_1, LastIncomingDetail_1, LastSourceDetail_1, LastOtherDetail_1
-- LastTime_2, LastTimeIn_2, LastTimeHeal_2, LastMaxHit_2, LastDmgIn_2, LastDmgOut_2, LastHPS_2, LastDPSIn_2, LastDPS_2, LastKills_2, LastRawHeal_2, LastOverHeal_2
-- LastOutgoingDetail_2, LastTargetDetail_2, LastIncomingDetail_2, LastSourceDetail_2, LastOtherDetail_2

	if not recap.Combatant[thisCombatant] then
		recap.Combatant[thisCombatant] = { WasInLast = false, WasInCurrent = false, Friend = false }
	end

	if seen and seen>0 then
		recap.Combatant[thisCombatant].Seen = seen
	end

	if flags and flags>0 then
		recap.Combatant[thisCombatant].Flags = flags
	end

	if not recap_temp.Last[thisCombatant] then
		Recap_InitializeLastFight(thisCombatant)
	end
end

-- this function is called only for player, party, or raid members, and their pets
-- if this is a pet, pass owner's name via ownerUnit
-- caller has already checked that thisCombatant is not nil
function Recap_AddFriend(thisCombatant, combatantUnit, ownerUnit)

	local thisOwner = nil

	if ownerUnit then
		thisOwner = (UnitName(ownerUnit))
		if (thisOwner == UNKNOWNOBJECT) then
			thisOwner = nil
		end
	end
	if not recap.Opt.IgnoreGUIDs.value then
		-- using GUIDs (the normal case)
		local g
		if thisOwner then
			g = (UnitGUID(ownerUnit))
			if g then
				thisOwner = thisOwner.."_"..Recap_TrimGUID(g)
			end
		end
	end

	if thisOwner then
		-- creating a pet (force them to be friends)
		local ownerPetNameGUID = Recap_CreatePet(thisCombatant, nil, thisOwner, nil, true, 0)
		-- remember as part of the group (i.e. a true friend not just a combatant marked as a friend)
		recap_temp.InGroup[ownerPetNameGUID] = true
	else
		-- creating a combatant that is not a pet
		if not recap.Combatant[thisCombatant] then
			Recap_CreateBlankCombatant(thisCombatant, 0, nil)
		end
		if not recap_temp.Last[thisCombatant] then
			Recap_InitializeLastFight(thisCombatant)
		end
		recap.Combatant[thisCombatant].Friend = true
		recap_temp.InFriend[thisCombatant] = true
		-- remember as part of the group (i.e. a true friend not just a combatant marked as a friend)
		recap_temp.InGroup[thisCombatant] = true
	end
end

function Recap_MakeFriends()

	local i, j, thisCombatant

	-- UnitName now returns two values, name and realm; and if used as an argument will insert two arguments into the call,
	--   which is usually undesirable.  To force UnitName to return one value only, wrap it in parentheses in the context
	--   of a call.  I've done the wrapping everywhere, whether necessary or not.

	-- clear list of group members (this churns every half second, but I can't think of a cheaper way to do this at the moment)
	recap_temp.InGroup = {}

	-- the following strict ordering is probably not necessary with patch 2.4 if always using GUIDs

	-- to avoid anomalies when players have the same names as pets, do pets first, then players
	-- what then happens is that NAME is marked as a pet (by Recap_AddFriend), then re-marked as a player (by Recap_AddFriend)
	if UnitExists("pet") then
		thisCombatant = Recap_GetUnitInfo("pet")
		if thisCombatant then
			Recap_AddFriend(thisCombatant, "pet", "player")
		end
	end

	-- create blank entries for group pets if we are in sync, or if we are including group, or if we are storing everyone
	if (not recap.Opt.HideGroup.value) or (recap.Opt.SyncState.value == "Member") or (recap.Opt.SyncState.value == "Leader") or (not recap.Opt.StoreOnlyDisplayed.value) then
		for i=1,4 do
			j = "partypet"..i
			if UnitExists(j) then
				thisCombatant = Recap_GetUnitInfo(j)
				if thisCombatant then
					Recap_AddFriend(thisCombatant, j, "party"..i)
				end
			end
		end
		if (GetNumRaidMembers()>0) then
			for i=1,40 do
				j = "raidpet"..i
				if UnitExists(j) then
					thisCombatant = Recap_GetUnitInfo(j)
					if thisCombatant then
						Recap_AddFriend(thisCombatant, j, "raid"..i)
					end
				end
			end
		end
	end

	-- now do players (player with same name as pet might overwrite the pet designation) (should be rare or non-existent with patch 2.4)
	thisCombatant = Recap_GetUnitInfo("player")
	if thisCombatant then
		Recap_AddFriend(thisCombatant, "player", nil)
	end

	-- create blank entries for group members if we are in sync, or if we are including group, or if we are storing everyone
	if (not recap.Opt.HideGroup.value) or (recap.Opt.SyncState.value == "Member") or (recap.Opt.SyncState.value == "Leader") or (not recap.Opt.StoreOnlyDisplayed.value) then
		for i=1,4 do
			j = "party"..i
			if UnitExists(j) then
				thisCombatant = Recap_GetUnitInfo(j)
				if thisCombatant then
					Recap_AddFriend(thisCombatant, j, nil)
				end
			end
		end
		if (GetNumRaidMembers()>0) then
			for i=1,40 do
				j = "raid"..i
				if UnitExists(j) then
					thisCombatant = Recap_GetUnitInfo(j)
					if thisCombatant then
						Recap_AddFriend(thisCombatant, j, nil)
					end
				end
			end
		end
	end
end

-- if petFlags is nil this is called from MakeFriends
-- if petFlags is not nil this is called from SummonOrCreate (which does not include Mind Control)
function Recap_CreatePet(petNameGUID, petFlags, ownerNameGUID, ownerFlags, forceFriend, seen)

	local iExistingPet, iNewPet, ownerPetNameGUID

	iExistingPet = recap.Combatant[petNameGUID]
	iNewPet = nil

	-- if not already a Combatant, add an entry for the pet
	if not recap.Combatant[petNameGUID] then
		Recap_CreateBlankCombatant(petNameGUID, seen, petFlags)
	end
	if not recap_temp.Last[petNameGUID] then
		Recap_InitializeLastFight(petNameGUID)
	end

	-- pet could potentially do damage without owner, so add owner if not already present
	if not recap.Combatant[ownerNameGUID] then
		Recap_CreateBlankCombatant(ownerNameGUID, seen, ownerFlags)
	end
	if not recap_temp.Last[ownerNameGUID] then
		Recap_InitializeLastFight(ownerNameGUID)
	end
	if forceFriend then
		recap.Combatant[ownerNameGUID].Friend = true
		recap_temp.InFriend[ownerNameGUID] = true
	end

	-- if the owner is a friend, then the pet is too
	if forceFriend or recap.Combatant[ownerNameGUID].Friend then
		recap.Combatant[petNameGUID].Friend = true
		recap_temp.InFriend[petNameGUID] = true
	end

	-- set the pet to point to its new (and current) owner
	-- this unqualified pet combatant will probably not track anything
	recap.Combatant[petNameGUID].OwnedBy = ownerNameGUID

	-- create the fully qualified pet name which is "owner:pet"
	-- this is the combatant for which Recap will actually track damage and healing
	ownerPetNameGUID = ownerNameGUID..":"..petNameGUID
	if not recap.Combatant[ownerPetNameGUID] then
		Recap_CreateBlankCombatant(ownerPetNameGUID, seen, petFlags)
		-- remember that this is a brand new combatant
		iNewPet = recap.Combatant[ownerPetNameGUID]
	end
	if not recap_temp.Last[ownerPetNameGUID] then
		Recap_InitializeLastFight(ownerPetNameGUID)
	end
	-- if the owner is a friend, then the fully qualified pet name is too
	if forceFriend or recap.Combatant[ownerNameGUID].Friend then
		recap.Combatant[ownerPetNameGUID].Friend = true
		recap_temp.InFriend[ownerPetNameGUID] = true
	end

	-- we do not currently keep a list of the owner's pets



-- TODO: this crap might no longer be necessary
-- TODO this doesn't tell me enough
if false and iExistingPet and not iNewPet and string_find(Recap_NameFromNameGUID(petNameGUID), "Elemental", 1, true) and recap_temp.Last[petNameGUID] and (recap_temp.Last[petNameGUID].DmgOut > 0) then
	-- we're working with an Elemental
	local text = petNameGUID
	if petFlags then
		text = text.." new pet flags: "..string_format("0x%X", petFlags)
	else
		text = text.." new pet flags: nil"
	end
	text = text.." last damage out: "..tostring(recap_temp.Last[petNameGUID].DmgOut)
	text = text.." (compound pet exists: "..ownerPetNameGUID..")"
	if recap.Combatant[ownerPetNameGUID].Flags then
		text = text.." compound pet flags: "..string_format("0x%X", recap.Combatant[ownerPetNameGUID].Flags)
	else
		text = text.." compound pet flags: nil"
	end
	text = text.." (state "..recap.Opt.State.value..")"
	DEFAULT_CHAT_FRAME:AddMessage(text)
	Screenshot()
end



-- TODO: need to find out what a mind-controlled mob's flags look like when we get here
-- transition from not controlled to controlled (three bits worth) could perhaps be used to trigger the move
if false and iExistingPet and iNewPet and (petFlags == nil) and recap.Combatant[petNameGUID] and recap.Combatant[petNameGUID].Flags then
 DEFAULT_CHAT_FRAME:AddMessage("Existing "..petNameGUID.." to brand new "..ownerPetNameGUID.." -- existing pet flags: "..string_format("0x%X", recap.Combatant[petNameGUID].Flags))
-- okay, the controlled mob shows up first with pre-control flags of 0x10A48 or 0xA48 -- standard for a mob
-- when controlled the flags are then 0x1111 or similar (friendly, player-controlled, pet)
end



	-- it is possible that petNameGUID already exists and has some data (for example if we see a damage event before we see the matching summon event, or for some other reason)
	-- if so, *move* that information to the new ownerPetNameGUID
	-- we do this if
	--   1) we are creating the pet for Recap_MakeFriends (petFlags will be nil, but we know that this must be a true pet), or
	--   2) we are creating the pet for Recap_SummonOrCreateEvent with petFlags indicating that this is a pet, guardian, or object

	-- we must avoid doing this if we are mind controlling a player or a hostile mob (both of which come here too) -- we do not want to grab that combatant's past data

-- TODO: turned off at the moment since it does the wrong thing (copies) for a mind-controlled pet
-- we want to do this *only* for a summoned pet that is somehow getting licks in before ownership is established
-- what I can't understand is that in some samples there is a five second gap between the Water Elemental doing damage and that Water Elemental getting assigned to the mage as a pet -- is there no summon event ?
-- TODO: okay, we are only doing this when in combat -- is that right? -- perhaps we are losing those Water Elementals because they are opening the fight and for some reason the State is not yet Active -- can that happen ?
-- Definitely missing a lot of these in the latest Tempest Keep run (leading up to VR, and the fight against VR) -- must be mid-fight
	if false and iExistingPet and iNewPet and ((petFlags == nil) or (petFlags and (bit.band(petFlags, recap_temp.PetCombatantMask) ~= 0))) and (recap.Opt.State.value == "Active") then
		-- assert: iNewPet represents blank data

		local iExistingLast = recap_temp.Last[petNameGUID]
		local iNewLast = recap_temp.Last[ownerPetNameGUID]

-- TODO
-- TODO: damn, a mind-controlled critter comes in here as a result of MakeFriends, petFlags being nil, which ***incorrectly*** has us copying data
if petFlags then
 DEFAULT_CHAT_FRAME:AddMessage("Copying "..petNameGUID.." to "..ownerPetNameGUID.." flags: "..string_format("0x%X", petFlags))
else
 DEFAULT_CHAT_FRAME:AddMessage("Copying "..petNameGUID.." to "..ownerPetNameGUID.." flags: nil")
end



		-- copy the older value for seen
		if iExistingPet.Seen then
			iNewPet.Seen = iExistingPet.Seen
		end

		-- we copy only from the live Last Fight information
		-- if the existing pet was from an earlier fight (which we don't expect) then tough
		--   ( All Fights details are copied anyway, since they are created live )
		if iExistingLast.DmgIn then
			iNewLast.DmgIn = iExistingLast.DmgIn
			iExistingLast.DmgIn = 0
		end
		if iExistingLast.DmgOut then
			iNewLast.DmgOut = iExistingLast.DmgOut
			iExistingLast.DmgOut = 0
		end
		if iExistingLast.StartOut then
			iNewLast.StartOut = iExistingLast.StartOut
			iExistingLast.StartOut = 0
		end
		if iExistingLast.EndOut then
			iNewLast.EndOut = iExistingLast.EndOut
			iExistingLast.EndOut = 0
		end
		if iExistingLast.EndIn then
			iNewLast.EndIn = iExistingLast.EndIn
			iExistingLast.EndIn = 0
		end
		if iExistingLast.StartHeal then
			iNewLast.StartHeal = iExistingLast.StartHeal
			iExistingLast.StartHeal = 0
		end
		if iExistingLast.EndHeal then
			iNewLast.EndHeal = iExistingLast.EndHeal
			iExistingLast.EndHeal = 0
		end
		if iExistingLast.MaxHit then
			iNewLast.MaxHit = iExistingLast.MaxHit
			iExistingLast.MaxHit = 0
		end
		if iExistingLast.Kills then
			iNewLast.Kills = iExistingLast.Kills
			iExistingLast.Kills = 0
		end
		if iExistingLast.RawHeal then
			iNewLast.RawHeal = iExistingLast.RawHeal
			iExistingLast.RawHeal = 0
		end
		if iExistingLast.OverHeal then
			iNewLast.OverHeal = iExistingLast.OverHeal
			iExistingLast.OverHeal = 0
		end
		if iExistingPet.LastOutgoingDetail_1 then
			iNewPet.LastOutgoingDetail_1 = iExistingPet.LastOutgoingDetail_1
			iExistingPet.LastOutgoingDetail_1 = nil
		end
		if iExistingPet.LastOutgoingDetail_2 then
			iNewPet.LastOutgoingDetail_2 = iExistingPet.LastOutgoingDetail_2
			iExistingPet.LastOutgoingDetail_2 = nil
		end
		if iExistingPet.OutgoingDetail then
			iNewPet.OutgoingDetail = iExistingPet.OutgoingDetail
			iExistingPet.OutgoingDetail = nil
		end
		if iExistingPet.LastTargetDetail_1 then
			iNewPet.LastTargetDetail_1 = iExistingPet.LastTargetDetail_1
			iExistingPet.LastTargetDetail_1 = nil
		end
		if iExistingPet.LastTargetDetail_2 then
			iNewPet.LastTargetDetail_2 = iExistingPet.LastTargetDetail_2
			iExistingPet.LastTargetDetail_2 = nil
		end
		if iExistingPet.TargetDetail then
			iNewPet.TargetDetail = iExistingPet.TargetDetail
			iExistingPet.TargetDetail = nil
		end
		if iExistingPet.LastIncomingDetail_1 then
			iNewPet.LastIncomingDetail_1 = iExistingPet.LastIncomingDetail_1
			iExistingPet.LastIncomingDetail_1 = nil
		end
		if iExistingPet.LastIncomingDetail_2 then
			iNewPet.LastIncomingDetail_2 = iExistingPet.LastIncomingDetail_2
			iExistingPet.LastIncomingDetail_2 = nil
		end
		if iExistingPet.IncomingDetail then
			iNewPet.IncomingDetail = iExistingPet.IncomingDetail
			iExistingPet.IncomingDetail = nil
		end
		if iExistingPet.LastSourceDetail_1 then
			iNewPet.LastSourceDetail_1 = iExistingPet.LastSourceDetail_1
			iExistingPet.LastSourceDetail_1 = nil
		end
		if iExistingPet.LastSourceDetail_2 then
			iNewPet.LastSourceDetail_2 = iExistingPet.LastSourceDetail_2
			iExistingPet.LastSourceDetail_2 = nil
		end
		if iExistingPet.SourceDetail then
			iNewPet.SourceDetail = iExistingPet.SourceDetail
			iExistingPet.SourceDetail = nil
		end
		if iExistingPet.LastOtherDetail_1 then
			iNewPet.LastOtherDetail_1 = iExistingPet.LastOtherDetail_1
			iExistingPet.LastOtherDetail_1 = nil
		end
		if iExistingPet.LastOtherDetail_2 then
			iNewPet.LastOtherDetail_2 = iExistingPet.LastOtherDetail_2
			iExistingPet.LastOtherDetail_2 = nil
		end
		if iExistingPet.OtherDetail then
			iNewPet.OtherDetail = iExistingPet.OtherDetail
			iExistingPet.OtherDetail = nil
		end
		if iExistingPet.WasInCurrent then
			iNewPet.WasInCurrent = iExistingPet.WasInCurrent
			iExistingPet.WasInCurrent = false
		end
	end


	-- if the pet belongs to player, ensure that Self is set up for this pet (not qualified by GUID -- all Self info will amalgamate by name, ignoring GUID)
	if ownerNameGUID == recap_temp.PlayerGUID then
		local petIndex
		petIndex = recap_temp.s..":"..Recap_NameOnlyFromCombatant(petNameGUID)
		if not recap.Self[petIndex] then
			recap.Self[petIndex] = {}
		end
	end

	-- return the fully qualified pet name so it can be used when necessary to mark 'InGroup' membership
	return ownerPetNameGUID
end

function Recap_InitializeLastFight(thisCombatant)
	recap_temp.Last[thisCombatant] = { StartOut = 0, EndOut = 0, StartIn = 0, EndIn = 0, StartHeal = 0, EndHeal = 0, DmgIn = 0, DmgOut = 0, MaxHit = 0, Kills = 0, RawHeal = 0, OverHeal = 0 }
end

-- returns a string of seconds converted to 0:00:00 format
function Recap_FormatTime(rawTime)

	local hours, minutes, seconds
	local text

	seconds = Recap_Round0(rawTime)
	hours = math_floor(seconds/3600)
	seconds = seconds - hours*3600
	minutes = math_floor(seconds/60)
	seconds = seconds - minutes*60

	return ((hours>0) and (hours..":") or "") .. string_format("%02d:",minutes) .. string_format("%02d",seconds)
end

-- returns a string of seconds converted to 00:00:00.0 format (nearest tenth of a second)
function Recap_FormatTimeRecent(rawTime)

	local days, hours, minutes, seconds, deciseconds
	local text

	seconds = math_floor(rawTime)
	deciseconds = Recap_Round0(10 * (rawTime - seconds))
	if deciseconds == 10 then
		seconds = seconds + 1
		deciseconds = 0
	end
	hours = math_floor(seconds/3600)
	seconds = seconds - hours*3600
	minutes = math_floor(seconds/60)
	seconds = seconds - minutes*60

	days = math_floor(hours/24)
	hours = hours - days*24

	return string_format("%02d:",hours) .. string_format("%02d:",minutes) .. string_format("%02d",seconds) .. string_format(".%01d",deciseconds)
end

-- returns a string of seconds converted to 000.0 format (nearest tenth of a second)
function Recap_FormatTimePoM(rawTime)

	local seconds, deciseconds
	local text

	seconds = math_floor(rawTime)
	deciseconds = Recap_Round0(10 * (rawTime - seconds))
	if deciseconds == 10 then
		seconds = seconds + 1
		deciseconds = 0
	end

	return string_format("%d",seconds) .. string_format(".%01d",deciseconds)
end

-- returns a string of seconds converted to 00:00:00 format
function Recap_FormatTimeSeen(rawTime)

	local days, hours, minutes, seconds, deciseconds
	local text

	seconds = math_floor(rawTime)
	deciseconds = Recap_Round0(10 * (rawTime - seconds))
	if deciseconds == 10 then
		seconds = seconds + 1
		deciseconds = 0
	end
	hours = math_floor(seconds/3600)
	seconds = seconds - hours*3600
	minutes = math_floor(seconds/60)
	seconds = seconds - minutes*60

	days = math_floor(hours/24)
	hours = hours - days*24

	return string_format("%02d:",hours) .. string_format("%02d:",minutes) .. string_format("%02d",seconds)
end

function Recap_SetClassIcon(id,class)

	local item
	local _G = getfenv(0)

	item = _G["RecapList"..id.."_Class"]
	if class and recap_temp.ClassIcons[class] then
		item:SetTexCoord(recap_temp.ClassIcons[class].left,recap_temp.ClassIcons[class].right,recap_temp.ClassIcons[class].top,recap_temp.ClassIcons[class].bottom)
	else
		item:SetTexCoord(.9,1,.9,1)
	end
end

function Recap_SetFactionIcon(id,faction)

	local item
	local _G = getfenv(0)

	item = _G["RecapList"..id.."_Faction"]
	if faction and recap_temp.FactionIcons[faction] then
		item:SetTexture(recap_temp.FactionIcons[faction])
	else
		item:SetTexture("")
	end
end

function Recap_SetRecapFrameBackdrop()
	if RecapFrame then
		if recap.Opt.Minimized.value then
			if recap.Opt.MinBack.value then
				RecapFrame:SetBackdropColor(1,1,1,1)
				RecapFrame:SetBackdropBorderColor(.5,.5,.5,1)
			else
				RecapFrame:SetBackdropColor(0,0,0,0)
				RecapFrame:SetBackdropBorderColor(0,0,0,0)
			end
		else
			if recap.Opt.OpaqueBackground.value then
				-- took quite a while to find an actual solid black background file
				RecapFrame:SetBackdrop({bgFile = "Interface\\CharacterFrame\\UI-Party-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
				RecapOptFrame:SetBackdrop({bgFile = "Interface\\CharacterFrame\\UI-Party-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
				RecapPanel:SetBackdrop({bgFile = "Interface\\CharacterFrame\\UI-Party-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
				RecapRecent:SetBackdrop({bgFile = "Interface\\CharacterFrame\\UI-Party-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
			else
				RecapFrame:SetBackdrop({bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
				RecapOptFrame:SetBackdrop({bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
				RecapPanel:SetBackdrop({bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
				RecapRecent:SetBackdrop({bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
										tile = true, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 }})
			end
		end
	end
end

-- not that as well as player, party, raid, and pet, the argument 'unit' could also be "target" or "mouseover"
function Recap_GetUnitInfo(unit)

	local ug = (UnitName(unit))

	if ug and ug~=UNKNOWNOBJECT then

		if not recap.Opt.IgnoreGUIDs.value then
			-- using GUIDs (the normal case)
			local g = (UnitGUID(unit))
			if g then
				ug = ug.."_"..Recap_TrimGUID(g)
			end
		end

		local iCombatant = recap.Combatant[ug]

		if iCombatant then
			_, iCombatant.Class = UnitClass(unit) -- returns localized class
			_, iCombatant.Faction = UnitFactionGroup(unit) -- returns localized faction
			if UnitCreatureFamily(unit) and iCombatant.Faction then
				iCombatant.Class = "Pet"
			end
			iCombatant.Level = tonumber(UnitLevel(unit))
		end

		if unit~="target" and unit~="mouseover" then

			-- create the 'recap_temp.Last' table for this unit so that we
			--   can have HP information from the beginning of a fight, in order to
			--   be able to calculate more accurate overhealing information.  The
			--   code will add party members or raid members, who would normally
			--   soon become combatants anyway.  We're just doing it a bit early.
			--   See also the call to Recap_MakeFriends in Recap_StartFight.
			if not iCombatant then
				Recap_CreateBlankCombatant(ug, 0, nil)
			end
			if not recap_temp.Last[ug] then
				Recap_InitializeLastFight(ug)
			end
			if UnitIsVisible(unit) then
				local maxHealth, maxPower
				recap_temp.Last[ug].HP = UnitHealth(unit)
				maxHealth = UnitHealthMax(unit)
				recap_temp.Last[ug].MaxHP = maxHealth

				-- remember min and max of MaxHP (will be displayed in tooltip)
				-- test for greater than 100 to avoid accidentally recording a percentage
				if iCombatant and maxHealth and (maxHealth > 100) then
					if iCombatant.MinMaxHealth then
						iCombatant.MinMaxHealth = math_min(iCombatant.MinMaxHealth, maxHealth)
					else
						iCombatant.MinMaxHealth = maxHealth
					end
					if iCombatant.MaxMaxHealth then
						iCombatant.MaxMaxHealth = math_max(iCombatant.MaxMaxHealth, maxHealth)
					else
						iCombatant.MaxMaxHealth = maxHealth
					end
				end
				-- similarly for power (i.e. mana, rage, energy)
				maxPower = UnitManaMax(unit)
				if iCombatant and maxPower and (maxPower > 100) then
					if iCombatant.MinMaxPower then
						iCombatant.MinMaxPower = math_min(iCombatant.MinMaxPower, maxPower)
					else
						iCombatant.MinMaxPower = maxPower
					end
					if iCombatant.MaxMaxPower then
						iCombatant.MaxMaxPower = math_max(iCombatant.MaxMaxPower, maxPower)
					else
						iCombatant.MaxMaxPower = maxPower
					end
					-- remember what kind of power it is
-- TODO: does it ever change for a combatant?
					powerType = UnitPowerType(unit)
					if powerType then
						iCombatant.PowerType = powerType
					end
				end
			end
		end

		-- and return the thisCombatant
		return ug
	else
		return nil
	end
end

-- populates .Opt with a copy of DefaultOpt
function Recap_LoadDefaultOpt()

	local i, j

	recap.Opt = {}
	for i in pairs(recap_temp.DefaultOpt) do
		recap.Opt[i] = {}
		for j in pairs(recap_temp.DefaultOpt[i]) do
			recap.Opt[i][j] = recap_temp.DefaultOpt[i][j]
		end
	end
end

function Recap_SaveOpt(user)

	local i, j

	if user and string_len(user)>0 then
		recap[user].Opt = {}
		for i in pairs(recap.Opt) do
			if recap.Opt[i] then
				text = ""
				for j in pairs(recap.Opt[i]) do
					text = text..tostring(j).."="..tostring(recap.Opt[i][j]).." "
				end
				recap[user].Opt[i] = text
			end
		end
	end
end

function Recap_LoadOpt(user)

	local i, j

	if user and recap[user] and recap[user].Opt then
		for i in pairs(recap[user].Opt) do
			if recap_temp.DefaultOpt[i] then
				for t,v in string_gmatch(recap[user].Opt[i],"(%w+)=(%d-%--%w+) ") do
					if v=="true" then
						v = true
					elseif v=="false" then
						v = false
					elseif tonumber(v) then
						-- haven't yet figured it out, but this definitely borks non-integer numbers, forcing them to false -- but not all the time
						v = tonumber(v)
					end
					if t=="type" and recap_temp.DefaultOpt[i][t] then
						v = recap_temp.DefaultOpt[i][t]
					end
					if recap.Opt[i] then
						recap.Opt[i][t] = v
					end
				end
			end
		end

		-- populate current user with any Default options that are missing,
		--   to prevent many 'nil' errors result from missing options.
		for i in pairs(recap_temp.DefaultOpt) do
			if not recap[user].Opt[i] then
				recap[user].Opt[i] = {}
				for j in pairs(recap_temp.DefaultOpt[i]) do
					recap[user].Opt[i][j] = recap_temp.DefaultOpt[i][j]
				end
			end
		end

		-- fix for 3.59 and later
		if recap[user].Opt.PanelDetail then
			recap[user].Opt.PanelDetail = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.PanelDetail then
			recap_temp.DefaultOpt.PanelDetail = nil -- keep as nil
		end

		-- fix for upgrading to 3.74
		if not (recap.Opt.SyncLeader.value == false) then
			if recap.Opt.SyncLeaderName.value == false then
				recap.Opt.SyncLeaderName.value = Recap_SyncExtractName(recap.Opt.SyncLeader.value)
			end
			if recap.Opt.SyncLeaderServer.value == false then
				recap.Opt.SyncLeaderServer.value = Recap_SyncExtractServer(recap.Opt.SyncLeader.value)
			end
		end

		-- fix for 3.75 and later
		if recap.Opt.RefreshPartyFrame then
			recap.Opt.RefreshPartyFrame = nil
		end

		-- fix for 4.00 and later
		if recap.Opt.MaxEventRange then
			recap.Opt.MaxEventRange = nil
		end
		if recap[user].Opt.MaxEventRange then
			recap[user].Opt.MaxEventRange = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.MaxEventRange then
			recap_temp.DefaultOpt.MaxEventRange = nil -- keep as nil
		end
		if recap.Opt.SpamRows then
			recap.Opt.SpamRows = nil
		end
		if recap[user].Opt.SpamRows then
			recap[user].Opt.SpamRows = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.SpamRows then
			recap_temp.DefaultOpt.SpamRows = nil -- keep as nil
		end

		-- fix for 4.07 and later
		if recap.Opt.PartyData then
			recap.Opt.PartyData = nil
		end
		if recap[user].Opt.PartyData then
			recap[user].Opt.PartyData = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.PartyData then
			recap_temp.DefaultOpt.PartyData = nil -- keep as nil
		end
		if recap.Opt.SelfData then
			recap.Opt.HideGroup = true
			recap.Opt.HideOthers = true
			recap.Opt.SelfData = nil
		end
		if recap[user].Opt.SelfData then
			recap[user].Opt.HideGroup = true
			recap[user].Opt.HideOthers = true
			recap[user].Opt.SelfData = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.SelfData then
			recap_temp.DefaultOpt.HideGroup = true
			recap_temp.DefaultOpt.HideOthers = true
			recap_temp.DefaultOpt.SelfData = nil -- keep as nil
		end
		if recap.Opt.HideFoe then
			recap.Opt.HideOthers = true
			recap.Opt.HideFoe = nil
		end
		if recap[user].Opt.HideFoe then
			recap[user].Opt.HideOthers = true
			recap[user].Opt.HideFoe = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.HideFoe then
			recap_temp.DefaultOpt.HideOthers = true
			recap_temp.DefaultOpt.HideFoe = nil -- keep as nil
		end
		if recap.Opt.StoreFriendsPartySelf then
			recap.Opt.StoreOnlyDisplayed = true
			recap.Opt.StoreFriendsPartySelf = nil
		end
		if recap[user].Opt.StoreFriendsPartySelf then
			recap[user].Opt.StoreOnlyDisplayed = true
			recap[user].Opt.StoreFriendsPartySelf = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.StoreFriendsPartySelf then
			recap_temp.DefaultOpt.StoreOnlyDisplayed = true
			recap_temp.DefaultOpt.StoreFriendsPartySelf = nil -- keep as nil
		end
		if recap.Opt.WriteFriends then
			recap.Opt.WriteGroup = true
			recap.Opt.WriteFriends = nil
		end
		if recap[user].Opt.WriteFriends then
			recap[user].Opt.WriteGroup = true
			recap[user].Opt.WriteFriends = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.WriteFriends then
			recap_temp.DefaultOpt.WriteGroup = true
			recap_temp.DefaultOpt.WriteFriends = nil -- keep as nil
		end
		if recap.Opt.SaveFriends then
			recap.Opt.SaveGroup = true
			recap.Opt.SaveFriends = nil
		end
		if recap[user].Opt.SaveFriends then
			recap[user].Opt.SaveGroup = true
			recap[user].Opt.SaveFriends = nil -- keep as nil
		end
		if recap_temp.DefaultOpt.SaveFriends then
			recap_temp.DefaultOpt.SaveGroup = true
			recap_temp.DefaultOpt.SaveFriends = nil -- keep as nil
		end
	end
end

-- saves current data to data set "filen"
function Recap_SaveCombatants(filen, lastfightonly, forcesavenongroup)

	local i, j, k, iCombatant, needEntry
	local sd = {}

	if not filen or string_len(filen)<1 then
		return
	end

	recap_set[filen] = {}
	-- Windows lua code does not support %e
	recap_set[filen].TimeStamp = date("%Y-%m-%d %H:%M %a %d %b")
	if not lastfightonly then
		recap_set[filen].TotalDuration = recap[recap_temp.p].TotalDuration
		recap_set[filen].TotalDurationIn = recap[recap_temp.p].TotalDurationIn
		recap_set[filen].TotalDurationHeal = recap[recap_temp.p].TotalDurationHeal
	else
		recap_set[filen].TotalDuration = recap[recap_temp.p].LastDuration
		recap_set[filen].TotalDurationIn = recap[recap_temp.p].LastDurationIn
		recap_set[filen].TotalDurationHeal = recap[recap_temp.p].LastDurationHeal
	end
	recap_set[filen].MergePets = Recap_PetsMerged() -- not 100% reliable if the user changed the setting after the data was collected
	if recap.Opt.IgnoreGUIDs then -- bandaid, not sure why this wouldn't exist
		recap_set[filen].IgnoreGUIDs = recap.Opt.IgnoreGUIDs.value -- not 100% reliable if the user changed the setting after the data was collected
	else
		recap_set[filen].IgnoreGUIDs = false -- not 100% reliable if the user changed the setting after the data was collected
	end
	recap_set[filen].Combatant = {}

	local AllLastOutgoingDetail = "OutgoingDetail"
	local AllLastIncomingDetail = "IncomingDetail"
	local AllLastOtherDetail = "OtherDetail"
	local AllLastTargetDetail = "TargetDetail"
	local AllLastSourceDetail = "SourceDetail"
	if lastfightonly then
		-- we save from the displayed Last Fight details (not from the details being actively accumulated)
		AllLastOutgoingDetail = "LastOutgoingDetail_"..recap_temp.DisplayLastFight
		AllLastIncomingDetail = "LastIncomingDetail_"..recap_temp.DisplayLastFight
		AllLastOtherDetail = "LastOtherDetail_"..recap_temp.DisplayLastFight
		AllLastTargetDetail = "LastTargetDetail_"..recap_temp.DisplayLastFight
		AllLastSourceDetail = "LastSourceDetail_"..recap_temp.DisplayLastFight
	end

	-- silent repair
	if not recap.Combatant then
		recap.Combatant = {}
	end
	for i in pairs(recap.Combatant) do
		if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) and
		   (forcesavenongroup or (not recap.Opt.SaveGroup.value or recap.Combatant[i].Friend)) and
		   (not lastfightonly or recap.Combatant[i].WasInLast) then

			iCombatant = recap.Combatant[i]

			-- the value of Seen is too large for %u
			local highSeen, lowSeen
			lowSeen = (iCombatant.Seen or 0)
			highSeen = math_floor(lowSeen / 2147483648)
			lowSeen = lowSeen - highSeen * 2147483648

			-- the value of Flags is too large for %u
			local highFlags, lowFlags
			lowFlags = (iCombatant.Flags or 0)
			highFlags = math_floor(lowFlags / 2147483648)
			lowFlags = lowFlags - highFlags * 2147483648

			if not lastfightonly then
				-- we exclude 'combatants' with no real data
				if ((iCombatant.TotalDmgIn == nil) or (iCombatant.TotalDmgIn == 0)) and
				   ((iCombatant.TotalDmgOut == nil) or (iCombatant.TotalDmgOut == 0)) and
				   ((iCombatant.TotalRawHeal == nil) or (iCombatant.TotalRawHeal == 0)) and
				   ((iCombatant.TotalKills == nil) or (iCombatant.TotalKills == 0)) and
				   (iCombatant[AllLastOutgoingDetail] == nil) and
				   (iCombatant[AllLastIncomingDetail] == nil) and
				   (iCombatant[AllLastOtherDetail] == nil) and
				   (iCombatant[AllLastTargetDetail] == nil) and
				   (iCombatant[AllLastSourceDetail] == nil) then
					-- no damage in or out, no healing, no details, don't save
				else
					recap_set[filen].Combatant[i] = string_format("%s %d %d %d %d %d %d %.3f %.3f %.3f %d %d %d %d ~%d %d %d",
													tostring(iCombatant.Friend),
													(iCombatant.TotalDmgIn or 0),
													(iCombatant.TotalDmgOut or 0),
													(iCombatant.TotalMaxHit or 0),
													(iCombatant.TotalRawHeal or 0),
													(iCombatant.TotalOverHeal or 0),
													(iCombatant.TotalKills or 0),
													(iCombatant.TotalTime or 0),
													(iCombatant.TotalTimeIn or 0),
													(iCombatant.TotalTimeHeal or 0),
													highSeen, lowSeen, highFlags, lowFlags,
													Recap_MakeKey(iCombatant.Faction),
													Recap_MakeKey(iCombatant.Class),
													(iCombatant.Level or 0))
				end
			else
				-- added to write Last Fight data only
				-- we exclude 'combatants' with no real data
				if ((iCombatant["LastDmgIn_"..recap_temp.DisplayLastFight] == nil) or (iCombatant["LastDmgIn_"..recap_temp.DisplayLastFight] == 0)) and
				   ((iCombatant["LastDmgOut_"..recap_temp.DisplayLastFight] == nil) or (iCombatant["LastDmgOut_"..recap_temp.DisplayLastFight] == 0)) and
				   ((iCombatant["LastRawHeal_"..recap_temp.DisplayLastFight] == nil) or (iCombatant["LastRawHeal_"..recap_temp.DisplayLastFight] == 0)) and
				   ((iCombatant["LastKills_"..recap_temp.DisplayLastFight] == nil) or (iCombatant["LastKills_"..recap_temp.DisplayLastFight] == 0)) and
				   (iCombatant[AllLastOutgoingDetail] == nil) and
				   (iCombatant[AllLastIncomingDetail] == nil) and
				   (iCombatant[AllLastOtherDetail] == nil) and
				   (iCombatant[AllLastTargetDetail] == nil) and
				   (iCombatant[AllLastSourceDetail] == nil) then
					-- no damage in or out, no healing, no details, don't save
				else
					recap_set[filen].Combatant[i] = string_format("%s %d %d %d %d %d %d %.3f %.3f %.3f %d %d %d %d ~%d %d %d",
													tostring(iCombatant.Friend),
													(iCombatant["LastDmgIn_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastDmgOut_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastMaxHit_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastRawHeal_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastOverHeal_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastKills_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastTime_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastTimeIn_"..recap_temp.DisplayLastFight] or 0),
													(iCombatant["LastTimeHeal_"..recap_temp.DisplayLastFight] or 0),
													highSeen, lowSeen, highFlags, lowFlags,
													Recap_MakeKey(iCombatant.Faction),
													Recap_MakeKey(iCombatant.Class),
													(iCombatant.Level or 0))
				end
			end
			if recap.Opt.LightData and not recap.Opt.LightData.value then

				if iCombatant[AllLastOutgoingDetail] then
					for j in pairs(iCombatant[AllLastOutgoingDetail]) do
						needEntry = true
						for k in pairs(iCombatant[AllLastOutgoingDetail][j]) do
							if k~="Element" then
								if needEntry then
									recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."["..j..">"
									if iCombatant[AllLastOutgoingDetail][j].Element then
										recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..iCombatant[AllLastOutgoingDetail][j].Element..">"
									else
										recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."?"..">"
									end
									needEntry = false
								end
								recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..detkey[k]..iCombatant[AllLastOutgoingDetail][j][k]
							end
						end
						if not needEntry then
							recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."]"
						end
					end
				end

				if iCombatant[AllLastTargetDetail] then
					for j in pairs(iCombatant[AllLastTargetDetail]) do
						needEntry = true
						for k in pairs(iCombatant[AllLastTargetDetail][j]) do
							-- name, amount pairs where name (k) is "Total" or a pet name (which may include colons)
							if needEntry then
								recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."@"..j..">"
								needEntry = false
							end
							recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..k..iCombatant[AllLastTargetDetail][j][k]
						end
						if not needEntry then
							recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."@"
						end
					end
				end

				if iCombatant[AllLastIncomingDetail] then
					for j in pairs(iCombatant[AllLastIncomingDetail]) do
						needEntry = true
						for k in pairs(iCombatant[AllLastIncomingDetail][j]) do
							if k~="Element" then
								if needEntry then
									recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."*"..j..">"
									if iCombatant[AllLastIncomingDetail][j].Element then
										recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..iCombatant[AllLastIncomingDetail][j].Element..">"
									else
										recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."?"..">"
									end
									needEntry = false
								end
								recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..detkey[k]..iCombatant[AllLastIncomingDetail][j][k]
							end
						end
						if not needEntry then
							recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."*"
						end
					end
				end

				if iCombatant[AllLastSourceDetail] then
					for j in pairs(iCombatant[AllLastSourceDetail]) do
						needEntry = true
						for k in pairs(iCombatant[AllLastSourceDetail][j]) do
							-- name, amount pairs where name (k) is "Total" or a pet name (which may include colons)
							if needEntry then
								recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."&"..j..">"
								needEntry = false
							end
							recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..k..iCombatant[AllLastSourceDetail][j][k]
						end
						if not needEntry then
							recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."&"
						end
					end
				end

				if iCombatant[AllLastOtherDetail] then
					for j in pairs(iCombatant[AllLastOtherDetail]) do
						needEntry = true
						for k in pairs(iCombatant[AllLastOtherDetail][j]) do
							if k~="Attribute" then
								if needEntry then
									recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."#"..j..">"
									if iCombatant[AllLastOtherDetail][j].Attribute then
										recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..iCombatant[AllLastOtherDetail][j].Attribute..">"
									else
										recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."?"..">"
									end
									needEntry = false
								end
								recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i]..detkey[k]..iCombatant[AllLastOtherDetail][j][k]
							end
						end
						if not needEntry then
							recap_set[filen].Combatant[i] = recap_set[filen].Combatant[i].."#"
						end
					end
				end

			end
		end
	end

	fight_count = 0
	for i in pairs(recap_set[filen].Combatant) do
		fight_count = fight_count + 1
	end
	recap_set[filen].ListSize = fight_count
end


function Recap_MakeKey(thisIndex)

	local i
	local key = 0

	for i in pairs(recap_temp.Keys) do
		if thisIndex==recap_temp.Keys[i] then
			key = i
		end
	end
	return key
end

function Recap_GetKey(thisIndex)

	local key = nil -- keep as nil

	if not thisIndex or not tonumber(thisIndex) or thisIndex==0 then
		key=nil -- keep as nil
	else
		key=recap_temp.Keys[tonumber(thisIndex)]
	end
	return key
end

function Recap_LoadCombatants(filen)

	local i, j, k, m, iCombatant, needArray
	local sd = {} -- temp holding place for fields that may not exist
	local found, set_friend, set_dmgin, set_dmgout, set_maxhit, set_rawheal, set_overheal, set_heal, set_kills, set_timeOut, set_timeIn, set_timeHeal, set_highseen, set_lowseen, set_highflags, set_lowflags, set_faction, set_class, set_level, set_element, set_attribute, set_rest, set_petname
	local okayToLoad

	if not filen or string_len(filen)<1 or not recap_set[filen] then
		return
	end

	Recap_ResetAllCombatants(false)
	collectgarbage("collect")

	-- we always load into the main details, so no need to switch (for example) between OutgoingDetail and LastOutgoingDetail

	-- update of durations to version 3.59 and later
	if not recap[recap_temp.p].TotalDurationIn then
		recap[recap_temp.p].TotalDurationIn = recap[recap_temp.p].TotalDuration
	end
	if not recap[recap_temp.p].TotalDurationHeal then
		recap[recap_temp.p].TotalDurationHeal = recap[recap_temp.p].TotalDuration
	end
	recap[recap_temp.p].TotalDuration = recap[recap_temp.p].TotalDuration + recap_set[filen].TotalDuration
	if recap_set[filen].TotalDurationIn then
		recap[recap_temp.p].TotalDurationIn = recap[recap_temp.p].TotalDurationIn + recap_set[filen].TotalDurationIn
	else
		recap[recap_temp.p].TotalDurationIn = recap[recap_temp.p].TotalDurationIn + recap_set[filen].TotalDuration
	end
	if recap_set[filen].TotalDurationHeal then
		recap[recap_temp.p].TotalDurationHeal = recap[recap_temp.p].TotalDurationHeal + recap_set[filen].TotalDurationHeal
	else
		recap[recap_temp.p].TotalDurationHeal = recap[recap_temp.p].TotalDurationHeal + recap_set[filen].TotalDuration
	end

	-- silent repair
	if not recap_set[filen].Combatant then
		recap_set[filen].Combatant = {}
	end
	recap_temp.InFriend = {}
	for i in pairs(recap_set[filen].Combatant) do

		if not recap.Combatant[i] then
			Recap_CreateBlankCombatant(i, 0, nil)
		end

		iCombatant = recap.Combatant[i]

		-- format from 4.07 and later
		found,_,set_friend,set_dmgin,set_dmgout,set_maxhit,set_rawheal,set_overheal,set_kills,set_timeOut,set_timeIn,set_timeHeal,set_highseen,set_lowseen,set_highflags,set_lowflags = string_find(recap_set[filen].Combatant[i], "(%w+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+%.?%d+) (%d+%.?%d+) (%d+%.?%d+) (%d+) (%d+) (%d+) (%d+)")
		if found then
			if (tonumber(set_highseen) > 0) or (tonumber(set_lowseen) > 0) then
				iCombatant.Seen = 2147483648 * set_highseen + set_lowseen
			end
			if (tonumber(set_highflags) > 0) or (tonumber(set_lowflags) > 0) then
				iCombatant.Flags = 2147483648 * set_highflags + set_lowflags
			end
			if set_friend == "true" then
				iCombatant.Friend = true
				recap_temp.InFriend[i] = true
			end
			if (tonumber(set_maxhit) > 0) then
				if tonumber(set_maxhit) > (iCombatant.TotalMaxHit or 0) then
					iCombatant.TotalMaxHit = tonumber(set_maxhit)
				end
			end
			if (tonumber(set_dmgin) > 0) then
				iCombatant.TotalDmgIn = (iCombatant.TotalDmgIn or 0) + tonumber(set_dmgin)
			end
			if (tonumber(set_dmgout) > 0) then
				iCombatant.TotalDmgOut = (iCombatant.TotalDmgOut or 0) + tonumber(set_dmgout)
			end
			if (tonumber(set_kills) > 0) then
				iCombatant.TotalKills = (iCombatant.TotalKills or 0) + tonumber(set_kills)
			end
			if (tonumber(set_rawheal) > 0) then
				iCombatant.TotalRawHeal = (iCombatant.TotalRawHeal or 0) + tonumber(set_rawheal)
				m = math_min(((iCombatant.TotalOverHeal or 0) + tonumber(set_overheal)), (iCombatant.TotalRawHeal or 0))
				if m>0 then
					iCombatant.TotalOverHeal = m
				end
			end
			if (tonumber(set_timeOut) > 0) then
				iCombatant.TotalTime = (iCombatant.TotalTime or 0) + tonumber(set_timeOut)
			end
			if  iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
				iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
			end
			if (tonumber(set_timeIn) > 0) then
				iCombatant.TotalTimeIn = (iCombatant.TotalTimeIn or 0) + tonumber(set_timeIn)
			end
			if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
				iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
			end
			if (tonumber(set_timeHeal) > 0) then
				iCombatant.TotalTimeHeal = (iCombatant.TotalTimeHeal or 0) + tonumber(set_timeHeal)
			end
			if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and (((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)) > 0) then
				iCombatant.TotalHPS = Recap_Div1(((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)), iCombatant.TotalTimeHeal)
			end
		else
			-- format from 4.00 and later
			found,_,set_friend,set_dmgin,set_dmgout,set_maxhit,set_rawheal,set_overheal,set_kills,set_timeOut,set_timeIn,set_timeHeal,set_highseen,set_lowseen = string_find(recap_set[filen].Combatant[i], "(%w+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+%.?%d+) (%d+%.?%d+) (%d+%.?%d+) (%d+) (%d+)")
			if found then
				if (tonumber(set_highseen) > 0) or (tonumber(set_lowseen) > 0) then
					iCombatant.Seen = 2147483648 * set_highseen + set_lowseen
				end
				if set_friend == "true" then
					iCombatant.Friend = true
					recap_temp.InFriend[i] = true
				end
				if (tonumber(set_maxhit) > 0) then
					if tonumber(set_maxhit) > (iCombatant.TotalMaxHit or 0) then
						iCombatant.TotalMaxHit = tonumber(set_maxhit)
					end
				end
				if (tonumber(set_dmgin) > 0) then
					iCombatant.TotalDmgIn = (iCombatant.TotalDmgIn or 0) + tonumber(set_dmgin)
				end
				if (tonumber(set_dmgout) > 0) then
					iCombatant.TotalDmgOut = (iCombatant.TotalDmgOut or 0) + tonumber(set_dmgout)
				end
				if (tonumber(set_kills) > 0) then
					iCombatant.TotalKills = (iCombatant.TotalKills or 0) + tonumber(set_kills)
				end
				if (tonumber(set_rawheal) > 0) then
					iCombatant.TotalRawHeal = (iCombatant.TotalRawHeal or 0) + tonumber(set_rawheal)
					m = math_min(((iCombatant.TotalOverHeal or 0) + tonumber(set_overheal)), (iCombatant.TotalRawHeal or 0))
					if m>0 then
						iCombatant.TotalOverHeal = m
					end
				end
				if (tonumber(set_timeOut) > 0) then
					iCombatant.TotalTime = (iCombatant.TotalTime or 0) + tonumber(set_timeOut)
				end
				if  iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
					iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
				end
				if (tonumber(set_timeIn) > 0) then
					iCombatant.TotalTimeIn = (iCombatant.TotalTimeIn or 0) + tonumber(set_timeIn)
				end
				if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
					iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
				end
				if (tonumber(set_timeHeal) > 0) then
					iCombatant.TotalTimeHeal = (iCombatant.TotalTimeHeal or 0) + tonumber(set_timeHeal)
				end
				if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and (((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)) > 0) then
					iCombatant.TotalHPS = Recap_Div1(((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)), iCombatant.TotalTimeHeal)
				end
			else
				-- format from 3.61 and later
				found,_,set_friend,set_dmgin,set_dmgout,set_maxhit,set_rawheal,set_overheal,set_kills,set_timeOut,set_timeIn,set_timeHeal = string_find(recap_set[filen].Combatant[i], "(%w+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+%.?%d+) (%d+%.?%d+) (%d+%.?%d+)")
				if found then
					if set_friend == "true" then
						iCombatant.Friend = true
						recap_temp.InFriend[i] = true
					end
					if (tonumber(set_maxhit) > 0) then
						if tonumber(set_maxhit) > (iCombatant.TotalMaxHit or 0) then
							iCombatant.TotalMaxHit = tonumber(set_maxhit)
						end
					end
					if (tonumber(set_dmgin) > 0) then
						iCombatant.TotalDmgIn = (iCombatant.TotalDmgIn or 0) + tonumber(set_dmgin)
					end
					if (tonumber(set_dmgout) > 0) then
						iCombatant.TotalDmgOut = (iCombatant.TotalDmgOut or 0) + tonumber(set_dmgout)
					end
					if (tonumber(set_kills) > 0) then
						iCombatant.TotalKills = (iCombatant.TotalKills or 0) + tonumber(set_kills)
					end
					if (tonumber(set_rawheal) > 0) then
						iCombatant.TotalRawHeal = (iCombatant.TotalRawHeal or 0) + tonumber(set_rawheal)
						m = math_min(((iCombatant.TotalOverHeal or 0) + tonumber(set_overheal)), (iCombatant.TotalRawHeal or 0))
						if m>0 then
							iCombatant.TotalOverHeal = m
						end
					end
					if (tonumber(set_timeOut) > 0) then
						iCombatant.TotalTime = (iCombatant.TotalTime or 0) + tonumber(set_timeOut)
					end
					if iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
						iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
					end
					if (tonumber(set_timeIn) > 0) then
						iCombatant.TotalTimeIn = (iCombatant.TotalTimeIn or 0) + tonumber(set_timeIn)
					end
					if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
						iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
					end
					if (tonumber(set_timeHeal) > 0) then
						iCombatant.TotalTimeHeal = (iCombatant.TotalTimeHeal or 0) + tonumber(set_timeHeal)
					end
					if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and (((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)) > 0) then
						iCombatant.TotalHPS = Recap_Div1(((iCombatant.TotalRawHeal or 0) - (iCombatant.TotalOverHeal or 0)), iCombatant.TotalTimeHeal)
					end
				else
					-- format from 3.54 and later
					found,_,set_friend,set_dmgin,set_dmgout,set_maxhit,set_heal,set_kills,set_timeOut,set_timeIn,set_timeHeal = string_find(recap_set[filen].Combatant[i], "(%w+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+%.?%d+) (%d+%.?%d+) (%d+%.?%d+)")
					if found then
						if set_friend == "true" then
							iCombatant.Friend = true
							recap_temp.InFriend[i] = true
						end
						if (tonumber(set_maxhit) > 0) then
							if tonumber(set_maxhit) > (iCombatant.TotalMaxHit or 0) then
								iCombatant.TotalMaxHit = tonumber(set_maxhit)
							end
						end
						if (tonumber(set_dmgin) > 0) then
							iCombatant.TotalDmgIn = (iCombatant.TotalDmgIn or 0) + tonumber(set_dmgin)
						end
						if (tonumber(set_dmgout) > 0) then
							iCombatant.TotalDmgOut = (iCombatant.TotalDmgOut or 0) + tonumber(set_dmgout)
						end
						if (tonumber(set_kills) > 0) then
							iCombatant.TotalKills = (iCombatant.TotalKills or 0) + tonumber(set_kills)
						end
						if (tonumber(set_rawheal) > 0) then
							-- details of overhealing lost on update to 3.61
							iCombatant.TotalRawHeal = (iCombatant.TotalRawHeal or 0) + tonumber(set_heal)
						end
						if (tonumber(set_timeOut) > 0) then
							iCombatant.TotalTime = (iCombatant.TotalTime or 0) + tonumber(set_timeOut)
						end
						if iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
							iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
						end
						if (tonumber(set_timeIn) > 0) then
							iCombatant.TotalTimeIn = (iCombatant.TotalTimeIn or 0) + tonumber(set_timeIn)
						end
						if iCombatant.TotalTimeIn and (iCombatant.TotalTimeIn > recap_temp.MinTime) and ((iCombatant.TotalDmgIn or 0) > 0) then
							iCombatant.TotalDPSIn = Recap_Div1((iCombatant.TotalDmgIn or 0), iCombatant.TotalTimeIn)
						end
						if (tonumber(set_timeHeal) > 0) then
							iCombatant.TotalTimeHeal = (iCombatant.TotalTimeHeal or 0) + tonumber(set_timeHeal)
						end
						if iCombatant.TotalTimeHeal and (iCombatant.TotalTimeHeal > recap_temp.MinTime) and ((iCombatant.TotalRawHeal or 0) > 0) then
							iCombatant.TotalHPS = Recap_Div1((iCombatant.TotalRawHeal or 0), iCombatant.TotalTimeHeal)
						end
					else
						-- load format from 3.53 and earlier
						found,_,set_friend,set_dmgin,set_dmgout,set_maxhit,set_heal,set_kills,set_timeOut = string_find(recap_set[filen].Combatant[i], "(%w+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+%.?%d+)")
						if found then
							if set_friend == "true" then
								iCombatant.Friend = true
								recap_temp.InFriend[i] = true
							end
							if (tonumber(set_maxhit) > 0) then
								if tonumber(set_maxhit) > (iCombatant.TotalMaxHit or 0) then
									iCombatant.TotalMaxHit = tonumber(set_maxhit)
								end
							end
							if (tonumber(set_dmgin) > 0) then
								iCombatant.TotalDmgIn = (iCombatant.TotalDmgIn or 0) + tonumber(set_dmgin)
							end
							if (tonumber(set_dmgout) > 0) then
								iCombatant.TotalDmgOut = (iCombatant.TotalDmgOut or 0) + tonumber(set_dmgout)
							end
							if (tonumber(set_kills) > 0) then
								iCombatant.TotalKills = (iCombatant.TotalKills or 0) + tonumber(set_kills)
							end
							if (tonumber(set_rawheal) > 0) then
								-- details of overhealing lost on update to 3.61
								iCombatant.TotalRawHeal = (iCombatant.TotalRawHeal or 0) + tonumber(set_heal)
							end
							if (tonumber(set_timeOut) > 0) then
								iCombatant.TotalTime = (iCombatant.TotalTime or 0) + tonumber(set_timeOut)
							end
							if iCombatant.TotalTime and (iCombatant.TotalTime > recap_temp.MinTime) and ((iCombatant.TotalDmgOut or 0) > 0) then
								iCombatant.TotalDPS = Recap_Div1((iCombatant.TotalDmgOut or 0), iCombatant.TotalTime)
							end
						end
					end
				end
			end
		end
		found,_,set_faction,set_class,set_level = string_find(recap_set[filen].Combatant[i], "~(%d+) (%d+) (%-?%d+)")
		if not found then
			found,_,set_faction,set_class = string_find(recap_set[filen].Combatant[i], "~(%d+) (%d+)")
		end
		if found then
			if set_faction and (tonumber(set_faction) > 0) then
				iCombatant.Faction = Recap_GetKey(set_faction)
			end
			if set_class and (tonumber(set_class) > 0) then
				iCombatant.Class = Recap_GetKey(set_class)
			end
			if set_level and (tonumber(set_level) ~= 0) then
				iCombatant.Level = iCombatant.Level or set_level
			end
		end

		if not recap.Opt.LightData.value then

			-- process incoming: ^a000b000c000^
			for k in pairs(sd) do sd[k] = nil end -- wipe temp buffer -- keep as nil
			found,_,j = string_find(recap_set[filen].Combatant[i], "%^(.-)%^")
			if found then -- new saved method
				for k in pairs(inckey) do
					_,_,sd[k] = string_find(j, inckey[k].."(%d+)")
				end
			elseif string_find(recap_set[filen].Combatant[i], "%^%d+ %d+ %d+ %d+ %d+ %d+ %d+ %d+ %d+ %d+ %d+ %d+ %d+") then
				-- old saved method
				found,_,sd.MeleeDamage,sd.MeleeMax,sd.MeleeHits,sd.MeleeCrits,sd.MeleeMissed,sd.MeleeDodged,sd.MeleeParried,sd.MeleeBlocked,
					sd.NonMeleeDamage,sd.NonMeleeMax,sd.NonMeleeHits,sd.NonMeleeCrits,sd.NonMeleeMissed =
						string_find(recap_set[filen].Combatant[i], "%^(%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+)")
			end
			if found then
				if sd.MeleeDamage and tonumber(sd.MeleeDamage)>0 then
					if not iCombatant.IncomingDetail then
						iCombatant.IncomingDetail = {}
					end
					if not iCombatant.IncomingDetail["1Melee"] then
						iCombatant.IncomingDetail["1Melee"] = {}
					end
					iCombatant.IncomingDetail["1Melee"].Element = "Melee"
					iCombatant.IncomingDetail["1Melee"].HitsDmg = sd.MeleeDamage
					if sd.MeleeHits then
						iCombatant.IncomingDetail["1Melee"].Hits = sd.MeleeHits
						iCombatant.IncomingDetail["1Melee"].CritsEvents = sd.MeleeHits
					end
					if sd.MeleeCrits then
						iCombatant.IncomingDetail["1Melee"].Crits = sd.MeleeCrits
						iCombatant.IncomingDetail["1Melee"].CritsEvents = (iCombatant.IncomingDetail["1Melee"].CritsEvents or 0) + sd.MeleeCrits
					end
					if sd.MeleeMax then
						iCombatant.IncomingDetail["1Melee"].HitsMax = sd.MeleeMax
					end
					if sd.MeleeMissed then
						iCombatant.IncomingDetail["1Melee"].Missed = sd.MeleeMissed
					end
					if sd.MeleeBlocked then
						iCombatant.IncomingDetail["1Melee"].Blocked = sd.MeleeBlocked
					end
					if sd.MeleeParried then
						iCombatant.IncomingDetail["1Melee"].Parried = sd.MeleeParried
					end
					if sd.MeleeDodged then
						iCombatant.IncomingDetail["1Melee"].Dodged = sd.MeleeDodged
					end
				end
				if sd.NonMeleeDamage and tonumber(sd.NonMeleeDamage)>0 then
					if not iCombatant.IncomingDetail then
						iCombatant.IncomingDetail = {}
					end
					if not iCombatant.IncomingDetail["1Non-Melee"] then
						iCombatant.IncomingDetail["1Non-Melee"] = {}
					end
					iCombatant.IncomingDetail["1Non-Melee"].Element = "Non-Melee"
					iCombatant.IncomingDetail["1Non-Melee"].HitsDmg = sd.NonMeleeDamage
					if sd.NonMeleeHits then
						iCombatant.IncomingDetail["1Non-Melee"].Hits = sd.NonMeleeHits
						iCombatant.IncomingDetail["1Non-Melee"].CritsEvents = sd.NonMeleeHits
					end
					if sd.NonMeleeCrits then
						iCombatant.IncomingDetail["1Non-Melee"].Crits = sd.NonMeleeCrits
						iCombatant.IncomingDetail["1Non-Melee"].CritsEvents = (iCombatant.IncomingDetail["1Non-Melee"].CritsEvents or 0) + sd.NonMeleeCrits
					end
					if sd.NonMeleeMax then
						iCombatant.IncomingDetail["1Non-Melee"].HitsMax = sd.NonMeleeMax
					end
					if sd.NonMeleeTicks then
						iCombatant.IncomingDetail["1Non-Melee"].Ticks = sd.NonMeleeTicks
					end
					if sd.NonMeleeMissed then
						iCombatant.IncomingDetail["1Non-Melee"].Missed = sd.NonMeleeMissed
					end
				end
			end

			-- test whether old format Detail or new format Outgoing Detail (from 3.59)
			local newFormat = false
			for j in string_gmatch(recap_set[filen].Combatant[i],"%[.-%]") do
				found,_,set_name,set_element = string_find(j, "%[(.-)%>(.-)%>")
				if found then
					newFormat = true
				end
				break
			end
			if newFormat then
				-- process Outgoing Detail: [name>element>a000b000c000]
				for j in string_gmatch(recap_set[filen].Combatant[i],"%[.-%]") do
					for k in pairs(sd) do sd[k] = nil end -- wipe temp buffer -- keep as nil
					found,_,set_name,set_element,set_rest = string_find(j, "%[(.-)%>(.-)%>(.*)")
					if found then -- new saved method
						-- fix to allow multi-letter keys
						-- prefix the string j with a digit and use digits to delimit the keys
						set_rest = "1"..set_rest
						for k in pairs(detkey) do
							_,_,sd[k] = string_find(set_rest, "%d"..detkey[k].."(%d+)")
						end
						-- skip effects that have zero damage
						local total
						total = (sd.GlancesDmg or 0) + (sd.HitsDmg or 0) + (sd.CritsDmg or 0) + (sd.CrushDmg or 0) + (sd.TicksDmg or 0)
						if total == 0 then
							found = false
						end
					end
					if found then
						needArray = true
						for k in pairs(sd) do
							if sd[k] and tonumber(sd[k])>0 then
								if needArray then
									if not iCombatant.OutgoingDetail then
										iCombatant.OutgoingDetail = {}
									end
									iCombatant.OutgoingDetail[set_name] = {}
									iCombatant.OutgoingDetail[set_name].Element = set_element
									needArray = false
								end
								iCombatant.OutgoingDetail[set_name][k] = tonumber(sd[k])
							end
						end
					end
				end
			else
				-- process old format Outgoing Detail: [name>a000b000c000]
				for j in string_gmatch(recap_set[filen].Combatant[i],"%[.-%]") do
					for k in pairs(sd) do sd[k] = nil end -- wipe temp buffer
					found,_,set_name,set_rest = string_find(j, "%[(.-)%>(.*)")
					if found then -- new saved method
						if not iCombatant.OutgoingDetail then
							iCombatant.OutgoingDetail = {}
						end
						iCombatant.OutgoingDetail[set_name] = {}
						-- fix to allow multi-letter keys
						-- prefix the string j with a digit and use digits to delimit the keys
						set_rest = "1"..set_rest
						for k in pairs(detkey) do
							_,_,sd[k] = string_find(set_rest, "%d"..detkey[k].."(%d+)")
						end
					else
						found,_,set_name = string_find(j, "%[(.-) %d")
						if found then -- old saved method
							if not iCombatant.OutgoingDetail then
								iCombatant.OutgoingDetail = {}
							end
							iCombatant.OutgoingDetail[set_name] = {}
							_,_,sd.HitsDmg,sd.Hits,sd.HitsMax,sd.CritsDmg,sd.Crits,sd.CritsMax,sd.CritsEvents,sd.Missed =
								string_find(j, "%[.+ (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+) (%d+)") -- old method
						end
					end
					if found then
						if not iCombatant.OutgoingDetail then
							iCombatant.OutgoingDetail = {}
						end
						iCombatant.OutgoingDetail[set_name].Element = "?"
						for k in pairs(sd) do
							if sd[k] and tonumber(sd[k])>0 then
								iCombatant.OutgoingDetail[set_name][k] = tonumber(sd[k])
							end
						end
					end
				end
			end

			-- legacy (from 4.00 .. 4.05) Target Detail: @name>petName>v000@
			for j in string_gmatch(recap_set[filen].Combatant[i],"%@.-%@") do
				found,_,set_name,set_petname,set_rest = string_find(j, "%@(.-)%>(.-)%>(.*)")
				if found then
					_,_,k,m = string_find(set_rest, "(.)(%d+)")
					if not iCombatant.TargetDetail then
						iCombatant.TargetDetail = {}
					end
					iCombatant.TargetDetail[set_name] = {}
					if (set_petname == "?") then
						-- assert that k will equal 'v'
						iCombatant.TargetDetail[set_name]["Total"] = tonumber(m)
					else
						iCombatant.TargetDetail[set_name][set_petname] = tonumber(m)
					end
				else
					-- Target Detail: @name>tgt000tgt000tgt000@
					--   tgt can be 'Total', owner:pet, or owner:owner:pet (without GUIDs)
					--   legacy abbreviation of 'v' needs to be replaced by 'Total'
					found,_,set_name,set_rest = string_find(j, "%@(.-)%>(.*)")
					if found then
						needArray = true
						for k,m in string_gmatch(set_rest, "([^%d]+)(%d+)") do
							-- relies on tgt containing no digits
							if needArray then
								if not iCombatant.TargetDetail then
									iCombatant.TargetDetail = {}
								end
								iCombatant.TargetDetail[set_name] = {}
								needArray = false
							end
							if (k == "v") then
								iCombatant.TargetDetail[set_name]["Total"] = tonumber(m)
							else
								iCombatant.TargetDetail[set_name][k] = tonumber(m)
							end
						end
					end
				end
			end

			-- process Incoming Detail: *name>element>a000b000c000*
			for j in string_gmatch(recap_set[filen].Combatant[i],"%*.-%*") do
				for k in pairs(sd) do sd[k] = nil end -- wipe temp buffer -- keep as nil
				found,_,set_name,set_element,set_rest = string_find(j, "%*(.-)%>(.-)%>(.*)")
				if found then -- new saved method
					-- fix to allow multi-letter keys
					-- prefix the string j with a digit and use digits to delimit the keys
					set_rest = "1"..set_rest
					for k in pairs(detkey) do
						_,_,sd[k] = string_find(set_rest, "%d"..detkey[k].."(%d+)")
					end
					-- skip effects that have zero damage
					local total
					total = (sd.GlancesDmg or 0) + (sd.HitsDmg or 0) + (sd.CritsDmg or 0) + (sd.CrushDmg or 0) + (sd.TicksDmg or 0)
					if total == 0 then
						found = false
					end
				end
				if found then
					needArray = true
					for k in pairs(sd) do
						if sd[k] and tonumber(sd[k])>0 then
							if needArray then
								if not iCombatant.IncomingDetail then
									iCombatant.IncomingDetail = {}
								end
								iCombatant.IncomingDetail[set_name] = {}
								iCombatant.IncomingDetail[set_name].Element = set_element
								needArray = false
							end
							iCombatant.IncomingDetail[set_name][k] = tonumber(sd[k])
						end
					end
				end
			end

			-- legacy (from 4.00 .. 4.05) Source Detail: &name>petName>v000&
			for j in string_gmatch(recap_set[filen].Combatant[i],"%&.-%&") do
				found,_,set_name,set_petname,set_rest = string_find(j, "%&(.-)%>(.-)%>(.*)")
				if found then
					_,_,k,m = string_find(set_rest, "(.)(%d+)")
					if not iCombatant.SourceDetail then
						iCombatant.SourceDetail = {}
					end
					iCombatant.SourceDetail[set_name] = {}
					if (set_petname == "?") then
						-- assert that k will equal 'v'
						iCombatant.SourceDetail[set_name]["Total"] = tonumber(m)
					else
						iCombatant.SourceDetail[set_name][set_petname] = tonumber(m)
					end
				else
					-- Source Detail: &name>src000src000src000&
					--   src can be 'Total', owner:pet, or owner:owner:pet (without GUIDs)
					--   legacy abbreviation of 'v' needs to be replaced by 'Total'
					found,_,set_name,set_rest = string_find(j, "%&(.-)%>(.*)")
					if found then
						needArray = true
						for k,m in string_gmatch(set_rest, "([^%d]+)(%d+)") do
							-- relies on src containing no digits
							if needArray then
								if not iCombatant.SourceDetail then
									iCombatant.SourceDetail = {}
								end
								iCombatant.SourceDetail[set_name] = {}
								needArray = false
							end
							if (k == "v") then
								iCombatant.SourceDetail[set_name]["Total"] = tonumber(m)
							else
								iCombatant.SourceDetail[set_name][k] = tonumber(m)
							end
						end
					end
				end
			end

			-- process Other Detail: #name>attribute>a000b000c000#
			for j in string_gmatch(recap_set[filen].Combatant[i],"%#.-%#") do
				for k in pairs(sd) do sd[k] = nil end -- wipe temp buffer -- keep as nil
				found,_,set_name,set_attribute,set_rest = string_find(j, "%#(.-)%>(.-)%>(.*)")
				if found then -- new saved method
					-- fix to allow multi-letter keys
					-- prefix the string j with a digit and use digits to delimit the keys
					set_rest = "1"..set_rest
					for k in pairs(detkey) do
						_,_,sd[k] = string_find(set_rest, "%d"..detkey[k].."(%d+)")
					end
					needArray = true
					for k in pairs(sd) do
						if sd[k] and tonumber(sd[k])>0 then
							if needArray then
								if not iCombatant.OtherDetail then
									iCombatant.OtherDetail = {}
								end
								iCombatant.OtherDetail[set_name] = {}
								if set_attribute ~= "?" then
									iCombatant.OtherDetail[set_name].Attribute = set_attribute
								end
								needArray = false
							end
							iCombatant.OtherDetail[set_name][k] = tonumber(sd[k])
						end
					end
				end
			end
		end
	end

	-- special case migration of Detail to OutgoingDetail, and Incoming to IncomingDetail for 3.59 and later, prior to saving
	Recap_MigrateDetails()

	-- special case correction to add OutgoingDetail.Ticks, OutgoingDetail.Ticks, and Self.Ticks
	Recap_CorrectForTicks("OutgoingDetail", "IncomingDetail") -- no need to do this for Last
end

function Recap_ResetAllCombatants(overrideLocks)

	local current_state = recap.Opt.State.value
	RecapUpdateFrame:Hide() -- disable OnUpdate
	recap.Opt.State.value = "Stopped"
	recap.Opt.SkipNextFight.value = false

	-- scan combatants and save a copy of any combatant marked as Locked (unless in synchronization)
	local i
	local local_save = {}
	-- silent repair
	if not recap.Combatant then
		recap.Combatant = {}
	end
	if not overrideLocks then
		for i in pairs(recap.Combatant) do
			if recap.Combatant[i].Locked then
				local_save[i] = recap.Combatant[i]
			end
		end
	end
	-- clear all combatants
	recap.Combatant = {}
	-- restore the saved combatants (unless in synchronization)
	if not overrideLocks then
		for i in pairs(local_save) do
			recap.Combatant[i] = local_save[i]
		end
	end
	recap_temp.Last = {}
	for i in pairs(recap.Combatant) do
		Recap_InitializeLastFight(i)
	end
	-- Note that LastDuration (etc.) refers only to a displayed Last Fight
	recap[recap_temp.p].LastDuration = 0
	recap[recap_temp.p].LastDurationIn = 0
	recap[recap_temp.p].LastDurationHeal = 0
	recap[recap_temp.p].TotalDuration = 0
	recap[recap_temp.p].TotalDurationIn = 0
	recap[recap_temp.p].TotalDurationHeal = 0
	recap[recap_temp.p].LastFightStart = 0
	recap[recap_temp.p].LastFightEnd = 0
	recap_temp.FightStart = 0
	recap_temp.FightEnd = 0
	recap_temp.FightStartIn = 0
	recap_temp.FightEndIn = 0
	recap_temp.FightStartHeal = 0
	recap_temp.FightEndHeal = 0
	recap_temp.ListSize = 0
	recap_temp.List = {}

	-- clear variables for live DPS and HPS
	recap_temp.PlayerDPSLast = 0
	recap_temp.PlayerDPSAll = 0
	recap_temp.PlayerHPSLast = 0
	recap_temp.PlayerHPSAll = 0
	recap_temp.TotalGroupDmgIn = 0
	recap_temp.TotalGroupDmgOut = 0
	recap_temp.TotalGroupHeal = 0
	recap_temp.GroupDPSOutLast = 0
	recap_temp.GroupDPSInLast = 0
	recap_temp.GroupHPSLast = 0
	recap_temp.GroupDPSOutAll = 0
	recap_temp.GroupDPSInAll = 0
	recap_temp.GroupHPSAll = 0

	-- if Recent Data Mode is active, clear the buffers
	if recap.Opt.RecentData.value then
		Recap_CreateRecent()
	else
		recap_temp.Recent = {}
	end

	-- make sure end of fight delay timer is off
	recap_temp.EndFightDelayTimer = -1

	-- need to set idle timer if we were in combat
	if current_state == "Active" then
		-- start idle timers
		if not recap.Opt.LimitFights.value and recap.Opt.IdleFight.value then
			recap_temp.IdleTimer = 0
		end
		if not recap.Opt.LimitFights.value then
			recap_temp.HiddenIdleTimer = 0
		end
	end

	recap.Opt.State.value = current_state
	RecapUpdateFrame:Show() -- enable OnUpdate
end

function Recap_LoadSelf()

	recap.Self = recap.Self or {}
	recap.Self[recap_temp.s] = recap.Self[recap_temp.s] or {}
	-- not at the moment being done for the player's pets

	recap_temp.SelfListSize = 1
	recap_temp.SelfList = {}
end

local oldkeys = { ["Krieger"]="WARRIOR", ["Magier"]="MAGE", ["Schurke"]="ROGUE", ["Druide"]="DRUID",
				  ["J\195\164ger"]="HUNTER", ["Schamane"]="SHAMAN", ["Priester"]="PRIEST", ["Hexenmeister"]="WARLOCK",
				  ["Kriegerin"]="WARRIOR", ["Magierin"]="MAGE", ["Schurkin"]="ROGUE", ["Druidin"]="DRUID",
				  ["J\195\164gerin"]="HUNTER", ["Schamanin"]="SHAMAN", ["Priesterin"]="PRIEST", ["Hexenmeisterin"]="WARLOCK",
				  ["Guerrier"]="WARRIOR", ["Voleur"]="ROGUE", ["Chasseur"]="HUNTER", ["Chaman"]="SHAMAN",
				  ["Prêtre"]="PRIEST", ["Démoniste"]="WARLOCK", ["Familier"]="Pet",
				  ["Guerri\195\168re"]="WARRIOR", ["Voleuse"]="ROGUE", ["Chasseuse"]="HUNTER", ["Chamane"]="SHAMAN",
				  ["Prêtresse"]="PRIEST" }


function Recap_InitializeData()

	local i, j, k, l, m, iCombatant, user, combatant, dataset
	local temp_user = recap_temp.p

	if not recap.DataVersion then
		-- wipe all pre-2.6 data
		recap = {}
		recap.DataVersion = 3.2
	elseif recap.DataVersion < 3.0 then
		-- convert 2.6 to 3.0, make classes uppercase
		if not recap.Combatant then
			recap.Combatant = {}
		end
		for i in pairs(recap.Combatant) do
			iCombatant = recap.Combatant[i]
			if iCombatant.Faction and (iCombatant.Faction == "Allianz") then
				iCombatant.Faction = "Alliance"
			end
			if iCombatant.Class then
				-- convert classes to english, which all locales will use for icon reference
				if oldkeys[iCombatant.Class] then
					iCombatant.Class = oldkeys[iCombatant.Class]
				end
				iCombatant.Class = string_upper(iCombatant.Class)
			end
		end
		recap.DataVersion = 3.0
	end

	-- If no Opt, create a default one
	if not recap.Opt then
		Recap_LoadDefaultOpt()
	end

	-- if this user hasn't been made before
	if not recap[recap_temp.p] then
		recap[recap_temp.p] = {}
		recap[recap_temp.p].LastDuration = 0
		recap[recap_temp.p].LastDurationIn = 0
		recap[recap_temp.p].LastDurationHeal = 0
		recap[recap_temp.p].TotalDuration = 0
		recap[recap_temp.p].TotalDurationIn = 0
		recap[recap_temp.p].TotalDurationHeal = 0
		recap[recap_temp.p].LastFightStart = 0
		recap[recap_temp.p].LastFightEnd = 0
	end

	-- if this is a different or first-time user, or if this is first run of a new version,
	--   save the current combatants and options and then load them based on default
	-- any changes in combatant or option format must come with a version change
	if (recap.User ~= recap_temp.p) or (not recap.DataVersion) or (recap.DataVersion and recap.DataVersion<Recap_Version) then

		-- special case migration with 4.00 of name format from NameServer to Name_Server
		if recap.DataVersion and recap.DataVersion < 4.0 then
			local somethingChanged = true
			local safetyCount = 0
			while somethingChanged do
				-- need to loop indefinitely since a single pass through recap or recap_set does not catch everything (due to the mutable order of tables)
				somethingChanged = false
				safetyCount = safetyCount + 1
				if recap then
					for user in pairs(recap) do
						local found, name, server, item, newNameServer
						found, _, name, server = string_find(user, "^(%u%l+)(%u[^_]+)$")
						if found and name and server and (user ~= "DataVersion") and (type(recap[user]) == "table") then
							-- old style NameServer, change
							newNameServer = name.."_"..server
							recap[newNameServer] = {}
							for item in pairs(recap[user]) do
								recap[newNameServer][item] = recap[user][item]
							end
							recap[user] = nil
							somethingChanged = true
						end
					end
				end
				if recap and recap.Self then
					for combatant in pairs(recap.Self) do
						local found, owner, pet, newCombatant
						owner = Recap_ExtractOwner(combatant)
						if owner then
							-- "owner:pet"
							found, _, name, server = string_find(owner, "^(%u%l+)(%u[^_]+)$")
							if found and name and server and (type(recap.Self[combatant]) == "table") then
								-- old style NameServer, change
								pet = Recap_ExtractCombatant(combatant)
								-- we don't change the pet
								newCombatant = name.."_"..server..":"..pet
								recap.Self[newCombatant] = {}
								for item in pairs(recap.Self[combatant]) do
									recap.Self[newCombatant][item] = recap.Self[combatant][item]
								end
								recap.Self[combatant] = nil
								somethingChanged = true
							end
						else
							-- "combatant"
							found, _, name, server = string_find(combatant, "^(%u%l+)(%u[^_]+)$")
							if found and name and server and (type(recap.Self[combatant]) == "table") then
								-- old style NameServer, change
								newCombatant = name.."_"..server
								recap.Self[newCombatant] = {}
								for item in pairs(recap.Self[combatant]) do
									recap.Self[newCombatant][item] = recap.Self[combatant][item]
								end
								recap.Self[combatant] = nil
								somethingChanged = true
							end
						end
					end
				end
				if recap_set then
					for dataset in pairs(recap_set) do
						local found, userData, name, server, item, newUserDataNameServer
						found, _, userData, name, server = string_find(dataset, "^(UserData:)(%u%l+)(%u[^_]+)$")
						if found and userData and name and server and (type(recap_set[dataset]) == "table") then
							-- old style NameServer, change
							newUserDataNameServer = "UserData:"..name.."_"..server
							recap_set[newUserDataNameServer] = {}
							for item in pairs(recap_set[dataset]) do
								recap_set[newUserDataNameServer][item] = recap_set[dataset][item]
							end
							recap_set[dataset] = nil
							somethingChanged = true
						end
					end
				end
				if safetyCount >= 20 then
					-- hmm, 20 loops, time to bail (paranoia check)
					break
				end
			end
		end

		-- if we have data from previous user, save them
		if recap.User then

			-- special case migration of Detail to OutgoingDetail, and Incoming to IncomingDetail for 3.59 and later, prior to saving
			Recap_MigrateDetails()

			recap_temp.p = recap.User
			Recap_SaveCombatants("UserData:"..recap.User, false, true)
			Recap_SaveOpt(recap.User)
			recap_temp.p = temp_user
		end
		recap.User = recap_temp.p

		-- wipe slate, load defaults and overlay saved data on top
		Recap_LoadDefaultOpt()
		Recap_ResetAllCombatants(false)
		collectgarbage("collect")

		-- convert data sets to new format (see readme)
		if recap.DataVersion < 3.2 then
			for i in pairs(recap_set) do
				if recap_set[i].Combatant then
					j = recap_set[i].TimeStamp
					Recap_LoadCombatants(i)
					for k in pairs(recap_set[i].Combatant) do
						l = (l or 0) + string_len(recap_set[i].Combatant[k])
					end
					Recap_SaveCombatants(i, false, true)
					for k in pairs(recap_set[i].Combatant) do
						m = (m or 0) + string_len(recap_set[i].Combatant[k])
					end
					recap_set[i].TimeStamp = j
				end
			end
			recap.DataVersion = 3.2
		end

		Recap_LoadCombatants("UserData:"..recap_temp.p)
		Recap_LoadOpt(recap_temp.p)

		recap.DataVersion = Recap_Version
	end

	-- silent repair
	if not recap.Combatant then
		recap.Combatant = {}
	end
	for i in pairs(recap.Combatant) do
		Recap_InitializeLastFight(i)

		-- special case correction for some damage:healing DoT:HoT spells
		iCombatant = recap.Combatant[i]
		Recap_CorrectDualDoTHoT(iCombatant, "OutgoingDetail", "IncomingDetail") -- no need to do for Last, which should be empty
	end

	-- for Recap 4.00 and later, remove obsolete items
	for i in pairs(recap.Combatant) do
		if recap.Combatant[i].Name then
			recap.Combatant[i].Name = nil
		end
		if recap.Combatant[i].OwnsPet then
			recap.Combatant[i].OwnsPet = nil
		end
	end

	-- remove current UserData: - it's now "live" in recap.Combatant and elsewhere
	if recap_set and recap_set["UserData:"..recap_temp.p] then
		recap_set["UserData:"..recap_temp.p] = nil -- keep as nil
	end

	Recap_LoadSelf()

	-- special case migration of Detail to OutgoingDetail, and Incoming to IncomingDetail for 3.59 and later, prior to saving
	Recap_MigrateDetails()

	-- special case correction for some damage:healing DoT:HoT spells
	Recap_CorrectDualDoTHoTForSelf()

	-- special case correction to add OutgoingDetail.Ticks, OutgoingDetail.Ticks, and Self.Ticks (and for IncomingDetails)
	Recap_CorrectForTicks("OutgoingDetail", "IncomingDetail")
end


--[[ Fight report functions ]]

function Recap_AutoPostGetStatID(myType)

	if myType=="Damage" then
		return "DmgOut"
	elseif myType=="Tanking" then
		return "DmgIn"
	elseif myType=="Healing" then
		return "Heal"
	end
	return "DPS"
end

function Recap_PostFight()
	Recap_PostSpamRows(Recap_AutoPostGetStatID(recap.Opt.AutoPost.Stat), true)
end

function Recap_PostLeader()

	local print_chat = SendChatMessage
	local i, chatchan, chatnum, leader, statid, thisCombatant

	if recap_temp.ListSize>2 then
		chatchan = recap.Opt.AutoPost.Channel
		-- de-localize the channel name
		for i=1,5 do -- hardcoded
			if chatchan == recap_temp.Localize.ChannelDropList[i] then
				chatchan = recap_temp.CanonicalChannelDropList[i]
				break
			end
		end
		if not chatchan then
			chatchan = "Self"
		end
		_,_,chatnum = string_find(chatchan, "(%d+)")
		if chatnum then
			chatchan = "CHANNEL"
		end
		if chatchan=="Self" then
			print_chat = Recap_Print
		end

		-- if the focus is on a WIM (WoW Instant Messenger) edit box, post there
		if WIM_EditBoxInFocus then
			print_chat = Recap_WIM_Print
		end

		-- If the Clip box on the Options : Reports panel is open, post there instead of to chat or console
		if RecapClipEditBox:IsVisible() then
			print_chat = Recap_LineToClipboard
		end

		statid = Recap_AutoPostGetStatID(recap.Opt.AutoPost.Stat)

		if not recap.Combatant[recap_temp.StatLeader] then -- in case leader reset
			recap_temp.StatLeader = false
		end

		thisCombatant = recap_temp.List[1].Name
		if (not recap_temp.StatLeader) and thisCombatant and recap.Combatant[thisCombatant] and recap.Combatant[thisCombatant].Friend then
			recap_temp.StatLeader = thisCombatant
			leader = recap_temp.StatLeader -- spam initial leader
		end

		if recap_temp.StatLeader and recap.Combatant[recap_temp.StatLeader] then
			for i in pairs(recap_temp.InFriend) do
				local iCombatant = recap.Combatant[i]
				if iCombatant and iCombatant["Total"..statid] and recap.Combatant[recap_temp.StatLeader]["Total"..statid] and (iCombatant["Total"..statid] > recap.Combatant[recap_temp.StatLeader]["Total"..statid]) then
					leader = i
				end
			end
			if leader then
				recap_temp.StatLeader = leader
				print_chat(string_format(recap_temp.Localize.NewLeader.." "..recap_temp.Localize.For.." %s: %s "..recap_temp.Localize.With.." %s",recap.Opt.AutoPost.Stat,Recap_StripGUIDsFromCombatant(leader),Recap_FormatStat(statid,leader)), chatchan, nil, chatnum)
			end
		end
	end
end

-- function to package channel info calculation
function Recap_GetChannelInfo(recapAuto, forceSelf)

	local chatchan, chatnum

	chatchan = false
	chatnum = false
	if recapAuto then
		chatchan = recap.Opt.AutoPost.Channel
		-- de-localize the channel name
		for i=1,5 do -- hardcoded
			if chatchan == recap_temp.Localize.ChannelDropList[i] then
				chatchan = recap_temp.CanonicalChannelDropList[i]
				break
			end
		end
	else
		if forceSelf then
			chatchan = "Self"
		else
			if ChatFrameEditBox:IsVisible() then
				chatchan = ChatFrameEditBox:GetAttribute("chatType")
				if chatchan=="WHISPER" then
					chatnum = ChatFrameEditBox:GetAttribute("tellTarget")
				elseif chatchan=="CHANNEL" then
					chatnum = ChatFrameEditBox:GetAttribute("channelTarget")
				end
			end
		end
	end
	if not chatchan then
		chatchan = "Self"
	end
	if not chatnum then
		_,_,chatnum = string_find(chatchan, "(%d+)")
		if chatnum then
			chatchan = "CHANNEL"
		end
	end

	return chatchan, chatnum
end

-- type = stat to report by (DPS, DmgOut, etc)
function Recap_PostSpamRows(myType, recapAuto)

	local i
	local headertext = nil -- keep as nil
	local print_chat = SendChatMessage
	local chatchan, chatnum
	local maxLines = recap.Opt.MaxRank.value
	local iType = string_gsub(myType, "P$","") -- strip trailing P from % ids

	chatchan, chatnum = Recap_GetChannelInfo(recapAuto, nil)
	if chatchan=="Self" then
		print_chat = Recap_Print
	end

	-- if the focus is on a WIM (WoW Instant Messenger) edit box, post there
	if WIM_EditBoxInFocus then
		print_chat = Recap_WIM_Print
	end

	-- If the Clip box on the Options : Reports panel is open, post there instead of to chat or console
	if RecapClipEditBox:IsVisible() then
		print_chat = Recap_LineToClipboard
		maxLines = recap_temp.MaxRecentEventLinesClipboard
	end

	headertext = "__ Recap ("..Recap_PetsMergedText().."): "..recap_temp.Localize.LinkRank[iType].." "..recap_temp.Localize.For.." "..recap_temp.LastAll[recap.Opt.View.value].." __"
	for i=1, maxLines do
		if recap_temp.List[i] and recap_temp.List[i].Name then
			thisCombatant = recap_temp.List[i].Name
			if (i<recap_temp.ListSize) and thisCombatant and recap.Combatant[thisCombatant] and recap.Combatant[thisCombatant].Friend then
				text = i..". "..Recap_StripGUIDsFromCombatant(thisCombatant)..": "..Recap_FormatStat(iType,i)
				if recap_temp.List[i][iType] > 0 then
					if headertext then
						print_chat(headertext, chatchan, nil, chatnum)
						headertext = nil -- keep as nil
					end
					print_chat(text, chatchan, nil, chatnum)
				end
			end
		end
	end
end

-- two function to post Recent Events removed

-- filterType can be "Group" or "NonGroup"
function Recap_FormatRecentEvent(i, outgoing, filterCombatant, filterEffect, filterType)

	local printTime = Recap_FormatTimeRecent(recap_temp.Recent[i].T)
	local source = recap_temp.Recent[i].S
	local dest = recap_temp.Recent[i].D
	local effectType = string_sub(recap_temp.Recent[i].E,1,1)
	local printEffect = string_sub(recap_temp.Recent[i].E,2)
	local printElement
	if recap_temp.Recent[i].L then
		printElement = recap_temp.Recent[i].L
	else
		printElement = "?"
	end
	local amountNumeric = false
	if type(recap_temp.Recent[i].A) == "number" then
		amountNumeric = true
	end
	local printAmount = tostring(recap_temp.Recent[i].A)
	local printCrit
	printCrit = ""
	if recap_temp.Recent[i].C then
		-- critical blow
		printCrit = "*"
	end
	if recap_temp.Recent[i].R then
		-- crushing blow
		printCrit = "**"
	end

	local okayCombatant = false
	if filterType and filterType=="Group" then
		if outgoing and filterEffect and source and recap.Combatant[source] and recap.Combatant[source].Friend then
			okayCombatant = true
		elseif (not outgoing) and filterEffect and dest and recap.Combatant[dest] and recap.Combatant[dest].Friend then
			okayCombatant = true
		end
	elseif filterType and filterType=="NonGroup" then
		if outgoing and filterEffect and source and recap.Combatant[source] and (not recap.Combatant[source].Friend) then
			okayCombatant = true
		elseif (not outgoing) and filterEffect and dest and recap.Combatant[dest] and (not recap.Combatant[dest].Friend) then
			okayCombatant = true
		end
	else
		if filterCombatant then
			-- filtering by combatant
			if Recap_PetsMerged() then
				if outgoing then
					if source then
						if Recap_IsOwnedBy(source, filterCombatant) then
							okayCombatant = true
						end
					end
				else
					if dest then
						if Recap_IsOwnedBy(dest, filterCombatant) then
							okayCombatant = true
						end
					end
				end
			end
			-- check the combatant
			if outgoing and source and (filterCombatant == source) then
				okayCombatant = true
			elseif not outgoing and (filterCombatant == dest) then
				okayCombatant = true
			end
			-- if not already chosen, check death event since we want deaths to show up on both incoming and outgoing panels
			if not okayCombatant then
				if (effectType == "5") and (filterCombatant == dest) then
					okayCombatant = true
				end
			end
		else
			-- no filtering by combatant
			okayCombatant = true
		end
	end

	if okayCombatant == false then
		return nil
	end

	-- filter by effect
	if filterEffect and (filterEffect ~= printEffect) then
		return nil
	end

	-- trim the source and dest to remove GUIDs
	if source then
		source = Recap_StripGUIDsFromCombatant(source)
	end
	if dest then
		dest = Recap_StripGUIDsFromCombatant(dest)
	else
		-- needed if Store Only Displayed Combatants is enabled
		dest = "?"
	end

	-- prepare the message
	if source then
		if amountNumeric then
			if effectType == "1" then
				-- damage
				if (printEffect == printElement) or (printElement == recap_temp.Localize.ElementOther) then
					return printTime.." "..source.." ("..printEffect..") ==> "..dest.." ( -"..printAmount..printCrit.." )"
				else
					return printTime.." "..source.." ("..printEffect.." ("..printElement..")) ==> "..dest.." ( -"..printAmount..printCrit.." )"
				end
			else
				-- heal
				if (printEffect == printElement) or (printElement == recap_temp.Localize.ElementOther) then
					return printTime.." "..source.." ("..printEffect..") ==> "..dest.." ( +"..printAmount..printCrit.." )"
				else
					return printTime.." "..source.." ("..printEffect.." ("..printElement..")) ==> "..dest.." ( +"..printAmount..printCrit.." )"
				end
			end
		else
			if effectType == "5" then
				-- death
				return printTime.." "..source.." ("..printEffect..") ==> "..dest
			elseif effectType == "6" then
				-- dispel (should have source and dest) (in this context the Effect is the name of whatever is doing the dispelling, and the Element is whatever is being dispelled)
				return printTime.." "..source.." ("..printEffect..") "..recap_temp.Localize.Dispels.." ==> "..dest.." ("..printElement..")"
			elseif printAmount == "cast" then
				-- successful cast, target not available, not sure how best to display this
				return printTime.." "..source.." ("..printEffect..") ==> "..dest
			else
				return printTime.." "..source.." ("..printEffect..") ==> "..dest.." ( "..printAmount.." )"
			end
		end
	else
		if amountNumeric then
			if effectType == "1" then
				-- damage
				if (printEffect == printElement) or (printElement == recap_temp.Localize.ElementOther) then
					return printTime.." ("..printEffect..") ==> "..dest.." ( -"..printAmount..printCrit.." )"
				else
					return printTime.." ("..printEffect.." ("..printElement..")) ==> "..dest.." ( -"..printAmount..printCrit.." )"
				end
			else
				-- heal
				if (printEffect == printElement) or (printElement == recap_temp.Localize.ElementOther) then
					return printTime.." ("..printEffect..") ==> "..dest.." ( +"..printAmount..printCrit.." )"
				else
					return printTime.." ("..printEffect.." ("..printElement..")) ==> "..dest.." ( +"..printAmount..printCrit.." )"
				end
			end
		else
			if effectType == "5" then
				-- death
				return printTime.." "..dest.." ( "..printEffect.." )"
			else
				if printAmount == "cast" then
					-- successful cast, target not available, not sure how best to display this
					return printTime.." ("..printEffect..") ==> "..dest
				else
					return printTime.." ("..printEffect..") ==> "..dest.." ( "..printAmount.." )"
				end
			end
		end
	end
end

-- returns a formatted string for type stat ("DPS" "Time" etc) for combatant (number is .List[], non-number is name)
function Recap_FormatStat(myType, combatant)

	local i, iCombatant
	local iType = string_gsub(myType,"P$","")

	if not tonumber(combatant) then

		for i=1,(recap_temp.ListSize-1) do
			if recap_temp.List[i].Name==combatant then
				combatant = i
				i = recap_temp.ListSize
			end
		end
	end

	if tonumber(combatant) then
		iCombatant = recap_temp.List[combatant]
		if iType=="DPS" then
			return string_format("%.1f",iCombatant.DPS)
		elseif iType=="DPSIn" then
			return string_format("%.1f",iCombatant.DPSIn)
		elseif iType=="HPS" then
			return string_format("%.1f",iCombatant.HPS)
		elseif iType=="DPSvsAll" then
			return string_format("%.1f",iCombatant.DPSvsAll)
		elseif iType=="Time" then
			return Recap_FormatTime(iCombatant.Time)
		elseif iType=="TimeIn" then
			return Recap_FormatTime(iCombatant.TimeIn)
		elseif iType=="TimeHeal" then
			return Recap_FormatTime(iCombatant.TimeHeal)
		elseif iType=="DmgOut" then
			return iCombatant.DmgOut.." ("..iCombatant.DmgOutP.."%)"
		elseif iType=="DmgIn" then
			return iCombatant.DmgIn.." ("..iCombatant.DmgInP.."%)"
		elseif iType=="Heal" then
			return iCombatant.Heal.." ("..iCombatant.HealP.."%)".." ("..iCombatant.Over.."%>)"
		elseif iType=="Over" then
			return iCombatant.Over.."%"
		elseif iType=="Seen" then
			return Recap_FormatTimeSeen((iCombatant.Seen or tonumber(0))/1000)
		else
			return iCombatant[iType]
		end
	end

	return ""
end

function Recap_Print(text)
	DEFAULT_CHAT_FRAME:AddMessage(tostring(text))
end

function Recap_WIM_Print(text)
	-- get the user (the target of the whisper) from the WIM chat edit box and whisper to them (WIM deals only in whispers)
	SendChatMessage(tostring(text), "WHISPER", nil, WIM_ParseNameTag(WIM_EditBoxInFocus:GetParent().theUser))
end


--[[ Assorted repair / migration functions ]]

-- Special case for combined DoT:HoT spells where the healing event comes as if it were a non-HoT
-- This code is designed to correct existing errors after the fact
-- TODO: at the moment it doesn't correct the element subtotal
function Recap_CorrectDualDoTHoT(iCombatant, AllLastOutgoingDetail, AllLastIncomingDetail)

	local thisTypeEffect, thisType, healEffect, damageEffect, iHealEffect

	if iCombatant[AllLastOutgoingDetail] then
		for thisTypeEffect in pairs(iCombatant[AllLastOutgoingDetail]) do
			-- checking every effect for every combatant
			thisType = tonumber(string_sub(thisTypeEffect,1,1))
			if (thisType == 3) or (thisType == 4) then
				-- only check from healing effects
				healEffect = thisTypeEffect
				iHealEffect = iCombatant[AllLastOutgoingDetail][healEffect]
				-- compute the matching damage effect, and see if it exists
				damageEffect = tostring(thisType-2)..string_sub(healEffect,2)
				if iCombatant[AllLastOutgoingDetail][damageEffect] then
					-- there is a matching damage effect, so this is a dual damage:healing effect
					if iCombatant[AllLastOutgoingDetail][damageEffect].TicksDmg and iHealEffect.HitsDmg then
						-- the damage half of the effect is a DoT and the healing half is (incorrectly) non-HoT healing
						-- convert the healing half of the effect into a HoT
						iHealEffect.Ticks = (iHealEffect.Ticks or 0) + (iHealEffect.CritsEvents or 0)
						iHealEffect.Hits = nil -- keep as nil
						iHealEffect.CritsEvents = nil -- keep as nil
						iHealEffect.TicksDmg = (iHealEffect.TicksDmg or 0) + (iHealEffect.HitsDmg or 0)
						iHealEffect.HitsDmg = nil -- keep as nil
						iHealEffect.TicksMax = math_max((iHealEffect.TicksMax or 0),(iHealEffect.HitsMax or 0))
						iHealEffect.HitsMax = nil -- keep as nil
						iHealEffect.TicksMin = Recap_Min(iHealEffect.TicksMin,iHealEffect.HitsMin)
						iHealEffect.HitsMin = nil -- keep as nil
					end
				end
			end
		end
	end
	if iCombatant[AllLastIncomingDetail] then
		for thisTypeEffect in pairs(iCombatant[AllLastIncomingDetail]) do
			-- checking every effect for every combatant
			thisType = tonumber(string_sub(thisTypeEffect,1,1))
			if (thisType == 3) or (thisType == 4) then
				-- only check from healing effects
				healEffect = thisTypeEffect
				iHealEffect = iCombatant[AllLastIncomingDetail][healEffect]
				-- compute the matching damage effect (in the *Outgoing* table), and see if it exists
				damageEffect = tostring(thisType-2)..string_sub(healEffect,2)
				if iCombatant[AllLastOutgoingDetail] and iCombatant[AllLastOutgoingDetail][damageEffect] then
					-- there is a matching damage effect, so this is a dual damage:healing effect
					if iCombatant[AllLastOutgoingDetail][damageEffect].TicksDmg and iHealEffect.HitsDmg then
						-- the damage half of the effect is a DoT and the healing half is (incorrectly) non-HoT healing
						-- convert the healing half of the effect into a HoT
						iHealEffect.Ticks = (iHealEffect.Ticks or 0) + (iHealEffect.CritsEvents or 0)
						iHealEffect.Hits = nil -- keep as nil
						iHealEffect.CritsEvents = nil -- keep as nil
						iHealEffect.TicksDmg = (iHealEffect.TicksDmg or 0) + (iHealEffect.HitsDmg or 0)
						iHealEffect.HitsDmg = nil -- keep as nil
						iHealEffect.TicksMax = math_max((iHealEffect.TicksMax or 0),(iHealEffect.HitsMax or 0))
						iHealEffect.HitsMax = nil -- keep as nil
						iHealEffect.TicksMin = Recap_Min(iHealEffect.TicksMin,iHealEffect.HitsMin)
						iHealEffect.HitsMin = nil -- keep as nil
					end
				end
			end
		end
	end
end
function Recap_CorrectDualDoTHoTForSelf()

	local thisTypeEffect, thisType, healEffect, damageEffect, iHealEffect

	-- for the Self table (is Outgoing only)
	if not recap.Self[recap_temp.s] then
		recap.Self[recap_temp.s] = {}
	end
	for thisTypeEffect in pairs(recap.Self[recap_temp.s]) do
		thisType = tonumber(string_sub(thisTypeEffect,1,1))
		if (thisType == 3) or (thisType == 4) then
			-- only check from healing effects
			healEffect = thisTypeEffect
			iHealEffect = recap.Self[recap_temp.s][healEffect]
			-- compute the matching damage effect, and see if it exists
			damageEffect = tostring(thisType-2)..string_sub(healEffect,2)
			if recap.Self[recap_temp.s][damageEffect] then
				-- there is a matching damage effect, so this is a dual damage:healing effect
				if recap.Self[recap_temp.s][damageEffect].TicksDmg and iHealEffect.HitsDmg then
					-- the damage half of the effect is a DoT and the healing half is non-HoT healing
					-- convert the healing half of the effect into a HoT
					iHealEffect.Ticks = (iHealEffect.Ticks or 0) + (iHealEffect.CritsEvents or 0)
					iHealEffect.Hits = nil -- keep as nil
					iHealEffect.CritsEvents = nil -- keep as nil
					iHealEffect.TicksDmg = (iHealEffect.TicksDmg or 0) + (iHealEffect.HitsDmg or 0)
					iHealEffect.HitsDmg = nil -- keep as nil
					iHealEffect.TicksMax = math_max((iHealEffect.TicksMax or 0),(iHealEffect.HitsMax or 0))
					iHealEffect.HitsMax = nil -- keep as nil
					iHealEffect.TicksMin = Recap_Min(iHealEffect.TicksMin,iHealEffect.HitsMin)
					iHealEffect.HitsMin = nil -- keep as nil
				end
			end
		end
	end
end

-- special case correction to add OutgoingDetail.Ticks and Self.Ticks (conversion from v3.56 or earlier to v 3.57 or later)
function Recap_CorrectForTicks(AllLastOutgoingDetail, AllLastIncomingDetail)

	local i, thisTypeEffect, subtotal, iCombatant, iEffect

	-- silent repair
	if not recap.Combatant then
		recap.Combatant = {}
	end
	-- first for the Combatant table
	for i in pairs(recap.Combatant) do
		if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) then
			iCombatant = recap.Combatant[i]
			if iCombatant[AllLastOutgoingDetail] then
				for thisTypeEffect in pairs(iCombatant[AllLastOutgoingDetail]) do
					iEffect = iCombatant[AllLastOutgoingDetail][thisTypeEffect]
					if iEffect.TicksDmg and not iEffect.Ticks then
						-- DoT:HoT without a tick count, add one
						if iEffect.HitsDmg then
							-- DoT:HoT with a direct damage component
							subtotal = math_floor(((iEffect.CritsEvents or 0) - (iEffect.Crits or 0))/2)
						else
							subtotal = (iEffect.CritsEvents or 0) - (iEffect.Crits or 0)
						end
						iEffect.Ticks = (iEffect.Hits or 0)
						iEffect.Hits = subtotal
						iEffect.CritsEvents = subtotal + (iEffect.Crits or 0)
					end
				end
			end
			if iCombatant[AllLastIncomingDetail] then
				for thisTypeEffect in pairs(iCombatant[AllLastIncomingDetail]) do
					iEffect = iCombatant[AllLastIncomingDetail][thisTypeEffect]
					if iEffect.TicksDmg and not iEffect.Ticks then
						-- DoT:HoT without a tick count, add one
						if iEffect.HitsDmg then
							-- DoT:HoT with a direct damage component
							subtotal = math_floor(((iEffect.CritsEvents or 0) - (iEffect.Crits or 0))/2)
						else
							subtotal = (iEffect.CritsEvents or 0) - (iEffect.Crits or 0)
						end
						iEffect.Ticks = (iEffect.Hits or 0)
						iEffect.Hits = subtotal
						iEffect.CritsEvents = subtotal + (iEffect.Crits or 0)
					end
				end
			end
		end
	end

	-- silent repairs
	if not recap.Self then
		recap.Self = {}
	end
	if not recap.Self[recap_temp.s] then
		recap.Self[recap_temp.s] = {}
	end
	-- second for the Self table
	for thisTypeEffect in pairs(recap.Self[recap_temp.s]) do
		iEffect = recap.Self[recap_temp.s][thisTypeEffect]
		if iEffect.TicksDmg and not iEffect.Ticks then
			-- DoT:HoT without a tick count, add one
			if iEffect.HitsDmg then
				-- DoT:HoT with a direct damage component
				subtotal = math_floor(((iEffect.CritsEvents or 0) - (iEffect.Crits or 0))/2)
			else
				subtotal = (iEffect.CritsEvents or 0) - (iEffect.Crits or 0)
			end
			iEffect.Ticks = (iEffect.Hits or 0)
			iEffect.Hits = subtotal
			iEffect.CritsEvents = subtotal + (iEffect.Crits or 0)
		end
	end
end

-- migrate Detail to OutgoingDetail; and Incoming to IncomingDetail (for 3.59 and later)
function Recap_MigrateDetails()

	local i, thisEffect, j, iCombatant, iEffect

	-- silent repair
	if not recap.Combatant then
		recap.Combatant = {}
	end
	for i in pairs(recap.Combatant) do
		if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) then
			iCombatant = recap.Combatant[i]
			if iCombatant.Detail then
				if iCombatant.OutgoingDetail then
					iCombatant.OutgoingDetail = {}
				end
				for thisEffect in pairs(iCombatant.Detail) do
					if not iCombatant.OutgoingDetail[thisEffect] then
						iCombatant.OutgoingDetail[thisEffect] = {}
					end
					for j in pairs(iCombatant.Detail[thisEffect]) do
						iCombatant.OutgoingDetail[thisEffect][j] = iCombatant.Detail[thisEffect][j]
					end
				end
				iCombatant.Detail = nil -- keep as nil
			end

			if iCombatant.Incoming then
				if iCombatant.Incoming.MeleeDamage then
					if not iCombatant.IncomingDetail then
						iCombatant.IncomingDetail = {}
					end
					if not iCombatant.IncomingDetail["1Melee"] then
						iCombatant.IncomingDetail["1Melee"] = {}
					end
					iEffect = iCombatant.IncomingDetail["1Melee"]
					iEffect.Element = "Melee"
					iEffect.HitsDmg = iCombatant.Incoming.MeleeDamage
					if iCombatant.Incoming.MeleeHits then
						iEffect.Hits = iCombatant.Incoming.MeleeHits
						iEffect.CritsEvents = iCombatant.Incoming.MeleeHits
					end
					if iCombatant.Incoming.MeleeCrits then
						iEffect.Crits = iCombatant.Incoming.MeleeCrits
						iEffect.CritsEvents = (iEffect.CritsEvents or 0) + iCombatant.Incoming.MeleeCrits
					end
					if iCombatant.Incoming.MeleeMax then
						iEffect.HitsMax = iCombatant.Incoming.MeleeMax
					end
					if iCombatant.Incoming.MeleeMissed then
						iEffect.Missed = iCombatant.Incoming.MeleeMissed
					end
					if iCombatant.Incoming.MeleeBlocked then
						iEffect.Blocked = iCombatant.Incoming.MeleeBlocked
					end
					if iCombatant.Incoming.MeleeParried then
						iEffect.Parried = iCombatant.Incoming.MeleeParried
					end
					if iCombatant.Incoming.MeleeDodged then
						iEffect.Dodged = iCombatant.Incoming.MeleeDodged
					end
				end
				if iCombatant.Incoming.NonMeleeDamage then
					if not iCombatant.IncomingDetail then
						iCombatant.IncomingDetail = {}
					end
					if not iCombatant.IncomingDetail["1Non-Melee"] then
						iCombatant.IncomingDetail["1Non-Melee"] = {}
					end
					iEffect = iCombatant.IncomingDetail["1Non-Melee"]
					iEffect.Element = "Non-Melee"
					iEffect.HitsDmg = iCombatant.Incoming.NonMeleeDamage
					if iCombatant.Incoming.NonMeleeHits then
						iEffect.Hits = iCombatant.Incoming.NonMeleeHits
						iEffect.CritsEvents = iCombatant.Incoming.NonMeleeHits
					end
					if iCombatant.Incoming.NonMeleeCrits then
						iEffect.Crits = iCombatant.Incoming.NonMeleeCrits
						iEffect.CritsEvents = (iEffect.CritsEvents or 0) + iCombatant.Incoming.NonMeleeCrits
					end
					if iCombatant.Incoming.NonMeleeMax then
						iEffect.HitsMax = iCombatant.Incoming.NonMeleeMax
					end
					if iCombatant.Incoming.NonMeleeTicks then
						iEffect.Ticks = iCombatant.Incoming.NonMeleeTicks
					end
					if iCombatant.Incoming.NonMeleeMissed then
						iEffect.Missed = iCombatant.Incoming.NonMeleeMissed
					end
				end

				iCombatant.Incoming = nil -- keep as nil
			end
		end
	end
end


--[[ Functions directly or indirectly related to the GUIDs ]]

-- throw away leading '0x' and leading zeroes
function Recap_TrimGUID(longGUID)
	local shortGUID
	_, _, shortGUID = string_find(longGUID, "^0x0*(%x%x-)$")
	return shortGUID
end

-- format could be "owner:owner:pet" or "owner:owner:pet" or "owner:pet" or "combatant", and each of those could be "name" or "name_GUID"
function Recap_NameOnlyFromCombatant(thisCombatant)
	local found, pet
	found, _, pet = string_find(thisCombatant, "^.+:.+:(.+)$")
	if found then
		-- "owner:owner:pet"
		return Recap_NameFromNameGUID(pet)
	else
		-- "owner:pet" or "combatant"
		found, _, pet = string_find(thisCombatant, "^.+:(.+)$")
		if found then
			-- "owner:pet"
			return Recap_NameFromNameGUID(pet)
		else
			-- "combatant"
			return Recap_NameFromNameGUID(thisCombatant)
		end
	end
end

function Recap_StripGUIDsFromCombatant(thisCombatant)
	local found, firstOwner, secondOwner, pet
	found, _, firstOwner, secondOwner, pet = string_find(thisCombatant, "^(.+):(.+):(.+)$")
	if found then
		-- "owner:owner:pet"
		return Recap_NameFromNameGUID(firstOwner)..":"..Recap_NameFromNameGUID(secondOwner)..":"..Recap_NameFromNameGUID(pet)
	else
		found, _, firstOwner, pet = string_find(thisCombatant, "^(.+):(.+)$")
		if found then
			-- "owner:pet"
			return Recap_NameFromNameGUID(firstOwner)..":"..Recap_NameFromNameGUID(pet)
		else
			-- "combatant"
			return Recap_NameFromNameGUID(thisCombatant)
		end
	end
end

function Recap_StripOwnerAndGUIDsFromCombatant(thisCombatant)
	local found, pet
	found, _, pet = string_find(thisCombatant, "^.+:.+:(.+)$")
	if found then
		-- "owner:owner:pet"
		return Recap_NameFromNameGUID(pet)
	else
		found, _, pet = string_find(thisCombatant, "^.+:(.+)$")
		if found then
			-- "owner:pet"
			return Recap_NameFromNameGUID(pet)
		else
			-- "combatant"
			return Recap_NameFromNameGUID(thisCombatant)
		end
	end
end

-- format could be "effect" or "name_GUID: effect" or "owner_GUID:name_GUID: effect" or "owner_GUID:owner_GUID:name_GUID: effect"
--   (where effect itself could be, for example, "Shadow Word: Pain")
function Recap_StripOwnerAndGUIDsFromEffect(thisEffect)
	local found, iPet, effectFirstPart, effectSecondPart, iEffect
	found, _, iPet, effectFirstPart, effectSecondPart = string_find(thisEffect, "^(.+): (.+): (.+)$")
	if found then
		-- "pet: effectFirstPart: effectSecondPart"
		-- iPet must be a combatant, strip out the GUIDs
		return Recap_StripOwnerAndGUIDsFromCombatant(iPet)..": "..effectFirstPart..": "..effectSecondPart
	else
		found, _, iPet, iEffect = string_find(thisEffect, "^(.+): (.+)$")
		if found then
			-- "owner_GUID:owner_GUID:name_GUID: effect" or "owner_GUID:name_GUID: effect" or "name_GUID: effect" or "effectFirstPart: effectSecondPart"
			-- the following code does the right thing no matter which of the above it is (!)
			return Recap_StripOwnerAndGUIDsFromCombatant(iPet)..": "..iEffect
		else
			-- simple "effect"
			return thisEffect
		end
	end
end

-- format could be "effect" or "name_GUID: effect" or "owner_GUID:name_GUID: effect" or "owner_GUID:owner_GUID:name_GUID: effect"
--   (where effect itself could be, for example, "Shadow Word: Pain")
function Recap_StripOwnerAndPetFromEffect(thisEffect)
	local found, iPet, effectFirstPart, effectSecondPart, iEffect
	found, _, iPet, effectFirstPart, effectSecondPart = string_find(thisEffect, "^(.+): (.+): (.+)$")
	if found then
		-- "pet: effectFirstPart: effectSecondPart"
		-- iPet must be a combatant
		return effectFirstPart..": "..effectSecondPart
	else
		found, _, iPet, iEffect = string_find(thisEffect, "^(.+): (.+)$")
		if found then
			-- "owner_GUID:owner_GUID:name_GUID: effect" or "owner_GUID:name_GUID: effect" or "name_GUID: effect" or "effectFirstPart: effectSecondPart"
			-- the following is heuristic only
			if recap.Combatant[iPet] then
				return iEffect
			else
				return thisEffect
			end
		else
			-- simple "effect"
			return thisEffect
		end
	end
end

-- format is "name_GUID: effect" where effect itself could be, for example, "Shadow Word: Pain"
function Recap_PetFromFromPetEffect(thisEffect)
	local found, iPet, effectFirstPart, effectSecondPart
	found, _, iPet, effectFirstPart, effectSecondPart = string_find(thisEffect, "^(.+): (.+): (.+)$")
	if found then
		-- "name_GUID: effectFirstPart: effectSecondPart"
		return Recap_NameFromNameGUID(iPet)
	else
		-- "name_GUID: effect"
		found, _, iPet = string_find(thisEffect, "^(.+): .+$")
		return Recap_NameFromNameGUID(iPet)
	end
end

-- format is "name_GUID: effect" where effect itself could be, for example, "Shadow Word: Pain"
function Recap_EffectFromFromPetEffect(thisEffect)
	local found, effectFirstPart, effectSecondPart, iEffect
	found, _, effectFirstPart, effectSecondPart = string_find(thisEffect, "^.+: (.+): (.+)$")
	if found then
		-- "name_GUID: effectFirstPart: effectSecondPart"
		return effectFirstPart..": "..effectSecondPart
	else
		-- "name_GUID: effect" or "effect"
		found, _, iEffect = string_find(thisEffect, "^.+: (.+)$")
		return iEffect
	end
end

-- format could be "name" or "name_GUID"
function Recap_NameFromNameGUID(nameGUID)
	local found, name
	found, _, name = string_find(nameGUID, "^(.+)_.-$")
	if found then
		return name
	else
		-- no GUID, use nameGUID as is
		return nameGUID
	end
end

-- note that if we have "firstOwner:secondOwner:name" or "firstOwner:secondOwner:name" this checks against "firstOwner"
function Recap_IsOwnedBy(thisCombatant, potentialOwner)
	local found, firstOwner, secondOwner
	found, _, firstOwner, secondOwner = string_find(thisCombatant, "^(.+):(.+):.+$")
	if found and (firstOwner == potentialOwner) then
		return true
	else
		found, _, firstOwner = string_find(thisCombatant, "^(.+):.+$")
		if found and (firstOwner == potentialOwner) then
			return true
		else
			return false
		end
	end
end

-- note that if we have "firstOwner:secondOwner:name" or "firstOwner:secondOwner:name" this returns "firstOwner" (the ultimate owner)
function Recap_ExtractOwner(thisCombatant)
	local found, firstOwner, secondOwner
	found, _, firstOwner, secondOwner = string_find(thisCombatant, "^(.+):(.+):.+$")
	if found then
		return firstOwner
	else
		found, _, firstOwner = string_find(thisCombatant, "^(.+):.+$")
		if found then
			return firstOwner
		else
			return nil
		end
	end
end

function Recap_ExtractCombatant(thisCombatant)
	local found, combatant
	found, _, combatant = string_find(thisCombatant, "^.+:.+:(.+)$")
	if found then
		return combatant
	else
		found, _, combatant = string_find(thisCombatant, "^.+:(.+)$")
		if found then
			return combatant
		else
			return nil
		end
	end
end

function Recap_CombatantIsTrash(thisCombatant)
	local i, count, name, owner

-- TODO: trickier than it looks, bosses who reset get new GUIDs ...

	-- shortcuts
	if recap.Combatant[thisCombatant].Friend then -- redundant check, call it paranoia
		return false
	end
	if recap.Combatant[thisCombatant].Trash == false then
		-- marked as not trash, take that as definitive
		return false
	end
	if recap.Combatant[thisCombatant].Level and tonumber(recap.Combatant[thisCombatant].Level)==-1 then
		-- if this is a high level boss (level is -1) then mark it as not trash
		recap.Combatant[thisCombatant].Trash = false
		return false
	end
	if recap.Combatant[thisCombatant].Trash == true then
		-- marked as trash, but if it is a pet of a combatant who is marked as not trash, change the pet to not trash also
		owner = Recap_ExtractOwner(thisCombatant)
		if owner and recap.Combatant[owner] and (recap.Combatant[owner].Trash ~= nil) and (recap.Combatant[owner].Trash == false) then
			recap.Combatant[thisCombatant].Trash = false
			return false
		end
		return true
	end

	-- we don't know about this combatant, count the hard way
	count = 0
	name = Recap_NameOnlyFromCombatant(thisCombatant)
	for i in pairs(recap.Combatant) do
		if (i ~= recap_temp.GroupTotal) and (i ~= recap_temp.NonGroupTotal) then
			if (Recap_NameOnlyFromCombatant(i) == name) then
				count = count + 1
				if count > 1 then
					-- we have found more than one of this combatant, mark this one as trash
					recap.Combatant[i].Trash = true
					-- we carry on looping to mark any others
					-- the first one we found may not be marked yet
				end
			end
		end
	end

	if count > 1 then
		recap.Combatant[thisCombatant].Trash = true
		return true
	else
		-- only found one, and can't tell whether it is trash or not
		return false
	end
end


RecapAux_lua_411 = true
