-- Vars -------------------------------------------------------------------
ReagentRestocker = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0","AceEvent-2.0")

local moduleName = 'ReagentRestocker'

-- Item property labels
local ITEM_NAME = "0"
local ITEM_LINK = "1"
local ITEM_RARITY = "2"
local ITEM_LEVEL = "3"
local ITEM_MIN_LEVEL = "4"
local ITEM_TYPE = "5"
local ITEM_SUB_TYPE = "6"
local ITEM_STACK_COUNT = "7"
local ITEM_EQUIP_LOC = "8"
local ITEM_TEXTURE = "9"
local QUANTITY_TO_STOCK = "qty"

-- Shared variables
local LockedBagSlotIDList = {}

local TransactionLock = false

-- Events waiting to be performed
local QueuedActions = {}

-- Static variables
local SHOPPING_TYPE = "shopping"
local SELLING_TYPE = "selling"

-- Event IDs
local PLAYER_MONEY_EVENT = "PLAYER_MONEY"
local ITEM_LOCK_CHANGED_EVENT = "ITEM_LOCK_CHANGED"
local MERCHANT_SHOW_EVENT = "MERCHANT_SHOW"
local MERCHANT_UPDATE_EVENT = "MERCHANT_UPDATE"
local BANKFRAME_OPENED_EVENT = "BANKFRAME_OPENED"
local VARIABLES_LOADED_EVENT = "VARIABLES_LOADED"
local BAG_UPDATE_EVENT = "BAG_UPDATE"

-- Helpers
local map = table.foreach

-- Waterfall
local waterfall = AceLibrary("Waterfall-1.0")

-- UI
local Menu = { 
	type='group',
	args = {
		shoppingOptions = {
			type = "group",
			name = "Shopping",
			desc = "Change how Reagent Restocker shops",
			args = {
				addTitle = {
					type = "header",
					name = "Add to Shopping List",
					order = 1					
				},
			
				addReagent = {
					type = "text",
					name = "Type the item's name and hit enter",
					desc = "Adds a new item to your Shopping List",
					get = "",
					usage = "<reagent name>",
					set = function (x) ReagentRestocker:addItemToShoppingList(x); end,
					order = 2
				},
				
				discountTitle = {
					type = "header",
					name = " ",
					order = 3				
				},
				
				discountSpacerTwo = {
					type = "header",
					name = "Required Faction Discount",
					order = 4
				},
				none = {
					type = "toggle",
					name = "None (0%)",
					desc = "Reagent Restocker will purchase Shopping List items from any vendors",
					get = function () return (ReagentRestockerDB.Options.RequiredDiscount == 0) end,
					set = function (x) ReagentRestockerDB.Options.RequiredDiscount = 0 end,
					order = 5
				},
				friendly = {
					type = "toggle",
					name = "Friendly (5%)",
					desc = "Reagent Restocker will only purchase Shopping List items from vendors with a 5% (or better) discount",
					get = function () return (ReagentRestockerDB.Options.RequiredDiscount == 5) end,
					set = function (x) ReagentRestockerDB.Options.RequiredDiscount = 5 end,
					order = 6
				},
				honored = {
					type = "toggle",
					name = "Honored (10%)",
					desc = "Reagent Restocker will only purchase Shopping List items from vendors with a 10% (or better) discount",
					get = function () return (ReagentRestockerDB.Options.RequiredDiscount == 10) end,
					set = function (x) ReagentRestockerDB.Options.RequiredDiscount = 10 end,
					order = 7
				},
				revered = {
					type = "toggle",
					name = "Revered (15%)",
					desc = "Reagent Restocker will only purchase Shopping List items from vendors with a 15% (or better) discount",
					get = function () return (ReagentRestockerDB.Options.RequiredDiscount == 15) end,
					set = function (x) ReagentRestockerDB.Options.RequiredDiscount = 15 end,
					order = 8
				},						
				exalted = {
					type = "toggle",
					name = "Exalted (20%)",
					desc = "Reagent Restocker will only purchase Shopping List items from vendors with a 20% discount",
					get = function () return (ReagentRestockerDB.Options.RequiredDiscount == 20) end,
					set = function (x) ReagentRestockerDB.Options.RequiredDiscount = 20 end,
					order = 9
				},
				
				otherSpacer = {
					type = "header",
					name = " ",
					order = 10
				},
				otherTitle = {
					type = "header",
					name = "Other",
					order = 11
				},				
				AutoBuy = {
					type = "toggle",
					name = "Enable auto-shop",
					desc = "Enables auto-shopping; if you turn this option off, Reagent Restocker will not purchase items automatically",
					get = function () return ReagentRestockerDB.Options.AutoBuy end,
					set = function (x) ReagentRestockerDB.Options.AutoBuy = x end,
					order = 12
				},
				Overstock = {
					type = "toggle",
					name = "Overstock",
					desc = "If this option is disabled, Reagent Restocker will never stock more than the quantity you specify (otherwise, overstocking may occur when a vendor sells an item in stacks instead of one-by-one)",
					get = function () return not ReagentRestockerDB.Options.NotOverstock; end,
					set = function (x) ReagentRestockerDB.Options.NotOverstock = not x; end,
					order = 12
				}				
			},
			order = 1
		},
		sellingOptions = {
			type = "group",
			name = "Selling",
			desc = "Options for automatically selling items",
			args = {
				addTitle = {
					type = "header",
					name = "Add to Selling List",
					order = 1				
				},				
				addSellingList = {
					type = "text",
					name = "Type the item's name and hit enter",
					desc = "Adds an item to your Selling List",
					get = false,
					usage = "<reagent name>",
					set = function (x) ReagentRestocker:addItemToSellingList(x); end,
					order = 2
				},
				
				otherSpacer = {
					type = "header",
					name = " ",
					order = 3
				},
				otherTitle = {
					type = "header",
					name = "Other",
					order = 4					
				},	
				AutoPopulate = {
					type = "toggle",
					name = "Auto-populate Selling List",
					desc = "With this option enabled, Reagent Restocker will watch what items you sell to the vendor and automatically add them to the Selling List",
					get = function() return ReagentRestockerDB.Options.AutoPopulate end,
					set = function(x) ReagentRestockerDB.Options.AutoPopulate = x end,
					order = 5
				},					
				AutoSellGrays = {
					type = "toggle",
					name = "Sell gray items",
					desc = "Do you want Reagent Restocker to automatically sell gray (usless) items from your inventory?  (This is like adding all gray items to your Selling List)",
					get = function() return ReagentRestockerDB.Options.AutoSellGrays end,
					set = function(x) ReagentRestockerDB.Options.AutoSellGrays = x end,
					order = 6
				},
				AutoSell = {
					type = "toggle",
					name = "Enable auto-sell",
					desc = "Enable auto-selling; if you turn this option off, Reagent Restocker will not sell items automatically",
					get = function () return ReagentRestockerDB.Options.AutoSell end,
					set = function (x) ReagentRestockerDB.Options.AutoSell = x end,
					order = 7
				},			
			},
			order = 2
		},
		BankOptions = {
			type = "group",
			name = "Bank",
			desc = "Options for interacting with your bank",
			order = 3,
			args = {
				pullFromBank = {
					type = "toggle",
					name = "Stock items from bank",
					desc = "Pulls items on your Shopping List from your bank if you need them",
					get = function () return ReagentRestockerDB.Options.PullFromBank end,
					set = function (x) ReagentRestockerDB.Options.PullFromBank = x end
				},
				overstockToBank = {
					type = "toggle",
					name = "Overstock to bank",
					desc = "Puts 'extra' Shopping List items (i.e., more than the quantity you keep stocked) into your bank",
					get = function() return ReagentRestockerDB.Options.OverstockToBank end,
					set = function(x) ReagentRestockerDB.Options.OverstockToBank = x end
				}				
			}
		},
		RepairOptions = {
			type = "group",
			name = "Repair",
			desc = "Options for interacting with your bank",
			order = 4,
			args = {
				AutoRepair = {
					type = "toggle",
					name = "Auto-repair",
					desc = "Automatically repair your gear when you visit a repair-able vendor",
					get = function () return ReagentRestockerDB.Options.AutoRepair end,
					set = function (x) ReagentRestockerDB.Options.AutoRepair = x end
				},
				UseGuildBankFunds = {
					type = "toggle",
					name = "Use guild bank funds",
					desc = "Use guild bank funds when auto-repairing, if possible",
					get = function() return ReagentRestockerDB.Options.UseGuildBankFunds end,
					set = function(x) ReagentRestockerDB.Options.UseGuildBankFunds = x end
				},
				RequireDiscount = {
					type = "toggle",
					name = "Require discount (see Shopping options)",
					desc = "Only repair if the repair vendor meets the required vendor discounts (as chosen in the Shopping options)",
					get = function() return ReagentRestockerDB.Options.RepairDiscount end,
					set = function(x) ReagentRestockerDB.Options.RepairDiscount = x end
				}					
			}
		},
		MiscOptions = {
			type = "group",
			name = "Misc.",
			desc = "Other Reagent Restocker options",
			order = 5,
			args = {
				QuietMode = {
					type = "toggle",
					name = "Quiet mode",
					desc = "Disable Reagent Restocker messages",
					get = function() return ReagentRestockerDB.Options.QuietMode end,
					set = function(x) ReagentRestockerDB.Options.QuietMode = x end
				}					
			}
		}	
	}
}

--=========--
-- Helpers --
--=========--

-- Returns the "difference" between two tables with numerical values
local function tDiff(ta, tb, onlyInA)
	local diff = {}
	for k,v in pairs(ta) do
		if tb[k] then
			diff[k] = ta[k] - tb[k]
		else
			diff[k] = ta[k]
		end
	end
	if not onlyInA then
		for k,v in pairs(tb) do
			if not diff[k] then -- Don't use keys we've already used; also, anything in a is already in diff
				diff[k] = -1 * tb[k]		
			end
		end
	end
	return diff
end

-- Returns true if the provided item info indicates the item should be included in the offset list for bank stocking
local function bankOffset(itemID, qty)
	if qty > 0 then 
		return ReagentRestockerDB.Options.PullFromBank
	else
		return ReagentRestockerDB.Options.OverstockToBank
	end
end

-- Returns the number of all items table t; #<table> returns array count only
local function tcount(t)
	local i = 0
	map(t,function () i = i + 1; end)
	return i
end

-- Returns true if item is a value in the table; false otherwise
local function inT(tab,item)
	for _,v in pairs(tab) do
		if v == item then
			return true
		end
	end
	return false
end

-- Returns a string representation of a price (in copper)
local function nCTS(price)
	if price < 100 then
		return price .. "|cFFB87333c|r"
	elseif price < 10000 then
		return price/100 .. "|cFFC0C0C0s|r"
	else
		return ceil(price/100 - 0.005)/100 .. "|cFFCDAD00g|r"
	end
end

-- Returns a string representation of a table
local function strT (tab)
	if type(tab) == type({}) then
		local mystr = ""
		for k,v in pairs(tab) do
			mystr = mystr .. "[" .. tostring(k) .. " = '" .. strT(v) .. "']"
		end
		return mystr
	else
		return tostring(tab)
	end
end

-- Returns the item id of parsed from the provided item link
local function getIDFromItemLink(itemLink)
	if itemLink then
		return tonumber(string.match(itemLink, "|c[0-9a-fA-F]+|Hitem:([0-9]+):.*"))
	end
end

-- Returns the item name from the provided item link
local function getNameFromItemLink(itemLink)
	return string.match(itemLink, ".+%[([^%]]+)%].+")
end

-- Returns true if the 'clue' is found in the item name
local function isItemNameInLink(clue,link)
	-- Remember to escape any -s in the string
	return string.find(string.lower(clue),string.gsub(string.lower(getNameFromItemLink(link)),"-","%%-"))
end

-- Returns a table array containing the bag IDs for bank bags
local function getBankBagIDList()
	local bankBagIDList = {}
	
	-- Add the bank container
	table.insert(bankBagIDList, BANK_CONTAINER)
	
	-- Add the remaining bags
	for bagID=NUM_BAG_SLOTS+1, NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
		table.insert(bankBagIDList, bagID)
	end
	
	return bankBagIDList
end

-- Returns a table array containing the bag IDs for player bags
local function getPlayerBagIDList()
	return {0,1,2,3,4}
end

-- Prints the message if messages are enabled
function ReagentRestocker:say(msg)
	if not ReagentRestockerDB.Options.QuietMode and msg ~= "" then
		self:Print(tostring(msg))
	end
end

--========================--
-- Reagent Restocker Core --
--========================--

-- If the items in cache, update the stored values and return it; otherwise, return what is in cache, if it exists; otherwise return nil
function ReagentRestocker:safeGetItemInfo(itemID)
	if ReagentRestockerDB.Items[itemID] then
		if not GetItemInfo(itemID) then	
			return ReagentRestockerDB.Items[itemID][ITEM_NAME],ReagentRestockerDB.Items[itemID][ITEM_LINK],ReagentRestockerDB.Items[itemID][ITEM_RARITY],ReagentRestockerDB.Items[itemID][ITEM_LEVEL],ReagentRestockerDB.Items[itemID][ITEM_MIN_LEVEL],ReagentRestockerDB.Items[itemID][ITEM_TYPE],ReagentRestockerDB.Items[itemID][ITEM_SUB_TYPE],ReagentRestockerDB.Items[itemID][ITEM_STACK_COUNT],ReagentRestockerDB.Items[itemID][ITEM_EQUIP_LOC],ReagentRestockerDB.Items[itemID][ITEM_TEXTURE]
		else
			ReagentRestockerDB.Items[itemID][ITEM_NAME],ReagentRestockerDB.Items[itemID][ITEM_LINK],ReagentRestockerDB.Items[itemID][ITEM_RARITY],ReagentRestockerDB.Items[itemID][ITEM_LEVEL],ReagentRestockerDB.Items[itemID][ITEM_MIN_LEVEL],ReagentRestockerDB.Items[itemID][ITEM_TYPE],ReagentRestockerDB.Items[itemID][ITEM_SUB_TYPE],ReagentRestockerDB.Items[itemID][ITEM_STACK_COUNT],ReagentRestockerDB.Items[itemID][ITEM_EQUIP_LOC],ReagentRestockerDB.Items[itemID][ITEM_TEXTURE] = GetItemInfo(itemID)
			return ReagentRestockerDB.Items[itemID][ITEM_NAME],ReagentRestockerDB.Items[itemID][ITEM_LINK],ReagentRestockerDB.Items[itemID][ITEM_RARITY],ReagentRestockerDB.Items[itemID][ITEM_LEVEL],ReagentRestockerDB.Items[itemID][ITEM_MIN_LEVEL],ReagentRestockerDB.Items[itemID][ITEM_TYPE],ReagentRestockerDB.Items[itemID][ITEM_SUB_TYPE],ReagentRestockerDB.Items[itemID][ITEM_STACK_COUNT],ReagentRestockerDB.Items[itemID][ITEM_EQUIP_LOC],ReagentRestockerDB.Items[itemID][ITEM_TEXTURE]
		end
	else
		return GetItemInfo(itemID)
	end
end

-- Given an item's name, return the item's ID if it is found; nil on failure
function ReagentRestocker:discoverItemID(itemClue)
	-- Look in the player's backpack/bank
	local bagIDList = {}
	map(getBankBagIDList(),function (_,bagID) table.insert(bagIDList,bagID) end)
	map(getPlayerBagIDList(),function (_,bagID) table.insert(bagIDList,bagID) end)
	for _,bagID in pairs(bagIDList) do
		for bagSlotID=1,GetContainerNumSlots(bagID) do
			currentItemLink = GetContainerItemLink(bagID,bagSlotID)
			if currentItemLink then 
				if isItemNameInLink(itemClue,currentItemLink) then
					return getIDFromItemLink(currentItemLink)
				end
			end
		end
	end
	
	-- Look in the merchant window, if it is open
	if GetMerchantNumItems() then
		for i=1, GetMerchantNumItems() do
			currentItemLink = GetMerchantItemLink(i)
			if currentItemLink then
				if isItemNameInLink(itemClue,currentItemLink) then
					return getIDFromItemLink(currentItemLink)
				end	
			end
		end
	end
	
	-- Look in the items list
	for itemID,data in pairs(ReagentRestockerDB.Items) do
		_, currentItemLink = self:safeGetItemInfo(itemID)
		if currentItemLink then
			if isItemNameInLink(itemClue,currentItemLink) then
				return getIDFromItemLink(currentItemLink)
			end	
		end				
	end
	
	-- If we don't find the item, return nil
	return nil
end

-- Adds a value to the ReagentRestockerDB.Items table
function ReagentRestocker:addToItems(itemID, var, value)
	if not ReagentRestockerDB.Items[itemID] then
		ReagentRestockerDB.Items[itemID] = {}
	end
		
	ReagentRestockerDB.Items[itemID][var] = value
	self:safeGetItemInfo(itemID)
end 

-- Add an item to the shopping list with a starting value of 0
function ReagentRestocker:addItemToShoppingList(reagent)
	self:addToList(reagent,0)
end

-- Add an item to the selling list
function ReagentRestocker:addItemToSellingList(reagent)
	self:addToList(reagent,-1)
end

-- Returns a table of the form {itemID = qtyOff}, indicating how far "off" the player's current stock of items is from "ideal"
function ReagentRestocker:getOffsetList(filter)
	local sl = {}
	for itemID, data in pairs (ReagentRestockerDB.Items) do
		if (self:listType(itemID) == SHOPPING_TYPE) then
			if not filter then
				sl[itemID] = data[QUANTITY_TO_STOCK] - self:countItemInBags(getPlayerBagIDList(),itemID)
			else
				local count = data[QUANTITY_TO_STOCK] - self:countItemInBags(getPlayerBagIDList(),itemID)
				if filter(itemID,count) then
					sl[itemID] = count				
				end
			end			
		end
	end
	
	return sl
end

-- Returns shoppping if the item ID is on the shopping list; selling if it is on the selling list; nil if it is not on any list
function ReagentRestocker:listType(itemID)
	if not ReagentRestockerDB.Items[itemID] then
		return nil
	else
		if ReagentRestockerDB.Items[itemID][QUANTITY_TO_STOCK] > -1 then
			return SHOPPING_TYPE
		else
			return SELLING_TYPE
		end
	end
end

-- Attempts to add an item to the appropriate list
function ReagentRestocker:addToList(reagent,qty)
	-- Make sure something was entered
	if (reagent == "") then
		self:say("Please enter an item name.")
	else
		-- Attempt to find the item; if we do, add it to items
		local itemID
		if type(reagent) == type(1) then
			itemID = reagent
		else
			itemID = self:discoverItemID(reagent)
		end
		
		if itemID then
			local _, itemLink = self:safeGetItemInfo(itemID)
			if ReagentRestockerDB.Items[itemID] then
				-- The item already exists in the item list
				self:say(string.format("%s is already on your %s list.", itemLink, self:listType(itemID)))
			else
				self:addToItems(itemID, QUANTITY_TO_STOCK, qty)
				local msg = string.format("%s has been added to your %s list.", itemLink, self:listType(itemID))
				if listType == SHOPPING_TYPE then
					msg = msg .. "  |cffff8000You must choose a stock quantity before ReagentRestocker will purchase this item.|r"
				end
				self:say(msg)
			end
		else
			self:say(string.format("Reagent Restocker cannot find %q; try adding it when you have the item in your bags, bank, or while visiting a vendor selling it.", reagent))
		end
		
		-- Now that the list is changed, update the options with the changes
		self:synchronizeOptionsTable()
	end

	-- Turn off notifier, since the user performed an action
	ReagentRestockerDB.Options.UnusedNotification = false
end

-- Synchronizes the options table with the current settings
function ReagentRestocker:synchronizeOptionsTable()
	-- Clear the items out of the options
	for _,t in pairs({Menu.args.shoppingOptions.args,Menu.args.sellingOptions.args}) do
		for k,_ in pairs(t) do
			if type(k) == type(1) then
				t[k] = nil
			end
		end
	end	
	
	-- Populate the lists
	for itemID,data in pairs(ReagentRestockerDB.Items) do
		local itemName, _, _, _, _, _, _, itemStackCount, _, itemTexture = self:safeGetItemInfo(itemID)
		if (self:listType(itemID) == SHOPPING_TYPE) then
			-- Helper: Returns a suggested maximum-to-be-stocked quantity based on provided stack size
			-- Common stack sizes are 1, 5, 10, 20, 200
			local maxFromSC = function (stack)	
				if stack >= 1000 then
					return 28000
				elseif stack > 100 then
					return 6400
				elseif stack > 20 then
					return 3200
				elseif stack > 10 then
					return 640
				elseif stack > 1 then
					return 320
				else
					return 32
				end
			end	
			
			-- Add this item to the shopping section of the menu
			Menu.args.shoppingOptions.args[itemID] = {
				type = "group",
				name = itemName .. " (" .. tostring(ReagentRestockerDB.Items[itemID][QUANTITY_TO_STOCK]) .. ")",
				desc = "Currently stocking " .. tostring(ReagentRestockerDB.Items[itemID][QUANTITY_TO_STOCK]),
				icon = itemTexture,
				args = {		
					range = {	
						type = "range",
						step = ceil(itemStackCount/20),
						bigStep = itemStackCount,
						min = 0,
						max = maxFromSC(itemStackCount),
						name = "Stock how many?",
						desc = "Quantity of " .. itemName .. " you wish to keep stocked; Reagent Restocker will purchase enough so that you will have this many after leaving the vendor",
						get = function() return ReagentRestockerDB.Items[itemID][QUANTITY_TO_STOCK]; end,
						set = function(x) ReagentRestockerDB.Items[itemID][QUANTITY_TO_STOCK] = x; self:synchronizeOptionsTable(); end,
						order = 1
					},
					deleteSpacer1 = {
						type = "header",
						name = " ",
						order = 2
					},						
					deleteTitle = {
						type = "header",
						name = "Remove this item from your Shopping List?",
						order = 3
					},
					deleteSpacer2 = {
						type = "header",
						name = " ",
						order = 4
					},						
					delete = {
						type = "execute",
						desc = "Click to remove " .. itemName .. " from your Shopping List",
						name = "Remove",
						func = function() ReagentRestockerDB.Items[itemID] = nil; self:synchronizeOptionsTable();	end,
						order = 5
					}
				}
			}
		elseif (self:listType(itemID) == SELLING_TYPE) then	
			-- Add the item to the selling list
			Menu.args.sellingOptions.args[itemID] = {	
				type = "group",
				name = itemName,
				desc = itemName,
				icon = itemTexture,
				args = {		
					deleteSpacer1 = {
						type = "header",
						name = " ",
						order = 2
					},						
					deleteTitle = {
						type = "header",
						name = "Remove this item from your Selling List?",
						icon = itemTexture,
						order = 3
					},
					deleteSpacer2 = {
						type = "header",
						name = " ",
						order = 4
					},				
					delete = {
						type = "execute",
						desc = "Click to remove " .. itemName .. " from your Shopping List",						
						name = "Remove",
						func = function() ReagentRestockerDB.Items[itemID] = nil; self:synchronizeOptionsTable(); end,
						order = 6
					}
				}
			}
		end
	end
end

-- Locks out transactions to avoid multi-click problems; returns true if the transaction is has been locked; false if it is already locked
function ReagentRestocker:lockTransaction(sec)
	if not TransactionLock then
		TransactionLock = time()
		self:queueAction(function() return time() > TransactionLock + sec end, function() TransactionLock = false end)
		return true
	else
		-- It is already locked
		return false
	end
end

-- Prints a message letting the player know usage
function ReagentRestocker:notifyPlayer()
	if (tcount(ReagentRestockerDB.Items) == 0 and ReagentRestockerDB.Options.UnusedNotification) then
		self:say("Your Shopping List is currently empty.  |cffff8000Type /rr to get started (and get rid of this annoying message).|r")
	end
end

--======================--
-- Merchant Interaction --
--======================--

-- Handles auto-population of the selling list; adds new items in the buyback list to the selling list, if appropriate
function ReagentRestocker:MERCHANT_UPDATE()
	self:triggerAction(MERCHANT_UPDATE_EVENT)
	
	if ReagentRestockerDB.Options.AutoPopulate then
		for i=1,GetNumBuybackItems() do
			local itemLink = GetBuybackItemLink(i)
			if itemLink then 
				local itemID = getIDFromItemLink(itemLink)
				if not ReagentRestockerDB.Items[itemID] then
					-- Doesn't apply this rule to blue or better items; also doesn't apply to items that are to be sold
					local _, _, rarity = self:safeGetItemInfo(itemID)			
					if rarity < 3 and not self:isToBeSold(itemID) then
						self:addItemToSellingList(itemID)
					end
				end
			end
		end
	end
end

-- Handles auto-purching, -selling, and -repairing when the vendor window is opened
function ReagentRestocker:MERCHANT_SHOW()
	self:triggerAction(MERCHANT_SHOW_EVENT)

	if not self:lockTransaction(2) then
		self:say("You are attempting to begin too many transactions in a short time; ignoring ...")
		return
	end	
		
	-- Remind the player how to open RR
	self:notifyPlayer()
	
	-- Do the purchasing, selling, and repairing
	local endMoney, msgs, soldItemsInfo, queueReport = GetMoney(), {}, {}, false
	if ReagentRestockerDB.Options.AutoBuy and self:isReagentVendor() then
		local cost, msg = self:buy()
		endMoney = endMoney - cost
		table.insert(msgs,msg)
		if cost > 0 then queueReport = true; end
	end
	if ReagentRestockerDB.Options.AutoRepair and CanMerchantRepair() then
		local cost, msg = self:repair()
		endMoney = endMoney - cost
		table.insert(msgs,msg)
		-- TODO - Potential bug: Cost can be 0 if guild bank repair occured, but transction is occuring
		if cost > 0 then queueReport = true; end
	end
	if ReagentRestockerDB.Options.AutoSell then
		soldItemsInfo = self:sell()
	end
	
	-- If we're waiting on a transaction to complete, let the player know
	if not queueReport and #soldItemsInfo == 0 then
		self:say(table.concat(msgs))
	else
		self:say("Working, please wait ...")
		self:queueAction(
			function() return self:areSlotsUnlocked(soldItemsInfo); end,
			function() 
				if #soldItemsInfo > 0 then table.insert(msgs,string.format("Sold items for a profit of %s.",nCTS(GetMoney() - endMoney))); end
				self:say(table.concat(msgs," ")); 
			end,
			PLAYER_MONEY_EVENT
		);
	end
end

-- Purchases as close to as possible the specified quantity of the item
function ReagentRestocker:purchaseItems(itemID, toBuy)
	local itemIndex
	for i=1,GetMerchantNumItems() do
		local merchantItem = GetMerchantItemLink(i)
		if merchantItem then
			if getIDFromItemLink(GetMerchantItemLink(i)) == itemID then
				itemIndex = i
				break
			end
		end
	end
	
	if not itemIndex then
		return
	end
	
	-- Purchases only allow for 1 stack at maximum per "click"; iteratively buy stacks, then buy the last bit if any remain
	local _, _, _, _, _, _, _, itemStackSize = self:safeGetItemInfo(itemID)
	local _, _, _, qtyPerPurchase = GetMerchantItemInfo(itemIndex)
	while toBuy > itemStackSize do
		BuyMerchantItem(itemIndex, floor(itemStackSize/qtyPerPurchase))
		toBuy = toBuy - itemStackSize
	end
	if toBuy > 0 then
		BuyMerchantItem(itemIndex, floor(toBuy/qtyPerPurchase))
	end
end	

-- Returns true if the item is able to be sold and if preferences dictate it should be; false otherwise
function ReagentRestocker:isToBeSold(itemID)
	local _, _, quality = self:safeGetItemInfo(itemID)
	if (ReagentRestockerDB.Options.AutoSellGrays and quality == 0) or (self:listType(itemID) == SELLING_TYPE) then
		return true
	else
		return false
	end
end

-- Returns true if the merchant sells items on the player's shopping list; false otherwise
function ReagentRestocker:isReagentVendor()
	for i=1,GetMerchantNumItems() do
		local merchantItem = GetMerchantItemLink(i)
		if merchantItem then
			if (self:listType(getIDFromItemLink(merchantItem)) == SHOPPING_TYPE) then
				return true
			end
		end
	end	
	return false
end

-- Repairs the character's equipment, if necessary; returns the cost and a report string
function ReagentRestocker:repair()
	if CanMerchantRepair() then
		local msg, cost = "", GetRepairAllCost()
		
		if ReagentRestockerDB.Options.RepairDiscount and not self:meetsDiscountRequirements() and cost > 0 then
			return 0, string.format("This vendor is not offering a %d%% or better discount; your items will not be repaired.",ReagentRestockerDB.Options.RequiredDiscount)
		end		
		
		-- Warn the player if he's not capable of making guild bank repairs
		if not CanGuildBankRepair() and ReagentRestockerDB.Options.UseGuildBankFunds then
			msg = "You are not authorized to make repairs via the guild bank.  "
		end
	
		-- Do the repairing
		if cost > 0 then 
			if CanGuildBankRepair() and ReagentRestockerDB.Options.UseGuildBankFunds then
				-- TODO: If the repair is only partially from the guild, the cost is not actually 0
				msg = msg .. string.format("Your gear has been repaired using the guild bank's funds, costing it %s.", nCTS(cost))
				cost = 0
				RepairAllItems(1)
			elseif GetRepairAllCost() <= GetMoney() then
				msg = msg .. string.format("Your gear has been repaired, costing you %s.", nCTS(cost))
				RepairAllItems()
			elseif GetRepairAllCost() > GetMoney() then
				msg = msg .. string.format("Insufficient funds to repair (%s required).", nCTS(cost))
				cost = 0
			end
		else
			msg = "You are already fully repaired."
			cost = 0
		end
		
		return cost, msg
	else
		return 0, "This merchant cannot repair gear."
	end
end

-- Buys the necessary reagents based on those that are currently in the player's inventory; returns the cost and a reporting string
function ReagentRestocker:buy()
	-- Keep track of what items will actually be bought
	local buyingList, cost, shoppingList = {}, 0, self:getOffsetList()
	
	-- Grab only the items that we need from the shopping list
	map(shoppingList, function(itemID,qty) if qty < 1 then shoppingList[itemID] = nil; end end)
	
	-- Look through all of the merchant's items
	for i=1,GetMerchantNumItems() do
		local merchantItem = GetMerchantItemLink(i)
		if merchantItem then
			local itemID = getIDFromItemLink(merchantItem)
			local numDesired = shoppingList[itemID]
			local _, _, itemPrice, qtyPerPurchase, itemQtyAvailable = GetMerchantItemInfo(i)
			if numDesired then
				local buyQty
				if itemQtyAvailable == -1 then -- Unlimited quantity
					buyQty = numDesired
				else -- Limited quantity; take at most the number that are available
					buyQty = min(numDesired, itemQtyAvailable)
				end
				-- Round down to quantity to an integer multiple of the quantity in which they can be purchased (some items can't be purchased 1 by 1)
				if ReagentRestockerDB.Options.NotOverstock then
					buyingList[itemID] = floor(buyQty/qtyPerPurchase)*qtyPerPurchase
				else
					buyingList[itemID] = ceil(buyQty/qtyPerPurchase)*qtyPerPurchase
				end
				cost = cost + (buyingList[itemID]/qtyPerPurchase) * itemPrice
			end
		end
	end
	
	-- Filter out useless purchases
	for k,v in pairs(buyingList) do
		if v == 0 then
			buyingList[k] = nil
		end
	end
	
	-- Is there actually anything to be purchased?
	if tcount(buyingList) == 0 then
		return 0, "Already stocked on this vendor's items."
	end	
	
	-- Make sure the player has enough money
	if cost > GetMoney() then
		return 0, string.format("Insufficient funds to purchase reagents (%s required).",nCTS(cost))
	end
	
	-- Make sure the vendor is offering the appropriate discount
	if not self:meetsDiscountRequirements() then
		return 0, string.format("This vendor is not offering a %d%% or better discount; no items will be purchased.",ReagentRestockerDB.Options.RequiredDiscount)
	end
		
	-- Purchase the items
	for itemID,qty in pairs(buyingList) do
		self:purchaseItems(itemID,qty)
	end
	
	-- Builds a readable list of strings and quantities
	local purchasedItemLinkList = {}
	map(buyingList,function(itemID,qty) local _, l = self:safeGetItemInfo(itemID); table.insert(purchasedItemLinkList,string.format("%d %s",qty,l)); end)
	
	-- Build the message to report
	return cost, string.format("Purchased %s, costing a total of %s.",table.concat(purchasedItemLinkList,", "),nCTS(cost))
end

-- Sells the appropriate item from the player's inventory; returns list of items sold
function ReagentRestocker:sell()
	local soldItemsInfo = {} -- {bagID, bagSlotID} list
	for _,bagID in pairs(getPlayerBagIDList()) do
		for bagSlotID=1,GetContainerNumSlots(bagID) do
			local itemLink = GetContainerItemLink(bagID,bagSlotID)
			if itemLink then		
				if self:isToBeSold(getIDFromItemLink(itemLink)) then
					local _, qty = GetContainerItemInfo(bagID, bagSlotID)
					table.insert(soldItemsInfo,{bagID,bagSlotID,getIDFromItemLink(itemLink),qty})
					UseContainerItem(bagID,bagSlotID)
				end
			end
		end
	end
	return soldItemsInfo
end

-- Returns true if the player has required reputation with current merchant window; false otherwise
function ReagentRestocker:meetsDiscountRequirements()
	-- Return true if the available discount is anywhere from required to 20%
	if self:getMerchantDiscount() >= ReagentRestockerDB.Options.RequiredDiscount then
		return true
	end
	return false
end

-- Returns the integer percentage that the merchant is offering
function ReagentRestocker:getMerchantDiscount()
	-- Helper: Returns true if removing the discount from the price results in more 0s than the original price
	local isDiscounted = function (price, discountPercent)
		-- Helper: Returns the number of 0s in the [decimal] number
		local numZeroes = function (num)
			local zeroCount = 0
			while floor(num/10) == num/10 and num > 9 do
				num = num/10
				zeroCount = zeroCount + 1
			end
			
			return zeroCount
		end
		
		return numZeroes(price) < numZeroes(floor(price/(1 - discountPercent/100) + 0.5))
	end

	local merchantPriceList = {}
	for i=1, GetMerchantNumItems() do
		local _, _, price = GetMerchantItemInfo(i)
		if price then
			table.insert(merchantPriceList,price)
		end
	end	

	-- Check every multiple of 5% from under the required discount (or lower) to 20%
	for discountPercent = floor(ReagentRestockerDB.Options.RequiredDiscount/5)*5, 20, 1 do
		local numDiscounted = 0
		for _,price in ipairs(merchantPriceList) do
			if isDiscounted(price, discountPercent) then
				numDiscounted = numDiscounted + 1
			end
		end
		
		if (0.4 < numDiscounted/#merchantPriceList) then
			return discountPercent
		end
	end
	
	-- At this point, can't assess discount; assume no discount
	return 0
end

--========--
-- Events --
--========--

function ReagentRestocker:OnInitialize()
	self:RegisterEvent(MERCHANT_SHOW_EVENT)	-- Buying and selling items
	self:RegisterEvent(MERCHANT_UPDATE_EVENT)	-- Adding items to selling list
	self:RegisterEvent(BANKFRAME_OPENED_EVENT)	-- Initiate bank transfers
	self:RegisterEvent(VARIABLES_LOADED_EVENT)	-- Initializing variables
	self:RegisterEvent(ITEM_LOCK_CHANGED_EVENT)	-- Queued events
	self:RegisterEvent(PLAYER_MONEY_EVENT)
	self:RegisterEvent(BAG_UPDATE_EVENT)
	
	-- Create an instance of waterfall
	waterfall:Register(
		moduleName,
		'aceOptions',Menu,
		'title','Reagent Restocker',
		'treeLevels',10,
		'hideTreeRoot',true,
		'colorR', 0.7, 'colorG', 0.7, 'colorB', 0.7
		)
	
	self:RegisterChatCommand({"/rr", "/reagentrestocker"}, function () ReagentRestockerDB.Options.UnusedNotification = false; waterfall:Open(moduleName); end)
end

-- Adds an action to action queue
function ReagentRestocker:queueAction(evaluator, action, eventID)
	table.insert(QueuedActions,{evaluator,action,eventID})
end

function ReagentRestocker:triggerAction(eventID)
	-- self:say(string.format("A [%s] event has been triggered.",tostring(eventID)))

	for i=#QueuedActions,1,-1 do
		-- Only run the action if the correct eventID is specified -or- if there is no event specified
		if not QueuedActions[i][3] or QueuedActions[i][3] == eventID then
			if QueuedActions[i][1]() then
				-- If the evaluator for an action is true, then its associated action is performed; then the action is deleted			
				local theAction = QueuedActions[i][2]
				table.remove(QueuedActions,i)
				theAction()
			end			
		end
	end
end

function ReagentRestocker:PLAYER_MONEY()
	self:triggerAction(PLAYER_MONEY_EVENT)
end

function ReagentRestocker:BAG_UPDATE()
	self:triggerAction(BAG_UPDATE_EVENT)
end

function ReagentRestocker:ITEM_LOCK_CHANGED(bagID, bagSlotID)
	self:triggerAction(ITEM_LOCK_CHANGED_EVENT)	
end

function ReagentRestocker:VARIABLES_LOADED()
	-- Initialize variables
	if not ReagentRestockerDB then
		ReagentRestockerDB = {}
		ReagentRestockerDB.Options = {}
		ReagentRestockerDB.Options.UnusedNotification = true
		ReagentRestockerDB.Options.AutoBuy = true
		ReagentRestockerDB.Options.AutoSell = true
		ReagentRestockerDB.Options.RequiredDiscount = 0
		ReagentRestockerDB.Items = {}
	end
	
	self:synchronizeOptionsTable()
	self:notifyPlayer()
end
--==========--
-- Bank/Bags --
--==========--

function ReagentRestocker:BANKFRAME_OPENED()
	self:triggerAction(BANKFRAME_OPENED_EVENT)
	
	if ReagentRestockerDB.Options.PullFromBank or ReagentRestockerDB.Options.OverstockToBank then
		if not self:lockTransaction(2) then
			self:say("You are attempting to begin too many transactions in a short time; ignoring ...")
			return
		end

		if self:recursiveMove(self:getOffsetList(bankOffset),getPlayerBagIDList(),getBankBagIDList()) then
			self:say("Working, please wait ...")
		end
	end
end

-- Returns true if the item in the specified slot is locked; false otherwise
function ReagentRestocker:isSlotUnlocked(bagID,bagSlotID)
	local _, _, isLocked = GetContainerItemInfo(bagID, bagSlotID)
	return not isLocked
end

-- Returns true if all of the items in the specified slot are unlocked; false otherwise
function ReagentRestocker:areSlotsUnlocked(bagIDSlotPairs)
	for i=1,#bagIDSlotPairs do
		if not self:isSlotUnlocked(bagIDSlotPairs[i][1],bagIDSlotPairs[i][2]) then
			return false
		end
	end

	return true
end

-- Returns the number of the specified item in the specified bag
function ReagentRestocker:countItemInBag(bagID,itemID)
	local count = 0
	for bagSlotID=1,GetContainerNumSlots(bagID) do
		local itemLink = GetContainerItemLink(bagID,bagSlotID)
		if itemLink then	
			if getIDFromItemLink(itemLink) == itemID then	
				local _, slotCount = GetContainerItemInfo(bagID,bagSlotID)
				count = count + slotCount
			end
		end
	end
	
	return count
end

-- Returns the number of the specified item in the specified bags
function ReagentRestocker:countItemInBags(bagIDList,itemID)
	local count = 0
	for _,bagID in pairs(bagIDList) do
		count = count + self:countItemInBag(bagID,itemID)
	end
	
	return count
end

-- Attempts to safely move an item from one slot to another; splitting often causes problems unless it is 100% necessary
function ReagentRestocker:safeContainerMove(fromBagID, fromBagSlotID, toBagID, toBagSlotID, qty)
	local _, sourceStackCount = GetContainerItemInfo(fromBagID,fromBagSlotID)
	local _, destStackCount = GetContainerItemInfo(toBagID,toBagSlotID)
	
	if not sourceStackCount then sourceStackCount = 0 end
	if not destStackCount then destStackCount = 0 end			
	
	local doMove = false
	if destStackCount > 0 then
		local _, _, _, _, _, _, _, destStackableCount = self:safeGetItemInfo(getIDFromItemLink(GetContainerItemLink(toBagID,toBagSlotID)))
		if destStackCount + qty >= destStackableCount then
			doMove = true
		end
	end
	
	ClearCursor()
	if sourceStackCount > qty and not doMove then
		SplitContainerItem(fromBagID,fromBagSlotID,qty)
	else
		PickupContainerItem(fromBagID,fromBagSlotID)
	end
	PickupContainerItem(toBagID,toBagSlotID)	
end

-- Attempts to move items from one baglist to another, based on shopping list
function ReagentRestocker:recursiveMove(startOffsetList, fromBags, toBags)
	-- Attempt to find an appropriate move
	local offsetList = self:getOffsetList(bankOffset)
	local fromBagID, fromBagSlotID, toBagID, toBagSlotID, qty, itemID = self:findNecessaryMove(offsetList,fromBags,toBags)	
	if fromBagID then -- If there is one, perform the move, and queue up another recursiveMove
		self:safeContainerMove(fromBagID, fromBagSlotID,toBagID, toBagSlotID, qty)
		local _, pSSC = GetContainerItemInfo(fromBagID,fromBagSlotID)
		local _, pDSC = GetContainerItemInfo(toBagID,toBagSlotID)
		self:queueAction(
			function()
				local _, sSC = GetContainerItemInfo(fromBagID,fromBagSlotID)
				local _, dSC = GetContainerItemInfo(toBagID,toBagSlotID)
				return self:areSlotsUnlocked({{fromBagID, fromBagSlotID},{toBagID, toBagSlotID}}) and pSSC ~= sSC and pDSC ~= dSC
			end,
			function() 
				self:recursiveMove(startOffsetList, fromBags, toBags)
			end
			)
		return true
	else -- If there isn't one, return false, and report our findings from movesThusFar
		local withdrawList, depositList, msgs = {}, {}, {} 
		for itemID, qty in pairs(tDiff(startOffsetList, offsetList)) do
			local _, itemLink = self:safeGetItemInfo(itemID)
			if qty > 0 then table.insert(depositList,string.format("%d %s",qty,itemLink)); else if qty < 0 then table.insert(withdrawList,string.format("%d %s",-1*qty,itemLink)); end end
		end		
		if #withdrawList == 0 and #depositList == 0 then
			self:say("No bank transactions are necessary.")
		else	
			if #withdrawList > 0 then 
				table.insert(msgs,string.format("Deposited %s.",table.concat(withdrawList,", ")))
			end
			if #depositList > 0 then
				table.insert(msgs,string.format("Withdrew %s.",table.concat(depositList,", ")))
			end
			self:say(table.concat(msgs,"  "))
		end
	end
end

-- Returns a necessary move; nil if there are none
function ReagentRestocker:findNecessaryMove(offsetList, fromBags, toBags)
	map(offsetList,function(itemID, qty) if qty == 0 then offsetList[itemID] = nil; end end)
	for itemID, qty in pairs(offsetList) do
		local lFromBags, lToBags = fromBags, toBags
		if qty < 0 then -- We are moving FROM the TOBAGS to TO the FROMBAGS, so switch 'em
			lToBags, lFromBags, qty = fromBags, toBags, -1*qty
		end
		local fromBagID, fromBagSlotID, fromQty = self:findOptimalItemsToMove(lToBags, itemID)
		if fromBagID then -- fromBagID will be nil if fOITM was unsuccessful
			local toBagID, toBagSlotID, toQty = self:findOptimalDestinationInBags(lFromBags, itemID)
			if toBagID then -- toBagID will be nil if fODIB was unsuccessful
				return fromBagID, fromBagSlotID, toBagID, toBagSlotID, min(fromQty, toQty, qty), itemID				
			end
		end
	end
	return nil
end

-- Returns the location and quantity of an item that should be moved; nil if there are none
function ReagentRestocker:findOptimalItemsToMove(bagIDList, itemID)
	-- Find the optimal bag to remove the item from (least of that item)
	local itemInBagCountList = {}
	map(bagIDList,function(_,bagID) local count = self:countItemInBag(bagID,itemID); if count > 0 then table.insert(itemInBagCountList,{bagID,count}); end end)
	table.sort(itemInBagCountList,	function(x, y) if (x[2] ~= y[2]) then return x[2] < y[2]; else return (x[1] > y[1]); end end)
	for _, data in pairs(itemInBagCountList) do
		local bagSlotItemCount = {}
		for bagSlotID=1,GetContainerNumSlots(data[1]) do
			local itemLink = GetContainerItemLink(data[1], bagSlotID)
			if itemLink then
				local _, itemCount = GetContainerItemInfo(data[1], bagSlotID)
				local sourceItemID = getIDFromItemLink(itemLink)
				if sourceItemID == itemID then
					table.insert(bagSlotItemCount,{bagSlotID,itemCount})
				end
			end
		end
		-- Find the slot with the lowest value and return it
		table.sort(bagSlotItemCount, function(x,y) return x[2] < y[2]; end)
		for i=1,#bagSlotItemCount do
			return data[1], bagSlotItemCount[i][1], bagSlotItemCount[i][2]
		end
	end	
	return nil
end

-- Returns the "optimal" slot in the bags for the item to be placed and the number that can be moved there; nil if impossible
function ReagentRestocker:findOptimalDestinationInBags(bagIDList, itemID)
	-- First, find the optimal bag to place the item in (the once with the most of it)
	local itemInBagCountList = {}
	map(bagIDList,function(_,bagID) table.insert(itemInBagCountList,{bagID,self:countItemInBag(bagID,itemID)}); end)
	table.sort(itemInBagCountList,	function(x, y) if (x[2] ~= y[2]) then return x[2] > y[2]; else return (x[1] < y[1]); end end)
	for _, data in pairs(itemInBagCountList) do
		-- TODO: We're mostly avoiding special bag types
		if data[2] > 0 or not self:isSpecialBagType(data[1]) then
			local bagID, bagSlotID, qty = self:findOptimalDestinationInBag(data[1],itemID)
			if bagID then -- bagID is nil if fODIB returns a negative result (no space)
				return data[1], bagSlotID, qty
			end
		end
	end
	return nil
end

-- Returns the "optimal" slot in the bag for the item to be placed and the number that can be moved there; nil if impossible
function ReagentRestocker:findOptimalDestinationInBag(bagID, sourceItemID)
	-- First, look for any stacks of the item that are not full
	local _, _, _, _, _, _, _, itemStackSize = self:safeGetItemInfo(sourceItemID)
	local slotStatusList, OPEN, CLOSED, SAME = {}, 0, -1, 1
	for bagSlotID=1,GetContainerNumSlots(bagID) do
		local itemLink = GetContainerItemLink(bagID, bagSlotID)
		if itemLink then
			local _, itemCount = GetContainerItemInfo(bagID, bagSlotID)
			local itemID = getIDFromItemLink(itemLink)
			if sourceItemID == itemID then
				slotStatusList[bagSlotID] = SAME
				if itemCount < itemStackSize then
					return bagID, bagSlotID, itemStackSize - itemCount
				end
			else
				slotStatusList[bagSlotID] = CLOSED
			end
		else
			slotStatusList[bagSlotID] = OPEN
		end
	end
	
	-- No un-full stacks were found; look for a pattern in the full ones
	local emptySlotList, sameItemSlotList, hopSizeList, bestDiffVal = {}, {}, {}, 1
	map(slotStatusList,function(slotID, status) if status == SAME then table.insert(sameItemSlotList,slotID); end end)
	map(slotStatusList,function(slotID, status) if status == OPEN then table.insert(emptySlotList,slotID); end end)
	
	-- If there are no empty slots, return nil
	if #emptySlotList == 0 then
		return
	end
	
	if #sameItemSlotList > 1 then
		-- We have more than 1 stack of the item in the bag; find the modal difference in slot IDs
		for i=1,#sameItemSlotList-1 do
			local diff = abs(sameItemSlotList[i+1] - sameItemSlotList[i])
			if hopSizeList[diff] then
				hopSizeList[diff] = hopSizeList[diff] + 1
			else
				hopSizeList[diff] = 1	
			end
		end
		local bestDiffCount = 0
		map(hopSizeList, function (diffVal, diffCount) if diffCount > bestDiffCount then bestDiffVal = diffVal; bestDiffCount = diffCount; end end)
	end
	
	-- Look in each slot "diff" away from the slot in which the same item exists; if it is available, use it
	for _, slot in pairs(sameItemSlotList) do
		if inT(emptySlotList,slot-bestDiffVal) then
			return bagID, slot-bestDiffVal, itemStackSize
		elseif inT(emptySlotList,slot+bestDiffVal) then
			return bagID, slot+bestDiffVal, itemStackSize		
		end
	end
	
	-- No good places to put the item; just return the [at least] one we know exists
	return bagID, emptySlotList[1], itemStackSize
end

-- Returns true if the bag is of a special bag type; false otherwise
function ReagentRestocker:isSpecialBagType(bagID)
	-- Every item can go in the backpack (0) or bank slots (-1)
	if bagID == 0 or bagID == -1 then
		return false
	end

	local inventoryID = ContainerIDToInventoryID(bagID)
	local itemLink = GetInventoryItemLink("player",inventoryID)

	-- If there's no bag, it's not compatible
	if not itemLink then
		return true
	end
	
	local _, _, _, _, _, _, itemSubType = self:safeGetItemInfo(getIDFromItemLink(itemLink))
	
	-- No way to check the type without localizing; if the name is long, it's a special bag, so NO
	if strlen(itemSubType) < 8 then	
		return false
	else		
		return true
	end
end
