-- RaidSnapshot script
-- Copyright 2008 Breezze (EU-Hellscream) breezzedotwowatgmaildotcom

-- Usage:
-- /rss <command>
-- or /raidsnapshot <command>
--
-- Commands:
-- /rss save [<id>]
--   Stores the current configuration. e.g. "/rss restore" or "/rss store tactic1"
--
-- /rss restore [<id>]
--   Restores the raid configuration. e.g. "/rss restore" or "/rss restore tactic1"
--
-- /rss list
--   Lists the id's of all snapshots.
--
-- /rss view [<id>]
--   View the contents of the snapshot (player names per group)
--
-- /rss remove <id>
--   Removes a snapshot. e.g. "/rss remove tactic1"
--
-- /rss clear
--   Removes all stored raid configurations. Use with caution.
--
-- /rss bgautosave
--   Toggles auto restoring the raid in battlegrounds.
--




--
--
-- Addon initialization.
--
--

RaidSnapshot = LibStub( "AceAddon-3.0" ):NewAddon( "RaidSnapshot", "AceConsole-3.0", "AceHook-3.0", "AceEvent-3.0" )

local options = {
    name = "RaidSnapshot",
    handler = RaidSnapshot,
    type = "group",
    args = {
        save = {
            type = "input",
            name = "save",
            desc = "Save the current raid configuration.",
            usage = "<id>",
            set = function( info, value )
                      local id = RaidSnapshot:makeSnapshotId( value )

                      RaidSnapshot:storeCurrent( id )
                      RaidSnapshot:Print( "Saved snapshot as \"" .. id .. "\"." )
                  end,
        },

        restore = {
            type = "input",
            name = "restore",
            desc = "Restore a snapshot.",
            usage = "<id>",
            set = function( info, value )
                      local id = RaidSnapshot:makeSnapshotId( value )

                      if ( RaidSnapshot:verifySnapshotExists( id ) ) then
                          RaidSnapshot:restore( id )
                          RaidSnapshot:Print( "Restored snapshot \"" .. id .. "\"." )
                      end
                  end,
        },


        list = {
            type = "input",
            name = "list",
            desc = "View a list of all snapshots.",
            usage = "",
            set = function( info, value )
                      --RaidSnapshot:Print( "Listing all snapshots: " )
                      RaidSnapshot:listAllIds( )
                  end,
        },


        view = {
            type = "input",
            name = "view",
            desc = "View the contents of a snapshot.",
            usage = "<id>",
            set = function( info, value )
                      local id = RaidSnapshot:makeSnapshotId( value )

                      if ( RaidSnapshot:verifySnapshotExists( id ) ) then
                          RaidSnapshot:Print( "Contents of snapshot \"" .. id .. "\":" )
                          RaidSnapshot:printSnapshot( RaidSnapshot:getSnapshot( id ) )
                      end
                  end,
        },


        remove = {
            type = "input",
            name = "remove",
            desc = "Remove a snapshot.",
            usage = "<id>",
            set = function( info, value )
                      local id = RaidSnapshot:makeSnapshotId( value )

                      if ( RaidSnapshot:verifySnapshotExists( id ) ) then
                          RaidSnapshot:removeSnapshot( id )
                          RaidSnapshot:Print( "Snapshot \"" .. id .. "\" removed." )
                      end
                  end,
        },


        copy = {
            type = "input",
            name = "copy",
            desc = "Copy a snapshot.",
            usage = "<id> <dest>",
            set = function( info, value )
                      local pattern = "(.+)%s(.+)"
                      local srcValue, destValue = string.match( value, pattern )

                      if ( not ( srcValue and destValue ) ) then
                          RaidSnapshot:Print( "Usage: <src> <dest>" )
                          return
                      end


                      local srcId = RaidSnapshot:makeSnapshotId( srcValue )
                      local destId = RaidSnapshot:makeSnapshotId( destValue )

                      if ( RaidSnapshot:verifySnapshotExists( srcId )
                          and RaidSnapshot:verifySnapshotExists( srcId ) ) then

                          if ( RaidSnapshot:copySnapshot( srcId, destId ) ) then
                              RaidSnapshot:Print( "Snapshot \"" .. srcId .. "\" copied to \"" .. destId .. "\"" )
                          else
                              RaidSnapshot:Print( "Unable to rename " .. srcId .. "\" to \"" .. destId .. "\"" )
                          end
                      end
                  end,
        },

        rename = {
            type = "input",
            name = "rename",
            desc = "Copy a snapshot.",
            usage = "<oldname> <newname>",
            set = function( info, value )
                      local pattern = "(.+)%s(.+)"
                      local srcValue, destValue = string.match( value, pattern )

                      if ( not ( srcValue and destValue ) ) then
                          RaidSnapshot:Print( "Usage: <oldname> <newname>" )
                          return
                      end


                      local srcId = RaidSnapshot:makeSnapshotId( srcValue )
                      local destId = RaidSnapshot:makeSnapshotId( destValue )

                      if ( RaidSnapshot:verifySnapshotExists( srcId )
                          and RaidSnapshot:verifySnapshotExists( srcId ) ) then

                          if ( RaidSnapshot:renameSnapshot( srcId, destId ) ) then
                              RaidSnapshot:Print( "Snapshot \"" .. srcId .. "\" renamed to \"" .. destId .. "\"" )
                          else
                              RaidSnapshot:Print( "Unable to rename " .. srcId .. "\" to \"" .. destId .. "\"" )
                          end
                      end
                  end,
        },


        clear = {
            type = "input",
            name = "clear",
            desc = "Remove all snapshots; use with caution.",
            usage = "<id>",
            set = function( info, value )
                      RaidSnapshot:clearAll( )
                      RaidSnapshot:Print( "Snapshots are cleared" )
                  end,
        },



        bgauto = {
            name = "Battlegrounds auto restore options",
            type = "group",

            args = {
                enable = {
                    type = "toggle",
                    name = "enable",
                    desc = "Enable/disable autorestore in battlegrounds.",
                    set = function( info, value )
                              RaidSnapshot:setAutoRestore( value )

                              if ( RaidSnapshot:isAutoRestore( ) ) then
                                  RaidSnapshot:enableAutoRestore( )
                                  RaidSnapshot:Print( "Autorestore enabled" )
                              else
                                  RaidSnapshot:disableAutoRestore( )
                                  RaidSnapshot:Print( "Autorestore disabled" )
                              end
                          end,
                    get = function( info )
                              return RaidSnapshot:isAutoRestore( )
                          end,
                },
            },
        },
    },
}



function RaidSnapshot:OnInitialize( )
    LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "RaidSnapshot", options, { "raidsnapshot", "rss" } )
    self.rosterlib = AceLibrary( "Roster-2.1" )


    -- Initialize DB
    local AceDB = LibStub( "AceDB-3.0" )
    self.db = AceDB:New( "RaidSnapshotDB" )

    if ( nil == self.db.profile.snapshots ) then
        self.db.profile.snapshots = { }
    end

    self.MAX_SUBGROUP_SIZE = 5

    self.isDebug = false

    self:initializeAutoRestore( )

end



function RaidSnapshot:OnEnable( )
    if ( self:isAutoRestore( ) ) then
        self:enableAutoRestore( )
    end

    self:Print( "RaidSnapshot enabled." )
end



function RaidSnapshot:OnDisable( )
    self:disableAutoRestore( )
    
    self:Print( "RaidSnapshot disabled." )
end



