local L = AceLibrary("AceLocale-2.2"):new("RaidMarshal")
local BabbleClass = AceLibrary("Babble-Class-2.2")
local Tourist = AceLibrary("Tourist-2.0")
local Dewdrop = AceLibrary("Dewdrop-2.0")

RM_TAB_CHARACTERS = L["Characters"]
RM_TAB_RAIDS = L["Raids"]
RM_TITLE = L["Raid Marshal"]
RM_NAME = L["Name"]
RM_SCORE = L["Score"]
RM_ATTENDANCE = L["Attnd"]
RM_CLASS = L["Class"]
RM_ROLE = L["Role"]
RM_ZONE = L["Zone"]
RM_DATE = L["Date"]
RM_DURATION = L["Duration"]
RM_LOOTS = L["Loots"]

RMBrowser = {}

RMBrowser.CHAR_COL_NAME = 1
RMBrowser.CHAR_COL_SCORE = 2
RMBrowser.CHAR_COL_CLASS = 3
RMBrowser.CHAR_COL_ROLE = 4

RMBrowser.CHAR_COMPARATORS = {
	NAME = function(a,b) return a.name < b.name end,
	NAME_REVERSE = function(a,b) return a.name > b.name end,
	SCORE = function(a,b) return RMRaid:GetNeedScore(a) > RMRaid:GetNeedScore(b) end,
	SCORE_REVERSE = function(a,b) return RMRaid:GetNeedScore(a) < RMRaid:GetNeedScore(b) end,
	ATTENDANCE = function(a,b) return RMRaid:GetAttendance(a) > RMRaid:GetAttendance(b) end,
	ATTENDANCE_REVERSE = function(a,b) return RMRaid:GetAttendance(a) < RMRaid:GetAttendance(b) end,		
	CLASS = function(a,b) return a.className < b.className end,
	CLASS_REVERSE = function(a,b) return a.className > b.className end,
	ROLE = function(a,b)
		local atest, btest
		if not a.role then atest="" else atest = a.role end
		if not b.role then btest="" else btest = b.role end 
		return atest < btest 
	end,
	ROLE_REVERSE = function(a,b)
		local atest, btest
		if not a.role then atest="" else atest = a.role end
		if not b.role then btest="" else btest = b.role end 
		return atest > btest 
	end,
}

RMBrowser.RAID_COL_ZONE = 1
RMBrowser.RAID_COL_DATE = 2
RMBrowser.RAID_COL_DURATION = 3
RMBrowser.RAID_COL_LOOTS = 4

RMBrowser.RAID_COMPARATORS = {
	ZONE = function(a,b) return a.zoneName < b.zoneName end,
	ZONE_REVERSE = function(a,b) return a.zoneName > b.zoneName end,	
	DATE = function(a,b) return a.lastUpdated < b.lastUpdated end,
	DATE_REVERSE = function(a,b) return a.lastUpdated > b.lastUpdated end,
	DURATION = function(a,b) return a.duration < b.duration end,
	DURATION_REVERSE = function(a,b) return a.duration > b.duration end,	
	LOOTS = function(a,b) return #a.lootTable < #b.lootTable end,
	LOOTS_REVERSE = function(a,b) return #a.lootTable > #b.lootTable end
}

function RMBrowser.SetTab(parent, id)
  PanelTemplates_SetTab(parent, id)
  for i=1,parent.numTabs do
  	local pageName = parent:GetName().."Page"..i
    local page = getglobal(parent:GetName().."Page"..i)
    if i ~= id then      
      page:Hide()
    else   	  
      page:Show()
    end
  end
end

function RMBrowser.GetFilteredRaidTable()
	local raidTable = RMRaid:LoadAll()
	local newTable = {}	
	for i, raid in ipairs(raidTable) do
		local isDeleted = raid.isDeleted ~= nil and raid.isDeleted == true		
		-- check online if show offline is not checked		
		if (not isDeleted or RaidMarshal.db.profile.showDeleted) and 
			((RaidMarshal.db.profile.show10 and RMRaid:GetSize(raid)=="10") or
			(RaidMarshal.db.profile.show25 and RMRaid:GetSize(raid)=="25"))  			
			then																			
				tinsert(newTable, raid)
		else
			--RaidMarshal:Print("Filtered out raid key="..raid.key.."; zone="..raid.zoneName)			
		end						
	end
	return newTable
end

function RMBrowser.GetCharacterTotals(charTable)
	local players = 0
	local tanks = 0
	local healers = 0
	local dps = 0
	local nonRaiders = 0
	local playersTable = {}	
	for i, char in ipairs(charTable) do
		if not char.isDeleted then
			local name = RMCharacter:GetMainName(char)
			local attendance = RMRaid:GetAttendance(char)
			if not playersTable[name] and attendance > 0 then
				players = players + 1
				playersTable[name] = name
				if not char.role then nonRaiders = nonRaiders + 1				
				elseif char.role == RMCharacter.ROLE_TANK then tanks = tanks + 1				
				elseif char.role == RMCharacter.ROLE_HEALER then healers = healers + 1
				elseif char.role == RMCharacter.ROLE_DPS then dps = dps + 1 end
			end						
		end		
	end 
	return players, tanks, healers, dps, nonRaiders	
end

function RMBrowser.GetRaidTotals(raidTable)
	local needLoot = 0	
	local greedLoot = 0
	local duration = 0
	local participants = 0
	local onlinePlayers = 0	
	local participantTable = {}
	local onlinePlayersTable = {}
	local raids = 0 
	for i, raid in ipairs(raidTable) do
		if not raid.isDeleted then
			raids = raids + 1
			for i, loot in ipairs(raid.lootTable) do
				if loot.isNeed then
					needLoot = needLoot + 1
				else
					greedLoot = greedLoot + 1
				end
			end
			duration = duration + raid.duration
			for charName, dur in pairs(raid.participantDurationTable) do
				local name = RMCharacter:GetMainName(charName)
				if not participantTable[name] then
					participantTable[name] = name
					participants = participants + 1
				end
				if not onlinePlayersTable[name] then
					onlinePlayersTable[name] = name
					onlinePlayers = onlinePlayers + 1
				end							
			end
			for charName, dur in pairs(raid.benchDurationTable) do
				local name = RMCharacter:GetMainName(charName)
				if not onlinePlayersTable[name] then
					onlinePlayersTable[name] = name
					onlinePlayers = onlinePlayers + 1
				end											
			end
		end		
	end 
	return raids, duration, needLoot, greedLoot, participants, onlinePlayers	
end

function RMBrowser.UpdateRaidTable()					
  	if not RMBrowserFrame:IsShown() then 
  		return 
  	end
	  	    		
	local raidTable = RMBrowser.GetFilteredRaidTable()		  
	sort(raidTable, RMBrowser.RAID_COMPARATORS[RaidMarshal.db.profile.raidComparator])	 	
  	local scrollbar_shown = FauxScrollFrame_Update(RaidScrollFrame, #raidTable, 15, 16)--, "EPGPListingEntry", 298, 330)
  	if (scrollbar_shown) then
    	RMZoneColumnHeader:SetWidth(111)
  	else
    	RMZoneColumnHeader:SetWidth(131)
  	end

  	for i=1,15 do
	    local j = i + FauxScrollFrame_GetOffset(RaidScrollFrame)
	    local row = getglobal("RaidRow"..i)
	    if j <= #raidTable then
	    	local raid = raidTable[j]								
			row.raid = raid	      		      							
      		getglobal(row:GetName().."Zone"):SetText(raid.zoneName)
			if not raid.endTime then
				-- flag active raids as orange on list
				getglobal(row:GetName().."Zone"):SetTextColor(BabbleClass:GetColor("Druid"))
			elseif raid.isDeleted then
				-- if deleted set row color to gray
				getglobal(row:GetName().."Zone"):SetTextColor(0.5, 0.5, 0.5)				
			else
				getglobal(row:GetName().."Zone"):SetTextColor(BabbleClass:GetColor("Hunter"))
			end
			
			if raid.isDeleted then
				getglobal(row:GetName().."Date"):SetTextColor(0.5, 0.5, 0.5)
				getglobal(row:GetName().."Duration"):SetTextColor(0.5, 0.5, 0.5)
				getglobal(row:GetName().."Loots"):SetTextColor(0.5, 0.5, 0.5)
	    	else
				getglobal(row:GetName().."Date"):SetTextColor(1.0, 1.0, 1.0)
				getglobal(row:GetName().."Duration"):SetTextColor(1.0, 1.0, 1.0)
				getglobal(row:GetName().."Loots"):SetTextColor(1.0, 1.0, 1.0)
	    	end
				    		
      		if scrollbar_shown then
        		getglobal(row:GetName().."Zone"):SetWidth(92)
      		else
        		getglobal(row:GetName().."Zone"):SetWidth(112)
      		end
	      	getglobal(row:GetName().."Date"):SetText(date("%m/%d/%y",raid.lastUpdated))										
	      	--getglobal(row:GetName().."EP"):SetAlpha(backend:IsBelowThreshold(EP) and 0.5 or 1.0)
	      	getglobal(row:GetName().."Duration"):SetText(RMBrowser.FormatDuration(raid.duration))								
			getglobal(row:GetName().."Loots"):SetText(#raid.lootTable)			
	      	row:Show()
		else
	      row:Hide()
    	end
	end
	local raids, duration, needLoot, greedLoot, participants, onlinePlayers	= RMBrowser.GetRaidTotals(raidTable)		
	RMRaidStatusLabel:SetText("Raids: "..raids.."    Need Loot: "..needLoot.."    Greed Loot: "..greedLoot.."\nParticipants: "..participants.."    Online Players: "..onlinePlayers.."\nTotal Raiding Duration: "..RMBrowser.FormatDuration(duration))
end

function RMBrowser.FormatDuration(duration)
	 -- units / 86400 
	local days = math.floor(duration / 86400)
	-- units / 3600 
	local hours = math.floor((duration - (days * 86400)) / 3600)   
	-- units / 60
	local minutes = math.floor((duration - ( (days * 86400) + (hours * 3600)) )/ 60)
	-- same unit
	local durationString = minutes .. "m"
	if hours > 0 then
		durationString = hours .. "h ".. durationString
	end
	if days > 0 then
			durationString = days .. "d ".. durationString
	end 	
	return durationString 	
end


function RMBrowser.GetFilteredCharacterTable(charTable)
	local newTable = {}
	-- lower the case of the search text to make the search case insensitive	
	local searchText = strlower(RMCharacterSearchBox:GetText())			
	for i, char in ipairs(charTable) do							
		-- RaidMarshal:Print("showAlts="..RaidMarshal.db.profile.showAlts.."; mainName="..char.mainName.."; online="..RMCharacter.onlineMap[char.name])
		-- check online if show offline is not checked
		if (RaidMarshal.db.profile.showOffline or RMCharacter.onlineMap[char.name]) and
			(RaidMarshal.db.profile.showNonRaiders or char.role) and					
			(RaidMarshal.db.profile.showAlts or not char.mainName) and			
			(searchText=="search" 
			or searchText==""
			or searchText == nil
			or string.find(strlower(char.name), searchText, 1, true)
			or string.find(strlower(char.className), searchText, 1, true)
			or (char.role and string.find(strlower(char.role), searchText, 1, true))) then										
				tinsert(newTable, char)								
		end		
	end
	return newTable
end

function RMBrowser.UpdateCharacterTable()					
  	if not RMBrowserFrame:IsShown() then 
  		return 
  	end
	  	  
  	local charTable = RMCharacter:LoadAll()	
	charTable = RMBrowser.GetFilteredCharacterTable(charTable)		 
	sort(charTable, RMBrowser.CHAR_COMPARATORS[RaidMarshal.db.profile.characterComparator])	 	
  	local scrollbar_shown = FauxScrollFrame_Update(CharacterScrollFrame, #charTable, 15, 16)--, "EPGPListingEntry", 298, 330)
  	if (scrollbar_shown) then
    	RMNameColumnHeader:SetWidth(111)
  	else
    	RMNameColumnHeader:SetWidth(131)
  	end

  	for i=1,15 do
	    local j = i + FauxScrollFrame_GetOffset(CharacterScrollFrame)
	    local row = getglobal("CharRow"..i)
	    if j <= #charTable then
	    	local char = charTable[j]      		      		
			row.char = char						
      		getglobal(row:GetName().."Name"):SetText(RMBrowser.GetNameWithMain(char))      
      		if scrollbar_shown then
        		getglobal(row:GetName().."Name"):SetWidth(92)
      		else
        		getglobal(row:GetName().."Name"):SetWidth(112)
      		end
			getglobal(row:GetName().."Attendance"):SetText(RMRaid:GetAttendance(char))
	      	getglobal(row:GetName().."Score"):SetText(RMRaid:GetNeedScore(char))
	      	--getglobal(row:GetName().."EP"):SetAlpha(backend:IsBelowThreshold(EP) and 0.5 or 1.0)
	      	getglobal(row:GetName().."Class"):SetText(char.className)
		  	getglobal(row:GetName().."Class"):SetTextColor(BabbleClass:GetColor(char.className))			     
	      	getglobal(row:GetName().."Role"):SetText(char.role)
			if role then      			
				getglobal(row:GetName().."Role"):SetTextColor(RMBrowser.GetRoleColor(char.role))
			end			
	      	row:Show()
		else
	      row:Hide()
    	end
	end	
	local players, tanks, healers, dps, nonRaiders = RMBrowser.GetCharacterTotals(charTable)		
	RMCharacterStatusLabel:SetText("Tanks: "..tanks.."    Healers: "..healers.."    DPS: "..dps.."    Non Raiders: "..nonRaiders.."\nTotal Attending Players: "..players)
end

function RMBrowser.BuildCharacterAltTable(character)
	if character.mainName then
		return nil
	end
	local altTable = {
		name = "Alts",
		order=20,
		type = "group",
		desc = "Alternate characters of " .. character.name,
		args = {			
			newAlt = {
				name = "Add Alt",
				type = "text",
				disabled = not RMCharacter:CanEdit(character.name), 
				usage = "Add a new alt",
				desc = "Map an alternate character for " .. character.name,
				get = function() return "" end,
				set = function(altName)				
					local altCharacter = RMCharacter:Load(altName)
					if not altCharacter then
						message("You can only add characters that are already in the database!")
						return
					end 	
					RMCharacter:MapAlt(altCharacter, character)
					RMBrowser.UpdateCharacterTable()
				end,
				order = 2
			}					
		},                            
		order = 30
	}
	
	local altNames = RMCharacter:GetAltNames(character.name)
	if altNames then
		for key, altName in pairs(altNames) do
			local altCharacter = RMCharacter:Load(altName)
			--RaidMarshal:Print("Adding menu line for "..altName)
			tinsert(altTable.args, {
				name = RMBrowser.GetNameWithColor(altCharacter),
				order = 1,
				type = "execute",
				func = function() message("clicked on alt") end,
				desc = "Alternate characters of " .. character.name							
			})
		end
	end
	-- args = RMBrowser.BuildCharacterTable(altCharacter)
	
	return altTable	
end

function RMBrowser.BuildRaidLootTable(raid)
	-- return nil if no loots for this raid
	if not raid.lootTable or #raid.lootTable == 0 then
		return nil
	end
	local lootTable = {
		name = "Loot",
		type = "group",
		desc = "All items master looted during this raid",
		order = 25,
		args = {}		
	}
	-- otherwise build loot table by lootedFrom
	for i, loot in ipairs(raid.lootTable) do
		if not lootTable.args[loot.lootedFrom] then		
			lootTable.args[loot.lootedFrom] = {
				name = loot.lootedFrom,
				type = "group",
				order = 1,
				desc = "All items looted from "..loot.lootedFrom,
				args = {} 				
			}
		end		
		local lootType = "Greed"
		
		local itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount,
					itemEquipLoc, itemTexture 
		if loot.itemId then 				
			itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount,
				itemEquipLoc, itemTexture = GetItemInfo()
		else
			itemLink = loot.itemString
		end
		if loot.isNeed then lootType = "Need" end
		tinsert(lootTable.args[loot.lootedFrom].args, {
			name = itemLink,
			type = "group",
			order = 1,
			desc = "Shows further loot details for "..loot.itemString,
			args = {
				NameLine = {
					name = loot.itemString,
					type = "header",
					desc = "Item Name",
					order = 1
				},				
				RecipientLine = {
					name = "Recipient: "..loot.recipient,
					type = "header",
					desc = "The person who received this item",
					order = 2
				},
				LootTypeLine = {
					name = "Priority: "..lootType,
					type = "header",
					desc = "Loot priority used, i.e. need or greed",
					order = 3
				},	
				TimeLine = {					
					name = "Time: "..date("%m/%d/%y %I:%M %p",loot.atTime),
					type = "header",
					desc = "Time at which the item was looted",
					order = 4
				},
				SetItemNameLine = {
					name="Set Item Name",
					type="text",
					desc="Set the item name",
					usage="Set the item name",
					get = function() return loot.itemString end,
					set = function(itemString) loot.itemString = itemString end,
					hidden = not RMRaid:CanEdit(raid),
					order = 5,
					
				},				
				SetLootedFromLine = {
					name="Set Looted From",
					type="text",
					desc="Set who the item was looted from",
					usage="Set who the item was looted from",
					get = function() return loot.lootedFrom end,
					set = function(lootedFrom) loot.lootedFrom = lootedFrom end,
					hidden = not RMRaid:CanEdit(raid),
					order = 6,
					
				},
				DeleteLine = {					
					name = "Delete",
					type = "execute",
					confirm = true,
					func = function()
						tremove(raid.lootTable, i)
						RMRaid:Save(raid)
					end,
					desc = "Delete this loot record from the raid",
					hidden = not RMRaid:CanEdit(raid),
					order = 10
				}												
			}
		})				
	end
	return lootTable		 
end

function RMBrowser.BuildCharacterLootTable(isNeed, charName)		
	local lootMap
	if isNeed then
		lootMap = RMRaid.needLootMap
	else
		lootMap = RMRaid.greedLootMap
	end
	local lootTable = {}	
	 
	if lootMap[charName] then
		--RaidMarshal:Print("Found loot for char")		
		for i, loot in ipairs(lootMap[charName]) do
			--RaidMarshal:Print("Building entry for loot "..loot.itemString)
			
			if loot.itemId then 				
				itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount,
					itemEquipLoc, itemTexture = GetItemInfo()
			else
				itemLink = loot.itemString
			end						
			tinsert(lootTable, {
				name = itemLink,
				type = "group",
				desc = loot.itemString,
				order = loot.atTime,
				args = {
					ItemName = {
						name = loot.itemString,
						type = "header",
						desc = loot.itemString,
						order = 1
					},
					LootedFrom = {
						name = "Looted From: "..loot.lootedFrom,
						type = "header",
						desc = loot.itemString,
						order = 2						
					},
					AtTime = {
						name = "Time: "..date("%m/%d/%y %I:%M %p",loot.atTime),
						type = "header",
						desc = loot.itemString,
						order = 3								
					}					
				}
			})			
		end
		return lootTable
	else
		--RaidMarshal:Print("No loot found for char")
		return nil
	end 
end

function RMBrowser.BuildCharacterTable(character)
	local characterTable = {
		type='group',				
		args = { 				
			NameLine = {
				type="header",
				name=RMBrowser.GetNameWithColor(character),
				order = 10
			},
			AttendanceLine = {
				type="header",
				name="Attendance: "..RMRaid:GetAttendance(character).."%",
				order = 11
			},						
			NeedScoreLine = {
				type="header",
				name="Need Score: "..RMRaid:GetNeedScore(character),
				order = 12				
			},
			GreedScoreLine = {
				type="header",
				name="Greed Score: "..RMRaid:GetGreedScore(character),
				order = 13				
			},						
			LastUpdatedLine = {
				type="header",
				name="Updated: "..date("%m/%d/%y %I:%M %p",character.lastUpdated),
				order = 14
			},																							
			RoleGroup = {
				name="Role",
				type="group",
				desc="Role in a raid group",
				args = {
					tank = {
						name=RMBrowser.GetRoleWithColor("Tank"),
						desc="Tank",
						type="toggle",
						order = 10,                            
						set=function()
							if not RMCharacter:CanEdit(character.name) then RMBrowser.WarnPermissions() end
							if character.role ~= RMCharacter.ROLE_TANK then
								character.role = RMCharacter.ROLE_TANK
							else
								character.role = nil
							end
							RMCharacter:Save(character)
							RMBrowser:UpdateCharacterTable()									
						end,
						get=function() return character.role == RMCharacter.ROLE_TANK end
					},
					dps = {
						name=RMBrowser.GetRoleWithColor("DPS"),
						type="toggle",
						desc="DPS",
						order = 20,
						set=function()
							if not RMCharacter:CanEdit(character.name) then RMBrowser.WarnPermissions() end
							if character.role ~= RMCharacter.ROLE_DPS then
								character.role = RMCharacter.ROLE_DPS
							else
								character.role = nil
							end
							RMCharacter:Save(character)		
							RMBrowser:UpdateCharacterTable()							
						end,
						get=function() return character.role == RMCharacter.ROLE_DPS end
					},                        
					healer = {
						name=RMBrowser.GetRoleWithColor("Healer"),
						type="toggle",
						desc="Healer",
						order = 30,
						set=function()
							if not RMCharacter:CanEdit(character.name) then RMBrowser.WarnPermissions() end
							if character.role ~= RMCharacter.ROLE_HEALER then
								character.role = RMCharacter.ROLE_HEALER
							else
								character.role = nil
							end
							RMCharacter:Save(character)		
							RMBrowser:UpdateCharacterTable()							
						end,
						get=function() return character.role == RMCharacter.ROLE_HEALER end
					}
				},
				order = 20
			},
			AltsLine = RMBrowser.BuildCharacterAltTable(character),
			setMain = {
				name = "Set Main",
				type = "text",
				usage = "Set main",
				desc = "Map an alternate character for " .. character.name,
				get = function() return character.mainName end,
				set = function(mainName)
					if not RMCharacter:CanEditAlts(character.name) then RMBrowser.WarnPermissions() end										  
					local mainCharacter = RMCharacter:Load(mainName)
					if not mainCharacter and mainName ~="" and mainName ~= nil then
						message("You can only add characters that are already in the database!")
						return
					else 					 											
						RMCharacter:MapAlt(character, mainCharacter)				
						RMBrowser.UpdateCharacterTable() 
					end
				end,
				order = 30
			},				
			removePlayer = {
				name = "Delete",
				type = "execute",
				desc = "Delete a player from the database",
				disabled = not RMCharacter:CanEdit(character.name),
				confirm = true,
				order = 40,
				func = function() 
					RaidMarshal.db.profile.characters[character.name] = nil
					RMCharacter:BuildAltMap()
					RMBrowser.UpdateCharacterTable() 
				end
			},
			CloseLine = {
				name = "Close",
				type = "execute",
				desc = "Closes this menu",
				confirm = false,
				order = 50,
				func = function() 
					Dewdrop:Close() 
				end
			}																						
		}  								
	}
	-- add loot tables if any loots by this character
	local needLootsTable = RMBrowser.BuildCharacterLootTable(true, character.name)	
	local greedLootsTable = RMBrowser.BuildCharacterLootTable(false, character.name)
	
	if needLootsTable then
		--RaidMarshal:Print("Adding need lootTable")
		characterTable.args.NeedLootsLine = {
			name = "Need Loots",
			type = "group",
			order = 16,
			desc = "List of need loots for "..character.name,
			args = needLootsTable			
		}
	end
	
	if greedLootsTable then
		--RaidMarshal:Print("Adding greed lootTable")
		characterTable.args.GreedLootsLine = {
			name = "Greed Loots",
			type = "group",
			order = 17,
			desc = "List of greed loots for "..character.name,
			args = greedLootsTable			
		}
	end	
		
	return characterTable
end

function RMBrowser.OpenCharacterDewdrop(charRow)						
	Dewdrop:Open(
		charRow,
		'point', 'RIGHT',
		'children', function()
			Dewdrop:FeedAceOptionsTable(RMBrowser.BuildCharacterTable(charRow.char))
		end
	)
end

function RMBrowser.BuildBenchTable(raid)	
	local benchTable = {
		name = "Bench",
		order=40,
		type = "group",
		desc = "Bench",
		args = {
			AddLine = {
				name = "Add minutes to all",
				desc = "Add minutes to everyone on the bench",
				type = "text",
				usage =  "Adds minutes to everyone on the bench",
				hidden = not RMRaid:CanEdit(raid),
				get = function() return "0" end,		
				order = 2,	
				set = function(add)
					if not tonumber(add) or tonumber(add) < 0 then return end										
					for charName, duration in pairs(raid.benchDurationTable) do
						if (duration + (add * 60)) > raid.duration then
							raid.benchDurationTable[charName] = raid.duration
						else
							raid.benchDurationTable[charName] = duration + (add * 60)
						end						
					end 
					RMRaid:Save(raid)
				end
			},
			SetLine = {
				name = "Set minutes for all",
				desc = "Set minutes to everyone on the bench",
				type = "text",
				usage =  "Adds minutes to everyone on the bench",
				hidden = not RMRaid:CanEdit(raid),
				get = function() return "0" end,		
				order = 3,	
				set = function(newVal)
					if not tonumber(newVal) or tonumber(newVal) < 0 then return end
					local newValToSet = newVal
					if (newValToSet * 60) > raid.duration then newValToSet = raid.duration end					
					for charName, duration in pairs(raid.benchDurationTable) do
						raid.benchDurationTable[charName] = (newValToSet * 60) 
					end
					RMRaid:Save(raid) 
				end
			},
			NewLine = {
				name = "New bench member",
				desc = "Add new bench member to list",
				type = "text",
				usage =  "Add new character to bench",
				hidden = not RMRaid:CanEdit(raid),
				get = function() return "0" end,		
				order = 1,	
				set = function(newName)									
					if RMCharacter:Load(newName) then
						raid.benchDurationTable[newName] = 0
					end
					RMRaid:Save(raid)															
				end
			},																
		},                            		                       
		order = 40
	}			
	for charName, duration in pairs(raid.benchDurationTable) do
		local character = RMCharacter:Load(charName)		
		tinsert(benchTable.args, {
			name = math.floor(duration / 60).."m "..RMBrowser.GetNameWithColor(character),
			order = 100000-duration,
			type = "text",
			usage = "Sets the bench time in minutes",			
			desc = character.name,
			hidden = not RMRaid:CanEdit(raid),			
			get = function() return math.floor(raid.benchDurationTable[character.name]/60) end,
			set = function(duration)
				if not tonumber(duration) or tonumber(duration) < 0 then return end
				if tonumber(duration) > 0 then
					local newDuration = duration * 60
					if newDuration > raid.duration then
						newDuration = raid.duration
					end			
					raid.benchDurationTable[character.name]=newDuration
				else
					raid.benchDurationTable[character.name]=nil
				end
				RMRaid:Save(raid)							
			end,						
		})
	end	
	return benchTable	
end

function RMBrowser.BuildParticipantTable(raid)	
	local participantTable = {
		name = "Participants",
		order=30,
		type = "group",
		desc = "Participants",
		args = {
			AddLine = {
				name = "Add minutes to all",
				desc = "Add minutes to everyone in the raid",
				type = "text",
				usage =  "Adds minutes to everyone in the raid",
				hidden = not RMRaid:CanEdit(raid),
				get = function() return "0" end,		
				order = 10,	
				set = function(add)					
					if not tonumber(add) or tonumber(add) < 0 then return end					
					for charName, duration in pairs(raid.participantDurationTable) do
						if (duration + (add * 60)) > raid.duration then
							raid.participantDurationTable[charName] = raid.duration
						else
							raid.participantDurationTable[charName] = duration + (add * 60)
						end 
					end 
					RMRaid:Save(raid)	
				end
			},
			SetLine = {
				name = "Set minutes for all",
				desc = "Add minutes to everyone in the raid",
				type = "text",
				usage =  "Adds minutes to everyone in the raid",
				hidden = not RMRaid:CanEdit(raid),
				get = function() return "0" end,		
				order = 11,	
				set = function(newVal)
					if not tonumber(newVal) or tonumber(newVal) < 0 then return end
					local newValToSet = newVal
					if (newValToSet * 60) > raid.duration then newValToSet = raid.duration end					
					for charName, duration in pairs(raid.participantDurationTable) do
						raid.participantDurationTable[charName] = (newValToSet * 60) 
					end 
					RMRaid:Save(raid)	
				end
			},	
			NewLine = {
				name = "New raid member",
				desc = "Add new raid member to list",
				type = "text",
				usage =  "Add new character to raid",
				hidden = not RMRaid:CanEdit(raid),
				get = function() return "0" end,		
				order = 1,	
				set = function(newName)					
					if RMCharacter:Load(newName) then
						raid.participantDurationTable[newName] = 0
					end
					RMRaid:Save(raid)															
				end
			},														
			
		},                            
		order = 30
	}			
	for charName, duration in pairs(raid.participantDurationTable) do
		local character = RMCharacter:Load(charName)				
		tinsert(participantTable.args, {
			name = math.floor(duration / 60).."m "..RMBrowser.GetNameWithColor(character),
			order = 100000-duration,
			type = "text",			
			desc = character.name,
			usage = "Sets the participation time in minutes",
			disabled = not RMRaid:CanEdit(raid),			
			get = function() return math.floor(raid.participantDurationTable[character.name]/60) end,
			set = function(duration)
				if not tonumber(duration) or tonumber(duration) < 0 then return end
				if tonumber(duration) > 0 then
					local newDuration = duration * 60
					if newDuration > raid.duration then
						newDuration = raid.duration
					end			
					raid.participantDurationTable[character.name]=newDuration
				else
					raid.participantDurationTable[character.name]=nil
				end
				RMRaid:Save(raid)							
			end,									
		})
	end		
	return participantTable	
end

function RMBrowser.BuildRaidTable(raid)	
	local raidTable = {
		type='group',				
		args = { 				
			ZoneLine = {
				type="header",
				name=raid.zoneName.." - "..raid.owner,
				order = 10
			},
			StartTimeLine = {
				type="header",
				name="Start Time: "..date("%m/%d/%y %I:%M %p",raid.startTime),
				order = 12				
			},			
			DurationLine = {
				type="header",
				name="Duration: "..RMBrowser.FormatDuration(raid.duration),
				order = 13
			},							
			LastUpdatedLine = {
				type="header",
				name="Updated: "..date("%m/%d/%y %I:%M %p",raid.lastUpdated),
				order = 16
			},																		
			ParticipantsLine = RMBrowser.BuildParticipantTable(raid),
			BenchLine = RMBrowser.BuildBenchTable(raid),
			AddLootLine = {
				name="Add Loot",
				type="group",
				order=21,
				desc="Add Loot",
				args = {
					NeedLine = {
						name="Need To",
						type="text",
						order=1,
						usage="Add need loot to a specified character",
						desc="Add loot to a specified character distributed as needed",						
						get=function() return "" end,
						set=function(recipient) 
							if raid.participantDurationTable[recipient] then
								RMRaid:AddLoot(raid, recipient, "Unknown", "Loot Token", true)
							else
								RaidMarshal:Print("You can only add loot to raid participants")
							end
						end,						
					},
					GreedLine = {
						name="Greed To",
						type="text",
						order=2,						
						usage="Add greed loot to a specified character",
						desc="Add loot to a specified character distributed as greed",
						get=function() return "" end,
						set=function(recipient) 
							if raid.participantDurationTable[recipient] then
								RMRaid:AddLoot(raid, recipient, "Unknown", "Loot Token", false)
							else
								RaidMarshal:Print("You can only add loot to raid participants")
							end
						end,							
					}
				}
			},
			SetDurationLine = {
				name = "Set Duration",
				type="text",
				hidden = not RMRaid:CanEdit(raid),				
				set=function(newDuration)
					if not tonumber(newDuration) or tonumber(newDuration) < 0 then return end  
					raid.duration = newDuration * 60
					tables = {raid.participantDurationTable, raid.benchDurationTable}
					for i, table in ipairs(tables) do
						for name, duration in pairs(table) do
							if duration > raid.duration then
								table[name] = raid.duration
							end
						end
					end
					RMRaid:Save(raid)
				end,					
				get=function() return math.floor(raid.duration / 60) end,
				order = 22,
				usage="Sets new raid duration in minutes",
				desc="Sets new raid duration in minutes"
			},
			ContinueLine = {
				name = "Continue",
				type = "execute",
				desc = "Continue a raid from the database",
				hidden = not RMRaid:CanEdit(raid),
				confirm = true,
				order = 50,
				func = function()
					RaidMarshal:Continue(raid)					 
				end
			},						
			DeleteLine = {
				name = "Delete",
				type = "toggle",
				desc = "Delete a raid from the database",
				disabled = not RMRaid:CanEdit(raid),
				confirm = true,
				order = 60,
				set = function(set) 
					raid.isDeleted = set
					RMRaid:Save(raid)									
				end,
				get = function() return raid.isDeleted end
			},									
			CloseLine = {
				name = "Close",
				type = "execute",
				desc = "Closes this menu",
				confirm = false,
				order = 70,
				func = function() 
					Dewdrop:Close() 
				end
			}																						
		}  								
	}	
	local lootTable = RMBrowser.BuildRaidLootTable(raid)
	if lootTable then
		raidTable.args["LootLine"] = lootTable
	end	
	return raidTable
end

function RMBrowser.OpenRaidDewdrop(raidRow)				
	Dewdrop:Open(
		raidRow,
		'point', 'RIGHT',
		'children', function()
			Dewdrop:FeedAceOptionsTable(RMBrowser.BuildRaidTable(raidRow.raid))
		end
	)
end

function RMBrowser.GetRoleColor(roleName) 
	if roleName and roleName == RMCharacter.ROLE_TANK then
		return BabbleClass:GetColor("Warrior")
	elseif roleName and roleName == RMCharacter.ROLE_HEALER then
		return BabbleClass:GetColor("Priest")
	elseif roleName and roleName == RMCharacter.ROLE_DPS then
		return BabbleClass:GetColor("Rogue")																
	else -- non colors are mage color
		return BabbleClass:GetColor("Mage")
	end	
end

function RMBrowser.GetRoleHexColor(roleName) 
	if roleName and roleName == RMCharacter.ROLE_TANK then
		return BabbleClass:GetHexColor("Warrior")
	elseif roleName and roleName == RMCharacter.ROLE_HEALER then
		return BabbleClass:GetHexColor("Priest")
	elseif roleName and roleName == RMCharacter.ROLE_DPS then
		return BabbleClass:GetHexColor("Rogue")																
	else
		return BabbleClass:GetHexColor("Mage") 	       
	end	
end

function RMBrowser.GetRoleWithColor(roleName)	
	return "|cFF"..RMBrowser.GetRoleHexColor(roleName)..roleName.."|r"	     
end

function RMBrowser.GetClassHexColor(className)
	return BabbleClass:GetHexColor(className)
end

function RMBrowser.GetNameWithColor(character)
    local classColor
    if character.className then
        classColor = BabbleClass:GetHexColor(character.className)
    else
        -- else set the color to gray to indicate unknown
        classColor = "ADADAD"
    end
    return "|cFF"..classColor..character.name.."|r"
end


function RMBrowser.GetNameWithMain(character)
	local mainString = ""
	if character.mainName then
		local mainCharacter = RMCharacter:Load(character.mainName)
		mainColor = BabbleClass:GetHexColor(mainCharacter.className)
		mainString = " |cFF"..mainColor.."("..mainCharacter.name..")|r"
	end
	return RMBrowser.GetNameWithColor(character)..mainString
end

function RMBrowser:UpdateRaidMenu(group)
    -- update key if nil (bad data)
    if not group.key then group.key = group.owner .. group.startTime end
    
    self.options.args.raids.args[group.key] = {
        type="group",
        name=group.zoneName..' ('..group.owner..'): '..date("%m/%d/%y - %H:%M",group.startTime),
        desc="Raid in "..group.zoneName.."at "..group.startTime,
        args = {
            leaders = {
                type="group",
                name="Leaders",
                desc="Players that were at some point raid leader",
                args = {}
            },
            officers = {
                type="group",
                name="Officers",
                desc="Players that were at some point raid assistants",
                args = {}
            },
            participants = {
                type="group",
                name="Participants",
                desc="Players that participated in the raid",
                args = {}
            }
        }
    }
    
    --[[ build leader list for raid
        for i, leader in ipairs(group.leaders) do
            self.options.args.raids.args[groupKey].args.leaders.args[leader] = {
                name=leader,
                type="execute",
                func= function() end
            }
        end ]]
        
    -- update participant table
    for playerName, duration in pairs(group.participantDurationTable) do
        local player = self:UpdatePlayer(playerName, group)
        local playerMenu = self:UpdatePlayerMenu(player)
        
        self.options.args.raids.args[group.key].args.participants.args[playerName] = {
            name=duration .. "m " .. playerName,
            type="execute",
            desc=playerName .. " was present in the raid for " .. duration .. " minutes",
            func= function() end
        }                
        
        self.options.args.players.args[playerName].args.raids.args[group.key] = self.options.args.raids.args[group.key]
    end
    
    return self.options.args.raids.args[group.key]
end
