-- Window.lua
-- VFL_Profiler
-- (C)2007 Bill Johnson and The VFL Project
--
-- The main interface for the VFL profiler.

local FormatMicro = VFL.Time.FormatMicro;

local profw = VFLUI.Window:new(UIParent);
profw:SetFrameStrata("LOW");
VFLUI.Window.SetDefaultFraming(profw, 24);
profw:SetText("VFL Profiler");
profw:SetTitleColor(0,0,.6);
profw:SetWidth(510); profw:SetHeight(380);
profw:SetPoint("CENTER", UIParent, "CENTER");
profw:SetMovable(true); profw:SetToplevel(nil);
VFLUI.Window.StdMove(profw, profw:GetTitleBar());
profw:Hide();
local ca = profw:GetClientArea();

----------- Detailed prof toggle
local toggle = VFLUI.Button:new(ca);
toggle:SetPoint("TOPLEFT", ca, "TOPLEFT");
toggle:SetWidth(100); toggle:SetHeight(25); toggle:Show();
if VFLP.IsEnabled() then
	toggle:SetTextColor(1,0,0); toggle:SetText("Stop Profiling");
	toggle:SetScript("OnClick", function()
		VFLUI.MessageBox("Confirm", "Stop profiling and reload the UI?", nil, "Yes", function()
			SetCVar("scriptProfile", 0); ReloadUI();
		end, "No", VFL.Noop);
	end);
else
	toggle:SetTextColor(1,1,0); toggle:SetText("Start Profiling");
	toggle:SetScript("OnClick", function()
		VFLUI.MessageBox("Confirm", "Start profiling and reload the UI? |cFFFF0000WARNING: Profiling uses up a lot of UI resources and can cause significant slowdowns and 'jerky' behavior.", nil, "Yes", function()
			SetCVar("scriptProfile", 1); ReloadUI();
		end, "No", VFL.Noop);
	end);
end


---------------------------------------------------------------
-- DATA LIST
---------------------------------------------------------------
-- Create a cell in the data table
local function CreateCell(parent)
	local self = VFLUI.AcquireFrame("Frame");
	VFLUI.StdSetParent(self, parent); self:SetHeight(10);

	-- Create the columns.
	local w, af, ap = 0, self, "TOPLEFT";
	self.col = {};
	for i=1,8 do
		-- Create the text object.
		local colText = VFLUI.CreateFontString(self);
		colText:SetDrawLayer("OVERLAY");
		VFLUI.SetFont(colText, Fonts.Default, 9);
		colText:SetShadowOffset(1,-1);
		colText:SetShadowColor(0,0,0,.6);
		colText:SetTextColor(1,1,1,1); colText:SetJustifyH("LEFT");
		colText:SetPoint("TOPLEFT", af, ap); 
		colText:SetHeight(10);
		colText:Show();
		-- Save it
		self.col[i] = colText;
		-- Update iterative variables
		af = colText; ap = "TOPRIGHT";
	end

	-- Create the selection texture
	local selTexture = VFLUI.CreateTexture(self);
	selTexture:Hide(); selTexture:SetDrawLayer("BACKGROUND");	selTexture:SetAllPoints(self);
	selTexture:SetTexture(0, 0, 0.6, 0.6);
	self.selTexture = selTexture;

	-- Update the Destroy function
	self.Destroy = VFL.hook(function(s)
		-- Destroy columns
		for _,column in pairs(s.col) do column:Destroy();	end
		s.col = nil;
		-- Destroy textures
		if s.selTexture then s.selTexture:Destroy(); s.selTexture = nil; end
	end, self.Destroy);
	self.OnDeparent = self.Destroy;

	return self;
end

local data_decor = VFLUI.AcquireFrame("Frame");
data_decor:SetParent(ca);
data_decor:SetPoint("TOPLEFT", ca, "TOPLEFT", 0, -25);
data_decor:SetWidth(500); data_decor:SetHeight(310);
data_decor:SetBackdrop(VFLUI.DefaultDialogBorder); data_decor:Show();

local data = VFLUI.List:new(ca, 12, CreateCell);
data:SetPoint("TOPLEFT", data_decor, "TOPLEFT", 5, -5);
data:SetWidth(490); data:SetHeight(300); data:Show(); data:Rebuild();

local function SetupLayout(colspec, cols)
	for i=1,8 do
		local cs, col = colspec[i], cols[i];
		if cs then
			col:Show(); col:SetWidth(colspec[i].width); col:SetJustifyH(cs.jh or "LEFT");
		else
			col:Hide();
		end
	end
end

local function PaintTitle(colspec, cols)
	for i=1,8 do
		local cs, col = colspec[i], cols[i];
		if cs then
			col:SetTextColor(1,1,1); col:SetText(cs.title);
		end
	end
end

local function PaintData(colspec, cols, data)
	for i=1,8 do
		local cs,col = colspec[i], cols[i];
		if cs then
			col:SetTextColor(cs.r, cs.g, cs.b);
			cs.paint(col, data);
		end
	end
end

local colspec_summ = {
	{ title = "Name"; width = 110; r=1; g=1; b=1;
	  paint = function(cell, data) cell:SetText(data.title); end; };
	{ title = "mem"; width = 55; r=.95; g=.95; b=.45; jh="RIGHT";
	  paint = function(cell, data) cell:SetText(string.format("%0.2fk", data.mem)); end; };
	{ title = "mem/sec"; width = 55; r=.75; g=.75; b=.25; jh="RIGHT";
		paint = function(cell, data) cell:SetText(string.format("%0.2fk", data.raMem / VFLP.GetSummaryUpdateInterval())); end; };
	{ title = "CPU%"; width = 55; r=1; g=1; b=1; jh="RIGHT";
		paint = function(cell, data) cell:SetText(string.format("%0.2f%%", data.raCPU * 100 / VFLP.GetSummaryUpdateInterval())); end; };
	{ title = "CPU/frame"; width = 55; r=.75; g=.35; b=.35; jh="RIGHT";
	  paint = function(cell, data) cell:SetText(FormatMicro(data.raCPU / VFLP.GetFramesPerSummaryUpdate())); end; };
	{ title = "FPS impact"; width = 55; r=0;g=.7;b=0; jh="RIGHT";
		paint = function(cell, data)
			local timePerFrame = data.raCPU / VFLP.GetFramesPerSummaryUpdate();
			if timePerFrame > 0.0000001 then
				local fr = GetFramerate();
				local nfr = (1/fr) - timePerFrame;
				if nfr > 0.00001 then nfr = 1/nfr else nfr = fr; end
				cell:SetText(string.format("%0.2f", nfr - fr));
			else
				cell:SetText(0);
			end
		end; };
};

local function SetSummaryMode()
	for _,cell in data:_GetGrid():StatelessIterator(1) do
		SetupLayout(colspec_summ, cell.col);
	end
	local alist = VFLP._GetAddonSummary();
	data:SetDataSource(function(cell, data, pos, absPos)
		local col = cell.col;
		if(absPos == 1) then
			cell.selTexture:Show(); cell.selTexture:SetTexture(0,0,0.6,0.6);
			PaintTitle(colspec_summ, col);
		else
			cell.selTexture:Hide();
			PaintData(colspec_summ, col, data);
		end
	end, function() return #alist + 1; end, function(n) if n == 1 then return true; else return alist[n-1]; end end);
	-- Unbind any events previousl bound to the window
	VFLP.Events:Unbind(profw);
	-- Notify to update on repaint
	VFLP.Events:Bind("SUMMARY_PROFILE_UPDATED", data, data.Update, profw);
end

local colspec_obj = {
	{ title = "Name"; width = 110; r=1; g=1; b=1;
	  paint = function(cell, data) cell:SetText(data.title); end; };
	{ title = "call"; width = 55; r=.95; g=.95; b=.45; jh="RIGHT";
	  paint = function(cell, data) cell:SetText(data.calls); end; };
	{ title = "call/sec"; width = 55; r=.75; g=.75; b=.25; jh="RIGHT";
		paint = function(cell, data) cell:SetText(string.format("%0.2f", data.raCalls / VFLP.GetObjectUpdateInterval())); end; };
	{ title = "call/frame"; width = 52; r=.75; g=.75; b=.25; jh="RIGHT";
		paint = function(cell, data) cell:SetText(string.format("%0.2f", data.raCalls / VFLP.GetFramesPerObjectUpdate())); end; };
	{ title = "sec/call"; width = 50; r=.75; g=.35; b=.35; jh="RIGHT";
	  paint = function(cell, data) 
			cell:SetText( ((data.type == "category") or (data.calls == 0)) and "" or FormatMicro(data.CPU / data.calls)); 
		end; };
	{ title = "sec/frame"; width = 50; r=.75; g=.35; b=.35; jh="RIGHT";
		paint = function(cell, data) cell:SetText(FormatMicro(data.raCPU / VFLP.GetFramesPerObjectUpdate())); end; };
	{ title = "CPU%"; width = 45; r=1;g=1;b=1; jh="RIGHT";
		paint = function(cell, data) cell:SetText(string.format("%0.2f%%", data.raCPU * 100 / VFLP.GetObjectUpdateInterval())); end; };
	{ title = "sec"; width = 55; r=.95;g=.5;b=.5; jh="RIGHT";
		paint = function(cell, data) cell:SetText(FormatMicro(data.CPU)); end; };
};

local function SetObjectMode()
	for _,cell in data:_GetGrid():StatelessIterator(1) do
		SetupLayout(colspec_obj, cell.col);
	end
	local alist = VFLP._flist;
	data:SetDataSource(function(cell, data, pos, absPos)
		local col = cell.col;
		if(absPos == 1) then
			cell.selTexture:Show(); cell.selTexture:SetTexture(0,0,0.6,0.6);
			PaintTitle(colspec_obj, col);
		else
			if data.type == "category" then
				cell.selTexture:Show(); cell.selTexture:SetTexture(0,0.6,0.6,1);
			else
				cell.selTexture:Hide();
			end
			PaintData(colspec_obj, col, data);
		end
	end, function() return #alist + 1; end, function(n) if n == 1 then return true; else return alist[n-1]; end end);
	-- Unbind any events previousl bound to the window
	VFLP.Events:Unbind(profw);
	-- Notify to update on repaint
	VFLP.Events:Bind("OBJECT_PROFILE_UPDATED", data, data.Update, profw);
end

--------- Mode buttons
local mode = 1;
local UpdateMode;

local mb1 = VFLUI.Button:new(ca);
mb1:SetPoint("LEFT", toggle, "RIGHT", 15, 0);
mb1:SetWidth(60); mb1:SetHeight(25); mb1:Show();
mb1:SetText("Addons");
mb1:SetScript("OnClick", function() UpdateMode(1); end);

local mb2 = VFLUI.Button:new(ca);
mb2:SetPoint("LEFT", mb1, "RIGHT");
mb2:SetWidth(60); mb2:SetHeight(25); mb2:Show();
mb2:SetText("Events");
mb2:SetScript("OnClick", function() UpdateMode(2); end);

local mb3 = VFLUI.Button:new(ca);
mb3:SetPoint("LEFT", mb2, "RIGHT");
mb3:SetWidth(60); mb3:SetHeight(25); mb3:Show();
mb3:SetText("Objects");
mb3:SetScript("OnClick", function() UpdateMode(3); end);

function UpdateMode(n)
	mb1:UnlockHighlight(); mb2:UnlockHighlight(); mb3:UnlockHighlight();
	mode = n;
	if(n == 1) then
		mb1:LockHighlight(); SetSummaryMode();
	elseif(n == 3) then
		mb3:LockHighlight(); SetObjectMode();
	end
end

profw:SetScript("OnHide", function(self) VFLP.Events:Unbind(self); end);
profw:SetScript("OnShow", function() UpdateMode(mode); end);

local closebtn = VFLUI.CloseButton:new();
closebtn:SetScript("OnClick", function() profw:Hide(); end);
profw:AddButton(closebtn);

SLASH_PROFILER1 = "/vflprofiler";
SlashCmdList["PROFILER"] = function() profw:Show(); end

RDX.RegisterMainMenuFunction(function(ent)
	ent.text = i18n("Profiler");
	ent.OnClick = function() VFL.poptree:Release(); profw:Show(); end;
end);
