-- Core.lua
-- VFL_Profiler
-- (C)2007 Bill Johnson and The VFL Project
--
-- Core updating routines for the profiler.

-----------------------------------------------------
-- SUMMARY PROFILING
-----------------------------------------------------
local summary_update_interval = 1.0;

--- Gets the summary update interval.
-- This is used for converting to per-second stats.
-- (usage per second = usage per interval * intervals per sec = upi / spi)
function VFLP.GetSummaryUpdateInterval() return summary_update_interval; end

local sig_ProfUpdated = VFLP.Events:LockSignal("SUMMARY_PROFILE_UPDATED");

-- List of addons currently being profiled
local alist = {};
function VFLP._GetAddonSummary() return alist; end

local function UpdateAList()
	alist[1] = {
		addon_index = 1;
		title = "--- TOTAL ---";
		mem = 0; CPU = 0; raMem = 0; raCPU = 0;
	};
	local n = 2;
	for i=1,GetNumAddOns() do if IsAddOnLoaded(i) then
		alist[n] = {
			addon_index = i;
			title = GetAddOnInfo(i);
			mem = 0; CPU = 0; raMem = 0; raCPU = 0;
		};
		n = n + 1;
	end end
	sig_ProfUpdated:Raise();
end
WoWEvents:Bind("ADDON_LOADED", nil, UpdateAList);

local function UpdateMemorySummary()
--	if VFLP.IsEnabled() then UpdateAddOnMemoryUsage(); end

	local mem, usage, tot, x, aindex = 0, 0, 0, nil, 1;
	-- For each addon...
	for i=2,#alist do
		x = alist[i]; aindex = x.addon_index;
		mem = GetAddOnMemoryUsage(aindex); tot = tot + mem;

		-- Update usage for this frame
		usage = mem - x.mem; x.raMem = ( (x.raMem * 4) + usage ) / 5; -- weight-5 rolling average
		x.mem = mem;
	end

	-- Now update totals row
	x = alist[1];
	usage = tot - x.mem;
	x.raMem = ( (x.raMem * 4) + usage ) / 5;
	x.mem = tot;
end

-- By default, no CPU summary
local UpdateCPUSummary = VFL.Noop;
-- If profiling is enabled, populate it with a real function
if VFLP.IsEnabled() then
	function UpdateCPUSummary()
		UpdateAddOnCPUUsage();
		local cpu, usage, x, aindex = 0, 0, nil, 1;
		for i=2,#alist do
			x = alist[i];	aindex = x.addon_index;
			cpu = GetAddOnCPUUsage(aindex) / 1000; usage = cpu - x.CPU;
			x.raCPU = ( (x.raCPU * 4) + usage ) / 5; 
			x.CPU = cpu;
		end
		x = alist[1]; cpu = GetScriptCPUUsage() / 1000; usage = cpu - x.CPU;
		x.raCPU = ( (x.raCPU * 4) + usage ) / 5;
		x.CPU = cpu;
	end
end

-- Summary updater
local last_su, framecount, last_su_frame, frames_per_su = 0, 0, 0, 30;

--- Usage/frame = usage/interval * intervals/frame = u/i * 1/(f/i)
function VFLP.GetFramesPerSummaryUpdate() return frames_per_su; end

local suf = CreateFrame("Frame");
local function SummaryUpdate()
	framecount = framecount + 1;
	local t = GetTime();
	if (t-last_su) < summary_update_interval then return; end
	last_su = t;

	-- Measure 
	frames_per_su = ( (frames_per_su * 4) + (framecount - last_su_frame) ) / 5;
	last_su_frame = framecount;

	UpdateMemorySummary();
	UpdateCPUSummary();
	sig_ProfUpdated:Raise();
end

------------------------------------------------------------------------------
-- Object PROFILING
------------------------------------------------------------------------------
local object_update_interval = 0.5;
--- Gets the object update interval.
-- This is used for converting to per-second stats.
-- (usage per second = usage per interval * intervals per sec = upi / spi)
function VFLP.GetObjectUpdateInterval() return object_update_interval; end

local sig_OProfUpdated = VFLP.Events:LockSignal("OBJECT_PROFILE_UPDATED");
local oblist = VFLP._flist;

local function UpdateObjProfile()
	local curCat, totCPU, totCalls, delta, cpu, calls = nil, 0, 0, 0, 0, 0;
	for _,obj in ipairs(oblist) do
		if (obj.type == "category") and (obj ~= curCat) then
			-- We just changed categories. Update the current category
			if curCat then
				-- update total calls
				delta = totCalls - curCat.calls;
				curCat.raCalls = ( (curCat.raCalls * 4) + delta ) / 5;
				curCat.lastCalls = curCat.calls; curCat.calls = totCalls;
				-- update total CPU
				delta = totCPU - curCat.CPU;
				curCat.raCPU = ( (curCat.raCPU * 4) + delta ) / 5;
				curCat.lastCPU = curCat.CPU; curCat.CPU = totCPU;
			end
			totCPU = 0; totCalls = 0; curCat = obj;
		else
			if obj.type == "function" then
				cpu, calls = GetFunctionCPUUsage(obj.object, obj.includeSubObjects);
				cpu = cpu / 1000;
			elseif obj.type == "frame" then
				cpu, calls = GetFrameCPUUsage(obj.object, obj.includeSubObjects);
				cpu = cpu / 1000;
			end
			totCPU = totCPU + cpu; totCalls = totCalls + calls;
			-- update calls
			delta = calls - obj.calls;
			obj.raCalls = ( (obj.raCalls * 4) + delta ) / 5;
			obj.lastCalls = obj.calls; obj.calls = calls;
			-- update cpu
			delta = cpu - obj.CPU;
			obj.raCPU = ( ( obj.raCPU * 4) + delta ) / 5;
			obj.lastCPU = obj.CPU; obj.CPU = cpu;
		end
	end
	-- Update last category
	if curCat then
		-- update total calls
		delta = totCalls - curCat.calls;
		curCat.raCalls = ( (curCat.raCalls * 4) + delta ) / 5;
		curCat.lastCalls = curCat.calls; curCat.calls = totCalls;
		-- update total CPU
		delta = totCPU - curCat.CPU;
		curCat.raCPU = ( (curCat.raCPU * 4) + delta ) / 5;
		curCat.lastCPU = curCat.CPU; curCat.CPU = totCPU;
	end
end

-- Object updater
local last_ou, last_ou_frame, frames_per_ou = 0, 0, 0, 30;

--- Usage/frame = usage/interval * intervals/frame = u/i * 1/(f/i)
function VFLP.GetFramesPerObjectUpdate() return frames_per_ou; end

local ouf = CreateFrame("Frame");
local function ObjectUpdate()
	local t = GetTime();
	if (t-last_ou) < object_update_interval then return; end
	last_ou = t;

	-- Measure 
	frames_per_ou = ( (frames_per_ou * 4) + (framecount - last_ou_frame) ) / 5;
	last_ou_frame = framecount;

	UpdateObjProfile();
	sig_OProfUpdated:Raise();
end

-- Add the profiling core to the profiler!
VFLP.RegisterCategory("VFL Profiler");
VFLP.RegisterFunc("VFL Profiler", "Mem Updater", UpdateMemorySummary, true);
VFLP.RegisterFunc("VFL Profiler", "CPU Updater", UpdateCPUSummary, true);
VFLP.RegisterFunc("VFL Profiler", "Object Updater", UpdateObjProfile, true);

WoWEvents:Bind("VARIABLES_LOADED", nil, function()
	if type(VFLConfig.prof_summary_update_interval) == "number" then
		summary_update_interval = VFL.clamp(VFLConfig.prof_summary_update_interval, 0.1, 10);
	else
		VFLConfig.prof_summary_update_interval = summary_update_interval;
	end

	if type(VFLConfig.prof_op_enabled) ~= "number" then VFLConfig.prof_op_enabled = 1; end

	if VFLP.IsEnabled() and (VFLConfig.prof_op_enabled == 1) then
		ouf:SetScript("OnUpdate", ObjectUpdate);
		VFL.print("|cFFFF0000[VFL]|r Warning : Profiler Activated !!!");
	end

	suf:SetScript("OnUpdate", SummaryUpdate);
end);
