-- WindowManager.lua
-- RDX - Raid Data Exchange
-- (C)2006 Raid Informatics
--
-- THIS FILE CONTAINS COPYRIGHTED CONTENT SUBJECT TO THE TERMS OF A SEPARATE LICENSE.
-- UNLICENSED COPYING IS PROHIBITED.
--
-- The WindowManager is responsible for laying out RDX windows on the screen
-- and saving and recalling the states of windows on demand.
--
-- The WindowManager explicitly manages position, scale, alpha, anchorage.

RDXWM = RegisterVFLModule({
	name = "RDXWM";
	description = "RDX - Window Manager";
	parent = RDX;
});

-- Flag: whether or not to show the "dock parent" markers on windows.
local show_parent_marks = nil;

----------------------------------------------------
-- Helper functions
----------------------------------------------------

--- Print an error message from inside compiled paint code.
local _err_norepeat = {};
setmetatable(_err_norepeat, {__mode = 'k'});
function RDXWM.PrintError(win, info, err)
	if (type(win) == "table") and (not _err_norepeat[win]) then
		_err_norepeat[win] = true;
		local ident, path = "<unknown>", "<unknown>";
		if (type(win.WMGetIdentity) == "function") then ident = tostring(win:WMGetIdentity()); end
		if (type(win._path) == "string") then
			path = win._path;
			if(ident == "<unknown>") then ident = path; end
		end
		VFL.TripError("RDX", i18n("Window <") .. ident .. i18n("> caused a paint error.", "Pointer: ") .. tostring(win) .. i18n("\nIdentity: ") .. ident ..i18n("\nPath: ") .. path .. i18n("\nError\n--------\n") .. err);
	end
end

--- Gets the corner of the screen (quadrant) best containing the box with the
-- given universal coordinates.
--
-- The screen is divided into four quadrants based on universal coordinates.
-- Information about these quadrants is used to make decisions about window layouts.
function RDXWM.GetBoxQuadrant(left, top, right, bottom)
	left = math.abs(left - VFLUI.uScreenLeft);
	top = math.abs(top - VFLUI.uScreenTop);
	right = math.abs(right - VFLUI.uScreenRight);
	bottom = math.abs(bottom - VFLUI.uScreenBottom);
	if(top < bottom) then
		if(left < right) then return "TOPLEFT"; else return "TOPRIGHT"; end
	else
		if(left < right) then return "BOTTOMLEFT"; else return "BOTTOMRIGHT"; end
	end
end

--- Gets the corner of the screen (quadrant) best containing the given frame.
function RDXWM.GetFrameQuadrant(frame)
	return RDXWM.GetBoxQuadrant(GetUniversalBoundary(frame));
end
--- Given frame coordinates and an anchor point, anchors the frame appropriately based on
-- that point.
function RDXWM.AnchorFrame(frame, ap, left, top, right, bottom)
	if(ap == "TOPLEFT") then
		frame:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, left, top));
	elseif(ap == "TOP") then
		frame:SetPoint("TOP", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, (left+right)/2, top));
	elseif(ap == "TOPRIGHT") then
		frame:SetPoint("TOPRIGHT", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, right, top));
	elseif(ap == "RIGHT") then
		frame:SetPoint("RIGHT", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, right, (top+bottom)/2));
	elseif(ap == "BOTTOMRIGHT") then
		frame:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, right, bottom));
	elseif(ap == "BOTTOM") then
		frame:SetPoint("BOTTOM", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, (left+right)/2, bottom));
	elseif(ap == "BOTTOMLEFT") then
		frame:SetPoint("BOTTOMLEFT", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, left, bottom));
	elseif(ap == "LEFT") then
		frame:SetPoint("LEFT", UIParent, "BOTTOMLEFT", GetLocalCoords(frame, left, (top+bottom)/2));
	else
		error(i18n("RDXWM.AnchorFrame: invalid anchorpoint"));
	end
end

-- Based on a set of coordinates, force a frame into a fixed-position layout.
-- All coordinates must be universal coordinates relative to the frame itself.
local function LayoutByPosition(frame, posf, left, top, right, bottom)
	posf:ClearAllPoints();
	local ap = "TOPLEFT";
	if (frame._wm_data) and (frame._wm_data.ap) then 
		ap = frame._wm_data.ap 
	else
		ap = RDXWM.GetBoxQuadrant(left, top, right, bottom);
	end
--	RDXWM:Debug(5, "LayoutByPosition(" .. frame:WMGetIdentity() .. "): parent[" .. tostring(posf:GetParent()) .. "] ap[" .. ap .. "] coords[" .. string.format("%0.2f", left) .. "," .. string.format("%0.2f", top) .. "," .. string.format("%0.2f", right) .. "," .. string.format("%0.2f", bottom) .. "]");
	RDXWM.AnchorFrame(posf, ap, left, top, right, bottom);
end

--- Determine if this window is a "managed window"
local function IsManagedWindow(w)
	if w and (w._wm_data) then return true; else return nil; end
end
RDXWM.IsManagedWindow = IsManagedWindow;

--- Create a docking entry for this frame.
local function CreateDockEntry(win, localPoint, remoteID, remotePoint)
	local d = win._wm_data;
	if not d.dock then d.dock = {}; end
	d.dock[localPoint] = { id = remoteID, point = remotePoint };
end

--- Determine if this frame is the dockgroup parent
local function IsDGP(frame)
	return frame._wm_data.dgp;
end

--- Set this frame's DGP status
local function SetDGP(frame, val)
	frame._wm_data.dgp = val;
end

--- Get the dockpoint table for this frame.
local function GetDockPoints(frame)
	return frame._wm_data.dock;
end

--- Clear a specific dock point on tihs frame.
local function ClearDockPoint(frame, pt)
	local d =  frame._wm_data;
	if d.dock then d.dock[pt] = nil; end
end

--- Undock a window from this frame, verifying identity.
local function UndockOther(win, localPoint, otherWindow)
	local d = win._wm_data;
	if not d.dock then return; end
	local di = d.dock[localPoint];
	if di and (di.id == otherWindow:WMGetIdentity()) then
		d.dock[localPoint] = nil;
	end
end

-----------------------------------------------------
-- WINDOW CLASSES
-----------------------------------------------------
local classes = {};

function RDXWM.RegisterWindowClass(tbl)
	if (not tbl) then error(i18n("expected table, got nil")); end
	if (not tbl.name) then error(i18n("attempt to register anonymous class")); end
	local n = tbl.name;
	if classes[n] then error(i18n("duplicate WindowClass registration")); end
	classes[n] = tbl;
end

function RDXWM.GetWindowClass(cn)
	if not cn then return nil; end
	return classes[cn];
end

-----------------------------------------------------
-- WINDOW DATABASE
-----------------------------------------------------
-- The main window manager database - maps idents to saved data
local db = nil;
-- Transient ident database - maps an ident to a frame
local identMap = {};

--- Create a "blank" profile.
local function CreateBlankWindowProfile()
	local ret = {};
	ret.scale = 1; ret.alpha = 1;
	ret.strata = "MEDIUM"; ret.level = 0;
	return ret;
end

local function GetWindowByIdentity(ident)
	if not ident then return nil; end
	return identMap[ident];
end

----------------------------------------------------------
-- LAYOUT CORE
----------------------------------------------------------

-- Helpers for IterateDockGroup
local function IDGRecursive(win, fn, touched)
	-- Don't recurse over windows alraedy visited
	if touched[win] then return; end
	-- Do the operation
	fn(win); touched[win] = true;
	-- Recurse
	local di = GetDockPoints(win);
	if not di then return; end
	for _,ri in pairs(di) do
		local rw = GetWindowByIdentity(ri.id);
		if rw then 
			IDGRecursive(rw, fn, touched); 
		end
	end
end

-- Iterate over all windows in the dock group of the given window, calling the 
-- function fn for each.
local _idg_tbl = {};
local function IterateDockGroup(win, fn)
	VFL.empty(_idg_tbl);
	IDGRecursive(win, fn, _idg_tbl);
end
RDXWM.IterateDockGroup = IterateDockGroup;

-- Make w the parent of its dock group
local function MakeDockGroupParent(w)
	if RDXWM.IsDocked(w) then
		IterateDockGroup(w, function(x) SetDGP(x, nil); end);		
		SetDGP(w, true);
	else
		SetDGP(w, nil);
	end
end

-- Find the parent of w's dock group.
local function FindDockGroupParent(w)
	local ret = nil;
	if RDXWM.IsDocked(w) then
		IterateDockGroup(w, function(x) 
			if(IsDGP(x)) then ret = x; end 
		end);
	end
	return ret;
end

-- Clear the parent of w's dock group
local function ClearDockGroupParent(w)
	IterateDockGroup(w, function(x) SetDGP(x, nil); end);
end

-- Determine if two windows are in the same dock group
local function Interdocked(w1, w2)
	local ret = nil;
	IterateDockGroup(w1, function(w) if(w == w2) then ret = true; end end);
	return ret;
end

------------------- UNLAYOUT CODE
local function Unlayout(w)
	if w._wm_drag then self:WMStopDrag(true); end
	if not w._wm_nolayout then
		local pf = w:WMGetPositionalFrame();
		if pf then pf:ClearAllPoints(); end
		w._wm_layout = nil;
	end
end
local function ResetLayout()
	for _,wi in pairs(identMap) do Unlayout(wi); end
end
local function ResetDockGroupLayout(win)
	IterateDockGroup(win, Unlayout);
end

--------------------- LAYOUT CODE
-- Force layout a window by position.
local function _DoFixedLayout(win, pf)
	RDXWM:Debug(8, "* fixed layout on " .. win:WMGetIdentity());
	local l,t,r,b = RDXWM.GetSavedPosition(win);
	if l then
		LayoutByPosition(win, pf, l, t, r, b);
		win._wm_layout = true; return true;
	end
	pf:SetPoint("CENTER", UIParent, "CENTER");
	RDXWM.SavePosition(win);
	win._wm_layout = true; return true;
end

-- Attempt to layout the given window based on the current state.
local function LayoutWindow(win)
	-- If this window doesn't need to be laid out, don't.
	if win._wm_nolayout or win._wm_layout then return true; end
	-- Don't layout secure windows during ICLD
	if win.secure and InCombatLockdown() then return true; end
	-- Everything here works by positional frame..
	local posf = win:WMGetPositionalFrame();
	local wmdata = win._wm_data;
	if (not posf) then 
		error(i18n("Window ") .. win:WMGetIdentity() .. i18n(" does not have a positional frame."));
	end

	RDXWM:Debug(8, "LayoutWindow(" .. win:WMGetIdentity() .. ")");

	-- Apply scale, alpha, stratum, level
	win:SetScale(wmdata.scale);
	win:SetAlpha(wmdata.alpha);
	win:SetFrameStrata(wmdata.strata);
	
	-- Clear preexisting layout.
	posf:ClearAllPoints();
	if (not show_parent_marks) and (win._wm_mark) then
		VFLUI.ReleaseRegion(win._wm_mark); win._wm_mark = nil;
	end

	-- If we're the DGP...
	if IsDGP(win) then
		RDXWM:Debug(8, "* locking in dock group parent " .. win:WMGetIdentity());
		-- Lay us out by locked position.
		win:SetClampedToScreen(nil);
		_DoFixedLayout(win, posf);
		-- Show the "parent" mark if necessary.
		if (show_parent_marks) and (not win._wm_mark) then
			local tx = VFLUI.CreateTexture(win);
			tx:SetHeight(12); tx:SetWidth(12); tx:SetPoint("CENTER", win, "TOPLEFT");
			tx:Show(); tx:SetTexture(1,0,0,1);
			win._wm_mark = tx;
		end
		return true;
	else
		if(win._wm_mark) then
			-- destroy the "parent" mark
			VFLUI.ReleaseRegion(win._wm_mark); win._wm_mark = nil;
		end
	end
	
	-- If we're docked...
	local di = GetDockPoints(win);
	if di then
		-- For each neighbor...
		for localPoint, remoteInfo in pairs(di) do
			local rw = GetWindowByIdentity(remoteInfo.id);
			-- If the neighbor was successfully laid out...
			if rw and rw._wm_layout then
				RDXWM:Debug(8, "* docking " .. win:WMGetIdentity() .. ":" .. localPoint .. " to " .. rw:WMGetIdentity() .. ":" .. remoteInfo.point);
				-- Dock us!
				win:SetClampedToScreen(nil);
				local actualLocalPoint, dxl, dyl = win:WMGetDockSourcePoint(localPoint);
				local actualRemotePoint, dxr, dyr = rw:WMGetDockTargetPoint(remoteInfo.point);
				posf:SetPoint(actualLocalPoint, rw:WMGetPositionalFrame(), actualRemotePoint, dxl+dxr, dyl+dyr);
				-- We're done!
				win._wm_layout = true; return true;
			end
		end
	end

	-- Otherwise we're just floating, layout by position.
	win:SetClampedToScreen(wmdata.cts);
	return _DoFixedLayout(win, posf);
end

-- Layout a dock group.
local function LayoutDockGroup(w)
	RDXWM:Debug(4, "LayoutDockGroup(" .. w:WMGetIdentity() ..")");
	
	if RDXWM.IsDocked(w) then
		-- Find the dock group's parent; start there.
		local dgp = FindDockGroupParent(w);
		if not dgp then
			RDXWM:Debug(4, "LayoutDockGroup(" .. w:WMGetIdentity() .."): window doesn't have a DGP.");
			dgp = w;
		end
		IterateDockGroup(dgp, LayoutWindow);
	else
		LayoutWindow(w);
	end
end


-- Layout all windows
local function LayoutAll()
	for _,wi in pairs(identMap) do
		if (not wi._wm_nolayout) and (not wi._wm_layout) then
			LayoutDockGroup(wi);
		end
	end
end

-------------------------------------------------------------------------
-- DOCKING CORE
-------------------------------------------------------------------------
-- Distance functions
local dockDistance = 15;
local dist = function(x1, y1, x2, y2)
	local dx, dy = x2-x1, y2-y1; 
	return math.sqrt(dx*dx + dy*dy);
end

-- Undock the given window from all its neighbors.
local function CompletelyUndock(win, bump)
	if not IsManagedWindow(win) then return; end
	if win.secure and InCombatLockdown() then
		VFL.TripError("RDX", i18n("Attempt to undock secure window while in combat."), i18n("Window ID <") .. win:WMGetIdentity() .. ">");
		return;
	end
	-- Save my last known good position
	RDXWM.SavePosition(win);
	-- If I was the DGP, get rid of it.
	SetDGP(win, nil);
	-- Reset the layout for my dock group
	ResetDockGroupLayout(win);
	-- Undock all attached windows.
	local di = GetDockPoints(win);
	if di then
		-- For each dock point...
		for lp,ri in pairs(di) do
			-- Remove the local docking
			di[lp] = nil;
			-- Try to get the remote window...
			local rw = GetWindowByIdentity(ri.id);
			if rw then
				-- Remove the reciprocal docking entry.
				UndockOther(rw, ri.point, win);
				-- Try to find the remote window's dock parent. If it doesnt' have one,
				-- make IT into the new docking parent!
				if not FindDockGroupParent(rw) then 
					RDXWM:Debug(7, "Undock(<" .. win:WMGetIdentity() .. ">): undockee <" .. rw:WMGetIdentity() .. "> is dock-orphaned; promoting to DGP.");
					MakeDockGroupParent(rw); 
				end
			end
		end
	end
	if bump then
		if win._wm_data.t then win._wm_data.t = win._wm_data.t - 5; end
		if win._wm_data.l then win._wm_data.l = win._wm_data.l + 5; end
	end
	LayoutAll();
end

-- Attempt to create the table entries necessary to dock the given window with 
-- the other window at the given attach points
local function DoDock(win1, pt1, win2, pt2)
	RDXWM:Debug(8, "->DoDock(" .. win1:WMGetIdentity() .. "," .. pt1 .. ":" .. win2:WMGetIdentity() .. "," .. pt2);
	-- Get the docking info
	local di1, di2 = GetDockPoints(win1), GetDockPoints(win2);
	-- Check if this docking has already been made.
	if (di1 and di2) and (di1[pt1] and di2[pt2]) then
		if (di1[pt1].id == win2:WMGetIdentity()) and (di2[pt2].id == win1:WMGetIdentity()) then
			RDXWM:Debug(8, "* Dock already made.");
			return true;
		end
	end
	-- If the docking hasn't already been made, but either of the two dock points are occupied, reject.
	if (di1 and di1[pt1]) or (di2 and di2[pt2]) then
		RDXWM:Debug(8, "* Dockpoint already occupied.");
		return false;
	end

	-- The "source" window loses its dock ancestry
	ClearDockGroupParent(win1);
	-- Generate the new docking structure.
	CreateDockEntry(win1, pt1, win2:WMGetIdentity(), pt2);
	CreateDockEntry(win2, pt2, win1:WMGetIdentity(), pt1);

	-- If there wasn't already a remote docking structure, create it.
	if not FindDockGroupParent(win2) then 
		RDXWM:Debug(8, "* Making " .. win2:WMGetIdentity() .. " the DGP.");
		MakeDockGroupParent(win2); 
	end
	
	RDXWM:Debug(8, "* Docked.");
	return true;
end

-- Check if a dock should be considered legal between the docker and the target
local function IsDockLegal(docker, target)
	-- Sanity check inputs
	if (not IsManagedWindow(docker)) or (not IsManagedWindow(target)) then return false; end
	-- Can't dock to self.
	if docker == target then return false; end
	-- No-layout flag on docker = no good.
	if docker._wm_nolayout then return false; end
	-- If docker is secure and we're locked down, no good.
	if docker.secure and InCombatLockdown() then return false; end
	-- Local checks
	if not target:WMCanOtherDockToMe(docker) then return false; end
	if not docker:WMCanIDockToOther(target) then return false; end
	-- Sometimes windows don't have coordinates; reject these windows.
	if (not docker:WMGetDockBoundary()) or (not target:WMGetDockBoundary()) then return false; end
	-- OK
	return true;
end

-- Check if the docker is in dock interaction range of the target.
local function CheckDockInteraction(docker, target)
	RDXWM:Debug(7, "CheckDockInteraction(" .. docker:WMGetIdentity() .. ", " .. target:WMGetIdentity() .. ")");
	if not IsDockLegal(docker, target) then return false; end
	if Interdocked(docker, target) then return false; end
	-- Honor noattach and nohold flags.
	local dLeft, dTop, dRight, dBottom = docker:WMGetDockBoundary();
	local tLeft, tTop, tRight, tBottom = target:WMGetDockBoundary();
	-- Check if my topleft is near his bottomleft (bottom edge join)
	if (dist(dLeft, dTop, tLeft, tBottom) < dockDistance) then
		VFLUI:Debug(7, "* TOPLEFT->BOTTOMLEFT (Bottom edge join)");
		return DoDock(docker, "TOPLEFT", target, "BOTTOMLEFT");
	end
	-- Check if my topleft is near his topright (right edge join)
	if (dist(dLeft, dTop, tRight, tTop) < dockDistance) then
		VFLUI:Debug(7, "* TOPLEFT->TOPRIGHT (Right edge join)");
		return DoDock(docker, "TOPLEFT", target, "TOPRIGHT");
	end
	-- Check if my bottomleft is near his topleft (top edge join)
	if (dist(dLeft, dBottom, tLeft, tTop) < dockDistance) then
		VFLUI:Debug(7, "* BOTTOMLEFT->TOPLEFT (Top edge join)");
		return DoDock(docker, "BOTTOMLEFT", target, "TOPLEFT");
	end
	-- Check if my topright is near his topleft (left edge join)
	if (dist(dRight, dTop, tLeft, tTop) < dockDistance) then
		VFLUI:Debug(7, "* TOPRIGHT->TOPLEFT (Left edge join)");
		return DoDock(docker, "TOPRIGHT", target, "TOPLEFT");
	end
	-- Check if my bottomright is near his topright (left edge join)
	if (dist(dRight, dBottom, tRight, tTop) < dockDistance) then
		VFLUI:Debug(7, "* BOTTOMRIGHT->TOPRIGHT (Top edge join)");
		return DoDock(docker, "BOTTOMRIGHT", target, "TOPRIGHT");
	end
	-- No dice.
	return nil;
end

----------------------------------------------------------------------------
-- WM API
----------------------------------------------------------------------------

--- Programmatically dock two windows.
-- w1 will be docked to w2, which will be the parent in the relationship.
-- w1 will have its current layout obliterated in the process.
function RDXWM.ForceDock(w1, p1, w2, p2)
	-- Sanity check the input windows
	if not IsDockLegal(w1, w2) then return false; end
	CompletelyUndock(w1, nil);
	return DoDock(w1, p1, w2, p2);
end

--- Force a re-layout operation on the given window.
function RDXWM._Layout(win)
	Unlayout(win); LayoutWindow(win);
end

--- Save a window's current position.
function RDXWM.SavePosition(win)
	local rgn = win:WMGetPositionalFrame();
	if (not rgn) or (not rgn:GetLeft()) then return; end -- BUGFIX: sometimes there are no bdry coordinates?
	local l,t,r,b = GetUniversalBoundary(rgn);
	local d = win._wm_data;
	d.l = l; d.t = t; d.r = r; d.b = b;
end

--- Get the saved position of a window
function RDXWM.GetSavedPosition(win)
	local d = win._wm_data;
	if not d then return nil; end
	return d.l, d.t, d.r, d.b;
end

--- Determine if this frame is docked
function RDXWM.IsDocked(win)
	local d = win._wm_data.dock;
	if not d then return nil; end
	if(VFL.tsize(d) > 0) then return true; else return nil; end
end

--- Determine if this frame is being dragged.
function RDXWM.IsBeingDragged(win)
	return win._wm_drag;
end

--- Close a window
function RDXWM.CloseWindow(win)
	-- BUGFIX: make sure we're still a managed frame...
	if not IsManagedWindow(win) then return; end
	-- Stop any pending dragops
	win:WMStopDrag(true); 
	-- Mark the window as closed
	win._wm_data.open = nil;
	-- Don't allow secure windows to be closed in combat
	if win.secure and InCombatLockdown() then
		VFL.TripError("RDX", i18n("Window Manager: Attempt to close a secure window while in combat."), i18n("Window ID: ") .. win:WMGetIdentity());
	end
	-- Verify metadata
	local cls, id = win:WMGetClass(), win:WMGetIdentity();
	if (not cls) or (not cls.Close) then return; end
	-- If the window is closeable...
	if cls.Close(win, id) then
		CompletelyUndock(win);
		identMap[id] = nil;
		return true;
	else
		return nil;
	end
end

--- Rebuild all classful windows
function RDXWM.RebuildAll()
	for id,wi in pairs(identMap) do
		local cls = wi:WMGetClass();
		if cls and cls.Rebuild then cls.Rebuild(wi, id); end
	end
end

--------------------------------------------------------------------------------------------------
-- INDIVIDUAL WINDOW FUNCTIONALITY
--------------------------------------------------------------------------------------------------
-- The default frame abstraction handlers.
local function Default_GetPositionalFrame(win)
	return win;
end
local function Default_GetDockPoint(win, point)
	return point, 0, 0;
end
local function Default_GetDockBoundary(win)
	local df = win:WMGetPositionalFrame();
	return GetUniversalBoundary(df);
end
local function Default_CanOtherDockToMe(thisWin, otherWin)
	-- If I am secure, don't allow a non secure to dock to me.
--	if thisWin.secure and (not otherWin.secure) then return nil; end
	-- Honor the nohold attribute
	if thisWin._wm_data.nohold then return nil; end
	return true;
end
local function Default_CanIDockToOther(thisWin, otherWin)
	-- If I am not secure, don't allow me to dock to a secure
--	if (not thisWin.secure) and otherWin.secure then return nil; end
	-- Honor the nograb attribute
	if thisWin._wm_data.noattach then return nil; end
	return true;
end


-- Default drag handlers
local function Default_WMDrag(self)
	if (self._wm_nolayout) or (self._wm_drag) then return; end
	-- When we drag, we're actually dragging the dock group parent.
	local dragTarget = FindDockGroupParent(self) or self; 
	if self.OnDrag then self:OnDrag(true); end
	self._wm_drag = dragTarget;
	dragTarget:WMGetPositionalFrame():StartMoving();
end
local function Default_WMStopDrag(self, quick)
		-- Sanity checks
	if (self._wm_nolayout) or (not self._wm_drag) then return; end
	-- Stop moving
	self._wm_drag:WMGetPositionalFrame():StopMovingOrSizing();
	if self.OnDrag then self:OnDrag(nil); end
	RDXWM.SavePosition(self._wm_drag);
	self._wm_drag = nil;
	-- Don't allow any docking to happen while in combat.
	if quick or InCombatLockdown() then return; end

	-- Figure out if this frame is dockable to any other frames nearby
	local iDocked = nil;
	for id2,w2 in pairs(identMap) do
		if CheckDockInteraction(self, w2) then iDocked = true; break; end
	end
	-- Relayout the dock group.
	if iDocked then
		ResetDockGroupLayout(self);
		LayoutDockGroup(self);
	end
	-- Save the new position for the entire dock group.
	IterateDockGroup(self, function(x) RDXWM.SavePosition(x); end);
end

local function Default_GetClass(self)
	return RDXWM.GetWindowClass(self._wm_data.class);
end

local function WMRebuild(self)
	ResetDockGroupLayout(self);
	LayoutDockGroup(self);
end

local function WMSetSecure(self, flag)
end

-- Undo what was done by ImbueManagedFrame.
local function UnimbueManagedFrame(frame)
	-- Clear vars
	frame._wm_layout = nil; frame._wm_nolayout = nil;
	frame._wm_drag = nil; frame._wm_data = nil;
	-- Quash functions
	frame.WMDrag = nil; frame.WMStopDrag = nil;
	frame.WMGetIdentity = nil; frame.WMGetClass = nil;
	frame.WMSetSecure = nil; frame.WMRebuild = nil;

	frame.WMGetPositionalFrame = nil; frame.WMGetDockBoundary = nil;
	frame.WMGetDockTargetPoint = nil; frame.WMGetDockSourcePoint = nil;
	frame.WMCanOtherDockToMe = nil; frame.CanIDockToOther = nil;
end

-- Make a frame into a WM-managed frame.
local function ImbueManagedFrame(frame, ident, data)
	if frame._wm_data then error(i18n("ImbueManagedFrame on an already managed frame")); end

	-- Setup default internals
	frame._wm_data = data; frame._wm_layout = nil;

	--- Non-overridable default methods
	function frame:WMGetIdentity()
		return ident;
	end
	frame.WMGetClass = Default_GetClass; 	frame.WMSetSecure = WMSetSecure;
	frame.WMDrag = Default_WMDrag; frame.WMStopDrag = Default_WMStopDrag;
	frame.WMRebuild = WMRebuild;

	-- Overridable methods
	if not frame.WMGetPositionalFrame then frame.WMGetPositionalFrame = Default_GetPositionalFrame; end
	if not frame.WMGetDockBoundary then frame.WMGetDockBoundary = Default_GetDockBoundary; end
	if not frame.WMGetDockTargetPoint then frame.WMGetDockTargetPoint = Default_GetDockPoint; end
	if not frame.WMGetDockSourcePoint then frame.WMGetDockSourcePoint = Default_GetDockPoint; end
	if not frame.WMCanOtherDockToMe then frame.WMCanOtherDockToMe = Default_CanOtherDockToMe; end
	if not frame.WMCanIDockToOther then frame.WMCanIDockToOther = Default_CanIDockToOther; end

	-- Destructor
	frame.Destroy = VFL.hook(function(s)
		-- Destroy mark if it exists
		if s._wm_mark then
			VFLUI.ReleaseRegion(s._wm_mark);
			s._wm_mark = nil;
		end
		-- Destroy layout
		Unlayout(s);
		-- Remove this frame's record as an open frame
		identMap[s:WMGetIdentity()] = nil;
		-- Destroy managed frame entries
		UnimbueManagedFrame(s);
	end, frame.Destroy);
end

-------------------------------------------------------------------------------
-- THE DESKTOP OBJECT
--
-- A Desktop is a window manager database. It stores the positioning and layout
-- information for all visible windows.
-------------------------------------------------------------------------------

-- Set the currently active desktop to the database pointed to by newdb.
local function _SetActiveDesktop(newdb, extraEventArg, override)
	-- Prevent accidental noop
	if (not override) and (db == newdb) then return; end

	-- PASS 1: Destroy all extant windows that aren't open on the new desktop.
	RDXWM:Debug(3, "> Desktop pass1 (Close closeables)");
	-- For each id,window pair currently active...
	for ident,win in pairs(identMap) do
		Unlayout(win);
		local cls = win:WMGetClass();
		-- If the window has a class, and it has no entry in the new database, or
		-- it is closed in the new database...
		if cls then
			if ((not newdb[ident]) or (not newdb[ident].open)) then
				RDXWM:Debug(3, "* Closing window " .. ident);
				-- Remove it from the active data structures
				if cls.Close(win, ident) then
					identMap[ident] = nil;
				end
			end
		end
	end
	
	-- PASS 2: Apply the new data set to each window that exists in the new DB.
	RDXWM:Debug(3, "> Desktop pass2 (Imbue new data on old windows)");
	for ident,win in pairs(identMap) do
		if newdb[ident] then
			RDXWM:Debug(4,"* Imbuing window id: " .. ident);
			win._wm_data = newdb[ident];
			Unlayout(win);
		end
	end
	
	-- PASS 3: Recreate all windows in the new database that don't already exist
	RDXWM:Debug(3, "> Desktop pass3 (Create open windows that didn't exist)");
	for ident,dbe in pairs(newdb) do
		if (not identMap[ident]) and (dbe.class) and (dbe.open) then
			local wclass = RDXWM.GetWindowClass(dbe.class)
			if wclass then
				RDXWM:Debug(1, "* opening saved window '" .. ident .. "' of class '" .. dbe.class .. "'");
				local newWin = wclass.Open(ident, dbe);
				if newWin then
					ImbueManagedFrame(newWin, ident, dbe);
					identMap[ident] = newWin;
				else
					RDXWM:Debug(1,"* could not reopen window '" .. ident .. "', ignoring.");
				end
			else
				RDXWM:Debug(1, "* ignoring saved window of nonexistent class '" .. dbe.class .. "'");
			end
		end
	end

	-- Update the database pointer
	db = newdb;

	-- Inform interested parties that the desktop has changed.
	RDXEvents:Dispatch("DESKTOP_CHANGED", db, extraEventArg);

	-- Now, for every classless window still open, make sure it has a new database entry.
	for ident,win in pairs(identMap) do
		if not win:WMGetClass() then
			if not db[ident] then
				db[ident] = CreateBlankWindowProfile();
				win._wm_data = db[ident];
				RDXWM:Debug(1, "* creating new profile for classless window <" .. ident .. ">");
			end
		end
	end

	-- Now redo the layouts.
	RDXWM:Debug(3, "> Desktop LAYOUT!");
	LayoutAll();
end

local curDesktopPath = nil;

--- Load the window database described by the given table
function RDXWM.LoadDesktop(newdb)
	_SetActiveDesktop(newdb);
end

--- Load a desktop by file path.
function RDXWM.LoadDesktopFile(fPath, fData)
	RDXWM:Debug(1, "LoadDesktopFile<", fPath, ">");
	if not fData.data then return; end
	curDesktopPath = fPath;
	_SetActiveDesktop(fData.data, fPath);
end

--- Get the package:file for the currently active desktop.
function RDXWM.GetCurrentDesktopPath()
	return curDesktopPath;
end
-----------------------------------------------------------------
-- LOCKDOWN ACTION QUEUE
-- Execute a series of actions after combat lockdown ends.
-----------------------------------------------------------------
local caq = {};
function RDXWM.QueueLockdownAction(object, method)
	if not InCombatLockdown() then method(object); end
	if not caq[object] then caq[object] = method; end
end

VFLEvents:Bind("PLAYER_COMBAT", nil, function(flag)
	if not flag then
		for k,v in pairs(caq) do
			v(k); caq[k] = nil;
		end
	end
end);

-----------------------------------------------------------------
-- REBUILD QUEUE
-- In WoW 2.0, we can't rebuild windows in combat. Use a queue system
-- to queue up windows to be rebuilt while in combat, and then just
-- rebuild them once we get out.
-----------------------------------------------------------------
local rbq = {};
local function DoRebuild(w)
	RDXWM:Debug(2, "Rebuilding window: ", w:WMGetIdentity());
	local cls = w:WMGetClass();
	if cls then cls.Rebuild(w, w:WMGetIdentity()); end
end

--- Rebuild a single window.
function RDXWM.RebuildWindow(w)
	if InCombatLockdown() then
		rbq[w] = true;
	else
		DoRebuild(w);
	end
end

VFLEvents:Bind("PLAYER_COMBAT", nil, function(flag)
	if not flag then
		for k,_ in pairs(rbq) do
			DoRebuild(k); rbq[k] = nil;
		end
	end
end);

--- Rebuild all secure windows.
--[[
function RDXWM.RebuildSecureWindows()
	RDXWM:Debug(2, "Rebuilding secure windows");
	for ident,win in pairs(identMap) do
		if win.secure then
			RDXWM.RebuildWindow(win);
		end
	end
end

-- On switching from party to raid, rebuild all secure windows.
RDXEvents:Bind("PARTY_IS_RAID", nil, RDXWM.RebuildSecureWindows);
RDXEvents:Bind("PARTY_IS_NONRAID", nil, RDXWM.RebuildSecureWindows);
]]--

-----------------------------------------------------------------
-- WINDOW DATABASE MANAGEMENT
--
-- Functions for creating and examining classful and classless windows on the
-- given desktop.
-----------------------------------------------------------------

--- If an object with the given identity has a WM profile (but not necessarily instantiated)
-- it is returned. Otherwise returns NIL.
function RDXWM.GetProfile(ident)
	if not ident then return nil; end
	return db[ident];
end

--- If an object with the given identity is currrently instantiated in the WM, returns
-- it. Otherwise returns NIL.
RDXWM.GetWindow = GetWindowByIdentity;

--- Check if an object with the given identity has an "open" flag in the WM
function RDXWM.IsProfileOpen(ident)
	if not ident then return nil; end
	local data = db[ident]; if not data then return nil; end
	return data.open;
end

--- Use a window class's Open method to add a new window to this desktop.
function RDXWM.OpenWindow(id, className)
	if (not db) then error(i18n("Attempt to call RDXWM.OpenWindow() before WM init.")); end
	if (not id) then error(i18n("Usage: RDXWM.OpenWindow(className, id)")); end
	local cls = RDXWM.GetWindowClass(className);
	if not cls then error(i18n("Invalid window class name.")); end
	-- Check if the window with this ID is already open
	if identMap[id] then return nil, i18n("A window with that identity is already open"); end
	-- Determine if we have data on this window; create if not
	local d = db[id];
	if not d then
		d = CreateBlankWindowProfile(); db[id] = d;
		d.class = className;
	else
		if d.class ~= className then
			return nil, i18n("Window '") .. id .. i18n("' class conflict.");
		end
	end
	-- Now create the window object
	d.open = true;
	local win = cls.Open(id);
	if not win then return nil, className .. i18n(".Open failed"); end
	ImbueManagedFrame(win, id, d);
	identMap[id] = win;
	-- Layout the window
	ResetDockGroupLayout(win); LayoutDockGroup(win);
	-- Done
	return win;
end

--- Add a user-controlled classless window to this desktop.
function RDXWM.ManageWindow(frame, id)
	if not db then error(i18n("Attempt to call RDXWM.ManageWindow() before WM init.")); end
	if (not frame) or (not id) then error(i18n("Usage: RDXWM.ManageWindow(frame, id)")); end
	if identMap[id] then return error(i18n("RDXWM.ManageWindow: duplicate id ") .. tostring(id)); end
	-- DB entry
	local d = db[id];
	if not d then
		d = CreateBlankWindowProfile(); db[id] = d;
	end
	-- Layout setup
	ImbueManagedFrame(frame, id, d);
	identMap[id] = frame;
	ResetDockGroupLayout(frame); LayoutDockGroup(frame);
	return frame;
end

--- Add a standard "move handle" to the given managed frame
function RDXWM.StdMove(w, handle, clickFunc)
	if type(clickFunc) ~= "function" then clickFunc = VFL.Noop; end
	handle:SetScript("OnMouseDown", function()
		if(arg1 == "LeftButton") and IsShiftKeyDown() then w:WMDrag(); end
	end);
	handle:SetScript("OnMouseUp", function()
		if(arg1 == "LeftButton") then
			if w._wm_drag then
				w:WMStopDrag();
			else
				clickFunc(w);
			end
		elseif(arg1 == "RightButton") then
			RDXWM.WindowProperties(w);
		end
	end);
end

--- Close a window by identity, if it exists.
function RDXWM.CloseWindowByID(id)
	if not id then return; end
	local w = identMap[id]; if not w then return; end
	RDXWM.CloseWindow(w);
end

--- Delete a window's profile by identity.
function RDXWM.DestroyProfile(id)
	if not id then return; end
	RDXWM.CloseWindowByID(id);
	db[id] = nil;
end

-------------------------------------------------------
-- Props (properties) menu
--
-- The props are assembled by listing the basic WM properties
-- (undock, hide, close, etc) followed by any custom entries specified
-- by the window class's PropsMenu routine
-------------------------------------------------------
function RDXWM.WindowProperties(frame)
	-- Make sure this is a managed frame.
	-- Properties menus are not available during InCombatLockdown().
	if (not frame) or (not frame._wm_data) or (not frame:IsVisible()) or InCombatLockdown() then 
		return; 
	end
	local mnu = {};
	table.insert(mnu, { text = i18n("Layout..."), OnClick = function() RDXWM.LayoutPropsDialog(frame); VFL.poptree:Release(); end });
	table.insert(mnu, { text = i18n("Drag to move"), 
		OnMouseDown = function() frame:WMDrag(); end, 
		OnMouseUp = function() VFL.poptree:Release(); frame:WMStopDrag(); end 
	});
	if RDXWM.IsDocked(frame) then
		table.insert(mnu, {text = i18n("Undock"), OnClick = function() CompletelyUndock(frame, true); VFL.poptree:Release(); end });
		if not IsDGP(frame) then
			table.insert(mnu, {text = i18n("Make Dock Parent"), OnClick = function() 
				VFL.poptree:Release();
				RDXWM.SavePosition(frame);
				MakeDockGroupParent(frame);
				ResetDockGroupLayout(frame); LayoutDockGroup(frame);
			end });
		end
	end
	local cls = frame:WMGetClass()
	if cls then
		table.insert(mnu, { text = i18n("Close"), OnClick = function() 
			VFL.poptree:Release();
			RDXWM.CloseWindow(frame);
		end });
		if cls.Props then
			cls.Props(frame, frame:WMGetIdentity(), mnu);
		end
	end
	-- Allow downstream menu hooks for class-less windows.
	if type(frame._WindowMenu) == "function" then
		frame:_WindowMenu(mnu, frame:WMGetIdentity());
	end
	
	VFL.poptree:Begin(150, 12, frame, "TOPLEFT", GetRelativeLocalMousePosition(frame));
	VFL.poptree:Expand(nil, mnu);
end

-------------------------------------------------------
-- Props dialog
-- Allows scale, alpha, stratum, level to be changed
-------------------------------------------------------
local _strata = {
	{ text = "BACKGROUND" },
	{ text = "LOW" },
	{ text = "MEDIUM" },
	{ text = "HIGH"} ,
	{ text = "DIALOG" } ,
	{ text = "FULLSCREEN" } ,
	{ text = "FULLSCREEN_DIALOG" } ,
	{ text = "TOOLTIP" } ,
};
local function _fnStrata() return _strata; end

local _anchors = {
	{ text = "Auto" },
	{ text = "TOPLEFT" },
	{ text = "TOPRIGHT" },
	{ text = "BOTTOMLEFT" },
	{ text = "BOTTOMRIGHT" },	
};
local function _fnAnchors() return _anchors; end

local function _MakeNudgeButton(parent, texture, dgp, dx, dy)
	local btn = VFLUI.Button:new(parent);
	btn:SetWidth(24); btn:SetHeight(24); btn:Show();
	local tex = VFLUI.CreateTexture(btn);
	tex:SetPoint("TOPLEFT", btn, "TOPLEFT", 6, -6);
	tex:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", -6, 6);
	tex:SetDrawLayer("OVERLAY"); tex:SetTexture(texture); tex:Show();
	btn.Destroy = VFL.hook(function() VFLUI.ReleaseRegion(tex); end, btn.Destroy);

	btn:SetScript("OnClick", function()
		local wdata = dgp._wm_data; if not wdata then return; end
		if not wdata.l then RDXWM.SavePosition(dgp); end
		if not wdata.l then return; end
		
		wdata.l = wdata.l + dx; wdata.r = wdata.r + dx;
		wdata.t = wdata.t + dy; wdata.b = wdata.b + dy;
		ResetDockGroupLayout(dgp); LayoutDockGroup(dgp); 
		RDXWM.SavePosition(dgp);
	end);
	return btn;
end

function RDXWM.LayoutPropsDialog(frame)
	if (not frame) or (not IsManagedWindow(frame)) then return; end
	local dd, dgp = frame._wm_data, (FindDockGroupParent(frame) or frame);

	local x = nil;
	
	local dlg = VFLUI.Window:new(VFLTop);
	VFLUI.Window.SetDefaultFraming(dlg, 20);
	dlg:SetTitleColor(0,0,.6);
	dlg:SetText(i18n("Layout Properties"));
	dlg:SetWidth(250); dlg:SetHeight(300); dlg:SetPoint("CENTER", UIParent, "CENTER");
	dlg:Show();
	local ca = dlg:GetClientArea();

	local lblScale = VFLUI.MakeLabel(nil, dlg, i18n("Scale"));
	lblScale:SetPoint("TOPLEFT", ca, "TOPLEFT"); lblScale:SetHeight(25);
	local edScale = VFLUI.Edit:new(dlg, true); edScale:SetHeight(25); edScale:SetWidth(50);
	edScale:SetPoint("TOPRIGHT", ca, "TOPRIGHT", 0, 0); edScale:Show();
	local slScale = VFLUI.HScrollBar:new(dlg);
	slScale:SetPoint("RIGHT", edScale, "LEFT", -16, 0); slScale:SetWidth(100);
	slScale:Show();
	slScale:SetMinMaxValues(.1, 3.0); slScale:SetValue(dd.scale);
	VFLUI.BindSliderToEdit(slScale, edScale);


	local lblAlpha = VFLUI.MakeLabel(nil, dlg, i18n("Alpha"));
	lblAlpha:SetPoint("TOPLEFT", lblScale, "BOTTOMLEFT"); lblAlpha:SetHeight(25);
	local edAlpha = VFLUI.Edit:new(dlg, true); edAlpha:SetHeight(25); edAlpha:SetWidth(50);
	edAlpha:SetPoint("TOPRIGHT", edScale, "BOTTOMRIGHT", 0, 0); edAlpha:Show();
	local slAlpha = VFLUI.HScrollBar:new(dlg);
	slAlpha:SetPoint("RIGHT", edAlpha, "LEFT", -16, 0); slAlpha:SetWidth(100);
	slAlpha:Show();
	slAlpha:SetMinMaxValues(0, 1); slAlpha:SetValue(dd.alpha);
	VFLUI.BindSliderToEdit(slAlpha, edAlpha);

	local lblStrata = VFLUI.MakeLabel(nil, dlg, i18n("Stratum"));
	lblStrata:SetPoint("TOPLEFT", lblAlpha, "BOTTOMLEFT"); lblStrata:SetHeight(25);
	local ddStrata = VFLUI.Dropdown:new(dlg, _fnStrata);
	ddStrata:SetPoint("TOPRIGHT", edAlpha, "BOTTOMRIGHT", 0, 0); ddStrata:SetWidth(132);
	ddStrata:SetSelection(dd.strata);
	ddStrata:Show();

	local lblAP = VFLUI.MakeLabel(nil, dlg, i18n("Anchor point"));
	lblAP:SetPoint("TOPLEFT", lblStrata, "BOTTOMLEFT"); lblAP:SetHeight(25);
	local ddAP = VFLUI.Dropdown:new(dlg, _fnAnchors);
	ddAP:SetPoint("TOPRIGHT", ddStrata, "BOTTOMRIGHT", 0, 0); ddAP:SetWidth(132);
	ddAP:SetSelection(dd.ap or "Auto");
	ddAP:Show();

	local chkCTS = VFLUI.Checkbox:new(dlg); chkCTS:Show();
	chkCTS:SetHeight(20); chkCTS:SetWidth(200);
	chkCTS:SetPoint("TOPLEFT", lblAP, "BOTTOMLEFT");
	chkCTS:SetText(i18n("Clamp to screen (only if undocked)"));
	chkCTS:SetChecked(dd.cts);

	local chkNoAttach = VFLUI.Checkbox:new(dlg); chkNoAttach:Show();
	chkNoAttach:SetHeight(20); chkNoAttach:SetWidth(250);
	chkNoAttach:SetPoint("TOPLEFT", chkCTS, "BOTTOMLEFT");
	chkNoAttach:SetText(i18n("Prevent this window from attaching to others"));
	chkNoAttach:SetChecked(dd.noattach);

	local chkNoHold = VFLUI.Checkbox:new(dlg); chkNoHold:Show();
	chkNoHold:SetHeight(20); chkNoHold:SetWidth(250);
	chkNoHold:SetPoint("TOPLEFT", chkNoAttach, "BOTTOMLEFT");
	chkNoHold:SetText(i18n("Prevent other windows from attaching to this one"));
	chkNoHold:SetChecked(dd.nohold);

	local lblNudge = VFLUI.MakeLabel(nil, dlg, i18n("Nudge"));
	lblNudge:SetPoint("TOPLEFT", chkNoHold, "BOTTOMLEFT", 0, -25);
	local nudgeLeft = _MakeNudgeButton(dlg, "Interface\\Addons\\VFL\\Skin\\sb_left", dgp, -1, 0);
	nudgeLeft:SetPoint("LEFT", lblNudge, "RIGHT");
	local nudgeUp = _MakeNudgeButton(dlg, "Interface\\Addons\\VFL\\Skin\\sb_up", dgp, 0, 1);
	nudgeUp:SetPoint("BOTTOMLEFT", nudgeLeft, "RIGHT");
	local nudgeDown = _MakeNudgeButton(dlg, "Interface\\Addons\\VFL\\Skin\\sb_down", dgp, 0, -1);
	nudgeDown:SetPoint("TOPLEFT", nudgeLeft, "RIGHT");
	local nudgeRight = _MakeNudgeButton(dlg, "Interface\\Addons\\VFL\\Skin\\sb_right", dgp, 1, 0);
	nudgeRight:SetPoint("LEFT", nudgeDown, "TOPRIGHT");

	local txtCurDock = VFLUI.CreateFontString(dlg);
	txtCurDock:SetPoint("TOPRIGHT", chkNoHold, "BOTTOMRIGHT");
	txtCurDock:SetWidth(130); txtCurDock:SetHeight(60);
	txtCurDock:SetJustifyV("TOP");
	txtCurDock:SetJustifyH("LEFT");
	txtCurDock:SetFontObject(Fonts.Default10); txtCurDock:Show();
	local str = i18n("Docks:\n");
	if dd.dock then
		for k,v in pairs(dd.dock) do
			str = str .. k .. ": " .. v.id .. "\n";
		end
	else
		str = str .. "(none)";
	end
	txtCurDock:SetText(str);

	----------- ok/cancel
	local btnCancel = VFLUI.CancelButton:new(dlg);
	btnCancel:SetHeight(25); btnCancel:SetWidth(75); btnCancel:SetPoint("BOTTOMRIGHT", ca, "BOTTOMRIGHT");
	btnCancel:SetText(i18n("Cancel")); btnCancel:Show();
	btnCancel:SetScript("OnClick", function() dlg:Destroy(); end);

	local btnOK = VFLUI.OKButton:new(dlg);
	btnOK:SetHeight(25); btnOK:SetWidth(75); btnOK:SetPoint("RIGHT", btnCancel, "LEFT");
	btnOK:SetText(i18n("OK")); btnOK:Show();
	btnOK:SetScript("OnClick", function()
		dd.scale = slScale:GetValue();
		dd.alpha = slAlpha:GetValue();
		dd.strata = ddStrata:GetSelection();
		if ddAP:GetSelection() == "Auto" then dd.ap = nil; else dd.ap = ddAP:GetSelection(); end
		dd.cts = chkCTS:GetChecked();
		dd.noattach = chkNoAttach:GetChecked();
		dd.nohold = chkNoHold:GetChecked();
		dlg:Destroy();
		ResetDockGroupLayout(frame);
		LayoutDockGroup(frame);
	end);

	dlg.Destroy = VFL.hook(function(s)
		btnCancel:Destroy(); btnCancel = nil;
		btnOK:Destroy(); btnOK = nil;
		slScale:Destroy(); slScale = nil; edScale:Destroy(); edScale = nil;
		slAlpha:Destroy(); slAlpha = nil; edAlpha:Destroy(); edAlpha = nil;
		ddStrata:Destroy(); ddStrata = nil; ddAP:Destroy(); ddAP = nil;
		chkCTS:Destroy(); chkCTS = nil; chkNoAttach:Destroy(); chkNoAttach = nil;
		chkNoHold:Destroy(); chkNoHold = nil;
		nudgeLeft:Destroy(); nudgeLeft = nil; nudgeRight:Destroy(); nudgeRight = nil;
		nudgeUp:Destroy(); nudgeUp = nil; nudgeDown:Destroy(); nudgeDown = nil;
		VFLUI.ReleaseRegion(txtCurDock); txtCurDock = nil;
	end, dlg.Destroy);
end

-------------------------------------------------------
-- Show/hide entire desktop
-------------------------------------------------------
function RDXWM.IsRDXHidden()
	if RDXU then return RDXU.hidden; else return nil; end
end

function RDXWM.ShowRDX()
	if InCombatLockdown() then 
		RDX.print(i18n("Cannot change show/hide state while in combat."));
		return; 
	end
	if (not RDXWM.IsRDXHidden()) then return; end
	RDXU.hidden = nil;
	RDXParent:Show();
end

function RDXWM.HideRDX()
	if InCombatLockdown() then 
		RDX.print(i18n("Cannot change show/hide state while in combat."));
		return; 
	end
	if RDXWM.IsRDXHidden() then return; end
	RDXU.hidden = true;
	RDXParent:Hide();
end

function RDXWM.ToggleRDX()
	if RDXWM.IsRDXHidden() then
		RDXWM.ShowRDX();
	else
		RDXWM.HideRDX();
	end
end

-- /rdx show and /rdx hide
RDX.RegisterSlashCommand("show", RDXWM.ShowRDX);
RDX.RegisterSlashCommand("hide", RDXWM.HideRDX);

-------------------------------------------------------
-- Lock/unlock entire desktop
-------------------------------------------------------

local lockbtn = VFLUI.AcquireFrame("Button");
local phtex = VFLUI.CreateTexture(lockbtn);
phtex:SetAllPoints(lockbtn);
phtex:Show();
lockbtn:SetHighlightTexture(phtex);
phtex:SetBlendMode("DISABLE");
if RDX._skin == "boomy" then
	lockbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\boomy\\lock");
	phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\lock");
elseif RDX._skin == "kids" then
	lockbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\kids\\encrypted");
	phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\encrypted");
else
	lockbtn:SetNormalTexture("Interface\\Addons\\VFL\\Skin\\minus");
	phtex:SetTexture("Interface\\Addons\\VFL\\Skin\\minus");
end
phtex:SetVertexColor(0.8, 0, 0);

function RDXWM.IsRDXLocked()
	if RDXU then return RDXU.locked; else return nil; end
end

function RDXWM.UnlockRDX()
	if InCombatLockdown() then 
		RDX.print(i18n("Cannot change lock/unlock state while in combat."));
		return; 
	end
	if (not RDXWM.IsRDXLocked()) then return; end
	RDX.print(i18n("Unlocking Frames."));
	RDXU.locked = nil;
	RDXWM.RebuildAll();
	if RDX._skin == "boomy" then
		lockbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\boomy\\key");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\key");
	elseif RDX._skin == "kids" then
		lockbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\kids\\decrypted");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\decrypted");
	else
		lockbtn:SetNormalTexture("Interface\\Addons\\VFL\\Skin\\plus");
		phtex:SetTexture("Interface\\Addons\\VFL\\Skin\\plus");
	end
	phtex:SetVertexColor(0, 0.8, 0);
end

function RDXWM.LockRDX()
	if InCombatLockdown() then 
		RDX.print(i18n("Cannot change lock/unlock state while in combat."));
		return; 
	end
	if RDXWM.IsRDXLocked() then return; end
	RDX.print(i18n("Locking Frames."));
	RDXU.locked = true;
	RDXWM.RebuildAll();
	if RDX._skin == "boomy" then
		lockbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\boomy\\lock");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\lock");
	elseif RDX._skin == "kids" then
		lockbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\kids\\encrypted");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\encrypted");
	else
		lockbtn:SetNormalTexture("Interface\\Addons\\VFL\\Skin\\minus");
		phtex:SetTexture("Interface\\Addons\\VFL\\Skin\\minus");
	end
	phtex:SetVertexColor(0.8, 0, 0);
end

lockbtn:SetScript("OnClick", function()
	if RDXWM.IsRDXLocked() then RDXWM.UnlockRDX();
	else RDXWM.LockRDX();
	end
end);

RDXEvents:Bind("INIT_PRELOAD", nil, function() RDX.AddToolbarButton(lockbtn, true); end);

-------------------------------------------------------
-- LAYOUT RESET
-------------------------------------------------------
function RDX._ResetCurrentDesktop()
	if InCombatLockdown() then 
		RDX.print(i18n("Cannot reset desktop while in combat."));
		return; 
	end
	-- Show everything
	RDXWM.ShowRDX();
	-- Restore us from mini status.
	RDX.Maximize();

	-- If we don't have a current desktop, just bring us back to the default.
	local p = RDXWM.GetCurrentDesktopPath();
	if not p then RDX.SelectDesktop("default:desktop"); return; end

	local data = RDXDB.GetObjectData(p);
	if (not data) or (not data.data) then RDX.SelectDesktop("default:desktop"); return; end
	data = data.data; -- lol

	-- OK, let's iterate over the desktop and kill all stored layouts
	for ident,dbe in pairs(data) do
		dbe.scale = 1; dbe.alpha = 1;
		dbe.dock = nil;
		dbe.l = VFLUI.uxScreenCenter;
		dbe.r = VFLUI.uxScreenCenter;
		dbe.t = VFLUI.uyScreenCenter;
		dbe.b = VFLUI.uyScreenCenter;
	end

	-- Now force-reload the desktop.
	_SetActiveDesktop(data, p, true);

	-- Reset downstream stuff.
	RDXEvents:Dispatch("USER_RESET_UI");
end

RDX.RegisterSlashCommand("reset", function()
	VFLUI.MessageBox(
		i18n("Reset UI"),
		i18n("Are you sure you want to reset the current desktop? All windows will be undocked and moved to the center of the screen."),
		nil,
		i18n("No"),
		VFL.Noop,
		i18n("Yes"),
		RDX._ResetCurrentDesktop
	);
end);
-------------------------------------------------------
-- The Crosshair
-------------------------------------------------------
local xhbtn = VFLUI.AcquireFrame("Button");
local xhtex = VFLUI.CreateTexture(xhbtn);
xhtex:SetAllPoints(xhbtn);
xhtex:Show();
xhbtn:SetHighlightTexture(xhtex);
xhtex:SetBlendMode("DISABLE");
if RDX._skin == "boomy" then
	xhbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\boomy\\find");
	xhtex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\find");
elseif RDX._skin == "kids" then
	xhbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\kids\\hook");
	xhtex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\hook");
else
	xhbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\crosshair");
	xhtex:SetTexture("Interface\\Addons\\RDX\\Skin\\crosshair");
end
xhtex:SetVertexColor(0.6, 0, 0.6);
RDXEvents:Bind("INIT_PRELOAD", nil, function() RDX.AddToolbarButton(xhbtn, true); end);

local xhdrag = VFLUI.AcquireFrame("Frame");
xhdrag:SetFrameStrata("FULLSCREEN_DIALOG");
xhdrag:SetAllPoints(xhbtn); xhdrag:SetMovable(true); xhdrag:Hide();
local xhdtex = VFLUI.CreateTexture(xhdrag);
xhdtex:SetAllPoints(xhdrag); xhdtex:Show();
if RDX._skin == "boomy" then
	xhdtex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\find");
elseif RDX._skin == "kids" then
	xhdtex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\hook");
else
	xhdtex:SetTexture("Interface\\Addons\\RDX\\Skin\\crosshair");
end
xhbtn:SetScript("OnMouseDown", function()
	xhdrag:Show();
	xhdrag:StartMoving();
end);

xhbtn:SetScript("OnMouseUp", function()
	xhdrag:StopMovingOrSizing();
	xhdrag:SetAllPoints(xhbtn);
	xhdrag:Hide();
	-- Figure out which window, if any, the mouse is over.
	for ident, win in pairs(identMap) do
		if MouseIsOver(win) then
			RDXWM.WindowProperties(win);
		end
	end
end);

-------------------------------------------------------
-- Menu
-------------------------------------------------------
-- Old menu code
--[[
local function WMMenu(menu, cell)
	if not show_parent_marks then
		table.insert(mnu, {text = "Show parent marks", OnClick = function() 
			menu:Release();
			show_parent_marks = true;
			ResetLayout(); LayoutAll();
		end});
	else
		table.insert(mnu, {text = "Hide parent marks", OnClick = function() 
			menu:Release();
			show_parent_marks = nil;
			ResetLayout(); LayoutAll();
		end});
	end
end
]]--

RDX.systemMenu:RegisterMenuFunction(function(ent)
	if RDXWM.IsRDXHidden() then
		ent.text = i18n("Show/Hide |cFFFF0000[Hidden]|r");
		ent.OnClick = function() RDXWM.ShowRDX(); VFL.poptree:Release(); end;
	else
		ent.text = i18n("Show/Hide |cFF00FF00[Shown]|r");
		ent.OnClick = function() VFL.poptree:Release(); RDXWM.HideRDX(); end;
	end
end);

RDX.systemMenu:RegisterMenuEntry(i18n("Rebuild Desktop"), nil, function()
	VFL.poptree:Release();
	RDXWM.RebuildAll();
end);

RDX.systemMenu:RegisterMenuFunction(function(ent)
	if RDXWM.IsRDXLocked() then
		ent.text = i18n("Lock/Unlock frames |cFF00FF00[UnLock]|r");
		ent.OnClick = function() VFL.poptree:Release(); RDXWM.UnlockRDX(); end;
	else
		ent.text = i18n("Lock/Unlock frames |cFFFF0000[Lock]|r");
		ent.OnClick = function() VFL.poptree:Release(); RDXWM.LockRDX(); end;
	end
end);


-------------------------------------------------------
-- Core initialization
-------------------------------------------------------
local function WMInit()
	RDXWM:Debug(1, "**************** WMInit() ****************");
	-- Make sure default:desktop exists
	local dd = nil;
	if not RDXDB.CheckObject("default:desktop", "Desktop") then
		RDXDB.GetOrCreatePackage("default");
		dd = RDXDB._DirectCreateObject("default", "desktop");
		dd.ty = "Desktop"; dd.version = 1;
	else
		dd = RDXDB.GetObjectData("default:desktop");
	end

	-- Create an empty database for initialization purposes
	db = {};

	-- Dispatch INIT_WM.
	RDXEvents:Dispatch("INIT_WM"); RDXEvents:DeleteKey("INIT_WM");

	-- Restore hide/show setting
	if RDXU.hidden then
		RDXParent:Hide();
	else
		RDXParent:Show();
	end

	-- If no desktop is loaded then let's load the default shall we.
	if not curDesktopPath then
		RDXWM.LoadDesktopFile("default:desktop", dd)
	end
end

WoWEvents:Bind("PLAYER_LOGIN", nil, WMInit);
