-- Obj_Window.lua
-- RDX - Raid Data Exchange
-- (C)2006 Bill Johnson
--
-- This file contains copyrighted and licensed content. Unlicensed copying is prohibited.
-- See the accompanying LICENSE file for license terms.
--
-- Backend, registration and user interface for the Window object type.

--------------------------------------------------
-- The generic RDX Window object.
-- Provides slots for Create, Destroy, Show, and Hide events.
-- Based on the VFL Window object, which allows custom
-- framing.
--------------------------------------------------
RDX.Window = {};
function RDX.Window:new(parent)
	local self = VFLUI.InvertedControlWindow:new(parent);

	local hide, show, destroy, update = VFL.Noop, VFL.Noop, VFL.Noop, VFL.Noop;
	
	-- Properly invoke destructors.
	local function ProperDestroy()
		self._WindowMenu = nil;
		self:SetScript("OnUpdate", nil);
		if self:IsShown() and hide then hide(self); end
		if destroy then destroy(self); end
		if self.Multiplexer then self.Multiplexer:Close(self); self.Multiplexer = nil; end
		self.RepaintLayout = nil; self.RepaintSort = nil; self.RepaintData = nil; self.RepaintAll = nil;
		hide = nil; show = nil; destroy = nil; update = nil;
	end
	
	function self:UnloadState()
		self:SetClient(nil);
		ProperDestroy();
		self:TearDown();
	end

	function self:LoadState(state)
		self.RepaintAll = state:GetSlotFunction("RepaintAll");
		self.RepaintLayout = state:GetSlotFunction("RepaintAll"); -- COMPAT
		self.RepaintData = state:GetSlotFunction("RepaintData");
		self.RepaintSort = state:GetSlotFunction("RepaintSort");
		-- Run assembly functions
		(state:GetSlotFunction("Assemble"))(state, self);
		(state:GetSlotFunction("Create"))(self);
		-- Get API
		hide = state:GetSlotFunction("Hide");	show = state:GetSlotFunction("Show");
		update = state:GetSlotFunction("Update");	destroy = state:GetSlotFunction("Destroy");
		-- Bind API to window.
		if self:IsShown() then show(self); end
		self:SetScript("OnHide", hide);	self:SetScript("OnShow", show);
		if update ~= VFL.Noop then self:SetScript("OnUpdate", update); end
	end

	self.Destroy = VFL.hook(function(s)
		ProperDestroy();
		s.UnloadState = nil; s.LoadState = nil; s.Multiplexer = nil;
	end, self.Destroy);

	return self;
end

-----------------------------------------------
-- The WindowState for a generic window.
-----------------------------------------------
RDX.WindowState = {};
function RDX.WindowState:new()
	local st = RDX.ObjectState:new();
	st.OnResetSlots = function(state)
		state:AddSlot("Create", true);
		state:AddSlot("Hide", true);
		state:AddSlot("Show", true);
		state:AddSlot("Destroy", true);
		state:AddSlot("Update", true);
		state:AddSlot("Window", nil);
	end;
	st:Clear();
	return st;
end

--- The No Event Hinting feature - disables event hinting on the underlying window.
RDX.RegisterFeature({
	name = "No Hinting";
	IsPossible = function(state)
		if not state:Slot("Window") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		return true;
	end;
	ApplyFeature = function(desc, state)
		local mux = state:GetSlotValue("Multiplexer");
		mux:SetNoHinting(true);
	end;
	UIFromDescriptor = VFL.Nil;
	CreateDescriptor = function() return { feature = "No Hinting"; }; end;
});

RDX.GenericWindowState = {};
function RDX.GenericWindowState:new()
	local st = RDX.ObjectState:new();
	st.OnResetSlots = function(state)
		state:AddSlot("Create", true); state:AddSlot("Hide", true);	state:AddSlot("Show", true);
		state:AddSlot("Destroy", true);	state:AddSlot("Update", true); state:AddSlot("Window", nil);
		state:AddSlot("UnitWindow", nil);
		state:AddSlot("Menu");
		-- Each window gets a new multiplexer
		state:SetSlotValue("Multiplexer", RDX.Multiplexer:new());

		-- On assembly, open the multiplexer.
		state:Attach(state:Slot("Assemble"), true, function(s,w)
			-- Apply the multiplexer to the window.
			local mux = s:GetSlotValue("Multiplexer");
			w.Multiplexer = mux;
			mux:Open(w);
			-- Compat.
			w.RepaintLayout = w.RepaintAll;

			s:Attach("Create", true, function(theWindow)
				theWindow.Multiplexer:Bind(theWindow);
			end);
			s:Attach("Show", true, function(theWindow)
				-- On show, always repaint all.
				theWindow.RepaintAll();
			end);
			s:Attach("Destroy", true, function(theWindow)
				theWindow.Multiplexer:Unbind(theWindow);
			end);

			-- Attach downstream menu hooks to WMMenu.
			w._WindowMenu = s:GetSlotFunction("Menu");
		end);	
	end
	st:Clear();
	return st;
end

-- A general state object to be reused by this engine
local state = RDX.GenericWindowState:new();
RDX._exportedWindowState = RDX.GenericWindowState:new();

-----------------------------------------------------------
-- Window meta-control
-----------------------------------------------------------
-- Master priming function for compiling windows.
local function SetupWindow(path, win, desc)
	if (not win) or (not desc) then return nil; end
	RDX:Debug(5, "SetupWindow<", tostring(path), ">");

	-- Quash the old window
	win:UnloadState();

	-- Load the features.
	state:LoadDescriptor(desc);
	local _errs = VFL.Error:new();
	if not state:ApplyAll(_errs) then
		_errs:ToErrorHandler("RDX", i18n("Could not build window at <") .. tostring(path) .. ">");
		return nil;
	end
	_errs = nil;

	-- Sanity check; make sure there are layouts and frames
	if (not state:Slot("Frame")) or (not state:Slot("Layout")) then
		VFL.TripError("RDX", i18n("Cannot open window <") .. tostring(path) .. ">.", i18n("The window at <") .. tostring(path) .. i18n("> is missing a Frame or Layout."));
		return nil;
	end

	-- If we are in combat, and the pre- or post-build window is Secure, we can't rebuild it.
	if (win.secure or state:Slot("SecureSubframes")) and InCombatLockdown() then
		VFL.TripError("RDX", i18n("Attempt to build secure window while in combat."), i18n("Could not build secure window at <") .. tostring(path) .. i18n(">. Player was in combat."));
		return nil;
	end

	-- Setup the new window
	win._path = path;
	-- Apply the features to the window. If the window will be secure, mark it so.
	win:LoadState(state);
	if state:Slot("SecureSubframes") then win.secure = true; else win.secure = nil; end
	-- Destroy the state to prevent memory leakage
	state:Clear();

	-- Show the new window.
	win:Show();
	-- Tell the WM that we are rebuilding...
	if win.WMRebuild then win:WMRebuild(); end

	return true;
end

-- Called after the Feature Editor is closed. Repopulates the window.
local function UpdateWindowAfterEdit(path, md, newState)
	md.data = newState:GetDescriptor();
	local inst = RDXDB.PathHasInstance(path);
	if inst then
		SetupWindow(path, inst, md.data);
	end
end

-- Open the feature editor for the window.
local function EditWindow(path, md)
	if not RDX.IsFeatureEditorOpen() then
		state:Rebuild(md.data);
		RDX.FeatureEditor(state, function(x) UpdateWindowAfterEdit(path, md, x); end, path);
	end
end

-- The "gwin" generic window class.
RDXWM.RegisterWindowClass({
	name = "gwin",
	Open = function(id)
		if not RDXDB.CheckObject(id, "Window") then return nil; end
		RDX:Debug(5, "Open window<", id, ">");
		return RDXDB.GetObjectInstance(id);
	end,
	Close = function(win, id)
		if win.secure and InCombatLockdown() then
			VFL.TripError("RDX", i18n("Could not close secure window at path <") .. id .. ">", i18n("Player was in combat."));
			return nil;
		end
		RDX:Debug(5, "Close window<", id, ">");
		RDXDB._RemoveInstance(id);
		return true;
	end,
	Rebuild = function(win, id)
		RDX:Debug(5, "Rebuild window<", id, ">");
		local md = RDXDB.GetObjectData(id);
		if (not md) or (not md.data) then return; end
		SetupWindow(id, win, md.data);
	end,
	Props = function(win, id, mnu)
		table.insert(mnu, {
			text = i18n("Feature Editor..."),
			OnClick = function()
				VFL.poptree:Release();
				local md = RDXDB.GetObjectData(id);
				if md then EditWindow(id, md); end
			end
		});
		table.insert(mnu, {
			text = i18n("Rebuild"),
			OnClick = function()
				VFL.poptree:Release();
				local cls = win:WMGetClass();
				if cls then
					cls.Rebuild(win, id);
				end
			end
		});
	end
});

-- The Window object type.
RDXDB.RegisterObjectType({
	name = "Window";
	isFeatureDriven = true;
	New = function(path, md)
		md.version = 1;
	end,
	Open = function(path, md)
		RDX:Debug(5, "Open WindowObject<", path, ">");
		if not RDXDB.GetObjectInstance(path, true) then
			RDXWM.OpenWindow(path, "gwin");
		end
	end,
	Close = function(path, md)
		local inst = RDXDB.GetObjectInstance(path, true);
		if inst then 
			RDXWM.CloseWindow(inst);
		end
	end,
	Instantiate = function(path, obj)
		local w = RDX.Window:new(RDXParent); w:Show();
		-- Attempt to setup the window; if it fails, just bail out.
		if not SetupWindow(path, w, obj.data) then w:Destroy(); return nil; end
		-- move handle
		RDXWM.StdMove(w, w:GetTitleBar());
		return w;
	end,
	Deinstantiate = function(instance, path, obj)
		instance:Destroy();
		instance._path = nil; -- Remove the path previously stored
	end,
	GenerateBrowserMenu = function(mnu, path, md, dlg)
		table.insert(mnu, {
			text = i18n("Edit..."),
			OnClick = function()
				VFL.poptree:Release();
				EditWindow(path, md);
			end
		});
		if not RDXWM.GetWindow(path) then
			table.insert(mnu, {
				text = i18n("Open"),
				OnClick = function()
					VFL.poptree:Release();
					RDXWM.OpenWindow(path, "gwin");
				end
			});
		end
	end,
});
