-- Obj_Desktop.lua
-- RDX - Raid Data Exchange
-- (C)2006 Bill Johnson
-- THIS FILE CONTAINS COPYRIGHTED CONTENT SUBJECT TO THE TERMS OF A SEPARATE LICENSE.
-- UNLICENSED COPYING IS PROHIBITED.
--
-- The Desktop object class, and code to quick-switch between them.

local function ActivateDesktop(path, md)
	RDXWM.LoadDesktopFile(path, md);
end

local qd = nil;

-- On init, create the quick desktop list
RDXEvents:Bind("INIT_DATABASE_LOADED", nil, function()
	if not RDXG.qd then RDXG.qd = {}; end
	qd = RDXG.qd;
	-- Verify that all the quick desktops exist; remove those that do not.
	VFL.filterInPlace(qd, function(x)
		local _,_,_,ty = RDXDB.GetObjectData(x);
		if(ty == "Desktop") then return true; else return nil; end
	end);
end);

--- Determine if a desktop is on the Quick Desktops list.
function RDX.IsQuickDesktop(path)
	return VFL.vfind(qd, path);
end

-- Remove a desktop from the Quick Desktops list.
function RDX.RemoveQuickDesktop(path)
	VFL.filterInPlace(qd, function(x) return (x ~= path); end);
end

-- Add a desktop to the Quick Desktops list if it's not already there.
function RDX.AddQuickDesktop(path)
	-- Make sure it's not already there
	if RDX.IsQuickDesktop(path) then return nil; end
	-- Make sure it's a desktop
	local _,_,_,ty = RDXDB.GetObjectData(path); if(ty ~= "Desktop") then return nil; end
	-- Add it
	table.insert(qd, path);
end

RDXDB.RegisterObjectType({
	name = "Desktop",
	localOnly = true, -- Desktops are not cross-machine portable
	New = function(path, md)
		md.version = 1;
	end,
	GenerateBrowserMenu = function(mnu, path, md, dlg)
		table.insert(mnu, {
			text = i18n("Activate"),
			OnClick = function()
				VFL.poptree:Release();
				ActivateDesktop(path, md);
			end
		});
		if not RDX.IsQuickDesktop(path) then
			table.insert(mnu, {
				text = i18n("Add to Quick Desktops");
				OnClick = function()
					VFL.poptree:Release();
					RDX.AddQuickDesktop(path);
				end;
			});
		end
	end
});



local mnu = {};
local function BuildQuickDesktopMenu()
	VFL.empty(mnu);
	for _,name in ipairs(qd) do
		local path = name;
		table.insert(mnu, {
			text = path;
			OnClick = function()
				VFL.poptree:Release();
				if(arg1 == "LeftButton") then
					RDX.SelectDesktop(path);
				elseif(arg1 == "RightButton") then
					RDX.RemoveQuickDesktop(path);
				end
			end;
		});
	end
end


-- Toolbar button
local dtbtn = VFLUI.AcquireFrame("Button");
local dttex = VFLUI.CreateTexture(dtbtn);
dttex:SetAllPoints(dtbtn);
dttex:Show();
dtbtn:SetHighlightTexture(dttex);
dttex:SetBlendMode("DISABLE");
if RDX._skin == "boomy" then
	dtbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\boomy\\paint");
	dttex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\paint");
elseif RDX._skin == "kids" then
	dtbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\kids\\desktop");
	dttex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\desktop");
else
	dtbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\desktop");
	dttex:SetTexture("Interface\\Addons\\RDX\\Skin\\desktop");
end
dttex:SetVertexColor(0.55, 0.44, 0.44);
dtbtn:RegisterForClicks("LeftButtonUp", "RightButtonUp");
RDXEvents:Bind("INIT_PRELOAD", nil, function() RDX.AddToolbarButton(dtbtn, true); end);

dtbtn:SetScript("OnClick", function()
	if(arg1 == "LeftButton") then
		BuildQuickDesktopMenu();
		VFL.poptree:Begin(200, 14, UIParent, "TOPLEFT", GetRelativeLocalMousePosition(UIParent));
		VFL.poptree:Expand(nil, mnu);
	elseif(arg1 == "RightButton") then
		RDXWM.WindowList();
	end
end);
