-- UITools.lua
-- RDX - Raid Data Exchange
-- (C)2006 Bill Johnson
--
-- THIS FILE CONTAINS COPYRIGHTED MATERIAL SUBJECT TO THE TERMS OF A SEPARATE
-- LICENSE. UNLICENSED COPYING IS PROHIBITED.
--
-- Helpful tools and objects for the UnitFrames UIs.

---------------------------------------------------
-- Generate a right-embedded color swatch inside a container.
---------------------------------------------------
function RDXUI.GenerateColorSwatch(ctr, text)
	local er = RDXUI.EmbedRight(ctr, text);
	local swatch = VFLUI.ColorSwatch:new(er);
	swatch:Show();
	er:EmbedChild(swatch); er:Show();
	ctr:InsertFrame(er);
	return swatch;
end

----------------------------------------------------
-- Given a unitframe state, compose a list of available objects on that state.
----------------------------------------------------
function RDXUI.ComposeObjectList(state, prefix, includeBase)
	local tbl = {};
	if (not state) or (not state:Slot("Base")) then return tbl; end
	if includeBase then table.insert(tbl, {text = "Base"}); end
	for x in state:SlotsMatching("^" .. prefix) do
		local _,_,trim = string.find(x, "^" .. prefix .. "(.*)$");
		if trim then
			table.insert(tbl, {text = trim});
		end
	end
	return tbl;
end

function RDXUI.ComposeFrameList(state)
	return RDXUI.ComposeObjectList(state, "Frame_", true);
end

----------------------------------------------------
-- Generate a dropdown that selects slots with a given prefix from a state.
----------------------------------------------------
function RDXUI.MakeSlotSelectorDropdown(ctr, title, state, prefix, includeBase, x1, x2)
	local er = RDXUI.EmbedRight(ctr, title);
	local dd_array = RDXUI.ComposeObjectList(state, prefix, includeBase);
	if x1 then table.insert(dd_array, {text = x1}); end
	if x2 then table.insert(dd_array, {text = x2}); end
	local dd = VFLUI.ComboBox:new(er, function() return dd_array; end);
	dd:SetWidth(175); dd:Show();
	er:EmbedChild(dd); er:Show();
	ctr:InsertFrame(er);

	return dd;
end

-- Validity check for boolean variables (allow true/false constants)
function RDXUI.IsValidBoolVar(vn, state)
	if (not vn) then return nil; end
	if (vn == "true") or (vn == "false") or (state:Slot("BoolVar_" .. vn)) then return true; else return nil; end
end

----------------------------------------------------
-- Anchor selection control for unit frames
----------------------------------------------------
local anchorpoints = {
	{ text = "TOPLEFT" },
	{ text = "TOP" },
	{ text = "TOPRIGHT" },
	{ text = "RIGHT" },
	{ text = "BOTTOMRIGHT" },
	{ text = "BOTTOM" },
	{ text = "BOTTOMLEFT" },
	{ text = "LEFT" },
	{ text = "CENTER" }
};
local function amOnBuild() return anchorpoints; end
RDXUI.AnchorPointSelectionFunc = amOnBuild;

RDXUI.UnitFrameAnchorSelector = {};
function RDXUI.UnitFrameAnchorSelector:new(parent)
	local self = VFLUI.GroupBox:new(parent);
	VFLUI.GroupBox.MakeTextCaption(self, i18n("Anchor"));
	self:SetLayoutConstraint("WIDTH_DOWNWARD_HEIGHT_UPWARD");
	local ctr = VFLUI.CompoundFrame:new(self); ctr:Show();
	self:SetClient(ctr);

	-- Local anchor point
	local er = RDXUI.EmbedRight(ctr, i18n("Anchor local point:"));
	local dd_lp = VFLUI.Dropdown:new(er, amOnBuild);
	dd_lp:SetWidth(100); dd_lp:Show(); dd_lp:SetSelection("TOPLEFT");
	er:EmbedChild(dd_lp); er:Show();
	ctr:InsertFrame(er);

	-- Remote anchor frame
	local afArray = {};
	local afOnBuild = function() return afArray; end
	function self:SetAFArray(x) afArray = x; end

	er = RDXUI.EmbedRight(ctr, i18n("To frame:"));
	local dd_af = VFLUI.ComboBox:new(er, afOnBuild);
	dd_af:SetWidth(150); dd_af:Show();
	er:EmbedChild(dd_af); er:Show();
	ctr:InsertFrame(er);
	
	-- Remote anchor point
	local er = RDXUI.EmbedRight(ctr, i18n("and remote point:"));
	local dd_rp = VFLUI.Dropdown:new(er, amOnBuild);
	dd_rp:SetWidth(100); dd_rp:Show(); dd_rp:SetSelection("TOPLEFT");
	er:EmbedChild(dd_rp); er:Show();
	ctr:InsertFrame(er);

	-- OffsetY
	local edy = VFLUI.LabeledEdit:new(ctr, 75); edy:SetText(i18n("Offset X/Y:")); edy:Show();
	local edx = VFLUI.Edit:new(edy); edx:Show();
	edx:SetHeight(24); edx:SetWidth(75); edx:SetPoint("RIGHT", edy.editBox, "LEFT");
	edy.Destroy = VFL.hook(function() edx:Destroy(); end, edy.Destroy);
	ctr:InsertFrame(edy);

	------------------------ SETUP
	function self:SetAnchorInfo(ai)
		dd_lp:SetSelection(ai.lp); dd_rp:SetSelection(ai.rp); dd_af:SetText(ai.af);
		edx:SetText(ai.dx); edy.editBox:SetText(ai.dy);
	end
	function self:GetAnchorInfo()
		local dx = VFL.clamp(edx:GetNumber(), -1000, 1000);
		local dy = VFL.clamp(edy.editBox:GetNumber(), -1000, 1000);
		return {
			lp = dd_lp:GetSelection(), rp = dd_rp:GetSelection(), af = dd_af:GetSelection(),
			dx = dx, dy = dy
		};
	end

	----------------------- Destructor
	self.Destroy = VFL.hook(function(s)
		s.SetAFArray = nil; 
		afArray = nil; afOnBuild = nil;
		s.SetAnchorInfo = nil; s.GetAnchorInfo = nil;
	end, self.Destroy);

	return self;
end

--- A helper function to resolve a frame reference to a variable on the object.
function RDXUI.ResolveFrameReference(ref)
	if (not ref) or (ref == "") or (ref == "Base") then return "frame"; end
	if string.find(ref, "^Frame_") then
		VFL.TripError("RDX", i18n("Warning: deprecated frame reference"), i18n("Deprecated frame reference <") .. ref .. ">");
		return "frame." .. ref;
	else
		return "frame.Frame_" .. ref;
	end
end

--- A helper function to generate the appropriate arguments to SetPoint() given an
-- anchor descriptor
function RDXUI.AnchorCodeFromDescriptor(anchor)
	return "'" .. anchor.lp .. "'," .. RDXUI.ResolveFrameReference(anchor.af) .. ",'" .. anchor.rp .. "'," .. anchor.dx .. "," .. anchor.dy;
end

------------------------------------------
-- Dropdown helpers
------------------------------------------
local fontdd = {};
for k,_ in pairs(Fonts) do
	table.insert(fontdd, { text = k } );
end

function RDXUI.FontDropdownFunction() return fontdd; end

local hadd = {
	{ text = "LEFT" },
	{ text = "CENTER" },
	{ text = "RIGHT" }
};
function RDXUI.HAlignDropdownFunction() return hadd; end

local vadd = {
	{ text = "TOP" },
	{ text = "CENTER" },
	{ text = "BOTTOM" }
};
function RDXUI.VAlignDropdownFunction() return vadd; end

local dladd = {
	{ text = "BACKGROUND" },
	{ text = "BORDER" },
	{ text = "ARTWORK" },
	{ text = "OVERLAY" },
	{ text = "HIGHLIGHT" }
};
function RDXUI.DrawLayerDropdownFunction() return dladd; end


---------------------------------------------
-- Factored-out oft-repeated fragments
---------------------------------------------
function RDXUI.GenNameWidthHeightPortion(ui, desc)
	local ed_name = VFLUI.LabeledEdit:new(ui, 100); ed_name:Show();
	ed_name:SetText(i18n("Name"));
	ed_name.editBox:SetText(desc.name);
	ui:InsertFrame(ed_name);

	local ed_width = VFLUI.LabeledEdit:new(ui, 50); ed_width:Show();
	ed_width:SetText(i18n("Width"));
	if desc and desc.w then ed_width.editBox:SetText(desc.w); end
	ui:InsertFrame(ed_width);

	local ed_height = VFLUI.LabeledEdit:new(ui, 50); ed_height:Show();
	ed_height:SetText(i18n("Height"));
	if desc and desc.h then ed_height.editBox:SetText(desc.h); end
	ui:InsertFrame(ed_height);

	return ed_name, ed_width, ed_height;
end

--- Check to see if an object name is a valid name (alphanumeric/underscores only, 15 chars or less)
-- and is not previously taken on a state. Designed for use in ExposeFeature methods
function __UFFrameCheck_Proto(prefix, desc, state, errs)
	if desc and desc.name then
		if(desc.name == "Base") then
			VFL.AddError(errs, i18n("No object can be named 'Base.'"));
			return nil;
		end
		if not RDXDB.IsValidFileName(desc.name) then
			VFL.AddError(errs, i18n("Object names must be alphanumeric."));
			return nil;
		end
		if state:Slot(prefix .. desc.name) then
			VFL.AddError(errs, i18n("Duplicate object name '") .. desc.name .. "'.");
			return nil;
		end
		state:AddSlot(prefix .. desc.name);
		return true;
	else
		VFL.AddError(errs, i18n("Bad or missing object name."));
		return nil;
	end
end

function __UFFrameCheck(prefix)
	return function(desc, state, errs)
		return __UFFrameCheck_Proto(prefix, desc, state, errs);
	end;
end

__UFObjCheck = __UFFrameCheck("Obj_");

--- Check an anchor descriptor against a state to make sure we're anchoring to something that exists.
function __UFAnchorCheck(anchor, state, errs)
	if not anchor then
		VFL.AddError(errs, i18n("Invalid anchor definition."));
		return nil;
	end
	if (not anchor.lp) or (not anchor.rp) or (not anchor.dx) or (not anchor.dy) then
		VFL.AddError(errs, i18n("Missing anchor layout parameters."));
		return nil;
	end
	if (not anchor.af) then
		VFL.AddError(errs, i18n("Missing anchor target frame.")); return nil;
	end
	if (anchor.af == "Base") or (state:Slot("Frame_" .. anchor.af)) then return true; end
	if state:Slot(anchor.af) then
		VFL.AddError(errs, i18n("Warning: Deprecated Frame_ reference."));
		return true;
	end
	VFL.AddError(errs, i18n("Anchor target frame does not exist.")); return nil;
end

--- Check a frame owner exist.
function __UFOwnerCheck(owner, state, errs)
	if not owner then
		VFL.AddError(errs, i18n("Invalid owner definition."));
		return nil;
	end
	if (owner == "Base") or (state:Slot("Frame_" .. owner)) or (state:Slot("Subframe_" .. owner)) then return true; end
	VFL.AddError(errs, i18n("Owner frame does not exist.")); return nil;
end
