-- Text.lua
-- RDX - Raid Data Exchange
-- (C)2006 Bill Johnson
--
-- THIS FILE CONTAINS COPYRIGHTED MATERIAL SUBJECT TO THE TERMS OF A SEPARATE
-- LICENSE. UNLICENSED COPYING IS PROHIBITED.
--
-- Unit frame features that add text to various places on the unit frame.

local function _EmitCreateFontString(desc)
	return [[
local txt = VFLUI.CreateFontString(]] .. RDXUI.ResolveFrameReference(desc.owner) .. [[);
txt:SetWidth(]] .. desc.w .. [[); txt:SetHeight(]] .. desc.h .. [[);
]];
end

--- Scripted custom text.
RDX.RegisterFeature({
	name = "txt_custom";	version = 1;	multiple = true;
	title = i18n("Custom Text");	category = i18n("Text");
	IsPossible = function(state)
		if not state:Slot("UnitFrame") then return nil; end
		if not state:Slot("Base") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		if not desc then VFL.AddError(errs, i18n("Missing descriptor.")); return nil; end
		if not VFLUI.isFacePathExist(desc.font.face) then VFL.AddError(errs, i18n("Font path not found.")); return nil; end
		local flg = true;
		flg = flg and __UFAnchorCheck(desc.anchor, state, errs);
		flg = flg and __UFFrameCheck_Proto("Frame_", desc, state, errs);
		flg = flg and __UFOwnerCheck(desc.owner, state, errs);
		if flg then 
			state:AddSlot("Text_" .. desc.name);
		end
		return flg;
	end;
	ApplyFeature = function(desc, state)
		local objname = "Frame_" .. desc.name;
		
		---- Generate the code.
		local createCode = _EmitCreateFontString(desc) .. [[
txt:SetPoint(]] .. RDXUI.AnchorCodeFromDescriptor(desc.anchor) .. [[);
]] .. VFLUI.GenerateSetFontCode("txt", desc.font, nil, true) .. [[
txt:Show();
frame.]] .. objname .. [[ = txt;
]];

		local destroyCode = [[
frame.]] .. objname .. [[:Destroy(); frame.]] .. objname .. [[ = nil;
]];

		local cleanupCode = [[
frame.]] .. objname .. [[:SetText("");
]];

		-- Apply the custom code.
		local md,_,_,ty = RDXDB.GetObjectData(desc.script);
		if (md) and (ty == "Script") and (md.data) and ( md.data.script) then
			local paintCode = [[
local text = ]] .. (desc.useNil and 'nil' or '""') .. [[;

]] .. md.data.script .. [[

if text then frame.]] .. objname .. [[:SetText(text); end
]];
			state:Attach(state:Slot("EmitPaint"), true, function(code) code:AppendCode(paintCode); end);
		end

		state:Attach(state:Slot("EmitCreate"), true, function(code) code:AppendCode(createCode); end);
		state:Attach(state:Slot("EmitDestroy"), true, function(code) code:AppendCode(destroyCode); end);
		state:Attach(state:Slot("EmitCleanup"), true, function(code) code:AppendCode(cleanupCode); end);
		
		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);

		local ed_name, ed_width, ed_height = RDXUI.GenNameWidthHeightPortion(ui, desc);

		-- Owner
		local owner = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Owner"), state, "Subframe_", true);
		if desc and desc.owner then owner:SetSelection(desc.owner); end

		local anchor = RDXUI.UnitFrameAnchorSelector:new(ui); anchor:Show();
		anchor:SetAFArray(RDXUI.ComposeFrameList(state));
		if desc and desc.anchor then anchor:SetAnchorInfo(desc.anchor); end
		ui:InsertFrame(anchor);

		local er = RDXUI.EmbedRight(ui, i18n("Font"));
		local fontsel = VFLUI.MakeFontSelectButton(er, desc.font); fontsel:Show();
		er:EmbedChild(fontsel); er:Show();
		ui:InsertFrame(er);

		local chk_useNil = VFLUI.Checkbox:new(ui); 
		chk_useNil:Show(); chk_useNil:SetText(i18n("Preserve existing content if text local variable is undefined"))
		if desc and desc.useNil then chk_useNil:SetChecked(true); end
		ui:InsertFrame(chk_useNil);

		local scriptsel = RDXDB.ObjectFinder:new(ui, function(_,_,d) return d and (d.ty == "Script"); end);
		scriptsel:SetLabel(i18n("Script object")); scriptsel:Show();
		if desc and desc.script then scriptsel:SetPath(desc.script); end
		ui:InsertFrame(scriptsel);

		function ui:GetDescriptor()
			return { 
				feature = "txt_custom"; version = 1;
				name = ed_name.editBox:GetText();
				owner = owner:GetSelection();
				w = ed_width.editBox:GetNumber(), h = ed_height.editBox:GetNumber(), 
				anchor = anchor:GetAnchorInfo();
				font = fontsel:GetSelectedFont();
				useNil = chk_useNil:GetChecked();
				script = scriptsel:GetPath();
			};
		end
		
		return ui;
	end;
	CreateDescriptor = function()
		return { 
			feature = "txt_custom", version = 1,
			name = "customText", w = 50, h = 14, owner = "Base",
			anchor = { lp = "LEFT", af = "Base", rp = "LEFT", dx = 0, dy = 0 }, 
			font = VFL.copy(Fonts.Default);
		};
	end;
});

-- Static text.
RDX.RegisterFeature({
	name = "txt_static";	version = 1;	multiple = true;
	title = i18n("Static Text");	category = i18n("Text");
	IsPossible = function(state)
		if not state:Slot("UnitFrame") then return nil; end
		if not state:Slot("Base") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		if not desc then VFL.AddError(errs, i18n("Missing descriptor.")); return nil; end
		if not VFLUI.isFacePathExist(desc.font.face) then VFL.AddError(errs, i18n("Font path not found.")); return nil; end
		-- Verify our texte
		if (not desc.txt) then
			VFL.AddError(errs, i18n("Invalid texte")); return nil;
		end
		if not __UFFrameCheck_Proto("Frame_", desc, state, errs) then return nil; end
		state:AddSlot("Text_" .. desc.name);
		return true;
	end;
	ApplyFeature = function(desc, state)
		local objname = "Frame_" .. desc.name;
		local colorVar, colorBoo = strtrim(desc.color or ""), "false";
		if colorVar ~= "" then colorBoo = "true"; end
		---- Generate the code.
		local createCode = _EmitCreateFontString(desc) .. [[
txt:SetPoint(]] .. RDXUI.AnchorCodeFromDescriptor(desc.anchor) .. [[);
]] .. VFLUI.GenerateSetFontCode("txt", desc.font, nil, true) .. [[
txt:Show();
frame.]] .. objname .. [[ = txt;
]];
		local destroyCode = [[
frame.]] .. objname .. [[:Destroy(); frame.]] .. objname .. [[ = nil;
]];
		local cleanupCode = [[
frame.]] .. objname .. [[:SetText("");
]];
		-- Apply the static text
		local paintCode = [[
frame.]] .. objname .. [[:SetText("]] .. desc.txt .. [[");
if ]] .. colorBoo .. [[ then
	frame.]] .. objname .. [[:SetTextColor(explodeRGBA(]] .. colorVar .. [[));
end
]];
		state:Attach(state:Slot("EmitPaint"), true, function(code) code:AppendCode(paintCode); end);
		state:Attach(state:Slot("EmitCreate"), true, function(code) code:AppendCode(createCode); end);
		state:Attach(state:Slot("EmitDestroy"), true, function(code) code:AppendCode(destroyCode); end);
		state:Attach(state:Slot("EmitCleanup"), true, function(code) code:AppendCode(cleanupCode); end);
		
		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);

		local ed_name, ed_width, ed_height = RDXUI.GenNameWidthHeightPortion(ui, desc);

		-- Owner
		local owner = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Owner"), state, "Subframe_", true);
		if desc and desc.owner then owner:SetSelection(desc.owner); end

		local anchor = RDXUI.UnitFrameAnchorSelector:new(ui); anchor:Show();
		anchor:SetAFArray(RDXUI.ComposeFrameList(state));
		if desc and desc.anchor then anchor:SetAnchorInfo(desc.anchor); end
		ui:InsertFrame(anchor);

		local er = RDXUI.EmbedRight(ui, i18n("Font"));
		local fontsel = VFLUI.MakeFontSelectButton(er, desc.font); fontsel:Show();
		er:EmbedChild(fontsel); er:Show();
		ui:InsertFrame(er);

		local txt = VFLUI.LabeledEdit:new(ui, 200);
		txt:SetText(i18n("Static Text"));
		txt:Show();
		if desc and desc.txt then txt.editBox:SetText(desc.txt); end
		ui:InsertFrame(txt);
		
		local colorVar = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Color variable"), state, "ColorVar_");
		if desc and desc.color then colorVar:SetSelection(desc.color); end

		function ui:GetDescriptor()
			local scolorVar = strtrim(colorVar:GetSelection() or "");
			if scolorVar == "" then scolorVar = nil; end
			return { 
				feature = "txt_static"; version = 1;
				name = ed_name.editBox:GetText();
				owner = owner:GetSelection();
				w = ed_width.editBox:GetNumber(), h = ed_height.editBox:GetNumber(), 
				anchor = anchor:GetAnchorInfo();
				font = fontsel:GetSelectedFont();
				txt =  txt.editBox:GetText();
				color = scolorVar;
			};
		end
		
		return ui;
	end;
	CreateDescriptor = function()
		return { 
			feature = "txt_static", version = 1,
			name = "staticText", w = 50, h = 14, owner = "Base",
			anchor = { lp = "LEFT", af = "Base", rp = "LEFT", dx = 0, dy = 0 }, 
			font = VFL.copy(Fonts.Default);
		};
	end;
});

--- Dynamic text.
RDX.RegisterFeature({
	name = "txt_dyn"; version = 1; multiple = true;
	title = i18n("Info Text"); category = i18n("Text");
	IsPossible = function(state)
		if not state:Slot("UnitFrame") then return nil; end
		if not state:Slot("Base") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		if not desc then VFL.AddError(errs, i18n("Missing descriptor.")); return nil; end
		if not VFLUI.isFacePathExist(desc.font.face) then VFL.AddError(errs, i18n("Font path not found.")); return nil; end
		-- Verify our texte
		if (not desc.txt) or (not state:Slot("Txt_" .. desc.txt)) then
			VFL.AddError(errs, i18n("Invalid texte object pointer.")); return nil;
		end
		if not __UFFrameCheck_Proto("Frame_", desc, state, errs) then return nil; end
		state:AddSlot("Frame_" .. desc.name);
		return true;
	end;
	ApplyFeature = function(desc, state)
		local objname = "Frame_" .. desc.name;
		local colorVar, colorBoo = strtrim(desc.color or ""), "false";
		if colorVar ~= "" then colorBoo = "true"; end
		---- Generate the code.
		local createCode = _EmitCreateFontString(desc) .. [[
txt:SetPoint(]] .. RDXUI.AnchorCodeFromDescriptor(desc.anchor) .. [[);
]] .. VFLUI.GenerateSetFontCode("txt", desc.font, nil, true) .. [[
txt:Show();
frame.]] .. objname .. [[ = txt;
]];
		local destroyCode = [[
frame.]] .. objname .. [[:Destroy(); frame.]] .. objname .. [[ = nil;
]];
		local cleanupCode = [[
frame.]] .. objname .. [[:SetText("");
]];
		-- Apply the static
		local paintCode = [[
frame.]] .. objname .. [[:SetText(]] .. desc.txt .. [[);
if ]] .. colorBoo .. [[ then
	frame.]] .. objname .. [[:SetTextColor(explodeRGBA(]] .. colorVar .. [[));
end
]];
		state:Attach(state:Slot("EmitPaint"), true, function(code) code:AppendCode(paintCode); end);
		state:Attach(state:Slot("EmitCreate"), true, function(code) code:AppendCode(createCode); end);
		state:Attach(state:Slot("EmitDestroy"), true, function(code) code:AppendCode(destroyCode); end);
		state:Attach(state:Slot("EmitCleanup"), true, function(code) code:AppendCode(cleanupCode); end);
		
		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);

		local ed_name, ed_width, ed_height = RDXUI.GenNameWidthHeightPortion(ui, desc);

		-- Owner
		local owner = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Owner"), state, "Subframe_", true);
		if desc and desc.owner then owner:SetSelection(desc.owner); end

		local anchor = RDXUI.UnitFrameAnchorSelector:new(ui); anchor:Show();
		anchor:SetAFArray(RDXUI.ComposeFrameList(state));
		if desc and desc.anchor then anchor:SetAnchorInfo(desc.anchor); end
		ui:InsertFrame(anchor);

		local er = RDXUI.EmbedRight(ui, i18n("Font"));
		local fontsel = VFLUI.MakeFontSelectButton(er, desc.font); fontsel:Show();
		er:EmbedChild(fontsel); er:Show();
		ui:InsertFrame(er);

		local txt = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Texte"), state, "Txt_");
		if desc and desc.txt then txt:SetSelection(desc.txt); end
		
		local colorVar = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Color variable"), state, "ColorVar_");
		if desc and desc.color then colorVar:SetSelection(desc.color); end

		function ui:GetDescriptor()
			local scolorVar = strtrim(colorVar:GetSelection() or "");
			if scolorVar == "" then scolorVar = nil; end
			return { 
				feature = "txt_dyn"; version = 1;
				name = ed_name.editBox:GetText();
				owner = owner:GetSelection();
				w = ed_width.editBox:GetNumber(), h = ed_height.editBox:GetNumber(), 
				anchor = anchor:GetAnchorInfo();
				font = fontsel:GetSelectedFont();
				txt =  txt.editBox:GetText();
				color = scolorVar;
			};
		end
		
		return ui;
	end;
	CreateDescriptor = function()
		return { 
			feature = "txt_dyn", version = 1,
			name = "infoText", w = 50, h = 14, owner = "Base",
			anchor = { lp = "LEFT", af = "Base", rp = "LEFT", dx = 0, dy = 0 }, 
			font = VFL.copy(Fonts.Default);
		};
	end;
});



----------------- Status text types and code
local statusText = {};
local statusIndex = {};
function RDX.RegisterStatusTextType(tbl)
	if (type(tbl) ~= "table") or (type(tbl.name) ~= "string") then return; end
	statusText[tbl.name] = tbl;
	table.insert(statusIndex, {value = tbl.name, text = tbl.title});
end

local function hpPrereq(desc, state, errs)
	return true; -- Fractional health no longer required.
end
local function hpHint(desc, state)
	local mux = state:GetContainingWindowState():GetSlotValue("Multiplexer");
	local mask = mux:GetPaintMask("HEALTH");
	mux:Event_UnitMask("UNIT_HEALTH", mask);
end
local function mpPrereq(desc, state, errs)
	return true;
end
local function mpHint(desc, state)
	local mux = state:GetContainingWindowState():GetSlotValue("Multiplexer");
	local mask = mux:GetPaintMask("MANA");
	mux:Event_UnitMask("UNIT_MANA", mask);
end

RDX.RegisterStatusTextType({
	name = "fdld";
	title = i18n("Feigned/Dead/LD");
	OnExpose = VFL.Noop;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
if unit:IsFeigned() then
	frame.]] .. objname .. [[:SetText(i18n("Feign"));
elseif unit:IsDead() then
	frame.]] .. objname .. [[:SetText(i18n("|cFFFF0000Dead|r"));
elseif not unit:IsOnline() then
	frame.]] .. objname .. [[:SetText(i18n("|cFF777777LD|r"));
else
	frame.]] .. objname .. [[:SetText("");
end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "fdld2";
	title = i18n("Feigned/Dead");
	OnExpose = VFL.Noop;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
if unit:IsFeigned() then
	frame.]] .. objname .. [[:SetText(i18n("Feign"));
elseif unit:IsDead() then
	frame.]] .. objname .. [[:SetText(i18n("|cFFFF0000Dead|r"));
else
	frame.]] .. objname .. [[:SetText("");
end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "gn";
	title = i18n("Group Number");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, _white, nil, unit:GetGroup());
]]; end;
});

RDX.RegisterStatusTextType({
	name = "gn2";
	title = i18n("Group");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	if UnitInRaid(uid) then
		RDX.SetStatusText(frame.]] .. objname .. [[, 1, _white, nil, "G:" .. unit:GetGroup());
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "class";
	title = i18n("Class");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local text;
	if UnitIsPlayer(uid) then
		text = UnitClass(uid) or "";
	else
		if UnitCreatureType(uid) == "Beast" and UnitCreatureFamily(uid) then
			text = UnitCreatureFamily(uid);
		else
			text = UnitCreatureType(uid) or "";
		end
	end
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, unit:GetClassColor(), nil, text);
]]; end;
});

RDX.RegisterStatusTextType({
	name = "class4";
	title = i18n("Class 4");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local text;
	if UnitIsPlayer(uid) then
		text = UnitClass(uid) or "";
	else
		if UnitCreatureType(uid) == "Beast" and UnitCreatureFamily(uid) then
			text = UnitCreatureFamily(uid);
		else
			text = UnitCreatureType(uid) or "";
		end
	end
	if string.len(text) > 4 then text = string.sub(text,1,4); end
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, unit:GetClassColor(), nil, text); 
]]; end;
});

RDX.RegisterStatusTextType({
	name = "race";
	title = i18n("Race");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local text;
	if UnitIsPlayer(uid) then
		text = (UnitRace(uid) or "");
	else
		if UnitCreatureType(uid) == "Beast" and UnitCreatureFamily(uid) then
			text = UnitCreatureFamily(uid);
		else
			text = UnitCreatureType(uid) or "";
		end
	end
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, unit:GetClassColor(), nil, text); 
]]; end;
});

RDX.RegisterStatusTextType({
	name = "guild";
	title = i18n("Guild");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local text, guild, pguild;
	local color = _grey;
	if UnitIsPlayer(uid) then
		guild = GetGuildInfo(uid);
		if guild then
			text = guild;
			pguild = GetGuildInfo("player")
			if guild == pguild then
				if string.len(text) > 22 then text = string.sub(text,1,22); end
				color = _green;
			else
				if string.len(text) > 22 then text = string.sub(text,1,22); end
				color = _red;
			end
		end
	end
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, color, nil, text); 
]]; end;
});

RDX.RegisterStatusTextType({
	name = "mobtype";
	title = i18n("Mob type");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local str = UnitClassification(uid);
	local text = "";
	local color = _grey;
	if (str ~= "normal") then
		if (str == "worldboss") then
			text = "Boss";
			color = _white;
		elseif (str == "rareelite") then
			text = "Rare Elite";
			color = _yellow;
		elseif (str == "elite") then
			text = "Elite Mob";
			color = _yellow;
		elseif (str == "rare") then
			text = "Rare Mob";
			color = _grey;
		end
	end
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, color, nil, text); 
]]; end;
});

RDX.RegisterStatusTextType({
	name = "level";
	title = i18n("Level");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	if UnitExists(uid) then
		local n = UnitLevel(uid);
		if (n < 0) then
			RDX.SetStatusText(frame.]] .. objname .. [[, 1, GetDifficultyColor(UnitLevel(uid)), nil, "??");
		else
			RDX.SetStatusText(frame.]] .. objname .. [[, 1, GetDifficultyColor(UnitLevel(uid)), nil, tostring(n));
		end
	else
		RDX.SetStatusText(frame.]] .. objname .. [[, 1, GetDifficultyColor(UnitLevel(uid)), nil, "");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "realm";
	title = i18n("Realms");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local lname, lrealm = UnitName(uid);
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, _white, nil, lrealm);
]]; end;
});



RDX.RegisterStatusTextType({
	name = "cp";
	title = i18n("Combo Points");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local _cp = GetComboPoints()
	if _cp == 0 then
		frame.]] .. objname .. [[:SetText("");
	elseif _cp then
		RDX.SetStatusText(frame.]] .. objname .. [[, 1, _white, nil, _cp);
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "hpp";
	title = i18n("HP%");
	OnExpose = hpPrereq;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, string.format("%0.0f%%", unit:FracHealth()*100));
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "hp";
	title = i18n("HP");
	OnExpose = hpPrereq;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, unit:Health());
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "hpm";
	title = i18n("HP Missing");
	OnExpose = hpPrereq;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, "-" .. unit:MissingHealth());
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "hpi";
	title = i18n("HP Incoming");
	OnExpose = VFL.Noop;
	OnApply = VFL.Noop;
	GeneratePaintCode = function(objname) return [[
	local text = "";
	if (ih or 0) > 0 then text = "+" .. ih; end
	RDX.SetStatusText(frame.]] .. objname .. [[, 1, _white, nil, text);
]]; end;
});

RDX.RegisterStatusTextType({
	name = "hptxt";
	title = i18n("HP / HP Max");
	OnExpose = hpPrereq;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
	local text = "";
	if unit:IsIncapacitated() then
		text = "";
	elseif UnitIsDead(uid) then
		text = "Dead";
	elseif UnitIsGhost(uid) then
		text = "Ghost";
	elseif not UnitIsConnected(uid) then
		text = "LinkDead";
	elseif UnitIsPlayer(uid) then
		text = UnitHealth(uid) .. " / " .. UnitHealthMax(uid);
	else
		if MobHealth3 then
			local cur, max, found = MobHealth3:GetUnitHealth(uid, UnitHealth(uid), UnitHealthMax(uid));
			text = cur .. " / " .. max;
		else
			text = UnitHealth(uid) .. " / ".. UnitHealthMax(uid);
		end
	end
	RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, text);
]]; end;
});

RDX.RegisterStatusTextType({
	name = "hptxt2";
	title = i18n("HP% | HP / HP Max");
	OnExpose = hpPrereq;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		local text = "";
		if UnitIsDead(uid) then
			text = "Dead";
		elseif UnitIsGhost(uid) then
			text = "Ghost";
		elseif not UnitIsConnected(uid) then
			text = "LinkDead";
		elseif UnitIsPlayer(uid) then
			text = string.format("%0.0f%%", unit:FracHealth()*100) .. " | " .. UnitHealth(uid) .. " / " .. UnitHealthMax(uid);
		else
			if MobHealth3 then
				local cur, max, found = MobHealth3:GetUnitHealth(uid, UnitHealth(uid), UnitHealthMax(uid));
				text = string.format("%0.0f%%", unit:FracHealth()*100) .. " | " .. cur .. " / " .. max;
			else
				text = string.format("%0.0f%%", unit:FracHealth()*100) .. " | " .. UnitHealth(uid) .. " / ".. UnitHealthMax(uid);
			end
		end
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, text);
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "mpp";
	title = i18n("Mana%");
	OnExpose = mpPrereq;
	OnApply = mpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracMana(), _white, _red, string.format("%0.0f%%", unit:FracMana()*100));
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "mp";
	title = i18n("Mana");
	OnExpose = mpPrereq;
	OnApply = mpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() and (unit:PowerType() == 0) then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracMana(), _white, _red, unit:Mana());
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "mpm";
	title = i18n("Mana Missing");
	OnExpose = mpPrereq;
	OnApply = mpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() and (unit:PowerType() == 0) then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracMana(), _white, _red, "-" .. unit:MissingMana());
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "mptxt";
	title = i18n("Mana / Mana Max");
	OnExpose = mpPrereq;
	OnApply = mpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracMana(), _white, _red, UnitMana(uid) .. " / " .. UnitManaMax(uid));
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "mptxt2";
	title = i18n("Mana% | Mana / Mana Max");
	OnExpose = mpPrereq;
	OnApply = mpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() and (unit:PowerType() == 0) then
		RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracMana(), _white, _red, string.format("%0.0f%%", unit:FracMana()*100) .. " | " .. UnitMana(uid) .. " / " .. UnitManaMax(uid));
	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

------------------------
-- By giemer
------------------------

RDX.RegisterStatusTextType({
	name = "mihp";
	title = i18n("HPMI");
	OnExpose = hpPrereq;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		if MobHealth3 then
			local cur
			cur = MobHealth3:GetUnitHealth(uid);
			RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, string.format("%d",cur));
		else
			RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, unit:Health());
		end

	else
		frame.]] .. objname .. [[:SetText("");
	end
]]; end;
});

RDX.RegisterStatusTextType({
	name = "mihpmax";
	title = i18n("HPMImax");
	OnExpose = hpPrereq;
	OnApply = hpHint;
	GeneratePaintCode = function(objname) return [[
	if not unit:IsIncapacitated() then
		if MobHealth3 then
			local cur, max
			cur, max = MobHealth3:GetUnitHealth(uid);
			RDX.SetStatusText(frame.]] .. objname .. [[, unit:FracHealth(), _white, _red, string.format("%d",max));
		else
			frame.]] .. objname .. [[:SetText("");
		end
	else
		frame.]] .. objname .. [[:SetText("Mobhealth3 required to have HPMI work");
	end
]]; end;
});

-------------------- STATUS TEXT
RDX.RegisterFeature({
	name = "txt_status"; version = 1; multiple = true;
	title = i18n("Status Text");
	category = i18n("Text");
	IsPossible = function(state)
		if not state:Slot("UnitFrame") then return nil; end
		if not state:Slot("Base") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		if (not desc) then VFL.AddError(errs, i18n("Missing descriptor.")); return nil; end
		if not VFLUI.isFacePathExist(desc.font.face) then VFL.AddError(errs, i18n("Font path not found.")); return nil; end
		if not __UFFrameCheck_Proto("Frame_", desc, state, errs) then return nil; end
		if (not desc.ty) or (not statusText[desc.ty]) then VFL.AddError(errs,i18n("Invalid text type.")); return nil; end
		statusText[desc.ty].OnExpose(desc, state, errs);
		if VFL.HasError(errs) then return nil; end
		state:AddSlot("Frame_" .. desc.name);
		return true;
	end;
	ApplyFeature = function(desc, state)
		local objname = "Frame_" .. desc.name;
		
		-- Colorization
		local colorExpr;
		if desc.staticColor then
			local sc = desc.staticColor;
			colorExpr = [[
frame.]] .. objname .. [[:SetTextColor(]] .. sc.r .. "," .. sc.g .. "," .. sc.b .. "," .. sc.a .. [[);
]];
		elseif desc.color then
			local colorVar = strtrim(desc.color or "");
			colorExpr = [[ 
frame.]] .. objname .. [[:SetTextColor(explodeRGBA(]] .. colorVar .. [[));
]];
		else
			colorExpr = [[ ]];
		end

		---- Generate the code.
		local createCode = _EmitCreateFontString(desc) .. [[
txt:SetPoint(]] .. RDXUI.AnchorCodeFromDescriptor(desc.anchor) .. [[);
]] .. VFLUI.GenerateSetFontCode("txt", desc.font, nil, true) .. [[
txt:Show();
frame.]] .. objname .. [[ = txt;
]];

		local destroyCode = [[
frame.]] .. objname .. [[:Destroy(); frame.]] .. objname .. [[ = nil;
]];

		local cleanupCode = [[
frame.]] .. objname .. [[:SetText(""); frame.]] .. objname .. [[:SetTextColor(1,1,1,1);
]];

		local paintCode = statusText[desc.ty].GeneratePaintCode(objname) .. colorExpr;

		state:Attach(state:Slot("EmitCreate"), true, function(code) code:AppendCode(createCode); end);
		state:Attach(state:Slot("EmitDestroy"), true, function(code) code:AppendCode(destroyCode); end);
		state:Attach(state:Slot("EmitCleanup"), true, function(code) code:AppendCode(cleanupCode); end);
		state:Attach(state:Slot("EmitPaint"), true, function(code) code:AppendCode(paintCode); end);

		statusText[desc.ty].OnApply(desc, state);
		
		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);

		local ed_name, ed_width, ed_height = RDXUI.GenNameWidthHeightPortion(ui, desc);

		-- Owner
		local owner = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Owner"), state, "Subframe_", true);
		if desc and desc.owner then owner:SetSelection(desc.owner); end

		local anchor = RDXUI.UnitFrameAnchorSelector:new(ui); anchor:Show();
		anchor:SetAFArray(RDXUI.ComposeFrameList(state));
		if desc and desc.anchor then anchor:SetAnchorInfo(desc.anchor); end
		ui:InsertFrame(anchor);

		local er = RDXUI.EmbedRight(ui, i18n("Font"));
		local fontsel = VFLUI.MakeFontSelectButton(er, desc.font); fontsel:Show();
		er:EmbedChild(fontsel); er:Show();
		ui:InsertFrame(er);
		
		er = RDXUI.EmbedRight(ui, i18n("Display"));
		local display = VFLUI.Dropdown:new(er, function() return statusIndex; end);
		display:SetWidth(250); display:Show();
		display:SetSelection(statusText[desc.ty].title, desc.ty);
		er:EmbedChild(display); er:Show(); ui:InsertFrame(er);
		
		local chk_static = VFLUI.Checkbox:new(ui); chk_static:Show();
		chk_static:SetText(i18n("Use static color"));
		local swatch = VFLUI.ColorSwatch:new(chk_static);
		swatch:SetPoint("RIGHT", chk_static, "RIGHT"); swatch:Show();
		if desc and desc.staticColor then 
			chk_static:SetChecked(true);
			swatch:SetColorTable(desc.staticColor);
		else 
			chk_static:SetChecked(); 
			swatch:SetColor(1,1,1,1);
		end
		ui:InsertFrame(chk_static);
		
		local colorVar = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Color variable"), state, "ColorVar_");
		if desc and desc.color then colorVar:SetSelection(desc.color); end

		function ui:GetDescriptor()
			local _,ty = display:GetSelection();
			local sc = nil;
			if chk_static:GetChecked() then sc = swatch:GetColor(); end
			local scolorVar = strtrim(colorVar:GetSelection() or "");
			if scolorVar == "" then scolorVar = nil; end
			return { 
				feature = "txt_status"; version = 1;
				ty = ty;
				name = ed_name.editBox:GetText();	owner = owner:GetSelection();
				w = ed_width.editBox:GetNumber(), h = ed_height.editBox:GetNumber(), 
				anchor = anchor:GetAnchorInfo();
				font = fontsel:GetSelectedFont();
				staticColor = sc;
				color = scolorVar;
			};
		end
		
		ui.Destroy = VFL.hook(function() swatch:Destroy(); end, ui.Destroy);
		
		return ui;
	end;
	CreateDescriptor = function()
		return { 
			feature = "txt_status"; version = 1;
			ty = "fdld";
			name = "status_text", w = 40, h = 14, anchor = { lp = "RIGHT", af = "Base", rp = "RIGHT", dx = 0, dy = 0 };
			font = VFL.copy(Fonts.Default);
		};
	end;
});

-------------------- NAMEPLATE
RDX.RegisterFeature({
	name = "txt_np"; version = 1; multiple = true;
	title = i18n("Nameplate");
	category = i18n("Text");
	IsPossible = function(state)
		if not state:Slot("UnitFrame") then return nil; end
		if not state:Slot("Base") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		if (not desc) then VFL.AddError(errs, i18n("Missing descriptor.")); return nil; end
		if not VFLUI.isFacePathExist(desc.font.face) then VFL.AddError(errs, i18n("Font path not found.")); return nil; end
		if not __UFFrameCheck_Proto("Frame_", desc, state, errs) then return nil; end
		return true;
	end;
	ApplyFeature = function(desc, state)
		local objname = "Frame_" .. desc.name;

		-- Text
		local textExpr;
		if desc.trunc then
			textExpr = "unit:GetProperName():sub(1, " .. desc.trunc .. ")";
		else
			textExpr = "unit:GetProperName()";
		end
		
		-- Color
		local colorclassBoo, colorVar, colorBoo = "false", strtrim(desc.color or ""), "false";
		if desc.classColor then colorclassBoo = "true"; end
		if colorVar ~= "" then colorBoo = "true"; end

		local createCode = _EmitCreateFontString(desc) .. [[
txt:SetPoint(]] .. RDXUI.AnchorCodeFromDescriptor(desc.anchor) .. [[);
]] .. VFLUI.GenerateSetFontCode("txt", desc.font, nil, true) .. [[
txt:Show();
frame.]] .. objname .. [[ = txt;
]];

		local destroyCode = [[
frame.]] .. objname .. [[:Destroy(); frame.]] .. objname .. [[ = nil;
]];

		local cleanupCode = [[
frame.]] .. objname .. [[:SetText(""); frame.]] .. objname .. [[:SetTextColor(1,1,1,1);
]];

		local paintCode = [[
if UnitExists(uid) then
frame.]] .. objname .. [[:SetText(]] .. textExpr .. [[);
else
frame.]] .. objname .. [[:SetText("");
end
if ]] .. colorclassBoo .. [[ then
	frame.]] .. objname .. [[:SetTextColor(explodeColor(unit:GetClassColor()));
end

if ]] .. colorBoo .. [[ then
	frame.]] .. objname .. [[:SetTextColor(explodeRGBA(]] .. colorVar .. [[));
end

]];

		state:Attach(state:Slot("EmitCreate"), true, function(code) code:AppendCode(createCode); end);
		state:Attach(state:Slot("EmitDestroy"), true, function(code) code:AppendCode(destroyCode); end);
		state:Attach(state:Slot("EmitCleanup"), true, function(code) code:AppendCode(cleanupCode); end);
		state:Attach(state:Slot("EmitPaint"), true, function(code) code:AppendCode(paintCode); end);
		
		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);

		local ed_name, ed_width, ed_height = RDXUI.GenNameWidthHeightPortion(ui, desc);

		-- Owner
		local owner = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Owner"), state, "Subframe_", true);
		if desc and desc.owner then owner:SetSelection(desc.owner); end

		local anchor = RDXUI.UnitFrameAnchorSelector:new(ui); anchor:Show();
		anchor:SetAFArray(RDXUI.ComposeFrameList(state));
		if desc and desc.anchor then anchor:SetAnchorInfo(desc.anchor); end
		ui:InsertFrame(anchor);

		local er = RDXUI.EmbedRight(ui, i18n("Font"));
		local fontsel = VFLUI.MakeFontSelectButton(er, desc.font); fontsel:Show();
		er:EmbedChild(fontsel); er:Show();
		ui:InsertFrame(er);

		local ed_trunc = VFLUI.LabeledEdit:new(ui, 50); ed_trunc:Show();
		ed_trunc:SetText(i18n("Max name length (blank = no truncation)"));
		if desc and desc.trunc then ed_trunc.editBox:SetText(desc.trunc); end
		ui:InsertFrame(ed_trunc);

		local chk_colorclass = VFLUI.Checkbox:new(ui); chk_colorclass:Show();
		chk_colorclass:SetText(i18n("Use class color"));
		if desc and desc.classColor then chk_colorclass:SetChecked(true); else chk_colorclass:SetChecked(); end
		ui:InsertFrame(chk_colorclass);
		
		local colorVar = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Color variable"), state, "ColorVar_");
		if desc and desc.color then colorVar:SetSelection(desc.color); end

		function ui:GetDescriptor()
			local trunc = tonumber(ed_trunc.editBox:GetText());
			if trunc then trunc = VFL.clamp(trunc, 1, 50); end
			local scolorVar = strtrim(colorVar:GetSelection() or "");
			if scolorVar == "" then
				scolorVar = nil;
			end
			return { 
				feature = "txt_np"; version = 1;
				trunc = trunc;
				classColor = chk_colorclass:GetChecked();
				name = ed_name.editBox:GetText();	owner = owner:GetSelection();
				w = ed_width.editBox:GetNumber(), h = ed_height.editBox:GetNumber(), 
				anchor = anchor:GetAnchorInfo();
				font = fontsel:GetSelectedFont();
				color = scolorVar;
			};
		end
		
		return ui;
	end;
	CreateDescriptor = function()
		return { 
			feature = "txt_np"; version = 1;
			name = "np", w = 50, h = 14, anchor = { lp = "LEFT", af = "Base", rp = "LEFT", dx = 0, dy = 0 };
			font = VFL.copy(Fonts.Default); 
		};
	end;
});

------------------------------------------------
-- COMPATIBILITY: Port old text features over
------------------------------------------------
_GenerateReplaceTextFeature(i18n("Custom Text"), "txt_custom", 1);
_GenerateReplaceTextFeature(i18n("Nameplate"), "txt_np", 1);
_GenerateReplaceTextFeature(i18n("Status Text"), "txt_status", 1, function(desc) desc.ty = "fdld"; end);

