-- FreeTimer.lua
-- RDX - Raid Data Exchange
-- (C)2007 Bill Johnson
--
-- THIS FILE CONTAINS COPYRIGHTED MATERIAL SUBJECT TO THE TERMS OF A SEPARATE
-- LICENSE. UNLICENSED COPYING IS PROHIBITED.
--
-- A "free timer" that runs independent of unit frame repainting, which allows event-driven
-- timers to work without a Periodic Repaint.

-- Modify by Sigg

FreeTimer = {};
-- Some useful constants
FreeTimer.SB_Hide = "ft.bar:Hide();";
FreeTimer.SB_Empty = "ft.bar:SetValue(0);";
FreeTimer.SB_Full = "ft.bar:SetValue(1);";

FreeTimer.Text_None = "ft.text:SetText('');";
FreeTimer.Text_Unknown = "ft.text:SetText('?');";

FreeTimer.TextFilter_Largest = "Emm";
FreeTimer.TextFilter_MinSec = "VFL.Time.FormatMinSec";
FreeTimer.TextFilter_Hundredths = "Hundredths";

local _TextTypes = {
	{ text = "Largest" },
	{ text = "MinSec" },
	{ text = "Hundredths" },
};
local function _tt_types() return _TextTypes; end

FreeTimer.Formula_CountUP = "1 - dt/dur";
FreeTimer.Formula_CountDOWN = "dt/dur";

local _CountTypes = {
	{ text = "CountUP" },
	{ text = "CountDOWN" },
};
local function _cc_types() return _CountTypes; end

local function ftSetTimer(self, start, dur)
	self.start = start; self.dur = dur;
end

FreeTimer.TextInfo_None = "ft.textInfo:SetText('');";
FreeTimer.TextInfo_Unknown = "ft.textInfo:SetText('?');";

FreeTimer.TexIcon_Hide = "ft.texIcon:Hide();";
FreeTimer.TexIcon_None = "ft.texIcon:SetTexture('');";
FreeTimer.TexIcon_Unknown = "ft.texIcon:SetTexture('Interface\\InventoryItems\\WoWUnknownItem01.blp');";

local function ftSetData(self, txt, tex)
	self.txt = txt;
	self.tex = tex;
end

local function ftDestroy(self)
	self.start = nil; self.dur = nil;
	self.bar = nil; self.text = nil; self.SetTimer = nil;
	self.textInfo = nil; self.texIcon = nil; self.SetData = nil;
end

function FreeTimer.CreateFreeTimerClass(statusBar, textTimer, formula, textFilter, textInfo, texIcon, unknownSB, unknownText, unknownTextInfo, unknownTexIcon, expiredSB, expiredText, expiredTextinfo, expiredTexIcon)
	-- Build the onupdate script.
	local onUpdate = [[
return function(ft)
	local dt,dur = ft.start or 0, ft.dur or 0;
	local txt, tex = ft.txt or "", ft.tex or "Interface\\InventoryItems\\WoWUnknownItem01.blp";
	if dt == 0 then
]];
	-- "Unknown"/"no timer" case.
	if statusBar then onUpdate = onUpdate .. unknownSB; end
	if textTimer then onUpdate = onUpdate .. unknownText; end
	if textInfo then onUpdate = onUpdate .. unknownTextInfo; end
	if texIcon then onUpdate = onUpdate .. unknownTexIcon; end
	-- End unknown case
	onUpdate = onUpdate .. [[
	  return;
	end
	dt = (dt + dur) - GetTime();
	if dt > 0 then
]];
	-- Within-bounds case
	if statusBar then onUpdate = onUpdate .. "ft.bar:Show(); ft.bar:SetValue(" .. formula .. ");"; end
	if textTimer then onUpdate = onUpdate .. "ft.text:SetText(" .. textFilter .. "(dt, dur));"; end
	if textInfo then onUpdate = onUpdate .. "ft.textInfo:SetText(txt);"; end
	if texIcon then onUpdate = onUpdate .. "ft.texIcon:Show(); ft.texIcon:SetTexture(tex);"; end
	-- End within bounds case
	onUpdate = onUpdate .. [[
	else
]];
	-- Expired case
	if statusBar then onUpdate = onUpdate .. expiredSB; end
	if textTimer then onUpdate = onUpdate .. expiredText; end
	if textInfo then onUpdate = onUpdate .. expiredTextinfo; end
	if texIcon then onUpdate = onUpdate .. expiredTexIcon; end
	-- End expired case
	onUpdate = onUpdate .. [[
	end
end;
]];
	local updater = loadstring(onUpdate);
	if updater then updater = updater(); else updater = VFL.Noop; end

	return function(parent, sb, textTimer, textInfo, texIcon)
		local f = VFLUI.AcquireFrame("Frame");
		f:SetParent(parent);
		f:Show();
		f.bar = sb; f.text = textTimer; f.SetTimer = ftSetTimer;
		f.textInfo = textInfo; f.texIcon = texIcon; f.SetData = ftSetData;
		f:SetScript("OnUpdate", updater);
		f.Destroy = VFL.hook(ftDestroy, f.Destroy);
		return f;
	end
end

RDX.RegisterFeature({
	name = "free_timer"; version = 1;	multiple = true;
	title = i18n("Timer Shader"); category = i18n("Shaders");
	IsPossible = function(state)
		if not state:HasSlots("UnitFrame", "EmitClosure", "EmitCreate", "EmitPaint", "EmitDestroy") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		if not desc then VFL.AddError(errs, i18n("Missing descriptor.")); return nil; end
		return true;
	end;
	ApplyFeature = function(desc, state, errs)
		local objname = "FreeTimer_" .. math.random(10000000); -- Generate a random ID.
		local sb = strtrim(desc.statusBar or "");
		local textTimer = strtrim(desc.text or "");
		local ct = desc.countType or "CountUP";
		local tet = desc.textType or "Hundredths";
		local textInfo = strtrim(desc.textInfo or "");
		local texIcon = strtrim(desc.texIcon or "");
		local textInfodata = desc.txt or "";
		local texIcondata = desc.tex or "";
		local sbPresent, txtPresent, txtInfoPresent, texIconPresent = "true", "true", "true", "true";
		if sb == "" then sbPresent = "false"; sb = "nil"; else sb = RDXUI.ResolveFrameReference(desc.statusBar); end
		if textTimer == "" then txtPresent = "false"; textTimer = "nil"; else textTimer = RDXUI.ResolveFrameReference(desc.text); end
		if textInfo == "" then txtInfoPresent = "false"; textInfo = "nil"; else textInfo = RDXUI.ResolveFrameReference(desc.textInfo); end
		if texIcon == "" then texIconPresent = "false"; texIcon = "nil"; else texIcon = "frame.Tex_" .. desc.texIcon; end
		if textInfodata == "" then textInfodata = "nil"; end
		if texIcondata == "" then texIcondata = "nil"; end
		
		--- Closure
		-- Create a free timer class for our frame (this avoids the nasty situation where we have to recompile
		-- the code every time a frame is created.)
		local closureCode = [[
local ftc_]] .. objname .. [[ = FreeTimer.CreateFreeTimerClass(]] .. sbPresent .. [[,]] .. txtPresent .. [[, FreeTimer.Formula_]] .. ct .. [[, FreeTimer.TextFilter_]] .. tet .. [[, ]] .. txtInfoPresent .. [[, ]] .. texIconPresent .. [[, FreeTimer.SB_Hide, FreeTimer.Text_None, FreeTimer.TextInfo_None, FreeTimer.TexIcon_Hide, FreeTimer.SB_Hide, FreeTimer.Text_None, FreeTimer.TextInfo_None, FreeTimer.TexIcon_Hide);
]];
		state:Attach("EmitClosure", true, function(code) code:AppendCode(closureCode); end);

		--- Creation
		-- The free timer is just a frame with an OnUpdate routine that updates the linked objects.
		local createCode = [[
frame.]] .. objname .. [[ = ftc_]] .. objname .. [[(frame, ]] .. sb .. [[, ]] .. textTimer .. [[, ]] .. textInfo .. [[, ]] .. texIcon .. [[);
]];
		state:Attach("EmitCreate", true, function(code) code:AppendCode(createCode); end);

		--- Paint
		local paintCode = [[
frame.]] .. objname .. [[:SetTimer(]] .. desc.timerVar .. [[_start, ]] .. desc.timerVar .. [[_duration);
frame.]] .. objname .. [[:SetData(]] .. textInfodata .. [[, ]] .. texIcondata .. [[);
]];
		state:Attach("EmitPaint", true, function(code) code:AppendCode(paintCode); end);

		--- Destruction
		local destroyCode = [[
frame.]] .. objname .. [[:Destroy(); frame.]] .. objname .. [[ = nil;
]];
		state:Attach("EmitDestroy", true, function(code) code:AppendCode(destroyCode); end);

		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);
		
		local timerVar = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Timer variable"), state, "TimerVar_");
		if desc and desc.timerVar then timerVar:SetSelection(desc.timerVar); end
		
		local statusBar = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Status Bar"), state, "StatusBar_");
		if desc and desc.statusBar then statusBar:SetSelection(desc.statusBar); end
		
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Text Timer")));
		
		local text = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Text Timer"), state, "Text_");
		if desc and desc.text then text:SetSelection(desc.text); end
		
		local er = RDXUI.EmbedRight(ui, i18n("Count Type:"));
		local dd_countType = VFLUI.Dropdown:new(er, _cc_types);
		dd_countType:SetWidth(150); dd_countType:Show();
		if desc and desc.countType then 
			dd_countType:SetSelection(desc.countType); 
		else
			dd_countType:SetSelection("CountUP");
		end
		er:EmbedChild(dd_countType); er:Show();
		ui:InsertFrame(er);
		
		--local cflag = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Count Type variable"), state, "BoolVar_");
		--if desc and desc.cflag then cflag:SetSelection(desc.cflag); end
		
		local tt = RDXUI.EmbedRight(ui, i18n("Text Timer Type:"));
		local dd_textType = VFLUI.Dropdown:new(tt, _tt_types);
		dd_textType:SetWidth(150); dd_textType:Show();
		if desc and desc.textType then 
			dd_textType:SetSelection(desc.textType); 
		else
			dd_textType:SetSelection("Hundredths");
		end
		tt:EmbedChild(dd_textType); tt:Show();
		ui:InsertFrame(tt);
		
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Text Info (spell name, aura name ...)")));
		
		local textInfo = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Text Info"), state, "Text_");
		if desc and desc.textInfo then textInfo:SetSelection(desc.textInfo); end
		
		local txt = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Text Info data"), state, "Txt_");
		if desc and desc.txt then txt:SetSelection(desc.txt); end
		
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Icon texture")));
		
		local texIcon = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Texture Icon"), state, "Tex_");
		if desc and desc.texIcon then texIcon:SetSelection(desc.texIcon); end

		local tex = RDXUI.MakeSlotSelectorDropdown(ui, i18n("Texture Icon data"), state, "TexNameVar_");
		if desc and desc.tex then tex:SetSelection(desc.tex); end

		function ui:GetDescriptor()
			return {
				feature = "free_timer"; version = 1;
				timerVar = timerVar:GetSelection();
				statusBar = statusBar:GetSelection();
				text = text:GetSelection();
				countType = dd_countType:GetSelection();
				--cflag = cflag:GetSelection();
				textType = dd_textType:GetSelection();
				textInfo = textInfo:GetSelection(); -- frame
				txt =  txt:GetSelection(); -- data
				texIcon = texIcon:GetSelection(); -- texture
				tex = tex:GetSelection(); -- data
			};
		end

		return ui;
	end;
	CreateDescriptor = function()
		return {
			feature = "free_timer"; version = 1;
		};
	end;
});

