-- AuraBars.lua 
-- Code for aura bars attached to unitframes.
-- Sigg Rashgarroth EU
--

local function NEditor(ctr, nC, label, ew)
	label = label or ""; ew = ew or 50;
	
	local f = VFLUI.AcquireFrame("Frame");
	VFLUI.StdSetParent(f, ctr);
	f:SetHeight(25);
	f.DialogOnLayout = VFL.Noop; f:Show();

	local t1 = VFLUI.CreateFontString(f);
	t1:SetFontObject(Fonts.Default10); t1:SetWidth(150); t1:SetHeight(25);
	t1:SetJustifyH("LEFT");
	t1:SetPoint("LEFT", f, "LEFT");
	t1:SetText(label); t1:Show();

	f.edit = {};
	local af = t1;
	for i=1,nC do
		local ed = VFLUI.Edit:new(f);
		ed:SetHeight(25); ed:SetWidth(ew);
		ed:SetPoint("LEFT", af, "RIGHT", 1, 0); ed:Show();
		af = ed; f.edit[i] = ed;
	end

	function f:SetNumbers(n1,n2,n3,n4)
		if n1 and self.edit[1] then self.edit[1]:SetText(n1); end
		if n2 and self.edit[2] then self.edit[2]:SetText(n2); end
		if n3 and self.edit[3] then self.edit[3]:SetText(n3); end
		if n4 and self.edit[4] then self.edit[4]:SetText(n4); end
	end
	
	function f:GetNumbers(cmin, cmax)
		local n = {};
		for i=1,nC do
			n[i] = VFL.clamp(self.edit[i]:GetNumber(), cmin, cmax);
		end
		return n[1], n[2], n[3], n[4];
	end

	f.Destroy = VFL.hook(function(s)
		s.SetNumbers = nil; s.GetNumbers = nil;
		for _,editor in pairs(s.edit) do editor:Destroy(); end
		VFLUI.ReleaseRegion(t1);
		s.edit = nil;
	end, f.Destroy);

	return f;
end

function __SetAuraBar(btn, meta, tex, apps, dur, tl, showTime, dispelType)
	btn:Show(); 
	btn.meta = meta;
	--btn.tex:SetTexture(tex);
	
	if dispelType then
		btn:SetBackdropColor(explodeColor(DebuffTypeColor[dispelType]));
	else
		btn:SetBackdropColor(0,0,0,1);
	end
	-- Cooldown
	if dur and dur>0 then
		btn.cd:SetCooldown(GetTime() + tl - dur , dur);
	else
		btn.cd:SetCooldown(0, 0);
	end
	-- Text
	if not showTime then
		if apps and (apps > 1) then btn.cd.fs:SetText(apps); else btn.cd.fs:SetText(""); end
	end
	
	return true;
end

function __fraccalc(ftimeleft, fdur)
	local fracc = 1;
	if ftimeleft and fdur then 
		if ftimeleft <= 0.1 then ftimeleft = 0.1; end
		fracc = 1 - VFL.clamp(ftimeleft / fdur, 0, 1);
		if fracc == 0 then fracc = 1; end
	end
	return fracc
end

--------------- Code emitter helpers
local function _EmitCreateCode(objname, desc)
	local ty = '"Frame"'; 
	local cdgfx = "false";
	local cdtext = "nil"; if desc.text == "TIME" then cdtext = "true"; end
	local createCode = [[
frame.]] .. objname .. [[ = {};
local btn, btnOwner = nil, ]] .. RDXUI.ResolveFrameReference(desc.owner) .. [[;
for i=1,]] .. desc.nIcons .. [[ do
	btn = VFLUI.AcquireFrame(]] .. ty .. [[); 
	btn:SetParent(btnOwner);
	btn:SetFrameLevel(btnOwner:GetFrameLevel());
	btn:SetBackdrop(VFLUI.WhiteBackdrop);
	btn.tex = VFLUI.CreateTexture(btn);
	btn.tex:SetPoint("TOPLEFT", btn, "TOPLEFT", 1, -1);
	btn.tex:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", -1, 1);
	btn.tex:Show();
	btn.cd = RDXUI.CooldownCounter:new(btn, ]] .. cdtext .. ", " .. cdgfx .. [[, true, 0.3);
	btn.cd:SetAllPoints(btn); btn.cd:Show();
]];
	createCode = createCode .. VFLUI.GenerateSetTextureCode("btn.tex", desc.texture);
	createCode = createCode .. VFLUI.GenerateSetFontCode("btn.cd.fs", desc.font, nil, true);
	createCode = createCode .. [[
	frame.]] .. objname .. [[[i] = btn;
end
frame.]] .. objname .. [[[1]:SetPoint(]] .. RDXUI.AnchorCodeFromDescriptor(desc.anchor) .. [[);
for i=2,]] .. desc.nIcons .. [[ do ]];
	local opri1, opri2, osec1, osec2 = '"RIGHT"', '"LEFT"', '"TOP"', '"BOTTOM"';
	if desc.orientation == "RIGHT" then
		opri1 = '"LEFT"'; opri2 = '"RIGHT"';
	elseif desc.orientation == "DOWN" then
		opri1 = '"TOPLEFT"'; opri2 = '"BOTTOMLEFT"'; osec1 = '"LEFT"'; osec2 = '"RIGHT"';
	elseif desc.orientation == "UP" then
		opri1 = '"BOTTOM"'; opri2 = '"TOP"'; osec1 = '"LEFT"'; osec2 = '"RIGHT"';
	end
	-- Single-row code
	createCode = createCode .. [[frame.]] .. objname .. [[[i]:SetPoint(]] .. opri1 .. [[, frame.]] .. objname .. [[[i-1], ]] .. opri2 .. [[);]];
	createCode = createCode .. [[
end
]];
	return createCode;
end


local _orientations = {
	{ text = "LEFT"},
	{ text = "RIGHT"},
	{ text = "DOWN"},
	{ text = "UP"},
};
local function _dd_orientations() return _orientations; end

local _types = {
	{ text = "BUFFS" },
	{ text = "DEBUFFS" },
};
local function _dd_types() return _types; end

-----------------------------
-- AURA BARS
-----------------------------
RDX.RegisterFeature({
	name = "aura_bars"; version = 1; title = i18n("Aura Bars"); category = i18n("Buffs/Debuffs");
	multiple = true;
	deprecated = true;
	IsPossible = function(state)
		if not state:Slot("UnitFrame") then return nil; end
		if not state:Slot("Base") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		local flg = true;
		if not desc then VFL.AddError(errs, i18n("No descriptor")); return nil; end
		if not desc.auraType then VFL.AddError(errs, i18n("Bad aura type")); return nil; end
		if desc.name and (not state:Slot("Bars_" .. desc.name)) then
			state:AddSlot("Bars_" .. desc.name);
		else
			flg = nil; VFL.AddError(errs, i18n("Missing or duplicate object name."));
		end
		return flg;
	end;
	ApplyFeature = function(desc, state)
		local objname = "Bars_" .. desc.name;
		local loadCode = "RDX.LoadBuffFromUnit";

		-- Event hinting.
		local mux, mask = state:GetContainingWindowState():GetSlotValue("Multiplexer"), 0;
		if desc.auraType == "DEBUFFS" then
			mask = mux:GetPaintMask("DEBUFFS");
			mux:Event_UnitMask("UNIT_DEBUFF_*", mask);
			loadCode = "RDX.LoadDebuffFromUnit";
		else
			mask = mux:GetPaintMask("BUFFS");
			mux:Event_UnitMask("UNIT_BUFF_*", mask);
		end
		mask = bit.bor(mask, 1);

		-- If there's an external filter, add a quick menu to the window to edit it.
		if desc.externalNameFilter then
			local path = desc.externalNameFilter; local afname = desc.name;
			state:GetContainingWindowState():Attach("Menu", true, function(win, mnu)
				table.insert(mnu, {
					text = i18n("Edit AuraFilter: ") .. afname;
					OnClick = function()
						VFL.poptree:Release();
						RDXDB.OpenObject(path, "Edit");
					end;
				});
			end);
		end

		------------ Closure (name filter array)
		if desc.filterName then
			local closureCode = [[
local ]] .. objname .. [[_fnames = ]];
			if desc.externalNameFilter then
				closureCode = closureCode .. [[RDXDB.GetObjectInstance(]] .. string.format("%q", desc.externalNameFilter) .. [[);
]];
			else
				-- Internal filter
				closureCode = closureCode .. [[{};
]];
				for _,name in pairs(desc.filterNameList) do
					closureCode = closureCode .. objname .. "_fnames[" .. string.format("%q", name) .. "] = true; ";
				end
			end
			state:Attach("EmitClosure", true, function(code) code:AppendCode(closureCode); end);
		end

		----------------- Creation
		local createCode = _EmitCreateCode(objname, desc);
		state:Attach("EmitCreate", true, function(code) code:AppendCode(createCode); end);

		------------------- Destruction
		local destroyCode = [[
local btn = nil;
for i=1,]] .. desc.nIcons .. [[ do
	btn = frame.]] .. objname .. [[[i]
	btn.meta = nil;
	VFLUI.ReleaseRegion(btn.tex); btn.tex = nil;
	btn.cd:Destroy(); btn.cd = nil;
	btn:Destroy();
end
frame.]] .. objname .. [[ = nil;
]];
		state:Attach("EmitDestroy", true, function(code) code:AppendCode(destroyCode); end);

		------------------- Paint
		local engselftimer = "false"; if desc.selftimer then engselftimer = "true"; end
		local engauratimer = "false"; if desc.auratimer then engauratimer = "true"; end
		local raidfilter = "nil"; if desc.raidfilter then raidfilter = "true"; end
		local timefilter = "true"; if desc.timefilter then timefilter = "(_dur and _dur > 0)"; end
		local namefilter = "true";
		if desc.filterName == "include" then
			namefilter = "(" .. objname .. "_fnames[_bn] or " .. objname .. "_fnames[_meta.category])";
		elseif desc.filterName == "exclude" then
			namefilter = "(not (" .. objname .. "_fnames[_bn] or " .. objname .. "_fnames[_meta.category]))"
		end
		local showTime = "nil"; if desc.text == "TIME" then showTime = "true"; end

		local paintCode = [[
if band(paintmask, ]] .. mask .. [[) ~= 0 then
	local _i,_j,_bn,_tex,_apps,_meta,_dur,_tl,_dispelt, fraca = 1,1,nil,nil,nil,nil,nil,nil,0.1;
	local _icons = frame.]] .. objname .. [[;
	while true do
		if (_j > ]] .. desc.nIcons .. [[) then break; end
		_, _bn, _, _apps, _, _tex, _meta, _dur, _tl, _dispelt = ]] .. loadCode .. [[(uid, _i, ]] .. raidfilter .. [[, ]] .. engselftimer .. [[, ]] .. engauratimer .. [[);
		if not _meta then break; end
		if (not _meta.isInvisible) and ]] .. timefilter .. [[ and ]] .. namefilter .. [[ then
			__SetAuraBar(_icons[_j], _meta, tex, _apps, _dur, _tl, ]] .. showTime .. [[, _dispelt);
			
			fraca = __fraccalc(_tl,_dur);
			
			_icons[_j]:SetWidth(lerp1(fraca,]] .. desc.w1 .. "," .. desc.w2 .. [[));
			_icons[_j]:SetHeight(lerp1(fraca,]] .. desc.h1 .. "," .. desc.h2 .. [[));
			
			
			_j = _j + 1;
		end
		_i = _i + 1;
	end
	while _j <= ]] .. desc.nIcons .. [[ do
		_icons[_j]:Hide(); _j = _j + 1;
	end
end
]];
		state:Attach("EmitPaint", true, function(code) code:AppendCode(paintCode); end);
		------------------- Cleanup
		local cleanupCode = [[
local btn = nil;
for i=1,]] .. desc.nIcons .. [[ do
	btn = frame.]] .. objname .. [[[i];
	btn:Hide(); btn.meta = nil;
end
]];
		state:Attach("EmitCleanup", true, function(code) code:AppendCode(cleanupCode); end);


		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);

		------------- Core
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Core Parameters")));

		local ed_name = VFLUI.LabeledEdit:new(ui, 100); ed_name:Show();
		ed_name:SetText(i18n("Name"));
		ed_name.editBox:SetText(desc.name);
		ui:InsertFrame(ed_name);

		local er = RDXUI.EmbedRight(ui, i18n("Aura Type:"));
		local dd_auraType = VFLUI.Dropdown:new(er, _dd_types);
		dd_auraType:SetWidth(75); dd_auraType:Show();
		if desc and desc.auraType then 
			dd_auraType:SetSelection(desc.auraType); 
		else
			dd_auraType:SetSelection("BUFFS");
		end
		er:EmbedChild(dd_auraType); er:Show();
		ui:InsertFrame(er);

		------------- Layout
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Layout")));

		local owner = RDXUI.MakeSlotSelectorDropdown(ui, "Owner", state, "Subframe_", true);
		if desc and desc.owner then owner:SetSelection(desc.owner); end

		local anchor = RDXUI.UnitFrameAnchorSelector:new(ui); anchor:Show();
		anchor:SetAFArray(RDXUI.ComposeFrameList(state));
		if desc and desc.anchor then anchor:SetAnchorInfo(desc.anchor); end
		ui:InsertFrame(anchor);

		local ed_nbbars = VFLUI.LabeledEdit:new(ui, 50); ed_nbbars:Show();
		ed_nbbars:SetText(i18n("Max bars"));
		if desc and desc.nIcons then ed_nbbars.editBox:SetText(desc.nIcons); end
		ui:InsertFrame(ed_nbbars);

		local er = RDXUI.EmbedRight(ui, i18n("Orientation:"));
		local dd_orientation = VFLUI.Dropdown:new(er, _dd_orientations);
		dd_orientation:SetWidth(75); dd_orientation:Show();
		if desc and desc.orientation then 
			dd_orientation:SetSelection(desc.orientation); 
		else
			dd_orientation:SetSelection("DOWN");
		end
		er:EmbedChild(dd_orientation); er:Show();
		ui:InsertFrame(er);
		
		local wh1 = NEditor(ui, 2, i18n("Empty width/height"), 50);
		if desc then wh1:SetNumbers(desc.w1, desc.h1); end
		ui:InsertFrame(wh1);
		
		local wh2 = NEditor(ui, 2, i18n("Full width/height"), 50);
		if desc then wh2:SetNumbers(desc.w2, desc.h2); end
		ui:InsertFrame(wh2);
		
		-------------- Display
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Display")));
		
		local etx = RDXUI.EmbedRight(ui, i18n("Texture"));
		local tsel = VFLUI.MakeTextureSelectButton(etx, desc.texture); tsel:Show();
		etx:EmbedChild(tsel); etx:Show();
		ui:InsertFrame(etx);

		local er = RDXUI.EmbedRight(ui, i18n("Font"));
		local fontsel = VFLUI.MakeFontSelectButton(er, desc.font); fontsel:Show();
		er:EmbedChild(fontsel); er:Show();
		ui:InsertFrame(er);

		local rg_text = VFLUI.RadioGroup:new(ui); rg_text:Show();
		rg_text:SetLayout(2,2);
		rg_text.buttons[1]:SetText(i18n("Stack")); rg_text.buttons[2]:SetText(i18n("Time Left"));
		rg_text:SetValue(1);
		if desc and desc.text == "TIME" then rg_text:SetValue(2); end
		ui:InsertFrame(rg_text);
		
		-------------- Timer
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Timer")));
		
		local eng_selftimer = VFLUI.Checkbox:new(ui); eng_selftimer:Show();
		eng_selftimer:SetText(i18n("Use GetPlayerBuffName"));
		if desc and desc.selftimer then eng_selftimer:SetChecked(true); else eng_selftimer:SetChecked(); end
		ui:InsertFrame(eng_selftimer);
		
		local eng_auratimer = VFLUI.Checkbox:new(ui); eng_auratimer:Show();
		eng_auratimer:SetText(i18n("Use Aura Timer Engine"));
		if desc and desc.auratimer then eng_auratimer:SetChecked(true); else eng_auratimer:SetChecked(); end
		ui:InsertFrame(eng_auratimer);

		------------ Filter
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Filtering")));

		local chk_raidfilter = VFLUI.Checkbox:new(ui); chk_raidfilter:Show();
		chk_raidfilter:SetText(i18n("Use Blizzard raid filter"));
		if desc and desc.raidfilter then chk_raidfilter:SetChecked(true); else chk_raidfilter:SetChecked(); end
		ui:InsertFrame(chk_raidfilter);

		local chk_timefilter = VFLUI.Checkbox:new(ui); chk_timefilter:Show();
		chk_timefilter:SetText(i18n("Show only auras for which timers are known"));
		if desc and desc.timefilter then chk_timefilter:SetChecked(true); else chk_timefilter:SetChecked(); end
		ui:InsertFrame(chk_timefilter);

		local chk_filterName = VFLUI.Checkbox:new(ui); chk_filterName:Show();
		chk_filterName:SetText(i18n("Filter by aura name"));
		if desc and desc.filterName then chk_filterName:SetChecked(true); else chk_filterName:SetChecked(); end
		ui:InsertFrame(chk_filterName);

		local rg_ie = VFLUI.RadioGroup:new(ui); rg_ie:Show();
		rg_ie:SetLayout(2,2);
		rg_ie.buttons[1]:SetText(i18n("Include")); rg_ie.buttons[2]:SetText(i18n("Exclude"));
		if desc and desc.filterName == "include" then
			rg_ie:SetValue(1);
		elseif desc and desc.filterName == "exclude" then
			rg_ie:SetValue(2);
		end
		ui:InsertFrame(rg_ie);

		local chk_external = RDXUI.CheckEmbedRight(ui, i18n("Use external aura list"));
		local file_external = RDXDB.ObjectFinder:new(chk_external, function(p,f,md) return (md and md.ty and string.find(md.ty, "AuraFilter$")); end);
		file_external:SetWidth(200); file_external:Show();
		chk_external:EmbedChild(file_external); chk_external:Show();
		ui:InsertFrame(chk_external);
		if desc.externalNameFilter then
			chk_external:SetChecked(true); file_external:SetPath(desc.externalNameFilter);
		else
			chk_external:SetChecked();
		end

		local le_names = VFLUI.ListEditor:new(ui, desc.filterNameList or {}, function(cell,data) cell.text:SetText(data); end);
		le_names:SetHeight(183); le_names:Show();
		ui:InsertFrame(le_names);
		
		function ui:GetDescriptor()
			local w1,h1 = wh1:GetNumbers(0.1, 1000);
			local w2,h2 = wh2:GetNumbers(0.1, 1000);
			local filterName, filterNameList, ext = nil, nil, nil;
			if chk_filterName:GetChecked() then
				filterName = "include";
				if rg_ie:GetValue() == 2 then filterName = "exclude"; end
				filterNameList = le_names:GetList();
				for k,v in pairs(filterNameList) do filterNameList[k] = string.lower(v); end
				if chk_external:GetChecked() then	ext = file_external:GetPath(); end
			end
			local text = "STACK";
			if rg_text:GetValue() == 2 then text = "TIME"; end
			return { 
				feature = "aura_bars"; version = 1;
				name = ed_name.editBox:GetText();
				auraType = dd_auraType:GetSelection();
				owner = owner:GetSelection();
				anchor = anchor:GetAnchorInfo();
				nIcons = VFL.clamp(ed_nbbars.editBox:GetNumber(), 1, 40);
				orientation = dd_orientation:GetSelection();
				w1 = w1; h1 = h1; w2 = w2; h2 = h2;
				texture = tsel:GetSelectedTexture();
				font = fontsel:GetSelectedFont();
				text = text;
				selftimer = eng_selftimer:GetChecked();
				auratimer = eng_auratimer:GetChecked();
				raidfilter = chk_raidfilter:GetChecked();
				timefilter = chk_timefilter:GetChecked();
				filterName = filterName;
				externalNameFilter = ext;
				filterNameList = filterNameList;
			};
		end

		return ui;
	end;
	CreateDescriptor = function()
		local font = VFL.copy(Fonts.Default); font.size = 8;
		return { 
			feature = "aura_bars"; version = 1;
			name = "ai2"; auraType = "BUFFS";
			owner = "Base";
			anchor = { lp = "LEFT", af = "Base", rp = "RIGHT", dx = 0, dy = 0};
			nIcons = 4; orientation = "DOWN";
			w1 = 100; h1 = 14; w2 = 0.1; h2 = 14;
			font = font; text = "STACK";
		};
	end;
});

--------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------

-- aura bar 2

function __SetAuraBar2(btn, meta, tex, apps, dur, tl, showTime, dispelType, showBorder, auratype, usedebuffcolor, auranametrunc, auranameab)
	btn:Show(); btn.meta = meta;
	btn.icontex:SetTexture(tex);
	if apps and (apps > 1) then btn.icontxt:SetText(apps); else btn.icontxt:SetText(""); end
	if auranameab then
		local word, anstr = nil, "";
		for word in string.gmatch(meta.properName, "%a+")
			do anstr = anstr .. word:sub(1, 1);
		end
		btn.sbtxt:SetText(anstr);
	elseif auranametrunc then
		btn.sbtxt:SetText(string.sub(meta.properName, 1, auranametrunc));
	else
		btn.sbtxt:SetText(meta.properName);
	end
	if auratype == "DEBUFFS" and usedebuffcolor then
		--if showBorder then
		--	if dispelType then
				--btn:SetBackdropBorderColor(explodeColor(DebuffTypeColor[dispelType]));
		--	end
		--else
			if dispelType then
				btn.sb:SetColorTable(DebuffTypeColor[dispelType]);
			else
				btn.sb:SetColorTable(_grey);
			end
		--end
	end
	
	if dur and dur > 0 then
		btn.ftc:SetTimer(GetTime() + tl - dur , dur);
	else
		btn.ftc:SetTimer(0, 0);
	end
	return true;
end

--------------- Code emitter helpers
local function _EmitCreateCode2(objname, desc)
	local nRows = VFL.clamp(desc.rows, 1, 40);
	local ty = '"Frame"'; if (not desc.ephemeral) then ty = '"Button"'; end
	local bs = 1;
	--if not desc.bkd then desc.bkd = {} end
	if desc.bkd.edgeSize then bs = desc.bkd.edgeSize/3; end
	local createCode = [[
frame.]] .. objname .. [[ = {};
local btn, btnOwner = nil, ]] .. RDXUI.ResolveFrameReference(desc.owner) .. [[;
for i=1,]] .. desc.nBars .. [[ do
	btn = VFLUI.AcquireFrame(]] .. ty .. [[);
	btn:SetParent(btnOwner);
	btn:SetFrameLevel(btnOwner:GetFrameLevel());
	btn:SetWidth(]] .. desc.bwidth .. [[); btn:SetHeight(]] .. desc.bheight .. [[);
	VFLUI.SetBackdrop(btn, ]] .. Serialize(desc.bkd) .. [[);
	
	btn.icontex = VFLUI.CreateTexture(btn);
	btn.icontex:SetPoint("TOPLEFT", btn, "TOPLEFT",0,0);
	btn.icontex:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", -(]] .. desc.bwidth .. [[-]] .. desc.bheight .. [[), 0);
	btn.icontex:SetTexCoord(0.08, 1-0.08, 0.08, 1-0.08);
	btn.icontex:Show();
	btn.icontxt = VFLUI.CreateFontString(btn);
	btn.icontxt:SetAllPoints(btn.icontex); 
	btn.icontxt:Show();
]];
	createCode = createCode .. VFLUI.GenerateSetFontCode("btn.icontxt", desc.iconfont, nil, true);
	createCode = createCode .. [[
	btn.sb = VFLUI.StatusBarTexture:new(btn);
	btn.sb:SetOrientation("]] .. desc.borientation .. [[");
	btn.sb:SetPoint("TOPLEFT", btn, "TOPLEFT",]] .. desc.bheight .. [[,0);
	btn.sb:SetWidth(]] .. desc.bwidth .. [[-]] .. desc.bheight .. [[); btn.sb:SetHeight(]] .. desc.bheight .. [[);
	btn.sb:Show();
]];
	createCode = createCode .. VFLUI.GenerateSetTextureCode("btn.sb", desc.sbtexture);
	createCode = createCode .. [[
	btn.sbtxt = VFLUI.CreateFontString(btn);
	btn.sbtxt:SetPoint("TOPLEFT", btn, "TOPLEFT",]] .. desc.bheight .. [[,0);
	btn.sbtxt:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", 0, 0);
	btn.sbtxt:Show();
]];
	createCode = createCode .. VFLUI.GenerateSetFontCode("btn.sbtxt", desc.sbfont, nil, true);
	createCode = createCode .. [[
	btn.sbtimetxt = VFLUI.CreateFontString(btn);
	btn.sbtimetxt:SetPoint("TOPLEFT", btn, "TOPLEFT",]] .. desc.bheight .. [[,0);
	btn.sbtimetxt:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", 0, 0);
	btn.sbtimetxt:Show();
]];
	createCode = createCode .. VFLUI.GenerateSetFontCode("btn.sbtimetxt", desc.sbtimerfont, nil, true);
	createCode = createCode .. [[
	btn.ftc = ftctoto(frame, btn.sb, btn.sbtimetxt);
]];
	if (not desc.ephemeral) then createCode = createCode .. [[
	btn:SetScript("OnEnter", __AuraIconOnEnter);
	btn:SetScript("OnLeave", __AuraIconOnLeave);
	btn:RegisterForClicks("RightButtonUp");
	btn:SetScript("OnClick", __AuraIconOnClick);
]]; 
	end
createCode = createCode .. [[
	frame.]] .. objname .. [[[i] = btn;
end
frame.]] .. objname .. [[[1]:SetPoint(]] .. RDXUI.AnchorCodeFromDescriptor(desc.anchor) .. [[);
for i=2,]] .. desc.nBars .. [[ do ]];
	local opri1, opri2, osec1, osec2, csx, csy, csxm, csym = '"RIGHT"', '"LEFT"', '"TOP"', '"BOTTOM"', -tonumber(desc.iconspx), 0, 0, -tonumber(desc.iconspy);
	if desc.orientation == "RIGHT" then
		opri1 = '"LEFT"'; opri2 = '"RIGHT"'; csx = tonumber(desc.iconspx); csy = 0;
	elseif desc.orientation == "DOWN" then
		opri1 = '"TOP"'; opri2 = '"BOTTOM"'; osec1 = '"LEFT"'; osec2 = '"RIGHT"'; csx = 0; csy = -tonumber(desc.iconspy); csxm = tonumber(desc.iconspx); csym = 0;
	elseif desc.orientation == "UP" then
		opri1 = '"BOTTOM"'; opri2 = '"TOP"'; osec1 = '"LEFT"'; osec2 = '"RIGHT"'; csx = 0; csy = tonumber(desc.iconspy); csxm = tonumber(desc.iconspx); csym = 0;
	end
	if nRows == 1 then 
		-- Single-row code
		createCode = createCode .. [[frame.]] .. objname .. [[[i]:SetPoint(]] .. opri1 .. [[, frame.]] .. objname .. [[[i-1], ]] .. opri2 .. [[, ]] .. csx .. [[, ]] .. csy .. [[);]];
	else 
		-- Multi-row code
		createCode = createCode .. [[
if( VFL.mmod ( i + ]] .. nRows .. [[-1,]] .. nRows .. [[)  == 0 ) then 
    frame.]] .. objname .. [[[i]:SetPoint(]] .. osec1 .. [[, frame.]] .. objname .. "[i-" .. nRows .. [[], ]] .. osec2 .. [[, ]] .. csxm .. [[, ]] .. csym .. [[);
else 
    frame.]] .. objname .. [[[i]:SetPoint(]] .. opri1 .. [[, frame.]] .. objname .. "[i-1], " .. opri2 .. [[, ]] .. csx .. [[, ]] .. csy .. [[);
end
]];
	end
	createCode = createCode .. [[
end
]];
	return createCode;
end

local tbl_hvert = { {text = "HORIZONTAL"}}; --, {text = "VERTICAL"} };
local function hvert_gen() return tbl_hvert; end

RDX.RegisterFeature({
	name = "aura_bars2"; version = 1; title = i18n("Aura Bars 2"); category = i18n("Buffs/Debuffs");
	multiple = true;
	IsPossible = function(state)
		if not state:Slot("UnitFrame") then return nil; end
		if not state:Slot("Base") then return nil; end
		return true;
	end;
	ExposeFeature = function(desc, state, errs)
		local flg = true;
		if not desc then VFL.AddError(errs, i18n("No descriptor")); return nil; end
		if not __UFOwnerCheck(desc.owner, state, errs) then return nil; end
		if not desc.auraType then VFL.AddError(errs, i18n("Bad aura type")); return nil; end
		if desc.name and (not state:Slot("Frame_" .. desc.name)) then
			state:AddSlot("Frame_" .. desc.name);
		else
			flg = nil; VFL.AddError(errs, i18n("Missing or duplicate object name."));
		end
		return flg;
	end;
	ApplyFeature = function(desc, state)
		local objname = "Frame_" .. desc.name;
		local loadCode = "RDX.LoadBuffFromUnit";

		-- Event hinting.
		local mux, mask = state:GetContainingWindowState():GetSlotValue("Multiplexer"), 0;
		if desc.auraType == "DEBUFFS" then
			mask = mux:GetPaintMask("DEBUFFS");
			mux:Event_UnitMask("UNIT_DEBUFF_*", mask);
			loadCode = "RDX.LoadDebuffFromUnit";
		else
			mask = mux:GetPaintMask("BUFFS");
			mux:Event_UnitMask("UNIT_BUFF_*", mask);
		end
		mask = bit.bor(mask, 1);

		-- If there's an external filter, add a quick menu to the window to edit it.
		if desc.externalNameFilter then
			local path = desc.externalNameFilter; local afname = desc.name;
			state:GetContainingWindowState():Attach("Menu", true, function(win, mnu)
				table.insert(mnu, {
					text = i18n("Edit AuraFilter: ") .. afname;
					OnClick = function()
						VFL.poptree:Release();
						RDXDB.OpenObject(path, "Edit");
					end;
				});
			end);
		end

		------------ Closure
		local closureCode = [[
local ftctoto = FreeTimer.CreateFreeTimerClass(true,true, FreeTimer.Formula_CountDOWN, FreeTimer.TextFilter_MinSec, false, false, FreeTimer.SB_Hide, FreeTimer.Text_None, FreeTimer.TextInfo_None, FreeTimer.TexIcon_Hide, FreeTimer.SB_Hide, FreeTimer.Text_None, FreeTimer.TextInfo_None, FreeTimer.TexIcon_Hide);
]];
		if desc.filterName then
			closureCode = closureCode .. [[
local ]] .. objname .. [[_fnames = ]];
			if desc.externalNameFilter then
				closureCode = closureCode .. [[RDXDB.GetObjectInstance(]] .. string.format("%q", desc.externalNameFilter) .. [[);
]];
			else
				-- Internal filter
				closureCode = closureCode .. [[{};
]];
				for _,name in pairs(desc.filterNameList) do
					closureCode = closureCode .. objname .. "_fnames[" .. string.format("%q", name) .. "] = true; ";
				end
			end
		end
		
		state:Attach("EmitClosure", true, function(code) code:AppendCode(closureCode); end);

		----------------- Creation
		local createCode = _EmitCreateCode2(objname, desc);
		state:Attach("EmitCreate", true, function(code) code:AppendCode(createCode); end);

		------------------- Destruction
		local destroyCode = [[
local btn = nil;
for i=1,]] .. desc.nBars .. [[ do
	btn = frame.]] .. objname .. [[[i]
	btn.meta = nil;
	VFLUI.ReleaseRegion(btn.icontex); btn.icontex = nil;
	VFLUI.ReleaseRegion(btn.icontxt); btn.icontxt = nil;
	btn.sb:Destroy(); btn.sb = nil;
	VFLUI.ReleaseRegion(btn.sbtxt); btn.sbtxt = nil;
	VFLUI.ReleaseRegion(btn.sbtimetxt); btn.sbtimetxt = nil;
	btn.ftc:Destroy(); btn.ftc = nil;
	btn:Destroy();
end
frame.]] .. objname .. [[ = nil;
]];
		state:Attach("EmitDestroy", true, function(code) code:AppendCode(destroyCode); end);

		------------------- Paint
		local engselftimer = "false"; if desc.selftimer then engselftimer = "true"; end
		local engauratimer = "false"; if desc.auratimer then engauratimer = "true"; end
		local raidfilter = "nil"; if desc.raidfilter then raidfilter = "true"; end
		local myaurasfilter = "true"; if desc.myaurasfilter then myaurasfilter = "mycast"; end
		local timefilter = "true"; if desc.timefilter then timefilter = "(_dur and _dur > 0)"; end
		local namefilter = "true";
		if desc.filterName == "include" then
			namefilter = "(" .. objname .. "_fnames[_bn] or " .. objname .. "_fnames[_meta.category])";
		elseif desc.filterName == "exclude" then
			namefilter = "(not (" .. objname .. "_fnames[_bn] or " .. objname .. "_fnames[_meta.category]))"
		end
		local showTime = "nil"; if desc.text == "TIME" then showTime = "true"; end
		local showBorder = "false"; if desc.bkd.edgeSize then showBorder = "true"; end
		local usedebuffcolor = "true"; if (not desc.sbcolor) then usedebuffcolor = "false"; end
		local auranametrunc = "nil"; if desc.trunc then auranametrunc = desc.trunc; end
		local auranameab = "true"; if (not desc.abr) then auranameab = "false"; end

		local paintCode = [[
if band(paintmask, ]] .. mask .. [[) ~= 0 then
	local _i,_j,_bn,_tex,_apps,_meta,_dur,_tl,_dispelt,mycast,who=1,1,nil,nil,nil,nil,nil,nil,nil,nil;
	local _icons = frame.]] .. objname .. [[;
	while true do
		if (_j > ]] .. desc.nBars .. [[) then break; end
		_, _bn, _, _apps, _, _tex, _meta, _dur, _tl, _dispelt, mycast, who = ]] .. loadCode .. [[(uid, _i, ]] .. raidfilter .. [[, ]] .. engselftimer .. [[, ]] .. engauratimer .. [[);
		if not _meta then break; end
		if (not _meta.isInvisible) and ]] .. myaurasfilter .. [[ and ]] .. timefilter .. [[ and ]] .. namefilter .. [[ then
			__SetAuraBar2(_icons[_j], _meta, _tex, _apps, _dur, _tl, ]] .. showTime .. [[, _dispelt, ]] .. showBorder .. [[, "]] .. desc.auraType .. [[", ]] .. usedebuffcolor .. [[, ]] .. auranametrunc .. [[, ]] .. auranameab .. [[);
			_j = _j + 1;
		end
		_i = _i + 1;
	end
	while _j <= ]] .. desc.nBars .. [[ do
		_icons[_j]:Hide(); _j = _j + 1;
	end
end
]];
		state:Attach("EmitPaint", true, function(code) code:AppendCode(paintCode); end);
		------------------- Cleanup
		local cleanupCode = [[
local btn = nil;
for i=1,]] .. desc.nBars .. [[ do
	btn = frame.]] .. objname .. [[[i];
	btn:Hide(); btn.meta = nil;
	btn.sb:SetValue(0);
end
]];
		state:Attach("EmitCleanup", true, function(code) code:AppendCode(cleanupCode); end);


		return true;
	end;
	UIFromDescriptor = function(desc, parent, state)
		local ui = VFLUI.CompoundFrame:new(parent);

		------------- Core
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Core Parameters")));

		local ed_name = VFLUI.LabeledEdit:new(ui, 100); ed_name:Show();
		ed_name:SetText(i18n("Name"));
		ed_name.editBox:SetText(desc.name);
		ui:InsertFrame(ed_name);

		local er = RDXUI.EmbedRight(ui, i18n("Aura Type:"));
		local dd_auraType = VFLUI.Dropdown:new(er, _dd_types);
		dd_auraType:SetWidth(75); dd_auraType:Show();
		if desc and desc.auraType then 
			dd_auraType:SetSelection(desc.auraType); 
		else
			dd_auraType:SetSelection("BUFFS");
		end
		er:EmbedChild(dd_auraType); er:Show();
		ui:InsertFrame(er);

		------------- Layout
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Layout")));

		local owner = RDXUI.MakeSlotSelectorDropdown(ui, "Owner", state, "Subframe_", true);
		if desc and desc.owner then owner:SetSelection(desc.owner); end

		local anchor = RDXUI.UnitFrameAnchorSelector:new(ui); anchor:Show();
		anchor:SetAFArray(RDXUI.ComposeFrameList(state));
		if desc and desc.anchor then anchor:SetAnchorInfo(desc.anchor); end
		ui:InsertFrame(anchor);

		local ed_mb = VFLUI.LabeledEdit:new(ui, 50); ed_mb:Show();
		ed_mb:SetText(i18n("Max bars"));
		if desc and desc.nBars then ed_mb.editBox:SetText(desc.nBars); end
		ui:InsertFrame(ed_mb);

		local ed_rows = VFLUI.LabeledEdit:new(ui, 50); ed_rows:Show();
		ed_rows:SetText(i18n("Row size"));
		if desc and desc.rows then ed_rows.editBox:SetText(desc.rows); end
		ui:InsertFrame(ed_rows);

		local er = RDXUI.EmbedRight(ui, i18n("Orientation:"));
		local dd_orientation = VFLUI.Dropdown:new(er, _dd_orientations);
		dd_orientation:SetWidth(75); dd_orientation:Show();
		if desc and desc.orientation then 
			dd_orientation:SetSelection(desc.orientation); 
		else
			dd_orientation:SetSelection("RIGHT");
		end
		er:EmbedChild(dd_orientation); er:Show();
		ui:InsertFrame(er);
		
		local ed_iconspx = VFLUI.LabeledEdit:new(ui, 50); ed_iconspx:Show();
		ed_iconspx:SetText(i18n("Bars spacing width"));
		if desc and desc.iconspx then ed_iconspx.editBox:SetText(desc.iconspx); else ed_iconspx.editBox:SetText("0"); end
		ui:InsertFrame(ed_iconspx);
		
		local ed_iconspy = VFLUI.LabeledEdit:new(ui, 50); ed_iconspy:Show();
		ed_iconspy:SetText(i18n("Bars spacing height"));
		if desc and desc.iconspy then ed_iconspy.editBox:SetText(desc.iconspy); else ed_iconspy.editBox:SetText("0"); end
		ui:InsertFrame(ed_iconspy);
		
		local er_bor = RDXUI.EmbedRight(ui, i18n("Bar orientation:"));
		local dd_borientation = VFLUI.Dropdown:new(er_bor, hvert_gen);
		dd_borientation:SetWidth(100); dd_borientation:Show();
		if desc and desc.borientation then 
			dd_borientation:SetSelection(desc.borientation); 
		else
			dd_borientation:SetSelection("HORIZONTAL");
		end
		er_bor:EmbedChild(dd_borientation); er_bor:Show();
		ui:InsertFrame(er_bor);
		
		local ed_width = VFLUI.LabeledEdit:new(ui, 50); ed_width:Show();
		ed_width:SetText(i18n("Bar width"));
		if desc and desc.bwidth then ed_width.editBox:SetText(desc.bwidth); end
		ui:InsertFrame(ed_width);
		
		local ed_height = VFLUI.LabeledEdit:new(ui, 50); ed_height:Show();
		ed_height:SetText(i18n("Bar height"));
		if desc and desc.bheight then ed_height.editBox:SetText(desc.bheight); end
		ui:InsertFrame(ed_height);

		-------------- Display
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Display")));
		
		local er2 = RDXUI.EmbedRight(ui, i18n("Backdrop style"));
		local bkd = VFLUI.MakeBackdropSelectButton(er2, desc.bkd); bkd:Show();
		er2:EmbedChild(bkd); er2:Show();
		ui:InsertFrame(er2);
		
		local er_if = RDXUI.EmbedRight(ui, i18n("Icon Font Stack"));
		local iconfontsel = VFLUI.MakeFontSelectButton(er_if, desc.iconfont); iconfontsel:Show();
		er_if:EmbedChild(iconfontsel); er_if:Show();
		ui:InsertFrame(er_if);
		
		local er_btx = RDXUI.EmbedRight(ui, i18n("Bar Texture"));
		local sbtexsel = VFLUI.MakeTextureSelectButton(er_btx, desc.sbtexture); sbtexsel:Show();
		er_btx:EmbedChild(sbtexsel); er_btx:Show();
		ui:InsertFrame(er_btx);
		
		local er_bf = RDXUI.EmbedRight(ui, i18n("Bar Font Aura name"));
		local barfontsel = VFLUI.MakeFontSelectButton(er_bf, desc.sbfont); barfontsel:Show();
		er_bf:EmbedChild(barfontsel); er_bf:Show();
		ui:InsertFrame(er_bf);
		
		local ed_trunc = VFLUI.LabeledEdit:new(ui, 50); ed_trunc:Show();
		ed_trunc:SetText(i18n("Max aura length (blank = no truncation)"));
		if desc and desc.trunc then ed_trunc.editBox:SetText(desc.trunc); end
		ui:InsertFrame(ed_trunc);
		
		local chk_abr = VFLUI.Checkbox:new(ui); chk_abr:Show();
		chk_abr:SetText(i18n("Use abbreviating"));
		if desc and desc.abr then chk_abr:SetChecked(true); else chk_abr:SetChecked(); end
		ui:InsertFrame(chk_abr);
		
		local er_tf = RDXUI.EmbedRight(ui, i18n("Bar Font Aura Timer"));
		local timerfontsel = VFLUI.MakeFontSelectButton(er_tf, desc.sbtimerfont); timerfontsel:Show();
		er_tf:EmbedChild(timerfontsel); er_tf:Show();
		ui:InsertFrame(er_tf);
		
		local chk_bc = VFLUI.Checkbox:new(ui); chk_bc:Show();
		chk_bc:SetText(i18n("Use Bar color debuff"));
		if desc and desc.sbcolor then chk_bc:SetChecked(true); else chk_bc:SetChecked(); end
		ui:InsertFrame(chk_bc);
		
		local chk_ephemeral = VFLUI.Checkbox:new(ui); chk_ephemeral:Show();
		chk_ephemeral:SetText(i18n("No tooltips on mouseover / No drop on click"));
		if desc and desc.ephemeral then chk_ephemeral:SetChecked(true); else chk_ephemeral:SetChecked(); end
		ui:InsertFrame(chk_ephemeral);
		
		-------------- Timer
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Timer")));
		
		local eng_selftimer = VFLUI.Checkbox:new(ui); eng_selftimer:Show();
		eng_selftimer:SetText(i18n("Use GetPlayerBuffName"));
		if desc and desc.selftimer then eng_selftimer:SetChecked(true); else eng_selftimer:SetChecked(); end
		ui:InsertFrame(eng_selftimer);
		
		local eng_auratimer = VFLUI.Checkbox:new(ui); eng_auratimer:Show();
		eng_auratimer:SetText(i18n("Use Aura Timer Engine"));
		if desc and desc.auratimer then eng_auratimer:SetChecked(true); else eng_auratimer:SetChecked(); end
		ui:InsertFrame(eng_auratimer);

		------------ Filter
		ui:InsertFrame(VFLUI.Separator:new(ui, i18n("Filtering")));

		local chk_raidfilter = VFLUI.Checkbox:new(ui); chk_raidfilter:Show();
		chk_raidfilter:SetText(i18n("Use Blizzard raid filter"));
		if desc and desc.raidfilter then chk_raidfilter:SetChecked(true); else chk_raidfilter:SetChecked(); end
		ui:InsertFrame(chk_raidfilter);
		
		local chk_myauras = VFLUI.Checkbox:new(ui); chk_myauras:Show();
		chk_myauras:SetText(i18n("Show only my auras cast"));
		if desc and desc.myaurasfilter then chk_myauras:SetChecked(true); else chk_myauras:SetChecked(); end
		ui:InsertFrame(chk_myauras);

		local chk_timefilter = VFLUI.Checkbox:new(ui); chk_timefilter:Show();
		chk_timefilter:SetText(i18n("Show only auras for which timers are known"));
		if desc and desc.timefilter then chk_timefilter:SetChecked(true); else chk_timefilter:SetChecked(); end
		ui:InsertFrame(chk_timefilter);

		local chk_filterName = VFLUI.Checkbox:new(ui); chk_filterName:Show();
		chk_filterName:SetText(i18n("Filter by aura name"));
		if desc and desc.filterName then chk_filterName:SetChecked(true); else chk_filterName:SetChecked(); end
		ui:InsertFrame(chk_filterName);

		local rg_ie = VFLUI.RadioGroup:new(ui); rg_ie:Show();
		rg_ie:SetLayout(2,2);
		rg_ie.buttons[1]:SetText(i18n("Include")); rg_ie.buttons[2]:SetText(i18n("Exclude"));
		if desc and desc.filterName == "include" then
			rg_ie:SetValue(1);
		elseif desc and desc.filterName == "exclude" then
			rg_ie:SetValue(2);
		end
		ui:InsertFrame(rg_ie);

		local chk_external = RDXUI.CheckEmbedRight(ui, i18n("Use external aura list"));
		local file_external = RDXDB.ObjectFinder:new(chk_external, function(p,f,md) return (md and md.ty and string.find(md.ty, "AuraFilter$")); end);
		file_external:SetWidth(200); file_external:Show();
		chk_external:EmbedChild(file_external); chk_external:Show();
		ui:InsertFrame(chk_external);
		if desc.externalNameFilter then
			chk_external:SetChecked(true); file_external:SetPath(desc.externalNameFilter);
		else
			chk_external:SetChecked();
		end

		local le_names = VFLUI.ListEditor:new(ui, desc.filterNameList or {}, function(cell,data) cell.text:SetText(data); end);
		le_names:SetHeight(183); le_names:Show();
		ui:InsertFrame(le_names);
		
		function ui:GetDescriptor()
			local trunc = tonumber(ed_trunc.editBox:GetText());
			if trunc then trunc = VFL.clamp(trunc, 1, 50); end
			local filterName, filterNameList, ext = nil, nil, nil;
			if chk_filterName:GetChecked() then
				filterName = "include";
				if rg_ie:GetValue() == 2 then filterName = "exclude"; end
				filterNameList = le_names:GetList();
				for k,v in pairs(filterNameList) do filterNameList[k] = string.lower(v); end
				if chk_external:GetChecked() then	ext = file_external:GetPath(); end
			end
			return { 
				feature = "aura_bars2"; version = 1;
				name = ed_name.editBox:GetText();
				auraType = dd_auraType:GetSelection();
				owner = owner:GetSelection();
				anchor = anchor:GetAnchorInfo();
				nBars = VFL.clamp(ed_mb.editBox:GetNumber(), 1, 40);
				rows = VFL.clamp(ed_rows.editBox:GetNumber(), 1, 40);
				orientation = dd_orientation:GetSelection();
				iconspx = VFL.clamp(ed_iconspx.editBox:GetNumber(), 0, 200);
				iconspy = VFL.clamp(ed_iconspy.editBox:GetNumber(), 0, 200);
				borientation = dd_borientation:GetSelection();
				bwidth = VFL.clamp(ed_width.editBox:GetNumber(), 1, 1000);
				bheight = VFL.clamp(ed_height.editBox:GetNumber(), 1, 200);
				-- Display
				bkd = bkd:GetSelectedBackdrop();
				iconfont = iconfontsel:GetSelectedFont();
				sbtexture = sbtexsel:GetSelectedTexture();
				sbfont = barfontsel:GetSelectedFont();
				trunc = trunc;
				abr = chk_abr:GetChecked();
				sbtimerfont = timerfontsel:GetSelectedFont();
				sbcolor = chk_bc:GetChecked();
				ephemeral = chk_ephemeral:GetChecked();
				-- Timer
				selftimer = eng_selftimer:GetChecked();
				auratimer = eng_auratimer:GetChecked();
				-- Filter
				raidfilter = chk_raidfilter:GetChecked();
				myaurasfilter = chk_myauras:GetChecked();
				timefilter = chk_timefilter:GetChecked();
				filterName = filterName;
				externalNameFilter = ext;
				filterNameList = filterNameList;
			};
		end

		return ui;
	end;
	CreateDescriptor = function()
		return { 
			feature = "aura_bars2"; version = 1;
			name = "ab1"; auraType = "BUFFS";
			owner = "Base";
			anchor = { lp = "LEFT", af = "Base", rp = "RIGHT", dx = 4, dy = 0};
			nBars = 4; rows = 1; orientation = "RIGHT"; iconspx = 0; iconspy = 0; 
			borientation = "HORIZONTAL"; bwidth = 90; bheight = 14;
			bkd = VFL.copy(VFLUI.defaultBackdrop);
			iconfont = VFL.copy(Fonts.Default);
			sbtexture = VFL.copy(VFLUI.defaultTexture);
			sbfont = VFL.copy(Fonts.Default);
			sbtimerfont = VFL.copy(Fonts.Default);
		};
	end;
});

