-- UDB.lua
-- RDX
-- (C)2006 Bill Johnson
--
-- The master unit database.
--
-- The primary responsibilities of the Unit Database are:
-- * Enforce the party/raid abstraction.
-- * Provide events for when people are added/removed from the group.
-- * Map units to ndata/edata.
-- * Promote WoW events to RDX events, intelligently up-converting unit references.
--

-- Imports
local tempty, strlower, strmatch, strgsub = VFL.empty, string.lower, string.match, string.gsub;

VFLP.RegisterCategory(i18n("RDX: UnitDB"));

------------------------------------------------------------------------------
-- Performance enhancements - precache commonly used signals
------------------------------------------------------------------------------
local _sig_rdx_debuff_star = RDXEvents:LockSignal("UNIT_DEBUFF_*");
local _sig_rdx_buff_star = RDXEvents:LockSignal("UNIT_BUFF_*");
local _sig_rdx_mybuff_star = RDXEvents:LockSignal("UNIT_MYBUFF_*");
local _sig_rdx_unit_health = RDXEvents:LockSignal("UNIT_HEALTH");
local _sig_rdx_unit_mana = RDXEvents:LockSignal("UNIT_MANA");
local _sig_rdx_unit_aura = RDXEvents:LockSignal("UNIT_AURA");
local _sig_rdx_unit_target = RDXEvents:LockSignal("UNIT_TARGET");
local _sig_rdx_unit_focus = RDXEvents:LockSignal("UNIT_FOCUS");
local _sig_rdx_unit_flags = RDXEvents:LockSignal("UNIT_FLAGS");
local _sig_rdx_unit_range = RDXEvents:LockSignal("UNIT_RANGED");
local _sig_rdx_unit_portrait_update = RDXEvents:LockSignal("UNIT_PORTRAIT_UPDATE");
local _sig_rdx_unit_faction = RDXEvents:LockSignal("UNIT_FACTION");

local _sig_rdx_unit_entering_world = RDXEvents:LockSignal("UNIT_ENTERING_WORLD");
local _sig_rdx_unit_totem_update = RDXEvents:LockSignal("UNIT_TOTEM_UPDATE");

local _sig_rdx_unit_buffweapon_update = RDXEvents:LockSignal("UNIT_BUFFWEAPON_UPDATE");

------------------------------------------------------------------------------
-- UNIT ID MAPPINGS/UNIT COUNTING
------------------------------------------------------------------------------
RDX.NUM_UNITS = 80; -- Number of internal unit structures.
local NUM_UNITS = RDX.NUM_UNITS;

----------- Party unit maps
-- player, party1..party4, pet, partypet1..partypet4
local party_id2num = {};
party_id2num["player"] = 1;
for i=1,4 do party_id2num["party" .. i] = i + 1; end
party_id2num["pet"] = 41;
for i=1,4 do party_id2num["partypet" .. i] = i + 41; end

local party_num2id = {};
party_num2id[1] = "player";
for i=1,4 do party_num2id[i+1] = "party" .. i; end
for i=6,40 do party_num2id[i] = false; end
party_num2id[41] = "pet";
for i=1,4 do party_num2id[41+i] = "partypet" .. i; end

----------- Raid unit maps
-- raid1..raid40, raidpet1..raidpet40
local raid_id2num = {};
for i=1,40 do	raid_id2num["raid" .. i] = i; end
for i=1,40 do raid_id2num["raidpet" .. i] = 40 + i; end

local raid_num2id = {};
for i=1,40 do	raid_num2id[i] = "raid" .. i; end
for i=1,40 do raid_num2id[40 + i] = "raidpet" .. i; end

local id2num, num2id = {}, {};

local function SetRaidIDDatabase()
	id2num = raid_id2num; num2id = raid_num2id;
end

local function SetPartyIDDatabase()
	id2num = party_id2num; num2id = party_num2id;
end

function RDX.GetNumberToUIDMap()
	return num2id;
end

function RDX.GetUIDToNumberMap()
	return id2num;
end

function RDX.NumberToUID(unum)
	return num2id[unum];
end

function RDX.UIDToNumber(uid)
	return id2num[uid] or 0;
end

-- Internal: raid or party status and member count
local function party_GetNumUnits()
	return GetNumPartyMembers() + 1;
end

local function raid_GetNumUnits()
	return GetNumRaidMembers();
end

--- @return The total number of units in the RDX unit database.
RDX.GetNumUnits = party_GetNumUnits;

local isRaid, isSolo = false, true;

--- Return TRUE iff we are currently in a raid group, nil otherwise.
function RDX.InRaid()	return isRaid; end
--- Return TRUE iff we are solo, nil otherwise.
function RDX.IsSolo() return isSolo; end
local IsSolo = RDX.IsSolo;

-------------------------------------------------------------------
-- UNIT AURA METADATA
-------------------------------------------------------------------
-- Local override for debuff categorization.
-- Example: debuffCategoryOverride[i18n("arcane blast")] = "@other";
local debuffCategoryOverride = {};

-- The aura metadata caches.
local function GenMetadataCacheFuncs(ncache)
	local Set = function(texture, name, category, properName, properCategory, descr)
		if not name then return; end
		local x = { name = name, texture = texture, properName = properName, category = category, properCategory = properCategory, descr = descr };
		ncache[name] = x;
		return x;
	end

	local NGet = function(name) 
		return ncache[name]; 
	end

	return Set, NGet;
end

local buffCacheN = {};
local debuffCacheN = {};

local buffcache_set, buffcache_nget = GenMetadataCacheFuncs(buffCacheN);
local debuffcache_set, debuffcache_nget = GenMetadataCacheFuncs(debuffCacheN);

-- store debuff duration (GetPlayerBuff) Player
local debuffCacheDuration = {};

local function Getdbcd(bbname, ttimeleft)
	local tduration = 0;
	if debuffCacheDuration[bbname] then
		if ttimeleft > debuffCacheDuration[bbname] then
			debuffCacheDuration[bbname] = ttimeleft;
			tduration = ttimeleft;
		else
			tduration = debuffCacheDuration[bbname];
		end
	else
		debuffCacheDuration[bbname] = ttimeleft;
		tduration = ttimeleft;
	end
	return tduration;
end

-- store buff duration (GetPlayerBuff) Player
local buffCacheDuration = {};

local function Getbcd(bbname, ttimeleft)
	local tduration = 0;
	if buffCacheDuration[bbname] then
		if ttimeleft > buffCacheDuration[bbname] then
			buffCacheDuration[bbname] = ttimeleft;
			tduration = ttimeleft;
		else
			tduration = buffCacheDuration[bbname];
		end
	else
		buffCacheDuration[bbname] = ttimeleft;
		tduration = ttimeleft;
	end
	return tduration;
end

-- INTERNAL: Get information about a debuff from a unit.
local function LoadDebuffFromUnit(uid, i, raidFilter, gpdb, atedb)
	local startime, mycast, who = nil, false, "unknown";
	-- Verify debuff info; get number of applications.
	local dn, _, tex, apps, dispelType, duration, timeLeft = UnitDebuff(uid, i, raidFilter);
	if (not dn) then return nil; end
	local ldn = strlower(dn);
	if duration and timeLeft then mycast = true; who = strlower(UnitName("player")); end
	
	-- found duration and timeleft using GetPlayerBuffName
	if gpdb and (uid == "player") and (not timeLeft) then
		local i = 1;
		local debuffIndex, bname = nil,nil;
		while true do
			debuffIndex = GetPlayerBuff(i, "HARMFUL");
			if debuffIndex == 0 then break; end
			bname = GetPlayerBuffName(debuffIndex);
			if dn == bname then
				tmptimeleft = GetPlayerBuffTimeLeft(debuffIndex);
				if tmptimeleft == 0 then break; end
					timeLeft = tmptimeleft;
					duration = Getdbcd(bname, tmptimeleft);
				break;
			end
			i = i +1 ;
		end
	end
	
	-- ATE
	if atedb and RDXATE.listDebuffs[ldn] and (not timeLeft) then
		local unit = RDX._FastProject(uid);
		if unit then
			_, duration, timeLeft = RDXATE.GetDebuffTimerbyStartime(unit:GetDebuffstartime(ldn), ldn);
		else
			_, duration, timeLeft, who = OmniDB.GetGUIDDebuff(UnitGUID(uid), ldn);
		end
		
	end
	
	if (not apps) or (apps < 1) then apps = 1; end
	
	-- Query cache; early out if we already have the infos.
	local info = debuffcache_nget(ldn);
	if info then
		return true, ldn, info.category, apps, nil, tex, info, duration, timeLeft, dispelType, mycast, who;
	end
	-- Munge category
	local category = dispelType;
	if debuffCategoryOverride[ldn] then
		category = debuffCategoryOverride[ldn];
	elseif category then
		category = "@" .. strlower(category);
	else
		category = "@other";
	end
	-- Stuff tooltip
	VFLTipTextLeft1:SetText(nil); VFLTipTextRight1:SetText(nil);
	VFLTipTextLeft2:SetText(nil);
	VFLTip:SetUnitDebuff(uid, i);
	-- Add to cache
	info = debuffcache_set(tex, ldn, category, dn, VFLTipTextRight1:GetText(), VFLTipTextLeft2:GetText());
	return true, ldn, category, apps, nil, tex, info, duration, timeLeft, nil, mycast, who;
end
RDX.LoadDebuffFromUnit = LoadDebuffFromUnit;
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "LoadDebuffFromUnit", LoadDebuffFromUnit, true);

-- INTERNAL: Get information about a buff from a unit.
local function LoadBuffFromUnit(uid, i, raidFilter, gpdb, atedb)
	local startime, mycast, who = nil, false, "unknown";
	-- Verify buff info; get number of applications.
	local dn, _, tex, apps, duration, timeLeft = UnitBuff(uid, i, raidFilter);
	if (not dn) then return nil; end
	local ldn = strlower(dn);
	if duration and timeLeft then mycast = true; who = strlower(UnitName("player")); end
	--duration = nil; timeLeft = nil;
	
	-- found duration and timeleft using GetPlayerBuffName
	if gpdb and (uid == "player") and (not timeLeft) then
		local i = 1;
		local buffIndex, bname = nil,nil;
		while true do
			buffIndex = GetPlayerBuff(i, "HELPFUL");
			if buffIndex == 0 then break; end
			bname = GetPlayerBuffName(buffIndex);
			if dn == bname then
				tmptimeleft = GetPlayerBuffTimeLeft(buffIndex);
				if tmptimeleft == 0 then break; end
					timeLeft = tmptimeleft;
					duration = Getbcd(bname, tmptimeleft);
				break;
			end
			i = i +1 ;
		end
	end
	
	-- ATE
	if atedb and RDXATE.listBuffs[ldn] and (not timeLeft) then
		local unit = RDX._FastProject(uid);
		if unit then
			_, duration, timeLeft = RDXATE.GetBuffTimerbyStartime(unit:GetBuffstartime(ldn), ldn);
		else
			_, duration, timeLeft, who = OmniDB.GetGUIDBuff(UnitGUID(uid), ldn);
		end
	end
	
	if (not apps) or (apps < 1) then apps = 1; end
	
	-- Attempt to get buff data from the cache
	local info = buffcache_nget(ldn);
	if info then 
		return true, ldn, info.category, apps, nil, tex, info, duration, timeLeft, nil, mycast, who; 
	else
		-- Stuff tooltip
		VFLTipTextLeft1:SetText(nil); VFLTipTextRight1:SetText(nil);
		VFLTipTextLeft2:SetText(nil);
		VFLTip:SetUnitBuff(uid, i);
		-- Write to cache and return
		info = buffcache_set(tex, ldn, "@other", dn, nil, VFLTipTextLeft2:GetText());
		return true, ldn, "@other", apps, nil, tex, info, duration, timeLeft, nil, mycast, who;
	end
end
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "LoadBuffFromUnit", LoadBuffFromUnit, true);

RDX.LoadBuffFromUnit = LoadBuffFromUnit;

-- INTERNAL: Get information about weapons buff.
-- "MainHandSlot"
-- "SecondaryHandSlot"
local function scanHand(hand)
	VFLTip:SetOwner(UIParent, "ANCHOR_NONE");
	VFLTip:ClearLines();
	local idslot = GetInventorySlotInfo(hand);
	VFLTip:SetInventoryItem("player", idslot);
	local mytext, strfound = nil, nil;
	local buffname, buffrank, bufftex;
	for i = 1, VFLTip:NumLines() do
		mytext = getglobal("VFLTipTextLeft" .. i);
		strfound = strmatch(mytext:GetText(), "^(.*) %(%d+ [^%)]+%)$");
		if strfound then break; end
	end
	if strfound then
		strfound = strgsub(strfound, " %(%d+ [^%)]+%)", "");
		buffname, buffrank = strmatch(strfound, "(.*) (%d*)$");
		if not buffname then
			buffname, buffrank = strmatch(strfound, "(.*) ([IVXLMCD]*)$");
		end
		if not buffname then
			buffname, buffrank = strmatch(strfound, "(.*)(%d)");
			-- specific fucking french language langue de feu
			if buffname then
				local a = string.len(buffname);
				buffname = string.sub(buffname, 1, a - 2);
			else 
				buffname = strfound;
			end
			--if buffname then VFL.print(buffname); VFL.print(a); end
		end
		if not buffname then
			buffname = "unknown parse";
		end
		bufftex = GetInventoryItemTexture("player", idslot);
	end
	VFLTip:Hide();
	return buffname, buffrank, bufftex;
end;

local function LoadWeaponsBuff()
	local hasMainHandEnchant, mainHandExpiration, mainHandCharges, hasOffHandEnchant, offHandExpiration, offHandCharges = GetWeaponEnchantInfo();
	if (not hasMainHandEnchant) and (not hasOffHandEnchant) then return nil; end
	local mainHandBuffName, mainHandBuffRank, mainHandBuffStart, mainHandBuffDur, mainHandTex, mainHandBuffTex, offHandBuffName, offHandBuffRank, offHandBuffStart, offHandBuffDur, offHandTex, offHandBuffTex;
	mainHandBuffStart = 0 ;
	if hasMainHandEnchant then
		mainHandBuffName, mainHandBuffRank, mainHandTex = scanHand("MainHandSlot");
		mainHandBuffDur, mainHandBuffTex = Logistics.getBuffWeaponInfo(mainHandBuffName);
		if mainHandBuffDur > 0 then
			mainHandBuffStart = GetTime() - (mainHandBuffDur - mainHandExpiration / 1000);
		else mainHandBuffStart = 0; end
	end
	if hasOffHandEnchant then
		offHandBuffName, offHandBuffRank, offHandTex = scanHand("SecondaryHandSlot");
		offHandBuffDur, offHandBuffTex = Logistics.getBuffWeaponInfo(offHandBuffName);
		if offHandBuffDur > 0 then
			offHandBuffStart = GetTime() - (offHandBuffDur - offHandExpiration / 1000);
		else offHandBuffStart = 0; end
	end
	return hasMainHandEnchant, mainHandBuffName, mainHandBuffRank, mainHandCharges, mainHandBuffStart, mainHandBuffDur, mainHandTex, mainHandBuffTex, hasOffHandEnchant, offHandBuffName, offHandBuffRank, offHandCharges, offHandBuffStart, offHandBuffDur, offHandTex, offHandBuffTex;
end
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "LoadWeaponsBuff", LoadWeaponsBuff, true);

RDX.LoadWeaponsBuff = LoadWeaponsBuff;

-- Debuff categories
RDXEvents:Bind("INIT_PRELOAD", nil, function()
	debuffCacheN["@curse"] = {	
		name = "@curse",
		texture = "Interface\\InventoryItems\\WoWUnknownItem01.blp",
		properName = i18n("(Any Curse)"),
		descr = i18n("Matches any curse."),
		set = RDX.GetDebuffSet("@curse"), isInvisible = true,
	};

	debuffCacheN["@magic"] = {
		name = "@magic",
		texture = "Interface\\InventoryItems\\WoWUnknownItem01.blp",
		properName = i18n("(Any Magic)"),
		descr = i18n("Matches any magic debuff."),
		set = RDX.GetDebuffSet("@magic"), isInvisible = true,
	};

	debuffCacheN["@poison"] = {
		name = "@poison",
		texture = "Interface\\InventoryItems\\WoWUnknownItem01.blp",
		properName = i18n("(Any Poison)"),
		descr = i18n("Matches any poison debuff."),
		set = RDX.GetDebuffSet("@poison"), isInvisible = true,
	};
	
	debuffCacheN["@disease"] = {
		name = "@disease",
		texture = "Interface\\InventoryItems\\WoWUnknownItem01.blp",
		properName = i18n("(Any Disease)"),
		descr = i18n("Matches any disease debuff."),
		set = RDX.GetDebuffSet("@disease"), isInvisible = true,
	};

	debuffCacheN["@other"] = {
		name = "@other",
		texture = "Interface\\InventoryItems\\WoWUnknownItem01.blp",
		properName = i18n("(Any Other)"),
		descr = i18n("Matches any debuff that is not disease, poison, magic, or curse."),
		set = RDX.GetDebuffSet("@other"), isInvisible = true,
	};
end);


------------- Metadata API --------------

--- @return A table containing information on the buff with the given texture, if seen this session.
-- Nil otherwise.
RDX.GetBuffInfoByName = buffcache_nget;
RDX.GetDebuffInfoByName = debuffcache_nget;

function RDX._GetBuffCache()
	return buffCacheN;
end

function RDX._GetDebuffCache()
	return debuffCacheN;
end

--- Reproduce an aura tooltip from an entry in an aura cache.
function RDX.ShowAuraTooltip(meta, frame, anchor)
	GameTooltip:SetOwner(frame, "ANCHOR_NONE");
	GameTooltip:SetPoint("TOPLEFT", frame, anchor);
	GameTooltip:ClearLines();
	GameTooltip:AddDoubleLine(meta.properName, meta.properCategory);
	GameTooltip:AddLine(meta.descr, 1, 1, 1, 1, true);
	GameTooltip:Show();
end

--- Determine if a unit has the given buff.
function RDX.UnitHasBuffByName(uid, buff)
	local i=1;
	while true do
		local name = UnitBuff(uid, i); if not name then break; end
		if strlower(name) == buff then return true; end
		i=i+1;
	end
end

--- Determine if a unit has the given debuff.
function RDX.UnitHasDebuffByName(uid, debuff)
	local i=1;
	while true do
		local name = UnitDebuff(uid, i); if not name then break; end
		if strlower(name) == debuff then return true; end
		i=i+1;
	end
end

-------------------------------------------------------------------
-- EDATA
-- Edata = engine data = data that follows a unit around by ID number.
-------------------------------------------------------------------

-- Temporary touched matrix for edata
local _touched = {};

-- Internal: Create a new engine data structure for the given unit.
local function NewEData(idx)
	local self = {};
	
	-- Custom fields
	local _efields = {};
	-- Buff/debuff lists
	local _buffs, _debuffs = {}, {};
	-- Only store mybuffs base on timer
	local _mybuffs = {};
	
	-- Store lefttimer for Buff/Debuff, use to fire new event
	local _buffslti, _debuffslti = {}, {};
	
	-- Store startime for Buff/Debuff, Use by Aura Timer Engine
	local _buffssti, _debuffssti = {}, {};
	
	-- Store last spell and rank from UNIT_SPELLCAST_SENT
	local _spellrank = {};
	
	-- Fixed fields
	local group = 0;

	self.SetGroup = function(x, g) group = g; end
	self.GetGroup = function() return group; end

	self.IsPet = VFL.Noop;
	if(idx > 0) and (idx <= 40) then
		self.IsPet = VFL.Nil;
	elseif(idx > 40) and (idx <= 80) then
		self.IsPet = VFL.True;
	end

	self.HasDebuff = function(x, debuff) return _debuffs[debuff]; end
 	self.Debuffs = function() return _debuffs; end

	self.HasBuff = function(x, buff) return _buffs[buff]; end
	self.Buffs = function() return _buffs; end
	
	self.HasMyBuff = function(x, mybuff) return _mybuffs[mybuff]; end
	self.MyBuffs = function() return _mybuffs; end
	
	self.GetDebuffstartime = function(x, buff) return _debuffssti[buff]; end
	self.Debuffsstartime = function() return _debuffssti; end
	
	self.GetBuffstartime = function(x, buff) return _buffssti[buff]; end
	self.Buffsstartime = function() return _buffssti; end
	
	-- Process auras
	self.ProcessAuras = function(rdxunit)
		local uid = num2id[idx]; tempty(_touched);
		local i, cont, name, category, apps, dur, tLeft, mycast, who = 0, nil, nil, nil, 0, nil, nil, nil, nil;
		
		RDX.BeginEventBatch();

		------------------ DEBUFFS
		-- Iterate over all debuffs
		i=1;
		while true do
			-- Get debuff name and category from cache, loading manually if
			-- not cached.
			cont, name, category, apps, _, _, _, dur, tleft, _, mycast, who = LoadDebuffFromUnit(uid, i, nil, true, true);
			if not cont then break; end
			-- Check local debuff info for given debuff name; update if necessary.
			if name then
				_touched[name] = true;
				--local ATEduration = RDXATE.listDebuffs[name];
				if (_debuffs[name] ~= apps) then
					_debuffs[name] = apps;
					_debuffssti[name] = GetTime();
					RDXEvents:Dispatch("UNIT_DEBUFF_" .. name, rdxunit, name, apps);
					_sig_rdx_debuff_star:Raise(rdxunit, name, apps);
					-- ATE for raid units
					--if mycast and ATEduration and RDXATE.sync then 
					--	OmniDB.SetGUIDDebuff(UnitGUID(uid), strlower(UnitName(uid)), name, dur, tleft, who, true);
					--end
				end
				
				-- base on time, if tleft is superior, fire event
				if tleft then
					if not _debuffslti[name] then _debuffslti[name] = (tleft - 0.15); break; end
					if (_debuffslti[name] > tleft) then
						_debuffslti[name] = tleft;
					else
						-- request from Xenios to disable
						_debuffssti[name] = GetTime();
						_sig_rdx_debuff_star:Raise(rdxunit, name, apps);
						-- ATE for raid units
						--if mycast and ATEduration and RDXATE.sync then 
						--	OmniDB.SetGUIDDebuff(UnitGUID(uid), strlower(UnitName(uid)), name, dur, tleft, who, true);
						--end
						_debuffslti[name] = (tleft - 0.15);
					end
				end
			end -- if name
			-- Mark category info
			if category then
				_touched[category] = true;
				if _debuffs[category] ~= 1 then
					_debuffs[category] = 1;
					RDXEvents:Dispatch("UNIT_DEBUFF_" .. category, rdxunit, name, 1);
					_sig_rdx_debuff_star:Raise(rdxunit, name, 1);
				end
			end -- if category
			-- Move on to next debuff
			i = i + 1;
		end

		-- Any debuff not touched has been removed; update as appropriate
		for k,v in pairs(_debuffs) do
			if not _touched[k] then
				_debuffs[k] = nil;
				_debuffslti[k] = nil;
				_debuffssti[k] = nil;
				RDXEvents:Dispatch("UNIT_DEBUFF_" .. k, rdxunit, k, 0);
				_sig_rdx_debuff_star:Raise(rdxunit, k, 0);
			end
		end

		------------------------- BUFFS
		tempty(_touched);
		cont, name, category, apps, dur, tLeft, mycast, who = nil, nil, nil, 0, nil, nil, nil, nil;
		
		-- Iterate over buffs
		i = 1;
		while true do
			cont, name, category, apps, _, _, _, dur, tleft, _, mycast, who = LoadBuffFromUnit(uid, i, nil, true, true);
			if not cont then break; end
			if name then
				_touched[name] = true;
				local ATEduration = RDXATE.listBuffs[name];
				if (_buffs[name] ~= apps) then
					_buffs[name] = apps;
					_buffssti[name] = GetTime();
					RDXEvents:Dispatch("UNIT_BUFF_" .. name, rdxunit, name, apps);
					if mycast then
						_mybuffs[name] = apps;
						RDXEvents:Dispatch("UNIT_MYBUFF_" .. name, rdxunit, name, apps);
					end
					_sig_rdx_buff_star:Raise(rdxunit, name, apps);
					
					-- ATE for raid units
					--if mycast and ATEduration and RDXATE.sync then
					--	OmniDB.SetGUIDBuff(UnitGUID(uid), strlower(UnitName(uid)), name, dur, tleft, who, true);
					--end
				end
				if tleft then
					if not _buffslti[name] then _buffslti[name] = (tleft - 0.15); break; end
					if (_buffslti[name] > tleft) then
						_buffslti[name] = tleft;
					else
						_buffssti[name] = GetTime();
						_sig_rdx_buff_star:Raise(rdxunit, name, apps);
						-- ATE for raid units
						--if mycast and ATEduration and RDXATE.sync then
						--	OmniDB.SetGUIDBuff(UnitGUID(uid), strlower(UnitName(uid)), name, dur, tleft, who, true);
						--end
						_buffslti[name] = (tleft - 0.15);
					end
				end
			end
			i = i + 1;
		end

		-- Any buff not touched was removed
		for k,v in pairs(_buffs) do
			if not _touched[k] then
				_buffs[k] = nil;
				_buffslti[k] = nil;
				_buffssti[k] = nil;
				RDXEvents:Dispatch("UNIT_BUFF_" .. k, rdxunit, k, 0);
				_sig_rdx_buff_star:Raise(rdxunit, k, 0);
			end
		end
		for k,v in pairs(_mybuffs) do
			if not _touched[k] then
				_mybuffs[k] = nil;
				RDXEvents:Dispatch("UNIT_MYBUFF_" .. k, rdxunit, k, 0);
			end
		end
		
		RDX.EndEventBatch();
	end
	
	self.GetLastSpellRank = function(a) return _spellrank[a]; end
	self.SetLastSpellRank = function(a, r) _spellrank[a] = r; end
	
	self.SetEField = function(x, f, v) _efields[f] = v; end
	self.GetEField = function(x, f) return _efields[f]; end
	self.EFields = function() return _efields; end

	RDXEvents:Dispatch("EDATA_CREATED", self, idx);

	return self;
end

-- Master edata table
local edata = {};

local function GetEData(i)
	return edata[i];
end

-- Edata[0] = empty, do-nothing edata.
local ed0 = NewEData(0);
ed0.ProcessAuras = VFL.Noop;
ed0.SetEField = VFL.Noop;
ed0.GetEField = VFL.Nil;
ed0.IsPet = VFL.False;
ed0.SetGroup = VFL.Noop;
ed0.GetGroup = VFL.Zero;
edata[0] = ed0;

-------------------------------------------------------------------
-- NDATA
-- Ndata = nominative data = data that follows a unit around by name.
-------------------------------------------------------------------
local function NewNData(name)
	local self = {};

	local _nfields = {};
	local leader, class = 0, 0;
	local feigned = nil;

	self.SetLeaderLevel = function(x, lv) leader = lv; end
	self.GetLeaderLevel = function(x) return leader; end
	self.IsLeader = function() return (leader > 0); end
	self.SetClassID = function(x, cn) class = cn; end
	self.GetClassID = function() return class; end
	self._SetFeigned = function(x, flg) feigned = flg; end
	self.IsFeigned = function() return feigned; end
	
	self.SetNField = function(x, f, v) _nfields[f] = v; end
	self.GetNField = function(x, f) return _nfields[f]; end
	self.NFields = function() return _nfields; end

	RDXEvents:Dispatch("NDATA_CREATED", self, name);

	return self;
end

-- Master ndata table
local ndata = {};

local function GetNData(name)
	local r = ndata[name];
	if not r then
		r = NewNData(name);
		ndata[name] = r;
	end
	return r;
end

-----------------------------------------------------------------
-- UNIT DATABASES
-----------------------------------------------------------------
-- UBI: Units by index
local ubi = {};
-- UBN: Units by name
local ubn = {};

local function ubn_to_ubi(x)
	return RDX.GetUnitByNumber(x.nid);
end
local function ubi_to_ubn(x)
	return RDX.GetUnitByName(x.name);
end

--- Get a reference to a unit by its unit number.
-- @param i The unit number to query.
-- @return A reference to the unit object with unit number i.
function RDX.GetUnitByNumber(i)
	if not i then return nil; end
	return ubi[i];
end

--- Get a reference to a unit by its name.
-- @param n The name (all lowercase) of the unit to query.
-- @return A reference to the unit object for the unit named n.
function RDX.GetUnitByName(n)
	if not n then return nil; end
	local r = ubn[n];
	if not r then
		r = RDX.Unit:new();	r.name = n; r:Invalidate();
		-- Apply nominative/indexed unit funcs
		r.GetNominativeUnit = VFL.Identity;
		r.IsNominativeUnit = VFL.True;
		r.GetIndexedUnit = ubn_to_ubi;
		r.IsIndexedUnit = VFL.Nil;
		-- BUGFIX: Defer this until after init.
		if RDX.initialized then VFL.mixin(r, GetNData(n), true); end
		ubn[n] = r;
	end
	return r;
end

-------------- Initial unit creation
-- Create the initial units by index
for i=1,NUM_UNITS do 
	local qq = RDX.Unit:new(); qq.nid = i; qq:Invalidate();
	-- Apply unit query functionality
	qq.GetNominativeUnit = ubi_to_ubn;
	qq.IsNominativeUnit = VFL.Nil;
	qq.GetIndexedUnit = VFL.Identity;
	qq.IsIndexedUnit = VFL.True;
	ubi[i] = qq;
end

-- The player unit. Refers to the player's RDX unit always.
RDXPlayer = RDX.GetUnitByName(strlower(UnitName("player")));

-- Defer the creation and application of edata and ndata until the VARS_LOADED phase.
-- This gives all mods a chance to load first.
RDXEvents:Bind("INIT_VARIABLES_LOADED", nil, function()
	-- Create and apply EData.
	for i=1,NUM_UNITS do
		edata[i] = NewEData(i);
		VFL.mixin(ubi[i], edata[i], true);
	end
	-- Create all NData that doesn't exist
	for n,unit in pairs(ubn) do VFL.mixin(unit, GetNData(n), true); end
end);

------------------------------------------------------------
-- UNIT_AURA HANDLING
------------------------------------------------------------
-- The queue of units whose auras are dirty
local auraq = {};
-- Perf: how many auras should I process per frame?
local aura_unitsPerUpdate = 3;

-- Process pending aura updates
local function ProcessAuraQueue()
	local i, unit, batchTrig = 1, nil, nil;
	for un,_ in pairs(auraq) do
		if(i > aura_unitsPerUpdate) then break; end
		if not batchTrig then batchTrig = true; RDX.BeginEventBatch(); end
		ubi[un]:ProcessAuras();
		auraq[un] = nil;
		i = i + 1;
	end
	if batchTrig then RDX.EndEventBatch(); end
end
local auraFrame = CreateFrame("Frame");
auraFrame:SetScript("OnUpdate", ProcessAuraQueue);
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "ProcessAuras", ProcessAuraQueue, true);

--- Force a reprocessing of auras for all valid units.
function RDX.FlushAuras()
	local u;
	tempty(auraq);
	for i=1,NUM_UNITS do
		u = RDX.GetUnitByNumber(i);
		if u:IsValid() then auraq[i] = true; end
	end
end
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "FlushAuras", RDX.FlushAuras, true);

-- On UNIT_AURA, add the aura'd unit to the aura queue.
WoWEvents:Bind("UNIT_AURA", nil, function()
	local x = id2num[arg1];
	if x then
		auraq[x] = true;
		_sig_rdx_unit_aura:Raise(ubi[x], x, arg1);
	end
end);

-- On PLAYER_ENTERING_WORLD, refresh all auras.
WoWEvents:Bind("PLAYER_ENTERING_WORLD", nil, RDX.FlushAuras);

----------------------------------------------------------------
-- ATE Processing
----------------------------------------------------------------

-- Engine to store startime for specific single unit
-- This is used to fill ATE
--[[
local debuffSTTarget = {};
local function getDebuffSTTarget(debuff)
	return debuffSTTarget[debuff] or 0;
end
local function setDebuffSTTarget(debuff, st)
	debuffSTTarget[debuff] = st;
end

local debuffSTFocus = {};
local function getDebuffSTFocus(debuff)
	return debuffSTFocus[debuff] or 0;
end
local function setDebuffSTFocus(debuff, st)
	debuffSTFocus[debuff] = st;
end

local debuffSTPet = {};
local function getDebuffSTPet(debuff)
	return debuffSTTarget[debuff] or 0;
end
local function setDebuffSTPet(debuff, st)
	debuffSTPet[debuff] = st;
end

local targetlock = false;
local focuslock = false;
local petlock = false;
local atelock = false;

local function ATEDebuff(uid)
	local i, cont, name, category, apps, dur, tLeft, mycast, who, startime = 0, nil, nil, nil, 0, nil, nil, nil, nil, 0;
	local getDebuffST, setDebuffST;
	if (uid == "target") then
		getDebuffST = getDebuffSTTarget;
		setDebuffST = setDebuffSTTarget;
		atelock = targetlock;
	elseif (uid == "focus") then
		getDebuffST = getDebuffSTFocus;
		setDebuffST = setDebuffSTFocus;
		atelock = focuslock;
	elseif (uid == "pet") then
		getDebuffST = getDebuffSTPet;
		setDebuffST = setDebuffSTPet;
		atelock = petlock;
	end
	if not atelock then
		atelock = true;
		i=1;
		while true do
			cont, name, category, apps, _, _, _, dur, tleft, _, mycast, who = LoadDebuffFromUnit(uid, i, nil, false, false);
			if not cont then break; end
			if name then
				local ATEduration = RDXATE.listDebuffs[name];
				if mycast and ATEduration then
					startime = GetTime() + tleft - dur;
					if (startime - getDebuffST(name)) > 1 then
						OmniDB.SetGUIDDebuff(UnitGUID(uid), strlower(UnitName(uid)), name, dur, tleft, who, true);
						setDebuffST(name, startime);
					end
				end
			end
			i = i + 1;
		end
		atelock = false;
	end
end

VFLP.RegisterFunc(i18n("RDX: UnitDB"), "ATEDebuff", ATEDebuff, true);
	
WoWEvents:Bind("UNIT_AURA", nil, function()
	if (arg1 == "target") or (arg1 == "focus") or (arg1 == "pet") then
		ATEDebuff(arg1);
	end
end);

-- ATE, in case of receving an update
function RDX.ATEProcess(obj)
	local uguid = RDX.GetUnitByGuid(obj.g);
	if (not uguid) or (not uguid:IsValid()) then return; end
	if not auraq[uguid.nid] then
		--auraq[uguid.nid] = true;
		--_sig_rdx_unit_aura:Raise(uguid, uguid.nid, uguid.uid);
	end
end

VFLP.RegisterFunc(i18n("RDX: UnitDB"), "ATEProcess", RDX.ATEProcess, true);
]]
-----------------------------------------------------------------
-- RAID ROSTER
-----------------------------------------------------------------

-- When the group is a party, get the roster info for the given unit.
local function party_GetRosterInfo(idx, uid)
	if not uid then return nil; end
	local _, class = UnitClass(uid);
	local llv = 0;
	if IsSolo() or UnitIsPartyLeader(uid) then llv = 2; end
	return UnitName(uid), llv, 1, UnitLevel(uid), nil, class;
end
local GetRosterInfo = VFL.Nil;

-- Main roster processing.
local _rtouched, _gtouched = {}, {};
function RDX.ProcessRoster()
	VFL.empty(_rtouched);
	VFL.empty(_gtouched);

	local roster_changed = nil;
	local my_ndata, my_edata;
	local iunit, uid, nunit;
	local name, leaderLevel, grp, level, class, guid;

	RDX.BeginEventBatch();

	-- Iterate over all valid units
	local n = RDX.GetNumUnits();
	RDX:Debug(2, "RDX.ProcessRoster(): processing ", n, " units.");
	for i=1,n do
		iunit = ubi[i];
		uid = RDX.NumberToUID(i);
		name, leaderLevel, grp, level, _, class = GetRosterInfo(i, uid);
		guid = UnitGUID(uid);
		if not guid then guid = "Unknown"; VFL.print(UnitName(uid) .. " guid unknown"); end
		if (not name) or (name == "Unknown") then
			-- This unit is now invalid...
			iunit:Invalidate();
		else
			iunit.rosterName = name;
			name = strlower(UnitName(uid));
			-- Mark engine unit as valid
			iunit.uid = uid; 
			-- patch 2.4
			iunit.guid = guid;
			iunit:Validate();
			-- If engine unit has changed, schedule it for rediscovery
			if(iunit.name ~= name) then 
				RDX:Debug(7, "Roster: NID<", tostring(i), "> ", tostring(iunit.name), " -> ", tostring(name));
				iunit.name = name;
				-- When an engine unit changes identities, update auras too.
				auraq[i] = true;
				roster_changed = true; 
			end
			
			-- Acquire nominative unit and mark as valid
			nunit = RDX.GetUnitByName(name);
			nunit.rosterName = iunit.rosterName;
			nunit.uid = uid;
			-- patch 2.4
			nunit.guid = guid;
			nunit:Validate();
			
			if(nunit.nid ~= i) then nunit.nid = i; roster_changed = true; end
			-- Mark unit as touched this session
			_rtouched[name] = iunit;
			_gtouched[guid] = iunit;
			
			-- Get unit data
			my_ndata = GetNData(name); my_edata = GetEData(i);

			-- Update unit data
			my_ndata.SetLeaderLevel(nil, leaderLevel);
			my_edata.SetGroup(nil, grp);
			my_ndata.SetClassID(nil, VFLGetClassID(class));

			-- Attach new edata to nunit
			VFL.mixin(nunit, my_edata, true);			
			-- Atach new ndata to eunit
			VFL.mixin(iunit, my_ndata, true);
		end
	end

	-- Iterate over all invalid units
	if (n < 40) and (ubi[n+1]:_ValidMetatable()) then roster_changed = true; end
	for i=(n+1),40 do
		if ubi[i]:Invalidate() then
			RDX:Debug(7, "Roster: NID<", i, "> quashed.");
		end
	end

	-- Invalidate all nominative units no longer present
	for k,v in pairs(ubn) do
		if not _rtouched[k] then v:Invalidate(); end
	end

	-- Notify of a roster update
	if roster_changed then RDXEvents:Dispatch("ROSTER_NIDS_CHANGED", _rtouched); end
	RDXEvents:Dispatch("ROSTER_UPDATE", _rtouched);
	RDX.EndEventBatch();
end
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "ProcessRoster", RDX.ProcessRoster, true);

-- Pet processing
local ProcessPets = VFL.CreatePeriodicLatch(1, function()
	RDX:Debug(2, "Roster: ProcessPets()");
	local unit, uid, changed;
	changed = nil;
	for i=41,80 do
		unit = ubi[i]; uid = RDX.NumberToUID(i);
		if UnitExists(uid) then
			if not unit:IsValid() then
				unit:Validate();
				unit.uid = uid;
				unit.name = strlower(UnitName(uid));
				unit.guid = UnitGUID(uid);
				if not unit.guid then unit.guid = "Unknow pet"; end
				auraq[i] = true; changed = true;
			end
		elseif unit:_ValidMetatable() then
			unit:Invalidate();
			changed = true;
		end
	end
	if changed then 
		RDX:Debug(2, "ROSTER_PETS_CHANGED");
		RDXEvents:Dispatch("ROSTER_PETS_CHANGED"); 
	end
end);
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "ProcessPets", RDX.ProcessPets, true);

-- Pets: whenever a major change in the raid roster happens, or a UNIT_PET happens
-- let's update the pets.
WoWEvents:Bind("UNIT_PET", nil, ProcessPets);

----------------------------------------------------------------------------
-- ROSTER EVENT BINDINGS
----------------------------------------------------------------------------
local SetRaid, SetNonRaid;

-- Called on the WoW RAID_ROSTER_UPDATE event.
-- Latched to prevent uberspam.
local OnRaidRosterUpdate = function()
	local n = GetNumRaidMembers();

	-- If we weren't in a raid, transition to raid status
	if not isRaid then
		if(n > 0) then SetRaid(); return; end
		RDX.ProcessRoster();
		return;
	end

	if(n == 0) then SetNonRaid(); return; end

	RDX.ProcessRoster();
end

-- Called on the WoW PARTY_MEMBERS_CHANGED event
-- Latched to prevent uberspam.
local OnPartyMembersChanged = VFL.CreatePeriodicLatch(1, function()
	if isRaid then return; end
	-- Check solo state
	local soloChanged = nil;
	local n = RDX.GetNumUnits();
	if n == 1 and (not isSolo) then
		isSolo = true; soloChanged = true;
	elseif n > 1 and isSolo then
		isSolo = false; soloChanged = true;
	end
	-- Process roster
	RDX.ProcessRoster();
	if soloChanged then RDXEvents:Dispatch("PARTY_IS_NONRAID"); end
end);

-- Internal: Flip between raid and nonraid status
function SetRaid()
	RDX:Debug(1, "SetRaid()");

	WoWEvents:Unbind("party_roster");
	
	isRaid = true; isSolo = false;
	SetRaidIDDatabase();
	RDX.GetNumUnits = raid_GetNumUnits;
	GetRosterInfo = GetRaidRosterInfo;

	RDX.BeginEventBatch();
	RDX.ProcessRoster();
	RDX.FlushAuras();
	RDX.EndEventBatch();

	RDXEvents:Dispatch("PARTY_IS_RAID");
end

function SetNonRaid(noReprocess)
	RDX:Debug(1, "SetNonRaid()");
	
	isRaid = nil;
	SetPartyIDDatabase();
	RDX.GetNumUnits = party_GetNumUnits;
	GetRosterInfo = party_GetRosterInfo;

	if RDX.GetNumUnits() == 1 then isSolo = true; else isSolo = false; end

	if not noReprocess then
		RDX.BeginEventBatch();
		RDX.ProcessRoster();
		RDX.FlushAuras();
		RDX.EndEventBatch();
	end

	WoWEvents:Bind("PARTY_MEMBERS_CHANGED", nil, OnPartyMembersChanged, "party_roster");

	RDXEvents:Dispatch("PARTY_IS_NONRAID");
end

-- Before everything loads, let's setup in a default nonraid state
RDXEvents:Bind("INIT_PRELOAD", nil, function()
	WoWEvents:Bind("RAID_ROSTER_UPDATE", nil, OnRaidRosterUpdate);
	SetNonRaid(true);
end);

-- After everything loads, let's double check our party/raid status.
--RDXEvents:Bind("INIT_VARIABLES_LOADED", nil, OnRaidRosterUpdate);
-- add because the guid is not available at INIT_VARIABLES_LOADED with UnitGUID(uid) sigg
RDXEvents:Bind("INIT_DEFERRED", nil, OnRaidRosterUpdate);




------------------------------------------------------------
-- ROSTER LOOKUP AND INDEXING
-----------------------------------------------------------
--- Get the RDX Unit for this raider if he's in the raid; otherwise
-- return nil.
function RDX.GetUnitByNameIfInGroup(name)
	return _rtouched[name];
end

function RDX.UnitInGroup(uid)
	return (UnitInParty(uid) or UnitInRaid(uid));
end

function RDX.ProjectUnitID(uid)
	for i=1,NUM_UNITS do
		if UnitIsUnit(num2id[i] or "none", uid) then return ubi[i]; end
	end
end

function RDX._FastProject(uid)
	if not RDX.UnitInGroup(uid) then return nil; end
	local un = id2num[uid]; if not un then return nil; end
	return ubi[un];
end

function RDX._ReallyFastProject(uid)
	if not uid then return nil; end
	local un = id2num[uid]; if not un then return nil; end
	return ubi[un];
end


-- Get the RDX Unit for this raider if he's in the raid; otherwise
-- return nil, no PET
function RDX.GetUnitByGuidIfInGroup(guid)
	return _gtouched[guid];
end

-- not really fast but all and PET
function RDX.GetUnitByGuid(guid)
	for i,_ in pairs(ubi) do
		if (ubi[i].guid == guid) then return ubi[i]; end
	end
end

--function RDX.TESTPrint()
--	for i,_ in pairs(ubi) do
--		if ubi[i].guid then VFL.print(ubi[i].guid); end
--	end
--end


-------------------------------
-- PROJECTIVE UNIT
-- A projective unit is a unit with a non-canonical id (i.e. not partyX or raidX)
-- that may or may not be equivalent to one of the canonical units. If it is,
-- there is a Project() operator that will figure out which one and match
-- edata and ndata appropriately.
-------------------------------
local function ProjUnit_Project(self)
	local uid, nid = self.uid, 0;
	-- Project
	if UnitExists(uid) then
		self.name = strlower(UnitName(uid));
		self.guid = UnitGUID(uid);
		for i=1,NUM_UNITS do
			if UnitIsUnit(num2id[i] or "none", uid) then nid = i; end
		end
	else
		self.name = "";
	end
	-- Store nid
	self.nid = nid;
	-- Get unit data
	VFL.mixin(self, GetEData(nid), true);
	if nid > 0 and nid < 41 then -- Only players in grp have ndata.
		VFL.mixin(self, GetNData(self.name), true);
	else
		VFL.mixin(self, GetNData(""), true);
	end
end

RDX.ProjectiveUnit = {};
function RDX.ProjectiveUnit:new()
	local x = RDX.Unit:new();
	x.uid = "none"; x.name = "unknown"; x.nid = 0; x.guid = "unknown";
	x._Project = ProjUnit_Project;
	x.Invalidate = VFL.Noop; x.Validate = VFL.Noop;

	return x;
end


------------------------------------------------------------
-- UNIT_HEALTH/UNIT_MANA HANDLING
------------------------------------------------------------

-- Propagate events only if they pertain to RDX-managed units, and
-- promote the units from raw unit IDs to full fledged RDX unit 
-- objects.
local function UnitHealthPropagator()
	local x = id2num[arg1];
	if x then
		_sig_rdx_unit_health:Raise(ubi[x], x, arg1, UnitHealth(arg1), UnitHealthMax(arg1));
	end
end
local function UnitManaPropagator()
	local x = id2num[arg1];
	if x then
		_sig_rdx_unit_mana:Raise(ubi[x], x, arg1, UnitMana(arg1), UnitManaMax(arg1));
	end
end

-- Raw bindings
WoWEvents:Bind("UNIT_HEALTH", nil, UnitHealthPropagator);
WoWEvents:Bind("UNIT_MAXHEALTH", nil, UnitHealthPropagator);
WoWEvents:Bind("UNIT_MANA", nil, UnitManaPropagator);
WoWEvents:Bind("UNIT_MAXMANA", nil, UnitManaPropagator);
WoWEvents:Bind("UNIT_ENERGY", nil, UnitManaPropagator);
WoWEvents:Bind("UNIT_RAGE", nil, UnitManaPropagator);

------------------------------------------------------------------
-- FEIGN DEATH CHECKING
------------------------------------------------------------------
-- If a unit's health drops low, check it for the FeignDeath buff. If so, mark as feigned.
RDXEvents:Bind("UNIT_HEALTH", nil, function(u, un, uid, uh)
	if(uh < 2) and (not u:IsFeigned()) and (UnitIsFeignDeath(uid)) then
		u:_SetFeigned(true);
		RDXEvents:Dispatch("UNIT_FEIGN_DEATH", u, un, uid, true);
	end
end);

-- When a FD unit's auras change, snap check to see if FD went away.
RDXEvents:Bind("UNIT_AURA", nil, function(u, un, uid)
	if u:IsFeigned() and (not UnitIsFeignDeath(uid)) then
		u:_SetFeigned(nil);
		RDXEvents:Dispatch("UNIT_FEIGN_DEATH", u, un, uid, false);
	end
end);

--------------------------------------------
-- TOTEM EVENTS
--------------------------------------------

WoWEvents:Bind("PLAYER_ENTERING_WORLD", nil, function()
	_sig_rdx_unit_entering_world:Raise(RDXPlayer, RDXPlayer.nid, RDXPlayer.uid);
end);

-- Target change
WoWEvents:Bind("PLAYER_TOTEM_UPDATE", nil, function()
	_sig_rdx_unit_totem_update:Raise(RDXPlayer, RDXPlayer.nid, RDXPlayer.uid);
end);

--------------------------------------------
-- BUFF WEAPON EVENTS
--------------------------------------------

-- there is no event to track enchant weapons...
RDXEvents:Bind("INIT_VARIABLES_LOADED", nil, function()
	local timemh, timeoh = 0, 0;
	VFL.AdaptiveSchedule("weaponsupdate_update", 1, function()
		local hasMainHandEnchant, mainHandExpiration, _, hasOffHandEnchant, offHandExpiration = GetWeaponEnchantInfo();
		if hasMainHandEnchant then
			if mainHandExpiration > timemh then
				_sig_rdx_unit_buffweapon_update:Raise(RDXPlayer, RDXPlayer.nid, RDXPlayer.uid);
			end
			timemh = mainHandExpiration;
		elseif timemh > 0 then
			_sig_rdx_unit_buffweapon_update:Raise(RDXPlayer, RDXPlayer.nid, RDXPlayer.uid);
			timemh = 0;
		end
		if hasOffHandEnchant then
			if offHandExpiration > timeoh then
				_sig_rdx_unit_buffweapon_update:Raise(RDXPlayer, RDXPlayer.nid, RDXPlayer.uid);
			end
			timeoh = offHandExpiration;
		elseif timeoh > 0 then
			_sig_rdx_unit_buffweapon_update:Raise(RDXPlayer, RDXPlayer.nid, RDXPlayer.uid);
			timeoh = 0;
		end
	end);
end);

--------------------------------------------
-- MISC EVENTS
--------------------------------------------
-- store spell and rank
WoWEvents:Bind("UNIT_SPELLCAST_SENT", nil, function()
	if not arg1 then return; end
	local x = id2num[arg1];
	local un = ubi[x];
	if un then 
		if arg2 and arg3 then
			un.SetLastSpellRank(arg2, string.match(arg3, i18n("Rank (%d+)")));
		end
	end
end);

-- Powertype change
WoWEvents:Bind("UNIT_DISPLAYPOWER", nil, function()
	local x = RDX.GetUnitByNameIfInGroup(strlower(arg1));
	if x then
		RDXEvents:Dispatch("UNIT_DISPLAYPOWER", x, x.nid, arg1);
	end
end);

-- Target change
WoWEvents:Bind("UNIT_TARGET", nil, function()
	local x = id2num[arg1];
	if x then _sig_rdx_unit_target:Raise(ubi[x], x, arg1); end
end);

-- Focus change
WoWEvents:Bind("UNIT_FOCUS", nil, function()
	local x = id2num[arg1];
	if x then _sig_rdx_unit_focus:Raise(ubi[x], x, arg1); end
end);

-- Flags change (combat etc)
local function flagprop()
	local x = id2num[arg1];
	if x then _sig_rdx_unit_flags:Raise(ubi[x], x, arg1); end
end
WoWEvents:Bind("UNIT_FLAGS", nil, flagprop);
WoWEvents:Bind("UNIT_DYNAMIC_FLAGS", nil, flagprop);

-- Rangedamage
local function rangeprop()
	local x = id2num[arg1];
	if x then _sig_rdx_unit_range:Raise(ubi[x], x, arg1); end
end
WoWEvents:Bind("UNIT_RANGEDDAMAGE", nil, rangeprop);
WoWEvents:Bind("UNIT_RANGED_ATTACK_POWER", nil, rangeprop);

-- Portrait change.
WoWEvents:Bind("UNIT_PORTRAIT_UPDATE", nil, function()
	local x = id2num[arg1];
	if x then _sig_rdx_unit_portrait_update:Raise(ubi[x], x, arg1); end
end);

-- faction change.
WoWEvents:Bind("UNIT_FACTION", nil, function()
	local x = id2num[arg1];
	if x then _sig_rdx_unit_faction:Raise(ubi[x], x, arg1); end
end);

-- Spell events
-- CAST_TIMER_UPDATE
local sigUNIT_CAST_TIMER_UPDATE = RDXEvents:LockSignal("UNIT_CAST_TIMER_UPDATE");
local function filter()
	local x = id2num[arg1];
	if x then sigUNIT_CAST_TIMER_UPDATE:Raise(ubi[x], x, arg1); end
end
WoWEvents:Bind("UNIT_SPELLCAST_CHANNEL_START", nil, filter);
WoWEvents:Bind("UNIT_SPELLCAST_CHANNEL_UPDATE", nil, filter);
WoWEvents:Bind("UNIT_SPELLCAST_DELAYED", nil, filter);
WoWEvents:Bind("UNIT_SPELLCAST_START", nil, filter);

-- CAST_TIMER_STOP
local sigUNIT_CAST_TIMER_STOP = RDXEvents:LockSignal("UNIT_CAST_TIMER_STOP");
local function filter()
	local x = id2num[arg1];
	if x then sigUNIT_CAST_TIMER_STOP:Raise(ubi[x], x, arg1); end
end
WoWEvents:Bind("UNIT_SPELLCAST_CHANNEL_STOP", nil, filter);
WoWEvents:Bind("UNIT_SPELLCAST_FAILED", nil, filter);
WoWEvents:Bind("UNIT_SPELLCAST_INTERRUPTED", nil, filter);
WoWEvents:Bind("UNIT_SPELLCAST_SUCCEEDED", nil, filter);
WoWEvents:Bind("UNIT_SPELLCAST_STOP", nil, filter);

-----------------------------------------------------
-- ITERATORS
-- Iterate over the units in the raid.
-----------------------------------------------------
-- Return an iterator over the current group.
local function giter(_, i)
	i=i+1;
	local u = ubi[i];
	if u and u:IsValid() then return i, u; end
end

function RDX.Raid()
	return giter, nil, 0;
end

-- Return an iterator for all, raids and pets
local function giter2(_, i)
	i=i+1;
	local u = ubi[i];
	if u then return i, u; end
end

function RDX.RaidAll()
	return giter2, nil, 0;
end

local function GroupStatelessIterator(gn, idx)
	local u = nil;
	while true do
		idx = idx + 1; u = ubi[idx];
		if (not u) or (not u:IsValid()) then break; end
		if (u:GetGroup() == gn) then return idx, u; end
	end
end

function RDX.Group(gn)
	if not gn then return giter, nil, 0; end
	return GroupStatelessIterator, gn, 0;
end

-----------------------------------------------------------------
-- "DISRUPTIVE EVENTS"
-- Disruptive events are events that tend to require the whole
-- UI to rebuild itself. We consolidate them into one big event.
-----------------------------------------------------------------
local function Disruption()
	RDX:Debug(1,"|cFFFF00FFDisruption: ", tostring(event), "|r");
	RDXEvents:Dispatch("DISRUPT_SETS");
	RDXEvents:Dispatch("DISRUPT_SORTS");
	RDXEvents:Dispatch("DISRUPT_WINDOWS");
end
VFLP.RegisterFunc(i18n("RDX: UnitDB"), "Disruptions", Disruption, true);

WoWEvents:Bind("PLAYER_ENTERING_WORLD", nil, Disruption);
RDXEvents:Bind("ROSTER_UPDATE", nil, Disruption);
RDXEvents:Bind("ROSTER_PETS_CHANGED", nil, Disruption);
VFLEvents:Bind("PLAYER_COMBAT", nil, Disruption);

function RDX._Disrupt()
	Disruption();
end

function RDX._Test_var()
	RDX_test = RDX._GetBuffCache();
end

