-- Obj_NominativeSet.lua
-- RDX - Raid Data Exchange
-- (C)2006 Bill Johnson
--
-- THIS FILE CONTAINS COPYRIGHTED MATERIAL SUBJECT TO THE TERMS OF A SEPARATE
-- LICENSE. UNLICENSED COPYING IS PROHIBITED.
--
-- Glue code for attaching NominativeSets to the user interface.

local dlg = nil;
local function EditNominativeSetDialog(parent, path, md)
	if dlg then
		RDX.print(i18n("A set editor is already open. Please close it first."));
		return;
	end

	-- Sanity checks
	if (not path) or (not md) or (not md.data) then return nil; end
	local inst = RDXDB.GetObjectInstance(path, true);

	if not parent then 
		RDX:Debug(1, "WARNING: Parentless dialog in EditNominativeSetDialog()");
		parent = UIParent; 
	end
	dlg = VFLUI.Window:new(parent);
	VFLUI.Window.SetDefaultFraming(dlg, 22);
	dlg:SetTitleColor(0,0,.6);
	dlg:SetBackdrop(VFLUI.BlackDialogBackdrop);
	dlg:SetPoint("CENTER", UIParent, "CENTER");
	dlg:SetWidth(270); dlg:SetHeight(350);
	dlg:SetText(i18n("Edit NominativeSet: ") .. path);
	VFLUI.Window.StdMove(dlg, dlg:GetTitleBar());
	dlg:Show();
	
	--------------- PREDECLARATIONS
	local list = md.data or {};

	---------------- CONTROLS
	local lbl = VFLUI.MakeLabel(nil, dlg, i18n("Enter name to add:"));
	lbl:SetPoint("TOPLEFT", dlg:GetClientArea(), "TOPLEFT");

	local le = VFLUI.ListEditor:new(dlg, list, function(cell, data) cell.text:SetText(data); end);
	le:SetPoint("TOPLEFT", lbl, "BOTTOMLEFT");
	le:SetWidth(260); le:SetHeight(263);
	le:Show();

	---------------- OK button
	local btnOK = VFLUI.OKButton:new(dlg);
	btnOK:SetText("OK");
	btnOK:SetHeight(25); btnOK:SetWidth(60);
	btnOK:SetPoint("BOTTOMRIGHT", dlg:GetClientArea(), "BOTTOMRIGHT");
	btnOK:Show();

	----------------- Close functionality
	dlg.Destroy = VFL.hook(function()
		btnOK:Destroy(); btnOK = nil;
		le:Destroy(); le = nil; list = nil;
		dlg = nil;
	end, dlg.Destroy);
	
	-- Escapement
	local esch = function() dlg:Destroy(); end
	VFL.AddEscapeHandler(esch);
	local closebtn = VFLUI.CloseButton:new();
	closebtn:SetScript("OnClick", function() VFL.EscapeTo(esch); end);
	dlg:AddButton(closebtn);

	-- OK handler
	btnOK:SetScript("OnClick", function()
		local lst = le:GetList();
		VFL.EscapeTo(esch);
		if lst then 
			-- Lowercase the list before saving.
			for k,name in ipairs(lst) do lst[k] = string.lower(name); end
			md.data = lst;
			if inst then inst:SetNameList(lst); end
		end
	end);
end

-- Object class registration
RDXDB.RegisterObjectType({
	name = "NominativeSet";
	New = function(path, md)
		md.version = 1;
		md.data = {};
	end;
	Edit = function(path, md, parent)
		EditNominativeSetDialog(parent or VFLHigh, path, md);
	end;
	Twiddle = function(path, md, name, val)
		if not name then return; end
		name = string.lower(name);
		local inst = RDXDB.GetObjectInstance(path);
		if not inst then return; end
		local cs = inst:CheckName(name);
		if (cs) and (not val) then
			inst:RemoveName(name);
			md.data = inst:GetNameArray();
		elseif (not cs) and (val) then
			inst:AddName(name);
			md.data = inst:GetNameArray();
		end
	end;
	Instantiate = function(path, obj)
		local x = RDX.NominativeSet:new();
		x.name = "NominativeSet<" .. path .. ">";
		x.path = path;
		RDX.RegisterSet(x);
		x:SetNameList(obj.data);
		return x;
	end;
	GenerateBrowserMenu = function(mnu, path, md, dlg)
		table.insert(mnu, {
			text = i18n("Edit..."),
			OnClick = function() 
				VFL.poptree:Release(); 
				EditNominativeSetDialog(dlg, path, md); 
			end
		});
	end;
});

------------------------------------------------------
-- NOMINATIVE SET EDITOR
-- The nominative set editor is a panel of 3 buttons that attaches to a window that has
-- a NominativeSet as its underlying set.
------------------------------------------------------
RDX.RegisterFeature({
	name = "NominativeSet Editor",
	IsPossible = function(state)
		if not state:Slot("UnitWindow") then return nil; end
		if not state:Slot("SetDataSource") then return nil; end
		if state:Slot("NominativeSetEditor") then return nil; end
		return true;
	end,
	ExposeFeature = function(desc, state, errs)
		state:AddSlot("NominativeSetEditor");
		local set = state:RunSlot("SetDataSource");
		if (not set) or (not set.name) then
			VFL.AddError(errs, i18n("Underlying set is missing or invalid."));
			return nil;
		end
		if not set.AddName then
			VFL.AddError(errs, i18n("Underlying set is not nominative."));
			return nil;
		end
		return true;
	end;
	ApplyFeature = function(desc, state)
		local btnPlus, btnMinus, btnStar;
		local set = state:RunSlot("SetDataSource");
		local path = set.path;
		local orientation = desc.orientation or "LEFT";

		-- Upon window creation, generate the framepool's fallback function.
		state:_Attach(state:Slot("Create"), true, function(w)
			
			-- Allow this window to be used for drag/drop operations from the
			-- roster screen.
			RDXUI.dc_RaidMembers:RegisterDragTarget(w);
			function w:OnDrop(dropped)
				local n = dropped.data; if type(n) ~= "string" then return; end
				RDXDB.OpenObject(path, "Twiddle", n, true);
			end

			btnPlus = VFLUI.TexturedButton:new(w, 16, "Interface\\AddOns\\VFL\\Skin\\plus");
			btnPlus:SetHighlightColor(0,1,0,1);
			w:AddButton(btnPlus);
			btnPlus:SetScript("OnClick", function()
				local n = UnitName("target"); if not n then return; end
				if not UnitIsPlayer("target") then return; end
				n = string.lower(n);
				RDXDB.OpenObject(path, "Twiddle", n, true);
			end);

			btnMinus = VFLUI.TexturedButton:new(w, 16, "Interface\\AddOns\\VFL\\Skin\\minus");
			btnMinus:SetHighlightColor(1,0,0,1);
			w:AddButton(btnMinus);		
			btnMinus:SetScript("OnClick", function()
				local n = UnitName("target"); if not n then return; end
				n = string.lower(n);
				RDXDB.OpenObject(path, "Twiddle", n, nil);
			end);

			btnStar = VFLUI.TexturedButton:new(w, 16, "Interface\\AddOns\\RDX\\Skin\\menu");
			btnStar:SetHighlightColor(0,1,1,1);
			w:AddButton(btnStar);		
			btnStar:SetScript("OnClick", function()
				RDXDB.OpenObject(path, "Edit");
			end);
		end);
		
		-- Upon window destruction, also destroy the underlying framepool.
		state:_Attach(state:Slot("Destroy"), true, function(w)
			RDXUI.dc_RaidMembers:UnregisterDragTarget(w); w.OnDrop = nil;
		end);

		return true;
	end,
--[[	UIFromDescriptor = function(desc, parent)
		local self = VFLUI.RadioGroup:new(parent);
		self:SetLayout(2,2);
		self.buttons[1]:SetText("Orient Left");
		self.buttons[2]:SetText("Orient Right");
		if desc and desc.orientation and desc.orientation == "RIGHT" then
			self:SetValue(2);
		else
			self:SetValue(1);
		end

		function self:GetDescriptor()		
			local ori = "LEFT";
			if self:GetValue() == 2 then ori = "RIGHT"; end
			return {feature = "NominativeSet Editor", orientation = ori;};
		end

		return self;
	end, ]] --
	UIFromDescriptor = VFL.Nil;
	CreateDescriptor = function() return { feature = "NominativeSet Editor" }; end,
});
