-- Log.lua
-- RDX - Project Omniscience
-- (C)2006 Bill Johnson
--
-- Combat logging facilities for Project Omniscience.

-- Imports
local match, gmatch, lower = string.find, string.gmatch, string.lower;
local GetUnitByName = RDX.GetUnitByNameIfInGroup;

-- The Omniscience log
local omniLog, logCap = {}, 3000;
local omniLogPet, logCapPet = {}, 3000;
local omniLogAll, logCapAll = {}, 3000;

--- Get the Omniscience local log table.
function Omni.GetLog()
	return omniLog;
end

function Omni.GetLogPet()
	return omniLogPet;
end

function Omni.GetLogAll()
	return omniLogAll;
end

local logTrigger = OmniEvents:LockSignal("LOG_ROW_ADDED");
local logTriggerAll = OmniEvents:LockSignal("LOGALL_ROW_ADDED");

-------------------------------------------
-- LOOKUP METADATA: BUFFS TO TRACK
-- Omniscience doesn't ordinarily record buffs because the log would be overspammed
-- with buff info. Certain exceptions (shield wall, last stand, etc.) are recorded here.
-------------------------------------------
--[[
local trackedBuffs = {};
trackedBuffs[i18n("Shield Block")] = true;
trackedBuffs[i18n("Shield Wall")] = true;
trackedBuffs[i18n("Gift of Life")] = true;
trackedBuffs[i18n("Last Stand")] = true;
trackedBuffs[i18n("Power Word: Shield")] = true;
Omni.trackedBuffs = trackedBuffs;
]]--

-------------------------------------------
-- LOOKUP METADATA: DAMAGE TYPES
-------------------------------------------
local dmgToType = {};
Omni.dmgToType = dmgToType;
local typeToDmg = {};
Omni.typeToDmg = typeToDmg;

--[[ 
1 = Physical
2 = Holy
3 = Fire
4 = Nature
5 = Frost
6 = Shadow
7 = Arcane
]]

local dmgTypeColors = {
	[1] = { r=1, g=1, b=0 }, [2] = { r=1, g=1, b=.7 }, [3] = { r=.95, g=.3, b= 0 }, 
	[4] = { r=.65, g=.9, b=.25 }, [5] = { r=.25, g=.9, b=.92 }, [6] = { r=.5, g=.13, b=.58 },
	[7] = { r=1, g=1, b=1 },
};

function Omni.GetDamageTypeColor(idx)
	if not idx then return dmgTypeColors[1]; end
	local ret = dmgTypeColors[idx];
	if not ret then return dmgTypeColors[1]; end
	return ret;
end

local unknown = i18n("Unknown");
function Omni.GetDamageTypeName(idx)
	if not idx then return unknown; end
	local ret = typeToDmg[idx];
	if not ret then return unknown; end
	return ret;
end

-------------------------------------------------
-- LOOKUP METADATA: LOG ROW TYPES
-- (1 = damage in, 2 = damage out, 3 = healing in, 4 = healing out) 
-- (5 = debuff applied, 6 = debuff removed, 7 = healing self, 8 = healing))
-- (11 = in combat, 12 = out of combat, 13 = enc start, 14 = enc stop)
-- (15 = enter bg, 16 = leave bg, 17 = killing blow, 18 = your death)
-------------------------------------------------

local rowTypeColors = {
	[1] = {r=.75, g=0, b=0}, [2] = {r=.9, g=.5, b=0},
	[3] = {r=0, g=1, b=0}, [4] = {r=.25, g=.7, b=1},
	[5] = {r=1,g=0,b=0}, [6] = {r=0.5,g=0.9,b=0},
	[7] = {r=0,g=1,b=.35}, [8] = {r=0,g=1,b=0.35},
	[9] = {r=0,g=1,b=0.75}, [10] = {r=1,g=0.5,b=0.75},
	[11] = {r=0.6,g=0.6,b=0}, [12] = {r=0.6,g=0.6,b=0},
	[13] = {r=1,g=1,b=1}, [14] = {r=1,g=1,b=1},
	[15] = {r=1,g=1,b=1}, [16] = {r=1,g=1,b=1},
	[17] = {r=0,g=0.5,b=0}, [18] = {r=0.5,g=0,b=0},
	[19] = {r=1,g=1,b=1}, [20] = {r=1,g=1,b=1},
	[21] = {r=.75, g=0, b=0}, [22] = {r=.9, g=.5, b=0},
};

local rowTypes = {
	[1] = "DamageIn", [2] = "DamageOut", 
	[3] = "HealingIn", [4] = "HealingOut",
	[5] = "+Debuff", [6] = "-Debuff", 
	[7] = "HealingSelf", [8] = "Healing",
	[9] = "+Buff", [10] = "-Buff",
	[11] = "+Combat", [12] = "-Combat", 
	[13] = "+Encounter", [14] = "-Encounter", 
	[15] = "+Battleground", [16] = "-Battleground", 
	[17] = "Killing Blow", [18] = "Death", 
	[19] = "Perform", [20] = "Cast", 
	[21] = "CastMob", [22] = "CastMobSuccess",
	[23] = "+BuffMob", [24] = "-BuffMob",
	[25] = "+DebuffMob", [26] = "-DebuffMob",
	[27] = "CastSucess",
};

local typeToRow = VFL.invert(rowTypes);

function Omni.GetMap_IndexToRowType() return rowTypes; end
function Omni.GetMap_RowTypeToIndex() return typeToRow; end

function Omni.GetRowTypeColor(idx)
	if not idx then return rowTypeColors[7]; end
	local ret = rowTypeColors[idx];
	return ret or rowTypeColors[7];
end

function Omni.GetRowType(idx)
	return rowTypes[idx] or "unknown";
end

-- An "impulse" is an event that actually does damage or healing...
local impulseTypes = { [1] = "DamageIn", [2] = "DamageOut", [3] = "HealingIn", [4] = "HealingOut", [7] = "HealingSelf", [8] = "Healing" };
function Omni.GetImpulseType(idx)
	if not idx then return nil; end
	return impulseTypes[idx];
end

-- Determine if a row has a source/target
function Omni.RowHasActors(idx)
	return (idx == 1) or (idx == 2) or (idx == 3) or (idx == 4) or (idx == 7) or (idx == 8) or (idx == 16) or (idx == 21) or (idx == 22);
end

-- If the actor is the SOURCE for this row type, return true.
function Omni.RowActorIsSource(idx)
	return (idx == 1) or (idx == 3) or (idx == 7) or (idx == 18) or (idx == 21);
end

-- If the actor is the TARGET for this row type, return true
function Omni.RowActorIsTarget(idx)
	return (idx == 2) or (idx == 4) or (idx == 16) or (idx == 22);
end


-------------------------------------------------------------------
-- LOOKUP METADATA: EXTENDED INFO
-- (1 = miss, 2 = dodge, 3 = parry, 4 = block, 5 = resist, 6 = crit)
-- (7 = absorb, 8 = crush, 9 = glance)
-------------------------------------------------------------------
local xiTypes = {};
Omni.xiTypes = xiTypes;
local i_xiTypes = {};
Omni.i_xiTypes = i_xiTypes;

function Omni.GetXiType(idx)
	if not idx then return nil; end
	return xiTypes[idx];
end

-- The following types represent "failures"
local failTypes = {
	[1] = "miss", [2] = "dodge", [3] = "parry", [4] = "block",
	[5] = "resist", [7] = "absorb",  [10] = "immune", [11] = "reflect",
	[12] = "evade",
};
function Omni.GetFailType(idx)
	return failTypes[idx];
end

-- Determine if a row is a crit.
function Omni.IsCritRow(row) return (row.e == 6); end

--- Get the miscellaneous string from a log row.
function Omni.GetMiscString(row)
	if not row then return ""; end
	local str = "";
	if row.absorb then str = str .. "A[|cFFFFFF00" .. row.absorb .. "|r] "; end
	if row.resist then str = str .. "R[|cFFFFFF00" .. row.resist .. "|r] "; end
	if row.block then str = str .. "B[|cFFFFFF00" .. row.block .. "|r] "; end
	if row.oh then str = str .. "OH[|cFF00FF00" .. row.oh .. "|r] "; end
	return str;
end

----------------------------------------------------------------------
-- LOGGING CORE
----------------------------------------------------------------------
-- Adds a row to the Omniscience combat log.
-- y = Type
-- c = Actor (Target of outgoing or source of incoming effects) no more sigg
-- s = Source
-- r = Target
-- sg = Sourceguid
-- rg = Targetguid
-- a = Ability (Name of ability or spell)
-- b = Ability Id or spell Id
-- k = rank spell
-- x = Amount
-- d = Damage type
-- dot = dot
-- hot = hot
-- di = dist
-- e = Extension type
-- m = Mitigation table
function Omni.AddLogRow(y, s, r, sg, rg, a, b, x, d, dot, hot, dist, e, mt1, mn1, mt2, mn2, mt3, mn3)
	local n, row = #omniLog, nil;
	-- Recycle a log row if possible.
	if(n >= logCap) then row = table.remove(omniLog, 1); VFL.empty(row); else	row = {};	end
	row.t = GetTimeTenths();
	-- For damage in and healing in types, record local health
	if(y == 1) or (y == 3) or (y == 7) then
		local uh, uhm = UnitHealth("player"), UnitHealthMax("player");
		if (y ~= 1) then -- overhealing
			local oh = uh + x - uhm; if(oh >= 1) then row.oh = oh; end
		end
		row.uh = uh; row.uhm = uhm;
	end
	-- For healing out, try to compute overhealing.
	if(y == 4) then
		local tu = GetUnitByName(lower(r));
		if tu and tu:IsValid() then
			local uh, uhm = tu:Health(), tu:MaxHealth();
			local oh = uh + x - uhm; if(oh >= 1) then row.oh = oh; end
		end
	end
	
	row.k = 0;
	-- For Healing out, damage out, get rank spell if possible
	if(y == 2) or (y == 4) or (y == 7) or (y == 8) then
		local tu = GetUnitByName(lower(s));
		if tu and tu:IsValid() then
			if a and a ~= "" then
				row.k = tu.GetLastSpellRank(a) or 0;
			end
		end
	end
	
	row.y = tonumber(y);
	row.s = s;
	row.r = r;
	row.sg = sg;
	row.rg = rg;
	row.a = a;
	row.b = b;
	row.x = tonumber(x);
	row.d = tonumber(d);
	row.dot =  tonumber(dot);
	row.hot =  tonumber(hot);
	row.di =  tonumber(dist);
	row.e = tonumber(e);
	if mt1 then row[mt1] = mn1; end
	if mt2 then row[mt2] = mn2; end
	if mt3 then row[mt3] = mn3; end
	table.insert(omniLog, row);
	logTrigger:Raise(omniLog, row);
end
local AddLogRow = Omni.AddLogRow;

-- LogALL
-- Adds a row to the Omniscience combat log.
-- y = Type
-- s = Source
-- r = Target
-- a = Ability (Name of ability or spell)
-- b = Ability Id or spell Id
-- x = Amount
-- d = Damage type
-- e = Extension type
-- m = Mitigation table
function Omni.AddLogRow2(y, s, r, sg, rg, a, b, x, d, dot, hot, dist, e, mt1, mn1, mt2, mn2, mt3, mn3)
	local n, row = #omniLogAll, nil;
	-- Recycle a log row if possible.
	if(n >= logCapAll) then 
		row = table.remove(omniLogAll, 1); 
		VFL.empty(row); 
	else
		row = {};
	end
	row.t = GetTimeTenths();
	-- For damage in and healing in types, record local health
	if(y == 1) or (y == 3) or (y == 7) then
		local uh, uhm = UnitHealth("player"), UnitHealthMax("player");
		if (y ~= 1) then -- overhealing
			local oh = uh + x - uhm; if(oh >= 1) then row.oh = oh; end
		end
		row.uh = uh; row.uhm = uhm;
	end
	-- For healing out, try to compute overhealing.
	if(y == 4) or (y == 8) then
		local tu = GetUnitByName(lower(r));
		if tu and tu:IsValid() then
			local uh, uhm = tu:Health(), tu:MaxHealth();
			local oh = uh + x - uhm; if(oh >= 1) then row.oh = oh; end
		end
	end
	
	-- For Healing out, damage out, get rank spell if possible
	if(y == 2) or (y == 4) or (y == 7) or (y == 8) then
		local tu = GetUnitByName(lower(s));
		if tu and tu:IsValid() then
			row.k = tu.GetLastSpellRank(a) or 0;
		end
	end
	
	row.y = tonumber(y); 
	row.s = s;
	row.r = r;
	row.sg = sg;
	row.rg = rg;
	row.a = a;
	row.b = b;
	row.x = tonumber(x);
	row.d = tonumber(d);
	row.dot =  tonumber(dot);
	row.hot =  tonumber(hot);
	row.di =  tonumber(dist);
	row.e = tonumber(e);
	if mt1 then row[mt1] = mn1; end
	if mt2 then row[mt2] = mn2; end
	if mt3 then row[mt3] = mn3; end
	-- disable by sigg
	--table.insert(omniLogAll, row);
	logTriggerAll:Raise(omniLog, row);
end
local AddLogRow2 = Omni.AddLogRow2;

-- Affliction table for debuff counting
local afflic = {}
local function Afflicted(abil, stacks)
	if(stacks == 1) then
		AddLogRow(5, nil, nil, abil);
	else
		AddLogRow(5, nil, nil, abil .. " (" .. stacks .. ")");
	end
	afflic[abil] = true;
end
local buff = {};
local function Buffed(abil, stacks)
--	if not trackedBuffs[abil] then return; end
	if stacks == 1 then
		AddLogRow(9, nil, nil, abil);
	else
		AddLogRow(9, nil, nil, abil .. " (" .. stacks .. ")");
	end
	buff[abil] = true;
end
local function Unafflicted(abil)
	if afflic[abil] then
		afflic[abil] = nil;
		AddLogRow(6, nil, nil, abil);
	elseif buff[abil] then
		buff[abil] = nil;
		AddLogRow(10, nil, nil, abil);
	end
end
Omni._Afflicted = Afflicted; Omni._Buffed = Buffed; Omni._Unafflicted = Unafflicted;

-----------------------------------------
-- PARSING CORE
-- The following table is populated by locale-specific parsing code
-- (e.g. Parse_enUS.lua)
-----------------------------------------
Omni.ParseFuncs = {};

