-- FeatureEditor.lua
-- RDX - Raid Data Exchange
-- (C)2006 Bill Johnson
--
-- The dialog box for editing an ObjectState's features.

local dlg = nil;

function RDX.IsFeatureEditorOpen() return dlg; end

local features = {};

local fle_active = { text = i18n("Active Features"), hlt = {r=0,g=.45,b=.70} };
local fle_possible = { text = i18n("Possible Features"), hlt = {r=0,g=.45,b=.70} };
local category_color = {r=.5,g=.4,b=.35};

function RDX.FeatureEditor(state, callback, augText)
	if (dlg) or (not state) then return nil; end
	
	dlg = VFLUI.Window:new(UIParent); dlg:SetFrameStrata("FULLSCREEN");
	VFLUI.Window.SetDefaultFraming(dlg, 24);
	dlg:SetBackdrop(VFLUI.BlackDialogBackdrop);
	dlg:SetTitleColor(0,.6,0);
	dlg:SetText(i18n("Feature Editor: ") .. augText);
	dlg:SetWidth(700); dlg:SetHeight(500);
	dlg:SetPoint("CENTER", UIParent, "CENTER");
	dlg:Show();
	VFLUI.Window.StdMove(dlg, dlg:GetTitleBar());
	local ca = dlg:GetClientArea();

	-- Predeclarations
	local featList, sf, ui;
	local activeFeat;
	local RebuildFeatureList, SetActiveFeature, HideErrors, ShowErrors;

	------ Helper functions
	local function AddFeatureClick(feat)
		state:InSituState();
		local q1, q2 = state:AddFeatureInSitu(feat.CreateDescriptor(state));
		if q1 then
			RebuildFeatureList();
		else
			VFLUI.MessageBox(i18n("Error"), i18n("AddFeature failed: ") .. tostring(q2));
		end
	end

	local function ResetFeatureUI()
		if ui then 
			ui:Hide();
			sf:SetScrollChild(nil);
			ui.GetDescriptor = nil; -- BUGFIX: Remove the GetDescriptor from a feature UI before freeing it.
			ui:Destroy(); 
			ui = nil; 
		end
	end

	-- Get the currently-edited state
	function dlg:GetActiveState() return state; end

	-- Callback: save the current feature
	local function SaveActiveFeature(noFeedback)
		if (not activeFeat) or (not ui) then return; end
		-- Find the feature on the local state; if not found, fail out.
		local fd = state:_GetFeatureByIndex(activeFeat);
		if not fd then return; end
		local feat = RDX.GetFeatureByDescriptor(fd);
		if not feat then return; end
		-- Rewind our in-situ state to the given index
		state:InSituState(activeFeat);
		-- Attempt to expose the newly created feature
		local descr = ui:GetDescriptor();
		feat = RDX.GetFeatureByDescriptor(descr);
		vflErrors:Clear();
		if state:_ExposeFeatureInSitu(descr, feat, vflErrors) then
			state:_SaveFeatureByIndex(activeFeat, descr);
			local idx = activeFeat; activeFeat = nil;
			SetActiveFeature(idx);
			-- 6.4.9
			RDXDB.PaintPreviewWindow(state);
		else
			ShowErrors(vflErrors);
		end
	end

	-- Callback: select a new active feature.
	function SetActiveFeature(idx)
		-- Activate the feature only if required
		if activeFeat == idx then return; end
		-- Find the feature on the local state; if not found, fail out.
		local fd = state:_GetFeatureByIndex(idx);
		if not fd then return; end
		local feat = RDX.GetFeatureByDescriptor(fd);
		if not feat then return; end
		-- Update the active feature. Rebuild the feature list.
		activeFeat = idx;
		state:InSituState(activeFeat);
		RebuildFeatureList();
		-- Now build the feature config UI.
		ResetFeatureUI();	
		if (not feat.UIFromDescriptor) then return; end -- no UI
		ui = feat.UIFromDescriptor(fd, sf, state); if not ui then return; end
		-- Layout and show the feature config UI.
		ui.isLayoutRoot = true;
		ui:SetParent(sf); sf:SetScrollChild(ui);
		ui:SetWidth(sf:GetWidth()); 
		if ui.DialogOnLayout then ui:DialogOnLayout(); end
		ui:Show();
		-- If there are any errors, show them
		state:InSituState(activeFeat);
		vflErrors:Clear();
		if state:_ExposeFeatureInSitu(fd, feat, vflErrors) then
			HideErrors();
		else
			ShowErrors(vflErrors);
		end
	end
	function dlg:SetActiveFeature(idx) SetActiveFeature(idx); end

	-- Remove the feature at the given index. Anything that depended on this
	-- feature will be blasted as well.
	local function RemoveFeatureAt(idx)
		-- Remove the feature
		local qq = VFL.copy(state:Features());
		local x = table.remove(qq, idx); if not x then return; end
		-- Destroy the existing features
		ResetFeatureUI(); HideErrors();
		activeFeat = nil;
		state:Clear();
		-- Readd each feature one at a time
		for ix,ft in ipairs(qq) do state:AddFeatureInSitu(ft); end
		-- Rebuild the list
		RebuildFeatureList();
		-- 6.4.9
		RDXDB.PaintPreviewWindow(state);
	end

	-- Move the active feature up.
	local function MoveFeatureUp()
		if (not activeFeat) or (activeFeat < 2) then return; end
		-- Swap the features
		local qq = state:Features();
		local tmp = qq[activeFeat];
		qq[activeFeat] = qq[activeFeat - 1];
		qq[activeFeat - 1] = tmp;
		activeFeat = activeFeat - 1;
		-- Rebuild the list
		RebuildFeatureList();
	end

	-- Move the active feature down.
	local function MoveFeatureDown()
		local qq = state:Features();
		if (not activeFeat) or (activeFeat > (table.getn(qq) - 1)) then return; end
		local tmp = qq[activeFeat];
		qq[activeFeat] = qq[activeFeat + 1];
		qq[activeFeat + 1] = tmp;
		activeFeat = activeFeat + 1;
		RebuildFeatureList();
	end

	-- Export the active feature to disk at the given location.
	local function ExportFeatureTo(file)
		RDX:Debug(1, "ExportFeatureTo " .. tostring(file));
		local qq = state:Features();
		if (not activeFeat) or (not qq[activeFeat]) then return; end
		file = RDXDB.TouchObject(file);
		if not file then
			-- TODO: error message here? shouldn't really ever happen but who knows.
			return;
		end
		-- Check file integrity
		if(file.ty == "Typeless") then file.ty = "FeatureData"; file.version = 1; end
		if(file.ty ~= "FeatureData") then
			VFLUI.MessageBox("Error", i18n("Cannot overwrite a non-FeatureData object with FeatureData. Delete it first."));
			return;
		end
		-- Write the data
		file.data = VFL.copy(qq[activeFeat]);
	end

	-- Import a feature from disk, adding it to the end of the featurelist.
	local function ImportFeatureFrom(file)
		RDX:Debug(1, "ImportFeatureFrom " .. tostring(file));
		local md = RDXDB.GetObjectData(file);
		-- Sanity check.
		if(type(md) ~= "table") or (md.ty ~= "FeatureData") or (type(md.data) ~= "table") or (type(md.data.feature) ~= "string") then
			VFLUI.MessageBox(i18n("Error"), i18n("Not a valid FeatureData file.")); return;
		end
		-- Import it.
		-- We'll VFL.copy() it just to be safe.
		state:AddFeature(VFL.copy(md.data));
		RebuildFeatureList();
	end

	------ The feature list.
	featList = VFLUI.List:new(dlg, 10, VFLUI.Selectable.AcquireCell);
	featList:SetPoint("TOPLEFT", ca, "TOPLEFT");
	featList:SetWidth(200); featList:SetHeight(448);
	featList:Rebuild(); featList:Show();
	featList:SetDataSource(function(cell, data, pos)
		cell.text:SetText(data.text);
		if activeFeat and (data.idx == activeFeat) then
			cell.selTexture:SetVertexColor(0,0,0.6); cell.selTexture:Show();
		elseif data.hlt then
			cell.selTexture:SetVertexColor(data.hlt.r, data.hlt.g, data.hlt.b); cell.selTexture:Show();
		else
			cell.selTexture:Hide();
		end
		cell:SetScript("OnClick", data.OnClick);
	end, VFL.ArrayLiterator(features));

	function RebuildFeatureList()
		VFL.empty(features);
		table.insert(features, fle_active);
		local feat, qq, text;
		for idx,fd in ipairs(state:Features()) do
			local blah = idx; 
			text = "";
			state:InSituState(idx);
			local feat = RDX.GetFeatureByDescriptor(fd);
			if not feat then
				text = i18n("(Invalid!)");
			elseif not feat.IsPossible(state) then
				text = "|cFFFF0000" .. feat.title .. "|r";
			elseif not state:_ExposeFeatureInSitu(fd, feat) then
				text = "|cFFFFFF00" .. feat.title .. "|r";
			elseif feat.deprecated then
				text = "|cFFDD0077" .. feat.title .. "|r";
			else
				text = feat.title;
			end
			table.insert(features, { idx = idx, text = text, OnClick = function() SetActiveFeature(blah); end });
		end
		----------------------------------- "Possible Features" list
		table.insert(features, fle_possible);
		-- Sort the features by category, then name.
		local flist = RDX._GetFeatureArray();
		table.sort(flist, function(f1, f2)
			if(f1.category == f2.category) then
				return (f1.title < f2.title);
			else
				return (f1.category < f2.category);
			end
		end);
		-- Traverse the list in order, inserting a category-header whenever the category changes.
		local curCat = "";
		for idx,feat in ipairs(flist) do
			if state:IsFeaturePossible(feat) and (not feat.invisible) then
				if(feat.category ~= curCat) then
					table.insert(features, {text = feat.category, hlt = category_color}); curCat = feat.category;
				end
				local blah, txt = feat, feat.title;
				if feat.deprecated then txt = "|cFFDD0077" .. txt .. "|r"; end
				table.insert(features, { text = txt, OnClick = function() AddFeatureClick(blah); end });
			end
		end
		state:InSituState();
		featList:Update();
	end
	dlg.RebuildFeatureList = RebuildFeatureList;

	------ The feature config ui.
	sf = VFLUI.VScrollFrame:new(dlg);
	sf:SetWidth(470); sf:SetHeight(440);
	sf:SetPoint("TOPLEFT", featList, "TOPRIGHT");
	sf:Show();

	------ The error frame
	local el = VFLUI.List:new(dlg, 10, VFLUI.Selectable.AcquireCell);
	local function elad(cell, data)
		cell.text:SetText(data);
		if(data ~= "Errors") then
			cell.selTexture:Hide();
		else
			cell.selTexture:SetVertexColor(0.75,0,0); cell.selTexture:Show();
		end
	end;
	el:SetPoint("TOPLEFT", sf, "BOTTOMLEFT");
	el:SetWidth(470); el:SetHeight(100); el:Rebuild(); el:Hide();
	function HideErrors()
		sf:SetHeight(440); el:Hide();
	end
	function ShowErrors(err)
		sf:SetHeight(340); el:Show();
		local ec, et = err:Count(), err:ErrorTable();
		el:SetDataSource(elad, function() return ec + 1; end, function(x)
			if(x == 1) then return "Errors"; else return et[x-1]; end
		end);
	end

	------ Save/revert buttons
	local btnSave = VFLUI.Button:new(dlg);
	btnSave:SetHeight(25); btnSave:SetWidth(65);
	btnSave:SetPoint("BOTTOMRIGHT", ca, "BOTTOMRIGHT");
	btnSave:SetText(i18n("Save")); btnSave:Show();
	btnSave:SetScript("OnClick", SaveActiveFeature);

	local btnRevert = VFLUI.Button:new(dlg);
	btnRevert:SetHeight(25); btnRevert:SetWidth(65);
	btnRevert:SetPoint("RIGHT", btnSave, "LEFT");
	btnRevert:SetText(i18n("Revert")); btnRevert:Show();
	btnRevert:SetScript("OnClick", function()
		local ix = activeFeat;
		if ix then
			SetActiveFeature(nil); SetActiveFeature(ix);
		end
	end);

	local btnRemove = VFLUI.Button:new(dlg);
	btnRemove:SetHeight(25); btnRemove:SetWidth(65);
	btnRemove:SetPoint("RIGHT", btnRevert, "LEFT");
	btnRemove:SetText(i18n("Remove")); btnRemove:Show();
	btnRemove:SetScript("OnClick", function()
		local ix = activeFeat;
		if ix then
			SetActiveFeature(nil); RemoveFeatureAt(ix);
		end
	end);

	local btnMoveUp = VFLUI.Button:new(dlg);
	btnMoveUp:SetHeight(25); btnMoveUp:SetWidth(65);
	btnMoveUp:SetPoint("BOTTOMLEFT", ca, "BOTTOMLEFT");
	btnMoveUp:SetText(i18n("Up")); btnMoveUp:Show();
	btnMoveUp:SetScript("OnClick", MoveFeatureUp);

	local btnMoveDown = VFLUI.Button:new(dlg);
	btnMoveDown:SetHeight(25); btnMoveDown:SetWidth(65);
	btnMoveDown:SetPoint("LEFT", btnMoveUp, "RIGHT");
	btnMoveDown:SetText(i18n("Down")); btnMoveDown:Show();
	btnMoveDown:SetScript("OnClick", MoveFeatureDown);

	local btnImport = VFLUI.Button:new(dlg);
	btnImport:SetHeight(25); btnImport:SetWidth(65);
	btnImport:SetPoint("LEFT", btnMoveDown, "RIGHT");
	btnImport:SetText(i18n("Import")); btnImport:Show();
	btnImport:SetScript("OnClick", function()
		local xp = RDXDB.ExplorerPopup(this);
		xp:SetPoint("BOTTOMLEFT", this, "TOPLEFT"); xp:Show();
		xp:SetFileFilter(function(_,_,md) return (md.ty == "FeatureData"); end); xp:Rebuild();
		xp:EnableFeedback(function(zz) ImportFeatureFrom(zz:GetPath()); end);
	end);

	local btnExport = VFLUI.Button:new(dlg);
	btnExport:SetHeight(25); btnExport:SetWidth(65);
	btnExport:SetPoint("LEFT", btnImport, "RIGHT");
	btnExport:SetText(i18n("Export")); btnExport:Show();
	btnExport:SetScript("OnClick", function()
		local xp = RDXDB.ExplorerPopup(this);
		xp:SetPoint("BOTTOMLEFT", this, "TOPLEFT"); xp:Show();
		xp:SetFileFilter(VFL.True); xp:Rebuild();
		xp:EnableFeedback(function(zz) ExportFeatureTo(zz:GetPathRaw()); end);
	end);


	------ Close procedure
	dlg.Destroy = VFL.hook(function(s)
		-- Remove exposed API
		s.SetActiveFeature = nil; s.RebuildFeatureList = nil;
		s.GetActiveState = nil;

		-- Tear down controls
		ResetFeatureUI();
		featList:Destroy(); featList = nil;
		el:Destroy(); el = nil;
		btnSave:Destroy(); btnSave = nil;	btnRevert:Destroy(); btnRevert = nil;
		btnRemove:Destroy(); btnRemove = nil;	btnMoveUp:Destroy(); btnMoveUp = nil;
		btnMoveDown:Destroy(); btnMoveDown = nil;	btnImport:Destroy(); btnImport = nil;
		btnExport:Destroy(); btnExport = nil;
		sf:Destroy(); sf = nil;
	end, dlg.Destroy);

	local function Close()
		dlg:Destroy(); dlg = nil;
		if callback then callback(state); end
	end

	local closebtn = VFLUI.CloseButton:new()
	closebtn:SetScript("OnClick", Close);
	dlg:AddButton(closebtn);

	RebuildFeatureList();

	return dlg;
end
