-- LogisticsWindow.lua
-- RDX - Raid Data Exchange
-- (C)2006 Raid Informatics
--
-- THIS FILE CONTAINS COPYRIGHTED CONTENT SUBJECT TO THE TERMS OF A SEPARATE
-- LICENSE. UNLICENSED COPYING IS PROHIBITED.
--
-- General-purpose windows for displaying results from logistics queries.

------------------ Logistics window support routines
local function LW_Sort(self, fld)
	local tbl, nl = self._dataList, self._nameMap;
	self.sortField = fld;
	table.sort(tbl, function(x1, x2)
		local nlx1, nlx2 = nl[x1], nl[x2];
		if nlx1 and nlx2 then
			return nlx1[fld] < nlx2[fld];
		elseif nlx2 then
			return true;
		elseif nlx1 then
			return nil;
		end
	end);
	self:RepaintAll();
end

local function LW_SortGT(self, fld)
	local tbl, nl = self._dataList, self._nameMap;
	self.sortField = fld;
	table.sort(tbl, function(x1, x2)
		local nlx1, nlx2 = nl[x1], nl[x2];
		if nlx1 and nlx2 then
			return nlx1[fld] > nlx2[fld];
		elseif nlx2 then
			return true;
		elseif nlx1 then
			return nil;
		end
	end);
	self:RepaintAll();
end

function LW_SetupNameMap(self, augmentor)
	if not augmentor then augmentor = VFL.Noop; end

	-- Name data map
	local nameMap, totalCount = {}, 0;
	for _,unit in RDX.Group() do
		totalCount = totalCount + 1;
		local t = { 
			name = unit:GetProperName(); 
			class = unit:GetClassID(); 
		};
		nameMap[unit.name] = t;
		augmentor(self, t);
	end
	self._nameMap = nameMap; self.totalCount = totalCount;

	-- Paint entry map
	local paintMap = self._dataList;
	VFL.empty(paintMap);
	for name,_ in pairs(nameMap) do	table.insert(paintMap, name);	end
end

--- Create a new logistics window with the given ID and title.
-- Return a handler to the window.
function RDX.CreateLogisticsWindow(id, title, periodicRepaint, totalWidth, t1width)
	totalWidth = totalWidth or 150;
	t1width = t1width or 70;
	local state = RDX.GenericWindowState:new();
	local win = RDX.Window:new(RDXParent);

	-- The data source will pull directly from a client-maintained data list.
	local dataList = {};
	state:AddSlot("DataSourceIterator");
	state:_SetSlotFunction("DataSourceIterator", function() return ipairs(dataList); end);
	state:AddSlot("DataSourceSize");
	state:_SetSlotFunction("DataSourceSize", function() return table.getn(dataList); end);
	-- ApplyData invokes a user provided function
	local applyData = VFL.Noop;
	state:AddSlot("_ApplyData");
	state:_SetSlotFunction("_ApplyData", function(frame, icv, member)
		applyData(win, frame, icv, member);
	end);
	function win:SetApplyData(fn) applyData = fn; end

	-- Features
	state:AddFeature({feature = "Frame: Lightweight", title = title, titleColor = {r=0,g=0,b=0,a=1}, bkdColor = {r=0,g=.15,b=.15,a=1}});
	state:AddFeature({feature = "Generic Subframe", w = totalWidth, h = 14, tdx = t1width});
	state:AddFeature({feature = "Grid Layout", cols = 2, axis = 1, dxn = 1});
	if periodicRepaint then
		state:AddFeature({feature = "Event: Periodic Repaint", interval=0.2, slot = "RepaintAll"});
	end

	-- Create the window.
	win:LoadState(state);
	win.RepaintAll = state:GetSlotFunction("RepaintAll");
	state = nil;
	win._dataList = dataList;
	win:GetClientArea():SetPoint("LEFT", UIParent, "LEFT", 50, 0);
	VFLUI.Window.StdMoveICW(win, win:GetTitleBar());

	local closebtn = VFLUI.CloseButton:new();
	closebtn:SetScript("OnClick", function() win:Destroy(); end);
	win:AddButton(closebtn);

	-- Prep methods
	win.Sort = LW_Sort; win.SortGT = LW_SortGT; win.SetupNameMap = LW_SetupNameMap;

	-- Destructor code
	win.Destroy = VFL.hook(function(s)
		dataList = nil;
		s._rpcid = nil; s._WindowMenu = nil;
		s.sortField = nil; 
		s._nameMap = nil; s.totalCount = nil; s._dataList = nil; 
		s.Sort = nil; s.SortRev = nil; s.SetupNameMap = nil;
		s.SetApplyData = nil; s.RepaintAll = nil;
	end, win.Destroy);

	return win;
end

