-- Encounters.lua
-- RDX - Raid Data Exchange
-- (C)2006 Bill Johnson
--
-- The RDX encounter system.

---------------------------------------------
-- Encounter internal variables
---------------------------------------------
local encTimer = VFL.CountUpTimer:new();
local encPane = nil;
local mouseoverMap = {};

---------------------------------------------
-- ENCOUNTER DATABASE
---------------------------------------------
local edb, ce = {}, nil;

-- Encounter category menus.
local ecats = {};
ecats[""] = {};
local _submenu_color = {r=0.2, g=0.9, b=0.9};
local function EncounterCatMenu(menu, cell, category)
	if not ecats[category] then menu:Release(); return; end
	menu:Expand(cell, ecats[category]);
end

local function MakeCatHierarchy(cat)
	if not ecats[cat] then
		local _,_,pcat,thiscat = string.find(cat, "^(.*)%/([^%/]*)$");
		if not pcat then pcat = ""; thiscat = cat; end
		if not ecats[pcat] then MakeCatHierarchy(pcat); end 
		table.insert(ecats[pcat], 1, {
			text = thiscat; color = _submenu_color; isSubmenu = true;
			sort = -1;
			OnClick = function() EncounterCatMenu(VFL.poptree, this, cat); end;
		});
		ecats[cat] = {};
	end
end

--- Register an encounter in the RDX encounter database.
-- @field name The name of the encounter.
-- @field title A text title for the encounter.
-- @field category A string category for the encounter. Hierarchical subcategories can be added by using "/" separators.
-- @field sort An integer used in table.sort to sort the encounter list.
function RDX.RegisterEncounter(tbl)
	if type(tbl) ~= "table" then error("expected table"); end
	if type(tbl.name) ~= "string" then error("bad or missing encounter name"); end
	if type(tbl.title) ~= "string" then error("bad or missing encounter title"); end
	if type(tbl.category) ~= "string" then tbl.category = ""; end

	local name = tbl.name;

	if edb[name] then
		-- Duplicate encounter registration.
		-- Check if it's the active encounter...
		if ce and (ce == edb[name]) then
			-- Stop and deactivate the encounter if it is the current encounter.
			RDX:Debug(1, "Replacing encounter <", name, "> in situ");
			RDX.StopEncounter(true); encTimer:Reset();
			if ce.DeactivateEncounter then ce.DeactivateEncounter(name); end
			-- Update the data
			VFL.copyOver(ce, tbl);
			-- Reactivate the encounter on the next frame.
			-- This gives the dynamic bossmods time to initialize themselves.
			encPane:SetInfoBundle(ce.title, "", 1);
			if ce.ActivateEncounter then
				VFL.schedule(.1, ce.ActivateEncounter, name, name);
			end
		else
			-- Just update the data
			VFL.copyOver(edb[name], tbl);
		end
	else
		-- Non-duplicate encounter
		-- Add the data
		edb[name] = tbl;
		-- Categorize the encounter; find its parent category.
		local cat = tbl.category;
		-- Create category related structures
		MakeCatHierarchy(cat);
		-- Add it to the menu for its category
		table.insert(ecats[cat], {
			text = tbl.title;
			name = name;
			sort = tonumber(tbl.sort) or 0;
			OnClick = function() VFL.poptree:Release(); RDX.SetActiveEncounter(name); end
		});
		table.sort(ecats[cat], function(x1,x2) return x1.sort < x2.sort; end);
	end
end

--- Remove an encounter previously added with RegisterEncounter.
function RDX.UnregisterEncounter(name)
	-- Sanity checks.
	if (name == "default") or (not edb[name]) then return; end;
	local cat = edb[name].category;

	-- swap to default if we're trying to unregister the current encounter
	if ce.name == name then RDX.SetActiveEncounter("default", true); end
	
	-- discard it from the menu
	VFL.filterInPlace(ecats[cat], function(x)
		if x.name == name then return false; else return true; end
	end);
	-- resort
	table.sort(ecats[cat], function(x1,x2) return x1.sort < x2.sort; end);

	-- remove from mouseover map
	for k,v in pairs(mouseoverMap) do if v == name then mouseoverMap[k] = nil; end	end

	-- remove from the database
	edb[name] = nil;
end


-- The default encounter
RDX.RegisterEncounter({name = "default"; title = "Default";});

--- Get the name of the currently-active encounter
function RDX.GetActiveEncounter()
	if not ce then return "default"; end
	return ce.name;
end

--- Change the currently-active encounter.
function RDX.SetActiveEncounter(en, nosync)
	if not en then error("Usage: RDX.SetActiveEncounter(encName, nosync)"); end
	if ce and ce.name == en then
		-- already set to this encounter
		return;
	end
	-- Check the new encounter
	RDX:Debug(2, "RDX.SetActiveEncounter(" .. en .. ")");
	if not edb[en] then
		RDX:Debug(1, "SetActiveEncounter(".. en .. "): invalid encounter");
		return nil;
	end
	-- Stop the existing encounter
	RDX.StopEncounter(nosync); encTimer:Reset();
	-- Deactivate the old encounter
	local olden = nil;
	if ce then 
		olden = ce.name;
		if ce.DeactivateEncounter then ce.DeactivateEncounter(olden); end
	end
	-- Switch to the new encounter.
	ce = edb[en];
	-- Update encounter pane
	encPane:SetInfoBundle(ce.title, "", 1);
	-- Invoke encounter's activate function
	if ce.ActivateEncounter then ce.ActivateEncounter(en, olden); end
	RDXEvents:Dispatch("ENCOUNTER_CHANGED", en, olden);
	-- Save the pref
	RDXU.active_encounter = en;
	-- Sync
	if (not nosync) then RDX.SyncEncounter(); end
	return true;
end

-----------------------------------------------
-- ENCOUNTER SYNC
-----------------------------------------------
--- Synchronize the current encounter
function RDX.SyncEncounter()
	if RDXPlayer:IsLeader() then
		RPC_Group:Flash("enc_set", RDX.GetActiveEncounter());
	end
end

RPC_Group:Bind("enc_set", function(ci, en) 
	if RPC.IsGroupLeader(ci) then
		RDX.SetActiveEncounter(en, true); 
	end
end);

---------------------------------------------
-- START/STOP
---------------------------------------------
local playbtn = VFLUI.AcquireFrame("Button");
local phtex = VFLUI.CreateTexture(playbtn);
phtex:SetAllPoints(playbtn);
phtex:Show();
playbtn:SetHighlightTexture(phtex);
phtex:SetBlendMode("DISABLE");

local function PlayBtn_SetPlay()
	if RDX._skin == "boomy" then
		playbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\boomy\\play");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\play");
	elseif RDX._skin == "kids" then
		playbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\kids\\click-n-run");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\click-n-run");
	else
		playbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\play");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\play");
	end
	phtex:SetVertexColor(0, 0.8, 0);
end
local function PlayBtn_SetStop()
	if RDX._skin == "boomy" then
		playbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\boomy\\stop");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\boomy\\stop");
	elseif RDX._skin == "kids" then
		playbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\kids\\agt-resume");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\kids\\agt-resume");
	else
		playbtn:SetNormalTexture("Interface\\Addons\\RDX\\Skin\\stop");
		phtex:SetTexture("Interface\\Addons\\RDX\\Skin\\stop");
	end
	phtex:SetVertexColor(0.8, 0, 0);
end

PlayBtn_SetPlay();

--- Stop the encounter if it's running, or start it if it isn't.
function RDX.StartOrStopEncounter()
	if encTimer:IsRunning() then RDX.StopEncounter(); else RDX.StartEncounter(); end
end

-- On click of the play button, start or stop
playbtn:SetScript("OnClick", RDX.StartOrStopEncounter);

--- Return TRUE iff the current encounter is running, nil otherwise.
function RDX.EncounterIsRunning()
	return encTimer:IsRunning();
end

--- Start the current encounter.
function RDX.StartEncounter(nosync)
	local ename = RDX.GetActiveEncounter();
	if encTimer:IsRunning() then return; end
	if (not nosync) and (RDXPlayer:IsLeader()) then
		RPC_Group:Flash("enc_start", ename);
	end
	PlayBtn_SetStop(); encTimer:Reset(); encTimer:Start();
	-- Inform the encounter that it's been started.
	if ce and ce.StartEncounter then
		ce.StartEncounter();
	end
	RDXEvents:Dispatch("ENCOUNTER_STARTED", ename);
end
-- Remote start request hdlr
RPC_Group:Bind("enc_start", function(ci, enc)
	if RPC.IsGroupLeader(ci) and (enc == RDX.GetActiveEncounter()) then 
		RDX.StartEncounter(true); 
	end
end);

--- Stop the current encounter.
function RDX.StopEncounter(nosync)
	local ename = RDX.GetActiveEncounter();
	PlayBtn_SetPlay();
	-- Synchronize encounter stop
	if (not nosync) and (RDXPlayer:IsLeader()) then
		RPC_Group:Flash("enc_stop", ename);
	end
	-- Stop the encounter if it was started.
	if(encTimer:IsRunning()) then
		encTimer:Stop();
		if ce and ce.StopEncounter then
			ce.StopEncounter();
		end
		RDXEvents:Dispatch("ENCOUNTER_STOPPED", ename);
	end
end
-- Remote stop request hdlr
RPC_Group:Bind("enc_stop", function(ci, enc)
	if RPC.IsGroupLeader(ci) and (enc == RDX.GetActiveEncounter()) then 
		RDX.StopEncounter(true); 
	end
end);

----------------------------------------
-- MOUSEOVER ENCOUNTER SWITCHING
----------------------------------------
function RDX.RegisterMouseoverEncounterTrigger(target, encid)
	if type(target) ~= "string" or type(encid) ~= "string" then return; end
	mouseoverMap[target] = encid;
end

WoWEvents:Bind("UPDATE_MOUSEOVER_UNIT", nil, function()
	local n = UnitName("mouseover"); if not n then return; end
	if UnitHealth("mouseover") < 2 then return; end -- Don't proc off dead bosses...
	if mouseoverMap[n] then
		RDX.SetActiveEncounter(mouseoverMap[n]);
	end
end);


----------------------------------------
-- ENCOUNTER MENU
----------------------------------------
-- Show the dropdown encounter menu
local function ShowEncounterMenu(frame)
	-- Show the menu
	VFL.poptree:Begin(160, 14, frame);
	EncounterCatMenu(VFL.poptree, nil, "")
end

----------------------------------------
-- THE MAIN ENCOUNTER PANE
----------------------------------------
local function CreateEncPane()
	local s = VFLUI.AcquireFrame("Frame");
	s:SetParent("RDXParent"); s:Hide();
	s:SetHeight(45); s:SetWidth(230); s:SetMovable(true);
	s:SetBackdrop(VFLUI.BorderlessDialogBackdrop);

	-- Divider
	local tx1 = VFLUI.CreateTexture(s);
	tx1:SetDrawLayer("ARTWORK");
	tx1:SetTexture("Interface\\TradeSkillFrame\\UI-TradeSkill-SkillBorder");
	tx1:SetTexCoord(0.1, 0.5, 0, 0.25);
	tx1:SetPoint("TOPLEFT", s, "TOPLEFT", 2, -17);
	tx1:SetPoint("BOTTOMRIGHT", s, "TOPRIGHT", -2, -25);
	tx1:Show();

	-- Topbar
	local tBtn = VFLUI.AcquireFrame("Button");
	tBtn:SetParent(s); tBtn:SetFrameLevel(5);
	tBtn:SetHeight(18);
	tBtn:SetPoint("TOPLEFT", s, "TOPLEFT", 2, -2);
	tBtn:Show();

	local tBar = VFLUI.AcquireFrame("StatusBar");
	tBar:SetFrameLevel(10);
	tBar:EnableKeyboard(nil); tBar:EnableMouse(nil); tBar:EnableMouseWheel(nil);
	tBar:SetParent(s); tBar:SetAllPoints(tBtn);
	tBar:SetMinMaxValues(0,1); tBar:SetValue(1);
	tBar:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar");
	tBar:Show();

	local tTxtLeft = VFLUI.CreateFontString(tBtn);
	tTxtLeft:SetFontObject(Fonts.Default); tTxtLeft:SetHeight(12);
	tTxtLeft:SetJustifyH("LEFT");
	tTxtLeft:SetPoint("LEFT", tBtn, "LEFT");
	tTxtLeft:Show();

	local tTxtRight = VFLUI.CreateFontString(tBtn);
	tTxtRight:SetFontObject(Fonts.Default10); tTxtRight:SetHeight(10);
	tTxtRight:SetJustifyH("RIGHT");
	tTxtRight:SetPoint("RIGHT", tBtn, "RIGHT");
	tTxtRight:Show();

	function s:GetInfoBundle()
		return tTxtLeft, tTxtRight, tBar, tBtn;
	end

	function s:SetInfoBundle(ltxt, rtxt, frac, c0, c1)
		if not frac then frac=1; end
		if not c0 then c0 = _blue; end
		if not c1 then c1 = c0; end
		tTxtLeft:SetText(ltxt);
		tTxtRight:SetText(rtxt);
		frac = VFL.clamp(frac, 0, 1);
		tBar:SetValue(frac);
		tempcolor:blend(c0, c1, frac);
		tBar:SetStatusBarColor(explodeColor(tempcolor));
	end

	-- Timer widget
	local tmr = RDXUI.TimerWidget:new(s);
	tmr:SetPoint("TOPLEFT", s, "TOPLEFT", 5, -25);
	tmr:SetWidth(70); tmr:SetHeight(16); tmr:Show();
	tmr:SetScript("OnUpdate", function() this:SetTime(encTimer:Get()); end);

	-- Click handlers
	tBtn:SetScript("OnMouseDown", function()
		if(arg1 == "LeftButton") and IsShiftKeyDown() then
			encPane:WMDrag();
		end
	end);
	tBtn:SetScript("OnMouseUp", function()
		if(arg1 == "LeftButton") then 
			encPane:WMStopDrag();
		elseif(arg1 == "RightButton") then
			ShowEncounterMenu(this);
		end
	end);

	-- Toolbar
	local tbw, lastBtn, btns = 0, nil, {};

	function s:AddToolbarButton(btn, autoWidth)
		btn:SetParent(self);
		btn:SetHeight(16); if autoWidth then btn:SetWidth(16); end
		if lastBtn then
			btn:SetPoint("LEFT", lastBtn, "RIGHT");
		else
			btn:SetPoint("LEFT", tmr, "RIGHT");
		end
		lastBtn = btn;
		table.insert(btns, btn);
		tbw = tbw + btn:GetWidth();
		btn:Show();
		self:SetWidth(math.max(230, tbw + 74));
	end

	local function Layout()
		tBtn:SetWidth(math.max(s:GetWidth() - 4, 0));
		tTxtRight:SetWidth(50);
		tTxtLeft:SetWidth(s:GetWidth() - 50);
	end
	s:SetScript("OnShow", Layout);
	s:SetScript("OnSizeChanged", Layout);

	s.Destroy = VFL.hook(function(s2)
		VFLUI.ReleaseRegion(tx1); tx1 = nil;
		VFLUI.ReleaseRegion(tTxtLeft); tTxtLeft = nil;
		VFLUI.ReleaseRegion(tTxtRight); tTxtRight = nil;
		tBar:Destroy(); tBar = nil;
		tBtn:Destroy(); tBtn = nil;
		for _,b in btns do b:Destroy(); end; btns = nil; lastBtn = nil;
		s.AddToolbarButton = nil; s.GetInfoBundle = nil;
	end, s.Destroy);

	s:AddToolbarButton(playbtn, true);

	return s;
end

-- Create the encounter pane
encPane = CreateEncPane();

function RDX.GetEncounterPane() return encPane; end

--- Add a button to the RDX Encounter Pane toolbar. If the setWidth argument
-- is true, autosets the width of the button.
function RDX.AddToolbarButton(btn, setWidth)
	encPane:AddToolbarButton(btn, setWidth);
end

------------------------
-- "MINIMIZED" RDX ICON
------------------------
local mini = VFLUI.AcquireFrame("Button");
mini:SetParent(UIParent); mini:SetScale(Minimap:GetEffectiveScale() / UIParent:GetEffectiveScale()); mini:SetMovable(true);
mini:SetPoint("CENTER", UIParent, "CENTER");
mini:SetHighlightTexture("Interface\\Minimap\\UI-Minimap-ZoomButton-Highlight");
mini:SetHeight(32); mini:SetWidth(32); mini:Show();
local tx1 = VFLUI.CreateTexture(mini);
tx1:SetPoint("TOPLEFT", mini, "TOPLEFT"); tx1:SetWidth(56); tx1:SetHeight(56);
tx1:SetDrawLayer("OVERLAY");
tx1:SetTexture("Interface\\Minimap\\MiniMap-TrackingBorder"); tx1:Show();
local tx2 = VFLUI.CreateTexture(mini);
tx2:SetPoint("CENTER", mini, "CENTER"); tx2:SetHeight(18); tx2:SetWidth(18);
tx2:SetDrawLayer("BACKGROUND");
tx2:SetTexture("Interface\\Addons\\RDX\\Skin\\menu"); tx2:Show();
mini:Hide();

function RDX.Minimize()
	if RDXU.minimized then return; end
	RDXU.minimized = true;
	encPane:Hide(); mini:Show();
end
function RDX.Maximize()
	if not RDXU.minimized then return; end
	RDXU.minimized = nil;
	encPane:Show(); mini:Hide();
end
local function restoreMiniState()
	if RDXU.minimized then encPane:Hide(); mini:Show(); else encPane:Show(); mini:Hide(); end
end

local mmvg = nil;
mini:SetScript("OnMouseDown", function()
	if(arg1 == "LeftButton") then
		if IsShiftKeyDown() then
			mmvg = true;
			mini:StartMoving();
			return;
		end
	end
end);
mini:SetScript("OnMouseUp", function()
	if mmvg then
		mmvg = nil;
		mini:StopMovingOrSizing();
		local l,t = GetUniversalBoundary(mini);
		RDXU.iconL = l; RDXU.iconT = t;
		return;
	end
	if(arg1 == "LeftButton") then 
		RDX.Maximize(); return; 
	elseif(arg1 == "RightButton") then
		RDX.ShowMainMenu(); return;
	end
end);

RDXEvents:Bind("USER_RESET_UI", nil, function()
	RDXU.iconL = nil; RDXU.iconT = nil;
	mini:ClearAllPoints();
	mini:SetPoint("CENTER", UIParent, "CENTER");
end);

------------------------
-- INIT
------------------------
-- After the window manager is loaded, we can register the encpane and show it
RDXEvents:Bind("INIT_WM", nil, function()
	-- Restore the last selected encounter.
	-- If the encounter is invalid, revert to default.
	if (not RDXU.active_encounter) or (not edb[RDXU.active_encounter]) then 
		RDXU.active_encounter = "default"; 
	end
	RDX.SetActiveEncounter(RDXU.active_encounter, true);

	-- Restore positioning info for the RDX frames.
	RDXWM.ManageWindow(encPane, "_root");
	if RDXU.iconL then
		mini:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", GetLocalCoords(mini, RDXU.iconL, RDXU.iconT));
	end
	restoreMiniState();
end);

