--Base Class Implementation
--The BaseClass class is an abstract (not really, actually, it could be inherited from if someone really wanted) class all other modules inherit from

local GetMacroIconInfo = GetMacroIconInfo
local GetNumMacroIcons = GetNumMacroIcons
local PlaySoundFile = PlaySoundFile
local IsAddOnLoaded = IsAddOnLoaded
local pcall = pcall
local pairs = pairs
local select = select
local next = next
local random = math.random
local GetTime = GetTime
local join = string.join
local sub = string.sub
local gsub = string.gsub
local match = string.match
local gmatch = string.gmatch
local trim = string.trim
local utf8sub = string.utf8sub
local LibStub = LibStub
local setmetatable = setmetatable
local type = type

setfenv(1, RBM.FunctionEnviroment)
local L = L

local BaseClass = BaseClass

LibStub("AceEvent-3.0"):Embed(BaseClass)
LibStub("AceTimer-3.0"):Embed(BaseClass)

local Core = Core

--#AUTODOC_NAMESPACE BaseClass

--Due to the public nature of all methods and members in Lua I've resorted to using 
--local tables and functions to keep them internal at the cost of slightly higher memory usage


--=================================================================--
--[[=================Object Information/Accessors====================]]--
--=================================================================--

--[[
Returns:
	*string - The name of the module
]]
function BaseClass:GetName()
	return self.Name
end

--[[
Notes:
	*Used internally to prevent conflicts between different types of modules with the same name
Returns:
	*string - Name prefixed by the module type
]]
function BaseClass:GetPrefixedName()
	return self:GetType() .. self:GetName()
end


--[[
Arguments:
	string - Type of module to check against
Returns:
	*boolean - True if the module is of the type passed in
Example:
	if module:IsType("Plugin") then
]]
function BaseClass:IsType(type)
	return self.Type == type
end

--[[
Returns:
	*string - The type of the module
]]
function BaseClass:GetType()
	return self.Type
end



--=================================================================--
--[[======================Object State Control=======================]]--
--=================================================================--

do
	local init, enabled = {}, {}
	--[[
	Notes:
		*Enables a module if it is currently disabled
		*Calls :OnEnable if available
	]]
	function BaseClass:Enable()
		if not enabled[self] then
			enabled[self] = true
			Profile.ModuleStates[self:GetPrefixedName()] = true
			self:OnEnable()
		end
	end
	--[[
	Arguments:
		[optional]boolean - If true the module's disabled state will not be saved (for when the core goes into standby)(default: false)
	Notes:
		*Disables a module, causing all bars to be canceled and mixins to be disabled
		*Calls :OnDisable if available
	]]
	function BaseClass:Disable(standby)
		if enabled[self] then
			enabled[self] = false

			self:CancelBars()
			if not standby then
				Profile.ModuleStates[self:GetPrefixedName()] = false
			end
			self:OnDisable()
		end
	end
	--[[
	Notes:
		*Toggles a module from enabled to disabled, or vice versa
	]]
	function BaseClass:Toggle()
		if self:IsEnabled() then
			self:Disable()
		else
			self:Enable()
		end
	end
	--Initializes a module, should only be called by the core, never directly
	--Calls :OnInitialize is available
	--#NODOC
	function BaseClass:Initialize()
		if not init[self] then
			init[self] = true
			enabled[self] = false

			self:OnInitialize()
		end
	end

	--[[
	Returns:
		*boolean - True if the module is currently enabled
	]]
	function BaseClass:IsEnabled()
		return enabled[self]
	end

	--[[
	Returns:
		*boolean - True if the module is currently disabled
	]]
	function BaseClass:IsDisabled()
		return not enabled[self]
	end

	--[[
	Returns:
		*boolean - True if the module has been initialized
	]]
	function BaseClass:IsInitialized()
		return init[self]
	end
end


--Called after the module is loaded and savedVars are available
BaseClass.OnInitialize = Noop

--When module is toggled inactive this method is called after all bars are canceled
BaseClass.OnDisable = Noop

--When module is toggled active this method is called and if the module is a bossmod its events are registered
BaseClass.OnEnable = Noop


--=================================================================--
--[[=======================Utility Functions=========================]]--
--=================================================================--

--[[
Arguments:
	tuple - Pairs of strings and booleans where the string is the label and the boolean is the default value
Notes:
	*Registers quick command toggles, their values will be saved between sessions
	*Use :CheckToggle(label) to get its current value
	*For best memory usage call during :OnInitialize
Example:
	*self:RegisterToggles("Color by Class", true, "Color by Category", false)
	*if self:CheckToggle("Color by Class") then
	*	--Color By Class
	*elseif self:CheckToggle("Color by Category") then
	*	--Color By Category
	*else
	*	--No color
	*end
]]
do
	local regLst = {}
	local cancelTimer
	function BaseClass:RegisterToggles(...)
		local t
		if self.db then
			if not self.db.profile.Toggles then
				self.db.profile.Toggles = {}
			end
			t = self.db.profile.Toggles
		else
			if not self._Toggles then
				self._Toggles = {}
			end
			t = self._Toggles
		end
		for i=1, select("#", ...), 2 do
			local key, state = select(i, ...)
			if type(key) == "string" then
				if t[key] == nil then
					t[key] = not not state
				end
				if not regLst[self:GetName()] then
					regLst[self:GetName()] = {}
				end
				regLst[self:GetName()][key] = true
			end
		end
		self:CancelTimer(cancelTimer, true)
		cancelTimer = self:ScheduleTimer("ClearToggleList", 0)
	end
	--#NO_DOC DO NOT CALL! INTERNAL USE ONLY!
	function BaseClass:ClearToggleList()
		for name in pairs(self.db.profile.Toggles) do
			if not regLst[self:GetName()][name] then
				self.db.profile.Toggles[name] = nil
			end
		end
		regLst[self:GetName()] = nil
		self.ConfigLoaded = true
		if IsAddOnLoaded("RBM_Config") then
			Core:RefreshAndReturnConfig()
		end
	end
end

--[[
Returns:
	*table|nil - A table describing the contents of the toggles registered to this module if available, nil otherwise
Notes:
	*The toggles table is described in the format {ToggleName = defaultBoolean}
	*Use :CheckToggle(ToggleName) to get the current state of the toggle as the table only describes the default
]]
function BaseClass:GetToggles()
	return self.db and self.db.profile.Toggles
end

--[[
Arguments:
	tuple - Pairs of strings and functions where the string is the label and the function is the execute called when it's clicked
Notes:
	*Registers quick command executes
	*Only used for simple executes, for more control add them directly to your options table
	*The module object is passed as the first and only argument to the function
Example:
	*self:RegisterExecutes("Click me!", function(self) self:Alert("I was clicked!") end)
]]
function BaseClass:RegisterExecutes(...)
	local t = self._Executes or {}
	for i=1, select("#", ...), 2 do
		local key, func = select(i, ...)
		if type(key) == "string" and type(func) == "function" then
			t[key] = func
		end
	end
	self._Executes = t
end


--[[
Returns:
	*table|nil - A table describing the executes registered to this module if available, nil otherwise
Notes:
	*The execute table is described in the format {ExecuteName = functionReference}
]]
function BaseClass:GetExecutes()
	return self._Executes
end

--[[
Arguments:
	[optional]boolean - If true the quick commands or not shown. (default: false)
Notes:
	*When set to true quick toggles and executes or not shown in the GUI
]]
function BaseClass:SetHideQuickCommands(state)
	self._HideQuickCommands = not not state
end

--[[
Returns:
	*boolean - True if the quick commands are hidden
]]
function BaseClass:GetHideQuickCommands()
	if self:IsBossMod() then
		return self:IsBossDead() or not self:IsEnabled()
	end
	return self._HideQuickCommands or false
end

do
	local trimString = setmetatable({}, {__index = function(self, key)
		local name = key
		local abrevname = ""
		if name:match("[%(%)]") then
			name = name:trim(name:gsub(name:gsub("%b()", ""), "  +"), " ")
		end
		
		for word in name:gmatch("([^ ]+)") do
			local firstlet, secondlet = word:utf8sub(1, 1), word:utf8sub(2)
			if secondlet:match(":") then
				abrevname = abrevname..firstlet..":"
			else
				abrevname = abrevname..firstlet
			end
		end
		self[key] = abrevname
		return abrevname
	end,})

	--[[
	Arguments:
		tuple - String to be trimmed.
	Returns:
		*string - The trimmed string
	Notes:
		*Used to trim strings between modules.
	Example:
		*spellname = self:TrimString("Silly Spell One")
	]]
	function BaseClass:TrimString(...)
		return trimString[...]
	end

	--[[
	Notes:
		*Used to clear the trimmed string table
	]]

	function BaseClass:ClearTrimTable()
		for i,v in pairs(trimString) do
			trimString[i] = nil
		end
	end
end
--[[
Arguments:
	tuple - Strings to be joined into a label
Returns:
	*string - The label
Notes:
	*Used to create standard labels across modules
Example:
	*local Onyxia_WingBuffet = module:CreateLabel("Onyxia", "Wing Buffet")
]]
function BaseClass:CreateLabel(...)
	return join(" :: ", ...) 
end

--[[
Arguments:
	string - The name of the toggle to check
Returns:
	*boolean|nil - True if it's checked, false if not, or nil if the toggle doesn't exist
Example:
	*self:RegisterToggles("Color by Class", true, "Color by Category", false)
	*if self:CheckToggle("Color by Class") then
	*	--Color By Class
	*elseif self:CeckToggle("Color by Category") then
	*	--Color By Category
	*else
	*	--No color
	*end
]]
function BaseClass:CheckToggle(name)
	return self.db and self.db.profile.Toggles and self.db.profile.Toggles[name]
end

--[[
Arguments:
	string - The name of the toggle to set
	[optional]boolean - The state to set the toggle(default: false)
Notes:
	*If the toggle doesn't exist then nothing will happen
]]
function BaseClass:SetToggle(name, state)
	if self.db and self.db.profile.Toggles and self.db.profile.Toggles[name] ~= nil then
		if state ~= nil then
			self.db.profile.Toggles[name] = not not state
		else
			self.db.profile.Toggles[name] = not self.db.profile.Toggles[name]
		end
	end
end

--[[
Arguments:
	string - The name of the toggle to set
Notes:
	*Removes a toggle, if it exists.
]]
function BaseClass:RemoveToggle(name)
	if self.db and self.db.profile.Toggles and self.db.profile.Toggles[name] ~= nil then
		self.db.profile.Toggles[name] = nil
	end
end

do
	local options = {}
	local ToggleActive = {
		type = "toggle",
		name = L["Enabled"],
		desc = L["Controls the active state of this module."],
		set = "Toggle",
		get = "IsEnabled",
		width = "full",
		--order = -.001,
		order = .001,
	}
	--[[
	Arguments:
		[optional]boolean - If true then a new options table will be returned regardless if one already exists (default: false)
	Notes:
		*If an options table is already set then that table will be returned unless force is set to true
		*Generally your options should be added to the args table inside the options table.
		*See AceConfig-3.0 API documentation for more information
	Returns:
		*table - A fresh options table that describes the options for the module
	Example:
		*local options = self:GetFreshOptionsTable()
		*options.args["New Option"] = {
		*	type = "boolean",
		*	name = "An option",
		*	desc = "A description",
		*}
	]]
	function BaseClass:GetFreshOptionsTable(force)
		if force or not options[self] then
			options[self] = {
				type = "group",
				handler = self,
				name = self:GetName(),
				desc = L["Configures this module."],
				args = {
					ToggleActive = ToggleActive,
				},
				
			}
		end
		return options[self]
	end

	--[[
	Arguments:
		[optional]boolean - If true a fresh options table will not be created if a table hasn't been set yet (default: false)
	Notes:
		*If an options table hasn't been set and noCreate is false then a fresh options table will be returned
	Returns:
		*table|nil - The options table registered to this module, nil if it doesn't exist and noCreate is true
	]]
	function BaseClass:GetOptionsTable(noCreate)
		if type(self.GetDynamicOptionsTable) == "function" then
			local opts = self:GetDynamicOptionsTable()
			if opts then
				return opts
			end
		end
		if options[self] then
			return options[self]
		elseif not noCreate then
			return self:GetFreshOptionsTable()
		end
	end

	--[[
	Arguments:
		table - The options table to set
	Notes:
		*If a table is missing important fields this function will fill them in automatically
	Example:
		*local opts = {
		*	args = {
		*		AnOption = {
		*			type = "boolean",
		*			name = "An option",
		*			desc = "A description",
		*		},
		*	},
		*}
		*self:SetOptionsTable(opts)
	]]
	function BaseClass:SetOptionsTable(opts)
		if type(opts) == "table" then
			opts.type = opts.type or "group"
			opts.handler = opts.handler or self
			opts.name = opts.name or self:GetName()
			opts.desc = opts.desc or L["Configures this module."]
			if opts.type == "group" and not opts.args then
				opts.args = {}
			end
			if opts.args and not opts.args.ToggleActive then
				opts.args.ToggleActive = ToggleActive
			end
			options[self] = opts
		end
	end
end

--[[
Arguments:
	[optional]boolean - If true then other players in your group will also receive the command (default: self:CheckToggle("Announce"))
	[optional]tuple - List of English upper-case classes, and/or the name of players in your group (can be mixed) to have their screen shake (default: all classes and players)
Notes:
	*Use this command sparingly, if the screen is constantly shaking then the importance is lost
	*Briefly shakes the screen, ideal for important information
	*Use the class filter to only shake certain class's screens
	*Use the player filter to only shake people affected by an event's screen
	*The player and class filter can be mixed
Example:
	*self:ShakeScreen(nil, "WARRIOR", "ROGUE", "MAGE") --if a spell needs to be interrupted
	*self:ShakeScreen(nil, "Someguy", "Anotherguy", "Thirdguy") --if these people need to take action 
	*self:ShakeScreen(nil, "Guywiththebomb", "PALADIN", "PRIEST") --Guywiththebomb needs to move and paladins and priests need to dispel him
]]
function BaseClass:ShakeScreen(announce, ...)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage("CoreMethod", "ShakeScreen", ...)--#NO_LOCALIZE
	end
	Core:ShakeScreen(...)
end

--[[
Arguments:
	[optional]boolean - If true then other players in your group will also receive the command (default: self:CheckToggle("Announce"))
	[optional]float[0,1] - The red value of the flash (default: .9)
	[optional]float[0,1] - The green value of the flash (default: .1)
	[optional]float[0,1] - The blue value of the flash (default: .1)
	[optional]tuple - List of English upper-case classes, and/or the name of players in your group (can be mixed) to have their screen shake (default: all classes)
Notes:
	*Use this command sparingly
	*Briefly flashes the screen, ideal for important information
	*Use the class filter to only flash certain class's screens
	*Use the player filter to only flash people affected by an event's screen
	*The player and class filter can be mixed
Example:
	*self:WarningFlash(nil, .7, .7, .7, "PRIEST", "PALADIN") --if a magic debuff needs dispelling
]]
function BaseClass:WarningFlash(announce, r, g, b, ...)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage("CoreMethod", "WarningFlash", r, g, b, ...)--#NO_LOCALIZE
	end
	Core:WarningFlash(r, g, b, ...)
end

--[[#FORCE_DOC
Notes:
	*Returns a locale table used to translate strings into the current locale
Example:
	*local L = self:GetTranslations()
	*self:Alert("Self", L["Translated message"])
Returns:
	*table - The translation table
]]
BaseClass.GetTranslations = Core.GetTranslations

--[[#FORCE_DOC
Arguments:
	string - The locale to register translations for ("enUS", "koKR", ect)
	tuple - Pairs of strings and strings or true to register translations for
Notes:
	*Registers translations to use within the module
	*If the pair's value is the boolean true then the key will be used as both the value and the key
Example:
	*self:RegisterTranslations("enUS",
	*	"Translated message", true,
	*	"Another message", true,
	*	"Short desc", "The description of something"
	*)
	--assuming enUS locale
	*print("Short desc") => "The description of something"
]]
BaseClass.RegisterTranslations = Core.RegisterTranslations

--=================================================================--
--[[=========================Communications==========================]]--
--=================================================================--

--[[
Arguments:
	string - One of "High", "Normal", "Low"
	tuple - Strings to be formatted in to the alert message
Notes:
	*Displays an alert, only the client running the module will see this
]]
function BaseClass:Alert(priority, ...)
	if not Profile.ShowAlerts then return end
	if Profile.PlayAlertSounds then
		if self:IsBossMod() then
			if Profile.BossSound then
				PlaySoundFile(MediaHandler:Fetch("sound", Profile.BossSound))
			end
		else
			if Profile.PluginSound then
				PlaySoundFile(MediaHandler:Fetch("sound", Profile.PluginSound))
			end
		end
	end
	Core:ShowAlert(priority, ...)
end


--=================================================================--
--[[======================Bar State Queries==========================]]--
--=================================================================--
local StartBar, CancelBar, PauseBar, UnpauseBar, IncreaseCount, DecreaseCount, SetCount, SetMaxCount, SetColor, SetIcon, SetLabel, SetTime, ShiftBar, SetMinMax, SetCurrentTime


local cache, activeBars = {}, {}
local modules = _G.setmetatable({}, {__index = function(self, key)
	self[key] = {}
	return self[key]
end,})

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*boolean|nil - True if the bar is paused, false if not; nil if the bar doesn't exist
]]
function BaseClass:IsBarPaused(label)
	return modules[self][label] and modules[self][label].paused ~= nil
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The time relative to GetTime() when the bar was paused; nil if the bar doesn't exist
]]
function BaseClass:GetBarPausedTime(label)
	return modules[self][label] and modules[self][label].paused
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*boolean - True if the bar is active, false if not
]]
function BaseClass:IsBarActive(label)
	return modules[self][label] ~= nil
end


--[[
Notes:
	*Iterates over all active bars, returning their label
Returns:
	*function - The iterator function
Example:
	*for label in self:IterateActiveBars() do
	*	if self:IsBarPaused(label)
	*		--do important stuff
	*	end
	*end
]]
do
	local iter
	function BaseClass:IterateActiveBars()
		if not iter then
			iter = function(table, last)
				return (next(table, last))
			end
		end
		return iter, modules[self], nil
	end
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*boolean|nil - True if a bar is a timed bar, false if it's a count or timeless bar; nil if the bar desn't exist
]]
function BaseClass:IsTimedBar(label)
	return modules[self][label] and modules[self][label].startTime ~= nil
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*boolean|nil - True if a bar is a timed bar, false if it's a count or timeless bar; nil if the bar desn't exist
]]
function BaseClass:IsCountBar(label)
	return modules[self][label] and modules[self][label].count ~= nil
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*boolean|nil - True if a bar is a timeless bar, false if it's a count or timed bar; nil if the bar desn't exist
]]
function BaseClass:IsTimelessBar(label)
	local bar = modules[self][label]
	return bar and bar.count == nil and bar.startTime == nil
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The current count of a count bar, nil if the bar isn't a count bar or the bar doesn't exist
]]
function BaseClass:GetBarCount(label)
	return modules[self][label] and modules[self][label].count
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The max count of a count bar, nil if the bar isn't a count bar or the bar doesn't exist
]]
function BaseClass:GetBarMaxCount(label)
	return modules[self][label] and modules[self][label].maxCount
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The custom red value the bar is using, nil if no custom coloring or the bar doesn't exist
	*number|nil - The custom green value the bar is using, nil if no custom coloring or the bar doesn't exist
	*number|nil - The custom blue value the bar is using, nil if no custom coloring or the bar doesn't exist
]]
function BaseClass:GetBarColor(label)
	local bar = modules[self][label]
	if bar then
		return bar.r, bar.g, bar.b
	end
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*string|nil - The path to the icon the bar is using, or nil if the bar doesn't exist
]]
function BaseClass:GetBarIcon(label)
	return modules[self][label] and modules[self][label].iconName
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*string|nil - The last label the bar was using, nil if no other label has been assigned or the bar doesn't exist
]]
function BaseClass:GetBarLastLabel(label)
	return modules[self][label] and modules[self][label].lastLabel
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*boolean|nil - True if the bar empties instead of fills, false otherwise, nil if the bar doesn't exist
]]
function BaseClass:IsBarReversed(label)
	return modules[self][label] and modules[self][label].reverse
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The start time relative to GetTime(), nil if the bar doesn't exist or the bar isn't a timed bar
]]
function BaseClass:GetBarStartTime(label)
	return modules[self][label] and modules[self][label].startTime
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The end time relative to GetTime(), nil if the bar doesn't exist or the bar isn't a timed bar
]]
function BaseClass:GetBarEndTime(label)
	return modules[self][label] and modules[self][label].endTime
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The duration of a timed bar, nil if the bar doesn't exist or the bar isn't a timed bar
]]
function BaseClass:GetBarDuration(label)
	local bar = modules[self][label]
	if bar and bar.startTime and bar.endTime then
		return bar.endTime - bar.startTime
	end
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - The time left on the bar, nil if the bar doesn't exist or the bar isn't a timed bar
]]
function BaseClass:GetBarTimeLeft(label)
	local bar = modules[self][label]
	if bar and bar.startTime then
		return (bar.paused or bar.endTime) - GetTime()
	end
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*number|nil - Number of times the bar will repeat, nil if it's not a repeating bar or if the bar doesn't exist or the bar isn't a timed bar
]]
function BaseClass:GetBarRepeatsLeft(label)
	return modules[self][label] and modules[self][label].numRepeat
end

--[[
Arguments:
	string - Label of the bar you wish to query
Notes:
	*Plugins generally use their name as the category
	*BossMods generally use "Boss Abilities" as their category
Returns:
	*string|nil - The category the bar belongs, nil if  the bar doesn't exist
]]
function BaseClass:GetBarCategory(label)
	return modules[self][label] and modules[self][label].category
end

--[[
Arguments:
	string - Label of the bar you wish to query
Returns:
	*string|nil - The sub-category the bar belongs, nil if  the bar doesn't exist
]]
function BaseClass:GetBarSubCategory(label)
	return modules[self][label] and modules[self][label].subCategory
end


--=================================================================--
--[[==========================Bar Creation===========================]]--
--=================================================================--

--All controls handled by modules
local Cancel = "CancelBar"		--#NO_LOCALIZE
local CancelM = "CancelModule"	--#NO_LOCALIZE
local Start = "StartBar"		--#NO_LOCALIZE
local Shift = "ShiftBar"		--#NO_LOCALIZE
local Time = "TimeBar"			--#NO_LOCALIZE
local Color = "ColorBar"		--#NO_LOCALIZE
local Pause = "PauseBar"		--#NO_LOCALIZE
local Unpause = "UnpauseBar"	--#NO_LOCALIZE
local ICount = "IncreaseCount"	--#NO_LOCALIZE
local DCount = "DecreaseCount"	--#NO_LOCALIZE
local SCount = "SetCount"		--#NO_LOCALIZE
local SMCount = "SetMaxCount"	--#NO_LOCALIZE
local Label = "LabelBar"		--#NO_LOCALIZE
local Icon = "IconBar"			--#NO_LOCALIZE
local MinMax = "SetMinMaxBar"	--#NO_LOCALIZE
local CTime = "CurrentTime"		--#NO_LOCALIZE
local Control = "Control"		--#NO_LOCALIZE
local IconCoords = "IconCoords"	--#NO_LOCALIZE

--[[=========================Advanced Bar Creation========================]]--

--#NODOC
--use class specific bar creation methods
function BaseClass:StartBar(announce, ...)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Start, self:GetPrefixedName(), ...)
	end
	Core:ExecuteFrameMethod(Start, self:GetPrefixedName(), ...)
	StartBar(self, ...)
end

--=================================================================--
--[[===========================Bar Control===========================]]--
--=================================================================--

--[[
Arguments:
	string - Label of the bar you wish to control
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Cancels a bar
]]
function BaseClass:CancelBar(label, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Cancel, label)
	end
	Core:ExecuteFrameMethod(Cancel, label)
	CancelBar(self, label)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	[optional]float[0,1] - The red value of the bar (default: category coloring)
	[optional]float[0,1] - The green value of the bar (default: category coloring)
	[optional]float[0,1] - The blue value of the bar (default: category coloring)
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Changes the bar coloring
	*Not all values have to be present for this to work, only the red value could be changed for example
]]
function BaseClass:SetBarColor(label, r, g, b, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Color, label, r, g, b)
	end
	Core:ExecuteFrameMethod(Color, label, r, g, b)
	SetColor(self, label, r, g, b)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Increases the current value of a count bar by 1
]]
function BaseClass:IncreaseBarCount(label, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, ICount, label)
	end
	Core:ExecuteFrameMethod(ICount, label)
	SetCount(self, label, 1, true)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Decreases the current value of a count bar by 1
]]
function BaseClass:DecreaseBarCount(label, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, DCount, label)
	end
	Core:ExecuteFrameMethod(DCount, label)
	SetCount(self, label, -1, true)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	number - The count the bar should be set to
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Sets a count bar to a specific value
]]
function BaseClass:SetBarCount(label, count, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, SCount, label, count)
	end
	Core:ExecuteFrameMethod(SCount, label, count)
	SetCount(self, label, count)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	number - The new maximum count of the count bar
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Sets the max possible value on a count bar
]]
function BaseClass:SetBarMaxCount(label, count, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, SMCount, label, count)
	end
	Core:ExecuteFrameMethod(SMCount, label, count)
	SetMaxCount(self, label, count)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Pauses a timed bar
]]
function BaseClass:PauseBar(label, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Pause, label)
	end
	Core:ExecuteFrameMethod(Pause, label)
	PauseBar(self, label)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Unpauses a timed bar
]]
function BaseClass:UnpauseBar(label, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Unpause, label)
	end
	Core:ExecuteFrameMethod(Unpause, label)
	UnpauseBar(self, label)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	string - The new label to set the bar to, use this label to control the bar from now on
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Changes a bar's label
]]
function BaseClass:SetBarLabel(label, newLabel, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Label, label, newLabel)
	end
	Core:ExecuteFrameMethod(Label, label, newLabel)
	SetLabel(self, label, newLabel)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	string - New path to the icon, the string "@random" will apply a random icon
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Changes a bar's icon
]]
function BaseClass:SetBarIcon(label, icon, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Icon, label, icon)
	end
	Core:ExecuteFrameMethod(Icon, label, icon)
	SetIcon(self, label, icon)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	number - The left offset of the icon
	number - the right offset of the icon
	number - The top offset of the icon
	number - the bottom offset of the icon
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Changes a bar's icon coordinates
	*Used when multiple icons are stored in the same texture file (game files typically)
Usage:
	* self:SetBarIconCoors(self:CreateLabel("this", "that"), 0.65, 0.32, 0.53, 0.32)
]]
function BaseClass:SetBarIconCoords(label, left, right, top, bottom, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, IconCoords, label, left, right, top, bottom)
	end
	Core:ExecuteFrameMethod(IconCoords, label, left, right, top, bottom)
	SetIconCoords(self, label, left, right, top, bottom)
end
--[[
Arguments:
	string - Label of the bar you wish to control
	number - The new time value of the bar, if negative it'll change the bar's previous "reverse" direction
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Resets a bar's max time to the new value, only works on bars with time
]]
function BaseClass:SetBarTime(label, time, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Time, label, time)
	end
	Core:ExecuteFrameMethod(Time, label, time)
	SetTime(self, label, time)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	number - The new time value of the bar, negatives will act as if they were positive
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Sets the bar's current time to a specific value, unlike :SetBarTime it keeps the starting and ending time intact
]]
function BaseClass:SetBarCurrentTime(label, time, announce)
	if time < 0 then
		time = -time
	end
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, CTime, label, time)
	end
	Core:ExecuteFrameMethod(CTime, label, time)
	SetCurrentTime(self, label, time)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	number - The amount of time in seconds you wish to shift the bar, use negatives to decrease time left and positives to increase time left
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Shifts a bar by a specified amount of time, for example a 20 sec bar that is currently at 15 seconds elapsed shifted by 10 sec would now be 5 seconds elapsed
	*Shifting a bar gives the appearance of the time increasing/decreasing
	*A bars starting time cannot be shifted past the current time, that is, start a 15 second bar, let it run 5 sec and try to shift the bar 10 seconds it'll only move 5 seconds back to 15
	*If a bar is shifted past the end time, then the bar will cancel itself automatically
]]
function BaseClass:ShiftBarTime(label, time, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, Shift, label, time)
	end
	Core:ExecuteFrameMethod(Shift, label, time)
	ShiftBar(self, label, time)
end

--[[
Arguments:
	string - Label of the bar you wish to control
	number - The minimum statusbar value relative to GetTime
	number - The maximum statusbar value relative to GetTime
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Provides direct control over a statusbar's min/max values
	*An advanced control method, when a bar is created its min is set to GetTime()'s value automatically, and max is GetTime() + time,
	use this to modify those values directly
]]
function BaseClass:SetMinMaxValues(label, min, max, announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, MinMax, label, min, max)
	end
	Core:ExecuteFrameMethod(MinMax, label, min, max)
	SetMinMax(self, label, min, max)
end

--[[
Arguments:
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Cancels every bar created by the module
]]
function BaseClass:CancelBars(announce)
	if announce or announce == nil and self:CheckToggle(L["Announce"]) then
		Core:SendMessage(Control, CancelM, self:GetPrefixedName())
	end
	Core:ExecuteFrameMethod(CancelM, self:GetPrefixedName())
	for label, bar in pairs(modules[self]) do
		modules[self][label] = nil
		activeBars[bar] = nil
		cache[#cache + 1] = bar
	end
end

--[[
Arguments:
	string - Label of the bar you wish to control
	[optional]boolean - If true the command will be sent across the group, if false the command is local only (default: self:CheckToggle("Announce"))
Notes:
	*Resets the bar color to SubCategory color.
]]
function BaseClass:ResetBarColor(label, announce)
	local bar = modules[self][label]
	if bar then
		local ct = Profile.ColorTables[bar.category][bar.subCategory]
		self:SetBarColor(label, ct.r, ct.g, ct.b, announce)
	end
end

--Internal functions.
local Frame = CreateFrame"Frame"

Frame:SetScript("OnUpdate", function()
	local now = GetTime()
	local empty = true
	for bar in pairs(activeBars) do
		if 0 > bar.endTime - now then
			if bar.numRepeat and bar.numRepeat > 0 then
				bar.numRepeat = bar.numRepeat - 1
				local time = bar.endTime - bar.startTime
				bar.startTime = now
				bar.endTime = time + now
			else
				modules[activeBars[bar]][bar.label] = nil
				activeBars[bar] = nil
				cache[#cache + 1] = bar
			end
		end
		empty = false
	end
	if empty then
		Frame:Hide()
	end
end)

local Icons = Icons
function StartBar(self, label, category, subCategory, time, icon, reverse, numRepeat, count, red, green, blue)
	if not label or not category or not subCategory or (time and time < 0) then return end
	local bar
	if #cache > 0 then
		 bar = cache[#cache]
		 cache[#cache] = nil
	else
		bar = {}
	end
	if time then
		activeBars[bar] = self
	end

	modules[self][label] = bar

	bar.label = label
	bar.startTime = time and GetTime() or nil
	bar.endTime = time and bar.startTime + time or nil

	bar.iconName = Icons[icon]

	bar.category = category
	bar.subCategory = subCategory
	bar.reverse = not not reverse
	bar.numRepeat = time and numRepeat or nil

	bar.r, bar.g, bar.b = red, green, blue
	bar.maxCount = not time and numRepeat or nil
	bar.count = bar.maxCount and (count or 1) or nil
	bar.lastLabel = nil
	bar.paused = nil

	Frame:Show()
end

function CancelBar(self, label)
	local bar = modules[self][label]
	if bar then
		modules[self][label] = nil
		activeBars[bar] = nil
		cache[#cache + 1] = bar
	end
end

function PauseBar(self, label)
	local bar = modules[self][label]
	if bar and bar.startTime and not bar.paused then
		bar.paused = GetTime()
	end
end

function UnpauseBar(self, label)
	local bar = modules[self][label]
	if bar and bar.paused then
		local pausedTime = GetTime() - bar.paused
		bar.paused = nil
		bar.startTime = bar.startTime + pausedTime
		bar.endTime = bar.endTime + pausedTime
	end
end


function SetCount(self, label, count, byOne)
	local bar = modules[self][label]
	if bar and bar.count and count then 
		if byOne then
			count = bar.count + count
		end
		if count > bar.maxCount then
			count = bar.maxCount 
		elseif count < 0 then
			count = 0
		end
		bar.count = count
	end
end

function SetMaxCount(self, label, maxCount)
	local bar = modules[self][label]
	if bar and bar.count and maxCount then 
		bar.maxCount = maxCount
		bar.count = bar.count <= bar.maxCount and bar.count or bar.maxCount
	end 
end

function SetColor(self, label, r, g, b)
	local bar = modules[self][label]
	if bar then
		bar.r, bar.g, bar.b = r or bar.r, g or bar.g, b or bar.b
	end
end

function SetIcon(self, label, icon)
	local bar = modules[self][label]
	if bar then
		bar.iconName = not icon and "random" or icon--#NO_LOCALIZE
	end
end

function SetIconCoords(self, label, l, r, t, b)
	local bar = modules[self][label]
	if bar then
		bar.iconCoors = {
			[1] = l,
			[2] = r,
			[3] = t,
			[4] = b,
		}
	end
end

function SetLabel(self, label, newLabel)
	local bar = modules[self][label]
	if bar then
		bar.lastLabel = label
		modules[self][label] = nil
		modules[self][newLabel] = bar
	end
end

function SetGUID(self, label, guid)
	local bar = modules[self][label]
	if bar then
		bar.GUID = guid
	end
end

function SetTime(self, label, time)
	local bar = modules[self][label]
	if bar and bar.startTime then
		if time < 0 then
			bar.reverse = not bar.reverse
			time = -time
		end
		bar.startTime = GetTime()
		bar.endTime = bar.startTime + time
		if bar.paused then
			bar.paused = bar.startTime
		end
	end
end

function SetCurrentTime(self, label, time)
	local bar = modules[self][label]
	if bar and bar.startTime then
		local now = GetTime()
		local diff = (bar.endTime - now) - time
		bar.startTime = bar.startTime - diff
		bar.endTime = bar.endTime - diff
		if bar.startTime > now then
			local adjust =  bar.startTime - now
			bar.startTime = bar.startTime - adjust
			bar.endTime = bar.endTime - adjust
		end
	end
end

function ShiftBar(self, label, value)
	local bar = modules[self][label]
	if bar and bar.startTime then
		bar.startTime = bar.startTime + value
		bar.endTime = bar.endTime + value
		local max = GetTime()
		if bar.startTime > max then
			local adjust =  bar.startTime - max
			bar.startTime = bar.startTime - adjust
			bar.endTime = bar.endTime - adjust
		end
	end
end

function SetMinMax(self, label, min, max)
	local bar = modules[self][label]
	if bar and bar.startTime then
		bar.startTime = min
		bar.endTime = max
	end
end