local _G = getfenv(0);

-- Constants
local modName = "QBar";
local btnSize = 36;
local slots = {
	"HeadSlot", "NeckSlot", "ShoulderSlot", "BackSlot", "ChestSlot", "ShirtSlot", "TabardSlot", "WristSlot",
	"HandsSlot", "WaistSlot", "LegsSlot", "FeetSlot", "Finger0Slot", "Finger1Slot", "Trinket0Slot", "Trinket1Slot",
	"MainHandSlot", "SecondaryHandSlot", "RangedSlot",
};

-- Work Vars
local shownItems = 0;
local lastName;
local ignoreList;
local bindFrame;
local updateTime = 0;
local x, y;

-- Config
QBar_Config = {};
local cfg;
local defConfig = {
	enabled = true,
	showTips = true,
	vertical = false,
	scale = 1,
	padding = 1,
};

--------------------------------------------------------------------------------------------------------
--                                                Main                                                --
--------------------------------------------------------------------------------------------------------

local f = CreateFrame("Frame",modName,UIParent);
f:SetMovable(1);
f:SetToplevel(1);
f:SetFrameStrata("MEDIUM");

local function OnMouseUp(self,button)
	f:StopMovingOrSizing();
	cfg.left, cfg.bottom = f:GetLeft(), f:GetBottom();
end

local function OnUpdate(self,elapsed)
	updateTime = (updateTime + elapsed);
	if (updateTime > 0.25) then
		f.UpdateButtons();
		f:SetScript("OnUpdate",nil);
	end
end

f:SetScript("OnMouseDown",function(self,button) f:StartMoving(); end);
f:SetScript("OnMouseUp",OnMouseUp);

f.tip = CreateFrame("GameTooltip",modName.."Tip",nil,"GameTooltipTemplate");
f.tip:SetOwner(UIParent,"ANCHOR_NONE");

--------------------------------------------------------------------------------------------------------
--                                             Local Funcs                                            --
--------------------------------------------------------------------------------------------------------

-- Lock Frame
local function SetLocked(lock)
	if (lock) then
		f:EnableMouse(nil);
		f:SetBackdrop(nil);
	else
		f:EnableMouse(1);
		f:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", tile = 1, tileSize = 16, insets = { left = 0, right = 0, top = 0, bottom = 0 } });
		f:SetBackdropColor(0.1,0.22,0.35,1);
	end
end

-- Bindings Frame
local function OpenBindingFrame()
	if (bindFrame) then
		bindFrame:Show();
		return;
	end

	bindFrame = CreateFrame("Frame",nil,UIParent);
	bindFrame:SetWidth(200);
	bindFrame:SetHeight(60);
	bindFrame:SetPoint("CENTER");
	bindFrame:SetFrameStrata("DIALOG");
	bindFrame:EnableKeyboard(1);
	bindFrame:SetBackdrop({ bgFile = "Interface\\ChatFrame\\ChatFrameBackground", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", tile = 1, tileSize = 16, edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 } });
	bindFrame:SetBackdropColor(0.1,0.22,0.35,1);
	bindFrame:SetBackdropBorderColor(0.1,0.1,0.1,1);

	function bindFrame.keyDownFunc(self,key)
		-- Check Key
		if (key == "ESCAPE") then
			SetBinding(cfg.bindKey);
			cfg.bindKey = nil;
			f.UpdateBinding();
			self:Hide();
			return;
		elseif (key:match("^.SHIFT") or key:match("^.CTRL") or key:match("^.ALT")) then
			return;
		end
		-- Prefix Modifier Key
		if (IsShiftKeyDown()) then
			key = "SHIFT-"..key;
		end
		if (IsControlKeyDown()) then
			key = "CTRL-"..key;
		end
		if (IsAltKeyDown()) then
			key = "ALT-"..key;
		end
		-- Validate
		if (GetBindingAction(key) ~= "") then
			AzMsg("|2Invalid|r: The hotkey |1"..key.."|r is already bound to |1"..GetBindingAction(key).."|r.");
			return;
		end
		if (cfg.bindKey) then
			SetBinding(cfg.bindKey);
		end
		-- Set New Hotkey
		cfg.bindKey = key;
		AzMsg("|2"..modName.."|r: The hotkey has been set to |1"..key.."|r.");
		f.UpdateBinding(1);
		self:Hide();
	end
	bindFrame:SetScript("OnKeyDown",bindFrame.keyDownFunc);

	bindFrame.hint = bindFrame:CreateFontString(nil,"ARTWORK","GameFontNormal");
	bindFrame.hint:SetPoint("CENTER");
	bindFrame.hint:SetText("Press the Desired Hotkey...\nEscape to Clear Binding");
end

-- SetEnabled
local function SetEnabledStatus()
	if (cfg.enabled) then
		f:RegisterEvent("BAG_UPDATE");
		f:RegisterEvent("UNIT_INVENTORY_CHANGED");
		f:RegisterEvent("ACTIONBAR_UPDATE_COOLDOWN");
		f:RegisterEvent("PLAYER_REGEN_ENABLED");
		f:RegisterEvent("UPDATE_BINDINGS");
		f.RequestUpdate();
	else
		f:UnregisterAllEvents();
	end
end

--------------------------------------------------------------------------------------------------------
--                                               Events                                               --
--------------------------------------------------------------------------------------------------------
local function OnEvent(self,event,p1,...)
	-- Variables Loaded
	if (event == "VARIABLES_LOADED") then
		cfg = QBar_Config;
		if (cfg.left and cfg.bottom) then
			f:SetPoint("BOTTOMLEFT",cfg.left,cfg.bottom);
		else
			f:SetPoint("CENTER");
		end
		for option, value in pairs(defConfig) do
			if (cfg[option] == nil) then
				cfg[option] = value;
			end
		end
		SetEnabledStatus();
		f:SetScale(cfg.scale);
	-- Update Cooldowns
	elseif (event == "ACTIONBAR_UPDATE_COOLDOWN") then
		if (shownItems > 0) then
			f.UpdateCooldowns();
		end
	-- Bindings Update
	elseif (event == "UPDATE_BINDINGS") then
		f.UpdateBinding();
	-- Inventory Changed
	elseif (event == "UNIT_INVENTORY_CHANGED") then
		if (p1 == "player") then
			f.RequestUpdate();
		end
	-- Other Events
	else
		f.RequestUpdate();
	end
end

f:SetScript("OnEvent",OnEvent);
f:RegisterEvent("VARIABLES_LOADED");

--------------------------------------------------------------------------------------------------------
--                                            Command Line                                            --
--------------------------------------------------------------------------------------------------------
_G["SLASH_"..modName.."1"] = "/qb";
SlashCmdList[modName] = function(cmd)
	-- Extract Parameters
	local param1, param2 = cmd:match("^([^%s]+)%s*(.*)$");
	param1 = (param1 and param1:lower() or cmd:lower());
	-- Enabled
	if (param1 == "toggle") then
		cfg.enabled = not cfg.enabled;
		SetEnabledStatus();
		AzMsg("|2"..modName.."|r has now been |1"..(cfg.enabled and "enabled" or "disabled").."|r.");
	-- Scale
	elseif (param1 == "scale") then
		if (param2 ~= "") and (type(tonumber(param2)) == "number") then
			cfg.scale = tonumber(param2);
			f:SetScale(cfg.scale);
			AzMsg("Button Scale set to |1"..cfg.scale.."|r.");
		end
	-- Padding
	elseif (param1 == "padding") then
		if (param2 ~= "") and (type(tonumber(param2)) == "number") then
			cfg.padding = tonumber(param2);
			f.RequestUpdate();
			AzMsg("Button Padding set to |1"..cfg.padding.."|r.");
		end
	-- Tips
	elseif (param1 == "tips") then
		cfg.showTips = not cfg.showTips;
		AzMsg("Item tips |1"..(cfg.showTips and "enabled" or "disabled").."|r.");
	-- Vertical
	elseif (param1 == "vertical") then
		cfg.vertical = not cfg.vertical;
		f.RequestUpdate();
		AzMsg("Vertical alignment |1"..(cfg.vertical and "enabled" or "disabled").."|r.");
	-- Lock
	elseif (param1 == "lock") then
		SetLocked(f:IsMouseEnabled());
		AzMsg("QBar Button Frame is now |1"..(f:IsMouseEnabled() and "unlocked" or "locked").."|r.");
	-- Bind
	elseif (param1 == "bind") then
		OpenBindingFrame();
	-- Clear Ignore
	elseif (param1 == "clearignore") then
		ignoreList = nil;
		f.RequestUpdate();
		AzMsg("Ignore List Cleared.");
	-- Invalid or No Command
	else
		UpdateAddOnMemoryUsage();
		AzMsg(format("----- |2%s|r |1%s|r ----- |1%.2f |2kb|r -----",modName,GetAddOnMetadata(modName,"Version"),GetAddOnMemoryUsage(modName)));
		AzMsg("The following |2parameters|r are valid for this addon:");
		AzMsg(" |2toggle|r = Toggles the mod being on or off");
		AzMsg(" |2scale 'value'|r = Sets the scale of the buttons");
		AzMsg(" |2padding 'value'|r = Space padding between the buttons");
		AzMsg(" |2tips|r = Toggles the showing of item tips when you mouse over buttons");
		AzMsg(" |2vertical|r = Sets whether or not the buttons anchor vertical or horizontal");
		AzMsg(" |2lock|r = Toggles the button frame being locked, use this command to move the buttons around");
		AzMsg(" |2clearignore|r = Show all items again, by clearing the ignore list");
		AzMsg(" |2bind|r = Set a key binding for the last item used");
	end
end
--------------------------------------------------------------------------------------------------------
--                                                Items                                               --
--------------------------------------------------------------------------------------------------------

f.items = {};

-- Button Scripts
local function Button_OnEnter(self)
	if (cfg.showTips) then
		GameTooltip:SetOwner(self,"ANCHOR_RIGHT");
		local bag, slot = self:GetAttribute("bag"), self:GetAttribute("slot");
		if (bag) then
			GameTooltip:SetBagItem(bag,slot);
		else
			GameTooltip:SetInventoryItem("player",slot);
		end
	end
end

local function HideGTT()
	GameTooltip:Hide();
end

local function Button_OnClick(self,button)
	-- Handle Modified Click
	if (HandleModifiedItemClick(self.link)) then
		return;
	-- Ignore
	elseif (IsShiftKeyDown()) then
		if (not ignoreList) then
			ignoreList = {};
		end
		ignoreList[self.name] = true;
		f.RequestUpdate();
	-- Set Hotkey
	else
		lastName = self.name;
		f.UpdateBinding(1);
	end
end

-- Make Loot Button
local function MakeItemButton()
	local b = CreateFrame("Button",modName..(#f.items + 1),f,"SecureActionButtonTemplate");
	b:SetWidth(btnSize);
	b:SetHeight(btnSize);
	b:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square");
	b:RegisterForClicks("LeftButtonUp","RightButtonUp");
	b:SetScript("OnEnter",Button_OnEnter);
	b:SetScript("OnLeave",HideGTT);
	b:HookScript("OnClick",Button_OnClick);
	b:SetAttribute("type*","item");

	b.icon = b:CreateTexture(nil,"ARTWORK");
	b.icon:SetAllPoints();

	b.count = b:CreateFontString(nil,"ARTWORK","NumberFontNormal");
	b.count:SetPoint("BOTTOMRIGHT",b.icon,-3,3);

	b.cooldown = CreateFrame("Cooldown",nil,b,"CooldownFrameTemplate");
	b.cooldown:SetAllPoints();

	b.bind = b:CreateFontString(nil,"ARTWORK","NumberFontNormalSmallGray");
	b.bind:SetPoint("TOPLEFT",b.icon,3,-3);
	b.bind:SetPoint("TOPRIGHT",b.icon,-3,-3);
	b.bind:SetJustifyH("RIGHT");

	if (#f.items == 0) then
		b:SetPoint("TOPLEFT",f,8,-8);
	end

	tinsert(f.items,b);
	return b;
end

-- Add Button
local function AddButton(index,bag,slot,link,count)
	button = f.items[index] or MakeItemButton();

	button.icon:SetTexture(select(10,GetItemInfo(link)));
	button.count:SetText(count and count > 1 and count or "");

	button.link = link;
	button.name = GetItemInfo(link);

	button:SetAttribute("bag",bag);
	button:SetAttribute("slot",slot);

	if (index > 1) then
		button:ClearAllPoints();
		if (cfg.vertical) then
			button:SetPoint("TOP",f.items[index - 1],"BOTTOM",0,cfg.padding * -1);
		else
			button:SetPoint("LEFT",f.items[index - 1],"RIGHT",cfg.padding,0);
		end
	end

	button:Show();
end

-- Check Item
local function CheckItem(link)
	local itemName, _, _, _, _, itemType, itemSubType = GetItemInfo(link);
	if (ignoreList) and (ignoreList[itemName]) then
		return;
	end
	f.tip:ClearLines();
	f.tip:SetHyperlink(link);
	local text;
	if (f.tip:NumLines() >= 3) and (itemType == "Quest" or itemSubType == "Quest" or _G[modName.."TipTextLeft2"]:GetText() == ITEM_BIND_QUEST) then
		for i = 3, f.tip:NumLines() do
			text = _G[modName.."TipTextLeft"..i]:GetText() or "";
--			if (text:find("^"..ITEM_SPELL_TRIGGER_ONUSE) or text:find("^"..ITEM_STARTS_QUEST)) then
			if (text:find("^"..ITEM_SPELL_TRIGGER_ONUSE)) then
				return 1;
			end
		end
	end
	return;
end

--------------------------------------------------------------------------------------------------------
--                                               Update                                               --
--------------------------------------------------------------------------------------------------------

-- Request a Button Update
function f.RequestUpdate()
	updateTime = 0;
	f:SetScript("OnUpdate",OnUpdate);
end

-- Update Buttons
function f.UpdateButtons()
	-- Check if we are locked by combat
	if (InCombatLockdown()) then
		return;
	end
	-- locals
	local index = 1;
	local link, slotId;
	-- Inventory
	for bag = 0, NUM_BAG_FRAMES do
		for slot = 1, GetContainerNumSlots(bag) do
			link = GetContainerItemLink(bag,slot);
			if (link) and (CheckItem(link)) then
				AddButton(index,bag,slot,link,select(2,GetContainerItemInfo(bag,slot)));
				index = (index + 1);
			end
		end
	end
	-- Equipped Items
	for _, slotName in ipairs(slots) do
		slotId = GetInventorySlotInfo(slotName);
		link = GetInventoryItemLink("player",slotId);
		if (link) and (CheckItem(link)) then
			AddButton(index,nil,slotId,link);
			index = (index + 1);
		end
	end
	-- Set Shown Items
	shownItems = (index - 1);
	for i = index, #f.items do
		f.items[i]:Hide();
	end
	-- Set Frame Dimention
	x = (btnSize * shownItems + (shownItems - 1) * cfg.padding + 16);
	y = (btnSize + 16);
	f:SetWidth(cfg.vertical and y or x);
	f:SetHeight(cfg.vertical and x or y);
	-- Update Misc
	f.UpdateBinding(1);
	f.UpdateCooldowns();
end

-- Update Cooldowns
function f.UpdateCooldowns()
	local bag, slot;
	for i = 1, shownItems do
		bag, slot = f.items[i]:GetAttribute("bag"), f.items[i]:GetAttribute("slot");
		if (bag) then
			CooldownFrame_SetTimer(f.items[i].cooldown,GetContainerItemCooldown(bag,slot));
		else
			CooldownFrame_SetTimer(f.items[i].cooldown,GetInventoryItemCooldown("player",slot));
		end
	end
end

-- Update Bind Text
function f.UpdateBinding(reBind)
	-- Can't rebind in combat
	if (InCombatLockdown()) and (reBind) then
		reBind = nil;
	end
	-- Unbind previous button first
	if (reBind) and (cfg.bindKey) then
		SetBinding(cfg.bindKey);
	end
	-- Update all visible buttons
	for i = 1, shownItems do
		if (reBind) and (cfg.bindKey) and (lastName == f.items[i].name) then
			SetBindingClick(cfg.bindKey,modName..i);
		end
		f.items[i].bind:SetText(GetBindingText(GetBindingKey("CLICK "..modName..i..":LeftButton"),"",1));
	end
end

--------------------------------------------------------------------------------------------------------
--                                Global Chat Message Function (Rev 3)                                --
--------------------------------------------------------------------------------------------------------
if (not AZMSG_REV or AZMSG_REV < 3) then
	AZMSG_REV = 3;
	function AzMsg(text)
		DEFAULT_CHAT_FRAME:AddMessage(tostring(text):gsub("|1","|cffffff80"):gsub("|2","|cffffffff"),128/255,192/255,255/255);
	end
end