PLANNER_VERSION = 2.3;

function Planner_OnLoad()
	-- Hook the Talent Frame's Update function
	hooksecurefunc( "TalentFrame_Update", Planner_TalentFrame_AfterUpdate );
	
	-- Replace the Talent Buttons' OnEnter and OnClick scripts
	for i = 1, MAX_NUM_TALENTS, 1 do
		local button = getglobal( "PlayerTalentFrameTalent"..i );
		button:RegisterForClicks( "LeftButtonUp", "RightButtonUp" );
		
		Planner_TalentFrameTalent_OnClick = button:GetScript( "OnClick" );
		button:SetScript( "OnClick", Planner_OnClick );
		
		Planner_TalentFrameTalent_OnEnter = button:GetScript( "OnEnter" );
		button:SetScript( "OnEnter", Planner_OnEnter );
	end
	
	-- Register for Loading Variables
	this:RegisterEvent( "VARIABLES_LOADED" );
	this:RegisterEvent( "PLAYER_ENTERING_WORLD" );
	this:RegisterEvent( "PLAYER_LEAVING_WORLD" );
	
	SlashCmdList["PLANNER"] = function ( msg )
		Planner_DoSlash( msg );
	end
	SLASH_PLANNER1 = "/planner";
	SLASH_PLANNER2 = "/tp";
	
	-- Default colors
	INVALID     = { r = 1.0, g = 0.1, b = 0.1, };  -- Red
	PLANNED     = { r = 0.1, g = 0.1, b = 1.0, };  -- Blue
	MAXPLANNED  = { r = 1.0, g = 0.1, b = 1.0, };  -- Purple
	BUYMENOW    = { r = 0.1, g = 1.0, b = 0.1, };  -- Green
	MAXEDOUT    = { r = 1.0, g = 0.82, b = 0.0, }; -- Gold
	NORMAL      = { r = 0.8, g = 0.8, b = 0.8, };  -- Grey
	DONTBUYME   = { r = 0.1, g = 1.0, b = 1.0, };  -- Cyan

	-- Locale/Gender-neutral class names FTW
	tp_classes = { [1] = "DRUID",
	               [2] = "HUNTER",
	               [3] = "MAGE",
	               [4] = "PALADIN",
	               [5] = "PRIEST",
	               [6] = "ROGUE",
	               [7] = "SHAMAN",
	               [8] = "WARLOCK",
	               [9] = "WARRIOR", };
end

function Planner_OnEvent( event )
	if ( event == "VARIABLES_LOADED" ) then
		if ( not tp_ver ) then
			tp_ver = 0.0;
		end
		
		if ( not tp_talents ) then
			tp_talents = {};
		end
		
		if ( not tp_saved ) then
			tp_saved = {};
		end
		
		for i = 1, MAX_NUM_TALENTS, 1 do
			local talent = getglobal( "PlayerTalentFrameTalent"..i );
			local rankBorder = getglobal( "PlayerTalentFrameTalent"..i.."RankBorder" );
			rankBorder:SetWidth( 64 );
			
			local goalRankBorder = talent:CreateTexture( "PlannerRankBorder"..i, "OVERLAY" );
			goalRankBorder:SetTexture( rankBorder:GetTexture() );
			goalRankBorder:SetPoint( "CENTER", talent, "BOTTOMLEFT" );
			
			local goalRank = talent:CreateFontString( "PlannerRank"..i, "OVERLAY" );
			goalRank:SetFontObject( GameFontNormalSmall );
			goalRank:SetPoint( "CENTER", talent, "BOTTOMLEFT" );
		end
	elseif ( event == "PLAYER_ENTERING_WORLD" ) then
		tp_name = UnitName( "player" ).." of "..GetRealmName();
		_, class = UnitClass( "player" );
		
		if ( tp_ver ~= PLANNER_VERSION ) then
			Planner_Upgrade();
		end

		for k, v in ipairs( tp_classes ) do
			if ( v == class ) then
				tp_class = k;
			end
		end
		
		if ( not tp_talents[tp_name] ) then
			tp_talents[tp_name] = { [1] = {}, [2] = {}, [3] = {}, };
		end
	elseif ( tp_name ) then
		if ( event == "PLAYER_LEAVING_WORLD" ) then
			tp_name = nil;
			tp_class = nil;
		end
	end
end

function Planner_DoSlash( msg )
	if ( not msg or msg == "" ) then
		for i = 1, MAX_NUM_TALENTS, 1 do
			local rankBorder = getglobal( "PlayerTalentFrameTalent"..i.."RankBorder" );
			
			local goalRank = getglobal( "PlannerRank"..i );
			local goalRankBorder = getglobal( "PlannerRankBorder"..i );
			
			if ( tp_standby ) then
				rankBorder:SetWidth( 64 );
				goalRank:Show();
				goalRankBorder:Show();
			else
				rankBorder:SetWidth( 32 );
				goalRank:Hide();
				goalRankBorder:Hide();
			end
		end
		
		tp_standby = not tp_standby;
		
		GameTooltip:Hide();
		TalentFrame_Update( PlayerTalentFrame );
		Planner_Log( PLANNER_PLANNER..( tp_standby and PLANNER_DE or PLANNER_EN ) );
	end
	
	local words = {};
	for word in string.gmatch( msg, "[^ ]+" ) do
		table.insert( words, word );
	end
	local cmd = words[1];
	local prm = words[2];
	
	if ( cmd == "reset" ) then
		tp_talents[tp_name] = { [1] = {}, [2] = {}, [3] = {}, };
		
		GameTooltip:Hide();
		TalentFrame_Update( PlayerTalentFrame );
		Planner_Log( PLANNER_REPORT_FORNAME..tp_name..PLANNER_REPORT_RESET );
	elseif ( cmd == "current" ) then
		tp_talents[tp_name] = { [1] = {}, [2] = {}, [3] = {}, };

		for page = 1, 3, 1 do
			for id = 1, MAX_NUM_TALENTS, 1 do
				_, _, row, col, rank = GetTalentInfo( page, id );
				if ( rank > 0 ) then
					tp_talents[tp_name][page][id] = rank;
				end
			end
		end
		
		GameTooltip:Hide();
		TalentFrame_Update( PlayerTalentFrame );
		Planner_Log( PLANNER_REPORT_FORNAME..tp_name..PLANNER_REPORT_CURRENT );
	elseif ( cmd == "save" ) then
		if ( not prm or prm == "" ) then
			return;
		end
		
		tp_saved[tp_name.."::"..prm] = { [1] = {}, [2] = {}, [3] = {}, };
		
		for page = 1, 3, 1 do
			tp_saved[tp_name.."::"..prm][page] = {};
			
			for k, v in pairs( tp_talents[tp_name][page] ) do
				tp_saved[tp_name.."::"..prm][page][k] = v;
			end
		end
		
		GameTooltip:Hide();
		TalentFrame_Update( PlayerTalentFrame );
		Planner_Log( PLANNER_REPORT_SETNAME.."\""..prm.."\""..PLANNER_REPORT_SAVED );
	elseif ( cmd == "load" ) then
		if ( not prm or prm == "" ) then
			return;
		end
		
		if ( tp_saved[tp_name.."::"..prm] ) then
			tp_talents[tp_name] = { [1] = {}, [2] = {}, [3] = {}, };
		
			for page = 1, 3, 1 do
				for k, v in pairs( tp_saved[tp_name.."::"..prm][page] ) do
					tp_talents[tp_name][page][k] = v;
				end
			end
		else
			Planner_Log( PLANNER_REPORT_SETNAME.."\""..prm.."\""..PLANNER_REPORT_NOTFOUND );
			return;
		end
		
		GameTooltip:Hide();
		TalentFrame_Update( PlayerTalentFrame );
		Planner_Log( PLANNER_REPORT_SETNAME.."\""..prm.."\""..PLANNER_REPORT_LOADED );
	elseif ( cmd == "delete" ) then
		if ( not prm or prm == "" ) then
			return;
		end
		
		if ( tp_saved[tp_name.."::"..prm] ) then
			tp_saved[tp_name.."::"..prm] = nil;
		else
			Planner_Log( PLANNER_REPORT_SETNAME.."\""..prm.."\""..PLANNER_REPORT_NOTFOUND );
			return;
		end
		
		GameTooltip:Hide();
		TalentFrame_Update( PlayerTalentFrame );
		Planner_Log( PLANNER_REPORT_SETNAME.."\""..prm.."\""..PLANNER_REPORT_DELETED );
	elseif ( cmd == "list" ) then
		Planner_Log( PLANNER_REPORT_LIST..tp_name..":" );
		
		local list = { n = 0, };
		local namelen = string.len( tp_name ) + 2;
		for name, set in pairs( tp_saved ) do
			if ( string.sub( name, 1, namelen ) == tp_name.."::" ) then
				table.insert( list, string.sub( name, namelen + 1 ) );
			end
		end
		
		if ( #list == 0 ) then
			Planner_Log( PLANNER_REPORT_NOLIST );
		else
			for k, v in ipairs( list ) do
				Planner_Log( PLANNER_CYAN..v );
			end
		end
		
		list = nil;
	end
end

function Planner_GetCount( page, row )
	local count = 0;
	
	if ( page and row ) then
		for i = 1, MAX_NUM_TALENTS, 1 do
			_, _, tier = GetTalentInfo( page, i );
			if ( tier < row and tp_talents[tp_name][page][i] ) then
				count = count + tp_talents[tp_name][page][i];
			end
		end
	elseif ( page ) then
		return Planner_GetCount( page, MAX_NUM_TALENT_TIERS );
	else
		return Planner_GetCount( 1 ) + Planner_GetCount( 2 ) + Planner_GetCount( 3 );
	end
	
	return count;
end

function Planner_GetRank( page, id )
	if ( tp_talents[tp_name][page] and tp_talents[tp_name][page][id] )  then
		return tp_talents[tp_name][page][id];
	else
		return 0;
	end
end

function Planner_GetPrereq( page, id )
	local reqRow, reqCol = GetTalentPrereqs( page, id );
	
	if ( reqRow and reqCol ) then
		reqIndex = PlayerTalentFrame.TALENT_BRANCH_ARRAY[reqRow][reqCol].id;
		reqName, _, _, _, _, reqNum = GetTalentInfo( page, reqIndex );

		return reqIndex, reqName, reqNum;
	end
end

function Planner_TalentFrame_AfterUpdate( frame )
	if ( frame ~= PlayerTalentFrame ) then
		return;
	end
	
	if ( tp_standby or not tp_name or not tp_class ) then
		return;
	end
	
	local page = PanelTemplates_GetSelectedTab( PlayerTalentFrame );
	local numTalents = GetNumTalents( page );
	
	for id = 1, MAX_NUM_TALENTS, 1 do
		if ( id <= numTalents ) then
			local name, texture, row, col, rank, maxRank = GetTalentInfo( page, id );
			local tabName, _, pointsSpent = GetTalentTabInfo( page );
			local isGlowing = getglobal( "PlayerTalentFrameTalent"..id.."RankBorder" ):IsVisible();
			local freePoints = UnitCharacterPoints( "player" );
			local goalRank = Planner_GetRank( page, id );
			local pagePoints = Planner_GetCount( page );
			local totalPoints = Planner_GetCount();

			local color = nil;
			if ( rank > goalRank ) then
				color = INVALID;
			elseif ( rank < goalRank ) then
				if ( freePoints > 0 and isGlowing ) then
					color = BUYMENOW;
				else
					if ( goalRank == maxRank ) then
						color = MAXPLANNED;
					else
						color = PLANNED;
					end
				end
			else
				if ( rank == maxRank ) then
					color = MAXEDOUT;
				elseif ( freePoints > 0 and isGlowing ) then
					color = DONTBUYME;
				else
					if ( goalRank == 0 ) then
						if ( Planner_CanInc( page, id ) ) then
							color = NORMAL;
						end
					else
						color = MAXEDOUT;
					end
				end
			end

			if ( color ) then
				getglobal( "PlayerTalentFrameTalent"..id.."Slot" ):SetVertexColor( color.r, color.g, color.b );
				getglobal( "PlayerTalentFrameTalent"..id.."RankBorder" ):Show();
				getglobal( "PlayerTalentFrameTalent"..id.."Rank" ):SetTextColor( color.r, color.g, color.b );
				getglobal( "PlayerTalentFrameTalent"..id.."Rank" ):SetText( rank.."/"..maxRank.." " );
				getglobal( "PlayerTalentFrameTalent"..id.."Rank" ):Show();
			else
				getglobal( "PlayerTalentFrameTalent"..id.."RankBorder" ):Hide();
				getglobal( "PlayerTalentFrameTalent"..id.."Rank" ):Hide();
			end
			
			if ( color and goalRank > 0 ) then
				getglobal( "PlannerRankBorder"..id ):Show();
				getglobal( "PlannerRank"..id ):SetTextColor( color.r, color.g, color.b );
				getglobal( "PlannerRank"..id ):SetText( goalRank );
				getglobal( "PlannerRank"..id ):Show();
			else
				getglobal( "PlannerRankBorder"..id ):Hide();
				getglobal( "PlannerRank"..id ):Hide();
			end

			local pagePointsText = HIGHLIGHT_FONT_COLOR_CODE..pointsSpent.." ("..pagePoints..PLANNER_PLANNED..FONT_COLOR_CODE_CLOSE;
			PlayerTalentFrameSpentPoints:SetText( tabName.." Talent Points: "..pagePointsText );

			local pointsToSpend = freePoints.." ("..(61 - totalPoints)..PLANNER_UNPLANNED;
			PlayerTalentFrameTalentPointsText:SetText( pointsToSpend );
		else
			getglobal( "PlannerRank"..id ):Hide();
		end
	end
end

function Planner_OnClick()
	if ( tp_standby or not tp_name or not tp_class ) then
		Planner_TalentFrameTalent_OnClick();
		return;
	end

	local page = PanelTemplates_GetSelectedTab( PlayerTalentFrame );
	local id = this:GetID();
	
	if ( arg1 == "LeftButton" ) then
		if ( IsModifiedClick( "CHATLINK" ) ) then
			local link = GetTalentLink( page, id );
			
			if ( link ) then
				ChatEdit_InsertLink( link );
			end
		elseif ( IsControlKeyDown() ) then
			LearnTalent( page, id );
		else
			Planner_IncTalent( page, id );
		end
	elseif ( arg1 == "RightButton" ) then
		Planner_DecTalent( page, id );
	end
end

function Planner_OnEnter()
	Planner_TalentFrameTalent_OnEnter();

	if ( tp_standby or not tp_name or not tp_class ) then
		return;
	end

	local page = PanelTemplates_GetSelectedTab( PlayerTalentFrame );
	local id = this:GetID();
	
	local name, texture, row, col, rank, maxRank = GetTalentInfo( page, id );
	local talent = tp_webdata[tp_class][page][id];
	local goalRank = Planner_GetRank( page, id );
	
	if ( ( goalRank == 1 and rank == 0 ) or ( goalRank == maxRank and rank == maxRank ) ) then
		local line2 = getglobal( "GameTooltipTextLeft2" );
		line2:SetText( PLANNER_PLANNED_RANK..goalRank.."/"..maxRank );
	elseif ( ( goalRank > 0 and goalRank < maxRank ) or ( goalRank == maxRank and rank < maxRank ) ) then
		GameTooltip:AddLine( " " );
		GameTooltip:AddLine( PLANNER_PLANNED_RANK..goalRank.."/"..maxRank, 1.0, 1.0, 1.0 );
		GameTooltip:AddLine( talent[goalRank], 1.0, 0.82, 0.0, 1.0, 1 );
		GameTooltip:Show();
	end
end

function Planner_IncTalent( page, id )
	local goalRank = Planner_GetRank( page, id );
	local pagePoints = Planner_GetCount( page );
	
	if ( Planner_CanInc( page, id ) ) then
		tp_talents[tp_name][page][id] = goalRank + 1;
		TalentFrame_Update( PlayerTalentFrame );
		Planner_OnEnter();
	end
end

function Planner_DecTalent( page, id )
	local goalRank = Planner_GetRank( page, id );
	
	if ( Planner_CanDec( page, id ) ) then
		if ( goalRank > 1 ) then
			tp_talents[tp_name][page][id] = goalRank - 1;
		else
			tp_talents[tp_name][page][id] = nil;
		end
		TalentFrame_Update( PlayerTalentFrame );
		Planner_OnEnter();
	end
end

function Planner_CanInc( page, id )
	local name, texture, row, col, rank, maxRank = GetTalentInfo( page, id );
	
	local talent = tp_webdata[tp_class][page][id];
	local goalRank = Planner_GetRank( page, id );
	local points = Planner_GetCount( page, row );
	local totalPoints = Planner_GetCount();
	
	-- Ensure we've not already maxed this talent and that we have points
	if ( goalRank == maxRank or totalPoints == 61 ) then
		return nil;
	end
	
	-- Ensure we've spent enough points to be on this row
	if ( ( (row - 1) * 5 > points ) ) then
		return nil;
	end
	
	-- Ensure we've done the listed prereqs
	local reqIndex, _, reqNum = Planner_GetPrereq( page, id );
	if ( reqIndex and tp_talents[tp_name][page][reqIndex] ~= reqNum ) then
		return nil;
	end
	
	return true;
end

function Planner_CanDec( page, id )
	local name, texture, row, col, rank, maxRank = GetTalentInfo( page, id );
	
	local talent = tp_webdata[tp_class][page][id];
	local goalRank = Planner_GetRank( page, id );
	
	-- Ensure we have goal ranks in this talent
	if ( goalRank == 0 ) then
		return nil;
	end
	
	-- Ensure we don't need this talent for future rows
	local numTalents = GetNumTalents( page );
	for i = id, numTalents, 1 do
		local reqName, _, tier = GetTalentInfo( page, i );
		local points = Planner_GetCount( page, tier );
		
		if ( tp_talents[tp_name][page][i] and row < tier and points <= (tier - 1) * 5 ) then
			return nil;
		end
		
		local reqRow, reqCol = GetTalentPrereqs( page, i );		
		if ( reqRow and reqCol ) then
			if ( row == reqRow and col == reqCol and tp_talents[tp_name][page][i] ) then
				return nil;
			end
		end
	end
	
	return true;
end

function Planner_Upgrade()
	if ( tp_ver == 0.0 ) then
		local name = UnitName( "player" );
		
		-- Move talents with this name and no realm to this character
		if ( tp_talents[name] ) then
			tp_talents[tp_name] = tp_talents[name];
			tp_talents[name] = nil;
		end

		-- Move saved sets with this name and no realm to this character
		tp_tmp = {};
		
		for k, v in pairs( tp_saved ) do
			for who, set in string.gmatch( k, "([^:]+)::(.+)" ) do
				if ( who == name ) then
					tp_tmp[tp_name.."::"..set] = v;
				else
					tp_tmp[k] = v;
				end
			end
		end
		
		tp_saved = tp_tmp;

		-- Check for other un-updated characters before promoting version
		for k, v in pairs( tp_talents ) do
			if ( not string.find( k, " of " ) ) then
				return;
			end
		end
		
		tp_ver = 2.3;
	end
end

function Planner_Log( text )
	SELECTED_CHAT_FRAME:AddMessage( text );
end

function Planner_Debug( text, r, g, b )
	ChatFrame3:AddMessage( text, r, g, b );
end
