
-- API Notes for addon authors looking to automatically add a page to Omnibus, or run a page from within Omnibus:
--	function Omnibus:RunPageInFolio(page,folio)
--		finds the page titled [page] in the folio titled [folio], and if it exists, executes it as defined by the page's tag
--	function Omnibus:ShowPageInFolio(page,folio)
--		finds the page titled [page] in the folio titled [folio], and displays it in the Omnibus editor window for editing.
--	function Omnibus:NewPageInFolio(folio,title,content,tag)
--		Creates a new page in the folio titled [folio], given the title indicated by [title], the page content from [content],
--		and the tag [tag] (which must be a valid tag such as "Lua")
--		Note, title, content, and tag may be set to nil, and each item will default to a default value.
--	function Omnibus:NewFolio(title)
--		Creates a new Folio with the given [title].  It will not create a new folio
--		with the same name as an older folio.
--	function Omnibus:GetPageText(pageid)
--		returns the text of the page indicated by [pageid], which must be retreived by Omnibus:FindPageInFolioIndex(page,folio)
--	function Omnibus:SetPageText(pageid,text)
--		sets the text of the given page with the new [text], if text is a string.
--	function Omnibus:FindPageInFolioIndex(page,folio)
--		returns the pageid used by GetPageText and SetPageText.  You should Find the pageid each time you need it within a given function call.
--	function Omnibus:FolioPages(folio)
--		Use in for loops like this:   for pageid,pagetitle in Omnibus:FolioPages(foliotitle) do ... end


local function debug(msg) end
local OB_RELEASE = nil
if not OB_RELEASE then
	debug = function(msg) DEFAULT_CHAT_FRAME:AddMessage(msg) end
end

-- current number of temp frames for the dropdown.
local maxtempframes = 0

local L = AceLibrary("AceLocale-2.2"):new("Omnibus")
local tablet = AceLibrary("Tablet-2.0")
local dewdrop = AceLibrary("Dewdrop-2.0")
local OBIndentationLib = IndentationLib

BINDING_HEADER_OMNIBUS = L["Omnibus"]
BINDING_NAME_OBUS_TOGGLE = L["Toggle Omnibus Window"]
BINDING_NAME_OBUS_EXEC = L["Execute Current Page"]
BINDING_NAME_OBUS_NEXTPAGE = L["Show Next Page In Folio"]
BINDING_NAME_OBUS_PREVPAGE = L["Show Previous Page In Folio"]

-- list of fonts to cycle through
local fonts = {
	{"Interface\\AddOns\\Omnibus\\VeraMono.ttf","Vera Monospace"},
	{"Fonts\\FRIZQT__.TTF","Friz QT"},
	{"Fonts\\ARIALN.TTF","Arial"},
	{"Fonts\\MORPHEUS.ttf","Morpheus"},
	{"Interface\\AddOns\\Omnibus\\JANCIENT.TTF","JSL Ancient"},
	{"Interface\\AddOns\\Omnibus\\JANCIENI.TTF","JSL Ancient Italic"},
	-- add fonts here {"fontpath","Font Display Name"},
}

--local useCart = false
--if Cartographer and Cartographer_Notes then
--	Omnibus = Cartographer:NewModule("Quests", "AceEvent-2.0", "AceDB-2.0", "FuBarPlugin-2.0", "AceComm-2.0", "AceConsole-2.0", "AceHook-2.1")
--	useCart = true
--else
	Omnibus = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "FuBarPlugin-2.0", "AceComm-2.0", "AceConsole-2.0")
--end

-- changed by profalbert to prevent a nil-error when trying to access "self".
-- occured when running "/omnibus run myscript"
local Omnibus, self = Omnibus, Omnibus
-- oldcode:
--		local Omnibus = Omnibus

Omnibus.hasIcon = true
Omnibus.defaultPosition = "RIGHT"
Omnibus.defaultMinimapPosition = 270
Omnibus.tooltipHiddenWhenEmpty = true
Omnibus.clickableTooltip = true
Omnibus.hideWithoutStandby = true
function Omnibus:GetText() return L["Omnibus"] end

Omnibus:RegisterDB("OmnibusDB")
Omnibus:RegisterDefaults("account",{
	Font = 2,
	book = {{"Untitled",""}},
	garbage = {},
	folios = {},
	autoruns = {},
	customtags = {
		{tagID = "Text",font = "Fonts\\MORPHEUS.ttf",fsize = 16,execFunc = "Omnibus.RunTextTag"},
		{tagID = "Lua",font = "Interface\\AddOns\\Omnibus\\VeraMono.ttf",fsize = 16,
			showFunc = "Omnibus.EnableFAIAP",hideFunc = "Omnibus.DisableFAIAP",execFunc = "Omnibus.RunLuaTag"},
	},
})

Omnibus.OnMenuRequest = {
	type = 'group',
	args = {
		toggle = {
			name = L["Toggle"],
			type = "execute",
			desc = L["Toggle the Omnibus Window"],
			func = function()
				Omnibus:Toggle()
			end
		},
		run = {
			name = L["Run"],
			type = "text",
			desc = L["Execute the first page found with a given Title"],
			get = false,
			set = function(str) Omnibus:TagExec(self.db.account.current_page) end,
			usage = L["Execute the first page found with a given Title"],
		},
		import = {
			name = L["Import Pages"],
			type = "execute",
			desc = L["Import pages from TinyPad."],
			func = function()
				Omnibus:ImportTinyPadPages()
			end
		},
	}
}
Omnibus:RegisterChatCommand({ "/omnibus", "/ob" }, Omnibus.OnMenuRequest)

-- Table used to fake a nil in the DewDrop menu... Shouldn't ever be used elsewhere.
local OBFAKENIL = {}

function Omnibus:FrameMenuFunc_MoveToFolios(level,v,v_1,v_2,v_3,v_4)
	v_1 = v_1 or self.db.account.current_page
	local page = self.db.account.book[v_1]
	if page.folio then
		dewdrop:AddLine('text',L["Default Folio"],'func',function() Omnibus:MovePageToFolio(page,v_1,self.db.account.default_folio) end,
			'closeWhenClicked',true,'tooltipTitle',L["Move Page"],'tooltipText',L["Move the page to the Default Folio"])
	end
	for i,folio in ipairs(self.db.account.folios) do
		if page.folio ~= folio.title then
			dewdrop:AddLine('text',folio.title,'func',function() Omnibus:MovePageToFolio(page,v_1,folio) end,'closeWhenClicked',true,
				'tooltipTitle',L["Move Page"],'tooltipText',string.format(L["Move the page to the %s Folio"],folio.title))
		end
	end
end

function Omnibus:FrameMenuFunc_PageTag(level,v,v_1,v_2,v_3,v_4)
	v_1 = v_1 or self.db.account.current_page
	local curpage = v_1 == self.db.account.current_page
	local page = self.db.account.book[v_1]
	local tag = page.tag or "Text"
	local _t = {}
	for tagID in pairs(self.tags) do _t[#_t+1] = tagID end
	table.sort(_t)
	for i,tagID in ipairs(_t) do
		local _tagID = tagID
		dewdrop:AddLine('text',tagID,'func',function() Omnibus:SetPageTag(v_1,_tagID) end,'checked',tag == tagID,
			'closeWhenClicked',true,'tooltipTitle',L["Tag Page"],'tooltipText',string.format(L["Tag this page with the %s tag"],tagID))
	end
end

function Omnibus:FrameMenuFunc_Page(level,v,v_1,v_2,v_3,v_4)
	local page = self.db.account.book[v]
	local curpage = self.db.account.current_page
	dewdrop:AddLine('text',L["Rename"],'hasArrow',true,'hasEditBox',true,'editBoxText',page[1],
		'editBoxFunc',function(newname)
			Omnibus:RenamePage(v,newname)
		end,'closeWhenClicked',true,'tooltipTitle',L["Rename Page"],'tooltipText',L["Rename the Page"])
	dewdrop:AddLine('text',L["Execute"],'func',function()
			if v == curpage then Omnibus:SavePage(true) end
			Omnibus:TagExec(v)
		end,'closeWhenClicked',true,'tooltipTitle',L["Execute Page"],'tooltipText',L["Run the page, as defined by it's Tag"])
	dewdrop:AddLine('text',L["Delete"],'func',function() StaticPopup_Show("OMNIBUSCONFIRM").data = v end,'closeWhenClicked',true,
		'tooltipTitle',L["Delete Page"],'tooltipText',L["Delete the Page"])
	dewdrop:AddLine('text',L["Move To Folio"],'hasArrow',true,'value',{Omnibus.FrameMenuFunc_MoveToFolios,v},
		'tooltipTitle',L["Move Page"],'tooltipText',L["Move the page to a different Folio"])
	dewdrop:AddLine('text',L["Tag"],'hasArrow',true,'value',{Omnibus.FrameMenuFunc_PageTag,v},
		'tooltipTitle',L["Tag Page"],'tooltipText',L["Change the Page's Tag"])
	dewdrop:AddLine('text',L["Send To"],'hasArrow',true,'hasEditBox',true,'editBoxText',"",
		'editBoxFunc',function(recip)
			Omnibus:SendTo(recip)
		end,'closeWhenClicked',true,
		'tooltipTitle',L["Send Page"],'tooltipText',L["Send the Page to another Omnibus User"])
	dewdrop:AddLine('text',L["Autorun on Login"],'func',function()
			page.autorun = (not page.autorun) or nil
			self.db.account.autoruns[v] = page.autorun
		end,'checked',page.autorun,
		'tooltipTitle',L["Autorun Page"],'tooltipText',L["Sets the page to automatically run on Login and ReloadUI"])
end

function Omnibus:FrameMenuFunc_DefaultTag(level,v,v_1,v_2,v_3,v_4)
	-- display each Tag. v_1 SHOULD be the folio to set the default tag on...
	local folio
	if v_1 == OBFAKENIL then
		folio = self.db.account.default_folio
	else
		folio = Omnibus:FindFolio(v_1)
	end
	local _t = {}
	for tagID in pairs(self.tags) do _t[#_t+1] = tagID end
	table.sort(_t)
	for i,tagID in ipairs(_t) do
		dewdrop:AddLine('text',tagID,'func',function() Omnibus:SetDefaultTag(folio,tagID) end, 'checked',tagID == (folio.tag or "Text"),
			'closeWhenClicked',true,'tooltipTitle',string.format(L["Default to %s"],tagID),'tooltipText',
			string.format(L["Sets the Folio's Default Tag to the %s Tag"],tagID))
	end
end

function Omnibus:FrameMenuFunc_FolioTag(level,v,v_1,v_2,v_3,v_4)
	-- display each Tag. v_1 SHOULD be the folio to set the default tag on...
	local folio
	if v_1 == OBFAKENIL then
		folio = self.db.account.default_folio
	else
		folio = Omnibus:FindFolio(v_1)
	end
	local _t = {}
	for tagID in pairs(self.tags) do _t[#_t+1] = tagID end
	table.sort(_t)
	for i,tagID in ipairs(_t) do
		dewdrop:AddLine('text',tagID,'func',function() Omnibus:ForEachInFolio(folio,Omnibus.SetPageTag,Omnibus,tagID) end,
			'closeWhenClicked',true,'tooltipTitle',string.format(L["Set Pages to %s"],tagID),'tooltipText',
			string.format(L["Sets all Pages in this Folio to the %s Tag"],tagID))
	end
end

function Omnibus:FrameMenuFunc_Folios(level,v,v_1,v_2,v_3,v_4)
	if level == 1 then
		-- Display current Folios, including the Default Folio.
		dewdrop:AddLine('text',L["Default Folio"],'hasArrow',true,'value',OBFAKENIL,
			'func',function() Omnibus:SetPage(self.db.account.default_folio[1]) end,'closeWhenClicked',true)
		if self.db.account.folios then
			for i,folio in ipairs(self.db.account.folios) do
				dewdrop:AddLine('text',folio.title,'hasArrow',true,'value',folio.title,'func',function() Omnibus:SetPage(folio[1]) end,'closeWhenClicked',true)
			end
		end
		-- Display Option to create a new Folio.
		dewdrop:AddLine('text',L["New Folio"],'func',function() Omnibus:NewFolio() end,'closeWhenClicked',true,
		'tooltipTitle',L["New Folio"],'tooltipText',L["Create a new Folio"])
	elseif level == 2 then
		-- Display the Pages in the folio Identified by v
		local folio
		if v == OBFAKENIL then
			folio = self.db.account.default_folio
		else
			folio = Omnibus:FindFolio(v)
		end
		for _,pnum in ipairs(folio) do
			dewdrop:AddLine('text',self.db.account.book[pnum][1],'func',function() Omnibus:SetPage(pnum) end,'closeWhenClicked',true,
				'hasArrow',true,'value',function(Omnibus,level,v,v_1,v_2,v_3,v_4) Omnibus:FrameMenuFunc_Page(level,pnum,v_1,v_2,v_3,v_4) end,
				'tooltipTitle',L["Edit Page"],'tooltipText',L["Click to edit this page, or make changes to the page in the sub-menu"])
		end
		-- Display the newpage button, which should make a new page in this folio.
		dewdrop:AddLine('text',"")
		dewdrop:AddLine('text',L["New Page"],'func',function() Omnibus:SetPage(Omnibus:NewPageInFolio(folio)) end,'closeWhenClicked',true,
			'hasArrow',true,'hasEditBox',true,'editBoxText',"Untitled",'editBoxFunc',function(newpagetitle)
				Omnibus:SetPage(Omnibus:NewPageInFolio(folio,newpagetitle))
			end,'tooltipTitle',L["New Page"],'tooltipText',L["Click to create an Untitled page, or type a Name for the new Page and hit Enter"])
		if v ~= OBFAKENIL then
			dewdrop:AddLine('text',L["Rename"],'hasArrow',true,'hasEditBox',true,'editBoxText',folio.title,
				'editBoxFunc',function(newname)
					Omnibus:RenameFolio(folio,newname)
				end,'closeWhenClicked',true,
				'tooltipTitle',L["Rename Folio"],'tooltipText',L["Rename this Folio"])
		end
		dewdrop:AddLine('text',L["Set Page Template"],'func',function() Omnibus:SetPageTemplate(folio) end,'closeWhenClicked',true,
			'tooltipTitle',L["Set Page Template"],'tooltipText',L["Set the default text for new pages in this Folio"])
		dewdrop:AddLine('text',L["Set Default Page Tag"],'hasArrow',true,'value',Omnibus.FrameMenuFunc_DefaultTag,
			'tooltipTitle',L["Set Default Page Tag"],'tooltipText',L["Set the default tag for new pages in this Folio"])
		dewdrop:AddLine('text',L["Set Tag on Folio Pages"],'hasArrow',true,'value',Omnibus.FrameMenuFunc_FolioTag,
			'tooltipTitle',L["Set Tag on Folio Pages"],'tooltipText',L["Set the tag on all pages in this Folio"])
		-- Display option to delete Folio, and make sure it pops a warning that all Pages within will be deleted too.
		if v ~= OBFAKENIL then
			dewdrop:AddLine('text',"")
			dewdrop:AddLine('text',L["Delete Folio"],'func',function() Omnibus:DeleteFolio(folio) end,'closeWhenClicked',true,
				'tooltipTitle',L["Delete Folio"],'tooltipText',L["Delete this Folio, and all Pages within it"])
		end
	end
end

function Omnibus:FrameMenuFunc_ChooseFont(level,v,v_1,v_2,v_3,v_4)
	local ctag = v_1
	for i,font in ipairs(fonts) do
		local f = font
		dewdrop:AddLine('text',f[2],'func',function() ctag.font = f[1] end,'checked',ctag.font == f[1],
			'tooltipTitle',L["Set Tag Font"],'tooltipText',string.format(L["Sets the Tag's display font to %s"],f[2]))
	end
end

function Omnibus:FrameMenuFunc_TagDetail(level,v,v_1,v_2,v_3,v_4)
	local ctag = v_1
	self:AddTag(ctag.tagID,ctag.font,ctag.fsize,ctag.showFunc,ctag.hideFunc,ctag.execFunc)
	dewdrop:AddLine('text',L["Font"],'hasArrow',true,'value',{Omnibus.FrameMenuFunc_ChooseFont,ctag},
		'tooltipTitle',L["Set Tag Font"],'tooltipText',L["Set the Tag's display font"])
	dewdrop:AddLine('text',L["Text Size"],'hasArrow',true,'hasSlider',true,'sliderMin',8,'sliderMax',24,'sliderValue',ctag.fsize,
		'sliderStep',1,'sliderFunc',function(newfsize) ctag.fsize = newfsize end,
		'tooltipTitle',L["Set Text Size"],'tooltipText',L["Set the Tag's text size"])
	dewdrop:AddLine('text',L["OnShow Function"],'hasArrow',true,'hasEditBox',true,'editBoxText',ctag.showFunc,
		'editBoxFunc',function(newshowFunc) ctag.showFunc = newshowFunc if ctag.showFunc == "" then ctag.showFunc = nil end end,
		'tooltipTitle',L["Set OnShow Function"],'tooltipText',L["Sets the Tag's OnShow function"])
	dewdrop:AddLine('text',L["OnHide Function"],'hasArrow',true,'hasEditBox',true,'editBoxText',ctag.hideFunc,
		'editBoxFunc',function(newhideFunc) ctag.hideFunc = newhideFunc if ctag.hideFunc == "" then ctag.hideFunc = nil end end,
		'tooltipTitle',L["Set OnHide Function"],'tooltipText',L["Sets the Tag's OnHide function"])
	dewdrop:AddLine('text',L["OnExec Function"],'hasArrow',true,'hasEditBox',true,'editBoxText',ctag.execFunc,
		'editBoxFunc',function(newexecFunc) ctag.execFunc = newexecFunc if ctag.execFunc == "" then ctag.execFunc = nil end end,
		'tooltipTitle',L["Set OnExec Function"],'tooltipText',L["Sets the Tag's OnExec function"])
end

function Omnibus:FrameMenuFunc_TagSettings(level,v,v_1,v_2,v_3,v_4)
	for i,ctag in ipairs(self.db.account.customtags) do
		dewdrop:AddLine('text',ctag.tagID,'hasArrow',true,'value',{Omnibus.FrameMenuFunc_TagDetail,ctag},
			'tooltipTitle',L["Tag Settings"],'tooltipText',"Set the "..ctag.tagID.." Tag's settings")
	end
	-- add the new tag line.
	dewdrop:AddLine('text',"")
	dewdrop:AddLine('text',L["New Tag"],'hasArrow',true,'hasEditBox',true,'editBoxText',"",
		'editBoxFunc',function(newtagname) Omnibus:NewTag(newtagname) end,
		'tooltipTitle',L["Create a New Tag"],'tooltipText',L["Creates a new tag"])
end

function Omnibus:FrameMenuFunc(level,v,v_1,v_2,v_3,v_4)
	if level == 1 then
		-- Add Current Page and current Folio settings here too, so it is easier to work with the current stuff.
		dewdrop:AddLine('text',L["Search"],'hasArrow',true,'hasEditBox',true,'editBoxText',"",
			'editBoxFunc',function(search)
				Omnibus:DoSearch(search)
			end,'closeWhenClicked',true,
			'tooltipTitle',L["Search"],'tooltipText',L["Search for text within all pages"])
		if type(TinyPadPages) == "table" then
			dewdrop:AddLine('text',"")
			dewdrop:AddLine('text',L["Import Pages"],'func',function()
					Omnibus:ImportTinyPadPages()
				end,'closeWhenClicked',true,
				'tooltipTitle',L["Import Pages"],'tooltipText',L["Import TinyPad Pages"])
		end
		if self.db.account.current_page then
			dewdrop:AddLine('text',"")
			dewdrop:AddLine('text',self.db.account.book[self.db.account.current_page][1],'isTitle',true,
				'icon',"Interface\\AddOns\\Omnibus\\PageIcon",'iconWidth',24,'iconHeight',24)
			Omnibus:FrameMenuFunc_Page(level,self.db.account.current_page,v_1,v_2,v_3,v_4)
		end
		dewdrop:AddLine('text',"")
		local curFolioName = Omnibus:FindFolio(self.db.account.current_folio).title or L["Default Folio"]
		dewdrop:AddLine('text',curFolioName,'isTitle',true,
			'icon',"Interface\\AddOns\\Omnibus\\FolioIcon",'iconWidth',24,'iconHeight',24)
		Omnibus:FrameMenuFunc_Folios(level+1,self.db.account.current_folio or OBFAKENIL,v,v_1,v_2,v_3)
	else
		if type(v) == "table" and type(v[1]) == "function" then v[1](Omnibus,level,v,v[2],v_2,v_3,v_4) return end
		if type(v) == "function" then v(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_1) == "function" then v_1(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_2) == "function" then v_2(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_3) == "function" then v_3(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_4) == "function" then v_4(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
	end
end

function Omnibus:FrameConfigFunc(level,v,v_1,v_2,v_3,v_4)
	if level == 1 then
		dewdrop:AddLine('text',L["Edit Frame Locked"],'func',function()
				self.db.account.Lock = not self.db.account.Lock
				self:UpdateLock()
			end,'checked',self.db.account.Lock,
			'tooltipTitle',L["Frame Lock"],'tooltipText',L["Locks the Size and location of the Omnibus Edit Frame"])
		dewdrop:AddLine('text',L["Tag Settings"],'hasArrow',true,'value',Omnibus.FrameMenuFunc_TagSettings,
			'tooltipTitle',L["Tag Settings"],'tooltipText',L["Make changes to, or add new, Tags"])
	else
		if type(v) == "table" and type(v[1]) == "function" then v[1](Omnibus,level,v,v[2],v_2,v_3,v_4) return end
		if type(v) == "function" then v(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_1) == "function" then v_1(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_2) == "function" then v_2(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_3) == "function" then v_3(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
		if type(v_4) == "function" then v_4(Omnibus,level,v,v_1,v_2,v_3,v_4) return end
	end
end

function Omnibus:OpenFrameMenu()
	if not dewdrop:IsOpen(OmnibusFrame) then
		dewdrop:Open(OmnibusFrame,'children',function(level,v,v_1,v_2,v_3,v_4) Omnibus:FrameMenuFunc(level,v,v_1,v_2,v_3,v_4) end,
			'cursorX',true,'cursorY',true)
	else
		dewdrop:Close()
	end
end

function Omnibus:OpenFolioMenu()
	if not dewdrop:IsOpen(OmnibusFrame) then
		dewdrop:Open(OmnibusFrame,'children',function(level,v,v_1,v_2,v_3,v_4) Omnibus:FrameMenuFunc_Folios(level,v,v_1,v_2,v_3,v_4) end,
			'cursorX',true,'cursorY',true)
	else
		dewdrop:Close()
	end
end

function Omnibus:OpenConfigMenu()
	if not dewdrop:IsOpen(OmnibusFrame) then
		dewdrop:Open(OmnibusFrame,'children',function(level,v,v_1,v_2,v_3,v_4) Omnibus:FrameConfigFunc(level,v,v_1,v_2,v_3,v_4) end,
			'cursorX',true,'cursorY',true)
	else
		dewdrop:Close()
	end
end

function Omnibus:RunOnClick()
	Omnibus:SavePage(true) -- we pass true here to not disable FAIAP
	Omnibus:TagExec(self.db.account.current_page)
end

function Omnibus:GetNewPageIndex()
	if #self.db.account.garbage > 0 then
		local newpage = self.db.account.garbage[#self.db.account.garbage]
		self.db.account.garbage[#self.db.account.garbage] = nil
		return newpage
	else
		return #self.db.account.book + 1
	end
end

StaticPopupDialogs["OMNIBUSCONFIRM"] = {
	text = L["Delete this page?"],
	button1 = L["Yes"],
	button2 = L["No"],
	timeout = 0,
	whileDead = 1,
	OnAccept = function(data) Omnibus:DeletePage(data) end
}

StaticPopupDialogs["OMNIBUSFOLIODELETECONFIRM"] = {
	text = L["Delete this Folio? WARNING: This will delete all pages within this Folio!"],
	button1 = L["Yes"],
	button2 = L["No"],
	timeout = 0,
	whileDead = 1,
	OnAccept = function(data) Omnibus:DeleteFolio(data,true) end
}

StaticPopupDialogs["OMNIBUSNOTINYPAD"] = {
	text = L["Cannot find TinyPad data.  Check that TinyPad is installed and active."],
	button1 = L["OK"],
	timeout = 10,
	whileDead = 1,
}

StaticPopupDialogs["OMNIBUSNOSENDTORESPONSE"] = {
	text = L["%s does not appear to have Omnibus installed.  (No Response)"],
	button1 = L["OK"],
	timeout = 10,
	whileDead = 1,
}

StaticPopupDialogs["OMNIBUSPAGEACCEPTED"] = {
	text = L["%s has accepted your page."],
	button1 = L["OK"],
	timeout = 10,
	whileDead = 1,
}

StaticPopupDialogs["OMNIBUSPAGEDECLINED"] = {
	text = L["%s has declined your page."],
	button1 = L["OK"],
	timeout = 10,
	whileDead = 1,
}

StaticPopupDialogs["OMNIBUSSNDCONFIRM"] = {
	text = L["Send this page to %s?\nWARNING: Users not running AceComm or an Ace2 Addon\nthat uses AceComm will see some strange whispers."],
	button1 = L["Yes"],
	button2 = L["No"],
	timeout = 0,
	whileDead = 1,
	OnAccept = function() Omnibus:SendTo() end
}

StaticPopupDialogs["OMNIBUSRCVCONFIRM"] = {
	text = L["%s has sent you an Omnibus page, Accept it?"],
	button1 = L["Yes"],
	button2 = L["No"],
	timeout = 0,
	whileDead = 1,
	OnAccept = function() Omnibus:AcceptReceivedPage() end,
	OnCancel = function() Omnibus:CancelReceivedPage() end
}

function Omnibus:OnInitialize()
	OmnibusFrame:SetMinResize(436,96)
	table.insert(UISpecialFrames,"OmnibusFrame")
	--table.insert(UISpecialFrames,"OmnibusSearchFrame")
end

function Omnibus:OnEnable()
	self.lastFUpdate = time()
	self.lastGRUpdate = self.lastFUpdate
	self:UpdateLock()
	-- Check for IndentationLib from "For all Indents and Purposes" mod.
	--if type(IndentationLib) == "table" and IndentationLib.revision >= 14 then
		--IndentationLib.addSmartCode(OmnibusEditBox)
	--end
	-- Register for Page Transfer Communications.
	self:RegisterComm("Omnibus","WHISPER")
	self:SetCommPrefix("Omnibus")
	-- Guild roster and friends list update.
	--self:RegisterEvent("PLAYER_GUILD_UPDATE")
	--if IsInGuild() then
		--self:ScheduleRepeatingEvent("ScheduledGuildRoster", GuildRoster, 15)
		--self:RegisterEvent("GUILD_ROSTER_UPDATE")
		--GuildRoster()
	--end
	--self:ScheduleRepeatingEvent(ShowFriends, 15)
	--self:RegisterEvent("FRIENDLIST_UPDATE")
	--ShowFriends()
	-- Build Default Folio table in DB if it doesn't exist.
	if not self.db.account.default_folio then
		self.db.account.default_folio = {}
		self.db.account.default_folio.template = self.db.account.template or ""
		for i = 1, #self.db.account.book do
			local page = self.db.account.book[i]
			if not page.folio then
				self.db.account.default_folio[#self.db.account.default_folio+1] = i
			end
		end
	end
	-- Now we run through the Custom Tags from the DB.
	for i,ctag in ipairs(self.db.account.customtags) do
		self:AddTag(ctag.tagID,ctag.font,ctag.fsize,ctag.showFunc,ctag.hideFunc,ctag.execFunc)
	end
	if not self.tags or not self.tags["Lua"] then
		self.db.account.customtags[#self.db.account.customtags + 1] = {
			tagID = "Lua", font = "Interface\\AddOns\\Omnibus\\VeraMono.ttf", fsize = 16,
			showFunc = "Omnibus.EnableFAIAP", hideFunc = "Omnibus.DisableFAIAP", execFunc = "Omnibus.RunLuaTag"
		}
		self:AddTag("Lua","Interface\\AddOns\\Omnibus\\VeraMono.ttf",16,
				"Omnibus.EnableFAIAP","Omnibus.DisableFAIAP","Omnibus.RunLuaTag")
	end
	if not self.tags["Text"] then
		self.db.account.customtags[#self.db.account.customtags + 1] = {
			tagID = "Text", font = "Fonts\\MORPHEUS.ttf", fsize = 16, execFunc = "Omnibus.RunTextTag"
		}
		self:AddTag("Text","Fonts\\MORPHEUS.ttf",16,nil,nil,"Omnibus.RunTextTag")
	end
	-- set up cartographer stuff
--	if useCart then
--		Cartographer_Notes:RegisterIcon("Omnibus",{
--			text = L["Omnibus"],
--			path = "Interface\\AddOns\\Omnibus\\OmnibusIcon"
--		})
--		self.db.account.cart = self.db.account.cart or {}
--		Cartographer_Notes:RegisterNotesDatabase("OmnibusDB",self.db.account.cart,self)
--		Cartographer_Notes:RefreshMap(true)
--	end
	-- Now finally we process autoruns.
	for pnum,run in pairs(self.db.account.autoruns) do
		if run then
			Omnibus:TagExec(pnum)
		end
	end
end

function Omnibus:OnDisable()
	if OmnibusFrame:IsVisible() then
		Omnibus:SavePage()
	end
--	Cartographer_Notes:UnregisterNotesDatabase("OmnibusDB")
--	Cartographer_Notes:UnregisterIcon("Omnibus")
--	Cartographer_Notes:RefreshMap(true)
end

function Omnibus:ImportTinyPadPage(page)
	Omnibus:SetPage(Omnibus:NewPageInFolio("TinyPad Imports",format(L["TinyPad Page %d"],page),TinyPadPages[page]))
	if TinyPadPages[page+1] then
		self:ScheduleEvent(self.ImportTinyPadPage,0.1,self,page+1)
	else
		self:UpdateTooltip()
	end
end

function Omnibus:ImportTinyPadPages()
	-- Save The current page...
	Omnibus:SavePage()
	if type(TinyPadPages) == "table" then
		Omnibus:NewFolio("TinyPad Imports")
		self:ImportTinyPadPage(1)
	else
		StaticPopup_Show("OMNIBUSNOTINYPAD")
	end
end

function Omnibus:OnCommReceive(prefix, sender, dist, cmd, title, text, z)
	if cmd == "PAGE" then
		local new_page = {[1] = title, [2] = text, z = z, sender = sender}
		self.rcvbook = self.rcvbook or {}
		table.insert(self.rcvbook,new_page)
		self:PopReceiveConfirm()
		-- send rcvd message to the other side.
		self:SendCommMessage("WHISPER",sender,"RCVD")
	elseif cmd == "RCVD" then
		-- cancel not received event quietly.
		self:CancelScheduledEvent("NOTRCVD")
	elseif cmd == "ACCEPTED" then
		-- popup the page accepted box
		StaticPopup_Show("OMNIBUSPAGEACCEPTED",sender)
	elseif cmd == "DECLINED" then
		-- popup the page declined box
		StaticPopup_Show("OMNIBUSPAGEDECLINED",sender)
	end
end

function Omnibus:PopReceiveConfirm()
	self.rcvbook = self.rcvbook or {}
	if not self.showingReceiveConfirm then
		self.showingReceiveConfirm = true
		StaticPopup_Show("OMNIBUSRCVCONFIRM",self.rcvbook[1].sender)
	end
end

function Omnibus:AcceptReceivedPage()
	local new_page = self.rcvbook[1]
	self:SendCommMessage("WHISPER",new_page.sender,"ACCEPTED")
	table.remove(self.rcvbook,1)
	local index = Omnibus:NewPageInFolio(self.db.account.default_folio,new_page[1],new_page[2])
	self:SortFolio()
	self:SetPage(index)
	self:UpdateTooltip()
	if self.rcvbook[1] then
		self:ScheduleEvent(StaticPopup_Show,1,"OMNIBUSRCVCONFIRM",self.rcvbook[1].sender)
	else
		self.showingReceiveConfirm = nil
	end
end

function Omnibus:CancelReceivedPage()
	self:SendCommMessage("WHISPER",self.rcvbook[1].sender,"DECLINED")
	table.remove(self.rcvbook,1)
	if self.rcvbook[1] then
		self:ScheduleEvent(StaticPopup_Show,1,"OMNIBUSRCVCONFIRM",self.rcvbook[1].sender)
	else
		self.showingReceiveConfirm = nil
	end
end

function Omnibus:SendPage(pnum,sendto)
	local page = self.db.account.book[pnum]
	if page then
		self:SendCommMessage("WHISPER",sendto,"PAGE",page[1],page[2])
		self:ScheduleEvent("NOTRCVD",StaticPopup_Show, 3,"OMNIBUSNOSENDTORESPONSE",sendto)
	end
end

function Omnibus:SendTo(recip)
	if recip then
		self.sendto = gsub(recip,"%s","")
		self.sendpage = self.db.account.current_page
		StaticPopup_Show("OMNIBUSSNDCONFIRM",self.sendto)
	else
		self:SendPage(self.sendpage,self.sendto)
		self.sendto = nil
		self.sendpage = nil
	end
end

-- toggles window on/off screen
function Omnibus:Toggle()
	if OmnibusFrame:IsVisible() then
		OmnibusFrame:Hide()
	else
		OmnibusFrame:Show()
	end
end

-- window movement
function Omnibus:StartMoving(arg1)
	if arg1=="LeftButton" and not self.db.account.Lock then
		OmnibusFrame:StartMoving()
		self.moving = true
	end
end

function Omnibus:StopMoving(arg1)
	if arg1=="LeftButton" then
		OmnibusFrame:StopMovingOrSizing()
		self.moving = nil
	end
end

--Fubar click
function Omnibus:OnClick(button)
	if button ~= "LeftButton" then return end
	self:Toggle()
end

function Omnibus:FindPageInFolioIndex(page,folio)
	if not folio then
		folio = self.db.account.default_folio
	elseif type(folio) ~= "table" then
		folio = Omnibus:FindFolio(folio)
	end
	for i,pnum in ipairs(folio) do
		if page == self.db.account.book[pnum][1] then
			return pnum
		end
	end
	return nil
end

function Omnibus:OmnibusCloseOnClick()
	--Omnibus:SavePage()
	OmnibusFrame:Hide()
end

function Omnibus:SetPage(new_page)
	if not new_page then return end
	if OmnibusFrame:IsVisible() then
		Omnibus:SavePage()
	end
	self.db.account.current_page = new_page
	self.db.account.current_folio = self.db.account.book[self.db.account.current_page].folio
	self:ShowPage()
end

-- removes the current page
function Omnibus:DeletePage(pnum)
	pnum = pnum or self.db.account.current_page
	self.db.account.autoruns[pnum] = false
	-- first, if the page we are deleting is the current page, then empty out the EditBox and the Page Title text.
	if pnum == self.db.account.current_page then
		self.db.account.current_page = nil
		OmnibusEditBox:SetText("")
		OmnibusPageTitle:SetText("")
	end
	-- now remove the page from the folio it appears in...
	local folio = self.db.account.default_folio
	if self.db.account.book[pnum].folio then folio = Omnibus:FindFolio(self.db.account.book[pnum].folio) end
	local removeat
	for i,v in ipairs(folio) do if v == pnum then removeat = i break end end
	table.remove(folio,removeat)
	Omnibus:SortFolio(folio)
	-- next, determine whether the page we are deleting is at the end of the book table.
	if pnum == #self.db.account.book then
		self.db.account.book[pnum] = nil
	else
		self.db.account.book[pnum] = "trash"
		self.db.account.garbage[#self.db.account.garbage+1] = pnum
	end
end

-- shows/updates the current page
function Omnibus:ShowPage()
	if not OmnibusFrame:IsVisible() then
		OmnibusFrame:Show()
	end
	if self.db.account.book[self.db.account.current_page] then
		Omnibus:TagShow(self.db.account.current_page)
		Omnibus:UpdatePageTitleDisplay()
		OmnibusEditBox:SetText(self:GetPageText(self.db.account.current_page))
	end
end

-- refreshes window when shown
function Omnibus:OnShow()
	self:ShowPage()
	OmnibusEditBox:SetWidth(OmnibusFrame:GetWidth()-50)
	if self.db.account.book[self.db.account.current_page] then
		Omnibus:UpdatePageTitleDisplay()
	else
		OmnibusPageTitle:SetText("")
	end
end

-- saves page when window hides
function Omnibus:OnHide()
	Omnibus:SavePage()
	OmnibusEditBox:ClearFocus()
end

function Omnibus:ShowCloseTooltip()
	GameTooltip_SetDefaultAnchor(GameTooltip,UIParent)
	GameTooltip:AddLine(L["Close Omnibus"])
	GameTooltip:AddLine(L["Close the book."],0.85,0.85,0.85,1)
	GameTooltip:Show()
end

function Omnibus:SavePage(noTagHide)
	if self.db.account.current_page and type(self.db.account.book[self.db.account.current_page]) == "table" then
		local newtext = OmnibusEditBox:GetText()
		self.db.account.book[self.db.account.current_page][2] = newtext
		self.db.account.book[self.db.account.current_page].z = nil
		if not noTagHide then Omnibus:TagHide(self.db.account.current_page) end
	end
end

-- changes border and resize grip depending on lock status
function Omnibus:UpdateLock()
	if self.db.account.Lock then
		OmnibusFrame:SetBackdropBorderColor(0,0,0,1)
		OmnibusResizeGrip:Hide()
	else
		OmnibusFrame:SetBackdropBorderColor(1,1,1,1)
		OmnibusResizeGrip:Show()
	end
	self:MakeESCable("OmnibusFrame",self.db.account.Lock)
end

-- adds frame to UISpecialFrames, removes frame if disable true
function Omnibus:MakeESCable(frame,disable)
	local idx
	for i=1,table.getn(UISpecialFrames) do
		if UISpecialFrames[i]==frame then
			idx = i
			break
		end
	end
	if idx and disable then
		table.remove(UISpecialFrames,idx)
	elseif not idx and not disable then
		table.insert(UISpecialFrames,1,frame)
	end
end
	
-- the following sorts the pages within the given folio
function Omnibus:SortFolio(folio)
	if not folio then folio = self.db.account.default_folio end
	if type(folio) ~= "table" then folio = self.db.account.folios[folio] end
	table.sort(folio,function(a,b) return self.db.account.book[a][1] < self.db.account.book[b][1] end)
end

-- The following sorts the folios themselves.
function Omnibus:SortFolios()
	table.sort(self.db.account.folios,function(a,b) return a.title < b.title end)
	-- now we update the indexes in the folios.
	for i,folio in ipairs(self.db.account.folios) do folio.index = i end
end

-- performs a search for the text in the search box
function Omnibus:DoSearch(searchstring)
	local start = self.db.account.current_page + 1
	if start > table.getn(self.db.account.book) then start = 1 end
	Omnibus:SavePage()
	local search = string.lower(searchstring)
	if string.len(search)<1 then
		return
	end
	for i = start, table.getn(self.db.account.book) do
		if string.find(string.lower(self:GetPageText(i)),search,1,1) then
			self.db.account.current_page = i
			self:ShowPage()
			return
		end
	end
	for i = 1, start - 1 do
		if string.find(string.lower(self:GetPageText(i)),search,1,1) then
			self.db.account.current_page = i
			self:ShowPage()
			return
		end
	end
end

function Omnibus:OnTextChanged(editBox)
	local scrollBar = getglobal(OmnibusEditScrollFrame:GetName().."ScrollBar")
	OmnibusEditScrollFrame:UpdateScrollChildRect()
	local min, max = scrollBar:GetMinMaxValues()
	if ( max > 0 and (editBox.max ~= max) ) then
		editBox.max = max
		scrollBar:SetMinMaxValues(min,max)
	end
end

function Omnibus:OnTooltipUpdate()
	local cat = tablet:AddCategory('columns',1)
	cat:AddLine("text",L["Toggle Omnibus Window"],'func',function() Omnibus:Toggle() end)
	cat = tablet:AddCategory('id',self.db.account.default_folio,'columns',1,'text',"Default Folio",
		'showWithoutChildren',true,'checked',true,'hasCheck',true,
		'checkIcon',self.db.account.default_hidden and "Interface\\Buttons\\UI-PlusButton-Up" or "Interface\\Buttons\\UI-MinusButton-Up",
		'func',function() self.db.account.default_hidden = not self.db.account.default_hidden end)
	if not self.db.account.default_hidden then
		for i, pnum in ipairs(self.db.account.default_folio) do
			local page = self.db.account.book[pnum]
			local _pnum = pnum
			cat:AddLine("text",page[1],"func",function()
				if IsControlKeyDown() then
					self:SetPage(_pnum)
				else
					Omnibus:TagExec(_pnum)
				end
			end)
		end
	end
	for i, folio in ipairs(self.db.account.folios) do
		cat = tablet:AddCategory('id',folio,'columns',1,'text',folio.title,
			'showWithoutChildren',true,'checked',true,'hasCheck',true,
			'checkIcon',folio.hidden and "Interface\\Buttons\\UI-PlusButton-Up" or "Interface\\Buttons\\UI-MinusButton-Up",
			'func',function() folio.hidden = not folio.hidden end)
		if not folio.hidden then
			for i, pnum in ipairs(folio) do
				local page = self.db.account.book[pnum]
				local _pnum = pnum
				cat:AddLine("text",page[1],"func",function()
					if IsControlKeyDown() then
						self:SetPage(_pnum)
					else
					Omnibus:TagExec(_pnum)
					end
				end)
			end
		end
	end
	tablet:SetHint(L["Click a page to execute it, Ctrl-Click a page to edit."])
end

function Omnibus:GetPageText(pnum)
	return self.db.account.book[pnum][2]
end

function Omnibus:SetPageText(pnum,text)
	if type(text) == "string" then
		self.db.account.book[pnum][2] = text
	end
end

function Omnibus:SetPageTag(pnum,tag)
	if self.db.account.current_page == pnum then
		Omnibus:TagHide(pnum)
		self.db.account.book[pnum].tag = tag
		Omnibus:TagShow(pnum)
	else
		self.db.account.book[pnum].tag = tag
	end
end

function Omnibus:ForEachInFolio(folio,func,ob,arg1,arg2,arg3,arg4)
	if ob then
		for pnum in Omnibus:FolioPages(folio) do
			func(ob,pnum,arg1,arg2,arg3,arg4)
		end
	else
		for pnum in Omnibus:FolioPages(folio) do
			func(pnum,arg1,arg2,arg3,arg4)
		end
	end
end

function Omnibus:FolioPages(folio)
	if not folio then
		folio = self.db.account.default_folio
	elseif type(folio) ~= "table" then
		folio = Omnibus:FindFolio(folio)
	end
	local i = 0
	return function()
		i = i + 1
		local p = folio[i]
		if p then
			return i, self.db.account.book[i][1]
		end
	end
end

function Omnibus:RunPageInFolio(page,folio)
	local page = Omnibus:FindPageInFolioIndex(page,folio)
	if page and self.db.account.book[page] and self.db.account.book[page][2] then
		Omnibus:TagExec(page)
	end
end

function Omnibus:ShowPageInFolio(page,folio)
	local page = Omnibus:FindPageInFolioIndex(page,folio)
	if page and self.db.account.book[page] and self.db.account.book[page][2] then
		Omnibus:SetPage(self.db.account.book[page][2])
	end
end

function Omnibus:NewPageInFolio(folio,title,content,tag)
	if type(folio) == "string" then folio = Omnibus:FindFolio(folio) end
	title = title or "Untitled"
	content = content or folio.template
	-- create the new page, and add it to the end of the book table.
	local newpage = {title,content,folio = folio.title,tag = tag or folio.tag}
	local newpageindex = Omnibus:GetNewPageIndex()
	self.db.account.book[newpageindex] = newpage
	folio[#folio+1] = newpageindex
	Omnibus:SortFolio(folio)
	return newpageindex
end

function Omnibus:OmnibusNextPageInFolio()
	if self.db.account.current_page then
		local pnum = self.db.account.current_page
		local newp
		local folio = Omnibus:FindFolio(self.db.account.current_folio)
		for i,p in ipairs(folio) do
			if p == pnum then
				local j = i + 1
				if j > #folio then j = #folio end
				newp = folio[j]
				break
			end
		end
		Omnibus:SetPage(newp)
	end
end

function Omnibus:OmnibusPrevPageInFolio()
	if self.db.account.current_page then
		local pnum = self.db.account.current_page
		local newp
		local folio = Omnibus:FindFolio(self.db.account.current_folio)
		for i,p in ipairs(folio) do
			if p == pnum then
				local j = i - 1
				if j < 1 then j = 1 end
				newp = folio[j]
				break
			end
		end
		Omnibus:SetPage(newp)
	end
end

function Omnibus:RenamePage(pnum,newname)
	local page = self.db.account.book[pnum]
	page[1] = newname
	Omnibus:SortFolio(Omnibus:FindFolioIndex(page.folio))
	if self.db.account.current_page == pnum then
		Omnibus:UpdatePageTitleDisplay()
	end
end

function Omnibus:RenameFolio(folio,newname)
	if type(folio) ~= "table" then
		folio = self.db.account.folios[folio]
	end
	folio.title = newname
	Omnibus:SortFolios()
	self:UpdateTooltip()
end

function Omnibus:UpdatePageTitleDisplay()
	local pagetitle = self.db.account.book[self.db.account.current_page][1]
	local foliotitle = self.db.account.book[self.db.account.current_page].folio or "Default Folio"
	OmnibusPageTitle:SetText(pagetitle..L["|cff7f7fff in|r "]..foliotitle)
end

function Omnibus:NewFolio(title)
	if Omnibus:FindFolioIndex(title) then
		return
	end
	title = title or "New Folio"
	local newfolio = {title = title,template = self.db.account.default_folio.template}
	self.db.account.folios[#self.db.account.folios+1] = newfolio
	newfolio.index = #self.db.account.folios
	Omnibus:SortFolios()
	self:UpdateTooltip()
end

function Omnibus:SetDefaultTag(folio,tagID)
	folio.tag = tagID
end

function Omnibus:DeleteFolio(folio,confirmed)
	if not confirmed then
		StaticPopup_Show("OMNIBUSFOLIODELETECONFIRM").data = folio
	else
		for i,pnum in ipairs(folio) do Omnibus:DeletePage(pnum) end
		table.remove(self.db.account.folios,folio.index)
		Omnibus:SortFolios()
		self:UpdateTooltip()
	end
end

function Omnibus:FindFolioIndex(fname)
	for i,folio in ipairs(self.db.account.folios) do if folio.title == fname then return i end end
end

function Omnibus:FindFolio(fname)
	for i,folio in ipairs(self.db.account.folios) do if folio.title == fname then return folio end end
	return self.db.account.default_folio
end

function Omnibus:MovePageToFolio(page,pnum,newfolio)
	local oldfolio = Omnibus:FindFolio(page.folio)
	page.folio = newfolio.title
	local oldremove
	for i,p in ipairs(oldfolio) do if p == pnum then oldremove = i break end end
	table.remove(oldfolio,oldremove)
	Omnibus:SortFolio(oldfolio)
	newfolio[#newfolio+1] = pnum
	Omnibus:SortFolio(newfolio)
	self:UpdateTooltip()
	if pnum == self.db.account.current_page then
		self.db.account.current_folio = page.folio
		Omnibus:UpdatePageTitleDisplay()
	end
end

function Omnibus:SetPageTemplate(folio)
	folio.template = OmnibusEditBox:GetText()
end

local function getdeep(str)
	if not str then return nil end
	local g = getfenv()
	for id in string.gmatch(str,"[^%.]+") do
		g = g[id]
		if not g then return nil end
	end
	return g
end

function Omnibus:AddTag(tagID,font,fsize,showFunc,hideFunc,execFunc)
	self.tags = self.tags or {}
	self.tags[tagID] = {font=font,fsize=fsize,show=getdeep(showFunc),hide=getdeep(hideFunc),exec=getdeep(execFunc)}
end

function Omnibus:NewTag(tagID)
	if not self.tags[tagID] then
		local ctag = {tagID = tagID,font = "Fonts\\MORPHEUS.ttf",fsize = 16,execFunc = "Omnibus.RunTextTag"}
		self.db.account.customtags[#self.db.account.customtags+1] = ctag
		self:AddTag(ctag.tagID,ctag.font,ctag.fsize,ctag.showFunc,ctag.hideFunc,ctag.execFunc)
	end
end

function Omnibus:TagShow(pnum)
	if not pnum then pnum = self.db.account.current_page end
	if not pnum then return end
	OmnibusEditBox:SetFont(self.tags[self.db.account.book[pnum].tag or "Text"].font,self.tags[self.db.account.book[pnum].tag or "Text"].fsize)
	if type(self.tags[self.db.account.book[pnum].tag or "Text"].show) == "function" then
		self.tags[self.db.account.book[pnum].tag or "Text"].show(pnum)
	end
end

function Omnibus:TagHide(pnum)
	if not pnum then pnum = self.db.account.current_page end
	if not pnum then return end
	if type(self.tags[self.db.account.book[pnum].tag or "Text"].hide) == "function" then
		self.tags[self.db.account.book[pnum].tag or "Text"].hide(pnum)
	end
end

function Omnibus:TagExec(pnum)
	if not pnum then pnum = self.db.account.current_page end
	if not pnum then return end
	if type(self.tags[self.db.account.book[pnum].tag or "Text"].exec) == "function" then
		self.tags[self.db.account.book[pnum].tag or "Text"].exec(pnum)
	end
end

function Omnibus.RunTextTag(pnum) Omnibus:SetPage(pnum) end

local FAIAPEnabled
function Omnibus.EnableFAIAP(pnum)
	-- enable faiap syntax higlighter.
	if not FAIAPEnabled then
		OBIndentationLib.enable(OmnibusEditBox,nil,2)
	end
	FAIAPEnabled = true
end
function Omnibus.DisableFAIAP(pnum)
	-- disable faiap syntax higlighter.
	if FAIAPEnabled then
		OBIndentationLib.disable(OmnibusEditBox)
	end
	FAIAPEnabled = false
end
function Omnibus.RunLuaTag(pnum)
	RunScript(Omnibus:GetPageText(pnum))
end
