local _, class = UnitClass("player")
if class ~= "SHAMAN" then return DisableAddOn("Numen") end

local _G = getfenv(0)
local L = AceLibrary("AceLocale-2.2"):new("Numen")
local Tablet = AceLibrary("Tablet-2.0")

Numen = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceDB-2.0", "AceEvent-2.0", "FuBarPlugin-2.0")
local mod = Numen

local function SpellName(spellID)
	local name = GetSpellInfo(spellID)
	return name
end

local defaults = {
	posX = nil,
	posY = nil,
	padding = 1,
	scale = 1,
	alpha = 1,
	vertical = true,
	growup = true,
	timer = true,
	pulse = true,
	hideTooltips = {
		["always"]=false,
		["combat"]=true,
	},
	tidemessage = "Tide",
	affinityOrder = {
		"Call",
		"Earth",
		"Fire",
		"Water",
		"Air",
		"Weapon",
		"Shield",
	},
	spellOrder = {
		Call = {
			SpellName(36936), -- Totemic Call
		},
		Earth = {
			SpellName(8143), -- Tremor Totem
			SpellName(5730), -- Stoneclaw Totem
			SpellName(2484), -- Earthbind Totem
			SpellName(8075), -- Strength of Earth Totem
			SpellName(8071), -- Stoneskin Totem
			SpellName(2062), -- Earth Elemental Totem
		},
		Fire = {
			SpellName(1535), -- Fire Nova Totem
			SpellName(30706), -- Totem of Wrath
			SpellName(3599), -- Searing Totem
			SpellName(8190), -- Magma Totem
			SpellName(8227), -- Flametongue Totem
			SpellName(8181), -- Frost Resistance Totem
			SpellName(2894), -- Fire Elemental Totem
		},
		Water = {
			SpellName(5675), -- Mana Spring Totem
			SpellName(5394), -- Healing Stream Totem
			SpellName(8166), -- Poison Cleansing Totem
			SpellName(8170), -- Disease Cleansing Totem
			SpellName(16190), -- Mana Tide Totem
			SpellName(8184), -- Fire Resistance Totem
		},
		Air = {
			SpellName(3738), -- Wrath of Air Totem
			SpellName(8177), -- Grounding Totem
			SpellName(8835), -- Grace of Air Totem
			SpellName(8512), -- Windfury Totem
			SpellName(15107), -- Windwall Totem
			SpellName(10595), -- Nature Resistance Totem
			SpellName(25908), -- Tranquil Air Totem
			SpellName(6495), -- Sentry Totem
		},
		Weapon = {
			SpellName(8017), -- Rockbiter Weapon
			SpellName(8024), -- Flametongue Weapon
			SpellName(8033), -- Frostbrand Weapon
			SpellName(8232), -- Windfury Weapon
		},
		Shield = {
			SpellName(974), -- Earth Shield
			SpellName(24398), -- Water Shield
			SpellName(324), -- Lightning Shield
		}
	},
	selectedSpell = {
		Call = nil,
		Earth = nil,
		Fire = nil,
		Water = nil,
		Air = nil,
		Weapon = nil,
		Shield = nil,
	},
	hideAffinity = {
		["*"] = false,
		["Call"] = true,
		["Shield"] = true,
	},
	hideSpell = {
		["*"] = false,
		[SpellName(6495)] = true, -- Sentry Totem
		[SpellName(15107)] = true, -- Windwall Totem
	},
	unsetableSpell = {
		["*"] = false,
		[SpellName(16190)] = true, -- Mana Tide Totem
		[SpellName(2062)] = true, -- Earth Elemental Totem
		[SpellName(2894)] = true, -- Fire Elemental Totem
	},
}

local options = {
	type = "group",
	args = {
		bar = {
			type = "group",
			name = L["Bar Configuration"],
			desc = L["Configuration for the Numen Bar."],
			order = 1,
			args = {
				vertical = {
					type = "toggle",
					name = L["Orientation"],
					desc = L["Set the orientation of the Numen Bar."],
					order = 1,
					get = function() return mod.db.profile.vertical end,
					set = function(v)
						mod.db.profile.vertical = v
						mod:UpdateLayout()
					end,
				},
				growup = {
					type = "toggle",
					name = L["Growup"],
					desc = L["Set the orientation of the Numen Bar."],
					order = 1,
					get = function() return mod.db.profile.growup end,
					set = function(v)
						mod.db.profile.growup = v
						mod:UpdateLayout()
					end,
				},
				scale = {
					type = "range",
					name = L["Main Scale"],
					desc = L["Set the scale of the main Numen buttons."],
					order = 4,
					step = 0.05,
					min = 0.25,
					max = 5.0,
					isPercent = false,
					get = function() return mod.db.profile.scale end,
					set = function(v)
						mod.db.profile.scale = v
						mod.root:SetScale(v)
					end
				},
				padding = {
					type = "range",
					name = L["Padding"],
					desc = L["Set the padding of the main Numen buttons."],
					order = 2,
					step = 1.0,
					min = -25.0,
					max = 25.0,
					isPercent = false,
					get = function() return mod.db.profile.padding end,
					set = function(v)
						mod.db.profile.padding = v
						mod:UpdateLayout()
					end
				},
				alpha = {
					type = "range",
					name = L["Main Alpha"],
					desc = L["Set the alpha of the main Numen buttons."],
					order = 3,
					step = 0.05,
					min = 0.05,
					max = 1.0,
					isPercent = false,
					get = function() return mod.db.profile.alpha end,
					set = function(v)
						mod.db.profile.alpha = v
						mod.root:SetAlpha(v)
					end
				},
				timer = {
					type = "toggle",
					name = L["Toggle Duration Timers"],
					desc = L["Toggles the duration timers."],
					order = 5,
					get = function() return mod.db.profile.timer end,
					set = function(v)
						mod.db.profile.timer = v
					end
				},
				pulse = {
					type = "toggle",
					name = L["Toggle Pulse Indicators"],
					desc = L["Toggles the pulses for totem ticks."],
					order = 5,
					get = function() return mod.db.profile.pulse end,
					set = function(v)
						mod.db.profile.pulse = v
					end
				},
			},
		},
		affinityOrder = {
			type = "group",
			name = L["Affinity Order"],
			desc = L["Configuration for the Numen Bar."],
			order = 2,
			args = {}
		},
		spellOrder = {
			type = "group",
			name = L["Spell Order"],
			desc = L["Configuration for the Numen Bar."],
			order = 3,
			args = {}
		},
		tideMessage = {
			type = "text",
			name = L["Manatide Message"],
			desc = L["Message that will be postet to party chat on manatide totem."],
			usage = "",
			order = 4,
			get = function() return mod.db.profile.tidemessage end,
			set = function(v)
				mod.db.profile.tidemessage = v
			end
		},
		hideAffinity = {
			type = "group",
			name = L["Hide special buttons"],
			desc = L["Hide the Totemic Call or Shield buttons"],
			order = 5,
			args = {
				toggleCall= {
					type = "toggle",
					name = L["Hide/Show Totemic Call"],
					desc = L["Show or hide Totemic Call in the Numen Bar."],
					order = 1,
					get = function() return mod.db.profile.hideAffinity.Call end,
					set = function(v)
						mod:ToggleCall(v)
					end,
				},
				toggleShield= {
					type = "toggle",
					name = L["Hide/Show Shields"],
					desc = L["Show or hide Shields in the Numen Bar."],
					order = 2,
					get = function() return mod.db.profile.hideAffinity.Shield end,
					set = function(v)
						mod:ToggleShield(v)
					end,
				},
				toggleWeapon= {
					type = "toggle",
					name = L["Hide/Show Weapon Buffs"],
					desc = L["Show or hide Weapon Buffs in the Numen Bar."],
					order = 3,
					get = function() return mod.db.profile.hideAffinity.Weapon end,
					set = function(v)
						mod:ToggleWeapon(v)
					end,
				},
			}
		},
		hideTooltips = {
			order = 6,
			name = L["Hide Tooltips"],
			desc = L["Configure the Button Tooltip."],
			type = "group",
			args = {
				always= {
					name = L["Always"],
					desc = L["Configure the Button Tooltip."],
					type = "toggle",
					map = { [false] = L["Disabled"], [true] = L["Enabled"] },
					get = function() return mod.db.profile.hideTooltips.always end,
					set = function(value) mod.db.profile.hideTooltips.always = value end,
				},
				combat= {
					name = L["In Combat"],
					desc = L["Configure the Button Tooltip."],
					type = "toggle",
					map = { [false] = L["Disabled"], [true] = L["Enabled"] },
					get = function() return mod.db.profile.hideTooltips.combat end,
					set = function(value) mod.db.profile.hideTooltips.combat = value end,
				},
			},
		},
		lock = {
			type = "toggle",
			name = L["Lock Bar"],
			desc = L["Lock the Numen Bar."],
			order = 7,
			get = function() return mod.isLocked end,
			set = function(v) mod:ToggleLock() end,
		},
	},
}

mod.name                   = "Numen"
mod.hasIcon                = "Interface\\Icons\\Spell_Fire_TotemOfWrath"
mod.defaultMinimapPosition = 0
mod.cannotDetachTooltip    = true
mod.independentProfile     = true
mod.defaultPosition        = "LEFT"
mod.hideWithoutStandby     = true

local db
function mod:OnInitialize()
	self:RegisterChatCommand("/numen", options)
	self.OnMenuRequest = options

	self:RegisterDB("NumenDB", "NumenDBPC")
	self:RegisterDefaults("profile", defaults)
	db = self.db.profile
end

function mod:OnEnable()
	BINDING_HEADER_Numen = "Numen"
	self.keyframe = CreateFrame("Frame", nil, UIParent)
	self:RegisterEvent("UPDATE_BINDINGS")
	self:RegisterEvent("LEARNED_SPELL_IN_TAB", "UpdateSpellList")
	self:RegisterEvent("PLAYER_REGEN_DISABLED", "EnterCombat")
	self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
	self:RegisterEvent("PLAYER_TOTEM_UPDATE");

	self:CreateFrame()
	self:UpdateAffinityOrder()
	self:UpdateSpellOrder()
	self:UpdateSpellList()
	self:RegisterOverrideBindings()
	self:ToggleLock()
end

function mod:OnDisable()
	self.root:Hide()
end

function mod:OnProfileEnable()
	db = self.db.profile
	self:UpdateAffinityOrder()
	self:UpdateSpellOrder()
	self:UpdateSpellList()
end

function mod:UpdateAffinityOrder()
	local option = options.args.affinityOrder
	for i, affinity in pairs(db.affinityOrder) do
		if not option.args[i] then option.args[i] = {} end
		local option = option.args[i]
		option.type = 'text'
		option.name = tostring(i)
		option.desc = tostring(i)
		option.get = function()
			return affinity
		end
		option.set = function(value)
			for order,oldValue in ipairs(db.affinityOrder) do
				if value == oldValue then
					table.remove(db.affinityOrder, order)
					table.insert(db.affinityOrder, i, value)
					self:UpdateLayout()
					self:UpdateAffinityOrder()
					break
				end
			end
		end
		option.validate = db.affinityOrder
	end
end

function mod:UpdateSpellOrder()
	local option = options.args.spellOrder
	for i, affinity in pairs(db.affinityOrder) do
		if not option.args[affinity] then option.args[affinity] = {} end
		local option = option.args[affinity]
		option.type = 'group'
		option.name = affinity
		option.desc = affinity
		option.order = i
		option.args = {}
		for j, spellName in pairs(db.spellOrder[affinity]) do
			if not option.args[j] then option.args[j] = {} end
			local option = option.args[j]
			option.type = 'text'
			option.name = tostring(j)
			option.desc = tostring(j)
			option.get = function()
				return spellName
			end
			option.set = function(value)
				for order,oldValue in ipairs(db.spellOrder[affinity]) do
					if value == oldValue then
						table.remove(db.spellOrder[affinity], order)
						table.insert(db.spellOrder[affinity], j, value)
						self:UpdateLayout()
						self:UpdateSpellOrder()
						break
					end
				end
			end
			option.validate = db.spellOrder[affinity]
		end
	end
end

function mod:UPDATE_BINDINGS()
	if not InCombatLockdown() then
		self:RegisterOverrideBindings()
	end
end

function mod:RegisterOverrideBindings()
	ClearOverrideBindings(self.keyframe)
	for affinity, button in pairs(self.buttons) do
		local name = button.button:GetName()
		local key1, key2 = GetBindingKey(name)
		if key1 then
			SetOverrideBindingClick(self.keyframe, false, key1, name)
		end
		if key2 then
			SetOverrideBindingClick(self.keyframe, false, key2, name)
		end
	end
end

function mod:UpdateSpellList()
	self.spellList = self.spellList or {}
	for spellName in pairs(self.spellList) do
		self.spellList[spellName] = nil
	end
	for i=1,GetNumSpellTabs() do
		local _, _, spellOffset, numSpells = GetSpellTabInfo(i)
		for j=spellOffset+1, spellOffset+numSpells do
			local spellName = GetSpellName(j, "spell")
			if self.totemSpells[spellName] then
				self.spellList[spellName] = j
			end
		end
	end
	self:UpdateLayout()
end

function mod:CreateFrame()
	self.root = CreateFrame("Frame", "NumenRoot", UIParent, "SecureStateHeaderTemplate")
	self.root:RegisterForDrag("LeftButton")
	self.root:EnableMouse(false)
	self.root:SetMovable(true)
	self.root.texture = self.root:CreateTexture()
	self.root.texture:SetTexture(0, 0, 0.5, 0)
	self.root.texture:SetAllPoints(self.root)
	self.root:SetFrameStrata("TOOLTIP")
	self:LoadPosition()

	self.buttons = {}
	self.popbuttons = {}
end

function mod:UpdateLayout()
	if InCombatLockdown() then
		return self:ScheduleLeaveCombatAction("UpdateLayout")
	end
	local i = 0
	for _, affinity in ipairs(db.affinityOrder) do
		local b = self.buttons[affinity]
		if not db.hideAffinity[affinity] then
			i = i + 1
			if not b then
				b = self.Button:new(affinity)
			end
			local j = 0
			local selectedSpell = db.selectedSpell[affinity]
			local setstate
			for i in ipairs(b.popbuttons) do
				b.popbuttons[i] = nil
			end
			for _, spellName in ipairs(db.spellOrder[affinity]) do
				local p = self.popbuttons[spellName]
				if self.spellList[spellName] and not db.hideSpell[affinity] then
					j = j + 1
					if not p then
						p = self.PopButton:new(spellName)
						self.popbuttons[spellName] = p
					end
					tinsert(b.popbuttons, j, p)
					p:Update(b, i, j)
					if (spellName == selectedSpell) then
						setstate = j
					end
				elseif p then
					p:Disable()
				end
			end
			if (j > 0) then
				b:Update(i, j, setstate)
			else
				i = i - 1
				b:Disable()
			end
		elseif b then
			b:Disable()
		end
	end
	local x, y = 36, 36
	local pos = ((36 + db.padding) * i) - db.padding
	if db.vertical then
		y = pos
	else
		x = pos
	end
	self.root:SetWidth(x)
	self.root:SetHeight(y)
	self.root:SetScale(db.scale)
	self.root:SetAlpha(db.alpha)
end

function mod:ShowTooltip(button)
	if mod.db.profile.hideTooltips.always == true then return end
	if mod.db.profile.hideTooltips.combat == true and InCombatLockdown() then return end
	local spellName = button.class.spellName
	if not spellName then return end
	local spellId = self.spellList[spellName]
	if (spellId) then
		GameTooltip:SetOwner(button)
		GameTooltip:SetSpell(spellId, BOOKTYPE_SPELL)
	end
end

function mod:HideTooltip()
	GameTooltip:Hide()
end

local function StartDrag()
	mod.root:StartMoving()
end

local function StopDrag()
	mod.root:StopMovingOrSizing()
	mod:SavePosition()
end

function mod:ToggleCall(v)
	mod.db.profile.hideAffinity["Call"] = v
	mod:UpdateAffinityOrder()
	mod:UpdateSpellOrder()
	mod:UpdateLayout()
end

function mod:ToggleShield(v)
	mod.db.profile.hideAffinity["Shield"] = v
	mod:UpdateAffinityOrder()
	mod:UpdateSpellOrder()
	mod:UpdateLayout()
end

function mod:ToggleWeapon(v)
	mod.db.profile.hideAffinity["Weapon"] = v
	mod:UpdateAffinityOrder()
	mod:UpdateSpellOrder()
	mod:UpdateLayout()
end

function mod:ToggleLock()
	if InCombatLockdown() then return end
	if self.isLocked then
		self.root:EnableMouse(true)
		self.root:SetScript("OnDragStart", StartDrag)
		self.root:SetScript("OnDragStop", StopDrag)
		self.root.texture:SetTexture(0, 0, 0.5, 0.5)
		self.root:SetFrameLevel(4)
		self.isLocked = false

		self.root:SetAttribute("state", -1)
	else
		self.root:EnableMouse(false)
		self.root:SetScript("OnDragStart", nil)
		self.root:SetScript("OnDragStop", nil)
		self.root.texture:SetTexture(0, 0, 0.5, 0)
		self.isLocked = true
		self.root:SetAttribute("state", 0)
	end
end

function mod:LoadPosition()
	local f = self.root
	local x, y, s = db.posX, db.posY, f:GetEffectiveScale()
	if x and y then
		x, y = x/s, y/s
	end
	f:ClearAllPoints()
	f:SetPoint(x and "TOPLEFT" or "CENTER", UIParent, y and "BOTTOMLEFT" or "CENTER", x or 0, y or 0)
end

function mod:SavePosition()
	local f = self.root
	local s = f:GetEffectiveScale()
	s = s / db.scale;
	local x, y = f:GetLeft(), f:GetTop()
	x, y = x * s, y  * s
	db.posX = x
	db.posY = y
end

function mod:OnTooltipUpdate()
	Tablet:SetHint(L["|cffeda55fLeft Click|r to lock Numen bar.\n|cffeda55fRight-Click|r to open config."])
end

function mod:OnClick(button)
	self:ToggleLock()
end

function mod:EnterCombat()
	if not self.isLocked then
		self:ToggleLock()
	end
end

mod.spellTimers = {}
mod.affinityDurations = {}
function mod:UNIT_SPELLCAST_SUCCEEDED(unit, spellName, spellRank)
		if (unit ~= "player") then return end
		local affinity = self.totemSpells[spellName]
		if not affinity then return end
		local slot = self:GetSlotFromAffinity(affinity)
		if not slot then return end
		local _, _, startTime, duration = GetTotemInfo(slot)
--[[		
		local duration = self.spellDuration[spellName]
		if type(duration) == "table" then
			duration = duration[spellRank]
		end
		if (unit ~= "player") or not duration then return end
]]--
		self.spellTimers[affinity] = startTime
		self.affinityDurations[affinity] = duration
		local pulse = self.spellPulse[spellName]
		self:ScheduleRepeatingEvent(affinity.."-Pulse", self.ShowPulse, pulse or duration, self, spellName)
end

function mod:PLAYER_TOTEM_UPDATE(slot)
	local _, name, _, duration = GetTotemInfo(slot)
	local affinity = self:GetAffinityFromSlot(slot)
	if (duration == 0) then
		self.spellTimers[affinity] = 0
	else
		local message = self.db.profile.tidemessage
		local tide = SpellName(16190) -- Mana Tide Totem
		if name == tide then
			if GetNumPartyMembers() > 0 then
				SendChatMessage(message, "PARTY")
				return
			end
		end
	end
end

function mod:ShowPulse(spellName)
	local affinity = self.totemSpells[spellName]
	local duration = self.affinityDurations[affinity] 
	local startTime = self.spellTimers[affinity]
	if not time or ((startTime + duration) <= GetTime()) then
		self.spellTimers[affinity] = nil
		self:CancelScheduledEvent(affinity.."-Pulse")
		return
	end
	if (db.pulse) then
		local v = self.buttons[affinity]
		if v.button:IsVisible() then
			v.button.pulse:SetCooldown(0, 0)
		end
	end
end

function mod:GetAffinityFromSlot(slot)
	if slot == 1 then
		return "Fire"
	elseif slot == 2 then
		return "Earth"
	elseif slot == 3 then
		return "Water"
	elseif slot == 4 then
		return "Air"
	end
end

function mod:GetSlotFromAffinity(affinity)
	if affinity == "Fire" then
		return 1
	elseif affinity == "Earth" then
		return 2
	elseif affinity == "Water" then
		return 3
	elseif affinity == "Air" then
		return 4
	end
end
