local dewdrop = AceLibrary("Dewdrop-2.0")
local tablet = AceLibrary("Tablet-2.0")
local L = AceLibrary("AceLocale-2.2"):new("MrPlow")

local dustBag = {}
local count = 0
local theWorks = {}
local eventCalled = nil
local delayCalled = nil

local theWorksBank = nil
local bankAvailable = nil
local plowingBank = nil
local swapping = nil
local itemCount = 0

BINDING_HEADER_MRPLOW = "Mr Plow"

--<< ================================================= >>--
-- Section II: AddOn Information.                        --
--<< ================================================= >>--
local sortCategories = {"Consumable.Potion", "Consumable.Water", "Consumable.Weapon Buff"}

local itemCategories = {
	ARMOR = 1,
	WEAPON = 2,
	GEM = 3,
	QUEST = 4,
	KEY = 5,
	RECIPE = 6,
	REAGENT = 7,
	TRADEGOODS = 8,
	CONSUMABLE = 9,
	CONTAINER = 10,
	QUIVER = 11,
	MISCELLANEOUS = 12,
	PROJECTILE = 13,
}

local armWepRank = {
	INVTYPE_AMMO = 0,
	INVTYPE_HEAD = 1,
	INVTYPE_NECK = 2,
	INVTYPE_SHOULDER = 3,
	INVTYPE_BODY = 4,
	INVTYPE_CHEST = 5,
	INVTYPE_ROBE = 5,
	INVTYPE_WAIST = 6,
	INVTYPE_LEGS = 7,
	INVTYPE_FEET = 8,
	INVTYPE_WRIST = 9,
	INVTYPE_HAND = 10,
	INVTYPE_FINGER = 11,
	INVTYPE_TRINKET = 12,
	INVTYPE_CLOAK = 13,
	INVTYPE_WEAPON = 14,
	INVTYPE_SHIELD = 15,
	INVTYPE_2HWEAPON = 16,
	INVTYPE_WEAPONMAINHAND = 18,
	INVTYPE_WEAPONOFFHAND = 19,
	INVTYPE_HOLDABLE = 20,
	INVTYPE_RANGED = 21,
	INVTYPE_THROWN = 22,
	INVTYPE_RANGEDRIGHT = 23,
	INVTYPE_RELIC = 24,
	INVTYPE_TABARD = 25,
}

-- Trade Good sorting
local ingredientRanking = {
	["Tradeskill.Mat.ByType.Bar"] = 1,
	["Tradeskill.Mat.ByType.Bolt"] = 2,
	["Tradeskill.Mat.ByType.Cloth"] = 3,
	["Tradeskill.Mat.ByType.Crystal"] = 4,
	["Tradeskill.Mat.ByType.Dust"] = 5,
	["Tradeskill.Mat.ByType.Dye"] = 6,
	["Tradeskill.Mat.ByType.Elemental"] = 7,
	["Tradeskill.Mat.ByType.Essence"] = 8,
	["Tradeskill.Mat.ByType.Flux"] = 9,
	["Tradeskill.Mat.ByType.Gem"] = 10,
	["Tradeskill.Mat.ByType.Grinding Stone"] = 11,
	["Tradeskill.Mat.ByType.Herb"] = 12,
	["Tradeskill.Mat.ByType.Hide"] = 13,
	["Tradeskill.Mat.ByType.Leather"] = 14,
	["Tradeskill.Mat.ByType.Mote"] = 15,
	["Tradeskill.Mat.ByType.Oil"] = 16,
	["Tradeskill.Mat.ByType.Ore"] = 17,
	["Tradeskill.Mat.ByType.Part"] = 18,
	["Tradeskill.Mat.ByType.Pearl"] = 19,
	["Tradeskill.Mat.ByType.Poison"] = 20,
	["Tradeskill.Mat.ByType.Powder"] = 21,
	["Tradeskill.Mat.ByType.Primal"] = 22,
	["Tradeskill.Mat.ByType.Rod"] = 23,
	["Tradeskill.Mat.ByType.Salt"] = 24,
	["Tradeskill.Mat.ByType.Scale"] = 25,
	["Tradeskill.Mat.ByType.Shard"] = 26,
	["Tradeskill.Mat.ByType.Spice"] = 27,
	["Tradeskill.Mat.ByType.Spider Silk"] = 28,
	["Tradeskill.Mat.ByType.Stone"] = 29,
	["Tradeskill.Mat.ByType.Thread"] = 30,
	["Tradeskill.Mat.ByType.Vial"] = 31,
}

local itemRanking = {
	[L["Armor"]] = itemCategories.ARMOR,
	[L["Weapon"]] = itemCategories.WEAPON,
	[L["Quest"]] = itemCategories.QUEST,
	[L["Key"]] = itemCategories.KEY,
	[L["Recipe"]] = itemCategories.RECIPE,
	[L["Reagent"]] = itemCategories.REAGENT,
	[L["Consumable"]] = itemCategories.CONSUMABLE,
	[L["Container"]] = itemCategories.CONTAINER,
	[L["Quiver"]] = itemCategories.QUIVER,
	[L["Miscellaneous"]] = itemCategories.MISCELLANEOUS,
	[L["Projectile"]] = itemCategories.PROJECTILE,
	[L["Trade Goods"]] = itemCategories.TRADEGOODS,
	[L["Gem"]] = itemCategories.GEM
}

local during = {
	L["There is a tinkle of broken glass and a soft \"D'oh!\""],
	L["It looks like two wet cats in a bag..."],
	L["You hear the theme music being hummed out of key."],
	L["The engine cuts off for a bit, followed by the sounds of an asthmatic cow and curses before spluttering on..."],
	L["You wince as something reverses into your lower back."],
	L["You hear a loud CRASH, and the ever present revolving noise of a single plate spinning on the ground."],
	L["The discordant noise of what sounds like a piano dropping onto a hessian sack full of lasagne fills your ears."],
}

local optionTable = {
	type = "group",
	args = {
		stack = {
			type = "execute",
			name = L["Stack"],
			desc = L["Compresses your inventory."],
			func = "ParseInventory",
			order = 100,
		},
		plow = {
			type = "execute",
			name = L["Plow"],
			desc = L["Defragments your inventory."],
			func = "Defrag",
			order = 101,
		},
		sort = {
			type = "execute",
			name = L["Sort"],
			desc = L["Sorts your inventory."],
			func = "SortAll",
			order = 102,
		},
		theworks = {
			type = "execute",
			name = L["The Works"],
			desc = L["Stacks, plows and sorts. All in one."],
			func = "Works",
			order = 103,
		},
		bank = {
			type = "group",
			name = L["Bank"],
			desc = L["Bank commands"],
			order = 104,
			args = {
				stack = {
					type = "execute",
					name = L["Stack"],
					desc = L["Compresses your bank."],
					func = function() MrPlow:ParseInventory(L["Bank"]) end,
					order = 100,
				},
				plow = {
					type = "execute",
					name = L["Plow"],
					desc = L["Defragments your bank."],
					func = function() MrPlow:Defrag(L["Bank"]) end,
					order = 101,
				},
				sort = {
					type = "execute",
					name = L["Sort"],
					desc = L["Sorts your bank."],
					func = function() MrPlow:SortAll(L["Bank"]) end,
					order = 102,
				},
				theworks = {
					type = "execute",
					name = L["The Works"],
					desc = L["Stacks, plows and sorts. All in one."],
					func = function() MrPlow:Works(L["Bank"]) end,
					order = 103,
				},
				bankstack = {
					type = "execute",
					name = L["Bank Stack"],
					desc = L["Moves stuff from your bags to your bank."],
					func = "BankStack",
					order = 104,
				},
			},
		},
		info = {
			type = "execute",
			name = L["Status"],
			desc = L["Shows the current options set for Mr Plow."],
			func = "Report",
			order = 105,
		},
		properties = {
			type = "group",
			name = L["Properties"],
			desc = L["Mr Plow Properties"],
			order = 106,
			args = {
				direction = {
					type     = "text",
					name     = L["Direction"],
					usage    = "<"..L["Direction"]..">",
					desc     = L["Change the plow direction."],
					get      = function() return MrPlow.db.profile.Direction end,
					set      = function(v)
									MrPlow.db.profile.Direction = v
									if v == "Forwards" then
										MrPlow:Print(L["Mr Plow does a three point turn and faces forward..."])
									else
										MrPlow:Print(L["Mr Plow changes gear into reverse..."])
									end
								end,
					validate = {["Forwards"] = L["Forwards"], ["Backwards"] = L["Backwards"]},
					order = 100,
				},
				gag = {
					type = "toggle",
					name = L["Gag"],
					desc = L["Silence output"],
					get  = function() return MrPlow.db.profile.Gag end,
					set  = function(v)
								if v then
									MrPlow:Print(L["You feel a great disturbance in the Force, as if millions of voices suddenly cried out in terror and were suddenly silenced..."])
									MrPlow:Print(L["Oh wait. No. Just one."])
								else
									MrPlow:Print(L["There is a grumbling as the gag is removed."])
								end
								MrPlow.db.profile.Gag = v
							end,
					order = 102,
				},
				bankstackstyle = {
					type    = "execute",
					name = L["Bank Stack Style"],
					desc    = L["Toggles the bank stack style."],
					func    = "BankStackStyle",
					order	= 103,
				},
				junkfilter = {
					type	= "toggle",
					name	= L["Filter Junk"],
					desc	= L["Moves all greys to the end."],
					get		= function() return MrPlow.db.profile.Junk end,
					set		= function() MrPlow.db.profile.Junk = not MrPlow.db.profile.Junk end,
					order	= 104,
				},
				clicktorun = {
					type	= "toggle",
					name = L["Click To Run"],
					desc	= L["Disables the click to run functionality."],
					get		= function() return MrPlow.db.profile.ClickToRun end,
					set		= function() MrPlow.db.profile.ClickToRun = not MrPlow.db.profile.ClickToRun end,
					order	= 105,
				},
			},
		},
		ignore = {
			type      = "group",
			name      = L["ignore"],
			desc      = L["Modifies your ignore list."],
			guiHidden = true,
			args      = {
				add = {
					 type = "text",
					 get  = false,
					 usage = "<[item]>",
					 name = L["add"],
					 desc = L["add item(s) to the ignore list, just shiftclick."],
					 set = "IgnoreAdd",
				},
				del = {
					 type = "text",
					 get  = false,
					 usage = "<[item]>",
					 name = L["del"],
					 desc = L["remove item(s) from the ignore list, just shiftclick."],
					 set = "IgnoreDel",
				},
				clear = {
					 type = "execute",
					 name = L["clear"],
					 desc = L["clear the ignore list completely."],
					 func = "IgnoreClear",
				},
				bag = {
					 type = "group",
					 name = L["bag"],
					 desc = L["ignore specific bags."],
					 args = { 
						add = {
							type = "text",
							get  = false,
							usage = "<0,1,2,3,4>",
							name = L["add"],
							desc = L["add a bag to the ignore list. Use '0' for the backpack, and 1-4 for the others. Use a single number or a comma separated list."],
							set = "IgnoreBagsAdd",
						},
						del = {
							type =  "text",
							get  = false,
							usage = "<0,1,2,3,4>",
							name = L["del"],
							desc = L["remove bags from the ignore list. Use '0' for the backpack, and 1-4 for the others, Use a single number or a comma separated list."],
							set = "IgnoreBagsDel",
						},
						clear = {
							type = "execute",
							name = L["clear"],
							desc = L["clear the ignore bag list."],
							func = "IgnoreBagsClear",
						}
					}
				},
				slotignore = {
					 type = "group",
					 name = L["slot"],
					 desc = L["ignore specific slots."],
					 args = { 
						add = {
							type = "text",
							get  = false,
							usage = "<[0,1,2,3,4]-[1-x]>",
							name = L["add"],
							desc = L["add slots to the ignore list. Use '0' for the backpack, and 1-4 for the others in the form bag-slot (eg 1-12). Use a single entry or a comma separated list."],
							set = "IgnoreSlotsAdd",
						},
						del = {
							type = "text",
							get  = false,
							name = L["del"],
							usage = "<[0,1,2,3,4]-[1-x]>",
							desc = L["remove slots from the ignore list. Use '0' for the backpack, and 1-4 for the others in the form bag-slot (eg 1-12), Use a single entry or a comma separated list."],
							set = "IgnoreSlotsDel",
						},
						addmouse = {
							type = "execute",
							name = L["addmouse"],
							desc = L["add slots to the ignore list. Place the cursor over the intended bagslot to ignore and run the command."],
							func = "IgnoreSlotsMouseAdd",
						},
						delmouse = {
							type = "execute",
							name = L["delmouse"],
							desc = L["remove slots from the ignore list. Place the cursor over the intended bagslot to unignore and run the command."],
							func = "IgnoreSlotsMouseDel",
						},
						clear = {
							type = "execute",
							name = L["clear"],
							desc = L["clear the ignore slot list."],
							func = "IgnoreSlotsClear",
						},
					},
				},
				ignorebank = {
					 type = "group",
					 name = L["bankbag"],
					 desc = L["ignore specific bankbags."],
					 args = {
						add = {
							type = "text",
							name = L["add"],
							get  = false,
							usage = "<[0,1,2,3,4,5,6]>",
							desc = L["add a bankbag to the ignore list. Use '0' for the main bank window, and 1-6 for the others. Use a single number or a comma separated list."],
							set = "IgnoreBankBagsAdd",
						},
						del = {
							type = "text",
							get  = false,
							name = L["del"],
							usage = "<[0,1,2,3,4,5,6]>",
							desc = L["remove bags from the ignore list. Use '0' for the main bank window, and 1-6 for the others, Use a single number or a comma separated list."],
							set = "IgnoreBankBagsDel",
						},
						clear = {
							type = "execute",
							name = L["clear"],
							desc = L["clear the ignore bankbag list."],
							func = "IgnoreBankBagsClear",
						},
					},
				},
			},
		},
	},
}

--<< ================================================= >>--
-- Section II: AddOn Information.                        --
--<< ================================================= >>--

local PT = LibStub("LibPeriodicTable-3.1")

MrPlow = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceConsole-2.0", "AceDebug-2.0", "FuBarPlugin-2.0")

MrPlow.hasIcon = true
MrPlow.hasNoColor = true
MrPlow.OnMenuRequest = optionTable
MrPlow.cannotDetachTooltip = true
MrPlow.independentProfile = true

function MrPlow:OnInitialize()
	self.version = self.version..".2.".. string.sub("$Revision: 56614 $", 12, -3)

	self:RegisterDB("MrPlowDB", "MrPlowIgnoreDB")
	self:RegisterDefaults("profile", {
		ClickToRun = true,
		Direction = "Forwards",
		BankStackStyle = 1,
	})
	self:RegisterDefaults("char", {
		ignoreList = {},
		ignoreBagList = {},
		ignoreBankBagList = {},
		ignoreSlotList = {}
	})

	self:RegisterChatCommand(L["AceConsole-commands"], optionTable, "MRPLOW")
	self:SetDebugging(false)
end

function MrPlow:OnEnable()
	self:RegisterEvent("BANKFRAME_OPENED")
	self:RegisterEvent("BANKFRAME_CLOSED")
end

function MrPlow:Report()
	if self.db.profile.Direction == "Backwards" then
		self:Print(L["Mr Plow is facing"].." ".. L["backward."])
	else
		self:Print(L["Mr Plow is facing"].." ".. L["forward."])
	end
	local tag = true
	for link,v in pairs(self.db.char.ignoreList) do
		if tag then
			self:Print(L["Mr Plow is ignoring:"])
			tag = nil
		end
		local linkstring = self:GetLink(link)
		if linkstring == "" then
			self.db.char.ignoreList[link] = nil
		else
			self:Print("     "..linkstring)
		end
	end
	for bag,slots in pairs(self.db.char.ignoreSlotList) do
		for slot in pairs(slots) do
			self:Print("     "..bag.."-"..slot)
		end
	end
	local ibag = "[ "
	for i,v in pairs(self.db.char.ignoreBagList) do
		ibag = ibag..(i-1).." "
	end
	if ibag ~= "[ " then
		self:Print(L["and bags: "]..ibag.." ]")
	end
	ibag = "[ "
	for i,v in pairs(self.db.char.ignoreBankBagList) do
		ibag = ibag..(i-1).." "
	end
	if ibag ~= "[ " then
		self:Print(L["and bank bags: "]..ibag.." ]")
	end

	self:PrintBankStyle()
end

function MrPlow:BankStackStyle()
	local style = self.db.profile.BankStackStyle or 0
	if style >= 2 then
		self.db.profile.BankStackStyle = 0
	else
		self.db.profile.BankStackStyle = style + 1
	end
	self:PrintBankStyle()
end

function MrPlow:PrintBankStyle()
	local style = self.db.profile.BankStackStyle or 0
	local message = {}
	local list = ""
	if style >= 0 then
		table.insert(message, L["filling up current stacks"])
	end
	if style >= 1 then
		table.insert(message, L["filling up the bag that contains the same item"])
	end
	if style >= 2 then
		table.insert(message, L["filling up any other free space"])
	end
	if #message > 1 then
		list = table.concat(message, ",", 1, #message - 1)..L[" and then "]..message[#message]
	else
		list = message[1]
	end
	self:Print(L["Mr Plow will stack items from your bag to your bank by "]..list)
end

function MrPlow:IgnoreAdd(list)
	self:ChangeIgnore(list, false)
end

function MrPlow:IgnoreDel(list)
	self:ChangeIgnore(list, true)
end

function MrPlow:IgnoreClear()
	self:ChangeIgnore(nil, true, true)
end

function MrPlow:IgnoreSlotsAdd(list)
	self:ChangeSlotsIgnore(list, false)
end

function MrPlow:IgnoreSlotsDel(list)
	self:ChangeSlotsIgnore(list, true)
end

function MrPlow:IgnoreSlotsMouseAdd(list)
	local bag = GetMouseFocus():GetParent():GetID()
	local slot = GetMouseFocus():GetID()
	if bag and slot then
		self:Debug("ignoring "..bag.." and "..slot)
		self:ChangeSlotsIgnore(bag.."-"..slot, false)
	end
end

function MrPlow:IgnoreSlotsMouseDel(list)
	local bag = GetMouseFocus():GetParent():GetID()
	local slot = GetMouseFocus():GetID()
	if bag and slot then
		self:Debug("ignoring "..bag.." and "..slot)
		self:ChangeSlotsIgnore(bag.."-"..slot, true)
	end
end

function MrPlow:IgnoreSlotsClear()
	self:ChangeSlotsIgnore(nil, true, true)
end

function MrPlow:IgnoreBagsAdd(list)
	self:ChangeBagIgnore(list, false)
end

function MrPlow:IgnoreBagsDel(list)
	self:ChangeBagIgnore(list, true)
end

function MrPlow:IgnoreBagsClear()
	MrPlow:ChangeBagIgnore(nil, true, true)
end


function MrPlow:IgnoreBankBagsAdd(list)
	MrPlow:ChangeBagIgnore(list, false, false, true)
end

function MrPlow:IgnoreBankBagsDel(list)
	self:ChangeBagIgnore(list, true, false, true)
end

function MrPlow:IgnoreBankBagsClear()
	MrPlow:ChangeBagIgnore(nil, true, true, true)
end


function MrPlow:Works(bank)
	if #theWorks == 0 then
		self:Debug("Running the works with ".. tostring(bank))
		theWorks = { "ParseInventory", "Defrag", "SortAll" }
		if bank == L["Bank"] then
			theWorksBank = L["Bank"]
		end
		self:SwapIt()
	end
end

function MrPlow:RegisterEvents()
	self:RegisterEvent("BAG_UPDATE", "ScanEvent")
	if bankAvailable then
		self:RegisterEvent("PLAYERBANKSLOTS_CHANGED", "ScanEvent")
	end
end

function MrPlow:UnregisterEvents()
	if self:IsEventRegistered("BAG_UPDATE") then
		self:UnregisterEvent("BAG_UPDATE")
	end
	if self:IsEventRegistered("PLAYERBANKSLOTS_CHANGED") then
		self:UnregisterEvent("PLAYERBANKSLOTS_CHANGED")
	end
end

function MrPlow:BANKFRAME_OPENED()
	bankAvailable = true
end

function MrPlow:BANKFRAME_CLOSED()
	bankAvailable = nil
	if plowingBank then
		plowingBank = nil
		self:UnregisterEvent("PLAYERBANKSLOTS_CHANGED")
		for k in pairs(dustBag) do dustBag[k] = nil end
		swapping = nil
	end
end


--[[
	This function will parse the inventory for stacks that can be restacked. They will be filled up from the back, forward, 
	so that if there is too many items to make all of the stacks full, the partial stack will remain at the front, to be 
	used by the player automatically (like ammo usage, or tradeskills)
]]--
function MrPlow:ParseInventory(bank)
	local t, mem = GetTime(), gcinfo()
	if #dustBag == 0 then
		local notFull = {}
		local dupe = {}
		local baglist

		if bank == L["Bank"] and bankAvailable then
			baglist = self:GetBagList(true)
			plowingBank = true
		elseif bank == L["Bank"] and not bankAvailable then
			self:Print("Your Bank window is not open!")
			for k in pairs(theWorks) do theWorks[k] = nil end
			theWorksBank = nil
			return
		else
			baglist = self:GetBagList()
		end
		
		for i,bag in pairs(baglist) do
			for slot=1, GetContainerNumSlots(bag) do
				if not (self.db.char.ignoreSlotList[bag] and self.db.char.ignoreSlotList[bag][slot]) and GetContainerItemLink(bag,slot) then
					local link = select(3, GetContainerItemLink(bag,slot):find("item:(%d+):%d+:%d+:%d+"))
					if link then
						if not self.db.char.ignoreList[link] then -- Skip over ignored items
							local stackSize = select(2, GetContainerItemInfo(bag,slot))
							local fullStack = select(8, GetItemInfo(link))

							-- If it exists, and is 1) in our NotFull list, and 2) isn't full itself add it to the dupe list
							if notFull[link] and stackSize < fullStack then
								if(not dupe[link]) then
									dupe[link] = {}
								end
								table.insert(dupe[link], 1, {bag, slot, stackSize})
							else -- Otherwise check if it's not full and put it in our list to fill
								if stackSize < fullStack then
									notFull[link] = {bag, slot, fullStack - stackSize, fullStack}
								end
							end
						end
					end
				end
			end
		end
		--[[
		-- Now we have two lists. The notFull has the last unfilled position, while the dupe table holds the rest 
		-- of the unfilled stacks of that object, but in forward order. So what we need to do now, is take the 
		-- first unfilled, and dump it on the last...
		--]]
		for link, data in pairs(dupe) do
			while #data > 0 do
				local dBag, dSlot, dStackSize = unpack(table.remove(data,1))
				local fillBag, fillSlot, fillDeficit, fullStack = unpack(notFull[link])
				local newTarget

				table.insert(dustBag, 1, {{dBag, dSlot}, {fillBag, fillSlot}})

				if dStackSize >= fillDeficit then -- If there is more than needed...
					if dStackSize > fillDeficit then -- Put the remainder back in the queue
						table.insert(data, 1, {dBag, dSlot, -(fillDeficit - dStackSize)})
					end --and retarget the module to the next last unfilled slot
					newTarget = table.remove(data)
					if newTarget then
						notFull[link] = {newTarget[1], newTarget[2], fullStack - newTarget[3], fullStack}
					end
				else -- Otherwise, update how much is needed to fill the target stack
					notFull[link][3] = fillDeficit - dStackSize
				end
			end
		end

		if #dustBag > 0 then
			self:Chat(L["Your bags rumble as Mr Plow gets busy and drives into your bags..."])
			self:RegisterEvents()
		else
			self:Chat(L["Mr Plow looks at your neat bags and shakes his head."])
		end
		self:SwapIt()
	end
	self:Debug(string.format("InventoryParse: |cff80ff80%.3f sec|r |cffff8000(%d KiB)", GetTime() - t, gcinfo() - mem))
end


function MrPlow:Defrag(bank)
	local t, mem = GetTime(), gcinfo()
	if #dustBag == 0 then
		local empty = {}
		local full = {}
		local qEmpty = {}
		local qFull = {}

		local function _BagDefrag(full, empty, bag)
			for slot = 1, GetContainerNumSlots(bag) do
				if not (self.db.char.ignoreSlotList[bag] and self.db.char.ignoreSlotList[bag][slot]) then
					local link
					if GetContainerItemLink(bag,slot) then
						link = select(3, GetContainerItemLink(bag,slot):find("item:(%d+):%d+:%d+:%d+"))
					end
					if not link or not self.db.char.ignoreList[link] then -- Skip over ignored items
						if self.db.profile.Direction == "Forwards" then
							if link then
								table.insert(full, 1, {bag, slot}) -- Insert the full spaces in reverse order
							else
								table.insert(empty, {bag, slot}) -- And the empty spaces in forward order
							end
						else
							if link then
								table.insert(full, {bag, slot}) -- Insert the full spaces in forward order
							else
								table.insert(empty, 1, {bag, slot}) -- And the empty spaces in reverse order
							end
						end
					end
				end
			end
		end
		
		local function _Defrag_Move(full, empty)
			if self.db.profile.Direction == "Forwards" then
				if #full <= #empty then
					for i,full in ipairs(full) do
						if full[1] > empty[i][1] or (full[1] == empty[i][1] and full[2] > empty[i][2]) then
							table.insert(dustBag, {{full[1], full[2]}, {empty[i][1], empty[i][2]}})
						end
					end
				else
					for i,empty in ipairs(empty) do
						if empty[1] < full[i][1] or (empty[1] == full[i][1] and empty[2] < full[i][2]) then
							table.insert(dustBag, {{full[i][1], full[i][2]}, {empty[1], empty[2]}})
						end
					end
				end
			else
				if #full <= #empty then
					for i,full in ipairs(full) do
						if full[1] < empty[i][1] or (full[1] == empty[i][1] and full[2] < empty[i][2]) then
							table.insert(dustBag, {{full[1], full[2]}, {empty[i][1], empty[i][2]}})
						end
					end
				else
					for i,empty in ipairs(empty) do
						if empty[1] > full[i][1] or (empty[1] == full[i][1] and empty[2] > full[i][2]) then
							table.insert(dustBag, {{full[i][1], full[i][2]}, {empty[1], empty[2]}})
						end
					end
				end
			end
		end

		local baglist = {}

		if bank == L["Bank"] and bankAvailable then
			baglist = self:GetBagList(true)
			plowingBank = true
		elseif bank == L["Bank"] and not bankAvailable then
			self:Print("Your Bank window is not open!")
			for k in pairs(theWorks) do theWorks[k] = nil end
			theWorksBank = nil
			return
		else
			baglist = self:GetBagList()
		end

		for i,bag in pairs(baglist) do
			if GetBagName(bag) or bankAvailable and bag == -1 then -- First, ignore all ammo pouches...
				local specialBag
				if bag > 0 then
					local bagslot = bag < 5 and bag + 19 or bag + 63
					local bagTLink = GetInventoryItemLink("player", bagslot) or nil
					self:Debug(bag)
					specialBag = PT:ItemInSet(bagTLink, "Misc.Bag.Special")
				end
				if not specialBag then
					_BagDefrag(full, empty, bag)
				else
					_BagDefrag(qFull, qEmpty, bag)
				end
			end
		end
		-- Now move the full items from the end to the empty spaces from the beginning
		for k in pairs(dustBag) do dustBag[k] = nil end
		_Defrag_Move(full, empty)
		_Defrag_Move(qFull, qEmpty)
		if #dustBag > 0 then
			self:Chat(L["Your bags rumble as Mr Plow gets busy and drives into your bags..."])
			self:RegisterEvents()
		else
			self:Chat(L["Mr Plow looks at your neat bags and shakes his head."])
		end
		self:SwapIt()
	end
	self:Debug(string.format("Defrag: |cff80ff80%.3f sec|r |cffff8000(%d KiB)", GetTime() - t, gcinfo() - mem))
end

function MrPlow:SortAll(bank)
	--[[
	-- For every item, I want to gather information about type/name/size and sort it in that order. Type will eventually be
	-- modifiable, but for now, lets get it working 
	--]]
	if #dustBag == 0 then
		local pile = {}
		local specialPile = {}
		local Stuff = {}
		Stuff.mt = {}
		local normalBlankSlate = {}
		local specialBlankSlate = {}
		local lookup = {}
		local indexLookup = {}
		local sIndexLookup = {}
		itemCount = 0

		function Stuff.new(num, ItemID, sbag, sslot)
			local stuff = {}
			setmetatable(stuff, Stuff.mt)
			stuff.OldPos = num
			stuff.itemName, stuff.itemLink, stuff.itemRarity, stuff.itemLevel, stuff.itemMinLevel, 
			stuff.itemType, stuff.itemSubType, stuff.itemStackCount, stuff.itemEquipLoc, stuff.itemTexture = GetItemInfo(ItemID)
			stuff.itemCount = select(2, GetContainerItemInfo(sbag, sslot))
			stuff.bag = -1
			stuff.bag = sbag
			stuff.slot = -1
			stuff.slot = sslot
			return stuff
		end
		
		function Stuff.Position(self,num)
			self.pos = num
		end
		
		function Stuff.mt.__tostring(stuff)
			return "Stuff: "..stuff.itemName.."("..stuff.itemLink..") at "..stuff.bag..":"..stuff.slot
		end

		function Stuff.mt.__lt(a, b)
			if a.itemLink == b.itemLink then
				if a.itemCount < b.itemCount then
					return true
				elseif a.itemCount > b.itemCount then
					return false
				else
					return a.OldPos < b.OldPos
				end
			end
	
			local itemA = itemRanking[a.itemType] or -1
			local itemB = itemRanking[b.itemType] or -1
			
			if self.db.profile.Junk then
				if a.itemRarity == 0 and b.itemRarity > 0 then
					return false
				elseif a.itemRarity > 0 and b.itemRarity == 0 then
					return true
				elseif a.itemRarity == 0 and b.itemRarity == 0 then
					return a.itemName < b.itemName
				end
			end
            
			if itemA < itemB then
				return true
			elseif itemA == itemB then
				if itemRanking[a.itemType] == itemCategories.TRADEGOODS then
					local aSet = select(2, PT:ItemInSet(a.itemLink, "Tradeskill.Mat.ByType", true))
					local bSet = select(2, PT:ItemInSet(b.itemLink, "Tradeskill.Mat.ByType", true))
					if type(aSet) == "string" and type(bSet) == "string" then
						local aRank = ingredientRanking[aSet] or -1
						local bRank = ingredientRanking[bSet] or -1
						if aRank < bRank then
							return true
						elseif aRank > bRank then
							return false
						end
					end
					return a.itemName < b.itemName

				elseif itemRanking[a.itemType] == itemCategories.ARMOR or itemRanking[a.itemType] == itemCategories.WEAPON then
					local aWep = armWepRank[a.itemEquipLoc] or -1
					local bWep = armWepRank[b.itemEquipLoc] or -1
					if aWep == bWep then
						if a.itemRarity == b.itemRarity then -- sort within by item rarity (common, etc) then itemLevel
							if a.itemLevel == b.itemLevel then
								return a.itemName < b.itemName
							else
								return a.itemLevel > b.itemLevel
							end
						else
							return a.itemRarity < b.itemRarity
						end
					else
						return aWep < bWep
					end
				elseif itemRanking[a.itemType] == itemCategories.CONSUMABLE then
					for i,v in ipairs(sortCategories) do
						local aitem = select(2, PT:ItemInSet(a.itemLink, v, true))
						local bitem = select(2, PT:ItemInSet(b.itemLink, v, true))
						if bitem == nil and aitem ~= nil then
							return false
						elseif aitem == nil and bitem ~= nil then
							return true
						end
					end
					return a.itemName < b.itemName
				end
				return a.itemName < b.itemName
			else
				return false
			end
		end

		local function _BagScan(storage, bag, blankSlate, indexLookup)
			self:Debug("Scanning "..bag)
			for slot=1, GetContainerNumSlots(bag) do
				if not(self.db.char.ignoreSlotList[bag] and self.db.char.ignoreSlotList[bag][slot]) and GetContainerItemLink(bag,slot) then
					local link = select(3, GetContainerItemLink(bag,slot):find("item:(%d+):%d+:%d+:%d+"))
					if not self.db.char.ignoreList[link] then
						local item = Stuff.new(itemCount, link, bag, slot)
						itemCount = itemCount + 1
						table.insert(storage, item)
						lookup[bag.."-"..slot] = item
						table.insert(indexLookup, #indexLookup + 1, bag.."-"..slot)
						table.insert(blankSlate, {bag, slot})
					end
				end
			end
		end

		local function _BagShift(pile, blankSlate)
			local item = table.remove(pile)
			while item do
				if not (item.bag == blankSlate[item.pos][1] and item.slot == blankSlate[item.pos][2]) then
					table.insert(dustBag, 1, {
						{item.bag, item.slot},
						{blankSlate[item.pos][1], blankSlate[item.pos][2]}
					}) -- Move the item into it's proper position
					local movedItem = lookup[blankSlate[item.pos][1].."-"..blankSlate[item.pos][2]]
					movedItem.bag = item.bag
					movedItem.slot = item.slot
					lookup[item.bag.."-"..item.slot] = movedItem
					for i,v in ipairs(pile) do
						if v == lookup[indexLookup[movedItem.pos]] then
							table.insert(pile, 1, table.remove(pile, i))
							break
						end
					end
				end
				item = table.remove(pile)
			end
		end

		local baglist

		if bank == L["Bank"] and bankAvailable then
			baglist = self:GetBagList(true)
			plowingBank = true
		elseif bank == L["Bank"] and not bankAvailable then
			self:Print("Your Bank window is not open!")
			for k in pairs(theWorks) do theWorks[k] = nil end
			theWorksBank = nil
			return
		else
			baglist = self:GetBagList()
		end

		for i,bag in pairs(baglist) do
			local specialBag
			if bag > 0 then
				local bagslot = bag < 5 and bag + 19 or bag + 63
				local bagTLink = GetInventoryItemLink("player", bagslot) or nil
				if bagTLink then
					specialBag = PT:ItemInSet(bagTLink, "Misc.Bag.Special")
				end
			end
			if not specialBag then
				_BagScan(pile, bag, normalBlankSlate, indexLookup)
			else
				_BagScan(specialPile, bag, specialBlankSlate, sIndexLookup)
			end
		end
		self:Debug("Original size = "..#pile + #specialPile)
		table.sort(pile)
		table.sort(specialPile)
		
		for i,v in ipairs(pile) do v.pos = i end -- rejig the iteration
		for i,v in ipairs(specialPile) do v.pos = i end
		
		for k in pairs(dustBag) do dustBag[k] = nil end
		_BagShift(pile, normalBlankSlate)
		_BagShift(specialPile, specialBlankSlate)
		self:Debug("Final:".. #dustBag)

		if #dustBag > 0 then
			self:Chat(L["Your bags rumble as Mr Plow gets busy and drives into your bags..."])
			self:RegisterEvents()
		else
			self:Chat(L["Mr Plow looks at your neat bags and shakes his head."])
		end

		self:SwapIt()
	end
end

function MrPlow:BankStack()
	if not bankAvailable then
		self:Print("Your Bank window is not open!")
		return
	end

	local bankContents = {}
	local bankbagContents = {}
	local bagContents = {}
	local bankSpace = {}

	local function _FindStacks(bankbag)
		bankbagContents[bankbag] = {}
		
		for slot = 1, GetContainerNumSlots(bankbag) do
			if GetContainerItemLink(bankbag,slot) then
				local stackSize = select(2, GetContainerItemInfo(bankbag,slot))
				--local link = select(3, GetContainerItemLink(bankbag,slot):find("item:(%d+):%d+:%d+:%d+"))
				local link = GetContainerItemLink(bankbag,slot)
				local fullStack = select(8, GetItemInfo(link))
				if not bankbagContents[bankbag][link] then bankbagContents[bankbag][link] = slot end
				if fullStack > 1 and fullStack == stackSize then
					-- If this is a stackable item... and is a full stack, add it
					if not bankContents[link] then bankContents[link] = {0} end
				elseif fullStack > 1 and fullStack ~= stackSize then
					-- If this is stackable and not a full stack add or overwrite the current one
					if not bankContents[link] or bankContents[link] == 0 then
						bankContents[link] = {fullStack - stackSize, bankbag, slot}
					end
				end
			else -- Add in as freespace
				if not bankSpace[bankbag] then bankSpace[bankbag] = {} end
				table.insert(bankSpace[bankbag], slot)
			end
		end
	end
	local function _FillStacks()
		for i,bag in ipairs(self:GetBagList()) do
			for slot=1, GetContainerNumSlots(bag) do
				if GetContainerItemLink(bag,slot) then
					--local link = select(3, GetContainerItemLink(bag,slot):find("item:(%d+):%d+:%d+:%d+"))
					local link = GetContainerItemLink(bag,slot)
					local stackSize = select(2, GetContainerItemInfo(bag,slot))
					if link then --Add to bagContents
						if bankContents[link] and bankContents[link][1] ~= 0 then
							if bankContents[link][1] < stackSize then -- if there's a matched unfull stack, add to it with as big a stack as possible
								table.insert(dustBag, {
									{bag, slot, bankContents[link][1]},
									{bankContents[link][2], bankContents[link][3]}
								})
								bankContents[link][1] = 0
								stackSize = stackSize - bankContents[link][1]
							else
								table.insert(dustBag, {
									{bag, slot, stackSize},
									{bankContents[link][2], bankContents[link][3]}
								})
								bankContents[link][1] = bankContents[link][1] - stackSize
								stackSize = 0
							end
						end
						if stackSize ~= 0 then
							if not bagContents[link] then bagContents[link] = {} end
							table.insert(bagContents[link], {bag, slot})
						end
					end
				end
			end
		end
	end
		-- We now have three tables, bankContents, bankSpace and bagContents.
	local function _OverflowBag(bankbag)
		if bankSpace[bankbag] then
			for i,v in ipairs(bankSpace[bankbag]) do
				for link, slot in pairs(bankbagContents[bankbag]) do
					if bagContents[link] and #bagContents[link] > 0 then
						local item = table.remove(bagContents[link])
						table.insert(dustBag, {{item[1], item[2]}, {bankbag, v}})
						table.remove(bankSpace[bankbag],i)
						if #bankSpace[bankbag] == 0 then bankSpace[bankbag] = nil end
						break
					end
				end
			end
		end
	end
	local function _OverflowRemainder()
		for bankBags, clearSpaces in pairs(bankSpace) do
			local specialBag
			if bankBags > 4 then
				local bagTLink = GetInventoryItemLink("player", bankBags + 63) or nil
				specialBag = PT:ItemInSet(bagTLink, "Misc.Bag.Special")
			end
			if not specialBag then
				local dobreak = nil
				for i, clearSlot in pairs(clearSpaces) do -- for each clear slot left.
					for link in pairs(bankContents) do -- look in the bank
						self:Debug(link)
						if bagContents[link] and #bagContents[link] > 0 then -- check if our bags hold it.
							local item = table.remove(bagContents[link])
							if bankSpace[bankBags][i] then
								table.insert(dustBag, {
									{item[1], item[2]},
									{bankBags, clearSlot}
								})
							end
							bankSpace[bankBags][i] = nil
							dobreak = true
							break
						end
					end
				end
			end
		end
	end
	
	--Right, first we get all bank contents...
	if bankAvailable then
		local baglist = self:GetBagList(true)
		local bsstyle = self.db.profile.BankStackStyle
		for i,v in pairs(baglist) do
			_FindStacks(v)
		end
		_FillStacks()
		if bsstyle >= 1 then
			for i,v in pairs(baglist) do
				_OverflowBag(v)
			end
		end
		if bsstyle >= 2 then
			_OverflowRemainder()
		end

		if #dustBag > 0 then
			self:Chat(L["Your bags rumble as Mr Plow gets busy and drives into your bags..."])
			self:RegisterEvents()
		else
			self:Chat(L["Mr Plow looks at your neat bags and shakes his head."])
		end	
		self:SwapIt()
	end
end

function MrPlow:DelayCalled()
	delayCalled = true
	if eventCalled then
		self:SwapIt()
		delayCalled = nil
		eventCalled = nil
	end
end

function MrPlow:ScanEvent()
	count = count + 1
	if math.fmod(count,2) == 0 then
		eventCalled = true
	end
	if eventCalled and delayCalled then
		self:SwapIt()
		delayCalled = nil
		eventCalled = nil
	else
		delayCalled = nil
		self:ScheduleEvent("MrPlow", self.DelayCalled, 0.15, self)
	end
end

function MrPlow:SwapIt()
	if swapping then
		return
	end
	if plowingBank and not bankAvailable then
		self:Print("Should be plowing bank, but it's not available!")
		for k in pairs(dustBag) do dustBag[k] = nil end
		self:BANKFRAME_CLOSED()
		return
	end
	swapping = true
	self:Debug("Size is: ".. #dustBag)
	if #dustBag > 0 then
		local detail = table.remove(dustBag)
		if detail then
			local from = detail[1]
			local to = detail[2]
			if random(1,24) == 4 then
				self:Chat(during[random(1, #during)])
			end
			if #from == 3 then
				SplitContainerItem(from[1], from[2], from[3])
			else
				PickupContainerItem(from[1], from[2])
			end
			if CursorHasItem() then
				PickupContainerItem(to[1], to[2])
			end
			if CursorHasItem() then
				PickupContainerItem(from[1], from[2])
			end
		end
	else
		self:UnregisterEvents()
		swapping = nil
		for k in pairs(dustBag) do dustBag[k] = nil end
		
		if #theWorks ~= 0 then
			func = table.remove(theWorks, 1)
			self:Debug("Running a new works with ".. tostring(theWorksBank))
			self[func](self, theWorksBank)
		else
			-- XXX test fix
			plowingBank = nil
			theWorksBank = nil
			self:Chat(L["As he drives off into the sunset, your bags deflate somewhat due to the extra room now available. That name again, is Mr Plow!"])
			collectgarbage()
		end
	end
	swapping = nil
end

function MrPlow:Chat(mess)
	if not self.db.profile.Gag then
		DEFAULT_CHAT_FRAME:AddMessage("|cffeda55f"..mess.."|r")
	end
end

function MrPlow:GetBagList(bank)
	local bagList
	local ignoreList

	self:Debug("Getting bag list for : "..tostring(bank))

	if bank then
		bagList = { -1, 5, 6, 7, 8, 9, 10, 11 }
		ignoreList = self.db.char.ignoreBankBagList or nil
	else
		bagList = { 0, 1, 2, 3, 4 }
		ignoreList = self.db.char.ignoreBagList or nil
	end
	
	if ignoreList then
		for i,v in pairs(ignoreList) do
			bagList[i] = nil
		end
	end
	return bagList
end


function MrPlow:ChangeIgnore(linklist, delete, purge)
	local tbl = self.db.char.ignoreList
	if purge then
		for k in pairs(tbl) do
			tbl[k] = nil
		end
		self:Print(L["Mr Plow has cleared the ignore list"])
	else
		local link
		for link in linklist:gmatch("%bH|") do
			local itemID = select(3, link:find("-*:(%d+):.*"))
			if itemID then
				if tbl[itemID] and delete then
					tbl[itemID] = nil
					self:Print(self:GetLink(itemID).." "..L["has been removed from the ignore list"])
				elseif not delete then
					tbl[itemID] = true
					self:Print(self:GetLink(itemID).." "..L["has been added to the ignore list"])
				end
			end
		end
	end
end

function MrPlow:ChangeSlotsIgnore(linklist, delete, purge)
	local tbl = self.db.char.ignoreSlotList
	if purge then
		for k in pairs(tbl) do
			tbl[k] = nil
		end
		self:Print(L["Mr Plow has cleared the ignore list"])
	else
		local link
		for link in linklist:gmatch("(%-?%d+%-%d+)[,]*") do
			local bag, slot = select(3, link:find("(%-?%d+)-(%d+)"))
			bag, slot = tonumber(bag), tonumber(slot)
			if bag and slot then
				if tbl[bag] and tbl[bag][slot] and delete then
					tbl[bag][slot] = nil
					self:Print(bag..","..slot.." "..L["has been removed from the ignore list"])
				elseif not delete then
					if not tbl[bag] then tbl[bag] = {} end
					tbl[bag][slot] = true
					self:Print(bag..","..slot.." "..L["has been added to the ignore list"])
				end
			end
		end
	end
end

function MrPlow:ChangeBagIgnore(list, delete, purge, bankbag)
	local tbl
	if bankbag then
		tbl = self.db.char.ignoreBankBagList
	else
		tbl = self.db.char.ignoreBagList
	end
	if purge then
		for k in pairs(tbl) do tbl[k] = nil end
		self:Print(L["Mr Plow has cleared the ignore list"])
		return
	end

	self:Debug(list)
	for bag in list:gmatch("([0-7]),?") do
		if bankbag then
			if delete then
				tbl[bag + 1] = nil
				self:Print(bag.." "..L["has been removed from the ignore list"])
			else
				tbl[bag + 1] = true
				self:Print(bag.." "..L["has been added to the ignore list"])
			end
		elseif tonumber(bag) < 5 then
			if delete then
				tbl[bag + 1] = nil
				self:Print(bag.." "..L["has been removed from the ignore list"])
			else
				tbl[bag + 1] = true
				self:Print(bag.." "..L["has been added to the ignore list"])
			end
		end
	end
end

function MrPlow:GetLink(linkInfo)
	local sName, sLink, iQuality = GetItemInfo(linkInfo)
	if sName then
		local color = select(4, GetItemQualityColor(iQuality))
		return string.format("%s%s|h|r", color, sLink)
	else
		return ""
	end
end

function MrPlow:OnTooltipUpdate()
	if not self.db.profile.ClickToRun then
		tablet:SetHint(L["|cffeda55fRight-Click|r for options."])
	else
		tablet:SetHint(L["|cffeda55fClick|r to run The Works on your bag, |cffeda55fShift-Click|r to run Bank The Works, |cffeda55fShift-Control-Click|r to run BankStack. |cffeda55fClick|r again to cancel."])
	end
end

function MrPlow:OnClick()
	if not self.db.profile.ClickToRun then
		return
	end
	if #dustBag == 0 then
		if IsShiftKeyDown() then
			if IsControlKeyDown() then
				self:BankStack()
			else
				self:Works(L["Bank"])
			end
		else
			self:Works()
		end
	else
		for k in pairs(dustBag) do dustBag[k] = nil end
		for k in pairs(theWorks) do theWorks[k] = nil end
		theWorksBank = nil
		self:Print(L["Stopping"])
	end
end

