--[[
    Mount by LordFarlander
    Version 1.4
    $Revision: 76986 $

    Based on Minipet.
]]--

--[[
Copyright (c) 2008, LordFarlander
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
]]--

local L = AceLibrary( "AceLocale-2.2" ):new( "Mount" );

_G["BINDING_NAME_CLICK Mount_Button:LeftButton"] = L["Summon Mount"];
BINDING_HEADER_MOUNTHEADER = L["Mount"];

local options = {
    type = "group",
    args = {
        [L["autoHideButton"]] = {
            type = "toggle",
            name = L["Automatically Hide Button"],
            desc = L["Hide the button when you can't mount?"],
            get = function()
                      return Mount.db.char.AutoHideButton;
                  end,
            set = function( val )
                      Mount.db.char.AutoHideButton = not Mount.db.char.AutoHideButton;
                      Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
                  end,
            },
--             toggleDebug = {
--                 type = "toggle",
--                 name = "Debug",
--                 desc = "Debug mode toggle",
--                 get = function()
--                           return Mount.DEBUG;
--                       end,
--                 set = function( val )
--                           Mount.DEBUG = not Mount.DEBUG;
--                       end,
--                 },
        [L["toggleButton"]] = {
            type = "toggle",
            name = L["Show Mount Button"],
            desc = L["Show/Hide the Mount Button."],
            get = function()
                      return Mount.button:IsVisible()
                  end,
            set = function( val )
                      if( Mount.button:IsVisible() ) then
                          Mount.button:Hide();
                      else
                          Mount.button:Show();
                      end
                  end,
            },
        [L["lockButton"]] = {
            type = "toggle",
            name = L["Lock Button Position"],
            desc = L["Lock the position of the button?"],
            get = function()
                      return Mount.db.char.LockButton;
                  end,
            set = function( val )
                      Mount.db.char.LockButton = not Mount.db.char.LockButton;
                  end,
            },
        [L["favorFlying"]] = {
            type = "toggle",
            name = L["Favor Flying Mounts"],
            desc = L["Toggles if flying mounts should be favored if in an area that allows them."],
            get = function()
                      return Mount.db.char.FavorFlying;
                  end,
            set = function( val )
                      Mount.db.char.FavorFlying = not Mount.db.char.FavorFlying;
                      Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
                  end,
            },
        [L["showModifierItems"]] = {
            type = "toggle",
            name = L["Show Modifier Items"],
            desc = L["Toggles if items that modify mounts (such as Fresh Holly) should be selected when holding Ctrl while mounted."],
            get = function()
                      return Mount.db.char.ShowModifierItems;
                  end,
            set = function( val )
                      Mount.db.char.ShowModifierItems = not Mount.db.char.ShowModifierItems;
                      Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
                  end,
            },
        [L["suggestOnlySoulbound"]] = {
            type = "toggle",
            name = L["Suggest Only Soulbound Mounts"],
            desc = L["Toggles if only mounts that are soulbound to you are to be suggested for use."],
            get = function()
                      return Mount.db.char.SuggestOnlySoulbound;
                  end,
            set = function( val )
                      Mount.db.char.SuggestOnlySoulbound = not Mount.db.char.SuggestOnlySoulbound;
                      Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
                  end,
            },
        [L["list"]] = {
            type = "execute",
            name = L["List all mounts"],
            desc = L["Show a list with all your mounts."],
            func = function()
                       Mount:ListMounts();
                   end,
            },
        [L["refresh"]] = {
            type = "execute",
            name = L["Refresh mount list"],
            desc = L["Force a refresh of the list of all your mounts."],
            func = function()
                       Mount:GetPlayersMounts();
                       Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
                   end,
            },
        },
    };

Mount = AceLibrary( "AceAddon-2.0" ):new( "AceConsole-2.0", "AceEvent-2.0", "AceDB-2.0" );
Mount.version = tonumber( string.match( "$Rev: 76986 $", "(%d+)" ) );
Mount:RegisterChatCommand( "/mount", "/mnt", options );
Mount:RegisterDB( "MountDB", "MountDBPC" );
Mount:RegisterDefaults( "char", {
        AutoHideButton = true,
        FavorFlying = true,
        LockButton = false,
        ShowModifierItems = true,
        SuggestOnlySoulbound = true,
        ButtonGroup = {
            SkinID = "Blizzard",
            Gloss = nil,
            Backdrop = nil,
            Colors = {},
            },
        FadeOptions = {
            fadeOut = false,
            fadeOutUpdateTime = 0.1,
            fadeOutCancelInCombat = true,
            fadeOutCancelOnShift = false,
            fadeOutCancelOnCtrl = false,
            fadeOutCancelOnAlt = false,
            alpha = 1.0,
            fadeOutDelay = 0.5,
            fadeOutAlpha = 0.2,
            fadeOutTime = 0.5,
            },
    } );

local PetAndMountDatabase = LibStub( "PetAndMountDatabase-1.0", true );
if( not PetAndMountDatabase ) then
    error( "Mount requires PetAndMountDatabase-1.0" );
    return;
end--if

local LibHelper = LibStub( "LibLordFarlander-1.0", true );
if( not LibHelper ) then
    error( "Mount requires LibLordFarlander-1.0" );
    return;
end--if

local ItemList = LibStub( "LibLordFarlander-ItemList-1.0", true );
if( not ItemList ) then
    error( "Mount requires LibLordFarlander-ItemList-1.0" );
    return;
end--if

local LibKeyBound = LibStub( "LibKeyBound-1.0", true );
if( not LibKeyBound ) then
    error( "Mount requires LibKeyBound-1.0" );
    return;
end--if

local LibStickyFrames = LibStub( "LibStickyFrames-2.0", true );
if( not LibStickyFrames ) then
    error( "Mount requires LibStickyFrames-2.0" );
    return;
end--if

--LibButtonFacade is optional
local LibButtonFacade = LibStub( "LibButtonFacade", true );

Mount.DEBUG = false;            --true = Debugging Mode
Mount.dprint = LibHelper.dprint;
Mount.button = nil;

--DO NOT CHANGE THESE VALUES
Mount.Ground = { Best = {}, NextBest = {}, Sets = {}, };
Mount.Flying = { Best = {}, NextBest = {}, Sets = {}, };

Mount.CurrentGround = nil; --Ground mount on the Button
Mount.CurrentFlying = nil; --Flying mount on the Button
Mount.CurrentMount = nil; --Current mount
Mount.CurrentMountItem = nil; --Current mount modifier item
Mount.NextCurrentGround = nil; --Ground mount on the Button
Mount.NextCurrentFlying = nil; --Flying mount on the Button
Mount.TempleOfAhnQiraj = nil; --Mount allowed in Temple Of Ahn'Qiraj
Mount.ReadyToInitialize = false;
Mount.Mounted = false;
--Mount.Indoors = false;
Mount.Dirty = false;
Mount.keyBoundMode = false;

function Mount:OnEnable()
    --Register the Events
    self:RegisterEvent( "BAG_UPDATE", "Update" );
    self:RegisterEvent( "SpecialEvents_LearnedSpell",
        function( spell, _ )
            local spellID = LibHelper.GetItemIDFromLink( GetSpellLink( spell ) );

            if( spellID ~= nil ) then

                if( PetAndMountDatabase:SpellInSet( spellID, "Spells.Mounts" ) ) then
                    Mount:Update();
                    return;
                end--if
            end--if
        end );
    self:RegisterEvent( "SpecialEvents_Mounted",
        function( _, _, _ )
            if( not Mount.Mounted ) then
                Mount.Mounted = true;
                Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
            end--if
        end );
    self:RegisterEvent( "SpecialEvents_Dismounted",
        function( _, _, _ )
            if( Mount.Mounted ) then
                Mount.Mounted = false;
                Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
            end--if
        end );
    self:RegisterEvent( "ZONE_CHANGED_NEW_AREA",
        function()
            if( Mount.DEBUG and LibHelper.IsInFlyableZone() ) then
                Mount:dprint( "Mount: In flyable zone" );
            end--if
            Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
        end );
--    self:RegisterEvent( "MINIMAP_UPDATE_ZOOM",
--        function()
--            Mount.Indoors = IsIndoors();
--            if( Mount.db.char.AutoHideButton and Mount.Indoors and Mount.CurrentMount ) then
--                Mount.button:Hide();
--            elseif( Mount.db.char.AutoHideButton and Mount.CurrentMount ) then
--                Mount.button:Show();
--            end--if
--        end );
    self:RegisterEvent( "MODIFIER_STATE_CHANGED",
        function( _, _ )
            Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
        end );
    self:RegisterEvent( "PLAYER_REGEN_ENABLED",
        function()
            Mount:SetButton( Mount:SetButtonBasedOnModifiers() );
        end );

    -- LibKeyBound support
    LibKeyBound.RegisterCallback( self, "LIBKEYBOUND_ENABLED" );
    LibKeyBound.RegisterCallback( self, "LIBKEYBOUND_DISABLED" );
    LibKeyBound.RegisterCallback( self, "LIBKEYBOUND_MODE_COLOR_CHANGED" );

    self.button = Mount_Button;
    LibStickyFrames:RegisterFrame( self.button );
    LibStickyFrames:SetFrameGroup( self.button, true );
    self.button.tooltip = Mount_Tooltip;

    self.button.tooltip.SetFromListEntry = ItemList.SetTooltipFromListEntry;
    self.button.SetFromListEntry = ItemList.SetButtonFromListEntry;
    self.button.SetImage = LibHelper.SetButtonImage;
    self.button.FadeOnUpdate = LibHelper.ButtonFadeOnUpdate;
    self.button.cooldown = _G[self.button:GetName() .. "Cooldown"];
    self.button.GetHotkey = function() return GetBindingKey( "BINDING_NAME_CLICK Mount_Button:LeftButton" ) end;
    self.button.GetActionName = function() return L["Mount"] end;

    if( LibButtonFacade ) then
        self.button.icon = _G[self.button:GetName() .. "Icon"];    
        LibButtonFacade:Group( "Mount" ):AddButton( self.button );
        LibButtonFacade:Group( "Mount" ):Skin( self.db.char.ButtonGroup.SkinID, self.db.char.ButtonGroup.Gloss, self.db.char.ButtonGroup.Backdrop, self.db.char.ButtonGroup.Colors );
        LibButtonFacade:RegisterSkinCallback( "Mount", self.SkinChanged, self );
    end--if

    --Upgrade check
    if( self.db.char.LastVersion ~= self.version ) then
        self.db.char.LastVersion = self.version;
    end--if

    self.Mounted = IsMounted();
--    self.Indoors = IsIndoors();

    self.Dirty = true
    --Create the Mount Inventory
    self:GetPlayersMounts();
    --Set a Mount
    self:ChooseRandomMounts();
end--Mount:OnEnable();

--For ButtonFacade support
function Mount:SkinChanged( SkinID, Gloss, Backdrop, barKey, buttonKey, Colors )
    self.db.char.ButtonGroup.SkinID = SkinID;
    self.db.char.ButtonGroup.Gloss = Gloss;
    self.db.char.ButtonGroup.Backdrop = Backdrop;
    self.db.char.ButtonGroup.Colors = Colors;
end--Mount:SkinChanged( SkinID, Gloss, Backdrop, barKey, buttonKey, Colors )

--For LibKeyBound support
function Mount:ColorButton()
    local overlayFrame = self.button;

    if( self.keyBoundMode ) then
        overlayFrame:SetBackdropColor( LibKeyBound:GetColorKeyBoundMode() );
    end--if
end--Mount:ColorButton()

--For LibKeyBound support
function Mount:LIBKEYBOUND_ENABLED()
	self.keyBoundMode = true;
	self:ColorButton();
end--Mount:LIBKEYBOUND_ENABLED()

--For LibKeyBound support
function Mount:LIBKEYBOUND_DISABLED()
	self.keyBoundMode = false;
	self:ColorButton();
end--Mount:LIBKEYBOUND_DISABLED()

--For LibKeyBound support
function Mount:LIBKEYBOUND_MODE_COLOR_CHANGED()
	self:ColorButton()
end--Mount:LIBKEYBOUND_MODE_COLOR_CHANGED()

--For LibStickyFrames support
function Mount:StartFrameMoving()
    LibStickyFrames:StartFrameMoving( self.button );
end--Mount:StartFrameMoving()

--For LibStickyFrames support
function Mount:StopFrameMoving()
    LibStickyFrames:StopFrameMoving( self.button );
end--Mount:StartFrameMoving()

function Mount:OnUpdate( elapsed )
    if( self.Dirty ) then
        --Update the list of mount items if needed
        self.Dirty = false;        
        if( self:HasMountListChanged() ) then
            self:ChooseRandomMounts();
        end--if
        self:SetButton( self:SetButtonBasedOnModifiers() );
    end--if
end--Mount:OnUpdate()

function Mount:Update()
    self.Dirty = true;
end--Mount:Update()

function Mount:SetButtonBasedOnModifiers()
    local IsFlyingModifierDown = IsAltKeyDown();
--    local IsPassengerModifierDown = IsControlKeyDown();
    local IsNextBestModifierDown = IsControlKeyDown() or IsShiftKeyDown();

    if( self.DEBUG ) then
        if( IsFlyingModifierDown ) then
            self:dprint( "IsFlyingModifierDown == true" );
        else
            self:dprint( "IsFlyingModifierDown == false" );
        end--if
        if( IsNextBestModifierDown ) then
            self:dprint( "IsNextBestModifierDown == true" );
        else
            self:dprint( "IsNextBestModifierDown == false" );
        end--if
    end--if
    if( self.Mounted and self.db.char.ShowModifierItems and IsControlKeyDown() and self.CurrentMountItem ) then
        return self.CurrentMountItem;
    elseif( self.Mounted ) then
        return nil
    elseif( ( GetZoneText() == L["Temple Of Ahn'Qiraj"] ) and self.TempleOfAhnQiraj ) then
        return self.TempleOfAhnQiraj;
    else
        --Pick a mount
        if( self.db.char.FavorFlying and LibHelper.IsInFlyableZone() and ( self.NextCurrentFlying or self.CurrentFlying ) ) then
            if( IsFlyingModifierDown and ( self.NextCurrentGround or self.CurrentGround ) ) then
                if( IsNextBestModifierDown and self.NextCurrentGround ) then
                    return self.NextCurrentGround;
                else
                    return self.CurrentGround;
                end--if
            else
                if( IsNextBestModifierDown and self.NextCurrentFlying ) then
                    return self.NextCurrentFlying;
                else
                    return self.CurrentFlying;
                end--if
            end--if
        elseif( LibHelper.IsInFlyableZone() and ( self.NextCurrentFlying or self.CurrentFlying ) ) then
            if( IsFlyingModifierDown or not ( self.NextCurrentGround or self.CurrentGround ) ) then
                if( IsNextBestModifierDown and self.NextCurrentFlying ) then
                    return self.NextCurrentFlying;
                else
                    return self.CurrentFlying;
                end--if
            else
                if( IsNextBestModifierDown and self.NextCurrentGround ) then
                    return self.NextCurrentGround;
                else
                    return self.CurrentGround;
                end--if
            end--if
        else
            if( IsNextBestModifierDown ) then
                return self.NextCurrentGround;
            else
                return self.CurrentGround;
            end--if
        end--if
    end--if
    return nil;
end--Mount:SetButtonBasedOnModifiers()

----------------------------
--+Pet/Inventory Functions--
function Mount:ChooseRandomMounts()
    self.CurrentGround = ItemList.GetRandomItemFromList( self.Ground.Best, self.CurrentGround );
    self.NextCurrentGround = ItemList.GetRandomItemFromList( self.Ground.NextBest, self.NextCurrentGround );
    
    self.CurrentFlying = ItemList.GetRandomItemFromList( self.Flying.Best, self.CurrentFlying );
    self.NextCurrentFlying = ItemList.GetRandomItemFromList( self.Flying.NextBest, self.NextCurrentFlying );

    self:SetButton( self:SetButtonBasedOnModifiers() );
end--Mount:ChooseRandomMounts()

function Mount.GetBestSets( sets )
    local Best = nil;
    local NextBest = nil;

    for _, set in pairs( sets ) do
        if( Best == nil ) then
            Best = set;
        else
            if( set.speed > Best.speed ) then
                if( ( NextBest == nil ) or ( Best.speed > NextBest.speed ) ) then
                    NextBest = Best;
                end--if
                Best = set;
            elseif( ( NextBest == nil ) or ( set.speed > NextBest.speed ) ) then
                NextBest = set;
            end--if
        end--if
    end--for
    if( Best == nil ) then
        Best = {};
    else
        Best = Best.Items;
    end--if
    if( NextBest == nil ) then
        NextBest = {};
    else
        NextBest = NextBest.Items;
    end--if
    return Best, NextBest;
end--Mount:GetBestSets()

function Mount.AddIfMount( ptsets, dest, bag, slot, itemID )
    for _, set in pairs( ptsets ) do
        local speed = PetAndMountDatabase:ItemInSet( itemID, set );

        if( speed ) then
            if( dest[set] == nil ) then
                dest[set] = { Items = {}, ["speed"] = tonumber( speed ), };
            end--if
            ItemList.AddBagItemToList( dest[set].Items, bag, slot );
            return true;
        end--if
    end--for
    return false;
end--Mount.GetPlayersMountsOnType( ptsets, dest )

function Mount.GetPlayerSpellMounts( spellDatabase, mountTable )
    local iterator, handle = PetAndMountDatabase:IterateSet( spellDatabase );

    if( iterator and handle ) then
        local spellID, speed = iterator( handle );
    
        while spellID do
            spellID = spellID * -1;
            if( LibHelper.HasPlayerSpell( spellID ) ) then
                Mount:dprint( "Found spell " .. spellID );
                if( mountTable.speed == nil ) then
                    mountTable.speed = tonumber( speed );
                end--if
                ItemList.AddSpellToList( mountTable.Items, spellID );
            end--if
            spellID, speed = iterator( handle );
        end--for
    end--if
end--Mount.GetPlayerSpellMounts( spellDatabase, mountTable )

function Mount.AddSpells( ptsets, dest )
    for _, set in pairs( ptsets ) do
        local SpellSet = ("%s.Spells"):format( set );

        Mount:dprint( "Checking for set " .. SpellSet );
        if( PetAndMountDatabase:HasSet( SpellSet ) ) then
            local destSet = dest[set];

            if( destSet == nil ) then
                destSet = { Items = {}, };
            end--if
            Mount.GetPlayerSpellMounts( set, destSet );
            if( ( dest[set] == nil ) and ( destSet.speed ~= nil ) ) then
                dest[set] = destSet;
            end--if
        end--if
    end--for
end--

--Gets all your pets from your inventory.
function Mount:GetPlayersMounts()
    -- First do ground mountItems, best first
    local opSuggestOnlySoulbound = self.db.char.SuggestOnlySoulbound;

    self.CurrentMountItem = nil;
    self.Ground = { Best = {}, NextBest = {}, Sets = {}, };
    self.Flying = { Best = {}, NextBest = {}, Sets = {}, };

    local GroundMounts = PetAndMountDatabase:GetSubsets( "Mounts.Ground" );
    local FlyingMounts = PetAndMountDatabase:GetSubsets( "Mounts.Flying" );
    local GroundSets = self.Ground.Sets;
    local FlyingSets = self.Flying.Sets;

    for bag = 0, NUM_BAG_SLOTS do
        for slot = 1, GetContainerNumSlots( bag ) do
            local itemLink = GetContainerItemLink( bag, slot );

            if( itemLink ) then
                local itemID = LibHelper.GetItemIDFromLink( itemLink );

                if( ( not opSuggestOnlySoulbound ) or LibHelper.ItemIsSoulbound( bag, slot ) ) then
                    self.AddIfMount( GroundMounts, GroundSets, bag, slot, itemID );
                    self.AddIfMount( FlyingMounts, FlyingSets, bag, slot, itemID );
                    if( ( GetZoneText() == L["Temple Of Ahn'Qiraj"] ) and PetAndMountDatabase:ItemInSet( itemID, "TempleOfAhnQiraj.Mounts" ) ) then
                        self.TempleOfAhnQiraj = ItemList.BuildBagItem( bag, slot );
                    end--if
                end--if
            end--if
        end--for
    end--for

    Mount.AddSpells( GroundMounts, GroundSets );
    Mount.AddSpells( FlyingMounts, FlyingSets );

    self.Ground.Best, self.Ground.NextBest = self.GetBestSets( GroundSets );
    self.Flying.Best, self.Flying.NextBest = self.GetBestSets( FlyingSets );

    local iterator, handle = PetAndMountDatabase:IterateSet( "Items.MountEnchant" );
    local item, _ = iterator( handle );

    while item do
        if( GetItemCount( item ) > 0 ) then
            self.CurrentMountItem = ItemList.BuildItem( item );
            break;
        end--if
        item, _ = iterator( handle );
    end--for
end--Mount:GetPlayersMounts()

function Mount:PrintMountList( mountTable, string )
    if( #mountTable > 0 ) then
        self:Print( L["%s mounts:"]:format( string ) );
        for _, mount in pairs( mountTable ) do
            self:Print( "-|cFF00FFFF" .. mount:GetName() );
        end--for
    end--if
end--Mount:PrintMountList()

function Mount:ListMountType( type, lists )
    local sorted = {};

    for name, _ in pairs( lists ) do
        table.insert( sorted, name );
    end--for
    self.CompareList = lists;
    table.sort( sorted, function( a, b ) return Mount.CompareList[a].speed < Mount.CompareList[b].speed end );
    self.CompareList = nil;
    for _, name in pairs( sorted ) do
        self:PrintMountList( lists[name].Items, L["%i%% speed %s"]:format( lists[name].speed, type ) );
    end--for
end--Mount.ListMountType( type, list )

--Prints out a list with all your mountItems.
function Mount:ListMounts()
    self:Print( L["List of your mounts:"] );

    self:ListMountType( L["ground"], self.Ground.Sets );
    self:ListMountType( L["flying"], self.Flying.Sets );
end--Mount:ListMounts()
---Mount/Inventory Functions--
----------------------------

-------------------
--+Help Functions--
function Mount:HasMountListChanged()
    local selfBestGround = self.Ground.Best;
    local selfBestFlying = self.Flying.Best;
    local selfNextBestGround = self.Ground.NextBest;
    local selfNextBestFlying = self.Flying.NextBest;
    local preGroundCount = #selfBestGround;
    local preFlyingCount = #selfBestFlying;
    local preNextGroundCount = #selfNextBestGround;
    local preNextFlyingCount = #selfNextBestFlying;
    local preBestGround =  {};
    local preBestFlying = {};
    local preNextBestGround =  {};
    local preNextBestFlying = {};

    LibHelper.CopyTable( preBestGround , selfBestGround );
    LibHelper.CopyTable( preBestFlying , selfBestFlying );
    LibHelper.CopyTable( preNextBestGround , selfNextBestGround );
    LibHelper.CopyTable( preNextBestFlying , selfNextBestFlying );

    self:GetPlayersMounts();
    
    selfBestGround = self.Ground.Best;
    selfBestFlying = self.Flying.Best;
    selfNextBestGround = self.Ground.NextBest;
    selfNextBestFlying = self.Flying.NextBest;

    -- differences in sizes always means the list has changed
    if( ( preGroundCount ~= #selfBestGround ) or ( preFlyingCount ~= #selfBestFlying ) or ( not self.ReadyToInitialize ) or
        ( preNextGroundCount ~= #selfNextBestGround ) or ( preNextFlyingCount ~= #selfNextBestFlying ) or ( not self.ReadyToInitialize ) ) then
        self.ReadyToInitialize = true;
        return true;
    end
    self.ReadyToInitialize = true;
    return not( LibHelper.CompareTables( preBestGround, selfBestGround ) and LibHelper.CompareTables( preBestFlying, selfBestFlying ) and
        LibHelper.CompareTables( preNextBestGround, selfNextBestGround ) and LibHelper.CompareTables( preNextBestFlying, selfNextBestFlying ) );
end--Mount:HasMountListChanged()
---Help Functions--
-------------------

-----------------
--+UI Functions--
function Mount:SetGameTooltips()
    local mntButton = self.button;

    if( not self.keyBoundMode ) then
        local mountItem = self.CurrentMount;
        local mntTooltip = mntButton.tooltip;
        local chardb = self.db.char;

        mntTooltip:SetOwner( mntButton, "RIGHT" );
        mntTooltip:SetBackdropColor( 0.0, 0.0, 0.0, 1.0 );
        if( self.Mounted and ( not ( chardb.ShowModifierItems and self.CurrentMountItem and IsControlKeyDown() ) ) ) then
            if( chardb.ShowModifierItems ) then
                mntTooltip:SetText( L["Dismount (if flying hold down ctrl or alt and click to dismount)"] );
            else
                mntTooltip:SetText( L["Dismount (if flying hold down shift, ctrl, or alt and click to dismount)"] );
            end--if
        elseif( mountItem == nil ) then
            mntTooltip:SetText( L["You have no mounts."] );
        else
            mntTooltip:SetFromListEntry( mountItem );
        end--if
    end--if
    -- LibKeyBound support
    if( mntButton.GetHotkey ) then
       LibKeyBound:Set( mntButton );
    end--if
end--Mount:SetGameTooltips()

function Mount:SetButton( mountItem )
    local mntButton = self.button;
    local chardb = self.db.char;

    if( not UnitAffectingCombat( "player" ) ) then
--        if( chardb.AutoHideButton and self.Indoors ) then
--            mntButton.cooldown:Hide();
--            mntButton:Hide();
        if( chardb.AutoHideButton and mountItem ) then
            mntButton:Show();
        end--if
        if( self.Mounted and ( not ( chardb.ShowModifierItems and self.CurrentMountItem and IsControlKeyDown() ) ) ) then
            mntButton.cooldown:Hide();
            mntButton:Show();
            mntButton:Enable();
            mntButton:SetAttribute( "*type1", "macro" );
            mntButton:SetAttribute( "macrotext", "/dismount [noflying][mod]" );

            mntButton:SetImage( "Interface\\AddOns\\Mount\\Img\\Dismount" );
    
            self:dprint( "Is mounted" );
        else
            if( mountItem == nil ) then
                if( chardb.AutoHideButton ) then
                    mntButton:Hide();
                end--if
                mntButton.cooldown:Hide();
                mntButton:SetAttribute( "*type1", "item" );
                mntButton:SetAttribute( "*bag1", nil );
                mntButton:SetAttribute( "*slot1", nil );
                mntButton:SetAttribute( "*item1", nil );

                mntButton:SetImage( "Interface\\AddOns\\Mount\\Img\\Mount" );
    
                self:dprint( "No Mounts" );
            else
                mntButton:Show();
                mntButton:Enable();
                mntButton:SetFromListEntry( mountItem, not LibHelper.CompareTables( self.CurrentMount, mountItem ) );
            end--if
        end--if
        self.CurrentMount = mountItem;
    end--if
end--Mount:SetButton( mountItem )
---UI Functions--
-----------------
