--[[
    Database of all minipets and mounts
    $Revision: 77084 $
]]--

--[[
Copyright (c) 2008, LordFarlander
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
]]--

local WOW_VERSION = "20403";
local MAJOR_VERSION = "PetAndMountDatabase-1.0";
local MINOR_VERSION = "$Rev: 77084 $";

local PetAndMountDatabase, oldminor = LibStub:NewLibrary( MAJOR_VERSION, MINOR_VERSION );
if( not PetAndMountDatabase ) then
    return;
end

local pt3 = LibStub( "LibPeriodicTable-3.1", true );

if( not pt3 ) then
    error( MAJOR_VERSION .. " requires LibPeriodicTable-3.1" );
end

pt3:AddData( "PetAndMountDatabase", MINOR_VERSION, {
        ["PetAndMountDatabase.Spells.Teleport.Druid"] = "-18960",
        ["PetAndMountDatabase.MiniPets.Normal"] = "29956,29958,29960,11110,25535,31760,22114,10360,10361,29363,23015,12529,31665,29901,29903,34535,20769,11023,32588,8485,8486,8487,8488,8489,8490,8491,8492,10398,23083,8495,8496,8497,32616,8499,29953,32622,29957,15996,13584,34425,30360,38628,27445,23002,19450,35350,29364,11826,13582,34493,32233,33993,12264,11027,11026,20371,10822,22235,29902,29904,21277,33816,13583,8498,35504,8500,35349,10392,33154,34955,34492,10394,23007,11474,11825,23713,8494,34478,33818,4401,8501,38050,32617,10393",
        ["PetAndMountDatabase.Mounts.Ground.VeryFast.Spells.Paladin.Alliance"] = "-23214:100",
        ["PetAndMountDatabase.Spells.Teleport.Translocate"] = "-35727,-30141,-35730,-25649,-25650,-25666,-25140,-25652,-25143,-32568,-35376,-29128,-29129,-26572,-32571,-32572,-45371,-45368,-32569",
        ["PetAndMountDatabase.Spells.Teleport.Items.Engineering.Gnomish"] = "-23452:18986,-30544:30544",
        ["PetAndMountDatabase.Mounts.Flying.Fast.Items"] = "25470:60,25474:60,25471:60,25475:60,34060:60,33176:60,25476:60,35225:60,25472:60",
        ["PetAndMountDatabase.Spells.Teleport.Mage.Horde"] = "-3566,-3567,-35715,-32272,-3563,-49358",
        ["PetAndMountDatabase.Mounts.Flying.VeryFast.Items"] = "25477:280,33999:280,32316:280,32857:280,32317:280,32859:280,32860:280,32861:280,32319:280,25531:280,25529:280,25532:280,34061:280,25533:280,25527:280,35226:280,25528:280,25473:280,33182:280,32318:280,32862:280,32858:280,32314:280",
        ["PetAndMountDatabase.Mounts.Flying.ExtremelyFast.Items"] = "35600:310,30609:310,34092:310,32458:310,35800:310",
        ["PetAndMountDatabase.Mounts.Ground.Fast.Spells.Warlock"] = "-5784:60",
        ["PetAndMountDatabase.Mounts.Ground.Fast.Spells.Paladin.Alliance"] = "-13819:60",
        ["PetAndMountDatabase.Mounts.Ground.Fast.Spells.Paladin.Horde"] = "-34769:60",
        ["PetAndMountDatabase.Spells.Teleport.Shaman"] = "-556",
        ["PetAndMountDatabase.MiniPets.Children"] = "31880,31881,18598,18597",
        ["PetAndMountDatabase.Mounts.Ground.Fast.Items"] = "1132:60,15290:60,14062:60,3321:60,5665:60,8588:60,8591:60,5872:60,33976:60,8592:60,5668:60,5873:60,8563:60,8595:60,29220:60,29221:60,29222:60,13322:60,8629:60,8628:60,8632:60,5655:60,8631:60,33224:60,2411:60,5656:60,8627:60,1134:60,21044:60,1041:60,33183:60,28481:60,2414:60,29743:60,29744:60,13331:60,28927:60,13332:60,5864:60,13333:60,15277:60",
        ["PetAndMountDatabase.Mounts.Ground.Slow.Items"] = "33189:0,23720:0",
        ["PetAndMountDatabase.MiniPets.Quest"] = "30803:10629,34253:11516",
        ["PetAndMountDatabase.Mounts.Ground.VeryFast.Spells.Warlock"] = "-23161:100",
        ["PetAndMountDatabase.MiniPets.Reagented.Snowball"] = "21308:17202,21309:17202,21301:17202,21305:17202",
        ["PetAndMountDatabase.Mounts.Ground.VeryFast.Items"] = "19030:100,18778:100,34129:100,18786:100,18788:100,18790:100,18794:100,30480:100,29466:100,29468:100,29470:100,29472:100,29224:100,29228:100,29230:100,29105:100,8586:100,29746:100,33225:100,29256:100,19872:100,33809:100,18241:100,18243:100,18245:100,18247:100,19902:100,12302:100,12303:100,18767:100,18773:100,19029:100,18777:100,13327:100,13328:100,13329:100,18785:100,18787:100,18789:100,18791:100,18793:100,13335:100,18797:100,29467:100,29469:100,29471:100,13086:100,29223:100,29227:100,29102:100,12330:100,29745:100,29747:100,29229:100,33977:100,13326:100,13317:100,35513:100,18246:100,15293:100,35906:100,13334:100,28936:100,18776:100,33184:100,18772:100,29231:100,29103:100,12351:100,31836:100,12353:100,12354:100,31834:100,31833:100,18902:100,31830:100,31829:100,18242:100,18244:100,31835:100,29104:100,31832:100,31831:100,32768:100,28915:100,18766:100,18798:100,18796:100,15292:100,18774:100,18795:100",
        ["PetAndMountDatabase.Mounts.Ground.VeryFast.Items.TempleOfAhnQiraj"] = "21176:100,21324:100,21323:100,21321:100,21218:100",
        ["PetAndMountDatabase.Mounts.Ground.VeryFast.Spells.Paladin.Horde"] = "-34767:100",
        ["PetAndMountDatabase.Spells.Teleport.Items"] = "-8690:6948,-46149:35230,-39937:28585",
        ["PetAndMountDatabase.Items.MountEnchant"] = "21212,21213",
        ["PetAndMountDatabase.MiniPets.Equipment.Head"] = "12185:HEADSLOT",
        ["PetAndMountDatabase.Spells.Teleport.Mage.Alliance"] = "-49359,-33690,-32271,-3561,-3565,-3562",
        ["PetAndMountDatabase.Spells.Teleport.Items.Engineering.Goblin"] = "-36890:30542,-23442:18984",

        -- Shortcuts
        ["PetAndMountDatabase.Spells.Mounts.Ground.VeryFast"] = "m,PetAndMountDatabase.Mounts.Ground.VeryFast.Spells";
        ["PetAndMountDatabase.Spells.Mounts.Ground.Fast"] = "m,PetAndMountDatabase.Mounts.Ground.Fast.Spells";
        ["PetAndMountDatabase.Spells.Alliance"] = "m,PetAndMountDatabase.Spells.Teleport.Mage.Alliance,PetAndMountDatabase.Mounts.Ground.Fast.Spells.Paladin.Alliance,PetAndMountDatabase.Mounts.Ground.VeryFast.Spells.Paladin.Alliance";
        ["PetAndMountDatabase.Spells.Horde"] = "m,PetAndMountDatabase.Spells.Teleport.Mage.Horde,PetAndMountDatabase.Mounts.Ground.Fast.Spells.Paladin.Horde,PetAndMountDatabase.Mounts.Ground.VeryFast.Spells.Paladin.Horde";
        ["PetAndMountDatabase.Spells.Druid"] = "m,PetAndMountDatabase.Spells.Teleport.Druid";
        ["PetAndMountDatabase.Spells.Mage"] = "m,PetAndMountDatabase.Spells.Teleport.Mage";
        ["PetAndMountDatabase.Spells.Paladin"] = "m,PetAndMountDatabase.Mounts.Ground.Fast.Spells.Paladin,PetAndMountDatabase.Mounts.Ground.VeryFast.Spells.Paladin";
        ["PetAndMountDatabase.Spells.Shaman"] = "m,PetAndMountDatabase.Spells.Teleport.Shaman";
        ["PetAndMountDatabase.Spells.Warlock"] = "m,PetAndMountDatabase.Mounts.Ground.Fast.Spells.Warlock,PetAndMountDatabase.Mounts.Ground.VeryFast.Spells.Warlock";
        ["PetAndMountDatabase.Items.Mounts.Ground.Slow"] = "m,PetAndMountDatabase.Mounts.Ground.Slow.Items";
        ["PetAndMountDatabase.Items.Mounts.Ground.Fast"] = "m,PetAndMountDatabase.Mounts.Ground.Fast.Items";
        ["PetAndMountDatabase.Items.Mounts.Ground.VeryFast"] = "m,PetAndMountDatabase.Mounts.Ground.VeryFast.Items";
        ["PetAndMountDatabase.Items.Mounts.Flying.Fast"] = "m,PetAndMountDatabase.Mounts.Flying.Fast.Items";
        ["PetAndMountDatabase.Items.Mounts.Flying.VeryFast"] = "m,PetAndMountDatabase.Mounts.Flying.VeryFast.Items";
        ["PetAndMountDatabase.Items.Mounts.Flying.ExtremelyFast"] = "m,PetAndMountDatabase.Mounts.Flying.ExtremelyFast.Items";
        ["PetAndMountDatabase.Items.MiniPets"] = "m,PetAndMountDatabase.MinPets";
        ["PetAndMountDatabase.TempleOfAhnQiraj.Mounts"] = "m,PetAndMountDatabase.Mounts.Ground.VeryFast.Items.TempleOfAhnQiraj";
        ["PetAndMountDatabase.MountEnchant"] = "m,PetAndMountDatabase.Items.MountEnchant";
        -- Future
        --["PetAndMountDatabase.Mounts.Flying.ExtremelyFast.Items.Passenger"] = "",
        --["PetAndMountDatabase.Mounts.Ground.VeryFast.Items.Passenger"] = "",
        --["PetAndMountDatabase.Mounts.Passenger.Flying.ExtremelyFast.Items"] = "m,PetAndMountDatabase.Mounts.Flying.ExtremelyFast.Items.Passenger";
        --["PetAndMountDatabase.Mounts.Passenger.Ground.VeryFast.Items"] = "m,PetAndMountDatabase.Mounts.Ground.VeryFast.Items.Passenger";
    } );
    
--[[
    Value meanings:
    
    Mounts.*: Maximum speed increase over running in percent, where running is 100%
    MiniPets.Reagented: ItemID of item needed to summon (snowball)
    MiniPets.Quest: QuestID of quest item is used for
    MiniPets.Equipment: SlotName of slot the equipment goes in
    
    Negative number is a SPELL instead of an ITEM
]]--

PetAndMountDatabase.Version = WOW_VERSION; -- Matches the WoW version number this is accurate for

-- Backwards compatibility stuff
PetAndMountDatabase.Collections = {
        MiniPets = {
                Normal = "MiniPets.Normal",
                Holiday = "MiniPets.Reagented.Snowball",
                Equipment = "MiniPets.Equipment",
                Children = "MiniPets.Children",
                Quest = "MiniPets.Quest",
                Reagented = "MiniPets.Reagented",
            }, -- MiniPets

        Mounts = {
                Ground = {
                        Slow = "Mounts.Ground.Slow",
                        Fast = "Mounts.Ground.Fast",
                        VeryFast = "Mounts.Ground.VeryFast",
                    }, -- Ground
                Flying = {
                        Fast = "Mounts.Flying.Fast",
                        VeryFast = "Mounts.Flying.VeryFast",
                        ExtremelyFast = "Mounts.Flying.ExtremelyFast",
                    }, -- Flying
            }, -- Mounts
    }; -- Collections
    
PetAndMountDatabase.FoundSets = {};
PetAndMountDatabase.SubSets = {};

function PetAndMountDatabase.NormalizeSetName( set )
    if( set:find( "PetAndMountDatabase." ) ~= 1 ) then
        set = "PetAndMountDatabase." .. set;
    end--if
    if( set == "PetAndMountDatabase.MiniPets.Holiday" ) then
        set = "PetAndMountDatabase.MiniPets.Reagented.Snowball";
    end--if
    return set;
end--PetAndMountDatabase.NormalizeSetName( set )

function PetAndMountDatabase:HasSet( set )
    local sets = pt3.sets;

    set = PetAndMountDatabase.NormalizeSetName( set );
    if( self.FoundSets[set] ) then
        return true;
    end--if
    for setName, _ in pairs( sets ) do
        if( setName:find( set ) == 1 ) then
            self.FoundSets[set] = true;
            return true;
        end--if
    end--if
    return false;
end--PetAndMountDatabase:HasSet( set )

function PetAndMountDatabase:GetSubsets( set )
    local sets = pt3.sets;

    set = PetAndMountDatabase.NormalizeSetName( set );
    if( self.SubSets[set] ) then
        return self.SubSets[set];
    end--if
    set = set .. ".";
    for setName, _ in pairs( sets ) do
        local start, ends = setName:find( set );

        if( ( start == 1 ) and ( ends ~= setName:len() ) ) then
            setName = setName:sub( ends + 1 );
            if( setName:find( "." ) ) then
                setName = ("%s%s"):format( set, strsplit( ".", setName ) );
            end--if
            if( self.SubSets[set] == nil ) then
                self.SubSets[set] = {};
            end--if
            self.SubSets[set][setName] = setName;
        end--if
    end--if
    return self.SubSets[set];
end--PetAndMountDatabase:GetSubsets( set )

function PetAndMountDatabase:ItemInSet( item, set )
    if( item == nil ) then
        return false;
    end--if
    set = PetAndMountDatabase.NormalizeSetName( set );
    return pt3:ItemInSet( item, set );
end--PetAndMountDatabase:ItemInSet( item, set )

function PetAndMountDatabase:SpellInSet( spellID, set )
    if( spellID == nil ) then
        return false;
    end--if
    if( spellID > 0 ) then
        spellID = spellID * -1;
    end--if
    return self:ItemInSet( spellID, set );
end--PetAndMountDatabase:SpellInSet( item, set )

function PetAndMountDatabase:IterateSet( set )
    set = PetAndMountDatabase.NormalizeSetName( set );
    return pt3:IterateSet( set );
end--PetAndMountDatabase:IterateSet( set )
