--[[
    Helper functions
    $Revision: 76985 $
]]--

--[[
Copyright (c) 2008, LordFarlander
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
]]--

local MAJOR_VERSION = "LibLordFarlander-ItemList-1.0";
local MINOR_VERSION = "$Rev: 76985 $";

local LibHelper = LibStub( "LibLordFarlander-1.0" );
if( not LibHelper ) then
    error( MAJOR_VERSION .. " requires LibLordFarlander-1.0" );
    return;
end

local ItemList, oldminor = LibStub:NewLibrary( MAJOR_VERSION, MINOR_VERSION );
if( not ItemList ) then
    return;
end

function ItemList.BuildBasicItem()
    return { GetName = ItemList.GetNameFromListEntry, GetType = ItemList.GetListEntryType, GetLink = ItemList.GetNameFromListEntry, };
end--ItemList.BuildBasicItem()

function ItemList.BuildBagItem( bag, slot )
    local returnVal = ItemList.BuildBasicItem();

    returnVal.bag = bag;
    returnVal.slot = slot;
    return returnVal;
end--ItemList.BuildBagItem( bag, slot )

function ItemList.BuildInventoryItem( invslot )
    local returnVal = ItemList.BuildBasicItem();

    if( type( invslot ) == string ) then
        invslot = GetInventorySlotInfo( invslot );
    end--if
    returnVal.invslot = invslot;
    return returnVal;
end--ItemList.BuildInventoryItem( bag, slot )

function ItemList.BuildSpell( spellID )
    local returnVal = ItemList.BuildBasicItem();
    
    returnVal.spellID = spellID;
    return returnVal;
end--ItemList.BuildSpell( bag, slot )

function ItemList.BuildItem( itemID )
    local returnVal = ItemList.BuildBasicItem();

    if( type( itemID ) == "string" ) then
        itemID = LibHelper.GetItemIDFromLink( LibHelper.GetItemLink( itemID ) );
    end--if
    returnVal.itemID = itemID;
    return returnVal;
end--ItemList.BuildItem( bag, slot )

function ItemList.AddBagItemToList( list, bag, slot )
    table.insert( list, ItemList.BuildBagItem( bag, slot ) );
end--LibHelperItemList.AddBagItemToList( list, bag, slot )

function ItemList.AddInventoryItemToList( list, invslot )
    table.insert( list, ItemList.BuildInventoryItem( invslot ) );
end--ItemList.AddBagItemToList( list, bag, slot )

function ItemList.AddSpellToList( list, spellID )
    table.insert( list, ItemList.BuildSpell( spellID ) );
end--ItemList.AddSpellToList( list, spellID )

function ItemList.AddItemToList( list, itemID )
    table.insert( list, ItemList.BuildItem( itemID ) );
end--ItemList.AddSpellToList( list, spellID )

function ItemList.GetListEntryType( item )
    if( item.spellID ) then
        --spell
        return "spell";
    elseif( item.invslot ) then
        --inventory
        return "inventoryslot";
    elseif( item.itemID ) then
        --item
        return "item";
    elseif( item.bag and item.slot ) then
        --bag slot
        return "bageditem";
    end--if
    return nil;
end--ItemList.GetListEntryType( item )

function ItemList.GetNameFromListEntry( item )
    local sName = nil;
    
    if( item.spellID ) then
        --spell
        sName, _, _, _, _, _, _, _, _ = GetSpellInfo( item.spellID );
    elseif( item.invslot ) then
        --inventory
        sName, _, _, _, _, _, _, _ = GetItemInfo( GetInventoryItemLink( "player", item.invslot ) );
    elseif( item.itemID ) then
        --item
        sName, _, _, _, _, _, _, _ = GetItemInfo( item.itemID );
    elseif( item.bag and item.slot ) then
        --bag slot
        sName, _, _, _, _, _, _, _ = GetItemInfo( GetContainerItemLink( item.bag, item.slot ) );
    end--if
    return sName;
end--ItemList.GetNameFromListEntry( item )

function ItemList.GetLinkFromListEntry( item )
    local sName = "";
    
    if( item.spellID ) then
        --spell
        return GetSpellLink( item.spellID );
    elseif( item.invslot ) then
        --inventory
        return GetInventoryItemLink( "player", item.invslot );
    elseif( item.itemID ) then
        --item
        return LibHelper.GetItemLink( item.itemID );
    elseif( item.bag and item.slot ) then
        --bag slot
        return GetContainerItemLink( item.bag, item.slot );
    end--if
    return nil;
end--ItemList.GetNameFromListEntry( item )

function ItemList.GetRandomItemFromList( list, currentItem )
    if( type( list ) == "table" ) then
        local amountOfItems = #list;

        if( amountOfItems > 0 ) then
            --math.random( upper ) generates integer numbers between 1 and upper.
            local randomItemIndex = math.random( amountOfItems );

            if( amountOfItems > 1 ) then --Lets make sure the player gets a different item if he has more then one, the quick and dirty way.
                while( LibHelper.CompareTables( list[randomItemIndex], currentItem ) ) do
                    randomItemIndex = math.random( amountOfItems );
                end--while
            end--if
            return list[randomItemIndex];
        end--if
    end--if
    return nil;
end--ItemList.GetRandomItemFromList( list, currentItem )

function ItemList.SetTooltipFromListEntry( tooltip, item )
    if( item.spellID ) then
        --spell
        tooltip:SetHyperlink( ItemList.GetLinkFromListEntry( item ) );
    elseif( item.invslot ) then
        --inventory
        tooltip:SetInventoryItem( "player", item.invslot );
    elseif( item.itemID ) then
        --item
        tooltip:SetHyperlink( ItemList.GetLinkFromListEntry( item ) );
    elseif( item.bag and item.slot ) then
        --bag slot
        tooltip:SetBagItem( item.bag, item.slot );
    end--if
end--ItemList.SetTooltipFromListEntry( item )

function ItemList.SetButtonFromListEntry( button, item, setCooldown, cooldownInicator )
    local sName, itemLink, sItemTexture;
    local startTime, duration, enable;

    if( not cooldownInicator ) then
        cooldownInicator = button.cooldown;
    end--if
    if( item.spellID ) then
        --spell
        sName, _, sItemTexture, _, _, _, _, _, _ = GetSpellInfo( item.spellID );
        startTime, duration, enable = GetSpellCooldown( sName, BOOKTYPE_SPELL );

        button:SetAttribute( "*type1", "spell" );
        button:SetAttribute( "*spell1", sName );
    elseif( item.tooltip ) then
        --inventory
        itemLink = ItemList.GetLinkFromListEntry( item );
        sName, _, _, _, _, _, _, _, _, sItemTexture = GetItemInfo( itemLink );
        startTime, duration, enable = GetItemCooldown( itemLink );

        button:SetAttribute( "*type1", "item" );
        button:SetAttribute( "*bag1", nil );
        button:SetAttribute( "*slot1", item.invitem );
        button:SetAttribute( "*item1", nil );
    elseif( item.itemID ) then
        --item
        itemLink = ItemList.GetLinkFromListEntry( item );
        sName, _, _, _, _, _, _, _, _, sItemTexture = GetItemInfo( itemLink );
        startTime, duration, enable = GetItemCooldown( itemLink );

        button:SetAttribute( "*type1", "item" );
        button:SetAttribute( "*bag1", nil );
        button:SetAttribute( "*slot1", nil );
        button:SetAttribute( "*item1", itemLink );
    elseif( item.bag and item.slot ) then
        --bag slot
        itemLink = ItemList.GetLinkFromListEntry( item );
        sName, _, _, _, _, _, _, _, _, sItemTexture = GetItemInfo( itemLink );
        startTime, duration, enable = GetItemCooldown( itemLink );

        button:SetAttribute( "*type1", "item" );
        button:SetAttribute( "*bag1", item.bag );
        button:SetAttribute( "*slot1", item.slot );
        button:SetAttribute( "*item1", nil );
    end--if
    if( button.SetImage ) then
        button:SetImage( sItemTexture );
    else
        LibHelper.SetButtonImage( button, sItemTexture );
    end--if
    if( cooldownInicator and enable and setCooldown ) then
        cooldownInicator:Show();
        cooldownInicator:SetCooldown( startTime, duration );
    else
        cooldownInicator:Hide();
    end--if
end--ItemList.SetButtonFromListEntry( item, button )
