--[[
    Helper functions
    $Revision: 76985 $
]]--

--[[
Copyright (c) 2008, LordFarlander
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
]]--

local MAJOR_VERSION = "LibLordFarlander-1.0";
local MINOR_VERSION = "$Rev: 76985 $";

local LibHelper, oldminor = LibStub:NewLibrary( MAJOR_VERSION, MINOR_VERSION );
if( not LibHelper ) then
    return;
end

local FlyableZones = nil;
local TooltipScan = nil;

-- Notes:
-- Checks if an item is soulbound or quest
-- Arguments:
--     number - bag the item is in
--    number - slot the item is in
-- Returns:
-- * boolean - if the item given is soulbound
function LibHelper.ItemIsSoulbound( bag, slot )
    if( not TooltipScan ) then
        TooltipScan = CreateFrame( "GameTooltip", "LibHelperTooltipScan", nil, "GameTooltipTemplate" );
    end--if
    
    TooltipScan:SetOwner( WorldFrame, "ANCHOR_NONE" );
    TooltipScan:ClearLines();
    TooltipScan:SetBagItem( bag, slot );
    return( ( getglobal( TooltipScan:GetName() .. "TextLeft2" ):GetText() == ITEM_SOULBOUND ) or
        ( getglobal( TooltipScan:GetName() .. "TextLeft3" ):GetText() == ITEM_SOULBOUND ) or
        ( getglobal( TooltipScan:GetName() .. "TextLeft4" ):GetText() == ITEM_SOULBOUND ) or
        ( getglobal( TooltipScan:GetName() .. "TextLeft2" ):GetText() == ITEM_BIND_QUEST ) or
        ( getglobal( TooltipScan:GetName() .. "TextLeft3" ):GetText() == ITEM_BIND_QUEST ) or
        ( getglobal( TooltipScan:GetName() .. "TextLeft4" ):GetText() == ITEM_BIND_QUEST ) );
end

-- Notes:
-- Gets and item string from a link
-- Arguments:
--     string - a link, can be any type of link
-- Returns:
-- * string - the item string from the link
function LibHelper.GetItemStringFromLink( itemLink )
    if( itemLink ) then
        local _, _, itemString = itemLink:find( "^|c%x+|H(.+)|h%[.+%]" );
    
        return itemString;
    end--if
    return nil;
end--LibHelper.GetItemStringFromLink( itemLink )

-- Notes:
-- Determines if the player is in a zone where flying is allowed
-- Returns:
-- * boolean - whether or not the player is in a zone where flying is allowed
function LibHelper.IsInFlyableZone()
    if( not FlyableZones ) then
        FlyableZones = table.concat( { GetMapZones( 3 ) }, "|" );
    end--if
    return FlyableZones:find( GetRealZoneText() ) ~= nil;
end--LibHelper.IsInFlyableZone()

-- Notes:
-- Gets an ID from a string
-- Arguments:
--     string - a string
-- Returns:
-- * number - the ID from the string
function LibHelper.GetItemIDFromString( itemID )
    if( itemID ) then
        return tonumber( itemID:match( "(%d+)" ) );
    end--if
    return nil;
end--LibHelper.GetItemIDFromString( itemID )

-- Notes:
-- Gets an ID from a link
-- Arguments:
--     string - a link, can be any type
-- Returns:
-- * number - the ID from the link
function LibHelper.GetItemIDFromLink( itemLink )
    if( itemLink ) then
        return LibHelper.GetItemIDFromString( LibHelper.GetItemStringFromLink( itemLink ) );
    end--if
    return nil;
end--LibHelper.GetItemIDFromLink( itemLink )

-- Notes:
-- Gets an item string from an ID
-- Arguments:
--     number - an item ID
-- Returns:
-- * string - the string from the ID
function LibHelper.GetItemString( itemID )
    return LibHelper.GetItemStringFromLink( LibHelper.GetItemLink( itemID ) );
end--LibHelper.GetItemString( itemID )

-- Notes:
-- Gets an item link from and ID
-- Arguments:
--     number - an item ID
-- Returns:
-- * string - the link from the ID
function LibHelper.GetItemLink( itemID )
    local _, itemLink, _, _, _, _, _, _, _, _ = GetItemInfo( itemID );

    return itemLink;
end--LibHelper.GetItemLink( itemID )

-- Notes:
-- Checks to see if the player has a spell
-- Arguments:
--     number, string - A spell ID or spell link
-- Returns:
-- * boolean - whether or not the player has the spell
function LibHelper.HasPlayerSpell( spell )
    local spellName, _, _, _, _, _, _, _, _ = GetSpellInfo( spell );

    if( spellName ) then
        return GetSpellLink( spell ) == GetSpellLink( spellName );
    end
    return false;
end--LibHelper.PlayerHasSpell( spell )

-- Notes:
-- Deep copies a table to another table
-- Arguments:
--     table - Destination table (MUST already be a table)
--     table - Source table
function LibHelper.CopyTable( to, from )
    for i, v in pairs( from ) do
        if( type( v ) == "table" ) then
            to[i] = {};
            LibHelper.CopyTable( to[i], v );
        else
            to[i] = v;
        end--if
    end--for
end--LibHelper.CopyTable( to, from )

-- #nodef
function LibHelper.SimpleCompareTables( table1, table2 )
    if( table1 and table2 ) then
        for i, v in pairs( table1 ) do
            if( type( v ) == "table" ) then
                if( type( table2[i] ) == "table" ) then
                    if( not LibHelper.SimpleCompareTables( v, table2[i] ) ) then
                        return false;
                    end--if
                else
                    return false;
                end--if
            elseif( table2[i] ~= v ) then
                return false;
            end--if
        end--for
    else
        return false;
    end--if
    return true;
end--LibHelper.SimpleCompareTables( table1, table2 )

-- Notes:
-- Compares two tables to see if they have all the same members and the values are the same
-- Arguments:
--     table - First table to compare
--     table - Second table to compare
-- Returns:
-- * boolean - whether or not the tables are exactly alike
function LibHelper.CompareTables( table1, table2 )
    return LibHelper.SimpleCompareTables( table1, table2 ) and LibHelper.SimpleCompareTables( table2, table1 );
end--LibHelper.CompareTables( table1, table2 )

-- Notes:
-- Sets the image for a button.
-- To set the image of the button's icon, make sure the button has an icon member set to its icon.
-- Arguments:
--     button - The button
--     string - Path to the image to set the button to
function LibHelper.SetButtonImage( button, img )
    if( button.icon ) then
        button.icon:SetTexture( img );
    else
        button:SetNormalTexture( img );
        button:SetHighlightTexture( img );
    end--if
end--LibHelper.SetButtonImage( button, img )

-- Notes:
-- Debug message printing, suitable for including in your addon's class.
-- Set your addon's DEBUG member to true to print.
-- Arguments:
--     addon - The addon object
--     string - The message to print
function LibHelper.dprint( addon, msg )
    if( addon.DEBUG ) then
        DEFAULT_CHAT_FRAME:AddMessage( msg, 1.0, 0.0, 0.0 );
    end--if
end--LibHelper.dprint( addon, msg )

function LibHelper.UpdateFadeOut( button, fadingOptions )
    if( fadingOptions.fadeOut ) then
        local cancelFade = ( InCombatLockdown() and fadingOptions.fadeOutCancelInCombat ) or
            MouseIsOver( button ) or
            ( IsShiftKeyDown() and fadingOptions.fadeOutCancelOnShift ) or
            ( IsControlKeyDown() and fadingOptions.fadeOutCancelOnCtrl ) or
            ( IsAltKeyDown() and fadingOptions.fadeOutCancelOnAlt );

        if( cancelFade ) then
            button:SetAlpha( fadingOptions.alpha );
            button.faded = nil
            button.fadeOutDelay = fadingOptions.fadeOutDelay
        elseif( not button.faded ) then
            local startAlpha = fadingOptions.alpha;
            local fadeOutAlpha = fadingOptions.fadeOutAlpha or 0;
            local fadeOutChunks = ( fadingOptions.fadeOutTime or 10 ) / fadingOptions.fadeOutUpdateTime;
            local decrement = ( startAlpha - fadeOutAlpha ) / fadeOutChunks;
            local alpha = button:GetAlpha() - decrement;

            if( alpha < fadeOutAlpha ) then
                alpha = fadeOutAlpha;
            end
            if( alpha > fadeOutAlpha ) then
                button:SetAlpha( alpha );
            else
                button:SetAlpha( fadeOutAlpha );
                button.faded = true;
            end--if
        end--if
    end--if
end--LibHelper.UpdateFadeOut( button, fadingOptions )

function LibHelper.ButtonFadeOnUpdate( button, elapsed, fadingOptions )
    if( fadingOptions.fadeOut ) then
        button.elapsed = button.elapsed + elapsed;
        if( button.fadeOutDelay ) then
           if( button.elapsed < button.fadeOutDelay ) then
              return;
           else
               button.elapsed = button.elapsed - self.fadeOutDelay;
               button.fadeOutDelay = nil;
           end--if
        end--if
        if( button.elapsed > fadingOptions.fadeOutUpdateTime ) then
           LibHelper.UpdateFadeOut( button, fadingOptions )
           button.elapsed = 0
        end--if
    end--if
end--LibHelper.ButtonFadeOnUpdate( button, elapsed, fadingOptions )