-- vim: ts=2 sw=2 ai et
--
-- Copyright(c) 2008 Johny Mattsson
--      a.k.a
-- Anyia of HordeYakka (Jubei'Thos)
--
-- Permission to copy and re-use as seen fit.
-- Credit would be nice.
--

local NAME = "MisInformation";
MisInformation = LibStub("AceAddon-3.0"):NewAddon (NAME, "AceConsole-3.0", "AceEvent-3.0", "AceComm-3.0", "AceSerializer-3.0")
MisInformation.hunters = {}; -- Need to have this before we try to call ReloadHunterList

local options = {
  type = 'group',
  handler = MisInformation,
  childGroups = 'tab',
  args = {
    options = {
      type = 'group',
      name = "Options",
      desc = "General configuration settings for MisInformation",
      order = 1,
      args = {
        visible = {
          type = "toggle",
          order = 1,
          name = "Visible",
          desc = "Show the MisInformation window",
          get = "GetBool",
          set = "SetHide", -- Need to refresh visibility, so custom func
        },
        onlyInParty = {
          type = "toggle",
          name = "Only in party/raid",
          desc = "Show the MisInformation window only when in a party/raid",
          get = "GetBool",
          set = "SetOnlyInParty", -- Need to refresh visibility
        },
        flash = {
          type = "toggle",
          name = "Flashing mark",
          desc = "Enable flashing target mark when misdirect is active on player",
          get = "GetBool",
          set = "SetBool",
        },
        sync = {
          type = "toggle",
          name = "Enable sync",
          desc = "Send cooldown sync notifications to party/raid. Required for 'Readiness' use to show properly.",
          get = "GetComms", -- don't change this yet, need to transition to new value name first
          set = "SetBool",
        },
        showTargetInRange = {
          type = "toggle",
          name = "Target range check",
          desc = "Indicate whether the current target is in range of a Misdirection cast.",
          get = "GetBool",
          set = "SetBool",
          order = 250,
        },
        showFocusInRange = {
          type = "toggle",
          name = "Focus range check",
          desc = "Indicate whether the current focus target is in range of a Misdirection cast.",
          get = "GetBool",
          set = "SetBool",
          order = 252,
        },
      },
    },
    colors = {
      type = 'group',
      name = "Colors",
      desc = "Color configuration",
      order = 9,
      args = {
        defaultBgColor = {
          type = "color",
          name = "Default background color.",
          desc = "Default frame background color (in-range color, if range checking enabled).",
          get = "GetColor",
          set = "SetColor",
          order = 1,
        },
        focusOutOfRangeColor = {
          type = "color",
          hasAlpha = false,
          name = "Focus out-of-range color.",
          desc = "Frame background color for indicating the focus target is out of Misdirection range.",
          get = "GetColor",
          set = "SetColor",
          order = 2,
        },
        targetOutOfRangeColor = {
          type = "color",
          hasAlpha = false,
          name = "Target out-of-range color.",
          desc = "Frame background color for indicating the current target is out of Misdirection range. Has priority over the focus target out-of-range color.",
          get = "GetColor",
          set = "SetColor",
          order = 3,
        },
      },
    },
    reset = {
      type = 'group',
      name = "Resets",
      desc = "Various reset functions",
      order = 10,
      args = {
        hunterlist = {
          type = "execute",
          name = "Refresh hunter list",
          desc = "Reload the hunter list",
          func = "ReloadHunterList",
        },
        window = {
          type = "execute",
          name = "Reset window",
          desc = "Reset the position of the MisInformation window",
          func = "ResetWinPos",
        },
      },
    },
    about = {
      type = 'group',
      name = "About",
      desc = "About MisInformation",
      order = -1,
      args = {
        __ = {
          type = 'header',
          name = "MisInformation " .. GetAddOnMetadata (NAME, "Version"),
          order = 1,
        },
        _1 = {
          type = 'description',
          name = GetAddOnMetadata (NAME, "Notes"),
          order = 10,
        },
        _2 = {
          type = 'description',
          name = "By " .. GetAddOnMetadata(NAME, "Author"),
          order = 20,
        },
        _3 = {
          type = 'description',
          name = "",
          order = 30,
        },
        _4 = {
          type = 'description',
          name = "Copyright(C) 2008 Johny Mattsson",
          order = 40,
        },
        _5 = {
          type = 'description',
          name = "",
          order = 50,
        },
        _6 = {
          type = 'description',
          name = "Build: " .. GetAddOnMetadata (NAME, "X-Build"):match("%d+"),
          order = 60,
        },
      },
    }
  },
}

-- Config defaults
local defaults = {
  profile = {
    x = 10,
    y = -10,
    height = 80,
    width = 160,
    visible = true,
    onlyInParty = false,
    flash = true,
    sync = true,
    defaultBgColor = { r = 0, g = 0, b = 0, a = 1 },
    showTargetInRange = false,
    showFocusInRange = false,
    targetOutOfRangeColor = { r = 0.9, g = 0, b = 0, a = 1 },
    focusOutOfRangeColor = { r = 0.5, g = 0, b = 0, a = 1 },
  }
}


function MisInformation:MaintainLocation()
  local mybottom = self.frame:GetBottom();
  local myleft = self.frame:GetLeft();
  local parbottom = self.frame:GetParent ():GetBottom();
  local parleft = self.frame:GetParent():GetLeft();
  self.db.profile.x = myleft - parleft;
  self.db.profile.y = mybottom - parbottom;
  self.db.profile.height = self.frame:GetHeight();
  self.db.profile.width = self.frame:GetWidth();
end


function MisInformation:UpdateRowVisibility()
  self:MaintainLocation();

  local bottom = self.frame:GetBottom() + 4; -- add a bit of margin
  local status = self.statuses;
  for i = 1, 40 do
    local f = status[i];
    if (f:GetBottom() < bottom) or (f:GetTop() < bottom) then
       f:Hide ();
    else
       f:Show ();
    end
  end
end


function MisInformation:GetMdCount(name)
  local i = 1;
  buff, _, _, count = UnitBuff (name, i);
  while (buff) do
    -- Don't know if this is i18n safe?
    if (buff == "Misdirection") then
      return count;
    end
    i = i + 1;
    buff, _, _, count = UnitBuff (name, i);
  end
  return 0;
end


function MisInformation:UpdateStatuses()
  local status = self.statuses;
  local i = 1;
  for name, hunter in pairs (self.hunters) do
    local msg = string.format("%4d  %s", hunter["cooldown"], name);
    local text = status[i].text;
    if (UnitIsDeadOrGhost(name) or not UnitIsConnected(name)) then
      hunter["active"] = 0; -- make sure we flag them as inactive, including ourselves
      text:SetTextColor(0.5, 0.5, 0.5);
      msg = "   -  " .. name;
    elseif (hunter["active"] > 0) then
      text:SetTextColor(0.9, 0, 0);
      if (hunter["target"]) then
        msg = "  [" .. self:GetMdCount (name) .. "] " .. name .. " -> (" .. 
              (30 - (120 - hunter["cooldown"])) .. ") " .. hunter["target"];
      end
    elseif (hunter["cooldown"] > 0) then
      text:SetTextColor(0.5, 0.5, 0.5);
    else
      text:SetTextColor(0, 0.9, 0);
    end
    text:SetText(msg);
    i = i + 1;
  end
  -- clear the remaining slots
  for count = i, 40 do
    status[count].text:SetText("");
  end
end

-- Config getters/setters
function MisInformation:GetBool (info)
  return self.db.profile[info[#info]];
end

function MisInformation:SetBool (info, v)
  self.db.profile[info[#info]] = v;
end

function MisInformation:GetColor (info)
  local c = self.db.profile[info[#info]];
  return c.r, c.g, c.b, (c.a or 1);
end

function MisInformation:SetColor (info, r, g, b, a)
  local c = { r = r, g = g, b = b, a = a };
  self.db.profile[info[#info]] = c;
end

function MisInformation:SetHide(info, v)
  self.db.profile.visible = v;
  self:FrameVisibilityCheck();
end

function MisInformation:ResetWinPos()
  self.frame:ClearAllPoints();
  self.frame:SetPoint("CENTER");
end

function MisInformation:SetOnlyInParty(info, v)
  self.db.profile.onlyInParty = v;
  self:FrameVisibilityCheck();
end

function MisInformation:GetComms()
  -- renamed comms to sync, but support old name for now
  return self.db.profile.sync or self.db.profile.comms;
end

-- end config support


function MisInformation:SetupUI()
  if (self.frame) then
     self.frame:Hide();
  end

  local overlayframe = CreateFrame ("Frame", nil, WorldFrame);
  overlayframe:SetClampedToScreen (true);
  overlayframe:SetFrameStrata ("BACKGROUND");
  overlayframe:SetAllPoints (UIParent);
  overlayframe:Show ();
  self.overlayframe = overlayframe;

  local texture = overlayframe:CreateTexture ("MisInformationOverlay", "BACKGROUND");
  texture:SetAlpha (0);
  texture:SetBlendMode ("ADD");
  texture:SetVertexColor (0.1, 0.5, 1);
  texture:SetAllPoints (overlayframe);
  texture:SetTexture ("Interface\\AddOns\\MisInformation\\Textures\\overlay");
  self.overlaytexture = texture;

  local frame = CreateFrame ("Frame", nil, UIParent);
  frame:SetWidth(self.db.profile.width);
  frame:SetHeight(self.db.profile.height);
  frame:SetPoint ("BOTTOMLEFT", UIParent, "BOTTOMLEFT", self.db.profile.x, self.db.profile.y);
  frame:EnableMouse(true);
  frame:SetResizable(true);
  frame:SetMovable(true);
  frame:SetScript("OnMouseDown", function() self.frame:StartMoving(); end);
  frame:SetScript("OnMouseUp", function() self.frame:StopMovingOrSizing(); self:UpdateRowVisibility(); end);
  frame:SetBackdrop({
    bgFile="Interface\\Tooltips\\UI-Tooltip-Background", 
    edgeFile="Interface\\Tooltips\\UI-Tooltip-Border", 
    tile=1, tileSize=10, edgeSize=10, 
    insets={left=3, right=3, top=3, bottom=3}
  });
  local c = self.db.profile.defaultBgColor;
  frame:SetBackdropColor(c.r, c.g, c.b, c.a);
  self.frame = frame;

  local grip = CreateFrame("Button", nil, frame);
  grip:SetNormalTexture("Interface\\AddOns\\MisInformation\\Textures\\grip");
  grip:SetHighlightTexture("Interface\\AddOns\\MisInformation\\Textures\\grip");
  grip:SetWidth(16);
  grip:SetHeight(16);
  grip:SetScript("OnMouseDown", function() self.frame:StartSizing(); end);
  grip:SetScript("OnMouseUp", function() self.frame:StopMovingOrSizing(); self:UpdateRowVisibility(); end);
  grip:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", 0, 0);

  self.grip = grip;

  local entry, count, anchor;
  entry = CreateFrame("Frame", nil, frame);
  entry:SetPoint("LEFT", frame, "LEFT");
  entry:SetPoint("RIGHT", frame, "RIGHT", -4, 0);
  entry:SetPoint("TOP", frame, "TOP", 0, -4);
  entry:SetHeight(14);
  entry.text = entry:CreateFontString(nil, "ARTWORK", "GameFontNormal");
  entry.text:SetAllPoints();
  entry.text:SetJustifyH("LEFT");
  self.statuses = {};
  self.statuses[1] = entry;
  anchor = entry;
  for count = 2, 40 do
    entry = CreateFrame("Frame", nil, frame);
    entry:SetPoint("LEFT", frame, "LEFT");
    entry:SetPoint("RIGHT", frame, "RIGHT", -4, 0);
    entry:SetPoint("TOP", anchor, "BOTTOM", 0, 2);
    entry:SetHeight(14);
    entry.text = entry:CreateFontString(nil, "ARTWORK", "GameFontNormal");
    entry.text:SetAllPoints();
    entry.text:SetJustifyH("LEFT");
    self.statuses[count] = entry;
    anchor = entry;
  end

  self:UpdateRowVisibility();
  self:FrameVisibilityCheck();
end


function MisInformation:DefaultStatus(name)
  local status = {};
  status["cooldown"] = 0;
  status["active"] = 0;
  status["name"] = name;
  status["target"] = "";
  return status;
end


function MisInformation:ReloadHunterList()
  local hunterlist = {};
  local class, classE, name;

  for index = 1, 40 do
    name = UnitName("raid" .. index) or UnitName("party" .. index);
    if name then
      class, classE = UnitClass(name);
      if (class == "Hunter") or (classE == "HUNTER") then
         hunterlist[name] = self.hunters[name] or self:DefaultStatus(name);
      end
    end
  end

  class, classE = UnitClass("player");
  name = UnitName("player");
  if (class == "Hunter") or (classE == "HUNTER") then
    hunterlist[name] = self.hunters[name] or self:DefaultStatus(name);
  end

  self.hunters = hunterlist;
end


function MisInformation:SelfCheck()
  local me = self.hunters[self.player];
  if (me and me.active and me.active == 0) then
    self:StopFlash ();
  end
end


function MisInformation:CoolDown()
  for name, hunter in pairs (self.hunters) do
    local cd = hunter["cooldown"];
    if (cd > 0) then
      hunter["cooldown"] = cd - 1;
      if (cd == 1) then
        hunter["active"] = 0;
      end
    else -- cd <= 0
      hunter["active"] = 0;
    end
  end
  self:SelfCheck(); -- switch off flashing if needed
end


function MisInformation:RangeCheck()
  local c = self.db.profile.defaultBgColor;
  local update = false; -- Don't burden the GUI with color changes unless range checking is enabled
  if (self.db.profile.showFocusInRange) then
    update = true;
    local inRange = IsSpellInRange ("Misdirection", "focus"); --IsSpellInRange (34477, "spell", "focus");
    if (inRange ~= nil and inRange ~= 1) then -- we have a valid focus target, but not in range
      c = self.db.profile.focusOutOfRangeColor;
    end
  end
  if (self.db.profile.showTargetInRange) then
    update = true;
    local inRange = IsSpellInRange ("Misdirection", "target");--IsSpellInRange (34477, "spell", "target");
    if (inRange ~= nil and inRange ~= 1) then -- valid MD target, but not in range.
      c = self.db.profile.targetOutOfRangeColor;
    end
  end
  if (update) then
    self.frame:SetBackdropColor (c.r, c.g, c.b, c.a);
  end
end


function MisInformation:FrameVisibilityCheck()
  if (self.db.profile.onlyInParty and
      (GetNumPartyMembers() == 0) and
      (GetNumRaidMembers () == 0))
      or
      not self.db.profile.visible
  then
    self.frame:Hide();
  else
    self.frame:Show();
  end
end


function MisInformation:SendCooldownMessage (cooldown)
  if (self.db.profile.comms) then
    self:SendCommMessage (NAME, self:Serialize ("t0", cooldown), "RAID", nil, "NORMAL");
  end
end


function MisInformation:RAID_ROSTER_UPDATE()
  self:ReloadHunterList();
  self:FrameVisibilityCheck();
  local me = self.hunters[self.player];
  if (me) then
    self:SendCooldownMessage (me.cooldown);
  end
end

function MisInformation:PARTY_MEMBERS_CHANGED()
  self:ReloadHunterList();
  self:FrameVisibilityCheck();
  local me = self.hunters[self.player];
  if (me) then
    self:SendCooldownMessage (me.cooldown);
  end
end


function MisInformation:DoFlash()
  if (self.db.profile.flash) then
    self.overlayframe:Show ();
    UIFrameFlash (self.overlaytexture, 0, 1.2, 30, false, 0, 0);
  end
end

function MisInformation:StopFlash()
  UIFrameFlashRemoveFrame (self.overlaytexture);
  self.overlaytexture:SetAlpha (0);
  self.overlayframe:Hide ();
end


function MisInformation:COMBAT_LOG_EVENT_UNFILTERED(eventName, time, event, srcID, srcName, srcFlags, dstID, dstName, dstFlags, spellID, spellName)
-- 34477 misdirect on hunter
-- 35079 misdirect gained by target
  if (event == "SPELL_AURA_APPLIED") and (spellID == 34477) then
    local hunter = self.hunters[dstName];
    if (hunter == nil) then return; end
    hunter["cooldown"] = 120;
    hunter["active"] = 1;
    if (dstName == self.player) then
      self:DoFlash ();
    end
    self:UpdateStatuses();
  elseif (event == "SPELL_AURA_REMOVED") and (spellID == 34477) then
    local hunter = self.hunters[dstName];
    if (hunter == nil) then return; end
    hunter["active"] = 0;
    self:SelfCheck ();
    self:UpdateStatuses();
  elseif (event == "SPELL_CAST_SUCCESS") and (spellID == 34477) then
    local hunter = self.hunters[srcName];
    if (hunter == nil) then return; end
    hunter["target"] = dstName;
    self:UpdateStatuses();
  end
end


function MisInformation:CheckTimer()
  local last_sec = self.timer_checked_at or 0;
  local sec_now = floor(GetTime());
  if (sec_now > last_sec) then
    local start, cooldown, enabled = GetSpellCooldown ("Misdirection");
    local me = self.hunters[self.player];
    if ((start == 0) and (me) and (me.cooldown > 0)) then
      -- Either we have miscalculated the cooldown, or Readiness has been used.
      -- Regardless, we're off cooldown and should update accordingly
      me.active = 0;
      me.cooldown = 0;
      self:SendCooldownMessage (me.cooldown);
    end

    self:CoolDown();
    self:UpdateStatuses();
    self:RangeCheck ();
  end
  self.timer_checked_at = sec_now; -- keep only the integer (second) part
end


MisInformation.OnCommReceive = { }; -- We use a table of handlers
-- t0: sender has specified a new cooldown value for him/herself in 'cooldown'
function MisInformation.OnCommReceive:t0 (sender, distribution, cooldown)
  local hunter = self.hunters[sender];
  if (hunter) then
    hunter["cooldown"] = cooldown;
    self:UpdateStatuses ();
  end
end


function MisInformation:OnCommDeMux (prefix, message, distribution, sender)
  if ((prefix ~= NAME) or (sender == self.player)) then return end;
  -- currently only using 'a', but might as well future proof a bit
  local success, msgtype, a, b, c, d = self:Deserialize (message);
  if not success then return end;
  if (self.OnCommReceive[msgtype]) then
    self.OnCommReceive[msgtype] (self, sender, distribution, a, b, c, d);
  --else
  --  self:Print ("debug: Received unknown MisInformation message type: " .. msgtype);
  end
end


function MisInformation:OnProfileChanged ()
  self:FrameVisibilityCheck();
end


function MisInformation:OnInitialize ()
  LibStub ("AceConfigRegistry-3.0"):RegisterOptionsTable (NAME, options);
  local ACD = LibStub ("AceConfigDialog-3.0");
  ACD:AddToBlizOptions (NAME, GetAddOnMetadata(NAME, "Title"));
  self:RegisterChatCommand ("misinformation", function() InterfaceOptionsFrame_OpenToFrame(ACD.BlizOptions[NAME].frame) end);
  LibStub("AceConfigCmd-3.0"):CreateChatCommand ("misi", NAME);

  self.db = LibStub ("AceDB-3.0"):New ("MisInformation2DB", defaults);
  options.args.profiles = LibStub("AceDBOptions-3.0"):GetOptionsTable (self.db);
  self.db.RegisterCallback(self, "OnProfileChanged", "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileCopied", "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileReset", "OnProfileChanged") 
end


function MisInformation:OnEnable()
  self.player = UnitName ("player");
  self:ReloadHunterList();
  self:SetupUI();

  self.frame:SetScript("OnUpdate", function() self:CheckTimer(); end);

  self:RegisterEvent("RAID_ROSTER_UPDATE");
  self:RegisterEvent("PARTY_MEMBERS_CHANGED");
  self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED");
  self:RegisterComm (NAME, "OnCommDeMux");
end

