--[[
 MiniPet Addon
 Version 4.5.118
 $Revision: 110 $

 Currently hosted by Breanni of Scarlet Crusade
 All credit for authoring this addon is listed below

 Visit WarcraftPets.com for all your minipet needs!
]]--

--[[
    MiniPet by Carra of Stormrage

    There are a number of events where the player looses his pet and we can notify him:
    -Entering the game                   >OK
    -Entering an instance                >OK
    -Dieing and getting back alive       >Usually OK :p
    -Using a Taxi Service                >OK
    -Teleporting/Summoning/Hearthstoning >OK but no summoning

    Based originally on MiniPetLeash (completely rewritten except for the events which are copied).

    Updated and converted to Ace3 by LordFarlander
]]--

local L = LibStub( "AceLocale-3.0" ):GetLocale( "MiniPet" );

_G["BINDING_NAME_CLICK MiniPet_Button:LeftButton"] = L["Summon Companion"];
BINDING_HEADER_MINIPETHEADER = L["MiniPet"];

if( not LibStub( "LibSpecialEvents-Mount-4.0", true ) ) then
    error( L["%s requires %s"]:format( L["MiniPet"], "LibSpecialEvents-Mount-4.0" ) );
    return;
end--if

if( not LibStub( "LibSpecialEvents-LearnSpell-4.0", true ) ) then
    error( L["%s requires %s"]:format( L["MiniPet"], "LibSpecialEvents-LearnSpell-4.0" ) );
    return;
end--if

local LibLordFarlander = LibStub( "LibLordFarlander-2.0", true );
if( not LibLordFarlander ) then
    error( L["%s requires %s"]:format( L["MiniPet"], "LibLordFarlander-2.0" ) );
    return;
end--if

local ItemList = LibStub( "LibLordFarlander-ItemList-2.0", true );
if( not ItemList ) then
    error( L["%s requires %s"]:format( L["MiniPet"], "LibLordFarlander-ItemList-2.0" ) );
    return;
end--if

if( not LibStub( "LibLordFarlander-UI-2.0", true ) ) then
    error( L["%s requires %s"]:format( L["MiniPet"], "LibLordFarlander-UI-2.0" ) );
    return;
end--if

local PetAndMountDatabase = LibStub( "LibPetAndMountDatabase-1.1", true );
if( not PetAndMountDatabase ) then
    error( L["%s requires %s"]:format( L["MiniPet"], "LibPetAndMountDatabase-1.1" ) );
    return;
end--if

MiniPet = LibStub( "AceAddon-3.0" ):NewAddon( "MiniPet", "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0" );
MiniPet.version = { major = 4, minor = 5, revision = tonumber( "118" ) };

if( LibStub( "LibDataBroker-1.1", true ) ) then
    MiniPet.ldbobj = LibStub( "LibDataBroker-1.1" ):NewDataObject( "MiniPet", {
            type = "data source",
            icon = "Interface\\AddOns\\MiniPet\\Img\\MiniPet",
            label = L["MiniPet"],
            text = L["Get a companion for some company!"],
            onClick = function( self, btn )
                          if( LibStub( "LibLordFarlander-UI-2.0" ).IsModifierDown( self.class.db.profile.CritterItemModifier ) and self.class.CurrentCompanionItem ) then
                              MiniPet:FindNextCompanionEnchant();
                          else
                              MiniPet:ChooseRandomCompanion();
                          end--if
                      end,
            OnTooltipShow = function( tooltip )
                                if( not tooltip ) then
                                    return;
                                end--if
                                MiniPet:SetTooltip( tooltip );
                            end,
        } );
end--if

--Global Values
MiniPet.DEBUG = false;            --true = Debugging Mode
MiniPet.dprint = LibLordFarlander.dprint;
MiniPet.button = nil;

MiniPet.UpdateInterval      = 0.2;   --Time between two flashes of the button in seconds
MiniPet.TotalFlashTime      = 10;    --Total time to flash the button in seconds

--DO NOT CHANGE THESE VALUES
MiniPet.PlayersCompanions    = {};
MiniPet.CurrentCompanion     = nil;   --Pet on the Button
MiniPet.BoardedTaxi          = false;
MiniPet.StartedHearthstone   = false;
MiniPet.HearthstoneComplete  = false;
MiniPet.PlayerDied           = false;
MiniPet.PlayerEnteringWorld  = true;
MiniPet.ReadyToInitialize    = false;
MiniPet.CompanionListOkay    = false;
MiniPet.LastCompanion        = nil;
MiniPet.QuestCompanion       = nil;
MiniPet.QuestCompanionQuest  = nil;
MiniPet.CurrentCompanionItem = nil;
MiniPet.LastZoneList         = {};

MiniPet.ShowButtonImg       = false; --Goes on and off if button flashes
MiniPet.FlashTime           = 0;     --Time since button flashes
MiniPet.FlashButton         = false; --On when the button flashes

MiniPet.Dirty               = false;
MiniPet.HasCompanions       = false;

MiniPet.keyBoundMode        = false;

MiniPet.ui = {};

MiniPet.dirtyBags = { [0] = true, [1] = true, [2] = true, [3] = true, [4] = true, [-254] = true, [-255] = true, };

MiniPet.CritterItemNames    = {
    [35223] = L["Papa Hummel's Old-Fashioned Pet Biscuit"],
    [37460] = L["Rope Pet Leash"],
    [37431] = L["Fetch Ball"],
    [43352] = L["Pet Grooming Kit"],
    [43626] = L["Happy Pet Snack"],
};

function MiniPet:OnInitialize()
    local AceConfigReg = LibStub( "AceConfigRegistry-3.0" );
    local AceConfigDialog = LibStub( "AceConfigDialog-3.0" );
    local defaults = {
            profile = {
                BattleGroundCheck = true,
                GiveTextWarning = true,
                GiveButtonWarning = true,
                GiveMountWarning = true,
                SuggestOnlySoulbound = true,
                ReagentCheck = true,
                EquipmentCheck = true,
                ForceQuestPet = true,
                CritterItemModifier = "Shift",
                -- List of favorite pets, these will be picked more often
                -- stored as [itemID]=percentage
                Favorites = {},
                -- List of pets per zone, if the pets are in the player's bags they will be picked
                -- when in the indicated zone
                -- stored as ["Localized zone name"]=itemID
                ZoneFavorites = {},
                Button = LibStub( "LibLordFarlander-UI-2.0" ).GetButtonSettingsDefaults(),
            },
        };
    local buttonOptions = LibStub( "LibLordFarlander-UI-2.0" ).GetButtonOptionGroup( self, {
            name = L["Button"],
            desc = L["Options relating to the button."],
            yes = L["yes"],
            no = L["no"],
            show = L["showing"],
            hide = L["hiding"],
            locked = L["locked"],
            unlocked = L["unlocked"],
            toggleButton = L["toggleButton"],
            toggleButton_name = L["Show button"],
            toggleButton_desc = L["Show/hide the button."],
            autoHideButton = L["autoHideButton"],
            autoHideButton_name = L["Automatically hide button"],
            autoHideButton_desc = L["Hide the button when you click it?"],
            lockButton = L["lockButton"],
            lockButton_name = L["Lock button position"],
            lockButton_desc = L["Lock/unlock the position of the button."],
            alpha = L["alpha"],
            alpha_name = L["Alpha of button"],
            alpha_desc = L["Sets the alpha value that the button has normally."],
            fadeOut = L["fadeOut"],
            fadeOut_name = L["Fade out"],
            fadeOut_desc = L["Fade the button out when the cursor is not over it?"],
            fadeOutInsteadHide = L["fadeOutInsteadHide"],
            fadeOutInsteadHide_name = L["Fade out instead of hiding"],
            fadeOutInsteadHide_desc = L["Fade the button out instead of hiding it?"],
            fadeOutCancelInCombat = L["fadeOutCancelInCombat"],
            fadeOutCancelInCombat_name = L["Cancel fade out if in combat"],
            fadeOutCancelInCombat_desc = L["Should the button not fade out if you're in combat?"],
            fadeOutCancelOnShift = L["fadeOutCancelOnShift"],
            fadeOutCancelOnShift_name = L["Cancel fade out if holding Shift"],
            fadeOutCancelOnShift_desc = L["Should the button not fade out if you're holding Shift?"],
            fadeOutCancelOnCtrl = L["fadeOutCancelOnCtrl"],
            fadeOutCancelOnCtrl_name = L["Cancel fade out if holding Ctrl"],
            fadeOutCancelOnCtrl_desc = L["Should the button not fade out if you're holding Ctrl?"],
            fadeOutCancelOnAlt = L["fadeOutCancelOnAlt"],
            fadeOutCancelOnAlt_name = L["Cancel fade out if holding Alt"],
            fadeOutCancelOnAlt_desc = L["Should the button not fade out if you're holding Alt?"],
            fadeOutTime = L["fadeOutTime"],
            fadeOutTime_name = L["Fade out time"],
            fadeOutTime_desc = L["Sets how long the button takes to fade out."],
            fadeOutDelay = L["fadeOutDelay"],
            fadeOutDelay_name = L["Fade out delay"],
            fadeOutDelay_desc = L["Sets the delay before the button starts to fade."],
            fadeOutAlpha = L["fadeOutAlpha"];
            fadeOutAlpha_name = L["Alpha after fadeout"],
            fadeOutAlpha_desc = L["Sets the alpha value that the button has after faded out."],
        }, 1 );
    local modifierItemsOptions = {
        type = "group",
        name = L["Modifier Items"],
        desc = L["Allows you to set the order that modifier items are picked."],
        handler = self,
        args = {
            favorites = {
                type = "select",
                order = 1,
                name = L["Modifier Items"],
                desc = L["Allows you to set the order that modifier items are picked."],
                width = "full",
                cmdHidden = true,
                values = function( info )
                             local favorites = {};

                             for _, item in pairs( info.handler.db.profile.enchantOrder ) do
                                 local sName, _, _, _, _, _, _, _, _, sTexture = GetItemInfo( item );
                                 
                                 if( sTexture ) then
                                     tinsert( favorites, ("|T%s:0|t %s"):format( sTexture, sName ) );
                                 else
                                     tinsert( favorites, info.handler.CritterItemNames[item] or item );
                                 end--if
                             end--if
                             if( not info.handler.ui.currentModifierItem ) then
                                 info.handler.ui.currentModifierItem = 1;
                             end--if
                             return favorites;
                         end,
                get = function( info )
                          return info.handler.ui.currentModifierItem;
                      end,
                set = function( info, val )
                          info.handler.ui.currentModifierItem = val;
                      end,
            },
            up = {
                type = "execute",
                order = 2,
                name = L["Move up"],
                desc = L["Moves the selected modifier up in the list."],
                cmdHidden = true,
                func = function( info )
                           if( info.handler.ui.currentModifierItem > 1 ) then
                               local olditem = info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem - 1];
                               
                               info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem - 1] = info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem];
                               info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem] = olditem;
                               info.handler.ui.currentModifierItem = info.handler.ui.currentModifierItem - 1;
                               info.handler:FindNextCompanionEnchant();
                           end--if
                       end,
            },
            down = {
                type = "execute",
                order = 2,
                name = L["Move down"],
                desc = L["Moves the selected modifier down in the list."],
                cmdHidden = true,
                func = function( info )
                           if( info.handler.ui.currentModifierItem < #info.handler.db.profile.enchantOrder ) then
                               local olditem = info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem + 1];

                               info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem + 1] = info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem];
                               info.handler.db.profile.enchantOrder[info.handler.ui.currentModifierItem] = olditem;
                               info.handler.ui.currentModifierItem = info.handler.ui.currentModifierItem + 1;
                               info.handler:FindNextCompanionEnchant();
                           end--if
                       end,
            },
        },
    };
    local favoritesOptions = {
        type = "group",
        name = L["Favorites"],
        desc = L["Companions that should be picked more often if you have them."],
        handler = self,
        args = {
            favorite = {
                type = "select",
                order = 1,
                name = L["Favorites"],
                desc = L["Companions that should be picked more often if you have them."],
                width = "full",
                cmdHidden = true,
                values = function( info )
                             info.handler.ui.Favorites = {};

                             local Favorites = info.handler.ui.Favorites;
                             local firstItem = 0;
                             local used = {};

                             for _, pet in pairs( info.handler:GetAllCompanionsList() ) do
                                 local id = pet:GetID();

                                 used[id] = true;
                                 table.insert( Favorites, id );
                                 if( type( info.handler.db.profile.Favorites[id] ) == "boolean" ) then
                                     info.handler.db.profile.Favorites[id] = 2.0;
                                 end--if
                                 if( ( type( info.handler.db.profile.Favorites[id] ) ~= "nil" ) and ( info.handler.db.profile.Favorites[id] == 1.0 ) ) then
                                     info.handler.db.profile.Favorites[id] = nil;
                                 end--if
                             end--for
                             for pet, howOften in pairs( info.handler.db.profile.Favorites ) do
                                 if( not used[pet] ) then
                                     table.insert( Favorites, pet );
                                 end--if
                             end--if
                             table.sort( Favorites, function( a, b ) return ( ( a < 0 ) and GetSpellInfo( a * -1 ) or GetItemInfo( a ) ) < ( ( b < 0 ) and GetSpellInfo( b * -1 ) or GetItemInfo( b ) ) end );
                             Favorites = {};
                             for index, pet in pairs( info.handler.ui.Favorites ) do
                                 local sName, sTexture;
                                 local sHowOften = L["default"];
                                 local howOften = info.handler.db.profile.Favorites[pet] or 1.0;

                                 if( pet < 0 ) then
                                     sName, _, sTexture, _, _, _, _, _, _ = GetSpellInfo( pet * -1 );
                                 else
                                     sName, _, _, _, _, _, _, _, _, sTexture = GetItemInfo( pet );
                                 end--if
                                 if( howOften == 0.0 ) then
                                     sHowOften = L["never"];
                                 elseif( howOften ~= 1.0 ) then
                                     sHowOften = L["%.1f times as often"]:format( howOften );
                                 end--if                                 
                                 if( sTexture ) then
                                     Favorites[index] = ("|T%s:0|t %s |cFFFF0000[%s]|r"):format( sTexture, sName, sHowOften );
                                 end--if
                                 if( firstItem == 0 ) then
                                     firstItem = index;
                                 end--if
                             end--for
                             if( ( not Favorites[info.handler.ui.currentFavorite] ) or ( info.handler.ui.currentFavorite == nil ) ) then
                                 info.handler.ui.currentFavorite = firstItem;
                             end--if
                             return Favorites;
                         end,
                get = function( info )
                          return info.handler.ui.currentFavorite;
                      end,
                set = function( info, val )
                          info.handler.ui.currentFavorite = val;
                      end,
            },
            frequency = {
                type = "range",
                order = 2,
                name = L["Frequency"],
                cmdHidden = true,
                min = 0.0,
                max = 5.0,
                bigStep = 0.1,
                isPercent = true,
                desc = L["Set how often the companion gets picked.  2 is twice as often and 0 is never."];
                get = function( info )
                          return info.handler.db.profile.Favorites[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] or 1.0;
                      end,
                set = function( info, val )
                          if( val == 1.0 ) then
                              info.handler.db.profile.Favorites[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = nil;
                          else
                              info.handler.db.profile.Favorites[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = val;
                          end--if
                      end,
            },
            [L["list"]] = {
                type = "execute",
                order = 4,
                guiHidden = true,
                name = L["List all favorite companions"],
                desc = L["Show a list with all of your favorite companions."],
                func = function( info )
                           info.handler:Print( L["List of your favorite companions:"] );
                           for pet, howOften in pairs( info.handler.db.profile.Favorites ) do
                               if( howOften ~= 1.0 ) then
                                   local link = ( pet > 0 ) and LibLordFarlander.GetItemLink( pet ) or GetSpellLink( pet * -1 );

                                   if( pet > 0 ) then
                                       sTexture = select( 10, GetItemInfo( pet ) );
                                   else
                                       sTexture = select( 3, GetSpellInfo( pet * -1 ) );
                                   end--if
                                   if( howOften == 0.0 ) then
                                       howOften = L["never"];
                                   else
                                       howOften = L["%.1f times as often"]:format( howOften );
                                   end--if

                                   if( sTexture ) then
                                       info.handler:Print( ("|T%s:0|t %s |cFFFF0000[%s]|r"):format( sTexture, link or tostring( pet ), howOften ) );
                                   else
                                       info.handler:Print( ("%s |cFFFF0000[%s]|r"):format( link or tostring( pet ), howOften ) );
                                   end--if
                               end--if
                           end--for
                       end,
            },
        },
    };
    local zoneTemplate = {
        header = {
            type = "header",
            order = 1,
            width = "full",
            cmdHidden = true,
            name = function( info )
                       return LibLordFarlander.UnnamifyString( info[#info - 1] );
                   end,
        },
        minipet = {
            type = "select",
            order = 2,
            name = L["Companion"],
            desc = L["Companions that should be picked more often if you have them in the selected zone."],
            width = "full",
            cmdHidden = true,
            values = function( info )
                         info.handler.ui.Favorites = {};

                         local Favorites = info.handler.ui.Favorites;
                         local used = {};
                         local ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];

                         for _, pet in pairs( info.handler:GetAllCompanionsList() ) do
                             local id = pet:GetID();

                             used[id] = true;
                             table.insert( Favorites, id );
                             if( type( info.handler.db.profile.Favorites[id] ) == "boolean" ) then
                                 info.handler.db.profile.Favorites[id] = 2.0;
                             end--if
                             if( ( type( info.handler.db.profile.Favorites[id] ) ~= "nil" ) and ( info.handler.db.profile.Favorites[id] == 1.0 ) ) then
                                 info.handler.db.profile.Favorites[id] = nil;
                             end--if                             
                         end--for
                         if( ZoneFavorite and ( type( ZoneFavorite ) ~= "table" ) ) then
                             info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )] = { [ZoneFavorite] = { frequency = 1.0, exclusive = true } };
                             ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                         end--if
                         if( not ZoneFavorite ) then
                             info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )] = {};
                             ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                         end--if                         
                         if( ZoneFavorite ) then
                             for pet, _ in pairs( ZoneFavorite ) do
                                 if( not used[pet] ) then
                                     table.insert( Favorites, pet );
                                     used[pet] = true;
                                 end--if
                             end--for
                         end--if
                         table.sort( Favorites, function( a, b ) return ( ( a < 0 ) and GetSpellInfo( a * -1 ) or GetItemInfo( a ) ) < ( ( b < 0 ) and GetSpellInfo( b * -1 ) or GetItemInfo( b ) ) end );
                         Favorites = {};
                         for index, pet in pairs( info.handler.ui.Favorites ) do
                             local sName, sTexture;
                             local howOften = 1.0;
                             local sHowOften = L["default"]
                             local exclusive = false;

                             if( ZoneFavorite[pet] ) then
                                 howOften = ZoneFavorite[pet].frequency or 1.0;
                                 exclusive = ZoneFavorite[pet].exclusive;
                             end--if
                             if( pet < 0 ) then
                                 sName, _, sTexture, _, _, _, _, _, _ = GetSpellInfo( pet * -1 );
                             else
                                 sName, _, _, _, _, _, _, _, _, sTexture = GetItemInfo( pet );
                             end--if
                             if( sTexture ) then
                                 Favorites[index] = ("|T%s:0|t %s"):format( sTexture, sName );
                             end--if
                             if( howOften == 0.0 ) then
                                 sHowOften = L["never"];
                             elseif( howOften ~= 1.0 ) then
                                 sHowOften = L["%.1f times as often"]:format( howOften );
                             end--if
                             if( exclusive ) then
                                 sHowOften = string.format( "%s, %s", sHowOften, L["exclusive"] );
                             end--if
                             if( sTexture ) then
                                 Favorites[index] = ("|T%s:0|t %s |cFFFF0000[%s]|r"):format( sTexture, sName, sHowOften );
                             end--if
                         end--for
                         if( #Favorites == 0 ) then
                             info.handler.ui.currentFavorite = nil;
                             Favorites[0] = L["none"];
                         elseif( ( info.handler.ui.currentFavorite == nil ) or ( info.handler.ui.currentFavorite > #Favorites ) ) then
                             info.handler.ui.currentFavorite = 1;
                         end--if
                         return Favorites;
                     end,
            get = function( info )
                      return info.handler.ui.currentFavorite;
                  end,
            set = function( info, val )
                      info.handler.ui.currentFavorite = val;
                  end,
            },
            frequency = {
                type = "range",
                order = 3,
                name = L["Frequency"],
                cmdHidden = true,
                min = 0.0,
                max = 5.0,
                bigStep = 0.1,
                isPercent = true,
                desc = L["Set how often the companion gets picked.  2 is twice as often and 0 is never."];
                get = function( info )
                          local ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];

                          if( ZoneFavorite and ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] ) then
                              return ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]].frequency or 1.0;
                          end--if
                          return 1.0;
                      end,
                set = function( info, val )
                          local ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          local CurrentFavorite;

                          if( val == 1.0 ) then
                              val = nil;
                          end--if
                          if( not ZoneFavorite ) then
                              info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )] = {};
                              ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          end--if
                          CurrentFavorite = ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]];
                          if( not CurrentFavorite ) then
                              ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = { frequency = val };
                              CurrentFavorite = ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]];
                          else
                              CurrentFavorite.frequency = val;
                          end--if
                          if( not ( CurrentFavorite.frequency or CurrentFavorite.active or CurrentFavorite.exclusive ) ) then
                              ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = nil;
                          end--if
                          if( info.handler:HasCompanionListChanged() ) then
                              info.handler:ChooseRandomCompanion();
                          end--if
                      end,
            },
            active = {
                type = "toggle",
                order = 4,
                name = L["Override favorites"],
                desc = L["Check this to override the favorites settings for this companion."];
                get = function( info )
                          local ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          
                          if( ZoneFavorite and ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] ) then
                              return ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]].active or false;
                          end--if
                          return false;
                      end,
                set = function( info, val )
                          local ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          local CurrentFavorite;

                          if( not ZoneFavorite ) then
                              info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )] = {};
                              ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          end--if
                          CurrentFavorite = ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]];
                          if( not CurrentFavorite ) then
                              ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = { active = val };
                              CurrentFavorite = ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]];
                          else
                              CurrentFavorite.active = val;
                          end--if
                          if( not ( CurrentFavorite.frequency or CurrentFavorite.active or CurrentFavorite.exclusive ) ) then
                              ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = nil;
                          end--if
                          if( info.handler:HasCompanionListChanged() ) then
                              info.handler:ChooseRandomCompanion();
                          end--if
                      end,
            },
            exclusive = {
                type = "toggle",
                order = 5,
                name = L["Exclusive"],
                desc = L["Companions marked Exclusive will be the only ones picked from."];
                get = function( info )
                          local ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          
                          if( ZoneFavorite and ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] ) then
                              return ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]].exclusive or false;
                          end--if
                          return false;
                      end,
                set = function( info, val )
                          local ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          local CurrentFavorite;

                          if( not ZoneFavorite ) then
                              info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )] = {};
                              ZoneFavorite = info.handler.db.profile.ZoneFavorites[LibLordFarlander.UnnamifyString( info[#info - 1] )];
                          end--if
                          CurrentFavorite = ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]];
                          if( not CurrentFavorite ) then
                              ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = { exclusive = val };
                              CurrentFavorite = ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]];
                          else
                              CurrentFavorite.exclusive = val;
                          end--if
                          if( not ( CurrentFavorite.frequency or CurrentFavorite.active or CurrentFavorite.exclusive ) ) then
                              ZoneFavorite[info.handler.ui.Favorites[info.handler.ui.currentFavorite]] = nil;
                          end--if
                          if( info.handler:HasCompanionListChanged() ) then
                              info.handler:ChooseRandomCompanion();
                          end--if                          
                      end,
            },
    };
    local continentTemplate = {
    };
    local zoneOptions = {
        type = "group",
        name = L["Zone Favorites"],
        desc = L["Set a companion to use when in a certain zone."],
        handler = self,
        childGroups = "tree",
        args = {
            [L["list"]] = {
                type = "execute",
                order = -1,
                guiHidden = true,
                name = L["List all favorite companions"],
                desc = L["Show a list with all of your favorite companions."],
                func = function( info )
                           info.handler:Print( L["List of your favorite companions:"] );
                           for zone, pets in pairs( info.handler.db.profile.ZoneFavorites ) do
                               info.handler:Print( ("%s:"):format( zone ) );
                               for pet, settings in pairs( pets ) do
                                   local howOften = settings.frequency or 1.0;
                                   local sHowOften =  "default";
                                   local exclusive = settings.exclusive;

                                   if( ( howOften ~= 1.0 ) or exclusive ) then
                                       local link = ( pet > 0 ) and LibLordFarlander.GetItemLink( pet ) or GetSpellLink( pet * -1 );
    
                                       if( pet > 0 ) then
                                           sTexture = select( 10, GetItemInfo( pet ) );
                                       else
                                           sTexture = select( 3, GetSpellInfo( pet * -1 ) );
                                       end--if
                                       if( howOften == 0.0 ) then
                                           sHowOften = L["never"];
                                       elseif( howOften ~= 1.0 ) then
                                           sHowOften = L["%.1f times as often"]:format( howOften );
                                       end--if
                                       if( exclusive ) then
                                           sHowOften = ("%s, %s"):format( sHowOften, L["exclusive"] );
                                       end--if
                                       if( sTexture ) then
                                           info.handler:Print( ("|T%s:0|t %s |cFFFF0000[%s]|r"):format( sTexture, link or tostring( pet ), sHowOften ) );
                                       else
                                           info.handler:Print( ("%s |cFFFF0000[%s]|r"):format( link or tostring( pet ), sHowOften ) );
                                       end--if
                                   end--if
                               end--if
                           end--if
                       end,
            },
        },
    };
    local modifierKeys = { ["Disabled"] = L["Disabled"], ["Alt"] = L["Alt"], ["Control"] = L["Control"], ["Shift"] = L["Shift"], };
    local options = {
        type = "group",
        handler = self,
        args = {
            [L["toggleTextWarning"]] = {
                type = "toggle",
                order = 1,
                name = L["Text warning"],
                desc = function( info )
                            local returnString = L["Show text warnings to alert you to summon your companion."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.GiveTextWarning and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.GiveTextWarning;
                      end,
                set = function( info, val )
                          info.handler.db.profile.GiveTextWarning = val;
                      end,
            },
            [L["toggleButtonWarning"]] = {
                type = "toggle",
                order = 2,
                name = L["Button warning"],
                desc = function( info )
                            local returnString = L["Flash the button to alert you to summon your companion."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.GiveButtonWarning and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.GiveButtonWarning;
                      end,
                set = function( info, val )
                          info.handler.db.profile.GiveButtonWarning = val;
                      end,
            },
            [L["toggleMountWarning"]] = {
                type = "toggle",
                order = 3,
                name = L["Mount warning"],
                desc = function( info )
                            local returnString = L["Show an alert when you dismount from your flying mount."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.GiveMountWarning and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.GiveMountWarning;
                      end,
                set = function( info, val )
                          info.handler.db.profile.GiveMountWarning = val;
                      end,
            },
            [L["toggleBG"]] = {
                type = "toggle",
                order = 4,
                name = L["Battleground check"],
                desc = function( info )
                            local returnString = L["Show minipet alerts while in a battleground."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.BattleGroundCheck and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.BattleGroundCheck;
                      end,
                set = function( info, val )
                          info.handler.db.profile.BattleGroundCheck = val;
                      end,
            },
            [L["suggestOnlySoulbound"]] = {
                type = "toggle",
                order = 5,
                name = L["Suggest only soulbound companions"],
                width = "double",
                desc = function( info )
                            local returnString = L["Toggles whether only soulbound companions will be suggested for use."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.SuggestOnlySoulbound and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.SuggestOnlySoulbound;
                      end,
                set = function( info, val )
                          info.handler.db.profile.SuggestOnlySoulbound = val;
                          info.handler:GetPlayersCompanions();
                          info.handler:ChooseRandomCompanion();
                      end,
            },
            [L["reagentCheck"]] = {
                type = "toggle",
                order = 6,
                name = L["Check for reagented companions that are missing their reagent"],
                width = "double",
                desc = function( info )
                            local returnString = L["Toggles whether companions that require a reagent will be suggested when you have none of the required reagent."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.ReagentCheck and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.ReagentCheck;
                      end,
                set = function( info, val )
                          info.handler.db.profile.ReagentCheck = val;
                          info.handler:GetPlayersCompanions();
                          info.handler:ChooseRandomCompanion();
                      end,
            },
            [L["equipmentCheck"]] = {
                type = "toggle",
                order = 7,
                name = L["Check to make sure that equipment companions are equipped"],
                width = "double",
                desc = function( info )
                            local returnString = L["Toggles whether companions summoned by equipment will be suggested when the item is unequipped."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.ReagentCheck and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.EquipmentCheck;
                      end,
                set = function( info, val )
                          info.handler.db.profile.EquipmentCheck = val;
                          info.handler:GetPlayersCompanions();
                          info.handler:ChooseRandomCompanion();
                      end,
            },
            [L["forceQuestPet"]] = {
                type = "toggle",
                order = 8,
                name = L["Force to pick a quest companion if availiable"],
                width = "double",
                desc = function( info )
                            local returnString = L["Toggles whether companions that are part of a quest are preferred instead of a random companion."];

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.db.profile.ForceQuestPet and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.db.profile.ForceQuestPet;
                      end,
                set = function( info, val )
                          info.handler.db.profile.ForceQuestPet = val;
                          info.handler:GetPlayersCompanions();
                          info.handler:ChooseRandomCompanion();
                      end,
            },
            minipetItemModifier = {
                type = "select",
                order = 9,
                name = L["Modifier key for modifier items"],
                values = modifierKeys,
                cmdHidden = true,
                desc = L["Changes the key that needs to be held down to pick a modifer item."],
                width = "double",
                get = function( info )
                          return info.handler.db.profile.CritterItemModifier;
                      end,
                set = function( info, val )
                          info.handler.db.profile.CritterItemModifier = val;
                      end,
            },
            [L["list"]] = {
                type = "execute",
                order = 10,
                name = L["List all companions"],
                desc = L["Show a list with all of your companions."],
                func = function( info )
                           info.handler:Print( L["List of your companions:"] );
                           for _, companion in pairs( self:GetAllCompanionsList() ) do
                               info.handler:Print( ("|T%s:0|t %s"):format( companion:GetTexture(), companion:GetLink() ) );
                           end--for
                       end,
            },
            [L["relist"]] = {
                type = "execute",
                order = 11,
                name = L["Refresh companion list"],
                desc = L["Force a refresh of the list of all your companions."],
                func = "GetPlayersCompanions",
            },
        },
        name = L["MiniPet"],
        desc = L["Addon options."],
        icon = "Interface\\AddOns\\MiniPet\\Img\\MiniPet",
    };

    self.db = LibStub( "AceDB-3.0" ):New( "MiniPetDB", defaults );
    self.db.RegisterCallback( self, "OnProfileChanged" );
    self.db.RegisterCallback( self, "OnProfileReset" );

    local profileOptions = LibStub( "AceDBOptions-3.0" ):GetOptionsTable( self.db );
    local slashOptions = {
        type = "group",
        handler = self,
        name = options.name,
        desc = options.desc,
        icon = options.icon,
        args = {
            [L["options"]] = options,
            [L["favorites"]] = favoritesOptions,
            [L["zoneFavorites"]] = zoneOptions,
            [L["button"]] = buttonOptions,
            [L["profile"]] = profileOptions,
            [L["gui"]] = {
                type = "execute",
                order = -1,
                name = L["Open GUI configurator"],
                desc = L["Open GUI configurator"],
                guiHidden = true,
                func = function( info )
                           InterfaceOptionsFrame_OpenToFrame( L[info.handler.name] );
                       end,
            },
            debug = {
                type = "toggle",
                hidden = true,
                name = "Debug",
                desc = "Debug mode toggle",
                desc = function( info )
                            local returnString = "Debug mode toggle";

                            if( info.uiType == "cmd" ) then
                                returnString = ("%s |cFFFF0000[%s]|r"):format( returnString, info.handler.DEBUG and L["yes"] or L["no"] );
                            end--if
                           return returnString;
                       end,
                get = function( info )
                          return info.handler.DEBUG;
                      end,
                set = function( info, val )
                          info.handler.DEBUG = val;
                          info.handler:Print( ("debug mode |cFFFF0000[%s]|r"):format( info.handler.DEBUG and L["yes"] or L["no"] ) );
                      end,
            },
        },
    };
    
    for continentindex, continent in pairs( { GetMapContinents() } ) do
        zoneOptions.args[LibLordFarlander.NamifyString( continent )] = {
            type = "group",
            handler = self,
            name = continent,
            desc = L["Zones in %s"]:format( continent ),
            childGroups = "tree",
            args = {},
        };

        local continentArgs = zoneOptions.args[LibLordFarlander.NamifyString( continent )].args;

        LibLordFarlander:CopyTable( continentTemplate, continentArgs );
        for index, zone in pairs( LibLordFarlander.GetMapZonesWithExtra( continentindex ) ) do
            continentArgs[LibLordFarlander.NamifyString( zone )] = {
                type = "group",
                handler = self,
                name = zone,
                desc = zone,
                args = zoneTemplate,
            };
        end--for
    end--for
    LibStub( "AceConfig-3.0" ):RegisterOptionsTable( L["MiniPet"], slashOptions, { L["minipet"], L["mp"] } );
    AceConfigReg:RegisterOptionsTable( "MiniPet Main Options", options );
    AceConfigReg:RegisterOptionsTable( "MiniPet Button", buttonOptions );
    AceConfigReg:RegisterOptionsTable( "MiniPet Zone Favorites", zoneOptions );
    AceConfigReg:RegisterOptionsTable( "MiniPet Favorites", favoritesOptions );
    AceConfigReg:RegisterOptionsTable( "MiniPet Profile", profileOptions );
    AceConfigReg:RegisterOptionsTable( "MiniPet Modifer Items", modifierItemsOptions );
    AceConfigDialog:AddToBlizOptions( "MiniPet Main Options", L["MiniPet"] );
    AceConfigDialog:AddToBlizOptions( "MiniPet Button", L["Button"], L["MiniPet"] );
    AceConfigDialog:AddToBlizOptions( "MiniPet Modifer Items", L["Modifier Items"], L["MiniPet"] );
    AceConfigDialog:AddToBlizOptions( "MiniPet Zone Favorites", L["Zone Favorites"], L["MiniPet"] );
    AceConfigDialog:AddToBlizOptions( "MiniPet Favorites", L["Favorites"], L["MiniPet"] );
    AceConfigDialog:AddToBlizOptions( "MiniPet Profile", L["Profile"], L["MiniPet"] );
end--MiniPet:OnInitialize()

function MiniPet:OnProfileReset()
    self:OnProfileChanged();
end--MiniPet:OnProfileReset()

function MiniPet:OnProfileChanged()
    LibStub( "LibLordFarlander-UI-2.0" ).OnProfileChanged( self );
    if( self.db.profile.enchantOrder ) then
        local iterator, handle = PetAndMountDatabase:IterateSet( "Items.CritterEnchant" );
        local item, value = iterator( handle );

        while item do
            local inList = false;

            for _, listItem in pairs( self.db.profile.enchantOrder ) do
                if( listItem == item ) then
                    inList = true;
                    break;
                end--if
            end--for
            if( not inList ) then
                tinsert( self.db.profile.enchantOrder, item );
            end--if
            item, value = iterator( handle );
        end--for
    else
        self.db.profile.enchantOrder = {};

        local iterator, handle = PetAndMountDatabase:IterateSet( "Items.CritterEnchant" );
        local item, value = iterator( handle );
        local found = false;
        local counter = 1;

        while item do
            if( type( value ) == "boolean" ) then
                value = counter;
            else
                value = tonumber( value );
            end--if
            self.db.profile.enchantOrder[value] = item;
            item, value = iterator( handle );
            counter = counter + 1;
        end--for
    end--if
    --Upgrade check
    if( ( self.db.profile.LastVersion == nil ) and ( ( MiniPet_BattleGroundCheck ~= nil ) or ( MiniPet_GiveTextWarning ~= nil ) or ( MiniPet_GiveButtonWarning ~= nil ) or ( MiniPet_HideButton ~= nil ) ) ) then
        --Upgrading from 2.0 or earlier, import the old settings
        self.db.profile.BattleGroundCheck = MiniPet_BattleGroundCheck;
        self.db.profile.GiveTextWarning = MiniPet_GiveTextWarning;
        self.db.profile.GiveButtonWarning = MiniPet_GiveButtonWarning;
        self.db.profile.Button.AutoHideButton = MiniPet_HideButton;
    end--if
    if( type( self.db.profile.LastVersion ) == "number" ) then
        --Upgrading from pre-3.5, need to convert zone favorites to the new format
        local newList = {};

        for zone, itemID in pairs( self.db.profile.ZoneFavorites ) do
            newList[zone] = { itemID = { exclusive = true, } };
        end--for
        self.db.profile.ZoneFavorites = newList;
    end--if
    if( ( type( self.db.profile.LastWoWVersion ) ~= "number" ) or ( floor( self.db.profile.LastWoWVersion / 10000 ) < floor( LibLordFarlander.GetTOCVersion() / 10000 ) ) ) then
        --Upgrading from pre-wotlk, need to convert itemIDs in favorites and zone favorites to spellIDs and to the new format
        local newID;
        local newList = {};

        for itemID, value in pairs( self.db.profile.Favorites ) do
            newID = PetAndMountDatabase:ItemInSet( itemID, "Conversion.Critters" );
            if( newID ) then
                itemID = tonumber( newID );
            end--if
            newList[itemID] = ( type( value ) == "number" ) and value or 2.0;
            if( newList[itemID] == 1.0 ) then
                newList[itemID] = nil;
            end--if
        end--for
        self.db.profile.Favorites = newList;
        newList = {};
        for zone, pets in pairs( self.db.profile.ZoneFavorites ) do
            newList[zone] = {};
            for itemID, info in pairs( pets ) do
                newID = PetAndMountDatabase:ItemInSet( itemID, "Conversion.Critters" );
                if( newID ) then
                    itemID = tonumber( newID );
                end--if
                newList[zone][itemID] = info;
                if( not ( info.frequency or info.active or info.exclusive ) ) then
                    newList[zone][itemID] = nil;
                end--if                
            end--for
        end--for
        self.db.profile.ZoneFavorites = newList;
    end--if
    self.db.profile.LastWoWVersion = LibLordFarlander.GetTOCVersion();
    self.db.profile.LastVersion = self.version;

    self.Dirty = true;
    --Create the Pet Inventory
    self:GetPlayersCompanions();
    --Set a Pet
    self:ChooseRandomCompanion();
end--MiniPet:OnProfileChanged()

function MiniPet:OnEnable()
    local LibUI = LibStub( "LibLordFarlander-UI-2.0" );
    local quixote = LibStub( "LibQuixote-2.0", true );

    --Register the Events
    self:RegisterEvent( "BAG_UPDATE",
        function( _, bagID )
            if( bagID and ( bagID >= 0 ) and ( bagID <= NUM_BAG_SLOTS ) ) then
                MiniPet.dirtyBags[bagID] = true;
                MiniPet.Dirty = true;
                if( not MiniPet.button:IsVisible() ) then
                    MiniPet:ScheduleTimer( "OnUpdate", 0 );
                end--if
            end--if
        end );
    self:RegisterEvent( "COMPANION_UPDATE",
        function( _, type )
            if( not type ) then
                MiniPet.Dirty = true;
                MiniPet.dirtyBags[-255] = true;
                if( not MiniPet.button:IsVisible() ) then
                    MiniPet:ScheduleTimer( "OnUpdate", 0 );
                end--if
            end--if
        end );
    self:RegisterEvent( "MODIFIER_STATE_CHANGED",
        function()
            MiniPet:SetButton( MiniPet:GetStateBasedOnModifiers() );
        end );
    self:RegisterEvent( "PLAYER_ALIVE",
        function()
            MiniPet.BoardedTaxi = UnitOnTaxi( "player" );
            if( MiniPet.HearthstoneComplete ) then
                MiniPet.HearthstoneComplete = false;
                MiniPet:dprint( "MiniPet: Activating Pet After Hearthstone!" );
                MiniPet:WarnPlayer();
            end--if
            if( MiniPet.PlayerDied and ( not UnitIsDeadOrGhost( "player" ) ) ) then
                MiniPet.PlayerDied = false;
                MiniPet:dprint( "MiniPet: no longer dead!" );
                MiniPet:WarnPlayer();
            end--if
        end );
    self:RegisterEvent( "PLAYER_CONTROL_GAINED",
        function()
            if( MiniPet.BoardedTaxi ) then
                MiniPet.BoardedTaxi = false;
                MiniPet:WarnPlayer();
            end--if
        end );
    self:RegisterEvent( "PLAYER_DEAD",
        function()
            MiniPet:dprint( "MiniPet: PlayerDied" );
            MiniPet.PlayerDied = true;
        end );
    self:RegisterEvent( "PLAYER_ENTERING_WORLD",
        function()
            MiniPet.PlayerDied = UnitIsDeadOrGhost( "player" );
            MiniPet.BoardedTaxi = UnitOnTaxi( "player" );
            MiniPet.PlayerEnteringWorld = false;
            MiniPet:WarnPlayer();
        end );
    self:RegisterEvent( "PLAYER_UNGHOST",
        function()
            if( MiniPet.PlayerDied ) then
                MiniPet.PlayerDied = false;
                MiniPet:dprint( "MiniPet: no longer ghost!" );
                MiniPet:WarnPlayer();
            end--if
        end );
    self:RegisterEvent( "TAXIMAP_CLOSED",
        function()
            MiniPet.BoardedTaxi = true;
        end );
    self:RegisterEvent( "UNIT_INVENTORY_CHANGED",
        function( _, unit )
            if( unit == "player" ) then
                MiniPet.dirtyBags[-254] = true;
                MiniPet.Dirty = true;
                if( not MiniPet.button:IsVisible() ) then
                    MiniPet:ScheduleTimer( "OnUpdate", 0 );
                end--if
            end--if
        end );
    self:RegisterEvent( "COMBAT_LOG_EVENT_UNFILTERED",
        function( _, _, event, sourceGUID, _, _, _, _, _, spellID )
            if( ( sourceGUID == UnitGUID( "player" ) ) and event:find( "SPELL_" ) ) then
                if( event == "SPELL_CAST_START" ) then
                    MiniPet:dprint( "Spellcast: " .. spellID );
                    MiniPet.StartedHearthstone = false;
                    if( PetAndMountDatabase:SpellInSet( spellID, "Spells.Teleport" ) ) then
                        MiniPet:dprint( "MiniPet: Started Teleporting!" );
                        MiniPet.StartedHearthstone = true;
                    end--if
                elseif( ( event == "SPELL_CAST_SUCCESS" ) and MiniPet.StartedHearthstone and PetAndMountDatabase:SpellInSet( spellID, "Spells.Teleport" ) ) then
                        MiniPet:dprint( "MiniPet: Finished Teleporting!" );
                        MiniPet.StartedHearthstone = false;
                        MiniPet.HearthstoneComplete = true;
                        MiniPet:WarnPlayer();
                elseif( ( event == "SPELL_CAST_FAILED" ) and MiniPet.StartedHearthstone ) then
                        MiniPet:dprint( "MiniPet: Interrupted Teleporting!" );
                        MiniPet.StartedHearthstone = false;
                end--if
            end--if
        end );
    self:RegisterEvent( "PLAYER_REGEN_ENABLED",
        function()
            MiniPet.button:OutOfCombatReset();
            MiniPet:SetButton( MiniPet:GetStateBasedOnModifiers() );
        end );
    self:RegisterEvent( "ZONE_CHANGED_NEW_AREA",
        function()
            MiniPet.ui.currentZone = GetRealZoneText();
            if( MiniPet.db.profile.ZoneFavorites[GetRealZoneText()] and MiniPet.HasCompanionListChanged() ) then
                MiniPet:ChooseRandomCompanion();
            end--if
        end );

    LibStub( "LibSpecialEvents-Mount-4.0" ).RegisterCallback( self, "Dismounted",
        function( _, _, _, flying )
            if( flying and MiniPet.db.profile.GiveMountWarning ) then
                MiniPet:WarnPlayer();
            end--if
        end );
    LibStub( "LibSpecialEvents-LearnSpell-4.0" ).RegisterCallback( self, "LearnedSpell",
        function( _, spell )
            local spellID = LibLordFarlander.GetItemIDFromLink( GetSpellLink( spell ) );

            if( ( spellID ~= nil ) and PetAndMountDatabase:SpellInSet( spellID, "Critters" ) ) then
                MiniPet.Dirty = true;
                MiniPet.dirtyBags[-255] = true;
                if( not MiniPet.button:IsVisible() ) then
                    MiniPet:ScheduleTimer( "OnUpdate", 0 );
                end--if
            end--if
        end );
    if( quixote ) then
        quixote.RegisterCallback( self, "Quest_Gained",
            function( _, _, uid )
                if( MiniPet.db.profile.ForceQuestPet and ( MiniPet.QuestCompanionQuest == uid ) ) then
                    MiniPet.dirtyBags = { [0] = true, [1] = true, [2] = true, [3] = true, [4] = true, [-254] = true, [-255] = true, };
                    MiniPet:GetPlayersCompanions();
                    MiniPet:ChooseRandomCompanion();
                end--if
            end );
        quixote.RegisterCallback( self, "Quest_Complete", "OnQuestDone" );
        quixote.RegisterCallback( self, "Quest_Failed", "OnQuestDone" );
        quixote.RegisterCallback( self, "Quest_Lost", "OnQuestDone" );
    else -- The hard way
        self:RegisterEvent( "UNIT_QUEST_LOG_CHANGED",
            function( _, unit )
                if( ( unit == "player" ) and MiniPet.db.profile.ForceQuestPet and MiniPet.QuestCompanionQuest ) then
                    if( MiniPet:HasQuestAndNotComplete( MiniPet.QuestCompanionQuest ) ) then
                        if( not MiniPet.QuestCompanion ) then
                            MiniPet.dirtyBags = { [0] = true, [1] = true, [2] = true, [3] = true, [4] = true, [-254] = true, [-255] = true, };
                            MiniPet:GetPlayersCompanions();
                            MiniPet:ChooseRandomCompanion();
                        end--if
                    else
                        MiniPet.QuestCompanionQuest = nil;
                        MiniPet.QuestCompanion = nil;
                        MiniPet:ChooseRandomCompanion();
                    end--if
                end--if
            end );
    end--if

    LibUI.SetupForUI( self );
    self.button = LibUI.CreateButton( self, self.name, "Interface\\AddOns\\MiniPet\\Img\\MiniPet", self.db.profile.ButtonGroup, self.SkinChanged, self );
    ItemList.AttachToButton( self.button );
    self.button:SetMovable( not self.db.profile.Button.LockButton );
    self.button:HookScript( "OnClick",
        function( self, mouse )
            self.class.FlashButton = false; --Stop Flashing the Button :)
            if( self.class.db.profile.Button.AutoHideButton ) then
                self:HideEx();
            end--if
            if( ( mouse == "RightButton" ) and LibStub( "LibLordFarlander-UI-2.0" ).IsModifierDown( self.class.db.profile.CritterItemModifier ) and self.class.CurrentCompanionItem ) then
                self.class:FindNextCompanionEnchant();
            else
                self.class:ChooseRandomCompanion();
            end--if
        end );

    self:OnProfileChanged();
end--MiniPet:OnEnable();

function MiniPet:OnQuestDone( _, _, uid )
    if( MiniPet.db.profile.ForceQuestPet and ( MiniPet.QuestCompanionQuest == uid ) ) then
        MiniPet.QuestCompanionQuest = nil;
        MiniPet.QuestCompanion = nil;
        MiniPet:ChooseRandomCompanion();
    end--if
end--MiniPet:OnQuestDone()

function MiniPet:OnDisable()
    local quixote = LibStub( "LibQuixote-2.0", true );

    if( quixote ) then
        quixote.UnregisterAll( self );
    end--if
    LibStub( "LibSpecialEvents-Mount-4.0" ).UnregisterAll( self );
    LibStub( "LibSpecialEvents-LearnSpell-4.0" ).UnregisterAll( self );
end--MiniPet:OnDisable()

function MiniPet:OnUpdate( elapsed )
    if( ( UnitName( "player" ) ~= UNKNOWNOBJECT ) and self.ReadyToInitialize ) then
        self:dprint( "MiniPet: PlayerEnteringWorld and ALIVE" );
        self.PlayerEnteringWorld = false;
        self.ReadyToInitialize = false;
        self:WarnPlayer();
    end--if
    
    if( self.Dirty ) then
        self.Dirty = false;
        if( self:HasCompanionListChanged() ) then
            self:ChooseRandomCompanion();
        end--if
        self:SetButton( self:GetStateBasedOnModifiers() );
    end--if

    --Code to let the button flash
    --elapsed = amount of seconds that passed since last update.
    --every updateinterval, change the button image
    --until you reach totalflashtime
    if( self.FlashButton and elapsed ) then
        local mpButton = self.button;

        if( mpButton.TimeSinceLastUpdate == nil ) then
            mpButton.TimeSinceLastUpdate = 0;
            self.FlashTime = 0;
        end--if
        mpButton.TimeSinceLastUpdate = mpButton.TimeSinceLastUpdate + elapsed;
        self.FlashTime = self.FlashTime + elapsed;
        if( self.FlashTime < self.TotalFlashTime ) then
            if( mpButton.TimeSinceLastUpdate > self.UpdateInterval ) then
                if( self.ShowButtonImg and ( self.CurrentCompanion ~= nil ) ) then
                    mpButton:SetFromListEntry( self.CurrentCompanion );
                    self.ShowButtonImg = false;
                else
                    mpButton:SetImage( "Interface\\AddOns\\MiniPet\\Img\\MiniPet" );
                    self.ShowButtonImg = true;
                end--if
                mpButton.TimeSinceLastUpdate = 0;
            end--if
        else
            self.FlashButton = false;
            self:SetButton( self:GetStateBasedOnModifiers() );
            self.FlashTime = 0;
        end--if
    end--if
end--MiniPet:OnUpdate( elapsed )

----------------------------
--+Pet/Inventory Functions--
function MiniPet:GetStateBasedOnModifiers()
    if( LibStub( "LibLordFarlander-UI-2.0" ).IsModifierDown( self.db.profile.CritterItemModifier ) and self.CurrentCompanionItem ) then
        return self.CurrentCompanionItem;
    end--if
    return self.CurrentCompanion;
end--MiniPet:GetStateBasedOnModifiers()

function MiniPet:GetZoneCompanion( set, ZoneList )
    if( ZoneList and LibLordFarlander.HasTableMembers( ZoneList ) ) then
        local profile = self.db.profile;
        local favorites = {};
        local exclusives = {};

        for pet, settings in pairs( ZoneList ) do
            if( settings.active ) then
                favorites[pet] = settings.frequency or 1.0;
            end--if
            if( settings.exclusive ) then
                exclusives[pet] = settings.frequency or 1.0;
            end--if
        end--if
        if( LibLordFarlander.HasTableMembers( favorites ) or LibLordFarlander.HasTableMembers( exclusives ) ) then
            local default = 1.0;

            if( LibLordFarlander.HasTableMembers( exclusives ) ) then
                favorites = exclusives;
                default = 0.0;
            else
                local profileFavorites = self.db.profile.Favorites;

                for _, item in pairs( set ) do
                    if( not favorites[item] ) then
                        favorites[item] = profileFavorites[item] or 1.0;
                    end--if
                end--if
            end--if
            return self:GetRandomFromListWithWeighing( set, favorites, default );
        end--if
    end--if
    return self:GetRandomFromListWithWeighing( set, self.db.profile.Favorites );
end--MiniPet:GetZoneCompanion( set, ZoneList )

function MiniPet:GetRandomFromListWithWeighing( set, favorites, default )
    if( not default ) then
        default = 1.0;
    end--if
    if( LibLordFarlander.HasTableMembers( favorites ) and set and ( LibLordFarlander.TableCount( set ) > 1 ) ) then
        local PlayerItemCount = LibLordFarlander.PlayerItemCount;
        local GetItemIDFromLink = LibLordFarlander.GetItemIDFromLink;
        local ItemIsSoulbound = LibLordFarlander.ItemIsSoulbound;
        local HasPlayerSpell = LibLordFarlander.HasPlayerSpell;
        local ItemInSet = PetAndMountDatabase.ItemInSet;
        local profile = self.db.profile;
        local opReagentCheck = profile.ReagentCheck;
        local opSuggestOnlySoulbound = profile.SuggestOnlySoulbound;
        local total = 0.0;
        local useset = {};
        local randomValue;
        local amountOfItems = 0;
        local index = 0;
        local item = nil;

        for _, item in pairs( set ) do
            local reagented = nil;
            local itemID = item:GetID();
            local howOften = favorites[itemID] or default;
            local BagItem = ( itemID > 0 ) and ItemList.FindInBags( itemID ) or nil;

            if( opReagentCheck ) then
                reagented = ItemInSet( PetAndMountDatabase, itemID, "Critters.Reagented" );
            end--if
            if( ( howOften > 0.0 ) and ( ( itemID < 0 ) or ( PlayerItemCount( itemID ) > 0 ) ) and ( ( not opReagentCheck ) or ( not reagented ) or ( GetItemCount( reagented ) > 0 ) ) and ( ( itemID < 0 ) or ( ( not opSuggestOnlySoulbound ) or ItemIsSoulbound( BagItem.bag, BagItem.slot ) ) ) ) then
                tinsert( useset, item );
                total = total + howOften;
                amountOfItems = amountOfItems + 1;
            end--if
        end--for
        randomValue = math.random() * total;
        repeat
            index = index + 1;
            item = useset[index];
            if( item ) then
                randomValue = randomValue - ( favorites[item:GetID()] or default );
            end--if
        until ( not item ) or ( randomValue <= 0 ) or ( index == amountOfItems );
        return item;
    else
        return ItemList.GetRandomItemFromList( set );
    end--if
end--MiniPet:GetRandomFromListWithWeighing( set, favorites, default )

--Pick a random pet and set the button.
function MiniPet:ChooseRandomCompanion()
    local profile = self.db.profile;

    if( profile.ForceQuestPet and self.QuestCompanion ) then
        self.CurrentCompanion = self.QuestCompanion;
    else
        self.CurrentCompanion = self:GetZoneCompanion( self:GetAllCompanionsList(), profile.ZoneFavorites[GetRealZoneText()] );
    end--if
    self:SetButton( self:GetStateBasedOnModifiers() );
end--MiniPet:ChooseRandomCompanion()

function MiniPet:GetAllCompanionsList()
    local returnVal = {};

    for _, item in pairs( self.PlayersCompanions ) do
        if( type( item ) == "table" ) then
            for _, subitem in pairs( item ) do
                tinsert( returnVal, subitem );
            end--for
        else
            tinsert( returnVal, item );
        end--if
    end--for
    return returnVal;
end--MiniPet:GetAllCompanionsList()

--Gets all your pets from your inventory.
function MiniPet:GetPlayersCompanions()
    local dirtyBags = self.dirtyBags;
    local playerPetList = self.PlayersCompanions;
    local itemLink;
    local reagented;
    local equipmentSlot;
    local opSuggestOnlySoulbound = self.db.profile.SuggestOnlySoulbound;
    local opReagentCheck = self.db.profile.ReagentCheck;
    local opEquipmentCheck = self.db.profile.EquipmentCheck;
    local ItemInSet = PetAndMountDatabase.ItemInSet;
    local ItemIsSoulbound = LibLordFarlander.ItemIsSoulbound;
    local AddBagItemToList = ItemList.AddBagItemToList;
    local AddSpellToList = ItemList.AddSpellToList;
    local ForceQuestPet = self.db.profile.ForceQuestPet;
    local GetItemIDFromLink = LibLordFarlander.GetItemIDFromLink;

    self.QuestCompanionQuest = nil;
    self.CurrentCompanionItem = nil;
    for bag = 0, NUM_BAG_SLOTS do
        if( dirtyBags[bag] ) then
            playerPetList[bag] = {};

            local bagList = playerPetList[bag];

            for slot = 1, GetContainerNumSlots( bag ) do
                itemLink = GetContainerItemLink( bag, slot );
                
                if( itemLink ) then
                    local itemID = GetItemIDFromLink( itemLink );

                    if( itemID and ItemInSet( PetAndMountDatabase, itemID, "Critters" ) ) then
                        if( opReagentCheck ) then
                            reagented = ItemInSet( PetAndMountDatabase, itemID, "Critters.Reagented" );
                        else
                            reagented = nil;
                        end--if
                        if( ( ( not opSuggestOnlySoulbound ) or ItemIsSoulbound( bag, slot ) ) and
                            ( ( not opReagentCheck ) or ( not reagented ) or ( GetItemCount( reagented ) > 0 ) ) and
                            ( ( not opEquipmentCheck ) or ( not ItemInSet( PetAndMountDatabase, itemID, "Critters.Equipment" ) ) ) ) then
                            AddBagItemToList( bagList, bag, slot );
                            self.HasCompanions = true;
                            if( ForceQuestPet ) then
                                local questID = ItemInSet( PetAndMountDatabase, itemID, "Critters.Quest" );
                                
                                if( questID ) then
                                    self.QuestCompanionQuest = questID;
                                end--if
                                if( ( questID and self:HasQuestAndNotComplete( tonumber( questID ) ) ) or
                                    ( ( self.QuestCompanion == nil ) and ItemInSet( PetAndMountDatabase, itemID, "Critters.Children" ) ) ) then
                                    self.QuestCompanion = ItemList.BuildBagItem( bag, slot );
                                end--if
                            end--if
                        end--if
                    end--if
                end--if
            end--for
        end--if
    end--for
    
    if( dirtyBags[-255] ) then
        playerPetList[-255] = {};

        if( PetAndMountDatabase.GetPlayersCompanions ) then
            local spellList = playerPetList[-255];

            for id, critter in pairs( PetAndMountDatabase:GetPlayersCompanions( "CRITTER" ) ) do
                if( opReagentCheck ) then
                    reagented = ItemInSet( PetAndMountDatabase, critter.spellID * -1, "Critters.Reagented" );
                else
                    reagented = nil;
                end--if
                if( ( not opReagentCheck ) or ( not reagented ) or ( GetItemCount( reagented ) > 0 ) ) then
                    AddSpellToList( spellList, critter.spellID );
                    self.HasCompanions = true;
                end--if
            end--for
        end--if
    end--if

    if( dirtyBags[-254] ) then
        playerPetList[-254] = {};
    
        local iterator, handle = PetAndMountDatabase:IterateSet( "Critters.Equipment" );
        local pet, equipmentSlot = iterator( handle );
        local equipped = playerPetList[-254];
        local AddInventoryItemToList = ItemList.AddInventoryItemToList;
    
        while pet do
            equipmentSlot = GetInventorySlotInfo( equipmentSlot );
            
            local itemLink = GetInventoryItemLink( "player", equipmentSlot );

            if( itemLink and opEquipmentCheck and ( GetItemIDFromLink( itemLink ) == pet ) ) then
                AddInventoryItemToList( equipped, equipmentSlot );
                self.HasCompanions = true;                
            end--if
            pet, equipmentSlot = iterator( handle );
        end--for
    end--if
    self.dirtyBags = {};

    self:FindNextCompanionEnchant();
end--MiniPet:GetPlayerPets()

function MiniPet:FindNextCompanionEnchant()
    local enchantOrder = self.db.profile.enchantOrder;
    local index = 1;    

    while( enchantOrder[index] and ( GetItemCount( enchantOrder[index] ) == 0 ) ) do
        index = index + 1;
    end--while
    if( enchantOrder[index] ) then
        self.CurrentMountItem = ItemList.BuildItem( enchantOrder[index] );
    else
        self.CurrentMountItem = nil;
    end--if
end--MiniPet:FindNextCompanionEnchant()
---Pet/Invenroty Functions--
----------------------------

-------------------
--+Help Functions--
function MiniPet.IsATeleportSpell( spellName )
    local spellID = LibLordFarlander.GetItemIDFromLink( GetSpellLink( spellName ) );

    return ( spellID ~= nil ) and ( PetAndMountDatabase:SpellInSet( spellID, "Spells.Teleport" ) ~= nil );
end--MiniPet.IsATeleportSpell( spellName )

function MiniPet:HasCompanionListChanged()
    local prePetList =  {};
    local preZoneFavorites = {};

    LibLordFarlander.CopyTable( prePetList, self.PlayersCompanions );
    LibLordFarlander.CopyTable( preLastZoneList, self.LastZoneList );

    self:GetPlayersCompanions();

    if( not self.CompanionListOkay ) then
        self.CompanionListOkay = true;
        return true;
    end--if
    self.CompanionListOkay = true;
    return ( not LibLordFarlander.CompareTables( prePetList, self.PlayersCompanions ) ) and ( not LibLordFarlander.CompareTables( preLastZoneList, self.LastZoneList ) );
end--MiniPet:HasCompanionListChanged()

--Check if a player is in a battleground
function MiniPet.IsInBattleground()
    local isInstanced, type = IsInInstance();

    return isInstanced and ( type == "pvp" );
end--MiniPet.IsInBattleground()

function MiniPet.HasQuestAndNotComplete( questID )
    if( _G["Baldrick"] ) then
        return _G["Baldrick"].HasQuestAndNotComplete( questID );
    else
        local quixote = LibStub( "LibQuixote-2.0", true );

        if( quixote ) then
            local uid, _, _, _, _, _, complete, _, _, _ = quixote:GetQuestByUid( questID );
        
            return ( uid ~= nil) and ( not complete );
        else
            local numEntries, _ = GetNumQuestLogEntries();
        
            if( numEntries > 0 ) then
                for id = 1, numEntries do
                    local complete = select( 6, GetQuestLogTitle( id ) );
                    
                    if( ( not complete ) and ( LibLordFarlander.GetItemIDFromLink( GetQuestLink( id ) ) == questID ) ) then
                        return true;
                    end--if
                end--for
            end--if
        end--if
    end--if
    return false;
end--Minipet.GetQuestID( questName )
---Help Functions--
-------------------

----------------------
--+Message Functions--
--Give the player a reminder.
function MiniPet:WarnPlayer()
    local charDB = self.db.profile;

    if( ( not ( self.IsInBattleground() and charDB.BattleGroundCheck ) ) and ( not self.PlayerDied ) and ( not self.BoardedTaxi ) ) then--BG + option check
        self.FlashButton = charDB.GiveButtonWarning;
        if( charDB.GiveTextWarning and self.HasCompanions ) then    --No use if player isn't cool and gots no pets!
            self.wprint( L["ALERT: Time to summon your companion!"] );
        end--if
        if( charDB.Button.AutoHideButton and ( not self.button:IsVisible() ) ) then
            self.button:ShowEx();
        end--if
    end--if
end--MiniPet:WarnPlayer()

--Shows a warning in the middle of the screen.
function MiniPet.wprint( msg )
    MinipetTextFrameText:SetTextColor( 1.0, 0.1, 0.1 );
    MinipetTextFrameText:SetText( msg );
    MinipetTextFrame.startTime = GetTime();
    MinipetTextFrame:Show();
end--MiniPet.wprint( msg )
---Message Functions
----------------------

-----------------
--+UI Functions--
function MiniPet:SetTooltip( tooltip )
    local pet = self.CurrentCompanion;

    if( LibStub( "LibLordFarlander-UI-2.0" ).IsModifierDown( self.db.profile.CritterItemModifier ) and self.CurrentCompanionItem ) then
        pet = self.CurrentCompanionItem;
    end--if
    if( pet == nil ) then
        tooltip:SetText( L["Get a companion for some company!"] );
    else
        ItemList.SetTooltipFromListEntry( tooltip, pet );
    end--if
end--Minipet:SetTooltip( tooltip )

function MiniPet:SetGameTooltips()
    local mpButton = self.button;

    if( not self.keyBoundMode ) then
        local mpTooltip = mpButton.tooltip;

        mpTooltip:SetOwner( mpButton, "RIGHT" );
        self:SetTooltip( mpTooltip );
    end--if
    LibStub( "LibLordFarlander-UI-2.0" ).AfterSetTooltip( mpButton );
end--MiniPet:SetGameTooltips()

function MiniPet:SetButton( critter )
    if( not InCombatLockdown() ) then
        local mpButton = self.button;
        local ldbobj = self.ldbobj;

        if( critter == nil ) then
            mpButton.cooldown:Hide();
            mpButton:SetAttribute( "type1", "item" );
            mpButton:SetAttribute( "*bag1", nil );
            mpButton:SetAttribute( "*slot1", nil );
            mpButton:SetAttribute( "*item1", nil );
            mpButton:SetImage( "Interface\\AddOns\\MiniPet\\Img\\MiniPet" );

            ldbobj.icon = "Interface\\AddOns\\MiniPet\\Img\\MiniPet";
            ldbobj.text = L["Get a companion for some company!"];
        else
            mpButton:SetFromListEntry( critter, not LibLordFarlander.CompareTables( self.LastCompanion, critter ) );
        end--if
        self.LastCompanion = critter;
    end--if
end--MiniPet:SetButton( pet )
---UI Functions--
-----------------
