-- MendWatch by Polima
-- 
-- Based upon Chronometer by Bhattu

-- ==========================================================================================
-- Setup
-- ==========================================================================================
MendWatch = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0",
                                           "AceConsole-2.0", "AceDebug-2.0",
                                           "AceHook-2.1", "CandyBar-2.0")

local L = AceLibrary("AceLocale-2.2"):new("MendWatch")
local DD = AceLibrary("Dewdrop-2.0")

local Spell_PoM = GetSpellInfo(33076)

-- ==========================================================================================
-- MendWatch:OnInitialize
-- ==========================================================================================
function MendWatch:OnInitialize()

	local paint = AceLibrary("PaintChips-2.0")
	paint:RegisterColor("gray", 0.5, 0.5, 0.5)
	paint:RegisterColor("forest", 0.0, 0.5, 0.0)
	paint:RegisterColor("maroon", 0.5, 0.0, 0.0)
	paint:RegisterColor("navy", 0.0, 0.0, 0.5)
	paint:RegisterColor("olive", 0.5, 0.5, 0.0)
	paint:RegisterColor("purple", 0.5, 0.0, 0.5)
	paint:RegisterColor("teal", 0.0, 0.5, 0.5)

	local colors = {"white", "black", "blue", "magenta", "cyan", "green", "yellow",
			"orange", "red", "gray", "forest", "maroon", "navy", "olive",
			"purple", "teal"
	}

	self.textures = {
		["default"] = nil,
		["banto"] = "Interface\\Addons\\MendWatch\\Textures\\banto",
		["smooth"] = "Interface\\Addons\\MendWatch\\Textures\\smooth",
		["perl"] = "Interface\\Addons\\MendWatch\\Textures\\perl",
		["glaze"] = "Interface\\Addons\\MendWatch\\Textures\\glaze",
		["cilo"] = "Interface\\Addons\\MendWatch\\Textures\\cilo",
		["charcoal"] = "Interface\\Addons\\MendWatch\\Textures\\Charcoal",
	}

	local options = {
		type = "group",
		args = {
			anchor = {
				name = L["Anchor"],
				desc = L["Show the dragable anchor."],
				type = "execute",
				func = "ToggleAnchor"
			},
			bar = {
				name = L["Bar"],
				desc = L["CandyBar Options"],
				type = "group",
				args = {
					texture = {
						name = L["Bar Texture"],
						desc = L["Change the texture of the timer bars."],
						type = "text",
						get = function () return self.db.profile.bartex end,
						set = function (v) self.db.profile.bartex = v end,
						validate = { "default", "banto", "smooth", "perl", "glaze", "cilo", "charcoal" },
					},
					color = {
						name = L["Bar Color"],
						desc = L["Set the default bar color."],
						type = "text",
						get = function () return self.db.profile.barcolor end,
						set = function (v) self.db.profile.barcolor = v end,
						validate = colors,
					},
					bgcolor = {
						name = L["Background Color"],
						desc = L["Set the bar background color."],
						type = "text",
						get = function () return self.db.profile.bgcolor end,
						set = function (v) self.db.profile.bgcolor = v end,
						validate = colors,
					},
					bgalpha = {
						name = L["Background Alpha"],
						desc = L["Set the bar background alpha."],
						type = "range",
						get = function () return self.db.profile.bgalpha end,
						set = function (v) self.db.profile.bgalpha = v end,
						min = 0.0,
						max = 1.0,
					},
					textcolor = {
						name = L["Text Color"],
						desc = L["Set the bar text color."],
						type = "text",
						get = function () return self.db.profile.textcolor end,
						set = function (v) self.db.profile.textcolor = v end,
						validate = colors,
					},
					textsize = {
						name = L["Text Size"],
						desc = L["Set the bar text size."],
						type = "range",
						get = function () return self.db.profile.textsize end,
						set = function (v) self.db.profile.textsize = v end,
						min = 8,
						max = 20,
					},
					scale = {
						name = L["Bar Scale"],
						desc = L["Set the bar scale."],
						type = "range",
						get = function () return self.db.profile.barscale end,
						set = function (v) self.db.profile.barscale = v end,
						min = 0.5,
						max = 1.5,
					},
					height = {
						name = L["Bar Height"],
						desc = L["Set the bar height."],
						type = "range",
						get = function () return self.db.profile.barheight end,
						set = function (v) self.db.profile.barheight = v end,
						min = 8,
						max = 30,
					},
					width = {
						name = L["Bar Width"],
						desc = L["Set the bar width."],
						type = "range",
						get = function () return self.db.profile.barwidth end,
						set = function (v) self.db.profile.barwidth = v end,
						min = 50,
						max = 300,
					},
					growth = {
						name = L["Bar Growth"],
						desc = L["Toggle the bar growing upwards or downwards."],
						type = "toggle",
						get = function () return self.db.profile.growup end,
						set = function (v) self.db.profile.growup = v
						self:SetCandyBarGroupGrowth("MendWatch", self.db.profile.growup) end,
					},
					reverse = {
						name = L["Reverse"],
						desc = L["Toggle the bar fill direction."],
						type = "toggle",
						get = function() return self.db.profile.reverse end,
						set = function(f) self.db.profile.reverse = f end,
					},
				},
			},
		},
	}

	options.args.config = {
		name = L["Config"],
		desc = L["Show the GUI configuration menu."],
		type = "execute",
		guiHidden = true,
		func = function() DD:Open(UIParent, 'children', function() DD:FeedAceOptionsTable(options) end, 'cursorX', true, 'cursorY', true) end,
	}

	local defaults = {
		bartex = "default",
		barcolor = "gray",
		bgcolor = nil,
		bgalpha = nil,
		textcolor = nil,
		textsize = nil,
		barscale = nil,
		barwidth = nil,
		barheight = nil,
		growup = false,
		reverse = false,
		barposition = {},
	}

	MendWatch:RegisterDB("MendWatchDB")
	MendWatch:RegisterDefaults('profile', defaults)
	MendWatch:RegisterChatCommand({'/mw', '/mendwatch'}, options)

	self.options = nil

	-- create the anchor frame
	self.anchor = self:CreateAnchor(L["MendWatch"], 0, 1, 0)

	-- create candybar group
	self:RegisterCandyBarGroup("MendWatch")
	self:SetCandyBarGroupPoint("MendWatch", "TOP", self.anchor, "BOTTOM", 0, 0)	
	self:SetCandyBarGroupGrowth("MendWatch", self.db.profile.growup)
end

-- ==========================================================================================
-- MendWatch:OnEnable
-- ==========================================================================================
function MendWatch:OnEnable()
	-- grab the Prayer of Mending texture for later
	self.tx = select(3, GetSpellInfo(33076))

	-- initialize the bars array
	self.bars = {}
	for i = 1, 20 do self.bars[i] = {} end

	-- watch for buff fade, gain and healing
  self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")

	-- watch for spellcasitng
	self.captive = {}
	self:RegisterEvent("UNIT_SPELLCAST_SENT")
	self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
	self:SecureHook("SpellStopCasting")
end

-- ==========================================================================================
-- MendWatch:OnDisable
-- ==========================================================================================
function MendWatch:OnDisable()
	-- clear out the bars
	if self.bars then
		for i = 1, 20 do
			if self.bars[i].id then
				self:StopCandyBar(self.bars[i].id)
			end
		end
		self.bars = nil
	end
	
	-- remove any captive spells
	if self.captive then self.captive = nil end

	-- undo all the events and hooks
	self:UnregisterAllEvents()
	self:UnhookAll()
end

-- ==========================================================================================
-- MendWatch:COMBAT_LOG_EVENT_UNFILTERED
-- ==========================================================================================
function MendWatch:COMBAT_LOG_EVENT_UNFILTERED(timestamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
  local spellId, spellName, spellSchool, auraType, amount, critical
  if eventType == "SPELL_AURA_REMOVED" then
    spellId, spellName, spellSchool, auraType = ...
    if spellName == Spell_PoM then
      self:Debug("COMBAT_LOG_EVENT_UNFILTERED - " .. timestamp .. " : " .. eventType .. " / " .. spellName .. " / " .. dstName .. " / " .. dstGUID .. " / " .. dstFlags)
      target = dstName
      for i = 20, 1, -1 do
        if self.bars[i].target == target then
          id = self.bars[i].id
          otext = target .. " (" .. self.bars[i].num .. ")"
          self:SetCandyBarColor(id, "red")
          self:SetCandyBarText(id, otext .. " - expired!")
          self:SetCandyBarFade(id, 5, true)
          self:StopCandyBar(id)
          self:StopBar(id)
        end
      end
    end
    return
  end
  
  if eventType == "SPELL_AURA_APPLIED" or eventType == "SPELL_PERIODIC_AURA_APPLIED" then
    spellId, spellName, spellSchool, auraType = ...
    if spellName == Spell_PoM then
      self:Debug("COMBAT_LOG_EVENT_UNFILTERED - " .. timestamp .. " : " .. eventType .. " / " .. spellName .. " / " .. dstName .. " / " .. dstGUID .. " / " .. dstFlags)

      name = spellName
      target = dstName

      -- look for a bar associated with this target and
      -- cancel if the spell was user iniated
      for i = 20, 1, -1 do
        if self.bars[i].target == target then
          if self.bars[i].initiated == 1 then
            return
          end
        end
      end

      -- 'target' is a name, but to check for the
      -- Prayer of Mending count we need a unitid. Run
      -- through the player, partyX and raidX unitids and
      -- find the one associated with 'target'
      local unit = nil

      if target == UnitName("player") then
        unit = "player"
      end

      if not unit then
        for i = 0, GetNumRaidMembers() do
          if target == UnitName("raid" .. i) then
            unit = "raid" .. i
            break
          end
        end
      end

      if not unit then
        for i = 0, GetNumPartyMembers() do
          if target == UnitName("partypet" .. i) then
            unit = "partypet" .. i
            break
          end
        end
      end

      if not unit then
        for i = 0, GetNumPartyMembers() do
          if target == UnitName("party" .. i) then
            unit = "party" .. i
            break
          end
        end
      end

      if unit then
        local count = 0
        for i = 1, 32 do
          local n, _, _, c = UnitBuff(unit, i)
          if n == Spell_PoM then
            count = c
          end
        end

        self:StartTimer(name, target, count)
      end
    end
    return
  end

  if eventType == "SPELL_HEAL" then
    spellId, spellName, spellSchool, amount, critical = ...
    if spellName == Spell_PoM then
      self:Debug("COMBAT_LOG_EVENT_UNFILTERED - " .. timestamp .. " : " .. eventType .. " / " .. spellName .. " / " .. amount .. " / " .. dstName .. " / " .. dstGUID .. " / " .. dstFlags)
      name = spellName
      target = dstName
      for i = 20, 1, -1 do
        if self.bars[i].target == target and amount ~= nil then
          id = self.bars[i].id
          otext = target .. " (" .. self.bars[i].num .. ")"
          self:SetCandyBarColor(id, "green")
          self:SetCandyBarText(id, otext .. " - " .. amount)
          self:SetCandyBarFade(id, 5, true)
          self:StopCandyBar(id)
          self:StopBar(id)
        end
      end
    end
    return
  end
end

-- ==========================================================================================
-- MendWatch:UNIT_SPELLCAST_SENT
-- ==========================================================================================
function MendWatch:UNIT_SPELLCAST_SENT(p, spell, rank, target)
	if spell == Spell_PoM then	
		self:Debug("UNIT_SPELLCAST_SENT - " .. p .. " / " .. spell .. " / " .. rank .. " / " .. target)
		table.insert(self.captive, 1, {s = spell, t = target, n = 6})
	end
end

-- ==========================================================================================
-- MendWatch:UNIT_SPELLCAST_SUCCEEDED
-- ==========================================================================================
function MendWatch:UNIT_SPELLCAST_SUCCEEDED(p, spell, rank)
	self:Debug("UNIT_SPELLCAST_SUCCEDED - " .. p .. " / " .. spell .. " / " .. rank)

	for idx, captive in pairs(self.captive) do
		if captive.s == spell then
			self:StartTimer(captive.s, captive.t, captive.n)
		end
	end

	self.captive = {}
end

-- ==========================================================================================
-- MendWatch:SpellStopCasting
-- ==========================================================================================
function MendWatch:SpellStopCasting()
	self:Debug("SpellStopCasting")
	self.captive = {}
end

-- ==========================================================================================
-- MendWatch:CreateAnchor
-- ==========================================================================================
function MendWatch:CreateAnchor(text, cRed, cGreen, cBlue)
	local f = CreateFrame("Button", nil, UIParent)
	f:SetWidth(200)
	f:SetHeight(25)

	f.owner = self

	if self.db.profile.barposition.x and self.db.profile.barposition.y and self.db.profile.barposition.anchor and self.db.profile.barposition.point then
		f:ClearAllPoints()
		f:SetPoint(self.db.profile.barposition.point, UIParent, self.db.profile.barposition.anchor, self.db.profile.barposition.x, self.db.profile.barposition.y)
	else
		f:SetPoint("CENTER", UIParent, "CENTER", 0, 50)
	end

	f:SetScript("OnDragStart", function() this:StartMoving() end )
	f:SetScript("OnDragStop",
							function()
								this:StopMovingOrSizing()
								local point, _, anchor, x, y = this:GetPoint()
								this.owner.db.profile.barposition.x = math.floor(x)
								this.owner.db.profile.barposition.y = math.floor(y)
								this.owner.db.profile.barposition.anchor = anchor
								this.owner.db.profile.barposition.point = point
							end)

	f:SetBackdrop({
									bgFile = "Interface/Tooltips/UI-Tooltip-Background",
									edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
									tile = false, tileSize = 16, edgeSize = 16,
									insets = { left = 5, right = 5, top = 5, bottom = 5 }
								})
	f:SetBackdropColor(cRed, cGreen, cBlue, .6)
	f:SetMovable(true)
	f:RegisterForDrag("LeftButton")

	f.Text = f:CreateFontString(nil, "OVERLAY")
	f.Text:SetFontObject(GameFontNormalSmall)
	f.Text:ClearAllPoints()
	f.Text:SetTextColor(1, 1, 1, 1)
	f.Text:SetWidth(200)
	f.Text:SetHeight(25)
	f.Text:SetPoint("TOPLEFT", f, "TOPLEFT")
	f.Text:SetJustifyH("CENTER")
	f.Text:SetJustifyV("MIDDLE")
	f.Text:SetText(text)

	f:Hide()

	return f
end

-- ==========================================================================================
-- MendWatch:ToggleAnchor
-- ==========================================================================================
function MendWatch:ToggleAnchor()
	if self.anchor:IsVisible() then
		self.anchor:Hide()
	else
		self.anchor:Show()
	end
end

-- ==========================================================================================
-- MendWatch:StartTimer
-- ==========================================================================================
function MendWatch:StartTimer(name, target, num)
	local id = name.."-"..num.."-"..target

	-- immediately kill any existing bar
	for i = 20, 1, -1 do
		if self.bars[i].id == id then
			self:SetCandyBarFade(id, 0, false)
			self:StopCandyBar(id)
			self:StopBar(id)
			break
		end
	end
	
	-- find a slot for the bar
	local slot = 0
	for i = 1, 20 do if not self.bars[i].id then slot = i; break end end

	-- setup the bar
	self.bars[slot].id = id
	self.bars[slot].name = name
	self.bars[slot].target = target

	if num == 6 then
		num = 5
		self.bars[slot].initiated = 1
	else
		self.bars[slot].initiated = 0
	end

	self.bars[slot].num = num

	-- do the CandyBar bit
	local text = target .. " (" .. num .. ")"
	local color = self.db.profile.barcolor

	self:RegisterCandyBar(id, 30, text, self.tx, color, color, color, "red")
	self:RegisterCandyBarWithGroup(id, "MendWatch")
	if self.db.profile.bartex then self:SetCandyBarTexture(id, self.textures[self.db.profile.bartex]) end
	if self.db.profile.bgcolor then self:SetCandyBarBackgroundColor(id, self.db.profile.bgcolor, self.db.profile.bgalpha) end
	if self.db.profile.textcolor then self:SetCandyBarTextColor(id, self.db.profile.textcolor) end
	if self.db.profile.textsize then self:SetCandyBarFontSize(id, self.db.profile.textsize) end
	if self.db.profile.barscale then self:SetCandyBarScale(id, self.db.profile.barscale) end
	if self.db.profile.barwidth then self:SetCandyBarWidth(id, self.db.profile.barwidth) end
	if self.db.profile.barheight then self:SetCandyBarHeight(id, self.db.profile.barheight) end
	self:SetCandyBarReversed(id, self.db.profile.reverse)
	self:SetCandyBarFade(id, 5, true)
	self:SetCandyBarCompletion(id, self.StopBar, self, id)
	self:StartCandyBar(id, true)
end

-- ==========================================================================================
-- MendWatch:StopBar
-- ==========================================================================================
function MendWatch:StopBar(id)
	for i = 1, 20 do
		if self.bars[i].id == id then
			self.bars[i] = {}
		end
	end
	for i = 1, 19 do
		if not self.bars[i].id then
			local temp = self.bars[i]
			for j = i + 1, 20 do
				if self.bars[j].id then
					self.bars[i] = self.bars[j]; self.bars[j] = temp; temp = nil
					break
				end
			end
			if temp then break end
		end
	end
end

