--[[ 

title: MarkBar
author: carahuevo
quick-description: Toolbar to set target icon easy and quickly.

This is an adjustable toolbar with raid target icon buttons; quicker and easier than the standard UI target popup menu. It also allows to set icons descriptions (ie sheep, sap,...) and party member assignments in order to dump that info in the chat (not implemented yet).

==Current Features==
  - Raid Marking Toolbar.
  - Scalable with /markbar scale value.
  - Transparency adjustable with /markbar alpha value.
  - Slash command for mark within macros. /markbar mark 0-8
  - Auto show/hide toolbar when join or leaving a party/raid (could be disabled).
  - Establish marks to your partners (select party/raid member and right click a toolbar mark).
==Help==
  - Click right lower corner to drag and reposition the toolbar.
  - Commands:
  - '''/markbar''' -> toogles show/hide toolbar.
  - '''/markbar auto n''' (n=0,1) -> 1=enables autoshow on join/leave party, 0=disables.
  - '''/markbar reset''' -> resets to default settings.
  - '''/markbar alpha n''' (n=0.5 - 1) -> sets toolbar transparency.
  - '''/markbar scale n''' (n=0.5 - 1) -> sets toolbar scale.
  - '''/markbar mark n''' (n=0-8) -> sets mark to current target (useful for use within macros).
  - '''/markbar allow n''' (n=0,1) -> 1=allow other party members to mark, 0=disallow.
  - '''/markbar showtooltip n''' (n=0,1) -> 1=shows tooltip on mouse over mark, 0=doesn't.
==Upcoming Features (sooner or later)== 
  - Profiles with different marks descriptions.
  - Allow other party members to put marks.

==HISTORY==

version 0.1
- Initial release.
version 0.9
- Settings panel (marks description, asignments, bar scale & alpha).
- Custom marks description/assignment for party/raid announcement (ie: Blue square->sheep for Blackmage).
- Some bugs fixed.

@ToDo list:
- Profile management
- Addon comm. to allow other addon users to set marks even if not party leader.
]]

-- Global Vars
MarkBar = {}
MarkBar.title       = "MarkBar";
MarkBar.version     = "1.1.6 int";
MarkBar.author      = "BrainInBlack, original by carahuevo";
--- comm. commands
MarkBar.CMD_PREFIX = "MARKBAR";
MarkBar.CMD_MARK   = "MARK";
MarkBar.CMD_TOKEN  = "#";


-- onload event handle
function MarkBar.OnLoad()
	--register slash commands
	SLASH_MARKBAR1 = "/markbar";
	SlashCmdList["MARKBAR"] = function(msg)
		MarkBar.Command(msg);
	end

	-- events to handle
	this:RegisterEvent("VARIABLES_LOADED");
	this:RegisterEvent("PARTY_MEMBERS_CHANGED");
	this:RegisterEvent("PARTY_LEADER_CHANGED");
	this:RegisterEvent("CHAT_MSG_ADDON");

	-- in order to close with ESC
	tinsert(UISpecialFrames,"MarkBar_SettingsFrame");

	-- MarkBar.Print("Version "..MarkBar.version.." by "..MarkBar.author.." loaded. Type \"/markbar\" for help.");
end

-- When vars are loadad
function MarkBar.OnVariablesLoaded()
	--Saved vars (gets defaults if needed)
	if (not MarkBarOptions) then
		MarkBarOptions = {
			XPos = 300,
			YPos = 200,
			Scale = 0.7,
			Alpha = 1,
			ShowTooltip = 1,
			AutoToogle = 1,
			AllowPartyMark = 0,
			SkipNotAssigned = 1,
			ActiveSet = "Default",
			Debug = 0,
			MarksSets = {
				["Default"] = {
					Name = "Default",
					Mark0 = MarkBar_REMOVEMARK,
					Mark1 = MarkBar_MARK.." 1",
					Mark2 = MarkBar_MARK.." 2",
					Mark3 = MarkBar_MARK.." 3",
					Mark4 = MarkBar_MARK.." 4",
					Mark5 = MarkBar_MARK.." 5",
					Mark6 = MarkBar_MARK.." 6",
					Mark7 = MarkBar_MARK.." 7",
					Mark8 = MarkBar_MARK.." 8"
				}
			}
		}

		if ( not MarkBarOptions.AllowPartyMark ) then
			MarkBarOptions.AllowPartyMark = 0;
		end
		if ( not MarkBarOptions.SkipNotAssigned ) then
			MarkBarOptions.SkipNotAssigned = 1;
		end
		if ( not MarkBarOptions.Debug ) then
			MarkBarOptions.Debug = 0;
		end
	end
	-- initial pos, size, etc
	MarkBar_MainFrame:SetPoint("CENTER", "UIParent", "CENTER", 0 , 0); -- ,MarkBarOptions.XPos, MarkBarOptions.YPos);
	MarkBar_MainFrame:SetScale(MarkBarOptions.Scale);
	MarkBar_MainFrame:SetAlpha(MarkBarOptions.Alpha);

	--Temp vars
	MarkBarVars = {
		closedByUser = nil,
		inGroup = nil,
		TimeSinceLastUpdate = 0,
		MarksAssignments = {
			Mark1 = "",
			Mark2 = "",
			Mark3 = "",
			Mark4 = "",
			Mark5 = "",
			Mark6 = "",
			Mark7 = "",
			Mark8 = ""
		},
		MarksNames = {
			Mark1 = MarkBar_MARKNAME1,
			Mark2 = MarkBar_MARKNAME2,
			Mark3 = MarkBar_MARKNAME3,
			Mark4 = MarkBar_MARKNAME4,
			Mark5 = MarkBar_MARKNAME5,
			Mark6 = MarkBar_MARKNAME6,
			Mark7 = MarkBar_MARKNAME7,
			Mark8 = MarkBar_MARKNAME8
		}
	}
end

-- slash function handle
function MarkBar.Command(msg)
	local cmd, param = MarkBar.ParamParser(msg);

	if ( cmd == "" ) then
		-- by default toggles visibility
		MarkBar.ToggleBar();
	elseif(cmd=="help") then
		MarkBar.ShowHelp();
	elseif(cmd=="reset") then
		MarkBar.Reset();
	elseif(cmd=="auto") then
		MarkBarOptions.AutoToogle = param; --@TODO:show log message of action
	elseif(cmd=="alpha") then
		MarkBarOptions.Alpha = param; --@TODO:show log message of action
		MarkBar_MainFrame:SetAlpha(MarkBarOptions.Alpha);
	elseif(cmd=="showtooltip") then
		MarkBarOptions.ShowTooltip = param; --@TODO:show log message of action
	elseif(cmd=="scale") then
		MarkBarOptions.Scale = param; --@TODO:show log message of action
		MarkBar_MainFrame:SetScale(MarkBarOptions.Scale);
	elseif(cmd=="allow") then --@TODO:show log message of action
		MarkBarOptions.AllowPartyMark = param;
	elseif(cmd=="skip") then --@TODO:show log message of action
		MarkBarOptions.SkipNotAssigned = param;
	elseif(cmd=="debug") then --@TODO:show log message of action
		MarkBarOptions.Debug = param;
	elseif(cmd=="mark") then
		if ( tonumber(param) >=0 and tonumber(param)<=8 ) then
			SetRaidTargetIcon("target", tonumber(param));
			return;
		end
		MarkBar.Print(MarkBar_ERRICONNUMBER);
	else
		MarkBar.Print(MarkBar_ERRUNKNOWNCMD);
		MarkBar.ShowHelp();
	end
end

-- Whenever an event triggers
function MarkBar.OnEvent()
	if ( event == "VARIABLES_LOADED" ) then
		MarkBar.OnVariablesLoaded();
	--Handles party/raid changes to Toggle bar if needed (player must be leader/assistant in order to mark targets)
	elseif ( event == "PARTY_MEMBERS_CHANGED" or event == "PARTY_LEADER_CHANGED" ) then
		local inParty, inRaid = MarkBar.GroupMode();
		MarkBarVars.inGroup = (inParty or inRaid);
		--checks if toolbar was not previously closed by user
		if ( MarkBarOptions.AutoToogle==1 and not MarkBarVars.closedByUser ) then
			if ( (inRaid and (IsRaidLeader() or IsRaidOfficer()) ) 
				  or (inParty and IsPartyLeader()) ) then -- in raid
				MarkBar_MainFrame:Show();
			else
				MarkBar_MainFrame:Hide();
			end
		end
		if (not MarkBarVars.inGroup) then
			MarkBarVars.closedByUser = nil; --resets when leave group
		end
	elseif ( event=="CHAT_MSG_ADDON" and arg1 == MarkBar.CMD_PREFIX and (arg3=="PARTY" or arg3=="RAID") ) then
		MarkBar.CommHandler(arg2, arg4); --message, sender
	end
end

-- reset assignments and some temp vars
function MarkBar.Reset()
	MarkBarVars.closedByUser = nil;
	for i=1,8 do
		MarkBarVars.MarksAssignments["Mark"..i] = "";
	end
end

-- prints command line help
function MarkBar.ShowHelp()
	MarkBar.Print(MarkBar.title.." "..MarkBar.version )
	MarkBar.Print(COLOR_GOLD.."/markbar "..COLOR_END..COLOR_WHITE..MarkBar_HELP1..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar auto n "..COLOR_END..COLOR_WHITE..MarkBar_HELP2..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar reset "..COLOR_END..COLOR_WHITE..MarkBar_HELP3..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar alpha n "..COLOR_END..COLOR_WHITE..MarkBar_HELP4..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar scale n "..COLOR_END..COLOR_WHITE..MarkBar_HELP5..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar mark n "..COLOR_END..COLOR_WHITE..MarkBar_HELP6..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar allow n "..COLOR_END..COLOR_WHITE..MarkBar_HELP7..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar showtooltip n "..COLOR_END..COLOR_WHITE..MarkBar_HELP8..COLOR_END);
	MarkBar.Print(COLOR_GOLD.."/markbar skip n "..COLOR_END..COLOR_WHITE..MarkBar_HELP9..COLOR_END);
end

-- [[ MainFrame ]]

-- Shows/Hides the main bar
function MarkBar.ToggleBar(arg)
	local show;

	if (arg) then
		show = arg;
	else
		if( MarkBar_MainFrame:IsVisible() ) then
			show = 0;
		else
			show = 1;
		end
	end
	
	if( show==1 ) then
		MarkBar_MainFrame:Show();
	else
		MarkBar_MainFrame:Hide();
		if (MarkBarVars.inGroup) then --means closed by user when in group
			MarkBarVars.closedByUser = 1; 
		end
	end
end

function MarkBar.MainFrame_OnUpdate(elapsed)
--[[
	local updateInterval = .1; --this delay fits only for scaling effect
	MarkBarVars.TimeSinceLastUpdate = MarkBarVars.TimeSinceLastUpdate + elapsed; 	
	if (this.TimeSinceLastUpdate > updateInterval) then
		local frame = MarkBar_MainFrame;
		if ( MarkBarVars.Scaling and frame:GetScale()>.3 and frame:GetScale()<1) then --min. scale = 0.3
			-- recalc scale
			--local cursorx, cursory = GetCursorPosition();
			--local deltaX = cursorX - MarkBarVars.CursorX;
			--@TODO:do calculations
			--frame:SetScale(newscale)
		end
		this.TimeSinceLastUpdate = 0 --resets 
	end
]]
end

function MarkBar.SetRaidTargetIcon(icon, id)
	local info = UnitPopupButtons["RAID_TARGET_"..id];
	icon:SetTexture("Interface\\TargetingFrame\\UI-RaidTargetingIcons");
	icon:SetTexCoord(info.tCoordLeft, info.tCoordRight, info.tCoordTop, info.tCoordBottom);
	--icon:SetTexture("Interface\\TargetingFrame\\UI-RaidTargetingIcons")
	--SetRaidTargetIconTexture(icon:GetTexture(),id);
end

-- [[ Mark Button handlers ]] 

function MarkBar.Mark_OnClick(arg1)
	local icon = this:GetID();
	this:SetChecked(0);
	if (arg1 == "LeftButton") then
		if ( UnitExists("target") ) then
			SetRaidTargetIcon("target", icon); -- function defined in Blizz's TargetFrame.lua
			--if not you are not the leader send addon message request to leader
			if ( not IsRaidLeader() and not IsRaidOfficer() ) then
				local cmd = MarkBar.CMD_MARK..MarkBar.CMD_TOKEN..icon; --param=icon
				SendAddonMessage(MarkBar.CMD_PREFIX, cmd, "RAID"); --doesn't matter if in party
			end
		end
	else
		--On right click assigns mark to current target if friendly (@TODO:in party/raid??)
		if( UnitIsFriend("player","target") ) then
			local mark =  "Mark"..icon;
			local textbox, assignlabel, icon = getglobal("MarkBar_Set"..icon.."Description"), getglobal("MarkBar_Set"..icon.."Assignment"), getglobal("MarkBar_Set"..icon.."Icon");
			assignlabel:SetText(UnitName("target"));
			icon:SetAlpha(1.0);
			MarkBarVars.MarksAssignments[mark] = UnitName("target");
			local msg = format(MarkBar_ASSIGNED, MarkBarVars.MarksNames[mark], UnitName("target"));
			if(MarkBarOptions.Debug==0) then
				SendChatMessage(msg, "RAID_WARNING"); -- warns all about the announcement
			else
				MarkBar.Print(msg);
			end
		else
			SetRaidTargetIcon("target", icon); -- function defined in Blizz's TargetFrame.lua
		end
	end
end

function MarkBar.Mark_OnEnter(arg1)
	if ( MarkBarOptions.ShowTooltip == 1 ) then
		local set, mark, assignedTo, color = MarkBarOptions.ActiveSet, "Mark"..this:GetID();
		assignedTo = MarkBarVars.MarksAssignments[mark];
		if (not assignedTo or assignedTo=="" ) then
			color = { r=0.7, g=0.7, b=0.7}; -- gray
			assignedTo = MarkBar_NOTASSIGNED;
		else
			color = { r=1.0, g=1.0, b=1.0};
		end
		GameTooltip:SetOwner(this, "ANCHOR_BOTTOMLEFT");
		GameTooltip:SetText(MarkBarOptions.MarksSets[set][mark]); -- mark description
		GameTooltip:AddLine(assignedTo, color.r, color.g, color.b); -- assignment
		GameTooltip:Show();
	end
end

function MarkBar.MoveHandler_OnMouseDown(arg1)
	if (arg1 == "LeftButton") then
		MarkBar_MainFrame:StartMoving();
	else
		--Start scaling
		--[[@TODO:
		MarkBarVars.ScaleWidth = MarkBar_MainFrame:GetWidth();
		MarkBarVars.Scaling = 1;
		MarkBarVars.CursorX, MarkBarVars.CursorY  = GetCursorPosition();
		]]
	end
end

function MarkBar.MoveHandler_OnMouseUp(arg1)
	if (arg1 == "LeftButton") then
		MarkBar_MainFrame:StopMovingOrSizing();
	else
		MarkBarVars.Scaling = nil;
	end
end

-- [[ Settings Frame ]]

-- load current profile onto settings
function MarkBar.LoadSettings()
	for i=1,8 do
		local set = MarkBarOptions.ActiveSet;
		local mark = "Mark"..i;
		local textbox, assignlabel, icon = getglobal("MarkBar_Set"..i.."Description"), getglobal("MarkBar_Set"..i.."Assignment"), getglobal("MarkBar_Set"..i.."Icon");
		local desc, assignment = MarkBarOptions.MarksSets[set][mark], MarkBarVars.MarksAssignments[mark];
		textbox:SetText(desc);
		assignlabel:SetText(assignment);
		if ( assignment==nil or assignment=="" ) then
			icon:SetAlpha(0.3);
		else
			icon:SetAlpha(1);
		end
	end
	MarkBar_ChkAutoToggle:SetChecked(MarkBarOptions.AutoToogle);
	MarkBar_ChkShowTooltip:SetChecked(MarkBarOptions.ShowTooltip);
	MarkBar_ChkSkipNA:SetChecked(MarkBarOptions.SkipNotAssigned);
	MarkBar_ScaleSlider:SetValue(MarkBarOptions.Scale);
	MarkBar_AlphaSlider:SetValue(MarkBarOptions.Alpha);
end

-- save current profile settings
function MarkBar.SaveSettings()
	for i=1,8 do
		local set, desc = MarkBarOptions.ActiveSet, getglobal("MarkBar_Set"..i.."Description"):GetText();
		MarkBarOptions.MarksSets[set]["Mark"..i] = desc;
	end
	if ( MarkBar_ChkAutoToggle:GetChecked() ) then
		MarkBarOptions.AutoToogle = 1;
	else
		MarkBarOptions.AutoToogle = 0;
	end
	if ( MarkBar_ChkShowTooltip:GetChecked() ) then
		MarkBarOptions.ShowTooltip = 1;
	else
		MarkBarOptions.ShowTooltip = 0;
	end
	if ( MarkBar_ChkSkipNA:GetChecked() ) then
		MarkBarOptions.SkipNotAssigned = 1;
	else
		MarkBarOptions.SkipNotAssigned = 0;
	end
end

function MarkBar.ToggleSettings()
	if ( MarkBar_SettingsFrame:IsVisible() ) then
		MarkBar.SaveSettings();
		MarkBar_SettingsFrame:Hide();
	else
		MarkBarTab_OnClick(1);
		MarkBar.LoadSettings();
		MarkBar_SettingsFrame:Show();
	end
end

function MarkBar.ClearAssign_OnClick()
	local p = this:GetParent();
	local i = p:GetID();
	--MarkBar.Print("DEBUG: "..p:GetID());
	local mark = "Mark"..i;
	MarkBarVars.MarksAssignments[mark] = "";
	local set = MarkBarOptions.ActiveSet;
	local textbox, assignlabel, icon = getglobal("MarkBar_Set"..i.."Description"), getglobal("MarkBar_Set"..i.."Assignment"), getglobal("MarkBar_Set"..i.."Icon");
	assignlabel:SetText("");
	icon:SetAlpha(0.3);
end

function MarkBarScaleSlider_OnValueChanged(value)
	MarkBarOptions.Scale = value;
	MarkBar_MainFrame:SetScale(MarkBarOptions.Scale);
end

function MarkBarAlphaSlider_OnValueChanged(value)
	MarkBarOptions.Alpha = value;
	MarkBar_MainFrame:SetAlpha(MarkBarOptions.Alpha);
end

function MarkBarTab_OnClick(index)
	if ( not index ) then
		index = this:GetID();
	end

	if ( MarkBar_SettingsFrame.numTabs == nil) then
		PanelTemplates_SetNumTabs(MarkBar_SettingsFrame, 2);
	end

	PanelTemplates_SetTab(MarkBar_SettingsFrame, index);
	MarkBar_SettingsFrameMarks:Hide();
	MarkBar_SettingsFrameSettings:Hide();
	if ( index == 1 ) then
		-- Marks Tab
		MarkBar_SettingsFrameSettings:Hide();
		MarkBar_SettingsFrameMarks:Show();
	elseif ( index == 2 ) then
		-- General Settings Tab
		MarkBar_SettingsFrameMarks:Hide();
		MarkBar_SettingsFrameSettings:Show();
	end
end


-- [[ Announce ]]
function MarkBar.Announce()
	-- guess the channel to use (raid or party)
	local inParty, inRaid = MarkBar.GroupMode();
	
	-----@todo:descomentar condicin
	if (not inParty and not inRaid and MarkBarOptions.Debug==0) then
		MarkBar.Print(MarkBar_NOTGROUPED);
		return;
	else
		-- guess proper channel
		local channel;
		if (inRaid) then
			channel = "RAID";
		else
			channel = "PARTY";
		end
		if (MarkBarOptions.Debug==0) then
			SendChatMessage(MarkBar_MARKSANNOUNCETITLE, "RAID_WARNING"); -- warns all about the announcement
		end
		--compose output
		local set, msg = MarkBarOptions.ActiveSet;
		for i=1,8 do
			local mark, name, assignedTo, description;
			mark = "Mark"..i;
			name = MarkBarVars.MarksNames[mark];
			assignedTo = MarkBarVars.MarksAssignments[mark];
			description = MarkBarOptions.MarksSets[set][mark];
			local skip = 0;
			if ( description==nil or description == "" ) then
				--skip = skip+1; 
				description = " ";
			end
			if ( assignedTo==nil or assignedTo == "" ) then
				if ( MarkBarOptions.SkipNotAssigned==1 ) then
					skip = skip+1;
				end
				assignedTo = " ";
			end


			if (skip<1) then
				msg=format("%s: %s >> %s", name, description, assignedTo);
				if (MarkBarOptions.Debug==0) then
					SendChatMessage(msg, channel);
				else
					MarkBar.Print("Debug> "..msg);
				end
			end
		end
	end
end

-- [[ Addon Comms / Message Format:	"COMMAND..TOKEN..PARAM" ]]
function MarkBar.CommHandler( msg, sender )
	if ( sender==UnitName("player") and MarkBarOptions.Debug==0 ) then return; end;
	
	--get sender's unitid 
	local target = MarkBar.GetUnitIdByName(sender);
	local a, b = string.find(msg, "(.+)"..MarkBar.CMD_TOKEN.."(.+)" );
	local cmd = strsub(msg, a);
	local param = strsub(msg, b	);

	if (MarkBarOptions.Debug==1) then
		MarkBar.Print("Debug>msg="..msg);
		MarkBar.Print("Debug>sender="..sender);
		MarkBar.Print("Debug>cmd="..cmd);
		MarkBar.Print("Debug>param="..param);	
		MarkBar.Print("Debug>target="..target);
	end
	if ( cmd==MarkBar.CMD_MARK ) then
		-- marks sender's target (param contains the icon number)
		if ( target and MarkBarOptions.AllowPartyMark==1 ) then
			target = target.."target";
			if ( UnitExists(target) ) then
				SetRaidTargetIcon(target, param);
			end
		end
	end
end

--get UnitId 
function MarkBar.GetUnitIdByName(name)
	local inParty, inRaid = MarkBar.GroupMode();
	local z, gtype, target;
	if ( inRaid ) then
		z = GetNumRaidMembers();
		gtype = "raid";
	elseif ( inParty ) then
		z = GetNumPartyMembers();
		gtype = "party";
	else
		return ""; 
	end
	for i = 1,z do
		if ( UnitName(gtype..i)==name ) then
			return gtype..i;
		end
	end
	return "";
end

-- Parse data after /slash command
function MarkBar.ParamParser(msg)
 	if msg then
 		local a,b,c=strfind(msg, "(%S+)"); --contiguous string of non-space characters
 		if a then
 			return c, strsub(msg, b+2);
 		else	
 			return "";
 		end
 	end
end;

-- print function
function MarkBar.Print (msg)
	DEFAULT_CHAT_FRAME:AddMessage(COLOR_ORANGE..MarkBar.title.."> "..COLOR_END..msg);
end

-- group mode function
function MarkBar.GroupMode()
	--returns notGrouped, inParty, inRaid
	local inRaid = (GetNumRaidMembers() > 0);
	local inParty = (GetNumPartyMembers() > 0);
	return inParty, inRaid; 
end