--[[
	Copyright (C) 2006-2007 Nymbia

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
]]
local L = AceLibrary("AceLocale-2.2"):new("Manufac")
Manufac = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceConsole-2.0", "AceHook-2.1", "AceModuleCore-2.0")
Manufac:RegisterDB("ManufacDB","ManufacPerCharDB")
Manufac.revision = "r"..("$Rev: 55252 $"):match("%d+")
local self = Manufac
local dewdrop = AceLibrary("Dewdrop-2.0")
local BS = AceLibrary("Babble-Spell-2.2")
local stitch = AceLibrary("Stitch-1.1")
local tablet = AceLibrary("Tablet-2.0")
local PT = LibStub("LibPeriodicTable-3.1")
local opentrade
local opencraft
local recentopen
local skills = {}
local professionpoints = {}
local items = {}
local modulelist = {}
local professions = {
	BS["Alchemy"],BS["Engineering"],BS["Enchanting"],BS["Leatherworking"],BS["Tailoring"],BS["Blacksmithing"],
	BS["First Aid"],BS["Cooking"],BS["Jewelcrafting"],BS["Mining"],BS["Poisons"],
}
local diffcolors = {
	["optimal"] = "|cFFFF8844",
	["medium"] = "|cFFFFFF00",
	["easy"] = "|cFF44CC44",
	["trivial"] = "|cFFAAAAAA"
}
local function colors(item)
	if self.db.profile.smoothdiffcolor then
		local diff = PT:ItemInSet(item.link,"Tradeskill.Crafted."..BS:GetReverseTranslation(item.profession))
		if not diff then
			return diffcolors[item.difficulty]
		end
		local max = professionpoints[item.profession]:match("%((%d+)%/")
		local percent = diff/max
		if percent > 1 then
			percent = 1
		end
		if percent >= 0.6666 then
			--RANGE: FF8844 (TOP) to FFFF00 (BOTTOM)
			--  1.000,0.531,0.265 to 1.000,1.000,0.000
			percent = (percent - 0.6666) * 3
			return ("|cff%02x%02x%02x"):format(255,math.floor((1-percent*0.469)*255),math.floor((percent*0.265)*255))
		elseif percent >= 0.3333 then
			--RANGE: FFFF00 (TOP) to 44CC44 (BOTTOM)
			--  1.000,1.000,0.000 to 0.265,0.786,0.265
			percent = (percent - 0.3333) * 3
			return ("|cff%02x%02x%02x"):format((0.265+(percent*0.735))*255,(0.786+(percent*0.214))*255,(0.265-(percent*0.265))*255)
		else
			--RANGE: 44CC44 (TOP) to AAAAAA (BOTTOM)
			--  0.265,0.786,0.265 to 0.664,0.664,0.664
			percent = percent * 3
			return ("|cff%02x%02x%02x"):format((0.664-(percent*0.399))*255,(0.664+(percent*0.122))*255,(0.664-(percent*0.399))*255)
		end
	else
		return diffcolors[item.difficulty]
	end
end
local diffindex = {
	["optimal"] = 1,
	["medium"] = 2,
	["easy"] = 3,
	["trivial"] = 4
}
local function profsort(a,b)
	if self.db.profile.sorttype == L["Alpha"] then
		return a.name < b.name
	else
		if self.db.profile.smoothdiffcolor then
			if not a.link or not b.link then
				return a.name < b.name
			end
			local alvl = PT:ItemInSet(a.link,"Tradeskill.Crafted."..BS:GetReverseTranslation(a.profession))
			local blvl = PT:ItemInSet(b.link,"Tradeskill.Crafted."..BS:GetReverseTranslation(b.profession))
			local adiff = ("%03d"):format(alvl or 375)
			local bdiff = ("%03d"):format(blvl or 375)
			if not adiff or not bdiff or adiff == bdiff then
				return a.name < b.name
			else
				return adiff > bdiff
			end
		else
			if diffindex[a.difficulty] == diffindex[b.difficulty] then
				return a.name < b.name
			else
				return diffindex[a.difficulty] < diffindex[b.difficulty]
			end
		end
	end
end
Manufac.options = {
	type = 'group',
	args = {
		reset = {
			type = 'execute',
			name = L["Reset"],
			desc = L["Reset the profession groups"],
			func = function()
				Manufac:Reset()
				Manufac:Print(L["Reset the profession groups"])
			end,
			order = 501,
		},
		unsorted = {
			type = 'execute',
			name = L["Unsorted"],
			desc = L["Unsorted"],
			func = function()
				Manufac:ListUnsortedRecipes()
			end,
			order = 500,
		},
		sorttype = {
			type = 'text',
			name = L["Sort Type"],
			desc = L["Change between Difficulty and Alphabetical sorting"],
			get = function()
				return Manufac.db.profile.sorttype
			end,
			set = function(v)
				Manufac.db.profile.sorttype = v
				Manufac:Print(L["Sort Type"]..": ["..v.."]")
			end,
			validate = {L["Alpha"], L["Difficulty"]},
			order = 5,
		},
		money = {
			name = L["Money Summary"],
			type = 'toggle',
			desc = L["Toggle money summary"],
			get = function()
				return Manufac.db.profile.money
			end,
			set = function(v)
				Manufac.db.profile.money = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 6,
		},
		menuposition = {
			type = 'text',
			name = L["Position"],
			desc = L["Change popup position behavior"],
			get = function()
				return Manufac.db.profile.position
			end,
			set = function(v)
				Manufac.db.profile.position = v
				Manufac:Position()
				Manufac:Print(L["Position"]..": ["..v.."]")
			end,
			validate = {L["Mouse"], L["Center"], L["Left"], L["Right"], L["Bottom"], L["BottomLeft"],
				L["BottomRight"], L["Top"], L["TopLeft"], L["TopRight"]},
			order = 7,
		},
		offsetx = {
			type = 'text',
			name = L["X Offset"],
			desc = L["Set X Offset"],
			usage = "<number>",
			get = function()
				return Manufac.db.profile.offsetx
			end,
			set = function(newx)
				Manufac.db.profile.offsetx = tonumber(newx)
				Manufac:Position()
				Manufac:Print(L["X Offset"]..": ["..newx.."]")
			end,
			validate = function(value)
				if tonumber(value) then
					return true
				else
					return false
				end
			end,
			order = 8,
		},
		offsety = {
			type = 'text',
			name = L["Y Offset"],
			desc = L["Set Y Offset"],
			usage = "<number>",
			get = function()
				return Manufac.db.profile.offsety
			end,
			set = function(newy)
				Manufac.db.profile.offsety = tonumber(newy)
				Manufac:Position()
				Manufac:Print(L["Y Offset"]..": ["..newy.."]")
			end,
			validate = function(value)
				if tonumber(value) then
					return true
				else
					return false
				end
			end,
			order = 9,
		},
		level2tooltips = {
			name = L["Toggle Level 2 Tooltips"],
			type = 'toggle',
			desc = L["Toggle Level 2 Tooltips"],
			get = function()
				return Manufac.db.profile.l2tooltips
			end,
			set = function(v)
				Manufac.db.profile.l2tooltips = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 12,
		},
		level3tooltips = {
			name = L["Toggle Level 3 Tooltips"],
			type = 'toggle',
			desc = L["Toggle Level 3 Tooltips"],
			get = function()
				return Manufac.db.profile.l3tooltips
			end,
			set = function(v)
				Manufac.db.profile.l3tooltips = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 12,
		},
		canmake = {
			name = L["Show Can Make"],
			type = 'toggle',
			desc = L["Show Can Make"],
			get = function()
				return Manufac.db.profile.canmake
			end,
			set = function(v)
				Manufac.db.profile.canmake = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 13,
		},
		oranges = {
			name = L["Show Oranges"],
			type = 'toggle',
			desc = L["Show Oranges"],
			get = function()
				return Manufac.db.profile.oranges
			end,
			set = function(v)
				Manufac.db.profile.oranges = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 14,
		},
		yellows = {
			name = L["Show Yellows"],
			type = 'toggle',
			desc = L["Show Yellows"],
			get = function()
				return Manufac.db.profile.yellows
			end,
			set = function(v)
				Manufac.db.profile.yellows = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 15,
		},
		icons = {
			name = L["Show Icons"],
			type = 'toggle',
			desc = L["Show Icons"],
			get = function()
				return Manufac.db.profile.icons
			end,
			set = function(v)
				Manufac.db.profile.icons = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 16,
		},
		smoothdiffcolor = {
			name = L["Smooth Difficulty Coloring"],
			type = 'toggle',
			desc = L["Smooth Difficulty Coloring"],
			get = function()
				return Manufac.db.profile.smoothdiffcolor
			end,
			set = function(v)
				Manufac.db.profile.smoothdiffcolor = v
			end,
			map = { [false] = "Disabled", [true] = "Enabled" },
			order = 17,
		},
		header = {
			order = 1000,
			type = 'header',
		},
		modules = {
			name = L["Modules"],
			type = 'group',
			desc = L["Modules"],
			args = {},
			order = 1001,
		},
	}
}
do
	local hasmodule = false
	for i=1,GetNumAddOns() do
		local metadata = GetAddOnMetadata(i, "X-Manufac-Module")
		if metadata and not IsAddOnLoaded(i) then
			local name, _, _, enabled = GetAddOnInfo(i)
			if enabled then
				modulelist[metadata] = true
				Manufac.options.args.modules.args[metadata] = {
					name = metadata,
					type = 'group',
					desc = metadata,
					args = {
						load = {
							name = L["Load"],
							type = 'execute',
							desc = L["Load"],
							func = function()
								LoadAddOn(name)
							end,
						},
					},
				}
			end
			hasmodule = true
		end
	end
	if not hasmodule then
		Manufac.options.args.modules = nil
	end
end
function Manufac:OnInitialize()
	self:RegisterChatCommand({"/manufac", "/manu"}, self.options)
	CreateFrame("Frame", "ManufacFrame")
	ManufacFrame:SetFrameStrata("BACKGROUND")
	ManufacFrame:SetWidth(1)
	ManufacFrame:SetHeight(1)
	ManufacFrame:SetPoint("CENTER",0,0)
	ManufacFrame:Show()
	self:RegisterDefaults("profile", {
		sorttype = L["Alpha"],
		position = L["Mouse"],
		money = true,
		offsetx = 0,
		offsety = 0,
		l2tooltips = true,
		l3tooltips = true,
		canmake = true,
		oranges = false,
		yellows = false,
		groups = {},
		icons = true,
		smoothdiffcolor = false,
	})
end
function Manufac:OnEnable()
	dewdrop:Register(ManufacFrame,
		'children', function(...)
			Manufac:Menu(...)
		end
	)
	self:RegisterEvent("TRADE_SKILL_SHOW")
	self:RegisterEvent("CRAFT_SHOW")
	self:RegisterEvent("SPELLS_CHANGED", "RefreshSkills")
	self:RegisterEvent("SKILL_LINES_CHANGED", "RefreshSkills")
	self:RegisterEvent("BAG_UPDATE", "RefreshMenu")
	self:RegisterEvent("CRAFT_CLOSE")
	self:RegisterEvent("TRADE_SKILL_CLOSE")
	self:RegisterEvent("ADDON_LOADED")
	if IsAddOnLoaded("Blizzard_TradeSkillUI") then
		self:Hook("TradeSkillFrame_Show", true)
	end
	if IsAddOnLoaded("Blizzard_CraftUI") then
		self:Hook("CraftFrame_Show", true)
	end
	self:Position()
	self:ScanActionButtons()
	stitch:EnableDataGathering("Manufac")
	stitch.data = self.db.char
	self.db.char = stitch.data
end
function Manufac:TradeSkillFrame_Show()
end
function Manufac:CraftFrame_Show()
	if GetCraftName() ~= BS["Enchanting"] then
		return self.hooks["CraftFrame_Show"]()
	end
end
function Manufac:OnDisable()
	dewdrop:Unregister(ManufacFrame)
	stitch:DisableDataGathering("Manufac")
end
function Manufac:ADDON_LOADED(addon)
	if addon == "Blizzard_TradeSkillUI" then
		self:Hook("TradeSkillFrame_Show", true)
	elseif addon == "Blizzard_CraftUI" then
		self:Hook("CraftFrame_Show", true)
	end
end
function Manufac:RefreshSkills()
	self:ScanActionButtons()
	self:RefreshMenu()
end
function Manufac:RefreshMenu()
	if dewdrop:IsOpen(FuBarPluginManufacFrameMinimapButton) or dewdrop:IsOpen(FuBarPluginManufacFrame) then
		dewdrop:Refresh()
	end
end
function Manufac:CRAFT_SHOW()
	recentopen = "craft"
	opencraft = GetCraftName()
	local parent = dewdrop:GetOpenedParent()
	if parent and parent ~= ManufacFrame then
		if opencraft == BS["Enchanting"] then
			self:ScanTrade(opencraft)
		end
		return
	end
	if opencraft == BS["Enchanting"] then
		self:ScanTrade(opencraft)
		if self.db.profile.position == L["Mouse"] then
			local x, y = GetCursorPosition()
			ManufacFrame:ClearAllPoints()
			ManufacFrame:SetPoint("BOTTOMLEFT",tonumber(x+(self.db.profile.offsetx or 0)),tonumber(y+(self.db.profile.offsety or 0)))
		else
			self:Position()
		end
		dewdrop:Open(ManufacFrame)
	end
end
function Manufac:TRADE_SKILL_SHOW()
	recentopen = "trade"
	opentrade = GetTradeSkillLine()
	local parent = dewdrop:GetOpenedParent()
	if parent and parent ~= ManufacFrame then
		self:ScanTrade(opentrade)
		return
	end
	self:ScanTrade(opentrade)
	if self.db.profile.position == L["Mouse"] then
		local x, y = GetCursorPosition()
		ManufacFrame:ClearAllPoints()
		ManufacFrame:SetPoint("BOTTOMLEFT",tonumber(x+(self.db.profile.offsetx or 0)),tonumber(y+(self.db.profile.offsety or 0)))
	else
		self:Position()
	end
	dewdrop:Open(ManufacFrame)
end
function Manufac:CRAFT_CLOSE()
	if dewdrop:GetOpenedParent() == ManufacFrame then
		dewdrop:Close()
	end
	opencraft = nil
end
function Manufac:TRADE_SKILL_CLOSE()
	if dewdrop:GetOpenedParent() == ManufacFrame then
		dewdrop:Close()
	end
	opentrade = nil
end
function Manufac:Position()
	ManufacFrame:ClearAllPoints()
	if self.db.profile.position==L["Center"] then
		ManufacFrame:SetPoint("CENTER",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["Left"] then
		ManufacFrame:SetPoint("LEFT",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["Right"] then
		ManufacFrame:SetPoint("RIGHT",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["Bottom"] then
		ManufacFrame:SetPoint("BOTTOM",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["BottomLeft"] then
		ManufacFrame:SetPoint("BOTTOMLEFT",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["BottomRight"] then
		ManufacFrame:SetPoint("BOTTOMRIGHT",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["Top"] then
		ManufacFrame:SetPoint("TOP",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["TopLeft"] then
		ManufacFrame:SetPoint("TOPLEFT",self.db.profile.offsetx,self.db.profile.offsety)
	elseif self.db.profile.position==L["TopRight"] then
		ManufacFrame:SetPoint("TOPRIGHT",self.db.profile.offsetx,self.db.profile.offsety)
	end
end
function Manufac:Reset()
	if not self.db.profile.groups then self.db.profile.groups = {} end
	if self.db.profile.groups[BS["Alchemy"]] then self.db.profile.groups[BS["Alchemy"]] = {} end
	if self.db.profile.groups[BS["Engineering"]] then self.db.profile.groups[BS["Engineering"]] = {} end
	if self.db.profile.groups[BS["Enchanting"]] then self.db.profile.groups[BS["Enchanting"]] = {} end
	if self.db.profile.groups[BS["Leatherworking"]] then self.db.profile.groups[BS["Leatherworking"]] = {} end
	if self.db.profile.groups[BS["Tailoring"]] then self.db.profile.groups[BS["Tailoring"]] = {} end
	if self.db.profile.groups[BS["Blacksmithing"]] then self.db.profile.groups[BS["Blacksmithing"]] = {} end
	if self.db.profile.groups[BS["First Aid"]] then self.db.profile.groups[BS["First Aid"]] = {} end
	if self.db.profile.groups[BS["Cooking"]] then self.db.profile.groups[BS["Cooking"]] = {} end
	if self.db.profile.groups[BS["Jewelcrafting"]] then self.db.profile.groups[BS["Jewelcrafting"]] = {} end
	if self.db.profile.groups[BS["Smelting"]] then self.db.profile.groups[BS["Smelting"]] = {} end
	if self.db.profile.groups[BS["Poisons"]] then self.db.profile.groups[BS["Poisons"]] = {} end
	self.db.profile.offsetx = 0
	self.db.profile.offsety = 0
	self:Position()
end

function Manufac:GetSkillLevel(arg1)
	local skilltext
	for i=1,GetNumSkillLines() do
		local skillName, isHeader, isExpanded, skillRank, numTempPoints, skillModifier, skillMaxRank, isAbandonable, stepCost, rankCost, minLevel, skillCostType, skillDescription = GetSkillLineInfo(i)
		if skillName and skillName == arg1 then
			skilltext = "|cFF7777FF ("..(skillRank+numTempPoints+skillModifier).."/"..skillMaxRank..")|r"
		end
	end
	return skilltext
end
function Manufac:GetPriceData(link)
	--sell, buy, buyout
	if not link then
		return 0,0,0
	end
	if self.db.profile.money==false then return nil, nil, nil end
	if KC_Items and not KC_Items.common then
		local buyout = select(10,AuctionSpy:GetItemInfo(UnitFactionGroup("player"), AuctionSpy:GetLongCode(link)))
		local sell, buy = SellValues:GetPrices(link)
		return sell, buy, buyout
	elseif KC_Items then
		local resultinfo = KC_Items.common:GetItemInfo(KC_Items.common:GetCode(link))
		local sSeen, sAvgStack, sMin, sBidcount, sBid, sBuycount, sBuy = KC_Auction:GetItemData(KC_Items.common:GetCode(link,true))
		return resultinfo["sell"], resultinfo["buy"],sBuy
	elseif Auctioneer and Auctioneer.HistoryDB and Auctioneer.Util and Auctioneer.ItemDB and Informant then
		local itemid = tonumber(link:match("item:(%d+)"))
		local itemTotals = Auctioneer.HistoryDB.GetItemTotals(Auctioneer.ItemDB.CreateItemKeyFromLink(link), Auctioneer.Util.GetAuctionKey())
		local avgBuy=0
		if itemTotals and (itemTotals.buyoutCount > 0) then
			avgBuy = math.floor(itemTotals.buyoutPrice / itemTotals.buyoutCount)
		end
		local venddata = Informant.GetItem(itemid)
		if venddata then
			return venddata.sell, venddata.buy, avgBuy
		else
			return 0, 0, avgBuy
		end
	end
end
function Manufac:GetMoneyText(cash)
	local g,s,c = floor(cash/(100*100)),mod(floor(cash/100),100),mod(floor(cash),100)
	local str=""
	if(g>0) then
		str = format("|cffffd700%sg", g)
	end
	if(s>0) then
		str = str .. format("|cffc7c7cf%ss", s)
	end
	if(c>0) then
		str = str .. format("|cffeda55f%sc", c)
	end
	return str
end
function Manufac:ScanTrade(profession)
	if not self.db.profile.groups then
		self.db.profile.groups = {}
	end
	if not self.db.profile.groups[profession] then
		self.db.profile.groups[profession] = {}
	end
	if profession == BS["Enchanting"] then
		for i=1,GetNumCrafts() do
			local skillname, _, skilltype = GetCraftInfo(i)
			if skilltype~="header" and skillname then
				local link = GetCraftItemLink(i)
				if not self.db.profile.groups[profession][skillname] or self.db.profile.groups[profession][skillname] == L["Unsorted"] then
					local _, set = PT:ItemInSet(link, "Manufac."..profession, true)
					if set then
						self.db.profile.groups[profession][skillname] = set:match("Manufac%.[^%.]+%.(.+)$")
					else
						self.db.profile.groups[profession][skillname] = L["Unsorted"]
					end
				end
			end
		end
	else
		for i=1,GetNumTradeSkills() do
			local skillname, skilltype = GetTradeSkillInfo(i)
			if skilltype~="header" and skillname then
				local link = GetTradeSkillItemLink(i)
				if not self.db.profile.groups[profession][skillname] or self.db.profile.groups[profession][skillname] == L["Unsorted"] then
					local _, set = PT:ItemInSet(link, "Manufac."..profession, true)
					if set then
						self.db.profile.groups[profession][skillname] = set:match("Manufac%.[^%.]+%.(.+)$")
					else
						self.db.profile.groups[profession][skillname] = L["Unsorted"]
					end
				end
			end
		end
	end
end
function Manufac:MakeDBChange(info, oldgroup, text)
	if text ~= L["Options"] then
		self.db.profile.groups[info.profession][info.name] = text
		dewdrop:Refresh(1)
		dewdrop:Refresh(2)
		dewdrop:Refresh(3)
	end
end
function Manufac:GetIDFromLink(link)
	if (not link) then return nil end
	local id = link:match("item:(%d+)")
	if id and tonumber(id) > 0 then
		return id
	else
		local id = link:match("enchant:(%d+)")
		return id
	end
end
function Manufac:ClickItemOne(info)
	if info.profession == BS["Enchanting"] then
		if opencraft == BS["Enchanting"] then
			DoCraft(info.index)
		else
			CastSpell(skills[info.profession],'spell')
		end
	else
		if opentrade == info.profession then
			DoTradeSkill(info.index)
		else
			CastSpell(skills[info.profession],'spell')
		end
	end
end
function Manufac:ClickItemAll(info)
	if opentrade == info.profession then
		DoTradeSkill(info.index, info.numcraftable)
	else
		CastSpell(skills[info.profession],'spell')
	end
end
function Manufac:ClickItemNum(info, num)
	if info.profession == BS["Enchanting"] then
		if opencraft == BS["Enchanting"] then
			DoCraft(info.index, num)
		else
			CastSpell(skills[info.profession],'spell')
		end
	else
		if opentrade == info.profession then
			DoTradeSkill(info.index, num)
		else
			CastSpell(skills[info.profession],'spell')
		end
	end
end
function Manufac:SetTooltip(link)
	GameTooltip_SetDefaultAnchor(GameTooltip, this)
	GameTooltip:SetHyperlink(link)
end
function Manufac:PrintMultiLink(itemtable)
	if ( ChatFrameEditBox:IsVisible() ) then
		ChatFrameEditBox:Insert(GetTradeSkillRecipeLink(itemtable.index))
	else
		ChatFrame_OpenChat(GetTradeSkillRecipeLink(itemtable.index))
	end
end
function Manufac:PrintSingleLink(link)
	if ( ChatFrameEditBox:IsVisible() ) then
		ChatFrameEditBox:Insert(link)
	else
		ChatFrame_OpenChat(link)
	end
end
function Manufac:ScanActionButtons()
	local _, _, _, numSpells = GetSpellTabInfo(1)
	skills = {}
	for h=1,numSpells do
		local spellName = GetSpellName(h, BOOKTYPE_SPELL)
		if spellName == BS["Alchemy"] then
			skills[BS["Alchemy"]] = h
			professionpoints[BS["Alchemy"]] = self:GetSkillLevel(BS["Alchemy"])
		end
		if spellName == BS["Engineering"] then
			skills[BS["Engineering"]] = h
			professionpoints[BS["Engineering"]] = self:GetSkillLevel(BS["Engineering"])
		end
		if spellName == BS["Enchanting"] then
			skills[BS["Enchanting"]] = h
			professionpoints[BS["Enchanting"]] = self:GetSkillLevel(BS["Enchanting"])
		end
		if spellName == BS["Leatherworking"] then
			skills[BS["Leatherworking"]] = h
			professionpoints[BS["Leatherworking"]] = self:GetSkillLevel(BS["Leatherworking"])
		end
		if spellName == BS["Tailoring"] then
			skills[BS["Tailoring"]] = h
			professionpoints[BS["Tailoring"]] = self:GetSkillLevel(BS["Tailoring"])
		end
		if spellName == BS["Blacksmithing"] then
			skills[BS["Blacksmithing"]] = h
			professionpoints[BS["Blacksmithing"]] = self:GetSkillLevel(BS["Blacksmithing"])
		end
		if spellName == BS["First Aid"] then
			skills[BS["First Aid"]] = h
			professionpoints[BS["First Aid"]] = self:GetSkillLevel(BS["First Aid"])
		end
		if spellName == BS["Cooking"] then
			skills[BS["Cooking"]] = h
			professionpoints[BS["Cooking"]] = self:GetSkillLevel(BS["Cooking"])
		end
		if spellName == BS["Jewelcrafting"] then
			skills[BS["Jewelcrafting"]] = h
			professionpoints[BS["Jewelcrafting"]] = self:GetSkillLevel(BS["Jewelcrafting"])
		end
		if spellName == BS["Smelting"] then
			skills[BS["Mining"]] = h
			professionpoints[BS["Mining"]] = self:GetSkillLevel(BS["Mining"])
		end
		if spellName == BS["Poisons"] then
			skills[BS["Poisons"]] = h
			professionpoints[BS["Poisons"]] = self:GetSkillLevel(BS["Poisons"])
		end
		--[[if spellName == BS["Disenchant"] then
			skills[BS["Disenchant"] ] = h
		end]]
	end
end
function Manufac:ListUnsortedRecipes()
	for k,v in pairs(self.db.profile.groups) do
		for a,b in pairs(v) do
			if a and b and b == L["Unsorted"] and stitch:GetItemDataByName(a) then
				self:Print(a,self:GetIDFromLink(stitch:GetItemDataByName(a).link))
			end
		end
	end
end
ManufacFu = AceLibrary("AceAddon-2.0"):new("AceDB-2.0", "FuBarPlugin-2.0")
ManufacFu:RegisterDB("ManufacDB")
function ManufacFu:OnInitialize()
	self.hasIcon = true
	self.hasNoColor = true
	self.hideWithoutStandby = true
	self.defaultMinimapPosition = 200
	self:SetIcon(true)
end
function ManufacFu:OnMenuRequest(arg1, arg2, arg3, arg4, arg5)
	dewdrop:FeedAceOptionsTable(Manufac.options)
end
function ManufacFu:OnTextUpdate()
	if self:IsTextShown() then
		self:SetText("Manufac")
	end
	if self:IsIconShown() then
		self:SetIcon("Interface\\Icons\\INV_Hammer_20")
	else
		self:HideIcon()
	end
end
function ManufacFu:OnEnable()
	self:OnTextUpdate()
end
function ManufacFu:OnTooltipUpdate()
	tablet:SetHint(L["TooltipHint"])
end
function ManufacFu:OnClick()
	if ManufacFu:IsMinimapAttached() then
		dewdrop:Open(FuBarPluginManufacFrameMinimapButton,
		'children', function(...)
			Manufac:PreMenu(...)
		end)
	else
		dewdrop:Open(FuBarPluginManufacFrame,
		'children', function(...)
			Manufac:PreMenu(...)
		end)
	end
end
local skillsort = {}
function Manufac:PreMenu(level, arg1, arg2, arg3, arg4, arg5, arg6)
	if arg1==L["Options"] or arg2==L["Options"] or arg3==L["Options"] or arg4==L["Options"] then
		dewdrop:FeedAceOptionsTable(self.options, 1)
		return
	end
	if not arg1 then
		for k,_ in ipairs(skillsort) do
			skillsort[k] = nil
		end
		if (self:HasModule("Snoop") and self:IsModuleActive("Snoop") and self.db.namespaces.Snoop.profile.showprofessions) or (modulelist.Snoop and not self:HasModule("Snoop")) then
			for _,v in ipairs(professions) do
				table.insert(skillsort,v)
			end
		else
			for k,_ in pairs(skills) do
				table.insert(skillsort,k)
			end
		end
		table.sort(skillsort)
		for _,v in ipairs(skillsort) do
			if skills[v] then
				dewdrop:AddLine(
					'text', v,
					'value', v,
					'func', function()
						CastSpell(skills[v],"spell")
					end,
					'hasArrow', true
				)
			else
				dewdrop:AddLine(
					'text', "|cffaaaaaa"..v.."|r",
					'value', "|SNOOP1|"..v,
					'hasArrow', true
				)
			end
		end
		dewdrop:AddLine()
		dewdrop:AddLine(
			'text', L["Options"],
			'value', L["Options"],
			'hasArrow', true
		)
		return
	end
	level = level - 1
	self:Menu(level, arg1, arg2, arg3, arg4, arg5, true)
end
local result = {}
local added = {}
function Manufac:Menu(level, arg1, arg2, arg3, arg4, arg5, fubarflag)
	if arg1==L["Options"] or arg2==L["Options"] or arg3==L["Options"] or arg4==L["Options"] then
		dewdrop:FeedAceOptionsTable(self.options, 1)
		return
	end
	if arg1 and arg1:sub(1,6) == "|SNOOP" then
		if not ManufacSnoop then
			LoadAddOn("ManufacSnoop")
			if not ManufacSnoop then
				return
			end
		end
		return ManufacSnoop:Menu(level, arg1, arg2, arg3, arg4, arg5)
	end
	if level == 1 then
		if not arg1 then
			if recentopen == "craft" then
				arg1 = opencraft
			elseif recentopen == "trade" then
				arg1 = opentrade
			end
		end
		if not arg1 then
			return
		end
		if not professionpoints[arg1] then
			self:ScanActionButtons()
		end
		if not professionpoints[arg1] then
			self:Print("didnt find skill level for",arg1)
			return
		end
		dewdrop:AddLine(
			'text', arg1..professionpoints[arg1],
			'isTitle', true
		)
		for k,v in pairs(result) do
			result[k] = nil
		end
		for k,v in pairs(added) do
			added[k] = nil
		end
		local haveresult=false
		if not self.db.profile.groups[arg1] then
			dewdrop:AddLine(
				'text', L["Refresh Needed, Click Here"],
				'func', function()
					CastSpell(skills[arg1],'spell')
				end
			)
			return
		end
		for k,v in pairs(self.db.profile.groups[arg1]) do
			if not added[v] then
				table.insert(result, v)
			end
			added[v] = true
		end
		sort(result)
		dewdrop:AddLine()
		local added
		for k,v in pairs(result) do
			dewdrop:AddLine(
				'text', v,
				'value', v,
				'hasArrow', true
			)
			added = true
		end
		if not added then
			dewdrop:AddLine(
				'text', L["Refresh Needed, Click Here"],
				'func', function()
					CastSpell(skills[arg1],'spell')
				end
			)
			return
		end
		if self.db.profile.oranges or self.db.profile.yellows then
			dewdrop:AddLine()
		end
		if self.db.profile.oranges then
			dewdrop:AddLine(
				'text', L["Oranges"],
				'value', "|oranges",
				'hasArrow', true
			)
		end
		if self.db.profile.yellows then
			dewdrop:AddLine(
				'text', L["Yellows"],
				'value', "|yellows",
				'hasArrow', true
			)
		end
		if self.db.profile.canmake then
			dewdrop:AddLine()
			dewdrop:AddLine(
				'text', L["Can Make"],
				'value', "|canmake",
				'hasArrow', true
			)
		end
		if not fubarflag then
			dewdrop:AddLine()
			dewdrop:AddLine(
				'text', L["Options"],
				'value', L["Options"],
				'hasArrow', true
			)
		end
	elseif level == 2 then
		
		if not arg1 then
			return
		end
		for k,v in ipairs(items) do
			items[k] = nil
		end
		if not arg2 then
			if recentopen == "craft" then
				arg2 = opencraft
			elseif recentopen == "trade" then
				arg2 = opentrade
			end
		end
		if arg1=="|canmake" then
			local maxindex
			local haveone=false
			if not self.db.char[arg2] then
				dewdrop:AddLine(
					'text', L["Refresh Needed, Click Here"],
					'func', function()
						CastSpell(skills[arg2],'spell')
					end
				)
				return
			end
			local max = 0
			for k,v in pairs(self.db.char[arg2]) do
				if k > max then
					max = k
				end
			end
			for i=1,max do
				local item = stitch:GetItemDataByIndex(arg2, i)
				if type(item) == "table" and item.numcraftable and item.numcraftable > 0 then
					haveone=true
					table.insert(items, item)
				end
			end
			if haveone==false then
				dewdrop:AddLine(
					'text', L["Cannot Create Anything"],
					'isTitle', true
				)
				return
			end
			table.sort(items,profsort)
			for k,v in ipairs(items) do
				self:AddItemLine(v)
			end
			return
		elseif arg1=="|yellows" then
			local maxindex
			local haveone=false
			if not self.db.char[arg2] then
				dewdrop:AddLine(
					'text', L["Refresh Needed, Click Here"],
					'func', function()
						CastSpell(skills[arg2],'spell')
					end
				)
				return
			end
			local max = 0
			for k,v in pairs(self.db.char[arg2]) do
				if k > max then
					max = k
				end
			end
			for i=1,max do
				local item = stitch:GetItemDataByIndex(arg2, i)
				if type(item) == "table" and item.difficulty == "medium" then
					haveone=true
					table.insert(items, item)
				end
			end
			if haveone==false then
				dewdrop:AddLine(
					'text', L["Cannot Create Anything"],
					'isTitle', true
				)
				return
			end
			table.sort(items,profsort)
			for k,v in ipairs(items) do
				self:AddItemLine(v)
			end
			return
		elseif arg1=="|oranges" then
			local maxindex
			local haveone=false
			if not self.db.char[arg2] then
				dewdrop:AddLine(
					'text', L["Refresh Needed, Click Here"],
					'func', function()
						CastSpell(skills[arg2],'spell')
					end
				)
				return
			end
			local max = 0
			for k,v in pairs(self.db.char[arg2]) do
				if k > max then
					max = k
				end
			end
			for i=1,max do
				local item = stitch:GetItemDataByIndex(arg2, i)
				if type(item) == "table" and item.difficulty == "optimal" then
					haveone=true
					table.insert(items, item)
				end
			end
			if haveone==false then
				dewdrop:AddLine(
					'text', L["Cannot Create Anything"],
					'isTitle', true
				)
				return
			end
			table.sort(items,profsort)
			for k,v in ipairs(items) do
				self:AddItemLine(v)
			end
			return
		end
		if not self.db.profile.groups[arg2] then
			dewdrop:AddLine(
				'text', L["Refresh Needed, Click Here"],
				'func', function()
					CastSpell(skills[arg2],'spell')
				end
			)
			return
		end
		for k,v in pairs (self.db.profile.groups[arg2]) do
			if v == arg1 then
				local info = stitch:GetItemDataByName(k)
				table.insert(items,info)
			end
		end
		table.sort(items,profsort)
		local madeline
		for k,v in ipairs(items) do
			self:AddItemLine(v)
			madeline = true
		end
		if not madeline then
			dewdrop:AddLine(
				'text', L["Refresh Needed, Click Here"],
				'func', function()
					CastSpell(skills[arg2],'spell')
				end
			)
			return
		end
	elseif level == 3 then
		if not arg1 then
			return
		end
		if not arg3 then
			if recentopen == "craft" then
				arg3 = opencraft
			elseif recentopen == "trade" then
				arg3 = opentrade
			end
		end
		local info = stitch:GetItemDataByName(arg1)
		if not info then
			dewdrop:AddLine(
				'text', L["Refresh Needed, Click Here"],
				'func', function()
					CastSpell(skills[arg3],'spell')
				end
			)
			return
		end
		if arg2=="|canmake" or arg2 == "|yellows" or arg2 == "|oranges" then
			arg2=self.db.profile.groups[arg3][info.name]
		end
		local totalprice = 0
		local displaytotalprice = true
		if info.profession == BS["Enchanting"] then
			local texture
			if self.db.profile.icons then
				texture = info.texture
			end
			local skilllvl = PT:ItemInSet(info.link,"Tradeskill.Crafted."..BS:GetReverseTranslation(info.profession))
			if skilllvl then
				skilllvl = " ("..skilllvl..")"
			else
				skilllvl = ""
			end
			if self.db.profile.l3tooltips then
				dewdrop:AddLine(
					'text', (diffcolors[info.difficulty]..info.name.."|r ["..info.numcraftable.."/"..info.numcraftablewbank.."]"..skilllvl),
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2', info.link,
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', (diffcolors[info.difficulty]..info.name.."|r ["..info.numcraftable.."/"..info.numcraftablewbank.."]"..skilllvl),
					'tooltipFunc', nil,
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'icon', texture
				)
			end
			dewdrop:AddLine()
			if opencraft == info.profession then
				dewdrop:AddLine(
					'text', "|cFF00FF00"..L["Cast"],
					'func', self.ClickItemOne,
					'arg1', self,
					'arg2', info
				)
			else
				dewdrop:AddLine(
					'text', "|cFF00FF00"..L["Open Profession to Make"],
					'func', function()
						CastSpell(skills[info.profession],'spell')
					end
				)
			end
			dewdrop:AddLine()
			if info.tools then
				dewdrop:AddLine(
					'text', info.tools,
					'notClickable', true
				)
			end
			--Note, if enchants ever get cooldowns, that goes here.
		else
			local texture
			if self.db.profile.icons then
				texture = info.texture
			end
			local skilllvl = PT:ItemInSet(info.link,"Tradeskill.Crafted."..BS:GetReverseTranslation(info.profession))
			if skilllvl then
				skilllvl = " ("..skilllvl..")"
			else
				skilllvl = ""
			end
			if self.db.profile.l3tooltips then
				dewdrop:AddLine(
					'text', (diffcolors[info.difficulty]..info.name.."|r ["..info.numcraftable.."/"..info.numcraftablewbank.."]"..skilllvl),
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2', info.link,
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', (diffcolors[info.difficulty]..info.name.."|r ["..info.numcraftable.."/"..info.numcraftablewbank.."]"..skilllvl),
					'tooltipFunc', nil,
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'icon', texture
				)
			end
			if self.db.profile.money then
				local sell, buy, buyout = self:GetPriceData(info.link)
				if buyout and buyout > 0 then
					buyout = buyout*info.nummade
					dewdrop:AddLine(
						'text', ("Market Value: "..self:GetMoneyText(buyout)),
						'isTitle', true
					)
				end
				if sell and sell > 0 then
					sell = sell*info.nummade
					dewdrop:AddLine(
						'text', ("Vendor Value: "..self:GetMoneyText(sell)),
						'isTitle', true
					)
				end
			end
			dewdrop:AddLine()
			if opentrade == info.profession then
				dewdrop:AddLine(
					'text', "|cFF00FF00"..string.format(L["Create %s"],info.nummade),
					'func', self.ClickItemOne,
					'arg1', self,
					'arg2', info
				)
				dewdrop:AddLine(
					'text', "|cFF00FF00"..L["Create X"],
					'hasArrow', true,
					'hasEditBox', true,
					'editBoxFunc', function(a,b) self:ClickItemNum(a,b) end,
					'editBoxArg1', info
				)
				dewdrop:AddLine(
					'text', "|cFF00FF00"..L["Create All"],
					'func', self.ClickItemAll,
					'arg1', self,
					'arg2', info
				)
			else
				dewdrop:AddLine(
					'text', "|cFF00FF00"..L["Open Profession to Make"],
					'func', function()
						CastSpell(skills[info.profession],'spell')
					end
				)
			end
			dewdrop:AddLine()
			if info.tools then
				dewdrop:AddLine(
					'text', info.tools,
					'notClickable', true
				)
			end
			local cooldown=GetTradeSkillCooldown(info.index)
			if cooldown ~= nil and cooldown >0 and opentrade == arg3 then
				dewdrop:AddLine(
					'text', "|cFFFF3333"..SecondsToTime(cooldown).."|r",
					'notClickable', true
				)
			end
		end
		for i=1,#info do
			local h=i
			local price
			if self.db.profile.money==true then
				local sell, buy, buyout = self:GetPriceData(info[i]["link"])
				if buy and buy > 0 and (((not buyout) or buyout == 0) or (not Auctioneer)) then
					price = buy*info[i]["needed"]
					totalprice = price + totalprice
					price = self:GetMoneyText(price)
				elseif buyout and buyout > 0 then
					price = buyout*info[i]["needed"]
					totalprice = price + totalprice
					price = self:GetMoneyText(price)
				else
					price = L["Unknown"]
					displaytotalprice = false
				end
			else
				price = ""
				displaytotalprice=false
			end
			self:AddItemReagentLine(info[i],price)
		end
		if displaytotalprice then
			totalprice = self:GetMoneyText(totalprice)
			dewdrop:AddLine(
				'text', ("Mats Cost: "..totalprice),
				'isTitle', true
			)
		end
		dewdrop:AddLine()
		dewdrop:AddLine(
			'text', ("|cFFFFFF00"..L["LinkResult"].."|r"),
			'func', self.PrintSingleLink,
			'arg1', self,
			'arg2', info.link
		)
		if opentrade == info.profession then
			dewdrop:AddLine(
				'text', ("|cFFFFFF00"..L["LinkRecipe!"].."|r"),
				'func', self.PrintMultiLink,
				'arg1', self,
				'arg2', info
			)
		end
		dewdrop:AddLine()
		dewdrop:AddLine(
			'text', L["Grouping"],
			'hasArrow', true,
			'hasEditBox', true,
			'editBoxFunc', function(item,group,text)
				self:MakeDBChange(item,group,text)
			end,
			'editBoxArg1', info,
			'editBoxArg2', arg2
		)
	elseif level == 4 then
		if not arg1 then
			return
		end
		if not arg4 then
			if recentopen == "craft" then
				arg4 = opencraft
			elseif recentopen == "trade" then
				arg4 = opentrade
			end
		end
		--Note, this will explode if an enchant ever uses a mat created by enchanting, ie a mana oil
		local info = stitch:GetItemDataByName(arg1)
		local group = self.db.profile.groups[arg4][arg1]
		if type(info) ~= "table" then
			return
		end
		dewdrop:AddLine(
			'text', "|cFF00FF00"..string.format(L["Create %s"],info.nummade),
			'func', self.ClickItemOne,
			'arg1', self,
			'arg2', info
		)
		dewdrop:AddLine(
			'text', "|cFF00FF00"..L["Create X"],
			'hasArrow', true,
			'hasEditBox', true,
			'editBoxFunc', function(a,b) self:ClickItemNum(a,b) end,
			'editBoxArg1', info
		)
		dewdrop:AddLine(
			'text', "|cFF00FF00"..L["Create All"],
			'func', self.ClickItemAll,
			'arg1', self,
			'arg2', info
		)
		dewdrop:AddLine()
		for i=1,#info do
			self:AddItemReagentLine(info[i])
		end
	end
end
function Manufac:AddItemLine(info)
	local texture
	if self.db.profile.icons then
		texture = info.texture
	end
	if info.numcraftablewbank > 0 then
		if self.db.profile.l2tooltips then
			dewdrop:AddLine(
				'text', (colors(info)..info.name.."|r ["..info.numcraftable.."/"..info.numcraftablewbank.."]"),
				'value', info.name,
				'hasArrow', true,
				'tooltipFunc', self.SetTooltip,
				'tooltipArg1', self, 'tooltipArg2', info.link,
				'icon', texture
			)
		else
			dewdrop:AddLine(
				'text', (colors(info)..info.name.."|r ["..info.numcraftable.."/"..info.numcraftablewbank.."]"),
				'value', info.name,
				'hasArrow', true,
				'tooltipFunc',nil,
				'icon', texture
			)
		end
	else
		if self.db.profile.l2tooltips then
			dewdrop:AddLine(
				'text', (colors(info)..info.name.."|r"),
				'value', info.name,
				'hasArrow', true,
				'tooltipFunc', self.SetTooltip,
				'tooltipArg1', self, 'tooltipArg2', info.link,
				'icon', texture
			)
		else
			dewdrop:AddLine(
				'text', (colors(info)..info.name.."|r"),
				'value', info.name,
				'hasArrow', true,
				'tooltipFunc',nil,
				'icon', texture
			)
		end
	end
end
function Manufac:AddItemReagentLine(info,price)
	local texture
	if self.db.profile.icons then
		texture = info.texture
	end
	if not price then
		price = ""
	end
	local inprof=false
	local reagentdata = stitch:GetItemDataByName(info.name)
	if reagentdata then
		inprof=reagentdata.index
	end
	if info.vendor then
		if tonumber(inprof) then
			if self.db.profile.l3tooltips==true then
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(vendor)"].." "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2', info.link,
					'hasArrow', true,
					'value', info.name,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(vendor)"].." "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', nil,
					'hasArrow', true,
					'value', info.name,
					'icon', texture
				)
			end
		else
			if self.db.profile.l3tooltips==true then
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(vendor)"].." "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2',info.link,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(vendor)"].." "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', nil,
					'icon', texture
				)
			end
		end
	elseif info.numwbank > info.num then
		if tonumber(inprof) then
			if self.db.profile.l3tooltips==true then
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(bank: "]..info.numwbank - info.num..") "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2', info.link,
					'hasArrow', true,
					'value', info.name,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(bank: "]..info.numwbank - info.num..") "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', nil,
					'hasArrow', true,
					'value', info.name,
					'icon', texture
				)
			end
		else
			if self.db.profile.l3tooltips==true then
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(bank: "]..info.numwbank - info.num..") "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2', info.link,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..L["(bank: "]..info.numwbank - info.num..") "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', nil,
					'icon', texture
				)
			end
		end
	else
		if tonumber(inprof) then
			if self.db.profile.l3tooltips==true then
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2', info.link,
					'hasArrow', true,
					'value', info.name,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', nil,
					'hasArrow', true,
					'value', info.name,
					'icon', texture
				)
			end
		else
			if self.db.profile.l3tooltips==true then
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', self.SetTooltip,
					'tooltipArg1', self, 'tooltipArg2', info.link,
					'icon', texture
				)
			else
				dewdrop:AddLine(
					'text', ("|cFF6666FF"..info.name.."|r ["..info.num.."/"..info.needed.."] "..price),
					'func', self.PrintSingleLink,
					'arg1', self,
					'arg2', info.link,
					'tooltipFunc', nil,
					'icon', texture
				)
			end
		end
	end
end
