----------------------------------------------------------------------------------------
-- M1Peng: enhances BG minimap (by Kamiikatze, Baelgun-EU)
----------------------------------------------------------------------------------------

local strgsub = string.gsub        
local strformat = string.format

local currentPlayer       -- player name
local currentlyHorde      -- nil or true
local currentBG           -- nil or "WS","AB","AV","ES"

local colozPfx = {}       -- which colors for Ally/Horde/Enemy
local colozMsg = {}       -- which bg messages to re-colorize
local finfoTab = {}       -- table[frameID]=frame's info

local Geti,Geti4Say       -- forward declarations (assigned with functions below)


----------------------------------------------------------------------------------------
-- Load 1 of the localization tables (OnLoad will check for error below):
--
-- Before this file is loaded, the locafiles fill global M1PengLocas[] with locatables:
-- M1Peng_deDE.lua inserts M1PengLocas["deDE"]={german locatable, with subtables}
-- M1Peng_enUS.lua inserts M1PengLocas["enUS"]={english locatable, with subtables}
-- ...
-- We load 1 of this locatables in our local variable LT (local is faster) and free the
-- whole M1PengLocas (frees memory resources).
----------------------------------------------------------------------------------------

local LT=M1PengLocas[GetLocale()]; M1PengLocas=nil


----------------------------------------------------------------------------------------
-- Colorize minimap's playericons for raid (colours by group, from first group==off==red 
-- to last group==deff==blue; blinking for attacked players). 
-- We don't create own player icons, we simply change WOW's player icon textures and let
-- WOW do the positioning. This is called every second by OnUpdate (see end of file).
----------------------------------------------------------------------------------------

local picoblink=0   -- toggles 0<-->1 on each call (every second)
local picosTab={}   -- save texture we set, to avoid unnecessary SetTexture() calls
local picosFL0      -- basic framelevel for player icons, need to be queried once


local function SetRaidiPico(i,g)
   local ui,infight="raid"..i,0
   -- if raid slot is empty, healthmax is 0
   local hp=UnitHealthMax(ui); if hp<1 then return end
   -- UnitAffectingCombat is nil, when out of range, so check by health percent
   if UnitAffectingCombat(ui)
   or (UnitHealth(ui)/hp<0.9 and not UnitIsDeadOrGhost(ui)) 
      then infight=1 g=g+picoblink end
   -- If infight, we set player icon 1 framelevel higher (should not be covered
   -- by non-infight player icons) and load texture 'GB' instead of 'G0' (where
   -- B is 0 or 1, it toggles every second, and G is 1,3,5,7 for the colors red
   -- orange,yellow,blue). Texture 'G1' has same color as 'G0', but is brighter
   -- which causes the blink effect for infight players.
   -- To avoid unecesary SetTexture calls (time consuming?) we save the texture
   -- we set in picosTab[], which is emptied in each MMMUpdate().
   local picf=getglobal("BattlefieldMinimapRaid"..i)
   local pico=getglobal("BattlefieldMinimapRaid"..i.."Icon")
   if not pico then return end
   if not picosFL0 then picosFL0=picf:GetFrameLevel() end  -- query basic framelevel once
   picf:SetFrameLevel(picosFL0+infight)
   if picosTab[i]~=g then -- SetTexture only, if there is a change
      pico:SetTexture("Interface\\AddOns\\M1Peng\\M1PengPico"..g)
      picosTab[i]=g end   -- Save the texture we set
   end;

local function SetRaidPicos()   -- called every second by OnUpdate (end of file)
   picoblink=1-picoblink        -- toggle 0<-->1 for blink
   -- 15 player bg: group1=red, group2=yellow, group3=blue, others=gray
   if currentBG=="AB" or currentBG=="ES" then
      for i=1,5 do SetRaidiPico(i,10) end
      for i=6,10 do SetRaidiPico(i,50) end
      for i=11,15 do SetRaidiPico(i,70) end
      for i=16,40 do SetRaidiPico(i,90) end
   -- 10 player bg: group1=red, group2=blue, others=gray
   elseif currentBG=="WS" then
      for i=1,5 do SetRaidiPico(i,10) end
      for i=6,10 do SetRaidiPico(i,70) end
      for i=11,40 do SetRaidiPico(i,90) end
   -- 40 player av: 
   -- group1/2 = 1st off (relief hut) = red
   -- group3/4 = 2nd off (frostwolf towers) = orange
   -- group5/6 = off/def (galvangar, retap bunkers) = yellow
   -- group7/8 = def (base/bunkers) = blue 
   elseif currentBG=="AV" then
      for i=01,10 do SetRaidiPico(i,10) end
      for i=11,20 do SetRaidiPico(i,30) end
      for i=21,30 do SetRaidiPico(i,50) end
      for i=31,40 do SetRaidiPico(i,70) end
      end
   end;


----------------------------------------------------------------------------------------
-- Parse faction's points from WorldState string, i=1/2 for ally/horde
----------------------------------------------------------------------------------------

local function GetABPoints(i)       -- returns: resource points (0-2000), nil for N/A
   local _,_,p = GetWorldStateUIInfo(i)
   if p then _,_,p=strfind(p,".+:.+: (%d+)/2000")
   if p then return tonumber(p) end end
   end;

local function GetAVPoints(i)       -- returns: reinforcement (0-600), nil for N/A
   local _,_,p = GetWorldStateUIInfo(i)
   if p then _,_,p=strfind(p,".+: (%d+)") 
   if p then return tonumber(p) end end
   end;

local function GetESPoints(i)       -- returns: winpoints (0-2000), nil for N/A
   local _,_,p = GetWorldStateUIInfo(i+1)    -- WorldState(1) is a captionbar in ES
   if p then _,_,p=strfind(p,".+:.+: (%d+)/2000")
   if p then return tonumber(p) end end
   end;

local function HasWSFlagPicked(i)   -- returns: has enemy flag? (0=no, 1=yes)
   local _,f=GetWorldStateUIInfo(i) return f-1
   end; 


----------------------------------------------------------------------------------------
-- Apply a replacement table={"oldtex1","newtex1","oldtex2","newtex2",...} to text.
-- Cannot use strgsub with table, since ORDER in table is significant (do "old1"="new1"
-- before "old2"=new2" before "old3"="new3" ...)
----------------------------------------------------------------------------------------

local function strtsub(tex,tab)
   for i=1,#tab,2 do tex=strgsub(tex,tab[i],tab[i+1]) end
   return tex
end;


----------------------------------------------------------------------------------------
-- Say a (maybe multilined) message to /bg chat.
-- A grammar table applies grammar rules, and last beautifications... With MyPengPlus, 
-- user can extend that, to modify messages or to add own texts (see MyPeng.lua). 
----------------------------------------------------------------------------------------

local html2utf8 = {            -- Can use html-entities in MyPengPlus, for [[ ]] strings
   "&Auml;","\195\132", "&auml;","\195\164",
   "&Ouml;","\195\150", "&ouml;","\195\182",
   "&Uuml;","\195\156", "&uuml;","\195\188",
   "&szlig;","\195\159" }

local myplusTab = {}       -- replacings for M1Peng's messages
local myjokeTab = {}       -- replacings for WOW's bg messages

function MyPengPlus(oldtex,newtex,magix)  -- inserts oldtex,newtex at end (tab order!)
    oldtex=strtsub(oldtex,html2utf8)
    newtex=strtsub(newtex,html2utf8)
    if not magix then                     -- escape magic chars, unless user wants them
       oldtex=strgsub(oldtex,"[%^%$%(%)%%%.%[%]%*%+%-%?]","%%%0") 
       newtex=strgsub(newtex,"%%","%%%0")
    elseif magix=="JOKE" then
       table.insert(myjokeTab,oldtex)
       table.insert(myjokeTab,newtex)
       return end 
    table.insert(myplusTab,oldtex)
    table.insert(myplusTab,newtex)
end;


local function BGSay(msg,cha,ply)
   -- MyPengPlusser after PLAY_ENT_WORLD has incorrect GetGuildInfo() sometimes, so
   -- we do it now (myplusTab is used here only). Call MyPengPlusser ONCE and free!
   if MyPengPlusser then MyPengPlusser() MyPengPlusser=nil end

-- cha="WHISPER"
   if not cha then cha="BATTLEGROUND" end               -- default output is to /bg
   if not ply then ply=currentPlayer                    -- default /w is to myself,
      local uname,realm=UnitName("target")              -- unless, we have a target
      if uname then ply=uname                           -- uname, if target exists
      if realm then ply=uname.."-"..realm end           -- realm, if not our realm
      end end
   msg=strtsub(msg,LT.Grammar4Say)                      -- apply 2-click grammatics
   msg=strtsub(msg,myplusTab)                           -- apply user modifications

   local p0,p1=1
   while (true) do                                      -- split multilined messages... 
      p1=strfind(msg,"\n",p0,true)
      if not p1 then break end
      SendChatMessage(strsub(msg,p0,p1-1),cha,nil,ply)  -- SendChatMsg("") does nothing
      p0=p1+1 end
   SendChatMessage(strsub(msg,p0),cha,nil,ply)          -- SendChatMsg("") does nothing
   end;


----------------------------------------------------------------------------------------
-- Create frames (for buttons/zones/timers) in BGMinimap
-- buttons: TYPE=1, IDs=101++, FRAMEs=M1PengF101++, TEMPLATE=M1PengF1 in M1Peng.xml
-- zones:   TYPE=2, IDs=201++, FRAMEs=M1PengF201++, TEMPLATE=M1PengF2 in M1Peng.xml
-- timers:  TYPE=3, IDs=301++, FRAMEs=M1PengF301++, TEMPLATE=M1PengF3 in M1Peng.xml
-- MMMUpdate() inits useFid={100,200,300}, so that ++(useFid[TYPE]) gives the ID to use
-- for next frame of type TYPE. Since there is no DeleteFrame(),we reuse old frames and 
-- hide unused old frames (see MMMUpdate).
----------------------------------------------------------------------------------------

local useFid                                           -- MMMUpdate sets {100,200,300}

local useFty={"button","button","frame"}
local function MMMF(u)
   useFid[u]=useFid[u]+1; local n="M1PengF"..useFid[u] -- get next ID for frame type u
   -- reuse, if frame M1PengF(ID) already exists, otherwise create frame M1PengF(ID)
   local f=getglobal(n) or CreateFrame(useFty[u],n,BattlefieldMinimap,"M1PengF"..u)
   return f,useFid[u]   -- return framehandle and ID
   end;

local function MMMPlace(f,l,t,w,h)                     -- position frame on BG minimap
   f:SetWidth(w) f:SetHeight(h) f:SetPoint("TOPLEFT",BattlefieldMinimap,"TOPLEFT",l,-t)
   end;

local function MMMBuNext() return useFid[1]+1 end      -- ID, the next button would have


----------------------------------------------------------------------------------------
-- Buttons (TYPE=1, IDs=101++)
--
-- The M1PengF1N.blp, M1PengF1P.blp, M1PengF1H.blp textures are 512x32 pixel stripes,
-- where the minibutton's pictures are in. To create a button on minimap, we say, where
-- the button's picture lies in minitexture (u=0..511, w=width in texels) and where the
-- button should be placed on bg minimap (l=left, t=top). For zoom=1.0, button's width
-- is w and height is 32.
-- In our frame-info-table we save finfoTab[id]=txt, where txt is the button text (from 
-- LT.GenButex or LT.##Butex, where ##=AB,AV,ES,WS). 
-- Some buttons say a message to /bg on click (1-click-messages); other buttons produce
-- messages in the 2-click-mechanism:  1st click the button (WHAT...), 2nd click a zone
-- (...WHERE) produces a "WHAT is WHERE" message.
----------------------------------------------------------------------------------------

local function MMMBu(txt,u,w,l,t,zoom)    -- MakeMyMinimapButton
   if not zoom then zoom=1 end 
   local f,i=MMMF(1); MMMPlace(f,l,t,w*zoom,32*zoom)
   f:GetNormalTexture():SetTexCoord(u/512,(u+w)/512,0,1)
   f:GetPushedTexture():SetTexCoord(u/512,(u+w)/512,0,1)
   f:GetHighlightTexture():SetTexCoord(u/512,(u+w)/512,0,1)
   finfoTab[i]=txt f:UnlockHighlight() 
   f:Show() f:SetID(i) return i
   end;


----------------------------------------------------------------------------------------
-- Zones (TYPE=2, IDs=201++)
--
-- Zones are white rectangular buttons (almost transparent, getting brighter when hove-
-- red or pushed). 
-- In WS, we have great zones: player icons will not cover the whole zone and zone will
-- be clickable, and we lay the zone in framelevel  UNDER the player icons  (so tooltip
-- when hovering player icons will work).
-- In ES,AB,AV we have small zones at the POIs (bases, towers...): player icons may co-
-- ver the whole zone and make it unclickable, so we lay zone OVER the player icons and
-- integrate player names in zone's tooltip  (zone buttons will fade-in only when hove-
-- red).
-- Zones produce messages in the 2-click-mechanism:  1st click on button (WHAT...), 2nd 
-- click on zone (...WHERE) produces a "WHAT is WHERE" message.
-- The location of zones is also saved in our frame-info-table, so GetMyZone() can find
-- the zone, the player is currently in (needed for "help to me!" function).
--
-- Each zone (IDs=201++) has an entry e in finfoTab[ID] in the following format:
--
-- e[1] = txt = zone's name (used for 2-click-messages and zone's tooltip)
-- e[2,3,4,5] = x,y,r,nil = a circular zone at center x,y with radius r (in minimap coords)
-- e[6] = timerID or nil (see timers section)
--
-- or:
--
-- e[1] = txt = zone's name (used for 2-click-messages and zone's tooltip)
-- e[2,3,4,5] = l,t,r,b = a rectangular zone (left,top,right,bottom in minimap coords)
-- e[6] = timerID or nil (see timers section)
--
-- or:
--
-- e[1] = txt = zone's name (used for 2-click-messages and zone's tooltip)
-- e[2,3,4,5] = nil,nil,nil,nil = a zone without area (WS: roof's area is in base's area)
-- e[6] = timerID or nil (see timers section)
-- 
-- In WS, we have a button for "roof" but no zone area for roof, only area for "base".
----------------------------------------------------------------------------------------

local zonesLevup                          -- controls overlay or underlay of zones
local zonesAlfalo,zonesAlfa,zonesAlfade   -- control fade-in and fade-out of zones

local function MMMZo(txt,l,t,r,b,zo1,zo2,zo3,zo4,tid)   -- MakeMyMinimapZone
   local w,h=r-l,b-t
   local f,i=MMMF(2); MMMPlace(f,l,t,w,h)
   f:SetFrameLevel(BattlefieldMinimap:GetFrameLevel()+zonesLevup)
   if zo1 then finfoTab[i]={txt,zo1,zo2,zo3,zo4} else finfoTab[i]={txt} end 
   if tid then finfoTab[i][6]=tid end
   f:SetAlpha(zonesAlfalo) f:UnlockHighlight() 
   f:Show() f:SetID(i) return i
   end;


-- This is a ipairs(finfoTab), which starts at finfoTab[201], to iterate zones.
-- Skips entries for zones with no zone area assigned.

local function zonesIta()
   return function(t,i) 
      repeat i=i+1 if not t[i] then return end until t[i][2]
      return i,t[i]
      end,
   finfoTab,200
   end;


-- Get player's current zone:
-- If inside a rectangular zone, this is a match. Otherwise, select nearest circular
-- zone. Returns ID (201++) of found zone, or nil, if outside of all zones. 

local function abstand(dx,dy) return sqrt(dx*dx+dy*dy) end -- german abstand==distance^^

local function GetMyZone()
   local b,x,y,z=999,GetPlayerMapPosition("player") x,y=225*x,150*y
   for i,e in zonesIta() do   -- iterates finfoTab[201++], skips non-area zones
   if e[5] then               -- a rectangular zone
      if e[2]<x and x<e[4] and e[3]<y and y<e[5] then return i end
   else                       -- a circular zone
      local a=abstand(e[2]-x,e[3]-y)
      if a<e[4] and a<b then b,z=a,i end
      end end
   return z  -- maybe z==nil, if no circular near zone found 
   end;


-- Get zone text for chat:
-- If a timer is associated with this zone (e[6]==timerID), then append timer's status
-- after zone name.

local function Gezo4Say(i)
   if i then
      local z,t,n=finfoTab[i] n,t=z[1],z[6]
      if t then n=n..Geti4Say(finfoTab[t],999,true) end
      return n end
   end;    


----------------------------------------------------------------------------------------
-- Timers (TYPE=3, IDs=301++)
--
-- Timers are bars or circles (for AB bases). MMMTi() creates timers from the M1PengF3
-- template and disables either the bar subelements or the circle subelements.
--
-- Each timer (IDs=301++) has an entry e in finfoTab[ID] in the following format:
-- e[1] = txt = timer's name (used to say/draw the timer, and maybe for its tooltip)
-- e[2] = nl = "" or "\n" to mark newlines for "say all timers to chat"
-- e[3] = pattern = to strfind in WOW's messages that start/stop the timer, OR:
-- e[3] = gettimerfunction = for special timers (win estimation etc, see below)
-- e[4] = duration = whole timer period (tapping time etc), in seconds
-- e[5] = endtime = is<0 for disabled, is>GetTime() for running, otherwise completed
-- e[6] = status = "A2H" for alliance-to-horde, "A2D" for alliance-to-destroyed etc...
-- e[7] = tooltip = if nil, then "timer for e[1]" is the tooltip
-- e[8] = getglobal-name for the frame = "M1PengF(ID)"
-- e[9] = zoneID for circle timers, nil for bar timers
--
-- e[5]<0 marks the timer as currently disabled (don't draw, dont say to chat).  Other-
-- wise, it is the endtime (GetTime() when timer expires). The Geti (Gettimer) function
-- converts it to "remaining time from now on" t: with t<0 for disabled timers, t>0 for 
-- running timers, and t==0 for completed timers.
--
-- To prevent disabled timers from being drawn (which may yield strange results), the
-- handling is as follows:  a timer is hidden after create, settimer with e[5]<0 will
-- hide it immediately, drawtimer will prepare & show it, when tested as non-disabled.
----------------------------------------------------------------------------------------

local getifElapsed=0        -- time since last evaluation of special gettimer function
local getifSignal=false     -- set to true on settimer,to awake special gettimer funcs

local function MMMTi(txt,l,t,r,eti,s2s,dur,pat,nl,tt,zid) -- MakeMyMinimapTimer
   local f,i=MMMF(3)
   local w,gg,ff=r-l,"M1PengF"..i
   f:SetFrameLevel(BattlefieldMinimap:GetFrameLevel()+4)  -- circle timer over playericons
   if zid then            -- a circle timer: init circle subelements, hide bar subelements
      MMMPlace(f,l,t,w,w)
      -- the left circle half
      ff=getglobal(gg.."CL") 
      ff:SetPoint("TOPLEFT",f,"TOPLEFT") ff:SetWidth(w/2) ff:SetHeight(w)
      ff:Show() getglobal(gg.."BL"):Hide()      
      getglobal(gg.."CLT"):SetAlpha(0.7)
      -- the right circle half 
      ff=getglobal(gg.."CR") 
      ff:SetPoint("TOPRIGHT",f,"TOPRIGHT") ff:SetWidth(w/2) ff:SetHeight(w)
      ff:Show() getglobal(gg.."BR"):Hide()
      getglobal(gg.."CRT"):SetAlpha(0.7)
      -- the centered time text, and circle-shaped highlite
      ff=getglobal(gg.."TCH") ff:SetAlpha(0) ff:Show() getglobal(gg.."TBH"):Hide()
      getglobal(gg.."TXC"):Show() getglobal(gg.."TXL"):Hide() getglobal(gg.."TXR"):Hide()
   else                     -- a bar timer: init bar subelements, hide circle subelements
      MMMPlace(f,l,t,w,13)
      -- the left bar half
      ff=getglobal(gg.."BL") 
      ff:SetPoint("TOPLEFT",f,"TOPLEFT") ff:SetWidth(1) ff:SetHeight(13)
      ff:Show() getglobal(gg.."CL"):Hide()
      -- the right bar half
      ff=getglobal(gg.."BR") 
      ff:SetPoint("TOPRIGHT",f,"TOPRIGHT") ff:SetWidth(1) ff:SetHeight(13) 
      ff:Show() getglobal(gg.."CR"):Hide() 
      -- the leftaligned name text, rightaligned time text, and bar-shaped highlite
      ff=getglobal(gg.."TBH") ff:SetAlpha(0) ff:Show() getglobal(gg.."TCH"):Hide()
      getglobal(gg.."TXL"):Show() getglobal(gg.."TXR"):Show() getglobal(gg.."TXC"):Hide()
      end
   finfoTab[i]={txt,nl or "",pat,dur,eti,s2s,tt or "",gg,zid}
   f:Hide() f:SetID(i) return i
   end;


-- This is a ipairs(finfoTab), which starts at finfoTab[301], to iterate timers (can also
-- start at 301+skip, to skip the first special timer, which is the win-estimation etc..)
-- Does NOT skip disabled timers with e[5]<0 (special gettimerfuncs do maybe e[5]>=0)....

local function timerIta(skip)
   if not skip then skip=0 end
   return function(t,i)
      i=i+1 if t[i] then return i,t[i] end
      end,
   finfoTab,300+skip
   end;


-- Set timer, can change the following elements while timer's lifetime:
-- txt = new timer text (from 'game starts in...' to 'win in...' or 'debuff in...')
-- tt  = new tooltip ("" defines 'timer for e[1]' as tooltip)
-- pat = set new eventcatch-pattern OR gettimerfunction
-- s2s = set new status like "A2H" or "U2U"
-- dur = set new duration = whole timer period (in seconds)
-- rem = disable for rem<0, otherwise: timer completes in rem*dur seconds (0<=rem<=1)

local function Seti(e,rem,s2s,dur,pat,txt,tt)
   if txt then e[1]=txt end
   if pat then e[3]=pat end 
   if dur then e[4]=dur end
   if rem then if rem<0 then getglobal(e[8]):Hide() e[5]=rem else e[5]=e[4]*rem+GetTime() end end
   if s2s then e[6]=s2s end
   if tt  then e[7]=tt  end
   getifSignal = true   -- this awakes special gettimer functions
   end;

-- .....................................................................................
-- SPECIAL GETTIMER FUNCTIONS:
-- Normal timers are started/stopped by catching events in WOW's bg messages (e[3] holds 
-- a pattern to match in WOW's message),  but some timers needs a special calculation to
-- get remaining time (e[3] indexes in getiTab[]  to select a special gettimer function,
-- which returns status and/or modifies timer's status).
-- .....................................................................................

local esTime=0            -- time for ES' gainrate calculation
local pointsA,pointsH=0,0 -- resources/reinforcement/winpoints
local gaininA,gaininH=0,0 -- gainrate of winpoints in ES


-- Get timer status by Landmark icons (blue/red/gray tower/mine/blacksmith etc...). We
-- only use clear states (blue, red, gray), no transitions (gray to blue etc..), since
-- the remaining time is unclear in transitions anyway.
-- "A2A" = Alliance-to-Alliance = blue
-- "H2H" = Horde-to-Horde = red
-- "D2D" = gray ("D" means dead, destroyed or neutral) 

local s2sByLama = {
   [14]="A2A",[10]="A2A",[18]="A2A",[23]="A2A",[28]="A2A",[33]="A2A",[38]="A2A",
   [12]="H2H",[ 9]="H2H",[20]="H2H",[25]="H2H",[30]="H2H",[35]="H2H",[40]="H2H",
   [ 7]="D2D",[ 5]="D2D",[16]="D2D",[21]="D2D",[26]="D2D",[31]="D2D",[36]="D2D"} 


-- For win estimation in AB and EotS we need a table "gainrate by basecount" (in points
-- per second). In these tables,entry[1] is for basecount==0, so index with basecount+1 
-- into these tables. For basecount==0, we assume a veeery small gaining rate, to avoid  
-- divide-by-zero errors.

local gaintabAB = { 0.01, 0.8333, 1.1111, 1.6667, 3.3333, 30.0000 }
local gaintabES = { 0.01, 1.0, 2.0, 5.0, 10.0 }


local function TabSortByE1(e1,e2) return (e1[1]<e2[1]) end;  -- sort tab by column[1]


-- The core of AB/EotS win estimation: resolve a tapping table.
-- This win estimation will not only count bases already held, but will take bases into
-- account, which are turning now, and will produce points at some time in the future.
-- A win estimation "if no one changes anything, the result will be..." should do so...
--
-- TapTable is {{time1,event1},{time2,event2},...} where time is in "seconds from now 
-- on" (0 is now), and event is one of 4 possible events:
-- 1 = APlus = ally wins 1 base
-- 2 = HPlus = horde wins 1 base
-- 3 = AMinus = ally loses 1 base
-- 4 = HMinus = horde loses 1 base
-- For bases held by ally/horde, we have APlus/HPlus events at time 0 in tapping table,
-- for bases in turning, we have APlus/HPlus events at "remaining tapping time".
--
-- We sort this table from earlier to later times, and then process it, until ally or
-- horde has 2000 points or TapTable ends (Taptable's end is marked with an entry for 
-- time==3600=1 hour).
-- Input: 
-- taptab = TapTable with tapping events (not sorted by time yet)
-- gaintab[basecount+1] = gainrate_for_basecount (for AB or EotS)
-- rA = remaining points for ally (2000 at start, 0 at end)
-- rH = remaining points for horde (2000 at start, 0 at end)
-- Result: 
-- tA<tH = ally completes and wins in tA seconds
-- tH<tA = horde completes and wins in tH seconds
-- tA==tH = both teams need same time or very long to complete

local function TaptabResolve(taptab,gaintab,rA,rH)
   local t1,bA,bH,tA,tH=0,1,1      -- start with basecount=1, since gaintab[1] is for basecount==0
   table.sort(taptab,TabSortByE1)  -- sort from earlier to later tap events   
   for _,e in ipairs(taptab) do
       local t2,a=e[1],e[2]                     -- time of coming event, action of coming event
       local gA=gaintab[bA] tA=t1+rA/gA         -- get gainrate by basecount, and time to earn the re-
       local gH=gaintab[bH] tH=t1+rH/gH         -- maining points, for ally and horde
       if tA<t2 or tH<t2 then return tA,tH end  -- ally and/or horde complete before next tapevent
       rA=rA-gA*(t2-t1)                         -- if not: subtract gained points until next tap-
       rH=rH-gH*(t2-t1)                         -- event from remainder, for ally and horde
       t1=t2                                    -- t1 = current accumulated time
       if a==1 then bA=bA+1 elseif a==2 then bH=bH+1 end   -- do action of tapevent: allybases++ or
       if a==3 then bA=bA-1 elseif a==4 then bH=bH-1 end   -- hordebases++ or allybases-- or horde--
       end
       return 9999,9999   -- !(tA<tH) !(tH<tA)  -- both teams need very long...
   end;


-- Simulate different scenarios, to get a more useful win estimation, e.g.: "horde wins in 3:44, but
-- 4:1 bases will turn the ally lose into win" (this is sometimes more interesting than win time: will 
-- 3:2 bases be enough? or 4:1 or 5.0?)
--
-- We build an AsapTable (as soon as possible tapping tab), which holds for each base one entry like
-- this: {APlus_time, HPlus_time, AMinus_time, HMinus_time}, which means: "If battle runs well for
-- ally, ally can get this base in APlus_time seconds and horde loses this base in HMinus_time sec-
-- onds. If battle runs well for horde, horde can get this base in HPlus_time seconds and ally loses 
-- this base in AMinus_time seconds."
-- Basing on this AsapTable, AsapFantasy simulates a battle with basesA bases for ally and the other
-- bases for horde, who would win in which time. AsapFantasy sorts the AsapTable by the APlus columns,
-- so those bases come first in table, that can be won by ally sooner. Then it picks the first baseA
-- entries from AsapTable and fills the pro-ally-scenario (APlus and HMinus events) in a TapTable. 
-- For the other bases, it fills the pro-horde-scenario (HPlus and AMinus events) in the TapTable. 
-- By using TaptapResolve, it finds the winner for this simulated TapTable.
--
-- Input: 
-- asaptab = AsapTable with possible events and their asap times (sorted by APlus_time)
-- gaintab[basecount+1] = gainrate_for_basecount (for AB or EotS)
-- rA = remaining points for ally (2000 at start, 0 at end)
-- rH = remaining points for horde (2000 at start, 0 at end)
-- basesA = bases to give to ally (other bases got to horde)
-- Result: 
-- tA<tH = ally completes and wins in tA seconds
-- tH<tA = horde completes and wins in tH seconds
-- tA==tH = both teams need same time or very long to complete

local function AsapFantasy(asaptab,gaintab,rA,rH,basesA)
   local taptab={{3600,0}}
   for _,e in ipairs(asaptab) do      -- AsapTab is sorted: smallest APlus_times come first
   if basesA>0 then                   -- Take the first basesA bases for Ally
      -- Fill APlus event into TapTable
      table.insert(taptab,{e[1],1})        -- in e[1] seconds do event 1 (APlus)
      -- Fill HPlus and HMinus events into TabTable, if HPlus happens BEFORE HMinus (this
      -- means: even if ally gets this base as soon as possible, the Horde gets this base 
      -- BEFORE turning begins, for example HPlus_time==0: horde already holds the base).
      -- We have HMinus_time<0, if HMinus event cannot happen (base is neutral etc...).
      if e[2]<e[4] then 
      table.insert(taptab,{e[2],2})        -- in e[2] secondes do event 2 (HPlus)
      table.insert(taptab,{e[4],4}) end    -- in e[4] secondes do event 4 (HMinus)
      basesA=basesA-1
   else                               -- Take the other basesA for Horde
      -- Same as above, but ally/horde swapped: always a HPlus event, and if APlus is
      -- earlier than AMinus, then also the APlus and AMinus events...
      table.insert(taptab,{e[2],2})
      if e[1]<e[3] then 
      table.insert(taptab,{e[1],1})
      table.insert(taptab,{e[3],3}) end
   end end
   return TaptabResolve(taptab,gaintab,rA,rH)  -- return result of simulated taptab
   end;


-- For AB and EotS, the final part of their special Gettimer functions is identical, 
-- this part is in this function.
-- Input: 
-- e = entry in finfoTab for this timer (to set e's new state, time etc...)
-- asaptab = AsapTable with possible events and their asap times (not sorted yet)
-- gaintab[basecount+1] = gainrate_for_basecount (for AB or EotS)
-- rA = remaining points for ally (2000 at start, 0 at end)
-- rH = remaining points for horde (2000 at start, 0 at end)
-- tA<tH = ally completes and wins in tA seconds
-- tH<tA = horde completes and wins in tH seconds
-- tA==tH = both teams need same time or very long to complete
-- Result: 
-- e's members are modified depending of win-lose

local function AsapFantasies(e,asaptab,gaintab,rA,rH,tA,tH)
   -- If both teams need very long, we get tA==tH>3600.
   -- Set timer's new state, depending on win-lose:
   -- We apply a little hysteresis here, to prevent the timer from flickering
   -- between "ally/horde/nobody wins" states (the treshold to go into "ally
   -- wins" is higher than the treshold to clear "ally wins", same for horde).
   local  cmp=(tA-tH)/(tH+1)             -- get ally time in ratio to horde time
   if abs(cmp)<0.05 then e[6]="U2U"      -- almost equal times: winner unknown
   elseif cmp<-0.09 then e[6]="U2A"      -- ally is faster than horde: win 2 ally
   elseif cmp>=0.09 then e[6]="U2H" end  -- horde is faster than ally: win 2 horde

   -- By the calculation e[4]=t*2000/(r+1), where t is remaining seconds to win and
   -- r is remaining points to 2000, we fake the timer's duration, for making 2000
   -- points == 100% full timer bar. (r+1 to avoid divide-by-0 errors).   

   local bases=#gaintab-1
   table.sort(asaptab,TabSortByE1)       -- make sure. asaptab is sorted by APlus times
   -- If ally wins, we simulate for allybasecount=maxbasecount-1 downto 0, whether a
   -- horde win is possible (find the highest allybasecount, where tH<tA is possible).
   if e[6]=="U2A" then
      -- put in ally's wintime, faked duration, and "ally wins" text
      e[5],e[4],e[1]=tA+GetTime(),tA*2000/(rA+1),LT.GenButex[15]
      for basesA=bases-1,0,-1 do
          tA,tH=AsapFantasy(asaptab,gaintab,rA,rH,basesA)
          if tH<tA then
             if currentlyHorde then basesA=bases-basesA end   -- a 1:4 is 4:1 for horde
             e[1]=strformat(LT.GenButex[16],basesA,bases-basesA) 
             break end
          end
   -- If horde wins, we simulate for allybasecount=1 to maxbasecount, whether an ally
   -- win is possible (find the lowest allybasecount, where tA<tH is possible).
   elseif e[6]=="U2H" then
      -- put in horde's wintime, faked duration, and "horde wins" text
      e[5],e[4],e[1]=tH+GetTime(),tH*2000/(rH+1),LT.GenButex[17]
      for basesA=1,bases do
          tA,tH=AsapFantasy(asaptab,gaintab,rA,rH,basesA)
          if tA<tH then 
             if currentlyHorde then basesA=bases-basesA end   -- a 1:4 is 4:1 for horde
             e[1]=strformat(LT.GenButex[18],basesA,bases-basesA) 
             break end
          end
      end
   end;


-- Normal timers have a pattern (such as "Mine") to catch WOW's messages that start or
-- stop the timer.  For special timers, the pattern is "!ABW" etc. and indexes in this
-- table with special gettimer functions:  

local getiTab = {

   -- AB win estimation ----------------------------------------------------------------

   ["!ABW"] = function(e)
      -- Disable timer (e[5]<0), if no data, or AB is completed
      local rA,rH=GetABPoints(1),GetABPoints(2)
      if rA==nil or rA==2000 or rH==nil or rH==2000 then e[5]=-1 return end
      rA,rH=2000-rA,2000-rH

      -- Recompute win estimation all 5 seconds, otherwise adapt the bar's length only.
      -- A new tap event sets getifSignal in Seti, and forces a new win estimation.
      if not (getifSignal or getifElapsed>5) then
         local t=e[5]-GetTime() 
         if t>0 then 
            if e[6]=="U2A" then e[4]=t*2000/(rA+1) end
            if e[6]=="U2H" then e[4]=t*2000/(rH+1) end
            end
         return end

      e[5]=-1   -- disable timer, if new winner unclear

      -- Make tapTab for REAL events (APlus/HPlus for bases held or within turning).
      -- Make asapTab for POSSIBLE events (APlus/HPlus asap, maybe AMinus/HMinus).
      local taptab={{3600,0}} 
      local asaptab={}
      for _,et in timerIta(1) do  -- all 5 base timers (skip timer1: win-estimator)
         local s,t=Geti(et)
         -- Base is gray (D2D): 
         -- REAL event: none yet 
         -- ASAP for ally: APlus after goto+turn, no HMinus
         -- ASAP for horde: HPlus after goto+turn, no AMinus
         if s=="D2D" then 
            table.insert(asaptab,{80,80,-1,-1})
         -- Base is gray-to-blue (D2A): t = remaining turn time
         -- Base is blue (A2A): t = 0 
         -- REAL event: APlus after t seconds 
         -- ASAP for ally: APlus after t seconds
         -- ASAP for horde: HPlus after goto+fight+turn, AMinus after goto+fight (the
         -- APlus and AMinus will not be realized by AsapFantasy, if goto+fight < t )
         elseif s=="D2A" or s=="A2A" then
            table.insert(asaptab,{t,95,30,-1})
            table.insert(taptab,{t,1})
         -- Base is gray-to-red (D2H) or red (H2H): same as above, but ally/horde swapped
         elseif s=="D2H" or s=="H2H" then
            table.insert(asaptab,{95,t,-1,30})
            table.insert(taptab,{t,2})
         -- Base is blue-to-red (A2H):
         -- REAL event: HPlus after t seconds (t = remaining turn time)
         -- ASAP for horde: HPlus after t seconds
         -- ASAP for ally: APlus after fight (if fight<t: turning can be interrupted), other-
         -- wise HMinus after fight, APlus after fight+turn (if fight>t: turning completes)
         elseif s=="A2H" then if t<15 
            then table.insert(asaptab,{80,t,-1,15})
            else table.insert(asaptab,{15,t,-1,-1}) end
            table.insert(taptab,{t,2})
         -- Base is red-to-blue (H2A): same as above, but ally/horde swapped
         elseif s=="H2A" then if t<15 
            then table.insert(asaptab,{t,80,15,-1})
            else table.insert(asaptab,{t,15,-1,-1}) end
            table.insert(taptab,{t,1})
            end
         end
            
      -- REAL tA,tH result from resolving taptab
      -- ASAP simulations, changing timer's state... by AsapFantasies(on asaptab)
      AsapFantasies(e,asaptab,gaintabAB,rA,rH,TaptabResolve(taptab,gaintabAB,rA,rH))
      getifElapsed,getifSignal=0,false  -- start interval until next win calulation
      end;

   -- "seismic" detection of Balinda's and Galvangar's dead in AV ----------------------
   -- There is no message or gray landmark icon to detect Bali's/Galv's dead, the only
   -- indication is a sudden reinforcement minus 100. So we watch reinforcement decrea-
   -- ses each second (when !BAL and !GAL are called by draw timers). 4 effects contri-
   -- bute to reinforcements:
   -- -100, when Balinda/Galvangar killed
   -- -n*75, when n bunkers/towers destroyed
   -- +2 per minute for each mine (therefore we use 145 and 95 instead of 150 and 100).
   -- -p, for p killed players (assuming p<<75 in one second)
   -- If a player enters AV after start, Bali's/Galv's state is unknown.....

   ["!BAL"] = function(e)  -- for Balinda/Ally
      local p=GetAVPoints(1)
      if p then
         local d=pointsA-p pointsA=p         -- d=decrease; we ignore small +/- changes
         while d>145 do d=d-75 end           -- >=2 bunkers destroyed: d >= 2*75
         if d>95 then e[6]="D2D" end         -- Bali killed: d >= 100
         if p>555 then e[6]="A2A" end        -- at start of battle: Bali must be alive
         end
      end;


   ["!GAL"] = function(e)  -- for Galvangar/Horde
      local p=GetAVPoints(2)
      if p then
         local d=pointsH-p pointsH=p         -- d=decrease; we ignore small +/- changes
         while d>145 do d=d-75 end           -- >=2 towers destroyed: d >= 2*75
         if d>95 then e[6]="D2D" end         -- Galv killed: d >= 100
         if p>555 then e[6]="H2H" end        -- at start of battle: Galv must be alive 
         end
      end;

   -- ES win estimation ----------------------------------------------------------------
   -- Much like for AB (see above). But unlike the 5 AB bases, the 4 EotS towers are no
   -- timers, since there is no turning with fixed time, and a flag contributes to the
   -- winpoints, too.
  
    ["!ESW"] = function(e)
      local rA,rH=GetESPoints(1),GetESPoints(2)
      if rA==nil or rA==2000 or rH==nil or rH==2000 then e[5]=-1 return end

      if not (getifSignal or getifElapsed>5) then
         local t=e[5]-GetTime() 
         if t>0 then 
            if e[6]=="U2A" then e[4]=t*2000/(2001-rA) end
            if e[6]=="U2H" then e[4]=t*2000/(2001-rH) end
            end
         return end

      e[5]=-1

      -- We track the current winpoints for ally/horde and compute an "average gain rate"
      -- for ally/horde from this: newest gaining rate is combined with an average of the
      -- gaining rate for the last minute. This will include the flag capturing too, not
      -- only the towers.
      -- esTime is the time since EotS start, but we truncate it to 60 sec, to average
      -- our gainrate over the last ~minute~ only.
      if esTime>99 then esTime=99 end               
      local gA=rA-pointsA pointsA=rA rA=2000-rA
      gA=(gA+gaininA*esTime)/(getifElapsed+esTime)                             
      local gH=rH-pointsH pointsH=rH rH=2000-rH
      gH=(gH+gaininH*esTime)/(getifElapsed+esTime)                             
      esTime=esTime+getifElapsed 

      -- Get the current owners of the towers from landmark icons (blue=ally, red=horde,
      -- gray=none). Count the towers owned by ally/horde and build a asaptab based upon
      -- the states of the 4 towers.
      local bA,bH=1,1   -- start with basecount==1, since gaintab[1] if for 0 bases
      local asaptab={}
      for i=1,4 do
          local _,_,s=GetMapLandmarkInfo(i)
          if s then s=s2sByLama[s] end
          -- Gray tower (D2D):
          -- ASAP for ally: APlus after goto+turn_gray2blue, no HMinus
          -- ASAP for horde: HPlus after goto+turn_gray2red, no AMinus
          if s=="D2D" then
             table.insert(asaptab,{30,30,-1,-1})
          -- Blue tower (A2A):
          -- ASAP for ally: APlus after 0 seconds (already held by ally), no HMinus
          -- ASAP for horde: HPlus after goto+fight+turn_blue2gray+turn_gray2red, 
          -- and AMinus after goto+fight+turn_blue2gray
          elseif s=="A2A" then bA=bA+1
             table.insert(asaptab,{00,50,40,-1})
          -- Red tower (H2H): same as above, but ally/horde swapped
          elseif s=="H2H" then bH=bH+1
             table.insert(asaptab,{50,00,-1,40})
          end end

      -- Get the gainrate from count of owned towers (using gaintab). If this gainrate
      -- is higher than our "averaged" gainrate, we take the higher one (near at game's
      -- end, our averaged gainrated is maybe to high, so we weaken its influence).
      local wg=rA; if wg>rH then wg=rH end; wg=wg/500; if wg>0.8 then wg=0.8 end
      gaininA=gaintabES[bA]; if gaininA<gA then gaininA=(1-wg)*gaininA+wg*gA end
      gaininH=gaintabES[bH]; if gaininH<gH then gaininH=(1-wg)*gaininH+wg*gH end

      -- REAL tA,tH result remaining points divided by gain rate
      -- ASAP simulations, changing timer's state... by AsapFantasies(on asaptab)
      AsapFantasies(e,asaptab,gaintabES,rA,rH,rA/gaininA,rH/gaininH)
      getifElapsed,getifSignal=0,false  -- start interval until next win calulation
      end;
   }


-- Geti(e): returns s,t = timer e's state and time (e=finfoTab[timerID]).
-- For t<0, timer is disabled, otherwise timer will complete in t seconds from now on.

Geti = function(e)
   local s,t,f
   f=getiTab[e[3]]; if f then f(e) end  -- if we have a special gettimerfunc, call it
   -- Some timers are created with state e[6]=i, because we can derive state from the 
   -- i-th landmark icon (blue is "A2A", red is "H2H" etc.). This is good, when player
   -- enters a BG after start and we miss some messages which tell the state. We try to
   -- read state from landmark icon, but we ignore icons in transition (gray-to-red etc),
   -- since the time is unclear then. We let e[6]==i until we find a clear landmark icon
   -- (red,blue,gray) somewhen or a message tells about timer's state.
   s=e[6]
   t=tonumber(s); if t then
      _,_,s=GetMapLandmarkInfo(t)
      if s then s=s2sByLama[s] end       -- if we can get state from landmark icon, we set
      if s then e[6]=s else s="U2U" end  -- e[6], otherwise we return "U2U" and let e[6]=i
      end
   t=e[5] if t<0                                  -- return t<0 for disabled timer
      then getglobal(e[8]):Hide()                 -- hide, to avoid crude drawing
      else t=t-GetTime() if t<0 then t=0 end end  -- return t=remaining time (0=completed) 
   return s,t
   end; 


-- .....................................................................................
-- SAY TIMER(S) TO CHAT:
-- .....................................................................................

-- Geti4Say(e): return a string to say timer e's state in chat (e=finfoTab[timerID]).
-- Possible results:
-- - "" if nothing to say (saying all timers will concatenate all Geti4Say results). Say 
--   nothing for timer with t<0 or unknown state (s=="U2U"). If a max is given, say only 
--   running timer (t>0) with remaining time t<max.
-- - "name{symbol}" for completed timers (t==0), if no max is given
-- - "name{symbol}=0:12" for running timers (t==12 sec remaining time)
-- - "{symbol}" for completed timers with ti4zo=true, if no max is given
-- - "{symbol}=0:12" for running timers with ti4zo=true.
-- ti4zo==true, if this timer is associated with a zone, and "{symbol}[=0:12]" should be
-- inserted after zone's name in chat (for "help to zone !!!" for example).
-- {symbol} is the blue box for ally (s=="?2A"), the red circle for horde (s=="?2H"), the 
-- skull for dead/destroyed (s=="?2D"), or "" otherwise.

local chatSymbols = { A="{rt6}", H="{rt2}", D="{rt8}" }

Geti4Say = function(e,max,ti4zo)
   local s,t=Geti(e)
   if t<0 or (max and t>max) or s=="U2U" then return "" end
   if t>0 then t=strformat("=%d:%02d",t/60,t%60) elseif max then return "" else t="" end
   s = chatSymbols[strsub(s,3)] or ""
   if ti4zo then return s..t else return e[1]..s..t.." " end 
   end;
   

-- Say all timers to chat. If max~=nil, it excludes all completed timers and all timers
-- with remaining time > max (this is done in Geti4Say). e[2] is "" or "\n" (we must in-
-- sert newlines, if we have many timers, since our /bg message would be truncated, if
-- the line is too long. For short messages (maybe many timers exclude due max), we de-
-- lete the "\n". The parameters cha and ply can redirect the output: it will not be said
-- to /bg, but to /g or /p, or /w to player ply.

local function SayTimers(max,cha,ply)
   local msg="" 
   for _,e in timerIta() do msg=msg..Geti4Say(e,max)..e[2] end
   if strlen(msg)<60 then msg=strgsub(msg,"\n","") end 
   BGSay(msg,cha,ply)
   end;


-- .....................................................................................
-- DRAW TIMERS (IN BAR OR CIRCLE FORM):
-- .....................................................................................

-- Define colors for bar timers for A=ally=blue, H=horde=red, D=destroyed/dead=gray,
-- Y=yellow, O=orange, R=red, U=unknown=white (need Y,O,R for warsong debuff timers)
local timColors = {
   A={0.1,0.6,1.0,0.6}, H={1.0,0.2,0.2,0.6}, D={0.4,0.4,0.4,0.6},
   Y={1.0,1.0,0.2,0.6}, O={1.0,0.6,0.2,0.6}, R={1.0,0.2,0.2,0.6},
   U={1.0,1.0,1.0,0.4}}

-- For circle timers, we have a texture with 3 circles, so we can draw D2A, D2H and 
-- A2H transitions (and their counterparts). A table entry {x,y,a} defines the tex-
-- ture coords x,y of the center of a circle, and the initial angle a (where 0.5 is
-- a half turning, 180 degrees).
local timCircle = {
   D2A={0.25,0.25,0.5}, A2D={0.25,0.25,0.0},
   H2A={0.75,0.25,0.5}, A2H={0.75,0.25,0.0},
   D2H={0.25,0.75,0.5}, H2D={0.25,0.75,0.0}}

-- Convert polar coordinates radius=r, angle=a into cartesian coordinates
local function Tex1Circle(x,y,r,a) 
   r,a=r,360*a return x+r*cos(a),y+r*sin(a)  -- a is 1.0 for 1 turning (360 degrees)
   end

-- We draw circular timers as a pie chart, built from a left and a right circle half.
-- We have a==0 for a starting timer, a==0.5 for a half completed timer and a==1 for
-- a completed timer. Furthermore, we have a texture, holding a circle (with center at
-- texcoords x,y), with left half in "old" color (blue for A2H) and right half in "new"
-- color (red for A2H).
-- If a<=0.5, the left half of our pie chart picks the left half of that texture circle,
-- and the right half of our pie chart does the same, but is rotated around x,y into the
-- right half of that texture circle, the more a grows.
-- If a>=0.5, the right half of our pie chart picks the right half of that texture circle,
-- and the left half of our pie chart uses the left half of that texture circle, but is
-- rotated around x,y into the right half of that texture circle, the more a-0.5 grows.
-- Tex4Circle returns texcoords for the quadpatch to pick from our circle-texture, for a
-- half of our pie chart.
local function Tex4Circle(x,y,a,mirror)
   local ULx,ULy=Tex1Circle(x,y,0.19*1.0000,0.250-a) -- point 0,+r in polar coords, +90
   local URx,URy=Tex1Circle(x,y,0.19*1.4142,0.125-a) -- point r,+r in polar coords, +45
   local LRx,LRy=Tex1Circle(x,y,0.19*1.4142,0.875-a) -- point r,-r in polar coords, -45
   local LLx,LLy=Tex1Circle(x,y,0.19*1.0000,0.750-a) -- point 0,-r in polar coords, -90
   if mirror 
      then return LRx,LRy,URx,URy,LLx,LLy,ULx,ULy
      else return ULx,ULy,LLx,LLy,URx,URy,LRx,LRy
      end
   end;

-- Draw all timers:
-- For disabled timers (t<0) a Hide() was done in Seti. For all other timers (t>=0) we
-- prepare and Show() them. The Show() is done here only, so we avoid drawing of unpre-
-- pared timers with crude visual results. Even unknown state (U2U) is drawn (as white).
local function DrawTimers()
   for _,e in timerIta() do
   local s,t=Geti(e)
   if t>=0 then
      local gg,r,w=e[8]
      if e[9] then
         -- A circular timer (pie chart): we must adjust the texcoords for left and right
         -- circle half of our pie chart (see Tex4Coords) and update the centered text with
         -- time (remaining seconds).
         -- The frame's subelements for BAR form were hidden in MMMTi, so nothing is to do.
         local x,y,aL,aR=unpack(timCircle[s])  -- this selects colors of pie chart by state
         r=t/e[4]; 
         if r<=0 then r,t=0,"" else t=strformat("%d",t) end
         if r>=1 then r=1 end
         if r<0.5 then aR=aL+0.5 aL=aR+r else aR=aL+r end
         getglobal(gg.."CLT"):SetTexCoord(Tex4Circle(x,y,aL,true))
         getglobal(gg.."CRT"):SetTexCoord(Tex4Circle(x,y,aR))
         getglobal(gg.."TXC"):SetText(t)
      else
         -- A bar timer: we must select colors of left/right part of bar depending on state
         -- (left part has "new" color, right part has "old" color), adjust the widths of 
         -- left/right part (left part grows as right part shrinks while timer progresses).
         -- Put name (can change, from "50% debuff" to "100% debuff" etc) in left text, and
         -- remaining time in right text ("" for completed timers).
         -- The frame's subelements for CIRCLE form were hidden in MMMTi, so nothing is to do.
         w=getglobal(gg):GetWidth() r=w*t/e[4]
         if r<=0 then r,t=0.1,"" else t=strformat("%d:%02d",t/60,t%60) end
         if r>=w then r=w-0.1 end; 
         getglobal(gg.."BR"):SetWidth(r) getglobal(gg.."BRT"):SetTexture(unpack(timColors[strsub(s,1,1)]))
         getglobal(gg.."BL"):SetWidth(w-r) getglobal(gg.."BLT"):SetTexture(unpack(timColors[strsub(s,3)]))  
         getglobal(gg.."TXL"):SetText(e[1])
         getglobal(gg.."TXR"):SetText(t)
         end
      getglobal(gg):Show()
      end end
   end;
   

----------------------------------------------------------------------------------------
-- Update all (minimap with buttons/zones, tables...), if a new BG. Cleanup, if no BG.
--
-- This is the central function, which initializes the addon. When initializing for a
-- BG, we dont know, whether we enter a BG before start, after start or after disconn.
-- Therefore, we init our win/begin timer as win-timer here (assuming, we have entered
-- the BG somewhen after start), reconfigure it as begin-timer when reading a "bg begins 
-- in..." message, and switch back to win-timer when readin the "bg begins" message.
--
-- Populate BG minimap (and our frame info table finfoTab) with buttons, zones, timers.
-- Their IDs are in these ranges:
--      101++ = buttons with special handling ("help!", "say all timers", "say info")
-- clikOIDs++ = buttons that say a static text (1-click-messages)
-- clik1IDs++ = buttons that are 1st click in 2-click-messages
-- clik2IDs++ = zones (are 2nd click in 2-click-messages)
-- timerIDs++ = timers (circle or bar form)
----------------------------------------------------------------------------------------

local clik1Timeo=0  -- time, when 1st of 2 clicks expires
local clik1         -- buttonID of 1st click in 2-click-message
local clikOIDs      -- buttonIDs, which are 1-click-messages, start here 
local clik1IDs      -- buttonIDs, which are 1st of 2-click-messages, start here
local clik2IDs      -- buttonIDs, which are 2nd of 2-click-messages, start here
local timerIDs      -- frameIDs for timers, start here

local function MMMUpdate()
   if not BattlefieldMinimap then return end
   ------------------------------------------------------------------
   -- get/update current vars; do nothing, if BG not changed
   ------------------------------------------------------------------
   local previousBG=currentBG

   currentBG=nil
   for i=1,3 do
       local status,name=GetBattlefieldStatus(i)
       if status=="active" then
          for k,e in pairs(LT.BGNames) do
          if strfind(strupper(name),e) then 
             currentBG=k 
             end end
          end 
       end
-- currentBG="AB"

   if UnitFactionGroup("player")=="Horde" 
      then currentlyHorde=true 
      else currentlyHorde=nil
      end

   currentPlayer=UnitName("player")

   if currentBG==previousBG then return end  -- nothing to do

   ------------------------------------------------------------------
   -- A minimap arrow for player, laid over original playerarrow:
   -- I could not find, which texture I have to replace to put my
   -- arrow into WOW's PlayerMiniArrowFrame, so I create my own frame
   -- with my arrow texture and bind it to WOW's arrow frame (let WOW
   -- do the positioning, only the facing I have to maintain (see On-
   -- Update at end of file).
   ------------------------------------------------------------------
   -- Revert playerblips to original (if no BG) 
   ------------------------------------------------------------------
   if not M1PengParrow then        -- create my arrow frame only once
      CreateFrame("frame","M1PengParrow",PlayerMiniArrowFrame,"M1PengParrowT")
      M1PengParrow:SetPoint("CENTER",PlayerMiniArrowFrame,"CENTER")
      M1PengParrow:SetFrameLevel(PlayerMiniArrowFrame:GetFrameLevel()+4)
      end

   if currentBG 
      then M1PengParrow:Show() 
      else M1PengParrow:Hide()
      for i in pairs(picosTab) do 
          getglobal("BattlefieldMinimapRaid"..i.."Icon"):SetTexture("Interface\\WorldMap\\WorldMapPartyIcon")
          end
      end

   picosTab={} picosFL0=nil           -- undone playrblips manipulate

   ------------------------------------------------------------------
   -- Build minimap with buttons/zones, and tables for new BG
   ------------------------------------------------------------------
   useFid={100,200,300} finfoTab={}   -- reset our frames
   colozMsg={} clik1=nil              -- reset colorize and 1st-click
   getifElapsed,getifSignal=0,false   -- reset getifunc's activation

   -- Get user's colors for ally/horde messages ---------------------

   local cc
   cc=ChatTypeInfo["BG_SYSTEM_ALLIANCE"] 
   colozPfx[1]=strformat("|cff%2X%2X%2X",cc.r*255,cc.g*255,cc.b*255)
   cc=ChatTypeInfo["BG_SYSTEM_HORDE"] 
   colozPfx[2]=strformat("|cff%2X%2X%2X",cc.r*255,cc.g*255,cc.b*255)

   -- For some buttons/zones we have 3 different texts: -------------
   -- text[i+0] = referring to MY faction (same for ally and horde)
   -- text[i+1] = referring to ENEMY faction, when I'm playing ally
   -- text[i+2] = referring to ENEMY faction, when I'm playing horde
   -- select text[i+b] for blue buttons and for ally-base in warsong
   -- select text[i+r] for red buttons and for horde-base in warsong
   local r,b=1,0; if currentlyHorde then r,b=0,2 end
   
   local GT,BT,ZT=LT.GenButex    -- use gener.button texts from locatab

   ------------------------------------------------------------------
   -- Populate minimap/tables for AB:
   ------------------------------------------------------------------
   if currentBG=="AB" then

   BT,ZT=LT.ABButex,LT.ABZones   -- use button/zone texts from locatab
   zonesLevup,zonesAlfalo=5,0 -- zones OVER playerblips,fade out to 0%

   -- buttons for help!,help?,supp!,deff!
   MMMBu(GT[1],376,38,157,0,0.8)
   MMMBu(GT[2],416,12,188,0,0.8)

   clikOIDs=
   MMMBu(BT[2],429,44, 33,125,0.8) 
   MMMBu(BT[3],475,35,157,125,0.8)

   -- buttons for ally/horde rogue,inc,big
   clik1IDs=
   MMMBu(GT[3+b],160,28, 8,33,0.8) 
   MMMBu(GT[6+b],188,28, 8,63,0.8)
   MMMBu(GT[9+b],216,28, 8,93,0.8)

   MMMBu(GT[3+r], 0,28,190,33,0.8)
   MMMBu(GT[6+r],28,28,190,63,0.8) 
   MMMBu(GT[9+r],56,28,190,93,0.8)

   -- zones for the 5 bases (associated with the 5 timers)
   clik2IDs=
   MMMZo(ZT[1], 77,34, 95,52, 84.2,43.8,20,nil,302)   -- stables
   MMMZo(ZT[3],124,34,141,51,129.3,46.2,20,nil,303)   -- mine
   MMMZo(ZT[5],100,60,117,76,107.0,63.0,18,nil,304)   -- blacksmith
   MMMZo(ZT[7], 78,82, 95,98, 91.1,83.5,20,nil,305)   -- lumber
   MMMZo(ZT[9],123,80,141,98,126.2,89.7,20,nil,306)   -- farm

   timerIDs=
   MMMTi("",8,5,118,-1,"U2U",123,"!ABW","",BT[1])     -- begin or win timer

   -- timers for the 5 bases (assiciated with the 5 zones)
   -- init timer states by index of landmark icons (see Geti)
   MMMTi(ZT[1], 57,34, 75,-1,5,66,ZT[ 2],nil,nil,201) -- stables
   MMMTi(ZT[3],144,31,162,-1,3,66,ZT[ 4],nil,nil,202) -- mine
   MMMTi(ZT[5],119,57,137,-1,1,66,ZT[ 6],nil,nil,203) -- blacksmith
   MMMTi(ZT[7], 57,82, 75,-1,4,66,ZT[ 8],nil,nil,204) -- lumber
   MMMTi(ZT[9],143,78,161,-1,2,66,ZT[10],nil,nil,205) -- farm
   
   ------------------------------------------------------------------
   -- Populate minimap/tables for AV:
   ------------------------------------------------------------------
   elseif currentBG=="AV" then

   BT,ZT=LT.AVButex,LT.AVZones   -- use button/zone texts from locatab
   zonesLevup,zonesAlfalo=5,0 -- zones OVER playerblips,fade out to 0%
   pointsA,pointsH=0,0                      -- reset for !BAL and !GAL
   colozMsg = LT.AVColoz      -- colorize herold/boss yells to blue,red

   -- buttons for help!,help?,timer,info
   MMMBu(GT[1],376,38,153,0,0.8)
   MMMBu(GT[2],416,12,184,0,0.8)
   MMMBu(BT[1],322,26,  5,3,0.7)
   MMMBu(BT[2],348,26, 25,3,0.7)
   clikOIDs=MMMBuNext()
   
   -- buttons for ally/horde rogue,inc,big
   clik1IDs=
   MMMBu(GT[3+b],160,28, 3,118,0.8) 
   MMMBu(GT[6+b],188,28,26,118,0.8)
   MMMBu(GT[9+b],216,28,49,118,0.8)

   MMMBu(GT[3+r], 0,28,147,118,0.8)
   MMMBu(GT[6+r],28,28,170,118,0.8) 
   MMMBu(GT[9+r],56,28,193,118,0.8)

   -- zones for the towers,GYs,bosses...
   clik2IDs=
   MMMZo(ZT[ 1], 84,  9, 97, 22, 95.3, 19.9,8)          -- vanndar
   MMMZo(ZT[ 2], 88, 23, 97, 32, 96.2, 23.7,8,nil,307)  -- aid station
   MMMZo(ZT[ 5], 98, 16,107, 25,101.4, 22.1,8,nil,305)  -- n bunker
   MMMZo(ZT[ 8], 98, 26,107, 35, 99.0, 27.7,8,nil,306)  -- s bunker
   MMMZo(ZT[11],108,  8,117, 17,111.0, 15.0,8)          -- n mine
   MMMZo(ZT[12],108, 18,117, 27,110.2, 22.2,8)          -- stormpike GY
   MMMZo(ZT[13],115, 52,124, 61,116.1, 53.5,8)          -- stonehearth GY
   MMMZo(ZT[14],110, 42,119, 51,113.9, 46.3,8,nil,304)  -- icewing bunker
   MMMZo(ZT[17],115, 62,124, 71,118.8, 66.1,8,nil,303)  -- stoneh. bunker
   MMMZo(ZT[20],101, 52,114, 65,110.0, 59.6,8,nil,302)  -- balinda
   MMMZo(ZT[23], 97, 66,106, 75,100.8, 68.5,8)          -- snowfall GY
   MMMZo(ZT[24], 93, 79,106, 92,103.5, 86.2,8,nil,308)  -- galvangar
   MMMZo(ZT[27],107, 83,116, 92,108.6, 87.9,8,nil,309)  -- iceblood tower
   MMMZo(ZT[30],109, 93,118,102,113.7, 98.4,8,nil,310)  -- tower point
   MMMZo(ZT[33],117, 83,126, 92,115.8, 90.2,8)          -- iceblood GY
   MMMZo(ZT[34],109,110,118,119,113.0,115.2,8)          -- frostwolf GY
   MMMZo(ZT[35], 99,104,108,113,103.0,107.0,8)          -- s mine
   MMMZo(ZT[36],111,120,120,129,111.3,126.8,8,nil,311)  -- e tower
   MMMZo(ZT[39],101,120,110,129,108.8,126.5,8,nil,312)  -- w tower
   MMMZo(ZT[42],109,130,118,139,110.9,132.1,8,nil,313)  -- relief hut
   MMMZo(ZT[45], 95,130,108,143,106.7,130.2,8)          -- drek'thar

   -- timers: begin timer, plus 2x6 important points...
   -- balinda/galvangar have no destroy time like bunkers/towers, they
   -- are simply alive or dead, but sometimes it is interesting to know
   -- whether they are alive or dead, without having to ride to them...
   timerIDs=
   MMMTi("",5,3,70,-1,"U2U",123,"NONE")            -- begin timer
   MMMTi(ZT[21],150, 30,215,0,"U2U",1,"!BAL")      -- balinda
   MMMTi(ZT[18],150, 45,215,0, 16,243,ZT[19])      -- stoneh. bunker
   MMMTi(ZT[15],150, 60,215,0, 12,243,ZT[16])      -- icewing bunker
   MMMTi(ZT[ 6],150, 75,215,0,  3,243,ZT[ 7])      -- n bunker
   MMMTi(ZT[ 9],150, 90,215,0,  4,243,ZT[10])      -- s bunker
   MMMTi(ZT[ 3],150,105,215,0, 19,243,ZT[ 4],"\n") -- aid station  
   MMMTi(ZT[25],  5, 30, 70,0,"U2U",1,"!GAL")      -- galvangar
   MMMTi(ZT[28],  5, 45, 70,0, 11,243,ZT[29])      -- iceblood tower
   MMMTi(ZT[31],  5, 60, 70,0, 21,243,ZT[32])      -- tower point
   MMMTi(ZT[37],  5, 75, 70,0,  5,243,ZT[38])      -- e tower
   MMMTi(ZT[40],  5, 90, 70,0, 22,243,ZT[41])      -- w tower
   MMMTi(ZT[43],  5,105, 70,0,  8,243,ZT[44])      -- relief hut

   ------------------------------------------------------------------
   -- Populate minimap/tables for ES:
   ------------------------------------------------------------------
   elseif currentBG=="ES" then

   BT,ZT=LT.ESButex,LT.ESZones   -- use button/zone texts from locatab
   zonesLevup,zonesAlfalo=5,0 -- zones OVER playerblips,fade out to 0%
   pointsA,pointsH,gaininA,gaininH,esTime=0,0,0,0,30 -- reset for !ESW
   colozMsg = LT.ESColoz      -- swap red-blue in "lost control" message

   -- buttons for help!,help?,supp!,deff!
   MMMBu(GT[1],376,38,157,0,0.8)
   MMMBu(GT[2],416,12,188,0,0.8)

   clikOIDs=
   MMMBu(BT[2],429,44, 33,125,0.8) 
   MMMBu(BT[3],475,35,157,125,0.8)

   -- buttons for ally/horde rogue,inc,big
   clik1IDs=
   MMMBu(GT[3+b],160,28, 8,33,0.8) 
   MMMBu(GT[6+b],188,28, 8,63,0.8)
   MMMBu(GT[9+b],216,28, 8,93,0.8)

   MMMBu(GT[3+r], 0,28,190,33,0.8)
   MMMBu(GT[6+r],28,28,190,63,0.8) 
   MMMBu(GT[9+r],56,28,190,93,0.8)

   -- zones for the 4 towers + middle
   clik2IDs=
   MMMZo(ZT[1], 82,57, 96,71, 91.7,62.8,16)  -- mage
   MMMZo(ZT[2],119,54,133,68,124.5,62.6,16)  -- draenei
   MMMZo(ZT[3],101,68,115,80,107.9,73.5,9)   -- middle
   MMMZo(ZT[4], 83,82, 98,96, 91.8,85.7,16)  -- fel reaver
   MMMZo(ZT[5],120,79,135,93,124.9,86.1,16)  -- blood elf
 
   -- timer for begin or win estimation
   timerIDs=
   MMMTi("",8,5,118,-1,"U2U",123,"!ESW","",BT[1])

   ------------------------------------------------------------------
   -- Populate minimap/tables for WS:
   ------------------------------------------------------------------
   elseif currentBG=="WS" then

   BT,ZT=LT.WSButex,LT.WSZones  -- use button/zone texts from locatab
   zonesLevup,zonesAlfalo=0,0.5 -- zones UNDER playerblips, fade out to 50%
   colozMsg = LT.WSColoz        -- make "flag is back to base" red/blue

   -- buttons for help!,help?,supp!,deff!,where ally/horde flag?
   MMMBu(GT[1],376,38,157,0,0.8)
   MMMBu(GT[2],416,12,188,0,0.8)

   clikOIDs=
   MMMBu(BT[2],429,44, 33,125,0.8) 
   MMMBu(BT[3],475,35,157,125,0.8)

   MMMBu(BT[4],294,25, 20,33,0.8)
   MMMBu(BT[5],134,25,157,33,0.8)

   -- buttons for ally/horde flag,!flag,rogue,inc,big
   clik1IDs=
   MMMBu(BT[6  ],244,25,20,63,0.8)
   MMMBu(BT[8+r],269,25,20,93,0.8)
   MMMBu(GT[3+b],160,28,45,33,0.8)
   MMMBu(GT[6+b],188,28,45,63,0.8)
   MMMBu(GT[9+b],216,28,45,93,0.8)

   MMMBu(BT[7  ], 84,25,157,63,0.8)
   MMMBu(BT[8+b],109,25,157,93,0.8)
   MMMBu(GT[3+r],  0,28,182,33,0.8)
   MMMBu(GT[5+r], 28,28,182,63,0.8)
   MMMBu(GT[9+r], 56,28,182,93,0.8)

   -- zones for ally/horde side, middlefield:
   -- There are buttons for BASE and ROOF, so user can distinguish
   -- between ROOF and BASE (all except roof). But roof has no zone
   -- area (roof's area is added to base's area), since we have no
   -- z player coordinate^^. However, the BASE button covers the
   -- flagroom (where is no roof), and the ROOF the remainder of the
   -- base. At left and right margin of the battlefield, the zone
   -- AREAS are bigger than the zone BUTTONS, to cover all players.
   -- There are 2 pixel gaps between zone buttons, to avoid wrong
   -- clicks due slipping with the mouse from desired zone to the
   -- neighboring zone.
   clik2IDs=
   MMMZo(ZT[ 1+b],101,15,124,23,101,00,165,33)     -- ally base
   MMMZo(ZT[ 4+b],101,24,124,32)                   -- ally roof
   MMMZo(ZT[ 7+b], 85,34,105,54, 60,00,106,55)     -- ally GY
   MMMZo(ZT[10+b],107,34,118,54,106,33,119,55)     -- ally tunnel
   MMMZo(ZT[13+b],120,34,140,54,119,33,165,55)     -- ally ramp

   MMMZo(ZT[ 1+r],101,135,124,143, 60,125,124,150) -- horde base
   MMMZo(ZT[ 4+r],101,126,124,134)                 -- horde roof
   MMMZo(ZT[ 7+r],120,104,140,124,119,103,165,150) -- horde GY
   MMMZo(ZT[10+r],107,104,118,124,106,103,119,125) -- horde tunnel
   MMMZo(ZT[13+r], 85,104,105,124, 60,103,106,125) -- horde ramp

   MMMZo(ZT[16], 85,56,101,102, 60,55,102,103)     -- westside
   MMMZo(ZT[17],103,56,122,102,102,55,123,103)     -- middle
   MMMZo(ZT[18],124,56,140,102,123,55,165,103)     -- east side

   -- timer for begin or debuff
   timerIDs=
   MMMTi("",3,3,90,-1,"U2U",123,"NONE")

   end  -- END SWITCH(currentBG)
 
   ------------------------------------------------------------------
   -- Hide old frames from us, not used now (have no DeleteFrame())
   -- and init for fading in/out of zones.
   -- We will end here, even if there is no current BG. Then we will
   -- hide all frames from us.
   ------------------------------------------------------------------
   zonesAlfa,zonesAlfade = zonesAlfalo

   -- For TYPE u=1,2,3 (button,zone,timer) we have in useFid[u] the 
   -- last currently used ID for this TYPE. We try to find frames above
   -- this ID (from earlier BGs with more frames) and hide them. Break
   -- on first frame not found (=not created). Do this for all 3 TYPES
   -- and all 3 ID intervals (101++, 201++, 301++).
   for u=1,#useFid do
   for i=useFid[u]+1,999 do
       local f=getglobal("M1PengF"..i)
       if f then f:Hide() else break end
       end 
       end
   
   end;


----------------------------------------------------------------------------------------
-- Action on button/zone/timer click
----------------------------------------------------------------------------------------

function M1PengOnClik(clik)
   local doclik1,dosay

   ------------------------------------------------------------------
   -- MMMUpdate() has populated minimap with IDs in these ranges:
   -- 101++ == special (help!,help?,say all timers,say info)
   -- clikOIDs++ = buttons for 1-click-messages (supp,deff,flag?...)
   -- clik1IDs++ = 1st click for 2-click-messages (rogue,inc...)
   -- clik2IDs++ = 2nd click for 2-click-messages (zones)
   -- timerIDs++ = click on a timer
   --
   -- We enter the following switch(ID) with:
   -- - clik1 = nil or ID of the button in a 1st click before this
   -- We leave the following switch(ID) with:
   -- - doclik1 = set clik1 to ID clicked now (nil for erase clik1)
   -- - dosay = a text to say in /bg (nil for nothing to say)
   ------------------------------------------------------------------
 
   -- special
   if clik<clikOIDs then
      -- help!: if it is the 1st click on help,we save it as 1st clik (a 2nd
      -- clik on a zone later will say "help to zone!"),otherwise (it is 2nd 
      -- click on help! == doubleclick) we say "help to MYZONE !!" (if there
      -- is no zone the player is currently in, we do emote /helpme).
      if clik==101 then
         if clik1~=clik then doclik1=clik else
            dosay=Gezo4Say(GetMyZone())
            if dosay -- finfoTab[clik1]==finfoTab[101]=="help 2# !!!" template
               then dosay=strgsub(finfoTab[clik1],"#",strupper(dosay))
               else DoEmote("helpme") end
            end
      -- help?: if it is the 1st click on help,we save it as 1st clik (a 2nd
      -- clik on a zone later will say "help to zone?"),otherwise (it is 2nd 
      -- click on help? == doubleclick) we say "anybody help ??"
      elseif clik==102 then
         if clik1~=clik then doclik1=clik else dosay=LT.GenButex[19] end
      elseif clik==103 then SayTimers(999)  -- say all running timers
      elseif clik==104 then SayTimers(nil)  -- say all running+completed timers
      end

   -- click for 1-click-message
   elseif clik<clik1IDs then
      dosay = finfoTab[clik]

   -- 1st click for 2-click-message (if doubleclick, then zone=myzone)
   elseif clik<clik2IDs then
      if clik1~=clik then doclik1=clik else
         dosay=Gezo4Say(GetMyZone())
         if dosay then dosay=strgsub(finfoTab[clik1],"#",dosay) end
         end

   -- 2nd click for 2-click-message (zone)
   elseif clik<timerIDs then
      dosay=Gezo4Say(clik)
      if clik1==101 then dosay=strupper(dosay) end
      if clik1 then dosay=strgsub(finfoTab[clik1],"#",dosay) end

   -- click on a timer: say it to /bg 
   -- (for disabled timers, Geti4Say returns "", and BGSay says nothing for "")
   else
      BGSay(Geti4Say(finfoTab[clik]))

   end -- END SWITCH(clik)

   ------------------------------------------------------------------
   -- Action after button/zone/timer click:
   -- dosay = a message to say in /bg chat
   -- doclik1 = a buttonID,clicked as 1st in a 2-click-message (after
   -- clik1Timeo seconds, the 1st clik is undone in OnUpdate,  if the
   -- user does no 2nd clik). Between 1st and 2nd click,the button of
   -- 1st click is highlited. We undo highliting for old clik1 button
   -- anyway (either 2nd click came now, or another click cancels the
   -- 2-click-message). We set highlite for new clik1 button, if any.
   ------------------------------------------------------------------
   if dosay then BGSay(dosay) end   -- applies 2-click-grammar, too

   if   clik1 then getglobal("M1PengF"..clik1):UnlockHighlight() end
   if doclik1 then getglobal("M1PengF"..doclik1):LockHighlight() end
   clik1=doclik1 clik1Timeo=9 

   end;


-- The timers are frames, no buttons. So we make the pushed look and click.
function M1PengOnMouseDU(i,down)
   if i>=timerIDs then
      local e,a=finfoTab[i],0.6 
      if down then a=1 M1PengOnClik(i) end 
      getglobal(e[8].."TBH"):SetAlpha(a)  -- one of these is shown on timer's create
      getglobal(e[8].."TCH"):SetAlpha(a)  -- the other is hidden (bar or circle form)
      end
   end;


----------------------------------------------------------------------------------------
-- Action on button/zone/timer hover (tooltip)
----------------------------------------------------------------------------------------

function M1PengOnEnter(i)
   local e,me,tt=finfoTab[i],getglobal("M1PengF"..i)

   -- timer: highlite 60% when hovered (100% when pushed)
   -- tooltip text is given in e[7], or a genic template "timer for NAME" is used
   if i>=timerIDs then
      getglobal(e[8].."TBH"):SetAlpha(0.6)  -- one of these is shown on timer's create
      getglobal(e[8].."TCH"):SetAlpha(0.6)  -- the other is hidden (bar or circle form)
      tt=e[7] if tt=="" then tt=strgsub(LT.GenButex[12],"#",e[1]) end
   -- zone: start fade-in of all zones (zonesAlfa+=zonesAlfade/second, in OnUpdate)
   -- tooltip is zone's text (2-click-grammatics will be stripped off by LT.Grammar4TT)
   elseif i>=clik2IDs then
      tt=e[1] zonesAlfade=1.5
   -- button:
   -- tooltip is button's text (2-click-grammatics will be stripped off by LT.Grammar4TT)
   else
      tt=e end

   -- I wanted no extra members for tooltip text in buttons and zones, I wanted to use the
   -- button/zone names itself, but I have to strip off the special 2-click-grammatics ...
   tt = strtsub(tt,LT.Grammar4TT)

   -- zone or timer: since zone/timer can cover playerblips, we must incorporate all
   -- players covered by our zone/timer frame into our tooltip
   if i>=clik2IDs then
      local nl,zR,zB,zL,zT=me:GetPoint()
      zT=-zT; zR=zL+me:GetWidth(); zB=zT+me:GetHeight()
      nl="\n\n"
      for ri=1,40 do
          local rx,ry=GetPlayerMapPosition("raid"..ri) rx,ry=225*rx,150*ry
          if zL<rx and rx<zR and zT<ry and ry<zB  -- 0,0 for unknown units
          and not UnitIsUnit("raid"..ri,"player") then
             tt=tt..nl..UnitName("raid"..ri) nl="\n" end
          end
      end

   -- Activate tooltip with our tooltip text
   if me:GetCenter() < me:GetParent():GetCenter()
      then GameTooltip:SetOwner(me,"ANCHOR_RIGHT")
      else GameTooltip:SetOwner(me,"ANCHOR_LEFT")
      end
   GameTooltip:SetText(tt)
   GameTooltip:Show()
   end;


function M1PengOnLeave(i)
   local e=finfoTab[i]
   -- timer: undo highlite, when hovering stops
   if i>=timerIDs then
      getglobal(e[8].."TBH"):SetAlpha(0)  -- one of these is shown on timer's create
      getglobal(e[8].."TCH"):SetAlpha(0)  -- the other is hidden (bar or circle form)
   -- zone: start fade-out of all zones (zonesAlfa+=zonesAlfade/second, in OnUpdate).
   -- Since fade-in goes until zonesAlfa=1.5, it will take some time until zonesAlfa<1,
   -- so fade-out of zones will start some time after leaving a zone. This prevents a
   -- flickering (fade-in, fade-out, fade-in...) when going with mouse from one zone to
   -- another.
   elseif i>=clik2IDs then
      zonesAlfade=-0.2 end
   GameTooltip:Hide()
   end;


----------------------------------------------------------------------------------------
-- Chathandler:
-- Catch WOW messages to start/stop timers, and recolorize some messages to make all
-- pro-ally messages blue and all pro-horde messages red.
-- Message patterns are taken from locatable (patterns don't have to be full messages,
-- but only patterns,that catch the desired message (or group of messages!) with strfind.
----------------------------------------------------------------------------------------

local function M1PengOnChev(ev)

   local GT,BT,ET,ZT=LT.GenButex

   ------------------------------------------------------------------
   -- React on event messages from chat for AB:
   -- the turn time (66sec) was put as duration into timers on create
   ------------------------------------------------------------------
   if currentBG=="AB" then 

   BT,ET=LT.ABButex,LT.ABEvents

   if ev=="CHAT_MSG_BG_SYSTEM_NEUTRAL" then
      local e=finfoTab[301]
      if strfind(arg1,ET[1]) then                   -- "begins in 1 minute"
         Seti(e,1.0,"U2O",60,"NONE",GT[14],GT[13])  -- activate begin timer, 100% of 60sec left
      elseif strfind(arg1,ET[2]) then               -- "begins in 30 seconds"
         Seti(e,0.5,"U2O",60,"NONE",GT[14],GT[13])  -- activate begin timer, 50% of 60sec left
      elseif strfind(arg1,ET[3]) then               -- "battle has begun"
         Seti(e,-1,"U2U",123,"!ABW","",BT[1])       -- begin-timer is win-timer now (unknown yet)
      end
   elseif ev=="CHAT_MSG_BG_SYSTEM_ALLIANCE" then    -- Ally taps/takes/defends a base
      for _,e in timerIta(1) do                     -- examine the 5 timers (skip 1 win timer)
      if strfind(arg1,e[3]) then                            -- base's name found in message 
         if strfind(arg1,ET[4]) then Seti(e,1,"D2A")        -- "claims": gray-to-blue, 100% to do
         elseif strfind(arg1,ET[5]) then Seti(e,1,"H2A")    -- "assault:" red-to-blue, 100% to do
         elseif strfind(arg1,ET[6]) or strfind(arg1,ET[7])  -- "taken/defended": status (A2A) is known now,
            then Seti(e,-1,"A2A") end                       -- but pie chart is visible while turning only
         break end end                                     
   elseif ev=="CHAT_MSG_BG_SYSTEM_HORDE" then       -- Horde taps/takes/defends a base (same as for ally)
      for _,e in timerIta(1) do                     
      if strfind(arg1,e[3]) then
         if strfind(arg1,ET[4]) then Seti(e,1,"D2H") 
         elseif strfind(arg1,ET[5]) then Seti(e,1,"A2H")
         elseif strfind(arg1,ET[6]) or strfind(arg1,ET[7]) 
            then Seti(e,-1,"H2H") end
         break end end
   end  -- END SWITCH(ev)

   ------------------------------------------------------------------
   -- React on event messages from chat for AV:
   -- At MMMUpdate() for AV we assume, that we entered somewhen after
   -- the start of battle: Balinda's and Galvangar's state is unknown,
   -- and "say timer" and "say info" buttons are displayed. Only if we
   -- read the "bg begins in..." messages, we turn on our begin timer,
   -- and turn it off upon "bg has begun" message (on which we also
   -- know Balinda and Galvangar alive).
   -- The turn time (4:03) was put as duration into timers on create.
   ------------------------------------------------------------------
   elseif currentBG=="AV" then 

   ET,ZT=LT.AVEvents,LT.AVZones

   if ev=="CHAT_MSG_BG_SYSTEM_NEUTRAL" then
      local e=finfoTab[301]                        -- the begin timer             
      if strfind(arg1,ET[1]) then                  -- "begins in 1 minute"
         M1PengF103:Hide()                         -- "say timers" covered by begin-timer
         M1PengF104:Hide()                         -- "say info" covered by begin timer
         Seti(e,1.0,"U2O",60,"NONE",GT[14],GT[13]) -- activate begin timer, 100% of 60sec left
      elseif strfind(arg1,ET[2]) then              -- "begins in 30 seconds"
         M1PengF103:Hide()                         -- "say timers" covered by begin-timer
         M1PengF104:Hide()                         -- "say info" covered by begin-timer
         Seti(e,0.5,"U2O",60,"NONE",GT[14],GT[13]) -- activate begin timer, 50% of 60sec left
      elseif strfind(arg1,ET[3]) then              -- "battle has begun"
         Seti(e,-1)                                -- hide begin-timer
         Seti(finfoTab[302],0,"A2A")               -- Balinda is alive at start
         Seti(finfoTab[308],0,"H2H")               -- Galvangar is alive at start
         M1PengF103:Show()                         -- "say timers" not covered by begin-timer
         M1PengF104:Show()                         -- "say info" not covered by begin-timer 
         end
   
   -- In AV the taking/destroying of towers/bunkers/GYs is yelled by the herald.
   -- We parse it, to get start/stop events for our timers (arg1=yell, arg2=yeller).
   elseif ev=="CHAT_MSG_MONSTER_YELL" then
      for _,e in timerIta(1) do  -- examine all timers, skip the 1 begin timer
      if e[3]=="!BAL" then       -- if Balinda is the yeller, we know: she is alive
         if strfind(arg2,ZT[22]) then Seti(e,0,"A2A") break end
      elseif e[3]=="!GAL" then   -- if Galvangar is the yeller, we know: he is alive
         if strfind(arg2,ZT[26]) then Seti(e,0,"H2H") break end
      elseif strfind(arg1,e[ 3]) then -- the tower's/bunker's/GY's name is in the yell
         if strfind(arg1,ET[ 4]) then Seti(e,1,"H2D") break end -- ally will destroy tower
         if strfind(arg1,ET[ 5]) then Seti(e,1,"A2D") break end -- horde will destroy bunker
         if strfind(arg1,ET[ 6]) then Seti(e,0,"D2D") break end -- ally has destroyed tower
         if strfind(arg1,ET[ 7]) then Seti(e,0,"D2D") break end -- horde has destroyed bunker
         if strfind(arg1,ET[ 8]) then Seti(e,1,"H2A") break end -- ally will take GY
         if strfind(arg1,ET[ 9]) then Seti(e,1,"A2H") break end -- horde will take GY
         if strfind(arg1,ET[10]) then Seti(e,0,"A2A") break end -- ally has taken GY / defended bunker
         if strfind(arg1,ET[11]) then Seti(e,0,"H2H") break end -- horde has taken GY / defended tower
         break end end
      -- Change color from yell-color to blue for pro-ally-news and red for pro-horde-news...
      -- When we prefix arg2 (the yeller) with a color code, WOW prints "yeller: yell" in that color.
      for i=1,#colozMsg,2 do   -- (no yell with 3)
      if strfind(arg1,colozMsg[i+1]) then 
         arg2=colozPfx[colozMsg[i]]..arg2 
         break end end

   end  -- END SWITCH(ev)

   ------------------------------------------------------------------
   -- React on event messages from chat for ES:
   -- Only a begin timer,  which is changed to be the win estimation
   -- bar after battle begin (same as in Arathi). The state of the 4
   -- towers we get from their landmark icons, not from catching the
   -- "has gained/losed tower" messages.  We might miss some of the-
   -- se messages,  when entering bg after start, but landmark icons
   -- are always present.
   -- Catch "flag won" and "gained/losed tower" to force new win es-
   -- timation (set getifSignal).
   ------------------------------------------------------------------
   elseif currentBG=="ES" then 

   BT,ET=LT.ESButex,LT.ESEvents

   if ev=="CHAT_MSG_BG_SYSTEM_ALLIANCE"
   or ev=="CHAT_MSG_BG_SYSTEM_HORDE"
   or ev=="CHAT_MSG_BG_SYSTEM_NEUTRAL" then
      local e=finfoTab[301]
      if strfind(arg1,ET[1]) then
         Seti(e,1.0,"U2O",60,"NONE",GT[14],GT[13])
      elseif strfind(arg1,ET[2]) then
         Seti(e,0.5,"U2O",60,"NONE",GT[14],GT[13])
      elseif strfind(arg1,ET[3]) then
         Seti(e,-1,"U2U",123,"!ESW","",BT[1])
      elseif strfind(arg1,ET[4]) or strfind(arg1,ET[5]) then
         getifSignal=true
      end
   end  -- END SWITCH(ev)

   ------------------------------------------------------------------
   -- React on event messages from chat for WS:
   -- A begin timer (same as in other BGs), which is changed to be
   -- the debuff timer, when a debuff situation appears.
   -- Debuff logic is as follows (I hope, this is correct):
   -- When BOTH flags are picked up, the debuff timer starts. 50% af-
   -- ter ~10min and 100% after ~15min. The timer stops, when BOTH
   -- flags are in their bases (after returning both, or winning a 
   -- flag). Instead of 10:00 min, I stopped 6:20 to 7:20 for 50% de-
   -- buff and ~6:10 from 50% to 100% debuff... so I can use average
   -- times only.
   -- Debuff timer states/colors: Y=yellow=0%,O=orange=50%,R=red=100%
   -- In MMMUpdate() for WS we set state U=unknown, since we maybe
   -- enter BG after start and don't know, whether a debuff is run-
   -- ning or not...
   ------------------------------------------------------------------
   elseif currentBG=="WS" then 

   BT,ET=LT.WSButex,LT.WSEvents

   if ev=="CHAT_MSG_BG_SYSTEM_ALLIANCE"
   or ev=="CHAT_MSG_BG_SYSTEM_HORDE"
   or ev=="CHAT_MSG_BG_SYSTEM_NEUTRAL" then
      local e=finfoTab[301]
      -- "begins in 1 minute": activate begin timer, 100% of 60sec left
      if strfind(arg1,ET[1]) then
         Seti(e,1.0,"U2O",60,"NONE",GT[14],GT[13])
      -- "begins in 30 seconds": activate begin timer, 50% of 60sec left
      elseif strfind(arg1,ET[2]) then
         Seti(e,0.5,"U2O",60,"NONE",GT[14],GT[13])
      -- "has begun": change begin-timer to be the debuff-timer: it
      -- is invisible (-1), but we KNOW there runs no debuff (Y2Y)
      elseif strfind(arg1,ET[3]) then
         Seti(e,-1,"Y2Y")
      -- "now is 50% debuff": we KNOW, there runs a debuff (from 50% to 100%, O2R)
      -- and the timer for 50%-to-100% starts now (377sec left).
      elseif strfind(arg1,ET[5]) then
         Seti(e,1,"O2R",377,"NONE",BT[12],BT[1]) 
      -- "now is 100% debuff": we KNOW, there is a debuff (100%, R2R)
      -- and the timer for 50%-to-100% has completed (0 sec left).
      elseif strfind(arg1,ET[6]) then
         Seti(e,0,"R2R",nil,"NONE",BT[12],BT[13]) 
      -- any other "flag picked/dropped/won..." message:
      -- Check this as LAST message, since our pattern for strfind is simplay "flag"
      -- and it will catch the debuff messages, too.
      -- We get the count of picked-up flags (0,1 or 2) simply by GetWorldStateUIInfo.
      -- 0 flags: debuff timer is invisible (-1) and we KNOW, there runs no debuff (Y2Y)
      -- 2 flags: start 0%-50%-debuff-timer, but only if we KNOW, that the debuff timer
      -- not already runs (Y2Y). I stopped 6:20 to 7:20 for 0%-50%-time, and use 6:50.
      elseif strfind(arg1,ET[4]) then
         local f=HasWSFlagPicked(1)+HasWSFlagPicked(2)
         if f==0 then Seti(e,-1,"Y2Y")
         elseif f==2 and e[6]=="Y2Y" then 
            Seti(e,1,"Y2O",411,"NONE",BT[11],BT[1]) end
      end
   end  -- END SWITCH(ev)

   end  -- END SWITCH(currentBG)

   ------------------------------------------------------------------
   -- AntiMessages: a blue message should be red, if it is bad for
   -- Alliance, or a red message should be blue, if bad for Horde. 
   -- Apply myjokeTab (replaces in WOW's messages, set by MyPeng).
   ------------------------------------------------------------------

   colozPfx[3]=nil
   if ev=="CHAT_MSG_BG_SYSTEM_ALLIANCE" then colozPfx[3]=colozPfx[2] end 
   if ev=="CHAT_MSG_BG_SYSTEM_NEUTRAL"  then colozPfx[3]="isn't nil" end
   if ev=="CHAT_MSG_BG_SYSTEM_HORDE"    then colozPfx[3]=colozPfx[1] end

   if colozPfx[3] then
      -- Do recolorize WOW's bg messages 
      for i=1,#colozMsg,2 do 
      if strfind(arg1,colozMsg[i+1]) then 
         arg1=colozPfx[colozMsg[i]]..arg1
         break end end
      -- Do modify WOW's bg messages, if myjokeTab is set by MyPeng...
      -- MyPengPlusser after PLAY_ENT_WORLD has incorrect GetGuildInfo() sometimes, so
      -- we do it now (myjokeTab is used here only). Call MyPengPlusser ONCE and free!
      if MyPengPlusser then MyPengPlusser() MyPengPlusser=nil end
      arg1=strtsub(arg1,myjokeTab)
      end

   end;


-- Hook the original handler for incoming chat messages.
-- Dispatch messages to M1PengOnChev, this function starts/stops
-- timers by catching messages and recolorizes some messages.
-- Some other messages are modified/suppressed by BGMsgFilter.

local  o_ChatFrame_MessageEventHandler
if not o_ChatFrame_MessageEventHandler then
       o_ChatFrame_MessageEventHandler = ChatFrame_MessageEventHandler
       ChatFrame_MessageEventHandler = function(ev,...)
           -- parse for events, colourize...
           if ev=="CHAT_MSG_MONSTER_YELL" 
           or ev=="CHAT_MSG_BG_SYSTEM_NEUTRAL" 
           or ev=="CHAT_MSG_BG_SYSTEM_ALLIANCE" 
           or ev=="CHAT_MSG_BG_SYSTEM_HORDE" then
              M1PengOnChev(ev)
           -- shorten or suppress bg join/leave messages...
           elseif ev=="CHAT_MSG_SYSTEM" and currentBG then
              local filtab=LT.BGMsgFilter
              for i=1,#filtab,2 do
              if strfind(arg1,filtab[i]) then
                 if not filtab[i+1] then return end
                 arg1=strgsub(arg1,filtab[i],filtab[i+1])
                 end end
           -- avoid BGSay(msg,whisper_to_self) writing twice... 
           elseif ev=="CHAT_MSG_WHISPER" and arg2==currentPlayer then
              return end
           return o_ChatFrame_MessageEventHandler(ev,...) 
           end 
       end


----------------------------------------------------------------------------------------
-- Slashcommand /m1peng: say help, timer, tactic texts...
----------------------------------------------------------------------------------------

-- Player types: /m1peng command [param] [/s|/y|/g|/p|/w [playername]]
-- M1Peng(cmdline) gets: command [param] [/s|/y|/g|/p|/w [playername]]
-- We split it into: 
-- cmd = command [param]
-- cha = nil|"SAY"|"YELL"|"GUILD"|"PARTY"|"WHISPER" (nil will use "BATTLEGROUND")
-- ply = nil|playername (nil will use MYNAME for whisper)
-- This allows to redirect command's output to /s or /y or /g or /p or /w (otherwise,
-- /bg is used). This is interesting for testing a textmacro (whisper to myself) or
-- redirecting the "say all timers" into /g channel...

local function ChatypSuffix(cmdline)
   for k,e in pairs(LT.Chatypes) do
   local p1,p2=strfind(cmdline,e)
   if p1 then 
      local ply=strsub(cmdline,p2+2)
      if ply=="" then ply=nil end
      return strsub(cmdline,1,p1-1),k,ply 
      end end
   return cmdline
   end;


-- Our slash commands /m1peng or /m1p
local function M1Peng(cmdline)
   local cmd,cha,ply=ChatypSuffix(strtrim(cmdline))
   
   -- /m1peng ===> print usage help
   if cmd=="" then
      for i=2,#(LT.UsageHelp) do DEFAULT_CHAT_FRAME:AddMessage(LT.UsageHelp[i]) end

   -- /m1peng helpme ===> like doubleclick on "help!" on minimap (to integrate in macros)
   elseif cmd=="helpme" then
      local myz=Gezo4Say(GetMyZone()) 
      if myz
         then BGSay(strgsub(LT.GenButex[1],"#",strupper(myz)),cha,ply)
         else DoEmote("helpme") end

   -- /m1peng timer [max] ===> say all timers (if max given, only running timers<max)
   elseif strfind(cmd,"timer")==1 then
      SayTimers(tonumber(strsub(cmd,7),10),cha,ply) 

   -- /m1peng textmacroname ===> /bg textmacrobody
   -- By using MyPeng.lua, user can define textmacros, BGSay will apply them   
   else
      BGSay(cmd,cha,ply) 
      end

   end;


-- Register our /m1peng

SlashCmdList["M1PENG"]=M1Peng
SLASH_M1PENG1="/m1peng"
SLASH_M1PENG2="/m1p"


----------------------------------------------------------------------------------------
-- Frame for Eventhandling: 
----------------------------------------------------------------------------------------

local elapsed0,elapsed1=0,0  


M1PengEvha = {

OnLoad = function()
   -- A welcome message, if a locatab could be loaded (see at start of this file), 
   -- otherwise a warning that this addon will NOT work (user should disable it).
   local msg="|cffFF2222M1PENG has no localization for |r"..GetLocale().."|cffFF2222 - it will not work !!!"
   if LT then msg=LT.UsageHelp[1] end
   DEFAULT_CHAT_FRAME:AddMessage(msg)
   this:RegisterEvent("PLAYER_ENTERING_WORLD")
   end;


OnEvent = function(ev)
   -- (Re)init state upon entering/leaving a BG. After disconn, this seems not to work,
   -- so it is done in OnUpdate() too.
   if ev=="PLAYER_ENTERING_WORLD" then MMMUpdate() end
   end;


OnUpdate = function(el)

   elapsed0=elapsed0+el
   if elapsed0<0.05 then return end

   -- PART 1: things to be done relatively often (each 2nd or 3rd frame)

   -- for zonesAlfade==nil: we have no fading of zones now; otherwise zonesAlfade says direction 
   -- (fade in for zonesAlfade>0, fade out for zonesAlfade<0) and speed (change by zonesAlfade 
   -- units per second). A fade-in will rise zones alpha to 150% (altough they will be drawn with
   -- 100% alpha maximal), so a later fade-out will take some time to go from 150% under 100% alpha
   -- (this avoids a flickering fade-in-out-in-out when user is moving mouse from zone to zone; the
   -- fade-in of all zones is started when entering a zone, fade-out of all zones is started when 
   -- leaving a zone).
   if zonesAlfade then
      -- compute new zonesAlfa in a
      local a=zonesAlfa+zonesAlfade*elapsed0
      -- fade-out completed (a<minimum): zonesAlfade=nil, zonesAlfa=minimum
      if a<zonesAlfalo then a,zonesAlfade=zonesAlfalo
      -- fade-in completed (a>1 and fading-in): zonesAlfade=nil,zonesAlfa=150%,shown_alfa=100%
      elseif a>1 and zonesAlfade>0 then a,zonesAlfa,zonesAlfade=1,1.5 
      else zonesAlfa=a end
      -- set changed alpha for all zone buttons
      for i=201,useFid[2] do getglobal("M1PengF"..i):SetAlpha(a) end
      end

   -- My playerminiarrow frame is bound to WOW's playerminiarrow frame, so WOW does the positioning,
   -- but we must update the facing (rotating texcoords of our arrowtexture according to WOW's arrow
   -- facing). Doing this outside of BGs,is not harmful (WOW's arrowframe exists, my arrow is hidden).
   if M1PengParrowT then
      local a = 0.5+PlayerMiniArrowFrame:GetFacing()*0.159155
      local ULx,ULy=Tex1Circle(0.484375,0.578125,0.6,0.375-a)
      local URx,URy=Tex1Circle(0.484375,0.578125,0.6,0.125-a)
      local LRx,LRy=Tex1Circle(0.484375,0.578125,0.6,0.875-a)
      local LLx,LLy=Tex1Circle(0.484375,0.578125,0.6,0.625-a)
      M1PengParrowT:SetTexCoord(ULx,ULy,LLx,LLy,URx,URy,LRx,LRy)
      end

   elapsed1=elapsed1+elapsed0 elapsed0=0
   if elapsed1<1.0 then return end

   -- PART 2: things to be done not so often (once per second is good for playerblips blinkrate)

   -- All tests like "if not currentPlayer then MMMUpdate() end" didn't help. If I want to have
   -- a reinit after disconnect, only a re-check of MMMUpdate() each 1-3 sec helps. MMMUpdate()
   -- exits as long the minimap is not displayed (can be after DC), and MMMUpdate() exits fast,
   -- if nothing is to do (currentBG not changed), so we have no performance problem here.
   MMMUpdate()
      
   if not currentBG then return end

   -- Forget the 1st click of a 2-click-message, if the user waits to long until 2nd click
   if clik1Timeo>0 then
      clik1Timeo=clik1Timeo-elapsed1
      if clik1Timeo<=0 and clik1 then 
         getglobal("M1PengF"..clik1):UnlockHighlight() 
         clik1=nil end
      end

   -- Color playerblips by group and let them blink, when player is attacked. Our update period
   -- of 1 second yields a comfortable blinkrate.
   SetRaidPicos()

   -- Draw all timers. This can call special gettimer functions. Some gettimer functions (like
   -- win estimation in AB or EotS) wait some getifElapsed seconds, until they recalculate the
   -- win estimation).  Our update period of 1 second  matches to the printed remainig time in 
   -- timers, which decreases each second.
   getifElapsed=getifElapsed+elapsed1 elapsed1=0    
   DrawTimers()
   
   end;

}; -- ENDFRAME M1PengEvha