-- LinkWrangler

-- Original Author: Fallan/Jenavive
-- Programming Credits / Other authors: Legorol, Amavana
-- Suggestions:  DwFMagik, dreamtgm, Crusadebank

-------------------------------------------------------------------------------------------
-- VARIABLES
-------------------------------------------------------------------------------------------
-- Note: Globals are at the end of the file

local LWVersion = 1.74
-- Version info: fix release

-- Control variables
local LWDebugEnable = false
local LWMasterEnable = true

-- variables for ItemRef hook
local LWOriginalSetItemRef

-- internal jump tables
local LWQuickActionJump			= {}
local LWSlashJump 			= {}

-- local references to external data tables
local LWA = LinkWranglerAlias		-- from Alias.lua (loaded before this file)
LinkWranglerAlias = nil
local LWL = LinkWranglerLocal		-- from localization.lua (loaded before this file)
LinkWranglerLocal = nil
local LWLM = LWL.Message		-- output message table
-- Placeholders for subtables of LinkWranglerSaved
local LWSConfig, LWSClick, LWSButtons, LWSLayout, LWSAddOns

-- data tables for status of tooltip and compare windows
local LWTooltipData			= {}	-- central access for tooltip data
local LWTooltipDataCount		= 0	-- current number of tooltip frames
local LWCompareData			= {}	-- central access for compare tooltip data
local LWCompareDataCount		= 0	-- current number of compare frames

-- table storing details of callback functions from other addons
local LWCallbackData = {
	enable = {},
	enablecomp = {},
	list = {},
	refresh = {},
	}
-- Callback Types are codes to tell the RegisterCallback function what to do
-- As this is still in development, codes may be changed at a later date
local LWCallbackType = {
	refresh = 1, redirect = 1,
	show =2, hide = 2, showcomp = 2, hidecomp = 2, refreshcomp = 2, maximize = 2, minimize = 2,
	allocate = 4, allocatecomp = 5,
	destroy = 0, destroycomp = 0, scan = 0,
	}

-- variables & constants for various timers
local LWTimerFrame			-- set in main OnLoad handler
local LWTimerTimeoutList		= {} -- table to hold partially-loaded windows
local LWTimerTimeoutElapsed		= 0
local LWTimerResizeList			= {} -- frames that need resize on next update
local LWTimerRefreshList		= {} -- frames that need refresh event on next update
local LWTimerRedrawList			= {} -- frames that need to be redrawn; change detected after last refresh
local LWTimerRedrawElapsed		= 0

-- Whisper button: buffer tables for storing chat lines and names
local LWChatBufferTextList		= {}
local LWChatBufferNameList		= {}
local LWChatBufferIndex			= 1	-- index of next message to be stored
local LWChatBufferStored		= 0	-- number of messages actually stored

-- Compare button: inventory hook-flag and data
local LWInventoryHooked			= false
local LWInventorySlotTable -- prototype as local
do -- create closure for 'g'
local g = GetInventorySlotInfo
LWInventorySlotTable = {
	[INVTYPE_2HWEAPON]		= {[1] = g"MAINHANDSLOT",
						[2] = g"SECONDARYHANDSLOT"},
	[INVTYPE_AMMO]			= {[1] = g"AMMOSLOT"},
	[INVTYPE_BODY]			= {[1] = g"SHIRTSLOT"},
	[INVTYPE_CHEST]			= {[1] = g"CHESTSLOT"},
	[INVTYPE_CLOAK]			= {[1] = g"BACKSLOT"},
	[INVTYPE_FEET]			= {[1] = g"FEETSLOT"},
	[INVTYPE_FINGER]		= {[1] = g"FINGER0SLOT",
						[2] = g"FINGER1SLOT"},
	[INVTYPE_HAND]			= {[1] = g"HANDSSLOT"},
	[INVTYPE_HEAD]			= {[1] = g"HEADSLOT"},
	[INVTYPE_HOLDABLE]		= {[1] = g"SECONDARYHANDSLOT"},
	[INVTYPE_LEGS]			= {[1] = g"LEGSSLOT"},
	[INVTYPE_NECK]			= {[1] = g"NECKSLOT"},
	[INVTYPE_RANGED]		= {[1] = g"RANGEDSLOT"},
	[INVTYPE_RANGEDRIGHT]		= {[1] = g"RANGEDSLOT"},
	[INVTYPE_RELIC]			= {[1] = g"RANGEDSLOT"},
	[INVTYPE_ROBE]			= {[1] = g"CHESTSLOT"},
	[INVTYPE_SHIELD]		= {[1] = g"SECONDARYHANDSLOT"},
	[INVTYPE_SHOULDER]		= {[1] = g"SHOULDERSLOT"},
	[INVTYPE_TABARD]		= {[1] = g"TABARDSLOT"},
	[INVTYPE_THROWN]		= {[1] = g"RANGEDSLOT"},
	[INVTYPE_TRINKET]		= {[1] = g"TRINKET0SLOT",
						[2] = g"TRINKET1SLOT"},
	[INVTYPE_WAIST]			= {[1] = g"WAISTSLOT"},
	[INVTYPE_WEAPON]		= {[1] = g"MAINHANDSLOT",
						[2] = g"SECONDARYHANDSLOT"},
	[INVTYPE_WEAPONMAINHAND]	= {[1] = g"MAINHANDSLOT"},
	[INVTYPE_WEAPONOFFHAND]		= {[1] = g"SECONDARYHANDSLOT"},
	[INVTYPE_WRIST]			= {[1] = g"WRISTSLOT"},
	}
end -- closure

-- variables for interaction with Auctioneer/EnhTooltip
local LWEnhTooltipLoaded		= false

--[[
Fix for handling addons trying to register before LWSAddOns gets loaded {fix1731}
Temporarily store in this table, until they can be processed properly
Appears this happens where another AddOn uses the "ADDON_LOADED" event,
and it loads before LinkWrangler.
--]]
local LWEarlyLoadAddOns = {}

-------------------------------------------------------------------------------------------
-- DEBUG AND ERROR FUNCTIONS AND VARIABLES
-------------------------------------------------------------------------------------------

local function LWPrintDebug(msg)
	if LWDebugEnable then
		DEFAULT_CHAT_FRAME:AddMessage("LWDebug: "..msg,0.8,0.3,0.9)
	end
end

-- table that can be used to store session values. All keys and values must be printable
--local LWLocalDebug = {}

-------------------------------------------------------------------------------------------
-- INTERNAL UTILITY FUNCTIONS
-------------------------------------------------------------------------------------------

local function LWPrint (msg, noHeader, noVerbose)
	if noVerbose and not LWSConfig.verbose then
		return
	end
	if not noHeader then
		msg = "LinkWrangler: "..msg
	end
	DEFAULT_CHAT_FRAME:AddMessage(msg,0.8,0.3,0.9)
end

local function LWReportError (msg, level)
	if LWDebugEnable then
		if not level then
			level = 2
		elseif level > 0 then
			level = level + 1
		end
		error (msg, level)
	else
		LWPrint ("LinkWrangler Error: "..msg, true, true)
	end
	return nil, msg
end

local function LWBoolStr (option)
	return option and LWLM.ENABLED or LWLM.DISABLED
end

-------------------------------------------------------------------------------------------
-- SAVED VARIABLE DEFAULTS AND CHECKS
-------------------------------------------------------------------------------------------

local function LWGetDefaultClickTable ()
	return {
		action = "open",
		shift = "bypass",
		ctrl = "bypass",
		alt = "open",
	}
end

local function LWGetDefaultSaveTable ()
	return {
		Version = LWVersion,
		Config = {
			verbose = true,
			alllinks = true,
			savelayout = true,
			maxwindows = 100,
			scale = 1,
			--alpha = 1,
		},
		Click = {
			LeftButton = LWGetDefaultClickTable (),
			RightButton = LWGetDefaultClickTable (),
			MiddleButton = LWGetDefaultClickTable (),
		},
		Buttons = {
			close = true,
			minimize = true,
			compare = true,
			whisper = true,
			relink = true,
			dressup = true,
			capture = true,
		},
		AddOns = {},
		Layout = {},
	}
end

local function LWSetLocalVariables ()
	LWSConfig = LinkWranglerSaved.Config
	LWSClick = LinkWranglerSaved.Click
	LWSButtons = LinkWranglerSaved.Buttons
	LWSLayout = LinkWranglerSaved.Layout
	LWSAddOns = LinkWranglerSaved.AddOns
end

-- Warning: self deleting function
function LinkWranglerRunOnce_CheckSavedVariables ()
	if type (LinkWranglerSaved) ~= "table" then
		-- First time use, or saved variables file deleted or badly corrupted
		LWPrint (LWLM.DEFAULTSAVED)
		LinkWranglerSaved = LWGetDefaultSaveTable ()
	elseif LinkWranglerSaved.Version ~= LWVersion then
		-- Generate new saved variables table and copy in valid values from old table
		-- This only happens when the version changes
		LWPrint (LWLM.TESTSAVED)
		local newsaved = LWGetDefaultSaveTable ()
		local newtable, oldtable
		-- Config
		oldtable = LinkWranglerSaved.Config
		if type (oldtable) == "table" then
			newtable = newsaved.Config
			for option, value in pairs (newtable) do
				if type (oldtable[option]) == type (value) then
					newtable[option] = oldtable[option]
				end
			end
		end
		-- Click
		oldtable = LinkWranglerSaved.Click
		if type (oldtable) == "table" then
			newtable = newsaved.Click
			for button, newsub in pairs (newtable) do
				local oldsub = oldtable[button]
				if type (oldsub) == "table" then
					for option, value in pairs (newsub) do
						if type (oldsub[option]) == type (value) then
							newsub[option] = oldsub[option]
						end
					end
				end
			end
		end
		-- Buttons
		oldtable = LinkWranglerSaved.Buttons
		if type (oldtable) == "table" then
			newtable = newsaved.Buttons
			for option, value in pairs (newtable) do
				if type (oldtable[option]) == type (value) then
					newtable[option] = oldtable[option]
				end
			end
		end
		-- Layout
		oldtable = LinkWranglerSaved.Layout
		if newsaved.Config.savelayout and type (oldtable) == "table" then
			newtable = newsaved.Layout
			local counter = 1
			repeat
				local savestring = oldtable[counter]
				if type (savestring) == "string" then
					savestring = strmatch (savestring, "%-?%d+,%-?%d+")
					newtable[counter] = savestring -- if nil, changes nothing
				end
				counter = counter + 1
			until not savestring
		end
		-- AddOns
		oldtable = LinkWranglerSaved.AddOns
		if type (oldtable) == "table" then
			newtable = newsaved.AddOns
			for option, value in pairs (oldtable) do
				if type (value) == "number" then
					newtable[option] = value
				end
			end
		end

		LinkWranglerSaved = newsaved
	end

	LWSetLocalVariables ()

	-- Self-delete
	LinkWranglerRunOnce_CheckSavedVariables = nil
end

-------------------------------------------------------------------------------------------
-- SAVED VARIABLE FUNCTIONS
-------------------------------------------------------------------------------------------

local function LWLogoutSaveLayout ()
	local x, y

	if not LWSConfig.savelayout then
		return
	end

	for frame, info in pairs (LWTooltipData) do
		x = floor(frame:GetLeft()*LWSConfig.scale+.5)
		y = floor(frame:GetTop()*LWSConfig.scale-GetScreenHeight()+.5)
		LWSLayout[info.index] = x..","..y
	end
end

local function LWSetFrameLayout (frame, index)
	local x, y, saved
	local scale = LWSConfig.scale
	frame:ClearAllPoints()
	-- try to restore saved layout for this frame
	if LWSConfig.savelayout then
		saved = LWSLayout[index]
		if saved then
			x,y = strmatch (saved, "(%-?%d+),(%-?%d+)")
			if x and y then
				x = tonumber (x)
				y = tonumber (y)
				if x and y then
					frame:SetPoint ("TOPLEFT",nil,"TOPLEFT",x/scale,y/scale)
					return
				end
			end
			LWReportError ("Invalid saved layout string found: "..saved, 1)
		end
	end
	-- use default layout
	local screenx = GetScreenWidth()
	local screeny = GetScreenHeight()
	if index > 10 then
		x = screenx /2 -70
		y = -.7 *screeny
	elseif index > 5 then
		x = (index-6) *screenx /5 +30
		y = -.5 *screeny
	else -- index in range 1-5
		x = (index-1) *screenx /5 +30
		y = -.3 * screeny
	end
	frame:SetPoint ("TOPLEFT", nil, "TOPLEFT", x/scale, y/scale)
end

local function LWLogoutSaveAddOns ()
	local enable = LWCallbackData.enable
	local enablecomp = LWCallbackData.enablecomp
	local bor = bit.bor
	local saved

	for count, addon in ipairs (LWCallbackData.list) do
		saved = bor (enable[addon] and 0 or 1, enablecomp[addon] and 0 or 2)
		if saved == 0 then saved = nil end
		LWSAddOns[addon] = saved
	end
end

local function LWSetAddOnSaved (addon)
	local enable = true
	local enablecomp = true
	local band = bit.band
	local saved = LWSAddOns[addon]

	if saved then
		if band (saved, 1) ~= 0 then
			enable = false
		end
		if band (saved, 2) ~= 0 then
			enablecomp = false
		end
	end

	LWCallbackData.enable[addon] = enable
	LWCallbackData.enablecomp[addon] = enablecomp

end

local function LWSetAddOnFirstCallback (addon)
	if LWSAddOns then
		LWSetAddOnSaved (addon)
	else
		-- fix for addons which try to register before LWSAddOns gets loaded {fix1731}
		LWEarlyLoadAddOns[addon] = true
		--LWPrintDebug (format("Stored AddOn %s registered before variables loaded", addon))
		-- set temporary values for now
		LWCallbackData.enable[addon] = true
		LWCallbackData.enablecomp[addon] = true
	end

	tinsert (LWCallbackData.list, addon)
end

-------------------------------------------------------------------------------------------
-- ADDON SUPPORT
-------------------------------------------------------------------------------------------

local function LWCallbackRefresh (frame, link)
	-- refresh gets special handling as it is called most frequently
	local enable = LWCallbackData.enable
	-- refresh callback event
	for addon, func in pairs (LWCallbackData.refresh) do
		if enable[addon] then
			func (frame, link)
		end
	end
	frame:Show() -- check frame size and layout
end

local function LWCallbackAction (frame, link, request, compare)
	if LWCallbackData[request] then
		local enable
		if compare then
			enable = LWCallbackData.enablecomp
		else
			enable = LWCallbackData.enable
		end
		for addon, func in pairs (LWCallbackData[request]) do
			if enable[addon] then
				func (frame, link, request)
			end
		end
	end
end

local function LWRegisterCallback (addonname, callbackfunc, ...)
	local errorstring
	local numparams = select("#",...)
	local redirectlevel = (numparams > 0 and select (1,...) == "redirect") and 3 or 2
	-- error checking
	if type(callbackfunc) == "string" then
		callbackfunc = getglobal(callbackfunc)
		if not callbackfunc then
			errorstring = "Function name must evaluate to a valid function"
		end
	end
	if type(addonname) ~= "string" or #addonname < 1 then
		errorstring = "Addon name must be a valid string"
	elseif type(callbackfunc) ~= "function" and type(callbackfunc) ~= "nil" then
		errorstring = "Callback function parameter must be a valid function, global function name, or nil"
	elseif not errorstring then
		-- The Work Stuff
		if type (LWCallbackData.enable[addonname]) ~= "boolean" then -- check if AddOn already registered
			LWSetAddOnFirstCallback (addonname)
		end
		if numparams < 1 then -- Default action when no extra params passed
			callbacktable.refresh = callbackfunc
		end
		for i=1,numparams do
			local param = (select(i,...))
			if type(param) ~= "string" then
				errorstring = "Callback request parameters must be strings, or nil"
				break
			else
				param = strlower (param)
			end
			local requesttype = LWCallbackType[param]
			if not requesttype then
				errorstring = "Unknown callback request parameter: "..param
				break
			end
			if requesttype == 1 then
				LWCallbackData.refresh[addonname] = callbackfunc
			elseif requesttype >= 2 then
				local requesttable = LWCallbackData[param]
				if not requesttable then
					requesttable = {}
					LWCallbackData[param] = requesttable
				end
				requesttable[addonname] = callbackfunc
			end
			if callbackfunc then
				if requesttype == 4 then
					for frame,_ in pairs (LWTooltipData) do
						callbackfunc (frame, nil, param)
					end
				elseif requesttype == 5 then
					for frame, _ in pairs (LWCompareData) do
						callbackfunc (frame, nil, param)
					end
				end
			end
		end
	end

	if errorstring then
		return LWReportError (errorstring, redirectlevel)
	end
	return true
end

local function LWModifyCallback (addonRef, command, extra)
	local addonname
	local enable = LWCallbackData.enable
	local enablecomp = LWCallbackData.enablecomp
	local errorstring

	if type(addonRef) == "string" then
		addonname = addonRef
	elseif type(addonRef) == "number" then
		addonname = LWCallbackData.list[addonRef]
	end

	if command == "test" then -- special handling
		if addonname and enable[addonname] ~= nil then
			return true, enable[addonname], enablecomp[addonname]
		end
		return nil -- do not return any errors for 'test' command
	end

	if not addonname then
		errorstring = "Must provide a valid AddOn name or number"
	elseif enable[addonname] == nil then
		errorstring = "AddOn "..addonname.." is not registered with LinkWrangler"
	elseif type (command) ~= "string" then
		errorstring = "Command parameter must be a string"
	elseif command == "enable" then
		enable[addonname] = true
	elseif command == "disable" then
		enable[addonname] = false
	elseif command == "enablecomp" then
		enablecomp[addonname] = true
	elseif command == "disablecomp" then
		enablecomp[addonname] = false
	else
		errorstring = "Invalid command parameter: "..command
	end

	if errorstring then
		return LWReportError (errorstring, 2)
	end
	return true, enable[addonname], enablecomp[addonname]
end

-- this function may be unnecessary - consider deleting in future?
local function LWGetLinkInfo (data)
	local frame, info
	local datatype = type(data)
	if datatype == "string" then
		for fra, inf in pairs (LWTooltipData) do
			if (inf.link == data or inf.textlink == data) and inf.state > 0 then
				frame = fra
				info = inf
				break
			end
		end
	elseif datatype == "table" then -- assume it's a frame
		frame = data
		info = LWTooltipData[frame]
	end
	if info then
		return frame, info.state, info.link, info.textlink, info.name, info.whisper
	end
end

-------------------------------------------------------------------------------------------
-- TIMER FUNCTIONS
-------------------------------------------------------------------------------------------
--[[
LinkWrangler currently uses a single OnUpdate handler, attached to the Event frame
It handles the various windows through a set of lists held in tables
Timer is started or stopped by Showing or Hiding the Event frame
--]]

local function LWStopTimer ()
	LWTimerFrame:Hide()
	-- clear out lists
	for frame,_ in pairs (LWTimerTimeoutList) do
		LWTimerTimeoutList[frame] = nil
	end
	for frame,_ in pairs (LWTimerRedrawList) do
		LWTimerRedrawList[frame] = nil
	end
	for frame,_ in pairs (LWTimerResizeList) do
		LWTimerResizeList[frame] = nil
	end
	for frame,_ in pairs (LWTimerRefreshList) do
		LWTimerRefreshList[frame] = nil
	end
end

local function LWStartRefreshTimer (frame, info)
	LWTimerRefreshList[frame] = info
	LWTimerFrame:Show()
end

local function LWStartResizeTimer (frame, info)
	LWTimerResizeList[frame] = info
	LWTimerFrame:Show()
end

local function LWStartTimeoutTimer (frame, info)
	LWTimerTimeoutList[frame] = info
	LWTimerTimeoutElapsed = 0 -- reset elapsed time
	LWTimerFrame:Show()
end

local function LWStartRedrawTimer (frame, info)
	LWTimerRedrawList[frame] = info
	LWTimerRedrawElapsed = 0 -- reset elapsed time
	LWTimerFrame:Show()
end

-------------------------------------------------------------------------------------------
-- COMPARE TOOLTIP FUNCTIONS
-------------------------------------------------------------------------------------------

local function LWGetAvailableCompare ()
	-- check for existing available compare frame
	for fra, inf in pairs (LWCompareData) do
		if inf.state == 0 then
			return fra, inf
		end
	end

	-- no available frame so create new compare frame
	LWCompareDataCount = LWCompareDataCount + 1
	local frameName = "LinkWranglerCompare"..LWCompareDataCount
	local frame = CreateFrame ("GameTooltip", frameName, UIParent, "LinkWranglerCompareTemplate")

	if not frame then
		local errmsg = "Error creating Compare frame number "..LWCompareDataCount
		return LWReportError (errmsg, 1)
	end

	-- create new data subtable
	local info = {
		buttonFrames = {
			close = getglobal(frameName.."CloseButton"),
			relink = getglobal(frameName.."RelinkButton"),
			capture = getglobal(frameName.."CaptureButton"),
		},
		state = 0,
	}
	LWCompareData[frame] = info
	frame:SetScale(LWSConfig.scale)
	--frame:SetAlpha(LWSConfig.alpha)

	LWCallbackAction (frame, nil, "allocatecomp", true)

	return frame, info
end

local function LWShowCompareFrame (frame, link, anchor)
	local info = LWCompareData[frame]
	local closeButton = info.buttonFrames.close
	local relinkButton = info.buttonFrames.relink
	local captureButton = info.buttonFrames.capture
	local offset = 0
	local padding = 0

	-- setup and display frame, record state, do callbacks - if needed
	if link and (not frame:IsShown() or info.link ~= link) then
		frame:SetOwner(UIParent, "ANCHOR_PRESERVE")
		--frame:ClearLines() -- todo: is this needed?
		frame:SetHyperlink (link)
		info.link = link
		info.state = 1
		--frame:SetAlpha (LWSConfig.alpha)
		LWCallbackAction (frame, link, "showcomp", true)
		LWCallbackAction (frame, link, "refreshcomp", true)
	end

	-- button layout and padding
	if LWSButtons.close then
		closeButton:Show()
		offset = -20
		padding = 16
	else
		closeButton:Hide()
	end
	if LWSButtons.relink then
		relinkButton:SetPoint ("TOPRIGHT", closeButton, "TOPRIGHT", 0, offset)
		relinkButton:Show()
		offset = offset - 20
		padding = 16
	else
		relinkButton:Hide()
	end
	if LWSButtons.capture then
		captureButton:SetPoint ("TOPRIGHT", closeButton, "TOPRIGHT", 0, offset)
		captureButton:Show()
		padding = 16
	else
		captureButton:Hide()
	end

	frame:SetPadding (padding)

--[[ 20.07.08
Note that LW does not check the height of buttons in Compare tooltips:
We assume that all equippable items will generate a sufficient number of lines in the tooltip that we don't need to worry
May need to look at this later though
--]]

	-- set anchor
	if anchor then
		frame:ClearAllPoints()
		frame:SetPoint("BOTTOMLEFT",anchor,"TOPLEFT",0,1)
	end
	frame:Raise()
	frame:Show() -- checks layout and resizes tooltip
end

local function LWHideCompareFrame (compareframe)
	local compareinfo = LWCompareData[compareframe]
	if compareframe:IsShown() then
		LWCallbackAction (compareframe, compareinfo.link, "hidecomp", true)
	end
	compareframe:Hide()
	-- unlink from main tooltip
	LWTooltipData[compareinfo.parentTooltip].compareFrames[compareinfo.compareIndex] = nil
	-- clear data
	compareinfo.state = 0
	compareinfo.parentTooltip = nil
	compareinfo.compareIndex = nil
	compareinfo.link = nil
end

local function LWCloseOneCompareFrame (compareframe)
	-- Function dealing with the case where only 1 compare tooltip is closed
	-- There may be another compare tooltip which requires adjustment
	local compareinfo = LWCompareData[compareframe]
	local tooltipframe = compareinfo.parentTooltip
	local tooltipinfo = LWTooltipData[tooltipframe]
	local frametable = tooltipinfo.compareFrames
	local otherframe, otheranchor

	tooltipinfo.compare = tooltipinfo.compare - 1
	if tooltipinfo.compare < 0 then
		for index, cframe in pairs (tooltipinfo.compareFrames) do
			LWHideCompareFrame (cframe)
		end
		tooltipinfo.compare = 0 -- try to recover
		return LWReportError ("compare windows counter is negative", 1)
	end

	-- if the other compare frame is anchored to the frame we just hid,
	-- then we want to anchor it to the parent tooltip frame instead
	-- first run checks to see if this is needed, and save the result for later
	if compareinfo.compareIndex == 1 then
		otherframe = frametable[2]
		if otherframe then
			local _,anchor = otherframe:GetPoint()
			if anchor == compareframe then
				_,anchor = compareframe:GetPoint()
				if anchor == tooltipframe then
					otheranchor = tooltipframe
				end
			end
		end
	end

	LWHideCompareFrame (compareframe) -- this will also remove compareframe from tooltipinfo.compareFrames

	if otherframe then
		LWCompareData[otherframe].compareIndex = 1
		frametable[1]=otherframe -- renumber other tooltip in tooltipinfo.compareFrames table
		frametable[2]=nil
		LWShowCompareFrame (otherframe, nil, otheranchor) -- re-anchor the other compare tooltip
	end

	-- error checking, in case something went wrong...
	if tooltipinfo.compare ~= #(tooltipinfo.compareFrames) then
		for index, cframe in pairs (tooltipinfo.compareFrames) do
			LWHideCompareFrame (cframe)
		end
		tooltipinfo.compare = 0
		return LWReportError ("Compare frame counter doesn't match compareFrames table", 1)
	end
end

local function LWOpenCompareWindow (tframe, tinfo, clink)
	local cframe, cinfo = LWGetAvailableCompare ()
	if not cframe then
		return nil, cinfo
	end

	local compareindex = tinfo.compare + 1
	if compareindex > 2 then
		return LWReportError ("Attempt to add more than 2 Compare windows to "..tframe:GetName(), 1)
	end
	local anchor = (compareindex > 1) and tinfo.compareFrames[compareindex - 1] or tframe
	tinfo.compareFrames[compareindex] = cframe
	tinfo.compare = compareindex
	cinfo.compareIndex = compareindex
	cinfo.parentTooltip = tframe
	cinfo.link = nil
	LWShowCompareFrame (cframe, clink, anchor)
	if cinfo.state ~= 1 then
		LWCloseOneCompareFrame (cframe) -- attempt to recover
		return LWReportError ("Opened Compare window has incorrect state", 1)
	end

	return true
end

local function LWCheckCompareButtonStatus (frame, info)
	if info.state == 1 then
		local match = false
		if info.numslots > 0 then
			LWInventoryHooked = true
			for s = 1, info.numslots do
				local test = GetInventoryItemLink("player", info.slots[s])
				if test then
					match = true
					break
				end
			end
		end
		if match then
			info.buttonFrames.compare:SetButtonState("NORMAL")
		else
			info.buttonFrames.compare:SetButtonState("DISABLED")
		end
	end
end

local function LWCheckInventory ()
	LWInventoryHooked = false
	if not LWMasterEnable or not LWSButtons.compare then
		return
	end
	for frame, info in pairs (LWTooltipData) do
		LWCheckCompareButtonStatus (frame, info)
	end
end

-------------------------------------------------------------------------------------------
-- MAIN TOOLTIP FUNCTIONS
-------------------------------------------------------------------------------------------

local function LWResetTooltipData (info)
	info.state = 0
	info.minimized = false
	info.buttonHeight = 0
	info.compare = 0
	info.EnhTooltipEmbedded = nil
	info.externalLink = nil
	info.openMinimize = nil
	info.link = nil
	info.textlink = nil
	info.whisper = nil
	info.name = nil
	info.slots = nil
	info.numslots = 0
	info.canDressup = false
	info.countSetLink = 0
	info.refreshStatus = 0
	info.originalLink = nil
end

local function LWCloseTooltipWindow (frame, inhibit)
	local info = LWTooltipData [frame]

	if info.state > 0 then
		LWCallbackAction (frame, info.link, "hide")
	end

	LWResetTooltipData (info)

	-- ensure frame is not held in timer lists
	LWTimerTimeoutList[frame] = nil
	LWTimerResizeList[frame] = nil
	LWTimerRefreshList[frame] = nil
	LWTimerRedrawList[frame] = nil

	-- hide all child compare frames
	for index, cframe in pairs (info.compareFrames) do
		LWHideCompareFrame (cframe)
	end
	frame:Hide()

	if not inhibit then -- inhibit when closing all windows or when window not fully open
		LWCheckInventory ()
	end
end

local function LWCloseAllAction()
	LWStopTimer()
	for frame, _ in pairs (LWTooltipData) do
		LWCloseTooltipWindow (frame, true)
	end
	LWInventoryHooked = false
end

local function LWCreateTooltipFrame ()
	if LWTooltipDataCount >= LWSConfig.maxwindows then -- maximum number of frames reached
		LWPrint (LWLM.MAXWINDOWS,nil,true)
		return
	end

	LWTooltipDataCount = LWTooltipDataCount + 1
	local frameName = "LinkWranglerTooltip"..LWTooltipDataCount
	local frame = CreateFrame ("GameTooltip", frameName, UIParent, "LinkWranglerTooltipTemplate")

	if not frame then
		return LWReportError ("Error creating Tooltip frame number "..LWTooltipDataCount, 1)
	end

	tinsert (UISpecialFrames, frameName) -- register for ESC key
	frame:SetScale(LWSConfig.scale)
	--frame:SetAlpha(LWSConfig.alpha)
	LWSetFrameLayout (frame, LWTooltipDataCount) -- set starting position

	-- start constructing data table for this window
	local info = {
		-- "constant" values
		index = LWTooltipDataCount,
		titleFrame = getglobal(frameName.."TextLeft1"), -- frame containing Title text
		-- sub-tables
		buttonFrames = {
			close = getglobal(frameName.."CloseButton"),
			minimize = getglobal(frameName.."MinButton"),
			compare = getglobal(frameName.."CompButton"),
			whisper = getglobal(frameName.."WhisperButton"),
			relink = getglobal(frameName.."RelinkButton"),
			dressup = getglobal(frameName.."DressupButton"),
		},
		compareFrames = {},
	}
	-- Initialize all other data fields
	LWResetTooltipData (info)
	-- Add new table to the data array
	LWTooltipData[frame] = info
	-- Notify other AddOns of tooltip creation
	LWCallbackAction (frame, nil, "allocate", false)

	return frame, info
end

local function LWGetCheckAvailableTooltip (link)
	local frame, info
	local index = LWTooltipDataCount + 1
	-- Scan existing windows
	for fra, inf in pairs (LWTooltipData) do
		if inf.link == link then
			if (inf.state < 0) then -- timer is running for this link
				LWCloseTooltipWindow (fra)
				return fra, inf -- force immediate retry in same fra
			else
				-- close the window
				LWCloseTooltipWindow (fra)
				return nil, nil
			end
		elseif inf.state == 0 and inf.index <= index then
			frame = fra
			info = inf
			index = inf.index
		end
	end

	if not frame then -- Attempt to create new frame
		frame, info = LWCreateTooltipFrame ()
	end

	if frame and info then
		return frame, info
	end
	-- any errors have already been handled, just return nil
end

local function LWTooltipButtonLayout (frame, info)
	-- local references to buttons
	local closeButton = info.buttonFrames.close
	local minButton = info.buttonFrames.minimize
	local compareButton = info.buttonFrames.compare
	local whisperButton = info.buttonFrames.whisper
	local relinkButton = info.buttonFrames.relink
	local dressupButton = info.buttonFrames.dressup

	-- counter variables
	local offset = 0
	local height = 12
	local nobuttons = true

	if LWSButtons.close then
		closeButton:Show()
		offset = -20
		height = height + 20
		nobuttons = false
	else
		closeButton:Hide()
	end
	if info.minimized then
		if LWSButtons.minimize then
			minButton:SetPoint("TOPRIGHT",closeButton,"TOPRIGHT",offset,0)
			minButton:Show()
		else
			minButton:Hide()
		end
		compareButton:Hide()
		whisperButton:Hide()
		relinkButton:Hide()
		dressupButton:Hide()
	else
		if LWSButtons.minimize then
			minButton:SetPoint("TOPRIGHT",closeButton,"TOPRIGHT",0,offset)
			offset = offset - 20
			height = height + 20
			nobuttons = false
			minButton:Show()
		else
			minButton:Hide()
		end
		if LWSButtons.compare and info.slots then
			compareButton:SetPoint("TOPRIGHT",closeButton,"TOPRIGHT",0,offset)
			offset = offset - 20
			height = height + 20
			nobuttons = false
			compareButton:Show()
		else
			compareButton:Hide()
		end
		if LWSButtons.whisper and info.whisper then
			whisperButton:SetPoint("TOPRIGHT",closeButton,"TOPRIGHT",0,offset)
			offset = offset - 20
			height = height + 20
			nobuttons = false
			whisperButton:Show()
		else
			whisperButton:Hide()
		end
		if LWSButtons.relink and info.textlink then
			relinkButton:SetPoint("TOPRIGHT",closeButton,"TOPRIGHT",0,offset)
			offset = offset - 20
			height = height + 20
			nobuttons = false
			relinkButton:Show()
		else
			relinkButton:Hide()
		end
		if LWSButtons.dressup and info.canDressup then
			dressupButton:SetPoint("TOPRIGHT",closeButton,"TOPRIGHT",0,offset)
			offset = offset - 20
			height = height + 20
			nobuttons = false
			dressupButton:Show()
		else
			dressupButton:Hide()
		end

		frame:SetPadding (nobuttons and 0 or 16)
		info.buttonHeight = height
	end
end

-- keep this function synchronised with LWTooltipOnEnter
local function LWEnhTooltipAction (frame, info)
	if LWEnhTooltipLoaded then
		EnhTooltip.ClearTooltip()
		local linecount = frame:NumLines()
		EnhTooltip.TooltipCall(frame, info.name, info.textlink, -1, 1, 0, true, info.link)
		info.EnhTooltipEmbedded = frame:NumLines() > linecount -- check for embedding
	end
end

local function LWMinimizeTooltip (frame)
	local info = LWTooltipData[frame]
	local link = info.link

	if info.minimized then -- do maximize
		-- set indicators
		info.minimized = false
		info.state = 1
		LWCallbackAction (frame, link, "maximize")
		-- clear and redraw frame
		info.countSetLink = 0
		info.refreshStatus = 0
		LWTimerRedrawList[frame] = nil
		frame:SetHyperlink (link)
		frame:Raise()
		LWEnhTooltipAction (frame, info)
		LWStartRefreshTimer (frame, info)
	else -- do minimize
		-- set indicators
		info.minimized = true
		info.state = 2
		LWCallbackAction (frame, link, "minimize")
		-- ensure anchored by top left corner (other mods may set anchor to different points)
		frame:ClearAllPoints()
		frame:SetPoint ("TOPLEFT", nil, "TOPLEFT", frame:GetLeft(), frame:GetTop()-GetScreenHeight()/LWSConfig.scale)
		-- clear everything except the title
		local colR,colG,colB = info.titleFrame:GetTextColor()
		--frame:ClearLines() -- todo: is this needed?
		frame:SetText(info.name, colR, colG, colB)
		--frame:SetAlpha (LWSConfig.alpha * .9) -- not working??
		frame:Lower()
		-- close any other associated frames
		if LWEnhTooltipLoaded then
			local _,anchor = EnhancedTooltip:GetPoint()
			if anchor == frame then
				EnhTooltip.ClearTooltip()
			end
		end
		for _,cframe in pairs (info.compareFrames) do
			LWHideCompareFrame (cframe)
		end
		info.compare = 0
	end

	LWTooltipButtonLayout (frame, info)
	LWStartResizeTimer (frame, info)
	LWCheckCompareButtonStatus (frame, info)
	frame:Show()
end

local function LWMinimizeAllAction () -- used for keybinding
	-- scan for open / unminized tooltips
	local dropout = true
	local maxfound

	--scan to check states of tooltips
	for _, info in pairs (LWTooltipData) do
		if info.state > 0 then
			dropout = false -- found an open window
			if not info.minimized then
				maxfound = true
				break
			end
		end
	end

	if dropout then -- no further action
		return
	end

	if maxfound then -- at least 1 window open and not minimized
		for frame, info in pairs (LWTooltipData) do
			if not info.minimized then
				LWMinimizeTooltip (frame)
			end
		end
	else -- at least 1 window open and all minimized
		for frame,_ in pairs (LWTooltipData) do
			LWMinimizeTooltip (frame)
		end
	end
end

-------------------------------------------------------------------------------------------
-- SLASH HANDLER
-------------------------------------------------------------------------------------------

local function LWSlashUnknownCommand()
	LWPrint (format(LWLM.UNKNOWN, LWVersion, LWBoolStr(LWMasterEnable)))
end

local function LWSlashEnableDisable(option, bool)
	if (option == "enable") then
		LWMasterEnable = true
	elseif (option == "disable") then
		LWMasterEnable = false
	elseif (option == "toggle") then
		LWMasterEnable = not LWMasterEnable
	elseif (option == "switch") then
		LWMasterEnable = not (not bool)
	end
	if not LWMasterEnable then
		LWCloseAllAction()
	end
	LWPrint (LWBoolStr(LWMasterEnable))
end
LWSlashJump["enable"]=LWSlashEnableDisable
LWSlashJump["disable"]=LWSlashEnableDisable
LWSlashJump["toggle"]=LWSlashEnableDisable

local function LWSlashClickStatus (button, options) -- subfunction used by LWSlashStatus and LWSlashClick
	local textout = format (LWLM.CLICKSTATUS,(LWL.Display[button] or button))
	for action, code in pairs (options) do
		textout = textout.." ".. action.."="..code
	end
	LWPrint (textout, true)
end

local function LWSlashButtonStatus () -- subfunction used by LWSlashStatus and LWlashButtons
	local textout = LWLM.BUTTONS
	local none = true
	for but, bool in pairs (LWSButtons) do
		if (bool) then
			textout=textout.." "..but
			none = false
		end
	end
	if (none) then
		textout=textout.." none"
	end
	LWPrint (textout, true)
end

local function LWSlashStatus(x,subcommand)
	if subcommand == "default" then
		--LWPrintDebug "setting to default"
		LinkWranglerSaved = LWGetDefaultSaveTable ()
		LWSetLocalVariables ()
		LWPrint (LWLM.DEFAULTSAVED)
	elseif subcommand == "test" then
		LinkWranglerSaved.Version = 0 -- force test on next reload
		LWPrint (LWLM.RESETCHECK)
		return -- don't show current status for this option
	elseif subcommand == "purge" then
		LinkWranglerSaved.AddOns = {}
		LinkWranglerSaved.Layout = {}
		LWSetLocalVariables ()
		LWPrint (LWLM.PURGE)
		return
	end
	LWPrint (format(LWLM.STATUS, LWVersion, LWBoolStr(LWMasterEnable)))
	for clk, ops in pairs (LWSClick) do
		LWSlashClickStatus (clk, ops)
	end
	LWSlashButtonStatus ()
	for op, val in pairs (LWSConfig) do
		if type (val) == "boolean" then
			LWPrint (op.." : "..LWBoolStr(val), true)
		elseif type (val) == "number" or type (val) == "string" then
			LWPrint (op.." : "..val, true)
		end
	end
end
LWSlashJump["status"]=LWSlashStatus

local function LWSlashHelp(x,y)
	for i = 1, #(LWLM.HELP) do
		LWPrint (LWLM.HELP[i], true)
	end
end
LWSlashJump["help"]=LWSlashHelp

local function LWSlashList(x, y)
	LWPrint (LWLM.LIST)
	local enable = LWCallbackData.enable
	local enablecomp = LWCallbackData.enablecomp
	for count, addon in ipairs (LWCallbackData.list) do
		LWPrint (format (LWLM.LISTLINE, count, addon, LWBoolStr(enable[addon]), LWBoolStr(enablecomp[addon])), true)
	end
end
LWSlashJump["list"]=LWSlashList

local function LWSlashAddOns (option, extra)
	local _, pos, subcommand, tail, addon, addonnumber
	local enable = LWCallbackData.enable
	local enablecomp = LWCallbackData.enablecomp

	_,pos,subcommand = strfind (extra, "(%S+)")
	subcommand = LWA[subcommand] or subcommand or "list"
	if subcommand == "list" then
		return LWSlashList ()
	elseif subcommand == "purge" then
		LinkWranglerSaved.AddOns = {}
		LWSetLocalVariables ()
		LWPrint (LWLM.PURGE)
		return
	end

	if pos then
		_,_, tail = strfind (extra, "%s+(.+)", pos+1)
	end
	if tail then
		-- obtain addon's number and properly capitalized name
		addonnumber = tonumber (tail)
		if addonnumber then
			addon = LWCallbackData.list[addonnumber]
		end
		if not addon then
			for number, name in ipairs (LWCallbackData.list) do
				if tail == strlower (name) then
					addonnumber = number
					addon = name
					break
				end
			end
		end
	else
		tail = "<blank>"
	end
	if not addon then
		LWPrint (format(LWLM.UNKNOWNADDON, tail))
		return
	end

	if subcommand == "enable" then
		enable[addon] = true
	elseif subcommand == "disable" then
		enable[addon] = false
	elseif subcommand == "enablecomp" then
		enablecomp[addon] = true
	elseif subcommand == "disablecomp" then
		enablecomp[addon] = false
	else
		LWPrint (format(LWLM.INVALID, subcommand))
		return
	end
	LWPrint (format (LWLM.LISTLINE, addonnumber, addon, LWBoolStr(enable[addon]), LWBoolStr(enablecomp[addon])))
end
LWSlashJump["addons"]=LWSlashAddOns

local function LWSlashConfig(option, extra)
	extra = LWA[extra] or extra
	if (extra == "toggle" or extra == "") then
		LWSConfig[option] = not LWSConfig[option]
	elseif (extra == "enable") then
		LWSConfig[option] = true
	elseif (extra == "disable") then
		LWSConfig[option] = false
	else -- unknown extra param
		LWPrint(format (LWLM.USAGE, option))
	end
	LWPrint (option.." : "..LWBoolStr(LWSConfig[option]))
end
LWSlashJump["verbose"]=LWSlashConfig
LWSlashJump["alllinks"]=LWSlashConfig
LWSlashJump["savelayout"]=LWSlashConfig

local function LWSlashMaxwindows (command, extra)
	local val = tonumber (extra)
	if val then
		if val >= 1 and val <= 200 then
			LWSConfig.maxwindows = val
		else
			LWPrint (LWLM.MAXWINDOWSUSAGE)
		end
	else
		LWPrint (format(LWLM.INVALID, param))
	end
	LWPrint ("maxwindows : "..LWSConfig.maxwindows)
end
LWSlashJump["maxwindows"]=LWSlashMaxwindows

local function LWSlashScale (command, extra)
	local val = tonumber (extra)
	local oldval = LWSConfig.scale
	local x, y
	local screeny = GetScreenHeight ()
	if val then
		if val >= .2 and val <= 2.5 then
			LWSConfig.scale = val
			for frame,info in pairs (LWTooltipData) do
				--LWPrintDebug(format("before frame %s left %.2f top %.2f", frame:GetName(), frame:GetLeft(), frame:GetTop()))
				x = frame:GetLeft () * oldval / val
				y = (frame:GetTop () * oldval - screeny) / val
				frame:SetScale(val)
				frame:SetPoint ("TOPLEFT", nil, "TOPLEFT", x, y)
				if frame:IsVisible () then
					frame:Show()
					LWStartResizeTimer (frame, info) -- todo: are both of these needed?
				end
				--LWPrintDebug(format("after frame %s left %.2f top %.2f", frame:GetName(), frame:GetLeft(), frame:GetTop()))
			end
			for frame,_ in pairs (LWCompareData) do
				frame:SetScale(val)
			end
		else
			LWPrint (LWLM.SCALEUSAGE)
		end
	else
		LWPrint (format(LWLM.INVALID, param))
	end
	LWPrint ("scale : "..LWSConfig.scale)
end
LWSlashJump["scale"]=LWSlashScale

--[[ disabled code
local function LWSlashAlpha (command, extra)
	local val = tonumber (extra)
	if val then
		if val >= 0 and val <= 1 then
			LWSConfig.alpha = val
			for frame,_ in pairs (LWTooltipData) do
				frame:SetAlpha(val)
			end
			for frame,_ in pairs (LWCompareData) do
				frame:SetAlpha(val)
			end
		else
			LWPrint (LWLM.ALPHAUSAGE)
		end
	else
		LWPrint (format(LWLM.INVALID, param))
	end
	LWPrint ("alpha : "..LWSConfig.alpha)
end
LWSlashJump["alpha"]=LWSlashAlpha
-- end disabled code ]]

local function LWSlashClick (click, extra)
	LWPrint (format(LWLM.CLICK,LWL.Display[click]))
	local saved=LWSClick[click]
	-- special cases first
	if extra == "default" then
		LWSClick[click] = LWGetDefaultClickTable ()
	elseif extra == "disable" then
		saved.action = "bypass"
		saved.shift = "bypass"
		saved.ctrl = "bypass"
		saved.alt = "bypass"
	else
		local action, code, pos
		_, pos, action, code = strfind (extra, "(%w+)%W+(%w+)")
		if not action or not code then
			LWPrint (format(LWLM.INVALID,extra))
		else
			while pos do
				action = LWA[action] or action
				code = LWA[code] or code
				if not (action=="action" or action=="shift" or action=="ctrl" or action=="alt") then
					LWPrint (format(LWLM.INVALID,action))
				elseif not LWQuickActionJump[code] then
					LWPrint (format(LWLM.INVALID,code))
				else
					saved[action]=code
				end
				_,pos,action,code = strfind (extra, "(%w+)%W+(%w+)", pos+1)
			end
		end
	end

	LWSlashClickStatus (click, LWSClick[click])
end
LWSlashJump["LeftButton"]=LWSlashClick
LWSlashJump["RightButton"]=LWSlashClick
LWSlashJump["MiddleButton"]=LWSlashClick

local function LWSlashAllClick (c, extra)
	for click, _ in pairs (LWSClick) do
		LWSlashClick(click, extra)
	end
end
LWSlashJump["allclick"]=LWSlashAllClick

local function LWSlashButtons (cmd, extra)
	for p in string.gmatch (extra, "%S+") do
		local param = LWA[p] or p or "<unknown>"
		if (param == "closeonly") then
			LWSButtons.close     = true
			LWSButtons.minimize  = false
			LWSButtons.compare   = false
			LWSButtons.whisper   = false
			LWSButtons.relink    = false
			LWSButtons.dressup   = false
			LWSButtons.capture   = false
		elseif (param == "closemin") then
			LWSButtons.close     = true
			LWSButtons.minimize  = true
			LWSButtons.compare   = false
			LWSButtons.whisper   = false
			LWSButtons.relink    = false
			LWSButtons.dressup   = false
			LWSButtons.capture   = false
		elseif (param == "all" or param == "default") then
			LWSButtons.close     = true
			LWSButtons.minimize  = true
			LWSButtons.compare   = true
			LWSButtons.whisper   = true
			LWSButtons.relink    = true
			LWSButtons.dressup   = true
			LWSButtons.capture   = true
		elseif param == "close" then LWSButtons.close = true
		elseif param == "noclose" then LWSButtons.close = false
		elseif param == "minimize" then LWSButtons.minimize = true
		elseif param == "nominimize" then LWSButtons.minimize = false
		elseif param == "compare" then LWSButtons.compare = true
		elseif param == "nocompare" then LWSButtons.compare = false
		elseif param == "whisper" then LWSButtons.whisper = true
		elseif param == "nowhisper" then LWSButtons.whisper = false
		elseif param == "relink" then LWSButtons.relink = true
		elseif param == "norelink" then LWSButtons.relink = false
		elseif param == "dressup" then LWSButtons.dressup = true
		elseif param == "nodressup" then LWSButtons.dressup = false
		elseif param == "capture" then LWSButtons.capture = true
		elseif param == "nocapture" then LWSButtons.capture = false
		else
			LWPrint (format(LWLM.INVALID,param))
		end
	end
	LWSlashButtonStatus ()
end
LWSlashJump["buttons"]=LWSlashButtons

--[[ switchable debug code
local function LWSlashDebug (command, extra)
	local subcommand = strmatch (extra, "(%S+)")
	subcommand = LWA[subcommand] or subcommand
	--process debug commands
	if subcommand == "clear" and LWLocalDebug then
		for key,_ in pairs (LWLocalDebug) do
			LWLocalDebug[key] = nil
		end
	elseif subcommand == "enable" then
		LWDebugEnable = true
	elseif subcommand == "disable" then
		LWDebugEnable = false
	elseif subcommand == "toggle" then
		LWDebugEnable = not LWDebugEnable
	end
	-- display debug data
	LWPrint ("Debug : "..LWBoolStr(LWDebugEnable))
	if not LWDebugEnable then
		return
	end
	if LWLocalDebug then
		for chk, val in pairs (LWLocalDebug) do
			LWPrint (chk.." : "..val, true)
		end
	end
end
LWSlashJump["debug"]=LWSlashDebug
-- end switchable code ]]

local function LWdoSlashCommand (cmd)
	cmd = strlower (cmd)
	local pos, command, extra

	_,pos,command = strfind (cmd, "(%S+)")
	if pos then
		_,_,extra = strfind (cmd, "%s+(.+)", pos+1)
	end
	command = LWA[command] or command
	extra = extra or ""

	local sfunc= LWSlashJump[command]
	if sfunc then
		sfunc (command, extra)
		return
	end
	-- unknown - basic help
	LWSlashUnknownCommand()
end

-------------------------------------------------------------------------------------------
-- QUICK ACTION SUBFUNCTIONS
-------------------------------------------------------------------------------------------

function LWQuickActionJump.bypass(retval)
	retval.bypass = true
end

function LWQuickActionJump.open()
	-- do nothing
end

function LWQuickActionJump.openmin (retval)
	retval.openMinimize = true
end

function LWQuickActionJump.relink(retval,link,text)
--[[
--Recent changes to WoW itself prevent this function from working as intended
--This appears to be caused by the Blizzard_CombatLog AddOn
--]]
	-- use "official" link text, if it is in the cache
	local _,linktext = GetItemInfo(link)
	linktext = linktext or text

	if (linktext) then
		if ( not ChatFrameEditBox:IsVisible() ) then
			ChatFrame_OpenChat(linktext)
		else
			ChatFrameEditBox:Insert(linktext)
		end
		retval.dropout = true
		return
	end
	retval.bypass = true
end

function LWQuickActionJump.dressup (retval, link)
	local _,_,_,_,_,itemtype,_,_,equip = GetItemInfo(link)
	-- test to see if item is equippable
	if (equip ~= "" and equip ~= "INVTYPE_BAG" and equip ~= "INVTYPE_AMMO") then --or (itemtype == LW_TYPESTRING_RECIPE) then
		DressUpItemLink(link)
	else
		LWPrint (LWLM.NODRESSUP,nil,true)
	end

	retval.dropout = true
	return
end

-------------------------------------------------------------------------------------------
-- HOOK AND EXTERNAL LINK FUNCTIONS
-------------------------------------------------------------------------------------------

local function LWTooltipSetHyperlink (frame, link, info, textlink, state)
	frame:SetOwner(UIParent, "ANCHOR_PRESERVE") -- todo: is this needed?
	frame:ClearLines() -- todo: is this needed?

	info.link = link
	info.state = state
	info.textlink = textlink
	info.countSetLink = 0
	info.refreshStatus = 0
	LWTimerRedrawList[frame] = nil

	local linktype = strmatch (link, "^(%l+):") -- debug
	--info.linktype = linktype -- debug
	if linktype == "item" then -- debug
		info.infoNeeded = true -- set flag to call GetItemInfo later
	end
	local errOK, errTXT = pcall(frame.SetHyperlink, frame, link)
	if not errOK then -- trap any errors
		LWCloseTooltipWindow (frame)
		local msg = "SetHyperlink error"
		if info.externalLink then
			msg = msg .. " (external link)"
		end
		msg = msg .. ": " .. errTXT
		return LWReportError (msg, 2)
	elseif frame:NumLines()==0 then --test to see if SetHyperlink failed during call - may succeed later
		LWStartTimeoutTimer(frame, info)
	end

	return errOK
end

local function LWHookSetItemRef(link, text, button, ...) -- ... is for future proofing
	if not LWMasterEnable then
		return LWOriginalSetItemRef(link,text,button,...)
	end
	--LWPrintDebug ("Link = "..link)

	local linktype = strmatch (link, "^(%l+):")
	if linktype == "item" or LWSConfig.alllinks and (linktype == "enchant" or linktype == "spell" or linktype == "quest") then
		local LWQuickReturnValues = {}
		local clickOption = LWSClick[button]
		if clickOption then
			if IsShiftKeyDown() then
				LWQuickActionJump[clickOption.shift](LWQuickReturnValues,link,text,button)
			elseif IsControlKeyDown() then
				LWQuickActionJump[clickOption.ctrl](LWQuickReturnValues,link,text,button)
			elseif IsAltKeyDown() then
				LWQuickActionJump[clickOption.alt](LWQuickReturnValues,link,text,button)
			else
				LWQuickActionJump[clickOption.action](LWQuickReturnValues,link,text,button)
			end
		end

		if LWQuickReturnValues.bypass then
			return LWOriginalSetItemRef(link,text,button,...)
		end
		if LWQuickReturnValues.dropout then
			return
		end

		local frame, info = LWGetCheckAvailableTooltip (link)
		if not frame then
			return
		end

		-- insert special flags from Quick Action Functions
		for key, value in pairs (LWQuickReturnValues) do
			info[key] = value
		end
		info.originalLink = text -- preserve link exactly as it appeared in chat

		LWTooltipSetHyperlink (frame, link, info, text, -1)

	else -- Call original if not handled above
		return LWOriginalSetItemRef(link,text,button,...)
	end
end

local function LWExternalTooltip (extlink, options)
	if not LWMasterEnable then
		return
	end
	local errorstring, link, textlink, frame, info

	if type (extlink) ~= "string" then
		errorstring = "OpenTooltip: link parameter must be a string"
	elseif options ~= nil and type (options) ~= "table" then
		errorstring = "OpenTooltip: options parameter must be a table or nil"
	end
	if errorstring then
		return LWReportError (errorstring, 2)
	end

	-- Item link?
	link = strmatch (extlink, "(item:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+)")
	if not link and LWSConfig.alllinks then
		-- Enchant, Spell or Quest link?
		link = strmatch (extlink, "(enchant:%-?%d+)") or strmatch (extlink, "(spell:%-?%d+)") or strmatch (extlink, "(quest:%-?%d+:%-?%d+)")
		textlink = strmatch (extlink, "(\124c%x+\124H.+\124h%[.+%]\124h\124r)")
	end
	if not link then
		return
	end

	frame, info = LWGetCheckAvailableTooltip (link)
	if not frame then
		return
	end
	if type(options)=="table" then
		if type(options.whisper)=="string" then
			info.whisper = options.whisper
		end
		if options.openmin then
			info.openMinimize = true
		end
		if type(options.textlink)=="string" then
			textlink = strmatch (options.textlink, "(\124c%x+\124H.+\124h%[.+%]\124h\124r)")
		end
	end
	info.externalLink = true
	LWTooltipSetHyperlink (frame, link, info, textlink, -2)

	return frame
end

local function LWCaptureTooltip ()
	-- Opens a LinkWrangler tooltip using the link from the current GameTooltip
	-- i.e. captures link from whatever item is under the mouse
	if GameTooltip:IsVisible() and LWMasterEnable then
		-- find the link and convert to correct format
		local _, link = GameTooltip:GetItem()
		link = link and strmatch (link, "(item:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+)")
		if link then
			-- Get tooltip frame and Set it to link
			local frame, info = LWGetCheckAvailableTooltip (link)
			if frame and info then
				LWTooltipSetHyperlink (frame, link, info, nil, -3)
				GameTooltip:Hide()
			end
		end
	end
end

local function LWCaptureCompare (cframe)
	local cinfo = LWCompareData[cframe]
	local link, textlink
	if cinfo then
		textlink = cinfo.link
	end
	if textlink then
		link = strmatch (textlink, "(item:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+:%-?%d+)")
	end
	if link then
		-- Get tooltip frame and Set it to link
		local frame, info = LWGetCheckAvailableTooltip (link)
		if frame and info then
			LWTooltipSetHyperlink (frame, link, info, textlink, -4)
			LWCloseOneCompareFrame (cframe)
		end
	end
end

-------------------------------------------------------------------------------------------
-- BUTTON HANDLERS
-------------------------------------------------------------------------------------------

-- Whisper button
local function LWWhisperButton(button)
	local member = LWTooltipData[button:GetParent()].whisper

	if member then
		if not ChatFrameEditBox:IsVisible() then
			ChatFrame_OpenChat("/w "..member.." ")
		else
			ChatFrameEditBox:SetText("/w "..member.." ")
		end
	end
end

-- Relink button
local function LWRelinkButton(button, isComp)
	local frame = button:GetParent()
	local textlink
	if isComp then -- it's a compare window
		textlink = LWCompareData[frame].link
	else
		textlink = LWTooltipData[frame].textlink
	end

	if textlink then
		if not ChatFrameEditBox:IsVisible() then
			ChatFrame_OpenChat(textlink)
		else
			ChatFrameEditBox:Insert (textlink)
		end
	end
end

-- Dress Up button
local function LWDressupButton(button)
	DressUpItemLink (LWTooltipData[button:GetParent()].link)
end

-- Minimize button
local function LWMinimizeButton (button)
	LWMinimizeTooltip (button:GetParent())
end

-- Compare button
local function LWCompareButton(button)
	local frame = button:GetParent()
	local info = LWTooltipData[frame]

	if info.slots == nil or info.numslots == 0 then -- should not happen
		return
	end

	if info.compare == 0 then
		for index, slot in pairs (info.slots) do
			local link = GetInventoryItemLink ("player", slot)
			if link then
				LWOpenCompareWindow (frame, info, link)
			end
		end
	else
		info.compare = 0
		for _,cframe in pairs (info.compareFrames) do
			LWHideCompareFrame (cframe)
		end
	end
end

local function LWCloseButton (button, isComp)
	if IsShiftKeyDown() then
		LWCloseAllAction()
	elseif isComp then
		LWCloseOneCompareFrame (button:GetParent())
	else
		LWCloseTooltipWindow (button:GetParent())
	end
	GameTooltip:Hide()
end

local function LWCaptureButton (button)
	LWCaptureCompare (button:GetParent())
end

local function LWButtonGameTooltip (button, buttontype)
	GameTooltip:SetOwner (button, "ANCHOR_RIGHT")
	GameTooltip:SetText (LWL.Tooltip[buttontype])
	GameTooltip:Show()
end

-------------------------------------------------------------------------------------------
-- EVENTFRAME EVENT HANDLERS
-------------------------------------------------------------------------------------------

-- Warning: global name will be deleted during initialization
function LinkWranglerEventFrame_OnEvent (frame, eventname, data1, data2)
	if eventname == "UNIT_INVENTORY_CHANGED" then
		if LWInventoryHooked and data1 == "player" then
			LWCheckInventory ()
		end
	elseif eventname == "ADDON_LOADED" then
		if data1 == "LinkWrangler" then
			LinkWranglerRunOnce_CheckSavedVariables ()
			-- fix for addons that loaded before LWSAddOns got set {fix1731}
			for addon,_ in pairs (LWEarlyLoadAddOns) do
				LWSetAddOnSaved (addon)
			end
			LWEarlyLoadAddOns = nil
			if EnhTooltip and EnhancedTooltip then -- EnhTooltip normally loads before LinkWrangler
				LWEnhTooltipLoaded = true
			end
		elseif data1 == "EnhTooltip" then -- can occur if loaded on demand
			if EnhTooltip and EnhancedTooltip then -- check they both exist, i.e. no errors on loading
				LWEnhTooltipLoaded = true
			end
		end
	elseif eventname == "PLAYER_LOGOUT" then
		LWLogoutSaveLayout ()
		LWLogoutSaveAddOns ()
	-- at this point, event must be a chat message of some sort
	elseif LWMasterEnable and LWSButtons.whisper and (strfind(data1, "\124h")) then -- only save lines which might have links in
		LWChatBufferTextList[LWChatBufferIndex] = data1
		LWChatBufferNameList[LWChatBufferIndex] = data2

		-- keep count of how many strings are stored
		if LWChatBufferIndex > LWChatBufferStored then
			LWChatBufferStored = LWChatBufferIndex
		end

		LWChatBufferIndex = LWChatBufferIndex + 1
		-- reuse oldest indexes once we reach maximum size
		if (LWChatBufferIndex > 32) then
			LWChatBufferIndex = 1
		end
	end
end

-- Warning: global name will be deleted during initialization
function LinkWranglerEventFrame_OnUpdate (self, elapsed)

	-- Perform refresh callbacks
	for frame, info in pairs (LWTimerRefreshList) do
		--LWTimerRedrawList[frame] = nil -- don't think this is wanted here
		LWTimerRefreshList[frame] = nil
		info.refreshStatus = info.refreshStatus + 1 -- should now be an odd number
		if info.refreshStatus < 8 then -- debug: stops runaways
		--LWPrintDebug (format ("Refresh count for %s is %d", frame:GetName(), info.refreshStatus))
		LWCallbackRefresh (frame, info.link)
		--frame:SetAlpha (LWSConfig.alpha)
		end
	end

	-- Check for frames that need resizing
	for frame,info in pairs (LWTimerResizeList) do
		if info.minimized then
			frame:SetHeight(36)
			local titleWidth = info.titleFrame:GetStringWidth() + 36
			if LWSButtons.close then
				titleWidth = titleWidth + 22
			end
			frame:SetWidth (titleWidth)
		else
			if frame:GetHeight() < info.buttonHeight then
				frame:SetHeight(info.buttonHeight)
			end
			if LWEnhTooltipLoaded then
				local _,anchor = EnhancedTooltip:GetPoint()
				if anchor == frame then
					local fWidth = frame:GetWidth()
					local aWidth = EnhancedTooltip:GetWidth()
					if (fWidth > aWidth) then
						EnhancedTooltip:SetWidth(fWidth)
					else
						frame:SetWidth(aWidth)
					end
				end
			end
		end
		LWTimerResizeList[frame] = nil
	end

	-- stop timer if other lists are empty
	if not next (LWTimerTimeoutList) and not next (LWTimerRedrawList) then
		LWStopTimer()
		return
	end


-- In some cases after calling SetHyperlink, the frame does not open immediately
-- This is caused by items not held in the local cache, and a delay downloading the required info from the server
-- In most cases the info will download within 2 seconds, and the window will then open
-- the following code is a failsafe in case that does not happen, so that the window is not left in a half-open state
	LWTimerTimeoutElapsed = LWTimerTimeoutElapsed + elapsed
	if LWTimerTimeoutElapsed > 2 then -- 2 second timeout
		-- close all windows in timeout list
		for frame, _ in pairs (LWTimerTimeoutList) do
			LWCloseTooltipWindow (frame, true) -- also removes from list
		end
	end

-- In some cases after a window opens, not all information has been included in the tooltip
-- This happens where the tooltip includes info about items other that the one being viewed,
-- for example recipes and slotted items, and where the 'extra' info is not held in the local cache.
-- The following code causes a refresh (after a small delay in case more info needs to be downloaded)
-- This is important for some tooltip scanning mods
	LWTimerRedrawElapsed = LWTimerRedrawElapsed + elapsed
	if LWTimerRedrawElapsed > .15 then -- 0.15 second delay
		-- force refresh for frames where delayed info has been received
		for frame, info in pairs (LWTimerRedrawList) do
			frame:ClearLines()
			info.countSetLink = 0
			info.refreshStatus = info.refreshStatus + 1 -- should now be an even number
			LWTimerRedrawList[frame] = nil
			frame:SetHyperlink(info.link)
			if info.EnhTooltipEmbedded then
				-- If EnhTooltip data is embedded, need to call here, otherwise not needed
				LWEnhTooltipAction (frame, info)
				-- note: will cause EnhancedTooltip to jump back to this frame
			end
			LWStartResizeTimer (frame, info) -- todo: is this needed?
			LWStartRefreshTimer (frame, info)
		end
	end
end

-- Warning: self-deleting function
function LinkWranglerEventFrame_OnLoad (frame)
	-- setup slash command handler
	SLASH_LinkWrangler1="/linkwrangler"
	SLASH_LinkWrangler2="/lw"
	if LWL.AltSlashCommand then
		SLASH_LinkWrangler3=LWL.AltSlashCommand
	end
	SlashCmdList["LinkWrangler"]=LWdoSlashCommand

	-- setup hooks
	LWOriginalSetItemRef = SetItemRef
	SetItemRef = LWHookSetItemRef

	-- setup scripts
	frame:SetScript ("OnEvent", LinkWranglerEventFrame_OnEvent)
	frame:SetScript ("OnUpdate", LinkWranglerEventFrame_OnUpdate)

	-- setup timer
	LWTimerFrame = frame -- frame is created as 'hidden' so timer is stopped
	--LWStopTimer() -- reset timer

	-- register events
	frame:RegisterEvent("ADDON_LOADED")
	frame:RegisterEvent("UNIT_INVENTORY_CHANGED")
	frame:RegisterEvent("PLAYER_LOGOUT")
	-- chat events
	frame:RegisterEvent("CHAT_MSG_GUILD")
	frame:RegisterEvent("CHAT_MSG_PARTY")
	frame:RegisterEvent("CHAT_MSG_CHANNEL")
	frame:RegisterEvent("CHAT_MSG_WHISPER")
	frame:RegisterEvent("CHAT_MSG_RAID")
	frame:RegisterEvent("CHAT_MSG_OFFICER")
	frame:RegisterEvent("CHAT_MSG_SAY")
	frame:RegisterEvent("CHAT_MSG_LOOT")
	frame:RegisterEvent("CHAT_MSG_YELL")

	-- code not needed after load; free up some memory
	frame:SetScript ("OnLoad", nil)
	LinkWranglerEventFrame_OnLoad = nil
	LinkWranglerEventFrame_OnEvent = nil
	LinkWranglerEventFrame_OnUpdate = nil
end

-------------------------------------------------------------------------------------------
-- TOOLTIP EVENT HANDLERS
-------------------------------------------------------------------------------------------

local function LWTooltipOnSizeChanged(frame)
	LWStartResizeTimer (frame, LWTooltipData[frame])
end

-- keep code synchronised with LWEnhTooltipAction
local function LWTooltipOnEnter(frame, mouse)
	if not mouse then return; end -- mouse is false if mouse didn't move (i.e. window got moved over mouse cursor)

	frame:Raise() -- bring frame to front

	if not LWEnhTooltipLoaded then return; end

	local _,anchor = EnhancedTooltip:GetPoint()
	if anchor == frame then return; end -- EnhTooltip already attached to this frame - no further action

	local info = LWTooltipData [frame]

	if info.minimized then return; end -- don't attach EnhTooltip to minimized LW Tooltip

	local name = info.name
	local link = info.link
	local textlink = info.textlink

	if info.EnhTooltipEmbedded then
		frame:ClearLines() -- needed - otherwise SetHyperlink will close this frame
		info.countSetLink = 0
		info.refreshStatus = 0
		LWTimerRedrawList[frame] = nil
		frame:SetHyperlink(link)
	end

	EnhTooltip.ClearTooltip()
	EnhTooltip.TooltipCall(frame, name, textlink, -1, 1, 0, true, link)
	--EnhancedTooltip:SetFrameLevel (frame:GetFrameLevel ())
	--EnhancedTooltip:Raise()

	if info.EnhTooltipEmbedded then
		LWStartRefreshTimer (frame, info)
	end

	LWStartResizeTimer (frame, info) -- todo: is this needed?
	frame:Show() -- todo: is this needed?
end

local function LWTooltipOnHide (frame)
	GameTooltip_OnHide()
	if LWTooltipData[frame].state ~= 0 then
		LWCloseTooltipWindow (frame)
	end
end

local function LWTooltipSetLink (frame, spell)
	local info = LWTooltipData[frame]

	-- how many times has LWTooltipSetLink been called since last SetHyperlink
	local count = info.countSetLink + 1
	info.countSetLink = count

	local slots, buttons

--[[ 02.04.08 version 1.713
The GetItemInfo call has been moved to here, as it was not always succeeding in the OnShow handler
It appears that OnShow now gets called before the local cache is updated
However it also appears that OnTooltipSetItem gets called each time the local cache is updated
So this is now the best place to call GetItemInfo (it should succeed ... eventually)
But we want to avoid calling it multiple times
Currently I'm using a control flag, which will have been set to true for item links earlier
]]
	if info.infoNeeded then
		local _,textlink,_,_,_,itemtype,_,_,equip = GetItemInfo(info.link)
		if textlink or itemtype or equip then
			info.infoNeeded = nil
		end
		if equip then
			-- setup compare slots: try to match to Equip code passed from getItemInfo
			slots = LWInventorySlotTable[getglobal(equip)]
			if slots and not info.slots then
				info.slots = slots
				info.numslots = #slots
				LWCheckCompareButtonStatus (frame, info) -- set compare button according to inventory
				buttons = true
			end
			-- Set Dressup button
			if equip ~= "" and equip ~= "INVTYPE_BAG" and equip ~= "INVTYPE_AMMO" then
				info.canDressup = true
				buttons = true
			end
		end

		if textlink then
			info.textlink = textlink
			buttons = true
		end
	end

	if count >1 and info.refreshStatus == 1 or info.refreshStatus == 3 then
		-- We have detected a call to OnTooltipSetItem after a "refresh" event has been fired
		-- This can happen when some of the info for the tooltip is not held in the local cache at first -
		-- When the info is eventually downloaded, OnTooltipSetItem gets called again (possibly many times)
		-- We're going to wait a fraction of a second (to see if any more info arrives) then refresh the tooltip
		LWStartRedrawTimer (frame, info)
		--LWPrintDebug (format ("TooltipSetLink: delayed cache change. count %d refreshStatus %d", count, info.refreshStatus))
	end

	if count == 2 and not info.slots then -- possibly a recipe; check if the finished item is equippable
		--[[22.04.08 version 1.721
		Code that used to check if the link is a recipe has been removed,
		as it was more complex than really necessary
		Now simply check whenever LWTooltipSetLink gets called for the second time
		--]]

		-- if tooltip is fully open, scan text to search for equippable types
		if info.state == 1  then --and info.isRecipe then
			local frametext = frame:GetName().."TextLeft"
			local textfield --, fieldcontents
			for i=3,12 do -- the line we want is most likely line 7 or 8, but search either side
				textfield = getglobal(frametext..i)
				slots = LWInventorySlotTable[(textfield and textfield:GetText())]
				if slots then
					break -- found match
				end
			end
		end
		if slots then
			info.slots = slots
			info.numslots = #slots
			LWCheckCompareButtonStatus (frame, info) -- set compare button according to inventory
			buttons = true
		end
	end

	if buttons then
		-- Buttons settings have been changed - re-do layout
		LWTooltipButtonLayout (frame, info)
		LWStartResizeTimer (frame, info)
		frame:Show() -- todo: test if required?
	end
end

--[[
This function does most of the work of initializing the tooltip window
once it has been activated by a successful SetHyperlink call
--]]
local function LWTooltipOnShow(frame)
	local info = LWTooltipData [frame]
	local name = info.titleFrame:GetText() -- get (localised) item name as it appears in tooltip

--[[
Uncached items: some functions do not always work at this point, if the item is uncached. Found this problem with:
GetItemInfo, frame:GetItem
--]]

	-- Set Whisper Player
	if info.originalLink then
		-- originalLink holds the full text link as it appeared in chat
		-- may be different from textlink, e.g. due to localisation
		-- only gets set when clicking a link in chat (i.e. via SetItemRef)
		local foundIndex = 0
		local _,_,itemText = strfind (info.originalLink, "(%[.+%])")
		if itemText then
			local counter = LWChatBufferIndex
			for i=1,LWChatBufferStored do
				-- find most recent whisperer for the named item
				counter = counter - 1
				if counter == 0 then
					counter = LWChatBufferStored
				end
				local startPos = strfind(LWChatBufferTextList[counter],itemText,1,true)
				if startPos then
					foundIndex = counter
					break
				end
			end
		end
		if foundIndex > 0 and LWChatBufferNameList[foundIndex] and strlen(LWChatBufferNameList[foundIndex]) > 0 then
			info.whisper = LWChatBufferNameList[foundIndex]
		end
		info.originalLink = nil
	end

	-- store name/chatlink (localized version if available)
	info.name = name

	-- other Addon stuff
	info.state = 1
	LWTimerTimeoutList[frame] = nil
	info.minimized = false -- todo: should not be needed as should already be false
	LWCallbackAction (frame, info.link, "show") -- notify addons frame is being opened

	if info.openMinimize then
		info.openMinimize = nil
		LWMinimizeTooltip (frame)
	else
		LWEnhTooltipAction (frame, info)
		LWStartRefreshTimer (frame, info)

		-- More drawing stuff
		LWStartResizeTimer (frame, info) -- todo: is this required here?
		LWCheckCompareButtonStatus (frame, info) -- set compare button according to inventory
		LWTooltipButtonLayout (frame, info)
		frame:Raise()
		frame:Show() -- todo: is this needed?
	end
end

-------------------------------------------------------------------------------------------
-- GLOBAL VARIABLES FOR EXPORT
-------------------------------------------------------------------------------------------

-- Official LinkWrangler export table
LinkWrangler = {
-- Documented Exports and functions
	Version = LWVersion,				-- version number (float)
	CloseAllWindows = LWCloseAllAction,
	MinimizeAllWindows = LWMinimizeAllAction,
	MasterSwitch = LWSlashEnableDisable,		-- master enable disable function
	SlashHandler = LWdoSlashCommand,		-- expose the slash command handler for other addons (caution!)
	RegisterCallback = LWRegisterCallback,		-- called by other mods to register their callback function
	CallbackData = LWModifyCallback,		-- check/change settings of callbacks
	GetLinkInfo = LWGetLinkInfo,			-- get info about frame or link
	OpenTooltip = LWExternalTooltip,		-- Open LW tooltip by supplying a link
	CaptureTooltip = LWCaptureTooltip,		-- Open LW tooltip from Gametooltip's info

-- Internal functions
	TooltipOnSizeChanged = LWTooltipOnSizeChanged,	-- Tooltip handlers
	TooltipOnEnter = LWTooltipOnEnter,
	TooltipOnShow = LWTooltipOnShow,
	TooltipOnHide = LWTooltipOnHide,
	TooltipSetItem = LWTooltipSetLink,
	WhisperButton = LWWhisperButton,		-- Button handlers
	RelinkButton = LWRelinkButton,
	DressupButton = LWDressupButton,
	MinButton = LWMinimizeButton,
	CompButton = LWCompareButton,
	CloseButton = LWCloseButton,
	CaptureButton = LWCaptureButton,
	ButtonTooltip = LWButtonGameTooltip,
}

--[[
This is the original Addon hook created by Fallan
This will continue to be supported for the forseeable future, for compatability
The metatable redirects table inserts so that they get included in the real callback table
--]]
LINK_WRANGLER_CALLER = setmetatable({},{__newindex=function (t,k,v) LWRegisterCallback(k, v, "redirect"); end})

--[[
The following globals are included to prevent certain other AddOns from breaking.
They will be removed once they have been written out of those AddOns
--]]
IR_MOD_AUCTIONEER_INDEX = 0
