﻿
-- module setup
local me = { name = "bossmod"}
local mod = thismod
mod[me.name] = me

--[[
BossMod.lua

See also: BossModSchema.lua: contains the schema for a boss mod definition.
See also: BossModImplentation.lua: contains the default built-in boss mods.

This module is responsible for loading the boss mod data and turning it into usable data structures. This means:
	
1) For speech entries, remove the item if the appropriate localisation can't be found. While it would be possible to identify a boss's localised name given its mobid, you would still need a localised search string for the actual message he yells / emotes / etc, so you may as well add the boss's name too.

2) For MobIDs, convert to a hex search string that will match the start of a GUID string.

3) Warn if particular spells are being handled by more than one boss mod.

4) Make lookups of spellids linking to their boss mobs.

---------------------------------------------------------------
	Variables defined:
---------------------------------------------------------------

1) me.data is an aggregate of the successfully loaded <me.mybossmods> tables defined in other modules.

	me.data = 
	{
		(module name) = (copy of the <me.mybossmods> table for that module)
	}

2) me.speech is a two level lookup, the key is (name of the mob that speaks) and (search string for the message). The value identifies the bossmod and particular speech entry.

	me.speech = 
	{
		(localised source name) = 
		{
			(search string) = <locator>
			...
		}
	}

	<locator> = (this is used in the other tables too)
	{
		module = (name of the module that defined the bossmod)
		bossmod = (key of the bossmod in mod[module].mybossmods table)
		name = (key of the entry in <bossmod>.speech table)
		type = "speech" or "debuffs" or "attacks" or "casts"
	}
	
3) Debuffs, Attacks, Casts 

	me.spells = 
	{
		spellid = <locator>       -- "type" property used to differentiate debuffs, attacks, etc.
	}
]]

me.data = { }

--[[
-- this is full trace enabled
me.mytraces = 
{
	default = "info",
}
]]

--[[
-------------------------------------------------------------------------------
		Loader Service
-------------------------------------------------------------------------------
]]

--[[
First loading phase. This is called by the loader service once for each module. We check for boss mod definitions and validate them if they exist.
]]
me.onmoduleload = function(module)
		
	-- ignore if the module does not have a schema
	if module.mybossmods == nil then
		return
	end
	
	-- try validate
	local errormessage = mod.typevalidation.validate(mod.bmschema, "lookup_bossmodset", module.mybossmods)
	
	if errormessage then
		-- warn and exit
		if mod.trace.check("warning", me, "badschema") then
			mod.trace.printf("The boss mods in the module '%s' is invalid. The error returned was:\n '%s'", module.name, errormessage)
		end
	
		return
	end
	
	-- success: add
	me.data[module.name] = module.mybossmods
	
	-- debug
	if mod.trace.check("info", me, "schemaload") then
		mod.trace.printf("Successfully loaded the boss mods for the '%s' module.", module.name)
	end
	
end

--[[
Second phase of the loading process. Convert configuration data into working data structures.
]]
me.onloadcomplete = function()
	
	for modulename, moduledata in pairs(me.data) do
		
		for bossmodname, bossdata in pairs(moduledata) do
			
			--[[ convert mobid to a 6-digit guid substring.
			%X		format as hexadecimal integer
			06		left-pad with zeros so that total length = 6.		
			]]
			bossdata.mobidstring = string.format("%06X", bossdata.mobid)
			
			-- load speech
			if bossdata.speech then
				mod.bmspeech.loadspeechdata(modulename, bossmodname, bossdata.speech)
			end

			-- load debuffs
			if bossdata.debuffs then
				mod.bmdebuffs.loaddebuffdata(modulename, bossmodname, bossdata.debuffs)
			end
			
			-- load casts
			if bossdata.casts then
				mod.bmcasts.loaddata(modulename, bossmodname, bossdata.casts)
			end
			
			-- load attacks
			if bossdata.attacks then
				mod.bmattacks.loaddata(modulename, bossmodname, bossdata.attacks)
			end
		end
	end
end

--[[
-------------------------------------------------------------------------------
					Public Methods
-------------------------------------------------------------------------------
]]

--[[
mod.bossmod.resetmobidunknowninstance(modulename, bossmodname)

Call this method to reset a bossmod where the exact unit (guid) is unknown. This will come from e.g. speech events, or resets generated by an external source.

<modulename> and <bossmodname> identify the bossmod. From that we can determine the resetting mob's unit type id, i.e. guid AND 0xFFF000. 

Now, we expect (hope) there will be exactly one known (i.e. on our threat list) mob matching the unit type id. If we find 0 or multiple hits, we will warn but keep going.
]]
me.resetmobidunknowninstance = function(modulename, bossmodname)
	
	-- debug print
	if mod.trace.check("info", me, "reset") then
		mod.trace.printf("Resetting the boss mod '%s.%s'", modulename, bossmodname)
	end
	
	-- get mobid
	local mobidstring = me.data[modulename][bossmodname].mobidstring
	
	-- debug protection
	if mod.mythreat == nil then return end

	-- TODO: consider moving some of this code into MyThreat module to avoid excessive linkage. Perhaps a method ResetByMobTypeID which returns the number of matches.

	-- look for matches to mobidstring
	local hits = 0
	
	for mobguid, mobdata in pairs(mod.mythreat.datalookup) do
		
		if string.sub(mobguid, 7, 12) == mobidstring then
			mod.mythreat.resettarget(mobguid)
			hits = hits + 1
		end
	end
		
	if hits ~= 1 then
		
		-- info and ok
		if mod.trace.check("info", me, "nomobidmatch") then
			mod.trace.printf("Found '%s' matches for the mob reset of the '%s' mob type caused by the bossmod '%s.%s'", hits, mobidstring, modulename, bossmodname)
		end
	end
	
end