﻿
-- module setup
local me = { name = "main"}
local mod = thismod
mod[me.name] = me

-- key = <frame>, value = addon name or ""
me.knownframes = { }

-- key = <addon name> value = {onupdate = , total =, onevent = { memory = <number>, time = <number>}
me.data = { }

me.originalhandlers = 
{
	onupdate = { },
	onevent = { },
}

me.myonupdates = 
{
	checknewframes = 1.0,
}

me.checknewframes = function()
	
	if me.loaded then
		me.load()
	end
		
end

me.myevents = { "ADDON_LOADED" }

me.onevent = function()
	
	if me.loaded then
		me.load()
	end
	
end

-- this catches every event
me.onload = function()
	
	me.frame = CreateFrame("Frame", nil, mod.loader.frame)
	me.frame:Show()
	me.frame:RegisterAllEvents()
	me.frame:SetScript("OnEvent", me.frameonevent)
	
end 

me.updatetick = 0

me.frameonupdate = function()
	
	me.updatetick = me.updatetick + 1
	
end

me.frameonevent = function()
end


--[[
------------------------------------------------------------------------------------
							Console Services
------------------------------------------------------------------------------------
]]

me.myconsole = 
{
	start = "start",
}

--[[
insertion point.

/kpm s
]]
me.start = function()
	
	if me.isloaded == true then
		mod.print("The Performance Monitor is already loaded!")
		return
	end

	me.load()
	
end


--[[
------------------------------------------------------------------------------------
							Event Services
------------------------------------------------------------------------------------
]]

me.load = function()
	
	local nextframe
	
	while true do

		nextframe = EnumerateFrames(nextframe)
	
		if nextframe == nil then
			
			-- finished
			if me.isloaded == nil then
				mod.print("Done!")
				me.isloaded = true
			end
			
			return
		end
	
		-- do next frame
		me.processframe(nextframe)
	end

end

me.processframe = function(frame)
		
	-- do we already have it?
	if me.knownframes[frame] then
		return
	end
		
	-- get source and set
	local issecure, addon = me.getframesourcename(frame)
		
	me.knownframes[frame] = addon or ""
	
	-- only hook frames we can
	if addon == nil then
		return
		
	-- create <data> entry for this addon if it's new 
	elseif me.data[addon] == nil then
		me.data[addon] = 
		{
			onupdate = 
			{
				memory = 0,
				time = 0,
			},
			onevent = 
			{
				memory = 0,
				time = 0,
			},
			total = 
			{
				memory = 0,
				time = 0,
			},
		}
	end
	
	-- special hooking for secure frames
	if issecure then
		
		if frame:GetScript("OnUpdate") then
			frame:HookScript("OnUpdate", me.secureonupdatehook)
		end
		
		if frame:GetScript("OnEvent") then
			frame:HookScript("OnEvent", me.secureoneventhook)
		end
	
	else	-- normal frame hooking
	
		if frame:GetScript("OnUpdate") then
			me.originalhandlers.onupdate[frame] = frame:GetScript("OnUpdate")	
			frame:SetScript("OnUpdate", me.onupdatehook)
		end
		
		if frame:GetScript("OnEvent") then
			me.originalhandlers.onevent[frame] = frame:GetScript("OnEvent")	
			frame:SetScript("OnEvent", me.oneventhook)
		end
	end
		
end

me.securedata = 
{
	lastevent = "", 	-- value of <event>, or "onupdate"
	timestart = 0, 	-- debugprofilestop()
	memorystart = 0, 	-- gcinfo()
	updatetick = 0,	-- me.updatetick. kind of
}

me.secureoneventhook = function(self, event)
	
	local timenow = debugprofilestop()
	local memorynow = gcinfo()
	local method = "onevent"
	
	if me.securedata.lastevent == event then
		
		-- we have valid data for this frame. yay.
		local addon = me.knownframes[self]
		local data = me.data[addon]
		
		local takentime = timenow - me.securedata.timestart
		local takenmemory = math.max(0, memorynow - me.securedata.memorystart)
		
		local addon = me.knownframes[self]
		local data = me.data[addon]
		
		data[method].memory = data[method].memory + takenmemory
		data.total.memory = data.total.memory + takenmemory
		
		data[method].time = data[method].time + takentime
		data.total.time = data.total.time + takentime
	end
	
	-- prepare for next secure handler
	me.securedata.lastevent = event
	me.securedata.timestart = debugprofilestop()
	me.securedata.memorystart = gcinfo()	
	
end

me.secureonupdatehook = function(self)
	
	local timenow = debugprofilestop()
	local memorynow = gcinfo()
	local method = "onupdate"
	
	if me.securedata.lastevent == "onupdate" then
		
		-- we have valid data for this frame. yay.
		local addon = me.knownframes[self]
		local data = me.data[addon]
		
		local takentime = timenow - me.securedata.timestart
		local takenmemory = math.max(0, memorynow - me.securedata.memorystart)
		
		local addon = me.knownframes[self]
		local data = me.data[addon]
		
		data[method].memory = data[method].memory + takenmemory
		data.total.memory = data.total.memory + takenmemory
		
		data[method].time = data[method].time + takentime
		data.total.time = data.total.time + takentime
	end
	
	me.securedata.lastevent = "onupdate"
	me.securedata.timestart = debugprofilestop()
	me.securedata.memorystart = gcinfo()	
	
end

me.oneventhook = function(self, ...)
	
	local originalhandler = me.originalhandlers.onevent[self]
	me.runhook(originalhandler, "onevent", self, ...)
	
	me.securedata.lastevent = event
	me.securedata.timestart = debugprofilestop()
	me.securedata.memorystart = gcinfo()
	
end

me.onupdatehook = function(self, ...)
	
	local originalhandler = me.originalhandlers.onupdate[self]
	me.runhook(originalhandler, "onupdate", self, ...)
	
	me.securedata.lastevent = "onupdate"
	me.securedata.timestart = debugprofilestop()
	me.securedata.memorystart = gcinfo()
	
end

me.runhook = function(handler, method, self, ...)
	
	-- deactivate secure stream
	me.securedata.issecurelast = false
	
	local starttime = debugprofilestop()
	local startmemory = gcinfo()
	
	handler(self, ...)	
	
	local takentime = debugprofilestop() - starttime
	local takenmemory = math.max(0, gcinfo() - startmemory)
	
	local addon = me.knownframes[self]
	local data = me.data[addon]
	
	data[method].memory = data[method].memory + takenmemory
	data.total.memory = data.total.memory + takenmemory
	
	data[method].time = data[method].time + takentime
	data.total.time = data.total.time + takentime
	
end

--[[
me.getframesourcename(frame)

Returns: nil if the frame is unhookable or has nothing to hook; otherwise string = addon name.

returns: 

nil - if the frame has no handlers
<issecure>, <name> - if the frame has handlers

	<issecure> boolean
	<name> string
]]
me.getframesourcename = function(frame)

	local handler = frame:GetScript("OnUpdate") or frame:GetScript("OnEvent")
	
	if handler == nil then
		return nil -- none
	end

	local issecure = false

	-- check for blizz frames
	if frame:GetName() and issecurevariable(getfenv(0), frame:GetName()) then
		issecure = true
	end

	local success, dump = pcall(string.dump, handler)
	if success == nil then
		return nil -- we might change this later, but it seems a bit risky atm.
	end

	local blizzaddon = string.match(dump, "@Interface\\FrameXML\\([%w_]+).lua")
	
	-- forget blizzard frames
	if blizzaddon then
		return true, blizzaddon
	end

	local directory, addon = string.match(dump, "@Interface\\([%w_]+)\\([%w_]+)\\")
	
	if directory == nil then
		-- this seems to happen when the handler is defined in XML		
		return issecure, me.getframenameheader(frame, frame:GetName(), dump)
			
	else
		return false, addon
	end

end

--[[
Finds the name of a named frame. ...
]]
me.getframenameheader = function(frame, name, dump, issecure)

	if name == nil then
		
		_, name = string.match(dump, "@(.-)([%w_]+)")
			
	end
		
	local length = string.len(name)

	-- really short stuff, to stop annoying cases
	if length < 6 then
		return name
	end

	-- check 4th char
	local current, previous, next_
	local position = 3
	
	previous = string.sub(name, position - 1, position - 1)
	current = string.sub(name, position, position)
	next_ = string.sub(name, position + 1, position + 1)

	local x = 0
	while true do 
	
		x = x + 1
		if x > 20 then
			mod.out.print("emergency stop!")
			return name
		end
	
		-- iterate
		position = position + 1
		previous = current
		current = next_
		next_ = string.sub(name, position + 1, position + 1)
		
		-- check for end of string
		if next_ == "" then
			return name
		end
		
		-- check for underscore
		if current == "_" then
			return string.sub(name, 1, position)
		end
		
		-- check for UUl
		if string.find(previous, "[A-Z]") and string.find(current, "[A-Z]") then
			-- if the string is 5 or longer, that's enough
			if position >= 5 then
				return string.sub(name, 1, position)
			end
			
		end
		
		-- check for lU
		if string.find(current, "[a-z]") and string.find(next_, "[A-Z]") then
			return string.sub(name, 1, position)
		end

	end
	
end