﻿
-- module setup
local me = { name = "event"}
local mod = thismod
mod[me.name] = me

--[[
Events.lua

This module provides an <onupdate> and <onevent> service to the rest of the addon.

1) OnUpdate service.

Implement a method called "onupdate" in your module and it will get called on every OnUpdate trigger. 

Alternatively, if you have a number of different methods that you want called periodically with different, specific periods, implement a table "myonupdates" as in this example

		me.myonupdates = 
		{
			updatescores = 0.0,
			updatenames = 0.5,
		}
		
		me.updatescores = function()
			(...)
		end
		
		me.updatenames = function()
			(...)
		end
		
The keys of the table must match functions in your module. The values are the minimum number of seconds allowed between subsequent method calls. 

2) OnEvent Service.

First implement a list of strings "myevents" with the names of the events you want, e.g.

		me.myevents = { "CHAT_MSG_SPELL_SELF_BUFF", "CHAT_MSG_SYSTEM", }
	
Then implement a method "onevent", e.g.

		me.onevent = function()
			(...)
		end
		
No arguments will be passed to this method, so you'll have to refer to the global variables <event>, <arg1>, <arg2>, etc. Make sure not to edit them, since other addons might need them!
]]

--[[
------------------------------------------------------------------------------------------
			Data Structures to remember OnUpdate and OnEvent subscriptions
------------------------------------------------------------------------------------------
]]

--[[ 
<me.events> records which module wants which events. e.g.

me.events = 
{
	["CHAT_MSG_SPELL_SELF_BUFF"] = {"combat", "regex"} -- combat and regex module register the event
	["CHAT_MSG_SYSTEM"] = {"netin"},
}
]]
me.events = { }

--[[
<me.onupdates> records which modules want onupdates. This is only for modules that have a <myonupdates> table - modules with <onupdate> we will just rescan for when an updates comes through. 

Suppose there is a method <mod.combat.updatedamage()> that should be called at most every 0.4 seconds, and a method <mod.combat.updatehealth()> that should be called every OnUpdate. Then e.g.

me.onupdates = 
{
	combat = 
	{
		updatedamage = 
		{
			lastupdate = 1234.203, 	-- value of GetTime()
			interval = 0.4,
		},
		updatehealth = 
		{
			lastupdate = 1235.120,	
			interval = 0.0,			-- 0.0 sec is minimum time between updates
		}
	}
}
]]
me.onupdates = { }

--[[
------------------------------------------------------------------------------------------
				Startup: Loading the Events Module
------------------------------------------------------------------------------------------

Loader.lua provides runs our startup functions. First <me.onload> is called, which creates our frame. Then <me.onmoduleload> is called with other modules as arguments, which will register all the necessary events on our frame. Then <me.onloadcomplete> is called, which creates the OnEvent script handler on our frame, which will start the event stream.
]]


--[[
me.onload() - special function called by Loader.lua

Creates the frame we will generate Events and OnUpdates with.
]]
me.onload = function()
	
	-- Create a frame to receive events and onupdates
	me.frame = CreateFrame("Frame", nil, mod.loader.frame)
	me.frame:Show()

end

--[[
me.onmoduleload() is a special function called by Core.lua when the addon starts up. It will be called once for each other module in the addon. We check whether they wish to receive events or onupdates, and implement them.
]]
me.onmoduleload = function(module)

	-- 1) Check for OnEvent subscription

	-- check for .onevent / .myevents mismatch
	if module.myevents and not module.onevent then
		if mod.trace.check("warning", me, "events") then
			mod.trace.print(string.format("The module |cffffff00%s|r has a |cffffff00.myevents|r list but no |cffffff00.onevent|r function.", module.name))
		end
		
	elseif module.onevent and not module.myevents then
		if mod.trace.check("warning", me, "events") then
			mod.trace.print(string.format("The module |cffffff00%s|r has a |cffffff00.onevent|r function but no |cffffff00.myevents|r list.", module.name))
		end
	
	-- normal modules: if they have both
	elseif module.myevents then
		
		local event, index, isnew, index2, knownmodule
		
		
		for index, event in pairs(module.myevents) do
			
			-- first level table is by the event
			if me.events[event] == nil then
				me.events[event] = { }
				me.frame:RegisterEvent(event)
			end
			
			-- prevent multiple occurrences of the same module-event definition
			isnew = true
			
			for index2, knownmodule in pairs(me.events[event]) do
				if knownmodule == module.name then
					isnew = false
					break
				end
			end
			
			if isnew then
				table.insert(me.events[event], module.name)
			end
		end
	end

	-- 2) Check for timed OnUpdates subscription
	if module.myonupdates then
		
		me.onupdates[module.name] = { }
		
		for method, interval in pairs(module.myonupdates) do
			
			if type(module[method]) ~= "function" then
				
				-- the module has not defined the method it promised
				if mod.trace.check("error", me, "onupdates") then
					mod.trace.print(string.format("The function |cffffff00%s|r in the onupdate list of the module |cffffff00%s|r is not defined.", method, module.name))
				end
				
			else
				me.onupdates[module.name][method] = 
				{
					lastupdate = 0,
					interval = interval,
				}
			end
		end
	end
	
end

--[[
me.onloadcomplete() is a special function called by Loader.lua after every module has loaded. This ensures that modules don't receive events until they have loaded.
]]
me.onloadcomplete = function()

	-- set handlers
	me.frame:SetScript("OnEvent", me.frameonevent)
	me.frame:SetScript("OnUpdate", me.frameonupdate)
		
	-- KLHBL compatability
	me.frame.klhblignore = true
	
end

--[[
mod.event.lateaddevent(me, event)

Registers an event for a module. Call this if you need to receive a new event after the mod has already loaded. This method won't perform any correctness checking.

<module>		table; reference to the module that wants the event
<event>		string; the event, e.g. "CHAT_MSG_SYSTEM"
]]
me.lateaddevent = function(module, event)
	
	if me.events[event] == nil then
		me.events[event] = { }
		me.frame:RegisterEvent(event)
	end
	
	-- HELP LATER
	-- don't dup
	for x = 1, #me.events[event] do
		if me.events[event][x] == module.name then
			return
		end
	end

	table.insert(me.events[event], module.name)
	
end

--[[
me.frameonevent() is the OnEvent handler of <me.frame>.

We pass on the event to all modules that have registered for it.
]]
me.frameonevent = function()
	
	-- don't send events unless the mod is enabled and loaded
	if mod.loader.isloaded == false or mod.isenabled == false then
		return
	end
	
	-- get all the modules that have registered this event
	local subtable, index, key
		
	for index, key in pairs(me.events[event]) do
		
		subtable = mod[key]

		mod.diag.logmethodcall(key, "onevent")
	end
	
end

--[[
me.frameonupdate() is the OnUpdate handler of <me.frame>.

We pass the onupdate to all modules that have registered for it.
]]
me.frameonupdate = function()
	
	-- don't send events unless the mod is enabled and loaded
	if mod.loader.isloaded == false or mod.isenabled == false then
		return
	end
	
	local key, subtable, module, moduledata, functiondata
	
	-- call all onupdates
	for key, subtable in pairs(mod) do
		if type(subtable) == "table" and type(subtable.onupdate) == "function" then
			mod.diag.logmethodcall(key, "onupdate")
		end
	end
	
	-- call all timed update methods
	local timenow = GetTime()
	
	for module, moduledata in pairs(me.onupdates) do
		 
		for functionname, functiondata in pairs(moduledata) do
			if timenow > functiondata.lastupdate + functiondata.interval then
				
				mod.diag.logmethodcall(module, "onupdate", mod[module][functionname])
				functiondata.lastupdate = timenow
			end
		end
	end
	
end