local _G = getfenv(0)
local __DEBUG__ = false
local KC_Items, tonumber = _G.KC_Items, _G.tonumber

local L = _G.AceLibrary("AceLocale-2.2"):new("SellValues")

L:RegisterTranslations("enUS", function()
	return {
		["Sell Values"] = true,
		["Toggle the Sell Values module"] = true,
        ["SellValues related options"] = true,
        ["Import Prices Database"] = true,
        ["Import the master prices database"] = true,
        ["Toggle"] = true,
        ["Import the master prices database. \n |cffcc0000Note:|r This will revert any data thats been updated. New items will not be changed."] = true,
        ["Yes"] = true,
        ["No"] = true,
	}
end)

L:RegisterTranslations("deDE", function()
	return {
		["Sell Values"] = "Verkaufs Preise",
		["Toggle the Sell Values module"] = "Aktiviert das Verkaufs Preise Module",
	}
end)

local SellValues = KC_Items:NewModule("SellValues", "AceEvent-2.0", "AceHook-2.1", "AceConsole-2.0", "AceDebug-2.0")

    function SellValues:OnInitialize()
        self.db = KC_Items:AcquireDBNamespace("SellValues")
        
        if __DEBUG__ then self:SetDebugging(true) end
        
        if type(self.db.account.data) == "string" then self.db.account.data = {} end
        
        KC_Items:RegisterDefaults("SellValues", 'account', {
            data = {
                ['*'] = ";",
            },
            maintenance = {
                ['importedMaster'] = false
            }
        })
        
        KC_Items:RegisterDefaults("SellValues", 'profile', {
            labels = {
                sv = "|cffffff78Vendor pays|r",
                bv = "|cffffff78Vendor charges|r",
                sep = " ",
            },
        })

        KC_Items.options.args.sellvalues = {
            type = "group",
            name = L["Sell Values"],
            desc = L["SellValues related options"],
            args = {
                toggle = {
                    name = L["Toggle"],
                    desc = L["Toggle the Sell Values module"],
                    type = "toggle",
                    get = function() return KC_Items:IsModuleActive("SellValues") end,
                    set = function(v) KC_Items:ToggleModuleActive("SellValues", v) end
                },
                importDB = {
                    name = L["Import Prices Database"],
                    desc = L["Import the master prices database"],
                    type = "execute",
                    func = function() self:PromptForMasterImport() end,
                }
            }
        }
        
        
        if not self.db.account.maintenance.version then
            KC_Items:ResetDB("SellValues")
            self:Print("Due to a data format change, and bugs that resulted in invalid data, the prices had to be reset.")
        end
        
        self.db.account.maintenance.version = self.version
    end

    function SellValues:OnEnable()
        self:RegisterEvent("MERCHANT_SHOW")
        self:RegisterEvent("KC_Items_ItemTooltipShown", "ItemTooltipShown")
        
        if not self.db.account.maintenance.importedMaster then
            self:PromptForMasterImport()
        end    
    end

    function SellValues:ItemTooltipShown(tt)
        if self._scanning then return end
        local sv, bv = self:GetValues(tt.link)
        
        tt:Prep()
        
        if sv and sv > 0 then tt:AddPriceLine(self:Round(sv * tt.qty), self.db.profile.labels.sv, self.db.profile.labels.sep) end
        if bv and bv > 0 then tt:AddPriceLine(self:Round(bv * tt.qty), self.db.profile.labels.bv, self.db.profile.labels.sep) end
        
        tt:CleanUp()
    end

    function SellValues:MERCHANT_SHOW()

        self:Hook("SetTooltipMoney",true)
        self._itemPrice = nil
        self._scanning = true
        
        for bag = 0, 4 do
            for slot = 1, GetContainerNumSlots(bag) do
                _G.GameTooltip:SetBagItem(bag, slot)
                if self._itemPrice and _G.GetContainerItemInfo(bag,slot) then
                    local _, qty = _G.GetContainerItemInfo(bag, slot)
                    self:SetSellValue(_G.GetContainerItemLink(bag, slot), self._itemPrice/qty)
                    self._itemPrice = nil
                end
            end
        end
        self:Unhook("SetTooltipMoney")
        
        for item = 1, _G.GetMerchantNumItems() do
            if _G.GetMerchantItemLink(item) then
                local _, _, price, qty = _G.GetMerchantItemInfo(item)
                self:SetBuyValue(_G.GetMerchantItemLink(item), price/qty)
            end
        end 
        
        self._scanning = false
    end

    function SellValues:SetTooltipMoney(obj, money)
        self._itemPrice = money
        return self.hooks.SetTooltipMoney(obj, money)
    end
    
    function SellValues:SetBuyValue(item, buyvalue)
        --Buy Value is what the vendor will sell you something for.
        self:SetValues(item, nil, self:Round(buyvalue))
    end
    
    function SellValues:SetSellValue(item, sellvalue)
        --Sell Value is how much a vendor will pay for something.
        self:SetValues(item, self:Round(sellvalue))
    end
    
    function SellValues:GetValues(item)
        local code = tonumber(item) or self:GetShortCode(item)
        local data = self.db.account.data[math.fmod(code, 50)]
        local sv, bv = data:match(';'..code..',(%d+),(%d+);')
        return tonumber(sv), tonumber(bv)
    end
    
    function SellValues:SetValues(item, sellvalue, buyvalue)
        local code = tonumber(item) or self:GetShortCode(item)
        local data = self.db.account.data[math.fmod(code, 50)]
        local sv, bv = self:GetValues(code)
        
        self:Debug('Setting Item: %s. nsv: %s. sv: %s. nbv: %s. bv: %s.', item, sellvalue, sv, buyvalue, bv)
        
        if sv or bv then
            data = data:gsub(';'..code..',(%d+),(%d+);', (';%s,%s,%s;'):format(code, sellvalue or sv or 0, buyvalue or bv or 0))
        else
            data = ("%s%s,%s,%s;"):format(data, code, sellvalue or 0, buyvalue or 0)
        end
        
        self.db.account.data[math.fmod(code, 50)] = data
        
        if __DEBUG__ then
            sv, bv = self:GetValues(code)
            if sellvalue then assert(sv == sellvalue, ("SellValue set incorrectly.  Item: %s. Target: %s. Actual: %s."):format(code, tostring(sellvalue), tostring(sv))) end
            if buyvalue then assert(bv == buyvalue, ("BuyValue set incorrectly.  Item: %s. Target: %s. Actual: %s."):format(code, tostring(buyvalue), tostring(bv))) end
        end
    end

    function SellValues:PromptForMasterImport()    
        local popupName = "KC_SELLVALUES_MASTERMERGE"
        _G.StaticPopupDialogs[popupName] = {
            text = L["Import the master prices database. \n |cffcc0000Note:|r This will revert any data thats been updated. New items will not be changed."],
            button1 = L["Yes"],
            button2 = L["No"],
            OnAccept = function()
                self:ImportMasterDB()
                _G.StaticPopup_Hide(popupName)
            end,
            OnHide = function()
                self.db.account.maintenance.importedMaster = true
            end,
            timeout = 0,
            whileDead = 1,
            hideOnEscape = 1
        }
        
        _G.StaticPopup_Show(popupName)
    end

    function SellValues:ImportMasterDB()
        for key, data in pairs(self:GetMasterDB()) do
            for code, sv, bv in data:gmatch(';(%d+),(%d+),(%d+);') do
                sv, bv = tonumber(sv), tonumber(bv)
                
                sv = sv ~= 0 and sv or nil
                bv = bv ~= 0 and bv or nil
                
                self:SetValues(code, sv, bv) 
            end
        end
    end

    function SellValues:GetMasterDB()
        return {
            ";14201,2060,0;7101,5,0;13051,22911,0;16251,7500,0;1951,1258,0;2751,375,0;5951,41,0;10751,8699,0;13951,11005,0;14851,7416,0;10501,3662,0;16651,62,0;3201,4496,0;12551,11594,0;15751,5000,0;8201,5350,0;14601,7307,0;15501,269,0;2851,44,0;7751,1614,0;20001,0,45000;5001,1038,0;6601,1437,0;12051,8996,0;15251,17685,0;3301,102,0;7051,2696,0;9751,86,0;13851,312,0;1251,10,0;2151,582,2353;2951,656,0;16701,14997,0;14751,3454,0;10401,235,0;5201,3048,0;6801,4456,0;9251,62,0;12451,1500,0;15651,15785,0;2601,100,0;3401,81,0;10151,25456,0;14251,3332,0;17751,12787,0;9901,3294,0;1701,376,0;2251,12,0;6551,409,0;8151,250,0;8751,2618,0;15151,3949,0;10801,11704,0;7001,3535,0;16051,5000,0;17001,10314,0;14901,3344,0;2701,400,0;4251,1178,0;7451,700,0;15801,35035,0;4701,252,0;6301,20,0;13501,0,30000;851,404,1821;1351,705,0;1751,676,0;3151,87,0;18301,29363,0;15551,4438,0;14401,1750,0;10051,2500,0;15301,108,0;6051,62,0;14151,1055,0;13001,10648,0;15051,21736,0;17051,0,70000;1401,14,0;1801,398,0;2451,0,34903;5351,403,0;16801,17322,0;14801,8156,0;3851,6258,0;9301,2500,9000;10001,5257,0;2901,16,73;14551,10601,0;9801,708,0;18351,8740,0;10201,5960,0;13401,51440,0;3951,5897,0;9601,65,0;4051,1669,0;14301,5451,0;1201,84,472;9951,6795,0;12001,4971,0;4601,100,400;4401,100,0;3651,383,0;3801,1031,0;8251,12080,0;201,455,2277;6451,400,0;14951,5458,0;13101,14318,0;9651,26828,0;9851,3355,0;10601,500,0;16851,27663,0;6901,1523,0;12651,36055,0;2651,3,0;9451,25,0;8301,14463,0;1501,80,0;3451,607,0;5751,1028,0;7351,58,0;8401,1375,0;8501,1250,5000;2101,0,3;15601,10345,0;4101,26,0;17201,0,240;6201,54,0;14451,5536,0;10101,15756,0;51,1,0;5051,1,0;8951,200,0;15351,1090,0;3751,2661,0;", -- [1]
            ";8952,200,800;4602,100,320;3902,1317,0;52,1,0;11902,34194,0;13952,56683,0;4352,275,0;7552,2542,0;19202,0,19800;14852,19420,0;6402,4699,0;12552,10980,0;15752,5000,0;5252,1151,0;8202,9022,0;18452,6910,0;11402,1205,0;3202,584,0;14602,2995,0;4002,2463,0;13452,1250,0;15502,411,0;11152,750,0;4552,530,0;7752,6902,0;20002,1000,0;12052,837,0;15252,27906,0;6602,803,0;1602,11664,0;14102,60,0;9752,40,0;12952,12956,0;7052,1488,0;11802,14980,0;3302,319,0;16702,8592,0;14752,978,0;2152,1170,4725;10402,385,0;9252,62,0;15652,7139,0;5202,1072,0;6802,17951,0;11302,7130,0;14502,13051,0;12202,87,0;3402,602,0;17752,28785,0;14252,4914,0;9902,5357,0;15152,4449,0;6552,601,0;8152,500,0;9652,13466,0;16052,5000,0;7002,2679,0;1302,262,0;1702,320,0;18802,5000,0;9402,25308,0;12602,35212,0;15802,11648,0;202,286,1714;14652,4215,0;6302,628,0;13502,11038,0;18302,7530,0;11202,1100,0;3152,28,0;13252,12363,0;1752,286,0;12102,9042,0;15302,47,0;14152,22915,0;6052,300,0;9802,1204,0;2802,1625,0;13002,14617,0;15052,31145,0;852,331,1390;10702,5379,0;12752,21948,0;2452,15,0;14802,9749,0;3652,364,0;1802,337,0;9302,2250,8100;8752,5217,0;10252,20959,0;18352,27552,0;10052,1500,0;14552,15378,0;4652,10126,0;13102,6879,0;10202,3988,0;13402,21168,0;3702,498,0;18102,16853,0;3602,27,116;12252,12619,0;10002,5162,0;14302,6517,0;8402,1375,0;4252,1387,0;14902,7432,0;12002,6322,0;3352,1250,0;15202,501,0;20452,250,0;4402,250,0;9952,5255,0;10102,15020,0;1852,336,1516;2902,1305,0;10302,1250,0;14952,5861,0;4852,300,0;6452,28,0;13052,41677,0;10602,750,3000;3802,627,0;18652,3000,0;8302,11060,0;6902,877,0;9452,8210,0;9852,3820,0;2302,24,0;5752,3109,0;7352,1173,0;18402,18662,0;5002,1535,0;2852,67,0;2102,1,0;2652,7,0;15602,6979,0;18852,3750,13500;4052,2699,0;14452,8836,0;1202,0,1839;1502,60,0;4102,33,0;", -- [2]
            ";15353,3377,0;14203,2754,0;203,229,1147;13053,59186,0;1503,109,0;2303,61,0;5503,16,0;8703,6492,0;13953,56880,0;19203,0,19800;7553,2542,0;9603,234,0;12803,500,0;14853,4057,0;16653,62,0;6403,1771,0;10503,5169,0;15753,5000,0;16403,7573,0;5253,7952,0;11403,1592,0;14603,4383,0;10253,15024,0;3203,5436,0;4003,4090,0;15503,1091,0;4553,411,0;7753,7414,0;10003,2859,0;12053,837,0;15253,30203,0;5003,1713,0;6603,2334,0;14103,9716,0;9753,248,0;12953,21584,0;17203,100000,0;7053,2240,0;15003,20,0;1203,23505,0;3303,70,0;12703,10000,0;15903,1623,8116;16703,8138,0;14753,2610,0;10403,421,0;18503,19757,0;15653,8053,0;2153,0,4742;18253,15,0;6803,4885,0;11303,580,2547;14503,17071,0;10153,16712,0;12203,87,0;15403,275,0;14253,2819,0;3403,321,0;13103,3453,0;8753,5198,0;15153,2626,0;4953,88,0;6553,525,0;8153,5,0;9653,6745,0;17003,44982,0;7003,981,0;10553,85,0;4253,962,0;7453,875,0;9403,4436,0;12603,24532,0;20603,12,0;1303,403,0;1703,81,0;6303,1,0;15553,3061,0;11203,1100,0;10053,4499,0;12103,12157,0;15303,260,0;14153,28182,0;6053,200,800;13003,49640,0;15053,15898,0;1753,439,0;3603,27,116;16803,17806,0;6953,7443,0;14803,9087,0;9903,7742,0;9303,2000,0;2453,25,90;2853,17,0;13453,1250,0;18353,55030,0;14553,17733,0;153,1,0;2653,3,0;4653,7178,0;10203,10397,0;13403,8540,0;18103,15252,0;3153,33,0;853,481,2168;1803,465,0;3703,36,0;14303,16209,0;3753,1635,0;9953,8467,0;53,1,0;15203,621,0;18853,3750,0;17103,68649,0;3853,10153,0;4353,300,0;14953,9158,0;6453,62,0;18653,4000,0;10603,825,0;14653,6854,0;9453,6268,0;12653,34964,0;1853,338,1521;3803,1086,0;6903,1549,0;9853,1368,0;2403,0,220;2553,25,0;17753,25877,0;8203,8628,0;2953,2321,0;5753,1884,0;7353,2998,0;15603,3546,0;8253,5107,0;14453,15677,0;8953,200,640;3453,46,0;4603,4,0;10103,5687,0;9803,729,0;20253,0,14400;", -- [3]
            ";7354,1864,0;2654,2,0;15354,1600,0;17404,6,25;11904,13815,0;15104,3188,0;5504,22,0;2304,15,0;19204,0,36000;13954,20315,0;9604,5632,0;12804,2000,0;7554,1039,0;1504,32,0;14854,14496,0;16654,62,0;10504,7069,0;6404,3242,0;18454,1154,0;11404,2080,0;14604,5986,0;5254,308,0;20254,0,18000;15504,1066,0;4004,2357,0;10004,5717,0;4554,708,0;7754,1959,0;12054,837,0;15254,34114,0;6604,1200,0;2854,225,0;15004,42,0;7054,4700,0;16704,12910,0;12704,7500,0;15904,4071,21707;4304,300,0;18754,10461,0;10404,864,0;6354,1,0;7954,23118,0;15654,8569,0;18254,18,0;11304,972,4374;14504,10000,0;10154,13337,0;20054,0,82910;12204,112,0;15404,337,0;17754,17394,0;754,23556,0;14254,7451,0;9904,5420,0;4504,2222,0;3404,181,0;15154,2372,0;10804,29551,0;6554,209,0;8154,250,0;17004,52607,0;16054,6000,0;14904,9245,0;7004,984,0;10554,105,0;13754,6,0;3854,13980,0;12604,14368,0;15804,9574,0;7454,5915,0;14654,3931,0;9154,300,0;15554,1853,0;11204,1100,0;14404,2471,0;1304,279,0;10054,3000,0;15304,414,0;5604,380,0;3154,1094,0;14154,29028,0;17304,2,10;9804,1874,0;13004,33590,0;804,2500,0;17054,28788,0;19104,64227,0;18854,3750,0;4604,1,4;4054,3625,0;11604,19428,0;14804,15471,0;10104,14077,0;1154,102,0;1754,359,0;8204,4676,0;15704,6145,0;18354,17466,0;954,50,0;18104,17284,0;10204,19888,0;13404,15704,0;9854,4620,0;8254,8639,0;9654,16928,0;9304,2000,0;14304,6194,0;15054,14559,0;4254,927,0;2504,5,26;2904,595,0;18404,6714,0;3754,1492,0;12004,9163,0;15204,779,0;13054,15805,0;1404,10306,0;1804,478,0;1204,12577,0;8754,3937,0;9704,3573,0;9754,70,0;4404,25,100;14954,15234,0;15854,64502,0;10604,825,0;6454,225,0;16604,17,0;854,604,2719;9454,1927,0;18654,4000,0;9954,12125,0;8304,10513,0;4954,6,0;3454,105,0;6904,4470,0;2204,1133,0;2954,2485,0;12404,75,0;5754,2538,0;2454,20,0;20204,0,44918;14454,5598,0;18504,14126,0;6204,2047,0;2604,12,45;", -- [4]
            ";7005,16,74;1205,25,90;10105,19429,0;6205,922,0;15355,1620,0;14205,1042,0;1455,2931,0;13055,29367,0;11905,6203,0;19205,0,36000;13955,16367,0;9605,1356,0;4355,375,0;7555,7123,0;14855,6502,0;16655,62,0;10505,250,0;6405,3167,0;12555,11406,0;15755,5500,0;14605,6868,0;1505,48,0;3555,1032,0;10255,8934,0;13455,750,0;15505,561,0;17705,32854,0;13205,34218,0;3205,209,0;4555,155,0;7755,4140,0;12055,8375,0;15255,42545,0;6605,971,0;16955,42472,0;7055,1503,0;10655,9,0;16705,9127,0;15905,343,2130;18755,42026,0;4305,600,0;18505,13076,0;10405,409,0;2505,0,53;3305,492,0;7955,228,0;12455,1500,0;15655,19862,0;18255,15,0;5205,31,0;10155,20168,0;12205,112,0;15405,245,0;17755,5818,0;3755,5447,0;14255,2628,0;9905,8251,0;13105,1976,0;4505,1974,0;8755,3269,0;15155,2571,0;2605,25,90;6555,130,0;14905,3217,0;5405,7,0;13755,7,0;555,8,0;755,1,0;955,37,120;15805,14662,0;3055,1526,0;3855,11248,0;7455,5937,0;14655,10252,0;10305,100,0;13505,84070,0;4705,1449,0;15555,2802,0;11205,1250,0;10055,3000,0;13255,11972,0;15305,412,0;5605,150,0;14155,20000,0;1705,600,0;4455,3096,0;6055,375,0;11855,782,0;15055,16019,0;17055,26747,0;19105,16476,0;10705,3666,0;16805,17583,0;6505,2881,0;9755,73,0;14805,20577,0;2805,2352,0;5355,947,0;9805,709,0;10455,6464,0;20205,0,26398;9155,400,0;9305,2000,0;15705,30703,0;3655,1022,0;10205,4111,0;805,250,0;13405,13102,0;1755,597,0;1155,5144,0;15455,3541,0;9655,7092,0;4605,6,22;14305,14349,0;9955,8530,0;4055,2481,0;12005,8813,0;15205,1063,0;2905,53,0;20705,21968,0;55,1,0;13005,1331,0;9705,3586,0;12905,13633,0;4405,125,0;4255,2366,0;14955,5418,0;2555,40,0;8255,6010,0;3455,188,0;10605,875,0;1405,1184,0;1805,351,0;3355,50,180;16605,17,0;12655,500,0;15855,7888,0;2455,10,32;8305,11743,0;18405,14744,0;6905,4010,0;9855,1883,0;9405,2055,0;12405,9239,0;15605,3858,0;7355,1031,0;8205,4346,0;14455,15686,0;", -- [5]
            ";4306,150,0;3806,1170,0;8956,200,0;9856,2839,0;13056,48330,0;11906,32723,0;3456,6375,0;15106,14447,0;13956,10950,0;5506,71,0;7106,1180,0;16006,10000,0;16656,62,0;5956,3,16;7556,5387,0;18706,10031,0;10506,5227,0;18456,5613,22452;9356,217,0;6406,1890,0;8006,12472,0;856,875,0;8206,11002,0;3556,1285,0;1506,51,0;12056,8375,0;15256,47531,0;7756,1606,0;3206,4582,0;14106,15203,0;4006,2434,0;9756,294,0;10656,14,0;13856,5112,0;7056,2314,0;16706,25325,0;3656,2116,0;14756,1860,0;7506,500,0;18256,0,6000;15656,13366,0;7956,1944,0;11306,0,17520;3306,653,0;14506,10000,0;10156,12794,0;5206,37,0;12206,112,0;14256,2000,0;9906,3043,0;13106,1216,0;15156,5266,0;1206,400,0;16056,6000,0;6556,174,0;8156,2219,0;14906,2989,0;15806,51273,0;8256,10261,0;14656,4300,0;7456,4139,0;2256,2996,0;10306,100,0;15556,1697,0;11206,1250,0;14406,895,0;10056,1500,0;6756,1375,0;15306,208,0;5606,23,0;556,106,0;11856,15481,0;15056,20966,0;4456,1393,0;6056,500,1800;19106,73353,0;15906,0,2138;18856,3750,0;4906,21,0;6506,87,0;1156,812,0;1706,86,0;11406,168,0;756,4963,0;18606,0,45000;20206,0,13102;9406,4161,0;10256,10675,0;15706,33323,0;2506,57,285;4706,687,0;3056,1532,0;7406,1372,0;10206,4048,0;9656,4251,0;19206,0,54000;15456,3938,0;4656,1,0;6256,1,21;19906,26969,0;13206,19880,0;4256,3477,0;17406,6,0;12606,12668,0;14306,16830,0;10806,14021,0;2456,15,0;56,1,0;9956,5130,0;18306,7215,0;2406,25,0;1756,438,0;12006,1064,0;4556,903,0;18656,4000,0;2656,9,0;3606,15,147;3856,13633,0;9706,9212,0;9806,833,0;14956,3108,0;4406,600,0;7606,557,0;4606,25,90;13006,54761,0;10606,875,0;2156,615,2769;14206,1046,0;15356,3935,0;9456,6652,0;17056,7,0;15856,8788,0;10406,1976,0;8306,11787,0;3356,30,0;18506,21340,0;6906,1275,0;10106,12120,0;20406,1929,0;9206,700,0;12406,4464,0;15606,3585,0;5756,9545,0;7356,1411,0;", -- [6]
            ";6057,500,0;14457,7269,0;5757,1265,0;7357,1835,0;10107,7218,0;8957,0,800;15357,2693,0;1807,387,0;2207,478,1912;3807,1162,0;4607,50,180;14207,3076,0;17407,50,0;13057,5382,0;5057,1,0;6657,5,0;11907,41050,0;15107,5000,0;13957,12427,0;18957,250,0;5507,600,0;7107,1803,0;9607,1762,0;14857,9266,0;4357,4,16;5957,32,0;7557,7596,0;15757,5500,0;1457,1850,0;6407,1198,0;11407,108,0;14607,5402,0;13457,750,0;5257,3159,0;15507,364,0;17707,11631,0;7307,62,0;12057,8375,0;15257,55234,0;19507,1701,0;7757,12874,0;14107,13140,0;857,2500,0;1507,123,0;4007,3261,0;6607,2202,0;13857,10878,0;16707,19962,0;5457,23,0;7057,2257,0;15907,1647,8236;14757,3298,0;18507,15692,0;2057,16,0;2857,198,0;4307,29,0;20557,175,700;7507,400,0;10407,431,0;12457,1500,0;15657,22625,0;10157,29943,0;1557,918,0;5207,1081,0;12207,150,0;15407,500,0;14257,6194,0;13107,11291,0;2957,97,0;4507,8282,0;10807,14864,0;17007,11037,0;9657,5333,0;4957,250,0;6557,302,0;8157,5116,0;14907,6123,0;13757,10,0;1207,9892,0;1607,43420,0;12607,8048,0;15807,39,235;8257,6053,0;14657,4195,0;18307,11163,0;4257,1248,0;7457,2444,0;10307,112,0;3057,903,0;6307,1,0;11207,3000,0;14407,4266,0;10057,12711,0;13257,18088,0;6757,4630,0;15307,815,0;13007,13585,0;16207,1250,0;17057,7,0;15057,18728,0;19107,44171,0;4457,300,0;10707,7772,0;5007,1632,0;9757,363,0;18857,3750,0;12757,33362,0;10257,15108,0;9807,1255,0;4907,11,0;6507,44,0;8107,5938,0;14807,8211,0;18607,12500,45000;4757,4,0;15707,7215,0;20407,2581,0;10007,5526,0;1707,62,180;3857,125,450;3607,24,0;7407,3666,0;18457,0,10698;9407,4654,0;10207,5998,0;12257,3770,0;15457,1134,0;4557,225,0;4107,2038,0;19307,100,0;14307,12064,0;4707,881,0;57,1,0;2407,162,0;9957,13753,0;16357,375,0;4057,4032,0;12007,1064,0;9907,4581,0;9857,1493,0;2907,1755,0;14957,6368,0;3207,87,0;1757,728,0;2657,875,0;4407,1200,0;7607,1355,0;10607,900,3600;16607,5,0;2457,15,0;5107,139,0;15857,15335,0;8307,18717,0;18407,10696,0;2507,350,1576;9357,227,0;3357,75,0;6907,1872,0;15607,9305,0;20207,0,8490;8207,8978,0;", -- [7]
            ";1708,50,200;3358,175,0;14458,12573,0;13308,450,0;5758,250,0;7358,885,0;15358,3604,0;4608,100,360;2208,728,3285;9858,4882,0;13058,21833,0;11908,9301,0;15108,5000,0;9608,5028,0;12808,1000,0;7108,209,0;16658,1649,0;10508,8958,0;4358,30,0;5958,799,0;7558,1609,0;15758,5500,0;8208,10272,0;11408,898,0;14608,3162,0;6408,1577,0;2308,267,0;15508,365,0;1458,2623,0;10008,2619,0;13208,12520,0;4108,5403,0;19508,1423,0;14108,11112,0;12958,12500,0;11808,19292,0;15008,37,0;6608,1328,0;16708,18175,0;2408,125,0;5458,27,0;18508,17194,0;14758,2267,0;10408,2390,0;4308,45,0;7508,400,0;12458,1500,0;15658,17850,0;11308,0,78828;14508,15000,0;6358,4,0;7958,2435,0;858,25,80;13358,21036,0;12208,150,0;15408,500,0;5208,668,3006;3308,144,0;14258,3109,0;9908,4257,0;13108,2446,0;15158,4320,0;10808,7404,0;7708,6649,0;2158,0,1859;3758,1817,0;16058,10156,0;14908,8195,0;6558,180,0;8158,2236,0;10558,250,0;13758,4,0;15808,672,2912;2608,63,0;14658,7604,0;10308,112,0;13508,3185,0;7458,3983,0;1608,18940,0;15558,4128,0;11208,1350,0;14408,2275,0;4708,461,0;6308,2,0;2258,83,0;10058,6278,0;3058,679,0;13258,12605,0;3858,250,0;15308,241,0;14158,1152,0;9808,1848,0;13008,12425,0;5608,3463,0;17058,7,0;19108,43967,0;11858,4768,0;10708,8982,0;6058,6,0;9758,72,0;10258,12292,0;4908,6,0;6508,34,0;8108,3594,0;20658,15272,0;3158,60,0;9358,228,0;20408,1942,0;9308,38,0;15708,8965,0;10108,8203,0;9658,4086,0;10208,9586,0;1158,146,0;7408,2281,0;3558,1044,0;15458,1423,0;4558,1565,0;2508,5,0;10358,2309,0;16608,185,0;18308,12970,0;12008,895,0;14858,11740,0;3008,43,0;13858,9591,0;10858,750,2700;4058,6056,0;14958,12168,0;9408,16429,0;2958,78,0;4258,1449,0;2458,15,0;15058,14692,0;10608,950,0;4408,162,0;7608,1106,0;18658,6000,0;2108,8,0;9458,6917,0;2908,1409,0;3108,0,1;3208,25378,0;8308,18703,0;8258,17396,0;1758,534,0;20208,15192,0;12408,4998,0;15608,5820,0;6908,522,0;", -- [8]
            ";159,1,4;12409,10336,0;15609,11806,0;14459,10798,0;13309,250,0;1359,19,0;1759,732,0;5759,375,0;8959,160,0;15359,5955,0;14209,1196,0;4609,250,0;13059,37759,0;11909,8267,0;2209,1394,6403;6659,541,0;13959,9542,0;12809,10000,0;18709,5758,0;7109,74,0;14859,14820,0;16409,8240,0;18459,7907,0;1809,440,0;4359,12,0;15759,5500,0;8209,8375,0;11409,503,0;6409,1660,0;13459,100,0;15509,485,0;17709,500,0;10009,5284,0;13209,9615,0;15259,3306,0;14109,289,0;1009,490,0;1459,2380,0;11809,51287,0;15009,162,0;16709,25542,0;6609,2148,0;10659,4662,0;12709,57571,0;15909,0,22106;5459,472,0;7059,2725,0;10409,2409,0;4309,200,0;7509,275,0;12459,1500,0;15659,9508,0;1509,59,0;2059,863,0;6359,5,0;7959,31464,0;10159,11444,0;13359,25336,0;17759,10307,0;15409,1000,0;5209,770,3470;14259,5056,0;13109,8152,0;2509,82,373;3309,333,0;15159,6743,0;4509,2794,0;16059,100,400;14909,4897,0;1159,32,0;1959,1223,0;2159,0,3733;2959,32,0;3759,1216,0;16509,8866,39898;6559,533,0;8159,2244,0;10559,750,0;13759,10,0;9409,4490,0;12609,20498,0;15809,2938,11753;18309,11296,0;14659,9382,0;13509,5393,0;7459,4077,0;12359,600,0;15559,1450,0;14409,1707,0;7909,1000,0;10059,4739,0;13259,14525,0;15309,220,0;2259,75,0;3859,60,0;9809,1855,0;13009,15167,0;17059,55000,198000;19109,33920,0;5609,427,0;11859,7135,0;9759,138,0;19259,12500,0;18859,3750,12000;4459,150,0;6059,1,0;3209,6768,0;10309,125,0;4709,982,0;20659,12263,0;6509,44,0;8109,5011,0;9609,1009,0;15709,7270,0;59,1,0;9909,2848,0;14559,243,0;10109,7273,0;9859,2871,0;10209,7216,0;859,87,0;7409,2289,0;3559,473,0;12259,8072,0;4109,4393,0;11109,6,25;14309,6866,0;2309,268,0;3609,25,0;4659,23,0;9359,19504,0;2409,350,0;4259,1934,0;9959,4803,0;809,31520,0;12009,2174,0;5109,223,0;6709,545,0;19159,21141,0;2459,25,0;9259,64,0;16859,11715,0;8259,9144,0;4059,2184,0;14959,5143,0;10609,1000,0;3809,926,0;4409,200,0;7609,4210,0;9459,13560,0;18409,11145,0;8309,26278,0;10259,11030,0;5009,1696,0;", -- [9]
            ";1710,125,450;6910,2298,0;13310,500,0;15360,2372,0;14210,1636,0;13060,60784,0;1760,609,0;11910,12345,0;15110,351,0;13960,12828,0;5060,0,1500;6660,33232,0;16660,2200,0;3810,1145,0;18710,6484,0;14860,10523,0;7110,2069,0;16410,5630,0;10510,11369,0;15760,5500,0;4360,60,0;7560,300,1080;11410,578,0;4810,3183,0;6410,1514,0;1810,648,0;17710,33952,0;15260,11556,0;19260,12500,0;9760,73,0;12960,17125,0;4560,26,0;7760,6074,0;11810,10000,0;16710,10446,0;13860,8741,0;6610,2156,0;18510,20889,0;14760,6263,0;7060,2791,0;10410,1256,0;3210,5626,0;9260,400,1600;12460,1500,0;4310,180,0;810,45267,0;1010,99,0;11310,6163,0;10160,19948,0;13360,55557,0;6360,2581,0;7960,38548,0;12210,300,0;14260,2900,0;5210,0,5226;1510,150,0;15160,2430,0;17010,2000,0;19060,100,360;3310,442,0;16060,100,400;7710,12279,0;18810,40678,0;16510,5931,26690;10560,1000,0;13760,10,0;6560,490,0;8160,3378,0;2160,0,3747;15810,2029,10145;18310,47406,0;8260,6938,0;14660,7063,0;1560,513,0;10310,125,0;13510,5000,0;12360,5000,0;4260,2559,0;7460,2000,0;14410,2709,0;10060,6667,0;6310,150,0;7910,5000,0;860,89,0;15310,279,0;14160,251,0;17060,0,198000;2260,106,0;3860,400,0;19110,54922,0;11860,12970,0;15060,20257,0;5610,208,0;10260,11945,0;1210,250,0;18860,0,164710;10710,6130,0;2210,1,0;4460,175,0;9860,2382,0;6060,2,0;11610,3000,0;15610,4354,0;2910,1257,0;6510,54,0;2960,21,0;8110,3622,0;14560,484,0;9910,3087,0;3010,101,0;10210,12845,0;3160,470,0;4060,5648,0;13810,0,1080;9410,3702,0;9060,1000,0;12810,500,0;7410,1111,0;11110,2,0;14310,11615,0;18460,30762,0;2510,7,41;13210,17668,0;60,12,0;9960,5623,0;4660,264,0;2310,112,0;6260,12,45;12010,2469,0;3610,50,0;15210,825,0;3360,625,0;16110,0,12000;13110,4968,0;10110,7947,0;5110,257,0;710,72,0;14910,3033,0;14960,4870,0;8210,5537,0;9660,4227,0;18660,7500,0;4960,12,0;13010,1646,0;3560,2102,0;3760,1500,0;15860,7907,0;7610,5811,0;4710,561,0;1310,715,0;9810,1231,0;8310,19419,0;10360,1250,0;4860,741,0;9210,750,0;", -- [10]
            ";15611,4369,0;1711,75,0;2911,524,0;6911,1442,0;13311,2500,0;15361,2204,0;5761,30,0;9861,1669,0;6211,450,0;11911,12392,0;15111,609,0;10761,17193,0;13961,18028,0;12811,20000,0;18711,8700,0;18461,8577,38599;7111,3329,0;15761,6250,0;4361,120,480;5961,1089,0;7561,500,2000;8211,8002,0;11411,1484,0;14611,32471,0;10261,17229,0;13461,750,0;20011,0,45000;6411,7268,0;15511,1474,0;17711,14244,0;12061,40212,0;15261,13530,0;1411,68,0;1811,451,0;19261,12500,0;14111,9143,0;9761,58,0;7761,11110,0;11811,1500,0;15011,183,0;16711,17505,0;18761,39230,0;5011,1912,0;6611,770,0;14761,2024,0;10411,583,0;15661,9036,0;3211,373,0;20061,0,128505;7511,1203,0;11311,5110,0;10161,12713,0;13361,56730,0;6361,2,0;7961,25508,0;15411,10283,0;14261,4044,0;5211,1106,5246;13111,13993,0;15161,3319,0;17011,2000,0;1511,193,0;2511,264,1192;7711,758,0;14911,5327,0;10561,1000,0;6561,637,0;8161,3390,0;9411,5851,0;15811,5426,27132;18311,56644,0;14661,5461,0;2961,155,0;10311,750,3000;12361,7000,0;15561,4705,0;7461,2429,0;14411,1387,0;10061,7661,0;1161,18,0;1561,223,0;6311,187,0;7911,500,0;15311,982,0;14161,337,0;9811,725,0;13011,1031,0;9511,21060,0;4561,309,0;15061,13690,0;16811,26962,0;5611,452,0;4711,1112,0;10711,8377,0;9261,250,0;4061,3306,0;18611,27,0;711,0,25;4461,208,0;20661,9140,0;6061,23,0;9461,4586,0;10461,3082,0;1211,347,0;2011,3840,0;3511,267,0;4911,15,0;6511,112,0;8111,5890,0;9911,7228,0;14561,245,0;3561,895,0;61,12,0;9061,250,0;3761,3210,0;4611,50,0;2361,8,0;3161,291,0;3961,1503,0;7411,1769,0;14311,6524,0;2211,7,0;9961,5644,0;2311,150,0;13161,64611,0;1011,80,0;4661,929,0;6261,0,1000;12011,4649,0;15211,1868,0;961,2,0;14861,5045,0;16861,17786,0;3661,7,0;3011,2846,0;3811,1064,0;911,2944,0;18861,27030,0;14961,9259,0;10211,12223,0;20561,1,0;811,54123,0;8261,10617,0;10111,13444,0;8311,5354,0;", -- [11]
            ";20212,25508,0;15612,4285,0;11262,8142,0;2912,6731,0;12162,750,2400;3712,87,0;14212,2194,0;1712,62,0;9862,4260,0;13062,2991,0;15112,354,0;10762,6903,0;13962,12064,0;10512,2,0;7112,4258,0;2212,16,0;3012,50,160;15762,0,25000;14612,27185,0;4362,187,0;5962,2785,0;20012,0,45000;10262,25272,0;6412,4711,0;13212,10670,0;12062,40365,0;15262,21552,0;19262,12500,0;14112,13509,0;4112,4182,0;16712,11714,0;11812,13979,0;15012,296,0;18762,27631,0;1412,49,0;1812,452,0;18512,4000,0;15912,1148,0;6612,1112,0;11562,1000,0;14762,4074,0;15662,10801,0;14512,15000,0;4312,237,0;10162,28137,0;3212,929,0;13362,2000,0;15412,500,0;6362,4,0;1462,1306,0;13112,9624,0;5212,648,0;2862,3,0;11962,7939,0;3662,25,0;15162,5478,0;18812,32458,0;9662,4258,0;7712,881,0;10562,750,0;18312,18073,0;15812,9574,0;6562,508,0;8162,4900,0;1512,194,0;14662,9413,0;10312,1500,0;15562,3558,0;14412,2298,0;4262,2652,0;7462,2682,0;812,57018,0;12112,10406,0;15312,745,0;7912,100,0;14162,445,0;17062,550,1980;2312,181,0;19112,10984,0;9812,570,0;20062,100,0;13012,1241,0;2612,0,130;16812,18040,0;11862,7042,0;15062,25503,0;5612,38,0;2962,120,0;12662,600,0;18612,34,0;2262,8746,0;1262,111,0;4462,1425,0;10362,24760,0;6062,26,0;10462,3115,0;3562,823,0;9262,1000,0;6512,59,0;8112,8436,0;9762,92,0;11362,0,900;14962,11690,0;4712,1228,0;3812,971,0;10212,11069,0;18712,5230,0;1212,21,0;15462,653,0;4562,290,0;8262,14519,0;14312,13541,0;5812,1033,0;7412,1302,0;2562,575,0;10062,4758,0;13162,10477,0;3962,3079,0;2362,1,0;9962,9218,0;3462,2193,0;12012,2463,0;15212,3450,0;4662,24,0;4062,8359,0;14562,982,0;9312,12,0;15362,3871,0;3312,70,0;16112,550,1760;6712,12,0;14262,3156,0;18662,5,0;9912,5037,0;9412,16681,0;15862,28789,0;10112,19071,0;20562,1,0;8312,14810,0;8212,11585,0;", -- [12]
            ";11513,1,0;7613,500,0;10363,27396,0;15613,3688,0;1263,79064,0;2463,0,12326;3263,4,0;6463,1527,0;11263,8279,0;10113,13541,0;17413,7750,0;5313,650,0;12163,1100,0;15363,4558,0;14213,3449,0;2913,1111,0;5763,700,0;7363,525,0;13063,5851,0;11913,13309,0;10763,5197,0;1713,5350,0;6663,150,0;18463,39536,0;10513,5,0;7113,6707,0;9363,12,0;2213,24,0;4363,50,200;5963,3107,0;13463,100,0;17713,14641,0;15513,1264,0;6413,2497,0;11163,750,0;13213,9633,0;3463,9,0;5263,1,0;15263,28889,0;19263,12500,0;14113,139,0;16963,44900,0;12963,27717,0;16713,11196,0;11813,3000,0;15013,86,0;13863,6617,0;9513,305,0;12713,10000,0;6613,744,0;11563,1000,0;14763,1685,0;18263,16911,0;10413,307,0;1413,55,0;1813,522,0;2763,240,0;12463,42446,0;15663,10226,0;7513,8171,0;10163,20264,0;12213,300,0;15413,22174,0;7963,6488,0;14263,5987,0;13113,20546,0;5213,5101,0;15163,3344,0;2863,4,0;16513,11838,53960;14913,6697,0;7713,4683,0;18313,13694,0;15813,10113,0;3313,519,0;6563,295,0;8163,3688,0;14663,5141,0;12363,2000,0;1513,293,0;2163,46710,0;2963,41,0;3763,6761,0;7463,6202,0;10063,7165,0;12113,10443,0;15313,356,0;4713,926,0;14163,1072,0;9763,336,0;4813,33,0;10263,18116,0;20013,0,45000;9813,825,0;13013,12355,0;2613,107,486;16813,27866,0;15063,11701,0;18413,17070,0;10713,500,0;3663,125,0;5613,2482,0;15963,4887,0;9413,32248,0;14813,10601,0;2263,2346,0;4463,905,0;6063,15,0;9913,8490,0;9313,12,0;5113,250,0;3563,557,0;6513,28,0;8113,9694,0;3013,25,100;14563,275,0;8263,10373,0;10213,7850,0;863,8964,0;5363,20,0;3713,40,160;15463,787,0;2313,200,0;16413,11692,52617;14313,9298,0;8213,8150,0;7413,2157,0;9963,9211,0;13163,69537,0;16363,750,0;12013,4649,0;15213,8165,0;1213,87,0;1613,19139,0;2013,3644,0;3963,1402,0;4663,23,0;6263,589,0;4063,3596,0;16863,17917,0;3363,1,0;3813,1772,0;16113,1250,4000;6713,9,0;14963,8301,0;17013,16676,0;4963,30,0;4563,110,0;9663,9969,0;13813,300,1080;9863,3207,0;15863,28896,0;4113,4853,0;", -- [13]
            ";10364,24938,0;4414,462,0;15614,7026,0;12164,1100,4400;15364,2431,0;14214,2009,0;9864,3527,0;13064,10807,0;5764,3000,0;7364,550,0;16914,40692,0;10764,10366,0;13964,52306,0;6214,595,0;864,9716,0;16414,11735,52808;6664,892,0;2564,6817,0;1314,362,0;1714,2535,0;11414,1828,0;20014,0,45000;10264,25657,0;13464,100,0;4364,12,48;5964,2609,0;17714,10107,0;15514,1902,0;11164,750,0;4814,6,0;6414,2055,0;19264,12500,0;15264,41517,0;14114,242,0;12964,28757,0;16714,10541,0;1364,8,0;1764,99,0;11814,13083,0;13864,9365,0;4564,610,0;2314,465,0;8364,6,0;11564,1000,0;14764,2146,0;6614,1120,0;12464,8520,0;14514,15000,0;10164,14808,0;4314,331,0;7514,7921,0;1414,97,0;1814,603,0;12214,300,0;15414,1500,0;14264,8013,0;7964,40,0;9914,7557,0;3214,33,0;13114,2574,0;15164,9135,0;5214,3935,0;2864,630,0;7714,3834,0;18314,12100,0;1464,71,0;15814,41852,0;8264,7857,0;14664,15058,0;6564,512,0;13514,5820,0;12364,10000,0;15564,1000,0;10064,10774,0;2164,27031,0;2764,440,0;17064,45914,0;14164,287,0;4714,534,0;9764,300,0;1514,294,0;2964,36,0;3914,6250,0;13014,33805,0;18864,3750,0;15064,19129,0;3664,100,0;2614,232,928;1214,930,0;11514,1,0;8214,6230,0;2464,0,6184;15964,6364,0;16514,9024,40609;5614,6147,0;414,6,20;5114,96,0;9664,5002,0;4114,5826,0;4464,2380,0;19764,0,45000;20564,1,0;2264,1957,0;3864,800,0;3814,1891,0;4264,2804,0;2364,54,295;14564,304,0;15114,372,0;6514,42,0;8114,3648,0;19114,17887,0;10214,8271,0;3314,163,0;9814,691,0;2214,182,0;12264,1500,0;15464,4504,0;14914,3240,0;5364,27,0;14414,1276,0;14314,7497,0;15214,11356,0;13164,10539,0;5814,2351,0;7414,2999,0;9264,35,0;814,25,0;8314,5344,0;9964,13190,0;12014,6289,0;13314,29927,0;6264,746,0;9514,306,0;3164,33,0;14614,13201,0;11764,10398,0;17414,14750,0;2014,4617,0;10314,0,3200;15864,1797,0;", -- [14]
            ";15865,6562,0;5565,1250,4500;8315,5061,0;11515,1,0;1015,24,0;2015,4443,0;4415,550,0;12415,17003,0;15615,7584,0;14465,8596,0;4865,5,0;6465,768,0;15365,2415,0;4065,5311,0;5315,141,0;19215,5500,19800;9865,1942,0;13065,20945,0;5765,4000,0;11915,20124,0;15115,473,0;10765,3367,0;13965,16250,0;1265,11774,0;6215,445,0;16415,0,39750;15765,7500,0;3365,3,0;11415,200,0;14615,12430,0;20015,1658,0;13465,150,0;17715,10225,0;5965,2536,0;15515,844,0;11165,750,0;1315,13000,0;1715,9288,0;2215,81,0;6415,4565,0;19265,12500,0;12065,7953,0;15265,46823,0;9765,406,0;12965,21810,0;16715,17766,0;11815,10000,0;13865,13555,0;20565,1,0;11565,1000,0;2315,307,0;6615,824,0;12465,10261,0;15665,25520,0;5465,3,0;7065,2398,0;10165,6240,0;2765,811,0;3565,280,0;4315,425,0;7515,5382,0;15415,500,0;14265,9853,0;6365,180,0;13115,6990,0;11965,464,0;15165,3744,0;1415,72,0;1815,366,0;5215,8656,0;16515,0,54338;14915,11569,0;18315,12100,0;2065,113,0;2865,962,0;15815,11860,0;14665,6214,0;6565,393,0;8165,500,0;13515,9600,0;12365,250,0;15565,4274,0;1465,9824,0;3315,454,0;14415,3410,0;10065,12153,0;4265,650,0;7465,6747,0;12115,7012,0;17065,33381,0;14165,1041,0;4715,1026,0;6315,2546,0;9815,1517,0;13015,49153,0;16215,3000,0;10315,1750,0;18865,0,223580;5115,101,0;3965,2221,0;11865,8708,0;15065,22233,0;2465,0,11229;2165,45,0;2565,3113,0;2815,19669,0;15965,7996,0;1515,196,0;3415,3497,0;5615,1776,0;9915,5091,0;765,10,0;1215,469,0;4765,559,0;2965,189,0;4665,22,0;4465,1575,0;20165,0,28904;9415,8852,0;8365,4,0;14565,809,0;2265,477,0;3065,658,0;4915,3,0;6515,37,0;8115,5980,0;10215,15440,0;14365,214,0;15465,2713,0;8215,9017,0;10365,26276,0;14315,16550,0;9965,9973,0;3665,150,0;1965,36,0;7415,1484,0;12015,8811,0;15215,16744,0;3765,4811,0;14765,4427,0;2615,0,4073;865,5060,0;9665,4826,0;11765,12525,0;14965,4450,0;19765,0,45000;8265,19914,0;2915,55129,0;4565,31,0;", -- [15]
            ";8766,200,720;2366,59,298;3166,403,0;15866,562,0;8316,8078,0;20216,20364,0;5566,105,0;7166,170,0;15616,10151,0;4416,600,0;14466,3000,0;15366,5250,0;14216,4887,0;5316,1295,0;9866,5605,0;13066,8997,0;4066,4845,0;11916,11834,0;15116,1163,0;5766,1219,0;7366,1098,0;816,305,0;18716,18655,0;10766,9660,0;13966,16250,0;2916,4405,0;14866,13301,0;9366,3689,0;11416,328,0;5516,317,0;13466,250,0;19766,0,45000;4366,75,0;5966,1374,0;11166,1000,0;13216,13250,0;6416,2728,0;12066,11149,0;15266,54405,0;5266,11157,0;1716,5067,0;16716,11487,0;9766,361,0;12966,14018,0;15016,200,0;13866,10298,0;4566,631,0;11566,1000,0;14766,5426,0;6616,2203,0;2316,408,0;12466,5968,0;17766,44863,0;5466,8,0;1366,1,0;1766,131,0;10166,5653,0;15416,1000,0;4316,743,0;19316,7,0;2766,1564,0;3566,1807,0;13116,19370,0;4766,448,2406;7966,200,0;11966,164,0;15166,3945,0;5216,11684,0;16516,8223,37006;3216,55,0;1416,72,0;1816,486,0;8266,7468,0;2866,752,0;14666,6250,0;6566,411,0;15566,2178,0;14416,1285,0;10066,4154,0;2216,210,0;5116,303,0;16166,0,5;17066,41198,0;14166,238,0;9816,1781,0;9416,22583,0;13016,4746,0;15066,23650,0;10716,750,0;15966,9137,0;866,16538,0;18366,9805,0;9916,3566,0;3466,500,1800;2616,0,2127;1116,1250,0;3966,2625,0;1516,211,0;14566,1010,0;15666,18375,0;10216,6874,0;4916,9,0;8116,4632,0;2266,479,0;3066,497,0;2966,146,0;13166,14204,0;8216,6323,0;2066,68,0;14316,11271,0;14266,5606,0;3866,1000,0;766,57,0;9966,6743,0;10366,32052,0;12016,7781,0;15216,25932,0;7416,1442,0;4816,1503,6766;4716,1658,0;3816,2102,0;3666,100,0;4666,26,0;6266,205,0;19216,0,19800;1566,1132,0;20716,11225,0;11766,8606,0;18866,0,224377;20016,3635,0;16416,0,39890;20066,100,0;13816,10561,0;6716,200,0;", -- [16]
            ";117,1,5;13817,18796,0;5117,825,0;9467,18657,0;8317,8595,0;767,100,0;3167,68,0;3967,2903,0;15617,6672,0;4417,675,0;14467,3000,0;4867,8,0;14217,1638,0;5317,1360,0;9867,1778,0;13067,13385,0;11917,6334,0;16667,25778,0;2017,895,0;4067,6381,0;18717,68818,0;5767,25,0;7367,1818,0;13967,23338,0;6217,24,112;14867,17802,0;2117,3,0;2917,665,0;6667,992,0;11417,1204,0;14617,6250,0;10267,13337,0;13467,250,0;15517,850,0;1017,100,0;2567,2484,0;5967,209,0;11167,1000,0;14367,251,0;13217,8814,0;6417,2536,0;15267,51958,0;14117,163,0;2217,243,0;16717,12244,0;5267,59731,0;9767,616,0;12967,16884,0;16167,0,25;11817,39254,0;15017,611,0;4117,3119,0;18517,40000,0;13867,12570,0;20567,1,0;9517,13114,0;18267,5000,0;1317,3500,0;1717,2939,0;11567,1000,0;14767,4478,0;6617,1371,0;17767,13610,0;5467,7,0;7067,400,0;10167,9429,0;12217,300,0;15417,500,0;4317,1076,0;19317,7,0;7517,9372,0;14267,8816,0;17017,0,162000;13117,3830,0;867,5344,0;1367,1,0;1767,101,0;6367,3316,0;7967,250,0;11967,1087,0;15167,7484,0;18567,0,135000;14917,3931,0;2417,3074,12538;3217,134,0;4017,6910,0;18317,12503,0;6117,1,0;7717,18349,0;14667,10770,0;1417,65,0;1817,501,0;2067,186,0;3667,25,0;6567,1036,0;8167,100,0;14417,5888,0;17717,25637,0;10067,5295,0;17067,75452,0;14167,358,0;3267,25,0;8267,8927,0;9817,1426,0;13017,17209,0;16217,3000,0;18867,0,281441;11867,8769,0;15067,12650,0;2967,66,0;15967,10304,0;3567,922,0;2317,689,0;2817,165,0;5617,244,0;8367,18455,0;4717,2014,0;2617,1470,8793;14567,529,0;10217,16688,0;8217,1000,0;6517,35,0;8117,7161,0;15467,8630,0;2367,38,223;2267,711,0;3067,1275,0;5367,22,0;6967,648,0;9967,2900,0;13167,59410,0;16417,14827,0;9917,5519,0;15217,30996,0;16867,36358,0;3817,1585,0;5817,3145,0;7417,2759,0;4567,1049,0;3767,23,0;19217,0,36000;6267,149,0;11767,8637,0;10317,0,3200;14967,6812,0;18417,10000,0;", -- [17]
            ";6218,24,0;13818,13431,0;118,5,0;5118,71,0;10368,17868,0;5568,4,0;12418,10242,0;14468,3000,12000;10118,23381,0;15368,3674,0;6468,590,0;19218,0,36000;11018,146,0;14218,2713,0;9868,2160,0;13068,4785,0;768,113,0;11918,9537,0;15118,1562,0;18718,16235,0;10768,4541,0;13968,16250,0;7368,2612,0;4068,6918,0;14868,13109,0;2018,2856,0;10518,4696,0;15768,7500,0;6668,1210,0;17718,22569,0;10268,13978,0;13468,1000,0;14368,346,0;4368,383,0;13218,53388,0;15268,1024,0;6418,1619,0;16718,18185,0;9768,292,0;12968,16948,0;16168,0,400;3018,781,0;15018,706,0;13868,9665,0;4118,7313,0;818,49,0;15918,4848,0;14768,8566,0;4568,1694,0;10418,16464,0;17768,7891,0;1318,3066,0;1718,8040,0;10168,9317,0;13368,55027,0;7068,400,0;2318,15,0;12218,300,0;19318,250,0;14268,3486,0;4318,610,0;7518,9406,0;17018,0,72000;13118,10601,0;16768,37500,0;11968,997,0;15168,7951,0;6368,0,320;1368,2,0;1768,217,0;18318,21202,0;9418,29413,0;4018,6372,0;14668,13022,0;6118,1,0;7718,6867,0;868,21532,0;10318,1750,0;15568,1813,0;6568,920,0;8168,100,0;2218,501,0;4668,32,0;11418,604,0;10068,8309,0;1418,8,0;1818,1221,0;2868,807,0;14168,275,0;14418,3098,0;7468,6713,0;9718,11221,0;13018,31967,0;9818,1465,0;16218,3500,0;11868,6542,0;2418,0,12585;7918,3701,0;9918,10212,0;8268,8960,0;18368,12348,0;2968,33,0;3868,1250,0;3818,125,0;1468,28,0;8218,1000,0;9318,5,0;9968,2695,0;14918,3432,0;14568,797,0;6068,375,0;4718,2223,0;1168,38483,0;8368,1000,0;10218,17589,0;2618,0,21311;3968,3331,0;15468,1207,0;6518,70,0;8118,4436,0;718,284,1453;918,1250,0;8318,12761,0;11118,10795,0;13168,16486,0;5368,48,0;2268,146,0;15218,37047,0;4818,2854,0;10018,3275,0;7418,4256,0;18868,0,282437;19118,17344,0;2568,31,0;11768,7224,0;14968,6502,0;6268,234,0;", -- [18]
            ";14969,12241,0;6269,193,0;13819,19880,0;9469,14634,0;719,4,0;15869,50,0;6719,1097,0;8319,9181,0;19969,3264,0;10369,7377,0;5569,203,0;12419,15183,0;15619,5987,0;14469,0,12000;3169,18,0;4419,112,0;10119,16676,0;15369,3097,0;19219,0,54000;6469,2240,0;14219,2046,0;9869,2225,0;2819,3830,0;16669,21995,0;5319,323,0;11919,9572,0;15119,13683,0;10769,5277,0;13969,15256,0;16419,14934,0;7369,2955,0;11669,17157,0;14869,7577,0;3269,12,0;6219,144,0;5069,146,0;6669,1378,0;11419,1900,0;17719,32106,0;769,3,0;3719,1027,0;15519,607,0;14369,809,0;4369,1179,0;19269,12500,0;5969,781,0;10019,3631,0;15269,2847,0;6419,2246,0;10919,202,0;14119,378,0;16719,27192,0;9769,1637,0;12969,69550,0;16169,0,200;5269,95,0;11819,10000,0;15019,322,0;13869,9702,0;3019,423,0;18269,0,1500;14769,5554,0;4569,568,0;3469,38,0;10169,14230,0;13369,17450,0;5469,9,0;7069,400,0;15419,600,0;14269,5669,0;1319,462,0;4319,815,0;7519,7635,0;13119,3063,0;11969,1721,0;15169,5242,0;6369,550,1980;7969,250,0;14919,7717,0;9419,18076,0;12619,24058,0;8269,14362,0;11469,3563,0;14669,9654,0;1369,1,0;1769,163,0;2419,1470,6315;4019,8336,0;19869,21051,0;6119,1,0;7719,6863,0;6569,1044,0;8169,500,0;10069,5335,0;2869,1831,0;19769,0,45000;5419,13,0;9719,6250,0;3869,1250,0;16819,17189,0;2319,50,0;18869,0,256472;7469,6284,0;9819,2726,0;13019,4721,0;16219,3500,0;5119,118,0;11869,6542,0;869,18529,0;1419,19,0;1819,768,0;3319,110,0;4719,1676,0;6319,803,0;7919,2251,0;15969,425,0;10269,13854,0;18369,9915,0;3369,25,0;20069,0,550995;9969,3964,0;2169,943,0;2969,355,0;3669,195,0;14569,472,0;10219,12011,0;2369,25,0;1469,1180,0;3419,125,0;4919,5,0;6519,36,0;8119,12725,0;14319,10659,0;17119,6,0;3969,3256,0;13169,23948,0;16369,7806,37873;12019,4220,0;15219,41376,0;4119,7271,0;3819,100,0;9919,5149,0;1169,18815,0;2569,87,0;3069,1412,0;5819,2953,0;7419,1326,0;", -- [19]
            ";3770,25,100;120,1,0;10620,250,0;13820,12042,0;9470,7363,0;8320,22092,0;6720,3218,0;10370,6415,0;15620,5564,0;1520,71,0;14470,3000,0;10120,8797,0;13320,0,5000;15370,5439,0;6470,20,0;720,1070,0;920,2821,0;9870,3602,0;13070,13322,0;16670,19459,0;18720,16515,0;11920,40732,0;15120,786,0;10770,7835,0;2820,4662,0;20520,2000,0;5770,1807,0;7370,1012,0;1970,120,0;6220,4893,0;15770,7500,0;4070,7527,0;11420,1712,0;14620,8446,0;6670,1672,0;19770,0,45000;15520,879,0;2120,7,0;14370,472,0;19270,12500,0;4370,175,0;5970,408,0;2020,1049,0;15270,22775,0;16720,21490,0;14120,577,0;4820,0,7488;6420,3611,0;9770,416,0;3370,10,0;11820,22301,0;770,316,0;13870,5143,0;4120,4192,0;14770,3172,0;4570,922,0;17770,3605,0;1270,213,0;12470,8784,0;10170,10105,0;15420,100,0;5470,28,0;7070,400,0;17020,250,900;14270,4877,0;13120,9505,0;4320,896,0;7520,5747,0;2320,2,9;11970,1710,0;10820,138,0;6370,10,0;1720,24771,0;14920,8799,0;2770,5,0;12620,15905,0;820,947,0;7720,5356,0;3220,40,0;15570,2431,0;4020,11189,0;3020,2655,0;14420,2889,0;6570,363,0;8170,500,2000;1370,2,0;1770,143,0;3820,100,0;10070,15482,0;17070,109307,0;19120,16283,0;2420,1931,9550;9920,3721,0;9420,3820,0;2070,1,4;2870,2935,0;14170,259,0;3670,70,0;3970,4296,0;18420,71236,0;9820,1541,0;13020,8985,0;16220,4000,0;7470,3668,0;10720,750,0;4720,1234,0;6320,2711,0;7920,8053,0;18370,21612,0;18520,68652,0;3270,3,0;15970,587,0;13170,19230,0;2570,11,0;1420,18,0;1820,963,0;3070,11,0;11370,500,0;14570,1723,0;15470,2432,0;15170,13103,0;3870,750,0;10270,22396,0;10220,23263,0;4470,9,38;6070,6,0;870,19428,0;2970,333,0;2520,0,19702;8270,14351,0;20070,0,442362;11120,30144,0;2220,519,0;14320,6855,0;4920,7,0;6520,41,0;8120,6207,0;9970,5730,0;14670,20396,0;2620,3013,0;12020,3969,0;15220,43493,0;5120,193,0;6970,906,0;3470,5,0;3170,47,0;16170,0,100;10320,1750,0;2470,0,33952;4970,54,0;5820,1622,0;7420,2658,0;", -- [20]
            ";3771,50,200;18421,25578,0;7421,1602,0;14971,11076,0;13821,17993,0;4671,32,0;5121,162,0;10371,11149,0;15621,15766,0;2721,1982,0;5571,250,0;14471,3000,0;10121,10515,0;15371,3467,0;16921,43440,0;1121,1075,0;1521,19205,0;3171,6,0;3971,3811,0;9871,5425,0;13071,4061,0;16671,12586,0;11921,54167,0;10771,2506,0;5771,50,0;7371,500,1600;14621,12717,0;1171,55,0;19771,0,45000;4071,2939,0;5071,443,0;6671,2068,0;14371,805,0;19271,12500,0;10021,7946,0;16971,300,0;4371,200,800;5971,1136,0;15271,41497,0;16721,30190,0;14121,579,0;4821,1210,0;6421,2317,0;16171,0,800;18521,17606,0;11821,22384,0;20571,1,0;3371,1,3;13871,13436,0;4121,1380,0;14771,3030,0;20071,0,41230;10421,32,0;3021,2421,0;3821,150,0;6171,1,0;15671,9782,0;10171,6018,0;13371,6657,0;15421,17855,0;17021,175,560;5471,30,0;7071,100,0;14271,6566,0;9921,5537,0;13121,3428,0;7521,3560,0;11971,3969,0;15171,8697,0;2321,25,100;6371,12,0;7971,1000,0;14921,6407,0;12621,44729,0;1721,35815,0;8271,20358,0;14671,13012,0;19871,50287,0;13521,10000,0;7721,17618,0;3671,201,0;2821,1117,0;14421,3945,0;4021,7198,0;2571,16,0;6571,1166,0;8171,500,0;10071,8063,0;2771,12,0;10321,0,3600;2271,3255,0;19121,8585,0;16821,31268,0;5421,650,0;14171,738,0;18871,0,265576;9821,871,0;13021,16694,0;821,249,0;16221,4000,16000;7471,2455,0;771,38,0;11871,10666,0;3471,142,0;15071,11443,0;10721,3317,0;2421,0,6362;18371,20737,0;4721,2749,0;6321,1650,0;7921,5387,0;15971,1064,0;14821,5290,0;2521,0,24717;871,29627,0;4421,100,0;4571,979,0;19221,12,0;14221,1510,0;14571,496,0;3871,850,0;10221,9148,0;1421,28,0;1821,988,0;2971,108,0;4471,33,135;15471,2034,0;18321,49320,0;11121,2941,0;14321,9438,0;9521,14382,0;6521,47,0;8121,4843,0;9971,3995,0;13171,7000,0;18821,64030,0;12021,9327,0;3421,50,0;6971,1924,0;2471,0,17040;", -- [21]
            ";2622,2460,0;7422,1622,0;13822,11442,0;20222,0,320;6272,0,300;3072,1207,0;10372,11750,0;5122,287,0;6722,1241,0;12422,16204,0;15622,13809,0;17422,2,0;14472,3000,12000;5572,250,0;15372,3383,0;4422,112,0;16922,58135,0;11022,250,0;16672,14059,0;9872,4101,0;13072,10426,0;4872,95,0;18722,10688,0;3172,18,0;15122,659,0;10772,12170,0;1522,10224,0;16022,40000,0;18222,3070,0;5772,125,500;7372,500,0;2822,1402,0;15772,7500,0;14622,8510,0;10272,20865,0;4072,1340,0;17222,300,0;19272,12500,0;14372,1430,0;1172,968,0;19022,28580,0;15272,51740,0;5972,375,0;16722,8105,0;3722,213,0;9772,1328,0;6422,3768,0;18522,36253,0;922,0,12038;20572,1,0;11822,11794,0;18022,10709,0;20072,10307,32984;14772,3284,0;17772,7132,0;3022,941,0;14522,31318,0;10172,9603,0;13372,63769,0;6622,55400,0;17022,0,135000;14272,3820,0;5472,10,0;7072,150,0;9922,7991,0;2672,4,0;3472,71,0;11972,4649,0;15172,4706,0;4322,1810,0;7522,4962,0;10822,2500,0;14022,8991,0;16072,2500,0;6372,35,0;7972,400,0;18322,16137,0;11722,16610,0;14922,8273,0;14672,7309,0;2772,150,0;7722,9295,0;1322,68,0;3972,3891,0;1722,19407,0;2472,0,34211;16622,27307,0;17072,80794,0;10072,6177,0;6572,1323,0;8172,500,0;15222,1219,0;9622,5665,0;15322,7745,0;2422,1380,6385;4022,12059,0;16822,42603,0;8272,13697,0;18872,30394,0;14172,837,0;422,25,90;13022,25785,0;9822,1192,0;16222,5000,0;3572,972,0;2122,1,0;11872,8566,0;3872,800,0;7472,3423,0;14122,219,0;18372,54895,0;4722,2118,0;18822,125208,0;1372,2,0;1772,226,0;2572,73,0;10122,9007,0;13122,11489,0;15422,500,0;12522,182,0;3322,11,0;13422,10,0;14572,609,0;5622,556,0;10222,14187,0;2522,0,18038;2072,4414,0;3372,10,40;14222,1821,0;2172,7,0;4822,1349,6071;11122,7162,0;2972,60,0;19772,0,45000;9972,2475,0;6522,4,0;8122,7875,0;1422,6,0;1822,1096,0;12022,4718,0;9172,500,0;11922,43491,0;6972,2792,0;", -- [22]
            ";8173,1000,0;20723,46934,0;6973,2465,0;1423,18,0;1823,725,0;3423,0,16000;7423,3941,0;10623,24748,0;13823,11924,0;9473,11535,0;14723,290,0;3073,1211,0;6723,8155,0;15623,10715,0;5573,875,0;10123,15239,0;15373,6930,0;2723,12,45;14223,2698,0;16673,13440,0;9873,5114,0;13073,9080,0;18723,41953,0;4873,15,0;6473,956,0;11923,41263,0;15123,2463,0;3973,3657,0;5323,1632,0;9623,4182,0;5773,250,0;7373,2897,0;873,21770,0;17723,750,0;2823,1618,0;19773,0,45000;6223,4482,0;14623,17085,0;10273,29323,0;13473,7164,0;15523,1474,0;17223,88049,0;19273,12500,0;14373,524,0;4073,2230,0;5523,15,0;10023,3284,0;15273,56429,0;16723,8625,0;4373,722,0;5973,162,585;14123,287,0;9773,1369,0;18523,34810,0;1173,47,0;1973,4618,0;2123,8,0;3723,68,0;20573,1,0;6423,7233,0;11823,19624,0;14773,3052,0;3373,14,0;4123,3804,0;17773,7132,0;15673,11075,0;17523,10758,0;6173,1,0;10173,5718,0;723,15,0;923,1731,7869;3023,734,3017;12223,4,0;15423,500,0;17023,0,144000;14273,6033,0;7073,6,0;13123,33387,0;18823,25776,0;11973,3971,0;15173,5387,0;2673,10,0;3473,299,0;4323,1889,0;7523,4993,0;10823,17796,0;14023,8991,0;18323,32988,0;6373,35,0;7973,46,0;14923,4604,0;8273,3574,0;14673,8303,0;2773,39,0;6123,1,0;7723,22267,0;11223,1450,5800;14423,3406,0;1273,1656,0;5123,117,0;17073,113631,0;3823,100,0;10073,9366,0;19123,4114,0;6573,872,0;8923,50,200;15323,17791,0;16823,32719,0;14173,281,0;2423,0,42770;3223,306,0;4023,6215,0;9823,925,0;13023,42378,0;16223,5000,0;13373,14846,0;20673,10199,0;11873,9661,0;7473,3710,0;19323,312980,0;15973,2313,0;3173,15,0;2073,742,0;3673,45,0;11623,8333,0;5173,0,100;6773,7366,0;14973,27810,0;14573,1289,0;3873,1100,0;2523,5657,22628;10223,9466,0;13423,125,0;20073,0,121395;15473,65,0;7673,8990,0;11123,15037,0;14323,8322,0;9923,6095,0;16873,2254,0;9973,3376,0;13173,11,0;2973,484,0;10373,17336,0;8123,11275,0;12023,4971,0;15223,2376,0;16623,7783,0;10323,1125,3600;", -- [23]
            ";8124,7932,0;18674,22003,0;20724,58873,0;2974,392,0;18424,20595,0;5374,87,0;6974,1497,0;20224,0,240;13824,9015,0;7424,2544,0;19974,0,10000;14724,253,0;6274,100,0;1824,1104,0;10374,11837,0;12424,5288,0;15624,4971,0;3074,476,0;10124,20396,0;5574,875,0;15374,7363,0;14224,3901,0;4424,100,0;16674,34435,0;9874,5801,0;11924,19650,0;4874,46,0;6474,137,0;10774,4749,0;18224,1728,0;3174,16,0;3974,5486,0;13724,300,1080;5774,275,0;7374,3760,0;19774,5000,0;14624,20578,0;4624,550,0;10274,20078,0;2824,32031,0;13474,24724,0;15524,2876,0;10024,5421,0;5524,21,0;16724,8519,0;3274,6,0;15274,31666,0;4074,4435,0;14124,292,0;4374,200,0;5974,350,0;9774,1374,0;12974,5419,0;4824,0,15170;6424,3825,0;2924,16,0;5274,1003,0;17774,6133,0;10424,750,0;4124,3991,0;3374,45,0;15674,18464,0;10174,8121,0;13374,13559,0;2224,21,0;3024,1419,6388;14274,10448,0;9924,10276,0;13124,3996,0;7074,4,0;1624,6172,0;2024,1215,4862;11974,4971,0;10824,5930,0;14024,52422,0;4324,1874,0;7524,4297,0;18324,55189,0;14924,14031,0;7974,50,0;724,25,0;10574,3535,0;9424,13594,0;15824,15061,0;8274,10251,0;14674,8270,0;19274,12500,0;9624,5267,0;13524,12458,0;1274,8,0;2324,6,22;11224,1450,0;9224,700,0;1074,491,0;6124,1,0;7724,5436,0;10074,13985,0;2774,28,0;4674,32,0;8924,0,100;15324,25565,0;6574,568,0;18874,0,268566;14174,368,0;15574,5150,0;9824,1599,0;13024,4863,0;16224,5000,20000;20674,21153,0;3724,81,0;2424,0,21461;4024,9821,0;18524,35596,0;18374,21418,0;19324,249410,0;20424,175,0;10724,4696,0;2674,12,0;2124,6,0;7474,3281,0;12774,30531,0;15974,1461,0;11624,7566,0;4724,1763,0;6324,1822,0;14774,6252,0;15174,6017,0;3824,150,0;14974,6009,0;3674,95,0;2074,1054,0;3874,1100,0;6774,2885,0;774,15,0;14574,1463,0;9474,13246,0;10224,10755,0;13074,8024,0;3474,216,0;2524,0,15354;3324,958,0;15474,23,0;11874,12124,0;11124,16980,0;14324,16706,0;4474,4197,0;14374,699,0;1374,3,0;1774,281,0;9974,16449,0;924,0,10972;14474,3000,0;12024,5396,0;15224,2695,0;", -- [24]
            ";12025,5282,0;15225,5958,0;25,3,0;8125,6367,0;1775,212,0;3775,13,0;6975,16766,0;20225,0,400;10625,27366,0;13825,10168,0;9475,33277,0;14725,292,0;6275,200,800;10375,7116,0;12425,6044,0;15625,5818,0;10125,14551,0;1425,37,0;1825,1548,0;3075,15073,0;5575,2500,0;15375,4856,0;14225,4694,0;16675,21421,0;4425,125,0;18725,59927,0;9875,2139,0;13075,19416,0;11925,16116,0;18475,7907,0;2725,375,0;10775,5331,0;9625,6344,0;1475,82,0;5775,350,0;7375,3773,0;8225,9099,0;10275,15385,0;13475,8109,0;15525,1634,0;19275,12500,0;5075,25,0;14375,528,0;16975,813,0;16725,11248,0;5525,20,0;15275,35714,0;1925,787,0;3275,1,0;4375,115,0;18525,13044,0;9775,315,0;12975,2205,0;4825,3942,0;6425,4478,0;13875,1,0;15925,439,0;5275,175,0;14775,8367,0;17775,7783,0;11325,150,600;14525,10561,0;1175,28,0;19325,188888,0;10175,9010,0;13375,30656,0;17025,0,144000;12225,187,0;15425,8272,0;14275,11115,0;9925,3439,0;13125,12954,0;2225,183,0;7075,400,0;11975,4739,0;15175,9692,0;14025,188,0;4325,2272,0;18325,13651,0;7525,3106,0;14925,10562,0;1625,14723,0;2025,970,5304;6375,250,0;7975,1500,0;15825,19302,0;11475,7,0;14675,7839,0;4075,1984,0;2825,13111,0;3175,100,0;10325,2500,0;15575,2300,0;11225,1550,0;14425,6223,0;2525,0,21191;19125,6356,0;6125,1,0;10075,9369,0;16825,21977,0;4775,28,0;925,0,7797;8925,125,500;15325,31784,0;2775,75,0;3575,200,0;6575,431,0;8175,5971,0;1275,1668,0;14175,737,0;2575,25,0;9825,2355,0;13025,10464,0;11875,5776,0;18375,14329,0;3825,110,0;10725,1500,0;3975,3896,0;2425,0,43077;4025,5070,0;4675,35,0;7475,2269,0;15975,2421,0;2125,6,0;3025,0,18478;11625,10452,0;14825,8595,0;4725,4736,0;2975,128,0;6325,0,40;3875,1250,0;4575,1487,0;3375,30,0;2675,11,0;1725,5000,0;2075,386,0;2175,2683,0;10225,10012,0;2325,250,1000;18875,31672,0;14325,11893,0;13175,30619,0;16375,625,0;4625,250,0;8275,22028,0;", -- [25]
            ";5976,2500,0;12026,7757,0;8126,15656,0;12926,14907,0;6976,17264,0;10626,36732,0;1376,4,0;1776,257,0;14726,337,0;19726,500,0;10376,12519,0;12426,12062,0;3426,1000,0;15626,8798,0;6726,2388,0;15376,8653,0;5576,2500,0;2276,4093,0;9876,3146,0;13076,4715,0;1826,1765,0;20526,500,0;10776,4799,0;19776,0,45000;15776,10000,0;8226,5319,0;3176,75,0;3976,6978,0;10276,9847,0;13476,3000,0;6226,780,0;1076,650,0;1476,6,0;19276,12500,0;15526,704,0;14376,551,0;10026,5459,0;16726,22270,0;15276,45263,0;826,732,0;18526,15255,0;9776,547,0;12976,1714,0;8626,250,0;11826,2500,0;4076,3301,0;13876,25,0;4826,0,13892;6426,4162,0;1926,687,0;15926,802,0;17776,7377,0;14776,6326,0;14526,5000,0;19326,0,19800;10176,5689,0;13376,13738,0;6176,15,0;17026,250,900;15426,13419,0;14276,3598,0;18826,31917,143628;11976,7778,0;15176,12121,0;7076,400,0;2226,4117,0;10826,25501,0;3026,726,3431;18326,9036,0;4326,2496,0;7526,3367,0;10576,6000,0;7976,2000,0;12626,9191,0;8276,3901,0;14676,13841,0;6826,548,0;2026,1212,6286;13526,10897,0;10126,8865,0;4676,254,0;15576,4462,0;7276,34,0;3826,105,0;2326,29,0;14426,2125,0;19126,4041,0;10076,6571,0;6126,1,0;7726,10460,0;16826,22057,0;18876,0,270533;15326,1,4;2126,11,0;2576,75,0;14176,872,0;9626,16243,0;6576,572,0;16676,14268,0;14226,1653,0;8176,4495,0;2776,500,0;9826,2600,0;13026,19441,0;3576,35,0;18376,52306,0;11876,15772,0;7026,22,0;3376,86,0;1276,2220,0;4776,41,0;11226,1550,0;10726,5520,0;13926,10000,0;11926,24590,0;13126,10374,0;9926,5768,0;7476,1979,0;2976,171,0;3776,175,0;14626,20732,0;2426,0,32568;11626,8081,0;14826,2311,0;4026,4362,0;15726,3000,0;19026,250,0;18726,11296,0;4726,2340,0;6326,0,32;7926,5952,0;926,1956,9784;4576,1184,0;4426,125,0;3726,125,0;14926,5598,0;2526,3867,15468;15226,8755,0;10226,15908,0;3676,106,0;9476,6765,0;15476,44,0;4126,6396,0;10326,0,4000;1326,10,0;1726,12157,0;16376,2000,0;6076,7,0;7676,30,0;", -- [26]
            ";3827,120,0;13177,16396,0;4477,4390,0;12027,6257,0;15227,21925,0;127,1,0;727,244,0;927,1390,6953;18427,0,5289;8127,5937,0;12927,19149,0;14977,7982,0;20227,0,300;6977,16030,0;10627,26974,0;14727,778,0;19727,15000,0;10377,7978,0;12427,12106,0;15627,14493,0;1377,1,0;1777,222,0;3427,1500,0;6727,2996,0;15377,3191,0;16677,24868,0;11027,2500,0;14227,2500,0;18727,13692,0;9877,2708,0;13077,7656,0;2277,3620,0;11927,16385,0;15127,1647,0;10777,4014,0;9627,2542,0;12827,5000,0;11677,9712,0;1427,27,0;1827,1282,0;19777,0,45000;7377,2269,0;10277,6985,0;13477,3000,12000;16977,1430,0;14377,602,0;19027,1250,0;6677,2401,0;16727,15298,0;1077,25,0;1477,87,0;5527,95,0;14127,999,0;18527,17988,0;9777,318,0;12977,355,0;4377,150,0;11827,675,0;13877,30,0;3277,30,0;4077,3298,0;6427,7016,0;9527,21613,0;15927,1864,0;17777,10815,0;1927,642,0;3727,125,0;19327,0,36000;10177,12833,0;13377,7,0;6177,69,0;15427,8536,0;14277,10131,0;18827,0,225217;9927,3927,0;13127,2808,0;11977,7896,0;15177,8525,0;7077,400,0;18327,8846,0;1177,3,0;2227,4133,0;3027,1207,5713;7527,11284,0;14927,4831,0;10577,2000,0;19877,43354,0;6377,250,1000;15827,22592,0;777,21,0;8277,3625,0;14677,18524,0;13527,13149,0;4327,3788,0;10127,4350,0;15577,2317,0;2877,8524,0;2077,5059,0;7277,23,0;19127,12258,0;4577,356,0;10077,16968,0;4827,749,3748;16827,22139,0;2577,75,0;2027,555,3051;2327,36,0;18877,60339,271529;3927,150,400;10027,5479,0;7727,2488,0;8927,100,0;15327,1,4;2527,0,26420;10227,22128,0;14177,631,0;827,923,0;4977,11893,0;6577,649,0;8177,71,0;9827,1114,0;13027,33551,0;3377,87,0;2977,260,0;2677,7,0;18377,13000,0;3177,50,0;15077,7615,0;3577,600,0;4677,70,0;10727,2000,0;13927,8,0;12777,38648,0;2127,12,0;3777,0,36;7477,7044,0;14827,2320,0;4127,8073,0;1127,25,0;2427,0,20146;3227,2480,0;4727,2791,0;6327,12135,0;7927,2987,0;12527,35332,0;15727,3000,0;2777,146,0;14577,21514,0;15477,162,0;9477,34123,0;5627,1379,0;14327,7659,0;", -- [27]
            ";2078,1070,0;2878,5143,0;13178,13782,0;3678,100,0;18678,26635,0;12028,4007,0;15228,28308,0;4478,10176,0;6078,15,0;6528,358,0;8128,5526,0;2528,0,51836;10628,27620,0;6978,672,0;7428,250,0;10378,19466,0;3778,1541,0;12428,8646,0;15628,9570,0;728,50,0;928,1972,9860;10128,11907,0;3428,100,400;15378,4155,0;16678,31694,0;18728,32466,0;14228,3559,0;1378,1,0;1778,337,0;9878,3424,0;18478,19768,0;11928,22045,0;12828,5500,0;4878,56,0;11678,15596,0;17728,12323,0;19778,0,45000;2728,375,0;15778,1250,0;5778,29,0;7378,2146,0;1828,1609,0;13478,3250,13000;19278,12500,0;16978,1158,0;10028,5793,0;6678,677,0;778,44,0;15278,50109,0;18528,20465,0;5528,0,720;12978,534,0;1478,62,0;15928,2387,0;4828,600,0;6428,2795,0;8028,2500,0;17778,4506,0;14778,4933,0;19328,0,19800;14528,28484,0;17028,175,630;10178,14475,0;2928,0,18;13378,21972,0;3728,300,0;12228,1250,5000;15428,8024,0;1928,917,0;18828,50225,226014;14278,6725,0;9928,9930,0;13128,10502,0;5028,5537,0;11978,7414,0;15178,4935,0;18328,14116,0;10828,40379,0;5478,70,0;7078,400,0;14928,12673,0;4328,1374,0;7528,9894,0;10578,3377,0;10278,6239,0;12628,16212,0;1178,7,0;6378,292,0;14678,13156,0;14478,3500,0;2578,224,0;6828,6268,0;18428,0,27000;2978,415,0;3478,10,0;15578,6962,0;8278,5477,0;17078,23981,0;10328,6482,0;14428,4717,0;2278,2247,0;10078,17515,0;16828,22221,0;18878,84625,0;10228,15940,0;8928,125,0;7728,3223,0;14378,997,0;14178,1532,0;3928,250,900;9828,1845,0;13028,52605,0;4678,108,0;6578,1302,0;8178,262,0;18378,31384,0;4428,331,0;2028,0,5065;15078,9649,0;4128,14587,0;9428,2096,0;10728,375,0;13928,8,0;5428,322,0;4778,1470,0;9478,28853,0;3578,287,0;4278,25,0;14828,3772,0;7478,6991,0;878,56,0;16728,20354,0;2778,133,0;12528,37833,0;15728,3000,0;6328,100,400;7928,4296,0;4078,4672,0;13878,32,0;2428,0,20222;3228,1098,0;14578,983,0;3828,150,0;3378,132,0;828,250,0;1678,4337,0;2678,0,1;5128,202,0;11128,500,0;7228,0,100;", -- [28]
            ";1179,6,22;5629,347,0;7229,285,0;18679,25136,0;79,10,0;12029,5395,0;15229,31924,0;2079,933,0;2879,3121,0;3679,100,0;4479,178,0;12929,19646,0;6529,0,45;8129,6253,0;10629,7541,0;6979,675,0;5829,804,0;7429,4620,0;10379,12185,0;12429,15193,0;15629,6374,0;1729,500,0;14479,3500,0;16929,41521,0;6729,7187,0;15379,7291,0;16679,22084,0;18729,38302,0;14229,2387,0;5579,32,0;9879,1964,0;13079,3359,0;11929,14315,0;10779,8807,0;929,75,300;4879,7,0;19779,0,45000;15779,15000,0;14629,8965,0;10279,11188,0;13479,3500,0;14379,1615,0;16729,14592,0;10029,5391,0;1429,7,0;1829,1563,0;3179,125,0;5079,4642,0;6679,4852,0;15279,17877,0;18529,9963,0;5529,125,0;12979,617,0;4379,2357,0;15929,4396,0;17779,5547,0;6429,4544,0;14779,13347,0;779,18,0;1479,47,0;4079,6753,0;9279,45,0;19329,0,36000;14529,500,0;10179,8632,0;13379,10734,0;15429,9049,0;2129,0,77;3729,300,0;18829,48836,0;6179,556,0;14279,4206,0;9929,3662,0;13129,7621,0;5029,5282,0;11979,7471,0;15179,18593,0;10829,10680,0;1529,700,0;1929,394,0;5479,125,0;7079,400,0;14929,8999,0;7529,6897,0;3779,2835,0;10129,4259,0;3829,150,0;6379,224,0;8279,5937,0;13529,32303,0;2479,0,96;6829,18714,0;9179,1000,0;15579,2122,0;10229,10331,0;11229,1696,0;14429,2644,0;1979,23273,0;2679,5,0;16829,34339,0;8529,175,0;7279,25,0;10079,6370,0;18879,53364,0;2379,9,74;3379,137,0;10329,2507,0;4829,830,0;6129,1,0;7729,5514,0;2579,37,0;14179,747,0;20679,500,0;9829,988,0;13029,5613,0;18379,23791,0;9479,10025,0;4979,2587,0;6579,667,0;8179,28,0;15079,21760,0;129,1,0;18429,0,23557;15979,7215,0;8429,31,0;14829,5049,0;2029,0,4418;7479,4873,0;17029,0,900;9779,334,0;15729,3000,12000;729,17,0;3279,98,0;4729,2991,0;14579,1791,0;2429,0,9124;2529,0,65031;", -- [29]
            ";11130,500,0;80,7,0;18680,37894,0;16380,1750,0;5630,293,0;7230,2878,0;12030,7143,0;18430,0,11512;4480,185,0;12930,10000,0;6530,25,100;8130,9458,0;1280,3696,0;10630,8664,0;14730,1223,0;10380,8468,0;7430,4194,0;15630,9868,0;19230,12500,0;10130,4706,0;2980,209,0;16680,13788,0;15380,4390,0;18730,9769,0;1730,48,0;14230,5164,0;18480,11860,0;9880,2130,0;5580,25,0;11930,14588,0;3430,11026,0;15130,3252,0;10780,5542,0;12830,5500,0;8080,5987,0;17730,40857,0;2280,3863,0;19780,0,45000;1380,4,0;1780,489,0;14630,250,1000;10280,15723,0;13480,3750,15000;5780,156,0;3530,28,0;16730,22523,0;730,16,0;10030,6007,0;15280,20603,0;18530,31331,0;3180,168,0;14130,12654,0;9780,344,0;5530,125,0;1430,7,0;1830,1783,0;4380,500,0;15930,9146,0;14780,14289,0;6430,9098,0;19330,0,54000;3280,33,0;14530,1000,0;17030,500,1800;10180,4907,0;13380,38443,0;1080,78,0;1480,954,0;9030,200,0;18830,0,257523;14280,5919,0;4580,787,0;6180,423,0;9930,4287,0;13130,25065,0;15180,6557,0;6630,2067,0;10830,750,0;7080,400,0;11730,12762,0;2380,7,37;4330,250,0;7530,4273,0;1930,408,0;9680,14054,0;2530,0,41775;2580,67,0;8280,8503,0;6380,654,0;3030,0,1;3830,500,0;2480,0,58;16980,20652,0;10330,9587,0;6830,22451,0;16830,22302,0;4430,4307,0;10080,7319,0;2680,10,0;3480,522,0;15330,1059,0;2780,374,0;4830,1894,0;7730,8593,0;1180,37,120;1980,6200,0;18380,20967,0;9830,3237,0;13030,13113,0;2130,0,49;4130,2171,0;15080,18230,0;6580,1300,0;8180,240,0;2880,25,90;2080,6590,0;3680,400,0;13930,5,0;8430,46,0;11630,5,0;14830,3801,0;9630,5780,0;15230,2755,0;3730,45,0;3780,1877,0;14430,3686,0;7480,3019,0;3380,365,0;14930,22950,0;14330,8962,0;14580,818,0;7930,7045,0;1630,66,0;2030,1108,5544;10230,25686,0;2930,0,50;880,2692,0;4680,91,0;5180,2777,0;2730,375,0;", -- [30]
            ";14331,11654,0;18681,15215,0;2431,0,9193;5631,30,0;12031,7894,0;15231,5033,0;4481,176,0;6531,417,0;8131,8321,0;10631,6086,0;10381,8094,0;2531,0,44935;4231,110,0;7431,3899,0;15631,16672,0;19231,12500,0;13331,0,640000;16681,13055,0;15381,12593,0;18731,500,0;2981,573,0;6731,871,0;14231,1904,0;9881,3464,0;13081,6529,0;5581,32,0;11931,57997,0;15131,1798,0;10781,12822,0;1731,92,0;3431,1317,0;8081,2834,0;2281,300,0;14631,13542,0;10281,11272,0;5781,576,0;16731,14870,0;10031,5906,0;15281,26032,0;18531,65383,0;14131,242,0;2381,0,75;3181,23,0;4381,540,0;15931,10646,0;731,27,0;1431,20,0;1831,1790,0;6431,5436,0;15681,11406,0;14531,59891,0;17031,250,900;10181,14640,0;13381,15270,0;7331,4330,0;12231,750,3000;15431,18518,0;18831,0,258520;11081,200,0;14281,7727,0;13131,2312,0;11981,496,0;15181,11053,0;1081,50,0;1481,3337,0;3731,55,0;6631,4053,0;20131,12500,0;7081,400,0;10131,7085,0;19781,0,45000;2581,20,0;4331,526,0;3781,3125,0;7531,6434,0;12631,8624,0;3581,1046,0;14681,17108,0;6381,625,0;3381,290,0;5181,1778,0;10331,2728,0;9531,5264,0;781,110,0;15581,2794,0;1131,1136,0;2131,10,49;6831,17222,0;14431,2877,0;2231,4386,0;16831,19950,0;8281,5982,0;9781,982,0;9631,3868,0;3831,550,0;7281,84,0;14781,4617,0;15331,555,0;3531,57,0;14181,775,0;3481,1200,0;7731,3482,0;2681,6,0;9831,1233,0;4581,862,0;13031,1401,0;4781,547,2735;8831,300,0;3231,1175,0;6581,422,0;8181,73,0;13931,12,0;3681,400,0;15981,7215,0;1181,25,100;1981,12830,0;3931,185,0;14831,4120,0;18381,21635,0;9381,8628,0;19281,12500,0;4131,3740,0;7481,4908,0;10081,7345,0;12531,28883,0;15731,3500,0;9931,4303,0;14581,2912,0;4831,1291,0;2781,335,0;4731,3440,0;6331,9378,0;7931,7955,0;10231,9815,0;9681,50,0;", -- [31]
            ";16382,3250,0;7932,8463,0;12282,6,0;15482,54,0;5182,1769,0;18682,25419,0;12032,5396,0;15232,5771,0;2432,0,6921;20232,0,240;2032,1665,0;11782,12720,0;4932,179,0;6532,0,250;8132,14485,0;17682,8750,0;2882,300,0;12682,3000,0;832,203,0;10382,18808,0;19232,12500,0;15632,4997,0;7432,2717,0;10132,7608,0;13332,0,640000;6282,2243,0;16682,14582,0;1282,2825,0;14232,2437,0;6732,2421,0;9882,6209,0;13082,13191,0;18232,10000,0;11932,58322,0;15132,1017,0;10782,4538,0;13982,69826,0;17732,10383,0;2632,581,0;1732,73,0;9382,3442,0;12582,51199,0;14632,18121,0;2282,139,0;10282,6712,0;19032,2500,0;5782,3211,0;16732,20725,0;2732,375,0;12082,13408,0;15282,30628,0;6682,1900,0;18282,66347,0;1382,24,0;12982,850,0;11832,10000,0;13882,60,0;15932,527,0;4382,600,2400;14782,4888,0;6432,3741,0;17032,500,1800;1832,362,0;10182,18370,0;13382,10850,0;18832,100336,0;12232,1250,5000;15432,14104,0;7332,6692,0;14282,4552,0;4082,11699,0;13132,7340,0;732,6,0;11982,1062,0;15182,5996,0;18082,28161,0;3732,801,0;14932,11017,0;5482,10,0;7082,400,0;10232,11732,0;1082,150,0;10582,3297,0;19282,12500,0;19182,25,0;2132,20,92;4332,500,0;7532,6975,0;4582,745,0;11382,750,0;3382,10,0;9932,6995,0;3282,235,0;13532,12762,0;4782,415,0;6382,523,0;15582,6544,0;2582,216,0;9682,3831,0;14432,4679,0;4632,50,0;18482,29652,0;3882,13,0;2232,812,0;10082,10752,0;8932,200,800;9632,3883,0;3682,400,0;3182,387,0;18632,0,200;7282,599,0;2982,565,0;18532,21629,0;14182,1167,0;3832,550,0;18382,15971,0;9832,1248,0;13032,4138,0;15382,5043,0;9782,1005,0;2682,25,0;11882,20520,0;3482,1317,0;4232,125,0;13932,12,0;6582,731,0;8182,40,0;12782,56295,0;15982,8142,0;13482,0,13500;14832,2363,0;2532,0,42086;2782,751,0;10332,3790,0;3782,3451,0;15732,3500,0;7482,4926,0;8282,15695,0;932,637,0;1182,44,0;1982,29513,0;4732,2632,0;6332,1153,0;", -- [32]
            ";8483,171,0;10233,11106,0;6333,1885,0;7933,5769,0;783,50,0;983,21,0;18683,51100,0;1183,29,0;2783,590,0;5183,1575,0;15233,10476,0;5633,150,0;11783,12701,0;10633,11684,0;17683,14750,0;6533,62,200;8133,10517,0;12683,3000,0;2033,967,0;2883,375,0;3683,400,0;10383,12521,0;15633,16210,0;7433,1684,0;10133,9429,0;13333,0,640000;16683,8765,0;4683,92,0;15383,5313,0;14233,4476,0;18483,37362,0;5133,300,0;13083,34694,0;833,28000,0;3783,3048,0;10783,10758,0;9633,5847,0;12833,20000,0;2633,25,0;8083,2261,0;12583,70102,0;15783,42148,0;14633,13012,0;13483,0,15000;1733,88,0;2283,140,0;16733,15207,0;10033,6013,0;4633,70,0;12083,7348,0;15283,37365,0;14133,950,0;12983,2204,0;15933,620,0;1783,247,0;4383,3183,0;14783,5424,0;4833,1731,0;17033,500,1600;10183,10954,0;13383,26864,0;12233,750,3000;15433,21849,0;14283,11646,0;1433,14,0;4083,4328,0;11983,1130,0;15183,6751,0;10833,8597,0;18083,4268,0;6633,2035,0;16083,2500,10000;9883,5026,0;11933,19646,0;14933,7884,0;9683,29952,0;3283,295,0;9433,4133,0;12633,14859,0;2383,11,57;7533,5557,0;8283,17138,0;14683,11932,0;4583,812,0;10333,2132,0;733,100,0;9783,811,0;2933,3750,0;6383,1519,0;4233,200,0;15583,5344,0;14433,6764,0;10083,10663,0;13283,14907,0;16583,0,9000;18633,0,22;933,2500,0;2233,1690,0;3033,0,1;3833,24,0;7283,519,0;14183,1710,0;18383,9529,0;2983,280,0;18533,10984,0;9833,3031,0;13033,8094,0;3383,100,0;4983,16837,0;15083,6872,0;1933,905,0;3483,965,0;19933,2080,0;6583,425,0;8183,2426,0;12783,58215,0;15983,8387,0;2583,339,0;11633,11730,0;14833,2988,0;13133,22015,0;2533,0,48886;9933,9394,0;9533,7092,0;4133,1794,0;9333,73,0;15733,3500,0;4333,1200,0;7483,4238,0;14583,1333,0;9233,500,0;", -- [33]
            ";4734,4208,0;7934,5753,0;18684,36141,0;13184,12827,0;6784,1966,0;12034,5012,0;15234,11354,0;5634,75,0;20234,0,320;11784,32597,0;6084,290,0;7684,2184,0;10634,10812,0;12684,3000,0;8134,15993,0;15634,11441,0;18984,5000,0;11284,1,4;14484,4000,0;7434,2536,0;2034,939,0;16684,9093,0;10134,7664,0;13334,0,8000000;18734,17194,0;12184,87,0;4684,61,0;16434,1900,0;3334,49,0;14234,5240,0;9884,5448,0;13084,6614,0;5134,92,0;11934,19921,0;10784,18432,0;17734,10457,0;8484,68,0;9384,9789,0;12584,0,223364;15784,11488,0;8084,2703,0;19284,12500,0;3434,3,0;14634,625,0;13484,0,13500;15534,2308,0;16734,14536,0;5784,75,0;2284,211,0;10034,2000,0;18534,71503,0;15284,3039,0;4634,87,0;1734,39,0;18284,0,1500;14134,11280,0;9784,498,0;12984,1422,0;13884,75,0;15934,2596,0;11584,1,0;4384,1000,0;17034,50,180;19084,15100,0;10184,6523,0;13384,8988,0;18834,3750,12000;14284,13135,0;9934,6577,0;13134,7716,0;11984,2189,0;15184,6882,0;4084,10136,0;10234,17801,0;16084,2500,8000;17184,0,34;1434,43,0;14934,6754,0;10584,2905,0;7084,350,0;2134,0,74;9434,5648,0;2934,7,0;3734,400,0;2734,375,0;8284,10514,0;4334,550,0;7534,3827,0;13534,37727,0;9534,8847,0;4234,150,0;4784,360,0;6384,250,0;16834,34614,0;2584,142,0;3384,20,0;14434,8145,0;3484,1767,0;11634,5671,0;10084,15246,0;4584,937,0;3284,79,0;15334,828,0;4984,5630,0;3184,1364,0;18484,53288,0;2534,0,45430;18384,16093,0;2384,1,37;7284,586,0;19934,1400,0;2234,1806,0;9834,2074,0;3834,32,0;13034,14710,0;11884,1846,0;15084,7311,0;6134,1,0;7734,15495,0;934,10682,0;15134,3295,0;16384,1750,0;2984,195,0;12784,72854,0;15984,9614,0;6584,1416,0;8184,2791,0;9634,3381,0;2684,20,0;14834,2777,0;1934,731,0;15734,3500,0;2434,0,4648;11384,70,0;14584,2481,0;7484,3062,0;", -- [34]
            ";3385,30,120;11385,145,0;7485,3073,0;10235,16896,0;935,1692,0;15485,344,0;4735,3621,0;6335,1581,0;7935,8368,0;13185,16422,0;12035,5513,0;15235,21090,0;20235,0,160;2785,1062,0;5635,45,0;12935,23196,0;6085,243,0;7685,5468,0;2435,0,27683;4035,1988,0;8135,10884,0;10385,13247,0;19235,12500,0;6985,690,0;15635,5896,0;11285,2,4;3685,71,0;16685,8694,0;7435,2749,0;10135,20663,0;18735,17253,0;8985,150,0;15385,11449,0;18485,34325,0;14235,1789,0;785,20,80;13085,9137,0;3335,46,0;5135,142,0;11935,13162,0;10785,13200,0;9385,12280,0;8085,5253,0;14635,750,0;10285,1000,0;1685,6250,0;15535,1270,0;16735,8738,0;5785,500,0;10035,1621,0;18535,30231,0;15285,4061,0;18285,4558,0;12985,1153,0;835,23,0;1735,67,0;2735,375,0;15935,3323,0;8385,875,0;14785,9558,0;4385,1146,0;9285,2388,0;17035,100,360;3185,8016,0;10185,7412,0;13385,13237,0;18835,0,157486;1785,464,0;2835,2,0;9935,9404,0;13135,14001,0;16335,10000,36000;18335,2520,0;11985,2144,0;4585,583,0;6185,9,0;10835,24645,0;8485,1000,0;2385,0,37;285,711,3200;4835,2110,0;7085,350,0;9435,2916,0;4135,1130,0;35,8,0;8285,5979,0;4335,1500,0;885,2264,0;7535,3947,0;16835,42315,0;14285,6905,0;6385,250,0;14435,2778,0;18585,14853,0;10085,10884,0;18635,100,400;2585,638,0;15335,580,0;18385,20688,0;14185,647,0;7285,588,0;9835,2843,0;13035,22273,0;15085,22732,0;3835,1106,0;16485,7596,0;13885,50,0;2985,737,0;15985,9887,0;2535,0,55400;6585,558,0;8185,10853,0;11635,35451,0;14835,8192,0;9885,4179,0;4235,200,0;9785,268,0;3485,1223,0;12535,34925,0;15735,3500,0;3235,412,0;2635,16,0;3785,3650,0;", -- [35]
            ";11386,676,0;3486,100,0;7486,9325,0;15486,245,0;18686,24527,0;6336,305,0;7936,6739,0;8836,95,0;12036,5982,0;15236,27485,0;6786,59,0;1936,1113,0;12936,10777,0;5636,75,0;2786,1173,0;11786,50739,0;3586,1034,0;17686,9937,0;10636,4,0;7686,4308,0;936,11620,0;4936,18,0;6536,488,0;19236,12500,0;10386,8865,0;4036,681,0;15636,6333,0;6986,50,0;18986,5000,0;1986,14861,0;16686,14400,0;10136,8530,0;7436,2111,0;16436,1963,0;2886,5,0;15386,7287,0;14236,2909,0;18236,2716,0;9886,2589,0;11936,284,0;15136,1248,0;10786,11977,0;13986,17469,0;17736,9083,0;12836,20000,0;5586,16,0;2036,238,0;8486,1000,0;9386,9859,0;2986,718,0;15786,18826,0;14636,9983,0;8086,3730,0;16986,850,0;16736,9295,0;15536,4107,0;16486,5695,22782;18536,38664,0;5786,137,0;1686,733,0;18286,2953,0;14136,15310,0;9786,234,0;6686,2627,0;13886,50,0;15936,8950,0;8386,1000,0;14786,11247,0;4386,175,0;19086,15209,0;3186,4436,0;13386,16376,0;9036,20,0;15436,10519,0;18586,14853,0;9936,6403,0;13136,1456,0;2836,7,0;11986,1745,0;15186,15373,0;10836,27564,0;4586,713,0;6186,1372,0;9686,25914,0;7786,5357,0;18486,21683,0;1786,683,0;14286,4620,0;5136,177,0;4086,6386,0;4236,225,0;8286,9427,0;4736,2616,0;4336,1500,0;7536,13048,0;3736,500,0;3836,3053,0;9286,6040,0;10286,400,0;14436,5270,0;15286,5568,0;10236,22615,0;6386,2692,0;17036,200,720;1436,366,0;10086,8438,0;2686,0,50;10036,1591,0;3786,3957,0;15336,1669,0;5236,0,12954;18386,19448,0;286,592,2906;236,551,2795;4786,278,1393;4636,110,0;3386,35,0;3286,57,0;9836,3677,0;13036,40988,0;36,6,0;7286,25,0;9486,5483,0;3936,985,0;15086,13154,0;10186,6966,0;16886,5492,0;19936,830,0;4536,1,4;6136,1,0;17186,0,243;9636,3154,0;2236,3355,0;3036,515,0;3986,5332,0;6586,505,0;8186,2231,0;4136,6021,0;1486,1202,0;4836,2000,0;4686,67,0;", -- [36]
            ";2287,6,25;15737,4000,0;11387,1013,0;14587,2012,0;10237,16287,0;2687,25,0;7487,8665,0;14337,8546,0;16387,5500,0;4737,5743,0;6337,245,0;7937,6763,0;12037,87,0;15237,30526,0;3137,0,1;3937,2018,0;20237,0,300;5187,991,0;14087,11,0;5637,75,0;11787,13732,0;17687,9937,0;1537,62,0;1937,1700,0;2787,10,0;3587,0,14314;6087,1239,0;9487,3632,0;17187,215,1078;6537,198,0;8137,2807,0;10387,8475,0;15637,14270,0;4037,1821,0;11287,94,0;14487,53222,0;16687,19711,0;10137,13484,0;16437,0,58845;4237,34,0;7437,1554,0;15387,5287,0;18487,10000,40000;537,87,0;937,12867,0;18237,1622,0;13087,5896,0;15137,1516,0;5137,217,0;10787,6382,0;17737,10163,0;9637,4749,0;5587,512,0;8487,1000,0;6037,1250,0;9387,3438,0;12587,20480,0;15787,22592,0;14637,25048,0;16987,807,0;2037,469,0;19037,5877,0;16737,9794,0;15537,1163,0;11187,3,0;16487,5716,0;3437,23,0;5787,150,0;7387,2571,0;15287,12052,0;18287,0,200;14137,18113,0;4637,150,0;12987,906,0;787,1,5;1687,243,0;6687,9930,0;15937,7922,0;8387,1000,0;14787,8323,0;17037,350,1120;4387,400,0;9287,2406,0;14537,15912,0;18837,0,158666;13387,15016,0;1737,139,0;3187,14350,0;3987,6837,0;6887,0,400;14287,13282,0;9937,4004,0;13137,5874,0;20387,0,1000;11987,2885,0;15187,10918,0;8287,7087,0;2987,376,0;18737,52427,0;2087,252,0;10837,38801,0;6187,613,0;7787,3960,0;9687,1554,0;18437,0,12349;4987,15584,0;14937,11406,0;4837,0,8002;37,5,0;10587,1500,0;287,0,2196;1187,1081,0;1787,102,0;3287,199,0;4087,8722,0;7087,300,0;14687,23357,0;9887,7071,0;1287,703,0;4537,6,20;4337,750,0;7537,14012,0;12637,10316,0;14437,12002,0;3737,550,0;18637,600,0;4787,577,0;6387,1228,0;10087,3393,0;18537,66135,0;887,82,0;5237,18,0;237,0,2524;14187,717,0;3487,1498,0;9837,1287,0;13037,4729,0;2587,200,0;3387,30,0;7287,100,0;15087,17892,0;837,0,1123;9187,600,0;18587,2000,0;2237,75,0;19937,780,0;4587,807,0;9587,6250,0;10287,964,0;2437,0,27889;10187,11785,0;9787,358,0;3037,4814,0;3837,4405,0;6587,1146,0;", -- [37]
            ";4338,250,0;4638,200,0;15738,4000,0;38,1,0;14588,1937,0;9088,250,0;15488,967,0;7488,6038,0;6338,125,0;7938,4028,0;12038,9092,0;15238,36496,0;6788,3331,0;17688,3155,0;888,840,0;9488,4419,0;17188,0,2038;3588,0,14365;10388,12895,0;4938,236,0;6538,492,0;8138,7666,0;15638,9614,0;16688,21530,0;11288,1535,0;18738,37593,0;10138,12274,0;4038,3562,0;15388,6357,0;7438,1560,0;14238,3691,0;18238,3151,0;13088,7413,0;2888,10,0;11938,213,0;15138,15198,0;10788,5338,0;6738,7040,0;17738,33351,0;1988,2852,0;8488,1000,0;9388,3101,0;4438,1692,0;14638,20954,0;19038,11145,0;8088,0,13335;16738,3893,0;2988,314,0;15538,1412,0;18288,0,1000;15288,33743,0;5788,162,0;10938,0,640;14138,20042,0;9788,312,0;238,354,1901;6238,98,0;6688,2059,0;9538,6463,0;15938,9550,0;1288,185,0;1688,806,0;17038,500,1800;9288,2414,0;4388,600,0;16788,5095,0;2738,375,0;14538,13569,0;13388,20974,0;12238,2,0;14288,14132,0;6888,10,0;18338,36595,0;9938,6029,0;3188,879,0;13138,9376,0;2438,0,20996;11988,7113,0;4138,10140,0;4238,200,0;1738,147,0;10838,34665,0;6188,305,0;2838,60,0;10188,16717,0;2238,26,0;4838,2000,8002;10588,5326,0;4738,3189,0;18838,0,218064;8288,10127,0;7088,1250,5000;2488,107,482;3288,250,0;4088,18523,0;7538,12322,0;12988,1137,0;1788,176,0;18638,12500,0;8838,60,0;10088,3154,0;13288,625,2250;4788,466,2633;6388,2043,0;3488,613,0;2138,38,0;15338,1387,0;14188,982,0;5238,0,32153;14938,6470,0;838,225,1128;3938,1189,0;9838,1174,0;13038,9630,0;9638,6875,0;11888,4879,0;15088,8191,0;14438,6192,0;7288,125,0;4588,900,0;18588,200,0;19938,580,0;3388,40,0;13938,39304,0;4538,25,80;6138,1,0;7738,211,0;1438,70,0;10238,7650,0;14838,2892,0;10288,1410,0;18788,0,8000000;6588,458,0;8188,6414,0;", -- [38]
            ";3389,40,0;8189,7268,0;15739,4000,0;239,282,1272;839,113,566;14589,3210,0;1839,36,166;3039,1610,0;5439,0,90;10239,4962,0;18689,16460,0;15489,363,0;4289,12,45;7489,3740,0;6339,24,0;7939,10899,0;8839,375,0;12039,4224,0;15239,42409,0;14089,30,0;12939,60241,0;12689,3500,0;17189,2408,9634;10389,19770,0;16939,64217,0;3589,109,576;6539,162,0;8139,3053,0;15639,7040,0;16689,13187,0;14489,5000,0;18739,20124,0;10139,12370,0;15389,10146,0;1939,168,675;4039,2215,0;4239,71,0;5839,1,0;7439,5425,0;11039,200,0;18239,0,3500;13089,8039,0;6289,1,0;11939,671,0;15139,3055,0;17739,9100,0;2889,60,0;6739,2991,0;12839,20000,0;16039,52512,0;5589,41,0;7189,4274,0;9389,4217,0;12589,10180,0;15789,9574,0;6039,1250,0;14639,375,0;10289,1557,0;13489,3750,0;16739,7300,0;8089,0,16064;15539,1282,0;11189,5,0;2989,1001,0;18289,9701,0;5789,700,0;14139,15840,0;12989,2854,0;1639,28460,0;2039,750,0;4639,162,0;6239,160,0;11839,9560,0;6689,8267,0;15939,10455,0;8389,1250,0;14789,17541,0;2289,87,350;9289,3635,0;15689,4040,0;16789,5692,0;4389,750,3000;14539,17836,0;18839,0,900;13389,21318,0;12239,1750,7000;14289,5131,0;6889,4,0;13139,21029,0;11989,7471,0;15189,15722,0;4689,106,0;3189,99,0;3989,3154,0;4139,2749,0;5739,2738,0;9889,4240,0;9639,22798,0;9789,741,0;2589,13,0;3889,1574,7129;4589,530,0;6189,624,0;14939,6793,0;10189,11872,0;789,1969,0;1739,255,0;3239,3,0;8489,1500,0;9939,4317,0;12639,13501,0;8289,16578,0;13539,8535,0;3939,2256,0;3439,30,0;15589,5462,0;2489,68,308;4339,1250,0;5939,14,0;7539,12692,0;14439,3662,0;18639,12500,0;3489,917,0;10089,4749,0;18389,16830,0;4789,391,1998;6389,2245,0;7989,2500,0;15339,2466,0;1389,58,0;1789,127,0;2139,7,51;3739,1250,0;5239,0,34857;9839,1426,0;13039,17892,0;4089,14500,0;39,1,0;10739,5292,0;4539,50,160;7739,480,0;14839,4701,0;", -- [39]
            ";3390,35,0;40,1,0;15740,4000,16000;6590,1734,0;8190,37286,0;18690,20183,0;2240,151,0;3840,2571,0;18440,0,5154;14340,21778,0;7490,5680,0;1440,1230,0;1840,37,166;12040,4164,0;6340,875,0;14090,119,0;12940,54812,0;6790,1540,0;840,170,853;5640,25,100;9490,6121,0;15890,25831,0;17190,0,27687;7690,2232,0;1490,8910,0;10390,14175,0;15640,20233,0;6540,431,0;8140,2837,0;18740,10101,0;3590,102,578;16440,0,39649;10140,9085,0;13340,16498,0;18490,18655,0;12190,250,0;15390,12402,0;11040,1,0;14240,2294,0;7440,5042,0;3240,10,0;4040,1347,0;11940,389,0;15140,2529,0;6290,1,0;5140,6,25;6740,984,0;8490,1500,0;12590,75624,0;19040,19468,0;14640,10408,0;16740,942,0;13490,4000,0;15540,2827,0;8090,0,10751;1990,10077,0;10040,1767,0;10940,0,640;7390,4135,0;9790,674,0;12990,2589,0;11840,7137,0;6240,161,0;15940,10957,0;6690,3280,0;1640,15637,0;8390,1250,0;5540,2107,0;9290,3940,0;15690,4912,0;18840,0,219658;4390,500,0;12240,1750,7000;2740,375,0;14290,7725,0;10290,0,2500;9390,2146,0;2440,0,14099;6890,6,0;9940,6954,0;11990,8306,0;15190,9614,0;4840,142,0;2290,75,0;5740,25,90;10240,12576,0;15990,6850,0;3190,99,0;3990,8237,0;14940,4059,0;4140,1875,0;9640,4791,0;4240,300,0;12640,21894,0;2840,4,0;15090,22002,0;2690,6,0;17740,11352,0;7090,250,0;1740,97,0;15590,2674,0;2590,5,0;3940,1630,0;940,12565,0;14440,5347,0;4340,87,350;11190,5,0;5940,153,0;3890,0,8161;7540,8845,0;890,3517,0;2490,103,540;10090,5148,0;4090,19883,0;18390,26295,0;10190,6669,0;3490,2651,0;11290,3157,0;3040,885,0;15340,1016,0;4790,831,0;7990,2500,0;790,1966,0;14590,1821,0;2990,933,0;14190,1796,0;13890,70,0;9840,1432,0;13040,28494,0;5240,1244,0;16690,22311,0;2140,323,0;2940,43,0;3740,2416,0;4590,655,0;3340,31,0;9890,2919,0;10740,10398,0;7290,400,0;1790,93,0;12790,74619,0;4690,71,0;19440,500,0;14840,7925,0;4540,1,5;", -- [40]
            ";4791,133,0;14841,3974,0;4541,6,25;791,7069,0;16891,1472,0;1391,1392,0;1791,78,0;2591,5,0;3391,20,0;6591,1007,0;8191,7222,0;11391,205,0;14591,4180,0;10241,8425,0;5441,250,900;15491,211,0;16391,5652,25435;18441,8830,16713;14341,1250,4500;2241,615,0;3041,3769,0;4291,125,500;7491,6158,0;15241,3012,0;4741,5438,0;7941,12520,0;14091,119,0;5191,2964,0;6791,3464,0;9491,1308,0;12691,4000,0;10391,8193,0;15641,7587,0;16691,13110,0;4941,11,0;6541,159,0;8141,4613,0;11291,0,4500;10141,20103,0;18491,40695,0;1491,2207,0;15391,12178,0;14241,2311,0;7441,3514,0;13091,10637,0;11941,5896,0;4041,3149,0;17741,10476,0;6291,1,0;9641,13041,0;16041,3000,12000;6741,3014,0;1941,203,815;5591,67,0;15791,7172,0;19041,9946,0;14641,15777,0;6041,1500,0;13491,4000,0;18791,0,8000000;15541,3441,0;3341,1461,0;8091,0,10791;11191,45,0;15291,19345,0;7391,4253,0;11841,12527,0;6241,99,0;6691,8866,0;14791,6056,0;5541,3693,0;9291,3517,0;15691,6909,0;16791,1562,0;18841,275,990;14541,68426,0;2041,1337,0;2291,36330,0;16541,16875,75937;13391,15000,0;14291,4830,0;20391,1,0;9891,2524,0;19441,1500,0;13141,12093,0;16341,0,19282;11991,6317,0;15191,9034,0;5741,111,0;10041,7161,0;14941,2515,0;6191,615,0;2091,213,0;8291,6784,0;9391,12831,0;3641,15,0;7091,250,0;15591,8739,0;2841,50,0;14441,7166,0;4341,0,450;5941,115,0;7541,5479,0;10091,7441,0;3841,3106,0;18391,11246,0;2491,96,484;15741,4000,14400;15341,1545,0;1741,112,0;3291,202,0;4091,22479,0;6391,150,0;7991,2500,0;2691,10,0;9841,3163,0;13041,4503,0;3591,0,1887;5241,820,0;3491,2622,0;3891,0,9827;1191,82,0;15091,6542,0;2991,705,0;2141,0,4701;9791,1644,0;12791,39141,0;10191,7777,0;9941,5599,0;", -- [41]
            ";2592,33,0;19442,0,90000;2142,524,2358;7742,600,0;8192,4782,0;11392,403,0;14592,5016,0;6592,2447,0;13442,500,0;3392,1655,0;15492,212,0;18442,0,16000;1792,208,0;14342,4000,0;4292,200,0;7492,3015,0;12042,4749,0;3042,4577,0;6342,75,300;7942,14659,0;5192,1803,0;6792,4738,0;3492,4543,0;17192,0,4399;5642,450,1800;10392,1250,0;6092,420,0;16692,9372,0;9242,2421,0;15642,12162,0;18742,21444,0;14492,5000,0;4942,89,0;6542,219,0;8142,4568,0;16442,0,79853;10142,12752,0;15392,6382,0;5392,25,0;14242,5140,0;9892,2857,0;4242,343,0;11942,5303,0;15142,2603,0;17742,12451,0;6292,8,0;16042,3000,12000;4042,2062,0;6742,2830,0;10542,4310,0;5592,54,0;19042,9984,0;2892,30,0;6042,250,0;1942,316,1265;11192,2,0;8092,0,16250;10042,6644,0;18042,10,0;9792,823,0;12992,3035,0;7392,200,0;11842,14208,0;3792,521,0;6242,207,0;6692,9788,0;3442,44,0;9292,3684,0;15692,6763,0;5542,370,0;892,369,0;10192,13032,0;13392,12211,0;4392,2500,0;14292,6795,0;792,41,166;2042,3588,0;16342,8830,35323;3892,0,21939;15242,4420,0;11992,7396,0;15192,10595,0;6892,62,250;10842,17595,0;2742,375,0;4542,25,100;9492,3172,0;14942,2524,0;10592,150,0;4592,1,4;3192,495,0;19142,18903,0;2492,0,284;9842,2391,0;14742,1118,0;8292,11476,0;5042,0,50;8492,1000,0;9642,2092,0;16392,9693,0;10242,5318,0;15592,8465,0;7092,250,0;12542,12356,0;2842,100,0;14442,4481,0;3642,22,0;16992,30874,0;18392,56240,0;14842,12483,0;10092,5602,0;2692,10,40;4342,0,2500;7542,7106,0;942,4500,0;3842,2906,0;13142,11641,0;2092,3,0;9742,49,0;3342,137,0;14192,1946,0;3592,420,1894;6392,886,0;7992,2000,0;2642,33,0;3992,3082,0;13042,17596,0;3292,203,0;16242,5500,0;4092,1296,0;2392,77,413;4692,68,0;2992,408,0;15092,7656,0;5242,589,0;1742,111,0;9942,9341,0;16542,12703,0;12792,39255,0;15992,250,0;", -- [42]
            ";13443,400,1600;12793,16538,0;15993,1500,0;43,1,0;14843,5036,0;193,0,48;943,42863,0;1743,299,0;2143,788,3550;2943,537,0;4543,3605,0;12543,7156,0;15743,5000,0;8193,8708,0;11393,780,0;14593,2210,0;18693,11425,0;6593,922,0;10243,5035,0;15493,647,0;18443,10000,32000;3393,250,0;20243,0,400;4293,162,0;7493,3268,0;12043,5396,0;15243,7871,0;17943,34622,0;14093,21,0;1793,177,0;2243,63309,0;4743,5430,0;7943,15891,0;9743,67,0;5193,940,0;6793,3062,0;12693,5000,0;15893,1082,0;3493,2841,0;5643,500,2000;10393,1250,0;16693,16335,0;6093,4418,0;15643,6675,0;18743,13555,0;793,27,111;16443,0,74528;1443,21125,0;1843,145,725;18493,14233,0;6543,147,0;8143,7227,0;15393,6479,0;5393,32,0;14243,3844,0;13093,3381,0;3593,0,4929;4243,461,0;7443,2243,0;15143,1439,0;17743,40496,0;6293,33,0;9643,15583,0;16043,3000,0;1893,1663,0;4043,1958,0;6743,1462,0;16993,30989,0;9393,7882,0;15793,50,0;15543,4179,0;11193,16938,0;8093,0,27186;1943,638,0;9793,550,0;15943,27752,0;14793,6985,0;17043,3054,0;6693,1816,0;16793,4353,0;15693,12718,0;1993,2100,0;5543,450,0;14543,10257,0;16543,14445,76501;10193,19775,0;13393,62380,0;12243,45717,0;18343,14615,0;9893,6689,0;14293,9857,0;3893,0,22018;13143,21372,0;21243,100,0;11993,874,0;10843,10595,0;9943,11271,0;18843,0,222028;2043,1500,0;14943,2534,0;2643,28,0;3943,2244,0;10043,2033,0;9843,3739,0;4593,4,100;12643,75,0;19143,20188,0;893,137,0;8293,8542,0;2393,41,206;3193,2072,0;2593,0,135;15593,5532,0;14443,6745,0;3643,71,0;18393,14944,0;10093,18903,0;4343,25,0;5943,840,0;7543,8123,0;3993,4940,0;3843,3882,0;8343,500,2000;4443,3472,0;14193,2006,0;4793,744,0;6393,760,0;7993,2500,0;13043,32389,0;16243,5500,22000;19943,2000,0;2493,140,701;843,164,1077;15093,8368,0;5093,247,0;4093,713,0;9293,1250,0;", -- [43]
            ";19444,7500,27000;15994,2500,0;16894,1993,0;12544,7407,0;15744,5000,0;4544,50,200;11394,580,0;14594,2236,0;10244,12039,0;13444,1500,0;6594,1120,0;1744,260,0;18444,0,24000;15494,598,0;11144,1000,0;5444,548,0;20244,0,200;3394,250,0;12044,5145,0;15244,12769,0;7494,3543,0;14094,121,0;9744,34,0;4744,1986,0;6344,100,0;7944,20092,0;2244,51857,0;10644,500,0;5194,2854,0;6794,2641,0;1394,740,0;1794,277,0;14744,2310,0;10394,1250,0;16694,20178,0;18744,11371,0;15644,7915,0;6094,1278,0;14494,5500,0;18494,21444,0;10144,7834,0;13344,16194,0;8144,5036,0;14244,6870,0;5394,4,0;1844,84,728;9894,4661,0;13094,2646,0;17744,9031,0;11944,8821,0;15144,4232,0;7444,3223,0;3594,0,4947;16044,4000,0;4694,515,0;794,55,223;10544,15,0;6744,1190,0;16994,9884,0;9394,3225,0;4044,4584,0;11444,0,800;15544,2448,0;6044,450,0;11194,20298,0;2894,12,50;18294,250,0;10044,6666,0;194,0,49;13244,12068,0;8094,0,27289;15294,20135,0;20094,0,49873;9794,829,0;12994,787,0;1944,231,0;15944,512,0;17044,15038,0;2194,3110,0;14794,10446,0;19094,9488,0;6694,6309,0;844,144,720;9294,2000,0;16544,12795,0;18594,3000,0;13394,17871,0;3444,90,0;9944,8570,0;15444,1404,0;9644,9885,0;1994,16696,0;14294,5223,0;3794,774,0;3944,2277,0;10194,7345,0;13144,6865,0;6444,228,0;15194,22412,0;11994,1312,0;4394,750,0;44,1,0;10844,43439,0;2644,11,0;17594,6956,0;9844,3709,0;8544,400,0;17344,1,0;14944,4245,0;7344,5000,0;3344,133,0;2744,375,0;3644,47,0;9144,250,0;9444,1120,0;12644,200,0;6194,5308,0;4444,1490,0;3844,5080,0;8294,12651,0;2044,7045,0;15794,39,0;8194,13877,0;2394,1,416;15594,2520,0;3994,4404,0;4594,6,200;10094,3311,0;15394,14614,0;14444,10877,0;2594,375,1350;744,10000,0;4344,11,0;5944,311,0;7544,4296,0;2844,106,0;14194,756,0;3894,0,29986;944,83000,0;13044,52800,0;4794,703,0;6394,1041,0;1194,0,94;2494,74,401;18844,49516,222826;13944,26808,0;5244,3465,0;", -- [44]
            ";4245,2000,0;1645,100,360;10745,7887,0;13945,5000,0;19445,7500,27000;45,1,0;15995,19739,0;195,0,36;15745,5000,0;16645,62,0;11395,830,0;14595,4500,0;18695,34450,0;13445,500,0;2145,518,0;18445,5804,0;6595,1124,0;12295,282,0;15495,220,0;11145,1250,0;15245,22341,0;1745,155,0;3395,250,0;14095,36,0;9745,54,0;4745,3230,0;7945,17291,0;10645,750,0;2245,23076,0;3045,1489,0;5195,351,0;6795,500,0;12695,5000,0;8345,7421,0;16695,14375,0;18745,16177,0;15645,14577,0;1395,1,0;1795,235,0;6095,688,0;18495,16770,0;10145,9755,0;13345,15457,0;18245,0,900000;4945,37,0;6545,675,0;15395,10379,0;14245,2295,0;5395,3,0;13095,6646,0;15145,4943,0;7445,3660,0;10795,5542,0;16045,4000,0;1445,583,0;1845,175,877;3595,0,23;14895,4904,0;10545,3720,0;16995,13686,0;15795,7631,0;8245,12123,0;2445,0,3090;4045,2683,0;5595,235,0;18545,42546,0;6045,650,0;11195,13655,0;10045,23,0;795,56,223;1495,58,0;18045,300,0;15295,22712,0;20095,0,24749;5345,705,0;9795,626,0;15945,837,0;14795,12690,0;1945,247,0;5095,3,0;6695,3002,0;9295,2000,0;16545,12842,57793;14545,21997,0;13395,11403,0;9895,5221,0;4695,327,0;3445,226,0;4395,1600,0;8845,375,0;14295,13236,0;12045,7767,0;8395,1000,0;13145,2704,0;16345,49483,222674;2495,100,504;6445,88,0;3945,3852,0;11995,914,0;10245,8548,0;8495,1000,0;10845,14233,0;2295,70,0;12895,29461,0;8545,600,0;11745,6565,0;14945,6432,0;1195,47,0;3795,981,0;17745,22519,0;9645,6768,0;9445,2110,0;2745,375,0;4595,75,0;6195,416,0;17045,14846,0;8295,7988,0;845,289,1447;2595,0,1800;2545,1796,0;9945,4062,0;9395,1565,0;3995,6284,0;15595,2950,0;3195,1344,0;14445,12353,0;18395,36645,0;2395,64,322;7095,32,0;10095,10295,0;15345,1786,0;3345,2544,0;4345,100,0;7545,4658,0;14195,1252,0;2645,11,0;9845,2136,0;2195,13,0;13045,10738,0;16245,6000,0;2845,109,0;3645,189,0;4795,705,0;6395,1187,0;7995,0,5400;15095,20885,0;", -- [45]
            ";18046,3000,12000;10746,4549,0;19446,0,27000;15996,2500,0;5246,3489,0;14846,3846,0;4096,608,0;7296,56,0;12546,7033,0;15746,5000,0;16646,62,0;8196,13979,0;18696,29121,0;14596,3388,0;4546,533,0;7746,8430,0;1696,606,0;13446,1000,0;12296,722,0;15496,461,0;6596,2114,0;2146,955,4328;7046,1494,0;8846,250,0;12046,5757,0;15246,45093,0;14096,325,0;4296,525,0;7496,8824,0;9746,56,0;3396,250,0;4746,4324,0;6346,100,0;7946,21660,0;1746,332,0;5196,1830,0;6796,750,0;2246,30000,0;14746,589,0;16696,9058,0;12446,17,0;15646,19506,0;18496,14233,0;14496,5500,0;4496,125,450;6096,1,0;10146,16194,0;13346,23118,0;18246,0,900000;15396,558,0;4946,67,0;6546,533,0;8146,500,0;1396,1,0;1796,400,0;14246,4451,0;9896,2913,0;13096,7913,0;17746,5723,0;8746,682,0;15146,3730,0;10796,8982,0;4246,125,0;7446,3092,0;16046,4000,0;8496,1000,0;14896,3418,0;4696,5537,0;6296,28,0;16996,41451,0;3596,0,23;19046,12500,0;9396,7319,0;1446,563,0;1846,157,880;8246,7917,0;13496,6000,0;5596,91,0;15546,4793,0;4046,6783,0;18296,11489,0;11196,16875,0;13246,53622,0;15296,34267,0;14146,14084,0;12996,1527,0;15046,22482,0;10696,39766,0;13896,0,55103;15946,802,0;14796,16983,0;796,52,238;16796,33990,0;9296,2000,0;13396,30219,0;2996,40,0;14296,7451,0;4396,6000,0;5996,95,0;13046,34679,0;9846,1429,0;2646,31,0;13146,34704,0;3446,592,0;11996,1713,0;15196,2500,8000;6446,532,0;14046,5000,0;17596,5696,0;10846,15262,0;9946,11396,0;3946,2431,0;2296,50,0;11746,9180,0;3996,4659,0;16396,7073,0;3796,493,0;2596,0,108;19146,19533,0;13946,5000,0;9446,7564,0;1996,1720,0;16846,34014,0;8346,5363,0;1196,432,1771;8296,20861,0;4596,25,0;17046,4796,0;13546,62,0;18396,54051,0;3846,3787,0;846,82,1452;10046,32,0;20096,0,10692;10096,6130,0;2446,0,4946;15846,7500,0;7096,5,0;14946,4151,0;10196,7593,0;2396,26,215;3196,1377,0;15346,2788,0;14196,1601,0;4346,100,0;9796,1038,0;7546,4802,0;2046,2461,0;16246,6000,0;9646,6791,0;6396,2814,0;", -- [46]
            ";4797,852,0;6397,985,0;7997,81,0;19447,0,54000;147,1,0;647,70024,0;12797,53693,0;847,263,1749;15997,10,0;2847,110,0;3647,433,0;5247,7961,0;14847,4090,0;4097,305,0;15747,5000,0;16647,62,0;8197,7158,0;4547,8319,0;6147,37,0;7747,7754,0;13447,1250,0;15497,387,0;6597,1707,0;1297,2038,0;1697,445,0;3747,1092,0;7047,824,0;12047,4996,0;15247,50576,0;14097,236,0;9747,197,0;17197,10,0;6347,100,0;7947,24892,0;10647,0,2000;5197,1597,0;14747,668,0;16697,8577,0;897,1022,0;12447,10,0;15647,13125,0;4497,0,18000;6097,1,0;10147,10319,0;18247,0,900000;15397,641,0;6547,267,0;14247,4340,0;20047,0,59557;9897,7368,0;13097,2164,0;17747,500,0;8747,1036,0;15147,1753,0;10797,32538,0;1797,161,0;7447,2068,0;9647,7962,0;16047,4000,0;14897,2286,0;4697,435,0;6297,7,0;10547,74,0;9397,3672,0;15797,7000,0;3597,0,865;15547,1234,0;1447,22775,0;4447,901,0;6047,0,4400;18047,24428,0;20097,0,33619;9797,1528,0;12997,1727,0;5347,0,14141;6947,5,0;11847,7,0;10697,40812,0;13897,595,2976;15947,1887,0;17047,1313,0;14797,11427,0;1497,71,0;2547,1,0;16797,21738,0;18847,0,225196;15697,2741,0;20647,40097,0;10197,7623,0;13397,15201,0;12247,5674,0;8497,500,0;14297,13556,0;3197,9295,0;2397,36,215;4047,3959,0;4397,5000,0;5997,5,0;9947,5193,0;2447,10,36;4297,500,0;47,1,0;11997,6469,0;797,35,158;15197,2500,9000;1547,2795,0;3447,34,0;6447,523,0;14047,400,0;3947,3258,0;14947,11128,0;16997,41603,0;10097,10452,0;9447,3237,0;18497,9213,0;8297,9113,0;4197,2765,0;4597,250,0;18397,22464,0;18647,450,0;10047,164,0;14397,1600,0;8247,4997,0;2997,100,0;4247,1049,0;9197,600,0;1197,480,2132;1997,2538,0;14447,8801,0;2697,100,0;8397,1000,0;8347,5979,0;7097,1,0;15347,955,0;14197,925,0;3997,7929,0;4347,150,0;9847,1779,0;13047,56588,0;9297,2500,0;3047,996,0;", -- [47]
            ";2398,0,434;3198,2444,0;48,1,0;19448,0,72000;4798,1166,5249;6398,989,0;9598,94,0;14848,6854,0;5248,6849,0;2048,19,0;18948,1589,0;10498,16,0;16648,62,0;12548,7102,0;15748,5000,0;7298,456,0;8198,3829,0;14598,2655,0;10248,9371,0;6148,44,0;15498,674,0;6598,2426,0;848,351,1755;15248,1791,0;7048,714,0;14098,43,0;3748,1304,0;9748,274,0;10648,0,500;6348,125,0;14748,1550,0;5198,1635,0;16698,16488,0;10398,1000,0;12448,19,0;15648,22212,0;18498,47443,0;3048,1919,0;14498,6250,0;3848,1426,0;18248,0,900000;13348,72769,0;4498,625,2250;1748,232,0;15398,108,0;14248,2372,0;4948,326,0;6548,225,0;17748,7257,0;9898,2718,0;2698,100,0;15148,1613,0;5398,8,0;6998,774,0;9648,6392,0;7448,2076,0;14898,2295,0;3948,3092,0;16998,35631,0;10548,2500,0;1798,279,0;6748,897,0;10298,1243,0;3598,209,868;15548,1589,0;10048,935,0;13248,29669,0;4448,1991,0;6048,100,0;20098,0,16684;15298,305,0;4048,1767,0;1448,405,0;12998,1020,0;11848,3,0;15048,24409,0;2898,32,0;13898,0,288699;17048,400,0;19098,17912,0;14798,24037,0;9298,2250,0;15698,4273,0;9998,4815,0;10198,12167,0;1498,57,0;18848,50215,225971;13398,19875,0;2598,30,0;4998,837,0;7148,21,0;11098,500,0;4398,900,0;14298,9628,0;3798,902,0;15798,50,0;13148,62629,0;16548,8492,38947;11998,2896,0;15198,10000,36000;9948,7647,0;2148,581,2326;8498,2500,0;14048,2000,0;10148,11273,0;6448,1942,0;4698,269,0;9398,2488,0;9698,1504,0;17348,250,0;2648,58,0;3448,6,0;11748,25083,0;13498,18274,0;14948,10256,0;12798,57150,0;19148,25399,0;8248,7032,0;9448,1154,0;7348,690,0;3998,5656,0;8298,8565,0;9798,1588,0;10098,8736,0;18648,450,0;4598,212,0;6198,1113,0;798,70,319;15598,2763,0;2798,25,0;18398,27103,0;14448,5555,0;5048,0,45;10798,6263,0;8398,1125,0;2098,3020,0;8348,8044,0;2748,375,0;13098,15282,0;2448,0,12926;8948,200,720;148,1,0;15348,792,0;5498,200,0;7098,6,0;1198,535,2140;1998,7239,0;14198,1531,0;9848,1578,0;13048,5712,0;4298,162,0;4348,175,0;", -- [48]
            ";6149,120,480;4599,100,400;13049,6191,0;49,1,0;10749,9279,0;3199,2262,0;4799,0,2738;6399,1639,0;9599,142,0;12799,7000,0;14849,8271,0;5249,8658,0;10499,2105,0;16649,62,0;12549,13669,0;15749,5000,0;2849,1135,0;8199,24661,0;14599,4663,0;10249,13435,0;4549,2092,0;12299,16,0;15499,307,0;4999,1052,0;6599,2678,0;13199,4398,0;15249,3877,0;7049,898,0;14099,47,0;9749,275,0;2149,0,3518;2949,1039,0;6349,125,500;849,265,1326;14749,1325,0;16699,21202,0;1299,390,0;3399,81,0;5199,739,0;10399,1467,0;15649,7808,0;18499,34589,0;14499,7500,0;10149,10915,0;13349,58428,0;2249,91,0;3049,2119,0;4499,0,80000;15399,139,0;14249,7066,0;17749,11425,0;4949,1706,0;6549,101,0;9899,7353,0;13099,1431,0;8749,1603,0;1749,396,0;2699,200,0;5399,11,0;10799,39135,0;4249,703,0;7449,625,0;14899,3731,0;16999,10709,0;10549,162,0;3949,6517,0;4699,213,0;6299,1,0;15799,7108,0;6749,897,0;10299,3778,0;13499,6000,0;899,1215,0;1399,12,0;1799,211,0;2799,67,0;3599,2,0;13249,67782,0;20099,0,7207;4449,3332,0;6049,170,0;2449,20,72;4049,832,0;12999,793,0;15049,20472,0;6949,20,0;1449,1875,0;1849,277,1249;14799,17308,0;16799,15852,0;18849,3750,0;15699,25,0;8349,14478,0;20649,22425,0;14549,6899,0;3299,48,0;18349,12332,0;9999,4832,0;13399,45172,0;15449,421,0;9149,250,0;14299,8602,0;4299,500,0;1499,51,0;4399,50,200;11999,5538,0;15199,10000,36000;17599,11518,0;10199,17261,0;3849,5328,0;6449,701,0;9699,1812,0;3799,1460,0;2099,45040,0;11749,20141,0;14949,3907,0;2649,1,0;3449,207,0;8249,5336,0;18949,500,1800;9449,9564,0;8299,14353,0;9849,2385,0;18649,450,1800;8499,2500,0;2299,8756,0;4099,1131,0;6199,650,0;3999,7109,0;15599,6631,0;18399,27103,0;799,71,319;14449,10125,0;10099,6567,0;9949,4030,0;8399,1250,0;8949,200,0;15349,1549,0;2549,4328,0;2749,375,0;7099,6,0;9799,599,0;17049,0,81000;", -- [49]
            [0] = ";9850,2394,0;2750,375,0;4350,200,0;9600,93,0;12800,10000,0;16000,3750,0;4800,0,5498;6400,3089,0;4000,1797,0;10500,3478,0;5250,2484,0;16650,62,0;2000,8827,0;8200,7275,0;14600,2675,0;18450,15814,0;15500,1377,0;4550,2092,0;7750,1712,0;20000,0,108000;6600,867,0;15250,6894,0;14100,12089,0;7050,999,0;9750,57,0;4300,700,0;2150,541,2344;6350,295,0;16700,22546,0;14750,1068,0;10400,563,0;5200,2323,0;18500,31340,0;3400,2642,0;15650,22230,0;1300,1854,0;10150,17254,0;15400,109,0;4500,8750,0;14250,3872,0;9900,3195,0;13100,5421,0;6550,196,0;8150,250,0;850,176,883;15150,2951,0;10800,6844,0;7000,650,0;19300,0,400;16050,5000,0;8500,1250,5000;14900,4298,0;7450,500,0;1750,298,0;15800,36082,0;4700,379,0;6300,443,0;8250,8034,0;200,0,1816;6750,897,0;15550,3316,0;14400,1215,0;10050,2500,0;3600,4,0;15300,86,0;6050,300,0;1800,479,0;9800,2027,0;13000,65225,0;15050,25383,0;10700,4983,0;16800,25717,0;10450,396,0;3850,7304,0;2950,3386,0;3300,9,0;1850,238,1254;18350,13063,0;4050,2374,0;16900,48673,0;10200,12314,0;8750,1863,0;13400,9143,0;15350,2384,0;12250,0,40341;10300,1250,0;15450,533,0;10250,14866,0;14300,7685,0;17600,11866,0;4250,705,0;15200,0,18000;4400,0,2000;3950,4157,0;18600,12000,0;14450,7364,0;6450,200,0;11750,34859,0;14950,3630,0;6900,4685,0;16850,27564,0;14850,10442,0;2850,1439,0;9450,2181,0;2650,3,0;8300,20172,0;10750,34877,0;12550,6464,0;5750,641,0;7350,27,0;18650,1250,5000;11150,750,0;8350,1130,0;2450,25,0;18400,14853,0;15600,3787,0;4600,0,340;6200,1095,0;10550,14,0;8400,1250,0;2100,24539,0;2300,165,0;3800,1068,0;10100,11108,0;9950,13418,0;8950,200,800;2800,1390,0;17050,12536,0;2700,100,0;2900,89,0;14200,2137,0;5500,750,0;7100,7,0;",
        }
    end