﻿function JQuest:Log_Init()
	if not JQuest.var.frameSave then
		UIPanelWindows["JQuestLogFrame"] = { area = "left", pushable = 3, whileDead = 1 }
	end
	JQuestLogFrame:EnableMouse(true)
	JQuestLogFrame:SetMovable(true)
	JQuestLogFrame:SetScript("OnMouseDown", JQuest.Log_OnMouseDown)
	JQuestLogFrame:SetScript("OnMouseUp", JQuest.Log_OnMouseUp)

	JQuestLogVersionText:SetText("JasonQuest v" .. self.version)
	JQuestLogNoQuestsText:SetText("Select a member tab\nto view their log.")

	self:Log_SetWidth()
	self:Log_FrameBlank(who)

	local name, title, notes, enabled, loadable, reason, security = GetAddOnInfo("JasonQuestLH")
	if loadable then
		local jqtab = CreateFrame("CheckButton", "JQuestLHTab", JQuestLogFrame, "SpellBookSkillLineTabTemplate")
		jqtab:SetPoint("BOTTOMLEFT", JQuestLogFrame,"BOTTOMRIGHT", 0, 52)
		jqtab:SetNormalTexture("Interface\\AddOns\\LightHeaded\\images\\wh_icon")
		jqtab.tooltip = "Load LightHeaded_JasonQuest"
		jqtab:SetChecked(nil)
		jqtab:Show()
		jqtab:SetScript("OnClick", function()
			local l, r = LoadAddOn("JasonQuestLH")
			if r then self:Log("JasonQuest LightHeaded not loaded (" .. r .. ")") end

			if this:GetChecked() then
				LightHeadedFrame:Show()
			elseif not QuestLogFrame:IsVisible() then
				LightHeadedFrame:Hide()
			else
				this:SetChecked(1)
			end
		end)
	elseif reason then
		self:Log("JasonQuest LightHeaded not loadable (" .. reason .. ")")
	end

	self.LogFrame.waitingForInit = nil
end
function JQuest:Log_Toggle()
	if self.LogFrame.waitingForInit then
		self:Log("JQuest is not ready, please wait.", 1)
	elseif JQuestLogFrame:IsVisible() then
		HideUIPanel(JQuestLogFrame)
	else
		ShowUIPanel(JQuestLogFrame)
	end
end
function JQuest:Log_OnShow()
	PlaySound("igQuestLogOpen")
	self:Log_UpdateSearchBoxPosition()
end
function JQuest:Log_OnHide()
	this:StopMovingOrSizing()
	PlaySound("igQuestLogClose")
end
function JQuest:Log_OnMouseDown()
	this:StartMoving()
	this:SetUserPlaced(JQuest.var.frameSave)
end
function JQuest:Log_OnMouseUp()
	this:StopMovingOrSizing()
end

-- ********************
-- Single / Double Wide
-- ********************
function JQuest:Log_SetWidth()
	local wide = self.var.wide
	if wide then
		self:Log_SetDoubleWide()
	else
		self:Log_SetSingleWide()
	end
	self:Log_RefreshTextures(wide)
	self:Log_RefreshList()
	self:Log_RefreshDetails()
	if JQuest_SearchEditBox then
		self:Log_UpdateSearchBoxPosition()
	end
end
function JQuest:Log_SetSingleWide()
	JQuestLogFrame:SetAttribute("UIPanelLayout-width", 350)

	JQuestLogFrame:SetWidth(350)

	JQuestLogDetailScrollFrame:ClearAllPoints()
	JQuestLogDetailScrollFrame:SetPoint("TOPLEFT", JQuestLogListScrollFrame, "BOTTOMLEFT", 0, -7)
	JQuestLogDetailScrollFrame:SetHeight(261)

	JQuestLogListScrollFrame:SetHeight(93)

	self.LogFrame.QuestsDisplayed = 6

	for i = 7, 23 do
		if getglobal("JQuestLogTitle" .. i) then
			getglobal("JQuestLogTitle" .. i):Hide()
		end
	end
end
function JQuest:Log_SetDoubleWide()
	JQuestLogFrame:SetAttribute("UIPanelLayout-width", 679)

	JQuestLogFrame:SetWidth(679)

	JQuestLogDetailScrollFrame:ClearAllPoints()
	JQuestLogDetailScrollFrame:SetPoint("TOPLEFT", JQuestLogListScrollFrame, "TOPRIGHT", 29, 0)
	JQuestLogDetailScrollFrame:SetHeight(360)

	JQuestLogListScrollFrame:SetHeight(360)

	self.LogFrame.QuestsDisplayed = 23

	for i = 7, self.LogFrame.QuestsDisplayed do
		if not getglobal("JQuestLogTitle" .. i) then
			local button = CreateFrame("Button", "JQuestLogTitle" .. i, JQuestLogFrame, "JQuestLogTitleButtonTemplate")
			button:SetID(i)
			button:Hide()
			button:ClearAllPoints()
			button:SetPoint("TOPLEFT", getglobal("JQuestLogTitle" .. (i-1)), "BOTTOMLEFT", 0, 1)
		end
	end
end
function JQuest:Log_CreateSingleTextures()
	local textures = {
		TopLeft = "Interface\\QuestFrame\\UI-QuestLog-TopLeft",
		TopRight = "Interface\\QuestFrame\\UI-QuestLog-TopRight",
		BotLeft = "Interface\\QuestFrame\\UI-QuestLog-BotLeft",
		BotRight = "Interface\\QuestFrame\\UI-QuestLog-BotRight"}
	local xOff = { Left = 0, Right = 256 }
	local yOff = { Top = 0, Bot = -256 }
	local height = { Top = 256, Bot = 209 }
	local width = { Left = 256, Right = 94 }
	local tWidth = { Left = 256, Right = 128 }
	local region, yofs, xofs

	for name, path in pairs(textures) do
		yofs, xofs = name:match("^([A-Z][a-z]+)([A-Z][a-z]+)$")
		region = JQuestLogFrame:CreateTexture("JQuest" .. yofs .. xofs, "ARTWORK")
		region:ClearAllPoints()
		region:SetTexture(path)
		region:SetWidth(width[xofs])
		region:SetHeight(height[yofs])
		region:SetTexCoord(0, width[xofs]/tWidth[xofs], 0, height[yofs]/256)
		region:SetPoint("TOPLEFT", JQuestLogFrame, "TOPLEFT", xOff[xofs], yOff[yofs])
	end
end
function JQuest:Log_CreateWideTextures()
	local left = JQuestLogFrame:CreateTexture("JQuestWLeft", "ARTWORK")
	left:ClearAllPoints()
	left:SetTexture("Interface\\AddOns\\JasonQuest\\DoubleWide\\JQuestLeft")
	left:SetWidth(512)
	left:SetHeight(465)
	left:SetTexCoord(0, 1, 0, 465 / 512)
	left:SetPoint("TOPLEFT", JQuestLogFrame)

	local right = JQuestLogFrame:CreateTexture("JQuestWRight", "ARTWORK")
	right:ClearAllPoints()
	right:SetTexture("Interface\\AddOns\\JasonQuest\\DoubleWide\\JQuestRight")
	right:SetWidth(167)
	right:SetHeight(465)
	right:SetTexCoord(0, 167 / 256, 0, 465 / 512)
	right:SetPoint("TOPLEFT", JQuestLogFrame, 512, 0)
end
function JQuest:Log_RefreshTextures(wide)
	if wide then
		if self.Log_CreateWideTextures then
			self:Log_CreateWideTextures()
			self.Log_CreateWideTextures = nil
		else
			JQuestWLeft:Show()
			JQuestWRight:Show()
		end
		if JQuestTopLeft then
			JQuestTopLeft:Hide()
			JQuestTopRight:Hide()
			JQuestBotLeft:Hide()
			JQuestBotRight:Hide()
		end
	else
		if self.Log_CreateSingleTextures then
			self:Log_CreateSingleTextures()
			self.Log_CreateSingleTextures = nil
		else
			JQuestTopLeft:Show()
			JQuestTopRight:Show()
			JQuestBotLeft:Show()
			JQuestBotRight:Show()
		end
		if JQuestWLeft then
			JQuestWLeft:Hide()
			JQuestWRight:Hide()
		end
	end
end

-- *************
-- Tab Functions
-- *************
function JQuest:Log_CreateTabBars()
	local bar
	for i = 1, MAX_PARTY_MEMBERS do
		bar = getglobal("JQuestTab" .. i):CreateTexture("JQuestTab" .. i .. "Bar", "OVERLAY")
		bar:SetTexture("Interface\\Tooltips\\UI-Tooltip-Background")
		bar:ClearAllPoints()
		bar:SetPoint("RIGHT")
		bar:SetWidth(32)
		bar:SetHeight(30)
		bar:SetVertexColor(0.6, 0, 0)
	end
end
function JQuest:UNIT_PORTRAIT_UPDATE()
	if self.Player.waitingForInit then return end
	self:Log("Event: UNIT_PORTRAIT_UPDATE")
	self:Log_RefreshTabs()
	if self.var.debugMode then
		SetPortraitTexture(JQuestTab5Portrait, "player")
	end
end
function JQuest:Log_RefreshTabs()
	if self.Log_CreateTabBars then
		self:Log_CreateTabBars()
		self.Log_CreateTabBars = nil
	end

	local tab, pic, overlay, unit, name, status, sending, model
	for i = 1, MAX_PARTY_MEMBERS do
		tab = getglobal("JQuestTab" .. i)
		pic = getglobal("JQuestTab" .. i .. "Portrait")
		overlay = getglobal("JQuestTab" .. i .. "Bar")
		unit = "party" .. i
		name = UnitName(unit)
		if self.Party[name] then
			status = self.Party[name]
		end
		sending = self.PartySending[name]

		if GetPartyMember(i) then
			SetPortraitTexture(pic, unit)
			pic:Show()

			if status and status > 0 and self.PartyQuests[name] then
				tab.tooltip = name .. "'s Quest Log"
				if sending then
					quests = 0
					for q in pairs(self.PartyQuests[name]) do
						quests = quests + 1
					end
					status = 32-((32/sending)*quests)
					if status > 0 and status <= 32 then
						overlay:SetWidth(status)
					end
					overlay:Show()
				else
					overlay:Hide()
				end
			elseif status == 0 then
				tab.tooltip = name .. " is using an incompatible\nversion of JasonQuest."
				overlay:SetWidth(32)
				overlay:SetVertexColor(0.5, 0.5, 0.5)
				overlay:Show()
			else
				tab.tooltip = "No data for " .. name .. "."
				overlay:SetWidth(32)
				overlay:SetVertexColor(0.6, 0, 0)
				overlay:Show()
			end
			tab.who = name
			tab.unit = unit
			tab:Show()
		else
			tab:Hide()
		end
		if self.LogFrame.selectedWho and self.LogFrame.selectedWho == name then
			tab:SetChecked(true)
		else
			tab:SetChecked(false)
		end
	end
end
function JQuest:Log_Tab_OnClick()
	if this:GetChecked() then
		self.LogFrame.selectedWho = this.who
		self.LogFrame.selectedQuest = nil
	    JQuestLogTitleText:SetText(this.who .. "'s " .. QUEST_LOG)
        SetPortraitTexture(JQuestPortrait, this.unit)
		self:Log_RefreshList()
		self:Log_RefreshDetails()
	else
		self.LogFrame.selectedWho = nil
		self.LogFrame.selectedQuest = nil
		self:Log_FrameBlank()
	end
	for i = 1, MAX_PARTY_MEMBERS + 1 do
		if this:GetID() ~= i then
			getglobal("JQuestTab" .. i):SetChecked(false)
		end
	end
end
function JQuest:Log_Tab_OnEnter()
	GameTooltip:SetOwner(this, "ANCHOR_RIGHT")
	GameTooltip:AddLine(this.tooltip)

	if self.PartyXP[this.who] then
		local pxp = self.PartyXP[this.who]
		GameTooltip:AddLine(" ")
		GameTooltip:AddLine("|c00FFFFFFXP Stats|r |c00808080(Level " .. pxp.level .. ")|r")
		GameTooltip:AddLine("Current XP: |c00FFFFFF" .. string.format("%s/%s (%3.0f%%)", pxp.xp, pxp.xpmax, (pxp.xp/pxp.xpmax)*100) .. "|r")
		GameTooltip:AddLine("To Level: |c00FFFFFF" .. string.format("%d (%2.f%%)", (pxp.xpmax - pxp.xp), ((pxp.xpmax - pxp.xp)/pxp.xpmax)*100) .. "|r")
		GameTooltip:AddLine("Rested XP: |c00FFFFFF" .. string.format("%d (%2.f%%)", pxp.xprest, (pxp.xprest/pxp.xpmax)*100) .. "|r")
	end
	GameTooltip:Show()
end

-- ******************
-- Core Log Functions
-- ******************
function JQuest:Log_BuildFrameTable(who)
	if not self.PartyQuests[who] then
		self.PartyFrameData[who] = {}
		return
	end
	local o, headers, byId = {}, {}, {}

	-- The full data set, by zone
	for title, data in pairs(self.PartyQuests[who]) do
		if not o[data.header] then
			o[data.header] = {}
		end
		tinsert(o[data.header], {title = title, level = data.level})
	end
	for zone, data in pairs(o) do
		tinsert(headers, { title = zone })
	end
	table.sort(headers, function(a, b) return a.title < b.title end)

	-- Full data set, by ID
	for idx, zone in pairs(headers) do
		tinsert(byId, {title = zone.title, isHeader = true})
		table.sort(o[zone.title], function(a, b)
			if a.level == b.level then
				return a.title < b.title
			else
				return a.level < b.level
			end
		end)

		for title, data in pairs(o[zone.title]) do
			tinsert(byId, {title = data.title})
		end
	end
	self.PartyFrameData[who] = byId
	return true
end
function JQuest:Log_FrameBlank(who)
	EmptyJQuestLogFrame:Show()
	JQuestFramePushQuestButton:Disable()
	JQuestLogDetailScrollFrame:Hide()
	JQuestLogListScrollFrame:Hide()
	JQuestLogHighlightFrame:Hide()
	JQuestLogQuestCount:SetText("")
	for i=1, self.LogFrame.QuestsDisplayed do
		getglobal("JQuestLogTitle"..i):Hide()
	end
	if not who then
		JQuestFrameSyncButton:Disable()
		JQuestLogTitleText:SetText(QUEST_LOG)
		JQuestLogNoQuestsText:SetText("Select a member tab\nto view their log.")
		JQuestPortrait:SetTexture("Interface\\QuestFrame\\UI-QuestLog-BookIcon")
	else
		JQuestFrameSyncButton:Enable()
		JQuestLogNoQuestsText:SetText("No data for this member.")
	end
end
function JQuest_Log_UpdateList()
	JQuest:Log_RefreshList(1)
end
function JQuest:Log_RefreshList(skipRebuild)
	local who = self.LogFrame.selectedWho
	if not who then
		return
	elseif not skipRebuild then
		self:Log_BuildFrameTable(who)
	end
	local numEntries, numQuests = self:Log_GetNumQuests(who)

	if numQuests == 0 then
		self:Log_FrameBlank(who)
		return
	elseif not self.LogFrame.selectedQuest and self.PartyFrameData[who][2].title then
		self.LogFrame.selectedQuest = self.PartyFrameData[who][2].title
		self:Log_RefreshDetails()
	end

	EmptyJQuestLogFrame:Hide()
	JQuestFrameSyncButton:Enable()
	JQuestLogListScrollFrame:Show()

	-- Update Quest Count
	JQuestLogQuestCount:SetText(format(QUEST_LOG_COUNT_TEMPLATE, numQuests, MAX_QUESTLOG_QUESTS))
	JQuestLogCountMiddle:SetWidth(JQuestLogQuestCount:GetWidth())

	-- ScrollFrame Update
	FauxScrollFrame_Update(JQuestLogListScrollFrame, numEntries, self.LogFrame.QuestsDisplayed, QUESTLOG_QUEST_HEIGHT, nil, nil, nil, JQuestLogHighlightFrame, 293, 316)

	-- Update the quest listing
	JQuestLogHighlightFrame:Hide()

	local listItem, itemTag, itemCheck, itemGroup, itemNormalText
	local listId, title, tag, group, level, isHeader, pushable, status, daily, questLink
	local titlePlusLevel, partyOnQuest, tempWidth, textWidth, color, quest, data, member, questName, playerOnQuest
	for i = 1, self.LogFrame.QuestsDisplayed do
		listId = i + FauxScrollFrame_GetOffset(JQuestLogListScrollFrame)
		listItem = getglobal("JQuestLogTitle" .. i)
		itemTag = getglobal("JQuestLogTitle" .. i .. "Tag")
		itemGroup = getglobal("JQuestLogTitle" .. i .. "GroupMates")
		itemCheck = getglobal("JQuestLogTitle" .. i .. "Check")
		itemNormalText = getglobal("JQuestLogTitle" .. i .. "NormalText")
		partyOnQuest = 0
		playerOnQuest = 0
		if listId <= numEntries then
			title, level, tag, group, isHeader, pushable, status, daily, questLink = self:Log_GetQuestDetails(listId, who)
			if isHeader then
				listItem:SetText(title)
				itemGroup:SetText("")
				itemCheck:Hide()
			else
				listItem.questLink = questLink
				titlePlusLevel = self:AddQuestLevel(title, level, tag, group)
				listItem:SetText("  " .. titlePlusLevel)
				JQuestLogDummyText:SetText("  " .. titlePlusLevel)
				listItem:SetNormalTexture("")

				for member, quest in pairs(self.PartyQuests) do
					for questName, data in pairs(quest) do
						if questName == title and data.level == level and member ~= UnitName("player") then
							partyOnQuest = partyOnQuest + 1
						end
					end
				end
				if self.PlayerQuests[title] and self.PlayerQuests[title].level == level then
					playerOnQuest = 1
				end
				if partyOnQuest > 1 or playerOnQuest > 0 then
					itemGroup:SetText("[" .. partyOnQuest + playerOnQuest .. "]")
				else
					itemGroup:SetText("")
				end
			end
			listItem.isHeader = isHeader
			listItem.questName = title
			listItem.questLevel = level

			if status and status < 0 then
				tag = FAILED
			elseif status and status > 0 then
				tag = COMPLETE
			elseif daily then
				if tag then
					tag = format(DAILY_QUEST_TAG_TEMPLATE, tag)
				else
					tag = DAILY
				end
			end
			if tag then
				itemTag:SetText("(" .. tag .. ")")
				tempWidth = 260 - itemTag:GetWidth()

				if JQuestLogDummyText:GetWidth() > tempWidth then
					textWidth = tempWidth
				else
					textWidth = JQuestLogDummyText:GetWidth()
				end

				itemNormalText:SetWidth(tempWidth)

				-- If there's quest tag position check accordingly
				itemCheck:Hide()
				if self:QW_IsWatched(who, title) then
					if itemNormalText:GetWidth() + 24 < 275 then
						itemCheck:SetPoint("LEFT", listItem, "LEFT", textWidth+24, 0)
					else
						itemCheck:SetPoint("LEFT", listItem, "LEFT", textWidth+10, 0)
					end
					itemCheck:Show()
				end
			else
				itemTag:SetText("")
				itemNormalText:SetWidth(260)

				-- Show check if quest is being watched
				itemCheck:Hide()
				if self:QW_IsWatched(who, title) then
					if (itemNormalText:GetWidth() + 24) < 275 then
						itemCheck:SetPoint("LEFT", itemNormalText, "LEFT", itemNormalText:GetWidth()+24, 0)
					else
						itemCheck:SetPoint("LEFT", itemNormalText, "LEFT", itemNormalText:GetWidth()+10, 0)
					end
					itemCheck:Show()
				end
			end

			if isHeader then
				color = QuestDifficultyColor["header"]
			else
				color = GetDifficultyColor(level)
			end
			if playerOnQuest > 0 then
				itemGroup:SetTextColor(.80, .80, 1)
			else
				itemGroup:SetTextColor(color.r, color.g, color.b)
			end
			itemTag:SetTextColor(color.r, color.g, color.b)
			listItem:SetTextColor(color.r, color.g, color.b)
			listItem.r = color.r
			listItem.g = color.g
			listItem.b = color.b
			listItem:Show()

			if self.LogFrame.selectedQuest and self.LogFrame.selectedQuest == title then
				JQuestLogHighlightFrame:SetPoint("TOPLEFT", "JQuestLogTitle" .. i, "TOPLEFT")
				JQuestLogSkillHighlight:SetVertexColor(color.r, color.g, color.b)
				JQuestLogHighlightFrame:Show()
				itemTag:SetTextColor(HIGHLIGHT_FONT_COLOR.r, HIGHLIGHT_FONT_COLOR.g, HIGHLIGHT_FONT_COLOR.b)
				listItem:LockHighlight()
				if pushable then
					JQuestFramePushQuestButton:Enable()
				else
					JQuestFramePushQuestButton:Disable()
				end
			else
				listItem:UnlockHighlight()
			end
			if not JQuest_SearchEditBox.searchText or (JQuest_SearchEditBox.searchText and strfind(strlower(title), JQuest_SearchEditBox.searchText)) then
				listItem:SetAlpha(1)
			else
				listItem:SetAlpha(self.var.unmatchedAlpha)
			end
		else
			listItem:Hide()
		end
	end
end
function JQuest:Log_RefreshDetails(doNotScroll)
	local who = self.LogFrame.selectedWho
	local quest = self.LogFrame.selectedQuest
	if not who or not quest or not self.PartyQuests[who] or not self.PartyQuests[who][quest] then
		JQuestLogDetailScrollFrame:Hide()
		return
	end

	local questData = self.PartyQuests[who][quest]

	local questTitle = self:AddQuestLevel(quest, questData.level, questData.tag, questData.group)
	JQuestLogQuestTitle:SetText(questTitle)
	JQuestLogObjectivesText:SetText(questData.obj)
	JQuestLogObjective1:SetPoint("TOPLEFT", JQuestLogObjectivesText, "BOTTOMLEFT", 0, -10)

	local numObj, status, text, line, groupStr
	if questData.leader and getn(questData.leader) > 0 then
		numObj = getn(questData.leader)
	else
		numObj = 0
	end

	for i = 1, numObj do
		status = questData.leader[i].status
		text = questData.leader[i].text
		line = getglobal("JQuestLogObjective" .. i)
		if status and status > 0 then
			line:SetTextColor(0.2, 0.2, 0.2)
			text = text .. " (" .. COMPLETE .. ")"
		else
			line:SetTextColor(0, 0, 0)
		end
		line:SetText(text)
		line:Show()
	end

	for i = numObj + 1, MAX_OBJECTIVES do
		getglobal("JQuestLogObjective" .. i):Hide()
	end

	-- If there's money required then anchor and display it
	if questData.requiredMoney then
		if numObj > 0 then
			JQuestLogRequiredMoneyText:SetPoint("TOPLEFT", "JQuestLogObjective"..numObj, "BOTTOMLEFT", 0, -4)
		else
			JQuestLogRequiredMoneyText:SetPoint("TOPLEFT", JQuestLogObjectivesText, "BOTTOMLEFT", 0, -10)
		end
		
		MoneyFrame_Update("JQuestLogRequiredMoneyFrame", abs(questData.requiredMoney))

		if questData.requiredMoney < 0 then
			-- Not enough money
			JQuestLogRequiredMoneyText:SetTextColor(0, 0, 0)
			SetMoneyFrameColor("JQuestLogRequiredMoneyFrame", 1.0, 0.1, 0.1)
		else
			JQuestLogRequiredMoneyText:SetTextColor(0.2, 0.2, 0.2)
			SetMoneyFrameColor("JQuestLogRequiredMoneyFrame", 1.0, 1.0, 1.0)
		end
		JQuestLogRequiredMoneyText:Show();
		JQuestLogRequiredMoneyFrame:Show();
	else
		JQuestLogRequiredMoneyText:Hide();
		JQuestLogRequiredMoneyFrame:Hide();
	end

	if questData.group > 0 then
		groupStr = format(QUEST_SUGGESTED_GROUP_NUM, questData.group)
		JQuestLogSuggestedGroupNum:SetText(groupStr)
		JQuestLogSuggestedGroupNum:Show()
		JQuestLogSuggestedGroupNum:ClearAllPoints()
		if questData.requiredMoney then
			JQuestLogSuggestedGroupNum:SetPoint("TOPLEFT", JQuestLogRequiredMoneyText, "BOTTOMLEFT", 0, -4)
		elseif numObj > 0 then
			JQuestLogSuggestedGroupNum:SetPoint("TOPLEFT", "JQuestLogObjective" .. numObj, "BOTTOMLEFT", 0, -4)
		else
			JQuestLogSuggestedGroupNum:SetPoint("TOPLEFT", JQuestLogObjectivesText, "BOTTOMLEFT", 0, -10)
		end
		JQuestLogDescriptionTitle:SetPoint("TOPLEFT", JQuestLogSuggestedGroupNum, "BOTTOMLEFT", 0, -10);
	else
		JQuestLogSuggestedGroupNum:Hide()
		if numObj > 0 then
			JQuestLogDescriptionTitle:SetPoint("TOPLEFT", "JQuestLogObjective"..numObj, "BOTTOMLEFT", 0, -10)
		else
			JQuestLogDescriptionTitle:SetPoint("TOPLEFT", JQuestLogObjectivesText, "BOTTOMLEFT", 0, -10)
		end
	end

	JQuestLogQuestDescription:SetText(questData.desc)
	QuestFrame_SetAsLastShown(JQuestLogQuestDescription, JQuestLogSpacerFrame)

	local numRewards, numChoices, money = self:GetPartyRewardInfo(questData.rewardInfo)
	if (numRewards + numChoices + money) > 0 then
		JQuestLogRewardTitleText:Show()
		QuestFrame_SetAsLastShown(JQuestLogRewardTitleText, JQuestLogSpacerFrame)
	else
		JQuestLogRewardTitleText:Hide()
	end

	if numRewards + numChoices + money == 0 then
		JQuestLogRewardTitleText:Hide()
	else
		JQuestLogRewardTitleText:Show()
		QuestFrame_SetAsLastShown(JQuestLogRewardTitleText, JQuestLogSpacerFrame)
	end
	if money == 0 then
		JQuestLogMoneyFrame:Hide()
	else
		JQuestLogMoneyFrame:Show()
		QuestFrame_SetAsLastShown(JQuestLogMoneyFrame, JQuestLogSpacerFrame)
		MoneyFrame_Update("JQuestLogMoneyFrame", money)
	end

	-- Hide unused rewards
	for i=(numRewards + numChoices + 1), MAX_NUM_ITEMS do
		getglobal("JQuestLogItem"..i):Hide()
	end

	local questItem
	local count = 0
	local id, name, texture, numItems, isUsable

	-- Setup choosable rewards
	if numChoices > 0 then
		JQuestLogItemChooseText:Show()
		QuestFrame_SetAsLastShown(JQuestLogItemChooseText, JQuestLogSpacerFrame)
		for i = 1, numChoices do
			id = string.match(questData.rewardInfo.choices[i][1], "item:(%d+)")
			isUsable = questData.rewardInfo.choices[i][2]
			name, _, _, _, _, _, _, numItems, _, texture = GetItemInfo(questData.rewardInfo.choices[i][1])
			questItem = getglobal("JQuestLogItem" .. i)
			questItem.itemId = id
			questItem:Show()
			QuestFrame_SetAsLastShown(questItem, JQuestLogSpacerFrame)
			getglobal("JQuestLogItem"..i.."Name"):SetText(name)
			SetItemButtonCount(questItem, numItems)
			SetItemButtonTexture(questItem, texture)
			if isUsable > 0 then
				SetItemButtonTextureVertexColor(questItem, 1.0, 1.0, 1.0)
				SetItemButtonNameFrameVertexColor(questItem, 1.0, 1.0, 1.0)
			else
				SetItemButtonTextureVertexColor(questItem, 0.9, 0, 0)
				SetItemButtonNameFrameVertexColor(questItem, 0.9, 0, 0)
			end
			if i > 1 then
				if mod(i,2) == 1 then
					questItem:SetPoint("TOPLEFT", "JQuestLogItem"..(i - 2), "BOTTOMLEFT", 0, -2)
				else
					questItem:SetPoint("TOPLEFT", "JQuestLogItem"..(i - 1), "TOPRIGHT", 1, 0)
				end
			else
				questItem:SetPoint("TOPLEFT", JQuestLogItemChooseText, "BOTTOMLEFT", -3, -5)
			end
			count = count + 1
		end
	else
		JQuestLogItemChooseText:Hide()
	end

	-- Setup spell rewards
	if questData.rewardInfo and questData.rewardInfo.spell then
		local spell = questData.rewardInfo.spell
		JQuestLogSpellLearnText:Show()
		QuestFrame_SetAsLastShown(JQuestLogSpellLearnText, JQuestLogSpacerFrame)

		--Anchor JQuestLogSpellLearnText if there were choosable rewards
		if count > 0 then
			JQuestLogSpellLearnText:SetPoint("TOPLEFT", "JQuestLogItem"..count, "BOTTOMLEFT", 3, -5)
		else
			JQuestLogSpellLearnText:SetPoint("TOPLEFT", JQuestLogRewardTitleText, "BOTTOMLEFT", 0, -5)
		end

		if spell.isTradeskillSpell then
			JQuestLogSpellLearnText:SetText(REWARD_TRADESKILL_SPELL)
		elseif ( not spell.isSpellLearned ) then
			JQuestLogSpellLearnText:SetText(REWARD_AURA)
		else
			JQuestLogSpellLearnText:SetText(REWARD_SPELL)
		end

		count = count + 1
		questItem = getglobal("JQuestLogItem"..count)
		questItem:Show()
		-- For the tooltip
		questItem.itemId = nil
		SetItemButtonCount(questItem, 0)
		SetItemButtonTexture(questItem, spell.texture)
		SetItemButtonTextureVertexColor(questItem, 1.0, 1.0, 1.0)
		SetItemButtonNameFrameVertexColor(questItem, 1.0, 1.0, 1.0)
		getglobal("JQuestLogItem"..count.."Name"):SetText(spell.name)
		questItem:SetPoint("TOPLEFT", JQuestLogSpellLearnText, "BOTTOMLEFT", -3, -5)
	else
		JQuestLogSpellLearnText:Hide()
	end

	-- Setup mandatory rewards
	if numRewards > 0 or money > 0 then
		local index
		if questData.rewardInfo.spell then
			JQuestLogItemReceiveText:SetText(REWARD_ITEMS)
			JQuestLogItemReceiveText:SetPoint("TOPLEFT", "JQuestLogItem"..count, "BOTTOMLEFT", 3, -5)
		elseif numChoices > 0 then
			JQuestLogItemReceiveText:SetText(REWARD_ITEMS)
			index = numChoices
			if mod(index, 2) == 0 then
				index = index - 1
			end
			JQuestLogItemReceiveText:SetPoint("TOPLEFT", "JQuestLogItem"..index, "BOTTOMLEFT", 3, -5)
		else 
			JQuestLogItemReceiveText:SetText(REWARD_ITEMS_ONLY)
			JQuestLogItemReceiveText:SetPoint("TOPLEFT", JQuestLogRewardTitleText, "BOTTOMLEFT", 3, -5)
		end
		JQuestLogItemReceiveText:Show()
		QuestFrame_SetAsLastShown(JQuestLogItemReceiveText, JQuestLogSpacerFrame)

		local baseIndex = count
		for i=1, numRewards do
			index = i + baseIndex
			id = string.match(questData.rewardInfo.rewards[i][1], "item:(%d+)")
			isUsable = questData.rewardInfo.rewards[i][2]
			name, _, _, _, _, _, _, numItems, _, texture = GetItemInfo(questData.rewardInfo.rewards[i][1])
			questItem = getglobal("JQuestLogItem" .. index)
			questItem.itemId = id
			questItem:Show()
			QuestFrame_SetAsLastShown(questItem, JQuestLogSpacerFrame)
			getglobal("JQuestLogItem"..index.."Name"):SetText(name)
			SetItemButtonCount(questItem, numItems)
			SetItemButtonTexture(questItem, texture)
			if isUsable > 0 then
				SetItemButtonTextureVertexColor(questItem, 1.0, 1.0, 1.0)
				SetItemButtonNameFrameVertexColor(questItem, 1.0, 1.0, 1.0)
			else
				SetItemButtonTextureVertexColor(questItem, 0.9, 0, 0)
				SetItemButtonNameFrameVertexColor(questItem, 0.9, 0, 0)
			end
			if i > 1 then
				if mod(i,2) == 1 then
					questItem:SetPoint("TOPLEFT", "JQuestLogItem"..(index - 2), "BOTTOMLEFT", 0, -2)
				else
					questItem:SetPoint("TOPLEFT", "JQuestLogItem"..(index - 1), "TOPRIGHT", 1, 0)
				end
			else
				questItem:SetPoint("TOPLEFT", JQuestLogItemReceiveText, "BOTTOMLEFT", -3, -5)
			end
			count = count + 1
		end
	else
		JQuestLogItemReceiveText:Hide()
	end

	if not doNotScroll then
		JQuestLogDetailScrollFrameScrollBar:SetValue(0)
	end
	JQuestLogDetailScrollFrame:UpdateScrollChildRect()

	JQuestLogDetailScrollFrame:Show()
end
function JQuest:GetPartyRewardInfo(rewardInfo)
	if not rewardInfo then
		return 0, 0, 0
	end
	local numRewards, numChoices, money

	if rewardInfo.rewards then
		numRewards = getn(rewardInfo.rewards)
	else
		numRewards = 0
	end
	if rewardInfo.choices then
		numChoices = getn(rewardInfo.choices)
	else
		numChoices = 0
	end
	if rewardInfo.money then
		money = rewardInfo.money
	else
		money = 0
	end
	return numRewards, numChoices, money
end
function JQuest:Log_TitleButton_OnEnter()
	-- Set highlight
	getglobal(this:GetName().."Tag"):SetTextColor(HIGHLIGHT_FONT_COLOR.r, HIGHLIGHT_FONT_COLOR.g, HIGHLIGHT_FONT_COLOR.b)

	-- Set group info tooltip
	if this.isHeader or (self:Party_GetNum() == 0 and not self.var.debugMode) then
		return
	end
	GameTooltip_SetDefaultAnchor(GameTooltip, this)
	GameTooltip:SetText(this.questName)
	GameTooltip:AddLine(HIGHLIGHT_FONT_COLOR_CODE.."Party members that are on this quest:"..FONT_COLOR_CODE_CLOSE)

	if self.PlayerQuests[this.questName] and self.PlayerQuests[this.questName].level == this.questLevel then
		GameTooltip:AddLine(LIGHTYELLOW_FONT_COLOR_CODE..UnitName("player")..FONT_COLOR_CODE_CLOSE)
	end

	for member, quest in pairs(self.PartyQuests) do
		for questName, data in pairs(quest) do
			if questName == this.questName and data.level == this.questLevel and member ~= UnitName("player") then
				GameTooltip:AddLine(LIGHTYELLOW_FONT_COLOR_CODE..member..FONT_COLOR_CODE_CLOSE)
			end
		end
	end
	GameTooltip:Show()
end
function JQuest:Log_TitleButton_OnLeave()
	if self.LogFrame.selectedQuest and self.LogFrame.selectedQuest ~= this.questName then
		getglobal(this:GetName().."Tag"):SetTextColor(this.r, this.g, this.b)
	end
	GameTooltip:Hide()
end
function JQuest:Log_TitleButton_OnClick(button)
	if IsShiftKeyDown() then
		-- Try to put it into chat
		if ChatFrameEditBox:IsVisible() then
			if this.questLink then
				ChatFrameEditBox:Insert(this.questLink)
			else
				ChatFrameEditBox:Insert(strtrim(this:GetText()))
			end
		elseif WIM_EditBoxInFocus then
			if this.questLink then
				WIM_EditBoxInFocus:Insert(this.questLink)
			else
				WIM_EditBoxInFocus:Insert(strtrim(this:GetText()))
			end
		elseif self.LogFrame.selectedWho and not this.isHeader then
			if self:QW_IsWatched(self.LogFrame.selectedWho, this.questName) then
				self:QW_Remove(self.LogFrame.selectedWho, this.questName)
			else
				if self:QW_GetNumBoards(self.LogFrame.selectedWho, this.questName) == 0 then
					UIErrorsFrame:AddMessage(QUEST_WATCH_NO_OBJECTIVES, 1.0, 0.1, 0.1, 1.0)
					return
				end
				if self:QW_GetNum() >= MAX_WATCHABLE_QUESTS then
					UIErrorsFrame:AddMessage(format(QUEST_WATCH_TOO_MANY, MAX_WATCHABLE_QUESTS), 1.0, 0.1, 0.1, 1.0)
					return
				end
				self:QW_Add(self.LogFrame.selectedWho, this.questName)
			end
		end
	end
	if not this.isHeader then
		self.LogFrame.selectedQuest = this.questName
		self:Log_RefreshList()
		self:Log_RefreshDetails()
	end
end
function JQuest:Log_CacheRewards(rewardInfo)
	if not rewardInfo then return end

	if rewardInfo.choices then
		for i=1, getn(rewardInfo.choices) do
			if not GetItemInfo(rewardInfo.choices[i][1]) then
				self:Log("Reward choice " .. i .. " not found, doing server lookup...")
				self.tt:SetHyperlink(rewardInfo.choices[i][1])
			end
		end
	end
	if rewardInfo.rewards then
		for i=1, getn(rewardInfo.rewards) do
			if not GetItemInfo(rewardInfo.rewards[i][1]) then
				self:Log("Reward item " .. i .. " not found, doing server lookup...")
				self.tt:SetHyperlink(rewardInfo.rewards[i][1])
			end
		end
	end
end
function JQuest:RewardItem_OnClick(button, itemId)
	if not itemId then return end
	local _, link = GetItemInfo(itemId)
	if IsControlKeyDown() and link then
		DressUpItemLink(link)
	elseif IsShiftKeyDown() and itemId then
		-- Try to put it into chat
		if ChatFrameEditBox:IsVisible() and link then
			ChatEdit_InsertLink(link)
		elseif WIM_EditBoxInFocus then
			WIM_EditBoxInFocus:Insert(link)
		end
	end
end

-- ********************
-- QuestWatch Functions
-- ********************
function JQuest:QW_GetNum()
	if self.watchedQuests then return getn(self.watchedQuests) end
	return 0
end
function JQuest:QW_GetNumBoards(who, title)
	if self.PartyQuests[who][title] then
		local q = self.PartyQuests[who][title]
		if q and q.leader then
			return getn(q.leader)
		else
			return 0
		end
	end
end
function JQuest:QW_IsWatched(who, title)
	for i = 1, self:QW_GetNum() do
		if self.watchedQuests[i][1] == who and self.watchedQuests[i][2] == title then
			if self.watchedQuests[i][3] then
				return self.watchedQuests[i][3]
			else
				return true
			end
		end
	end
	return false
end
function JQuest:QW_Add(who, title, duration)
	self:Log("QW_Add(): " .. who .. " - " .. title)
	if self:QW_IsWatched(who, title) then
		if duration then
			for i = 1, self:QW_GetNum() do
				if self.watchedQuests[i][1] == who and self.watchedQuests[i][2] == title then
					self.watchedQuests[i][3] = duration
					break
				end
			end
		end
	elseif self:QW_GetNum() < MAX_WATCHABLE_QUESTS then
			tinsert(self.watchedQuests, {who, title, duration})
	end
	if not self:IsEventScheduled("QW_CheckTimers") then
		self:ScheduleRepeatingEvent("QW_CheckTimers", self.QW_CheckTimers, 1, self)
	end
	if self.LogFrame.selectedWho and self.LogFrame.selectedWho == who then
		self:Log_RefreshList()
	end
	self:QW_Refresh()
end
function JQuest:QW_Remove(who, title)
	self:Log("QW_Remove(): " .. who .. " - " .. title)
	for i = 1, self:QW_GetNum() do
		if self.watchedQuests[i][1] == who and self.watchedQuests[i][2] == title then
			tremove(self.watchedQuests, i)
			if self.LogFrame.selectedWho and self.LogFrame.selectedWho == who then
				self:Log_RefreshList()
			end
			self:QW_Refresh()
			return
		end
	end
end
function JQuest:QW_CheckTimers()
	local i = 0
	while self:QW_GetNum() > 0 and i < self:QW_GetNum() do
		i = i + 1
		if self.watchedQuests[i] and self.watchedQuests[i][3] then
			if GetTime() >= self.watchedQuests[i][3] then
				self:QW_Remove(self.watchedQuests[i][1], self.watchedQuests[i][2])
				i = i - 1
			end
		end
	end
	if self:QW_GetNum() == 0 then
		self:CancelScheduledEvent("QW_CheckTimers")
		self:QW_Refresh()
	end
end
function JQuest:QW_Refresh()
	local who, questTitle, isComplete, numObjectives
	local watchLine, tempWidth, objectivesCompleted
	local watchLineIndex = 1
	local maxWidth = 0

	for i = 1, self:QW_GetNum() do
		who = self.watchedQuests[i][1]
		questTitle = self.watchedQuests[i][2]
		questData = self.PartyQuests[who][questTitle]

		if questData then
			numObjectives = getn(questData.leader)

			if numObjectives > 0 then
				watchLine = getglobal("JQuestWatchLine" .. watchLineIndex)
				watchLine:SetText("[" .. who .. "] " .. questTitle)
				tempWidth = watchLine:GetWidth()
				if watchLineIndex > 1 then
					watchLine:SetPoint("TOPLEFT", "JQuestWatchLine" .. (watchLineIndex - 1), "BOTTOMLEFT", 0, -4)
				end
				watchLine:Show()
				if tempWidth > maxWidth then
					maxWidth = tempWidth
				end
				watchLineIndex = watchLineIndex + 1
				objectivesCompleted = 0
				for j = 1, numObjectives do
					watchLine = getglobal("JQuestWatchLine" .. watchLineIndex)
					watchLine:SetText(" - " .. questData.leader[j].text)
					if questData.leader[j].status then
						watchLine:SetTextColor(HIGHLIGHT_FONT_COLOR.r, HIGHLIGHT_FONT_COLOR.g, HIGHLIGHT_FONT_COLOR.b)
						objectivesCompleted = objectivesCompleted + 1
					else
						watchLine:SetTextColor(0.8, 0.8, 0.8)
					end
					tempWidth = watchLine:GetWidth()
					if tempWidth > maxWidth then
						maxWidth = tempWidth
					end
					watchLine:SetPoint("TOPLEFT", "JQuestWatchLine" .. (watchLineIndex - 1), "BOTTOMLEFT", 0, 0)
					watchLine:Show()
					watchLineIndex = watchLineIndex + 1
				end

				-- Brighten the quest title if all the quest objectives were met
				watchLine = getglobal("JQuestWatchLine" .. watchLineIndex-numObjectives-1)
				if objectivesCompleted == numObjectives then
					watchLine:SetTextColor(NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b)
				else
					watchLine:SetTextColor(0.75, 0.61, 0)
				end
			end
		end
	end
	if watchLineIndex == 1 then
		JQuestWatchFrame:Hide()
		return
	else
		JQuestWatchFrame:Show()
		JQuestWatchFrame:SetHeight(watchLineIndex * 13)
		JQuestWatchFrame:SetWidth(maxWidth + 10)
	end

	-- Hide unused watch lines
	for i=watchLineIndex, 30 do
		getglobal("JQuestWatchLine"..i):Hide()
	end
end


function JQuest:QW_UpdateLock()
	if self.var.locked then
		JQuestWatchFrame:SetMovable(false)
		JQuestWatchFrame:EnableMouse(false)
	else
		JQuestWatchFrame:SetMovable(1)
		JQuestWatchFrame:EnableMouse(1)
	end
end
function JQuest:QW_OnDragStart()
	if not JQuest.var.locked then
		JQuest.var.reset = nil
		JQuestWatchFrame:StartMoving()
	end
end
function JQuest:QW_OnDragStop()
	JQuestWatchFrame:StopMovingOrSizing()
	JQuestWatchFrame.positionX = JQuestWatchFrame:GetRight()
	JQuestWatchFrame.positionY = JQuestWatchFrame:GetTop()
	JQuest:QW_Move()
end
function JQuest:QW_Move()
	if not self.var.reset then
		JQuestWatchFrame:ClearAllPoints()
		JQuestWatchFrame:SetPoint("TOPRIGHT", "UIParent", "BOTTOMLEFT", JQuestWatchFrame.positionX, JQuestWatchFrame.positionY)
		JQuest.var.questWatchX = JQuestWatchFrame.positionX
		JQuest.var.questWatchY = JQuestWatchFrame.positionY
	end
end
function JQuest:QW_ResetPos()
	JQuestWatchFrame:ClearAllPoints()
	JQuestWatchFrame:SetPoint("TOPRIGHT", QuestWatchFrame, "BOTTOMRIGHT", 0, -25)
	self.var.reset = true
end
function JQuest:QW_ClearAll()
	self.watchedQuests = {}
	self:QW_Refresh()
	self:Log_RefreshList()
end
function JQuest:QW_Init()
	JQuestWatchFrame:RegisterForDrag("LeftButton")
	JQuestWatchQuestName:SetText("JasonQuest")
	JQuestWatchQuestName:SetTextColor(1.0, 0.53, 0)

	JQuestWatchFrame:SetScript("OnDragStart", JQuest.QW_OnDragStart)
	JQuestWatchFrame:SetScript("OnDragStop", JQuest.QW_OnDragStop)
	JQuestWatchFrame:SetScript("OnHide", JQuest.QW_OnDragStop)

	JQuestWatchFrame.reset = self.var.reset or true
	JQuestWatchFrame.positionX = self.var.questWatchX or JQuestWatchFrame:GetRight()
	JQuestWatchFrame.positionY = self.var.questWatchY or JQuestWatchFrame:GetTop()

	self:QW_Move()
	self:QW_Refresh()
	self:QW_UpdateLock()
end

-- **********************
-- Aux QuestLog Functions
-- **********************
function JQuest:Log_GetNumQuests(who)
	if not who then who = self.LogFrame.selectedWho end
	local numQuests, numEntries = 0, 0
	for idx, data in pairs(self.PartyFrameData[who]) do
		if data.isHeader then
			numEntries = numEntries + 1
		else
			numQuests = numQuests + 1
			numEntries = numEntries + 1
		end
	end
	return numEntries, numQuests
end
function JQuest:Log_GetQuestDetails(listId, who)
	if not who then who = self.LogFrame.selectedWho end
	local o = self.PartyFrameData[who][listId]
	if o then
		if o.isHeader then
			return o.title, false, false, false, true
		else
			local q = self.PartyQuests[who][o.title]
			return o.title, q.level, q.tag or false, q.group, false, q.pushable or false, q.status or false, q.daily, q.questLink
		end
	end
end
function JQuest:Log_CreateSearchEditBox()
	-- Create Baggins Search EditBox
	local editBox = CreateFrame('EditBox', 'JQuest_SearchEditBox', JQuestLogFrame)
	editBox:SetWidth(100)
	editBox:SetHeight(24)

	editBox:SetFontObject(ChatFontNormal)
	editBox:SetTextInsets(8, 8, 0, 0)
	editBox:SetAutoFocus(false)

	editBox:SetBackdrop({edgeFile = "Interface/Tooltips/UI-Tooltip-Border", edgeSize = 16, insets = {left = 2, right = 2, top = 2, bottom = 2}})
	editBox:SetBackdropBorderColor(0.6, 0.6, 0.6)

	local background = editBox:CreateTexture(nil, "BACKGROUND")
	background:SetTexture("Interface/ChatFrame/ChatFrameBackground")
	background:SetPoint("TOPLEFT", 4, -4)
	background:SetPoint("BOTTOMRIGHT", -4, 4)
	background:SetGradientAlpha("VERTICAL", 0, 0, 0, 0.9, 0.2, 0.2, 0.2, 0.9)

	editBox:SetScript("OnHide", function() this:SetText(""); this.searchText = nil; JQuest_SearchLabel:Show() end)
	editBox:SetScript("OnEnterPressed", function() this:ClearFocus() end)
	editBox:SetScript("OnEscapePressed", function() this:SetText(""); this.searchText = nil; this:ClearFocus(); JQuest_SearchLabel:Show() end)
	editBox:SetScript("OnEditFocusGained", function() if IsControlKeyDown() then this:SetText(""); this.searchText = nil; this:ClearFocus(); JQuest_SearchLabel:Show() else JQuest_SearchLabel:Hide(); this:HighlightText() end end)
	editBox:SetScript("OnTextChanged", function() JQuest:Log_Search(this:GetText()) end)
	editBox:SetScript("OnEnter", function()
		GameTooltip_SetDefaultAnchor(GameTooltip, this)
		GameTooltip:SetText("JQuest Search")
		GameTooltip:AddLine("|c00FFFFFFv" .. self.version .. "|r")
		GameTooltip:Show()
		end)
	editBox:SetScript("OnLeave", function() GameTooltip:Hide() end)

	local label = editBox:CreateFontString("JQuest_SearchLabel", "OVERLAY", "GameFontHighlight")
	label:SetAlpha(0.2)
	label:SetText("Search")
	label:SetPoint("TOPLEFT", 8, 0)
	label:SetPoint("BOTTOMLEFT", -8, 0)
	label:Show()

	if not self.var.unmatchedAlpha then
		self.var.unmatchedAlpha = 0.2
	end
end
function JQuest:Log_UpdateSearchBoxPosition()
	JQuest_SearchEditBox:ClearAllPoints()
	JQuest_SearchEditBox:SetPoint("TOPRIGHT", JQuestLogFrame, "BOTTOMRIGHT", 0, 0)
	JQuest_SearchEditBox:SetWidth(JQuestLogFrame:GetWidth() - 10)
	if JQuestLogFrame:IsVisible() then
		JQuest_SearchEditBox:Show()
	else
		JQuest_SearchEditBox:Hide()
	end
end
function JQuest:Log_Search(searchText)
	if not searchText then
		JQuest_SearchEditBox.searchText = nil
	else
		JQuest_SearchEditBox.searchText = strlower(searchText)
	end
	JQuest:Log_RefreshList(1)
end
