﻿-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaTaxi - Handles character's taxi flights.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaTaxi-1.0", 0.1
local JambaTaxi = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaTaxi then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:Initialize
-- Initializes JambaTaxi.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Taxi", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Taxi", "Jamba-Taxi" )
				
end

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:GetOptionsConfiguration
-- Returns the configuration options table for JambaTaxi.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:GetOptionsConfiguration()

	if not self.configuration then
	
		self.configuration = {
			name = "Jamba-Taxi",
			handler = JambaTaxi,
			type = 'group',
			args = {	
				takeMastersTaxi = {
					type = "toggle",
					name = L["Take Master's Taxi"],
					desc = L["Take the same flight as the master did (slaves's must have NPC Flight Master window open)."],
					get = "CanTakeMastersTaxi",
					set = "ToggleTakeMastersTaxi",
					width = "full",
					order = 0,
				}, 
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 1,			
				},				
			},
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaTaxi.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:RefreshOptions
-- Refresh the JambaTaxi options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Taxi" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:PushSettingsToEnabledCharacters
-- Push the settings for JambaTaxi to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Taxi" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:ReceiveSettings
-- Receive settings for JambaTaxi.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:ReceiveSettings( settings )

	-- Set the new settings.
	self.db.takeMastersTaxi = settings.takeMastersTaxi
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi: CanTakeMastersTaxi
-- Return the take masters taxi flag.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:CanTakeMastersTaxi( info )

    return self.db.takeMastersTaxi
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi: ToggleTakeMastersTaxi
-- Toggle the take masters taxi flag.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:ToggleTakeMastersTaxi( info, value )

    self.db.takeMastersTaxi = value
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- TAXI PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:HookTakeTaxiNode
-- Hooked function - happens when the character has just taken a flight.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:HookTakeTaxiNode( taxiNodeIndex )

	-- If the take masters taxi option is on.
	if self.db.takeMastersTaxi then
		-- Get the name of the node flown to.
		local nodeName = TaxiNodeName( taxiNodeIndex )
		-- If this character is the master then tell the other characters about the taxi.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_TAKE_TAXI, nodeName )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTaxi:DoTakeTaxi
-- Called when another character has just taken flight.
-------------------------------------------------------------------------------------------------------------
function JambaTaxi:DoTakeTaxi( sender, nodeName )
	
	-- If the take masters taxi option is on.
	if self.db.takeMastersTaxi then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
		
			-- Find the index of the taxi node to fly to.
			local nodeIndex = nil
			for iterateNodes = 1, NumTaxiNodes() do
				if TaxiNodeName( iterateNodes ) == nodeName then
					nodeIndex = iterateNodes
					break
				end
			end
			
			-- If a node index was found...
			if nodeIndex then
				-- Take a taxi.
				TakeTaxiNode( nodeIndex )
			else
				-- Tell the master that could not take the same flight.
				self.comms:TellMaster( self.playerName, L["I cannot take the same flight as you.  I do not know where to go."] )				
			end
			
		end
	end
	
end
