-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaTalk - Handles character's whispers.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaTalk-1.0", 0.1
local JambaTalk = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaTalk then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:Initialize
-- Initializes JambaTalk.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Talk", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Talk", "Jamba-Talk" )

	-- Initialize static popups.
	self.InitializePopUps()	
	
	-- Remember the last sender to whisper this character.
	self.lastSender = nil

	-- A list of selected filter messages.
	self.selectedFilterMessages	= {}
	
	-- A list of selected chat snippets.
	self.selectedChatSnippets = {}
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:InitializePopUps
-- Initializes popups.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:InitializePopUps()

   -- Ask for part of a system message to filter on.
   StaticPopupDialogs["JAMBATALK_ASK_FILTER_MESSAGE"] = {
        text = L["Enter part of a system message to forward on:"],
        button1 = ACCEPT,
        button2 = CANCEL,
        hasEditBox = 1,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
		OnShow = function()
            local name = this:GetName()
            getglobal(name.."EditBox"):SetText("")
            getglobal(name.."Button1"):Disable()
            getglobal(name.."EditBox"):SetFocus()
        end,		
        OnAccept = function()
		    local filterMessage = getglobal(this:GetParent():GetName().."EditBox"):GetText()
			JambaTalk:AddFilterMessage( nil, filterMessage )
		end,
        EditBoxOnTextChanged = function()
            local name = this:GetParent():GetName()
            local button = getglobal(name.."Button1")
            local filterMessage = getglobal(this:GetParent():GetName().."EditBox"):GetText()
            if not filterMessage or filterMessage:trim() == "" then
				button:Disable()
            else
                button:Enable()
            end
        end,		
		EditBoxOnEnterPressed = function()
            if getglobal(this:GetParent():GetName().."Button1"):IsEnabled() then
                local filterMessage = getglobal(this:GetParent():GetName().."EditBox"):GetText()
                JambaTalk:AddFilterMessage( nil, filterMessage )
            end
            this:GetParent():Hide()
        end,		
    }

   -- Confirm removing filter messages.
   StaticPopupDialogs["JAMBATALK_CONFIRM_REMOVE_FILTER_MESSAGE"] = {
        text = L["Are you sure you wish to remove the selected forward messages?"],
        button1 = ACCEPT,
        button2 = CANCEL,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
        OnAccept = function()
			JambaTalk:RemoveFilterMessage()
		end,
    }

   -- Ask for a chat snippet.
   StaticPopupDialogs["JAMBATALK_ASK_SNIPPET"] = {
        text = L["Enter a chat snippet in the form name=text.  (e.g. no=Sorry, I am busy right now.) :"],
        button1 = ACCEPT,
        button2 = CANCEL,
        hasEditBox = 1,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
		OnShow = function()
            local name = this:GetName()
            getglobal(name.."EditBox"):SetText("")
            getglobal(name.."Button1"):Disable()
            getglobal(name.."EditBox"):SetFocus()
        end,		
        OnAccept = function()
		    local chatSnippet = getglobal(this:GetParent():GetName().."EditBox"):GetText()
			JambaTalk:AddSnippet( nil, chatSnippet )
		end,
        EditBoxOnTextChanged = function()
            local name = this:GetParent():GetName()
            local button = getglobal(name.."Button1")
            local chatSnippet = getglobal(this:GetParent():GetName().."EditBox"):GetText()
            if not chatSnippet:find( "%w+=.+" ) then
				button:Disable()
            else
                button:Enable()
            end
        end,		
		EditBoxOnEnterPressed = function()
            if getglobal(this:GetParent():GetName().."Button1"):IsEnabled() then
                local chatSnippet = getglobal(this:GetParent():GetName().."EditBox"):GetText()
                JambaTalk:AddSnippet( nil, chatSnippet )
            end
            this:GetParent():Hide()
        end,
    }

   -- Confirm removing chat snippets.
   StaticPopupDialogs["JAMBATALK_CONFIRM_REMOVE_CHAT_SNIPPET"] = {
        text = L["Are you sure you wish to remove the selected chat snippets?"],
        button1 = ACCEPT,
        button2 = CANCEL,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
        OnAccept = function()
			JambaTalk:RemoveSnippet()
		end,
    }
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:GetOptionsConfiguration
-- Returns the configuration options table for JambaTalk.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:GetOptionsConfiguration()

	if not self.configuration then
	
		self.configuration = {
			name = "Jamba-Talk",
			handler = JambaTalk,
			type = "group",
			childGroups  = "tab",
			args = {		
				groupConfiguration = {
					name = L["Options"],
					handler = JambaTalk,
					type = "group",
					order = 0,
					args = {	
						forwardWhispers = {
							type = "toggle",
							name = L["Forward Whispers And Relay"],
							desc = L["Forward whispers to character to the master and relay replies back."],
							get = "CanForwardWhispers",
							set = "ToggleForwardWhispers",
							width = "full",
							order = 0,
						},        
						forwardWhispersInformationA = {
							type = "description",
							name = L["Whisper forwarding will forward a whisper directed to a slave to the master.  Reply to the slave from the master and the slave will reply to the original whisper with the masters reply."],
							order = 1,
							cmdHidden = true,
						},							
						forwardWhispersInformationB = {
							type = "description",
							name = L["For commonly used replies, set up a chat snippet on the slave to save typing.  To send a chat snippet, use a ! followed by the name of the chat snippet."],
							order = 2,
							cmdHidden = true,
						},							
						forwardWhispersInformationC = {
							type = "description",
							name = L["To direct a whisper to another character from the slave, begin the whisper to the slave with @ followed by the name of the character."],
							order = 3,
							cmdHidden = true,
						},							
						requestWhoInformation = {
							type = "toggle",
							name = L["Request Who Information"],
							desc = L["Request /who information from server about character that whispered.  To view on master, enable forwarding of system messages."],
							get = "CanRequestWhoInformation",
							set = "ToggleRequestWhoInformation",
							disabled = function() return not JambaTalk:CanForwardWhispers() end,
							width = "full",
							order = 4,
						},   
						requestWhoInformationInformationA = {
							type = "description",
							name = L["With this option and the forwarding system messages option, the master receives /who information about the whisperer and a clickable link to the orginal whisperer."],
							order = 5,
							cmdHidden = true,
						},
						forwardSystemMessages = {
							type = "toggle",
							name = L["Forward System Messages"],
							desc = L["Forward system messages (like /who information) to master."],
							get = "CanForwardSystemMessages",
							set = "ToggleForwardSystemMessages",
							width = "full",
							order = 6,
						}, 
						forwardSystemMessagesInformationA = {
							type = "description",
							name = L["Forwarding system messages will forward the specifed system messages (yellow text) to the master.  To specify a message to be forwarded, you can add a part of the message to forward to the forward message list."],
							order = 7,
							cmdHidden = true,
						},						  
					},  
				},  
				groupFilterSystemMessages = {
					name = L["Forward Messages"],
					handler = JambaTalk,
					type = "group",
					order = 1,
					args = {				 
						systemMessagesToFilter = {
							type = "multiselect",
							name = L["System Messages To Forward"],
							desc = L["System messages to forward."],
							values = self.db.systemMessagesToFilter,
							get = "IsFilterMessageSelected",
							set = "ToggleFilterMessageSelected",
							width = "full",
							order = 4,					
							cmdHidden = true,
						},			
						addSystemMessageToFilterGui = {
							type = "execute",
							name = L["Add"],
							desc = L["Add a partial system message to forward."],
							func = "AddFilterMessageGui",
							width = "half",
							order = 5,					
							cmdHidden = true,
						},					
						removeSystemMessageToFilterGui = {
							type = "execute",
							name = L["Remove"],
							desc = L["Remove selected partial system messages."],
							func = "RemoveFilterMessageGui",
							width = "half",
							order = 6,					
							cmdHidden = true,
						},						
						addMessage = {
							type = "execute",
							name = L["Add"],
							desc = L["Add a partial system message to forward."],
							func = "AddFilterMessage",
							order = 100,					
							guiHidden = true,
						},	
						removeMessage = {
							type = "execute",
							name = L["Remove"],
							desc = L["Remove a partial system message to forward."],
							func = "RemoveFilterMessage",
							order = 101,					
							guiHidden = true,
						},	
					},
				},		
				groupChatSnippets = {
					name = L["Chat Snippets"],
					handler = JambaTalk,
					type = "group",
					order = 2,
					args = {				
						chatSnippets = {
							type = "multiselect",
							name = L["Chat Snippets"],
							desc = L["Snippets of text to send and their associated shortcut text."],
							values = self.db.chatSnippets,
							get = "IsChatSnippetSelected",
							set = "ToggleChatSnippetSelected",
							width = "full",
							order = 7,					
							cmdHidden = true,
						},
						addSnippetToFilterGui = {
							type = "execute",
							name = L["Add"],
							desc = L["Add a snippet of chat text."],
							func = "AddSnippetGui",
							width = "half",
							order = 8,					
							cmdHidden = true,
						},					
						removeSnippetToFilterGui = {
							type = "execute",
							name = L["Remove"],
							desc = L["Remove a snippet of chat text."],
							func = "RemoveSnippetGui",
							width = "half",
							order = 9,					
							cmdHidden = true,
						},						
						addSnippet = {
							type = "execute",
							name = L["Add"],
							desc = L["Add a snippet of chat text."],
							func = "AddSnippet",
							order = 102,					
							guiHidden = true,
						},	
						removeSnippet = {
							type = "execute",
							name = L["Remove"],
							desc = L["Remove a snippet of chat text."],
							func = "RemoveSnippet",
							order = 103,					
							guiHidden = true,
						},					
					},
				},
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 3,			
				},	
			},				
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaTalk.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:RefreshOptions
-- Refresh the JambaTalk options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Talk" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:PushSettingsToEnabledCharacters
-- Push the settings for JambaTalk to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Talk" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:ReceiveSettings
-- Receive settings for JambaTalk.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ReceiveSettings( settings )

	-- Set the new settings.
	self.db.forwardWhispers = settings.forwardWhispers
    self.db.requestWhoInformation = settings.requestWhoInformation
	self.db.forwardSystemMessages = settings.forwardSystemMessages
    self.utils:ClearTable( self.db.systemMessagesToFilter )
	self.utils:ClearTable( self.selectedFilterMessages )
	for messageKey, messageValue in pairs( settings.systemMessagesToFilter ) do
		self.db.systemMessagesToFilter[messageKey] = messageValue
	end
	self.utils:ClearTable( self.db.chatSnippets )
	self.utils:ClearTable( self.selectedChatSnippet )
	for snippetKey, snippetValue in pairs( settings.chatSnippets ) do
		self.db.chatSnippets[snippetKey] = snippetValue
	end
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: CanForwardWhispers
-- Return the forward whispers flag.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:CanForwardWhispers( info )

    return self.db.forwardWhispers
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: ToggleForwardWhispers
-- Toggle the forward whispers flag.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ToggleForwardWhispers( info, value )

    self.db.forwardWhispers = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: CanRequestWhoInformation
-- Return the request who information flag.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:CanRequestWhoInformation( info )

    return self.db.requestWhoInformation
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: ToggleRequestWhoInformation
-- Toggle the request who information flag.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ToggleRequestWhoInformation( info, value )

    self.db.requestWhoInformation = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: CanForwardSystemMessages
-- Return the forward system messages flag.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:CanForwardSystemMessages( info )

    return self.db.forwardSystemMessages
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: ToggleForwardSystemMessages
-- Toggle the forward system messages flag.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ToggleForwardSystemMessages( info, value )

    self.db.forwardSystemMessages = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: IsFilterMessageSelected
-- Return if a filter message is selected or not.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:IsFilterMessageSelected( info, key )

	local message = key
	local selected = false
	
	if message then
		if self.selectedFilterMessages[message] then
			selected = true
		end
	end
	
	return selected
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: ToggleFilterMessageSelected
-- Toggle a filter message selection state.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ToggleFilterMessageSelected( info, key, state )

	local message = key
	local isSelected = state
	
	if message then
		if isSelected then
			self.selectedFilterMessages[message] = true
		else
			self.selectedFilterMessages[message] = nil
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: AddFilterMessageGui
-- Provides a GUI for a user to enter a filter message which is added to the filter message list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:AddFilterMessageGui()
	
	StaticPopup_Show( "JAMBATALK_ASK_FILTER_MESSAGE" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: AddFilterMessage
-- Add a filter message to the filter message list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:AddFilterMessage( info, value )
	
	local message = value
	
	-- If the message is not nil then...
	if message then
		-- Add to the message list.
		self.db.systemMessagesToFilter[message] = message
	end
	
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: RemoveFilterMessageGui
-- Provides a GUI for a user to confirm removing selected filter messages from the filter message list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:RemoveFilterMessageGui()
	
	StaticPopup_Show( "JAMBATALK_CONFIRM_REMOVE_FILTER_MESSAGE" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: RemoveFilterMessage
-- Remove selected filter messages from the filter message list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:RemoveFilterMessage()

	for message, enabled in pairs( self.selectedFilterMessages ) do
		self.db.systemMessagesToFilter[message] = nil
		self.selectedFilterMessages[message] = nil
	end
	
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: IsChatSnippetSelected
-- Return if a chat snippet is selected or not.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:IsChatSnippetSelected( info, key )

	local snippet = key
	local selected = false
	
	if snippet then
		if self.selectedChatSnippets[snippet] then
			selected = true
		end
	end
	
	return selected
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: ToggleChatSnippetSelected
-- Toggle a chat snippet selection state.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ToggleChatSnippetSelected( info, key, state )

	local snippet = key
	local isSelected = state
	
	if snippet then
		if isSelected then
			self.selectedChatSnippets[snippet] = true
		else
			self.selectedChatSnippets[snippet] = nil
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: AddSnippetGui
-- Provides a GUI for a user to enter a snippet which is added to the snippet list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:AddSnippetGui()
	
	StaticPopup_Show( "JAMBATALK_ASK_SNIPPET" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: AddSnippet
-- Add a snippet to the snippet list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:AddSnippet( info, value )
	
	local snippet = value
	
	-- If the snippet is not nil then...
	if snippet then
		-- Add to the snippet list.
		self.db.chatSnippets[snippet] = snippet
	end
	
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: RemoveSnippetGui
-- Provides a GUI for a user to confirm removing selected snippets from the snippet list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:RemoveSnippetGui()
	
	StaticPopup_Show( "JAMBATALK_CONFIRM_REMOVE_CHAT_SNIPPET" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk: RemoveSnippet
-- Remove selected snippets from the snippet list.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:RemoveSnippet()
	
	for snippet, enabled in pairs( self.selectedChatSnippets ) do
		self.db.chatSnippets[snippet] = nil
		self.selectedChatSnippets[snippet] = nil
	end
	
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- WHISPER PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:ProcessWhisperReceived
-- Process the whisper received event.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ProcessWhisperReceived()

	-- Does this character forward whispers?
	if self.db.forwardWhispers then
		
		-- Yes, get the relevant information from the event arguments.
		local message = arg1
		local sender = arg2
		local flag = arg6
		
		-- Set a GM flag if this whisper was from a GM.
		local isGM = false
		if flag == L["GM"] then
			isGM = true
		end

		-- Was the sender the master?
		if self.comms:CharacterIsTheMaster( sender ) then
			-- Yes, relay the masters message to others.
			self:ForwardWhisperFromMaster( message )
		else		
			-- Not the master, forward the whisper to the master.
			self:ForwardWhisperToMaster( message, sender, isGM )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:ProcessSystemMessageReceived
-- Process the system message received event.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ProcessSystemMessageReceived()

	-- Does this character forward system messages?
	if self.db.forwardSystemMessages then

		-- Yes, get the relevant information from the event arguments.
		local message = arg1

		if message then
			-- If not the master then...
			if not self.comms:CharacterIsTheMaster( self.playerName ) then
				-- Check to see if this message should be filtered through.
				local filterMessage = false
				-- For each filter message...
				for filterKey, filterValue in pairs( self.db.systemMessagesToFilter ) do
					-- See if this filter message is in the system message (a plain substring search, no pattern matches).				
					if message:find( filterValue, 1, true ) then
						filterMessage = true
						break
					end
				end
				-- If the message is filtered then...
				if filterMessage then
					-- Tell the master.
					self.comms:TellMasterSpecial( self.playerName, message )
				end
			end
		end
		
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:ForwardWhisperToMaster
-- Forward a whisper to the master.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ForwardWhisperToMaster( message, sender, isGM )

	-- From who and forward message?
	local fromCharacter = sender
	local forwardMessage = true
	
	-- Don't relay messages to the master or self (causes infinite loop, which causes disconnect).
	if (self.comms:CharacterIsTheMaster( self.playerName )) or (self.playerName == sender) then
		forwardMessage = false
	end
	
	-- Don't relay messages from the master either (not that this situation should happen).
	if self.comms:CharacterIsTheMaster( sender ) then
		forwardMessage = false
	end

	-- Allowed to forward the message?	
	if forwardMessage then
	
		-- Send who information to the master?
		if self.db.requestWhoInformation then
			-- Request who information from wow.
			SendWho( sender )
		end
		
		-- Set the from character information.
		if isGM then
			fromCharacter = L["<GM>"]..fromCharacter
		end
		
		-- Whisper the master.
		ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_BULK, nil, fromCharacter..": "..message, "WHISPER", nil, self.comms:GetMasterName() )		
		-- Remember this sender as the most recent sender.
		self.lastSender = sender
		
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:ForwardWhisperFromMaster
-- Forward a whisper from the master to another character.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:ForwardWhisperFromMaster( messageFromMaster )

	-- Who to send to and what to send?
	local sendTo 
	local messageToInspect
	local snippetName
	local messageToSend
	
	-- Check the message to see if there is a character to whisper to; character name is preceeded by @.
	-- No match will return nil for the parameters.
	sendTo, messageToInspect = select( 3, messageFromMaster:find( "^@(%w+)%s*(.*)$" ) )
		
	-- If no sender found in message...
	if not sendTo then
		-- Then send to last sender.
		sendTo = self.lastSender
		-- Send the full message.
		messageToInspect = messageFromMaster
	end

	-- Check to see if there is a snippet name in the message (text with a leading !).
	snippetName = select( 3, messageToInspect:find( "^!(%w+)$" ) )
	-- If a snippet name was found...
	if snippetName then
		-- Then look up the associated text.
		messageToSend = self:GetTextForSnippet( snippetName )
	else
		-- Otherwise, use the message as it is.
		messageToSend = messageToInspect
	end
	
	-- If there is a valid character to send to...
	if sendTo then
		if messageToSend:trim() ~= "" then
			-- Send the message.
			ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_BULK, nil, messageToSend, "WHISPER", nil, sendTo )		
		end
		-- Remember this sender as the most recent sender.
		self.lastSender = sendTo
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaTalk:GetTextForSnippet
-- Look up text from a snippet name.
-------------------------------------------------------------------------------------------------------------
function JambaTalk:GetTextForSnippet( snippetName )

	local snippet = ""
	
	for snippetKey, snippetValue in pairs( self.db.chatSnippets ) do
		local tempSnippetName, tempSnippetText = select( 3, snippetValue:find( "(%w+)=(.+)" ) )
		if tempSnippetName == snippetName then
			snippet = tempSnippetText
			break
		end
	end
	
	return snippet
	
end
