﻿-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaQuest - Handles character's questing.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaQuest-1.0", 0.1
local JambaQuest = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaQuest then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:Initialize
-- Initializes JambaQuest.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Quest", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Quest", "Jamba-Quest" )

	-- The name of the last quest to be abandoned.
	self.currentSelectedQuest = ""

	-- The name of the last quest that a select command was sent for.
	self.lastQuestSelectedCommandSent = ""
	
	-- Jamba quest log frame.
	self.jambaQuestLogFrame = nil
	self.jambaQuestLogFrameQuestName = nil
	self.jambaQuestRemoveWatchButton = nil
	self.jambaQuestAddWatchButton = nil
	self.jambaQuestAbandonButton = nil
	self.jambaQuestSelectButton = nil

	JambaQuest:CreateJambaQuestLogFrame()
	
end
		
-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CreateJambaQuestLogFrame
-- Create the quest log frame.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CreateJambaQuestLogFrame()

	-- The frame.
	local frame = CreateFrame( "Frame", "JambaQuestLogWindowFrame", UIParent )
	frame.obj = self
	frame:SetWidth( 256 )
	frame:SetHeight( 128 )
	frame:SetFrameStrata( "DIALOG" )
	frame:SetToplevel( true )
	frame:SetClampedToScreen( true )
	frame:EnableMouse()
	frame:SetMovable( true )	
	frame:ClearAllPoints()
	frame:SetPoint( "CENTER", UIParent )
	frame:SetUserPlaced( true )
	frame:RegisterForDrag( "LeftButton" )
	frame:SetScript( "OnDragStart", 
		function( this ) 
			this:StartMoving() 
		end )
	frame:SetScript( "OnDragStop", 
		function( this ) 
			this:StopMovingOrSizing() 
		end	)

	-- The pretty picture.
	local border = frame:CreateTexture( nil, "OVERLAY" )
	border:SetWidth( 256 )
	border:SetHeight( 128 )
	border:SetPoint( "TOPLEFT", frame, "TOPLEFT", 0, 0 )
	local texname = border:SetTexture( "Interface\\AddOns\\Jamba\\Textures\\JambaQuestFrame" )

	-- The frame title.
	local title = frame:CreateFontString( nil, "OVERLAY", "GameFontNormal" )
	title:SetPoint( "TOP", frame, "TOP", 0, -8 )
	title:SetTextColor( 1.00, 1.00, 1.00 )
	title:SetText( L["Jamba-Quest"] )

	-- The select quest name.
	local label = frame:CreateFontString( nil, "OVERLAY", "GameFontNormal" )
	label:SetPoint( "TOPLEFT", frame, "TOPLEFT", 23, -43 )
	label:SetWidth( 210 )
	label:SetHeight( 20 )
	label:SetJustifyH( "CENTER" )
	label:SetJustifyV( "TOP" )
	label:SetText( L["(No Quest Selected)"] )

	-- Remember the place to put the name.
	self.jambaQuestLogFrameQuestName = label		

	-- The close button.
	local closeButton = CreateFrame( "Button", nil, frame, "UIPanelCloseButton" )
	closeButton:SetScript( "OnClick", JambaQuestCloseButtonClicked )
	closeButton:SetPoint( "TOPRIGHT", frame, "TOPRIGHT", 1, 1 )
	closeButton.obj = self

	-- The select button.
	local selectButton = CreateFrame( "Button", nil, frame, "UIPanelButtonTemplate" )
	selectButton.obj = self
	selectButton:SetScript( "OnClick", JambaQuestSelectButtonClicked )
	selectButton:SetPoint( "TOPLEFT", frame, "TOPLEFT", 16, -72 )
	selectButton:SetHeight( 20 )
	selectButton:SetWidth( 110 )
	selectButton:SetText( L["Select"] )		
	self.jambaQuestSelectButton = selectButton
	
	-- The abandon button.		
	local abandonButton = CreateFrame( "Button", nil, frame, "UIPanelButtonTemplate" )
	abandonButton:SetScript( "OnClick", JambaQuestAbandonButtonClicked )
	abandonButton:SetPoint( "TOPLEFT", frame, "TOPLEFT", 16, -94 )
	abandonButton:SetHeight( 20 )
	abandonButton:SetWidth( 110 )
	abandonButton:SetText( L["Abandon"] )		
	abandonButton.obj = self
	self.jambaQuestAbandonButton = abandonButton
		
	-- The add watch button.
	local addWatchButton = CreateFrame( "Button", nil, frame, "UIPanelButtonTemplate" )
	addWatchButton:SetScript( "OnClick", JambaQuestAddWatchButtonClicked )
	addWatchButton:SetPoint( "TOPLEFT", frame, "TOPLEFT", 130, -72 )
	addWatchButton:SetHeight( 20 )
	addWatchButton:SetWidth( 110 )
	addWatchButton:SetText( L["Add Watch"] )		
	addWatchButton.obj = self
	self.jambaQuestAddWatchButton = addWatchButton
	
	-- The remove watch button.	
	local removeWatchButton = CreateFrame( "Button", nil, frame, "UIPanelButtonTemplate" )
	removeWatchButton:SetScript( "OnClick", JambaQuestRemoveWatchButtonClicked )
	removeWatchButton:SetPoint( "TOPLEFT", frame, "TOPLEFT", 130, -94 )
	removeWatchButton:SetHeight( 20 )
	removeWatchButton:SetWidth( 110 )
	removeWatchButton:SetText( L["Remove Watch"] )		
	removeWatchButton.obj = self
	self.jambaQuestRemoveWatchButton = removeWatchButton

	self.jambaQuestLogFrame = frame

	-- Show or hide the form as it last was.
	self:ToggleShowQuestCommandWindow( nil, self.db.show )

end


-------------------------------------------------------------------------------------------------------------
-- JambaQuestCloseButtonClicked
-- Fires when the close button is clicked.
-------------------------------------------------------------------------------------------------------------
function JambaQuestCloseButtonClicked( this, button, down )

	JambaQuest:ToggleShowQuestCommandWindow( nil, false )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuestSelectButtonClicked
-- Fires when the select button is clicked.
-------------------------------------------------------------------------------------------------------------
function JambaQuestSelectButtonClicked( this, button, down )

	JambaQuest:SendSelectQuestCommand()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuestAbandonButtonClicked
-- Fires when the abandon button is clicked.
-------------------------------------------------------------------------------------------------------------
function JambaQuestAbandonButtonClicked( this, button, down )

	JambaQuest:SendAbandonQuestCommand()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuestAddWatchButtonClicked
-- Fires when the add watch button is clicked.
-------------------------------------------------------------------------------------------------------------
function JambaQuestAddWatchButtonClicked( this, button, down )

	JambaQuest:SendAddWatchQuestCommand()

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuestRemoveWatchButtonClicked
-- Fires when the remove watch button is clicked.
-------------------------------------------------------------------------------------------------------------
function JambaQuestRemoveWatchButtonClicked( this, button, down )

	JambaQuest:SendRemoveWatchQuestCommand()

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:GetOptionsConfiguration
-- Returns the configuration options table for JambaQuest.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:GetOptionsConfiguration()

	if not self.configuration then

		self.configuration = {
			name = "Jamba-Quest",
			handler = JambaQuest,
			type = 'group',
			childGroups  = "tab",
			args = {					
				headerQuestNpcSafe = {
					type = "header",
					name = L["Quest NPC (Safe)"],
					order = 0,
				},
				mirrorMasterQuestSelection = {
					type = "toggle",
					name = L["Mirror Master Quest Selection"],
					desc = L["Slaves follow the same steps as the master for quest selection.  All toons must have the quest dialog open."],
					get = "CanMirrorMasterQuestSelection",
					set = "ToggleMirrorMasterQuestSelection",
					width = "full",
					order = 1,
				}, 			
				mirrorMasterCompleteQuestNoReward = {
					type = "toggle",
					name = L["Mirror Master Complete Quest (No Reward Choice)"],
					desc = L["Slaves follow the same steps as the master for completing quests that have no reward choices.  All toons must have the quest dialog open."],
					get = "CanMirrorMasterCompleteQuestNoReward",
					set = "ToggleMirrorMasterCompleteQuestNoReward",
					disabled = function() return JambaQuest:CanSlaveAutoCompleteQuest() end,
					width = "full",
					order = 2,
				}, 
				mirrorMasterCompleteQuestWithReward = {
					type = "toggle",
					name = L["Mirror Master Complete Quest (WITH Reward Choice)"],
					desc = L["Slaves follow the same steps as the master for completing quests that have a reward choice.  All toons must have the quest dialog open."],
					get = "CanMirrorMasterCompleteQuestWithReward",
					set = "ToggleMirrorMasterCompleteQuestWithReward",
					width = "full",
					order = 3,
				},				
				mirrorMasterDeclineQuest = {
					type = "toggle",
					name = L["Mirror Master Decline Quest"],
					desc = L["Slaves follow the same steps as the master for declining quests.  All toons must have the quest dialog open."],
					get = "CanMirrorMasterDeclineQuest",
					set = "ToggleMirrorMasterDeclineQuest",
					width = "full",
					order = 4,
				}, 
				headerQuestLogSafe = {
					type = "header",
					name = L["Quest Log (Safe)"],
					order = 5,
				},
				mirrorMasterAbandonQuest = {
					type = "toggle",
					name = L["Mirror Master Abandon Quest"],
					desc = L["Slaves follow the same steps as the master for abandoning quests.  All toons must have the quest dialog open."],
					get = "CanMirrorMasterAbandonQuest",
					set = "ToggleMirrorMasterAbandonQuest",
					width = "full",
					order = 6,
				}, 
				mirrorMasterQuestWatch = {
					type = "toggle",
					name = L["Mirror Master Quest Watch"],
					desc = L["Slaves follow the same steps as the master for watching quests."],
					get = "CanMirrorMasterQuestWatch",
					set = "ToggleMirrorMasterQuestWatch",
					width = "full",
					order = 7,
				},				
				headerQuestNpcAdvanced = {
					type = "header",
					name = L["Quest NPC (Advanced)"],
					order = 8,
				},	
				slaveAutoAcceptEscortQuest = {
					type = "toggle",
					name = L["Slave Auto Accept Escort Quest"],
					desc = L["Slave automatically accepts escort quests."],
					get = "CanSlaveAutoAcceptEscortQuest",
					set = "ToggleSlaveAutoAcceptEscortQuest",
					width = "full",
					order = 9,
				},							
				slaveAutoAcceptQuest = {
					type = "toggle",
					name = L["Slave Auto Accept ANY Quests"],
					desc = L["Slave automatically accepts quests when interacting with an npc or when a quest is shared with them."],
					get = "CanSlaveAutoAcceptQuest",
					set = "ToggleSlaveAutoAcceptQuest",
					width = "full",
					order = 10,
				},  				
				slaveAutoCompleteQuest = {
					type = "toggle",
					name = L["Slave Auto Complete Quest (No Reward Choice)"],
					desc = L["Slave automatically completes quests that have no reward choices."],
					get = "CanSlaveAutoCompleteQuest",
					set = "ToggleSlaveAutoCompleteQuest",
					disabled = function() return JambaQuest:CanMirrorMasterCompleteQuestNoReward() end,
					width = "full",
					order = 11,
				}, 	
				headerOtherOptions = {
					type = "header",
					name = L["Other Options"],
					order = 12,
				},
				showJambaQuestLogWithWowLog = {
					type = "toggle",
					name = L["Show Jamba-Quest Log With WoW Quest Log"],
					desc = L["Show the Jamba-Quest log with the normal WoW quest log."],
					get = "CanShowJambaQuestLogWithWowLog",
					set = "ToggleShowJambaQuestLogWithWowLog",
					width = "full",
					order = 13,
				},
				questSilently = {
					type = "toggle",
					name = L["Quest Silently"],
					desc = L["Do not tell the master about this toon's quests."],
					get = "CanQuestSilently",
					set = "ToggleQuestSilently",
					width = "full",
					order = 14,
				},
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 15,			
				},	
				show = {
					type = "toggle",
					name = L["Show Quest Command"],
					desc = L["Show or hide the quest command window."],
					get = "CanShowQuestCommandWindow",
					set = "ToggleShowQuestCommandWindow",
					guiHidden = true,
				},										
			},
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaQuest.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:RefreshOptions
-- Refresh the JambaQuest options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Quest" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:PushSettingsToEnabledCharacters
-- Push the settings for JambaQuest to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Quest" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ReceiveSettings
-- Receive settings for JambaQuest.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ReceiveSettings( settings )

	-- Set the new settings.
	self.db.mirrorMasterQuestSelection = settings.mirrorMasterQuestSelection
	self.db.slaveAutoAcceptQuest = settings.slaveAutoAcceptQuest
	self.db.slaveAutoAcceptEscortQuest = settings.slaveAutoAcceptEscortQuest
	self.db.mirrorMasterDeclineQuest = settings.mirrorMasterDeclineQuest
	self.db.mirrorMasterAbandonQuest = settings.mirrorMasterAbandonQuest
	self.db.mirrorMasterCompleteQuestNoReward = settings.mirrorMasterCompleteQuestNoReward
	self.db.mirrorMasterCompleteQuestWithReward = settings.mirrorMasterCompleteQuestWithReward
	self.db.slaveAutoCompleteQuest = settings.slaveAutoCompleteQuest
	self.db.mirrorMasterQuestWatch = settings.mirrorMasterQuestWatch
	self.db.questSilently = settings.questSilently
	self.db.showJambaQuestLogWithWowLog = settings.showJambaQuestLogWithWowLog
	                	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanMirrorMasterQuestSelection
-- Return the mirror master quest selection flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanMirrorMasterQuestSelection( info )

    return self.db.mirrorMasterQuestSelection
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleMirrorMasterQuestSelection
-- Toggle the mirror master quest selection flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleMirrorMasterQuestSelection( info, value )

    self.db.mirrorMasterQuestSelection = value
    
    if self.db.mirrorMasterQuestSelection == true then
		self.jambaQuestSelectButton:Enable()
	else
		self.jambaQuestSelectButton:Disable()
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanSlaveAutoAcceptQuest
-- Return the slave auto accept quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanSlaveAutoAcceptQuest( info )

    return self.db.slaveAutoAcceptQuest
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleSlaveAutoAcceptQuest
-- Toggle the slave auto accept quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleSlaveAutoAcceptQuest( info, value )

    self.db.slaveAutoAcceptQuest = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest: CanSlaveAutoAcceptEscortQuest
-- Return the slave auto accept escort quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanSlaveAutoAcceptEscortQuest( info )

    return self.db.slaveAutoAcceptEscortQuest
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleSlaveAutoAcceptEscortQuest
-- Toggle the slave auto accept escort quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleSlaveAutoAcceptEscortQuest( info, value )

    self.db.slaveAutoAcceptEscortQuest = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanMirrorMasterDeclineQuest
-- Return the mirror master decline quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanMirrorMasterDeclineQuest( info )

    return self.db.mirrorMasterDeclineQuest
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleMirrorMasterDeclineQuest
-- Toggle the mirror master decline quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleMirrorMasterDeclineQuest( info, value )

    self.db.mirrorMasterDeclineQuest = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanMirrorMasterAbandonQuest
-- Return the mirror master abandon quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanMirrorMasterAbandonQuest( info )

    return self.db.mirrorMasterAbandonQuest
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleMirrorMasterAbandonQuest
-- Toggle the mirror master abandon quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleMirrorMasterAbandonQuest( info, value )

    self.db.mirrorMasterAbandonQuest = value

    if self.db.mirrorMasterAbandonQuest == true then
		self.jambaQuestAbandonButton:Enable()
	else
		self.jambaQuestAbandonButton:Disable()
	end
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanMirrorMasterCompleteQuestNoReward
-- Return the mirror master complete quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanMirrorMasterCompleteQuestNoReward( info )

    return self.db.mirrorMasterCompleteQuestNoReward
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleMirrorMasterCompleteQuestNoReward
-- Toggle the mirror master complete quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleMirrorMasterCompleteQuestNoReward( info, value )

    self.db.mirrorMasterCompleteQuestNoReward = value
    if value == true and self.db.slaveAutoCompleteQuest == true then
		self.db.slaveAutoCompleteQuest = false
	end
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanMirrorMasterCompleteQuestWithReward
-- Return the mirror master complete quest with reward flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanMirrorMasterCompleteQuestWithReward( info )

    return self.db.mirrorMasterCompleteQuestWithReward
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleMirrorMasterCompleteQuestWithReward
-- Toggle the mirror master complete quest with reward flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleMirrorMasterCompleteQuestWithReward( info, value )

    self.db.mirrorMasterCompleteQuestWithReward = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanSlaveAutoCompleteQuest
-- Return the slave auto complete quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanSlaveAutoCompleteQuest( info )

    return self.db.slaveAutoCompleteQuest
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleSlaveAutoCompleteQuest
-- Toggle the slave auto complete quest flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleSlaveAutoCompleteQuest( info, value )

    self.db.slaveAutoCompleteQuest = value
    if value == true and self.db.mirrorMasterCompleteQuestNoReward == true then
		self.db.mirrorMasterCompleteQuestNoReward = false
	end
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanMirrorMasterQuestWatch
-- Return the mirror master quest watch flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanMirrorMasterQuestWatch( info )

    return self.db.mirrorMasterQuestWatch
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleMirrorMasterQuestWatch
-- Toggle the mirror master quest watch flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleMirrorMasterQuestWatch( info, value )

    self.db.mirrorMasterQuestWatch = value

    if self.db.mirrorMasterQuestWatch == true then
		self.jambaQuestAddWatchButton:Enable()
		self.jambaQuestRemoveWatchButton:Enable()
	else
		self.jambaQuestAddWatchButton:Disable()
		self.jambaQuestRemoveWatchButton:Disable()
	end
	    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanShowJambaQuestLogWithWowLog
-- Return the show jamba quest log with wow log flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanShowJambaQuestLogWithWowLog( info )

    return self.db.showJambaQuestLogWithWowLog
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleShowJambaQuestLogWithWowLog
-- Toggle the show jamba quest log with wow log flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleShowJambaQuestLogWithWowLog( info, value )

    self.db.showJambaQuestLogWithWowLog = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanQuestSilently
-- Return the quest silently flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanQuestSilently( info )

    return self.db.questSilently
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleQuestSilently
-- Toggle the quest silently flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleQuestSilently( info, value )

    self.db.questSilently = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CanShowQuestCommandWindow
-- Return the show quest command window flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CanShowQuestCommandWindow( info )

    return self.db.show
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ToggleShowQuestCommandWindow
-- Toggle the show quest command window flag.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ToggleShowQuestCommandWindow( info, value )

    self.db.show = value
    
    if self.db.show == true then	
		self.jambaQuestLogFrame:Show()
    else
		self.jambaQuestLogFrame:Hide()
    end
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- QUEST EVENT PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ProcessQuestDetail
-- Process the quest detail event (fires when the quest detail frame is displayed).
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ProcessQuestDetail()

	-- If the auto accept quest option is on.
	if self.db.slaveAutoAcceptQuest then
		-- And not the master.
		if not self.comms:CharacterIsTheMaster( self.playerName ) then
			-- Tell the master a quest was accepted?
			if not self.db.questSilently then
				self.comms:TellMaster( self.playerName, L["Accepted Quest: "]..GetTitleText() )
			end
			AcceptQuest()	
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ProcessQuestAcceptConfirm
-- Process the quest accept confirm event (fires for escort quests - do you want to join question).
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ProcessQuestAcceptConfirm()

	-- If the auto accept quest option is on.
	if self.db.slaveAutoAcceptEscortQuest then	
		-- And not the master.
		if not self.comms:CharacterIsTheMaster( self.playerName ) then			
			-- Tell the master a quest was accepted?
			if not self.db.questSilently then
				self.comms:TellMaster( self.playerName, L["Accepted Quest: "]..GetTitleText() )
			end
			ConfirmAcceptQuest()
			StaticPopup_Hide( "QUEST_ACCEPT" )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:ProcessQuestComplete
-- Handle the quest complete event (fires for before the complete quest button is shown).
-------------------------------------------------------------------------------------------------------------
function JambaQuest:ProcessQuestComplete()

	-- If the auto accept quest option is on.
	if self.db.slaveAutoCompleteQuest then	
		-- And not the master.
		if not self.comms:CharacterIsTheMaster( self.playerName ) then		
			-- If there is 1 choice, then complete the quest.	
			self.comms:TellSelf( self.playerName, L["Quest has X reward choices."]( GetNumQuestChoices() ) )
			if GetNumQuestChoices() <= 1 then
				GetQuestReward( GetNumQuestChoices() )			
			end
			-- Tell the master a quest was completed?
			if not self.db.questSilently then
				self.comms:TellMaster( self.playerName, L["Completed Quest: "]..GetTitleText() )
			end			
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- NPC QUEST PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookSelectGossipOption
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookSelectGossipOption( questIndex )

	-- If mirror master is on...
	if self.db.mirrorMasterQuestSelection then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_SELECT_GOSSIP_OPTION, tostring( questIndex )  )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoSelectGossipOption
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoSelectGossipOption( sender, questIndex )

	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
			SelectGossipOption( questIndex )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookSelectGossipActiveQuest
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookSelectGossipActiveQuest( questIndex )

	-- If mirror master is on...
	if self.db.mirrorMasterQuestSelection then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_SELECT_GOSSIP_ACTIVE_QUEST, tostring( questIndex )  )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoSelectGossipActiveQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoSelectGossipActiveQuest( sender, questIndex )

	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
			SelectGossipActiveQuest( questIndex )
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookSelectGossipAvailableQuest
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookSelectGossipAvailableQuest( questIndex )

	-- If mirror master is on...
	if self.db.mirrorMasterQuestSelection then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_SELECT_GOSSIP_AVAILABLE_QUEST, tostring( questIndex )  )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoSelectGossipAvailableQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoSelectGossipAvailableQuest( sender, questIndex )

	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
			SelectGossipAvailableQuest( questIndex )
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookSelectActiveQuest
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookSelectActiveQuest( questIndex )

	-- If mirror master is on...
	if self.db.mirrorMasterQuestSelection then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_SELECT_ACTIVE_QUEST, tostring( questIndex )  )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoSelectActiveQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoSelectActiveQuest( sender, questIndex )

	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
			SelectActiveQuest( questIndex )
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookSelectAvailableQuest
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookSelectAvailableQuest( questIndex )

	-- If mirror master is on...
	if self.db.mirrorMasterQuestSelection then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_SELECT_AVAILABLE_QUEST, tostring( questIndex )  )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoSelectAvailableQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoSelectAvailableQuest( sender, questIndex )

	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then	
			SelectAvailableQuest( questIndex )
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookAcceptQuest
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookAcceptQuest()

	-- If mirror master is on...
	if self.db.mirrorMasterQuestSelection then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_ACCEPT_QUEST )
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoAcceptQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoAcceptQuest( sender )

	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
			AcceptQuest()
			if not self.db.questSilently then
				self.comms:TellMaster( self.playerName, L["Accepted Quest: "]..GetTitleText() )
			end
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookDeclineQuest
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookDeclineQuest()

	-- If mirror master is on...
	if self.db.mirrorMasterDeclineQuest then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_DECLINE_QUEST  )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoDeclineQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoDeclineQuest( sender )

	-- If the mirror master option is on...
	if self.db.mirrorMasterDeclineQuest then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
			DeclineQuest()
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookCompleteQuest
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookCompleteQuest()
          	
	-- If mirror master is on for completing a quest...
	if self.db.mirrorMasterCompleteQuestNoReward or self.db.mirrorMasterCompleteQuestWithReward then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_COMPLETE_QUEST  )
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoCompleteQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoCompleteQuest( sender )

	-- If the mirror master option is on...
	if self.db.mirrorMasterCompleteQuestNoReward or self.db.mirrorMasterCompleteQuestWithReward then
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then			
			CompleteQuest()
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookGetQuestReward
-- Hooked function - happens when the character has completed a quest item.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookGetQuestReward( questIndex )

	-- If mirror master is on...
	if self.db.mirrorMasterCompleteQuestNoReward or self.db.mirrorMasterCompleteQuestWithReward then
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			self.comms:CommandSlaves( self.utils.COMMAND_CHOOSE_QUEST_REWARD, tostring( questIndex ) )
		end
	end
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoChooseQuestReward
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoChooseQuestReward( sender, questIndex )

	local canChooseQuestReward = false
		
	-- If mirror master complete quest either version is selected...
	if self.db.mirrorMasterCompleteQuestNoReward or self.db.mirrorMasterCompleteQuestWithReward then
	
		-- If complete quest with reward is on, then allow quest reward choosing.
		if self.db.mirrorMasterCompleteQuestWithReward then
		
			-- Can choose a quest reward.
			canChooseQuestReward = true
			
		else
		
			-- And if complete quest with no reward is on...
			if self.db.mirrorMasterCompleteQuestNoReward then		
				-- And the reward choice was less than one or equal to one...
				if not self.db.questSilently then
					self.comms:TellSelf( self.playerName, L["Quest has X reward choices."]( GetNumQuestChoices() ) )
				end
				if GetNumQuestChoices() <= 1 then
					-- Then allow a get quest reward action as there is only one reward choice.
					canChooseQuestReward = true
				end
			end
			
		end
		
	end
		
	if canChooseQuestReward then		
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then
			GetQuestReward( questIndex )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- QUEST LOG PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookSelectQuestLogEntry
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookSelectQuestLogEntry( questIndex )
	
	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
	
		-- If this character is the master then tell the other characters.
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			if questIndex then
				local questName, questlevel, questTag, suggestedGroup, 
					isHeader, isCollapsed, isComplete, isDaily = GetQuestLogTitle( questIndex )
				if questName and not isHeader then
				
					-- Remember this quest for sending to slaves.
					self.currentSelectedQuest = questName
					
					-- Show this quest in the jamba quest log window.
					local questColour = GetDifficultyColor( questlevel )
					self.jambaQuestLogFrameQuestName:SetText( self.currentSelectedQuest )
					self.jambaQuestLogFrameQuestName:SetTextColor( questColour.r, questColour.g, questColour.b )

				end
			end	
		end
		
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:SendSelectQuestCommand
-- Send the select quest command to all the slaves.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:SendSelectQuestCommand()

	if self.currentSelectedQuest ~= self.lastQuestSelectedCommandSent then										
		self.comms:CommandSlaves( self.utils.COMMAND_SELECT_QUEST_LOG_ENTRY, self.currentSelectedQuest )
		self.lastQuestSelectedCommandSent = self.currentSelectedQuest
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:CurrentSelectedQuestIsValid
-- Return true if the currently selected quest is valid.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:CurrentSelectedQuestIsValid()
	
	local isValid = false
	if self.currentSelectedQuest:trim() ~= "" and self.currentSelectedQuest ~= L["(No Quest Selected)"] then
		isValid = true
	end
	return isValid
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoSelectQuestLogEntry
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoSelectQuestLogEntry( sender, questName )
	
	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestSelection then
	
		-- If the sender was not this character and is the master then...
		if (sender ~= self.playerName) and (self.comms:CharacterIsTheMaster( sender )) then	
			local questIndex = GetQuestLogIndexByName( "  "..questName )
			if questIndex then
				SelectQuestLogEntry( questIndex )
				self:UpdateTheQuestLog( questIndex )
			else
				if not self.db.questSilently then
					self.comms:TellMaster( self.playerName, L["I do not have the quest: "]..questName )
				end
			end
		end
		
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:SendAbandonQuestCommand
-- Send the abandon quest command to all the slaves.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:SendAbandonQuestCommand()

	if self.db.mirrorMasterAbandonQuest then
		if JambaQuest:CurrentSelectedQuestIsValid() then
			-- If this character is the master, abandon the quest as well.
			if JambaComms:CharacterIsTheMaster( self.playerName ) then
				JambaQuest:DoAbandonQuest( self.playerName, self.currentSelectedQuest )
			end
			-- Tell the slaves to abandon the quest.		
			self.comms:CommandSlaves( self.utils.COMMAND_ABANDON_QUEST, self.currentSelectedQuest )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoAbandonQuest
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoAbandonQuest( sender, questName )

	-- If the mirror master option is on...
	if self.db.mirrorMasterAbandonQuest then
		-- If the sender is the master then...
		if self.comms:CharacterIsTheMaster( sender ) then
			local questIndex = GetQuestLogIndexByName( "  "..questName )
			if questIndex then
				SelectQuestLogEntry( questIndex )
				SetAbandonQuest()
				AbandonQuest()
				self:UpdateTheQuestLog( questIndex )
				if not self.db.questSilently then
					self.comms:TellMaster( self.playerName, L["I have abandoned the quest: "]..questName )
				end
			else
				if not self.db.questSilently then
					self.comms:TellMaster( self.playerName, L["I do not have the quest: "]..questName )
				end
			end
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:SendAddWatchQuestCommand
-- Send the abandon quest command to all the slaves.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:SendAddWatchQuestCommand()

	-- If mirror master is on...
	if self.db.mirrorMasterQuestWatch then
		if JambaQuest:CurrentSelectedQuestIsValid() then
			-- If this character is the master, watch the quest as well.
			if JambaComms:CharacterIsTheMaster( self.playerName ) then
				JambaQuest:DoQuestWatchAdd( self.playerName, self.currentSelectedQuest )
			end
			self.comms:CommandSlaves( self.utils.COMMAND_QUEST_WATCH_ADD, self.currentSelectedQuest )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoQuestWatchAdd
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoQuestWatchAdd( sender, questName )

	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestWatch then
	
		-- If the sender is the master then...
		if self.comms:CharacterIsTheMaster( sender ) then
			local questIndex = GetQuestLogIndexByName( "  "..questName )
			if questIndex then
				AddQuestWatch( questIndex )
				self:UpdateTheQuestLog( questIndex )
			else
				if not self.db.questSilently then
					self.comms:TellMaster( self.playerName, L["I do not have the quest: "]..questName )
				end
			end
		end
		
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:SendRemoveWatchQuestCommand
-- Send the abandon quest command to all the slaves.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:SendRemoveWatchQuestCommand()

	-- If mirror master is on...
	if self.db.mirrorMasterQuestWatch then
		if JambaQuest:CurrentSelectedQuestIsValid() then
			-- If this character is the master, watch the quest as well.
			if JambaComms:CharacterIsTheMaster( self.playerName ) then
				JambaQuest:DoQuestWatchRemove( self.playerName, self.currentSelectedQuest )
			end
			self.comms:CommandSlaves( self.utils.COMMAND_QUEST_WATCH_REMOVE, self.currentSelectedQuest )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:DoQuestWatchRemove
-- Called after another character has already done the action.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:DoQuestWatchRemove( sender, questName )
	
	-- If the mirror master option is on...
	if self.db.mirrorMasterQuestWatch then
	
		-- If the sender is the master then...
		if self.comms:CharacterIsTheMaster( sender ) then
			local questIndex = GetQuestLogIndexByName( "  "..questName )
			if questIndex then
				RemoveQuestWatch( questIndex )
				self:UpdateTheQuestLog( questIndex )
			else
				if not self.db.questSilently then
					self.comms:TellMaster( self.playerName, L["I do not have the quest: "]..questName )
				end
			end
		end
		
	end
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HandleCommand
-- Handle commands for JambaQuest.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HandleCommand( sender, command, questIndex )

	--LibStub( "AceConsole-3.0" ):Print( "JambaQuest Handle COMMAND:", command, questIndex )
	
	if command == self.utils.COMMAND_SELECT_GOSSIP_OPTION then		
		self:DoSelectGossipOption( sender, tonumber( questIndex ) )
	end
	if command == self.utils.COMMAND_SELECT_GOSSIP_ACTIVE_QUEST then		
		self:DoSelectGossipActiveQuest( sender, tonumber( questIndex ) )
	end
	if command == self.utils.COMMAND_SELECT_GOSSIP_AVAILABLE_QUEST then		
		self:DoSelectGossipAvailableQuest( sender, tonumber( questIndex ) )
	end
	if command == self.utils.COMMAND_SELECT_ACTIVE_QUEST then		
		self:DoSelectActiveQuest( sender, tonumber( questIndex ) )
	end
	if command == self.utils.COMMAND_SELECT_AVAILABLE_QUEST then		
		self:DoSelectAvailableQuest( sender, tonumber( questIndex ) )
	end
	if command == self.utils.COMMAND_ACCEPT_QUEST then		
		self:DoAcceptQuest( sender )
	end
	if command == self.utils.COMMAND_COMPLETE_QUEST then		
		self:DoCompleteQuest( sender )
	end		
	if command == self.utils.COMMAND_CHOOSE_QUEST_REWARD then		
		self:DoChooseQuestReward( sender, tonumber( questIndex ) )
	end
	if command == self.utils.COMMAND_DECLINE_QUEST then		
		self:DoDeclineQuest( sender )
	end
	if command == self.utils.COMMAND_ABANDON_QUEST then		
		self:DoAbandonQuest( sender, questIndex )
	end
	if command == self.utils.COMMAND_QUEST_WATCH_ADD then
		self:DoQuestWatchAdd( sender, questIndex )
	end
	if command == self.utils.COMMAND_QUEST_WATCH_REMOVE then
		JambaQuest:DoQuestWatchRemove( sender, questIndex )
	end
	if command == self.utils.COMMAND_SELECT_QUEST_LOG_ENTRY then
		self:DoSelectQuestLogEntry( sender, questIndex )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:HookToggleQuestLog
-- Called after the quest log is toggled.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:HookToggleQuestLog()

	if self.db.showJambaQuestLogWithWowLog then
		if self.comms:CharacterIsTheMaster( self.playerName ) then
			if JambaQuest:IsQuestLogFrameVisible() then
				JambaQuest:ToggleShowQuestCommandWindow( nil, true )		
			else
				JambaQuest:ToggleShowQuestCommandWindow( nil, false )
			end
		end
	end
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:IsQuestLogFrameVisible
-- Return true if the quest log frame is visible.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:IsQuestLogFrameVisible()

	-- Theses are IU functions in Blizzard's XML/LUA that are called to refresh certain GUI elements.
	-- If Blizzard were to change these; they only need to be changed here.  If a user has an addon that modifies the quest log 
	-- and does not provide these functions, then there is provision to detect said addons and dynamically change the function that
	-- is called if necessary.

	return QuestLogFrame:IsVisible()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaQuest:UpdateTheQuestLog
-- Update the characters quest log.
-------------------------------------------------------------------------------------------------------------
function JambaQuest:UpdateTheQuestLog( questIndex )

	-- Theses are IU functions in Blizzard's XML/LUA that are called to refresh certain GUI elements.
	-- If Blizzard were to change these; they only need to be changed here.  If a user has an addon that modifies the quest log 
	-- and does not provide these functions, then there is provision to detect said addons and dynamically change the function that
	-- is called if necessary.
	
	if questIndex then
		QuestLog_SetSelection( questIndex )
	end
	QuestLog_Update()
	QuestLog_UpdateQuestDetails()
	QuestWatch_Update()
	
end
