﻿-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaNo - Handles a character being bothered.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaNo-1.0", 0.1
local JambaNo = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaNo then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaNo:Initialize
-- Initializes JambaNo.
-------------------------------------------------------------------------------------------------------------
function JambaNo:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-No", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-No", "Jamba-No" )
				
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo:GetOptionsConfiguration
-- Returns the configuration options table for JambaNo.
-------------------------------------------------------------------------------------------------------------
function JambaNo:GetOptionsConfiguration()

	if not self.configuration then
	
		self.configuration = {
			name = "Jamba-No",
			handler = JambaNo,
			type = 'group',
			args = {
				autoDenyDuels = {
					type = "toggle",
					name = L["Auto Deny Duels"],
					desc = L["Automatically deny duel requests."],
					get = "CanAutoDenyDuels",
					set = "ToggleAutoDenyDuels",
					width = "full",
					order = 0,
				},        
				autoDenyDuelsSilently = {
					type = "toggle",
					name = L["Deny Duels Silently"],
					desc = L["Do not tell the master if the character has denied a duel."],
					get = "CanAutoDenyDuelsSilently",
					set = "ToggleAutoDenyDuelsSilently",
					disabled = function() return not JambaNo:CanAutoDenyDuels() end,
					width = "full",
					order = 1,
				},  
				autoDenyGuildInvites = {
					type = "toggle",
					name = L["Auto Deny Guild Invites"],
					desc = L["Automatically deny guild invitations."],
					get = "CanAutoDenyGuildInvites",
					set = "ToggleAutoDenyGuildInvites",
					width = "full",
					order = 2,
				},        
				autoDenyGuildInvitesSilently = {
					type = "toggle",
					name = L["Deny Guild Invites Silently"],
					desc = L["Do not tell the master if the character has denied a guild invite."],
					get = "CanAutoDenyGuildInvitesSilently",
					set = "ToggleAutoDenyGuildInvitesSilently",
					disabled = function() return not JambaNo:CanAutoDenyGuildInvites() end,
					width = "full",
					order = 3,
				},  
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 4,			
				},									 
			},
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaNo.
-------------------------------------------------------------------------------------------------------------
function JambaNo:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo:RefreshOptions
-- Refresh the JambaNo options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaNo:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-No" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo:PushSettingsToEnabledCharacters
-- Push the settings for JambaNo to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaNo:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-No" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo:ReceiveSettings
-- Receive settings for JambaNo.
-------------------------------------------------------------------------------------------------------------
function JambaNo:ReceiveSettings( settings )

	-- Set the new settings.
    self.db.autoDenyDuels = settings.autoDenyDuels
    self.db.autoDenyDuelsSilently = settings.autoDenyDuelsSilently
    self.db.autoDenyGuildInvites = settings.autoDenyGuildInvites
    self.db.autoDenyGuildInvitesSilently = settings.autoDenyGuildInvitesSilently
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaNo: CanAutoDenyDuels
-- Return the auto deny duels flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:CanAutoDenyDuels( info )

    return self.db.autoDenyDuels
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo: ToggleAutoDenyDuels
-- Toggle the auto deny duels flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:ToggleAutoDenyDuels( info, value )

    self.db.autoDenyDuels = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo: CanAutoDenyDuelsSilently
-- Return the deny duels quietly flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:CanAutoDenyDuelsSilently( info )

    return self.db.autoDenyDuelsSilently
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo: ToggleAutoDenyDuelsSilently
-- Toggle the deny duels quietly flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:ToggleAutoDenyDuelsSilently( info, value )

    self.db.autoDenyDuelsSilently = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo: CanAutoDenyGuildInvites
-- Return the auto deny guild invites flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:CanAutoDenyGuildInvites( info )

    return self.db.autoDenyGuildInvites
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo: ToggleAutoDenyGuildInvites
-- Toggle the auto deny guild invites flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:ToggleAutoDenyGuildInvites( info, value )

    self.db.autoDenyGuildInvites = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo: CanAutoDenyGuildInvitesSilently
-- Return the deny guild invites quietly flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:CanAutoDenyGuildInvitesSilently( info )

    return self.db.autoDenyGuildInvitesSilently
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo: ToggleAutoDenyGuildInvitesSilently
-- Toggle the deny guild invites quietly flag.
-------------------------------------------------------------------------------------------------------------
function JambaNo:ToggleAutoDenyGuildInvitesSilently( info, value )

    self.db.autoDenyGuildInvitesSilently= value
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- JUST SAY NO PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaNo:ProcessDuelRequested
-- Process the duel requested event.
-------------------------------------------------------------------------------------------------------------
function JambaNo:ProcessDuelRequested()

	-- If the deny duels option is on.
	if self.db.autoDenyDuels then

		-- Whose the challenger?
		local challenger = arg1
	
		-- Cancel the duel.
		CancelDuel()
		StaticPopup_Hide( "DUEL_REQUESTED" )
		
		-- Tell the master.
		if not self.db.autoDenyDuelsSilently then
			self.comms:TellMaster( self.playerName, L["I refused a duel from: "]..challenger )
		end

	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaNo:ProcessGuildInvite
-- Process the guild invite event.
-------------------------------------------------------------------------------------------------------------
function JambaNo:ProcessGuildInvite()

	-- If the deny duels option is on.
	if self.db.autoDenyGuildInvites then

		-- Whose the inviter?
		local inviter = arg1
		local guild = arg2
	
		-- Decline the invite.
		DeclineGuild()
		StaticPopup_Hide( "GUILD_INVITE" )
		
		-- Tell the master.
		if not self.db.autoDenyGuildInvitesSilently then
			self.comms:TellMaster( self.playerName, L["I refused a guild invite to: "]..guild..L[" from: "]..inviter )
		end

	end
	
end